/* Generic scrollbar implementation.
   Copyright (C) 1994, 1995 Board of Trustees, University of Illinois.
   Copyright (C) 1995 Free Software Foundation, Inc.
   Copyright (C) 1995 Sun Microsystems, Inc.
   Copyright (C) 1995 Darrell Kindred <dkindred+@cmu.edu>.
   Copyright (C) 2003, 2010 Ben Wing.

This file is part of XEmacs.

XEmacs is free software: you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation, either version 3 of the License, or (at your
option) any later version.

XEmacs is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with XEmacs.  If not, see <http://www.gnu.org/licenses/>. */

/* Synched up with: Not in FSF. */

/* This file has been Mule-ized. */

#include <config.h>
#include "lisp.h"

#include "buffer.h"
#include "commands.h"
#include "scrollbar.h"
#include "device-impl.h"
#include "frame-impl.h"
#include "glyphs.h"
#include "gutter.h"
#include "lisp.h"
#include "window.h"
#include "line-number.h"

Lisp_Object Qinit_scrollbar_from_resources;

Lisp_Object Qscrollbar_line_up;
Lisp_Object Qscrollbar_line_down;
Lisp_Object Qscrollbar_page_up;
Lisp_Object Qscrollbar_page_down;
Lisp_Object Qscrollbar_to_top;
Lisp_Object Qscrollbar_to_bottom;
Lisp_Object Qscrollbar_vertical_drag;

Lisp_Object Qscrollbar_char_left;
Lisp_Object Qscrollbar_char_right;
Lisp_Object Qscrollbar_page_left;
Lisp_Object Qscrollbar_page_right;
Lisp_Object Qscrollbar_to_left;
Lisp_Object Qscrollbar_to_right;
Lisp_Object Qscrollbar_horizontal_drag;

#define DEFAULT_SCROLLBAR_WIDTH 15
#define DEFAULT_SCROLLBAR_HEIGHT 15

/* Width and height of the scrollbar. */
Lisp_Object Vscrollbar_width;
Lisp_Object Vscrollbar_height;

/* Scrollbar visibility specifiers */
Lisp_Object Vhorizontal_scrollbar_visible_p;
Lisp_Object Vvertical_scrollbar_visible_p;

/* Scrollbar location specifiers */
Lisp_Object Vscrollbar_on_left_p;
Lisp_Object Vscrollbar_on_top_p;

Lisp_Object Vscrollbar_pointer_glyph;

EXFUN (Fcenter_to_window_line, 2);

static void update_scrollbar_instance (struct window *w, int vertical,
				       struct scrollbar_instance *instance);

static const struct memory_description scrollbar_instance_description [] = {
  { XD_LISP_OBJECT, offsetof (struct scrollbar_instance, mirror) },
  { XD_LISP_OBJECT, offsetof (struct scrollbar_instance, next) },
  { XD_END }
};


DEFINE_NODUMP_INTERNAL_LISP_OBJECT ("scrollbar-instance", scrollbar_instance,
				    scrollbar_instance_description,
				    struct scrollbar_instance);

static void
free_scrollbar_instance (struct scrollbar_instance *instance,
			 struct frame *frame)
{
  if (!instance)
    return;
  else
    {
      struct device *d = XDEVICE (frame->device);

      MAYBE_DEVMETH (d, free_scrollbar_instance, (instance));
      /* not worth calling free_normal_lisp_object() -- scrollbar instances
	 are not created that frequently and it's dangerous. */
    }
}

static void
free_window_mirror_scrollbars (struct window_mirror *mir)
{
  free_scrollbar_instance (mir->scrollbar_vertical_instance, mir->frame);
  mir->scrollbar_vertical_instance = 0;

  free_scrollbar_instance (mir->scrollbar_horizontal_instance, mir->frame);
  mir->scrollbar_horizontal_instance = 0;
}

static struct window_mirror *
free_scrollbars_loop (Lisp_Object window, struct window_mirror *mir)
{
  struct window_mirror *retval = NULL;

  while (mir)
    {
      assert (!NILP (window));

      if (mir->vchild)
	{
	  retval = free_scrollbars_loop (XWINDOW (window)->vchild,
					 mir->vchild);
	}
      else if (mir->hchild)
	{
	  retval = free_scrollbars_loop (XWINDOW (window)->hchild,
					 mir->hchild);
	}

      if (retval != NULL)
	return retval;

      if (mir->scrollbar_vertical_instance ||
	  mir->scrollbar_horizontal_instance)
	free_window_mirror_scrollbars (mir);

      mir = mir->next;
      window = XWINDOW (window)->next;
    }

  return NULL;
}

/* Destroy all scrollbars associated with FRAME.  Only called from
   delete_frame_internal. */
void
free_frame_scrollbars (struct frame *f)
{
  if (!HAS_FRAMEMETH_P (f, create_scrollbar_instance))
    return;

  if (f->mirror_dirty)
    update_frame_window_mirror (f);

  free_scrollbars_loop (f->root_window, XWINDOW_MIRROR (f->root_mirror));

  while (FRAME_SB_VCACHE (f))
    {
      struct scrollbar_instance *tofree = FRAME_SB_VCACHE (f);
      FRAME_SB_VCACHE (f) = FRAME_SB_VCACHE (f)->next;
      tofree->next = NULL;
      free_scrollbar_instance (tofree, f);
    }

  while (FRAME_SB_HCACHE (f))
    {
      struct scrollbar_instance *tofree = FRAME_SB_HCACHE (f);
      FRAME_SB_HCACHE (f) = FRAME_SB_HCACHE (f)->next;
      tofree->next = NULL;
      free_scrollbar_instance (tofree, f);
    }
}


static struct scrollbar_instance *
create_scrollbar_instance (struct frame *f, int vertical)
{
  struct device *d = XDEVICE (f->device);
  Lisp_Object obj = ALLOC_NORMAL_LISP_OBJECT (scrollbar_instance);
  struct scrollbar_instance *instance = XSCROLLBAR_INSTANCE (obj);

  MAYBE_DEVMETH (d, create_scrollbar_instance, (f, vertical, instance));

  return instance;
}


#define GET_SCROLLBAR_INSTANCE_INTERNAL(cache)				\
  do {									\
    if (FRAME_SB_##cache (f))						\
      {									\
        struct scrollbar_instance *retval = FRAME_SB_##cache (f);	\
        FRAME_SB_##cache (f) = FRAME_SB_##cache (f)->next;		\
        retval->next = NULL;						\
        return retval;							\
      }									\
  } while (0)

static struct scrollbar_instance *
get_scrollbar_instance (struct frame *f, int vertical)
{
  /* Check if there are any available scrollbars already in existence. */
  if (vertical)
    GET_SCROLLBAR_INSTANCE_INTERNAL (VCACHE);
  else
    GET_SCROLLBAR_INSTANCE_INTERNAL (HCACHE);

  return create_scrollbar_instance (f, vertical);
}
#undef GET_SCROLLBAR_INSTANCE_INTERNAL

#define RELEASE_SCROLLBAR_INSTANCE_INTERNAL(cache)			\
  do {									\
    if (!FRAME_SB_##cache (f))						\
      {									\
	instance->next = NULL;						\
	FRAME_SB_##cache (f) = instance;				\
      }									\
    else								\
      {									\
	instance->next = FRAME_SB_##cache (f);				\
	FRAME_SB_##cache (f) = instance;				\
      }									\
  } while (0)

static void
release_scrollbar_instance (struct frame *f, int vertical,
			    struct scrollbar_instance *instance)
{
  /* #### should we do "instance->mir = 0;" for safety? */
  if (vertical)
    RELEASE_SCROLLBAR_INSTANCE_INTERNAL (VCACHE);
  else
    RELEASE_SCROLLBAR_INSTANCE_INTERNAL (HCACHE);
}
#undef RELEASE_SCROLLBAR_INSTANCE_INTERNAL

#ifdef MEMORY_USAGE_STATS

struct scrollbar_instance_stats
{
  struct usage_stats u;
  Bytecount device_data;
};

Bytecount
compute_all_scrollbar_instance_usage (struct scrollbar_instance *inst)
{
  Bytecount total = 0;

  while (inst)
    {
      total += lisp_object_memory_usage (wrap_scrollbar_instance (inst));
      inst = inst->next;
    }

  return total;
}

static void
scrollbar_instance_memory_usage (Lisp_Object scrollbar_instance,
				 struct generic_usage_stats *gustats)
{
  struct scrollbar_instance_stats *stats =
    (struct scrollbar_instance_stats *) gustats;
  struct scrollbar_instance *inst = XSCROLLBAR_INSTANCE (scrollbar_instance);
  struct device *d = FRAME_XDEVICE (inst->mirror->frame);
  Bytecount total = 0;

  if (HAS_DEVMETH_P (d, compute_scrollbar_instance_usage))
    total += DEVMETH (d, compute_scrollbar_instance_usage, (d, inst,
							    &gustats->u));

  stats->device_data = total;
}

#endif /* MEMORY_USAGE_STATS */

void
update_window_scrollbars (struct window *w, struct window_mirror *mirror,
			  int active, int horiz_only)
{
  struct frame *f = XFRAME (w->frame);
  struct device *d = XDEVICE (f->device);
  int depth;

  if (!HAS_DEVMETH_P (d, create_scrollbar_instance))
    return;

  depth = enter_redisplay_critical_section_maybe ();

  /* It is possible for this to get called from the mirror update
     routines.  In that case the structure is in an indeterminate
     state but we know exactly what struct we are working with.  So we
     pass it in in that case.  We also take advantage of it at some
     other points where we know what the mirror struct is. */
  if (!mirror)
    mirror = find_window_mirror (w);

  if (!mirror->scrollbar_vertical_instance && active)
    mirror->scrollbar_vertical_instance = get_scrollbar_instance (f, 1);

  if (!mirror->scrollbar_horizontal_instance && active)
    mirror->scrollbar_horizontal_instance = get_scrollbar_instance (f, 0);

  if (!horiz_only && mirror->scrollbar_vertical_instance)
    {
      int size = (active ? window_scrollbar_width (w) : 0);
      struct scrollbar_instance *instance;

      instance = mirror->scrollbar_vertical_instance;
      instance->scrollbar_is_active = active;
      instance->mirror = mirror;

      if (active && size)
	update_scrollbar_instance (w, 1, instance);
      MAYBE_DEVMETH (d, update_scrollbar_instance_status,
		     (w, active, size, instance));

      if (!active)
 	{
 	  release_scrollbar_instance (f, 1, instance);
 	  mirror->scrollbar_vertical_instance = NULL;
 	}
    }

  if (mirror->scrollbar_horizontal_instance)
    {
      int size = (active ? window_scrollbar_height (w) : 0);
      struct scrollbar_instance *instance;

      instance = mirror->scrollbar_horizontal_instance;
      instance->scrollbar_is_active = active;
      instance->mirror = mirror;

      if (active && size)
	update_scrollbar_instance (w, 0, instance);
      MAYBE_DEVMETH (d, update_scrollbar_instance_status,
		     (w, active, size, instance));

      if (!active)
 	{
 	  release_scrollbar_instance (f, 0, instance);
 	  mirror->scrollbar_horizontal_instance = NULL;
 	}
    }

  exit_redisplay_critical_section_maybe (depth);
}

void
release_window_mirror_scrollbars (struct window_mirror *mir)
{
  struct device *d = XDEVICE (mir->frame->device);

  if (!HAS_DEVMETH_P (d, create_scrollbar_instance))
    return;

  if (mir->scrollbar_vertical_instance)
    {
      release_scrollbar_instance (mir->frame, 1,
				  mir->scrollbar_vertical_instance);
      MAYBE_DEVMETH (d, release_scrollbar_instance,
		     (mir->scrollbar_vertical_instance));
    }
  mir->scrollbar_vertical_instance = 0;

  if (mir->scrollbar_horizontal_instance)
    {
      release_scrollbar_instance (mir->frame, 0,
				  mir->scrollbar_horizontal_instance);
      MAYBE_DEVMETH (d, release_scrollbar_instance,
		     (mir->scrollbar_horizontal_instance));
    }
  mir->scrollbar_horizontal_instance = 0;
}

/*
 * If w->sb_point is on the top line then return w->sb_point else
 * return w->start.  If flag, then return beginning point of line
 * which w->sb_point lies on.
 */
static Bytebpos
scrollbar_point (struct window *w, Boolint flag)
{
  Bytebpos start_pos, end_pos, sb_pos;
  Lisp_Object buf;
  struct buffer *b;

  if (NILP (w->buffer)) /* non-leaf window */
    return 0;

  start_pos = marker_byte_position (w->start[CURRENT_DISP]);
  sb_pos = marker_byte_position (w->sb_point);

  if (!flag && sb_pos < start_pos)
    return start_pos;

  buf = get_buffer (w->buffer, 0);
  if (!NILP (buf))
    b = XBUFFER (buf);
  else
    return start_pos;

  if (flag)
    end_pos = byte_find_next_newline_no_quit (b, sb_pos, -1);
  else
    end_pos = byte_find_next_newline_no_quit (b, start_pos, 1);

  if (flag)
    return end_pos;
  else if (sb_pos > end_pos)
    return start_pos;
  else
    return sb_pos;
}

/*
 * Update a window's horizontal or vertical scrollbar.
 */
static void
update_scrollbar_instance (struct window *w, int vertical,
			   struct scrollbar_instance *instance)
{
  struct frame *f = XFRAME (w->frame);
  struct device *d = XDEVICE (f->device);
  struct buffer *b = XBUFFER (w->buffer);
  int scrollbar_width  = window_scrollbar_width  (w);
  int scrollbar_height = window_scrollbar_height (w);

  int new_line_increment = -1, new_page_increment = -1;
  int new_minimum = -1, new_maximum = -1;
  int new_slider_size = -1, new_slider_position = -1;
  int new_width = -1, new_height = -1, new_x = -1, new_y = -1;
#if 0
  struct window *new_window = 0; /* #### currently unused */
#endif

  if (vertical)
    {
      new_height = WINDOW_TEXT_HEIGHT (w);
      new_width = scrollbar_width;
    }
  else
    {
      new_height = scrollbar_height;
      new_width = WINDOW_TEXT_WIDTH (w);
    }

  /* If the height and width are not greater than 0, then later on the
     Motif widgets will bitch and moan. */
  if (new_height <= 0)
    new_height = 1;
  if (new_width <= 0)
    new_width = 1;

  assert (instance->mirror && XWINDOW (real_window(instance->mirror, 0)) == w);

  /* Vertical scrollbars are line-based, horizontal scrollbars are
     character-based. */

  instance->scrollbar_is_active = 1;
  new_line_increment = 1;
  new_page_increment = 1;

  /* We used to check for inhibit_scrollbar_slider_size_change here,
     but that seems bogus.  */
  {
    int x_offset, y_offset;

    /* Scrollbars are always the farthest from the text area, barring
       gutters. */
    if (vertical)
      {
	if (!NILP (w->scrollbar_on_left_p))
	  {
	    x_offset = WINDOW_LEFT (w);
	  }
	else
	  {
	    x_offset = WINDOW_RIGHT (w) - scrollbar_width;
	    if (window_needs_vertical_divider (w))
	      x_offset -= window_divider_width (w);
	  }
	y_offset = WINDOW_TEXT_TOP (w) + f->scrollbar_y_offset;
      }
    else
      {
	x_offset = WINDOW_TEXT_LEFT (w);
	y_offset = f->scrollbar_y_offset;

	if (!NILP (w->scrollbar_on_top_p))
	  {
	    y_offset += WINDOW_TOP (w);
	  }
	else
	  {
	    y_offset += WINDOW_TEXT_BOTTOM (w);
	  }
      }

    new_x = x_offset;
    new_y = y_offset;
  }

  /* A disabled scrollbar has its slider sized to the entire height of
     the scrollbar.  Currently the minibuffer scrollbar is
     disabled. */
  if (!MINI_WINDOW_P (w) && vertical)
    {
      if (!DEVMETH_OR_GIVEN (d, inhibit_scrollbar_slider_size_change, (), 0))
	{
          /* If the buffer size is so big that calculating the buffer line
             number makes no difference to the slider size when compared to
             calculating the buffer position, use the buffer position rather
             than calculating the line number, since the buffer position is so
             much cheaper. This improves performance when scrolling with huge
             buffers, where redisplay ended up tied up in calculating the line
             number.

             WINDOW_HEIGHT (w) is the window's pixel height. 

             The Bytecount of the visible region divided by 64 was my first
             empirical estimate as to how many lines per putative window
             displayed. The below calculation works much better, in particular
             because we prefer to err in the direction of calculating using
             line numbers rather than buffer positions, the performance
             difference is only really an issue when scrolling with huge
             buffers.

             With a window size of 863 pixels, the buffer size calculation
             kicks in with a buffer above 220k; for context, emacs.c is 149k,
             UnicodeData.txt is about 1700k (1.7 megabytes), Unihan.txt is
             close on 29 megabytes.

             This will get things wrong if we have a huge buffer with very few
             newlines, it will degrade to the older XEmacs performance. In the
             couple of decades I have been reading the XEmacs lists, the
             traditional XEmacs performance has never been reported as a
             problem. Aidan Kehoe, Thu Jan 21 20:19:09 GMT 2021 */
          if (WINDOW_HEIGHT (w)
              < ((BYTE_BUF_ZV (b) - BYTE_BUF_BEGV (b)) / 256))
            {
              new_minimum = BYTE_BUF_BEGV (b);
              new_maximum = BYTE_BUF_ZV (b);
              new_slider_position = scrollbar_point (w, 0);
              new_slider_size = window_displayed_height (w) * 512;
            }
          else
            {
              new_maximum = buffer_line_number (b, BYTE_BUF_ZV (b), 1, 1);
              /* window_displayed_height() gives the number of display lines,
                 which conceptually does not directly reflect the number of
                 newlines in the buffer. buffer_line_number () deals with the
                 latter. This conceptual dissonance doesn't matter. */
              new_slider_size = window_displayed_height (w);
              new_minimum = 0;
              new_slider_position
                = buffer_line_number (b, scrollbar_point (w, 0), 1, 1);
            }

          /* The end position must be strictly greater than the start
             position, at least for the Motify scrollbar.  It shouldn't hurt
             anything for other scrollbar implementations. */
          new_maximum = max (new_minimum + 1, new_maximum);
	}
    }
  else if (!MINI_WINDOW_P (w))
    {
      /* The minus one is to account for the truncation glyph. */
      int wcw = window_char_width (w, 0) - 1;
      int max_width, max_slide;

      assert (w->max_line_len != ((Charcount) 0xDEADBEEF));

      if (w->max_line_len < wcw)
	{
	  max_width = 1;
	  max_slide = 1;
	  wcw = 1;
	}
      else
	{
	  max_width = w->max_line_len + 2;
	  max_slide = max_width - wcw;
	}

      new_minimum = 0;
      new_maximum = max_width;
      new_slider_size = wcw;
      new_slider_position = min (w->hscroll, max_slide);
    }
  else /* MINI_WINDOW_P (w) */
    {
      new_minimum = 1;
      new_maximum = 2;
      new_slider_size = 1;
      new_slider_position = 1;
      instance->scrollbar_is_active = 0;
    }

  DEVMETH (d, update_scrollbar_instance_values, (w, instance,
						 new_line_increment,
						 new_page_increment,
						 new_minimum,
						 new_maximum,
						 new_slider_size,
						 new_slider_position,
						 new_width, new_height,
						 new_x, new_y));
}

void
init_frame_scrollbars (struct frame *f)
{
  struct device *d = XDEVICE (f->device);

  if (HAS_DEVMETH_P (d, create_scrollbar_instance))
    {
      int depth = unlock_ghost_specifiers_protected ();
      Lisp_Object frame = wrap_frame (f);

      call_critical_lisp_code (XDEVICE (FRAME_DEVICE (f)),
			       Qinit_scrollbar_from_resources,
			       frame);
      unbind_to (depth);
    }
}

void
init_device_scrollbars (struct device *d)
{
  if (HAS_DEVMETH_P (d, create_scrollbar_instance))
    {
      int depth = unlock_ghost_specifiers_protected ();
      Lisp_Object device = wrap_device (d);

      call_critical_lisp_code (d,
			       Qinit_scrollbar_from_resources,
			       device);
      unbind_to (depth);
    }
}

void
init_global_scrollbars (struct device *d)
{
  if (HAS_DEVMETH_P (d, create_scrollbar_instance))
    {
      int depth = unlock_ghost_specifiers_protected ();
      call_critical_lisp_code (d,
			       Qinit_scrollbar_from_resources,
			       Qglobal);
      unbind_to (depth);
    }
}

static void
vertical_scrollbar_changed_in_window (Lisp_Object UNUSED (specifier),
				      struct window *w,
				      Lisp_Object UNUSED (oldval))
{
  /* Hold on your cerebella guys. If we always show the dividers,
     changing scrollbar affects only how the text and scrollbar are
     laid out in the window. If we do not want the dividers to show up
     always, then we mark more drastic change, because changing
     divider appearance changes lotta things. Although we actually need
     to do this only if the scrollbar has appeared or disappeared
     completely at either window edge, we do this always, as users
     usually do not reposition scrollbars 200 times a second or so. Do
     you? */
  if (NILP (w->vertical_divider_always_visible_p))
    MARK_FRAME_WINDOWS_STRUCTURE_CHANGED (XFRAME (WINDOW_FRAME (w)));
  else
    MARK_WINDOWS_CHANGED (w);
}

/* This function is called as a result of a change to the
   `scrollbar-pointer' glyph.  */
static void
scrollbar_pointer_changed_in_window (Lisp_Object UNUSED (specifier),
				     struct window *w,
				     Lisp_Object UNUSED (oldval))
{
  struct frame *f = XFRAME (WINDOW_FRAME (w));

  if (f->init_finished)
    MAYBE_FRAMEMETH (f, scrollbar_pointer_changed_in_window, (w));
}

/* ####

   All of the following stuff is functions that handle scrollbar
   actions.  All of it should be moved into Lisp.  This may require
   adding some badly-needed primitives. */

/********** vertical scrollbar stuff **********/

/*
 * If the original point is still visible, put the cursor back there.
 * Otherwise, when scrolling down stick it at the beginning of the
 * first visible line and when scrolling up stick it at the beginning
 * of the last visible line.
 */

/* #### This function should be moved into Lisp */
static void
scrollbar_reset_cursor (Lisp_Object win, Bytebpos orig_pt)
{
  /* When this function is called we know that start is already
     accurate.  We know this because either set-window-start or
     recenter was called immediately prior to it being called. */
  Lisp_Object buf;
  struct window *w = XWINDOW (win);
  int selected = ((w == XWINDOW (Fselected_window (XFRAME (w->frame)->device)))
		  ? 1
		  : 0);
  Bytebpos start_pos;

  buf = Fwindow_buffer (win);
  if (NILP (buf))
    return;	/* the window was deleted out from under us */

  start_pos = marker_byte_position (w->start[CURRENT_DISP]);

  if (orig_pt < start_pos)
    {
      if (selected)
	Fgoto_char (w->start[CURRENT_DISP], buf);
      else
        set_window_point (win, marker_byte_position (w->start[CURRENT_DISP]));
    }
  else if (!point_would_be_visible (XWINDOW (win), start_pos, orig_pt, 0))
    {
      Fmove_to_window_line (make_fixnum (-1), win);

      if (selected)
        BYTE_BUF_SET_PT (XBUFFER (buf),
                         byte_find_next_newline_no_quit (XBUFFER (buf),
                                                         BYTE_BUF_PT (XBUFFER (buf)),
                                                         -1));
      else
	{
	  Bytebpos pos
            = byte_find_next_newline_no_quit
            (XBUFFER (buf), marker_byte_position (w->pointm[CURRENT_DISP]),
             -1);

          /* Adjust this so we don't have to call set_marker_restricted(). */
          pos = bytebpos_clip_to_bounds (BYTE_BUF_BEGV (XBUFFER (buf)),
                                         pos,
                                         BYTE_BUF_ZV (XBUFFER (buf)));

          set_marker_byte_position (w->pointm[CURRENT_DISP], pos, buf);
	}
    }
  else
    {
      if (selected)
        {
          BYTE_BUF_SET_PT (XBUFFER (buf), orig_pt);
        }
      else
        set_window_point (win, orig_pt);
    }
}

DEFUN ("scrollbar-line-up", Fscrollbar_line_up, 1, 1, 0, /*
Function called when the line-up arrow on the scrollbar is clicked.
This is the little arrow at the top of the scrollbar.  One argument, the
scrollbar's window.  You can advise this function to change the scrollbar
behavior.
*/
       (window))
{
  CHECK_LIVE_WINDOW (window);
  window_scroll (window, Qone, -1, ERROR_ME_NOT);
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-line-down", Fscrollbar_line_down, 1, 1, 0, /*
Function called when the line-down arrow on the scrollbar is clicked.
This is the little arrow at the bottom of the scrollbar.  One argument, the
scrollbar's window.  You can advise this function to change the scrollbar
behavior.
*/
       (window))
{
  CHECK_LIVE_WINDOW (window);
  window_scroll (window, Qone, 1, ERROR_ME_NOT);
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-page-up", Fscrollbar_page_up, 1, 1, 0, /*
Function called when the user gives the "page-up" scrollbar action.
\(The way this is done can vary from scrollbar to scrollbar.) One argument,
a cons containing the scrollbar's window and a value (#### document me!
This value is nil for Motif/Lucid scrollbars and a number for Athena
scrollbars).  You can advise this function to change the scrollbar
behavior.
*/
       (object))
{
  Lisp_Object window = Fcar (object);

  CHECK_LIVE_WINDOW (window);
  /* Motif and Athena scrollbars behave differently, but in accordance
     with their standard behaviors.  It is not possible to hide the
     differences down in lwlib because knowledge of XEmacs buffer and
     cursor motion routines is necessary. */

  if (NILP (XCDR (object)))
    window_scroll (window, Qnil, -1, ERROR_ME_NOT);
  else
    {
      Fmove_to_window_line (Qzero, window);
      Fvertical_motion (Fcdr (object), window, Qnil);
      Fcenter_to_window_line (Qzero, window);
    }

  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-page-down", Fscrollbar_page_down, 1, 1, 0, /*
Function called when the user gives the "page-down" scrollbar action.
\(The way this is done can vary from scrollbar to scrollbar.) One argument,
a cons containing the scrollbar's window and a value (#### document me!
This value is nil for Motif/Lucid scrollbars and a number for Athena
scrollbars).  You can advise this function to change the scrollbar
behavior.
*/
       (object))
{
  Lisp_Object window = Fcar (object);

  CHECK_LIVE_WINDOW (window);
  /* Motif and Athena scrollbars behave differently, but in accordance
     with their standard behaviors.  It is not possible to hide the
     differences down in lwlib because knowledge of XEmacs buffer and
     cursor motion routines is necessary. */

  if (NILP (XCDR (object)))
    window_scroll (window, Qnil, 1, ERROR_ME_NOT);
  else
    {
      Lisp_Object value = Fcdr (object);
      CHECK_FIXNUM (value);
      Fmove_to_window_line (value, window);
      Fcenter_to_window_line (Qzero, window);
    }

  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-to-top", Fscrollbar_to_top, 1, 1, 0, /*
Function called when the user invokes the "to-top" scrollbar action.
The way this is done can vary from scrollbar to scrollbar, but
C-button1 on the up-arrow is very common. One argument, the
scrollbar's window.  You can advise this function to change the
scrollbar behavior.
*/
       (window))
{
  struct window *w = decode_window (window);
  struct buffer *b = XBUFFER (w->buffer);
  Boolint selectedp
    = (w == XWINDOW (Fselected_window (XFRAME (w->frame)->device))
       && current_buffer == b);
  Bytebpos orig_pt = selectedp ? BYTE_BUF_PT (b)
    : marker_byte_position (w->pointm[CURRENT_DISP]);

  if (selectedp)
    {
      BOTH_BUF_SET_PT (b, BUF_BEGV (b), BYTE_BUF_BEGV (b));
    }
  else
    {
      set_window_point (window, BYTE_BUF_BEGV (b));
    }

  Fcenter_to_window_line (Qzero, window);
  scrollbar_reset_cursor (window, orig_pt);
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-to-bottom", Fscrollbar_to_bottom, 1, 1, 0, /*
Function called when the user invokes the "to-bottom" scrollbar action.
The way this is done can vary from scrollbar to scrollbar, but
C-button1 on the down-arrow is very common. One argument, the
scrollbar's window.  You can advise this function to change the
scrollbar behavior.
*/
       (window))
{
  struct window *w = decode_window (window);
  struct buffer *b = XBUFFER (w->buffer);
  Boolint selectedp
    = (w == XWINDOW (Fselected_window (XFRAME (w->frame)->device))
       && current_buffer == b);
  Bytebpos orig_pt = selectedp ? BYTE_BUF_PT (b)
    : marker_byte_position (w->pointm[CURRENT_DISP]);

  if (selectedp)
    {
      BOTH_BUF_SET_PT (b, BUF_ZV (b), BYTE_BUF_ZV (b));
    }
  else
    {
      set_window_point (window, BYTE_BUF_ZV (b));
    }

  Fcenter_to_window_line (make_fixnum (-3), window);
  scrollbar_reset_cursor (window, orig_pt);
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-vertical-drag", Fscrollbar_vertical_drag, 1, 1, 0, /*
Function called when the user drags the vertical scrollbar slider.
One argument, a cons containing the scrollbar's window and a value
between point-min and point-max.  You can advise this function to
change the scrollbar behavior.
*/
       (object))
{
  Lisp_Object window = Fcar (object);
  Lisp_Object value = Fcdr (object);
  struct buffer *b;
  struct window *w;
  Boolint selectedp;
  Bytebpos orig_pt;

  CHECK_WINDOW (window);

  w = XWINDOW (window);
  b = XBUFFER (w->buffer);
  selectedp = (w == XWINDOW (Fselected_window (XFRAME (w->frame)->device))
               && current_buffer == b);
  orig_pt = selectedp ? BYTE_BUF_PT (b)
    : marker_byte_position (w->pointm[CURRENT_DISP]);

  Fset_marker (XWINDOW (window)->sb_point, value, Fwindow_buffer (window));
  set_window_start (window, scrollbar_point (XWINDOW (window), 1), 1);
  scrollbar_reset_cursor (window, orig_pt);
  Fsit_for(Qzero, Qnil);
  zmacs_region_stays = 1;
  return Qnil;
}

DEFUN ("scrollbar-set-hscroll", Fscrollbar_set_hscroll, 2, 2, 0, /*
Set WINDOW's hscroll position to VALUE.
This ensures that VALUE is in the proper range for the horizontal scrollbar.
*/
       (window, value))
{
  struct window *w;
  int hscroll, wcw, max_len;

  CHECK_LIVE_WINDOW (window);
  if (!EQ (value, Qmax))
    CHECK_FIXNUM (value);

  w = XWINDOW (window);
  wcw = window_char_width (w, 0) - 1;
  /* #### We should be able to scroll further right as long as there is
     a visible truncation glyph.  This calculation for max is bogus.  */
  max_len = w->max_line_len + 2;

  if (EQ (value, Qmax) || (XFIXNUM (value) > (max_len - wcw)))
    hscroll = max_len - wcw;
  else
    hscroll = XFIXNUM (value);

  /* Can't allow this out of set-window-hscroll's acceptable range. */
  /* #### What hell on the earth this code limits scroll size to the
     machine-dependent SHORT size? -- kkm */
  if (hscroll < 0)
    hscroll = 0;
  else if (hscroll >= (1 << (SHORTBITS - 1)) - 1)
    hscroll = (1 << (SHORTBITS - 1)) - 1;

  if (hscroll != w->hscroll)
    Fset_window_hscroll (window, make_fixnum (hscroll));

  return Qnil;
}


/************************************************************************/
/*                            initialization                            */
/************************************************************************/

void
scrollbar_objects_create (void)
{
#ifdef MEMORY_USAGE_STATS
  OBJECT_HAS_METHOD (scrollbar_instance, memory_usage);
#endif
}
void
syms_of_scrollbar (void)
{
  INIT_LISP_OBJECT (scrollbar_instance);

  DEFSYMBOL (Qscrollbar_line_up);
  DEFSYMBOL (Qscrollbar_line_down);
  DEFSYMBOL (Qscrollbar_page_up);
  DEFSYMBOL (Qscrollbar_page_down);
  DEFSYMBOL (Qscrollbar_to_top);
  DEFSYMBOL (Qscrollbar_to_bottom);
  DEFSYMBOL (Qscrollbar_vertical_drag);

  DEFSYMBOL (Qscrollbar_char_left);
  DEFSYMBOL (Qscrollbar_char_right);
  DEFSYMBOL (Qscrollbar_page_left);
  DEFSYMBOL (Qscrollbar_page_right);
  DEFSYMBOL (Qscrollbar_to_left);
  DEFSYMBOL (Qscrollbar_to_right);
  DEFSYMBOL (Qscrollbar_horizontal_drag);

  DEFSYMBOL (Qinit_scrollbar_from_resources);

  /* #### All these functions should be moved into Lisp.
     See comment above. */
  DEFSUBR (Fscrollbar_line_up);
  DEFSUBR (Fscrollbar_line_down);
  DEFSUBR (Fscrollbar_page_up);
  DEFSUBR (Fscrollbar_page_down);
  DEFSUBR (Fscrollbar_to_top);
  DEFSUBR (Fscrollbar_to_bottom);
  DEFSUBR (Fscrollbar_vertical_drag);

  DEFSUBR (Fscrollbar_set_hscroll);
}

void
vars_of_scrollbar (void)
{
#ifdef MEMORY_USAGE_STATS
  OBJECT_HAS_PROPERTY
    (scrollbar_instance, memusage_stats_list,
     list1 (intern ("device-data")));
#endif /* MEMORY_USAGE_STATS */

  DEFVAR_LISP ("scrollbar-pointer-glyph", &Vscrollbar_pointer_glyph /*
*The shape of the mouse-pointer when over a scrollbar.
This is a glyph; use `set-glyph-image' to change it.
If unspecified in a particular domain, the window-system-provided
default pointer is used.
*/ );

  Fprovide (intern ("scrollbar"));
}

void
specifier_vars_of_scrollbar (void)
{
  DEFVAR_SPECIFIER ("scrollbar-width", &Vscrollbar_width /*
*Width of vertical scrollbars.
This is a specifier; use `set-specifier' to change it.
*/ );
  Vscrollbar_width = make_magic_specifier (Qnatnum);
  set_specifier_fallback
    (Vscrollbar_width,
     listu (
#ifdef HAVE_TTY
            Fcons (list1 (Qtty), Qzero),
#endif
	    Fcons (Qnil, make_fixnum (DEFAULT_SCROLLBAR_WIDTH)),
            Qunbound));
  set_specifier_caching (Vscrollbar_width,
			 offsetof (struct window, scrollbar_width),
			 vertical_scrollbar_changed_in_window,
			 offsetof (struct frame, scrollbar_width),
			 frame_size_slipped, 0);

  DEFVAR_SPECIFIER ("scrollbar-height", &Vscrollbar_height /*
*Height of horizontal scrollbars.
This is a specifier; use `set-specifier' to change it.
*/ );
  Vscrollbar_height = make_magic_specifier (Qnatnum);
  set_specifier_fallback
    (Vscrollbar_height,
#ifdef HAVE_TTY
     list2 (Fcons (list1 (Qtty), make_fixnum (0)),
	    Fcons (Qnil, make_fixnum (DEFAULT_SCROLLBAR_HEIGHT)))
#else
     list1 (Fcons (Qnil, make_fixnum (DEFAULT_SCROLLBAR_HEIGHT)))
#endif
     );
  set_specifier_caching (Vscrollbar_height,
			 offsetof (struct window, scrollbar_height),
			 some_window_value_changed,
			 offsetof (struct frame, scrollbar_height),
			 frame_size_slipped, 0);

  DEFVAR_SPECIFIER ("horizontal-scrollbar-visible-p", &Vhorizontal_scrollbar_visible_p /*
*Whether the horizontal scrollbar is visible.
This is a specifier; use `set-specifier' to change it.
*/ );
  Vhorizontal_scrollbar_visible_p = Fmake_specifier (Qboolean);
  set_specifier_fallback (Vhorizontal_scrollbar_visible_p,
			  list1 (Fcons (Qnil, Qt)));
  set_specifier_caching (Vhorizontal_scrollbar_visible_p,
			 offsetof (struct window,
				   horizontal_scrollbar_visible_p),
			 some_window_value_changed,
			 offsetof (struct frame,
				   horizontal_scrollbar_visible_p),
			 frame_size_slipped, 0);

  DEFVAR_SPECIFIER ("vertical-scrollbar-visible-p", &Vvertical_scrollbar_visible_p /*
*Whether the vertical scrollbar is visible.
This is a specifier; use `set-specifier' to change it.
*/ );
  Vvertical_scrollbar_visible_p = Fmake_specifier (Qboolean);
  set_specifier_fallback (Vvertical_scrollbar_visible_p,
			  list1 (Fcons (Qnil, Qt)));
  set_specifier_caching (Vvertical_scrollbar_visible_p,
			 offsetof (struct window,
				   vertical_scrollbar_visible_p),
			 vertical_scrollbar_changed_in_window,
			 offsetof (struct frame,
				   vertical_scrollbar_visible_p),
			 frame_size_slipped, 0);

  DEFVAR_SPECIFIER ("scrollbar-on-left-p", &Vscrollbar_on_left_p /*
*Whether the vertical scrollbar is on the left side of window or frame.
This is a specifier; use `set-specifier' to change it.
*/ );
  Vscrollbar_on_left_p = Fmake_specifier (Qboolean);

  {
    /* Kludge. Under X, we want athena scrollbars on the left,
       while all other scrollbars go on the right by default. */
    Lisp_Object fallback = list1 (Fcons (Qnil, Qnil));
#if defined (HAVE_X_WINDOWS)			\
    && !defined (LWLIB_SCROLLBARS_MOTIF)	\
    && !defined (LWLIB_SCROLLBARS_LUCID) 	\
    && !defined (LWLIB_SCROLLBARS_ATHENA3D)

    fallback = Fcons (Fcons (list1 (Qx), Qt), fallback);
#endif
    set_specifier_fallback (Vscrollbar_on_left_p, fallback);
  }

  set_specifier_caching (Vscrollbar_on_left_p,
			 offsetof (struct window, scrollbar_on_left_p),
			 vertical_scrollbar_changed_in_window,
			 offsetof (struct frame, scrollbar_on_left_p),
			 frame_size_slipped, 0);

  DEFVAR_SPECIFIER ("scrollbar-on-top-p", &Vscrollbar_on_top_p /*
*Whether the horizontal scrollbar is on the top side of window or frame.
This is a specifier; use `set-specifier' to change it.
*/ );
  Vscrollbar_on_top_p = Fmake_specifier (Qboolean);
  set_specifier_fallback (Vscrollbar_on_top_p,
			  list1 (Fcons (Qnil, Qnil)));
  set_specifier_caching (Vscrollbar_on_top_p,
			 offsetof (struct window, scrollbar_on_top_p),
			 some_window_value_changed,
			 offsetof (struct frame, scrollbar_on_top_p),
			 frame_size_slipped, 0);
}

void
complex_vars_of_scrollbar (void)
{
  Vscrollbar_pointer_glyph = Fmake_glyph_internal (Qpointer);

  set_specifier_caching (XGLYPH (Vscrollbar_pointer_glyph)->image,
			 offsetof (struct window, scrollbar_pointer),
			 scrollbar_pointer_changed_in_window,
			 0, 0, 0);
}
