// SPDX-License-Identifier: BSD-3-Clause
package org.islandoftex.albatross

import com.github.ajalt.clikt.core.CliktCommand
import com.github.ajalt.clikt.parameters.arguments.argument
import com.github.ajalt.clikt.parameters.arguments.multiple
import com.github.ajalt.clikt.parameters.arguments.transformAll
import com.github.ajalt.clikt.parameters.options.default
import com.github.ajalt.clikt.parameters.options.flag
import com.github.ajalt.clikt.parameters.options.option
import com.github.ajalt.clikt.parameters.options.versionOption
import com.github.ajalt.clikt.parameters.types.choice
import com.github.ajalt.mordant.rendering.BorderStyle
import com.github.ajalt.mordant.terminal.AnsiLevel
import java.time.LocalDate

class Application : CliktCommand(name = "albatross", printHelpOnEmptyArgs = true) {
    private val glyphs by argument(name = "glyphs").multiple(required = true).transformAll {
        it.map { s ->
            if (s.length == 1 || (s.length == 2 && Character.isSurrogatePair(s[0], s[1]))) {
                Integer.toHexString(s.codePointAt(0)).toUpperCase()
            } else {
                if (s.matches("0x[0-9a-fA-F]+".toRegex())) {
                    s.substring(2).toUpperCase()
                } else {
                    fail(
                        "Invalid input, either use\u0085" +
                                "the symbol itself or the corresponding Unicode code point."
                    )
                }
            }
        }
    }

    private val showStyles by option("--show-styles", "-s", help = "Show available font styles").flag()
    private val detailed by option("--detailed", "-d", help = "Show a detailed font list").flag()
    private val ansiLevel by option("--ansi-level", "-a", help = "Set the default ANSI level").choice(
        "n" to AnsiLevel.NONE,
        "a16" to AnsiLevel.ANSI16,
        "a256" to AnsiLevel.ANSI256,
        "tc" to AnsiLevel.TRUECOLOR
    ).default(AnsiLevel.NONE)
    private val borderStyle by option("--border-style", "-b", help = "Set the border style").choice(
        "1" to BorderStyle.ASCII,
        "2" to BorderStyle.ROUNDED,
        "3" to BorderStyle.BLANK,
        "4" to BorderStyle.DOUBLE,
        "5" to BorderStyle.SQUARE,
        "6" to BorderStyle.HEAVY
    ).default(BorderStyle.SQUARE)
    private val matchAny by option("--or", "-o", help = "Look for each glyph separately").flag()

    override fun run() {
        val configuration = Configuration(
            showStyles = showStyles,
            detailed = detailed,
            matchAny = matchAny,
            ansiLevel = ansiLevel,
            borderStyle = borderStyle
        )

        Viewer(configuration, glyphs).view()
    }
}

fun main(args: Array<String>) {
    println(
        """
                __ __           __
        .---.-.|  |  |--.---.-.|  |_.----.-----.-----.-----.
        |  _  ||  |  _  |  _  ||   _|   _|  _  |__ --|__ --|
        |___._||__|_____|___._||____|__| |_____|_____|_____|

        """.trimIndent()
    )

    Application().versionOption(Application::class.java.`package`.implementationVersion ?: "DEVELOPMENT BUILD",
        names = setOf("-V", "--version"),
        message = {
            """
            albatross $it
            Copyright (c) ${LocalDate.now().year}, Island of TeX
            All rights reserved.
            """.trimIndent()
        }).main(args)
}
