/* METAFONT display support routines for character user interface. */

#include <windows.h>
#include <stdio.h>
#include <win32dll/win32lib.h>

#define DEFAULT_WINDOW_WIDTH  500
#define DEFAULT_WINDOW_HEIGHT 500
#define MIN_WINDOW_WIDTH      300
#define MIN_WINDOW_HEIGHT     300

#define REGISTRY_KEY "Software\\Kazunori Asayama\\TeX CUI\\Metafont"

#define AREA_MARGIN 10

#define BUTTON_X      10
#define BUTTON_X_SEP  10
#define BUTTON_Y      10
#define BUTTON_WIDTH  80
#define BUTTON_HEIGHT 25

#define NEXT_BUTTON_X    BUTTON_X
#define PAUSE_BUTTON_X   BUTTON_X + BUTTON_WIDTH + BUTTON_X_SEP
#define COPY_BUTTON_X    ( BUTTON_X + (BUTTON_WIDTH + BUTTON_X_SEP)*2 )
#define INTERRUPT_BUTTON_X ( BUTTON_X + (BUTTON_WIDTH + BUTTON_X_SEP)*3 )

#define ID_OK    100
#define ID_PAUSE 101
#define ID_COPY  102
#define ID_INTERRUPT  103

/* Internal data */
typedef struct {
  HWND hMainWnd;
  HWND hDispWnd;
  HWND hMsgWnd;
  HWND hOkButton;
  HWND hCopyButton;
  HWND hPauseButton;
  HWND hInterruptButton;
  LONG winX, winY;
  DWORD winWidth, winHeight;
  LONG areaX, areaY;
  DWORD areaWidth, areaHeight;
  DWORD buttonY;
  HDC hBufDC;
  HBITMAP hBitmap;
  BOOL bPause;
  BOOL bOk;
  BOOL resize_f;
  DWORD wm_texdll;

  RECT lastSize;

  HBRUSH hBgBrush;
  HPEN hFgPen;
} DisplayInfo;

/*** Internal routines ***/

static LPCTSTR mainWindowClassName = "metafontClass";
static LPCTSTR displayWindowClassName = "displayClass";

static VOID flush_message(DisplayInfo *pdisp)
{
  MSG msg;
  while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  }
}

static VOID set_color(DisplayInfo *pdisp)
{
  if (pdisp->hFgPen) SelectObject(pdisp->hBufDC, pdisp->hFgPen);
  if (pdisp->hBgBrush) SelectObject(pdisp->hBufDC, pdisp->hBgBrush);
}

static VOID restore_color(DisplayInfo *pdisp)
{
  if (pdisp->hBgBrush)
    SelectObject(pdisp->hBufDC, GetStockObject(WHITE_BRUSH));
  if (pdisp->hFgPen)
    SelectObject(pdisp->hBufDC, GetStockObject(BLACK_PEN));
}

static VOID clear_rect(DisplayInfo *pdisp,
		       DWORD left, DWORD right,
		       DWORD top, DWORD bottom)
{
  RECT rect;
  rect.left = left;
  rect.top = top;
  rect.right = right;
  rect.bottom = bottom;
  FillRect(pdisp->hBufDC, &rect,
	   pdisp->hBgBrush ? pdisp->hBgBrush : GetStockObject(WHITE_BRUSH));
  flush_message(pdisp);
}

void pause(DisplayInfo *pdisp)
{
  if (pdisp->hOkButton) {
    MSG msg;
    pdisp->bOk = FALSE;
    while (!pdisp->bOk && pdisp->bPause && GetMessage(&msg, NULL, 0, 0)) {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }
  }
}

VOID redraw_display(DisplayInfo *pinfo, LPRECT prect)
{
  HDC hDC = GetDC(pinfo->hDispWnd);
  BitBlt(hDC,
	 prect->left, prect->top,
	 prect->right - prect->left + 1,
	 prect->bottom - prect->top + 1,
	 pinfo->hBufDC,
	 prect->left,
	 prect->top, SRCCOPY);
  ReleaseDC(pinfo->hDispWnd, hDC);
}

VOID resize_bitmap(DisplayInfo *pdisp)
{
  HDC hDC;
  if (pdisp->hBitmap) {
    DeleteObject(pdisp->hBitmap);
  }
  if (pdisp->hBufDC) {
    DeleteDC(pdisp->hBufDC);
  }
  hDC = GetDC(pdisp->hDispWnd);
  pdisp->hBufDC = CreateCompatibleDC(hDC);
  pdisp->hBitmap =
    CreateCompatibleBitmap(hDC, pdisp->areaWidth, pdisp->areaHeight);
  ReleaseDC(pdisp->hDispWnd, hDC);
  SelectObject(pdisp->hBufDC, pdisp->hBitmap);
  clear_rect(pdisp, 0, pdisp->areaWidth, 0, pdisp->areaHeight);
}

void resize_windows(DisplayInfo *pdisp)
{
  SetWindowPos(pdisp->hDispWnd, NULL, 0, 0,
	       pdisp->areaWidth, pdisp->areaHeight, SWP_NOMOVE);
  SetWindowPos(pdisp->hOkButton, NULL, NEXT_BUTTON_X, pdisp->buttonY,
	       0, 0, SWP_NOSIZE);
  SetWindowPos(pdisp->hCopyButton, NULL, COPY_BUTTON_X, pdisp->buttonY,
	       0, 0, SWP_NOSIZE);
  SetWindowPos(pdisp->hPauseButton, NULL, PAUSE_BUTTON_X, pdisp->buttonY,
	       0, 0, SWP_NOSIZE);
  SetWindowPos(pdisp->hInterruptButton, NULL, INTERRUPT_BUTTON_X, pdisp->buttonY,
	       0, 0, SWP_NOSIZE);
}

LRESULT CALLBACK DispWndProc(HWND hWnd,
				UINT uMsg,
				WPARAM wParam,
				LPARAM lParam)
{
  DisplayInfo *pinfo  = (DisplayInfo*)GetWindowLong(hWnd, 0);
  
  switch (uMsg) {
  case WM_PAINT:
    {
      RECT rect;
      if (GetUpdateRect(hWnd, &rect, FALSE)) {
	PAINTSTRUCT paint;
	BeginPaint(hWnd, &paint);
	redraw_display(pinfo, &rect);
	EndPaint(hWnd, &paint);
      }
    }
    break;
  case WM_CREATE:
    pinfo = ((LPCREATESTRUCT)lParam)->lpCreateParams;
    SetWindowLong(hWnd, 0, (ULONG)pinfo);
  default:
    return DefWindowProc(hWnd, uMsg, wParam, lParam);
  }
  return 0;
}

void PostTeXDLLMessage(DisplayInfo* pdisp, DWORD msg)
{
  if (pdisp->hMsgWnd)
    PostMessage(pdisp->hMsgWnd, pdisp->wm_texdll, msg, 0);
}

LRESULT CALLBACK MainWndProc(HWND hWnd,
				UINT uMsg,
				WPARAM wParam,
				LPARAM lParam)
{
  DisplayInfo *pinfo  = (DisplayInfo*)GetWindowLong(hWnd, 0);
  
  switch (uMsg) {
  case WM_SIZE:
    if (wParam != SIZE_MINIMIZED) {
      RECT rect;
      int diffWidth, diffHeight;
      GetWindowRect(hWnd, &rect);
      diffWidth = rect.right - rect.left -  pinfo->winWidth;
      diffHeight = rect.bottom - rect.top - pinfo->winHeight;
      pinfo->winWidth += diffWidth;
      pinfo->winHeight += diffHeight;
      pinfo->areaWidth += diffWidth;
      pinfo->areaHeight += diffHeight;
      pinfo->buttonY += diffHeight;
      resize_windows(pinfo);
      GetWindowRect(hWnd, &pinfo->lastSize);
      pinfo->resize_f = TRUE;
    }
    return DefWindowProc(hWnd, uMsg, wParam, lParam);
  case WM_COMMAND:
    switch (LOWORD(wParam)) {
    case ID_OK:
      pinfo->bOk = TRUE;
      break;
    case ID_PAUSE:
      if (pinfo->bPause) {
	pinfo->bPause = FALSE;
	SetWindowText(pinfo->hPauseButton, "Pause");
	EnableWindow(pinfo->hOkButton, FALSE);
	EnableWindow(pinfo->hCopyButton, FALSE);
      }
      else {
	pinfo->bPause = TRUE;
	SetWindowText(pinfo->hPauseButton, "No Pause");
	EnableWindow(pinfo->hOkButton, TRUE);
	EnableWindow(pinfo->hCopyButton, TRUE);
      }
      break;
    case ID_COPY:
      {
	HBITMAP hBitmap = CopyImage(pinfo->hBitmap, IMAGE_BITMAP, 0, 0,
				    LR_COPYRETURNORG);
	if (hBitmap && OpenClipboard(hWnd)) {
	  if (EmptyClipboard()) {
	    SetClipboardData(CF_BITMAP, hBitmap);
	  }
	  CloseClipboard();
	}
      }
      break;
    case ID_INTERRUPT:
      PostTeXDLLMessage(pinfo, TeXDLL_MESSAGE_INTERRUPT);
      pinfo->bOk = TRUE;
      SetActiveWindow(GetParent(pinfo->hMainWnd));
      break;
    default:
      return DefWindowProc(hWnd, uMsg, wParam, lParam);
    }
    break;
  case WM_CREATE:
    pinfo = ((LPCREATESTRUCT)lParam)->lpCreateParams;
    SetWindowLong(hWnd, 0, (ULONG)pinfo);
  default:
    return DefWindowProc(hWnd, uMsg, wParam, lParam);
  }
  return 0;
}

/*** main routines ***/

/* Initialization */
LPVOID WINAPI MFDisplayInitialize(HWND hMsgWnd)
{
  DisplayInfo *pdisp;
  DWORD value;
  LPCTSTR topKey = REGISTRY_KEY;
  
  WNDCLASS mainWndClass, displayWndClass;

  HINSTANCE hInstance = GetModuleHandle(NULL);
  
  pdisp = malloc(sizeof(DisplayInfo));
  if (pdisp == NULL) return NULL;
  
  pdisp->hMsgWnd = hMsgWnd;
  pdisp->wm_texdll = RegisterWindowMessage(TeXDLL_WM_TEXDLL);
  pdisp->bPause = TRUE;
  pdisp->winX = win32_get_registry_int(topKey, "windowX", 0);
  if (pdisp->winX < 0) pdisp->winX = 0;
  pdisp->winY = win32_get_registry_int(topKey, "windowY", 0);
  if (pdisp->winY < 0) pdisp->winY = 0;
  pdisp->winWidth =
    win32_get_registry_int(topKey, "windowWidth", DEFAULT_WINDOW_WIDTH);
  if (pdisp->winWidth < MIN_WINDOW_WIDTH)
    pdisp->winWidth = DEFAULT_WINDOW_WIDTH;
  pdisp->winHeight =
    win32_get_registry_int(topKey, "windowHeight", DEFAULT_WINDOW_HEIGHT);
  if (pdisp->winHeight < MIN_WINDOW_HEIGHT)
    pdisp->winHeight = DEFAULT_WINDOW_HEIGHT;
  
  mainWndClass.style = CS_HREDRAW | CS_VREDRAW | CS_NOCLOSE;
  mainWndClass.lpfnWndProc = MainWndProc;
  mainWndClass.cbClsExtra = 0;
  mainWndClass.cbWndExtra = sizeof(long);
  mainWndClass.hInstance = hInstance;
  mainWndClass.hIcon = NULL;
  mainWndClass.hCursor = LoadCursor(NULL, IDC_ARROW);
  mainWndClass.hbrBackground = (HBRUSH)(COLOR_3DFACE + 1);
  mainWndClass.lpszMenuName = NULL;
  mainWndClass.lpszClassName = mainWindowClassName;
  
  if (RegisterClass(&mainWndClass) == 0)
    return NULL;
  
  pdisp->hMainWnd =
    CreateWindowEx(WS_EX_WINDOWEDGE,
		   mainWindowClassName, "METAFONT Display",
		   WS_POPUP | WS_CAPTION | WS_MINIMIZEBOX | WS_SYSMENU |
		   WS_THICKFRAME,
		   pdisp->winX, pdisp->winY,
		   pdisp->winWidth, pdisp->winHeight,
		   NULL, NULL, hInstance, pdisp);
  if (!pdisp->hMainWnd) {
    free(pdisp);
    return NULL;
  }
  
  displayWndClass.style = CS_HREDRAW | CS_VREDRAW;
  displayWndClass.lpfnWndProc = DispWndProc;
  displayWndClass.cbClsExtra = 0;
  displayWndClass.cbWndExtra = sizeof(long);
  displayWndClass.hInstance = hInstance;
  displayWndClass.hIcon = NULL;
  displayWndClass.hCursor = LoadCursor(NULL, IDC_ARROW);
  displayWndClass.hbrBackground =
    pdisp->hBgBrush ? pdisp->hBgBrush : (HBRUSH)GetStockObject(WHITE_BRUSH);
  displayWndClass.lpszMenuName = NULL;
  displayWndClass.lpszClassName = displayWindowClassName;

  {
    RECT rectClient;
    DWORD widthClient, heightClient;
    GetClientRect(pdisp->hMainWnd, &rectClient);
    widthClient = rectClient.right - rectClient.left;
    heightClient = rectClient.bottom - rectClient.top;
    pdisp->areaX = AREA_MARGIN;
    pdisp->areaY = AREA_MARGIN;
    pdisp->areaWidth = widthClient - AREA_MARGIN * 2;
    pdisp->areaHeight = heightClient - AREA_MARGIN * 3 - BUTTON_HEIGHT;
    pdisp->buttonY = AREA_MARGIN * 2 + pdisp->areaHeight;
  }
  
  if (RegisterClass(&displayWndClass) == 0)
    return NULL;
  
  pdisp->hDispWnd =
    CreateWindowEx(WS_EX_CLIENTEDGE,
		   displayWindowClassName, "",
		   WS_CHILD | WS_VISIBLE,
		   pdisp->areaX, pdisp->areaY,
		   pdisp->areaWidth, pdisp->areaHeight,
		   pdisp->hMainWnd, NULL, hInstance, pdisp);
  
  {
    HDC hDC = GetDC(pdisp->hDispWnd);
    LONG fg = win32_get_registry_int(topKey, "colorForeground", -1);
    LONG bg = win32_get_registry_int(topKey, "colorBackground", -1);
    if (fg >= 0 && (ULONG)fg <= 0x00ffffff) {
      pdisp->hFgPen = CreatePen(PS_SOLID, 0, GetNearestColor(hDC, fg));
    }
    else pdisp->hFgPen = NULL;
    if (bg >= 0 && (ULONG)bg <= 0x00ffffff) {
      pdisp->hBgBrush = CreateSolidBrush(GetNearestColor(hDC, bg));
    }
    else pdisp->hBgBrush = NULL;
    ReleaseDC(pdisp->hDispWnd, hDC);
  }
  
#define CREATE_BUTTON(name, x, id) \
    CreateWindow("BUTTON", name, \
		 WS_CHILD | WS_VISIBLE, \
		 x, \
		 pdisp->buttonY, \
		 BUTTON_WIDTH, \
		 BUTTON_HEIGHT, \
		 pdisp->hMainWnd, (HMENU)(id), hInstance, NULL)

  pdisp->hOkButton = CREATE_BUTTON("Next", NEXT_BUTTON_X, ID_OK);
  pdisp->hPauseButton = CREATE_BUTTON("No Pause", PAUSE_BUTTON_X, ID_PAUSE);
  pdisp->hCopyButton = CREATE_BUTTON("Copy", COPY_BUTTON_X, ID_COPY);
  pdisp->hInterruptButton =
    CREATE_BUTTON("Interrupt", INTERRUPT_BUTTON_X, ID_INTERRUPT);
  
  resize_windows(pdisp);
  ShowWindow(pdisp->hMainWnd, SW_SHOWNOACTIVATE);
  flush_message(pdisp);
  
  pdisp->hBufDC = NULL;
  pdisp->hBitmap = NULL;
  pdisp->resize_f = FALSE;
  resize_bitmap(pdisp);
  clear_rect(pdisp, 0, pdisp->areaWidth, 0, pdisp->areaHeight);
  
  return pdisp;
}

/* update screen */
HDC WINAPI MFDisplayBeginChar(LPVOID pvinfo)
{
  DisplayInfo *pinfo = (DisplayInfo*)pvinfo;
  set_color(pinfo);
  return pinfo->hBufDC;
}

VOID WINAPI MFDisplayEndChar(LPVOID pvinfo)
{
  DisplayInfo *pinfo = (DisplayInfo*)pvinfo;
  RECT rect;
  restore_color(pinfo);
  GetWindowRect(pinfo->hDispWnd, &rect);
  OffsetRect(&rect, -rect.left, -rect.top);
  InvalidateRect(pinfo->hDispWnd, &rect, FALSE);
  UpdateWindow(pinfo->hDispWnd);
  flush_message(pinfo);
  pause(pinfo);
  if (pinfo->resize_f) {
    resize_bitmap(pinfo);
    pinfo->resize_f = FALSE;
  }
}

/* Uninitialize displaying */
VOID WINAPI MFDisplayUninitialize(LPVOID pvinfo)
{
  DisplayInfo *pdisp = (DisplayInfo*)pvinfo;
  LPCTSTR topKey = REGISTRY_KEY;

  win32_set_registry_int(HKEY_CURRENT_USER, topKey, "windowX",
			 pdisp->lastSize.left);
  win32_set_registry_int(HKEY_CURRENT_USER, topKey, "windowY",
			 pdisp->lastSize.top);
  win32_set_registry_int(HKEY_CURRENT_USER, topKey, "windowWidth",
			 pdisp->lastSize.right - pdisp->lastSize.left + 1);
  win32_set_registry_int(HKEY_CURRENT_USER, topKey, "windowHeight",
			 pdisp->lastSize.bottom - pdisp->lastSize.top + 1);

  if (pdisp->hBufDC)
    DeleteDC(pdisp->hBufDC);
  if (pdisp->hBitmap)
    DeleteObject(pdisp->hBitmap);
  flush_message(pdisp);
  DestroyWindow(pdisp->hMainWnd);
  if (pdisp->hFgPen) DeleteObject(pdisp->hFgPen);
  if (pdisp->hBgBrush) DeleteObject(pdisp->hBgBrush);
  free(pdisp);
}
