#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb Compiler
=============================================
HTML constructors
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Basic constructs
=============================================
|#

(defc html-nbsp "&nbsp;")

(deff line (&rest ct)
 (cons ct f-newline))

(deff html-begin-glue (&rest ct)
 (list "<" ct ">"))

(deff html-begin (&rest ct)
 (line (html-begin-glue ct)))

(deff html-end-glue (&rest ct)
 (html-begin-glue "/" ct))

(deff html-end (&rest ct)
 (line (html-end-glue ct)))

(deff html-wrap-glue (wrapper &rest ct)
 (list
  (html-begin-glue wrapper)
  ct
  (html-end-glue wrapper)))

(deff html-wrap (wrapper &rest ct)
 (list
  (html-begin wrapper)
  (line ct)
  (html-end wrapper)))

(deff html-wrap-parm-glue (wrapper parm &rest ct)
 (list
  (html-begin-glue wrapper " " parm)
  ct
  (html-end-glue wrapper)))

(deff html-wrap-parm (wrapper parm &rest ct)
 (list
  (html-begin wrapper " " parm)
  (line ct)
  (html-end wrapper)))

#|
=============================================
Simple formatting
=============================================
|#

(defc html-br (html-begin "br/"))

(deff html-angle (&rest ct)
 (list "&lt;" ct "&gt;"))

(deff html-string (&rest ct)
 (list "\"" ct "\""))

(deff html-p (&rest ct)
 (html-wrap "p" ct))

(deff html-p-right (&rest ct)
 (html-wrap-parm "p" "align=\"right\"" ct))

#|
=============================================
Style commands
=============================================
|#

(deff html-tt (&rest ct)
 (html-wrap-glue "tt" ct))

(deff html-bf (&rest ct)
 (html-wrap-glue "b" ct))

(deff html-it (&rest ct)
 (html-wrap-glue "i" ct))

#|
=============================================
Section commands
=============================================
|#

(deff html-h2 (&rest ct)
 (html-wrap "h2" ct))

(deff html-h3 (&rest ct)
 (html-wrap "h3" ct))

(deff html-h4 (&rest ct)
 (html-wrap "h4" ct))

#|
=============================================
Page commands
=============================================
|#

(deff html-head (&rest ct)
 (html-wrap "head" ct))

(deff html-title (&rest ct)
 (html-wrap "title" ct))

(deff html-address (&rest ct)
 (html-wrap "address" ct))

(deff html-body (&rest ct)
 (html-wrap "body" ct))

#|
=============================================
References
=============================================
(url-redirect ref path) takes a reference and a path and returns an url such as
  http://logiweb.eu/logiweb/server/relay/go/99999999999999/2/mypath
where "http://logiweb.eu/logiweb/server/relay" is the relay of the server to which the page is submitted, 99999999999999 is the reference of the current page, and /mypath is the given path.
|#

(deff url-redirect0 (ref)
 (:let ref (card2ref ref))
 (map 'list 'b-byte2hex ref))

(deff url-redirect1 (ref path)
 (list "go/" (url-redirect0 ref) "/2" path))

(deff url-redirect (ref path)
 (url-localrelay (url-redirect1 ref path)))

(deff url-abs-redirect (ref path)
 (url-relay (url-redirect1 ref path)))

#|
=============================================
Links
=============================================
|#

(deff html-ref (ref &rest ct)
 (html-wrap-parm-glue "a" (list "href=" (html-string ref)) ct))

(deff html-link0 (url label)
 (line (html-ref url label)))

(deff html-link (url label)
 (line (html-ref url label) "."))

(deff help-link (doc)
 (url-localhome "doc/" doc ".html"))

(deff help-abs-link (doc)
 (url-home "doc/" doc ".html"))

(defc html-signature
 (html-ref
  "http://www.diku.dk/~grue/index.html"
  "Klaus Grue"))

#|
=============================================
Number generation
=============================================
|#

(defc *nibble2hex*
 (list
  (char-code #\0)
  (char-code #\1)
  (char-code #\2)
  (char-code #\3)
  (char-code #\4)
  (char-code #\5)
  (char-code #\6)
  (char-code #\7)
  (char-code #\8)
  (char-code #\9)
  (char-code #\A)
  (char-code #\B)
  (char-code #\C)
  (char-code #\D)
  (char-code #\E)
  (char-code #\F)))

(deff nibble2hex (nibble)
 (nth nibble *nibble2hex*))

(deff html-number (value length radix)
 (html-number1 value length radix nil))

(deff html-number1 (value length radix result)
 (if (= length 0) result
  (html-number1
   (floor value radix)
   (- length 1)
   radix
   (cons (nibble2hex (mod value radix)) result))))

(deff html-long-number (value radix)
 (:when (= value 0) "0")
 (:let long-number (html-long-number1 (abs value) radix nil))
 (if (< value 0) (cons "-" long-number) long-number))

(deff html-long-number1 (value radix result)
 (if (= value 0) result
  (html-long-number1
   (floor value radix)
   radix
   (cons (nibble2hex (mod value radix)) result))))

(deff html-dec (value)
 (html-long-number value 10))

(deff html-hex (value)
 (html-long-number value 16))

#|
=============================================
Convert byte to various formats
=============================================
|#

(deff b-byte2hex (byte)
 (html-number byte 2 16))

(deff b-byte2dec (byte)
 (html-number byte 3 10))

(deff b-byte*2hex* (byte*)
 (map 'list 'b-byte2hex byte*))

#|
=============================================
Image generation
=============================================
|#

(deff html-singleton (&rest ct)
 (line "<" ct "/>"))

(deff html-img (&rest ct)
 (html-singleton "img " ct))

(deff html-image (&key alt height width align hspace src)
 (html-img
  (line "alt=" (html-string alt))
  (line "align=" (html-string align))
  (line "src=" (html-string src))
  (line "height=" (html-string (html-dec height)))
  (line "width=" (html-string (html-dec width)))
  (line "hspace=" (html-string (html-dec hspace)))))

(deff html-icon ()
 (html-image
  :alt "Logiweb(TM)"
  :height 62
  :width 46
  :align "right"
  :hspace 30
  :src (url-localhome "doc/logiweb.png")))

#|
=============================================
Favicon generation
=============================================
A 'favicon' = 'favorite icon' is a small icon which some browsers display in front of the url of a page.
|#

(deff html-favicon-link (name ref type)
 (html-singleton "link "
  (line "rel=" (html-string name))
  (line "href=" (html-string ref))
  (line "type=" (html-string type))))

(deff html-favicon ()
 (:let ref (url-localhome "doc/logiweb.ico"))
 (list
  (html-favicon-link "icon" ref "image/x-icon")
  (html-favicon-link "shortcut icon" ref "image/x-icon")))

#|
=============================================
Page generation
=============================================
|#

(deff html-footer (lgt)
 (html-p
  (html-address
   (line (html-ref (help-link "compiler/index") "The Logiweb compiler") ",")
   (line "version " *logiweb-version* " by")
   (line html-signature "," html-br)
   (line (html-ref (help-link "misc/time") (lgt-2-grd-utc-string lgt)))
   (line "=")
   (line (html-ref (help-link "misc/time") (lgt-2-mjd-tai-string lgt)))
   (line "=")
   (html-ref (help-link "misc/time") (lgt-2-string lgt)))))

(deff html-page (lgt title &rest body)
 (list
  (html-head (html-title title) (html-favicon))
  (html-body (html-icon) body (html-footer lgt))))

#|
=============================================
Make html directory
=============================================
|#

(deff dirp (string)
 (:when (equalp string "") nil)
 (equalp #\/ (aref string (- (length string) 1))))

(ttst (dirp "abc/"))
(ntst (dirp "abc"))
(ntst (dirp ""))

(deff html-dir (lgt dir)
 (:let dir (slash (ct2string dir)))
 (:let lgwdir (cat dir "lgwdir.html"))
 (:let index (cat dir "index.html"))
 (run-rm "-f" lgwdir)
 (:let page (html-dir0 lgt dir))
 (ct2file lgwdir page)
 (:when (probe-file index) nil)
 (:let dir1 (cd))
 (unwind-protect
  (progn
   (cd dir)
   (run-program "ln" :arguments '("-s" "lgwdir.html" "index.html") :input nil))
  (cd dir1)))

#|
(directory dir) returns a list of pathnames. If a directory contains a symbolic link then the returned pathname is the path of the directory or file pointed to. For that reason, the 'latest' symbolic links result in duplicate entries.

(directory ".../*" :full t) returns a list of four-element lists where the head of each four-element list is the symbolic link. For that reason, (map 'list 'car (directory ".../*" :full t)) returns what ls would return.

However, :full has no effect on (directory ".../*/" :full t) and it seems impossible to get the symbolic names of symbolic links to directories. For that reason the function below removes duplicates from (directory ".../*/"). Furthermore, the function below checks for the presence of "latest/".
|#

(deff html-dir0 (lgt dir)
 (:let name1 (directory (cat dir "*") :full t))
 (:let name1 (map 'list 'car name1))
 (:let name2 (directory (cat dir "*/")))
 (:let name* (append name1 name2))
 (:let name* (if (directory (cat dir "latest/")) (cons "latest/" name*) name*))
 (:let name* (map 'list 'pathname2filename name*))
 (:let name* (sort name* #'string<))
 (:let name* (remove-duplicates name* :test #'string=))
 (:let title (list "Index of " dir))
 (html-page lgt title
  (html-h2 title)
  (line (html-ref "../index.html" "Up."))
  (html-p (html-table "0" (map 'list 'html-dir1 name*)))))

#|
(deff html-dir1 (name)
 (html-row
  (if (dirp name) (html-ref (cat name "lgwdir.html") "dir") "")
  (html-ref name name)))
|#
(deff html-dir1 (name)
 (:when (not (dirp name)) (html-row "" (html-ref name name)))
 (html-row
  (html-ref (cat name "lgwdir.html") "dir")
  (html-ref (cat name "index.html") name)))

#|
=============================================
Table generation
=============================================
|#

(deff html-table (border &rest ct)
 (html-wrap-parm "table" (list "border=" (html-string border)) ct))

(deff html-row (&rest items)
 (html-wrap "tr" (html-row1 "td" items)))

(deff html-row-header (&rest items)
 (html-wrap-parm "tr" "align='left'" (html-row1 "th" items)))

(deff html-row1 (wrap items)
 (:when (null items) nil)
 (cons (html-wrap wrap (car items)) (html-row1 wrap (cdr items))))

(deff html-right-tt (&rest ct)
 (html-wrap-parm "td" "align='right'" (html-tt ct)))

(deff html-simple-row (&rest items)
 (html-wrap "tr" items))

(deff html-left (&rest ct)
 (html-wrap "td" ct))

(deff html-right (&rest ct)
 (html-wrap-parm "td" "align='right'" ct))

#|
=============================================
UTF-8 to html
=============================================
|#

(deff make-unicode2html-table ()
 (:let length 128)
 (:let table (make-array (list length)))
 (dotimes (i length) (setf (aref table i) (ct2string i)))
 (dotimes (i 32) (setf (aref table i) (ct2string (cons "^" (+ i 64)))))
 (setf (aref table f-newline) "<br>
")
 (setf (aref table (char-code #\<)) "&lt;")
 (setf (aref table (char-code #\>)) "&gt;")
 (setf (aref table (char-code #\&)) "&amp;")
 (setf (aref table (char-code #\")) "&quot;")
 (setf (aref table 127) "(DEL)")
 table)

(defc unicode2html-table (make-unicode2html-table))

(deff unicode2html (unicode)
 (:when (< unicode (length unicode2html-table))
  (aref unicode2html-table unicode))
 (format nil "&#x~x;" unicode))

(deff utf2html (utf)
 (:let unicode* (utf2unicode utf))
 (map 'list 'unicode2html unicode*))















