#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb Compiler
=============================================
Codification
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
OVERVIEW
=============================================
The present file consists of the following main sections:

Auxilliary functions
Unpacking
Codification
Compilation
Verification
Loading

Loading a page involves unpacking, codification, compilation, and verification. For more on these processes, see the comments in the section on loading.



The main data structures are:

Vector: A list of bytes which constitutes the input to the loading process.

Ref (reference cardinal): A cardinal that represents a page.

Id: A cardinal that identifies a symbol within a page. A ref/id pair identifies a symbol.

Bibliography: A list of references. The first entry of a bibliography (entry number zero) of a page is the reference of the page itself. Apart from this, bibliographies are non-circulsr.

Dictionary: An array that maps id's to arities.

Tree: A structures of form ((ref id path) tree ... tree) where 'path' is for debugging purposes, ref/id is the root of the tree, and the remaining trees are subtrees.

Body: The tree that results when unpacking a Logiweb page. Rendering of a Logiweb page is based on the body. The body corresponds to a Lisp S-expression that makes up a Lisp program.

Expansion: The tree that results when macro expanding the body.

Codex: A trie structure which represents the semantics of a page in a form suitable for mathematical reasoning and code generation. A codex is a four-dimensional array that maps symbol/aspect pairs to values. A codex is indexed by a symbol reference cardinal, a symbol id, an aspect reference cardinal, and an aspect id in that order.

Code: An optimized extract of the codex which speeds up evaluation but which is unsuited for mathematical reasoning since it is stored in an execute-only form. A code is a two-dimensional array that maps symbols to fct-structures (c.f. optimize.lisp).

Rack: A non-uniform array that maps 'hooks' to values. The array is non-uniform in the sense that the values have very different types. The rack of a page has the following hooks: vector, bibliography, cache, dictionary, body, codex, expansion, diagnose, and code.

Cache: An array that maps references to racks. A cache maps the reference of a page to the rack of the page.

Global cache: An array that maps references to caches. A global cache maps the reference of a page to the cache of the page.
|#

#|
=============================================
AUXILLIARY FUNCTIONS
=============================================
|#

#|
=============================================
Trees that represent predefined constructs
=============================================
|#

(deff card2tree (card)
 (list (list 0 card)))

(defc tree-lambda    (card2tree card-lambda   ))

(defc tree-proclaim  (card2tree card-proclaim ))
(defc tree-define    (card2tree card-define   ))
(defc tree-introduce (card2tree card-introduce))
(defc tree-hide      (card2tree card-hide     ))

#|
=============================================
Make simple error message
=============================================
Print given message and raise exception.
|#

(deff c-error (format &rest args)
 (apply 'format t format args)
 (terpri)
 (raise))

#|
=============================================
tree -> string
=============================================
When proclaiming symbols to denote predefined concepts, the predefined concept is identified by a tree that denotes a string.

(c-tree2string tree) converts a tree to a string.

(c-tree2card* tree) converts a tree to a string expressed as a list of cardinals.
|#

(deff c-tree2vector (tree)
 (:let ((ref id)) tree)
 ;2008-09-01: card2vector is for the sake of |compile ( " )|
 (:when (= ref 0) (card2vector id))
 #())

(deff c-tree2string (tree)
 (vector2string (c-tree2vector tree)))

(deff c-tree2card* (tree)
 (vector2card* (c-tree2vector tree)))

(etst (c-tree2string '((0 #(65 66 67))))
 "ABC")

#|
=============================================
Constructs
=============================================

A construct is a sequence of spaces, asterisks, and small letters from the English alphabet.

A construct has an 'arity' which equals the number of asterisks in it. As an example, the construct '* plus *' has arity two and 'if * then * else *' has arity three.

When parsing constructs, a sequence of spaces, newline characters, and comments counts as a single space. Spaces between letters separate words so that 'go to *' and 'goto *' are different constructs. Spaces before and after the construct and before and after asterisks are insignificant so that 'goto*' and ' goto * ' denote the same construct.

A construct is on 'internal normal form' if each word is preceeded by a space, asterisks are preceeded by no space, and there is no space at the end of the construct. As examples, ' go to*', ' goto*', '* plus*' and ' if* then* else*' are on internal normal form.

A construct is on 'external normal form' if there are spaces between words, spaces between words and asterisks, and no spaces at the end of the construct. As examples, 'go to *', 'goto *', '* plus *' and 'if * then * else *' are on external normal form.

A construct on internal normal form is said to be preopen (postopen) if it starts (ends) with an asterisk; it is said to be preclosed (postclosed) otherwise.

A construct is said to be
open if it is pre- and postopen,
closed if it is pre- and postclosed,
prefix if it is preopen and postclosed, and
suffix if it is preclosed and postopen.

Examples:
*+* is an open construct,
if*then*else* is a prefix construct,
*factorial is a suffix construct, and
begin*end is a closed construct.

Operators on constructs are named "f-..." because they were originally defined in frontend.lisp. But they were moved to codify.lisp when it was decided to compile/load codify.lisp before frontend.lisp.

f-placeholder is the asterisk character.

(f-arity construct) returns the arity of the given construct.

(f-internalize construct) normalises the given construct to internal form. The construct is required to be built from small letters, placeholders, and spaces.

(f-externalize construct) normalises the given construct to external form and converts it to a string. The construct is required to be built from small letters, placeholders, and spaces.

|#

(defc f-placeholder (char-code #\"))

(deff f-arity (construct)
 (count f-placeholder construct))

(etst (f-arity (ct2ct "\" plus \"")) 2)

(deff c-strip-leading-spaces (construct)
 (:when (atom construct) construct)
 (:let (char . rest) construct)
 (:when (unequal char f-space) construct)
 (c-strip-leading-spaces rest))

(deff f-internalize (construct)
 (f-internalize1 (c-strip-leading-spaces construct) nil))

(deff f-internalize1 (construct result)
 (:when (atom construct) (reverse result))
 (:let (char . construct) construct)
 (:when (unequal char f-space) (f-internalize1 construct (cons char result)))
 (:let construct (c-strip-leading-spaces construct))
 (:when (atom construct) (reverse result))
 (f-internalize1 construct (cons f-space result)))

(etst (f-internalize (ct2ct "go to \"")) (ct2ct "go to \""))
(etst (f-internalize (ct2ct "  goto  \"  ")) (ct2ct "goto \""))

(deff f-externalize (construct)
 (card*2string (f-internalize construct)))

(etst (f-externalize (ct2ct "go to \"")) "go to \"")
(etst (f-externalize (ct2ct "  goto  \"  ")) "goto \"")

#|
(deff f-internalize (construct)
 (f-internalize1 construct nil))

(deff f-internalize1 (construct inside-word-p)
 (:when (null construct) nil)
 (:let (char . construct) construct)
 (:when (equalp char f-space) (f-internalize1 construct nil))
 (:when (equalp char f-newline) (f-internalize1 construct nil))
 (:when (equalp char f-placeholder) (cons char (f-internalize1 construct nil)))
 (:when inside-word-p (cons char (f-internalize1 construct t)))
 (list* f-space char (f-internalize1 construct t)))

(etst (f-internalize (ct2ct "go to \"")) (ct2ct " go to\""))
(etst (f-internalize (ct2ct "  goto  \"  ")) (ct2ct " goto\""))
(etst (f-internalize (ct2ct "\"plus\"")) (ct2ct "\" plus\""))
(etst (f-internalize (ct2ct "if \" then \" else \"")) (ct2ct " if\" then\" else\""))
(etst (f-internalize (ct2ct "")) (ct2ct ""))
(etst (f-internalize (ct2ct "   ")) (ct2ct ""))
(etst (f-internalize (ct2ct "\"")) (ct2ct "\""))

(deff f-externalize (construct)
 (card*2string (cdr (f-externalize1 construct nil))))

(deff f-externalize1 (construct inside-word-p)
 (:when (null construct) nil)
 (:when (atom construct)
  (error "Internal check failed: wrong argument type for f-externalize"))
 (:let (char . construct) construct)
 (:when (equalp char f-space) (f-externalize1 construct nil))
 (:when (equalp char f-newline) (f-externalize1 construct nil))
 (:when (equalp char f-placeholder)
  (list* f-space char (f-externalize1 construct nil)))
 (:when inside-word-p (cons char (f-externalize1 construct t)))
 (list* f-space char (f-externalize1 construct t)))

(etst (f-externalize (ct2ct " go to\"")) "go to \"")
(etst (f-externalize (ct2ct "  goto  \"  ")) "goto \"")
(etst (f-externalize (ct2ct "\"plus\"")) "\" plus \"")
(etst (f-externalize (ct2ct "if \" then \" else \"")) "if \" then \" else \"")
(etst (f-externalize (ct2ct "")) "")
(etst (f-externalize (ct2ct "   ")) "")
(etst (f-externalize (ct2ct "\"")) "\"")
|#

#|
=============================================
Convert symbol to name
=============================================
Convert a symbol given by its ref and id to a name, represented as a string. The system returns the name if it is defined and otherwise falls back to a kana/decimal name.
|#

(deff c-symbol2name (ref id state)
 ;2008-09-01: Next line is for the sake of |compile ( " )|
 (:let id (vector2card id))
 (:when (equalp ref 0) (card2string id))
 (:let def (codex-get state ref id 0 card-name))
 (c-symbol2name0 ref id def))

(deff c-symbol2name-cache (ref id cache)
 ;2008-09-01: Next line is for the sake of |compile ( " )|
 (:let id (vector2card id))
 (:when (equalp ref 0) (card2string id))
 (:let def (aget cache ref card-codex ref id 0 card-name))
 (c-symbol2name0 ref id def))

(deff c-symbol2name0 (ref id def)
 (:let name (c-symbol2name1 def))
 (:when name name)
 (:let ref (card2card* 256 ref))
 (:let ref (subseq ref 0 3))
 (:let ref (card*2kana ref " " #'card2kana))
 (:let ref (ct2string ref))
 (format nil "~a...:~d" ref id))

(deff c-symbol2name1 (def)
 (:when (null def) nil)
 (:let tree (fourth def))
 (:let name (c-tree2card* tree))
 (:when (null name) nil)
 (f-externalize name))

(deff c-named-gensym (tree string cache)
 (:let ((ref id)) tree)
 (:let name (c-symbol2name-cache ref id cache))
 (:let name (cat name string))
 (gen-sym name))

#|
=============================================
Print code
=============================================
A 'code' is an associative structure which maps ref/id pairs to values which denote computable functions. One may interpret
  (aget code ref id)
as a compiled version of
  (codex-get state ref id 0 card-value)

The value of (aget code ref id) may be:

- nil, which represents an undefined construct, or

- 0, which represents lambda abstration,

- 1, which represents quoting,

- An fct-structure (fct symbol type arity) which can represent any other kind of construct.

Fct-structures are described in optimize.lisp.

The symbol may be

- one of the atoms ttag-apply, ttag-true, and ttag-if, which denote proclaimable computable constructs,

- a Lisp symbol which represents an introduced function, or

- a gensym, which represents a user defined function.

The type can be :lazy, :eager, and :unknown, c.f. optimize.lisp.

For all gensyms s, (symbol-value s) equals the definition of the symbol (e.g. 'define value of x pair y as lambda z dot if z then x else y').

The symbols ttag-apply, ttag-true, and ttag-if which denote proclaimable computable constructs are fboundp but not boundp: the symbol-function represents the concept and the absense of a symbol-value represents the fact that the symbol-function is defined in the source code of the Logiweb compiler rather than being user defined.

Symbols which represent introduced functions may or may not have a symbol-function defined in the source code. Introduced functions without a predefined symbol-function get their symbol-function from a user definition. For that reason, introduced functions which are not fboundp get their symbol-value set.

Gensyms are uninterned so that they can be garbage collected. This is irrelevant for a the present batch tool but is important for interactive systems in which the user may unload pages.

(c-print-code state ref &optional level) prints all value defined symbols accessible from the page with the given ref. For each symbol, the definition is printed as "lambda" or "quote" or as an fct-structure. If level is given, it should be a cardinal. When level is given, the definition of each defined symbol is printet in Logiweb source to the given level. A level of 0 causes definitions to be printet to all levels.
|#

(deff c-print-code (state ref &optional level)
 (:let cache (aget state (id-cache) ref))
 (c-print-code0 cache state level))

(deff c-print-code0 (cache state level)
 (:when (null cache) (terpri))
 (:let ((ref . rack) . cache) cache)
 (:let assoc (aget rack card-code))
 (c-print-code1 ref assoc state level)
 (c-print-code0 cache state level))

(deff c-print-code1 (ref assoc state level)
 (:when (null assoc) nil)
 (:let ((id . fct) . assoc) assoc)
 (c-print-code2 ref id fct state level)
 (c-print-code1 ref assoc state level))

(deff c-print-code2 (ref id fct state level)
 (:let name (utf2string (c-symbol2name ref id state)))
 (format t "~%~3d  ~a~%" id name)
 (:when (equalp fct fct-lambda) (format t "     lambda~%"))
 (:when (equalp fct fct-quote) (format t "     quote~%"))
 (format t "     ~s~%" fct)
 (:when (null level) nil)
 (:let (symbol :type :arity) (thefct fct))
 (when (boundp symbol)
  (c-print-tree 5 (+ 4 level) 20 (symbol-value symbol) state)
  (terpri)))

(deff c-name-split (vector index)
 (:let position (position f-placeholder vector :start index))
 (:when (null position) (list (subvector vector index)))
 (cons (subvector vector index position) (c-name-split vector (+ position 1))))

(etst (c-name-split (ct2vector "AB\"CD\"EF") 0) '(#(65 66) #(67 68) #(69 70)))

(deff c-symbol2nameaspect* (ref id state)
 (:let name (c-symbol2nameaspect ref id state))
 (:when (null name) (list (ct2vector (c-symbol2name ref id state))))
 (c-name-split name 0))

(deff c-symbol2nameaspect (ref id state)
 (:let def (codex-get state ref id 0 card-name))
 (:when (null def) nil)
 (:let (:def :aspect :lhs rhs) def)
 (:let ((ref id)) rhs)
 (:when (equalp ref 0) id)
 nil)

(deff c-print-name (indent name)
 (:when (null name) nil)
 (:let name (utf2string name))
 (:let name (string-trim " " name))
 (:when (equalp name "") nil)
 (indent indent)
 (format t "~a~%" name))

(deff c-looks-like-tree (tree state)
 (:when (atom (head tree)) nil)
 (:let ((ref idx) . tree*) tree)
 (:when (equalp ref 0) (and (null tree*) (or (integerp idx) (arrayp idx))))
 (:when (not (integerp ref)) nil)
 (aget state (id-cache) ref))

(deff c-print-tree (indent maxindent length tree state)
 (:when (equalp indent maxindent) (indent indent) (format t " ...~%"))
 (:when (not (c-looks-like-tree tree state))
  (format t "!")
  (indent indent)
  (format t "~s~%" (trace1 (- maxindent indent) length tree)))
 (:let ((ref id) . tree*) tree)
 (:when (equalp ref 0)
  (:let string (utf2string (card2vector id)))
  (:let string (safe-subseq string 0 length))
  (indent indent)
  (format t " ~s~%" string))
 (:let (name . name*) (c-symbol2nameaspect* ref id state))
 (c-print-name indent name)
 (c-print-tree* indent maxindent length tree* name* state))

(deff c-print-tree* (indent maxindent length tree* name* state)
 (:when (null tree*) nil)
 (:let (tree . tree*) tree*)
 (c-print-tree (+ indent 1) maxindent length tree state)
 (:let (name . name*) name*)
 (c-print-name indent name)
 (c-print-tree* indent maxindent length tree* name* state))

#|
=============================================
Print without newlines and indentation
=============================================
|#

(deff c-princ-name (name)
 (:when (null name) nil)
 (:let name (utf2string name))
 (format t "~a" name))

(deff c-princ-tree (indent maxindent length tree state)
 (:when (equalp indent maxindent) (format t "..."))
 (:when (not (c-looks-like-tree tree state))
  (format t "~s" (trace1 (- maxindent indent) length tree)))
 (:let ((ref id) . tree*) tree)
 (:when (equalp ref 0)
  (:let string (utf2string (card2vector id)))
  (:let string (safe-subseq string 0 length))
  (format t "~s" string))
 (:let (name . name*) (c-symbol2nameaspect* ref id state))
 (c-princ-name name)
 (c-princ-tree* indent maxindent length tree* name* state))

(deff c-princ-tree* (indent maxindent length tree* name* state)
 (:when (null tree*) nil)
 (:let (tree . tree*) tree*)
 (c-princ-tree (+ indent 1) maxindent length tree state)
 (:let (name . name*) name*)
 (c-princ-name name)
 (c-princ-tree* indent maxindent length tree* name* state))

#|
=============================================
Print unfit operations
=============================================
(c-print-unfit) prints the names of value defined operations which are unfit for optimization but whose rhs has a norm in its root indicating that the user probably inteded the operation to be fit for optimization.

Suspects and reasons for being lazy are recorded in *unfit* by eagerness analysis.
|#

(defc *unfit* nil)

(deff c-print-unfit ()
 (:when *unfit*
  (:let length (length *unfit*))
  (:when (or (option "unfit") (< length 20))
   (format t "Are the following constructs mistakenly unfit?~%")
   (unfit))
  (format t "~d mistakenly unfit operations?~%" length)
  (format t "Rerun lgc with unfit=yes for a list of suspects.~%")
  (format t "Alternatively, run lgc with quit=no and issue (unfit)~%")
  (format t "for a list of suspects~%")))

(deff unfit ()
 (dolist (item *unfit*) (format t "~a" item))
 (values))

#|
=============================================
Print unrecognized operations
=============================================
*unrecognized* collects items of form (cons fct-name fingerprint)
for unrecognized fingerprints. Then it combines known and unknown
constructs and (dump-unrecognized filename) dump them to the given
file.

This facility, which is deprecated, is intended to be invoked from the clisp mainloop. It allows to generate a fingerprint.lisp file taking advantage of all registrations in the old fingerprint.lisp file. The problem with this facility is that bugs in the old fingerprint.lisp file may live on in the new one.

A newer facility, invoked by the optidump option, generates a fingerprint.lisp file from scratch when lgc is invoked with optidump=fingerprint.lisp. That facility is described later.
|#

(defc *unrecognized* nil)

(deff c-print-unrecognized ()
 (:when (null *unrecognized*) nil)
 (format t "Unrecognized introductions of the following:~%")
 (unrecognized))

(deff unrecognized ()
 (dolist (item *unrecognized*) (format t "~s~%" (head item)))
 (values))

(deff dump-unrecognized (&optional (filename "fingerprint.lisp"))
 (with-open-file
  (stream filename
   :direction :output
   :if-exists :rename)
  (dump-unrecognized1 stream)))

; convert fingerprint-2-fct-assoc to symbol-2-fingerprint assoc
(deff opti2reg (opti)
 (:when (atom opti) nil)
 (:let ((fingerprint . fct) . opti) opti)
 (:let (name :type :arity) (thefct fct))
 (acons name fingerprint (opti2reg opti)))

; find length of longest symbol name in symbol-2-fingerprint-assoc
(deff max-reg-length (reg result)
 (:when (atom reg) result)
 (:let ((name . :fingerprint) . reg) reg)
 (:let length (length (format nil "~s" name)))
 (max-reg-length reg (max length result)))

; find all associations in new which update associations in old
(deff c-updated (new old result)
 (:when (atom new) result)
 (:let ((name . fingerprint) . new) new)
 (:when (not (member name old :key 'car)) (c-updated new old result))
 (:let result (acons name fingerprint result))
 (c-updated new old result))

(deff dump-unrecognized1 (s)
 (:let reg (opti2reg *opti*))
 (:let new *unrecognized*)
 (:let updated (c-updated new reg nil))
 (:let reg (append updated new reg))
 (:let reg (remove-duplicates reg :key 'car :from-end t))
 (:let reg (sort reg 'symbol< :key 'car))
 (:let length (max-reg-length reg 0))
 (format s "; Generated by (dump-unrecognized filename)~%~%")
 (format s "(setq *opti* nil)~%~%")
 (dump-unrecognized2 s reg length))

(deff dump-unrecognized2 (s reg length)
 (:when (atom reg) :done)
 (:let ((name . fingerprint) . reg) reg)
 (format s "(register ~vs" length name)
 (princ " " s)
 (write fingerprint :stream s :pretty nil :right-margin 10000)
 (princ ")" s)
 (terpri s)
 (terpri s)
 (dump-unrecognized2 s reg length))

#|
=============================================
Dump introduced operations
=============================================
dump-introduced dumps all introduced constructs to the given file.

*introduced* collects all introductions as a list of form
((fingerprint ref id) ...)

That list is then translated into a 'reg' (registration list) of form
((fct-name . fingerprint) ...)

That 'reg' is then dumped.

The three formats in use for the same thing (the format of *introduced*, of *unrecognized*, and of *opti*) are there of historical reasons.
|#

(defc *introduced* nil)

(deff init-introduced ()
 (:let optidump (option "optidump"))
 (:when (equalp optidump "") nil)
 (format t "Making backup of fingerprint file ~s~%" optidump)
 (with-open-file (stream optidump :direction :output :if-exists :rename)))
 
(deff dump-introduced (cache)
 (:let optidump (option "optidump"))
 (:when (equalp optidump "") nil)
 (format t "Dumping fingerprints to ~s~%" optidump)
 (with-open-file
  (stream optidump
   :direction :output
   :if-exists :supersede)
  (dump-introduced1 stream cache))
 (format t "Fingerprints are dumped during each iteration.~%")
 (format t "If lgc loops indefinitely, quit with ctrl-C~%")
 (format t "and then use the latest fingerprints.~%"))

(deff dump-introduced1 (s cache)
 (:let intro *introduced*) ; introductions in reverse order
 (:let reg (dump-translate intro cache nil))
 (:let reg-full (sort reg 'symbol< :key 'car))
 (:let reg (remove-duplicates reg-full :key 'cdr :test 'equal :from-end t))
 (dump-print-synonyms reg reg-full)
 (:let reg-old (opti2reg *opti*))
 (dump-print-removals reg reg-old)
 (dump-print-additions reg reg-old)
 (dump-print-renames reg reg-old)
 (:let length (max-reg-length reg 0))
 (format s "; Generated by the -optidump lgc option (c.f. man lgc)~%~%")
 (format s "(setq *opti* nil)~%~%")
 (dump-unrecognized2 s reg length))

; Translate ((fingerprint ref id) ...)
; to ((fct-name . fingerprint) ...)

(deff dump-translate (intro cache result)
 (:when (atom intro) result)
 (:let ((finger ref id) . intro) intro)
 (:let name (utf2string (c-symbol2name-cache ref id cache)))
 (:let fct-name (intern name))
 (dump-translate intro cache (acons fct-name finger result)))

; Whenever the same fingerprint has more than one associated fct-name,
; only the first name in the alphabet actually enters the fingerprint
; file. The following functions prints fct-name which fails to enter
; the fingerprint file for this reason and prints which other name
; they map to.

(deff dump-print-synonyms (reg reg-full)
 (:let delta (set-difference reg-full reg :key 'car))
 (:when (atom delta) nil)
 (format t "Synonyms (functions with identical fingerprints)~%")
 (format t "For each synonym pair, only the latter name enters~%")
 (format t "the fingerprint file~%")
 (format t "---~%")
 (dump-print-synonyms1 delta reg))

(deff dump-print-synonyms1 (delta reg)
 (:when (atom delta) nil)
 (:let ((fct-name . fingerprint) . delta) delta)
 (:let delta1 (member fingerprint reg :key 'cdr :test 'equal))
 (:let ((fct-name1 . :fingerprint) . :reg) delta1)
;fct-name and fct-name1 are almost always different when we come here,
;but if the old *opti* contains synonyms, fct-name and fct-name1 can be
;equal when dump-print-synonyms1 is called from dump-print-renames.
 (when (unequal fct-name fct-name1)
  (format t "~a~%~a~%---~%" fct-name fct-name1))
 (dump-print-synonyms1 delta reg))

; Print names of fingerprints that have been renamed

(deff same-fingerprint-different-name (x y)
 (and (unequal (car x) (car y)) (equal (cdr x) (cdr y))))

(deff dump-print-renames (new old)
 (:let delta (intersection old new :test 'same-fingerprint-different-name))
 (:when (atom delta) nil)
 (format t "Renames:~%")
 (format t "Fingerprints which are known currently but whose names~%")
 (format t "will change when the new fingerprint file is used.~%")
 (format t "For each rename pair, the old name is listed first.~%")
 (format t "---~%")
 (dump-print-synonyms1 delta new))

; Print names of fingerprints that have disappeared

(deff dump-print-removals (new old)
 (:let delta (set-difference old new :test 'equal :key 'cdr))
 (:when (atom delta) nil)
 (format t "Removals:~%")
 (format t "Fingerprints which are known currently but which will~%")
 (format t "disappear when the new fingerprint file is used.~%")
 (format t "---~%")
 (dump-print-names delta)
 (format t "---~%"))

(deff dump-print-names (delta)
 (:when (atom delta) nil)
 (:let ((fct-name . :fingerprint) . delta) delta)
 (format t "~a~%" fct-name)
 (dump-print-names delta))

; Print names of new fingerprints

(deff dump-print-additions (new old)
 (:let delta (set-difference new old :test 'equal :key 'cdr))
 (:when (atom delta) nil)
 (format t "Additions:~%")
 (format t "Fingerprints which are not currently known but which will~%")
 (format t "become known when the new fingerprint file is used~%")
 (format t "---~%")
 (dump-print-names delta)
 (format t "---~%"))

#|
=============================================
UNPACKING
=============================================
|#

#|
=============================================
Parse cardinal expressed by septets
=============================================
Cardinals expressed by septets consist of a sequence of middle septets followed by a single end septet. Each septet represents a value in the range 0..127 and cardinals are expressed little endian base 128. Middle septets are represented excess 128 so that e.g. the middle septet 130 represents the value 2. End septets are represented excess 0. The functions raise an exception on end-of-vector.
|#

(deff c-skip-card (vector index)
 (:when (>= index (length vector)) (raise))
 (:when (< (aref vector index) 128) (+ index 1))
 (c-skip-card vector (+ index 1)))

(etst (c-skip-card #(130 1 2 3) 0) 2)
(etst (c-skip-card #(130 1 2 3) 1) 2)
(etst (c-skip-card #(130 1 2 3) 2) 3)
(etst (c-skip-card #(130 1 2 3) 3) 4)
(xtst (c-skip-card #(130 1 2 3) 4))

(deff c-get-card (vector start end)
 (:when (<= end start) 0)
 (:let card (aref vector start))
 (:let card (if (< card 128) card (- card 128)))
 (+ card (* 128 (c-get-card vector (+ start 1) end))))

(etst (c-get-card #(130 1 2 3) 0 0) 0)
(etst (c-get-card #(130 1 2 3) 0 1) 2)
(etst (c-get-card #(130 1 2 3) 0 2) 130)

#|
=============================================
Parse body
=============================================
Parse cardinals expressed as septets until the end of the vector and return the list of cardinals found.

When a zero cardinal is met, the next cardinal is interpretted as a length field and the following subvector of the given length is inserted in the result without interpretation.
|#

(deff c-parse-body (vector index)
 (c-parse-body1 vector index nil))

(deff c-parse-body1 (vector index result)
 (:when (>= index (length vector)) (reverse result))
 (:let index1 (c-skip-card vector index))
 (:let card (c-get-card vector index index1))
 (:when (unequal card 0) (c-parse-body1 vector index1 (cons card result)))
 (:let index2 (c-skip-card vector index1))
 (:let card (c-get-card vector index1 index2))
 (:let index3 (+ card index2))
 (:when (> index3 (length vector)) (raise))
 (:let string (subvector vector index2 index3))
 (:let result (cons string result))
 (c-parse-body1 vector index3 result))

(etst
 (c-parse-body (ct2vector '(130 1 0 3 97 98 99 5)) 0)
 '(130 #(97 98 99) 5))

#|
=============================================
Parse dictionary
=============================================
Parse one dictionary, i.e. a zero-terminated list of pairs of cardinals expressed as septets. Returns (a . i) where a is an array representing the dictionary and i is the index of the first unparsed byte. Raises an exception on end-of-vector.
|#

(deff c-parse-dict (vector index)
 (c-parse-dict1 vector index '((0 . 0))))

(deff c-parse-dict1 (vector index0 result)
 (:let index1 (c-skip-card vector index0))
 (:let id (c-get-card vector index0 index1))
 (:when (= id 0) (cons result index1))
 (:let index2 (c-skip-card vector index1))
 (:let arity (c-get-card vector index1 index2))
 (:let result (aput result arity id))
 (c-parse-dict1 vector index2 result))

#|
=============================================
Parse reference
=============================================
Parse one reference. A reference is either the cardinal 0 or the cardinal 1 followed by 20 bytes followed by two cardinals m and e.

Note 2008-05-21: A length field has been added to references in the
bibliography. From now on, the internal syntax of references is not
needed in order to parse one.
|#

#|
(deff c-skip-ref (vector index0)
 (:catch nil (c-error "Format error in reference"))
 (:let index1 (c-skip-card vector index0))
 (:let version (c-get-card vector index0 index1))
 (:when (= version 0) index1)
 (:when (/= version 1)
  (:let string (safe-subvector vector index0 (+ index0 100)))
  (:let string (vector2string string))
  (format t "Vector:~a...~%" string)
  (c-error "Unknown version number: ~d" version))
 (:let index1 (+ index1 20))
 (:when (> index1 (length vector)) (raise))
 (:let index1 (c-skip-card vector index1))
 (:let index1 (c-skip-card vector index1))
 index1)

(deff c-get-ref (vector start end)
 (vector2card (subvector vector start end)))
|#

(deff c-skip-ref (vector index0)
 (:let index1 (c-skip-card vector index0))
 (:let length (c-get-card vector index0 index1))
 (+ index1 length))

(deff c-get-ref (vector index0 :end)
 (:let index1 (c-skip-card vector index0))
 (:let length (c-get-card vector index0 index1))
 (vector2card (subvector vector index1 (+ index1 length))))

#|
=============================================
Parse bibliography
=============================================
Parse bibliography, i.e. a zero-terminated list of references.
|#

(deff c-parse-bib (vector index)
 (c-parse-bib1 vector index nil))

(deff c-parse-bib1 (vector index result)
 (:let index1 (c-skip-ref vector index))
 (:let ref (c-get-ref vector index index1))
 (:when (= ref 1) (cons (reverse result) index1))
 (c-parse-bib1 vector index1 (cons ref result)))

#|
=============================================
Unpacking of a vector
=============================================
All pages start with two references: a self-reference and a supporting reference. We shall refer to these two references as the 'ref' and the 'bed' of the page.

The ref of a page must be a proper reference (not a nil reference, i.e. the reference with version number zero). The ref identifies the page in a world-wide unique way.

The bed of a page may be nil. A page whose bed is nil is a base page, i.e. a page supported by no other pages. Symbol number one of a base page is a 'proclamation symbol' which may be used for introducing other symbols. Base pages are always loaded using default procedures.

If the bed is non-empty, then the bed must be loaded before the page itself can be loaded. If the bed defines an unpacker, then that unpacker continues the processing. Otherwise, the system does default unpacking.

(c-unpack ref state) unpacks (rack-get state ref card-vector) into a reference, a bibliography, a dictionary, and a tree, adds subpages to the given state, and returns a state in which the ref, bib, dict, and tree have been added to the codex. It complaints and raises an exception in case of error.
|#

(deff c-get-unpacker (bib state)
 (:let (bed) bib)
 (:when (null bed) nil)
 (codex-get state bed 0 0 card-unpack))

(deff c-unpack (ref state)
 (:let vector (rack-get state ref card-vector))
 (:let (bib . index) (c-parse-bib vector 0))
 (:let vector (subvector vector index))
 (:let (ref1 . bib) bib)
 (:when (unequal ref ref1) (c-error "Page starts with wrong reference"))
 (:let state (c-load-bib bib state))
 (:let (dict . index) (c-parse-dict vector 0))
 (:let vector (subvector vector index))
 (c-finish-unpack ref bib dict vector state))

#|
 (:let unpacker (c-get-unpacker bib state))
 (:when (null unpacker) (c-default-unpack ref bib vector state))
 (c-user-unpack ref bib unpacker vector state))
|#

#|
=============================================
Default unpacking of a vector
=============================================
(c-default-unpack ref bib vector state) does default unpacking.
|#

#|
(deff c-default-unpack (ref bib vector state)
 (:let (dict . index) (c-parse-dict vector 0))
 (c-finish-unpack ref bib dict tree state))
|#

#|
=============================================
User unpacking of a vector
=============================================
(c-user-unpack ref bed unpacker vector state) does user unpacking.
|#

#|
(deff c-user-unpack (ref bib unpacker vector state)
 (ignoring ref bib unpacker vector state)
 (c-error "User defined unpacking not yet implemented"))
|#

#|
=============================================
Finish unpacking and start codification
=============================================
(c-finish-unpack bib dict tree state) converts the given tree from Polish prefix to a parse tree and replaces relative page references by absolute ones.
|#

(deff bib2dict (bib state)
 (bib2dict1 1 bib state nil))

(deff bib2dict1 (index bib state result)
 (:when (null bib) result)
 (:let (ref . bib) bib)
 (:let dict (rack-get state ref card-dictionary))
 (:let result (aput result dict index))
 (bib2dict1 (+ index 1) bib state result))

(deff c-unpack-root (shift card)
 (:mlet (id ref) (floor (- card 1) shift))
 (list ref id))

(deff c-parse-polish (dict shift bib adr polish)
 (:when (null polish) (c-error "End of file while parsing tree"))
 (:let (root . polish) polish)
 (:when (arrayp root)
  (:let root (list 0 root adr))
  (:let tree (list root))
  (cons tree polish))
 (:let (ref id) (c-unpack-root shift root))
 (:let arity (aget dict ref id))
 (unless (integerp arity)
  (error "Internal error: arity not a cardinal: ~a" arity))
 (:let ref (nth ref bib))
 (:when (null ref) (c-error "Internal error: Relative reference out of range"))
 (:let (tree* . polish)
  (c-parse-polish* 0 arity dict shift bib adr polish nil))
 (:let root (list ref id adr))
 (:let tree (cons root tree*))
 (cons tree polish))

(deff c-parse-polish* (index arity dict shift bib adr polish result)
 (:when (>= index arity) (cons (reverse result) polish))
 (:let (tree . polish)
  (c-parse-polish dict shift bib (cons index adr) polish))
 (:let result (cons tree result))
 (c-parse-polish* (+ index 1) arity dict shift bib adr polish result))

(deff c-finish-unpack (ref bib dict vector state)
 (:let full-dict (bib2dict bib state))
 (:let full-dict (aput full-dict dict 0))
 (:let shift (+ 1 (length bib)))
 (:let adr (list ref))
 (:let cache (bib2cache-put bib state nil))
 (:let state (rack-put state cache ref card-cache))
 (:let cache0 (bib2cache-merge bib state nil))
 (:let cache1 (aget state (id-cache) ref ref))
 (:let cache2 (aput cache0 cache1 ref))
 (:let state (aput state cache2 (id-cache) ref))
 (:let state (aput state ref (id-cache) ref 0))
 (:let state (rack-put state (cons ref bib) ref card-bibliography))
 (:let state (rack-put state dict ref card-dictionary))
 (:let unpacker (c-get-unpacker bib state))
 (:when unpacker
  (when (verbose '> 0) (format t "Load: Custom unpacking~%"))
  (:let cache3 (aget state (id-cache) ref))
  (:let (:define :unpack :lhs unpacker) unpacker)
  (:let arglist nil)
  (:let term (tree2term cache3 arglist unpacker))
  (:let term (term-apply term (term-tv cache3)))
  (:let term (term-apply term (term-tv vector)))
  (:catch () (c-error "The unpacker raised an exception"))
  (:let tree (tm2tv (term-eval term nil)))
  (:let tree (c-prune-tree "body" ref tree state))
  (:let state (rack-put state tree ref card-body))
  state)
 (when (verbose '> 0) (format t "Load: Default unpacking~%"))
 (:let tree (c-parse-body vector 0))
 (:let (tree) (c-parse-polish full-dict shift (cons ref bib) adr tree))
 (:let state (rack-put state tree ref card-body))
 state)



#|
=============================================
Convert ref to url
=============================================
|#

(defc c-ref2url-hash (make-hash-table :test 'equal))
(defc c-url2ref-hash (make-hash-table :test 'equal))

(deff c-server2string (server)
 (:let (server . udp) server)
 (format nil "udp/~a/~d" server udp))

(deff c-ref2string (ref)
 (ct2string (ref2kana ref)))

(deff c-ref2url (ref)
 (:let hashed (gethash ref c-ref2url-hash))
 (:when hashed hashed)
 (:let verb (option "verbose"))
 (when (> verb 0) (format t "Resolving ~a~%" (c-ref2string ref)))
 (:let adr (m-ref2vector ref))
 (:let server (cons (option-udphost) (option "udpport")))
 (progress "Load: searching Internet")
 (socket-protect fd (udp-open) (c-ref2url1 ref fd server adr verb 0 nil)))

(deff c-ref2url1 (ref fd server adr verb norm trace)
 (:let prefix (make-prefix))
 (:let prefix* (list prefix))
 (:let index 0)
 (:let request (m-get-url prefix* adr index))
 (when (> verb 0)
  (format t "Contacting ~a norm=~3d/~d~%"
   (c-server2string server) norm (head adr)))
 (udp-sendto fd request server)
 (:let timeout (time+ (option "patience") (lgt)))
 (c-ref2url2 ref fd server adr prefix* timeout verb norm trace))

(deff c-ref2url2 (ref fd server adr prefix* timeout verb norm trace)
 (:let time (lgt))
 (:when (time< timeout time)
  (c-ref2url-timeout ref fd server adr verb norm trace))
 (inet-select (time- timeout time) (list fd))
 (:let (msg . (:ip . :port)) (udp-recvfrom fd))
 (:when (equalp msg :none)
  (c-ref2url-timeout ref fd server adr verb norm trace))
 (:let msg (m-parse-message msg))
 (:let (id tag* . arg) msg)
 (:when (unequal tag* prefix*)
  (when (> verb 1) (format t "Received response with improper prefix~%"))
  (c-ref2url2 ref fd server adr prefix* timeout verb norm trace))
 (:let trace (acons server msg trace))
 (:when (unequal id :got)
  (c-ref2url-improper ref fd server adr msg verb norm trace))
 (:let (adr1 class :index norm1 count :time1 value) arg)
 (:when (unequal adr1 adr)
  (c-ref2url-improper ref fd server adr msg verb norm trace))
 (:when (unequal class m-id-url)
  (c-ref2url-improper ref fd server adr msg verb norm trace))
 (:let (length) adr)
 (:when (= count 0)
  (format t "~a says the refereced page does not exist~%"
   (c-server2string server))
  (c-ref2url-retry ref fd adr verb norm trace))
 (:let (:length . card*) value)
 (:when (equalp length norm1) card*)
 (:when (<= norm1 norm)
  (format t "~a said no more than was known before asking it~%")
  (c-ref2url-improper ref fd server adr msg verb norm trace))
 (:let (:relay protocol . server) (card*2server card*))
 (:when (unequal protocol :udp)
  (c-ref2url-improper ref fd server adr msg verb norm trace))
 (c-ref2url1 ref fd server adr verb norm1 trace))

(deff c-ref2url-timeout (ref fd server adr verb norm trace)
 (format t "Timeout from ~s~%" (c-server2string server))
 (:let trace (acons server :none trace))
 (c-ref2url-retry ref fd adr verb norm trace))

(deff c-ref2url-improper (ref fd server adr msg verb norm trace)
 (c-ref2url-improper1 server msg)
 (c-ref2url-retry ref fd adr verb norm trace))

(deff c-ref2url-improper1 (server msg)
 (:let (id :prefix . arg) msg)
 (:let server (c-server2string server))
 (:when (unequal id :notify) (c-ref2url-improper2 server msg))
 (:let (event) arg)
 (:when (equal event m-id-sorry)
  (format t "Server ~a too pressed to respond~%" server))
 (:when (equal event m-id-rejected)
  (format t "Server ~a refuses to respond~%" server))
 (c-ref2url-improper2 server msg))

(deff c-ref2url-improper2 (server msg)
 (format t "Improper response ~s from ~a~%" msg server))

(deff c-ref2url-retry (ref fd adr :verb norm trace)
 (format t "Could not resolve ~a~%" (c-ref2string ref))
 (format t "Hit return to try again or enter http or file ref manually~%")
 (format t "> ")
 (:let line (read-line))
 (:let line (string-trim *spaces* line))
 (:when (unequal line "") (ct2ct line))
 (:let server (cons (option-udphost) (option "udpport")))
 (c-ref2url1 ref fd server adr 2 norm trace))

#|
=============================================
Convert url to vector
=============================================
A call like (c-url2vector (string2card* "http://logiweb.eu/abc/def")) loads file /abc/def from logiweb.eu.
|#

(deff c-split (card card*)
 (:let position (position card card*))
 (:when (null position) (raise))
 (:let before (subseq card* 0 position))
 (:let after (subseq card* (+ position 1)))
 (cons before after))

(deff c-url2vector (url)
 (:catch () (c-error "Unable to load: ~a" (card*2string url)))
 (:let (tag . rest) (c-split (char2card #\:) url))
 (:when (equalp tag (string2card* "http")) (c-http2vector rest))
 (:when (unequal tag (string2card* "file"))
  (c-error "Unknown or missing tag: ~a" (card*2string url)))
 (:let vector (path2vector rest))
 (:when (equalp vector :does-not-exist)
  (c-error "File not found: ~a" (card*2string url)))
 vector)

(deff c-http2vector (http)
 (:let slash (char2card #\/))
 (:let (slash1 slash2 . http) http)
 (:when (unequal slash1 slash) (raise))
 (:when (unequal slash2 slash) (raise))
 (:let (domain . path) (c-split slash http))
 (:let colon (char2card #\:))
 (:unless (member colon domain) (c-http2vector1 domain 80 path))
 (:let (domain . port) (c-split colon domain))
 (:let (port . junk) (parse-int (card*2string port)))
 (:when (unequal junk "") (c-error "Malformed domain name"))
 (c-http2vector1 domain port path))

(deff c-http2vector1 (domain port path)
 (:let slash (char2card #\/))
 (:let domain (card*2string domain))
 (:let path (cons slash path))
 (:let time (option "patience"))
 (:let msg (ct2card* (list "GET " path #\Newline)))
 (tcp-query domain port time msg))

#|
=============================================
Convert url to ref
=============================================
(c-url2ref url) converts the given url to a ref. It reads the entire file associated with the given url (not very efficient).
|#

(deff c-http-default (url)
 (:when (not (string-prefix (ct2ct "http:") url)) url)
 (:when (string-prefix (ct2ct "http:/") url) url)
 (ct2card* (cons (option "url") (nthcdr (length "http:") url))))

(deff c-url2ref (url)
 (:let url (c-http-default url))
 (:let position (default 0 (position (ct2ct #\:) url)))
 (:let type (subseq url 0 position))
 (:let ref (safe-subseq url (+ position 1)))
 (:catch () (c-error "~a is no Logiweb page" (card*2string url)))
 (:when (equalp type (ct2ct "lgw")) (c-lgw-url-2-ref :go ref))
 (:when (equalp type (ct2ct "32")) (c-lgw-url-2-ref :base32 ref))
 (:when (equalp type (ct2ct "64")) (c-lgw-url-2-ref :base64 ref))
 (when (verbose '>= 0) (format t "Reverse lookup of ~a~%" (card*2string url)))
 (:let hashed (gethash url c-url2ref-hash))
 (:when hashed hashed)
 (:let vector (c-url2vector url))
 (:let index (c-skip-ref vector 0))
 (:let ref (c-get-ref vector 0 index))
;(:let (ref) (c-parse-ref vector))
 (setf (gethash ref c-ref2url-hash) url)
 (setf (gethash url c-url2ref-hash) ref)
 ref)

(deff c-lgw-url-2-ref (type ref)
 (:let vector (edge2ref type ref))
 (:let vector (ct2vector vector))
 (:let index (c-skip-ref vector 0))
 (:when (unequal index (length vector)) (raise))
 (c-get-ref vector 0 index))

#|
=============================================
Load reference
=============================================
(c-load-ref ref state) loads the page pointed out by the given reference and adds it to the state.

c-load-ref first accesses a Logiweb server to get an URL for the page. Then it parses the URL to verify that it has form
  http://<domain><path>
and then sends
  GET <path>
to port 80 of that domain.
|#

(deff c-load-message (loaded ref)
 (:when (verbose '> 1)
  (format t "Loading ref:~%")
  (format t "~a~%" (ct2string (ref2kana ref)))
  (:when loaded (format t "Already loaded~%")))
 (:when (verbose '= 1)
  (format t "Loading ref ~a ...~%" (ct2string (ref2short-kana 5 ref)))
  (:when loaded (format t "Already loaded~%")))
 (:when loaded nil)
 (:when (verbose '< 0) nil)
 (format t "Loading ref ~a ...~%" (ct2string (ref2short-kana 5 ref))))

#|
Maximum number of iterations done by c-codify-iterate (called from c-codify, called from c-load, called from c-load-ref and frontend). A value of 0 represents infinity.
|#
(defc *max-iter* 0)

(deff c-load-ref (ref state)
 (:let loaded (rack-get state ref card-vector))
 (c-load-message loaded ref)
 (:when loaded state)
 (:let rack (load-rack ref))
 (:when rack (c-load-ref1 ref rack state))
 (progress "Load: Resolving reference")
 (:let url (c-ref2url ref))
 (progress "Load: retrieving page")
 (format t "Loading url: ~a~%" (card*2string url))
 (:let vector (c-url2vector url))
 (:let ref1 (ref2card (vector2ref (vector2card* vector))))
 (:when (unequal ref ref1)
  (format t "Mismatch between reference~%~s~%and url~%~s~%" ref ref1)
  (raise))
 (format t "Codifying vector~%")
 (:let state (rack-put state vector ref card-vector))
 (setq *max-iter* 0)
 (:let state (c-load ref state))
 (dump-rack ref state)
 state)

(deff c-load-ref1 (ref rack state)
 (:let (:ref . bib) (aget rack card-bibliography))
 (:let state (c-load-bib bib state))
 (:let cache (bib2cache-merge bib state nil))
 (:let state (aput state cache (id-cache) ref))
 (:let state (aput state ref (id-cache) ref 0))
 (:let state (aput state rack (id-cache) ref ref))
 (:let cache (bib2cache-put bib state nil))
 (:let state (rack-put state cache ref card-cache))
 (:let state (c-codex2code ref state))
 state)

#|
=============================================
Load bibliography
=============================================
(c-load-bib bib state) loads all pages in the given bibliography and adds them to the state.
|#

(deff c-memorize (ref state)
 (:when (null *memorize*) nil)
 (:let cache (aget state (id-cache) ref))
 (setq *state* (aput *state* cache (id-cache) ref)))

(deff c-load-bib (bib state)
 (:when (null bib) state)
 (:let (ref . bib) bib)
 (:let state (c-load-ref ref state))
 (c-memorize ref state)
 (c-load-bib bib state))



#|
=============================================
CODIFICATION
=============================================
|#

#|
=============================================
Subcache computation
=============================================
(bib2cache-merge bib state nil) gives the cache of the referencing page
except that the rack of the referencing page is missing.

(bib2cache-put bib state nil) gives the cache slot of the rack of the
referencing page.
|#

(deff bib2cache-merge (bib state result)
 (:when (atom bib) result)
 (:let (ref . bib) bib)
 (:let cache (aget state (id-cache) ref))
 (:let result (array-merge cache result))
 (bib2cache-merge bib state result))

(deff bib2cache-put (bib state result)
 (:when (atom bib) result)
 (:let (ref . bib) bib)
 (:let cache (aget state (id-cache) ref))
 (:let result (aput result cache ref))
 (bib2cache-put bib state result))

#|
=============================================
Proclamations
=============================================
Logiweb recognizes or has recognized the following predefined concepts:

apply lambda true if
The four basic computation constructs. A symbol denotes the apply construct if the value aspect of the symbol on the symbols own page equals
  `((0 . ,card-apply))
Similar holds for the other three constructs.

quote
A fifth basic computation construct, added to simplify macro expansion.

pair
Computation construct that has been removed. x pair y equals lambda z dot if z then x else y.

proclaim
Proclamation construct. A symbol denotes the proclamation construct if the definition aspect of the symbol on the symbols own page is non-nil and the arity of the construct is two. When loading a base page, if symbol number one has arity two, then the definition aspect of that symbol is set to a non-nil value. That allows symbol number one to act as a proclamation symbol in the first codification iteration of the base page. Symbol number one must proclaim itself to be a proclamation symbol to remain a proclamation symbol in the next iteration.

define introduce
Definition constructs. A symbol denotes the define construct if the definition aspect of the symbol on the symbols own page equals
  `((0 . ,card-define)).
Similar holds for the other construct. Formerly, "introduce" was called "optimize", but that was changed because users might start using "optimize" always in the foolish hope that they would get faster code that way.

vector bibliography dictionary body expansion cache correctness
Names of fixed entries of the codex. The vector of a page is stored as aspect
  `(0 . ,card-vector)
of the page symbol. Similar holds for the other aspects. The 'dictionary' aspect was previously named 'arity' and was distributed over all non-nulary symbols of a page. The correctness aspect is new. When T, the page is correct. Otherwise, the value is non-T and may indicate what is wrong with the page in some manner. No proclamations proclaim a symbol to denote any one of these aspects and, hence, the user cannot define any of these aspects of any symbol.

name value message statement
Names of predefined aspects that the user may define for individual symbols. A symbol denotes the name aspect if its message aspect equals
  `((0 . ,card-name))
Similar holds for the other aspects. The name aspect is badly needed since it allows to view the page. The value aspect is badly needed since it is used for defining almost everything else. The message aspect could be omitted since a user could use the codify aspect to escape from the limitation of not having the message aspect.

tex
Name of aspect that was removed from the Logiweb system when the TeX rendering information was moved to the name aspect.
  Note. It has been added again now that support of MathML is removed. Now there are two aspects: tex and texname.
  Note 2. tex has been renamed to use and texname to show.

definition
Name of aspect that is used for recognizing define and introduce constructs. This aspect was previously called codify.

unpack codify claim macro priority
Names of aspects that the user may define for the page symbol. Actually, the user may define those aspects of any symbol, but Logiweb only reacts to these aspects of the page symbol. A symbol denotes the unpack aspect if its message aspect equals
  `((0 . ,card-unpack))
Similar holds for the other aspects. The codify aspect is needed since it is the aspect that allows to escape from limitations in the default codification. The claim aspect is needed to allow claims even at base pages. The unpack aspect could be omitted since one could use the codify aspect to escape from the limitation of not having the unpack aspect. The priority aspect is new and is intended to allow reconstruction of Logiweb source from pages. That aspect is needed even on base pages.

verify
Name of aspect that was removed from the Logiweb system when default verification was simplified.

macrodef
The macrodef aspect records which constructs denote the macroprio, macroexpand, and macrovar constructs for defining macros. The macrodef and macro aspects define macro definers and macros, respectively.

macroself macrostop
Symbols proclaimed to denote macroself macroexpand into the page symbol of the page on which they occur.

macrostop macroprotect
Constructs for disabling macro expansion. macrostop disables macro expansion for an entire tree, macroprotect merely disables one level.

macroprio macroexpand macrovar
Symbols for building up the right hand side of macro definitions. Macrovar was previously called macroall. Macroexpand was previously called macroreduce.

pre post
Symbols for declaring constructs to be pre- or postassociative.

infers modus metavar instance sum label endorses proves hence
Symbols for building theories, lemmas, and proofs. 



Summary:

Proclaimable:
apply lambda true if quote
proclaim define introduce
name use show value message macro statement
unpack codify claim priority
macroself macrostop macroprotect macroprio macroexpand macrovar
pre post
infers modus metavar instance sum label endorses proves hence

Nonproclaimable: proclaim vector bibliography dictionary body expansion cache correctness macrodef definition

Removed: pair name verify

Renamed: "introduce" was formerly called "optimize"

|#

(defc predef-assoc
 (list
  (list "apply"        card-apply        2 card-value     )
  (list "lambda"       card-lambda       2 card-value     )
  (list "true"         card-true         0 card-value     )
  (list "if"	       card-if           3 card-value     )
  (list "quote"	       card-quote        1 card-value     )

  (list "proclaim"     card-proclaim     2 card-definition)
  (list "define"       card-define       3 card-definition)
  (list "introduce"    card-introduce    3 card-definition)
  (list "hide"         card-hide       nil card-definition)

; The following are going to be deleted soon when users have migrated
; to aspect proclamations. 2006-05-19. And have migrated to new
; pre/post 2006-09-25.
#|
  (list "pre"          card-pre          2 card-priority  )
  (list "post"         card-post         2 card-priority  )

  (list "name"	       card-name          0 card-message   )
  (list "use"          card-tex          0 card-message   )
  (list "show"         card-texname      0 card-message   )
  (list "value"        card-value        0 card-message   )
  (list "message"      card-message      0 card-message   )
  (list "macro"        card-macro        0 card-message   )
  (list "unpack"       card-unpack       0 card-message   )
  (list "claim"        card-claim        0 card-message   )
  (list "priority"     card-priority     0 card-message   )
  (list "execute"      card-execute      0 card-message   )
|#
))

#|
=============================================
Harvesting
=============================================
|#

(deff c-harvest-proclamation (ref base-state tree)
 (:let (nil construct predef) tree)
 (:let predef (c-tree2string predef))
 (:let (predef-card arity . aspect*) (assoc-get predef predef-assoc))
 (:when (null aspect*)
  (when (unequal predef "")
   (format t "Warning. Unknown predefined construct: ~s~%" (utf2string predef)))
  base-state)
 (:let ((sref sid) . tree*) construct)
 (:when (equalp sref 0)
  (format t "Warning. Attempt to proclaim string.~%")
  base-state)
 (:when (and arity (unequal arity (length tree*)))
  (format t "Warning. Ignoring ~s proclamation:~%" (utf2string predef))
  (format t "Symbol in question has wrong arity.~%")
  base-state)
 (:when (unequal ref sref)
  (format t "Warning. Ignoring ~s proclamation:~%" (utf2string predef))
  (format t "Symbol in question is imported from another page.~%")
  base-state)
 (:let tree (card2tree predef-card))
;(format t "Proclaims symbol ~3d as ~s~%" sid predef)
 (c-harvest-proclamation1 ref sref sid aspect* tree base-state))

(deff c-harvest-proclamation1 (ref sref sid aspect* tree state)
 (:when (atom aspect*) state)
 (:let (aspect . aspect*) aspect*)
 (:let state
  (aput state tree (id-cache) ref ref card-codex sref sid 0 aspect))
 (c-harvest-proclamation1 ref sref sid aspect* tree state))

(deff tree2aspect (tree state)
 (:let ((aref aid)) tree)
 (:when (equalp aref 0) (list 0 (vector2card aid)))
 (:let def (codex-get state aref aid 0 card-message))
 (:let (:def :aspect :lhs ((aref aid))) def)
 (:when (null aref) nil)
 (list aref (vector2card aid)))

(defc predef-aspect
 (list
  card-name
  card-tex
  card-texname
  card-value
  card-message
  card-macro
  card-definition
  card-priority
  card-unpack
  card-render
  card-claim
  card-execute
  card-executables))

(deff c-harvest-warn (ref sref sid aref aid iter-state)
 (:when (equalp sref ref) nil)
 (:when (equalp sref 0) nil)
 (:when (unequal aref 0) nil)
 (:when (not (member aid predef-aspect)) nil)
 (:let aspect (utf2string (card2string aid)))
 (:let name (utf2string (c-symbol2name sref sid iter-state)))
 (format t "Warning. non-local ~a definition of ~a~%" aspect name))

(deff c-harvest-definition (ref iter-state base-state tree)
 (:let (nil aspect symbol) tree)
 (:let (aref aid) (tree2aspect aspect iter-state))
 (:when (null aref) base-state)
 (:let ((sref sid)) symbol)
 (:let sid (vector2card sid))
 (c-harvest-warn ref sref sid aref aid iter-state)
#|
 (:when (equalp sref 0)
  (format t "Warning. Attempt to define string.~%")
  base-state)
 (when
  (and
   (unequal ref sref)
   (unequal sref 0)
   (equalp aref 0)
   (member aid predef-aspect))
  (format t "Warning. non-local ~a definition of ~a~%"
   (utf2string (card2string aid))
   (utf2string (c-symbol2name sref sid iter-state))))
|#
 (aput base-state tree (id-cache) ref ref card-codex sref sid aref aid))

(deff c-harvest (ref iter-state base-state tree)
 (:let ((sref sid) . tree*) tree)
 (:let def (codex-get iter-state sref sid 0 card-definition))
 (:when (null def) (c-harvest* ref iter-state base-state tree*))
 (:when (equalp def tree-proclaim)
  (c-harvest-proclamation ref base-state tree))
 (:when (equalp def tree-define)
  (c-harvest-definition ref iter-state base-state tree))
 (:when (equalp def tree-introduce)
  (c-harvest-definition ref iter-state base-state tree))
 (:when (equalp def tree-hide)
  base-state)
 (error "Internal check failed in c-harvest"))

(deff c-harvest* (ref iter-state base-state tree*)
 (:when (null tree*) base-state)
 (:let (tree . tree*) tree*)
 (:let base-state (c-harvest ref iter-state base-state tree))
 (:let base-state (c-harvest* ref iter-state base-state tree*))
 base-state)

#|
=============================================
Equality of terms
=============================================
|#

(deff c-tree-equal (tree1 tree2)
 (:let ((sref1 sid1) . tree*1) tree1)
 (:let ((sref2 sid2) . tree*2) tree2)
 (and
  (equalp sref1 sref2)
  (equalp sid1 sid2)
  (c-tree-equal* tree*1 tree*2)))

(deff c-tree-equal* (tree*1 tree*2)
 (:when (and (null tree*1) (null tree*2)) t)
 (:when (or (null tree*1) (null tree*2)) nil)
 (and
  (c-tree-equal (car tree*1) (car tree*2))
  (c-tree-equal* (cdr tree*1) (cdr tree*2))))

#|
=============================================
Expansion
=============================================
|#

(deff c-get-page-aspect (ref aspect cache)
 (:let result (aget cache ref card-codex ref 0 0 aspect))
 (:when result result)
 (:let (:self bed) (aget cache ref card-bibliography))
 (:when (null bed) nil)
 (aget cache bed card-codex bed 0 0 aspect))

(deff c-expand (ref state tree)
 (:let cache (aget state (id-cache) ref))
 (:let expander (c-get-page-aspect ref card-macro cache))
 (:when (null expander) tree)
 (:let (:define :macro :lhs expander) expander)
 (:let arglist nil)
 (:let term (tree2term cache arglist expander))
 (:let term (term-apply term (term-tv cache)))
 (:catch () (c-error "Some macro expander raised an exception"))
 (:let tree (tm2tv (term-eval term nil)))
 (c-prune-tree "expansion" ref tree state))

(defc *prune* nil)
(defc *pruned* nil)

(deff c-prune-tree (msg ref tree state)
 (:let cache (aget state (id-cache) ref))
 (:unless (c-prune-tree-p tree cache) tree)
 (setq *prune* nil)
 (:let default (list (list ref 0)))
;(:let default (list (list 0 (string2card "prune"))))
 (:let pruned-tree (c-prune-tree0 default tree cache nil))
 (:let pruned (length *prune*))
 (when (= pruned 1)
  (format t "One branch pruned in ~a~%" msg)
  (format t "Address: ~a~%" (cdar *prune*))
  (format t "Reason: ~s~%" (caar *prune*)))
 (when (> pruned 1)
  (format t "~d branches pruned in ~a~%" (length *prune*) msg)
  (format t "Run lgc with quit=no and see *prune* for addresses and reasons~%"))
 (:let depth (option "spydepth"))
 (:let depth (case depth (-2 10) (-1 -1) (t depth)))
 (:let length (option "spylength"))
 (:let length (case length (-2 20) (-1 nil) (t length)))
;(:let length (if (< length 0) nil length))
 (when (verbose '> 1)
  (format t "Raw value:~%")
  (format t "~s~%" (trace1 depth length tree)))
 (when (verbose '> 1)
  (format t "Tree:~%")
  (c-print-tree 1 depth length tree state))
 pruned-tree)

(deff c-prune (reason default address)
 (push (cons reason address) *prune*)
;(format t "Warning: pruning tree at address ~a~%" (reverse address))
;(format t "~a~%" reason)
 default)

(deff c-prune-tree0 (default tree cache address)
 (:when (atom tree) (c-prune "No root" default address))
 (:let (root . tree*) tree)
 (:when (atom root) (c-prune "No reference" default address))
 (:let (ref . rest) root)
 (:unless (numberp ref) (c-prune "Reference must be a number" default address))
 (:when (atom rest) (c-prune "No id" default address))
 (:let (id) rest)
 (:unless (intp id) (c-prune "Identifier must be a number" default address))
 (:when (equalp ref 0)
  (:when tree* (c-prune "String must have arity zero" default address))
  tree)
 (:let arity (aget cache ref card-dictionary id))
 (:unless (numberp arity)
  (c-prune (list "Unknown symbol" ref id) default address))
 (:let tree* (c-prune-tree* default tree* cache address 0))
 (:when (unequal arity (length tree*))
  (c-prune "Wrong arity" default address))
 (cons root tree*))

(deff c-prune-tree* (default tree* cache address index)
 (:when (atom tree*) nil)
 (:let (tree . tree*) tree*)
 (:let tree (c-prune-tree0 default tree cache (cons index address)))
 (:let tree* (c-prune-tree* default tree* cache address (+ index 1)))
 (cons tree tree*))

(deff c-prune-tree-p (tree cache)
 (:when (atom tree) t)
 (:let (root . tree*) tree)
 (:when (atom root) t)
 (:let (ref . rest) root)
 (:unless (numberp ref) t)
 (:when (atom rest) t)
 (:let (id) rest)
 (:unless (intp id) t)
 (:when (= ref 0) tree*)
 (:let arity (aget cache ref card-dictionary id))
 (:when (unequal arity (length tree*)) t)
 (c-prune-tree-p* tree* cache))

(deff c-prune-tree-p* (tree* cache)
 (:when (atom tree*) nil)
 (:let (tree . tree*) tree*)
 (or (c-prune-tree-p tree cache) (c-prune-tree-p* tree* cache)))

#|
=============================================
Codification
=============================================

(c-change-codex oldcodex newcodex) returns the path of a "non-monotonic" change from oldcodex to newcodex. A change from nil to non-nil does not count as a "non-monotonic" change.
|#

(deff c-change-codex (oldcodex newcodex)
 (:catch (x) (reverse x))
 (c-change-codex1 4 nil oldcodex newcodex)
 nil)

(deff c-change-codex1 (level path oldcodex newcodex)
 (:when (<= level 0)
  (:when (unequal oldcodex newcodex) (raise path))
  nil)
 (:when (null oldcodex) nil)
 (:let (codex1 . codex2) oldcodex)
 (:when (numberp codex1)
  (c-change-codex1 (- level 1) (cons codex1 path) codex2
   (aget newcodex codex1)))
 (c-change-codex1 level path codex1 newcodex)
 (c-change-codex1 level path codex2 newcodex))

(deff c-warn-codex (oldcodex newcodex state)
 (:let path (c-change-codex oldcodex newcodex))
 (:when (null path) nil)
 (:let (sref sid aref aid) path)
 (format t "Reiterating because aspect '~a' of symbol '~a' has changed~%"
  (utf2string (c-symbol2name aref aid state))
  (utf2string (c-symbol2name sref sid state))))

(deff c-codify-init (ref state)
 (:let bib (rack-get state ref card-bibliography))
 (:when (cdr bib) state)
 (:let dictionary (rack-get state ref card-dictionary))
 (:let arity (aget dictionary 1))
 (:when (unequal arity 2)
  (format t "Warning: Proclamation symbol of base pages must have arity 2~%")
  state)
 (:let tree tree-proclaim)
 (format t "Initiates proclamation symbol ~%")
 (:let newstate (codex-put state tree ref 1 0 card-definition))
 newstate)

(deff c-codify-iterate (ref base-state body iter-state cnt)
 (format t "ITERATION ~d~%" cnt)
 (when (verbose '> 0) (progress "Load: compiling definitions"))
 (:let iter-state (c-codex2code ref iter-state))
 (dump-introduced (aget iter-state (id-cache) ref))
 (when (verbose '> 0) (progress "Load: macro expanding"))
 (setq *spy-state* iter-state);so we can render *spy* in case of stack overflow
 (:let expansion (c-expand ref iter-state body))
 (print-timers)
 (when (null *spy*) (setq *spy-state* nil))
 (:when (equalp cnt *max-iter*) (cons expansion iter-state))
 (when (verbose '> 0) (progress "Load: harvesting"))
 (:let next-state (c-harvest ref iter-state base-state expansion))
 (:let old-codex (rack-get iter-state ref card-codex))
 (:let new-codex (rack-get next-state ref card-codex))
 (:when (equalp old-codex new-codex)
  ; iter-state contains compiled functions, next-state does not
  (cons expansion iter-state))
 (c-warn-codex old-codex new-codex next-state)
;Next line added 2008-03-09
 (:let next-state (rack-put next-state expansion ref card-expansion))
 (c-codify-iterate ref base-state body next-state (+ cnt 1)))

(deff c-codify (ref state)
 (:let body (rack-get state ref card-body))
 (:let iter-state (c-codify-init ref state))
 (init-introduced)
 (:let (expansion . state)
  (c-codify-iterate ref state body iter-state 1))
 (format t "ITERATION ENDED~%")
 (c-print-unfit)
 (c-print-unrecognized)
 (:let state (rack-put state expansion ref card-expansion))
 state)

#|
=============================================
FINGERPRINTS
=============================================

The 'fingerprint' of a definition is a version of the definition in which variables are replaced by deBruijn indices and function names are replaced by function indices so that definitions that are identical except for naming of functions and variables have the same fingerprint. Fingerprints are used in connection with optimizing definitions in which known functions are replaced by handcoded versions of those functions.

As an example, consider the following set of definitions:

proclaim lambda x dot y as "lambda" then
proclaim if x then y else z as "if" then
introduce value of x pair y as lambda z dot if z then x else y.

The fingerprint of x pair y is
  ((2 (2 (1 0 1 2))) 13161 3359945486572)
Above, 13161 represents the proclamation of "if" and 3359945486572 represents the proclamation of "lambda". string2card defines the relation between proclamations and cardinals. As an example, 13161=(string2card "if"). (2 (2 (1 0 1 2))) represents the definition of x pair y. The representation of the definition of x pair y is constructed thus:

Step 1. Convert
  lambda z dot if z then x else y
to
  (lambda (z) (if (z) (x) (y)))

Step 2. Replace defined constructs by their index into the fingerprint (so that 'if' is replaced by 1 and 'lambda' by 2. This results in the following:

  (2 (z) (1 (z) (x) (y)))

Step 3. Replace variables by deBruijn indices (where each lambda stacks another variable). Delete all binding variables. This results in the following:

  (2 (1 0 1 2))

Step 4. Add the arity of the construct in front of the fingerprint. This results in the following:

  (2 (2 (1 0 1 2)))

The Logiweb compiler once had the property that any symbol defined using "introduce" which has fingerprint ((2 (2 (1 0 1 2))) 13161 3359945486572) was computed by a hand-coded pairing function. The Logiweb compiler no longer recognizes that particular pattern, but it recognizes many others.
|#

#|
=============================================
Proclaimedness test
=============================================
(proclaimedp def) is true if the given def (which is assumed to come from the value aspect of some symbol in a codex) denotes a proclaimed computable construct.
|#

(deff proclaimedp (def)
 (:let ((ref)) def)
 (equalp ref 0))

#|
=============================================
Parenthesis test
=============================================
(parenthesisp def) is true if the given def (which is assumed to come from the value aspect of some symbol in a codex) denotes the identity function.
|#

(deff parenthesisp (def)
 (:when (null def) nil)
 (:when (proclaimedp def) nil)
 (:let (nil nil lhs rhs) def)
 (c-tree-equal* (cdr lhs) (list rhs)))

#|
=============================================
Transitively used symbols
=============================================
(symbol2symbol* ref id cache result) finds all symbols that are transitively used for defining the symbol with the given ref and id. The symbols found are consed in front of the given 'result'. Symbols that have no value definition are considered to be variables and are not included.

In the value returned from symbol2symbol*, symbols are represented as the cons of their references and ids. Normally, symbols are represented as lists that may contain more than just the ref and id.

(tree2symbol* tree result) applies symbol2symbols to all symbols in the given tree.

(tree*2symbol* tree* result) applies tree2symbols to all trees in the given tree*.
|#

(deff symbol2symbol* (ref id cache result)
 (:when (equalp ref 0) result) ; ignore strings
 (:let def (aget cache ref card-codex ref id 0 card-value))
 (:when (null def) result) ; ignore variables
 (:when (parenthesisp def) result) ; ignore parentheses
 (:let result (acons ref id result))
 (:when (proclaimedp def) result)
 (:let (nil nil nil rhs) def)
 (tree2symbol* rhs cache result))

(deff tree2symbol* (tree cache result)
 (:let ((ref id) . tree*) tree)
 (:let result (tree*2symbol* tree* cache result))
 (:let symbol (cons ref id))
 (:when (equalp ref 0) result) ; ignore strings
 (:when (member symbol result :test 'equal) result)
 (symbol2symbol* ref id cache result))

(deff tree*2symbol* (tree* cache result)
 (:when (null tree*) result)
 (:let (tree . tree*) tree*)
 (:let result (tree2symbol* tree cache result))
 (:let result (tree*2symbol* tree* cache result))
 result)

#|
=============================================
Convert tree to fingerprint
=============================================
(tree2fingerprint arg* tree directory cache) converts the labels of the given tree from lists of form (ref id ...) to indices into the given directory.

Strings are converted to cardinals and have their sign reversed.

Trees that have no value aspect are considered to be variables and are translated into an index into arg*.

Hence, a fingerprint like (2 (3) 4) denotes a tree (u (v) w) where u and v are symbol 2 and 3 of the given directory and w is tree number 4 in arg*.

tree2fingerprint treats lambda abstraction specially in that the lambda variable is consed onto the given arg* so that variables are translated into deBruijn indices.
|#

(deff tree2fingerprint (arg* tree directory cache)
 (:let ((ref id) . tree*) tree)
 (:when (equalp ref 0) (- (vector2card id)))
 (:let def (aget cache ref card-codex ref id 0 card-value))
 (:when (null def) (position tree arg* :test 'c-tree-equal))
 (:when (parenthesisp def) ; skip past parenthesis construct
  (tree2fingerprint arg* (first tree*) directory cache))
 (:let index (position (cons ref id) directory :test 'equal))
 (:when (equalp def tree-lambda)
  (:let (arg . tree*) tree*)
  (:let arg* (cons arg arg*))
  (:let fingerprint* (tree*2fingerprint* arg* tree* directory cache))
  (cons index fingerprint*))
 (:let fingerprint* (tree*2fingerprint* arg* tree* directory cache))
 (cons index fingerprint*))

(deff tree*2fingerprint* (arg* tree* directory cache)
 (:when (null tree*) nil)
 (:let (tree . tree*) tree*)
 (:let fingerprint (tree2fingerprint arg* tree directory cache))
 (:let fingerprint* (tree*2fingerprint* arg* tree* directory cache))
 (cons fingerprint fingerprint*))

#|
=============================================
Convert list of symbols to fingerprint
=============================================
(symbol*2fingerprint symbol* directory cache result) converts the value aspects of the given list of symbols to fingerprints.
|#

(deff symbol*2fingerprint (symbol* directory cache result)
 (:when (null symbol*) (reverse result))
 (:let ((ref . id) . symbol*) symbol*)
 (:let def (aget cache ref card-codex ref id 0 card-value))
 (:when (proclaimedp def)
  (:let ((nil id)) def)
  (:let result (cons id result))
  (symbol*2fingerprint symbol* directory cache result))
 (:let (nil nil (nil . arg*) rhs) def)
 (:let arity (length arg*))
 (:let fingerprint (tree2fingerprint arg* rhs directory cache))
 (:let entry (list arity fingerprint))
 (:let result (cons entry result))
 (symbol*2fingerprint symbol* directory cache result))

#|
=============================================
Convert definition of symbol to fingerprint
=============================================
(symbol2fingerprint ref id cache) computes the fingerprint of the definition of the symbol with the given reference and id.

symbol2fingerprint does so by a call to symbol2symbol* which finds all symbols that are transitively used for defining the symbol. Then it reverses the symbol* and uses it both as symbol* and directory in a call to symbol*2fingerprint. Reversing symbol* and using it as directory has the effect that symbols are indexed 'chronologically', i.e. such that the most fundamental definitions have small indices and dependent definitions have larger indices.
|#

(deff symbol2fingerprint (ref id cache)
 (:let symbol* (symbol2symbol* ref id cache nil))
 (:when (null symbol*) '((1 0))) ; Fingerprint of parenthesis
 (:let symbol* (reverse symbol*))
 (:let fingerprint (symbol*2fingerprint symbol* symbol* cache nil))
 fingerprint)

#|
=============================================
COMPILATION
=============================================
|#

#|
=============================================
Translation of trees to terms
=============================================
(cache2cache cache ref) accesses the cache associated with the given ref from the given cache. cache2cache does so by first finding the ref1 with which the given cache is associated, and if ref1 equals ref then cache2cache returns the cache unchanged. Otherwise, cache2cache retrieves the cache associated with the given ref from the given cache. This yields the cache associated with the given ref provided the given ref is in the transitive bibliography of ref1.
|#

(deff cache2cache (cache ref)
 (:let ref1 (aget cache 0))
 (:when (= ref ref1) cache)
 (aget cache ref1 card-cache ref))

(deff tree2term (cache arglist tree)
 (:let ((ref id) . tree*) tree)
 (:when (= ref 0) (term-tv id))
 (:when (= id 0)
  (:let name (format nil "~a-" (c-symbol2name-cache ref id cache)))
  (term-big-tv name (cache2cache cache ref)))
 (:let fct (aget cache ref card-code id))
 (:when (null fct) (var2term arglist tree))
 (:when (equalp fct fct-lambda) (lambda2term cache arglist tree*))
 (:when (equalp fct fct-quote) (quote2term (car tree*)))
 (apply-fct fct (tree*2term* cache arglist tree*)))

(deff var2term (arglist tree)
 (:let position (position tree arglist :test 'c-tree-equal))
 (:when (null position) term-true)
 (term-var position))

(deff lambda2term (cache arglist tree*)
 (:let (var tree) tree*)
 (term-lambda (tree2term cache (cons var arglist) tree)))

(deff quote2term (tree)
 (term-tv tree))

(deff tree*2term* (cache arglist tree*)
 (:when (atom tree*) nil)
 (:let (tree . tree*) tree*)
 (:let term (tree2term cache arglist tree))
 (:let term* (tree*2term* cache arglist tree*))
 (cons term term*))

#|
=============================================
Compile symbols
=============================================
(compile-symbol symbol) compiles the symbol-function of the given symbol with error messages sent to *error-output*.

Enable *compilation-errors* to see error messages from the Lisp compiler
|#

(defc *compilation-errors* nil)
(defc *compilation-backlog* nil)

(deff compile-symbol (symbol)
 (when *compilation-errors* (format t "; Compiling ~s~%" symbol))
 (:when *compilation-errors* (compile symbol))
 (with-output-to-string (*error-output*) (compile symbol)))

(deff compile-later (symbol)
 (push symbol *compilation-backlog*))

(deff compile-backlog ()
 (dolist (symbol *compilation-backlog*) (compile-symbol symbol))
 (setq *compilation-backlog* nil))

#|
=============================================
Compile lazy function
=============================================
(c-compile-lazy symbol cache) translates the symbol-value of the given symbol to a symbol-function. c-compile-lazy is called for user definitions which are neither recognized by introduce, nor fit for optimized compilation.

(c-term2lambda term) translates the given term to a Lisp s-expression that denotes the corresponding term function. Evaluating that s-expression yields the function itself.
|#

(deff tracecompile (symbol)
 (tracecompile1 (option "tracecompile") (symbol-name symbol)))

(deff tracecompile1 (string* name)
 (:when (atom string*) nil)
 (:let (string . string*) string*)
 (:when (search string name) t)
 (tracecompile1 string* name))

(deff c-compile-lazy (symbol cache)
 (:let def (symbol-value symbol))
 (:let (:define :value (:lhs . arg*) rhs) def)
 (:let term (tree2term cache arg* rhs))
 (:let fct (c-term2lambda term))
 (when (tracecompile symbol)
  (format t "Compiling ~s~%" symbol)
 ;(limit-print 4 4 (format t "~s~%~%" fct))
  (format t "~s~%~%" fct))
 (setf (symbol-function symbol) (eval fct))
 (compile-later symbol))

(deff c-term2lambda (term)
 `(lambda (env &rest term*)
   (term2rnf ',term (term*2closure* term* env))))

#|
=============================================
Compile eager function
=============================================
(c-compile-eager symbol cache) translates the symbol-value of the given symbol to a function. c-compile-lazy is called for user definitions which are fit for optimized compilation.
|#

(defc c-accept '(
  (|" norm|                 . c-compile-norm)
  (|norm "|                 . c-compile-norm)
  (|!"|                     . c-compile-norm)

  (|" is val : "|           . c-compile-is-val)
  (|" is bool : "|          . c-compile-is-bool)
  (|" is int : "|           . c-compile-is-int)
  (|" is pair : "|          . c-compile-is-pair)
  (|" is map : "|           . c-compile-is-map)
  (|" is object : "|        . c-compile-is-object)
  (ttag-true                . c-compile-true)
  (|spy " : "|              . c-compile-spy)
  (|if " then " else "|     . c-compile-if)
  (|" .and. "|              . c-compile-and)
  (|" .or. "|               . c-compile-or)
  (|" catch|                . c-compile-catch)
  (|" catching maptag|      . c-compile-catching-maptag)
  (|map ( " )|              . c-compile-map)
  (|LET " BE "|             . c-compile-let)))

(deff c-compile-norm (gensym* arg* tree* cache)
 (:let (tree) tree*)
 (c-tree2lisp gensym* arg* tree cache))

(deff c-compile-is-val (gensym* arg* tree* cache)
 (:let (:guard tree) tree*)
 (c-tree2lisp gensym* arg* tree cache))

(deff c-compile-is-bool (gensym* arg* tree* cache)
 (:let (guard tree) (c-tree*2lisp* gensym* arg* tree* cache))
`(if (boolp ,guard) ,tree (raise)))

(deff c-compile-is-int (gensym* arg* tree* cache)
 (:let (guard tree) (c-tree*2lisp* gensym* arg* tree* cache))
`(if (intp ,guard) ,tree (raise)))

(deff c-compile-is-pair (gensym* arg* tree* cache)
 (:let (guard tree) (c-tree*2lisp* gensym* arg* tree* cache))
`(if (pairp ,guard) ,tree (raise)))

(deff c-compile-is-map (gensym* arg* tree* cache)
 (:let (guard tree) (c-tree*2lisp* gensym* arg* tree* cache))
`(if (mapp ,guard) ,tree (raise)))

(deff c-compile-is-object (gensym* arg* tree* cache)
 (:let (guard tree) (c-tree*2lisp* gensym* arg* tree* cache))
`(if (objectp ,guard) ,tree (raise)))

(deff c-compile-true (:gensym* :arg* :tree* :cache)
 nil)

(deff c-spy (x y)
 (setq *spy* x)
 (spycd)
 y)

(deff c-compile-spy (gensym* arg* tree* cache)
 (cons 'c-spy (c-tree*2lisp* gensym* arg* tree* cache)))

; 'then' and 'else' branch swap because NIL = Logiweb truth = Lisp falsehood
(deff c-compile-if (gensym* arg* tree* cache)
 (:let (cond then else) (c-tree*2lisp* gensym* arg* tree* cache))
 (list 'if cond else then))

; AND becomes OR because NIL = Logiweb truth = Lisp falsehood
(deff c-compile-and (gensym* arg* tree* cache)
 (cons 'or (c-tree*2lisp* gensym* arg* tree* cache)))

; OR becomes AND because NIL = Logiweb truth = Lisp falsehood
(deff c-compile-or (gensym* arg* tree* cache)
 (cons 'and (c-tree*2lisp* gensym* arg* tree* cache)))

(deff c-catch (x)
 (:when (exception-p x) (cons nil (head (exception-value x))))
 (cons :false x))

(deff c-compile-catch (gensym* arg* tree* cache)
 `(c-catch (catch :exception ,@(c-tree*2lisp* gensym* arg* tree* cache))))

(deff c-catching-maptag (x)
 (:when (not (exception-p x)) (make-tagmap :map (map-tv x)))
 (:let tv (exception-value x))
 (:let tm (map-tv tv))
 (:let tm (map-pair (map-pair map-0 map-3) tm))
 (make-tagmap :map tm))

(deff c-compile-catching-maptag (gensym* arg* tree* cache)
 `(c-catching-maptag
   (catch :exception ,@(c-tree*2lisp* gensym* arg* tree* cache))))

(deff c-compile-map (:gensym* :arg* tree* cache)
 (:let (tree) tree*)
 (:let arglist nil)
 (:let term (tree2term cache arglist tree))
 (:let env nil)
 (:let map (term2closure term env))
 (make-tagmap :map map))

(deff c-compile-let (gensym* arg* tree* cache)
 (:let (tree (:lambda arg body)) tree*)
 (:let lisp (c-tree2lisp gensym* arg* tree cache))
 (:let gensym (c-named-gensym arg "-" cache))
 (:let gensym* (cons gensym gensym*))
 (:let arg* (cons arg arg*))
 (:let body (c-tree2lisp gensym* arg* body cache))
 `(let ((,gensym ,lisp)) ,body))

(deff c-arg2lisp (gensym* arg* tree)
 (:let position (position tree arg* :test 'c-tree-equal))
 (nth position gensym*))

(deff c-tree2lisp (gensym* arg* tree cache)
 (:let ((ref id) . tree*) tree)
 (:when (equalp ref 0) id)
 (:when (equalp id 0)
  (:let name (format nil "~a-" (c-symbol2name-cache ref id cache)))
  (value2gensym name (cache2cache cache ref)))
 (:let fct (aget cache ref card-code id))
 (:when (null fct) (c-arg2lisp gensym* arg* tree))
 (:when (equalp fct fct-quote) `(quote ,(car tree*)))
 (:when (no-fct fct) (error "Internal check failed: unknown fct"))
 (:let (symbol :type :arity) (thefct fct))
 (:let compiler (assoc-get symbol c-accept))
 (:when compiler (funcall compiler gensym* arg* tree* cache))
 (cons symbol (c-tree*2lisp* gensym* arg* tree* cache)))

(deff c-tree*2lisp* (gensym* arg* tree* cache)
 (:when (atom tree*) nil)
 (:let (tree . tree*) tree*)
 (:let lisp (c-tree2lisp gensym* arg* tree cache))
 (:let lisp* (c-tree*2lisp* gensym* arg* tree* cache))
 (cons lisp lisp*))

(deff c-body2constant (body lhs cache)
 (:let gensym (c-named-gensym lhs "-CONSTANT-" cache))
 `((declare (special ,gensym))
   (unless (boundp ',gensym) (setq ,gensym ,(unite body)))
   ,gensym))

(deff c-arg*2gensym* (arg* result cache)
 (:when (atom arg*) (reverse result))
 (:let (arg . arg*) arg*)
 (c-arg*2gensym* arg* (cons (c-named-gensym arg "-" cache) result) cache))

(deff c-compile-eager (symbol cache)
 (:let def (symbol-value symbol))
 (:let (:define :value lhs rhs) def)
 (:let (:root . arg*) lhs)
 (:let gensym* (c-arg*2gensym* arg* nil cache))
 (:let declaration `(declare (ignorable ,@gensym*)))
 (:let body (list (c-tree2lisp gensym* arg* rhs cache)))
 (:let body (deff-trace symbol nil body))
 (:let body (if gensym* body (c-body2constant body lhs cache)))
 (:let fct (list* 'lambda gensym* declaration body))
 (when (tracecompile symbol)
  (format t "Compiling ~s~%" symbol)
 ;(limit-print 4 4 (format t "~s~%~%" fct))
  (format t "~s~%~%" fct))
 (setf (symbol-function symbol) (eval fct))
 (compile-later symbol))

#|
=============================================
Analyze code
=============================================
(c-analyze-code ref id* cache) makes 'eagerness' analysis of all constructs with the given ref and id's.

A construct is 'eager' if:

(1) Its arguments are evaluated left to right and exceptions are relayed immediately.

(2) (c x_1 ... x_n) = (norm (c x_1 ... x_n))

(3) (c x_1 ... x_n) = (c (norm x_1) ... (norm x_n))

As can be seen, eager functions need not be strict. As an example, eager plus, x+y, will not evaluate y if x raises an exception. This is why c-analyze-code is said to do 'eagerness' analysis instead of strictness analysis.

Requirement (2) and (3) refer to the normalization function for 'tagged maps', c.f. optimize.lisp. The norm function is a retract so that (norm (norm x)) = (norm x).

Each construct is defined by an equation of form (c x_1 ... x_n) = rhs where the right hand side rhs is built up from constructs and parameters.

The eagerness analysis considers a construct to be strict if the following, sufficient conditions are met:

(A) The rhs has form (norm (guard_1 x_1 ... (guard_n x_n F) ... )) where F is built up from constructs and parameters. The 'norm' in the root guarantees (2) above. The guards must belong to a list of valid guards which are known to satisfy

(A1) (guard x y) = x when x is an exception.

(A2) (norm (guard x y)) = (norm (guard (norm x) (norm y)))

Presense of the guards guarantees (1) above.

(B) The F from (A) above must be constructed from parameters plus constructs d which satisfy (norm (d x_1 ... x_n)) = (norm (d (norm x_1) ... (norm x_n))). This satisfies (3) above.

(c-analyze-code ref id* cache)
|#

(deff c-analyze-code (ref id* cache)
 (setq *unfit* nil)
 (:let matrix (c-code2matrix ref id* cache nil))
 (c-propagate-laziness ref id* cache matrix)
 (c-conclude-eagerness ref id* cache))

(deff c-propagate-laziness (ref id* cache matrix)
 (:when (atom id*) nil)
 (:let (id . id*) id*)
 (c-propagate-laziness1 ref id cache matrix)
 (c-propagate-laziness ref id* cache matrix))

(deff c-propagate-laziness1 (ref id cache matrix)
 (:let fct (aget cache ref card-code id))
 (:when (no-fct fct) nil)
 (:let (symbol type :arity) (thefct fct))
 (:when (unequal type :lazy) nil)
 (c-propagate-laziness2 ref symbol cache matrix (aget matrix id)))

(deff c-propagate-laziness2 (ref called cache matrix array)
 (:when (null array) nil)
 (:let (head . tail) array)
 (:when (integerp head) (c-propagate-laziness3 ref called cache matrix head))
 (c-propagate-laziness2 ref called cache matrix head)
 (c-propagate-laziness2 ref called cache matrix tail))

(deff c-push-unfit (reason)
 (:when (verbose '> 1) (format t "Mistakenly unfit?~%~a~%" reason))
 (when (null *unfit*) (format t "Mistakenly unfit function seen?~%"))
 (push reason *unfit*))

(deff c-propagate-laziness3 (ref called cache matrix id)
 (:let fct (aget cache ref card-code id))
 (:when (no-fct fct) nil)
 (:let (symbol type :arity) (thefct fct))
 (:when (unequal type :unknown) nil)
 (c-push-unfit (format nil "~a~%  Calls ~a~%" symbol called))
 (setf (second (fctstruct-data fct)) :lazy)
 (c-propagate-laziness2 ref symbol cache matrix (aget matrix id)))

(deff c-conclude-eagerness (ref id* cache)
 (:when (atom id*) nil)
 (:let (id . id*) id*)
 (c-conclude-eagerness1 ref id cache)
 (c-conclude-eagerness ref id* cache))

(deff c-conclude-eagerness1 (ref id cache)
 (:let fct (aget cache ref card-code id))
 (:when (no-fct fct) nil)
 (:let (:symbol type :arity) (thefct fct))
 (:when (unequal type :unknown) nil)
 (setf (second (fctstruct-data fct)) :eager))

(deff c-code2matrix (ref id* cache matrix)
 (:when (atom id*) matrix)
 (:let (id . id*) id*)
 (:let matrix (c-code2matrix1 ref id cache matrix))
 (:let matrix (c-code2matrix ref id* cache matrix))
 matrix)

; Given fct found to be lazy. Do not update incidence matrix
(deff c-lazy (reason fct)
 (:let list (thefct fct))
 (:let (symbol :type :arity) list)
 (when reason (c-push-unfit (format nil "~a~%  ~a~%" symbol reason)))
 (setf (second list) :lazy)
 (raise))

; Note in incidence matrix that id1 calls id2 (= id2 is called from id1)
(deff c-incidence (id1 id2 matrix)
 (aput matrix t id2 id1))

(deff c-code2matrix1 (ref id cache matrix)
 (:catch () matrix)
 (:let fct (aget cache ref card-code id))
 (:when (no-fct fct) matrix)
 (:let (symbol type :arity) (thefct fct))
 (:when (unequal type :unknown) matrix)
 (:let (:def :value (:lhs . arg*) rhs) (symbol-value symbol))
 (:let ((sref sid) tree) rhs)
 (:when (equalp sref 0) (c-lazy nil fct))
 (:let sfct (aget cache sref card-code sid))
 (:when (no-fct sfct) (c-lazy nil fct))
 (:let (symbol :type :arity) (thefct sfct))
 (:when (unequal symbol '|norm "|) (c-lazy nil fct))
 (c-code2matrix2 tree arg* fct id arg* cache matrix))

(defc c-guards '(
  (|" is val : "|    . t)
  (|" is bool : "|   . t)
  (|" is int : "|    . t)
  (|" is pair : "|   . t)
  (|" is map : "|    . t)
  (|" is object : "| . t)))

(deff c-code2matrix2 (tree arg* fct id args cache matrix)
 (:when (atom arg*) (c-code2matrix3 tree fct id args cache matrix))
 (:let (arg . arg*) arg*)
 (:let ((sref sid) tree1 tree2) tree)
 (:when (equalp sref 0) (c-lazy "String where guard expected" fct))
 (:let sfct (aget cache sref card-code sid))
 (:when (no-fct sfct)
  (c-lazy "Lambda, quote, or undefined where guard expected" fct))
 (:let (symbol :type :arity) (thefct sfct))
 (:when (unequal symbol '|" is val : "|)
  (c-lazy (format nil "~a not guard" symbol) fct))
 (:unless (c-tree-equal arg tree1)
  (c-lazy "Mismatch between parameter list and guards" fct))
 (c-code2matrix2 tree2 arg* fct id args cache matrix))

(deff c-code2matrix3 (tree fct id args cache matrix)
 (:let ((sref sid) . tree*) tree)
 (:when (equalp sref 0) matrix)
 (:when (equalp sid 0) matrix)
 (:let sfct (aget cache sref card-code sid))
 (:when (equalp sfct fct-lambda) (c-lazy "Calls lambda" fct))
 (:when (equalp sfct fct-quote) matrix)
 (:when (no-fct sfct)
  (:when (member tree args :test 'c-tree-equal) matrix)
  (:let name (c-symbol2name-cache sref sid cache))
  (c-lazy (format nil "Calls ~a" name) fct))
 (:let (symbol type :arity) (thefct sfct))
 (:when (equalp type :eager) (c-code2matrix3* tree* fct id args cache matrix))
 (:when (equalp type :unknown)
  (:let matrix (c-incidence id sid matrix))
  (c-code2matrix3* tree* fct id args cache matrix))
 (:when (equalp symbol '|map ( " )|)
  (:let (tree) tree*)
  (c-map2matrix tree fct id args cache matrix))
 (:when (equalp symbol '|LET " BE "|)
  (c-let2matrix tree* fct id args cache matrix))
 (:when (assoc-get symbol c-accept)
  (c-code2matrix3* tree* fct id args cache matrix))
 (c-lazy (format nil "Calls ~a" symbol) fct))

(deff c-let2matrix (tree* fct id args cache matrix)
 (:let (tree lambda) tree*)
 (:let matrix (c-code2matrix3 tree fct id args cache matrix))
 (:let ((sref sid) arg body) lambda)
 (:when (equalp sref 0) (c-lazy "Malformed let" fct))
 (:when (equalp sid 0) (c-lazy "Malformed let" fct))
 (:let sfct (aget cache sref card-code sid))
 (:when (unequal sfct fct-lambda) (c-lazy "Malformed let" fct))
 (c-code2matrix3 body fct id (cons arg args) cache matrix))

(deff c-code2matrix3* (tree* fct id args cache matrix)
 (:when (atom tree*) matrix)
 (:let (tree . tree*) tree*)
 (:let matrix (c-code2matrix3 tree fct id args cache matrix))
 (c-code2matrix3* tree* fct id args cache matrix))

(deff c-map2matrix (tree fct id args cache matrix)
 (:let ((sref sid) . tree*) tree)
 (:when (equalp sref 0) matrix)
 (:when (equalp sid 0) matrix)
 (:let sfct (aget cache sref card-code sid))
 (:when (equalp sfct fct-lambda)
  (:let (arg tree) tree*)
  (:let args (remove arg args :test 'c-tree-equal))
  (c-map2matrix tree fct id args cache matrix))
 (:when (equalp sfct fct-quote) matrix)
 (:when (no-fct sfct)
  (:unless (member tree args :test 'c-tree-equal) matrix)
  (:let name (c-symbol2name-cache sref sid cache))
  (c-lazy (format nil "Refers to argument ~a inside tagged map" name) fct))
 (c-map*2matrix* tree* fct id args cache matrix))

(deff c-map*2matrix* (tree* fct id args cache matrix)
 (:when (atom tree*) matrix)
 (:let (tree . tree*) tree*)
 (:let matrix (c-map2matrix tree fct id args cache matrix))
 (c-map*2matrix* tree* fct id args cache matrix))

#|
=============================================
Compile code
=============================================
(c-compile-code code cache cache) compiles the given code as a side effect and returns nothing useful. The given code is an array which maps id's to fct structures. c-compile-code1 scans code and calls c-compile-code2 on each fct structure.

c-compile-code2 below does not compile code for symbols that are fboundp. Such symbols can be:
(1) ttag-if, ttag-apply, or ttag-true
(2) introduced functions which are predefined in e.g. optimize.lisp
(3) introduced functions which are not predefined but which have been
user defined previously.
|#

(deff c-compile-code (code cache)
 (setq *compilation-backlog* nil)
 (c-compile-code1 code cache)
 (compile-backlog))

(deff c-compile-code1 (code cache)
 (:when (null code) nil)
 (:let (code1 . code2) code)
 (:when (numberp code1) (c-compile-code2 code2 cache))
 (c-compile-code1 code1 cache)
 (c-compile-code1 code2 cache))

(deff c-print-lhs (symbol cache)
 (:let def (symbol-value symbol))
 (:let (:definition :aspect ((ref id)) :rhs) def)
 (:let name (c-symbol2name-cache ref id cache))
 (format t "Compiling ~a~%" (utf2string name)))

(deff c-compile-code2 (fct cache)
 (:when (no-fct fct) nil)
 (:let (symbol type :arity) (thefct fct))
 (:when (fboundp symbol) nil)
 (:when (equalp type :eager) (c-compile-eager symbol cache))
 (:when (equalp type :lazy) (c-compile-lazy symbol cache))
 (error "Internal error: Unknown type of ~s" (thefct fct)))

#|
=============================================
Alternative to fingerprint.lisp
=============================================
The structure below allows to recognize optimized contructs
by name rather than by fingerprint when fingerprint.lisp is
empty.
|#

(defc *opti2* nil)

(defmacro register2 (symbol &optional (fingerprint symbol))
 `(register3 ',symbol ',fingerprint))

(deff register3 (symbol fingerprint)
 (:let arity (f-arity (ct2ct (symbol-name fingerprint))))
 (:let type (symbol-value symbol))
 (:let fct (fct symbol type arity))
 (push (cons fingerprint fct) *opti2*))

; Implement construct |( " )| by function |!"| etc.
(register2 |!"| |( " )|)
(register2 |!"| |hide " end hide|)
(register2 |!"| |hiding name " end name|)
(register2 |!"| |name " end name|)
(register2 |!"| |newline "|)
(register2 |!"| |show " end show|)
(register2 |!"| |hiding show " end show|)
(register2 |" catch| |catch ( " )|)

; Implement construct |!"| by function |!"| etc.
(register2 |!"|)
(register2 |" %0|)
(register2 |" %1|)
(register2 |" %2|)
(register2 |" %3|)
(register2 |" %4|)
(register2 |" %5|)
(register2 |" %6|)
(register2 |" %7|)
(register2 |" %8|)
(register2 |" %9|)
(register2 |" * "|)
(register2 |" + "|)
(register2 |" - "|)
(register2 |" .and. "|)
(register2 |" .or. "|)
(register2 |" .then. "|)
(register2 |" :: "|)
(register2 |" < "|)
(register2 |" <= "|)
(register2 |" = "|)
(register2 |" > "|)
(register2 |" >= "|)
(register2 |" apply "|)
(register2 |" boolp|)
(register2 |" catch|)
(register2 |" catching maptag|)
(register2 |" div "|)
(register2 |" head|)
(register2 |" intp|)
(register2 |" is bool : "|)
(register2 |" is int : "|)
(register2 |" is map : "|)
(register2 |" is object : "|)
(register2 |" is pair : "|)
(register2 |" is val : "|)
(register2 |" mapp|)
(register2 |" maptag|)
(register2 |" mod "|)
(register2 |" norm|)
(register2 |" objectp|)
(register2 |" pairp|)
(register2 |" raise|)
(register2 |" root|)
(register2 |" tail|)
(register2 |" untag|)
(register2 %% %%)
(register2 |+ "|)
(register2 |+"|)
(register2 |- "|)
(register2 |-"|)
(register2 |.not. "|)
(register2 |Base|)
(register2 |LET " BE "|)
(register2 YY YY)
(register2 |ash ( " , " )|)
(register2 |bottom|)
(register2 |bt2byte* ( " )|)
(register2 |bt2vector ( " )|)
(register2 |bt2vector* ( " )|)
(register2 |ceiling ( " , " )|)
(register2 |compile ( " )|)
(register2 |destruct ( " )|)
(register2 |evenp ( " )|)
(register2 |exception|)
(register2 |false|)
(register2 |floor ( " , " )|)
(register2 |half ( " )|)
(register2 |if " then " else "|)
(register2 |integer-length ( " )|)
(register2 |logand ( " , " )|)
(register2 |logandc1 ( " , " )|)
(register2 |logandc2 ( " , " )|)
(register2 |logbitp ( " , " )|)
(register2 |logcount ( " )|)
(register2 |logeqv ( " , " )|)
(register2 |logior ( " , " )|)
(register2 |lognand ( " , " )|)
(register2 |lognor ( " , " )|)
(register2 |lognot ( " )|)
(register2 |logorc1 ( " , " )|)
(register2 |logorc2 ( " , " )|)
(register2 |logtest ( " , " )|)
(register2 |logxor ( " , " )|)
(register2 |map ( " )|)
(register2 |norm "|)
(register2 |notnot "|)
(register2 |object ( " )|)
(register2 |print ( " )|)
(register2 |rack2sl ( " )|)
(register2 |ripemd ( " )|)
(register2 |round ( " , " )|)
(register2 |sl2rack ( " )|)
(register2 |spy ( " )|)
(register2 |timer ( " )|)
(register2 |trace ( " )|)
(register2 |truncate ( " , " )|)
(register2 |vector ( " )|)
(register2 |vector-empty ( " )|)
(register2 |vector-index ( " , " )|)
(register2 |vector-length ( " )|)
(register2 |vector-norm ( " )|)
(register2 |vector-prefix ( " , " )|)
(register2 |vector-subseq ( " , " , " )|)
(register2 |vector-suffix ( " , " )|)
(register2 |vector2byte* ( " )|)
(register2 |vector2vector* ( " )|)
(register2 |vt2byte* ( " )|)
(register2 |vt2vector ( " )|)
(register2 |vt2vector* ( " )|)

#|
=============================================
Compile revelations
=============================================
(c-codex2code ref state) compiles value revelations on the given page and adds them to the code of the state. The structure of 'code' is given in the section on printing code above.

(c-codex2assoc ref id* state assoc) compiles value revelations for the given ids on the given page and adds them to the given assoc. ids in id* that have no value definition are ignored.

(c-revelation2fct rev state) translates the given revelation to an fct-structure. If the definition is missing (null) or invalid somehow, then c-revelation2fct returns nil which is then ignored in the calling function. If the revelation is a definition then c-revelation2fct returns an fct-structure containing a gensym whose symbol-value equals the definition. If the revelation is an introduction then c-revelation2fct returns an fct-structure that denotes an introduced function. If no such introduced function is found, c-revelation2fct prints a warning that contains the unmatched fingerprint and treats the introduction as a definition.
|#

(deff c-proclamation2fct (id)
 (:when (equalp id card-lambda) fct-lambda)
 (:when (equalp id card-apply) fct-apply)
 (:when (equalp id card-true) fct-true)
 (:when (equalp id card-if) fct-if)
 (:when (equalp id card-quote) fct-quote)
 nil)

(deff c-definition2fct (def cache)
 (:let (:define :value lhs :rhs) def)
 (:let gensym (c-named-gensym lhs "-" cache))
 (setf (symbol-value gensym) def)
 (:let type :unknown)
 (:let arity (1- (length lhs)))
 (fct gensym type arity))

(deff c-introduction2fct (def cache)
 (:let (nil nil ((ref id))) def)
 (:let fingerprint (symbol2fingerprint ref id cache))
 (push (list fingerprint ref id) *introduced*)
 (:let name (utf2string (c-symbol2name-cache ref id cache)))
 (:let fct-name (intern name))
 (:let fct
  (if
   (equalp *opti* :undefined)
   (assoc-get fct-name *opti2*)
   (assoc-get fingerprint *opti*)))
 (:when fct
  (:let (symbol) (thefct fct))
  (:when (fboundp symbol) fct)
  (setf (symbol-value symbol) def)
  fct)
 (when (verbose '>= 0) (format t "Unrecognized introduction of ~s~%" fct-name))
 (push (cons fct-name fingerprint) *unrecognized*)
 (c-definition2fct def cache))

(deff c-revelation2fct (revelation cache)
 (:when (null revelation) nil)
 (:let ((ref id)) revelation)
 (:when (equalp ref 0) (c-proclamation2fct id))
 (:let ((ref id)) (aget cache ref card-codex ref id 0 card-definition))
 (:when (unequal ref 0) nil)
 (:when (equalp id card-define) (c-definition2fct revelation cache))
 (:when (equalp id card-introduce) (c-introduction2fct revelation cache))
 nil)

(deff c-codex2code1 (ref id* cache code)
 (:when (null id*) code)
 (:let (id . id*) id*)
 (:let revelation (aget cache ref card-codex ref id 0 card-value))
 (:let fct (c-revelation2fct revelation cache))
 (:let code (if (null fct) code (aput code fct id)))
 (:let code (c-codex2code1 ref id* cache code))
 code)

(deff c-compile-cache (cache)
 (setq *unrecognized* nil)
 (setq *introduced* nil)
 (:let ref (aget cache 0))
 (:let id* (adom cache ref card-codex ref))
 (:let code (c-codex2code1 ref id* cache nil))
 (:let cache (aput cache code ref card-code))
 (c-analyze-code ref id* cache)
 (c-compile-code code cache)
 cache)

(deff c-compile-verify-cache (cache)
 (setq *unrecognized* nil)
 (setq *introduced* nil)
 (:let ref (aget cache 0))
 (:let id* (adom cache ref card-codex ref))
 (:let code (c-codex2code1 ref id* cache nil))
 (:let cache (aput cache code ref card-code))
 (c-analyze-code ref id* cache)
 (c-compile-code code cache)
; This following is an incomplete implementation of verification.
; The diagnose is not normalized to a term.
 (:let cache (aput cache (make-tagmap :map  map-b) ref card-diagnose))
 (:let def (c-get-page-aspect ref card-claim cache))
 (:when (null def) cache)
 (:let (:define :claim :lhs claim) def)
 (:let arglist nil)
 (:let term (tree2term cache arglist claim))
 (:let term (term-apply term (term-tv cache)))
 (:let map (term2closure term nil))
 (:let cache (aput cache (make-tagmap :map map) ref card-diagnose))
 cache)

(deff c-codex2code (ref state)
 (:let cache (aget state (id-cache) ref))
 (:let cache (c-compile-cache cache))
 (:let state (aput state cache (id-cache) ref))
 state)

#|
=============================================
VERIFICATION
=============================================
|#

#|
=============================================
Verify claim
=============================================
(c-verify ref state) reads the claim aspect of the referenced page, translates it to a term with an empty arglist, and reduces it to root normal form in the empty environment, and hangs the result on the diagnose hook after pruning.
|#

(defc *diagnose* nil)

(deff c-verify (ref state)
 (:let state (rack-put state (maptag nil) ref card-diagnose))
 (:let cache (aget state (id-cache) ref))
 (:let def (c-get-page-aspect ref card-claim cache))
 (:when (null def) (format t "No claim found~%") state)
 (:let (:define :claim :lhs claim) def)
 (:let arglist nil)
 (:let term (tree2term cache arglist claim))
 (:let term (term-apply term (term-tv cache)))
 (setq *spy-state* state);so that we can render *spy* in case of stack overflow
 (:let rnf (term2rnf term nil))
 (print-timers)
 (when (null *spy*) (setq *spy-state* nil))
 (setq *diagnose* nil)
 (:let name (c-symbol2name ref 0 state))
 (:let result (ifnil rnf "succeeded" "failed"))
 (format t "~%Claim of page '~a' ~a.~%~%" (utf2string name) result)
 (:when rnf
  (:catch ()
   (format t "The verifier raised an exception~%")
   (rack-put state (maptag (list (list ref 0))) ref card-diagnose))
  (:let tree (tm2tv rnf))
  (:let tree (c-prune-tree "diagnose" ref tree state))
  (:let state (rack-put state (maptag tree) ref card-diagnose))
  (setq *diagnose* tree)
  (setq *spy-state* state))
 state)

#|
=============================================
CACHING
=============================================
(dump-rack ref state) stores the rack associated to ref by state.
The rack is stored such that sharing can be restored when reading
by (load-rack ref).

(dump-rack ref state) and (load-rack ref) access a file whose
path is composed from the "cache" option and the mixed endian hexadecimal representation of ref.

(dump-rack ref state) stores a cleaned version of the rack from
which the code and cache hooks are removed. The code hooks are
removed to avoid compiled functions which dump-rack cannot store.
The cache hook is removed because it is very large and easy to
restore.
|#

#|
=============================================
Dump state to file
=============================================
Structures built up from cons, nil, cardinals, and byte vectors are represented thus:

Each cons, cardinal, and byte vector get an id. Nil-nodes are always represented by the constant c-nil.

Each cardinal is written once to the file as a tag, c-tag, followed by the cardinal expressed base 128 as a sequence of middle septets followed by an end septet.

Each byte vector is written once to the file as a tag, c-array, followed by the length of the vector expressed base 128 as above followed by the bytes of the array.

Each cons is written as two cardinals base 128 where the two cardinals represent the head and the tail. Each cardinal can be c-nil or the index of a data structure.
|#

(deff c-byte2hex (byte)
 (ct2string (html-number byte 2 16)))

(deff c-card2hex (ref)
 (:let card* (card2ref ref))
 (:let hex* (map 'list 'c-byte2hex card*))
 (apply 'concatenate 'string hex*))

(etst (c-card2hex #x01EFCDAB) "ABCDEF")

(deff c-cache-adr (cache ref)
 (concatenate 'string cache (c-card2hex ref) "/page.lgr"))

(etst (c-cache-adr "abc/" #x01EFCDAB) "abc/ABCDEF/page.lgr")

(defc *share-id* 0)
(defc *share-hash* (make-hash-table :test 'equalp))

(deff c-write-card (stream card)
 (:when (null stream) (format t "~d~%" card))
 (:when (< card 128) (write-byte card stream))
 (write-byte (+ 128 (logand card 127)) stream)
 (c-write-card stream (ash card -7)))

(defc c-tag 0)
(defc c-nil 1)
(defc c-array 2)
(defc c-start 3)

(deff c-write-integer (stream id data)
 (:let id1 (gethash data *share-hash*))
 (:when id1 id)
 (c-write-card stream c-tag)
 (c-write-card stream data)
 (setf (gethash data *share-hash*) id)
 (+ id 1))

(deff c-write-array (stream id array)
 (:let id1 (gethash array *share-hash*))
 (:when id1 id)
 (c-write-card stream c-array)
 (c-write-card stream (length array))
 (dotimes (i (length array)) (write-byte (aref array i) stream))
 (setf (gethash array *share-hash*) id)
 (+ id 1))

(deff c-write-data (stream data)
 (:when (null data) (c-write-card stream c-nil))
 (:when (atom data) (c-write-card stream (gethash data *share-hash*)))
 (:let (:tag id) data)
 (c-write-card stream id))

#|
(c-mark-data stream dir id root node) traverses a data structure depth first left to right.

(c-mark-data stream dir c-start nil data) assigns an identifier to each cons node in 'data' by overwriting all cons nodes (head . tail) by (:share-root id head . tail) where id is the assigned id. The id's start with c-start and count up. Values below c-start are reserved for tags of atomic data structures.

c-mark-data visites each cons node C three times:

First time, c-mark-data arrives to C from above. In this case the 'dir' ('dir' for 'direction') is :down, the 'node' parameter is C, and the root parameter represents the node above C (or is nil if C is the root of the entire structure).

Second time, c-mark-data arrives to C from the head branch. In this case the 'dir' parameter is :up, the head of C is :share-head, and the tail of C has form (nil tail root) where tail is the right subnode of C and root is the node above C.

Third time, c-mark-data arrives to C from the tail branch. In this case the 'dir' parameter is :up, the head of C is :share-tail, and the tail of C has form (nil head root)
|#
(deff c-mark-data (stream dir id root node)
 (:when (equalp dir :down)
  (:when (consp node)
   (:let (head . tail) node)
   (:when (equalp head :share-root) (c-mark-data stream :up id root node))
   (setf (car node) :share-head)
   (setf (cdr node) (list* nil tail root))
   (c-mark-data stream :down id node head))
  (:when (null node) (c-mark-data stream :up id root node))
  (:when (and (integerp node) (<= 0 node))
   (:let id (c-write-integer stream id node))
   (c-mark-data stream :up id root node))
  (:when (arrayp node)
   (:let id (c-write-array stream id node))
   (c-mark-data stream :up id root node))
  (error "Unknown type of circular structure"))
 (:when (null root) (c-write-card stream id) node)
 (:let (tag) root)
 (:when (equalp tag :share-head)
  (:let (:tag :id tail . :root) root)
  (setf (first root) :share-tail)
  (setf (third root) node)
  (c-mark-data stream :down id root tail))
 (:when (equalp tag :share-tail)
  (:let (:tag :id head . up) root)
  (setf (first root) :share-root)
  (setf (second root) id)
  (setf (cdddr root) node)
  (c-write-data stream head)
  (c-write-data stream node)
  (c-mark-data stream :up (+ 1 id) up root))
 (error "Internal error in c-mark-data1"))

#|
(c-unmark-data dir root node) reverses the effect of c-mark-data, i.e. it removes the id's assigned to cons nodes.
|#
(deff c-unmark-data (dir root node)
 (:when (equalp dir :down)
  (:when (atom node) (c-unmark-data :up root node))
  (:let (tag) node)
  (:when (unequal tag :share-root) (c-unmark-data :up root node))
  (:let (:tag :id head :tail) node)
  (setf (first node) :share-head)
  (setf (third node) root)
  (c-unmark-data :down node head))
 (:when (null root) node)
 (:let (tag) root)
 (:when (equalp tag :share-head)
  (:let (:tag :id :root . tail) root)
  (setf (first root) :share-tail)
  (setf (cdddr root) node)
  (c-unmark-data :down root tail))
 (:when (equalp tag :share-tail)
  (:let (:tag :id up . head) root)
  (setf (car root) head)
  (setf (cdr root) node)
  (c-unmark-data :up up root))
 (error "Internal error in c-unmark-data1"))

(deff write-sharing (stream data)
 (clrhash *share-hash*)
 (:let data (c-mark-data stream :down c-start nil data))
 (c-unmark-data :down nil data)
 (clrhash *share-hash*))

(deff c-mark-test (node)
 (c-unmark-data :down nil (c-mark-data nil :down c-start nil node)))

(deff dump-rack1 (path rack)
 (with-open-file
  (stream path
   :direction :output
   :element-type 'unsigned-byte
   :if-exists :supersede)
  (write-sharing stream rack)))

(deff dump-rack (ref state)
 (:let cache (option "cache"))
 (:when (equalp cache "/") nil)
 (:when (not (option "test")) nil)
 (progress "Load: dumping to cache")
 (:let rack (aget state (id-cache) ref ref))
 (:let rack (aput rack nil card-code))
 (:let rack (aput rack nil card-cache))
 (:let diagnose (aget rack card-diagnose))
 (:let diagnose (untag diagnose))
 (:let rack (aput rack diagnose card-diagnose))
 (:let path (c-cache-adr cache ref))
 (ensure-directories-exist path)
 (dump-rack1 path rack)
 (progress "Load: dump ended")
 (:let name (c-symbol2name ref 0 state))
 (:when (equalp (aref name 0) #\0) nil)
 (:let link (concatenate 'string cache name))
 (:let relpath (concatenate 'string  (c-card2hex ref) "/page.lgr"))
 (run-program "rm" :arguments (list "-f" link) :input nil)
 (run-program "ln" :arguments (list "-s" relpath link) :input nil))

#|
=============================================
Load state from file
=============================================
|#

(deff c-read-card (stream)
 (:let card (read-byte stream nil -1))
 (:when (< card 128) card)
 (+ card -128 (ash (c-read-card stream) 7)))

(deff c-read-vector (stream length result)
 (:when (= length 0) (card*2vector (reverse result)))
 (:let card (read-byte stream nil -1))
 (:when (< card 0) :corrupted)
 (c-read-vector stream (- length 1) (cons card result)))

(deff shared-array-put (array id value)
 (:let length (length array))
 (:when (< id length)
  (setf (aref array id) value)
  array)
 (:let array (adjust-array array (* 2 length)))
 (setf (aref array id) value)
 array)

(deff read-sharing1 (stream array id)
 (:let card0 (c-read-card stream))
 (:when (< card0 0) :corrupted)
 (:let card1 (c-read-card stream))
 (:when (< card1 0)
  (if (= card0 id) (aref array (- id 1)) :corrupted))
 (:when (= card0 c-tag)
  (:let array (shared-array-put array id card1))
  (read-sharing1 stream array (+ id 1)))
 (:when (= card0 c-array)
  (:let vector (c-read-vector stream card1 nil))
  (:let array (shared-array-put array id vector))
  (read-sharing1 stream array (+ id 1)))
 (:let head (aref array card0))
 (:let tail (aref array card1))
 (:let cons (cons head tail))
 (:let array (shared-array-put array id cons))
 (read-sharing1 stream array (+ id 1)))

(deff read-sharing (stream)
 (:when (null stream) nil)
 (:let array (make-array 1000000))
 (setf (aref array c-nil) nil)
 (read-sharing1 stream array c-start))

(deff load-rack1 (path)
 (with-open-file
  (stream path
   :direction :input
   :element-type 'unsigned-byte
   :if-does-not-exist nil)
  (read-sharing stream)))

(deff load-rack (ref)
 (:let cache (option "cache"))
 (:when (equalp cache "/") nil)
 (:let path (c-cache-adr cache ref))
 (ensure-directories-exist path)
 (:when (null (probe-file path)) nil)
 (when (verbose '>= 0) (progress "Load: loading from cache"))
 (:let rack (load-rack1 path))
 (:when (null rack) nil)
 (:when (equalp rack :corrupted)
  (format t "Cache corrupted. Cleaning.~%")
  (delete-file path)
  nil)
 (:let diagnose (aget rack card-diagnose))
 (:let diagnose (maptag diagnose))
 (:let rack (aput rack diagnose card-diagnose))
 (when (verbose '> 0) (progress "Load: load from cache succeeded"))
 rack)

#|
=============================================
LOADING
=============================================
|#

#|
=============================================
Load vector
=============================================
Loading a page involves the following steps:

1) Unpacking, which transforms the vector of the page into a bibliography, a dictionary, and a parse tree. The entries of the bibliography are recursively loaded during unpacking. Unpacking is done by c-unpack

2) Codification, which installs the information in a codex. Codification is done by c-codify.

3) Compilation, which installs optimized versions of the value aspects of symbols in a code. Compilation is done by c-codex2code.

4) Verification, which tests that the claim aspect of the page symbol evaluates to 'true'. Verification is done by c-verify.

(c-load ref state) loads (rack-get state ref card-vector), adds it to the given state, and returns the state. It complaints and raises an exception in case of error.
|#

(deff c-load (ref state)
 (when (verbose '> 0) (progress "Load: unpacking page"))
 (:let state (c-unpack ref state))
 (when (verbose '> 0) (progress "Load: codifying page"))
 (:let state (c-codify ref state))
;(c-print-code state ref)
 (:when (and (not (option "test")) (level '< "submit")) state)
 (progress "Load: verifying page")
 (:let state (c-verify ref state))
 state)



















