#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb System
=============================================
Message parsing and synthesis
=============================================

Messages have the following syntax:

bit::=		     0 | 1
byte::=		     bit*8

Bit patterns whose length is a multiple of eight are interpretted
using 'mixed endian' convention: A bit string like
  1000 0000  0000 0000
is interpretted as a sequence of two bytes, the first of which has
its most significant bit set.

middle-septet::=     1 bit*7
end-septet::=	     0 bit*7
cardinal::=	     middle-septet* end-septet

A middle- or end-septet is interpretted as a number between 0 and
127, inclusive. As an example, 0000 0101 and 1000 0101 both represent
the number 'five'. A cardinal represents a non-negative integer
expressed base 128 with the least significant septet first. As an
example, 1000 0010 0000 0001 represents 130.

vector::=	     length byte*
length::=	     cardinal

A 'vector' consists of a 'length' followed by x bytes. The value y of the
'length' must satisfy x*8-7 <= y <= x*8.

A 'vector' represents a bit vector with y bits. Bit number zero of the
bit vector is the least significant bit of the first byte. Bit number
8*x+y is the bit with weight 2^y in byte number x. As an example,
  0000 1010  1000 0000  0000 0001
represents the bit vector
  (0 0 0 0  0 0 0 1  1 0)
which is the concatenation of the mirror images of 1000 0000 and 0000 0001,
truncated to ten bits.

x00::=               0000 0000 | 1000 0000 x00
x01::=               0000 0001 | 1000 0001 x00
x02::=               0000 0010 | 1000 0010 x00
x03::=               0000 0011 | 1000 0011 x00
x04::=               0000 0100 | 1000 0100 x00
x05::=               0000 0101 | 1000 0101 x00
x06::=               0000 0110 | 1000 0110 x00
x07::=               0000 0111 | 1000 0111 x00

x00 is the syntax class of all cardinals that represent 'zero'. Similar
holds for x01 to x07.

m-id-nop::=          x00
m-id-notify::=       x01
m-id-ping::=         x02
m-id-pong::=         x03
m-id-get::=          x04
m-id-got::=          x05
m-id-put::=          x06
m-id-prefix::=       x07

The classes above identify the eight message types in the Logiweb protocol.

L::=		     11001100
o::=		     11101111
g::=		     11100111
i::=		     11101001
w::=		     11110111
e::=		     11100101
b::=		     11100010
id-version::=	     x01
m-id-logiweb::=	     L o g i w e b id-version

m-id-logiweb identifies the Logiweb protocol

timestamp::=	     mantissa exponent
mantissa::=	     cardinal
exponent::=	     cardinal
bytes::=	     byte*

A timestamp (m e) represents the time instant m*10^(-e) after 00:00:00 International Atomic Time of Modified Julian Day 0.

message::=           nop | notify | ping | pong | get | got | put | prefix
nop::=               m-id-nop
notify::=            m-id-notify event
ping::=              m-id-ping
pong::=              m-id-pong id now
get::=               m-id-get address class index
got::=               m-id-got address class index norm total time value
put::=               m-id-put address class operation value
prefix::=            m-id-prefix tag message

event::=             cardinal
id::=                cardinal
now::=               timestamp
address::=           vector
class::=             cardinal
index::=             cardinal
norm::=              cardinal
total::=             cardinal
time::=              timestamp
value::=             vector
operation::=         cardinal
tag::=               cardinal

The lines above define the messages that can occur in the Logiweb protocol.
At present, messages are sent using the udp protocol with one and only
one message in each udp packet.

If Logiweb messages are sent over a connection based channel, they should be
sent back-to-back.

A server that receives a nop, notify, pong, or got message shall not respond
to that message.

A server that receives a ping shall respond with a pong message in which
'id' is m-id-logiweb and 'now' is the current server time.

A server that receives a get message shall respond with a got message.
The address, class, and index of the got message shall be identical
to the corresponding fields in the get message.

If the state of the server has a node with the given address, then the
remaining fields of the got message shall be as follows: The 'norm'
shall be equal to the length of the address. The 'total' shall be equal
to the number of attributes of the given class stored in the addressed
node. If total=0 then the value shall be the empty bit vector and the
'time' shall be the current server time. Otherwise, if 0 < index < total
then the 'value' shall be the value of the n'th-oldest attribute of the
given class and the 'time' shall be the time at which that value entered
the state of the server. If index=0 or index>=total then the value shall
be the newest value of the given class and the 'time' shall indicate at
what time the newest value entered the state.

If the state of the server has no node with the given address, then the
the server shall find the longest prefix of the address for which the
server does have a node. Then the norm, total, timestamp, and value
shall be set as if the 'get' message had this reduced address and as if
the class of the 'get' message was 'pointer'. The value if the 'index'
shall be ignored and, if total>0, then the returned value shall be a
random among the available pointer attributes. The address, class, and
index of the 'got' message still shall be copies of the address, class,
and index of the original 'get' message.

A server that receives a 'put' message shall respond with a 'notify'
message in which 'event' is set to 'received'.

A pressed server may respond to a 'ping', 'get', or 'put' message with a
'notify' message in which 'event' is set to 'sorry'. A very pressed server
may discard incomming udp-packets without notice.

A server that receives a prefix message with tag t and message m
shall process the message m. If the message m results in no response,
the server shall not respond to the prefix message. If the message m
results in a response m', the server shall respond with a prefix
message with tag t and message m'. This holds even if the server
responds with a 'notify' message in which 'event' is set to 'sorry'.

Messages can have any number of tags. Tags are handled recursively.

The semantics of individual fields of messages is described in the
following:

event::=             cardinal

At present, the following events are defined:
0: sorry (too many requests, try again later)
1: received (respond to notify)
2: rejected (address/value out of bound)

id::=                cardinal

At present, the id of a pong message shall be m-id-logiweb.

now::=               timestamp

The current server time.

address::=           vector

The state of a server is a binary, rooted tree whose edges are labelled
by bits. The address of a node n is defined as the vector of edge labels
of the path from the root to n.

class::=             cardinal

At present, the following attribute classes are defined:
0: update
1: type
2: left
3: right
4: pnt (also called 'sibling' or 'pointer')
5: url
6: leap

operation::=         cardinal

At present, the following operations are defined:

0: remove attribute with given value
1: add attribute with given value

tag::=               cardinal

Tags have no inherent meaning. Adding an random, unguessable tag
to a request can be used as a measure against malicious software that
sends garbage to a Logiweb server or client. Adding a unique tag to
a message can be used by proxies which relay Logiweb messages past a
firewall.

Attributes of class 'type' can have two values:
0: The node is a leaf
1: The node is a cons

=============================================
Message representation
=============================================

The parse functions defined later convert 'incomming messages' to
'internal messages'. The generator functions defined later convert
'internal messages' to 'outgoing messages'.

Incomming messages are represented as lists of bytes where each byte
is a Lisp integer between 0 and 255, inclusive.

Outgoing messages are represented as ct's (cardinal trees).

Internal messages are represented as lists (id tag* &rest args).
The id is one of m-id-nop, m-id-notify, m-id-ping, m-id-pong, m-id-get,
m-id-got, and m-id-put. The tag* is the list of tags prepended to the
message. The tag* is in 'reverse order' in the sense that the first tag
in the incomming or outgoing message is the last tag in tag*. The args
are the cardinals, timestamps, and vectors that constitute the arguments
of the message. Cardinals are represented by Lisp integers. Timestamps
are represented as a list (mantissa exponent) where the mantissa and
exponent are integers. Vectors are represented as a cons (length . byte*)
where the length is an integer and byte* is a list of bytes. We must have
x*8-7 <= length <= x*8 where x is the length of byte*.
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Conversion to internal vectors
=============================================
The following functions convert a bit vector, a byte vector, a ct, and a
reference to an internal vector:

(m-bit*2vector bit*)
(m-byte*2vector byte*)
(m-ct2vector ct)
(m-ref2vector ref)

m-empty-vector is the empty vector
|#

(defc m-empty-vector '(0))

(deff m-bit*2byte* (bit*)
 (:when (null bit*) nil)
 (:let bit*1 (nthcdr 8 bit*))
 (:when (null bit*1) (list (card*2card 2 bit*)))
 (cons (card*2card 2 (subseq bit* 0 8)) (m-bit*2byte* bit*1)))

(etst (m-bit*2byte* '(1 0 0 0  0 0 0 0   0 1)) '(1 2))

(deff m-bit*2vector (bit*)
 (:let length (length bit*))
 (:let bit* (m-bit*2byte* bit*))
 (cons length bit*))

(etst (m-bit*2vector '(1 0 0 0  0 0 0 0   0 1)) '(10 1 2))

(deff m-byte*2vector (byte*)
 (:let length (length byte*))
 (:let length (* 8 length))
 (cons length byte*))

(etst (m-byte*2vector '(1 2)) '(16 1 2))

(deff m-ct2vector (ct)
 (m-byte*2vector (ct2card* ct)))

(etst (m-ct2vector '((1) 2)) '(16 1 2))

(deff m-ref2vector (ref)
 (m-byte*2vector (card2ref ref)))

#|
=============================================
Convert vector to bit vector
=============================================
|#

(deff byte2bits (byte length bits)
 (:when (= length 0) bits)
 (cons (mod byte 2) (byte2bits (floor byte 2) (- length 1) bits)))

(deff vector2bits1 (length byte*)
 (:let byte (default 0 (car byte*)))
 (:when (<debug length 8) (byte2bits byte length nil))
 (byte2bits byte 8 (vector2bits1 (- length 8) (cdr byte*))))

(deff vector2bits (debug vector)
 (:let (length . byte*) vector)
 (when (null length) (format t "vector2bits ~s~%" debug))
 (vector2bits1 length byte*))

(etst (vector2bits :debug01 (list 7 4)) '(0 0 1 0 0 0 0))
(etst (vector2bits :debug02 (list 9 4 3)) '(0 0 1 0 0 0 0 0 1))

#|
=============================================
Convert between cardinal and internal vector
=============================================
(m-card2vector card) and (m-card2vector8 card) both represent a cardinal as an internal vector. m-card2vector and m-card2vector8 does so with a granularity of 1 and 8 bits, respectively.

(m-vector2card vector) converts back from both representations.
|#

(deff m-card2vector (card)
 (m-bit*2vector (card2card* 2 card)))

(etst (m-card2vector 0) '(1 0))
(etst (m-card2vector 1) '(1 1))
(etst (m-card2vector 2) '(2 2))
(etst (m-card2vector 258) '(9 2 1))

(deff m-card2vector8 (card)
 (m-byte*2vector (card2card* 256 card)))

(etst (m-card2vector8 0) '(8 0))
(etst (m-card2vector8 1) '(8 1))
(etst (m-card2vector8 2) '(8 2))
(etst (m-card2vector8 258) '(16 2 1))

(deff m-vector2card (debug vector)
 (when (null (head vector)) (format t "m-vector2card ~s~%" debug))
 (card*2card 2 (vector2bits :debug03 vector)))

(etst (m-vector2card :debug01 '(1 0)) 0)
(etst (m-vector2card :debug02 '(1 1)) 1)
(etst (m-vector2card :debug03 '(2 2)) 2)
(etst (m-vector2card :debug04 '(9 2 1)) 258)
(etst (m-vector2card :debug05 '(8 0)) 0)
(etst (m-vector2card :debug06 '(8 1)) 1)
(etst (m-vector2card :debug07 '(8 2)) 2)
(etst (m-vector2card :debug08 '(16 2 1)) 258)

#|
=============================================
Convert between leap and internal vector
=============================================
A 'leap' is a pair (delta . mjd) where mjd is the modified julian day number of a particular day and delta is a number of seconds (positive or negative) by which the given day is prolonged.

(leap2vector leap) represents a leap as a vector.

(vector2leap vector) converts a vector back to a leap.
|#

(deff card2septet* (card rest)
 (:when (<debug card 128) (cons card rest))
 (cons (+ 128 (mod card 128)) (card2septet* (floor card 128) rest)))

(etst (card2septet*     0 nil) '(0))
(etst (card2septet*   127 nil) '(127))
(etst (card2septet*   128 nil) '(128 1))
(etst (card2septet*   130 nil) '(130 1))
(etst (card2septet* 16386 nil) '(130 128 1))

(deff leap2vector (leap)
 (:let (delta . mjd) leap)
 (m-byte*2vector (card2septet* (int2card delta) (card2card* 256 mjd))))

(etst (leap2vector '(+1 . 0)) '(16 1 0))
(etst (leap2vector '(-1 . 0)) '(16 2 0))
(etst (leap2vector '(-8193 . 65538)) '(48 130 128 1 2 0 1))

(deff vector2leap (vector)
 (:let card* (m-bit*2byte* (vector2bits :debug04 vector)))
 (vector2leap1 card* nil))

(deff vector2leap1 (card* septet*)
 (:when (null card*) (vector2leap2 septet* card*))
 (:let (card . card*) card*)
 (:when (<debug card 128) (vector2leap2 (cons card septet*) card*))
 (vector2leap1 card* (cons (- card 128) septet*)))

(deff vector2leap2 (septet* card*)
 (:let delta (card2int (card*2card 128 (reverse septet*))))
 (:let mjd (card*2card 256 card*))
 (cons delta mjd))

(etst (vector2leap '(16 1 0)) '(+1 . 0))
(etst (vector2leap '(9 2 0)) '(-1 . 0))
(etst (vector2leap '(48 130 128 1 2 0 1)) '(-8193 . 65538))
(etst (vector2leap '(8 1)) '(+1 . 0))
(etst (vector2leap '(8 2)) '(-1 . 0))

#|
=============================================
Identifiers
=============================================
|#

; Message identifiers

(defc m-id-nop 0)
(defc m-id-notify 1)
(defc m-id-ping 2)
(defc m-id-pong 3)
(defc m-id-get 4)
(defc m-id-got 5)
(defc m-id-put 6)
(defc m-id-prefix 7)

; Logiweb identifier

(defc m-id-logiweb (card*2card 128 (append (string2card* "Logiweb") (list 1))))

; Event identifiers

(defc m-id-sorry 0)
(defc m-id-received 1)
(defc m-id-rejected 2)

; Class identifiers

(defc m-id-update 0)
(defc m-id-type   1)
(defc m-id-left   2)
(defc m-id-right  3)
(defc m-id-pnt    4)
(defc m-id-url    5)
(defc m-id-leap   6)

; Operation identifiers

(defc m-id-del 0)
(defc m-id-add 1)

; Type identifiers

(defc m-id-leaf 0)
(defc m-id-cons 1)

#|
=============================================
Generate cardinal
=============================================
(m-gen-card card) converts the given cardinal to a list of middle septets followed by an end septet.
|#

(deff m-gen-card (card)
 (:when (<debug card 128) card)
 (:mlet (card byte) (floor card 128))
 (cons (+ byte 128) (m-gen-card card)))

(etst (m-gen-card 2) 2)
(etst (m-gen-card 130) '(130 . 1))

#|
=============================================
Generate Logiweb identifier
=============================================
|#

(defc m-gen-logiweb (m-gen-card m-id-logiweb))

#|
=============================================
Generate timestamp
=============================================
(m-gen-time timestamp) converts the given timestamp to a ct.
|#

(deff m-gen-time (timestamp)
 (:let (mantissa exponent) timestamp)
 (cons (m-gen-card mantissa) (m-gen-card exponent)))

(etst (m-gen-time '(130 6)) '((130 . 1) . 6))

#|
=============================================
Generate vector
=============================================
(m-gen-vector vector) converts the internal vector (length . byte*) structure
to an outgoing vector. In practice this merely involves translation of the
length since byte* is already a ct.
|#

(deff m-gen-vector (vector)
 (:let (length . byte*) vector)
 (cons (m-gen-card length) byte*))

#|
=============================================
Generate messages
=============================================
The following constructs generate the main message types. Each message is prefixed by the given list of tags.

(m-nop    tag*)
(m-notify tag* event)
(m-ping   tag*)
(m-pong   tag* timestamp)
(m-get    tag* address class index)
(m-got    tag* address class index norm total timestamp value)
(m-put    tag* address class operation value)
(m-prefix tag* message)

Prefix messages are generated by (m-prefix tag* message) which adds the given tags to the given message (the first element of tag* is added first so that
the tags end up in reverse order).

General messages are generated by (m-msg tag* &rest message)
|#

(deff m-msg (tag* &rest message)
 (m-prefix tag* message))

(deff m-prefix (tag* message)
 (:when (atom tag*) message)
 (:let (tag . tag*) tag*)
 (:let tag (card2septet* tag nil))
 (m-msg tag* m-id-prefix tag message))

(deff m-nop (tag*)
 (m-msg tag* m-id-nop))

(deff m-notify (tag* event)
 (m-msg tag* m-id-notify (m-gen-card event)))

(deff m-ping (tag*)
 (m-msg tag* m-id-ping))

(deff m-pong (tag* timestamp)
 (m-msg tag* m-id-pong m-gen-logiweb (m-gen-time timestamp)))

(deff m-get (tag* address class index)
 (:let address (m-gen-vector address))
 (:let class (m-gen-card class))
 (:let index (m-gen-card index))
 (m-msg tag* m-id-get address class index))

(deff m-got
 (tag* address class index norm total time value)
 (:let address (m-gen-vector address))
 (:let class (m-gen-card class))
 (:let index (m-gen-card index))
 (:let norm (m-gen-card norm))
 (:let total (m-gen-card total))
 (:let time (m-gen-time time))
 (:let value (m-gen-vector value))
 (m-msg tag* m-id-got address class index norm total time value))

(deff m-put (tag* address class operation value)
 (:let address (m-gen-vector address))
 (:let class (m-gen-card class))
 (:let operation (m-gen-card operation))
 (:let value (m-gen-vector value))
 (m-msg tag* m-id-put address class operation value))

#|
=============================================
Generate messages instances
=============================================
The following constructs generate various instances of the main message types.
|#

(deff m-sorry (tag*)
 (m-notify tag* m-id-sorry))

(deff m-received (tag*)
 (m-notify tag* m-id-received))

(deff m-rejected (tag*)
 (m-notify tag* m-id-rejected))

(deff m-get-update (tag* address index)
 (m-get tag* address m-id-update index))

(deff m-get-type (tag* address)
 (m-get tag* address m-id-type 0))

(deff m-get-pnt (tag* address index)
 (m-get tag* address m-id-pnt index))

(deff m-get-url (tag* address index)
 (m-get tag* address m-id-url index))

(deff m-get-leap (tag* address index)
 (m-get tag* address m-id-leap index))

(deff m-add-pnt (tag* address value)
 (m-put tag* address m-id-pnt m-id-add value))

(deff m-add-url (tag* address value)
 (m-put tag* address m-id-url m-id-add value))

(deff m-add-leap (tag* address value)
 (m-put tag* address m-id-leap m-id-add value))

(deff m-del-pnt (tag* address value)
 (m-put tag* address m-id-pnt m-id-del value))

(deff m-del-url (tag* address value)
 (m-put tag* address m-id-url m-id-del value))

(deff m-del-leap (tag* address value)
 (m-put tag* address m-id-leap m-id-del value))

#|
=============================================
Convert message from internal to outgoing
=============================================
|#

(deff m-id2fct (id)
 (case id
  (:nop    'm-nop   )
  (:notify 'm-notify)
  (:ping   'm-ping  )
  (:pong   'm-pong  )
  (:get    'm-get   )
  (:got    'm-got   )
  (:put    'm-put   )
  (t       (error "Unknown id: ~s" id))))

(deff m-unparse (msg)
 (:let (id . rest) msg)
 (apply (m-id2fct id) rest))

#|
=============================================
Parse cardinal
=============================================
(m-parse-card card*) parses one cardinal from the given suffix and returns
(card . suffix') where suffix' is the suffix after parsing the cardinal.

Parse functions raise an exception if they read past the end of card*.
|#

(deff m-parse-card (suffix)
 (m-parse-card1 suffix nil))

(deff m-parse-card1 (suffix result)
 (:when (null suffix) (raise))
 (:let (card . suffix) suffix)
 (:when (<debug card 128) (cons (m-parse-card2 result card) suffix))
 (:let result (cons card result))
 (m-parse-card1 suffix result))

(deff m-parse-card2 (card* result)
 (:when (null card*) result)
 (:let (card . card*) card*)
 (:let result (+ card -128 (* 128 result)))
 (m-parse-card2 card* result))

#|
=============================================
Parse timestamp
=============================================
(m-parse-timestamp suffix) parses one timestamp and returns
((mantissa exponent) . suffix)
|#

(deff m-parse-timestamp (suffix)
 (:let (mantissa . suffix) (m-parse-card suffix))
 (:let (exponent . suffix) (m-parse-card suffix))
 (:let timestamp (list mantissa exponent))
 (cons timestamp suffix))

#|
=============================================
Parse vector
=============================================
(m-parse-vector suffix) parses one vector and returns
((length . bytes) . suffix)
|#

(deff m-parse-vector (suffix)
 (:let (length . suffix) (m-parse-card suffix))
 (:let length1 (ceiling length 8))
 (:let (bytes . suffix) (m-parse-bytes length1 suffix nil))
 (:let vector (cons length bytes))
 (cons vector suffix))

(deff m-parse-bytes (n suffix result)
 (:when (<= n 0) (cons (reverse result) suffix))
 (:when (null suffix) (raise))
 (:let (card . suffix) suffix)
 (:let result (cons card result))
 (m-parse-bytes (- n 1) suffix result))

#|
=============================================
Parse arguments of messages
=============================================
|#

(deff m-parse-nop (tag*)
 (list :nop tag*))

(deff m-parse-notify (tag* suffix)
 (:let (event   . :suffix) (m-parse-card      suffix))
 (list :notify tag* event))

(deff m-parse-ping (tag*)
 (list :ping tag*))

(deff m-parse-pong (tag* suffix)
 (:let (id      .  suffix) (m-parse-card      suffix))
 (:let (now     . :suffix) (m-parse-timestamp suffix))
 (:when (unequal id m-id-logiweb) (raise))
 (list :pong tag* id now))

(deff m-parse-get (tag* suffix)
 (:let (address .  suffix) (m-parse-vector    suffix))
 (:let (class   .  suffix) (m-parse-card      suffix))
 (:let (index   . :suffix) (m-parse-card      suffix))
 (list :get tag* address class index))

(deff m-parse-got (tag* suffix)
 (:let (address .  suffix) (m-parse-vector    suffix))
 (:let (class   .  suffix) (m-parse-card      suffix))
 (:let (index   .  suffix) (m-parse-card      suffix))
 (:let (norm    .  suffix) (m-parse-card      suffix))
 (:let (total   .  suffix) (m-parse-card      suffix))
 (:let (time    .  suffix) (m-parse-timestamp suffix))
 (:let (value   . :suffix) (m-parse-vector    suffix))
 (list :got tag* address class index norm total time value))

(deff m-parse-put (tag* suffix)
 (:let (address .  suffix) (m-parse-vector    suffix))
 (:let (class   .  suffix) (m-parse-card      suffix))
 (:let (op      .  suffix) (m-parse-card      suffix))
 (:let (value   . :suffix) (m-parse-vector    suffix))
 (list :put tag* address class op value))

(deff m-parse-prefix (tag* suffix)
 (:let (tag     .  suffix) (m-parse-card      suffix))
 (m-parse-message1 (cons tag tag*) suffix))

#|
=============================================
Parse message
=============================================
|#

(deff m-parse-message (suffix)
 (:catch () nil)
 (m-parse-message1 nil suffix))

(deff m-parse-message1 (tag* suffix)
 (:let (id . suffix) (m-parse-card suffix))
 (:when (= id m-id-nop)    (m-parse-nop    tag*))
 (:when (= id m-id-notify) (m-parse-notify tag* suffix))
 (:when (= id m-id-ping)   (m-parse-ping   tag*))
 (:when (= id m-id-pong)   (m-parse-pong   tag* suffix))
 (:when (= id m-id-get)    (m-parse-get    tag* suffix))
 (:when (= id m-id-got)    (m-parse-got    tag* suffix))
 (:when (= id m-id-put)    (m-parse-put    tag* suffix))
 (:when (= id m-id-prefix) (m-parse-prefix tag* suffix))
 (raise))


















