/*  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
*/

/**********
 * Config *
 **********/

/* Max percentage of available ram that can be allocated for the heap */
#ifndef HEAPMAX0
#define HEAPMAX0 90
#endif

/* Percentage of max heap required to be free after GC. Quit if not met. */
#ifndef HEAPMAX1
#define HEAPMAX1 90
#endif

/* Percentage of heap required to be free after GC. Allocate if not met. */
#ifndef HEAPMAX2
#define HEAPMAX2 70
#endif

/* Percentage of heap to be free after allocation if permitted by HEAPMAX0 */
#ifndef HEAPMAX3
#define HEAPMAX3 50
#endif

/* Size of memory area reserved for stack */
#ifndef STACKSIZE
#define STACKSIZE 50000000
#endif

/* Area reserved for defined constants */
#ifndef STORESIZE
#define STORESIZE 10000
#endif

/* When TRUE: do Garbage Collection (GC) for *each* allocation (takes time!) */
#ifndef ALWAYSGC
#define ALWAYSGC FALSE
#endif

/* When TRUE: Print Garbage Collection (GC) statistics for each GC */
#ifndef PRINTGC
#define PRINTGC FALSE
#endif



/************
 * INCLUDES *
 ************/

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdint.h>
#include <unistd.h>
#include <sys/wait.h>
#include <dlfcn.h>
#include <signal.h>
#include <sys/mman.h>
#ifndef __CYGWIN__
#include <sys/sysinfo.h>
#else
#include <process.h>
#endif /* __CYGWIN__ */



/******************
 * INCLUDED PAGES *
 ******************/

#include "pages.c"



/***********
 * DEFINES *
 ***********/

/*
cons(TAG_FALSE   ,NIL  ,NIL  ) The constant F
cons(TAG_INT     ,val  ,int  ) Val is the least significant word of the integer
cons(TAG_VECT    ,bytes,vect ) Bytes are the least significant word of the vect
cons(TAG_PAIR    ,head ,tail ) Tagged pair
cons(TAG_EX      ,arg  ,NIL  ) Exception
cons(TAG_MAP     ,map  ,info ) Tagged map. the 'info' is for the optimizer
cons(TAG_OBJ     ,tag  ,val  ) Object
cons(TAG_GARB    ,NIL  ,NIL  ) Garbage
cons(TTAG_APPLY  ,fct  ,arg  ) Functional applicationat term level
cons(TTAG_LAMBDA ,term ,NIL  ) Lambda abstraction at term level
cons(TTAG_VAR    ,index,NIL  ) The variable with the given deBruijn index
cons(TTAG_PAIR   ,head ,tail ) Raw pairing at term level
cons(TTAG_CONST  ,rnf  ,NIL  ) A term whose value is the given rnf
cons(TTAG_CALL   ,term ,term*) Evaluate term in given term* environment
cons(TTAG_ECALL  ,eterm,term*) Evaluate eager term in given term* environment
cons(MTAG_CLOSURE,term ,env  ) Value of term in given closure* environment
cons(MTAG_INDIR  ,NIL  ,rnf  ) Indirection to the given rnf
cons(MTAG_PAIR   ,head ,tail ) Raw pairing at map level
cons(MTAG_LAMBDA ,term ,env  ) \x.term in given closure* environment
cons(MTAG_FIX    ,NIL  ,NIL  ) Fixed point operator

Eager term tags:

ETAG_var
ETAG_ecall
ETAG_digit0    " %0
ETAG_digit1    " %1
ETAG_digit2    " %2
ETAG_digit3    " %3
ETAG_digit4    " %4
ETAG_digit5    " %5
ETAG_digit6    " %6
ETAG_digit7    " %7
ETAG_digit8    " %8
ETAG_digit9    " %9
ETAG_times     " * "
ETAG_plus      " + "
ETAG_minus     " - "
ETAG_and       " .and. "
ETAG_or        " .or. "
ETAG_then      " .then. "
ETAG_pair      " :: "
ETAG_lt        " < "
ETAG_le        " <= "
ETAG_equal     " = "
ETAG_gt        " > "
ETAG_ge        " >= "
ETAG_apply     " apply "
ETAG_boolp     " boolp
ETAG_catch     " catch
ETAG_maptag1   " catching maptag
ETAG_div       " div "
ETAG_head      " head
ETAG_intp      " intp
ETAG_boolg     " is bool : "
ETAG_intg      " is int : "
ETAG_mapg      " is map : "
ETAG_objg      " is object : "
ETAG_pairg     " is pair : "
ETAG_mapp      " mapp
ETAG_maptag    " maptag
ETAG_mod       " mod "
ETAG_norm      " norm
ETAG_objp      " objectp
ETAG_pairp     " pairp
ETAG_raise     " raise
ETAG_root      " root
ETAG_tail      " tail
ETAG_untag     " untag
ETAG_digitend  %%
ETAG_par       ( " )
ETAG_uplus     + "
ETAG_dplus     +"
ETAG_uminus    - "
ETAG_dminus    -"
ETAG_not       .not. "
ETAG_Base      Base
ETAG_LET       LET " BE "
ETAG_ash       ash ( " , " )
ETAG_bottom    bottom
ETAG_bt2bytes  bt2byte* ( " )
ETAG_bt2vects  bt2vector* ( " )
ETAG_bt2vector bt2vector ( " )
ETAG_ceiling   ceiling ( " , " )
ETAG_destruct  destruct ( " )
ETAG_evenp     evenp ( " )
ETAG_exception exception
ETAG_false     false
ETAG_floor     floor ( " , " )
ETAG_half      half ( " )
ETAG_if        if " then " else "
ETAG_intlength integer-length ( " )
ETAG_logand    logand ( " , " )
ETAG_logandc1  logandc1 ( " , " )
ETAG_logandc2  logandc2 ( " , " )
ETAG_logbitp   logbitp ( " , " )
ETAG_logcount  logcount ( " )
ETAG_logeqv    logeqv ( " , " )
ETAG_logior    logior ( " , " )
ETAG_lognand   lognand ( " , " )
ETAG_lognor    lognor ( " , " )
ETAG_lognot    lognot ( " )
ETAG_logorc1   logorc1 ( " , " )
ETAG_logorc2   logorc2 ( " , " )
ETAG_logtest   logtest ( " , " )
ETAG_logxor    logxor ( " , " )
ETAG_map       map ( " )
ETAG_prenorm   norm "
ETAG_notnot    notnot "
ETAG_object    object ( " )
ETAG_print     print ( " )
ETAG_round     round ( " , " )
ETAG_spy       spy ( " )
ETAG_timer     timer ( " )
ETAG_trace     trace ( " )
ETAG_truncate  truncate ( " , " )
ETAG_vector    vector ( " )
ETAG_vnorm     vector-norm ( " )
ETAG_vempty    vector-empty ( " )
ETAG_vindex    vector-index ( " , " )
ETAG_vlength   vector-length ( " )
ETAG_vprefix   vector-prefix ( " , " )
ETAG_vsubseq   vector-subseq ( " , " , " )
ETAG_vsuffix   vector-suffix ( " , " )
ETAG_v2bytes   vector2byte* ( " )
ETAG_v2vects   vector2vector* ( " )
ETAG_vt2bytes  vt2byte* ( " )
ETAG_vt2vects  vt2vector* ( " )
ETAG_vt2v      vt2vector ( " )
ETAG_yy        YY
ETAG_compile   compile ( " )
ETAG_ripemd    ripemd ( " )
ETAG_sl2rack   sl2rack ( " )
ETAG_rack2sl   rack2sl ( " )
*/

/* Accessors */
#define ROOT(x) (((cell*)(x))->root)
#define HEAD(x) (((cell*)(x))->head)
#define TAIL(x) (((cell*)(x))->tail)

/* Number of tags */
#define TAG_SIZE       256

/* Data tags */
#define TAG_FALSE        1
#define TAG_INT          2
#define TAG_VECT         3
#define TAG_PAIR         4
#define TAG_EX           5
#define TAG_MAP          6
#define TAG_OBJ          7
#define TAG_GARB         8

/* Term tags */
#define TTAG_APPLY      20
#define TTAG_LAMBDA     21
#define TTAG_VAR        22
#define TTAG_PAIR       23
#define TTAG_CONST      24
#define TTAG_CALL       25
#define TTAG_ECALL      26

/* Map tags */
#define MTAG_CLOSURE    30
#define MTAG_INDIR      31
#define MTAG_PAIR       32
#define MTAG_LAMBDA     33
#define MTAG_FIX        34

/* Class tags */
#define CTAG_MAP        40
#define CTAG_EMAP       41
#define CTAG_TRUE       42
#define CTAG_APPLY      43
#define CTAG_IF         44

/* Optimized function tags 
   (ETAG for "eager tag" since most optimized functions are eager) */
#define ETAG_var       100
#define ETAG_ecall     101
#define ETAG_digit0    102
#define ETAG_digit1    103
#define ETAG_digit2    104
#define ETAG_digit3    105
#define ETAG_digit4    106
#define ETAG_digit5    107
#define ETAG_digit6    108
#define ETAG_digit7    109
#define ETAG_digit8    110
#define ETAG_digit9    111
#define ETAG_times     112
#define ETAG_plus      113
#define ETAG_minus     114
#define ETAG_and       115
#define ETAG_or        116
#define ETAG_then      117
#define ETAG_pair      118
#define ETAG_lt        119
#define ETAG_le        120
#define ETAG_equal     121
#define ETAG_gt        122
#define ETAG_ge        123
#define ETAG_apply     124
#define ETAG_boolp     125
#define ETAG_catch     126
#define ETAG_maptag1   127
#define ETAG_div       128
#define ETAG_head      129
#define ETAG_intp      130
#define ETAG_boolg     131
#define ETAG_intg      132
#define ETAG_mapg      133
#define ETAG_objg      134
#define ETAG_pairg     135
/*      ETAG_valg      136*/
#define ETAG_mapp      137
#define ETAG_maptag    138
#define ETAG_mod       139
#define ETAG_norm      140
#define ETAG_objp      141
#define ETAG_pairp     142
#define ETAG_raise     143
#define ETAG_root      144
#define ETAG_tail      145
#define ETAG_untag     146
#define ETAG_digitend  147
#define ETAG_par       148
#define ETAG_uplus     149
#define ETAG_dplus     150
#define ETAG_uminus    151
#define ETAG_dminus    152
#define ETAG_not       153
#define ETAG_Base      154
#define ETAG_LET       155
#define ETAG_ash       156
#define ETAG_bottom    157
#define ETAG_bt2bytes  158
#define ETAG_bt2vects  205
#define ETAG_bt2vector 159
#define ETAG_ceiling   160
#define ETAG_destruct  161
#define ETAG_evenp     162
#define ETAG_exception 163
#define ETAG_false     164
#define ETAG_floor     165
#define ETAG_half      166
#define ETAG_if        167
#define ETAG_intlength 168
#define ETAG_logand    169
#define ETAG_logandc1  170
#define ETAG_logandc2  171
#define ETAG_logbitp   172
#define ETAG_logcount  173
#define ETAG_logeqv    174
#define ETAG_logior    175
#define ETAG_lognand   176
#define ETAG_lognor    177
#define ETAG_lognot    178
#define ETAG_logorc1   179
#define ETAG_logorc2   180
#define ETAG_logtest   181
#define ETAG_logxor    182
#define ETAG_map       183
#define ETAG_prenorm   184
#define ETAG_notnot    185
#define ETAG_object    186
#define ETAG_print     187
#define ETAG_round     188
#define ETAG_spy       189
#define ETAG_timer     190
#define ETAG_trace     191
#define ETAG_truncate  192
#define ETAG_vector    193
#define ETAG_vnorm     208
#define ETAG_vempty    194
#define ETAG_vindex    195
#define ETAG_vlength   196
#define ETAG_vprefix   197
#define ETAG_vsubseq   198
#define ETAG_vsuffix   199
#define ETAG_v2bytes   200
#define ETAG_v2vects   206
#define ETAG_vt2bytes  201
#define ETAG_vt2vects  207
#define ETAG_vt2v      202
#define ETAG_yy        203
#define ETAG_compile   204
#define ETAG_ripemd    209
#define ETAG_sl2rack   210
#define ETAG_rack2sl   211

/* True if x is *not* of given type */

#define GD_INT(x)  ((ROOT(x)|1)!=TAG_VECT)
#define GD_PAIR(x) (ROOT(x)!=TAG_PAIR)
#define GD_EX(x)   (ROOT(x)!=TAG_EX)
#define GD_MAP(x)  (ROOT(x)!=TAG_MAP)
#define GD_OBJ(x)  (ROOT(x)!=TAG_OBJ)

/* True if x is *not* of given type */

#define NO_BOOL(x) (((x)!=T)&&(ROOT(x)!=TAG_FALSE))
#define NO_INT(x)  (((x)==T)||GD_INT(x))
#define NO_PAIR(x) (((x)==T)||GD_PAIR(x))
#define NO_EX(x)   (((x)==T)||GD_EX(x))
#define NO_MAP(x)  (((x)==T)||GD_MAP(x))
#define NO_OBJ(x)  (((x)==T)||GD_OBJ(x))

/* True if x is of given type */

#define IS_BOOL(x) (!NO_BOOL(x))
#define IS_INT(x)  (!NO_INT (x))
#define IS_PAIR(x) (!NO_PAIR(x))
#define IS_EX(x)   (!NO_EX  (x))
#define IS_MAP(x)  (!NO_MAP (x))
#define IS_OBJ(x)  (!NO_OBJ (x))

/* NIL pointer, map truth, C truth and C falsehood */
#define NIL       0
#define T         0
#define TRUE      1
#define FALSE     0
#define ENDFILE 256

/* Conversion of negative x to -1 and other x to 0 */
#define SIGN(x) ((x)&signmask?M:Z)

/* Operations on halfwords */
#define HALFHEAD(x) ((x)&halfmask)
#define HALFTAIL(x) ((x)>>halfsize)
#define HALFCONS(x,y) (((x)&halfmask)+((y)<<halfsize))

/* Test constructs */
#define test(x) {testfct(#x);if(!(x)){printf("\nTest failed\n");exit(1);}}
#define ttst(x) test(T==(x))
#define ftst(x) test(F==(x))
#define xtst(x) test(X==(x))
#define ntst(x) test(NIL==(x))
#define ptst(x) test(NIL!=(x))
#define etst(x,y) test(T==EQ(x,y))
#define utst(x,y) test(F==EQ(x,y))
#define prn(x) {printf("%s\n",#x);spy0(x);}
#define N(x) str2int(#x)
#define V(x) str2vec(#x)
#define W(e,x) str2nonvec(e,#x)
#define SYM(x) keep(pair(Z,pair(V(x),T)))
#define TREE(x) keep(pair(pair(Z,pair(V(x),T)),T))

/* String operations */
#define STREQ(x,y) (!strcmp((char *)(x),(char *)(y)))

/* Top of stack */
#define TOP stack[sp]

/* Constant definitions */
#define SETQ(x,y) x=(testfct(#x),keep(y))
#define DEFC(x,y) pnt x=(testfct(#x),keep(y))

/* Disable and enable ALWAYSGC */
#define TRUSTGC always_gc=FALSE
#define UNTRUSTGC always_gc=ALWAYSGC

/* Safe constructor */
#define SCONS(x,y,z) (push(y),scons(x,z))
#define MAP_PAIR(x,y) SCONS(MTAG_PAIR,x,y)

/* Script processing */
#define PATHSIZE _POSIX_PATH_MAX
#define LF 10
#define CR 13

/* Rack reading */
#define RACK_INT 0
#define RACK_NIL 1
#define RACK_ARRAY 2
#define RACK_START 3



/*********
 * TYPES *
 *********/

/* The types must be such that x = (cell*)(pnt)x for all cell* x */

typedef uintptr_t pnt;
typedef intptr_t spnt;
typedef struct cell {pnt root,head,tail;} cell;
typedef int (*Trigger)(int);



/********************
 * GLOBAL VARIABLES *
 ********************/

pnt stack[STACKSIZE];       /* The stack */
pnt store[STORESIZE];       /* List of constants */
pnt store_type[STORESIZE];  /* Types of constants */
unsigned char
  *store_name[STORESIZE];   /* Names of constants */
pnt sp=STACKSIZE;           /* Stack pointer */
pnt storepnt=STORESIZE;     /* Store pointer */
pnt garb;                   /* list of garbage collected cells */
pnt heaplist;               /* list individually allocated heap chunks */
pnt heapsize;               /* total heap size */
pnt heapmax;                /* max allowable heap size */
pnt collected;              /* number of cells collected by last gc */
pnt always_gc=ALWAYSGC;     /* When 1 do GC on every allocation */
pnt global_x;               /* Head of cons in progress */
pnt global_y;               /* Tail of cons in progress */
pnt F;                      /* the constant F (false) */
pnt X;                      /* the constant X (exception) */
pnt Z;                      /* the constant 0 */
pnt M;                      /* the constant -1 */
pnt E;                      /* the empty vector */
pnt I[256];                 /* the constants 0..255 */
pnt bytevect[256];          /* byte vectors of length 1 */
pnt pntsize;                /* the number of bits in a pnt */
pnt signmask;               /* mask with most significant bit set */
pnt markmask;               /* mask with secondmost significant bit set */
pnt halfsize;               /* the half of pntsize */
pnt halfmask;               /* mask with lower half set */
pnt septetmask;             /* mask with upper seven bits set */
pnt bytesize;               /* 1/8 of pntsize */
pnt term_t;                 /* T as a term, translation of proclaimed 'true' */
pnt term_f;                 /* F as a term */
pnt var_0;                  /* the variable with deBruijn index 0 */
pnt var_1;                  /* the variable with deBruijn index 1 */
pnt var_2;                  /* the variable with deBruijn index 2 */
pnt varlist_1;              /* list containing var_0 */
pnt varlist_2;              /* list containing var_1 and var_0 */
pnt varlist_3;              /* list containing var_2, var_1, and var_0 */
pnt term_fix;               /* term used for defining fixed point operator */
pnt map_fix;                /* fixed point operator */
pnt V0;                     /* Eager variable with deBruijn index 0 */
pnt V1;                     /* Eager variable with deBruijn index 1 */
pnt V2;                     /* Eager variable with deBruijn index 2 */
pnt V3;                     /* Eager variable with deBruijn index 3 */
pnt term_delay;             /* V(0)'V(1) */
pnt map_t;                  /* T as a map */
pnt map_f;                  /* F as a map */
pnt maptag_int;             /* tag of int as general map */
pnt maptag_pair;            /* tag of pair as general map */
pnt maptag_ex;              /* tag of exception as general map */
pnt maptag_map;             /* tag of tagged map as general map */
pnt fct_lambda;             /* translation of proclaimed 'lambda' */
pnt fct_apply;              /* translation of proclaimed 'apply' */
pnt fct_if;                 /* translation of proclaimed 'if' */
pnt fct_true;               /* translation of proclaimed 'true' */
pnt fct_quote;              /* translation of proclaimed 'quote' */
pnt const_bib;              /* "bibliography" as a vector */
pnt const_dict;             /* "dictionary" as a vector */
pnt const_body;             /* "body" as a vector */
pnt const_vector;           /* "vector" as a vector */
pnt const_expansion;        /* "expansion" as a vector */
pnt const_diagnose;         /* "diagnose" as a vector */
pnt const_cache;            /* "cluster" as a vector */
pnt const_code;             /* "code" as a vector */
pnt const_codex;            /* "codex" as a vector */
pnt const_definition;       /* "definition" as a vector */
pnt const_name;             /* "name" as a vector */
pnt const_value;            /* "value" as a vector */
pnt const_unpack;           /* "unpack" as a vector */
pnt const_macro;            /* "macro" as a vector */
pnt const_claim;            /* "claim" as a vector */
pnt const_render;           /* "render" as a vector */
pnt const_execute;          /* "execute" as a vector */
pnt const_executables;      /* "executable" as a vector */
pnt const_lambda;           /* "lambda" as a vector */
pnt const_apply;            /* "apply" as a vector */
pnt const_if;               /* "if" as a vector */
pnt const_true;             /* "true" as a vector */
pnt const_quote;            /* "quote" as a vector */
pnt const_spy;              /* "spy" as a vector */
pnt const_trace;            /* "trace" as a vector */
pnt const_print;            /* "print" as a vector */
pnt const_timer;            /* "timer" as a vector */
pnt const_quit;             /* "quit" as a vector */
pnt const_write;            /* "write" as a vector */
pnt const_exec;             /* "exec" as a vector */
pnt const_read;             /* "read" as a vector */
pnt const_extend;           /* "extend" as a vector */
pnt sym_exec;               /* "exec" as a symbol */
pnt sym_extend;             /* "extend" as a symbol */
pnt sym_boot;               /* "boot" as a symbol */
pnt sym_read;               /* "read" as a symbol */
pnt tree_introduce;         /* "introduce" as a tree */
pnt tree_define;            /* "define" as a tree */
pnt tree_exit;              /* "exit" as a tree */
pnt tree_time;              /* "time" as a tree */
pnt tree_memory;            /* "memory" as a tree */
pnt code_cclaim;            /* compile_claim as code */
pnt state;                  /* Repository for loaded pages */
pnt test;                   /* Suppresses testsuite when false */
pnt spyvar=T;               /* Storage for 'spy' variable */
pnt spycnt=0;               /* Count which limits output from spy1 */
pnt record=TRUE;            /* TRUE until patterns have been recorded */
char *tag2string[TAG_SIZE]; /* tag name */
pnt tag2eager[TAG_SIZE];    /* T for introduced, eager constructs */
pnt tag2name[TAG_SIZE];     /* Logiweb name */
pnt tag2rhs[TAG_SIZE];      /* right hand side of template definition */
pnt tag2arg[TAG_SIZE];      /* argument list of template definition */
char lgw_path1[PATHSIZE];   /* value of LGW_PATH environment variable */
char *lgw_path=0;           /* zero or pointer to lgw_path1 */



/******************************************
 * Global variables for machine execution *
 ******************************************/

pnt inputs=T;
pnt outputs=T;
pnt event=T;
pnt extensions=T;
int return_value=0;



/************************
 * FORWARD DECLARATIONS *
 ************************/

void spy1(pnt x);
pnt nameget(pnt caches,pnt ref,pnt idx);
pnt term2rnf(pnt term,pnt env);
pnt eval(pnt term,pnt env);
pnt untag1(pnt x);
void testsuite();
pnt compile(pnt x);
pnt ripemd(pnt source);
pnt sl2rack(pnt x);
pnt rack2sl(pnt x);
void deathReport();

/*******************
 * ERROR FUNCTIONS *
 *******************/

void die(char *msg){
  printf("lgwam: %s\n",msg);
  deathReport();
  exit(1);}

void xdie(char *msg){
  printf("%s\n",msg);
  deathReport();
  exit(1);}

pnt pdie(char *msg){
  perror(msg);
  deathReport();
  exit(1);}



/*********
 * STACK *
 *********/

void push(pnt x){
  if(sp==0) die("Stack overflow");
  stack[--sp]=x;}

pnt save(pnt x){
  push(x);
  return x;}

pnt pop(){
  return stack[sp++];}

pnt ret(pnt x,pnt y){
  sp=sp+x;
  return y;}



/*********
 * STORE *
 *********/

/* The store is a stack for constants to be protected from gc */

pnt keep3(pnt type,unsigned char *name,pnt value){
  if(storepnt==0) die("Store overflow");
  storepnt--;
  store_type[storepnt]=type;
  store_name[storepnt]=name;
  store[storepnt]=value;
  return value;}

pnt keep(pnt value){
  return keep3((pnt)0,(unsigned char *)"",value);}

pnt store_find(pnt type,unsigned char *name){
  pnt i;
  for(i=storepnt;i<STORESIZE;i++)
  if((store_type[i]==type)&&(STREQ(store_name[i],name))) return store[i];
  return T;}



/************************
 * TEST SUITE FUNCTIONS *
 ************************/

void printesc(pnt x){
  printf("\\%d%d",(int)(x/16),(int)(x%16));}

void printchar(pnt x){
  x=x&0xFF;
  if(x=='"') printf("\\\""); else
  if(x=='\\') printf("\\\\"); else
  if(x==10) printf("\n"); else
  if(x<32) printesc(x); else
  if(x<127) printf("%c",(int)x); else
  printesc(x);}

void prn1(char *str,pnt x){
  printf("%s",str);
  printf("(");
  spy1(x);
  printf(")");}

void prn2(char *str,pnt x,pnt y){
  printf("%s",str);
  printf("(");
  spy1(x);
  printf(",");
  spy1(y);
  printf(")");}

pnt invalid_tag(pnt tag){
  if(tag>=TAG_SIZE) return TRUE;
  if(tag2string[tag]==NIL) return TRUE;
  return FALSE;}

char *safe_tag2string(pnt tag){
  if(invalid_tag(tag)) return "TAG_UNKNOWN";
  return tag2string[tag];}

void spytag(pnt tag){
  if(tag&signmask) printf("*");
  if(tag&markmask) printf("#");
  printf("%s",safe_tag2string(tag&~signmask&~markmask));}

void spytag0(pnt tag){
  spytag(tag);
  printf("\n");}

void spytagroot1(pnt x){
  if(x==T) printf("TAG_TRUE"); else spytag(ROOT(x));}

void spytagroot0(pnt x){
  spytagroot1(x);
  printf("\n");}

void unexpected_tag(char *msg,pnt value){
  printf("%s: unexpected tag: ",msg);
  spytagroot0(value);
  die("Goodbye");}

#define PNTBUFSIZE 100

char pntbuffer[PNTBUFSIZE];

int ptoa1(pnt x){
  int position;
  if(x==0) return 0;
  position=ptoa1(x/10);
  pntbuffer[position++]='0'+x%10;
  if(position>=PNTBUFSIZE) die("Internal error: pnt buffer overflow");
  return position;}

char *ptoa(pnt x){
  if(x==0){
    pntbuffer[0]='0';
    pntbuffer[1]=0;}
  else{
    pntbuffer[ptoa1(x)]=0;}
  return pntbuffer;}

void spydigits1(FILE *f,pnt x){
  if(x==0) return;
  spydigits1(f,x/10);
  fprintf(f,"%d",(int)(x%10));}

void spydigits(FILE *f,pnt x){
  if(x==0) fprintf(f,"0"); else spydigits1(f,x);}

void spydigitsln(char *msg,pnt x){
  fprintf(stderr,"%s ",msg);
  spydigits(stderr,x);
  fprintf(stderr,"\n");}

void spyint0(pnt x){
  spydigits(stdout,HEAD(x));
  for(x=TAIL(x);x;x=TAIL(x)){printf(" ");spydigits(stdout,HEAD(x));}}

void spyvect(pnt x){
  printf("\"");
  for(;TAIL(x);x=TAIL(x)){
    pnt i;
    pnt y=HEAD(x);
    for(i=0;i<bytesize;i++){
      printchar(y);
      y=y>>8;}}
  for(x=HEAD(x);x>1;x=x>>8) printchar(x);
  printf("\"");}

/*
void spyint1(pnt x){
  if((TAIL(x)==T)&&((HEAD(x)<10000)||((HEAD(x))>1+~10000))){
    printf("%5d",(int)(HEAD(x)));
    return;}
  printf("#");
  spyvect(x);}
*/

void spyname(pnt x,pnt arglist){
  for(;TAIL(x);x=TAIL(x)){
    pnt i;
    pnt y=HEAD(x);
    for(i=0;i<bytesize;i++){
      if((y&255)!='"'||NO_PAIR(arglist)) printchar(y);
      else {spy1(HEAD(arglist));arglist=TAIL(arglist);}
      y=y>>8;}}
  for(x=HEAD(x);x>1;x=x>>8)
  if((x&255)!='"'||NO_PAIR(arglist)) printchar(x);
  else {spy1(HEAD(arglist));arglist=TAIL(arglist);}
  if(arglist==T) return;
  printf(":");
  spy1(arglist);}

pnt root2name(pnt x){
  pnt root,ref,idx,name;
  root=HEAD(x);
  if(NO_PAIR(root)) return TRUE;
  ref=HEAD(root);
  if(NO_INT(ref)) return TRUE;
  if(NO_PAIR(TAIL(root))) return TRUE;
  idx=HEAD(TAIL(root));
  if(NO_INT(idx)) return TRUE;
  if(ref==Z){printf("(");spy1(idx);printf(")");return FALSE;}
  name=nameget(state,ref,idx);
  if(name==T) return TRUE;
  spyname(name,TAIL(x));
  return FALSE;}

void spy1(pnt x){
  pnt tag;
  if(spycnt==0){printf("...");return;}
  spycnt--;
  if(x==T){printf("T");return;}
  tag=ROOT(x);
  if(tag&signmask) printf("*");
  if(tag&markmask) printf("#");
  tag=tag&~signmask&~markmask;
  switch(tag){
  case TAG_FALSE:   printf("F");return;
  case TAG_INT:     spyint0(x);return;
  case TAG_VECT:    spyvect(x);return;
  case TAG_PAIR:    /*if(root2name(x))*/ prn2("P",HEAD(x),TAIL(x));return;
  case TAG_EX:      prn1("X",HEAD(x));return;
  case TAG_MAP:     printf("M(%s,",safe_tag2string(TAIL(x)));
                    spy1(HEAD(x));printf(")");return;
  case TAG_OBJ:     prn2("O",HEAD(HEAD(x)),TAIL(HEAD(x)));spy1(TAIL(x));return;
  case TAG_GARB:    printf("G(%s)",ptoa(x));return;
  case TTAG_VAR:    printf("TTAG_VAR(%s)",ptoa(HEAD(x)));return;
  case TTAG_LAMBDA: prn1("TTAG_LAMBDA",HEAD(x));return;
  case TTAG_CONST:  prn1("TTAG_CONST",HEAD(x));return;
  case MTAG_INDIR:  prn1("MTAG_INDIR",TAIL(x));return;
  case MTAG_CLOSURE:printf("CLOSURE");return;
  case ETAG_var:    printf("ETAG_var(%s)",ptoa(HEAD(x)));return;
  case ETAG_ecall:  prn2("ETAG_ecall",HEAD(x),TAIL(x));return;}
  if(invalid_tag(tag)) {printf("??");return;}
  prn2(tag2string[tag],HEAD(x),TAIL(x));}

void spy2(pnt x,pnt y){
  spycnt=x;
  spy1(y);}

pnt spy0(pnt x){
  spy2(100,x);
  printf("\n");
  return x;}

void spydie(pnt x,char *msg){
  spy0(x);
  die(msg);}

void backtrace(pnt x,pnt cnt){
  pnt c;
  pnt i;
  pnt y;
  pnt tag;
  if(cnt==0) return;
  printf("backtrace %3s: %10s\n",ptoa(cnt),ptoa(x));
  spy0(x);
  if(x==state){printf("Referenced from state\n");return;}
  if(x==global_x){printf("Referenced from head\n");return;}
  if(x==global_y){printf("Referenced from tail\n");return;}
  for(i=sp;i<STACKSIZE;i++)if(x==stack[i]){
    printf("Referenced from stack[%s]\n",ptoa(i));
    return;}
  for(i=storepnt;i<STORESIZE;i++)if(x==store[i]){
    printf("Referenced from store[%s]\n",ptoa(i));
    return;}
  for(c=heaplist;c!=0;c=HEAD(c))
  for(i=1;i<TAIL(c);i++){
    y=c+i*sizeof(cell);
    tag=ROOT(y)&~signmask;
    switch(tag){
    case TAG_FALSE:
    case TTAG_VAR:
    case TAG_GARB:
      continue;
    case TAG_EX:
    case TAG_MAP:
    case TTAG_LAMBDA:
    case TTAG_CONST:
      if(HEAD(y)==x){backtrace(y,cnt-1);return;}
      continue;
    case MTAG_INDIR:
    case TAG_INT:
    case TAG_VECT:
      if(TAIL(y)==x){backtrace(y,cnt-1);return;}
      continue;
    default:
      if(invalid_tag(tag)) unexpected_tag("backtrace",y);
      if(HEAD(y)==x){backtrace(y,cnt-1);return;}
      if(TAIL(y)==x){backtrace(y,cnt-1);return;}
      continue;}}
  printf("Not found\n");}

pnt testcnt=0;
void testfct(char *s){
  if(!test) return;
  if(STACKSIZE-sp) printf("Warning. Stack residue: %s\n",ptoa(STACKSIZE-sp));
  testcnt++;
  printf("%5s %s\n",ptoa(testcnt),s);}

void print3(pnt x){
  printf("%c",(int)(x&0xFF));}

void print2(pnt x){
  for(;TAIL(x);x=TAIL(x)){
    pnt i;
    pnt y=HEAD(x);
    for(i=0;i<bytesize;i++){
      print3(y);
      y=y>>8;}}
  for(x=HEAD(x);x>255;x=x>>8) print3(x);}

void print1(pnt x){
  pnt tag;
  if(x==T) return;
  tag=ROOT(x);
  switch(tag){
  case TAG_FALSE:   return;
  case TAG_INT:     print2(x);return;
  case TAG_VECT:    print2(x);return;
  case TAG_PAIR:    print1(HEAD(x));print1(TAIL(x));return;
  case TAG_MAP:     return;
  case TAG_OBJ:     return;}
  printf("\nInternal error in 'print': Unexpected tag");
  if(invalid_tag(tag)) printf("TAG=%s\n",ptoa(tag));
  else printf("TAG=%s\n",tag2string[tag]);
  deathReport();
  exit(1);}



/**********
 * Spying *
 **********/

#define TIMERS 1000
#define MEGA 1000000
int currentTimer=-1;
struct timeval timers[TIMERS];
int counters[TIMERS];

void initTimers(){
  int i;
  for(i=0;i<TIMERS;i++){
    timers[i].tv_sec=0;
    timers[i].tv_usec=0;
    counters[i]=0;}}

void setTimer(int i){
  struct timeval time;
  if(gettimeofday(&time,NULL)) pdie("setTimer()/gettimeofday()");
  if(0<=currentTimer&&currentTimer<TIMERS){
    timers[currentTimer].tv_sec+=time.tv_sec;
    timers[currentTimer].tv_usec+=time.tv_usec;
    if(timers[currentTimer].tv_usec>=MEGA){
      timers[currentTimer].tv_usec-=MEGA;
      timers[currentTimer].tv_sec+=1;}}
  currentTimer=i;
  if((0<=currentTimer)&(currentTimer<TIMERS)){
    counters[currentTimer]++;
    timers[currentTimer].tv_sec-=time.tv_sec;
    timers[currentTimer].tv_usec-=time.tv_usec;
    if(timers[currentTimer].tv_usec<0){
      timers[currentTimer].tv_usec+=MEGA;
      timers[currentTimer].tv_sec-=1;}}}

int deathReportAlreadyCalled=FALSE;
void deathReport(){
  int i,d,first,last,usec;
  if(deathReportAlreadyCalled){
    printf("Panic: error occurred during generation of death report\n");
    exit(1);}
  deathReportAlreadyCalled=TRUE;
#if(PRINTGC)
  spydigitsln("heapsize in cells ",heapsize);
  spydigitsln("max heapsize      ",heapmax);
  spydigitsln("stacksize         ",STACKSIZE-sp);
  spydigitsln("max stacksize     ",STACKSIZE);
#endif
  if(spyvar!=T){
    printf("Value of spy variable:\n");
    spy0(spyvar);}
  first=TRUE;
  last=currentTimer;
  setTimer(-1);
  for(i=0;i<TIMERS;i++)
  if(counters[i]){
    if(first){
      first=FALSE;
      printf("Values of timers\n");
      printf("Timer Calls     Seconds\n");}
    printf("%5d %5d %4d.",i,counters[i],(int)(timers[i].tv_sec));
    usec=timers[i].tv_usec;
    for(d=0;d<6;d++){usec*=10;printf("%1d",usec/MEGA);usec%=MEGA;}
    if(i==last) printf(" This timer was still counting");
    printf("\n");}}

void processSigint(int sig){
  printf("\n");
  printf("Sigint\n");
  deathReport();
  exit(0);}

void catchSigint(){
  signal(SIGINT,&processSigint);}

int safewrite(char *msg){
  return write(STDERR_FILENO,msg,sizeof(msg)-1);}

static void processSigsegv(int sig){
  safewrite("lgwam: Segmentation fault.\n");
  safewrite("This is probably due to overflow of the C stack.\n");
  safewrite("If you run bash try this: ulimit -s unlimited\n");
  safewrite("If you run csh try this: unlimit stacksize\n");
  safewrite("Goodbye.\n");
  abort();}

void catchSigsegv(){
  signal(SIGSEGV,&processSigsegv);}



/*********************
 * MEMORY ALLOCATION *
 *********************/

void chunkinit(pnt chunk,pnt size){
  pnt i;
  ROOT(chunk)=TAG_GARB;
  HEAD(chunk)=heaplist;
  TAIL(chunk)=size;
  heaplist=chunk;
  heapsize+=size-1;
  for(i=1;i<size;i++){
    chunk+=sizeof(cell);
    ROOT(chunk)=TAG_GARB;
    HEAD(chunk)=garb;
    garb=chunk;}}

void heapalloc(pnt size){
  pnt chunk=(pnt)malloc(size*sizeof(cell));
  if(chunk==0) die("Dynamic memory allocation (malloc) failed");
  chunkinit(chunk,size);}

/* To be called before using heap */
void heapinit(){
  pnt initsize=pages_len/sizeof(cell);
#ifndef __CYGWIN__
  struct sysinfo info;
  if(sysinfo(&info)) pdie("sysinfo()");
  if(info.totalram>info.mem_unit)
  info.totalram/=sizeof(cell); else
  info.mem_unit/=sizeof(cell);
  heapmax=info.totalram*info.mem_unit;
  heapmax=heapmax/100*HEAPMAX0;
#else
  heapmax=2000111000/sizeof(cell)/100*HEAPMAX0;
#endif /* __CYGWIN__ */
#if(PRINTGC)
  spydigitsln("max heapsize      ",heapmax);
#endif
  heapsize=0;
  heaplist=NIL;
  garb=NIL;
  heapalloc(initsize);}

/* To be called when built pages are loaded */
void pagereuse(){
  pnt page_begin=(pnt)pages;
  pnt page_end=page_begin+pages_len;
  pnt chunk=(page_begin+sizeof(cell)-1)/sizeof(cell)*sizeof(cell);
  pnt size=(page_end-chunk)/sizeof(cell);
  if(pages_len<3*sizeof(cell)) return;
  if(size<2) return;
  chunkinit(chunk,size);}



/**********************
 * GARBAGE COLLECTION *
 **********************/

void mark(pnt x){
  pnt root;
  if(!x) return;
  root=ROOT(x);
  if(root&signmask) return;
  ROOT(x)=root|signmask;
  switch(root&~markmask){
  case TAG_FALSE:
  case TTAG_VAR:
  case ETAG_var:
  case ETAG_yy:
    return;
  case TAG_EX:
  case TAG_MAP:
  case TTAG_LAMBDA:
  case TTAG_CONST:
    mark(HEAD(x));
    return;
  case TAG_PAIR:
  case TAG_OBJ:
  case TTAG_APPLY:
  case TTAG_CALL:
  case TTAG_ECALL:
  case TTAG_PAIR:
  case MTAG_CLOSURE:
  case MTAG_PAIR:
  case MTAG_LAMBDA:
  case MTAG_FIX:
  case ETAG_ecall:
  case ETAG_digit0:
  case ETAG_digit1:
  case ETAG_digit2:
  case ETAG_digit3:
  case ETAG_digit4:
  case ETAG_digit5:
  case ETAG_digit6:
  case ETAG_digit7:
  case ETAG_digit8:
  case ETAG_digit9:
  case ETAG_times:
  case ETAG_plus:
  case ETAG_minus:
  case ETAG_and:
  case ETAG_or:
  case ETAG_then:
  case ETAG_pair:
  case ETAG_lt:
  case ETAG_le:
  case ETAG_equal:
  case ETAG_gt:
  case ETAG_ge:
  case ETAG_apply:
  case ETAG_boolp:
  case ETAG_catch:
  case ETAG_maptag1:
  case ETAG_div:
  case ETAG_head:
  case ETAG_intp:
  case ETAG_boolg:
  case ETAG_intg:
  case ETAG_mapg:
  case ETAG_objg:
  case ETAG_pairg:
  case ETAG_mapp:
  case ETAG_maptag:
  case ETAG_mod:
  case ETAG_norm:
  case ETAG_objp:
  case ETAG_pairp:
  case ETAG_raise:
  case ETAG_root:
  case ETAG_tail:
  case ETAG_untag:
  case ETAG_digitend:
  case ETAG_par:
  case ETAG_uplus:
  case ETAG_dplus:
  case ETAG_uminus:
  case ETAG_dminus:
  case ETAG_not:
  case ETAG_Base:
  case ETAG_LET:
  case ETAG_ash:
  case ETAG_bottom:
  case ETAG_bt2bytes:
  case ETAG_bt2vects:
  case ETAG_bt2vector:
  case ETAG_ceiling:
  case ETAG_destruct:
  case ETAG_evenp:
  case ETAG_exception:
  case ETAG_false:
  case ETAG_floor:
  case ETAG_half:
  case ETAG_if:
  case ETAG_intlength:
  case ETAG_logand:
  case ETAG_logandc1:
  case ETAG_logandc2:
  case ETAG_logbitp:
  case ETAG_logcount:
  case ETAG_logeqv:
  case ETAG_logior:
  case ETAG_lognand:
  case ETAG_lognor:
  case ETAG_lognot:
  case ETAG_logorc1:
  case ETAG_logorc2:
  case ETAG_logtest:
  case ETAG_logxor:
  case ETAG_map:
  case ETAG_prenorm:
  case ETAG_notnot:
  case ETAG_object:
  case ETAG_print:
  case ETAG_round:
  case ETAG_spy:
  case ETAG_timer:
  case ETAG_trace:
  case ETAG_truncate:
  case ETAG_vector:
  case ETAG_vnorm:
  case ETAG_vempty:
  case ETAG_vindex:
  case ETAG_vlength:
  case ETAG_vprefix:
  case ETAG_vsubseq:
  case ETAG_vsuffix:
  case ETAG_v2bytes:
  case ETAG_v2vects:
  case ETAG_vt2bytes:
  case ETAG_vt2vects:
  case ETAG_vt2v:
  case ETAG_compile:
  case ETAG_ripemd:
  case ETAG_sl2rack:
  case ETAG_rack2sl:
    mark(HEAD(x));
    mark(TAIL(x));
    return;
  case MTAG_INDIR:
  case TAG_INT:
  case TAG_VECT:
    mark(TAIL(x));
    return;
  default:
    printf("Unknown tag found during GC\n");
    spy0(x);
    backtrace(x,100);
    die("Goodbye");}}

void sweep(){
  pnt c;
  pnt i;
  pnt y;
  pnt root;
  garb=NIL;
  collected=0;
  for(c=heaplist;c!=0;c=HEAD(c))
  for(i=1;i<TAIL(c);i++){
    y=c+i*sizeof(cell);
    root=ROOT(y);
    if(root&signmask) ROOT(y)=root&~signmask; else {
      collected++;
      ROOT(y)=TAG_GARB;
      HEAD(y)=garb;
      garb=y;}}}

/* Garbage collection */
/* At present we quit if garbage collection gives less then 30 percent
 * free cells. The last line below is superflous when checking for the
 * 30 percent limit. If the 30 percent limit is removed, one runs the
 * risk of very long executing times in cases where the system gives
 * up in the end anyway */
void gc(pnt x,pnt y){
  pnt i,used,heapsize1;
#if(PRINTGC)
  pnt percent;
  if(!ALWAYSGC) fprintf(stderr,"gc\n");
#endif
  global_x=x;
  global_y=y;
  mark(x);
  mark(y);
  for(i=sp;i<STACKSIZE;i++) mark(stack[i]);
  for(i=storepnt;i<STORESIZE;i++) mark(store[i]);
  mark(state);
  mark(inputs);
  mark(outputs);
  mark(event);
  mark(extensions);
  mark(spyvar);
  sweep();
#if(PRINTGC)
  percent=collected/(heapsize/100);
  if(!ALWAYSGC){
    fprintf(stderr,"released           ");
    spydigits(stderr,percent);
    fprintf(stderr," percent\n");
    fprintf(stderr,"released           ");
    spydigits(stderr,collected);
    fprintf(stderr," cells\n");
    fprintf(stderr,"stack              ");
    spydigits(stderr,sp);
    fprintf(stderr," free positions\n");}
#endif
  used=heapsize-collected;
  if(used<heapsize/100*HEAPMAX2) return;
  if(used>heapmax/100*HEAPMAX1) die("Ran out of physical RAM, giving up.");
  heapsize1=used/HEAPMAX3*100;
  if(heapsize1>heapmax) heapsize1=heapmax;
  if(heapsize1<=heapsize) return;
  heapsize1-=heapsize;
#if(PRINTGC)
  fprintf(stderr,"allocate           ");
  spydigits(stderr,heapsize1);
  fprintf(stderr," cells\n");
#endif
  heapalloc(heapsize1);
  if(!garb) die("Internal error in GC, goodbye.");}



/****************
 * CONSTRUCTORS *
 ****************/

/* Get fresh cell */
pnt fresh(pnt x,pnt y){
  pnt result;
  if(always_gc || !garb) gc(x,y);
  result=garb;
  /*if(ROOT(result)!=TAG_GARB) die("Attempt to allocate non-garbage.\n");*/
  garb=HEAD(result);
  return result;}

/* General constructor */
pnt cons(pnt root,pnt head,pnt tail){
  pnt result;
  result=fresh(head,tail);
  ROOT(result)=root;
  HEAD(result)=head;
  TAIL(result)=tail;
  return result;}

/* General constructor with cardinal head */
pnt cons2(pnt root,pnt head,pnt tail){
  pnt result=fresh(NIL,tail);
  ROOT(result)=root;
  HEAD(result)=head;
  TAIL(result)=tail;
  return result;}

/* General constructor with cardinal tail */
pnt cons1(pnt root,pnt head,pnt tail){
  pnt result=fresh(head,NIL);
  ROOT(result)=root;
  HEAD(result)=head;
  TAIL(result)=tail;
  return result;}

/* Part of safe constructor */
pnt scons(pnt root,pnt tail){
  pnt result;
  result=fresh(NIL,tail);
  ROOT(result)=root;
  HEAD(result)=pop();
  TAIL(result)=tail;
  return result;}



/************************
 * INTEGER CONSTRUCTORS *
 ************************/

/* Simple integer cons */
pnt icons(pnt head,pnt tail){
  return cons2(TAG_INT,head,tail);}

/* Normalization preserving integer cons */
pnt intcons(pnt head,pnt tail){
  if((head&signmask)==0){
    if(tail!=Z) return icons(head,tail);
    if(head==0) return Z;
    return icons(head,NIL);}
  else{
    if(tail!=M) return icons(head,tail);
    if(head==~(pnt)0) return M;
    return icons(head,NIL);}}

/* Integer tail with sign extend if needed */
pnt inttail(pnt cons){
  if(TAIL(cons)) return TAIL(cons);
  if((HEAD(cons)&signmask)==0) return Z; else return M;}

/* Convert C integer to twos complement pnt */
pnt int2pnt(spnt i){
  if(i>=0) return (pnt)i;
  return 1+~(pnt)(-i);}

/* Convert C integer to integer */
pnt int2int(spnt i){
  return icons(int2pnt(i),NIL);}

/* Convert C integer to normalized integer */
pnt JN(spnt i){
  if(i==0) return Z;
  if(i==-1) return M;
  return keep(int2int(i));}

pnt K2(spnt i,spnt j){
  return keep(intcons(int2pnt(i),JN(j)));}



/***********************
 * VECTOR CONSTRUCTORS *
 ***********************/

pnt consvec(pnt head,pnt tail){
  pnt result=fresh(NIL,tail);
  ROOT(result)=TAG_VECT;
  HEAD(result)=head;
  TAIL(result)=tail;
  return result;}

pnt int2vec(pnt x){
  if(NO_INT(x)) return x;
  return consvec(HEAD(x),TAIL(x));}

/* Destructive. Only for use in main */
pnt vec2int(pnt x){
  if(NO_INT(x)) die("vec2int applied to non-int");
  ROOT(x)=TAG_INT;
  return x;}

pnt str2vec2(unsigned char *str,pnt len){
  if(len==0) return 1;
  return (*str)|(str2vec2(str+1,len-1)<<8);}

pnt str2vec1(unsigned char *str,pnt len){
  if(len<bytesize) return consvec(str2vec2(str,len),NIL);
  return consvec(str2vec2(str,bytesize),str2vec1(str+bytesize,len-bytesize));}

pnt str2vec0(unsigned char *str){
  pnt result=store_find((pnt)1,str);
  if(result!=T) return result;
  return keep3((pnt)1,str,str2vec1(str,strlen((char *)str)));}

pnt str2vec(char *str){
  return str2vec0((unsigned char *)str);}

pnt str2nonvec2(pnt empty,unsigned char *str,pnt len){
  if(len==0) return empty;
  return (*str)|(str2nonvec2(empty,str+1,len-1)<<8);}

pnt str2nonvec1(pnt empty,unsigned char *str,pnt len){
  if(len<bytesize) return intcons(str2nonvec2(empty,str,len),Z);
  return intcons(
    str2nonvec2(empty,str,bytesize),
    str2nonvec1(empty,str+bytesize,len-bytesize));}

pnt str2nonvec0(pnt empty,unsigned char *str){
  pnt result=store_find(empty*256+3,str);
  if(result!=T) return result;
  return keep3(empty*100+3,str,str2nonvec1(empty,str,strlen((char *)str)));}

pnt str2nonvec(pnt empty,char *str){
  return str2nonvec0(empty,(unsigned char *)str);}



/****************
 * MISCELANEOUS *
 ****************/

/* Find number of bits in a pnt */
pnt findsize(){
  pnt result,mask;
  result=0;
  for(mask=1;mask;mask=mask<<1) result++;
  return result;}



/*****************************
 * OPTIMIZED EAGER FUNCTIONS *
 *****************************/

/* These functions assume that no arguments are exceptions */

pnt bottom(){/* bottom */
  printf("This takes forever.\n");
  deathReport();
  die("Goodbye.\n");
  return T;}

pnt true(){
  return T;}

pnt false(){ /* false */
  return F;}

pnt exception(){ /* exception */
  return X;}

pnt norm(pnt x){/* " norm */ /* norm " */
  return x;}

pnt then(pnt x,pnt y){/* " .then. " */
  return y;}

pnt boolp(pnt x){/* " boolp */
  if(NO_BOOL(x)) return F;
  return T;}

pnt intp(pnt x){/* " intp */
  if(NO_INT(x)) return F;
  return T;}

pnt pairp(pnt x){/* " pairp */
  if(NO_PAIR(x)) return F;
  return T;}

pnt mapp(pnt x){/* " mapp */
  if(NO_MAP(x)) return F;
  return T;}

pnt objp(pnt x){/* " objectp */
  if(NO_OBJ(x)) return F;
  return T;}

pnt boolg(pnt x,pnt y){/* " is bool : " */
  if(NO_BOOL(x)) return X;
  return y;}

pnt intg(pnt x,pnt y){/* " is int : " */
  if(NO_INT(x)) return X;
  return y;}

pnt pairg(pnt x,pnt y){/* " is map : " */
  if(NO_PAIR(x)) return X;
  return y;}

pnt mapg(pnt x,pnt y){/* " is object : " */
  if(NO_MAP(x)) return X;
  return y;}

pnt objg(pnt x,pnt y){/* " is pair : " */
  if(NO_OBJ(x)) return X;
  return y;}

pnt not(pnt x){/* .not. " */
  if(x==T) return F;
  if(GD_EX(x)) return T;
  return X;}

pnt notnot(pnt x){/* notnot " */
  if(x==T) return T;
  if(GD_EX(x)) return F;
  return X;}

pnt eq0(pnt x,pnt y){
  pnt x1;
  pnt y1;
  if(x==T) return y==T;
  if(y==T) return FALSE;
  x1=ROOT(x);
  y1=ROOT(y);
  x1=(x1==TAG_VECT)?TAG_INT:x1;
  y1=(y1==TAG_VECT)?TAG_INT:y1;
  if(x1!=y1) return FALSE;
  switch(x1){
  case TAG_FALSE:
  case TAG_MAP:
    return TRUE;
  case TAG_INT:
    while(TRUE){
      if(HEAD(x)!=HEAD(y)) return FALSE;
      x=TAIL(x);
      y=TAIL(y);
      if(!x){if(!y) return TRUE; else return FALSE;}
      if(!y) return FALSE;}
  case TAG_EX: /* only for the sake of etst */
    return eq0(HEAD(x),HEAD(y));
  case TAG_PAIR:
  case TAG_OBJ:
    return eq0(HEAD(x),HEAD(y)) && eq0(TAIL(x),TAIL(y));
  default:
    unexpected_tag("eq0",x);
    return T; /* This never happens*/}}

pnt eq(pnt x,pnt y){/* " = " */
  if(eq0(x,y)) return T; else return F;}

pnt pair(pnt head,pnt tail){/* " :: " */
  return cons(TAG_PAIR,head,tail);}

pnt head(pnt cons){/* " head */
  if(NO_PAIR(cons)) return cons;
  return HEAD(cons);}

pnt tail(pnt cons){/* " tail */
  if(NO_PAIR(cons)) return cons;
  return TAIL(cons);}

pnt lognot1(pnt x){
  pnt u=HEAD(x);
  pnt z;
  if(!TAIL(x)) z=NIL; else z=lognot1(TAIL(x));
  return intcons(~u,z);}

pnt lognot(pnt x){/* lognot ( " ) */
  if(NO_INT(x)) return X;
  return lognot1(x);}

pnt logand1(pnt x,pnt y){
  pnt u,v;
  u=HEAD(x)&HEAD(y);
  if(!TAIL(x) && !TAIL(y)) v=SIGN(u); else v=logand1(inttail(x),inttail(y));
  return intcons(u,v);}

pnt logand(pnt x,pnt y){/* logand ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  return logand1(x,y);}

pnt logior1(pnt x,pnt y){
  pnt u,v;
  u=HEAD(x)|HEAD(y);
  if(!TAIL(x) && !TAIL(y)) v=SIGN(u); else v=logior1(inttail(x),inttail(y));
  return intcons(u,v);}

pnt logior(pnt x,pnt y){/* logior ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  return logior1(x,y);}

pnt logxor1(pnt x,pnt y){
  pnt u,v;
  u=HEAD(x)^HEAD(y);
  if(!TAIL(x) && !TAIL(y)) v=SIGN(u); else v=logxor1(inttail(x),inttail(y));
  return intcons(u,v);}

pnt logxor(pnt x,pnt y){/* logxor ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  return logxor1(x,y);}

pnt logandc11(pnt x,pnt y){
  pnt u,v;
  u=(~HEAD(x))&HEAD(y);
  if(!TAIL(x) && !TAIL(y)) v=SIGN(u); else v=logandc11(inttail(x),inttail(y));
  return intcons(u,v);}

pnt logandc1(pnt x,pnt y){/* logandc1 ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  return logandc11(x,y);}

pnt logandc21(pnt x,pnt y){
  pnt u,v;
  u=HEAD(x)&(~HEAD(y));
  if(!TAIL(x) && !TAIL(y)) v=SIGN(u); else v=logandc21(inttail(x),inttail(y));
  return intcons(u,v);}

pnt logandc2(pnt x,pnt y){/* logandc2 ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  return logandc21(x,y);}

pnt lognand1(pnt x,pnt y){
  pnt u,v;
  u= ~(HEAD(x)&HEAD(y));
  if(!TAIL(x) && !TAIL(y)) v=SIGN(u); else v=lognand1(inttail(x),inttail(y));
  return intcons(u,v);}

pnt lognand(pnt x,pnt y){/* lognand ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  return lognand1(x,y);}

pnt logorc11(pnt x,pnt y){
  pnt u,v;
  u=(~HEAD(x))|HEAD(y);
  if(!TAIL(x) && !TAIL(y)) v=SIGN(u); else v=logorc11(inttail(x),inttail(y));
  return intcons(u,v);}

pnt logorc1(pnt x,pnt y){/* logorc1 ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  return logorc11(x,y);}

pnt logorc21(pnt x,pnt y){
  pnt u,v;
  u=HEAD(x)|(~HEAD(y));
  if(!TAIL(x) && !TAIL(y)) v=SIGN(u); else v=logorc21(inttail(x),inttail(y));
  return intcons(u,v);}

pnt logorc2(pnt x,pnt y){/* logorc2 ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  return logorc21(x,y);}

pnt lognor1(pnt x,pnt y){
  pnt u,v;
  u= ~(HEAD(x)|HEAD(y));
  if(!TAIL(x) && !TAIL(y)) v=SIGN(u); else v=lognor1(inttail(x),inttail(y));
  return intcons(u,v);}

pnt lognor(pnt x,pnt y){/* lognor ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  return lognor1(x,y);}

pnt logeqv1(pnt x,pnt y){
  pnt u,v;
  u= ~(HEAD(x)^HEAD(y));
  if(!TAIL(x) && !TAIL(y)) v=SIGN(u); else v=logeqv1(inttail(x),inttail(y));
  return intcons(u,v);}

pnt logeqv(pnt x,pnt y){/* logeqv ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  return logeqv1(x,y);}

pnt logtest1(pnt x,pnt y){
  if(HEAD(x)&HEAD(y)) return T;
  if(!TAIL(x) && !TAIL(y)) return F;
  return logtest1(inttail(x),inttail(y));}

pnt logtest(pnt x,pnt y){/* logtest ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  return logtest1(x,y);}

/* returns signum(x-y)+1 for signed integers x and y */
pnt cmp2(pnt x,pnt y){
  if(x==y) return 1;
  if((x^signmask)<(y^signmask)) return 0; else return 2;}

/* returns signum(x-y)+1 for tagged integers x and y */
pnt cmp1(pnt x,pnt y){
  pnt z;
  if(!TAIL(x) && !TAIL(y)) return cmp2(HEAD(x),HEAD(y));
  z=cmp1(inttail(x),inttail(y));
  if(z!=1) return z;
  if(HEAD(x)==HEAD(y)) return 1;
  if(HEAD(x)<HEAD(y)) return 0; else return 2;}

pnt lt(pnt x,pnt y){/* " < " */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(cmp1(x,y)<1) return T; else return F;}

pnt gt(pnt x,pnt y){/* " > " */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(cmp1(x,y)>1) return T; else return F;}

pnt le(pnt x,pnt y){/* " <= " */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(cmp1(x,y)<=1) return T; else return F;}

pnt ge(pnt x,pnt y){/* " >= " */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(cmp1(x,y)>=1) return T; else return F;}

pnt evenp(pnt x){/* evenp ( " ) */
  if(NO_INT(x)) return X;
  if(HEAD(x)&1) return F; else return T;}

pnt half1(pnt x){
  pnt v=HEAD(x);
  pnt y=v>>1;
  pnt z=TAIL(x);
  if(!z){if(v&signmask) return intcons(y+signmask,M); else return intcons(y,Z);}
  if(HEAD(z)&1) y=y+signmask;
  return intcons(y,half1(z));}

pnt half(pnt x){/* half ( " ) */
  if(NO_INT(x)) return X;
  return half1(x);}

pnt small(pnt x){
  if(NO_INT(x)) return X;
  if(x==Z) return T;
  if(x==M) return T;
  return F;}

pnt base(){/* Base */
  return I[10];}

pnt zero(){/* %% */
  return Z;}

pnt unaryPlus(pnt x){/* + " */
  if(NO_INT(x)) return X;
  return x;}

pnt plus1(pnt x,pnt y,pnt c){
  pnt X=HEAD(x);
  pnt Y=HEAD(y);
  pnt u=HALFHEAD(X)+HALFHEAD(Y)+c;
  pnt v=HALFTAIL(X)+HALFTAIL(Y)+HALFTAIL(u);
  pnt w=HALFCONS(u,v);
  if(TAIL(x)||TAIL(y))
  return intcons(w,plus1(inttail(x),inttail(y),HALFTAIL(v)));
  if(w&signmask){
    if((X&signmask)||(Y&signmask))
      return intcons(w,M);
    else
      return intcons(w,Z);}
  else{
    if((!(X&signmask))||(!(Y&signmask)))
      return intcons(w,Z);
    else
      return intcons(w,M);}}

pnt plus(pnt x,pnt y){/* " + " */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  return plus1(x,y,0);}

pnt minus(pnt x,pnt y){/* " - " */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  push(x);
  return ret(1,plus1(x,lognot(y),1));}

pnt unaryMinus(pnt x){/* - " */
  return minus(Z,x);}

pnt integerLength3(pnt x){
  if(!x) return 0;
  return 1+integerLength3(x>>1);}

pnt integerLength2(pnt x){
 if(x&signmask) return integerLength3(~x);
  return integerLength3(x);}

pnt integerCombine(pnt x,pnt y){
  pnt u=HALFHEAD(y)*pntsize+x;
  pnt v=HALFTAIL(y)*pntsize+HALFTAIL(u);
  return intcons(HALFCONS(u,v),intcons(HALFTAIL(v),Z));}

pnt integerLength1(pnt x,pnt y){
  if(TAIL(x)) return integerLength1(TAIL(x),y+1);
  return integerCombine(integerLength2(HEAD(x)),y);}

pnt integerLength(pnt x){/* integer-length ( " ) */
  if(NO_INT(x)) return X;
  return integerLength1(x,0);}

pnt logcount2(pnt x){
  if(!x) return 0;
  return (x&1)+logcount2(x>>1);}

pnt logcount1(pnt x,pnt y){
  y=y+logcount2(HEAD(x));
  if(y&signmask) die("logcount overflow");
  if(TAIL(x)) return logcount1(TAIL(x),y);
  return intcons(y,Z);}

pnt logcount(pnt x){/* logcount ( " ) */
  if(NO_INT(x)) return X;
  if(ge(x,Z)==T) return logcount1(x,0);
  return logcount1(lognot(x),0);}

/* compute 10*x+y */
pnt digit(pnt x,pnt y){
  pnt x0=HEAD(x);
  pnt x1=HALFHEAD(x0)*10+y;
  pnt x2=HALFTAIL(x0)*10+HALFTAIL(x1);
  pnt x3=HALFCONS(x1,x2);
  pnt y1=HALFTAIL(x2);
  if(!TAIL(x)&&(y1==0)&&!(x3&signmask)) return intcons(x3,Z);
  if(!TAIL(x)&&(y1==9)&&(x3&signmask)) return intcons(x3,M);
  return intcons(x3,digit(inttail(x),y1));}

pnt digit0(pnt x){if(NO_INT(x)) return X;return digit(x,0);} /* " %0 */
pnt digit1(pnt x){if(NO_INT(x)) return X;return digit(x,1);} /* " %1 */
pnt digit2(pnt x){if(NO_INT(x)) return X;return digit(x,2);} /* " %2 */
pnt digit3(pnt x){if(NO_INT(x)) return X;return digit(x,3);} /* " %3 */
pnt digit4(pnt x){if(NO_INT(x)) return X;return digit(x,4);} /* " %4 */
pnt digit5(pnt x){if(NO_INT(x)) return X;return digit(x,5);} /* " %5 */
pnt digit6(pnt x){if(NO_INT(x)) return X;return digit(x,6);} /* " %6 */
pnt digit7(pnt x){if(NO_INT(x)) return X;return digit(x,7);} /* " %7 */
pnt digit8(pnt x){if(NO_INT(x)) return X;return digit(x,8);} /* " %8 */
pnt digit9(pnt x){if(NO_INT(x)) return X;return digit(x,9);} /* " %9 */

pnt str2int1(char *str,pnt result){
  if(*str==0) return result;
  if(*str=='-') return unaryMinus(str2int1(str+1,result));
  if(*str<'0') return str2int1(str+1,result);
  if(*str>'9') return str2int1(str+1,result);
  return str2int1(str+1,digit(result,*str-'0'));}

pnt str2int0(unsigned char *str){
  pnt result=store_find((pnt)2,str);
  if(result!=T) return result;
  return keep3((pnt)2,str,str2int1((char *)str,Z));}

pnt str2int(char *str){
  return str2int0((unsigned char *)str);}

pnt alloc1(pnt x){
  if(!x) return NIL;
  return icons(0,alloc1(TAIL(x)));}

pnt alloc2(pnt x,pnt y){
  if(!x) return alloc1(y);
  return icons(0,alloc2(TAIL(x),y));}

pnt normalize(pnt x){
  pnt u=HEAD(x);
  if(TAIL(x)) return intcons(u,normalize(TAIL(x)));
  if(u==0) return Z;
  if(u==~(pnt)0) return M;
  return x;}

/*
x:  First factor
y:  Second factor
z:  Product
x1: Pointer which traverses x
y1: Pointer which traverses y
z1: Pointer which traverses z synchronously with x
z2: Pointer which traverses z synchronously with y
a:  Current word from x
b:  Current word from y
c1: Carry 1
c2: Carry 2
d0: Product, low half
d1: Product, high half
*/
pnt umul(pnt x,pnt y){
  pnt z,x1,y1,z1,z2,a,b,c1,c2,d0,d1,d2;
  z=alloc2(x,y);
  z1=z;
  for(x1=x;x1;x1=TAIL(x1)){
    c1=0;
    z2=z1;
    a=HEAD(x1);
    for(y1=y;y1;y1=TAIL(y1)){
      b=HEAD(y1);
      c2=HEAD(z2);
      d0=HALFHEAD(a)*HALFHEAD(b)+HALFHEAD(c1)+HALFHEAD(c2);
      d1=HALFHEAD(a)*HALFTAIL(b)+HALFTAIL(d0)+HALFTAIL(c1);
      d2=HALFTAIL(a)*HALFHEAD(b)+HALFHEAD(d1)+HALFTAIL(c2);
      c1=HALFTAIL(a)*HALFTAIL(b)+HALFTAIL(d1)+HALFTAIL(d2);
      HEAD(z2)=HALFCONS(d0,d2);
      z2=TAIL(z2);
    }
    HEAD(z2)=c1;
    z1=TAIL(z1);
  }
  return normalize(z);}

pnt times(pnt x,pnt y){/* " * " */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  push(x);
  push(y);
  if(ge(x,Z)==T){
    if(ge(y,Z)==T) return ret(2,umul(x,y));
    push(unaryMinus(y));
    return ret(3,unaryMinus(umul(x,stack[sp])));}
  push(unaryMinus(x));
  if(ge(y,Z)==T) return ret(3,unaryMinus(umul(stack[sp],y)));
  push(unaryMinus(y));
  return ret(4,umul(stack[sp],stack[sp+1]));}

pnt logbitp1(pnt x,pnt y){
  if(!y) die("logbitp internal error");
  if(x>=pntsize) return logbitp1(x-pntsize,TAIL(y));
  if((HEAD(y)>>x)&1) return T;
  return F;}

pnt logbitp(pnt x,pnt y){/* logbitp ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(lt(x,Z)==T) return X;
  if(le(integerLength(y),x)==T) return lt(y,Z);
  if(TAIL(x)) die("logbitp overflow");
  return logbitp1(HEAD(x),y);}

pnt asll(pnt x,pnt y){
  if(y==0) return x;
  return intcons(0,asll(x,y-1));}

pnt asrr(pnt x,pnt y){
  if(y==0) return x;
  return asrr(inttail(x),y-1);}

pnt asl1(pnt x,pnt y,pnt c){
  pnt x1=HEAD(x);
  pnt x2=(x1<<y)+c;
  pnt c2=(x1>>(pntsize-y));
  if(x==Z) return intcons(x2,Z);
  if(x==M) return intcons(x2,M);
  return intcons(x2,asl1(inttail(x),y,c2));}

pnt asl(pnt x,pnt y){
  if(y==0) return x;
  return asl1(x,y,0);}

pnt asr(pnt x,pnt y){
  if(y==0) return x;
  return inttail(asl1(x,pntsize-y,0));}

pnt ash(pnt x,pnt y){/* ash ( " , " ) */
  pnt z;
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(TAIL(y)) die("ash overflow");
  z=HEAD(y);
  if(ge(y,Z)==T) return asll(asl(x,z%pntsize),z/pntsize);
  z=1+~z;
  return asrr(asr(x,z%pntsize),z/pntsize);}

/*
The division algorithms below leave room for HUGE improvements.
*/

/* Returns 2x */
pnt twice0(pnt x){
  return asl1(x,1,0);}

/* Returns 2x+1 */
pnt twice1(pnt x){
  return asl1(x,1,1);}

/* pairUnaryMinus(q::r)=(-q)::(-r) */
pnt pairUnaryMinus(pnt x){
  pnt y=HEAD(x);
  pnt z=TAIL(x);
  push(z);
  y=unaryMinus(y);
  push(y);
  return ret(2,pair(y,unaryMinus(z)));}

/* divAdjust(q::r,d)=(q+1)::(r-d) */
pnt divAdjust(pnt x,pnt d){
  pnt q=HEAD(x);
  pnt r=TAIL(x);
  push(q);
  r=minus(r,d);
  push(r);
  return ret(2,pair(plus(q,I[1]),r));}

/*
Divide non-negative remainder with positive divisor.
floor1(x,y) divides x by y and returns pair(quotient,remainder).
The remainder satisfies 0 <= remainder < y.
floor1 does not secure its first argument against GC.
*/

pnt floor1(pnt dividend,pnt divisor){
  pnt result,quotient,remainder,remainder1;
  push(divisor);
  if(lt(dividend,divisor)==T) return ret(1,pair(Z,dividend));
  result=floor1(dividend,twice0(divisor));
  push(result);
  quotient=HEAD(result);
  remainder=TAIL(result);
  remainder1=minus(remainder,divisor);
  if(lt(remainder1,Z)==T) return ret(2,pair(twice0(quotient),remainder));
  push(remainder1);
  return ret(3,pair(twice1(quotient),remainder1));}

/* Version of floor which secures both arguments against GC */
pnt floor0(pnt dividend,pnt divisor){
  push(dividend);
  return ret(1,floor1(dividend,divisor));}

/* dividend>=0, divisor>0 */
pnt ceiling0(pnt dividend,pnt divisor){
  pnt result;
  result=floor0(dividend,divisor);
  if(TAIL(result)==Z) return result;
  return divAdjust(result,divisor);}

/* dividend>=0, divisor>0 */
pnt round0(pnt dividend,pnt divisor){
  pnt result,result1;
  push(divisor);
  result=floor0(dividend,divisor);
  push(result);
  result1=cmp1(twice0(TAIL(result)),divisor);
  if(result1==0) return ret(2,result);
  if(result1==2) return ret(2,divAdjust(result,divisor));
  if(evenp(HEAD(result))==T) return ret(2,result);
  return ret(2,divAdjust(result,divisor));}

pnt floorx(pnt x,pnt y){/* floor ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(le(y,Z)==T) return X;
  if(ge(x,Z)==T) return floor0(x,y);
  push(y);
  return ret(1,pairUnaryMinus(ceiling0(unaryMinus(x),y)));}

pnt ceilingx(pnt x,pnt y){/* ceiling ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(le(y,Z)==T) return X;
  if(ge(x,Z)==T) return ceiling0(x,y);
  push(y);
  return ret(1,pairUnaryMinus(floor0(unaryMinus(x),y)));}

pnt truncatex(pnt x,pnt y){/* truncate ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(le(y,Z)==T) return X;
  if(ge(x,Z)==T) return floor0(x,y);
  push(y);
  return ret(1,pairUnaryMinus(floor0(unaryMinus(x),y)));}

pnt roundx(pnt x,pnt y){/* round ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(le(y,Z)==T) return X;
  if(ge(x,Z)==T) return round0(x,y);
  push(y);
  return ret(1,pairUnaryMinus(round0(unaryMinus(x),y)));}

pnt divx(pnt x,pnt y){/* " div " */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(le(y,Z)==T) return X;
  if(ge(x,Z)==T) return HEAD(floor0(x,y));
  push(y);
  return ret(1,unaryMinus(HEAD(ceiling0(unaryMinus(x),y))));}

pnt modx(pnt x,pnt y){/* " mod " */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(le(y,Z)==T) return X;
  if(ge(x,Z)==T) return TAIL(floor0(x,y));
  push(y);
  return ret(1,unaryMinus(TAIL(ceiling0(unaryMinus(x),y))));}



pnt raise0(pnt x){/* " raise */
  return cons(TAG_EX,x,NIL);}

pnt object(pnt x){/* object ( " ) */
  pnt tag,ref,idx,val,cmp;
  if(NO_PAIR(x)) return X;
  tag=HEAD(x);
  val=TAIL(x);
  if(NO_PAIR(tag)) return X;
  ref=HEAD(tag);
  idx=TAIL(tag);
  if(NO_INT(ref)) return X;
  if(NO_INT(idx)) return X;
  cmp=cmp1(ref,Z);
  if(cmp==0) return X;
  if(T==lt(idx,I[(cmp==1)?5:0])) return X;
  return cons(TAG_OBJ,tag,val);}

pnt destruct(pnt x){/* destruct ( " ) */
  if(NO_OBJ(x)) return X;
  return pair(HEAD(x),TAIL(x));}

pnt vector2(pnt x){
  if(x<256) return 1;
  return (x&0xFF)|(vector2(x>>8)<<8);}

pnt vector1(pnt x){
  pnt u=HEAD(x);
  pnt y=TAIL(x);
  if(y==NIL||y==Z) return consvec(vector2(u),NIL);
  return consvec(u,vector1(y));}

pnt vector(pnt x){/* vector ( " ) */
  if(x==T) return X;
  switch(ROOT(x)){
  case TAG_VECT: return x;
  case TAG_INT:
    if(le(x,Z)==T) return E;
    return vector1(x);
  default: return X;}}

pnt endlength(pnt x){
  if(TAIL(x)) return endlength(TAIL(x));
  return integerLength3(HEAD(x));}

pnt vectorNorm(pnt x){/* vector-norm ( " ) */
  if(x==T) return X;
  switch(ROOT(x)){
  case TAG_VECT: return x;
  case TAG_INT:
    if(le(x,Z)==T) return T;
    if(endlength(x)%8!=1) return T;
    return vector1(x);
  default: return X;}}

pnt vectorEmpty(pnt x){/* vector-empty ( " ) */
  return le(x,I[255]);}

pnt vectorLength2(pnt x){
  if(x<256) return 0;
  return 1+vectorLength2(x>>8);}

pnt vectorLength1(pnt x){
  pnt y=TAIL(x);
  if((!y)||(y==Z)) return vectorLength2(HEAD(x));
  return bytesize+vectorLength1(y);}

pnt negative(pnt x){
  if(ROOT(x)==TAG_VECT) return FALSE;
  for(;TAIL(x);x=TAIL(x));
  return (HEAD(x)&signmask)!=0;}

pnt vectorLength(pnt x){/* vector-length ( " ) */
  if(NO_INT(x)) return X;
  if(negative(x)) return Z;
  return intcons(vectorLength1(x),Z);}

pnt vectorIndex2(pnt x,pnt y){
  if(y==0) return x&0xFF;
  return vectorIndex2(x>>8,y-1);}

pnt vectorIndex1(pnt x,pnt y){
  if(y<bytesize) return vectorIndex2(HEAD(x),y);
  return vectorIndex1(TAIL(x),y-bytesize);}

pnt vectorIndex(pnt x,pnt y){/* vector-index ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(negative(x)) return X;
  if(inttail(y)!=Z) return X;
  if(HEAD(y)>=vectorLength1(x)) return X;
  return intcons(vectorIndex1(x,HEAD(y)),Z);}

pnt consvec1(pnt x,pnt y){
  if(y) return consvec(x,y);
  if(x==0) return NIL;
  return consvec(vector2(x),NIL);}

pnt vectorSuffix2(pnt x,pnt y,pnt z){
  pnt u=HEAD(x);
  pnt v=TAIL(x);
  if(v) return consvec1((u>>y)|(HEAD(v)<<z),vectorSuffix2(v,y,z));
  return consvec1(u>>y,NIL);}

pnt vectorSuffix1(pnt x,pnt y){
  if(y==0) return vector(x);
  if(y<bytesize) return vectorSuffix2(x,y*8,pntsize-y*8);
  return vectorSuffix1(TAIL(x),y-bytesize);}

pnt vectorSuffix(pnt x,pnt y){/* vector-suffix ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(negative(y)) return vector(x);
  if(cmp1(vectorLength(x),y)<=1) return E;
  if(inttail(y)!=Z) die("vectorSuffix overflow");
  return vectorSuffix1(x,HEAD(y));}

pnt vectorPrefix2(pnt x,pnt y){
  if(y==0) return 1;
  return (x&0xFF)|(vectorPrefix2(x>>8,y-1)<<8);}

pnt vectorPrefix1(pnt x,pnt y){
  pnt u=HEAD(x);
  if(y<bytesize) return consvec(vectorPrefix2(u,y),NIL);
  return consvec(u,vectorPrefix1(TAIL(x),y-bytesize));}

pnt vectorPrefix(pnt x,pnt y){/* vector-prefix ( " , " ) */
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(negative(y)) return E;
  if(cmp1(vectorLength(x),y)<=1) return vector(x);
  if(inttail(y)!=Z) die("vectorPrefix overflow");
  return vectorPrefix1(x,HEAD(y));}

pnt vectorSubseq3(pnt x,pnt y1,pnt y2,pnt z){
  pnt u=HEAD(x);
  pnt v=TAIL(x);
  if(v) v=HEAD(v);
  return vectorPrefix2((u>>y1)|(v<<y2),z);}

pnt vectorSubseq2(pnt x,pnt y1,pnt y2,pnt z){
  pnt u,v;
  if(z<bytesize) return consvec(vectorSubseq3(x,y1,y2,z),NIL);
  u=HEAD(x);
  v=TAIL(x);
  return consvec((u>>y1)|(HEAD(v)<<y2),vectorSubseq2(v,y1,y2,z-bytesize));}

pnt vectorSubseq1(pnt x,pnt y,pnt z){
  if(y==0) return vectorPrefix1(x,z);
  if(y<bytesize) return vectorSubseq2(x,y*8,pntsize-y*8,z);
  return vectorSubseq1(TAIL(x),y-bytesize,z);}

pnt vectorSubseq(pnt x,pnt y,pnt z){/* vector-subseq ( " , " , " ) */
  pnt len;
  if(NO_INT(x)) return X;
  if(NO_INT(y)) return X;
  if(NO_INT(z)) return X;
  if(negative(y)) return vectorPrefix(x,z);
  len=vectorLength(x);
  if(cmp1(len,y)<=1) return E;
  if(cmp1(z,y)<=1) return E;
  if(cmp1(len,z)<=1) return vectorSuffix(x,y);
  if(inttail(z)!=Z) die("vectorSubseq overflow");
  return vectorSubseq1(x,HEAD(y),HEAD(z)-HEAD(y));}

pnt vector2bytes3(pnt x,pnt y){
  if(x<=255) return y;
  return pair(I[x&0xFF],vector2bytes3(x>>8,y));}

pnt vector2bytes2(pnt x,pnt y,pnt z){
  if(!y) return z;
  return pair(I[x&0xFF],vector2bytes2(x>>8,y-1,z));}

pnt vector2bytes1(pnt x,pnt y){
  if(!TAIL(x)) return vector2bytes3(HEAD(x),y);
  return vector2bytes2(HEAD(x),bytesize,vector2bytes1(TAIL(x),y));}

pnt vector2bytes(pnt x){/* vector2byte* ( " ) */
  if(NO_INT(x)) return X;
  if(negative(x)) return T;
  return vector2bytes1(x,T);}

pnt vector2vects3(pnt x,pnt y){
  if(x<=255) return y;
  return pair(bytevect[x&0xFF],vector2vects3(x>>8,y));}

pnt vector2vects2(pnt x,pnt y,pnt z){
  if(!y) return z;
  return pair(bytevect[x&0xFF],vector2vects2(x>>8,y-1,z));}

pnt vector2vects1(pnt x,pnt y){
  if(!TAIL(x)) return vector2vects3(HEAD(x),y);
  return vector2vects2(HEAD(x),bytesize,vector2vects1(TAIL(x),y));}

pnt vector2vects(pnt x){/* vector2vector* ( " ) */
  if(NO_INT(x)) return X;
  if(negative(x)) return T;
  return vector2vects1(x,T);}

pnt bt2bytes1(pnt x,pnt y){
  if(x==T) return y;
  switch(ROOT(x)){
  case TAG_FALSE:
  case TAG_MAP:
  case TAG_OBJ:
    return y;
  case TAG_INT:
  case TAG_VECT:
    if(TAIL(x)) return y;
    if(HEAD(x)>255) return y;
    return pair(x,y);
  case TAG_PAIR:
    push(x);
    return ret(1,bt2bytes1(HEAD(x),bt2bytes1(TAIL(x),y)));
  default:
    unexpected_tag("bt2bytes1",x);
    return T; /* This never happens */}}

pnt bt2bytes(pnt x){/* bt2byte* ( " ) */
  return bt2bytes1(x,T);}

pnt bt2vects1(pnt x,pnt y){
  if(x==T) return y;
  switch(ROOT(x)){
  case TAG_FALSE:
  case TAG_MAP:
  case TAG_OBJ:
    return y;
  case TAG_INT:
  case TAG_VECT:
    if(TAIL(x)) return y;
    if(HEAD(x)>255) return y;
    return pair(bytevect[HEAD(x)],y);
  case TAG_PAIR:
    push(x);
    return ret(1,bt2vects1(HEAD(x),bt2vects1(TAIL(x),y)));
  default:
    unexpected_tag("bt2vects1",x);
    return T; /* This never happens */}}

pnt bt2vects(pnt x){/* bt2vector* ( " ) */
  return bt2vects1(x,T);}

pnt vt2bytes1(pnt x,pnt y){
  if(x==T) return y;
  switch(ROOT(x)){
  case TAG_FALSE:
  case TAG_MAP:
  case TAG_OBJ:
    return y;
  case TAG_INT:
  case TAG_VECT:
    if(negative(x)) return y;
    return vector2bytes1(x,y);
  case TAG_PAIR:
    push(x);
    return ret(1,vt2bytes1(HEAD(x),vt2bytes1(TAIL(x),y)));
  default:
    unexpected_tag("bt2bytes1",x);
    return T; /* This never happens */}}

pnt vt2bytes(pnt x){/* vt2byte* ( " ) */
  return vt2bytes1(x,T);}

pnt vt2vects1(pnt x,pnt y){
  if(x==T) return y;
  switch(ROOT(x)){
  case TAG_FALSE:
  case TAG_MAP:
  case TAG_OBJ:
    return y;
  case TAG_INT:
  case TAG_VECT:
    if(negative(x)) return y;
    return vector2vects1(x,y);
  case TAG_PAIR:
    push(x);
    return ret(1,vt2vects1(HEAD(x),vt2vects1(TAIL(x),y)));
  default:
    unexpected_tag("vt2vects1",x);
    return T; /* This never happens */}}

pnt vt2vects(pnt x){/* vt2vector* ( " ) */
  return vt2vects1(x,T);}

pnt bytes2vector(pnt n,pnt x){
  pnt u,y;
  if(x==T) return consvec(1,NIL);
  u=HEAD(HEAD(x));
  if(n==bytesize) return consvec(u,bytes2vector(1,TAIL(x)));
  y=bytes2vector(n+1,TAIL(x));
  HEAD(y)=(HEAD(y)<<8)|u;
  return y;}

pnt bt2vector(pnt x){/* bt2vector ( " ) */
  return bytes2vector(1,bt2bytes(x));}

pnt vt2vector(pnt x){/* vt2vector ( " ) */
  return bytes2vector(1,vt2bytes(x));}

pnt spy(pnt x){/* spy ( " ) */
  spyvar=x;
  return const_spy;}

pnt trace(pnt x){/* trace ( " ) */
  spy0(x);
  return const_trace;}

pnt print(pnt x){/* print ( " ) */
  print1(x);
  return const_print;}

pnt timer(pnt x){/* timer ( " ) */
  if(x==T) setTimer(-1);
  else if(ROOT(x)!=TAG_INT&&ROOT(x)!=TAG_VECT) setTimer(-1);
  else if(TAIL(x)!=NIL) setTimer(-1);
  else if(HEAD(x)>TIMERS) setTimer(-1);
  else setTimer(HEAD(x));
  return const_timer;}



/*********************
 * GENERAL FUNCTIONS *
 *********************/

pnt term_apply(pnt fct,pnt arg){
  return cons(TTAG_APPLY,fct,arg);}

pnt term_lambda(pnt body){
  return cons(TTAG_LAMBDA,body,NIL);}

pnt term_lambdas(pnt arity,pnt body){
  if(arity==0) return body;
  return term_lambda(term_lambdas(arity-1,body));}

pnt term_call(pnt fct,pnt arg){
  return cons(TTAG_CALL,fct,arg);}

pnt term_ecall(pnt fct,pnt arg){
  return cons(TTAG_ECALL,fct,arg);}

pnt term_var(pnt index){
  return cons2(TTAG_VAR,index,NIL);}

pnt term_pair(pnt x,pnt y){
  return cons(TTAG_PAIR,x,y);}

pnt term_const(pnt x){
  return cons(TTAG_CONST,x,NIL);}

pnt closure(pnt term,pnt env){
  return cons(MTAG_CLOSURE,term,env);}

pnt overwrite(pnt closure,pnt rnf){
  ROOT(closure)=MTAG_INDIR;
  TAIL(closure)=rnf;
  return rnf;}

pnt map_pair(pnt x,pnt y){
  return cons(MTAG_PAIR,x,y);}

pnt map_lambda(pnt term,pnt env){
  return cons(MTAG_LAMBDA,term,env);}

pnt bit2map(pnt x){
  if(x&1) return map_f; else return map_t;}

pnt small2rnf(pnt card){
  if(card==0) return T;
  return map_pair(bit2map(card),small2rnf(card>>1));}

pnt card2rnf1(pnt card,pnt cnt,pnt result){
  if(cnt==0) return result;
  return map_pair(bit2map(card),card2rnf1(card>>1,cnt-1,result));}

pnt card2rnf(pnt card){
  pnt tmp1;
  if(TAIL(card)==T) return small2rnf(HEAD(card));
  tmp1=HEAD(card);
  return card2rnf1(tmp1,pntsize,card2rnf(TAIL(card)));}

pnt map2rnf(pnt map){
  if(map==T) return T;
  switch(ROOT(map)){
  case MTAG_INDIR: return TAIL(map);
  case MTAG_CLOSURE:
    push(map);
    return ret(1,overwrite(map,term2rnf(HEAD(map),TAIL(map))));
  case MTAG_PAIR:
  case MTAG_LAMBDA:
  case MTAG_FIX:
  case TAG_FALSE:
  case TAG_INT:
  case TAG_VECT:
  case TAG_PAIR:
  case TAG_EX:
  case TAG_MAP:
  case TAG_OBJ:
    return map;
  default:
    unexpected_tag("map2rnf",map);
    return T; /* This never happens */}}

pnt fixpoint(pnt map){
  pnt env=map_pair(T,map_pair(map,T));
  pnt result=closure(term_fix,env);
  HEAD(env)=result;
  return map2rnf(result);}

void spyenv(pnt env){for(;env;env=TAIL(env)) spy0(HEAD(env));}

pnt terms2closures(pnt terms,pnt env){
  pnt closure0;
  if(terms==NIL) return NIL;
  closure0=closure(HEAD(terms),env);
  push(closure0);
  return ret(1,map_pair(closure0,terms2closures(TAIL(terms),env)));}

pnt terms2tagged(pnt terms,pnt env){
  pnt tagged1,tagged2;
  if(terms==NIL) return NIL;
  tagged2=terms2tagged(TAIL(terms),env);
  if(IS_EX(tagged2)) return tagged2;
  push(tagged2);
  tagged1=untag1(term2rnf(HEAD(terms),env));
  if(IS_EX(tagged1)) return ret(1,tagged1);
  return ret(1,pair(tagged1,tagged2));}

pnt map_apply(pnt map,pnt arg){
  pnt rnf;
  push(arg);
  push(rnf=map2rnf(map));
  if(rnf==T) return ret(2,rnf);
  switch(ROOT(rnf)){
  case MTAG_PAIR:
    if(map2rnf(arg)==T) return ret(2,map2rnf(HEAD(rnf)));
    return ret(2,map2rnf(TAIL(rnf)));
  case MTAG_LAMBDA:
    return ret(2,term2rnf(HEAD(rnf),map_pair(arg,TAIL(rnf))));
  case MTAG_FIX:
    return ret(2,fixpoint(arg));
  case TAG_FALSE:
    map2rnf(arg); /* ensure infinite looping when appropriate */
    return ret(2,T);
  case TAG_INT:
  case TAG_VECT:
    if(T==map2rnf(arg)) return ret(2,maptag_int);
    if(negative(rnf)) return ret(2,map_pair(map_f,card2rnf(unaryMinus(rnf))));
    return ret(2,map_pair(map_t,card2rnf(rnf)));
  case TAG_PAIR:
    if(T==map2rnf(arg)) return ret(2,maptag_pair);
    return ret(2,map_pair(HEAD(rnf),TAIL(rnf)));
  case TAG_EX:
    if(T==map2rnf(arg)) return ret(2,maptag_ex);
    return ret(2,HEAD(rnf));
  case TAG_MAP:
    if(T==map2rnf(arg)) return ret(2,maptag_map);
    return ret(2,map2rnf(HEAD(rnf)));
  case TAG_OBJ:
    if(map2rnf(arg)!=T) return ret(2,TAIL(rnf));
    return ret(2,
    MAP_PAIR(card2rnf(HEAD(HEAD(rnf))),card2rnf(TAIL(HEAD(rnf)))));
  default:
    unexpected_tag("map_apply",rnf);
    return T; /* This never happens */}}

pnt term2rnf(pnt term,pnt env){
  push(term);
  push(env);
  switch(ROOT(term)){
  case TTAG_APPLY:{
    pnt rnf;
    rnf=term2rnf(HEAD(term),env);
    if(rnf==T) return ret(2,T);
    push(rnf);
    switch(ROOT(rnf)){
    case MTAG_PAIR:
      if(T==term2rnf(TAIL(term),env)) return ret(3,map2rnf(HEAD(rnf)));
      return ret(3,map2rnf(TAIL(rnf)));
    case MTAG_LAMBDA:
      return ret(3,
      term2rnf(HEAD(rnf),map_pair(closure(TAIL(term),env),TAIL(rnf))));
    case MTAG_FIX:
      return ret(3,fixpoint(term2rnf(TAIL(term),env)));
    case TAG_FALSE:
      term2rnf(TAIL(term),env); /* ensure infinite looping when appropriate */
      return ret(3,NIL);
    case TAG_INT:
    case TAG_VECT:
      if(T==term2rnf(TAIL(term),env)) return ret(3,maptag_int);
      if(negative(rnf)) return ret(3,map_pair(map_f,card2rnf(unaryMinus(rnf))));
      return ret(3,map_pair(map_t,card2rnf(rnf)));
    case TAG_PAIR:
      if(T==term2rnf(TAIL(term),env)) return ret(3,maptag_pair);
      return ret(3,map_pair(HEAD(rnf),TAIL(rnf)));
    case TAG_EX:
      if(T==term2rnf(TAIL(term),env)) return ret(3,maptag_ex);
      return ret(3,HEAD(rnf));
    case TAG_MAP:
      if(T==term2rnf(TAIL(term),env)) return ret(3,maptag_map);
      return ret(3,map2rnf(HEAD(rnf)));
    case TAG_OBJ:
      if(term2rnf(TAIL(term),env)) return ret(3,TAIL(rnf));
      return ret(3,
      MAP_PAIR(card2rnf(HEAD(HEAD(rnf))),card2rnf(TAIL(HEAD(rnf)))));
    default:
      unexpected_tag("term2rnf/apply",rnf);}}
  case TTAG_LAMBDA:
    return ret(2,map_lambda(HEAD(term),env));
  case TTAG_CALL:{
    /* current version allows call of n-ary CLOSED terms */
    return ret(2,term2rnf(HEAD(term),terms2closures(TAIL(term),env)));}
  case TTAG_ECALL:{
    pnt args;
    /* current version allows call of n-ary CLOSED terms */
    args=terms2tagged(TAIL(term),env);
    if(IS_EX(args)) return ret(2,args);
    push(args);
    return ret(3,eval(HEAD(term),args));}
  case TTAG_VAR:{
    pnt i;
    pnt n=HEAD(term);
    for(i=0;i<n;i++){env=TAIL(env);if(T==env) return ret(2,env);}
    return ret(2,map2rnf(HEAD(env)));}
  case TTAG_PAIR:
    return ret(2,MAP_PAIR(closure(HEAD(term),env),closure(TAIL(term),env)));
  case TTAG_CONST:
    return ret(2,map2rnf(HEAD(term)));
  default:
    unexpected_tag("term2rnf",term);
    return T; /* This never happens */}}



/****************************
 * Tagged general functions *
 ****************************/

pnt delay_apply(pnt fct,pnt arg){
  push(fct);
  return ret(1,closure(term_delay,pair(fct,pair(arg,NIL))));}

pnt map_head(pnt rnf){
  return map_apply(rnf,map_t);}

pnt map_tail(pnt rnf){
  return map_apply(rnf,map_f);}

pnt mapcons2(pnt tag,pnt x){
  return cons1(TAG_MAP,x,tag);}

pnt mapcons(pnt x){
  return mapcons2(CTAG_MAP,x);}

pnt maptag(pnt x){/* " maptag */
  return mapcons(x);}

pnt rnf2card(pnt rnf){
  pnt val=0;
  pnt pos;
  for(pos=0;pos<pntsize;pos++){
    if(rnf==T) return intcons(val,Z);
    if(map_head(rnf)!=T) val|=((pnt)1)<<pos;
    rnf=map_tail(rnf);}
  return intcons(val,rnf2card(rnf));}

pnt rnf2int(pnt x){
  if(map_head(x)==T) return rnf2card(map_tail(x));
  return unaryMinus(rnf2card(map_tail(x)));}

pnt objcons(pnt x,pnt y,pnt z){
  return cons(TAG_OBJ,pair(x,y),z);}

pnt untag1(pnt x){
  pnt tag,val,ref,idx;
  if(x==T) return T;
  switch(ROOT(x)){
  case TAG_FALSE:
  case TAG_INT:
  case TAG_VECT:
  case TAG_PAIR:
  case TAG_EX:
  case TAG_MAP:
  case TAG_OBJ:
    return x;
  case MTAG_CLOSURE:
  case MTAG_INDIR:
  case MTAG_PAIR:
  case MTAG_LAMBDA:
  case MTAG_FIX:
    break;
  case TAG_GARB:
    die("untag1: Attempt to reduce garbage");
  case ETAG_bottom:
    bottom();
  default:
    unexpected_tag("untag1",x);}
  x=map2rnf(x);
  push(x);
  push(tag=map_head(x));
  push(ref=rnf2card(map_head(tag)));
  push(idx=rnf2card(map_tail(tag)));
  if(ref!=Z) return ret(4,objcons(ref,idx,untag1(map_tail(x))));
  if(TAIL(idx)) return ret(4,objcons(ref,idx,untag1(map_tail(x))));
  switch(HEAD(idx)){
  case 0: if(x==T) return ret(4,T); else return ret(4,F);
  case 1: return ret(4,rnf2int(map_tail(x)));
  case 2:
    push(val=map_tail(x));
    push(untag1(map_head(val)));
    push(untag1(map_tail(val)));
    return ret(7,pair(stack[sp+1],stack[sp]));
  case 3: return ret(4,raise0(untag1(map_tail(x))));
  case 4: return ret(4,mapcons(delay_apply(x,map_f)));
  default: return ret(4,objcons(ref,idx,untag1(map_tail(x))));}}

pnt untag(pnt x){/* " untag */
  if(NO_MAP(x)) return X;
  return untag1(HEAD(x));}

pnt apply(pnt x,pnt y){/* " apply " */
  if(NO_MAP(x)) return X;
  if(NO_MAP(y)) return X;
  return mapcons(delay_apply(HEAD(x),HEAD(y)));}

pnt root(pnt x){/* " root */
  if(NO_MAP(x)) return X;
  if(map2rnf(HEAD(x))==T) return T;
  return F;}



/********************
 * Eager evaluation *
 ********************/

#define STACK0(tag,fct) \
  case tag: \
    result=fct(); \
    goto unstack;

#define STACK1(tag) \
  case tag: \
    push(I[tag]); \
    term=TAIL(term); \
    goto resume;
#define UNSTACK1(tag,fct) \
  case tag: \
    if(IS_EX(result)) goto unstack; \
    result=fct(result); \
    goto unstack;

#define STACK2(tag) \
  case tag: \
    push(TAIL(term)); \
    push(I[tag]); \
    term=HEAD(term); \
    goto resume;
#define UNSTACK2(tag,fct) \
  case tag: \
    term=pop(); \
    if(IS_EX(result)) goto unstack; \
    push(result); \
    push(bytevect[tag]); \
    goto resume; \
  case tag+256:  \
    term=pop(); \
    if(IS_EX(result)) goto unstack; \
    result=fct(term,result); \
    goto unstack;

#define STOP 0
#define ETAG_vsubseq2 1

pnt eval(pnt term,pnt env){
  pnt result=T;
  pnt cnt;
  push(I[STOP]);
resume:
  if(term==T){result=T;goto unstack;}
  switch(ROOT(term)){
  case TAG_FALSE:
  case TAG_INT:
  case TAG_VECT:
  case TAG_PAIR:
  case TAG_EX:
  case TAG_MAP:
  case TAG_OBJ:
    result=term;
    goto unstack;
  case ETAG_var:
    result=env;
    for(cnt=HEAD(term);cnt>0;cnt--) result=TAIL(result);
    result=HEAD(result);
    goto unstack;
  case ETAG_ecall:
    push(env);
    push(HEAD(term));
    term=TAIL(term);
    if(term==T){
      env=T;
      term=pop();
      push(env);
      push(bytevect[ETAG_ecall]);
      goto resume;}
    push(TAIL(term));
    push(T);
    push(I[ETAG_ecall]);
    term=HEAD(term);
    goto resume;
  STACK1(ETAG_digit0       )
  STACK1(ETAG_digit1       )
  STACK1(ETAG_digit2       )
  STACK1(ETAG_digit3       )
  STACK1(ETAG_digit4       )
  STACK1(ETAG_digit5       )
  STACK1(ETAG_digit6       )
  STACK1(ETAG_digit7       )
  STACK1(ETAG_digit8       )
  STACK1(ETAG_digit9       )
  STACK2(ETAG_times        )
  STACK2(ETAG_plus         )
  STACK2(ETAG_minus        )
  STACK2(ETAG_and          )
  STACK2(ETAG_or           )
  STACK2(ETAG_then         )
  STACK2(ETAG_pair         )
  STACK2(ETAG_lt           )
  STACK2(ETAG_le           )
  STACK2(ETAG_equal        )
  STACK2(ETAG_gt           )
  STACK2(ETAG_ge           )
  STACK2(ETAG_apply        )
  STACK1(ETAG_boolp        )
  STACK1(ETAG_catch        )
  STACK1(ETAG_maptag1      )
  STACK2(ETAG_div          )
  STACK1(ETAG_head         )
  STACK1(ETAG_intp         )
  STACK2(ETAG_boolg        )
  STACK2(ETAG_intg         )
  STACK2(ETAG_mapg         )
  STACK2(ETAG_objg         )
  STACK2(ETAG_pairg        )
  STACK1(ETAG_mapp         )
  STACK1(ETAG_maptag       )
  STACK2(ETAG_mod          )
  STACK1(ETAG_norm         )
  STACK1(ETAG_objp         )
  STACK1(ETAG_pairp        )
  STACK1(ETAG_raise        )
  STACK1(ETAG_root         )
  STACK1(ETAG_tail         )
  STACK1(ETAG_untag        )
  STACK0(ETAG_digitend     ,zero         )
  STACK1(ETAG_par          )
  STACK1(ETAG_uplus        )
  STACK1(ETAG_dplus        )
  STACK1(ETAG_uminus       )
  STACK1(ETAG_dminus       )
  STACK1(ETAG_not          )
  STACK0(ETAG_Base         ,base         )
  STACK2(ETAG_LET          )
  STACK2(ETAG_ash          )
  STACK0(ETAG_bottom       ,bottom       )
  STACK1(ETAG_bt2bytes     )
  STACK1(ETAG_bt2vects     )
  STACK1(ETAG_bt2vector    )
  STACK2(ETAG_ceiling      )
  STACK1(ETAG_destruct     )
  STACK1(ETAG_evenp        )
  STACK0(ETAG_exception    ,exception    )
  STACK0(ETAG_false        ,false        )
  STACK2(ETAG_floor        )
  STACK1(ETAG_half         )
  case ETAG_if:
    push(TAIL(term));
    push(I[ETAG_if]);
    term=HEAD(term);
    goto resume;
  STACK1(ETAG_intlength    )
  STACK2(ETAG_logand       )
  STACK2(ETAG_logandc1     )
  STACK2(ETAG_logandc2     )
  STACK2(ETAG_logbitp      )
  STACK1(ETAG_logcount     )
  STACK2(ETAG_logeqv       )
  STACK2(ETAG_logior       )
  STACK2(ETAG_lognand      )
  STACK2(ETAG_lognor       )
  STACK1(ETAG_lognot       )
  STACK2(ETAG_logorc1      )
  STACK2(ETAG_logorc2      )
  STACK2(ETAG_logtest      )
  STACK2(ETAG_logxor       )
  case ETAG_map:
    result=mapcons(TAIL(term));
    goto unstack;
  STACK1(ETAG_prenorm      )
  STACK1(ETAG_notnot       )
  STACK1(ETAG_object       )
  STACK1(ETAG_print        )
  STACK2(ETAG_round        )
  STACK1(ETAG_spy          )
  STACK1(ETAG_timer        )
  STACK1(ETAG_trace        )
  STACK2(ETAG_truncate     )
  STACK1(ETAG_vector       )
  STACK1(ETAG_vnorm        )
  STACK1(ETAG_vempty       )
  STACK2(ETAG_vindex       )
  STACK1(ETAG_vlength      )
  STACK2(ETAG_vprefix      )
  STACK2(ETAG_vsubseq      ) /* STACK2 also applicable for arity 3 */
  STACK2(ETAG_vsuffix      )
  STACK1(ETAG_v2bytes      )
  STACK1(ETAG_v2vects      )
  STACK1(ETAG_vt2bytes     )
  STACK1(ETAG_vt2vects     )
  STACK1(ETAG_vt2v         )
  STACK1(ETAG_compile      )
  STACK1(ETAG_ripemd       )
  STACK1(ETAG_sl2rack      )
  STACK1(ETAG_rack2sl      )
/* GRD-2009-04-19: Ensure eager constants evaluated once only. */
  case MTAG_INDIR:
    result=TAIL(term);
    goto unstack;
  case MTAG_CLOSURE:
    push(term);
    result=overwrite(term,term2rnf(HEAD(term),TAIL(term)));
    pop();
    goto unstack;
  default: unexpected_tag("eval",term);}
unstack:
  switch(HEAD(pop())){
  case STOP:
    return result;
  case ETAG_ecall:
    if(IS_EX(result)){sp+=4;goto unstack;}
    result=pair(result,pop());
    term=pop();
    if(term==T){
      env=result;
      term=pop();
      push(env);
      push(bytevect[ETAG_ecall]);
      goto resume;}
    push(TAIL(term));
    push(result);
    push(I[ETAG_ecall]);
    term=HEAD(term);
    goto resume;
  case ETAG_ecall+256:
    pop();
    env=pop();
    goto unstack;
  UNSTACK1(ETAG_digit0   ,digit0       )
  UNSTACK1(ETAG_digit1   ,digit1       )
  UNSTACK1(ETAG_digit2   ,digit2       )
  UNSTACK1(ETAG_digit3   ,digit3       )
  UNSTACK1(ETAG_digit4   ,digit4       )
  UNSTACK1(ETAG_digit5   ,digit5       )
  UNSTACK1(ETAG_digit6   ,digit6       )
  UNSTACK1(ETAG_digit7   ,digit7       )
  UNSTACK1(ETAG_digit8   ,digit8       )
  UNSTACK1(ETAG_digit9   ,digit9       )
  UNSTACK2(ETAG_times    ,times        )
  UNSTACK2(ETAG_plus     ,plus         )
  UNSTACK2(ETAG_minus    ,minus        )
  case ETAG_and:
    term=pop();
    if(result!=T) goto unstack;
    goto resume;
  case ETAG_or:
    term=pop();
    if(result==T) goto unstack;
    if(IS_EX(result)) goto unstack;
    goto resume;
  UNSTACK2(ETAG_then     ,then         )
  UNSTACK2(ETAG_pair     ,pair         )
  UNSTACK2(ETAG_lt       ,lt           )
  UNSTACK2(ETAG_le       ,le           )
  UNSTACK2(ETAG_equal    ,eq           )
  UNSTACK2(ETAG_gt       ,gt           )
  UNSTACK2(ETAG_ge       ,ge           )
  UNSTACK2(ETAG_apply    ,apply        )
  UNSTACK1(ETAG_boolp    ,boolp        )
  case ETAG_catch:
    result=IS_EX(result)?pair(T,HEAD(result)):pair(F,result);
    goto unstack;
  case ETAG_maptag1:
    result=mapcons(result);
    goto unstack;
  UNSTACK2(ETAG_div      ,divx         )
  UNSTACK1(ETAG_head     ,head         )
  UNSTACK1(ETAG_intp     ,intp         )
  UNSTACK2(ETAG_boolg    ,boolg        )
  UNSTACK2(ETAG_intg     ,intg         )
  UNSTACK2(ETAG_mapg     ,mapg         )
  UNSTACK2(ETAG_objg     ,objg         )
  UNSTACK2(ETAG_pairg    ,pairg        )
  UNSTACK1(ETAG_mapp     ,mapp         )
  UNSTACK1(ETAG_maptag   ,mapcons      )
  UNSTACK2(ETAG_mod      ,modx         )
  UNSTACK1(ETAG_norm     ,norm         )
  UNSTACK1(ETAG_objp     ,objp         )
  UNSTACK1(ETAG_pairp    ,pairp        )
  UNSTACK1(ETAG_raise    ,raise0       )
  UNSTACK1(ETAG_root     ,root         )
  UNSTACK1(ETAG_tail     ,tail         )
  UNSTACK1(ETAG_untag    ,untag        )
  UNSTACK1(ETAG_par      ,norm         )
  UNSTACK1(ETAG_uplus    ,unaryPlus    )
  UNSTACK1(ETAG_dplus    ,unaryPlus    )
  UNSTACK1(ETAG_uminus   ,unaryMinus   )
  UNSTACK1(ETAG_dminus   ,unaryMinus   )
  UNSTACK1(ETAG_not      ,not          )
  case ETAG_LET:
    term=pop();
    if(IS_EX(result)) goto unstack;
    env=pair(result,env);
    push(env);
    push(bytevect[ETAG_LET]);
    goto resume;
  case ETAG_LET+256:
    env=TAIL(pop());
    goto unstack;
  UNSTACK2(ETAG_ash      ,ash          )
  UNSTACK1(ETAG_bt2bytes ,bt2bytes     )
  UNSTACK1(ETAG_bt2vects ,bt2vects     )
  UNSTACK1(ETAG_bt2vector,bt2vector    )
  UNSTACK2(ETAG_ceiling  ,ceilingx     )
  UNSTACK1(ETAG_destruct ,destruct     )
  UNSTACK1(ETAG_evenp    ,evenp        )
  UNSTACK2(ETAG_floor    ,floorx       )
  UNSTACK1(ETAG_half     ,half         )
  case ETAG_if:
    if(result==T){term=HEAD(pop());goto resume;}
    if(IS_EX(result)){pop();goto unstack;}
    term=TAIL(pop());goto resume;
  UNSTACK1(ETAG_intlength,integerLength)
  UNSTACK2(ETAG_logand   ,logand       )
  UNSTACK2(ETAG_logandc1 ,logandc1     )
  UNSTACK2(ETAG_logandc2 ,logandc2     )
  UNSTACK2(ETAG_logbitp  ,logbitp      )
  UNSTACK1(ETAG_logcount ,logcount     )
  UNSTACK2(ETAG_logeqv   ,logeqv       )
  UNSTACK2(ETAG_logior   ,logior       )
  UNSTACK2(ETAG_lognand  ,lognand      )
  UNSTACK2(ETAG_lognor   ,lognor       )
  UNSTACK1(ETAG_lognot   ,lognot       )
  UNSTACK2(ETAG_logorc1  ,logorc1      )
  UNSTACK2(ETAG_logorc2  ,logorc2      )
  UNSTACK2(ETAG_logtest  ,logtest      )
  UNSTACK2(ETAG_logxor   ,logxor       )
  UNSTACK1(ETAG_prenorm  ,norm         )
  UNSTACK1(ETAG_notnot   ,notnot       )
  UNSTACK1(ETAG_object   ,object       )
  UNSTACK1(ETAG_print    ,print        )
  UNSTACK2(ETAG_round    ,roundx       )
  UNSTACK1(ETAG_spy      ,spy          )
  UNSTACK1(ETAG_timer    ,timer        )
  UNSTACK1(ETAG_trace    ,trace        )
  UNSTACK2(ETAG_truncate ,truncatex    )
  UNSTACK1(ETAG_vector   ,vector       )
  UNSTACK1(ETAG_vnorm    ,vectorNorm   )
  UNSTACK1(ETAG_vempty   ,vectorEmpty  )
  UNSTACK2(ETAG_vindex   ,vectorIndex  )
  UNSTACK1(ETAG_vlength  ,vectorLength )
  UNSTACK2(ETAG_vprefix  ,vectorPrefix )
  case ETAG_vsubseq:
    term=pop();
    if(IS_EX(result)) goto unstack;
    push(result);
    push(TAIL(term));
    push(bytevect[ETAG_vsubseq]);
    term=HEAD(term);
    goto resume;
  case ETAG_vsubseq+256:
    term=pop();
    if(IS_EX(result)){pop();goto unstack;}
    push(result);
    push(I[ETAG_vsubseq2]);
    goto resume;
  case ETAG_vsubseq2:
    term=pop();
    cnt=pop();
    if(IS_EX(result)) goto unstack;
    result=vectorSubseq(cnt,term,result);
    goto unstack;
  UNSTACK2(ETAG_vsuffix  ,vectorSuffix )
  UNSTACK1(ETAG_v2bytes  ,vector2bytes )
  UNSTACK1(ETAG_v2vects  ,vector2vects )
  UNSTACK1(ETAG_vt2bytes ,vt2bytes     )
  UNSTACK1(ETAG_vt2vects ,vt2vects     )
  UNSTACK1(ETAG_vt2v     ,vt2vector    )
  UNSTACK1(ETAG_compile  ,compile      )
  UNSTACK1(ETAG_ripemd   ,ripemd       )
  UNSTACK1(ETAG_sl2rack  ,sl2rack      )
  UNSTACK1(ETAG_rack2sl  ,rack2sl      )
  default:
    die("Internal error in eval: stack corrupted");
    return T; /* This never happens */}}

/*
GRD-2009-07-20: The eval() below was replaced by the one above
to save a lot of C-stack. Here, the 'C-stack' is the one used
by C for function calls. The change of eval() does not save
space in the stack[] array. Saving C-stack is useful on Linux
and essential on Cygwin.

pnt R0;

#define EVAL0(fct) \
return fct();

#define EVAL1(fct) \
R0=eval(TAIL(term),env); \
if(IS_EX(R0)) return R0; \
return fct(R0);

#define EVAL2(fct) \
R0=eval(HEAD(term),env); \
if(IS_EX(R0)) return R0; \
push(R0); \
R0=eval(TAIL(term),env); \
if(IS_EX(R0)) return ret(1,R0); \
return ret(1,fct(TOP,R0));

#define EVAL3(fct) \
R0=eval(HEAD(term),env); \
if(IS_EX(R0)) return R0; \
push(R0); \
R0=eval(HEAD(TAIL(term)),env); \
if(IS_EX(R0)) return ret(1,R0); \
push(R0); \
R0=eval(TAIL(TAIL(term)),env); \
if(IS_EX(R0)) return ret(2,R0); \
return ret(2,fct(stack[sp+1],TOP,R0));

pnt eval(pnt term,pnt env){
  pnt u,v;
  if(term==T) return T;
  switch(ROOT(term)){
  case TAG_FALSE:
  case TAG_INT:
  case TAG_VECT:
  case TAG_PAIR:
  case TAG_EX:
  case TAG_MAP:
  case TAG_OBJ:
    return term;
  case ETAG_var      :
    for(R0=HEAD(term);R0>0;R0--) env=TAIL(env);
    return HEAD(env);
  case ETAG_ecall    :
    push(T);
    for(u=TAIL(term);u!=T;u=TAIL(u)){
      v=eval(HEAD(u),env);
      if(IS_EX(v)) return ret(1,v);
      TOP=cons(TAG_PAIR,v,TOP);}
    return ret(1,eval(HEAD(term),TOP));
  case ETAG_digit0   : EVAL1(digit0       )
  case ETAG_digit1   : EVAL1(digit1       )
  case ETAG_digit2   : EVAL1(digit2       )
  case ETAG_digit3   : EVAL1(digit3       )
  case ETAG_digit4   : EVAL1(digit4       )
  case ETAG_digit5   : EVAL1(digit5       )
  case ETAG_digit6   : EVAL1(digit6       )
  case ETAG_digit7   : EVAL1(digit7       )
  case ETAG_digit8   : EVAL1(digit8       )
  case ETAG_digit9   : EVAL1(digit9       )
  case ETAG_times    : EVAL2(times        )
  case ETAG_plus     : EVAL2(plus         )
  case ETAG_minus    : EVAL2(minus        )
  case ETAG_and      :
    R0=eval(HEAD(term),env);
    if(R0!=T) return R0;
    return eval(TAIL(term),env);
  case ETAG_or       :
    R0=eval(HEAD(term),env);
    if(R0==T) return T;
    if(IS_EX(R0)) return R0;
    return eval(TAIL(term),env);
  case ETAG_then     : EVAL2(then         )
  case ETAG_pair     : EVAL2(pair         )
  case ETAG_lt       : EVAL2(lt           )
  case ETAG_le       : EVAL2(le           )
  case ETAG_equal    : EVAL2(eq           )
  case ETAG_gt       : EVAL2(gt           )
  case ETAG_ge       : EVAL2(ge           )
  case ETAG_apply    : EVAL2(apply        )
  case ETAG_boolp    : EVAL1(boolp        )
  case ETAG_catch    :
    R0=eval(TAIL(term),env);
    if(IS_EX(R0)) return pair(T,HEAD(R0));
    return pair(F,R0);
  case ETAG_maptag1  :
    return mapcons(eval(TAIL(term),env));
  case ETAG_div      : EVAL2(divx         )
  case ETAG_head     : EVAL1(head         )
  case ETAG_intp     : EVAL1(intp         )
  case ETAG_boolg    : EVAL2(boolg        )
  case ETAG_intg     : EVAL2(intg         )
  case ETAG_mapg     : EVAL2(mapg         )
  case ETAG_objg     : EVAL2(objg         )
  case ETAG_pairg    : EVAL2(pairg        )
  case ETAG_mapp     : EVAL1(mapp         )
  case ETAG_maptag   :
    R0=eval(TAIL(term),env);
    if(IS_EX(R0)) return R0;
    return mapcons(R0);
  case ETAG_mod      : EVAL2(modx         )
  case ETAG_norm     : EVAL1(norm         )
  case ETAG_objp     : EVAL1(objp         )
  case ETAG_pairp    : EVAL1(pairp        )
  case ETAG_raise    : EVAL1(raise0       )
  case ETAG_root     : EVAL1(root         )
  case ETAG_tail     : EVAL1(tail         )
  case ETAG_untag    : EVAL1(untag        )
  case ETAG_digitend : EVAL0(zero         )
  case ETAG_par      : EVAL1(norm         )
  case ETAG_uplus    : EVAL1(unaryPlus    )
  case ETAG_dplus    : EVAL1(unaryPlus    )
  case ETAG_uminus   : EVAL1(unaryMinus   )
  case ETAG_dminus   : EVAL1(unaryMinus   )
  case ETAG_not      : EVAL1(not          )
  case ETAG_Base     : EVAL0(base         )
  case ETAG_LET      :
    R0=eval(HEAD(term),env);
    if(IS_EX(R0)) return R0;
    R0=pair(R0,env);
    push(R0);
    return ret(1,eval(TAIL(term),R0));
  case ETAG_ash      : EVAL2(ash          )
  case ETAG_bottom   : EVAL0(bottom       )
  case ETAG_bt2bytes : EVAL1(bt2bytes     )
  case ETAG_bt2vects : EVAL1(bt2vects     )
  case ETAG_bt2vector: EVAL1(bt2vector    )
  case ETAG_ceiling  : EVAL2(ceilingx     )
  case ETAG_destruct : EVAL1(destruct     )
  case ETAG_evenp    : EVAL1(evenp        )
  case ETAG_exception: EVAL0(exception    )
  case ETAG_false    : EVAL0(false        )
  case ETAG_floor    : EVAL2(floorx       )
  case ETAG_half     : EVAL1(half         )
  case ETAG_if       :
    R0=eval(HEAD(term),env);
    if(R0==T) return eval(HEAD(TAIL(term)),env);
    if(IS_EX(R0)) return R0;
    return eval(TAIL(TAIL(term)),env);
  case ETAG_intlength: EVAL1(integerLength)
  case ETAG_logand   : EVAL2(logand       )
  case ETAG_logandc1 : EVAL2(logandc1     )
  case ETAG_logandc2 : EVAL2(logandc2     )
  case ETAG_logbitp  : EVAL2(logbitp      )
  case ETAG_logcount : EVAL1(logcount     )
  case ETAG_logeqv   : EVAL2(logeqv       )
  case ETAG_logior   : EVAL2(logior       )
  case ETAG_lognand  : EVAL2(lognand      )
  case ETAG_lognor   : EVAL2(lognor       )
  case ETAG_lognot   : EVAL1(lognot       )
  case ETAG_logorc1  : EVAL2(logorc1      )
  case ETAG_logorc2  : EVAL2(logorc2      )
  case ETAG_logtest  : EVAL2(logtest      )
  case ETAG_logxor   : EVAL2(logxor       )
  case ETAG_map      :
    return mapcons(TAIL(term));
  case ETAG_prenorm  : EVAL1(norm         )
  case ETAG_notnot   : EVAL1(notnot       )
  case ETAG_object   : EVAL1(object       )
  case ETAG_print    : EVAL1(print        )
  case ETAG_round    : EVAL2(roundx       )
  case ETAG_spy      : EVAL1(spy          )
  case ETAG_timer    : EVAL1(timer        )
  case ETAG_trace    : EVAL1(trace        )
  case ETAG_truncate : EVAL2(truncatex    )
  case ETAG_vector   : EVAL1(vector       )
  case ETAG_vnorm    : EVAL1(vectorNorm   )
  case ETAG_vempty   : EVAL1(vectorEmpty  )
  case ETAG_vindex   : EVAL2(vectorIndex  )
  case ETAG_vlength  : EVAL1(vectorLength )
  case ETAG_vprefix  : EVAL2(vectorPrefix )
  case ETAG_vsubseq  : EVAL3(vectorSubseq )
  case ETAG_vsuffix  : EVAL2(vectorSuffix )
  case ETAG_v2bytes  : EVAL1(vector2bytes )
  case ETAG_v2vects  : EVAL1(vector2vects )
  case ETAG_vt2bytes : EVAL1(vt2bytes     )
  case ETAG_vt2vects : EVAL1(vt2vects     )
  case ETAG_vt2v     : EVAL1(vt2vector    )
  case ETAG_compile  : EVAL1(compile      )
  case ETAG_ripemd   : EVAL1(ripemd       )
  case ETAG_sl2rack  : EVAL1(sl2rack      )
  case ETAG_rack2sl  : EVAL1(rack2sl      )
// GRD-2009-04-19: Ensure eager constants evaluated once only.
  case MTAG_INDIR    :
    return TAIL(term);
  case MTAG_CLOSURE:
    push(term);
    return ret(1,overwrite(term,term2rnf(HEAD(term),TAIL(term))));
  default: unexpected_tag("eval",term);}}
*/



/*******************
 * Array functions *
 *******************/

void printdom(pnt a){
  pnt head;
  if(a==T) return;
  if(GD_PAIR(a)) die("Unexpected type in array");
  head=HEAD(a);
  if(!NO_INT(head)){
    spy0(head);
    return;}
  printdom(head);
  printdom(TAIL(a));}

pnt aget(pnt a,pnt i){
  pnt head,i1,i2,bit;
  if(NO_INT(i)) return X;
  i2=i;
  i1=HEAD(i2);
  bit=0;
  for(;;){
    if(a==T) return T;
    if(GD_PAIR(a)) return X;
    head=HEAD(a);
    if(IS_INT(head)){if(eq0(head,i)) return TAIL(a); else return T;}
    if((i1>>bit)&1) a=TAIL(a); else a=head;
    bit++;
    if(bit<pntsize) continue;
    bit=0;
    i2=inttail(i2);
    i1=HEAD(i2);}}

pnt mget1(pnt a,pnt i1){
  return aget(a,i1);}

pnt mget2(pnt a,pnt i1,pnt i2){
  return mget1(aget(a,i1),i2);}

pnt mget3(pnt a,pnt i1,pnt i2,pnt i3){
  return mget2(aget(a,i1),i2,i3);}

pnt mget4(pnt a,pnt i1,pnt i2,pnt i3,pnt i4){
  return mget3(aget(a,i1),i2,i3,i4);}

pnt mget5(pnt a,pnt i1,pnt i2,pnt i3,pnt i4,pnt i5){
  return mget4(aget(a,i1),i2,i3,i4,i5);}

pnt mget6(pnt a,pnt i1,pnt i2,pnt i3,pnt i4,pnt i5,pnt i6){
  return mget5(aget(a,i1),i2,i3,i4,i5,i6);}

pnt mget7(pnt a,pnt i1,pnt i2,pnt i3,pnt i4,pnt i5,pnt i6,pnt i7){
  return mget6(aget(a,i1),i2,i3,i4,i5,i6,i7);}

pnt defget(pnt caches,pnt ref,pnt idx,pnt aspect){
  return mget7(caches,ref,ref,const_codex,ref,idx,Z,aspect);}

pnt codeget(pnt caches,pnt ref,pnt idx){
  return mget4(caches,ref,ref,const_code,idx);}

pnt codexget(pnt caches,pnt ref){
  return mget3(caches,ref,ref,const_codex);}

pnt semipair1(pnt x,pnt y){
  if((x==T)&&(y==T)) return T;
  return pair(x,y);}

pnt semipair2(pnt x,pnt y){
  if(y==T) return T;
  return pair(x,y);}

pnt getbit(pnt x,pnt y){
  if(x>=pntsize) return getbit(x-pntsize,inttail(y));
  return (HEAD(y)>>x)&1;}

pnt aput2(pnt i1,pnt a,pnt i2,pnt v2,pnt b){
  if(getbit(b,i1)==0){
    if(getbit(b,i2)==0) return pair(aput2(i1,a,i2,v2,b+1),T);
    return pair(a,pair(i2,v2));}
  else{
    if(getbit(b,i2)==1) return pair(T,aput2(i1,a,i2,v2,b+1));
    return pair(pair(i2,v2),a);}}

pnt aput1(pnt a,pnt i,pnt v,pnt b){
  pnt head;
  pnt tail;
  if(a==T) return semipair2(i,v);
  if(GD_PAIR(a)) return a;
  head=HEAD(a);
  tail=TAIL(a);
  if(IS_INT(head)){
    if(eq0(head,i)) return semipair2(i,v);
    if(v==T) return a;
    if(tail==T) return pair(i,v);
    return aput2(head,a,i,v,b);}
  if(getbit(b,i)==0) return semipair1(aput1(head,i,v,b+1),tail);
  return semipair1(head,aput1(tail,i,v,b+1));}

pnt aput0(pnt a,pnt i,pnt v){
  if(NO_INT(i)) return a;
  push(a);
  push(i);
  push(v);
  return ret(3,aput1(a,i,v,0));}

pnt mput1(pnt array,pnt value,pnt index1){
  return aput0(array,index1,value);}

pnt mput2(pnt array,pnt value,pnt index1,pnt index2){
  pnt subarray=aget(array,index1);
  push(array);
  push(index1);
  subarray=ret(2,mput1(subarray,value,index2));
  return mput1(array,subarray,index1);}

pnt mput3(pnt array,pnt value,pnt index1,pnt index2,pnt index3){
  pnt subarray=aget(array,index1);
  push(array);
  push(index1);
  subarray=ret(2,mput2(subarray,value,index2,index3));
  return mput1(array,subarray,index1);}

pnt mput4(pnt array,pnt value,pnt index1,pnt index2,pnt index3,pnt index4){
  pnt subarray=aget(array,index1);
  push(array);
  push(index1);
  subarray=ret(2,mput3(subarray,value,index2,index3,index4));
  return mput1(array,subarray,index1);}



/*************
 * Scripting *
 *************/

/*
In scripts, blank lines and lines starting with a hash mark
are ignored.

LF (10) as well as CR (13) are considered as line terminators.
A CRLF is considered as two linebreaks, but nobody will notice
since blank lines are ignored.

Sample scripts:

#!/home/kgr/code/red/red/red.exe script
codex
017451CF6643931035C71796AC493D382EC8357EE9A390D5D6DBCDAA0806
017451CF6643931035C71796AC493D382EC8357EE9A390D5D6DBCDAA0806
0
0
execute
/home/kgr/code/red/lisp/cache
/home/kgr/code/red/lisp/cache1

#!/home/kgr/code/red/red/red.exe script
assoc
017451CF6643931035C71796AC493D382EC8357EE9A390D5D6DBCDAA0806
017451CF6643931035C71796AC493D382EC8357EE9A390D5D6DBCDAA0806
0
0
executables
hello
/home/kgr/code/red/lisp/cache
/home/kgr/code/red/lisp/cache1

#!/home/kgr/code/red/red/red.exe script
execute
017451CF6643931035C71796AC493D382EC8357EE9A390D5D6DBCDAA0806
/home/kgr/code/red/lisp/cache
/home/kgr/code/red/lisp/cache1

#!/home/kgr/code/red/red/red.exe script
executables
017451CF6643931035C71796AC493D382EC8357EE9A390D5D6DBCDAA0806
hello
/home/kgr/code/red/lisp/cache
/home/kgr/code/red/lisp/cache1

*/

unsigned char *mbufstart;
unsigned char *mbufpnt;
unsigned char *mbufend;
off_t mbufsize;

void mmapdie(char *caller,char *filename,char *fct){
  printf("%s trying to mmap %s:\n",caller,filename);
  pdie(fct);}

int mmap1(char *caller,char *filename){
  int fd;
  struct stat stats;
  fd=open(filename,O_RDONLY);
  if(fd<0){if(errno==ENOENT) return -1; else mmapdie(caller,filename,"open");}
  if(fstat(fd,&stats)) mmapdie(caller,filename,"fstat");
  if(!S_ISREG(stats.st_mode)) return -1;
  mbufstart=mmap(0,stats.st_size,PROT_READ,MAP_SHARED,fd,0);
  if(mbufstart==MAP_FAILED) mmapdie(caller,filename,"mmap");
  if(close(fd)) mmapdie(caller,filename,"close");
  mbufsize=stats.st_size;
  mbufpnt=mbufstart;
  mbufend=mbufstart+mbufsize;
  return 0;}

void munmap1(){
  if (munmap(mbufstart,mbufsize)) pdie("munmap");}

/* Return next input byte or ENDFILE (256) */
pnt getch(){
  if(mbufpnt==mbufend) return ENDFILE;
  return *(mbufpnt++);}

/* Return TRUE if line read or FALSE at ENDFILE
   Skip comment lines (lines starting with hash mark) */
pnt readln(unsigned char line[PATHSIZE]){
  pnt i;
  pnt c;
  i=0;
  for(;;){
    if(i>=PATHSIZE) die("Line too long");
    c=getch();
    line[i]=0;
    if(c==ENDFILE) return i!=0;
    if((c!=CR)&&(c!=LF)) line[i++]=c;
    else if((i>0)&&(line[0]!='#')) return TRUE;
    else i=0;}}

/* Read one pnt, die on ENDFILE, die on overflow */
pnt readpnt(){
  pnt result;
  pnt byte=getch();
  if(byte==ENDFILE) die("Unexpected end of file");
  if(byte<128) return byte;
  result=readpnt();
  if(result&septetmask) die("Pnt overflow");
  return result*128+byte-128;}

/*************
 * Read rack *
 *************/

/*
Read one rack file and convert it into a structure with sharing.
The reference is given as a file descriptor.

The input file is treated as a sequence of bytes.

The output structure is composed of the constant NIL, cardinals,
and conses.

In the input file, most cardinals are expressed base 128. Each such
cardinal is represented as a sequence of 'middle septets' followed
by an 'end septet'. Middle septets are represented by bytes in
the range 128..255. End septets are represented by bytes in the
range 0..127. Cardinals are expressed in little endian. As an example,
"129 130 003" represents 1+128*2+16384*3.

An input file represents a sequence of values, the last of which is
the rack to be constructed.

Each value in the sequence has an "index". The index of the first
value is RACK_START, the index of the next is RACK_START+1, and
so on.

The constant NIL is represented by the cardinal RACK_NIL in base 128.

A cardinal x is represented by the cardinal RACK_INT in base 128
followed by the cardinal x in base 128.

A cons of two values y and z is represented by the index of y followed
by the index of z. The indices of y and z are smaller than the index
of the cons, so when scanning a rack file from start to end, the
arguments of a cons will be constructed before the cons is constructed.

A vector x is represented by the cardinal RACK_ARRAY in base 128
followed by the number of bytes in x expressed in base 128 followed
by the bytes of x. Since a vector ultimately represents a cardinal,
this is just a slightly more compact and less CPU consuming way of
representing cardinals.
*/
pnt readrack1(){
  pnt sp1=sp;
  pnt sp2=sp+1;
  pnt card0;
  pnt card1;
  pnt index;
  pnt vector;
  pnt position;
  pnt *location;
  pnt result;
  push(T); /* stack[sp1-RACK_NIL] represents T */
  push(T); /* stack[sp1-RACK_ARRAY] is unused */
  for(;;){
    index=sp2-sp;
    card0=readpnt();
    if(card0==RACK_INT){
      vector=icons(0,0);
      push(vector);
      location=stack+sp;
      position=0;
      do{
        card1=getch();
        HEAD(vector)|=((card1&127)<<position);
        position+=7;
        if(position>=pntsize){
          position-=pntsize;
          TAIL(vector)=icons((card1&127)>>(7-position),0);
          location=&TAIL(vector);
          vector=TAIL(vector);}}
      while(card1>127);
      if(HEAD(vector)<256) *location=I[HEAD(vector)];}
    else if(card0==RACK_ARRAY){
      vector=consvec(0,0);
      push(vector);
      for(card1=readpnt();card1>=bytesize;card1-=bytesize){
        for(position=0;position<bytesize;position++)
        HEAD(vector)|=(getch()<<(position*8));
        TAIL(vector)=consvec(0,0);
        vector=TAIL(vector);}
      for(position=0;position<card1;position++)
      HEAD(vector)|=(getch()<<(position*8));
      HEAD(vector)|=((pnt)1)<<(card1*8);}
    else if(card0<index){
      card1=readpnt();
      if(card1>=index) die("Malformed rack (Pointer too large)");
      if(card1==RACK_INT) die("Malformed rack (Invalid tail)");
      if(card1==RACK_ARRAY) die("Malformed rack (Invalid tail)");
      push(pair(stack[sp1-card0],stack[sp1-card1]));}
    else if(card0==index) break;
    else die("Malformed rack (Pointer too large)");}
  result=stack[sp];
  sp=sp1;
  return result;}

/*
Read one rack file and convert it into a structure with sharing.
The reference is given as a mixed endian hexadecimal string.
The readrack() function scans the path for a rack file with the
given name and then invokes readrack1 on the file found.
*/

char *memcpy1(char *dest,char *src,size_t n){
  memcpy(dest,src,n);
  return dest+n;}

/*
If p1="aa:bb", p2="cc", p3="dd", and p4="ee" construct aaccddee
If p1="aa:bb", p2="cc", p3="dd", and p4="" construct aaccddbb
*/
pnt make_path(char *result,pnt size,char *p1,char *p2,char *p3,char *p4){
  char *result1=result;
  char *match0=strchr(p1,';');
  pnt l0=match0?(pnt)(match0-p1):(pnt)strlen(p1);
  char *match1=memchr(p1,':',l0);
  pnt l1=match1?(pnt)(match1-p1):(pnt)strlen(p1);
  pnt l2=strlen(p2);
  pnt l3=strlen(p3);
  pnt l4=strlen(p4);
  if(match1==NULL) die("Missing colon in LGW_PATH element");
  if(l4==0){l4=l0-l1-1;p4=p1+l1+1;}
  if(l1+l2+l3+l4>=size) die("Path too long");
  result1=memcpy1(result1,p1,l1);
  result1=memcpy1(result1,p2,l2);
  result1=memcpy1(result1,p3,l3);
  result1=memcpy1(result1,p4,l4);
  result1=memcpy1(result1,"",1);
  return l0;}

#define DEFAULT_LGW_PATH "/.logiweb/logiweb/:/rack.lgr"
char *get_lgw_path(){
  char *path;
  if(lgw_path) return lgw_path;
  path=getenv("LGW_PATH");
  if(path&&path[0]){
    make_path(lgw_path1,sizeof(lgw_path1),":","","",path);
    lgw_path=lgw_path1;
    return lgw_path;}
  path=getenv("HOME");
  if(path&&path[0]){
    make_path(lgw_path1,sizeof(lgw_path1),":","",path,DEFAULT_LGW_PATH);
    lgw_path=lgw_path1;
    return lgw_path;}
  die("Neither LGW_PATH nor HOME set");
  return T; /* This never happens */}

pnt readrack(char *hexref){
  pnt result;
  char *path=get_lgw_path();
  pnt length;
  char filename[PATHSIZE];
  if(path==0) die("LGW_PATH not set");
  for(;;){
    length=make_path(filename,sizeof(filename),path,"",hexref,"");
    path+=(length+1);
    if(length==0) continue;
    if(mmap1("readrack",filename)==0){
      TRUSTGC;
      result=readrack1();
      UNTRUSTGC;
      if(getch()!=ENDFILE) die("Malformed rack (ENDFILE expected)");
      munmap1();
      return result;}
    if(errno!=ENOENT) pdie("lgwam: readrack: open()");
    if(*(path-1)==';') continue;
    printf("%s not found\n",hexref);
    printf("LGW_PATH=%s\n",get_lgw_path());
    die("Goodbye");}}



/*********************************************
 * Optimized function sl2rack for rack input *
 *********************************************/

pnt sl2rack_sl;
pnt sl2rack_X=(pnt)(-1);

/* Essentially the same as getch() */
pnt sl2rack_getch(){
  pnt result;
  if(NO_PAIR(sl2rack_sl)) return sl2rack_X;
  result=HEAD(sl2rack_sl);
  sl2rack_sl=TAIL(sl2rack_sl);
  if(NO_INT(result)) return sl2rack_X;
  if(TAIL(result)!=NIL) return sl2rack_X;
  result=HEAD(result);
  if((result>>8)!=1) return sl2rack_X;
  return result&0xFF;}

/* Essentially the same as readpnt() */
pnt sl2rack_readpnt(){
  pnt result;
  pnt byte=sl2rack_getch();
  if(byte==sl2rack_X) return sl2rack_X;
  if(byte<128) return byte;
  result=sl2rack_readpnt();
  if(result==sl2rack_X) return sl2rack_X;
  if(result&septetmask) die("sl2rack: Pnt overflow");
  result=result*128+byte-128;
  if(result==sl2rack_X) die("sl2rack: Pnt overflow");
  return result;}

/* Essentially the same as readrack1() */
pnt sl2rack_readrack1(){
  pnt sp1=sp-1;
  pnt sp2=sp;
  pnt card0;
  pnt card1;
  pnt byte;
  pnt index;
  pnt vector;
  pnt position;
  pnt *location;
  push(T); /* stack[sp1-RACK_INT] is unused */
  push(T); /* stack[sp1-RACK_NIL] represents T */
  push(T); /* stack[sp1-RACK_ARRAY] is unused */
  for(;;){
    index=sp2-sp;
    card0=sl2rack_readpnt();
    if(card0==sl2rack_X) return X;
    if(card0==RACK_INT){
      vector=icons(0,0);
      push(vector);
      location=stack+sp;
      position=0;
      do{
        card1=sl2rack_getch();
        if(card1==sl2rack_X) return X;
        HEAD(vector)|=((card1&127)<<position);
        position+=7;
        if(position>=pntsize){
          position-=pntsize;
          TAIL(vector)=icons((card1&127)>>(7-position),0);
          location=&TAIL(vector);
          vector=TAIL(vector);}}
      while(card1>127);
      if(HEAD(vector)<256) *location=I[HEAD(vector)];}
    else if(card0==RACK_ARRAY){
      vector=consvec(0,0);
      push(vector);
      card1=sl2rack_readpnt();
      if(card1==sl2rack_X) return X;
      for(;card1>=bytesize;card1-=bytesize){
        for(position=0;position<bytesize;position++){
          byte=sl2rack_getch();
          if(byte==sl2rack_X) return X;
          HEAD(vector)|=(byte<<(position*8));}
        TAIL(vector)=consvec(0,0);
        vector=TAIL(vector);}
      for(position=0;position<card1;position++){
        byte=sl2rack_getch();
        if(byte==sl2rack_X) return X;
        HEAD(vector)|=(byte<<(position*8));}
      HEAD(vector)|=((pnt)1)<<(card1*8);}
    else if(card0<index){
      card1=sl2rack_readpnt();
      if(card1==sl2rack_X) return X;
      if(card1>=index) die("sl2rack: Malformed rack (Pointer too large)");
      if(card1==RACK_INT) die("sl2rack: Malformed rack (Invalid tail)");
      if(card1==RACK_ARRAY) die("sl2rack: Malformed rack (Invalid tail)");
      push(pair(stack[sp1-card0],stack[sp1-card1]));}
    else if(card0==index) break;
    else die("sl2rack: Malformed rack (Pointer too large)");}
  return stack[sp];}

pnt sl2rack(pnt sl){/* sl2rack ( " ) */
  pnt result;
  pnt sp0=sp;
  push(sl);
  sl2rack_sl=sl;
  result=sl2rack_readrack1();
  sp=sp0;
  return result;}



/**********************************************
 * Optimized function rack2sl for rack output *
 **********************************************/

#define RACK2SL_KEY 19
pnt rack2sl_sp;
pnt rack2sl_size;
pnt rack2sl_pnt;
pnt rack2sl_index;

/*
Compare the integers x and y for equality
This function also occurs inside eq0
*/
pnt rack_eq_int(pnt x,pnt y){
  for(;;){
    if(HEAD(x)!=HEAD(y)) return FALSE;
    x=TAIL(x);
    y=TAIL(y);
    if(x==0||y==0) return x==y;}}

/*
During rack conversion, we use a hash table
The hash table is stored on the stack below stack[sp]
The garbarge collector does not see the hash table because it is below stack[sp]
rack_pnt2index(r) converts the pointer r to its index using the hash table
*/
pnt rack_pnt2index(pnt r){
  pnt root;
  pnt head;
  pnt tail;
  pnt hash;
  pnt s;
  if(r==T) return 1;
  root=ROOT(r);
  root=root&~markmask;
  if(root==TAG_INT||root==TAG_VECT){
    hash=HEAD(r);
    hash=hash%rack2sl_size;
    hash=hash*RACK2SL_KEY;
    hash=hash%rack2sl_size;
    for(;(s=stack[rack2sl_sp+hash*2])!=T;hash=(hash+1)%rack2sl_size){
      if((ROOT(s)&~markmask)==TAG_PAIR) continue;
      if(!rack_eq_int(r,s)) continue;
      return stack[rack2sl_sp+hash*2+1];}
    die("rack2sl: Internal error 1, cannot find cardinal");}
  if(root==TAG_PAIR){
    head=HEAD(r);
    tail=TAIL(r);
    hash=(head%rack2sl_size+tail%rack2sl_size)*RACK2SL_KEY;
    hash=hash%rack2sl_size;
    for(;(s=stack[rack2sl_sp+hash*2])!=T;hash=(hash+1)%rack2sl_size){
      if((ROOT(s)&~markmask)!=TAG_PAIR) continue;
      if(HEAD(s)!=head) continue;
      if(TAIL(s)!=tail) continue;
      return stack[rack2sl_sp+hash*2+1];}
    die("rack2sl: Internal error 2, cannot find pair");}
  unexpected_tag("Internal error 3 in rack2sl",r);
  return T; /* This never happens */}

/*
Return size of rack measured in entities (pairs and integers)
T counts as zero entities, entities for one each.
Only count shared entities once.
Put a mark on each entity when counted.
Return sl2rack_X, i.e. (pnt)(-1) in case of unexpected data
*/
pnt rack_size(pnt r){
  pnt root;
  pnt head;
  pnt tail;
  if(r==T) return 0;
  root=ROOT(r);
  if(root&markmask) return 0;
  if(root==TAG_INT){
    if(lt(r,Z)==T){ROOT(r)=root|markmask;return sl2rack_X;}
    ROOT(r)=root|markmask;return 1;}
  ROOT(r)=root|markmask;
  if(root==TAG_VECT) return 1;
  if(root!=TAG_PAIR) return sl2rack_X;
  head=rack_size(HEAD(r));
  if(head==sl2rack_X) return sl2rack_X;
  tail=rack_size(TAIL(r));
  if(tail==sl2rack_X) return sl2rack_X;
  return 1+head+tail;}

/*
Search rack for sharing.
Build a hash table which maps entities (pairs and integers) to indexes
Remove mark after processing an entity.
Turn duplicates into garbage and let the garbage point to its twin.
Link all duplicates into a list pointed at by rack2sl_pnt
*/
pnt rack_merge(pnt r){
  pnt root;
  pnt head;
  pnt tail;
  pnt hash;
  pnt s;
  if(r==T) return r;
  root=ROOT(r);
  if(!(root&markmask)) return (root==TAG_GARB)?HEAD(r):r;
  root=root&~markmask;
  ROOT(r)=root;
  if(root==TAG_GARB) return HEAD(r);
  if(root==TAG_INT||root==TAG_VECT){
    hash=HEAD(r);
    hash=hash%rack2sl_size;
    hash=hash*RACK2SL_KEY;
    hash=hash%rack2sl_size;
    for(;(s=stack[rack2sl_sp+hash*2])!=T;hash=(hash+1)%rack2sl_size){
      if(ROOT(s)==TAG_PAIR) continue;
      if(!rack_eq_int(r,s)) continue;
      ROOT(r)=TAG_GARB;
      HEAD(r)=s;
      TAIL(r)=rack2sl_pnt;
      rack2sl_pnt=r;
      return s;}
    stack[rack2sl_sp+hash*2]=r;
    stack[rack2sl_sp+hash*2+1]=rack2sl_index++;
    return r;}
  if(root==TAG_PAIR){
    head=rack_merge(HEAD(r));
    tail=rack_merge(TAIL(r));
    HEAD(r)=head;
    TAIL(r)=tail;
    hash=(head%rack2sl_size+tail%rack2sl_size)*RACK2SL_KEY;
    hash=hash%rack2sl_size;
    for(;(s=stack[rack2sl_sp+hash*2])!=T;hash=(hash+1)%rack2sl_size){
      if(ROOT(s)!=TAG_PAIR) continue;
      if(HEAD(s)!=head) continue;
      if(TAIL(s)!=tail) continue;
      ROOT(r)=TAG_GARB;
      HEAD(r)=s;
      TAIL(r)=rack2sl_pnt;
      rack2sl_pnt=r;
      return s;}
    stack[rack2sl_sp+hash*2]=r;
    stack[rack2sl_sp+hash*2+1]=rack2sl_index++;
    return r;}
  unexpected_tag("Internal error 4 in rack2sl",r);
  return T; /* This never happens */}

/*
Turn duplicates back from garbage to duplicates
*/
void rack2sl_restore(){
  pnt r,s;
  while(rack2sl_pnt!=T){
    r=rack2sl_pnt;
    rack2sl_pnt=TAIL(rack2sl_pnt);
    s=HEAD(r);
    ROOT(r)=ROOT(s);
    HEAD(r)=HEAD(s);
    TAIL(r)=TAIL(s);}}

/*
Push one byte onto the result
The result is pointed at by TAIL(stack[sp])
rack2sl_pnt points to the last cell of the result
*/
void rack_byte_dump(pnt r){
  pnt x=pair(bytevect[r],T);
  TAIL(rack2sl_pnt)=x;
  rack2sl_pnt=x;}

/*
Push one small cardinal onto the result
*/
void rack_int_dump(pnt r){
  for(;r>=128;r=r/128) rack_byte_dump(r%128+128);
  rack_byte_dump(r);}

/*
Find the integer-length of the cardinal x
Resembles integerLength(x)
*/
pnt rack_int_length(pnt x){
  pnt result=0;
  for(;TAIL(x);x=TAIL(x)) result+=pntsize;
  x=HEAD(x);
  if(x&signmask) return 0;
  for(;x;x=x>>1) result+=1;
  return result;}

/*
Push the rack (excluding the end mark) onto the result
*/
void rack_dump(pnt r){
  pnt root;
  pnt head;
  pnt tail;
  pnt len;
  pnt i;
  if(r==T) return;
  root=ROOT(r);
  if(root&markmask) return;
  ROOT(r)=root|markmask;
  if(root==TAG_INT||root==TAG_VECT){
    len=rack_int_length(r);
    if(len%8==1){
      rack_int_dump(2);
      len=len/8;
      rack_int_dump(len);
      for(;TAIL(r);r=TAIL(r))
      for(i=0;i<bytesize;i++)
      rack_byte_dump((HEAD(r)>>(8*i))&0xFF);
      r=HEAD(r);
      for(;r>1;r=r>>8) rack_byte_dump(r&0xFF);
      return;}
    if(TAIL(r)) die("rack2sl conversion of non-vector bigints not implemented");
    rack_int_dump(0);
    rack_int_dump(HEAD(r));
    return;}
  if(root==TAG_PAIR){
    head=HEAD(r);
    tail=TAIL(r);
    rack_dump(head);
    rack_dump(tail);
    rack_int_dump(rack_pnt2index(head));
    rack_int_dump(rack_pnt2index(tail));
    return;}
  unexpected_tag("Internal error 5 in rack2sl",r);}



/*
Scan the heap and remove marks on all entities
*/
void rack_unmark(pnt r){
  pnt root;
  if(r==T) return;
  root=ROOT(r);
  if(!(root&markmask)) return;
  root=root&~markmask;
  ROOT(r)=root;
  if(root!=TAG_PAIR) return;
  rack_unmark(HEAD(r));
  rack_unmark(TAIL(r));}

/*
Clear the part of the stack used for the hash table
The hash table is stored below stack[sp] so that gc() does not note it
*/
void clear_hash(){
  pnt sp1;
  for(sp1=0;sp1<rack2sl_size;sp1++) stack[rack2sl_sp+sp1*2]=T;}

/*
Convert the rack r to a singleton list
*/
pnt rack2sl(pnt r){/* rack2sl ( " ) */
  pnt size;
  push(r);
  size=rack_size(r);
  if(size==sl2rack_X){rack_unmark(r);return ret(1,X);}
  push(pair(T,T));
  rack2sl_size=size*2; /* Make hash twice as big as needed */
  rack2sl_size=(rack2sl_size/RACK2SL_KEY+1)*RACK2SL_KEY+1;
  if(sp<rack2sl_size*2) die("stack overflow in rack2sl");
  rack2sl_sp=sp-rack2sl_size*2;
  clear_hash();
  rack2sl_pnt=T;
  rack2sl_index=3;
  r=rack_merge(r);
  rack2sl_restore();
  rack2sl_pnt=stack[sp];
  rack_dump(r);
  rack_int_dump(rack2sl_index);
  rack_unmark(r);
  rack2sl_pnt=TAIL(stack[sp]);
  return ret(2,rack2sl_pnt);}



/************************
 * Reference conversion *
 ************************/

/* Functions for converting a reference to a mixed endian hexadecimal string
   and vice versa */

unsigned char pnt2hex(pnt p){
  switch(p&15){
    case 0x00:return '0';
    case 0x01:return '1';
    case 0x02:return '2';
    case 0x03:return '3';
    case 0x04:return '4';
    case 0x05:return '5';
    case 0x06:return '6';
    case 0x07:return '7';
    case 0x08:return '8';
    case 0x09:return '9';
    case 0x0A:return 'A';
    case 0x0B:return 'B';
    case 0x0C:return 'C';
    case 0x0D:return 'D';
    case 0x0E:return 'E';
    case 0x0F:return 'F';
    default: return '?'; /* This never happens */}}

pnt hex2pnt(unsigned char p){
  switch(p){
    case '0': return 0x00;
    case '1': return 0x01;
    case '2': return 0x02;
    case '3': return 0x03;
    case '4': return 0x04;
    case '5': return 0x05;
    case '6': return 0x06;
    case '7': return 0x07;
    case '8': return 0x08;
    case '9': return 0x09;
    case 'A': return 0x0A;
    case 'B': return 0x0B;
    case 'C': return 0x0C;
    case 'D': return 0x0D;
    case 'E': return 0x0E;
    case 'F': return 0x0F;
    default: die("Unexpected character in hex constant");
    return T; /* This never happens */}}

/* Convert a vector to a mixed endian hexadecimal string.
   Store result in the "refname" global variable */

unsigned char bufrefname[PATHSIZE];
void ref2string(pnt ref){
  pnt index=0;
  pnt i;
  pnt head,tail;
  if(NO_INT(ref)) die("ref2string called with non-integer\n");
  for(;;){
    head=HEAD(ref);
    tail=TAIL(ref);
    for(i=0;i<bytesize;i++){
      if(index+2>=PATHSIZE) die("ref2string overflow");
      if((head<256)&&(tail==NIL)){bufrefname[index]=0;return;}
      bufrefname[index++]=pnt2hex(head>>4);
      bufrefname[index++]=pnt2hex(head);
      head>>=8;}
    ref=tail;}}

/* Convert a mixed endian hexadecimal string to a vector */
pnt string2ref2(unsigned char *str,pnt len){
  if(len==0) return 1;
  return (hex2pnt(*str)<<4)|hex2pnt(*(str+1))|(string2ref2(str+2,len-1)<<8);}

pnt string2ref1(unsigned char *str,pnt len){
  if(len<bytesize) return icons(string2ref2(str,len),NIL);
  return 
  icons(string2ref2(str,bytesize),string2ref1(str+2*bytesize,len-bytesize));}

pnt string2ref(unsigned char *str){
  pnt len=strlen((char *)str);
  if(len&1) die("string2ref: uneven string");
  return string2ref1(str,len/2);}



/*****************
 * Introductions *
 *****************/

#define init_name(tag,eager,name) init_name1(tag,eager,#tag,name)

void init_name1(pnt tag,pnt eager,char *string,char *name){
  if(TAG_SIZE<=tag) die("init_name: Tag out of range");
  tag2string[tag]=string;
  tag2eager[tag]=eager;
  if(name==T) return;
  tag2name[tag]=str2vec(name);}

void init_names(){
  pnt i;
  for(i=0;i<TAG_SIZE;i++){
    tag2string[i]=NIL;
    tag2eager[i]=F;
    tag2name[i]=T;
    tag2rhs[i]=T;
    tag2arg[i]=T;}

  init_name(TAG_FALSE     ,F,T);
  init_name(TAG_INT       ,F,T);
  init_name(TAG_VECT      ,F,T);
  init_name(TAG_PAIR      ,F,T);
  init_name(TAG_EX        ,F,T);
  init_name(TAG_MAP       ,F,T);
  init_name(TAG_OBJ       ,F,T);
  init_name(TAG_GARB      ,F,T);

  init_name(TTAG_APPLY    ,F,T);
  init_name(TTAG_LAMBDA   ,F,T);
  init_name(TTAG_VAR      ,F,T);
  init_name(TTAG_PAIR     ,F,T);
  init_name(TTAG_CONST    ,F,T);
  init_name(TTAG_CALL     ,F,T);
  init_name(TTAG_ECALL    ,F,T);

  init_name(MTAG_CLOSURE  ,F,T);
  init_name(MTAG_INDIR    ,F,T);
  init_name(MTAG_PAIR     ,F,T);
  init_name(MTAG_LAMBDA   ,F,T);
  init_name(MTAG_FIX      ,F,T);

  init_name(CTAG_MAP      ,F,T);
  init_name(CTAG_EMAP     ,F,T);
  init_name(CTAG_TRUE     ,F,T);
  init_name(CTAG_APPLY    ,F,T);
  init_name(CTAG_IF       ,F,T);

  init_name(ETAG_var      ,F,T);
  init_name(ETAG_ecall    ,F,T);
  init_name(ETAG_digit0   ,T,"\" %0");
  init_name(ETAG_digit1   ,T,"\" %1");
  init_name(ETAG_digit2   ,T,"\" %2");
  init_name(ETAG_digit3   ,T,"\" %3");
  init_name(ETAG_digit4   ,T,"\" %4");
  init_name(ETAG_digit5   ,T,"\" %5");
  init_name(ETAG_digit6   ,T,"\" %6");
  init_name(ETAG_digit7   ,T,"\" %7");
  init_name(ETAG_digit8   ,T,"\" %8");
  init_name(ETAG_digit9   ,T,"\" %9");
  init_name(ETAG_times    ,T,"\" * \"");
  init_name(ETAG_plus     ,T,"\" + \"");
  init_name(ETAG_minus    ,T,"\" - \"");
  init_name(ETAG_and      ,F,"\" .and. \"");
  init_name(ETAG_or       ,F,"\" .or. \"");
  init_name(ETAG_then     ,F,"\" .then. \"");
  init_name(ETAG_pair     ,T,"\" :: \"");
  init_name(ETAG_lt       ,T,"\" < \"");
  init_name(ETAG_le       ,T,"\" <= \"");
  init_name(ETAG_equal    ,T,"\" = \"");
  init_name(ETAG_gt       ,T,"\" > \"");
  init_name(ETAG_ge       ,T,"\" >= \"");
  init_name(ETAG_apply    ,T,"\" apply \"");
  init_name(ETAG_boolp    ,T,"\" boolp");
  init_name(ETAG_catch    ,F,"\" catch");
  init_name(ETAG_maptag1  ,F,"\" catching maptag");
  init_name(ETAG_div      ,T,"\" div \"");
  init_name(ETAG_head     ,T,"\" head");
  init_name(ETAG_intp     ,T,"\" intp");
  init_name(ETAG_boolg    ,T,"\" is bool : \"");
  init_name(ETAG_intg     ,T,"\" is int : \"");
  init_name(ETAG_mapg     ,T,"\" is map : \"");
  init_name(ETAG_objg     ,T,"\" is object : \"");
  init_name(ETAG_pairg    ,T,"\" is pair : \"");
  init_name(ETAG_mapp     ,T,"\" mapp");
  init_name(ETAG_maptag   ,F,"\" maptag");
  init_name(ETAG_mod      ,T,"\" mod \"");
  init_name(ETAG_norm     ,T,"\" norm");
  init_name(ETAG_objp     ,T,"\" objectp");
  init_name(ETAG_pairp    ,T,"\" pairp");
  init_name(ETAG_raise    ,T,"\" raise");
  init_name(ETAG_root     ,T,"\" root");
  init_name(ETAG_tail     ,T,"\" tail");
  init_name(ETAG_untag    ,T,"\" untag");
  init_name(ETAG_digitend ,T,"%%");
  init_name(ETAG_par      ,F,"( \" )");
  init_name(ETAG_uplus    ,T,"+ \"");
  init_name(ETAG_dplus    ,T,"+\"");
  init_name(ETAG_uminus   ,T,"- \"");
  init_name(ETAG_dminus   ,T,"-\"");
  init_name(ETAG_not      ,T,".not. \"");
  init_name(ETAG_Base     ,T,"Base");
  init_name(ETAG_LET      ,F,"LET \" BE \"");
  init_name(ETAG_ash      ,T,"ash ( \" , \" )");
  init_name(ETAG_bottom   ,T,"bottom");
  init_name(ETAG_bt2bytes ,T,"bt2byte* ( \" )");
  init_name(ETAG_bt2vects ,T,"bt2vector* ( \" )");
  init_name(ETAG_bt2vector,T,"bt2vector ( \" )");
  init_name(ETAG_ceiling  ,T,"ceiling ( \" , \" )");
  init_name(ETAG_destruct ,T,"destruct ( \" )");
  init_name(ETAG_evenp    ,T,"evenp ( \" )");
  init_name(ETAG_exception,T,"exception");
  init_name(ETAG_false    ,T,"false");
  init_name(ETAG_floor    ,T,"floor ( \" , \" )");
  init_name(ETAG_half     ,T,"half ( \" )");
  init_name(ETAG_if       ,F,"if \" then \" else \"");
  init_name(ETAG_intlength,T,"integer-length ( \" )");
  init_name(ETAG_logand   ,T,"logand ( \" , \" )");
  init_name(ETAG_logandc1 ,T,"logandc1 ( \" , \" )");
  init_name(ETAG_logandc2 ,T,"logandc2 ( \" , \" )");
  init_name(ETAG_logbitp  ,T,"logbitp ( \" , \" )");
  init_name(ETAG_logcount ,T,"logcount ( \" )");
  init_name(ETAG_logeqv   ,T,"logeqv ( \" , \" )");
  init_name(ETAG_logior   ,T,"logior ( \" , \" )");
  init_name(ETAG_lognand  ,T,"lognand ( \" , \" )");
  init_name(ETAG_lognor   ,T,"lognor ( \" , \" )");
  init_name(ETAG_lognot   ,T,"lognot ( \" )");
  init_name(ETAG_logorc1  ,T,"logorc1 ( \" , \" )");
  init_name(ETAG_logorc2  ,T,"logorc2 ( \" , \" )");
  init_name(ETAG_logtest  ,T,"logtest ( \" , \" )");
  init_name(ETAG_logxor   ,T,"logxor ( \" , \" )");
  init_name(ETAG_map      ,F,"map ( \" )");
  init_name(ETAG_prenorm  ,T,"norm \"");
  init_name(ETAG_notnot   ,T,"notnot \"");
  init_name(ETAG_object   ,T,"object ( \" )");
  init_name(ETAG_print    ,T,"print ( \" )");
  init_name(ETAG_round    ,T,"round ( \" , \" )");
  init_name(ETAG_spy      ,T,"spy ( \" )");
  init_name(ETAG_timer    ,T,"timer ( \" )");
  init_name(ETAG_trace    ,T,"trace ( \" )");
  init_name(ETAG_truncate ,T,"truncate ( \" , \" )");
  init_name(ETAG_vector   ,T,"vector ( \" )");
  init_name(ETAG_vnorm    ,T,"vector-norm ( \" )");
  init_name(ETAG_vempty   ,T,"vector-empty ( \" )");
  init_name(ETAG_vindex   ,T,"vector-index ( \" , \" )");
  init_name(ETAG_vlength  ,T,"vector-length ( \" )");
  init_name(ETAG_vprefix  ,T,"vector-prefix ( \" , \" )");
  init_name(ETAG_vsubseq  ,T,"vector-subseq ( \" , \" , \" )");
  init_name(ETAG_vsuffix  ,T,"vector-suffix ( \" , \" )");
  init_name(ETAG_v2bytes  ,T,"vector2byte* ( \" )");
  init_name(ETAG_v2vects  ,T,"vector2vector* ( \" )");
  init_name(ETAG_vt2bytes ,T,"vt2byte* ( \" )");
  init_name(ETAG_vt2vects ,T,"vt2vector* ( \" )");
  init_name(ETAG_vt2v     ,T,"vt2vector ( \" )");
  init_name(ETAG_yy       ,F,"YY");
  init_name(ETAG_compile  ,T,"compile ( \" )");
  init_name(ETAG_ripemd   ,T,"ripemd ( \" )");
  init_name(ETAG_sl2rack  ,T,"sl2rack ( \" )");
  init_name(ETAG_rack2sl  ,T,"rack2sl ( \" )");
}



/******************
 * Term accessors *
 ******************/

pnt term2ref(pnt term){
  return head(head(term));}
pnt term2idx(pnt term){
  return head(tail(head(term)));}
pnt term2arg1(pnt term){
  return head(tail(term));}
pnt term2arg2(pnt term){
  return head(tail(tail(term)));}
pnt term2arg3(pnt term){
  return head(tail(tail(tail(term))));}
pnt def2lhs(pnt def){
  return term2arg2(def);}
pnt def2arg(pnt def){
  return tail(def2lhs(def));}
pnt def2rhs(pnt def){
  return term2arg3(def);}



/*******************
 * Translate names *
 *******************/

pnt name2sym4(pnt cache,pnt idx,pnt ref,pnt name){
  pnt cache1;
  cache1=mget2(cache,Z,const_name);
  if(cache1==T) return T;
  cache1=head(tail(tail(tail(cache1))));
  cache1=head(tail(head(cache1)));
  if(eq0(cache1,name)) return pair(ref,idx);
  return T;}

pnt name2sym3(pnt cache,pnt ref,pnt name){
  pnt result;
  if(cache==T) return T;
  if(IS_INT(head(cache)))
  return name2sym4(TAIL(cache),HEAD(cache),ref,name);
  result=name2sym3(HEAD(cache),ref,name);
  if(result!=T) return result;
  return name2sym3(TAIL(cache),ref,name);}

pnt name2sym2(pnt cache,pnt ref,pnt name){
  return name2sym3(mget3(cache,ref,const_codex,ref),ref,name);}

pnt name2sym1(pnt cache,pnt name){
  pnt result;
  if(cache==T) return T;
  if(IS_INT(head(cache))) return name2sym2(TAIL(cache),HEAD(cache),name);
  result=name2sym1(HEAD(cache),name);
  if(result!=T) return result;
  return name2sym1(tail(cache),name);}

pnt name2sym(char name1[]){
  pnt name;
  name=str2vec1((unsigned char *)name1,strlen(name1));
  return name2sym1(state,name);}

pnt name2ref(char name1[]){
  pnt sym=name2sym(name1);
  if(sym==T) {printf("%s\n",name1);die("No code found");}
  return head(sym);}

pnt name2idx(char name1[]){
  pnt sym=name2sym(name1);
  if(sym==T) {printf("%s\n",name1);die("No code found");}
  return tail(sym);}

pnt name2code(char name1[]){
  pnt sym;
  pnt ref;
  pnt idx;
  sym=name2sym(name1);
  if(sym==T) {printf("%s\n",name1);die("No code found");}
  ref=HEAD(sym);
  idx=TAIL(sym);
  return mget4(state,ref,ref,const_code,idx);}

pnt name2code1(pnt code,char name1[]){
  return mget1(code,name2idx(name1));}



/************************
 * Compilation, stage 1 *
 ************************/

/*
During stage 1, an initial value for the code array is constructed.

The code array is constructed in the top element of the stack.

The code array has one entry per construct with a non-empty value aspect.

The code array is indexed by the index of each construct.

For proclaimed constructs (lambda, quote, true, apply, and if),
code[idx] is set to the value code[idx] is eventually going to have.
(0 for lambda, 1 for quote, and various tagged maps for the other three).

For defined and introduced constructs, code[idx] is set to a tagged map
whose 'hidden tag' is set to CTAG_EMAP because any definition is deemed
eager until the converse has been proved. During later stages, this hidden
tag is updated to reflect the outcome of various static analysis. The
hidden tag has no influence on the semantics of the entries.

For defined and introduced constructs, the map part of code[idx] is set
to as many lambdas as the arity of the construct indicates followed by
a pair construct followed by the definition of the construct. During
stage 5 and 6, the pair is overwritten with the compiled version of the
definition. This allows to implement mutual recursion with circular
call structures.
*/

pnt term2fct(pnt arity,pnt term){
  if(arity==0) return closure(term,NIL);
  return map_lambda(term_lambdas(arity-1,term),NIL);}

void initcode(pnt codex){
  pnt hd;
  pnt tl;
  pnt def;
  pnt arg;
  pnt arity;
  if(codex==T) return;
  if(GD_PAIR(codex)) die("Unexpected type in codex");
  hd=HEAD(codex);
  tl=TAIL(codex);
  if(NO_INT(hd)){initcode(hd);initcode(tl);return;}
  def=mget2(tl,Z,const_value);
  if(def==T) return;
  if(!eq0(head(head(def)),Z)){
    arity=0;
    for(arg=tail(head(tail(tail(def))));arg!=T;arg=TAIL(arg)) arity++;
    TOP=aput0(TOP,hd,mapcons2(CTAG_EMAP,term2fct(arity,pair(def,T))));
    return;}
  def=head(tail(head(def)));
  if(eq0(def,const_lambda)){TOP=aput0(TOP,hd,fct_lambda); return;}
  if(eq0(def,const_apply)){TOP=aput0(TOP,hd,fct_apply); return;}
  if(eq0(def,const_if)){TOP=aput0(TOP,hd,fct_if); return;}
  if(eq0(def,const_true)){TOP=aput0(TOP,hd,fct_true); return;}
  if(eq0(def,const_quote)){TOP=aput0(TOP,hd,fct_quote); return;}
  spy0(int2vec(def));
  die("Unknown value proclamations");}



/*****************************************
 * Compilation, stage 2, record patterns *
 *****************************************/

/*
During the stage 2, the official definitions of optimized functions
are collected.

This is done on basis of the cache of a reference page. The constructs
on the reference page are recognized using their Logiweb names.

Optimized function i is supposed to have Logiweb name tag2name[i].
Once a construct with name tag2name[i] is located in the cache of
the reference page, the value definition of that construct is taken
to be the official definition of optimized function i.

Stage 2 of the compilation is only performed when translating the
reference page. A complete compilation consists of compilation of the
reference page, including stage 2, followed by compilation of the
source page, excluding stage 2.

Note that the source page can be identical to the reference page in
which case translation of the source page will be trivial since it
is already compiled.

Also note that the source page may transitively reference to reference
page in which case the reference page is only compiled once.

One could stop compilation of the reference page after stage 2 if it
is not transitively referenced by the source page, but the reference
page is translated to completion in all cases for simplicity.

The algorithm uses the code array produced in stage 1.
One could have based the algorithm on the codex instead of the
code, leading to a cleaner but more complex algorithm. The algorithm
actually used is "unclean" in the sense that it depends on the
particular output from stage 1 even though it could have depended
only on the completely standardized format of the cache and codex.
*/

pnt nameget(pnt caches,pnt ref,pnt idx){
  pnt def=defget(caches,ref,idx,const_name);
  if(def==T) return T;
  return term2idx(def2rhs(def));}

void nameprint3(pnt x){
  x=x&0xFF;
  if(x<32) printesc(x); else
  if(x<127) printf("%c",(int)x); else
  printesc(x);}

void nameprint2(pnt x){
  for(;TAIL(x);x=TAIL(x)){
    pnt i;
    pnt y=HEAD(x);
    for(i=0;i<bytesize;i++){nameprint3(y);y=y>>8;}}
  for(x=HEAD(x);x>1;x=x>>8) nameprint3(x);}

void nameprint1(pnt caches,pnt term){
  pnt ref=term2ref(term);
  pnt idx=term2idx(term);
  pnt name;
  if(term==T) printf("    T");
  else if(ref==Z){printf("string ");spy2(1,idx);}
  else {
    printf("Index ");
    spy2(1,term2idx(term));
    name=nameget(caches,ref,idx);
    if(name==T) return;
    printf(": ");
    nameprint2(name);}}

void nameprint0(pnt caches,pnt term){
  nameprint1(caches,term);
  printf("\n");}

/*
Skip the mtag_closure, mtag_lambda, and ttag_lambda constructs
at the root of code entries and return the pair which is eventually
going to be overwritten with the compiled version of the code entry.

The given term entry is supposed to be a mapcons so skiparg cannot
be used on lambdas and quotes.
*/
pnt code2pair(pnt term){
  for(term=HEAD(term);ROOT(term)!=TAG_PAIR;term=HEAD(term));
  return term;}

/*
Same as above, but return the untranslated definition instead of the pair
*/
pnt code2def(pnt term){
  return HEAD(code2pair(term));}

void record_pattern1(pnt caches,pnt code,pnt def){
  pnt lhs=term2arg2(def);
  pnt ref=term2ref(lhs);
  pnt idx=term2idx(lhs);
  pnt arg=tail(lhs);
  pnt rhs=term2arg3(def);
  pnt i;
  pnt name=nameget(caches,ref,idx);
  if(name==T) return;
  for(i=0;i<TAG_SIZE;i++)
  if(eq0(name,tag2name[i])){
    if(tag2rhs[i]!=T){
      printf("Multiple definitions. Name=");
      spy0(name);
      return;}
    tag2rhs[i]=rhs;
    tag2arg[i]=arg;
    TAIL(code)=i; /* Unnecessary, but saves time, c.f. recognize_intro */
    return;}
  return;}

void record_pattern(pnt caches,pnt code,pnt def){
  pnt ref=term2ref(def);
  pnt idx=term2idx(def);
  pnt aspect=defget(caches,ref,idx,const_definition);
  if(!eq0(aspect,tree_introduce)) return;
  record_pattern1(caches,code,def);}

void record_patterns(pnt caches,pnt code){
  pnt hd;
  pnt tl;
  if(code==T) return;
  if(GD_PAIR(code)) die("Unexpected type in code");
  hd=HEAD(code);
  tl=TAIL(code);
  if(NO_INT(hd)){
    record_patterns(caches,hd);
    record_patterns(caches,tl);
    return;}
  if(ROOT(tl)!=TAG_MAP) return; /* disregard quote and lambda */
  if(TAIL(tl)!=CTAG_EMAP) return; /* disregard true, apply, and if */
  record_pattern(caches,tl,code2def(tl));}



/********************************************
 * Compilation, stage 3, pattern comparison *
 ********************************************/

/*
sim_term(pnt c,pnt t1,pnt v1,pnt t2,pnt v2,pnt sim) checks whether or not
two terms, t1 and t2, have identical definitions except for names of
bound variables and except for names of auxiliary functions.

When calling sim_term, v1 and v2 are lists of bound variables such that
an occurence of the n'th element of v1 in t1 is considered identical to
an occurence of the n'th element of v2 in t2.

The c parameter must contain the loaded caches containing the definitions
of constructs in t1 and t2.

The sim parameter contains function identifications such that an occurence
of sim[ref][idx] in t2 is considered identical to an occurrence of the
construct with the given reference and index in t1.

sim_term accumulates function identifications in the sim parameter.
When calling sim_term, one should set sim to T.

When calling sim_term, one would typically set t1 and v1 to the right
hand side and argument list, respectively, of a reference definition
and t2 and v2 to the right hand side and argument list, respectively,
of a definition which is being compared against the reference definition.
*/

/* Term equality */
pnt tree_equal(pnt x,pnt y){
  if(!eq0(term2ref(x),term2ref(y))) return FALSE;
  if(!eq0(term2idx(x),term2idx(y))) return FALSE;
  for(;;){
    x=tail(x);
    y=tail(y);
    if(x==T) return y==T;
    if(y==T) return FALSE;
    if(!tree_equal(head(x),head(y))) return FALSE;}}

/* List length comparison */
pnt same_length(pnt x,pnt y){
  for(;;){
    if(x==T) return y==T;
    if(y==T) return FALSE;
    x=TAIL(x);
    y=TAIL(y);}}

/* True if the given definition defines a parenthesis */
pnt pardef(pnt d){
  pnt arg,rhs;
  if(d==T) return FALSE;/* Undefined */
  if(tail(tail(tail(d)))==T) return FALSE;/* Proclaimed */
  arg=def2arg(d);
  rhs=def2rhs(d);
  if(arg==T) return FALSE; /* Nulary construct */
  if(tail(arg)!=T) return FALSE; /* Arity two or more */
  return tree_equal(head(arg),rhs);}

pnt sim_term(pnt c,pnt t1,pnt v1,pnt t2,pnt v2,pnt sim);

pnt sim_terms(pnt c,pnt t1,pnt v1,pnt t2,pnt v2,pnt sim){
  t1=tail(t1);
  t2=tail(t2);
  for(;;){
    if(t1==T) return sim;
    TOP=sim=sim_term(c,head(t1),v1,head(t2),v2,sim);
    if(sim==F) return F;
    t1=tail(t1);
    t2=tail(t2);}}

pnt sim_term(pnt c2,pnt t1,pnt v1,pnt t2,pnt v2,pnt sim){
  pnt r1,i1,r2,i2,t3,d1,d2;
  r1=term2ref(t1);
  i1=term2idx(t1);
  r2=term2ref(t2);
  i2=term2idx(t2);
  d1=defget(state,r1,i1,const_value);/* Remove parentheses from root of t1 */
  while(pardef(d1)){
    t1=term2arg1(t1);
    r1=term2ref(t1);
    i1=term2idx(t1);
    d1=defget(state,r1,i1,const_value);}
  d2=defget(c2,r2,i2,const_value);/* Remove parentheses from root of t2 */
  while(pardef(d2)){
    t2=term2arg1(t2);
    r2=term2ref(t2);
    i2=term2idx(t2);
    d2=defget(c2,r2,i2,const_value);}
  if(eq0(r1,Z)){/* If true, t1 must be a string */
    if(!eq0(r2,Z)) return F;
    if(!eq0(i1,i2)) return F;
    return sim;}
  if(eq0(r2,Z)) return F;
  if(d1==T){/* If true, t1 must be a variable */
    if(d2!=T) return F;
    for(;;){/* Traverse v1,v2 to see that t1,t2 have same deBruijn index */
      if(v1==T) return F;
      if(tree_equal(t1,head(v1))){
        if(!tree_equal(t2,head(v2))) return F;
        return sim;}
      if(tree_equal(t2,head(v2))) return F;
      v1=tail(v1);
      v2=tail(v2);}}
  if(d2==T) return F;
  if(term2ref(d1)==Z){/* If true, t1 must be a proclamation */
    if(term2ref(d1)!=Z) return F;
    d1=term2idx(d1);
    d2=term2idx(d2);
    if(!eq0(d1,d2)) return F;
    /* Now we know that t1 and t2 are proclaimed the same way */
    if(eq0(d1,const_apply)) return sim_terms(c2,t1,v1,t2,v2,sim);
    if(eq0(d1,const_if)) return sim_terms(c2,t1,v1,t2,v2,sim);
    if(eq0(d1,const_true)) return sim;
    if(eq0(d1,const_quote)){
      if(eq0(term2arg1(t1),term2arg1(t2))) return sim;
      return FALSE;}
    /* Now we know that t1 and t2 are lambdas */
    v1=pair(term2arg1(t1),v1);
    push(v1);
    v2=pair(term2arg1(t2),v2);
    push(v2);
    push(sim);
    t1=term2arg2(t1);
    t2=term2arg2(t2);
    return ret(3,sim_term(c2,t1,v1,t2,v2,sim));}
  if(term2ref(d2)==Z) return F;
  /* t1 and t2 are defined */
  t3=mget2(sim,r1,i1);
  if(t3!=T){/* If true, t1 is already matched against some construct */
    if(!eq0(r2,term2ref(t3))) return F;
    if(!eq0(i2,term2idx(t3))) return F;
    return sim_terms(c2,t1,v1,t2,v2,sim);}
  if(!same_length(def2arg(d1),def2arg(d2))) return F;
  TOP=sim=mput2(sim,t2,r1,i1);
  TOP=sim=sim_terms(c2,t1,v1,t2,v2,sim);
  if(sim==F) return F;
  t1=def2rhs(d1);
  v1=def2arg(d1);
  t2=def2rhs(d2);
  v2=def2arg(d2);
  return sim_term(c2,t1,v1,t2,v2,sim);}



/********************************************
 * Compilation, stage 3, recognize patterns *
 ********************************************/

/*
During stage 3, optimized functions are recognized.

For each introduction, the right hand side and argument list of
the introduction are compared to all reference definitions to find
a match. If a match is found, the identity of the recognized function
is noted in the hidden tag of the code entry of the introduction.

The hidden tag of unrecognized introductions remain unchanged so that
they are treated exactly like definitions.
*/

void printesc_stderr(pnt x){
  fprintf(stderr,"\\%d%d",(int)(x/16),(int)(x%16));}

void printchar_stderr(pnt x){
  x=x&0xFF;
  if(x=='"') fprintf(stderr,"\\\""); else
  if(x=='\\') fprintf(stderr,"\\\\"); else
  if(x==10) fprintf(stderr,"\n"); else
  if(x<32) printesc_stderr(x); else
  if(x<127) fprintf(stderr,"%c",(int)x); else
  printesc_stderr(x);}

void spyvect_stderr(pnt x){
  fprintf(stderr,"\"");
  for(;TAIL(x);x=TAIL(x)){
    pnt i;
    pnt y=HEAD(x);
    for(i=0;i<bytesize;i++){
      printchar_stderr(y);
      y=y>>8;}}
  for(x=HEAD(x);x>1;x=x>>8) printchar_stderr(x);
  fprintf(stderr,"\"");}

void recognize_intro3(pnt caches,pnt code,pnt def){
  pnt lhs=def2lhs(def);
  pnt arg=def2arg(def);
  pnt rhs=def2rhs(def);
  pnt ref=term2ref(lhs);
  pnt idx=term2idx(lhs);
  pnt i;
  for(i=0;i<TAG_SIZE;i++){
    push(T);
    if(F==ret(1,sim_term(caches,tag2rhs[i],tag2arg[i],rhs,arg,TOP))) continue;
/*  printf("%-14s ",safe_tag2string(i));
    spy0(nameget(caches,ref,idx)); */
    TAIL(code)=i;
    return;}
  fprintf(stderr,"%-14s ","No match");
/*spy0(nameget(caches,ref,idx));*/
  spyvect_stderr(nameget(caches,ref,idx));
  fprintf(stderr,"\n");}

void recognize_intro2(pnt caches,pnt code,pnt def){
  pnt ref=term2ref(def);
  pnt idx=term2idx(def);
  pnt aspect=defget(caches,ref,idx,const_definition);
  if(!eq0(aspect,tree_introduce)) return;
  recognize_intro3(caches,code,def);}

void recognize_intro1(pnt caches,pnt code){
  recognize_intro2(caches,code,code2def(code));}

void recognize_intro(pnt caches,pnt code){
  pnt hd;
  pnt tl;
  if(code==T) return;
  if(GD_PAIR(code)) die("Unexpected type in code");
  hd=HEAD(code);
  tl=TAIL(code);
  if(NO_INT(hd)){
    recognize_intro(caches,hd);
    recognize_intro(caches,tl);
    return;}
  if(ROOT(tl)!=TAG_MAP) return;   /* disregard quote and lambda */
  if(TAIL(tl)!=CTAG_EMAP) return; /* disregard true, apply, if, and
                                   * constructs tagged by record_pattern1 */
  recognize_intro1(caches,tl);
  return;}



/*********************************************
 * Compilation, stage 4, strictness analysis *
 *********************************************/

/*
During stage 4, eager functions are recognized.

To be eager, a function must evaluate its arguments left to right
and, if an argument raises and exception, return that exception
without evaluating the remaining arguments.

Furthermore, eager functions must normalize their arguments and
thus force the arguments to be tagged values.

The algorithm for recognizing eager functions is thus: Any function
is eager until the converse is proved. A function is non-eager in
the following situations:

- The right hand side (rhs) of the definition does not start the
  required way, which is
  norm ( arg1 .then. arg2 .then. ... .then. term )

- The rhs contains a construct which is neither value defined nor bound.

- The rhs references a construct which is deemed non-eager.

- The rhs references optimized/proclaimed functions in an unauthorized pattern.
*/

/* Note that constructs in rhs call the construct with the given ref and idx */
void call_graph2(pnt caches,pnt ref,pnt idx,pnt rhs){
  pnt ref1=term2ref(rhs);
  pnt idx1=term2idx(rhs);
  pnt def=codeget(caches,ref1,idx1);
  if(def==T) return; /* disregard variables */
  if(def==Z) {call_graph2(caches,ref,idx,term2arg2(rhs)); return;} /* lamdba */
  if(ROOT(def)!=TAG_MAP) return; /* disregard quote */
  if(TAIL(def)==ETAG_map) return; /* disregard x in map(x) */
  if(eq0(ref,ref1)) TOP=aput0(TOP,idx1,pair(idx,aget(TOP,idx1)));
  for(rhs=tail(rhs);rhs!=T;rhs=tail(rhs))
  call_graph2(caches,ref,idx,head(rhs));}

/* Construct mapping from called to calling function */
void call_graph1(pnt caches,pnt ref,pnt code){
  pnt hd;
  pnt tl;
  if(code==T) return;
  if(GD_PAIR(code)) die("Unexpected type in code");
  hd=HEAD(code);
  tl=TAIL(code);
  if(NO_INT(hd)){
    call_graph1(caches,ref,hd);
    call_graph1(caches,ref,tl);
    return;}
  if(ROOT(tl)!=TAG_MAP) return; /* disregard quote and lambda */
  if(TAIL(tl)!=CTAG_EMAP) return; /* disregard already classified maps */
  call_graph2(caches,ref,hd,def2rhs(code2def(tl)));
  return;}

void mark_unstrict(pnt code,pnt graph,pnt idx){
  pnt code1=aget(code,idx);
  pnt idx1;
  if(TAIL(code1)!=CTAG_EMAP) return;
  TAIL(code1)=CTAG_MAP;
  for(idx1=aget(graph,idx);idx1!=T;idx1=TAIL(idx1))
  mark_unstrict(code,graph,HEAD(idx1));}

/*
There are two norm constructs, "norm" and "prenorm". The prenorm construct
is a prefix version of the norm construct. The definition of prenorm
reads norm x = x norm so the two norms really do the same, but they have
different definition patterns. Only the prenorm is supposed to occur
in eager definitions, but normp below also recognizes the other one as
a norm construct.
*/
pnt normp(pnt caches,pnt term){
  pnt ref=term2ref(term);
  pnt idx=term2idx(term);
  pnt code=codeget(caches,ref,idx);
  if(code==T) return FALSE; /* string, page symbol, variable */
  if(ROOT(code)!=TAG_MAP) return FALSE; /* lambda, quote */
  if(TAIL(code)==ETAG_prenorm) return TRUE; /* norm, version 1 */
  if(TAIL(code)==ETAG_norm) return TRUE; /* norm, version 2 */
  return FALSE;}

pnt thenp(pnt caches,pnt term){
  pnt ref=term2ref(term);
  pnt idx=term2idx(term);
  pnt code=codeget(caches,ref,idx);
  if(code==T) return FALSE; /* string, page symbol, variable */
  if(ROOT(code)!=TAG_MAP) return FALSE; /* lambda, quote */
  if(TAIL(code)==ETAG_then) return TRUE; /* then */
  return FALSE;}

/*
If pref==T: test for strictness
If pref!=T: search for reason for being non-strict
When pref!=T, pref should be the reference of the caller.
In this case, disregard references to non-strict, local constructs with
norm in their root since they cannot be the ultimate reason for
non-strictness.
*/
pnt strict1(pnt caches,pnt rhs,pnt pref){
  pnt ref=term2ref(rhs);
  pnt idx=term2idx(rhs);
  pnt code;
  pnt result;
  pnt subterm;
  if(ref==Z) return T; /* string */
  if(idx==Z) return T; /* page symbol */
  code=codeget(caches,ref,idx);
  if(code==T) return T; /* Variable */
  if(code==Z) return rhs; /* Lambda */
  if(ROOT(code)!=TAG_MAP) return T; /* Quote */
  switch(TAIL(code)){
  case CTAG_MAP:
    if(pref==T) return rhs;
    if(!eq0(ref,pref)||!normp(caches,def2rhs(code2def(code)))) return rhs;
    for(rhs=TAIL(rhs);rhs!=T;rhs=TAIL(rhs)){
      result=strict1(caches,HEAD(rhs),pref);
      if(result!=T) return result;}
    return T;
  case CTAG_APPLY:
  case CTAG_IF:
    return rhs;
  case ETAG_LET:
    result=strict1(caches,term2arg1(rhs),pref);
    if(result!=T) return result;
    subterm=term2arg2(rhs);
    ref=term2ref(subterm);
    idx=term2idx(subterm);
    if(ref==T) return rhs; /* string */
    if(idx==T) return rhs; /* page symbol */
    code=codeget(caches,ref,idx);
    if(code!=Z) return rhs; /* Not lambda */
    return strict1(caches,term2arg2(subterm),pref);
  case ETAG_map:
    return T;
  case CTAG_EMAP:
  case CTAG_TRUE:
  case ETAG_digit0:
  case ETAG_digit1:
  case ETAG_digit2:
  case ETAG_digit3:
  case ETAG_digit4:
  case ETAG_digit5:
  case ETAG_digit6:
  case ETAG_digit7:
  case ETAG_digit8:
  case ETAG_digit9:
  case ETAG_times:
  case ETAG_plus:
  case ETAG_minus:
  case ETAG_and:
  case ETAG_or:
  case ETAG_then:
  case ETAG_pair:
  case ETAG_lt:
  case ETAG_le:
  case ETAG_equal:
  case ETAG_gt:
  case ETAG_ge:
  case ETAG_apply:
  case ETAG_boolp:
  case ETAG_catch:
  case ETAG_maptag1:
  case ETAG_div:
  case ETAG_head:
  case ETAG_intp:
  case ETAG_boolg:
  case ETAG_intg:
  case ETAG_mapg:
  case ETAG_objg:
  case ETAG_pairg:
  case ETAG_mapp:
  case ETAG_maptag:
  case ETAG_mod:
  case ETAG_norm:
  case ETAG_objp:
  case ETAG_pairp:
  case ETAG_raise:
  case ETAG_root:
  case ETAG_tail:
  case ETAG_untag:
  case ETAG_digitend:
  case ETAG_par:
  case ETAG_uplus:
  case ETAG_dplus:
  case ETAG_uminus:
  case ETAG_dminus:
  case ETAG_not:
  case ETAG_Base:
  case ETAG_ash:
  case ETAG_bottom:
  case ETAG_bt2bytes:
  case ETAG_bt2vects:
  case ETAG_bt2vector:
  case ETAG_ceiling:
  case ETAG_destruct:
  case ETAG_evenp:
  case ETAG_exception:
  case ETAG_false:
  case ETAG_floor:
  case ETAG_half:
  case ETAG_if:
  case ETAG_intlength:
  case ETAG_logand:
  case ETAG_logandc1:
  case ETAG_logandc2:
  case ETAG_logbitp:
  case ETAG_logcount:
  case ETAG_logeqv:
  case ETAG_logior:
  case ETAG_lognand:
  case ETAG_lognor:
  case ETAG_lognot:
  case ETAG_logorc1:
  case ETAG_logorc2:
  case ETAG_logtest:
  case ETAG_logxor:
  case ETAG_prenorm:
  case ETAG_notnot:
  case ETAG_object:
  case ETAG_print:
  case ETAG_round:
  case ETAG_spy:
  case ETAG_timer:
  case ETAG_trace:
  case ETAG_truncate:
  case ETAG_vector:
  case ETAG_vnorm:
  case ETAG_vempty:
  case ETAG_vindex:
  case ETAG_vlength:
  case ETAG_vprefix:
  case ETAG_vsubseq:
  case ETAG_vsuffix:
  case ETAG_v2bytes:
  case ETAG_v2vects:
  case ETAG_vt2bytes:
  case ETAG_vt2vects:
  case ETAG_vt2v:
  case ETAG_compile:
  case ETAG_ripemd:
  case ETAG_sl2rack:
  case ETAG_rack2sl:
    for(rhs=TAIL(rhs);rhs!=T;rhs=TAIL(rhs)){
      result=strict1(caches,HEAD(rhs),pref);
      if(result!=T) return result;}
    return T;
  default:die("strict1: unexpected tag");
  return T; /* This never happens */}}

pnt strict(pnt caches,pnt def,pnt pref){
/*pnt lhs=def2lhs(def);*/
  pnt arg;
  pnt rhs=def2rhs(def);
  if(!normp(caches,rhs)) return F;
  rhs=term2arg1(rhs);
  for(arg=def2arg(def);arg!=T;arg=TAIL(arg)){
    if(!thenp(caches,rhs)) return Z;
    if(!tree_equal(HEAD(arg),term2arg1(rhs))) return Z;
    rhs=term2arg2(rhs);}
  return strict1(caches,rhs,pref);}

void strictness2(pnt caches,pnt code1,pnt code2,pnt graph){
  pnt hd;
  pnt tl;
  pnt strictness;
  if(code2==T) return;
  if(GD_PAIR(code2)) die("Unexpected type in code");
  hd=HEAD(code2);
  tl=TAIL(code2);
  if(NO_INT(hd)){
    strictness2(caches,code1,hd,graph);
    strictness2(caches,code1,tl,graph);
    return;}
  if(hd==Z) return; /* disregard strictness of page symbol */
  if(ROOT(tl)!=TAG_MAP) return; /* disregard quote and lambda */
  if(TAIL(tl)!=CTAG_EMAP) return;  /* disregard already classified maps */
  strictness=strict(caches,code2def(tl),T);
  if(strictness==T) return;
  mark_unstrict(code1,graph,hd);
  return;}

void strictness_report(pnt caches,pnt code){
  pnt hd;
  pnt tl;
  pnt def;
  pnt lhs;
  pnt strictness;
  if(code==T) return;
  if(GD_PAIR(code)) die("Unexpected type in code");
  hd=HEAD(code);
  tl=TAIL(code);
  if(NO_INT(hd)){
    strictness_report(caches,hd);
    strictness_report(caches,tl);
    return;}
  if(ROOT(tl)!=TAG_MAP) return; /* disregard quote and lambda */
  if(TAIL(tl)!=CTAG_MAP) return;  /* only consider lazy maps */
  def=code2def(tl);
  lhs=def2lhs(def);
  strictness=strict(caches,def,term2ref(lhs));
  if(strictness==F) return;
  if(strictness==T){
    printf("lgwam: Probably mistakenly unfit due to problem elsewhere:\n");
    printf("lgwam: ");
    nameprint0(caches,lhs);
    return;}
  if(strictness==Z){
    printf("lgwam: Probably mistakenly unfit due to header:\n");
    printf("lgwam: ");
    nameprint0(caches,lhs);
    return;}
  printf("lgwam: Probably mistakenly unfit:\n");
  printf("lgwam: ");
  nameprint0(caches,lhs);
  printf("lgwam: Caused by call to:\n");
  printf("lgwam: ");
  nameprint0(caches,strictness);}

void strictness_analysis(pnt caches,pnt ref,pnt code){
  push(T);
  call_graph1(caches,ref,code);
  strictness2(caches,code,code,TOP);
  pop();
  strictness_report(caches,code);
  return;}



/*************************************************
 * Compilation, stage 5, change eager definition *
 *************************************************/

/*
The given code is a code entry containing zero, one, or more
lambdas (depending on the arity of the construct) followed
by a a pair which is intended to be overwritten with the
compiled version of the entry. The uncompiled definition
of the entry was stored as the head of the pair in stage 1.
After stage 4 we know which definitions are eager. In stage 5,
each eager definition is modified by replacing the pair
with a ttag_ecall construct whose head is a new pair to be
overwritten in stage 6. The head of the new pair is the
uncompiled definition. The tail of the ttag_ecall construct
contains as many lazy variables as indicated by the arity
of the construct. This allows lazy functions to call eager
functions.
*/

void compile_overwrite(pnt dst,pnt src){
  ROOT(dst)=ROOT(src);
  HEAD(dst)=HEAD(src);
  TAIL(dst)=TAIL(src);}

pnt listlength(pnt list){
  pnt result=0;
  for(;list!=T;list=TAIL(list)) result++;
  return result;}

pnt varlist(pnt arity){
  push(T);
  for(;arity>0;arity--) TOP=pair(term_var(arity-1),TOP);
  return pop();}

void change_eager(pnt caches,pnt code){
  pnt hd;
  pnt tl;
  pnt tag;
  pnt root;
  pnt def;
  pnt vars;
  if(code==T) return;
  if(GD_PAIR(code)) die("Unexpected type in code");
  hd=HEAD(code);
  tl=TAIL(code);
  if(NO_INT(hd)){
    change_eager(caches,hd);
    change_eager(caches,tl);
    return;}
  if(ROOT(tl)!=TAG_MAP) return;    /* disregard quote and lambda */
  tag=TAIL(tl);
  if(tag!=CTAG_EMAP) return;    /* only consider eager definitions */
  root=code2pair(tl);
  def=HEAD(root);
  vars=varlist(listlength(def2arg(def)));
  push(vars);
  compile_overwrite(root,ret(1,term_ecall(pair(def,T),vars)));}



/*******************************************
 * Compilation, stage 6, compile lazy term *
 *******************************************/

/*
Optimize individual code entries
arg1 is the list of variables bound by lambdas in deBruijn index order.
arg2 is the list of arguments in order of appearance which is
opposite to the deBruijn index order.
*/

pnt compile_term(pnt caches,pnt arg1,pnt arg2,pnt term);

pnt compile_lambda(pnt caches,pnt arg1,pnt arg2,pnt term){
  arg1=pair(head(tail(term)),arg1);
  push(arg1);
  term=head(tail(tail(term)));
  return ret(1,term_lambda(compile_term(caches,arg1,arg2,term)));}

pnt compile_var(pnt arg1,pnt arg2,pnt term){
  pnt index;
  for(index=0;arg1!=T;index++,arg1=TAIL(arg1)){
    if(tree_equal(HEAD(arg1),term)) return term_var(index);}
  for(;arg2!=T;arg2=TAIL(arg2))
  if(tree_equal(HEAD(arg2),term)){
    for(arg2=TAIL(arg2);arg2!=T;arg2=TAIL(arg2)) index++;
    return term_var(index);}
  return term_t;}

pnt compile_terms(pnt caches,pnt arg1,pnt arg2,pnt terms){
  push(NIL);
  for(;terms!=T;terms=tail(terms))
  TOP=pair(compile_term(caches,arg1,arg2,head(terms)),TOP);
  return pop();}

pnt compile_other(pnt code,pnt caches,pnt arg1,pnt arg2,pnt terms){
  pnt terms1;
  if(ROOT(code)!=MTAG_LAMBDA) die("compile_other: parameter mismatch 1");
  code=HEAD(code);
  for(terms1=tail(terms);!eq0(terms1,T);terms1=tail(terms1)){
    if(ROOT(code)!=TTAG_LAMBDA) die("compile_other: parameter mismatch 2");
    code=HEAD(code);}
  return term_call(code,compile_terms(caches,arg1,arg2,terms));}

pnt compile_term(pnt caches,pnt arg1,pnt arg2,pnt term){
  pnt root=head(term);
  pnt ref=head(root);
  pnt idx=head(tail(root));
  pnt code;
  if(eq0(ref,Z)) return term_const(idx);
  if(eq0(idx,Z)) return term_const(aget(caches,ref));
  code=mget4(caches,ref,ref,const_code,idx);
  if(code==T) return compile_var(arg1,arg2,term);
  if(eq0(code,fct_lambda)) return compile_lambda(caches,arg1,arg2,term);
  if(eq0(code,fct_quote)) return term_const(term2arg1(term));
  code=HEAD(code); /* Remove M tag */
  term=tail(term); /* Set term to parameter list */
  if(eq0(term,T)) return term_const(code);
  return compile_other(code,caches,arg1,arg2,term);}



/**************************************************
 * Compilation, stage 6, compile lazy definition *
 **************************************************/

/*
The given code is a code entry containing zero, one, or more
lambdas (depending on the arity of the construct) followed
by a pair which is intended to be overwritten with the
compiled version of the entry. The uncompiled definition of
the entry was stored as the head of the pair in stage 1.
*/

void compile_lazy_def(pnt caches,pnt code){
  pnt root=code2pair(code);
  pnt def=HEAD(root);
  pnt arg=def2arg(def);
  pnt rhs=def2rhs(def);
  compile_overwrite(root,compile_term(caches,T,arg,rhs));}



/****************************************************
 * Compilation, stage 6, compile eager introduction *
 ****************************************************/

pnt compile_eager_intro1(pnt tag){
  pnt arg=tag2arg[tag];
  if(arg==T) return term_ecall(cons(tag,T,T),T);
  arg=TAIL(arg);
  if(arg==T) return term_ecall(cons(tag,T,V0),varlist_1);
  arg=TAIL(arg);
  if(arg==T) return term_ecall(cons(tag,V1,V0),varlist_2);
  arg=TAIL(arg);
  if(arg==T) return term_ecall(cons(tag,V2,pair(V1,V0)),varlist_3);
  die("compile_eager_intro1: unexpected arity>3");
  return T; /* This never happens */}

void compile_eager_intro(pnt code,pnt tag){
  pnt root=code2pair(code);
  pnt code1=compile_eager_intro1(tag);
  compile_overwrite(root,code1);}



/********************************************
 * Compilation, stage 6, compile eager term *
 ********************************************/

pnt compile_eterm(pnt caches,pnt arg1,pnt arg2,pnt term);

pnt term_evar(pnt index){
  return cons2(ETAG_var,index,NIL);}

pnt compile_evar(pnt arg1,pnt arg2,pnt term){
  pnt index;
  for(index=0;arg1!=T;index++,arg1=TAIL(arg1)){
    if(tree_equal(HEAD(arg1),term)) return term_evar(index);}
  for(;arg2!=T;arg2=TAIL(arg2))
  if(tree_equal(HEAD(arg2),term)){
    for(arg2=TAIL(arg2);arg2!=T;arg2=TAIL(arg2)) index++;
    return term_evar(index);}
  return T;}

pnt compile_eterms(pnt caches,pnt arg1,pnt arg2,pnt terms){
  pnt term;
  if(terms==T) return T;
  term=compile_eterm(caches,arg1,arg2,HEAD(terms));
  push(term);
  return ret(1,pair(term,compile_eterms(caches,arg1,arg2,TAIL(terms))));}

pnt remove_lambdas(pnt code,pnt subterms){
  if(subterms==T) die("remove_lambdas: too few arguments of function");
  subterms=TAIL(subterms);
  for(code=HEAD(code);ROOT(code)==TTAG_LAMBDA;code=HEAD(code)){
    if(subterms==T) die("remove_lambdas: too few arguments of function");
    subterms=TAIL(subterms);}
  if(subterms!=T) die("remove_lambdas: too many arguments of function");
  return code;}

pnt compile_eterm(pnt caches,pnt arg1,pnt arg2,pnt term){
  pnt subterms; /* Uncompiled subterms of given term */
  pnt subterm0,subterm1,subterm2; /* d versions of subterms */
  pnt ref=term2ref(term);
  pnt idx=term2idx(term);
  pnt code; /* Code associated to root of given term */
  pnt tag; /* Hidden tag of root of given term */
  if(eq0(ref,Z)) return idx; /* string */
  if(eq0(idx,Z)) return aget(caches,ref); /* page symbol */
  code=mget4(caches,ref,ref,const_code,idx);
  if(code==T) return compile_evar(arg1,arg2,term); /* variable */
  if(ROOT(code)!=TAG_MAP) return term2arg1(term); /* quote */
  tag=TAIL(code);
  switch(tag){ /*dispatch on hidden tag */
  case CTAG_EMAP:
    code=HEAD(code); /* Get rid of TAG_MAP */
    subterms=TAIL(term);
    switch(ROOT(code)){
    case MTAG_CLOSURE:
      if(subterms!=T) die("compile_eterm: constant applied to arguments");
      return code;
/* GRD-2009-04-19: Ensure eager constants evaluated once only.
      code=HEAD(code);
      break;
*/
    case MTAG_INDIR:
      if(subterms!=T) die("compile_eterm: constant applied to arguments");
      return TAIL(code);
    case MTAG_LAMBDA:
      code=remove_lambdas(code,subterms);
      break;
    default:
      unexpected_tag("compile_eterm",code);}
    if(ROOT(code)!=TTAG_ECALL) unexpected_tag("compile_eterm",code);
    code=HEAD(code);
    return cons(ETAG_ecall,code,compile_eterms(caches,arg1,arg2,subterms));
/*
    subterms=TAIL(term);
    code=remove_lambdas(code,subterms);
    if(code==T) return T;
    switch(ROOT(code)){
    case TTAG_ECALL:
      code=HEAD(code);
      return cons(ETAG_ecall,code,compile_eterms(caches,arg1,arg2,subterms));
    case TAG_FALSE:
    case TAG_INT:
    case TAG_VECT:
    case TAG_PAIR:
    case TAG_EX:
    case TAG_MAP:
    case TAG_OBJ:
      if(subterms!=T) die("compile_eterm: unexpected tag");
      return code;
    default:
      spytagroot0(code);
      die("compile_eterm: unexpected tag");}
*/
  case CTAG_TRUE:
    return T;
  case ETAG_LET:
    subterm0=compile_eterm(caches,arg1,arg2,term2arg1(term));
    push(subterm0);
    term=term2arg2(term);
    arg1=pair(term2arg1(term),arg1);
    push(arg1);
    subterm1=compile_eterm(caches,arg1,arg2,term2arg2(term));
    return ret(2,cons(ETAG_LET,subterm0,subterm1));
  case ETAG_map:
    return cons(ETAG_map,T,closure(compile_term(caches,T,T,term2arg1(term)),T));
  case ETAG_digit0:
  case ETAG_digit1:
  case ETAG_digit2:
  case ETAG_digit3:
  case ETAG_digit4:
  case ETAG_digit5:
  case ETAG_digit6:
  case ETAG_digit7:
  case ETAG_digit8:
  case ETAG_digit9:
  case ETAG_times:
  case ETAG_plus:
  case ETAG_minus:
  case ETAG_and:
  case ETAG_or:
  case ETAG_then:
  case ETAG_pair:
  case ETAG_lt:
  case ETAG_le:
  case ETAG_equal:
  case ETAG_gt:
  case ETAG_ge:
  case ETAG_apply:
  case ETAG_boolp:
  case ETAG_catch:
  case ETAG_maptag1:
  case ETAG_div:
  case ETAG_head:
  case ETAG_intp:
  case ETAG_boolg:
  case ETAG_intg:
  case ETAG_mapg:
  case ETAG_objg:
  case ETAG_pairg:
  case ETAG_mapp:
  case ETAG_maptag:
  case ETAG_mod:
  case ETAG_norm:
  case ETAG_objp:
  case ETAG_pairp:
  case ETAG_raise:
  case ETAG_root:
  case ETAG_tail:
  case ETAG_untag:
  case ETAG_digitend:
  case ETAG_par:
  case ETAG_uplus:
  case ETAG_dplus:
  case ETAG_uminus:
  case ETAG_dminus:
  case ETAG_not:
  case ETAG_Base:
  case ETAG_ash:
  case ETAG_bottom:
  case ETAG_bt2bytes:
  case ETAG_bt2vects:
  case ETAG_bt2vector:
  case ETAG_ceiling:
  case ETAG_destruct:
  case ETAG_evenp:
  case ETAG_exception:
  case ETAG_false:
  case ETAG_floor:
  case ETAG_half:
  case ETAG_if:
  case ETAG_intlength:
  case ETAG_logand:
  case ETAG_logandc1:
  case ETAG_logandc2:
  case ETAG_logbitp:
  case ETAG_logcount:
  case ETAG_logeqv:
  case ETAG_logior:
  case ETAG_lognand:
  case ETAG_lognor:
  case ETAG_lognot:
  case ETAG_logorc1:
  case ETAG_logorc2:
  case ETAG_logtest:
  case ETAG_logxor:
  case ETAG_prenorm:
  case ETAG_notnot:
  case ETAG_object:
  case ETAG_print:
  case ETAG_round:
  case ETAG_spy:
  case ETAG_timer:
  case ETAG_trace:
  case ETAG_truncate:
  case ETAG_vector:
  case ETAG_vnorm:
  case ETAG_vempty:
  case ETAG_vindex:
  case ETAG_vlength:
  case ETAG_vprefix:
  case ETAG_vsubseq:
  case ETAG_vsuffix:
  case ETAG_v2bytes:
  case ETAG_v2vects:
  case ETAG_vt2bytes:
  case ETAG_vt2vects:
  case ETAG_vt2v:
  case ETAG_compile:
  case ETAG_ripemd:
  case ETAG_sl2rack:
  case ETAG_rack2sl:
    subterms=TAIL(term);
    if(subterms==T) return cons(tag,T,T);
    subterm0=compile_eterm(caches,arg1,arg2,HEAD(subterms));
    subterms=TAIL(subterms);
    if(subterms==T) return cons(tag,T,subterm0);
    push(subterm0);
    subterm1=compile_eterm(caches,arg1,arg2,HEAD(subterms));
    subterms=TAIL(subterms);
    if(subterms==T) return ret(1,cons(tag,subterm0,subterm1));
    push(subterm1);
    subterm2=compile_eterm(caches,arg1,arg2,HEAD(subterms));
    subterms=TAIL(subterms);
    if(subterms==T) return ret(2,cons(tag,subterm0,pair(subterm1,subterm2)));
    die("compile_eager_intro1: unexpected arity>3");
  default:
    die("compile_eterm: unexpected tag");
    return T; /* This never happens */}}



/**************************************************
 * Compilation, stage 6, compile eager definition *
 **************************************************/

/*
The given code is a code entry containing zero, one, or more
lambdas (depending on the arity of the construct) followed
by a ttag_ecall construct followed by a pair which is intended
to be overwritten with the compiled version of the entry.
The uncompiled definition of the entry was stored as the head
of the pair in stage 1. The entry was modified in stage 5.
*/

void compile_eager_def(pnt caches,pnt code){
  pnt root=code2pair(code);
  pnt def=HEAD(root);
  pnt arg=def2arg(def);
  pnt rhs=def2rhs(def);
  compile_overwrite(root,compile_eterm(caches,T,arg,rhs));}



/*********************************************
 * Compilation, stage 6, compile definitions *
 *********************************************/

void compile_code(pnt caches,pnt code){
  pnt hd;
  pnt tl;
  pnt tag;
  if(code==T) return;
  if(GD_PAIR(code)) die("Unexpected type in code");
  hd=HEAD(code);
  tl=TAIL(code);
  if(NO_INT(hd)){
    compile_code(caches,hd);
    compile_code(caches,tl);
    return;}
  if(ROOT(tl)!=TAG_MAP) return;    /* disregard quote and lambda */
  tag=TAIL(tl);
  if(tag==CTAG_TRUE) return;  /* disregard true */
  if(tag==CTAG_APPLY) return; /* disregard apply */
  if(tag==CTAG_IF) return;    /* disregard if */
  if(tag==ETAG_yy){HEAD(tl)=map_fix;return;}
  if(tag==CTAG_EMAP){compile_eager_def(caches,tl);return;}
  if(tag2eager[tag]==T){compile_eager_intro(tl,tag);return;}
  compile_lazy_def(caches,tl);}



/******************************
 * Compilation, invoke stages *
 ******************************/

/*  the value part of the codex */

void progress(char *msg,pnt ref){
  ref2string(ref);
  printf("%s %s\n",msg,bufrefname);}

pnt compile_cache(pnt ref,pnt cache){
  pnt rack=aget(cache,ref);
  pnt codex=mget2(rack,const_codex,ref);
  pnt code;
  pnt caches;
  pnt diagnose=maptag(cons(ETAG_bottom,T,T));
  pnt diagnose1;
  push(diagnose);
  push(T);
  TRUSTGC;
  initcode(codex);
  code=pop();
  rack=aput0(rack,const_code,code);
  rack=aput0(rack,const_diagnose,diagnose);
  cache=aput0(cache,ref,rack);
  caches=aget(rack,const_cache);
  caches=aput0(caches,ref,cache);
  push(caches);
  if(record){
    record=FALSE;
    record_patterns(caches,code);
    code_cclaim=name2code1(code,"compile-claim ( \" , \" , \" )");}
  recognize_intro(caches,code);
  strictness_analysis(caches,ref,code);
  change_eager(caches,code);
  compile_code(caches,code);
  UNTRUSTGC;
  diagnose1=map_apply(HEAD(code_cclaim),ref);
  diagnose1=map_apply(diagnose1,cache);
  push(diagnose1);
  diagnose1=ret(1,map_apply(diagnose1,maptag(cache)));
  HEAD(diagnose)=HEAD(diagnose1);
  pop();
  pop();
  return cache;}

pnt compile(pnt x){/* compile ( " ) */
  return compile_cache(mget1(x,Z),x);}



/*********************************
 * Read and compile transitively *
 *********************************/

/*
Load the cache with the given reference and all its transitively
referenced caches. The ref must be given as a vector (i.e. integer).
*/

void loadcache1(pnt ref){
  pnt bib;
  pnt bibref;
  pnt bibcache;
  pnt bibrack;
  pnt rack;
  pnt cache;
  pnt diagnose;
  ref2string(ref);
/*printf("Loading %s\n",bufrefname);*/
  if(aget(state,ref)!=T){/*printf("Already loaded\n");*/return;}
  state=mput2(state,ref,ref,Z);
  rack=readrack((char *)bufrefname);
  state=mput2(state,rack,ref,ref);
  bib=aget(rack,const_bib);
  if(bib==T) die("loadcache1: empty bibliography");
  if(NO_PAIR(bib)) die("loadcache1: unexpected type of bibliography");
  if(!eq0(ref,HEAD(bib))) die("loadcache1: cache corrupted");
  for(bib=TAIL(bib);bib!=T;bib=TAIL(bib)){
    if(GD_PAIR(bib)) die("loadcache1: unexpected type in bibliography");
    bibref=HEAD(bib);
    loadcache1(bibref);
    bibcache=aget(state,bibref);
    bibrack=aget(bibcache,bibref);
    state=mput2(state,bibrack,ref,bibref);
    state=mput4(state,bibcache,ref,ref,const_cache,bibref);}
  cache=aget(state,ref);
  diagnose=mget2(cache,ref,const_diagnose);
  push(diagnose);
  cache=compile_cache(ref,cache);
  HEAD(mget2(cache,ref,const_diagnose))=ret(1,diagnose);
  state=mput1(state,cache,ref);}

pnt loadcache(pnt ref){
  push(ref);
  loadcache1(ref);
  return ret(1,aget(state,ref));}

void preload(pnt verbose){
  pnt rack;
  pnt bib;
  pnt ref;
  pnt bibref;
  pnt bibcache;
  pnt bibrack;
  pnt cache;
  pnt diagnose;
  mbufstart=pages;
  mbufsize=pages_len;
  mbufpnt=pages;
  mbufend=pages+pages_len;
  while(mbufpnt!=mbufend){
    TRUSTGC;
    rack=readrack1();
    UNTRUSTGC;
    push(rack);
    bib=aget(rack,const_bib);
    if(bib==T) die("preload: empty bibliography");
    if(NO_PAIR(bib)) die("preload: unexpected type of bibliography");
    ref=HEAD(bib);
    if(NO_INT(ref)) die("preload: cache corrupted");
    state=mput2(state,ref,ref,Z);
    state=ret(1,mput2(state,rack,ref,ref));
    for(bib=TAIL(bib);bib!=T;bib=TAIL(bib)){
      if(GD_PAIR(bib)) die("preload: unexpected type in bibliography");
      bibref=HEAD(bib);
      if(aget(state,bibref)==T)
      die("preloaded pages can only reference previously preloaded pages");
      bibcache=aget(state,bibref);
      bibrack=aget(bibcache,bibref);
      state=mput2(state,bibrack,ref,bibref);
      state=mput4(state,bibcache,ref,ref,const_cache,bibref);}
    cache=aget(state,ref);
    diagnose=mget2(cache,ref,const_diagnose);
    push(diagnose);
    cache=compile_cache(ref,cache);
    HEAD(mget2(cache,ref,const_diagnose))=ret(1,diagnose);
    state=mput1(state,cache,ref);
    if(verbose){
      pnt name=nameget(state,ref,Z);
      if(name!=T) nameprint2(name);
      printf("\n");
      ref2string(ref);
      printf("%s\n",bufrefname);}}}



/**************
 * RIPEMD-160 *
 **************/

/* Taken from http://homes.esat.kuleuven.be/~bosselae/ripemd160.html */

/********************************************************************\
 *
 *      FILE:     rmd160.h
 *
 *      CONTENTS: Header file for a sample C-implementation of the
 *                RIPEMD-160 hash-function. 
 *      TARGET:   any computer with an ANSI C compiler
 *
 *      AUTHOR:   Antoon Bosselaers, ESAT-COSIC
 *      DATE:     1 March 1996
 *      VERSION:  1.0
 *
 *      Copyright (c) Katholieke Universiteit Leuven
 *      1996, All Rights Reserved
 *
 *  Conditions for use of the RIPEMD-160 Software
 *
 *  The RIPEMD-160 software is freely available for use under the terms and
 *  conditions described hereunder, which shall be deemed to be accepted by
 *  any user of the software and applicable on any use of the software:
 * 
 *  1. K.U.Leuven Department of Electrical Engineering-ESAT/COSIC shall for
 *     all purposes be considered the owner of the RIPEMD-160 software and of
 *     all copyright, trade secret, patent or other intellectual property
 *     rights therein.
 *  2. The RIPEMD-160 software is provided on an "as is" basis without
 *     warranty of any sort, express or implied. K.U.Leuven makes no
 *     representation that the use of the software will not infringe any
 *     patent or proprietary right of third parties. User will indemnify
 *     K.U.Leuven and hold K.U.Leuven harmless from any claims or liabilities
 *     which may arise as a result of its use of the software. In no
 *     circumstances K.U.Leuven R&D will be held liable for any deficiency,
 *     fault or other mishappening with regard to the use or performance of
 *     the software.
 *  3. User agrees to give due credit to K.U.Leuven in scientific publications 
 *     or communications in relation with the use of the RIPEMD-160 software 
 *     as follows: RIPEMD-160 software written by Antoon Bosselaers, 
 *     available at http://www.esat.kuleuven.be/~cosicart/ps/AB-9601/.
 *
\********************************************************************/

#ifndef  RMD160H           /* make sure this file is read only once */
#define  RMD160H

/********************************************************************/

/* typedef 8 and 32 bit types, resp.  */
/* adapt these, if necessary, 
   for your operating system and compiler */
typedef uint8_t  byte;
typedef uint32_t dword;


/********************************************************************/

/* macro definitions */

/* collect four bytes into one word: */
#define BYTES_TO_DWORD(strptr)                    \
            (((dword) *((strptr)+3) << 24) | \
             ((dword) *((strptr)+2) << 16) | \
             ((dword) *((strptr)+1) <<  8) | \
             ((dword) *(strptr)))

/* ROL(x, n) cyclically rotates x over n bits to the left */
/* x must be of an unsigned 32 bits type and 0 <= n < 32. */
#define ROL(x, n)        (((x) << (n)) | ((x) >> (32-(n))))

/* the five basic functions F(), G() and H() */
#define F(x, y, z)        ((x) ^ (y) ^ (z)) 
#define G(x, y, z)        (((x) & (y)) | (~(x) & (z))) 
#define H(x, y, z)        (((x) | ~(y)) ^ (z))
#define I(x, y, z)        (((x) & (z)) | ((y) & ~(z))) 
#define J(x, y, z)        ((x) ^ ((y) | ~(z)))
  
/* the ten basic operations FF() through III() */
#define FF(a, b, c, d, e, x, s)        {\
      (a) += F((b), (c), (d)) + (x);\
      (a) = ROL((a), (s)) + (e);\
      (c) = ROL((c), 10);\
   }
#define GG(a, b, c, d, e, x, s)        {\
      (a) += G((b), (c), (d)) + (x) + 0x5a827999UL;\
      (a) = ROL((a), (s)) + (e);\
      (c) = ROL((c), 10);\
   }
#define HH(a, b, c, d, e, x, s)        {\
      (a) += H((b), (c), (d)) + (x) + 0x6ed9eba1UL;\
      (a) = ROL((a), (s)) + (e);\
      (c) = ROL((c), 10);\
   }
#define II(a, b, c, d, e, x, s)        {\
      (a) += I((b), (c), (d)) + (x) + 0x8f1bbcdcUL;\
      (a) = ROL((a), (s)) + (e);\
      (c) = ROL((c), 10);\
   }
#define JJ(a, b, c, d, e, x, s)        {\
      (a) += J((b), (c), (d)) + (x) + 0xa953fd4eUL;\
      (a) = ROL((a), (s)) + (e);\
      (c) = ROL((c), 10);\
   }
#define FFF(a, b, c, d, e, x, s)        {\
      (a) += F((b), (c), (d)) + (x);\
      (a) = ROL((a), (s)) + (e);\
      (c) = ROL((c), 10);\
   }
#define GGG(a, b, c, d, e, x, s)        {\
      (a) += G((b), (c), (d)) + (x) + 0x7a6d76e9UL;\
      (a) = ROL((a), (s)) + (e);\
      (c) = ROL((c), 10);\
   }
#define HHH(a, b, c, d, e, x, s)        {\
      (a) += H((b), (c), (d)) + (x) + 0x6d703ef3UL;\
      (a) = ROL((a), (s)) + (e);\
      (c) = ROL((c), 10);\
   }
#define III(a, b, c, d, e, x, s)        {\
      (a) += I((b), (c), (d)) + (x) + 0x5c4dd124UL;\
      (a) = ROL((a), (s)) + (e);\
      (c) = ROL((c), 10);\
   }
#define JJJ(a, b, c, d, e, x, s)        {\
      (a) += J((b), (c), (d)) + (x) + 0x50a28be6UL;\
      (a) = ROL((a), (s)) + (e);\
      (c) = ROL((c), 10);\
   }

/********************************************************************/

/* function prototypes */

void MDinit(dword *MDbuf);
/*
 *  initializes MDbuffer to "magic constants"
 */

void compress(dword *MDbuf, dword *X);
/*
 *  the compression function.
 *  transforms MDbuf using message bytes X[0] through X[15]
 */

void MDfinish(dword *MDbuf, byte *strptr, dword lswlen, dword mswlen);
/*
 *  puts bytes from strptr into X and pad out; appends length 
 *  and finally, compresses the last block(s)
 *  note: length in bits == 8 * (lswlen + 2^32 mswlen).
 *  note: there are (lswlen mod 64) bytes left in strptr.
 */

#endif  /* RMD160H */

/*********************** end of file rmd160.h ***********************/



/********************************************************************\
 *
 *      FILE:     rmd160.c
 *
 *      CONTENTS: A sample C-implementation of the RIPEMD-160
 *                hash-function.
 *      TARGET:   any computer with an ANSI C compiler
 *
 *      AUTHOR:   Antoon Bosselaers, ESAT-COSIC
 *      DATE:     1 March 1996
 *      VERSION:  1.0
 *
 *      Copyright (c) Katholieke Universiteit Leuven
 *      1996, All Rights Reserved
 *
 *  Conditions for use of the RIPEMD-160 Software
 *
 *  The RIPEMD-160 software is freely available for use under the terms and
 *  conditions described hereunder, which shall be deemed to be accepted by
 *  any user of the software and applicable on any use of the software:
 * 
 *  1. K.U.Leuven Department of Electrical Engineering-ESAT/COSIC shall for
 *     all purposes be considered the owner of the RIPEMD-160 software and of
 *     all copyright, trade secret, patent or other intellectual property
 *     rights therein.
 *  2. The RIPEMD-160 software is provided on an "as is" basis without
 *     warranty of any sort, express or implied. K.U.Leuven makes no
 *     representation that the use of the software will not infringe any
 *     patent or proprietary right of third parties. User will indemnify
 *     K.U.Leuven and hold K.U.Leuven harmless from any claims or liabilities
 *     which may arise as a result of its use of the software. In no
 *     circumstances K.U.Leuven R&D will be held liable for any deficiency,
 *     fault or other mishappening with regard to the use or performance of
 *     the software.
 *  3. User agrees to give due credit to K.U.Leuven in scientific publications 
 *     or communications in relation with the use of the RIPEMD-160 software 
 *     as follows: RIPEMD-160 software written by Antoon Bosselaers, 
 *     available at http://www.esat.kuleuven.be/~cosicart/ps/AB-9601/.
 *
\********************************************************************/

/*  header files */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/********************************************************************/

void MDinit(dword *MDbuf)
{
   MDbuf[0] = 0x67452301UL;
   MDbuf[1] = 0xefcdab89UL;
   MDbuf[2] = 0x98badcfeUL;
   MDbuf[3] = 0x10325476UL;
   MDbuf[4] = 0xc3d2e1f0UL;

   return;
}

/********************************************************************/

void compress(dword *MDbuf, dword *X)
{
   dword aa = MDbuf[0],  bb = MDbuf[1],  cc = MDbuf[2],
         dd = MDbuf[3],  ee = MDbuf[4];
   dword aaa = MDbuf[0], bbb = MDbuf[1], ccc = MDbuf[2],
         ddd = MDbuf[3], eee = MDbuf[4];

   /* round 1 */
   FF(aa, bb, cc, dd, ee, X[ 0], 11);
   FF(ee, aa, bb, cc, dd, X[ 1], 14);
   FF(dd, ee, aa, bb, cc, X[ 2], 15);
   FF(cc, dd, ee, aa, bb, X[ 3], 12);
   FF(bb, cc, dd, ee, aa, X[ 4],  5);
   FF(aa, bb, cc, dd, ee, X[ 5],  8);
   FF(ee, aa, bb, cc, dd, X[ 6],  7);
   FF(dd, ee, aa, bb, cc, X[ 7],  9);
   FF(cc, dd, ee, aa, bb, X[ 8], 11);
   FF(bb, cc, dd, ee, aa, X[ 9], 13);
   FF(aa, bb, cc, dd, ee, X[10], 14);
   FF(ee, aa, bb, cc, dd, X[11], 15);
   FF(dd, ee, aa, bb, cc, X[12],  6);
   FF(cc, dd, ee, aa, bb, X[13],  7);
   FF(bb, cc, dd, ee, aa, X[14],  9);
   FF(aa, bb, cc, dd, ee, X[15],  8);
                             
   /* round 2 */
   GG(ee, aa, bb, cc, dd, X[ 7],  7);
   GG(dd, ee, aa, bb, cc, X[ 4],  6);
   GG(cc, dd, ee, aa, bb, X[13],  8);
   GG(bb, cc, dd, ee, aa, X[ 1], 13);
   GG(aa, bb, cc, dd, ee, X[10], 11);
   GG(ee, aa, bb, cc, dd, X[ 6],  9);
   GG(dd, ee, aa, bb, cc, X[15],  7);
   GG(cc, dd, ee, aa, bb, X[ 3], 15);
   GG(bb, cc, dd, ee, aa, X[12],  7);
   GG(aa, bb, cc, dd, ee, X[ 0], 12);
   GG(ee, aa, bb, cc, dd, X[ 9], 15);
   GG(dd, ee, aa, bb, cc, X[ 5],  9);
   GG(cc, dd, ee, aa, bb, X[ 2], 11);
   GG(bb, cc, dd, ee, aa, X[14],  7);
   GG(aa, bb, cc, dd, ee, X[11], 13);
   GG(ee, aa, bb, cc, dd, X[ 8], 12);

   /* round 3 */
   HH(dd, ee, aa, bb, cc, X[ 3], 11);
   HH(cc, dd, ee, aa, bb, X[10], 13);
   HH(bb, cc, dd, ee, aa, X[14],  6);
   HH(aa, bb, cc, dd, ee, X[ 4],  7);
   HH(ee, aa, bb, cc, dd, X[ 9], 14);
   HH(dd, ee, aa, bb, cc, X[15],  9);
   HH(cc, dd, ee, aa, bb, X[ 8], 13);
   HH(bb, cc, dd, ee, aa, X[ 1], 15);
   HH(aa, bb, cc, dd, ee, X[ 2], 14);
   HH(ee, aa, bb, cc, dd, X[ 7],  8);
   HH(dd, ee, aa, bb, cc, X[ 0], 13);
   HH(cc, dd, ee, aa, bb, X[ 6],  6);
   HH(bb, cc, dd, ee, aa, X[13],  5);
   HH(aa, bb, cc, dd, ee, X[11], 12);
   HH(ee, aa, bb, cc, dd, X[ 5],  7);
   HH(dd, ee, aa, bb, cc, X[12],  5);

   /* round 4 */
   II(cc, dd, ee, aa, bb, X[ 1], 11);
   II(bb, cc, dd, ee, aa, X[ 9], 12);
   II(aa, bb, cc, dd, ee, X[11], 14);
   II(ee, aa, bb, cc, dd, X[10], 15);
   II(dd, ee, aa, bb, cc, X[ 0], 14);
   II(cc, dd, ee, aa, bb, X[ 8], 15);
   II(bb, cc, dd, ee, aa, X[12],  9);
   II(aa, bb, cc, dd, ee, X[ 4],  8);
   II(ee, aa, bb, cc, dd, X[13],  9);
   II(dd, ee, aa, bb, cc, X[ 3], 14);
   II(cc, dd, ee, aa, bb, X[ 7],  5);
   II(bb, cc, dd, ee, aa, X[15],  6);
   II(aa, bb, cc, dd, ee, X[14],  8);
   II(ee, aa, bb, cc, dd, X[ 5],  6);
   II(dd, ee, aa, bb, cc, X[ 6],  5);
   II(cc, dd, ee, aa, bb, X[ 2], 12);

   /* round 5 */
   JJ(bb, cc, dd, ee, aa, X[ 4],  9);
   JJ(aa, bb, cc, dd, ee, X[ 0], 15);
   JJ(ee, aa, bb, cc, dd, X[ 5],  5);
   JJ(dd, ee, aa, bb, cc, X[ 9], 11);
   JJ(cc, dd, ee, aa, bb, X[ 7],  6);
   JJ(bb, cc, dd, ee, aa, X[12],  8);
   JJ(aa, bb, cc, dd, ee, X[ 2], 13);
   JJ(ee, aa, bb, cc, dd, X[10], 12);
   JJ(dd, ee, aa, bb, cc, X[14],  5);
   JJ(cc, dd, ee, aa, bb, X[ 1], 12);
   JJ(bb, cc, dd, ee, aa, X[ 3], 13);
   JJ(aa, bb, cc, dd, ee, X[ 8], 14);
   JJ(ee, aa, bb, cc, dd, X[11], 11);
   JJ(dd, ee, aa, bb, cc, X[ 6],  8);
   JJ(cc, dd, ee, aa, bb, X[15],  5);
   JJ(bb, cc, dd, ee, aa, X[13],  6);

   /* parallel round 1 */
   JJJ(aaa, bbb, ccc, ddd, eee, X[ 5],  8);
   JJJ(eee, aaa, bbb, ccc, ddd, X[14],  9);
   JJJ(ddd, eee, aaa, bbb, ccc, X[ 7],  9);
   JJJ(ccc, ddd, eee, aaa, bbb, X[ 0], 11);
   JJJ(bbb, ccc, ddd, eee, aaa, X[ 9], 13);
   JJJ(aaa, bbb, ccc, ddd, eee, X[ 2], 15);
   JJJ(eee, aaa, bbb, ccc, ddd, X[11], 15);
   JJJ(ddd, eee, aaa, bbb, ccc, X[ 4],  5);
   JJJ(ccc, ddd, eee, aaa, bbb, X[13],  7);
   JJJ(bbb, ccc, ddd, eee, aaa, X[ 6],  7);
   JJJ(aaa, bbb, ccc, ddd, eee, X[15],  8);
   JJJ(eee, aaa, bbb, ccc, ddd, X[ 8], 11);
   JJJ(ddd, eee, aaa, bbb, ccc, X[ 1], 14);
   JJJ(ccc, ddd, eee, aaa, bbb, X[10], 14);
   JJJ(bbb, ccc, ddd, eee, aaa, X[ 3], 12);
   JJJ(aaa, bbb, ccc, ddd, eee, X[12],  6);

   /* parallel round 2 */
   III(eee, aaa, bbb, ccc, ddd, X[ 6],  9); 
   III(ddd, eee, aaa, bbb, ccc, X[11], 13);
   III(ccc, ddd, eee, aaa, bbb, X[ 3], 15);
   III(bbb, ccc, ddd, eee, aaa, X[ 7],  7);
   III(aaa, bbb, ccc, ddd, eee, X[ 0], 12);
   III(eee, aaa, bbb, ccc, ddd, X[13],  8);
   III(ddd, eee, aaa, bbb, ccc, X[ 5],  9);
   III(ccc, ddd, eee, aaa, bbb, X[10], 11);
   III(bbb, ccc, ddd, eee, aaa, X[14],  7);
   III(aaa, bbb, ccc, ddd, eee, X[15],  7);
   III(eee, aaa, bbb, ccc, ddd, X[ 8], 12);
   III(ddd, eee, aaa, bbb, ccc, X[12],  7);
   III(ccc, ddd, eee, aaa, bbb, X[ 4],  6);
   III(bbb, ccc, ddd, eee, aaa, X[ 9], 15);
   III(aaa, bbb, ccc, ddd, eee, X[ 1], 13);
   III(eee, aaa, bbb, ccc, ddd, X[ 2], 11);

   /* parallel round 3 */
   HHH(ddd, eee, aaa, bbb, ccc, X[15],  9);
   HHH(ccc, ddd, eee, aaa, bbb, X[ 5],  7);
   HHH(bbb, ccc, ddd, eee, aaa, X[ 1], 15);
   HHH(aaa, bbb, ccc, ddd, eee, X[ 3], 11);
   HHH(eee, aaa, bbb, ccc, ddd, X[ 7],  8);
   HHH(ddd, eee, aaa, bbb, ccc, X[14],  6);
   HHH(ccc, ddd, eee, aaa, bbb, X[ 6],  6);
   HHH(bbb, ccc, ddd, eee, aaa, X[ 9], 14);
   HHH(aaa, bbb, ccc, ddd, eee, X[11], 12);
   HHH(eee, aaa, bbb, ccc, ddd, X[ 8], 13);
   HHH(ddd, eee, aaa, bbb, ccc, X[12],  5);
   HHH(ccc, ddd, eee, aaa, bbb, X[ 2], 14);
   HHH(bbb, ccc, ddd, eee, aaa, X[10], 13);
   HHH(aaa, bbb, ccc, ddd, eee, X[ 0], 13);
   HHH(eee, aaa, bbb, ccc, ddd, X[ 4],  7);
   HHH(ddd, eee, aaa, bbb, ccc, X[13],  5);

   /* parallel round 4 */   
   GGG(ccc, ddd, eee, aaa, bbb, X[ 8], 15);
   GGG(bbb, ccc, ddd, eee, aaa, X[ 6],  5);
   GGG(aaa, bbb, ccc, ddd, eee, X[ 4],  8);
   GGG(eee, aaa, bbb, ccc, ddd, X[ 1], 11);
   GGG(ddd, eee, aaa, bbb, ccc, X[ 3], 14);
   GGG(ccc, ddd, eee, aaa, bbb, X[11], 14);
   GGG(bbb, ccc, ddd, eee, aaa, X[15],  6);
   GGG(aaa, bbb, ccc, ddd, eee, X[ 0], 14);
   GGG(eee, aaa, bbb, ccc, ddd, X[ 5],  6);
   GGG(ddd, eee, aaa, bbb, ccc, X[12],  9);
   GGG(ccc, ddd, eee, aaa, bbb, X[ 2], 12);
   GGG(bbb, ccc, ddd, eee, aaa, X[13],  9);
   GGG(aaa, bbb, ccc, ddd, eee, X[ 9], 12);
   GGG(eee, aaa, bbb, ccc, ddd, X[ 7],  5);
   GGG(ddd, eee, aaa, bbb, ccc, X[10], 15);
   GGG(ccc, ddd, eee, aaa, bbb, X[14],  8);

   /* parallel round 5 */
   FFF(bbb, ccc, ddd, eee, aaa, X[12] ,  8);
   FFF(aaa, bbb, ccc, ddd, eee, X[15] ,  5);
   FFF(eee, aaa, bbb, ccc, ddd, X[10] , 12);
   FFF(ddd, eee, aaa, bbb, ccc, X[ 4] ,  9);
   FFF(ccc, ddd, eee, aaa, bbb, X[ 1] , 12);
   FFF(bbb, ccc, ddd, eee, aaa, X[ 5] ,  5);
   FFF(aaa, bbb, ccc, ddd, eee, X[ 8] , 14);
   FFF(eee, aaa, bbb, ccc, ddd, X[ 7] ,  6);
   FFF(ddd, eee, aaa, bbb, ccc, X[ 6] ,  8);
   FFF(ccc, ddd, eee, aaa, bbb, X[ 2] , 13);
   FFF(bbb, ccc, ddd, eee, aaa, X[13] ,  6);
   FFF(aaa, bbb, ccc, ddd, eee, X[14] ,  5);
   FFF(eee, aaa, bbb, ccc, ddd, X[ 0] , 15);
   FFF(ddd, eee, aaa, bbb, ccc, X[ 3] , 13);
   FFF(ccc, ddd, eee, aaa, bbb, X[ 9] , 11);
   FFF(bbb, ccc, ddd, eee, aaa, X[11] , 11);

   /* combine results */
   ddd += cc + MDbuf[1];               /* final result for MDbuf[0] */
   MDbuf[1] = MDbuf[2] + dd + eee;
   MDbuf[2] = MDbuf[3] + ee + aaa;
   MDbuf[3] = MDbuf[4] + aa + bbb;
   MDbuf[4] = MDbuf[0] + bb + ccc;
   MDbuf[0] = ddd;

   return;
}

/********************************************************************/

void MDfinish(dword *MDbuf, byte *strptr, dword lswlen, dword mswlen)
{
   unsigned int i;                                 /* counter       */
   dword        X[16];                             /* message words */

   memset(X, 0, 16*sizeof(dword));

   /* put bytes from strptr into X */
   for (i=0; i<(lswlen&63); i++) {
      /* byte i goes into word X[i div 4] at pos.  8*(i mod 4)  */
      X[i>>2] ^= (dword) *strptr++ << (8 * (i&3));
   }

   /* append the bit m_n == 1 */
   X[(lswlen>>2)&15] ^= (dword)1 << (8*(lswlen&3) + 7);

   if ((lswlen & 63) > 55) {
      /* length goes to next block */
      compress(MDbuf, X);
      memset(X, 0, 16*sizeof(dword));
   }

   /* append length in bits*/
   X[14] = lswlen << 3;
   X[15] = (lswlen >> 29) | (mswlen << 3);
   compress(MDbuf, X);

   return;
}

/************************ end of file rmd160.c **********************/



/********************************************************************\
 *
 *      FILE:     hashtest.c
 *
 *      CONTENTS: test file for sample C-implementation of
 *                RIPEMD-160 and RIPEMD128
 *        * command line arguments:                                         
 *           filename  -- compute hash code of file read binary     
 *           -sstring  -- print string & hashcode                   
 *           -t        -- perform time trial                        
 *           -x        -- execute standard test suite, ASCII input
 *        * for linkage with rmd128.c: define RMDsize as 128
 *          for linkage with rmd160.c: define RMDsize as 160 (default)
 *      TARGET:   any computer with an ANSI C compiler
 *
 *      AUTHOR:   Antoon Bosselaers, ESAT-COSIC
 *      DATE:     18 April 1996
 *      VERSION:  1.1
 *      HISTORY:  bug in RMDonemillion() corrected
 *
 *      Copyright (c) Katholieke Universiteit Leuven
 *      1996, All Rights Reserved
 *
 *  Conditions for use of the RIPEMD-160 Software
 *
 *  The RIPEMD-160 software is freely available for use under the terms and
 *  conditions described hereunder, which shall be deemed to be accepted by
 *  any user of the software and applicable on any use of the software:
 * 
 *  1. K.U.Leuven Department of Electrical Engineering-ESAT/COSIC shall for
 *     all purposes be considered the owner of the RIPEMD-160 software and of
 *     all copyright, trade secret, patent or other intellectual property
 *     rights therein.
 *  2. The RIPEMD-160 software is provided on an "as is" basis without
 *     warranty of any sort, express or implied. K.U.Leuven makes no
 *     representation that the use of the software will not infringe any
 *     patent or proprietary right of third parties. User will indemnify
 *     K.U.Leuven and hold K.U.Leuven harmless from any claims or liabilities
 *     which may arise as a result of its use of the software. In no
 *     circumstances K.U.Leuven R&D will be held liable for any deficiency,
 *     fault or other mishappening with regard to the use or performance of
 *     the software.
 *  3. User agrees to give due credit to K.U.Leuven in scientific publications 
 *     or communications in relation with the use of the RIPEMD-160 software 
 *     as follows: RIPEMD-160 software written by Antoon Bosselaers, 
 *     available at http://www.esat.kuleuven.be/~cosicart/ps/AB-9601/.
 *
\********************************************************************/
#ifndef RMDsize
#define RMDsize 160
#endif

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>

#define TEST_BLOCK_SIZE 8000
#define TEST_BLOCKS 1250
#define TEST_BYTES ((long)TEST_BLOCK_SIZE * (long)TEST_BLOCKS)

/********************************************************************/

byte *RMD(byte *message)
/*
 * returns RMD(message)
 * message should be a string terminated by '\0'
 */
{
   dword         MDbuf[RMDsize/32];   /* contains (A, B, C, D(, E))   */
   static byte   hashcode[RMDsize/8]; /* for final hash-value         */
   dword         X[16];               /* current 16-word chunk        */
   unsigned int  i;                   /* counter                      */
   dword         length;              /* length in bytes of message   */
   dword         nbytes;              /* # of bytes not yet processed */

   /* initialize */
   MDinit(MDbuf);
   length = (dword)strlen((char *)message);

   /* process message in 16-word chunks */
   for (nbytes=length; nbytes > 63; nbytes-=64) {
      for (i=0; i<16; i++) {
         X[i] = BYTES_TO_DWORD(message);
         message += 4;
      }
      compress(MDbuf, X);
   }                                    /* length mod 64 bytes left */

   /* finish: */
   MDfinish(MDbuf, message, length, 0);

   for (i=0; i<RMDsize/8; i+=4) {
      hashcode[i]   =  MDbuf[i>>2];         /* implicit cast to byte  */
      hashcode[i+1] = (MDbuf[i>>2] >>  8);  /*  extracts the 8 least  */
      hashcode[i+2] = (MDbuf[i>>2] >> 16);  /*  significant bits.     */
      hashcode[i+3] = (MDbuf[i>>2] >> 24);
   }

   return (byte *)hashcode;
}

/********************************************************************/

void RMDstring(char *message)
{
   unsigned int  i;
   byte         *hashcode;

   hashcode = RMD((byte *)message);
   printf("%s\n",message);
   for (i=0; i<RMDsize/8; i++)
      printf("%02X", hashcode[i]);
   printf("\n");
}



/*********************
 * RIPEMD conversion *
 *********************/

unsigned char hashkey[RMDsize/4+1];

unsigned char *msg2hash(char *message){
  pnt i;
  byte *hashcode;
  hashcode=RMD((byte *)message);
  for (i=0;i<RMDsize/8;i++){
    hashkey[2*i+0]=pnt2hex(hashcode[i]>>4);
    hashkey[2*i+1]=pnt2hex(hashcode[i]>>0);}
  hashkey[RMDsize/4]=0;
  return hashkey;}

pnt vt2length(pnt x){
  if(x==T) return 0;
  switch(ROOT(x)){
  case TAG_FALSE:
  case TAG_MAP:
  case TAG_OBJ:
    return 0;
  case TAG_INT:
  case TAG_VECT:
    if(negative(x)) return 0;
    return vectorLength1(x);
  case TAG_PAIR:
    return vt2length(HEAD(x))+vt2length(TAIL(x));
  default:
    unexpected_tag("vt2length",x);
    return T; /* This never happens*/}}

dword RipemdBuf[RMDsize/32]; /* contains (A, B, C, D, E) */
dword RipemdChunk[16];       /* current 16-word chunk    */
pnt   RipemdIndex1;          /* index into RipemdChunk   */
pnt   RipemdIndex2;          /* index into RipemdChunk   */

void vt2hash_finish(dword lswlen,dword mswlen){
   /* append the bit m_n == 1 */
   RipemdChunk[(lswlen>>2)&15] ^= (dword)1 << (8*(lswlen&3) + 7);
   if ((lswlen & 63) > 55) {
      /* length goes to next block */
      compress(RipemdBuf, RipemdChunk);
      memset(RipemdChunk, 0, 16*sizeof(dword));}
   /* append length in bits*/
   RipemdChunk[14] = lswlen << 3;
   RipemdChunk[15] = (lswlen >> 29) | (mswlen << 3);
   compress(RipemdBuf, RipemdChunk);
   return;}

void byte2hash(pnt x){
  RipemdChunk[RipemdIndex1]|=(x&0xFF)<<RipemdIndex2;
  RipemdIndex2+=8;
  if(RipemdIndex2<32) return;
  RipemdIndex2=0;
  RipemdIndex1++;
  if(RipemdIndex1<16) return;
  RipemdIndex1=0;
  compress(RipemdBuf,RipemdChunk);
  memset(RipemdChunk,0,16*sizeof(dword));}

void vector2hash(pnt x){
  pnt i;
  for(;TAIL(x);x=TAIL(x))
  for(i=0;i<bytesize;i++)
  byte2hash(HEAD(x)>>(8*i));
  for(x=HEAD(x);x>255;x=x>>8)
  byte2hash(x);}

void vt2hash2(pnt x){
  if(x==T) return;
  switch(ROOT(x)){
  case TAG_FALSE:
  case TAG_MAP:
  case TAG_OBJ:
    return;
  case TAG_INT:
  case TAG_VECT:
    if(negative(x)) return;
    vector2hash(x);
    return;
  case TAG_PAIR:
    vt2hash2(HEAD(x));
    vt2hash2(TAIL(x));
    return;
  default:
    unexpected_tag("vt2length",x);}}

void vt2hash1(pnt source){
  pnt length=vt2length(source);
  RipemdIndex1=0;
  RipemdIndex2=0;
  MDinit(RipemdBuf);
  memset(RipemdChunk,0,16*sizeof(dword));
  vt2hash2(source);
  /* On Cygwin, length>>32 gives warning: right shift count >= width of type
     and compiles incorrectly */
  vt2hash_finish((dword)length,(dword)((length>>16)>>16));}

unsigned char *vt2hash(pnt source){
  pnt i;
  vt2hash1(source);
  for(i=0;i<RMDsize/8;i++){
    hashkey[2*i+0]=pnt2hex(RipemdBuf[i>>2]>>(8*(i&3)+4));
    hashkey[2*i+1]=pnt2hex(RipemdBuf[i>>2]>>(8*(i&3)+0));}
  hashkey[RMDsize/4]=0;
  return hashkey;}



/*****************************
 * Ripemd optimized function *
 *****************************/

pnt ripemd(pnt source){ /* ripemd ( " ) */
  pnt result,byte;
  int index;
  vt2hash1(source);
  result=T;
  for(index=RMDsize/8-1;index>=0;index--){
    byte=RipemdBuf[index>>2];
    byte=byte>>(8*(index&3));
    result=pair(bytevect[byte&0xFF],result);}
  return result;}



/* Essentially the same as vt2hash, but only accepts singleton
 * lists as input and produces a singleton list as output

pnt ripemd(pnt source){
  pnt i,j;
  int k;
  pnt length=vt2length(source);
  RipemdIndex1=0;
  RipemdIndex2=0;
  MDinit(RipemdBuf);
  memset(RipemdChunk,0,16*sizeof(dword));
  for(i=source;i!=T;i=TAIL(i)){
    if(NO_PAIR(i)) return X;
    j=HEAD(i);
    if(NO_INT(j)) return X;
    if(TAIL(j)!=NIL) return X;
    j=HEAD(j);
    if((j>>8)!=1) return X;
    byte2hash(j);}
  vt2hash_finish((dword)length,(dword)((length>>16)>>16));
  i=T;
  for(k=RMDsize/8-1;k>=0;k--){
    j=RipemdBuf[k>>2];
    j=j>>(8*(k&3));
    i=pair(bytevect[j&0xFF],i);}
  return i;}
*/



/***************************
 * Boot event construction *
 ***************************/

pnt strings2pnt(char **strings){
  pnt string;
  if(*strings==0) return T;
  string=str2vec(*strings);
  return pair(string,strings2pnt(strings+1));}

pnt fd_strings2pnt(){
  unsigned char line[PATHSIZE]; /* input line */
  pnt string;
  if(!readln(line)) return T;
  string=str2vec0(line);
  return pair(string,fd_strings2pnt());}

void add_boot_event(char *argv[],char *env[],pnt cache,pnt defaults){
  pnt argv1,env1,event1;
  push(argv1=strings2pnt(argv+2));/* argv[0]=/usr/bin/lgwam argv[1]=script */
  push(env1=strings2pnt(env));
  event1=pair(sym_boot,pair(argv1,pair(env1,pair(cache,pair(defaults,T)))));
  inputs=ret(3,pair(event1,inputs));/* unstack argv1, env1, and defaults */}



/*********************
 * MACHINE EXECUTION *
 *********************/

/*
As mentioned on the Logiweb "base" page, a Logiweb machine comprises
a Logiweb computing engine, an interface, a boot handler, and a cache.
The engine and interface is common to all Logiweb machines whereas
the boot handler h and cache c distinguishes one machine from the other.

The map2rnf and eval functions defined above implement the engine.
The interface is implemented in the following.

The architecture of the Logiweb machine is described in more detail
on the base page. But, in short, the handler h of a machine is an
executable function which can run on the engine. The return value
from h is piped to the interface. This return value is supposed to
be a list. The elements of that list are called "requests" and we
shall refer to the list as the "request list". The interface
"executes" each request, one at a time, and produces "replies" which
it collects into a "reply list" in opposite order of the associated
requests. If the interface meets an "exec" request then it discards
all further requests in the request list and instead applies the
handler to the reply list. There is more to it that this, but the
explanation above should give the picture.

A "virgin" Logiweb machine is able to execute read, write, extend,
exec, and quit requests. The exec and quit requests are special in
that they affect further request processing. Both requests make
the interface discard further requests in the request list, but they
differ in that the exec request makes the interface re-invoke the
handler whereas the quit request makes the interface return to the
operating system. A quit request contains a value to be returned to
the operating system using exit(). If the request list becomes empty,
the interface performs an implicit quit with return value 0.

A read request makes the interface read one byte from standard input
and push a read reply on the reply list. A write request makes the
interface write a vector-tree to standard output. The interface also
pushes a write reply onto the reply list, but the write reply contains
no information.

An extend request is much more complex than the other requests.
The semantics of extend requests is left implementation dependent, but
the intension of extend requests is that it allows to define new requests,
new so-called "interrupts", and new optimized functions. The present
implementation, however, only supports definition of new requests.

An extend request takes two arguments: a request and a vector-tree.
The interface flattens the vector-tree and pipes it to the C-compiler,
producing a dynamic link library (dll) which the interface then loads.
Then the interface stores an association from the given request to
the dll such that future instances of the given request makes the
interface call the dll. A Logiweb machine is said to be "virgin"
until first time it executes an extend request.

Dynamic link libraries communicate with the Logiweb machine using
"trigger functions" of type Trigger, i.e. of type (int *trigger(int)).

Trigger functions are exported from dynamic link libraries and are
called from the Logiweb machine. Each dynamic link library loaded
by the Logiweb machine must export a trigger function named "trigger".

The protocol for communication between Logiweb machine and dynamic
link library is described in the following.

We shall refer to the argument of a trigger function as the "input"
of the function and the return value as the "output". Input as well
as output is of type int, but input as well as output merely take
values in the range -1..255.

Communication between Logiweb machine and dynamic link library
comprises a sequence of "exchanges". One exchange cannot start before
the previous exchange is completed.

Exchanges between Logiweb machine and dynamic link library always
occur on the initiative of the Logiweb machine.

Exchanges occur in two phases named lgw2dll and dll2lgw. During the
lgw2dll phase, a sequence of bytes is passed from Logiweb machine to
dynamic link library. During the dll2lgw phase, a sequence of bytes is
passed the other way. The dll2lgw phase starts when the lgw2dll
phase ends. The whole exchange starts with the start of the
lgw2dll phase and ends with the end of the dll2lgw phase.

Each lgw2dll phase consists of a sequence of calls to the trigger
function. At each call, one byte (in the range 0..255) is passed as
input to the trigger function. The lgw2dll phase ends when the
Logiweb machine passes -1 to the trigger function. The output
from the trigger function is ignored during the lgw2dll phase,
but the trigger function should return 0, reserving other values
for possible, future extensions.

Each dll2lgw phase also consists of a sequence of calls to the
trigger function. At each call, -1 is passed as input to the
trigger functino. At each call, one byte (in the range 0..255)
is passed as output from the trigger function. The dll2lgw
phases ends when the trigger function outputs -1.
*/



/***************************
 * Read reply construction *
 ***************************/

void add_read_reply(){
  int c=getchar();
  if(c<0) return;
  if(c>255) return;
  inputs=pair(pair(sym_read,pair(bytevect[c],T)),inputs);}



/***************************
 * Exec reply construction *
 ***************************/

void add_exec_reply(pnt id,pnt process){
  push(id);
  push(process);
  inputs=ret(2,pair(pair(sym_exec,pair(id,pair(process,T))),inputs));}

void add_exit_reply(pnt process){
  add_exec_reply(tree_exit,process);}

void add_time_reply(pnt process){
  add_exec_reply(tree_time,process);}

void add_mem_reply(pnt process){
  add_exec_reply(tree_memory,process);}



/*****************************
 * Extend reply construction *
 *****************************/

void add_extend_reply(pnt s){
  inputs=pair(pair(sym_extend,pair(s,T)),inputs);}

void extend_success(){
  add_extend_reply(E);}

void extend_failure(){
  add_extend_reply(bytevect[0]);}



/***************************
 * Write request execution *
 ***************************/

void machine_write_char(pnt x){
  printf("%c",(unsigned char)x);}

void machine_write_string(pnt x){
  for(;TAIL(x);x=TAIL(x)){
    pnt i;
    pnt y=HEAD(x);
    for(i=0;i<bytesize;i++){
      machine_write_char(y);
      y=y>>8;}}
  for(x=HEAD(x);x>1;x=x>>8) machine_write_char(x);}

void machine_write(pnt x){
  if(x==T) return;
  switch(ROOT(x)){
  case TAG_FALSE:
  case TAG_EX:
  case TAG_MAP:
  case TAG_OBJ:
    return;
  case TAG_INT:
  case TAG_VECT:
    machine_write_string(x);
    return;
  case TAG_PAIR:
    machine_write(HEAD(x));
    machine_write(TAIL(x));
    return;
  default:
    printf("machine_write: unknown tag\n");
    spy0(x);
    backtrace(x,100);
    die("Goodbye");}}

void exec_write_request(){
  pnt arg;
  arg=TAIL(event);
  if(NO_PAIR(arg)) return;
  machine_write(HEAD(arg));}



/**************************
 * Quit request execution *
 **************************/

void exec_quit_request(){
  pnt arg;
  return_value=0;
  outputs=F;
  arg=TAIL(event);
  if(NO_PAIR(arg)) return;
  arg=HEAD(arg);
  if(NO_INT(arg)) return;
  return_value=(int)(spnt)(HEAD(arg));}



/**************************
 * Exec Request execution *
 **************************/

void exec_exec_request(){
  pnt arg;
  pnt process;
  pnt handler;
  arg=TAIL(event);
  if(NO_PAIR(arg)) die("exec_exec_request: malformed exec_request");
  process=HEAD(arg);
  arg=TAIL(arg);
  if(NO_PAIR(arg)) die("exec_exec_request: malformed exec_request");
  handler=HEAD(arg);
  add_exit_reply(process);
  outputs=untag(apply(handler,mapcons(inputs)));
  inputs=T;}



/**************************
 * Read request execution *
 **************************/

void exec_read_request(){
  add_read_reply();}



/****************************
 * Extend request execution *
 ****************************/

#define BUFSIZE 1000000
unsigned char buf[BUFSIZE];
int bufpnt=0;
int bufcnt=0;

void dldie(char *msg){
  printf("%s: %s\n",msg,dlerror());
  deathReport();
  exit(1);}

void bufferflush(int fd){
  ssize_t written;
  bufcnt=0;
  while(bufcnt<bufpnt){
    written=write(fd,&(buf[bufcnt]),bufpnt-bufcnt);
    if(written<0) pdie("write()");
    bufcnt+=written;}
  bufpnt=0;}

void putch(int fd,pnt c){
  if(bufpnt>=BUFSIZE) bufferflush(fd);
  buf[bufpnt++]=c;}

void vector2file(int fd,pnt x){
  pnt i;
  for(;TAIL(x);x=TAIL(x))
  for(i=0;i<bytesize;i++)
  putch(fd,HEAD(x)>>(8*i));
  for(x=HEAD(x);x>255;x=x>>8)
  putch(fd,x);}

void vt2file1(int fd,pnt x){
  if(x==T) return;
  switch(ROOT(x)){
  case TAG_FALSE:
  case TAG_MAP:
  case TAG_OBJ:
    return;
  case TAG_INT:
  case TAG_VECT:
    if(negative(x)) return;
    vector2file(fd,x);
    return;
  case TAG_PAIR:
    vt2file1(fd,HEAD(x));
    vt2file1(fd,TAIL(x));
    return;
  default:
    unexpected_tag("vt2file",x);}}

void vt2file(int fd,pnt x){
  bufpnt=0;
  vt2file1(fd,x);
  bufferflush(fd);}

/*
Create directories containing given path. Examples:
Given path abc/def/ghi create abc/ and abc/def/.
Given path abc/def/ghi/ create abc/, abc/def/, and abc/def/ghi/.
*/
void mkdir_p(char *path){
  char *bufpnt;
  int result=0;
  if(*path==0) return;
  for(bufpnt=path+1;*bufpnt!=0;bufpnt++)
  if(*bufpnt=='/'){*bufpnt=0;result=mkdir(path,0777);*bufpnt='/';}
  if(result==0||errno==EEXIST) return;
  printf("mkdir -p %s\n",path);
  pdie("mkdir()");}

void *vt2dl(pnt vt){
  char *path=get_lgw_path();
  char filename_so[PATHSIZE];
  char filename_c[PATHSIZE];
  char *hash=(char *)vt2hash(vt);
  int fd;
  pid_t pid;
  int status;
  void *dl;
  make_path(filename_so,PATHSIZE,path,"dll/",hash,".so");
  dl=dlopen(filename_so,RTLD_NOW);
  if(dl) return dl;
  make_path(filename_c,PATHSIZE,path,"dll/",hash,".c");
  mkdir_p(filename_c);
/*
  unsigned char filename_dir[PATHSIZE];
  make_path((char *)filename_dir,PATHSIZE,path,"","","dll/");
  if(mkdir(filename_dir,0777)<0) if(errno!=EEXIST){
    perror("mkdir()");
    return NIL;}
*/
  fd=open(filename_c,O_WRONLY|O_CREAT|O_EXCL,0666);
  if(fd>=0){vt2file(fd,vt);close(fd);}
  else if(errno!=EEXIST){printf("%s\n",filename_c);perror("open()");return NIL;}
  pid=fork();
  if(pid<0){perror("fork()");return NIL;}
  if(pid==0){
    execl("/usr/bin/gcc","/usr/bin/gcc","-x","c","-pipe",filename_c,
#ifndef __CYGWIN__
    "-fPIC",
#endif /* __CYGWIN__ */
    "-shared","-Wl,-soname,dl.so","-o",filename_so,(char *)NULL);
    perror("execl(/usr/bin/gcc)");
    return NIL;}
  if(wait(&status)<0){perror("wait()");return NIL;}
  if(!WIFEXITED(status)){printf("Abnormal termination of gcc()\n");return NIL;}
  if(WEXITSTATUS(status)!=0){printf("gcc() exited due to error\n");return NIL;}
  dl=dlopen(filename_so,RTLD_NOW);
  if(!dl){printf("dlopen(): %s\n",dlerror());return NIL;}
  return dl;}

pnt exec_extend_request1(){
  pnt arg;
  pnt arg1;
  pnt arg2;
  pnt ref;
  pnt idx;
  Trigger trigger;
  void *dl;
  arg=TAIL(event);
  if(NO_PAIR(arg)) return FALSE;
  arg1=HEAD(arg);
  arg=TAIL(arg);
  if(NO_PAIR(arg)) return FALSE; 
  arg2=HEAD(arg);
  if(NO_PAIR(arg1)) return FALSE;
  arg=HEAD(arg1);
  if(NO_PAIR(arg)) return FALSE;
  ref=HEAD(arg);
  if(NO_INT(ref)) return FALSE;
  arg=TAIL(arg);
  if(NO_PAIR(arg)) return FALSE;
  idx=HEAD(arg);
  if(NO_INT(idx)) return FALSE;
  dl=vt2dl(arg2);
  if(!dl) return FALSE;
  trigger=dlsym(dl,"trigger");
  if(!trigger){printf("dlsym(): %s\n",dlerror());return FALSE;}
  extensions=mput2(extensions,cons2(TAG_FALSE,(pnt)trigger,NIL),ref,idx);
  return TRUE;}

void exec_extend_request(){
  if(exec_extend_request1())
  extend_success();
  else
  extend_failure();}



/*****************************
 * Execution of other events *
 *****************************/

void dll_write_string(Trigger trigger,pnt x){
  pnt i;
  pnt y;
  for(;TAIL(x);x=TAIL(x)){
    y=HEAD(x);
    for(i=0;i<bytesize;i++){
      (*trigger)((int)(y&0xFF));
      y=y>>8;}}
  for(y=HEAD(x);y>1;y=y>>8) (*trigger)((int)(y&0xFF));}

void dll_write(Trigger trigger,pnt x){
  if(x==T) return;
  switch(ROOT(x)){
  case TAG_FALSE:
  case TAG_EX:
  case TAG_MAP:
  case TAG_OBJ:
    return;
  case TAG_INT:
  case TAG_VECT:
    dll_write_string(trigger,x);
    return;
  case TAG_PAIR:
    dll_write(trigger,HEAD(x));
    dll_write(trigger,TAIL(x));
    return;
  default:
    printf("dll_write: unknown tag\n");
    spy0(x);
    backtrace(x,100);
    die("Goodbye");}}

void exec_other_request(pnt root,pnt ref,pnt idx){
  pnt trigger1=mget2(extensions,ref,idx);
  Trigger trigger;
  pnt arg;
  int c;
  pnt list;
  if(trigger1==T){
    printf("Unknown event:\n");
    printf("ref=");
    spy0(ref);
    printf("idx=");
    spy0(idx);
    return;}
  trigger=(Trigger)(HEAD(trigger1));
  arg=TAIL(event);
  if(NO_PAIR(arg)) return;
  dll_write(trigger,HEAD(arg));
  list=T;
  for(;;){
    c=(*trigger)(-1);
    if(c<0) break;
    list=pair(bytevect[c],list);}
  if(list!=T){
    pnt before=T;
    pnt after=TAIL(list);
    for(;;){
      TAIL(list)=before;
      if(after==T) break;
      before=list;
      list=after;
      after=TAIL(list);}}
  inputs=pair(pair(root,pair(list,T)),inputs);}



/*********************
 * Machine execution *
 *********************/

pnt apply_handler1(pnt handler,pnt inputs){
  return untag1(map_apply(untag1(untag(handler)),inputs));}

pnt apply_handler(pnt handler,pnt inputs){
  return ret(2,untag(apply(save(untag(handler)),save(mapcons(inputs)))));}

int runmachine(pnt handler,char *argv[],char *env[],pnt cache,pnt defaults){
  pnt root;
  pnt ref;
  pnt idx;
  if(NO_MAP(handler)) die("runmachine: no handler found");
  inputs=T;
  add_boot_event(argv,env,cache,defaults);
  outputs=apply_handler(handler,inputs);
  inputs=T;
  while(outputs!=F){
    if(outputs==T) return 0;
    if(IS_EX(outputs)) die("runmachine: unhandled exception");
    if(GD_PAIR(outputs)) die("runmachine: output events do not form a list");
    event=head(outputs);
    outputs=tail(outputs);
    if(NO_PAIR(event)) die("runmachine: output events must be pairs");
    root=head(event);
    if(NO_PAIR(event)) die("runmachine: event roots must be pairs");
    ref=HEAD(root);
    if(NO_INT(ref)) die("runmachine: event references must be integers");
    idx=TAIL(root);
    if(NO_PAIR(idx)) die("runmachine: event root tails must be pairs");
    idx=HEAD(idx);
    if(NO_INT(idx)) die("runmachine: event indexes must be integers");
    if(ref!=Z) exec_other_request(root,ref,idx);
    else if(eq0(idx,const_quit)) exec_quit_request();
    else if(eq0(idx,const_write)) exec_write_request();
    else if(eq0(idx,const_exec)) exec_exec_request();
    else if(eq0(idx,const_read)) exec_read_request();
    else if(eq0(idx,const_extend)) exec_extend_request();
    else exec_other_request(root,ref,idx);}
  return return_value;}



/********************
 * Script execution *
 ********************/

pnt get_handler(pnt assoc,pnt index){
  if(tail(assoc)==T) return T;
  if(eq0(index,term2idx(term2arg1(term2arg1(assoc)))))
  return term2arg1(term2arg2(term2arg1(assoc)));
  return get_handler(term2arg2(assoc),index);}

pnt term2code(pnt caches,pnt term){
  pnt ref=term2ref(term);
  pnt idx=term2idx(term);
  pnt code;
  if(NO_INT(ref)) return T;
  if(NO_INT(idx)) return T;
  code=codeget(caches,ref,idx);
  if(NO_MAP(code)) return T;
  return code;}

/*
Interpret argv[2] as a script.
*/
int runscript(int argc,char *argv[],char *env[]){
  pnt ref;
  pnt cache;
  pnt handler;
  pnt defaults;
  unsigned char type[PATHSIZE]; /* script type */
  unsigned char pref[PATHSIZE]; /* page reference */
  unsigned char sref[PATHSIZE]; /* symbol reference */
  unsigned char sidx[PATHSIZE]; /* symbol index */
  unsigned char aref[PATHSIZE]; /* aspect reference */
  unsigned char aidx[PATHSIZE]; /* aspect index */
  unsigned char name[PATHSIZE]; /* machine name */
  preload(FALSE);
  pagereuse();
  if(argc<=2) xdie("Usage: lgwam script <filename>");
  if(mmap1("runscript",argv[2])) die("runscript: cannot open script file");
  readln(type);
  if(STREQ(argv[1],"testrun")){
    readln(pref);
    munmap1();
    ref=string2ref(pref);
    cache=loadcache(ref);
    testsuite();
    return 0;}
  else if(STREQ(type,"string")){
    readln(pref);
    readln(sidx);
    readln(aidx);
    push(defaults=fd_strings2pnt());
    munmap1();
    ref=string2ref(pref);
    cache=loadcache(ref);
    handler=codexget(state,ref);
    handler=aget(handler,Z);
    handler=aget(handler,str2vec0(sidx));
    handler=aget(handler,Z);
    handler=aget(handler,str2vec0(aidx));
    handler=def2rhs(handler);
    handler=term2arg1(handler);
    handler=term2code(state,handler);}
  else if(STREQ(type,"execute")){
    readln(pref);
    push(defaults=fd_strings2pnt());
    munmap1();
    ref=string2ref(pref);
    cache=loadcache(ref);
    handler=defget(state,ref,Z,const_execute);
    handler=def2rhs(handler);
    handler=term2code(state,handler);}
  else if(STREQ(type,"executables")){
    readln(pref);
    readln(name);
    push(defaults=fd_strings2pnt());
    munmap1();
    ref=string2ref(pref);
    cache=loadcache(ref);
    handler=defget(state,ref,Z,const_executables);
    handler=def2rhs(handler);
    handler=get_handler(handler,str2vec0(name));
    handler=term2code(state,handler);}
  else if(STREQ(type,"codex")){
    readln(pref);
    readln(sref);
    readln(sidx);
    readln(aref);
    readln(aidx);
    push(defaults=fd_strings2pnt());
    munmap1();
    ref=string2ref(pref);
    cache=loadcache(ref);
    handler=codexget(state,ref);
    handler=aget(handler,string2ref(sref));
    handler=aget(handler,str2int0(sidx));
    handler=aget(handler,string2ref(aref));
    handler=aget(handler,str2int0(aidx));
    handler=def2rhs(handler);
    handler=term2code(state,handler);}
  else if(STREQ(type,"assoc")){
    readln(pref);
    readln(sref);
    readln(sidx);
    readln(aref);
    readln(aidx);
    readln(name);
    push(defaults=fd_strings2pnt());
    munmap1();
    ref=string2ref(pref);
    cache=loadcache(ref);
    handler=codexget(state,ref);
    handler=aget(handler,string2ref(sref));
    handler=aget(handler,str2int0(sidx));
    handler=aget(handler,string2ref(aref));
    handler=aget(handler,str2int0(aidx));
    handler=def2rhs(handler);
    handler=get_handler(handler,str2vec0(name));
    handler=term2code(state,handler);}
  else{
    die("runscript: Unknown method");
    return 0; /* This never happens */}
  return runmachine(handler,argv,env,cache,defaults);}



/**************
 * TEST SUITE *
 **************/

/***********************
 * Auxiliary functions *
 ***********************/

/* Apply untagged function */
#define U_0(f)         U_0x(f)
#define U_1(f,x)       U_1x(f,save(x))
#define U_2(f,x,y)     U_2x(f,save(x),save(y))
#define U_3(f,x,y,z)   U_3x(f,save(x),save(y),save(z))
#define U_4(f,x,y,z,v) U_4x(f,save(x),save(y),save(z),save(v))
pnt U_0x(pnt f){
  return HEAD(f);}
pnt U_1x(pnt f,pnt x){
  return ret(1,map_apply(HEAD(f),x));}
pnt U_2x(pnt f,pnt x,pnt y){
  return ret(2,map_apply(map_apply(HEAD(f),x),y));}
pnt U_3x(pnt f,pnt x,pnt y,pnt z){
  return ret(3,map_apply(map_apply(map_apply(HEAD(f),x),y),z));}
pnt U_4x(pnt f,pnt x,pnt y,pnt z,pnt v){
  return ret(4,map_apply(map_apply(map_apply(map_apply(HEAD(f),x),y),z),v));}

/* Apply tagged function */
#define M_0(f)       M_0x(f)
#define M_1(f,x)     M_1x(f,save(x))
#define M_2(f,x,y)   M_2x(f,save(x),save(y))
#define M_3(f,x,y,z) M_3x(f,save(x),save(y),save(z))
pnt M_0x(pnt f)                  {return f;}
pnt M_1x(pnt f,pnt x)            {return ret(1,apply(f,x));}
pnt M_2x(pnt f,pnt x,pnt y)      {return ret(2,apply(apply(f,x),y));}
pnt M_3x(pnt f,pnt x,pnt y,pnt z){return ret(3,apply(apply(apply(f,x),y),z));}

/* Safe versions of functions */
/*#define PAIR(x,y) SCONS(TAG_PAIR,x,y)*/
/*#define EQ(x,y) (push(x),push(y),ret(2,eq(stack[sp],stack[sp+1])))*/
#define PAIR(x,y) ret(2,pair(save(x),save(y)))
#define EQ(x,y) ret(2,eq(save(x),save(y)))
#define APPLY(x,y) ret(2,apply(save(x),save(y)))

/* Term constructors */
pnt map_arg1(pnt x){
  return map_pair(closure(x,NIL),NIL);}
pnt map_arg2(pnt x,pnt y){
  return MAP_PAIR(closure(x,NIL),map_arg1(y));}
pnt map_arg3(pnt x,pnt y,pnt z){
  return MAP_PAIR(closure(x,NIL),map_arg2(y,z));}
pnt term_eval(pnt x){
  return term2rnf(x,NIL);}
pnt term_eval1(pnt x,pnt a){
  push(x);
  return ret(1,term2rnf(x,map_arg1(a)));}
pnt term_eval2(pnt x,pnt a,pnt b){
  push(x);
  return ret(1,term2rnf(x,map_arg2(a,b)));}
pnt term_eval3(pnt x,pnt a,pnt b,pnt c){
  push(x);
  return ret(1,term2rnf(x,map_arg3(a,b,c)));}
pnt term_if(pnt x,pnt y,pnt z){
  return term_apply(term_pair(y,z),x);}
pnt term_p(pnt head,pnt tail){
  return term_lambda(term_if(var_0,head,tail));}
pnt term_hd(pnt pair){
  return term_apply(pair,term_t);}
pnt term_tl(pnt pair){
  return term_apply(pair,term_f);}

/* Untagged cardinals */
#define UCARD(x) str2ucard(#x)
pnt str2ucard(char *str){
  return map_tail(map_tail(str2int(str)));}

/* Eager functions */
#define E0(fct) cons(ETAG_##fct,T,T)
#define E1(fct,x) cons(ETAG_##fct,T,x)
#define E2(fct,x,y) SCONS(ETAG_##fct,x,y)
#define E3(fct,x,y,z) SCONS(ETAG_##fct,x,PAIR(y,z))

pnt eval0(pnt term){
  push(term);
  return ret(1,eval(term,T));}

pnt eval1(pnt term,pnt x){
  push(term);
  push(pair(x,T));
  return ret(2,eval(term,TOP));}

pnt eval2(pnt term,pnt x,pnt y){
  push(term);
  push(x);
  push(pair(y,T));
  TOP=pair(x,TOP);
  return ret(3,eval(term,TOP));}

pnt term_ecall2(pnt fct,pnt arg1,pnt arg2){
  push(fct);
  push(arg1);
  push(pair(arg2,T));
  TOP=pair(arg1,TOP);
  return ret(3,term_ecall(fct,TOP));}



/***********************
 * Debugging functions *
 ***********************/

void diffdie(pnt x,pnt y,pnt z){
  spy0(x);
  spy0(y);
  spy0(z);
  die("diff");}

void diff0(pnt x,pnt y,pnt z){
  pnt x1;
  pnt y1;
spy0(z);
  if(x==T){if(y==T) return; else diffdie(x,y,z);}
  if(y==T) diffdie(x,y,z);
  x1=ROOT(x);
  y1=ROOT(y);
  x1=(x1==TAG_VECT)?TAG_INT:x1;
  y1=(y1==TAG_VECT)?TAG_INT:y1;
  if(x1!=y1) diffdie(x,y,z);
  switch(x1){
  case TAG_FALSE:
  case TAG_MAP:
    return;
  case TAG_INT:
    while(TRUE){
      if(HEAD(x)!=HEAD(y)) diffdie(x,y,z);
      x=TAIL(x);
      y=TAIL(y);
      if(!x){if(!y) return; else diffdie(x,y,z);}
      if(!y) diffdie(x,y,z);}
  case TAG_EX:
    diffdie(x,y,z);
  case TAG_PAIR:
  case TAG_OBJ:
    diff0(HEAD(x),HEAD(y),pair(N(0),z));
    diff0(TAIL(x),TAIL(y),pair(N(1),z));
    return;
  default:
    unexpected_tag("diff0",x);}}

void diff(pnt x,pnt y){
  push(x);
  push(y);
  diff0(x,y,T);
  pop();
  pop();}

#define EQFAIL {spy0(x);spy0(y);return FALSE;}
#define EQSUCC {return TRUE;}
#define EQCALL(n,z,x,y) (result=eq1(x,y),(result?0:eqspy(n,z)),result)

pnt eqspy(char *n,pnt z){
  printf("%s ",n);
  if(z==T){printf("T\n");return 0;}
  switch(ROOT(z)){
  case TAG_FALSE:
  case TAG_INT:
  case TAG_VECT:
    spy0(z);
    return 0;
  case TAG_MAP:
    printf("map\n");
    return 0;
  case TAG_PAIR:
    printf("pair\n");
    return 0;
  case TAG_OBJ:
    printf("object\n");
    return 0;
  default:
    unexpected_tag("eqspy",z);
    return T;}}

pnt eq1(pnt x,pnt y){
  pnt x1;
  pnt y1;
  pnt result;
  if(x==T){if(y==T) EQSUCC else EQFAIL}
  if(y==T) EQFAIL
  x1=ROOT(x);
  y1=ROOT(y);
  x1=(x1==TAG_VECT)?TAG_INT:x1;
  y1=(y1==TAG_VECT)?TAG_INT:y1;
  if(x1!=y1) EQFAIL
  switch(x1){
  case TAG_FALSE:
  case TAG_MAP:
    EQSUCC
  case TAG_INT:
    while(TRUE){
      if(HEAD(x)!=HEAD(y)) EQFAIL
      x=TAIL(x);
      y=TAIL(y);
      if(!x){if(!y) EQSUCC else EQFAIL}
      if(!y) EQFAIL}
  case TAG_EX: /* only for the sake of etst */
    if(eq1(HEAD(x),HEAD(y))) EQSUCC else EQFAIL
  case TAG_PAIR:
  case TAG_OBJ:
    return EQCALL("head",T,HEAD(x),HEAD(y))
    && EQCALL("tail",HEAD(x),TAIL(x),TAIL(y));
  default:
    unexpected_tag("eq1",x);
    return T;}}

pnt eq2(pnt x,pnt y){
  push(x);
  push(y);
  return ret(2,eq1(x,y));}



/************************
 * The testsuite itself *
 ************************/

#ifndef FULL_TESTSUITE
void testsuite(){
  testcnt=0;
  printf("This version of lgwam is compiled with a small testsuite\n");
  printf("which just contains a few test cases. To get the big one\n");
  printf("with thousands of tests, recompile lgwam.c using 'make full'.\n");
  etst(plus(N(-1000000000000),N(1000000)),N(-0999999000000))
  etst(times(N(-1000000000000),N(-100)),N(100000000000000))
  etst(vectorSubseq(W(2,abcdefghi),N( 2),N( 7)),V(cdefg))
  printf("%s tests succeeded.\n",ptoa(testcnt));}
#else
void testsuite(){
  pnt Z0000= 0;
  pnt Z0001= 1;
  pnt Z0010= 2;
  pnt Z0011= 3;
  pnt Z0100= 4;
  pnt Z0101= 5;
  pnt Z0110= 6;
  pnt Z0111= 7;
  pnt Z1000= 8;
/*pnt Z1001= 9; Happens to be unused */
  pnt Z1010=10;
/*pnt Z1011=11; Happens to be unused */
  pnt Z1100=12;
/*pnt Z1101=13; Happens to be unused */
  pnt Z1110=14;
/*pnt Z1111=15; Happens to be unused */
/*pnt M0000= 0-16; Happens to be unused */
  pnt M0001= 1-16;
/*pnt M0010= 2-16; Happens to be unused */
  pnt M0011= 3-16;
/*pnt M0100= 4-16; Happens to be unused */
  pnt M0101= 5-16;
  pnt M0110= 6-16;
  pnt M0111= 7-16;
  pnt M1000= 8-16;
/*pnt M1001= 9-16; Happens to be unused */
  pnt M1010=10-16;
  pnt M1011=11-16;
  pnt M1100=12-16;
  pnt M1101=13-16;
  pnt M1110=14-16;
/*pnt M1111=15-16; Happens to be unused */
  DEFC(TT,T);
  DEFC(FF,F);
  DEFC(XX,X);
  DEFC(I0,Z);
  DEFC(J1,M);
  DEFC(J2,JN(-2));
  DEFC(EE,E);
  DEFC(I0000,I[Z0000]);
  DEFC(I0001,I[Z0001]);
  DEFC(I0010,I[Z0010]);
  DEFC(I0011,I[Z0011]);
  DEFC(I0100,I[Z0100]);
  DEFC(I0101,I[Z0101]);
  DEFC(I0110,I[Z0110]);
  DEFC(I0111,I[Z0111]);
  DEFC(I1000,I[Z1000]);
/*DEFC(I1001,I[Z1001]); Happens to be unused */
  DEFC(I1010,I[Z1010]);
/*DEFC(I1011,I[Z1011]); Happens to be unused */
  DEFC(I1100,I[Z1100]);
/*DEFC(I1101,I[Z1101]); Happens to be unused */
  DEFC(I1110,I[Z1110]);
/*DEFC(I1111,I[Z1111]); Happens to be unused */
/*DEFC(J0000,JN(M0000)); Happens to be unused */
  DEFC(J0001,JN(M0001));
/*DEFC(J0010,JN(M0010)); Happens to be unused */
  DEFC(J0011,JN(M0011));
/*DEFC(J0100,JN(M0100)); Happens to be unused */
  DEFC(J0101,JN(M0101));
  DEFC(J0110,JN(M0110));
  DEFC(J0111,JN(M0111));
  DEFC(J1000,JN(M1000));
/*DEFC(J1001,JN(M1001)); Happens to be unused */
  DEFC(J1010,JN(M1010));
/*DEFC(J1011,JN(M1011)); Happens to be unused */
  DEFC(J1100,JN(M1100));
  DEFC(J1101,JN(M1101));
  DEFC(J1110,JN(M1110));
/*DEFC(J1111,JN(M1111)); Happens to be unused */
/*DEFC(I0000_0000,intcons(Z0000,I0000)); Happens to be unused */
  DEFC(I0001_1000,intcons(Z0001,I1000));
/*DEFC(I0010_0100,intcons(Z0010,I0100)); Happens to be unused */
  DEFC(I0011_1100,intcons(Z0011,I1100));
  DEFC(I0100_0010,intcons(Z0100,I0010));
  DEFC(I0101_1010,intcons(Z0101,I1010));
  DEFC(I0110_0110,intcons(Z0110,I0110));
  DEFC(I0111_1110,intcons(Z0111,I1110));
  DEFC(I1000_0001,intcons(Z1000,I0001));
/*DEFC(I1001_1001,intcons(Z1001,I1001)); Happens to be unused */
  DEFC(I1010_0101,intcons(Z1010,I0101));
/*DEFC(I1011_1101,intcons(Z1011,I1101)); Happens to be unused */
  DEFC(I1100_0011,intcons(Z1100,I0011));
/*DEFC(I1101_1011,intcons(Z1101,I1011)); Happens to be unused */
/*DEFC(I1110_0111,intcons(Z1110,I0111)); Happens to be unused */
/*DEFC(I1111_1111,intcons(Z1111,I1111)); Happens to be unused */
/*DEFC(J0000_0000,intcons(M0000,J0000)); Happens to be unused */
  DEFC(J0001_1000,intcons(M0001,J1000));
/*DEFC(J0010_0100,intcons(M0010,J0100)); Happens to be unused */
  DEFC(J0011_1100,intcons(M0011,J1100));
/*DEFC(J0100_0010,intcons(M0100,J0010)); Happens to be unused */
  DEFC(J0101_1010,intcons(M0101,J1010));
  DEFC(J0110_0110,intcons(M0110,J0110));
  DEFC(J0111_1110,intcons(M0111,J1110));
  DEFC(J1000_0001,intcons(M1000,J0001));
/*DEFC(J1001_1001,intcons(M1001,J1001)); Happens to be unused */
  DEFC(J1010_0101,intcons(M1010,J0101));
  DEFC(J1011_1101,intcons(M1011,J1101));
  DEFC(J1100_0011,intcons(M1100,J0011));
/*DEFC(J1101_1011,intcons(M1101,J1011)); Happens to be unused */
  DEFC(J1110_0111,intcons(M1110,J0111));
/*DEFC(J1111_1111,intcons(M1111,J1111)); Happens to be unused */
  DEFC(I12 ,intcons(1,intcons(2,Z)));
  DEFC(I123,intcons(1,intcons(2,intcons(3,Z))));
  DEFC(I923,intcons(9,intcons(2,intcons(3,Z))));
/*DEFC(I193,intcons(1,intcons(9,intcons(3,Z)))); Happens to be unused */
  DEFC(I129,intcons(1,intcons(2,intcons(9,Z))));
  DEFC(J12 ,intcons(1,intcons(2,J1)));
  DEFC(J123,intcons(1,intcons(2,intcons(3,J1))));
  DEFC(J923,intcons(9,intcons(2,intcons(3,J1))));
/*DEFC(J193,intcons(1,intcons(9,intcons(3,J1)))); Happens to be unused */
  DEFC(J129,intcons(1,intcons(2,intcons(9,J1))));
  DEFC(Z0,Z);
  DEFC(Z1,M);
  DEFC(Z2,I1000_0001);
  DEFC(Z3,J1000_0001);
  DEFC(P0,pair(T,T));
  DEFC(P1,pair(T,F));
  DEFC(P2,pair(F,T));
  DEFC(P3,pair(F,F));
  DEFC(X0,raise0(I[0]));
  DEFC(X1,raise0(I[1]));
  DEFC(X2,raise0(I[2]));
  DEFC(X3,raise0(I[3]));
  DEFC(B0,objcons(I[1],I[2],I[3]));
  DEFC(B1,objcons(I[9],I[2],I[3]));
  DEFC(B2,objcons(I[1],I[9],I[3]));
  DEFC(B3,objcons(I[1],I[2],I[9]));
  DEFC(B,E0(bottom));
  DEFC(BB,B);

  DEFC(UT            ,map_t);
  DEFC(UF            ,map_f);
  DEFC(UPTT          ,map_pair(map_t,map_t));
  DEFC(UPTF          ,map_pair(map_t,map_f));
  DEFC(UPFT          ,map_pair(map_f,map_t));
  DEFC(UPFF          ,map_pair(map_f,map_f));
  DEFC(UPPTTPTT      ,map_pair(UPTT,UPTT));
  DEFC(UPPTTPTF      ,map_pair(UPTT,UPTF));
  DEFC(UPPTTPFT      ,map_pair(UPTT,UPFT));
  DEFC(UPPTTPFF      ,map_pair(UPTT,UPFF));
  DEFC(UPPTFPTT      ,map_pair(UPTF,UPTT));
  DEFC(UPPTFPTF      ,map_pair(UPTF,UPTF));
  DEFC(UPPTFPFT      ,map_pair(UPTF,UPFT));
  DEFC(UPPTFPFF      ,map_pair(UPTF,UPFF));
  DEFC(UPPFTPTT      ,map_pair(UPFT,UPTT));
  DEFC(UPPFTPTF      ,map_pair(UPFT,UPTF));
  DEFC(UPPFTPFT      ,map_pair(UPFT,UPFT));
  DEFC(UPPFTPFF      ,map_pair(UPFT,UPFF));
  DEFC(UPPFFPTT      ,map_pair(UPFF,UPTT));
  DEFC(UPPFFPTF      ,map_pair(UPFF,UPTF));
  DEFC(UPPFFPFT      ,map_pair(UPFF,UPFT));
  DEFC(UPPFFPFF      ,map_pair(UPFF,UPFF));

  DEFC(term_i  ,term_lambda(var_0));
  DEFC(term_k  ,term_lambda(term_lambda(var_1)));
  DEFC(term_ki ,term_lambda(term_lambda(var_0)));
  DEFC(term_y1 ,term_lambda(term_apply(var_1,term_apply(var_0,var_0))));
  DEFC(term_y  ,term_lambda(term_apply(term_y1,term_y1)));
  DEFC(term_yk ,term_apply(term_y,term_k));
  DEFC(term_yki,term_apply(term_y,term_ki));
  DEFC(term_ptx,term_lambda(term_pair(term_t,var_0)));
  DEFC(term_pxt,term_lambda(term_pair(var_0,term_t)));
  DEFC(term_pxy,term_lambda(term_lambda(term_pair(var_1,var_0))));
  DEFC(term_pyx,term_lambda(term_lambda(term_pair(var_0,var_1))));

  DEFC(UI  ,term_eval(term_i  ));
  DEFC(UK  ,term_eval(term_k  ));
  DEFC(UKI ,term_eval(term_ki ));
/*DEFC(UY1 ,term_eval(term_y1 )); Happens to be unused */
  DEFC(UY  ,term_eval(term_y  ));
  DEFC(UYK ,term_eval(term_yk ));
  DEFC(UYKI,term_eval(term_yki));
  DEFC(UPTX,term_eval(term_ptx));
  DEFC(UPXT,term_eval(term_pxt));
  DEFC(UPXY,term_eval(term_pxy));
  DEFC(UPYX,term_eval(term_pyx));

  DEFC(MT            ,mapcons(UT      ));
  DEFC(MF            ,mapcons(UF      ));
  DEFC(MPTT          ,mapcons(UPTT    ));
  DEFC(MPTF          ,mapcons(UPTF    ));
  DEFC(MPFT          ,mapcons(UPFT    ));
  DEFC(MPFF          ,mapcons(UPFF    ));
  DEFC(MPPTTPTT      ,mapcons(UPPTTPTT));
  DEFC(MPPTTPTF      ,mapcons(UPPTTPTF));
  DEFC(MPPTTPFT      ,mapcons(UPPTTPFT));
  DEFC(MPPTTPFF      ,mapcons(UPPTTPFF));
  DEFC(MPPTFPTT      ,mapcons(UPPTFPTT));
  DEFC(MPPTFPTF      ,mapcons(UPPTFPTF));
  DEFC(MPPTFPFT      ,mapcons(UPPTFPFT));
  DEFC(MPPTFPFF      ,mapcons(UPPTFPFF));
  DEFC(MPPFTPTT      ,mapcons(UPPFTPTT));
  DEFC(MPPFTPTF      ,mapcons(UPPFTPTF));
  DEFC(MPPFTPFT      ,mapcons(UPPFTPFT));
  DEFC(MPPFTPFF      ,mapcons(UPPFTPFF));
  DEFC(MPPFFPTT      ,mapcons(UPPFFPTT));
  DEFC(MPPFFPTF      ,mapcons(UPPFFPTF));
  DEFC(MPPFFPFT      ,mapcons(UPPFFPFT));
  DEFC(MPPFFPFF      ,mapcons(UPPFFPFF));

  DEFC(MI  ,mapcons(UI  ));
  DEFC(MK  ,mapcons(UK  ));
  DEFC(MKI ,mapcons(UKI ));
/*DEFC(MY1 ,mapcons(UY1 )); Happens to be unused */
  DEFC(MY  ,mapcons(UY  ));
  DEFC(MYK ,mapcons(UYK ));
  DEFC(MYKI,mapcons(UYKI));
  DEFC(MPTX,mapcons(UPTX));
  DEFC(MPXT,mapcons(UPXT));
  DEFC(MPXY,mapcons(UPXY));
  DEFC(MPYX,mapcons(UPYX));

  DEFC(M0,MT);
  DEFC(M1,MF);
  DEFC(M2,MK);
  DEFC(M3,MKI);

/* UB is Bottom: map2rnf(UB) runs forever */
  DEFC(term_b1,term_lambda(term_apply(var_0,var_0)));
  DEFC(term_b ,term_apply(term_b1,term_b1));
  DEFC(UB,closure(term_b,T));
  DEFC(MB,mapcons(UB));

  DEFC(call_if,term_if(var_0,var_1,var_2));
  DEFC(call_apply,term_apply(var_1,var_0));
#define CALL_HEAD(x) term_call(call_apply,pair(term_t,pair(x,NIL)))
  DEFC(lambda_head,map_lambda(CALL_HEAD(var_0),NIL));

  printf("Testsuite\n");

  ttst(true())

  ftst(false())

  ttst(eq(TT,TT))
  ftst(eq(TT,FF))
  ftst(eq(TT,I0))
  ftst(eq(TT,EE))
  ftst(eq(TT,P0))
  ftst(eq(TT,M0))
  ftst(eq(TT,B0))
  ftst(eq(FF,TT))
  ttst(eq(FF,FF))
  ftst(eq(FF,I0))
  ftst(eq(FF,EE))
  ftst(eq(FF,P0))
  ftst(eq(FF,M0))
  ftst(eq(FF,B0))
  ftst(eq(I0,TT))
  ftst(eq(I0,FF))
  ttst(eq(I0,I0))
  ftst(eq(I0,EE))
  ftst(eq(I0,P0))
  ftst(eq(I0,M0))
  ftst(eq(I0,B0))
  ftst(eq(EE,TT))
  ftst(eq(EE,FF))
  ftst(eq(EE,I0))
  ttst(eq(EE,EE))
  ftst(eq(EE,P0))
  ftst(eq(EE,M0))
  ftst(eq(EE,B0))
  ftst(eq(P0,TT))
  ftst(eq(P0,FF))
  ftst(eq(P0,I0))
  ftst(eq(P0,EE))
  ttst(eq(P0,P0))
  ftst(eq(P0,M0))
  ftst(eq(P0,B0))
  ftst(eq(M0,TT))
  ftst(eq(M0,FF))
  ftst(eq(M0,I0))
  ftst(eq(M0,EE))
  ftst(eq(M0,P0))
  ttst(eq(M0,M0))
  ftst(eq(M0,B0))
  ftst(eq(B0,TT))
  ftst(eq(B0,FF))
  ftst(eq(B0,I0))
  ftst(eq(B0,EE))
  ftst(eq(B0,P0))
  ftst(eq(B0,M0))
  ttst(eq(B0,B0))

  ttst(eq(I12 ,I12 ))
  ftst(eq(I12 ,I123))
  ftst(eq(I12 ,I923))
  ftst(eq(I12 ,I129))
  ftst(eq(I123,I12 ))
  ttst(eq(I123,I123))
  ftst(eq(I123,I923))
  ftst(eq(I123,I129))
  ftst(eq(I923,I12 ))
  ftst(eq(I923,I123))
  ttst(eq(I923,I923))
  ftst(eq(I923,I129))
  ftst(eq(I129,I12 ))
  ftst(eq(I129,I123))
  ftst(eq(I129,I923))
  ttst(eq(I129,I129))

  ttst(eq(J12 ,J12 ))
  ftst(eq(J12 ,J123))
  ftst(eq(J12 ,J923))
  ftst(eq(J12 ,J129))
  ftst(eq(J123,J12 ))
  ttst(eq(J123,J123))
  ftst(eq(J123,J923))
  ftst(eq(J123,J129))
  ftst(eq(J923,J12 ))
  ftst(eq(J923,J123))
  ttst(eq(J923,J923))
  ftst(eq(J923,J129))
  ftst(eq(J129,J12 ))
  ftst(eq(J129,J123))
  ftst(eq(J129,J923))
  ttst(eq(J129,J129))

  ttst(eq(EE,I[1]))

  ttst(eq(P0,P0))
  ftst(eq(P0,P1))
  ftst(eq(P0,P2))
  ftst(eq(P0,P3))
  ftst(eq(P1,P0))
  ttst(eq(P1,P1))
  ftst(eq(P1,P2))
  ftst(eq(P1,P3))
  ftst(eq(P2,P0))
  ftst(eq(P2,P1))
  ttst(eq(P2,P2))
  ftst(eq(P2,P3))
  ftst(eq(P3,P0))
  ftst(eq(P3,P1))
  ftst(eq(P3,P2))
  ttst(eq(P3,P3))

  ttst(eq(B0,B0))
  ftst(eq(B0,B1))
  ftst(eq(B0,B2))
  ftst(eq(B0,B3))
  ftst(eq(B1,B0))
  ttst(eq(B1,B1))
  ftst(eq(B1,B2))
  ftst(eq(B1,B3))
  ftst(eq(B2,B0))
  ftst(eq(B2,B1))
  ttst(eq(B2,B2))
  ftst(eq(B2,B3))
  ftst(eq(B3,B0))
  ftst(eq(B3,B1))
  ftst(eq(B3,B2))
  ttst(eq(B3,B3))

  ttst(EQ(PAIR(PAIR(T,F),PAIR(Z,M)),PAIR(PAIR(T,F),PAIR(Z,M))))
  ftst(EQ(PAIR(PAIR(T,F),PAIR(Z,M)),PAIR(PAIR(F,F),PAIR(Z,M))))
  ftst(EQ(PAIR(PAIR(T,F),PAIR(Z,M)),PAIR(PAIR(T,T),PAIR(Z,M))))
  ftst(EQ(PAIR(PAIR(T,F),PAIR(Z,M)),PAIR(PAIR(T,F),PAIR(T,M))))
  ftst(EQ(PAIR(PAIR(T,F),PAIR(Z,M)),PAIR(PAIR(T,F),PAIR(Z,F))))

  xtst(lognot(TT))
  xtst(lognot(FF))
  xtst(lognot(P0))
  xtst(lognot(M0))
  xtst(lognot(B0))
  etst(lognot(Z),M)
  etst(lognot(M),Z)
  etst(lognot(I0011),J1100)
  etst(lognot(J1010_0101),I0101_1010)

  xtst(logand(TT,TT))
  xtst(logand(TT,FF))
  xtst(logand(TT,I0))
  xtst(logand(TT,P0))
  xtst(logand(TT,M0))
  xtst(logand(TT,B0))
  xtst(logand(FF,TT))
  xtst(logand(FF,FF))
  xtst(logand(FF,I0))
  xtst(logand(FF,P0))
  xtst(logand(FF,M0))
  xtst(logand(FF,B0))
  xtst(logand(I0,TT))
  xtst(logand(I0,FF))
  xtst(logand(I0,P0))
  xtst(logand(I0,M0))
  xtst(logand(I0,B0))
  xtst(logand(P0,TT))
  xtst(logand(P0,FF))
  xtst(logand(P0,I0))
  xtst(logand(P0,P0))
  xtst(logand(P0,M0))
  xtst(logand(P0,B0))
  xtst(logand(M0,TT))
  xtst(logand(M0,FF))
  xtst(logand(M0,I0))
  xtst(logand(M0,P0))
  xtst(logand(M0,M0))
  xtst(logand(M0,B0))
  xtst(logand(B0,TT))
  xtst(logand(B0,FF))
  xtst(logand(B0,I0))
  xtst(logand(B0,P0))
  xtst(logand(B0,M0))
  xtst(logand(B0,B0))
  etst(logand(I0011     ,I0101     ),I0001     )
  etst(logand(I1100     ,I1010     ),I1000     )
  etst(logand(I0011_1100,I0101_1010),I0001_1000)
  etst(logand(I0011     ,I0101_1010),I0001     )
  etst(logand(I0101_1010,I0011     ),I0001     )
  etst(logand(I0011     ,J0101     ),I0001     )
  etst(logand(I1100     ,J1010     ),I1000     )
  etst(logand(I0011_1100,J0101_1010),I0001_1000)
  etst(logand(I0011     ,J0101_1010),I0001     )
  etst(logand(I0101_1010,J0011     ),intcons(Z0001,I1010))
  etst(logand(J0011     ,I0101     ),I0001     )
  etst(logand(J1100     ,I1010     ),I1000     )
  etst(logand(J0011_1100,I0101_1010),I0001_1000)
  etst(logand(J0011     ,I0101_1010),intcons(Z0001,I1010))
  etst(logand(J0101_1010,I0011     ),I0001     )
  etst(logand(J0011     ,J0101     ),J0001     )
  etst(logand(J1100     ,J1010     ),J1000     )
  etst(logand(J0011_1100,J0101_1010),J0001_1000)
  etst(logand(J0011     ,J0101_1010),intcons(M0001,J1010))
  etst(logand(J0101_1010,J0011     ),intcons(M0001,J1010))
  etst(logand(I0011_1100,I1000_0001),Z)
  etst(logand(EE,I0),I0)
  etst(logand(EE,J1),EE)
  etst(logand(I0,EE),I0)
  etst(logand(J1,EE),EE)

  xtst(logior(TT,TT))
  xtst(logior(TT,FF))
  xtst(logior(TT,I0))
  xtst(logior(TT,P0))
  xtst(logior(TT,M0))
  xtst(logior(TT,B0))
  xtst(logior(FF,TT))
  xtst(logior(FF,FF))
  xtst(logior(FF,I0))
  xtst(logior(FF,P0))
  xtst(logior(FF,M0))
  xtst(logior(FF,B0))
  xtst(logior(I0,TT))
  xtst(logior(I0,FF))
  xtst(logior(I0,P0))
  xtst(logior(I0,M0))
  xtst(logior(I0,B0))
  xtst(logior(P0,TT))
  xtst(logior(P0,FF))
  xtst(logior(P0,I0))
  xtst(logior(P0,P0))
  xtst(logior(P0,M0))
  xtst(logior(P0,B0))
  xtst(logior(M0,TT))
  xtst(logior(M0,FF))
  xtst(logior(M0,I0))
  xtst(logior(M0,P0))
  xtst(logior(M0,M0))
  xtst(logior(M0,B0))
  xtst(logior(B0,TT))
  xtst(logior(B0,FF))
  xtst(logior(B0,I0))
  xtst(logior(B0,P0))
  xtst(logior(B0,M0))
  xtst(logior(B0,B0))
  etst(logior(I0011     ,I0101     ),I0111     )
  etst(logior(I1100     ,I1010     ),I1110     )
  etst(logior(I0011_1100,I0101_1010),I0111_1110)
  etst(logior(I0011     ,I0101_1010),intcons(Z0111,I1010))
  etst(logior(I0101_1010,I0011     ),intcons(Z0111,I1010))
  etst(logior(I0011     ,J0101     ),J0111     )
  etst(logior(I1100     ,J1010     ),J1110     )
  etst(logior(I0011_1100,J0101_1010),J0111_1110)
  etst(logior(I0011     ,J0101_1010),intcons(M0111,J1010))
  etst(logior(I0101_1010,J0011     ),J0111     )
  etst(logior(J0011     ,I0101     ),J0111     )
  etst(logior(J1100     ,I1010     ),J1110     )
  etst(logior(J0011_1100,I0101_1010),J0111_1110)
  etst(logior(J0011     ,I0101_1010),J0111     )
  etst(logior(J0101_1010,I0011     ),intcons(M0111,J1010))
  etst(logior(J0011     ,J0101     ),J0111     )
  etst(logior(J1100     ,J1010     ),J1110     )
  etst(logior(J0011_1100,J0101_1010),J0111_1110)
  etst(logior(J0011     ,J0101_1010),J0111     )
  etst(logior(J0101_1010,J0011     ),J0111     )
  etst(logior(J0011_1100,J1110_0111),M)
  etst(logior(EE,I0),EE)
  etst(logior(EE,J1),J1)
  etst(logior(I0,EE),EE)
  etst(logior(J1,EE),J1)

  xtst(logxor(TT,TT))
  xtst(logxor(TT,FF))
  xtst(logxor(TT,I0))
  xtst(logxor(TT,P0))
  xtst(logxor(TT,M0))
  xtst(logxor(TT,B0))
  xtst(logxor(FF,TT))
  xtst(logxor(FF,FF))
  xtst(logxor(FF,I0))
  xtst(logxor(FF,P0))
  xtst(logxor(FF,M0))
  xtst(logxor(FF,B0))
  xtst(logxor(I0,TT))
  xtst(logxor(I0,FF))
  xtst(logxor(I0,P0))
  xtst(logxor(I0,M0))
  xtst(logxor(I0,B0))
  xtst(logxor(P0,TT))
  xtst(logxor(P0,FF))
  xtst(logxor(P0,I0))
  xtst(logxor(P0,P0))
  xtst(logxor(P0,M0))
  xtst(logxor(P0,B0))
  xtst(logxor(M0,TT))
  xtst(logxor(M0,FF))
  xtst(logxor(M0,I0))
  xtst(logxor(M0,P0))
  xtst(logxor(M0,M0))
  xtst(logxor(M0,B0))
  xtst(logxor(B0,TT))
  xtst(logxor(B0,FF))
  xtst(logxor(B0,I0))
  xtst(logxor(B0,P0))
  xtst(logxor(B0,M0))
  xtst(logxor(B0,B0))
  etst(logxor(I0011     ,I0101     ),I0110     )
  etst(logxor(I1100     ,I1010     ),I0110     )
  etst(logxor(I0011_1100,I0101_1010),I0110_0110)
  etst(logxor(I0011     ,I0101_1010),intcons(Z0110,I1010))
  etst(logxor(I0101_1010,I0011     ),intcons(Z0110,I1010))
  etst(logxor(I0011     ,J0101     ),J0110     )
  etst(logxor(I1100     ,J1010     ),J0110     )
  etst(logxor(I0011_1100,J0101_1010),J0110_0110)
  etst(logxor(I0011     ,J0101_1010),intcons(M0110,J1010))
  etst(logxor(I0101_1010,J0011     ),intcons(M0110,J0101))
  etst(logxor(J0011     ,I0101     ),J0110     )
  etst(logxor(J1100     ,I1010     ),J0110     )
  etst(logxor(J0011_1100,I0101_1010),J0110_0110)
  etst(logxor(J0011     ,I0101_1010),intcons(M0110,J0101))
  etst(logxor(J0101_1010,I0011     ),intcons(M0110,J1010))
  etst(logxor(J0011     ,J0101     ),I0110     )
  etst(logxor(J1100     ,J1010     ),I0110     )
  etst(logxor(J0011_1100,J0101_1010),I0110_0110)
  etst(logxor(J0011     ,J0101_1010),intcons(Z0110,I0101))
  etst(logxor(J0101_1010,J0011     ),intcons(Z0110,I0101))
  etst(logxor(J0101_1010,J0101_1010),Z)
  etst(logxor(J0101_1010,I1010_0101),M)
  etst(logxor(EE,I0),EE)
  etst(logxor(EE,J1),J2)
  etst(logxor(I0,EE),EE)
  etst(logxor(J1,EE),J2)

  xtst(logandc1(TT,TT))
  xtst(logandc1(TT,FF))
  xtst(logandc1(TT,I0))
  xtst(logandc1(TT,P0))
  xtst(logandc1(TT,M0))
  xtst(logandc1(TT,B0))
  xtst(logandc1(FF,TT))
  xtst(logandc1(FF,FF))
  xtst(logandc1(FF,I0))
  xtst(logandc1(FF,P0))
  xtst(logandc1(FF,M0))
  xtst(logandc1(FF,B0))
  xtst(logandc1(I0,TT))
  xtst(logandc1(I0,FF))
  xtst(logandc1(I0,P0))
  xtst(logandc1(I0,M0))
  xtst(logandc1(I0,B0))
  xtst(logandc1(P0,TT))
  xtst(logandc1(P0,FF))
  xtst(logandc1(P0,I0))
  xtst(logandc1(P0,P0))
  xtst(logandc1(P0,M0))
  xtst(logandc1(P0,B0))
  xtst(logandc1(M0,TT))
  xtst(logandc1(M0,FF))
  xtst(logandc1(M0,I0))
  xtst(logandc1(M0,P0))
  xtst(logandc1(M0,M0))
  xtst(logandc1(M0,B0))
  xtst(logandc1(B0,TT))
  xtst(logandc1(B0,FF))
  xtst(logandc1(B0,I0))
  xtst(logandc1(B0,P0))
  xtst(logandc1(B0,M0))
  xtst(logandc1(B0,B0))
  etst(logandc1(lognot(I0011     ),I0101     ),I0001     )
  etst(logandc1(lognot(I1100     ),I1010     ),I1000     )
  etst(logandc1(lognot(I0011_1100),I0101_1010),I0001_1000)
  etst(logandc1(lognot(I0011     ),I0101_1010),I0001     )
  etst(logandc1(lognot(I0101_1010),I0011     ),I0001     )
  etst(logandc1(lognot(I0011     ),J0101     ),I0001     )
  etst(logandc1(lognot(I1100     ),J1010     ),I1000     )
  etst(logandc1(lognot(I0011_1100),J0101_1010),I0001_1000)
  etst(logandc1(lognot(I0011     ),J0101_1010),I0001     )
  etst(logandc1(lognot(I0101_1010),J0011     ),intcons(Z0001,I1010))
  etst(logandc1(lognot(J0011     ),I0101     ),I0001     )
  etst(logandc1(lognot(J1100     ),I1010     ),I1000     )
  etst(logandc1(lognot(J0011_1100),I0101_1010),I0001_1000)
  etst(logandc1(lognot(J0011     ),I0101_1010),intcons(Z0001,I1010))
  etst(logandc1(lognot(J0101_1010),I0011     ),I0001     )
  etst(logandc1(lognot(J0011     ),J0101     ),J0001     )
  etst(logandc1(lognot(J1100     ),J1010     ),J1000     )
  etst(logandc1(lognot(J0011_1100),J0101_1010),J0001_1000)
  etst(logandc1(lognot(J0011     ),J0101_1010),intcons(M0001,J1010))
  etst(logandc1(lognot(J0101_1010),J0011     ),intcons(M0001,J1010))
  etst(logandc1(EE,I0),I0)
  etst(logandc1(EE,J1),J2)
  etst(logandc1(I0,EE),EE)
  etst(logandc1(J1,EE),I0)

  xtst(logandc2(TT,TT))
  xtst(logandc2(TT,FF))
  xtst(logandc2(TT,I0))
  xtst(logandc2(TT,P0))
  xtst(logandc2(TT,M0))
  xtst(logandc2(TT,B0))
  xtst(logandc2(FF,TT))
  xtst(logandc2(FF,FF))
  xtst(logandc2(FF,I0))
  xtst(logandc2(FF,P0))
  xtst(logandc2(FF,M0))
  xtst(logandc2(FF,B0))
  xtst(logandc2(I0,TT))
  xtst(logandc2(I0,FF))
  xtst(logandc2(I0,P0))
  xtst(logandc2(I0,M0))
  xtst(logandc2(I0,B0))
  xtst(logandc2(P0,TT))
  xtst(logandc2(P0,FF))
  xtst(logandc2(P0,I0))
  xtst(logandc2(P0,P0))
  xtst(logandc2(P0,M0))
  xtst(logandc2(P0,B0))
  xtst(logandc2(M0,TT))
  xtst(logandc2(M0,FF))
  xtst(logandc2(M0,I0))
  xtst(logandc2(M0,P0))
  xtst(logandc2(M0,M0))
  xtst(logandc2(M0,B0))
  xtst(logandc2(B0,TT))
  xtst(logandc2(B0,FF))
  xtst(logandc2(B0,I0))
  xtst(logandc2(B0,P0))
  xtst(logandc2(B0,M0))
  xtst(logandc2(B0,B0))
  etst(logandc2(I0011     ,lognot(I0101     )),I0001     )
  etst(logandc2(I1100     ,lognot(I1010     )),I1000     )
  etst(logandc2(I0011_1100,lognot(I0101_1010)),I0001_1000)
  etst(logandc2(I0011     ,lognot(I0101_1010)),I0001     )
  etst(logandc2(I0101_1010,lognot(I0011     )),I0001     )
  etst(logandc2(I0011     ,lognot(J0101     )),I0001     )
  etst(logandc2(I1100     ,lognot(J1010     )),I1000     )
  etst(logandc2(I0011_1100,lognot(J0101_1010)),I0001_1000)
  etst(logandc2(I0011     ,lognot(J0101_1010)),I0001     )
  etst(logandc2(I0101_1010,lognot(J0011     )),intcons(Z0001,I1010))
  etst(logandc2(J0011     ,lognot(I0101     )),I0001     )
  etst(logandc2(J1100     ,lognot(I1010     )),I1000     )
  etst(logandc2(J0011_1100,lognot(I0101_1010)),I0001_1000)
  etst(logandc2(J0011     ,lognot(I0101_1010)),intcons(Z0001,I1010))
  etst(logandc2(J0101_1010,lognot(I0011     )),I0001     )
  etst(logandc2(J0011     ,lognot(J0101     )),J0001     )
  etst(logandc2(J1100     ,lognot(J1010     )),J1000     )
  etst(logandc2(J0011_1100,lognot(J0101_1010)),J0001_1000)
  etst(logandc2(J0011     ,lognot(J0101_1010)),intcons(M0001,J1010))
  etst(logandc2(J0101_1010,lognot(J0011     )),intcons(M0001,J1010))
  etst(logandc2(EE,I0),EE)
  etst(logandc2(EE,J1),I0)
  etst(logandc2(I0,EE),I0)
  etst(logandc2(J1,EE),J2)

  xtst(lognand(TT,TT))
  xtst(lognand(TT,FF))
  xtst(lognand(TT,I0))
  xtst(lognand(TT,P0))
  xtst(lognand(TT,M0))
  xtst(lognand(TT,B0))
  xtst(lognand(FF,TT))
  xtst(lognand(FF,FF))
  xtst(lognand(FF,I0))
  xtst(lognand(FF,P0))
  xtst(lognand(FF,M0))
  xtst(lognand(FF,B0))
  xtst(lognand(I0,TT))
  xtst(lognand(I0,FF))
  xtst(lognand(I0,P0))
  xtst(lognand(I0,M0))
  xtst(lognand(I0,B0))
  xtst(lognand(P0,TT))
  xtst(lognand(P0,FF))
  xtst(lognand(P0,I0))
  xtst(lognand(P0,P0))
  xtst(lognand(P0,M0))
  xtst(lognand(P0,B0))
  xtst(lognand(M0,TT))
  xtst(lognand(M0,FF))
  xtst(lognand(M0,I0))
  xtst(lognand(M0,P0))
  xtst(lognand(M0,M0))
  xtst(lognand(M0,B0))
  xtst(lognand(B0,TT))
  xtst(lognand(B0,FF))
  xtst(lognand(B0,I0))
  xtst(lognand(B0,P0))
  xtst(lognand(B0,M0))
  xtst(lognand(B0,B0))
  etst(lognand(I0011     ,I0101     ),lognot(I0001     ))
  etst(lognand(I1100     ,I1010     ),lognot(I1000     ))
  etst(lognand(I0011_1100,I0101_1010),lognot(I0001_1000))
  etst(lognand(I0011     ,I0101_1010),lognot(I0001     ))
  etst(lognand(I0101_1010,I0011     ),lognot(I0001     ))
  etst(lognand(I0011     ,J0101     ),lognot(I0001     ))
  etst(lognand(I1100     ,J1010     ),lognot(I1000     ))
  etst(lognand(I0011_1100,J0101_1010),lognot(I0001_1000))
  etst(lognand(I0011     ,J0101_1010),lognot(I0001     ))
  etst(lognand(I0101_1010,J0011     ),lognot(intcons(Z0001,I1010)))
  etst(lognand(J0011     ,I0101     ),lognot(I0001     ))
  etst(lognand(J1100     ,I1010     ),lognot(I1000     ))
  etst(lognand(J0011_1100,I0101_1010),lognot(I0001_1000))
  etst(lognand(J0011     ,I0101_1010),lognot(intcons(Z0001,I1010)))
  etst(lognand(J0101_1010,I0011     ),lognot(I0001     ))
  etst(lognand(J0011     ,J0101     ),lognot(J0001     ))
  etst(lognand(J1100     ,J1010     ),lognot(J1000     ))
  etst(lognand(J0011_1100,J0101_1010),lognot(J0001_1000))
  etst(lognand(J0011     ,J0101_1010),lognot(intcons(M0001,J1010)))
  etst(lognand(J0101_1010,J0011     ),lognot(intcons(M0001,J1010)))
  etst(lognand(EE,I0),J1)
  etst(lognand(EE,J1),J2)
  etst(lognand(I0,EE),J1)
  etst(lognand(J1,EE),J2)

  xtst(logorc1(TT,TT))
  xtst(logorc1(TT,FF))
  xtst(logorc1(TT,I0))
  xtst(logorc1(TT,P0))
  xtst(logorc1(TT,M0))
  xtst(logorc1(TT,B0))
  xtst(logorc1(FF,TT))
  xtst(logorc1(FF,FF))
  xtst(logorc1(FF,I0))
  xtst(logorc1(FF,P0))
  xtst(logorc1(FF,M0))
  xtst(logorc1(FF,B0))
  xtst(logorc1(I0,TT))
  xtst(logorc1(I0,FF))
  xtst(logorc1(I0,P0))
  xtst(logorc1(I0,M0))
  xtst(logorc1(I0,B0))
  xtst(logorc1(P0,TT))
  xtst(logorc1(P0,FF))
  xtst(logorc1(P0,I0))
  xtst(logorc1(P0,P0))
  xtst(logorc1(P0,M0))
  xtst(logorc1(P0,B0))
  xtst(logorc1(M0,TT))
  xtst(logorc1(M0,FF))
  xtst(logorc1(M0,I0))
  xtst(logorc1(M0,P0))
  xtst(logorc1(M0,M0))
  xtst(logorc1(M0,B0))
  xtst(logorc1(B0,TT))
  xtst(logorc1(B0,FF))
  xtst(logorc1(B0,I0))
  xtst(logorc1(B0,P0))
  xtst(logorc1(B0,M0))
  xtst(logorc1(B0,B0))
  etst(logorc1(lognot(I0011     ),I0101     ),I0111     )
  etst(logorc1(lognot(I1100     ),I1010     ),I1110     )
  etst(logorc1(lognot(I0011_1100),I0101_1010),I0111_1110)
  etst(logorc1(lognot(I0011     ),I0101_1010),intcons(Z0111,I1010))
  etst(logorc1(lognot(I0101_1010),I0011     ),intcons(Z0111,I1010))
  etst(logorc1(lognot(I0011     ),J0101     ),J0111     )
  etst(logorc1(lognot(I1100     ),J1010     ),J1110     )
  etst(logorc1(lognot(I0011_1100),J0101_1010),J0111_1110)
  etst(logorc1(lognot(I0011     ),J0101_1010),intcons(M0111,J1010))
  etst(logorc1(lognot(I0101_1010),J0011     ),J0111     )
  etst(logorc1(lognot(J0011     ),I0101     ),J0111     )
  etst(logorc1(lognot(J1100     ),I1010     ),J1110     )
  etst(logorc1(lognot(J0011_1100),I0101_1010),J0111_1110)
  etst(logorc1(lognot(J0011     ),I0101_1010),J0111     )
  etst(logorc1(lognot(J0101_1010),I0011     ),intcons(M0111,J1010))
  etst(logorc1(lognot(J0011     ),J0101     ),J0111     )
  etst(logorc1(lognot(J1100     ),J1010     ),J1110     )
  etst(logorc1(lognot(J0011_1100),J0101_1010),J0111_1110)
  etst(logorc1(lognot(J0011     ),J0101_1010),J0111     )
  etst(logorc1(lognot(J0101_1010),J0011     ),J0111     )
  etst(logorc1(EE,I0),J2)
  etst(logorc1(EE,J1),J1)
  etst(logorc1(I0,EE),J1)
  etst(logorc1(J1,EE),EE)

  xtst(logorc2(TT,TT))
  xtst(logorc2(TT,FF))
  xtst(logorc2(TT,I0))
  xtst(logorc2(TT,P0))
  xtst(logorc2(TT,M0))
  xtst(logorc2(TT,B0))
  xtst(logorc2(FF,TT))
  xtst(logorc2(FF,FF))
  xtst(logorc2(FF,I0))
  xtst(logorc2(FF,P0))
  xtst(logorc2(FF,M0))
  xtst(logorc2(FF,B0))
  xtst(logorc2(I0,TT))
  xtst(logorc2(I0,FF))
  xtst(logorc2(I0,P0))
  xtst(logorc2(I0,M0))
  xtst(logorc2(I0,B0))
  xtst(logorc2(P0,TT))
  xtst(logorc2(P0,FF))
  xtst(logorc2(P0,I0))
  xtst(logorc2(P0,P0))
  xtst(logorc2(P0,M0))
  xtst(logorc2(P0,B0))
  xtst(logorc2(M0,TT))
  xtst(logorc2(M0,FF))
  xtst(logorc2(M0,I0))
  xtst(logorc2(M0,P0))
  xtst(logorc2(M0,M0))
  xtst(logorc2(M0,B0))
  xtst(logorc2(B0,TT))
  xtst(logorc2(B0,FF))
  xtst(logorc2(B0,I0))
  xtst(logorc2(B0,P0))
  xtst(logorc2(B0,M0))
  xtst(logorc2(B0,B0))
  etst(logorc2(I0011     ,lognot(I0101     )),I0111     )
  etst(logorc2(I1100     ,lognot(I1010     )),I1110     )
  etst(logorc2(I0011_1100,lognot(I0101_1010)),I0111_1110)
  etst(logorc2(I0011     ,lognot(I0101_1010)),intcons(Z0111,I1010))
  etst(logorc2(I0101_1010,lognot(I0011     )),intcons(Z0111,I1010))
  etst(logorc2(I0011     ,lognot(J0101     )),J0111     )
  etst(logorc2(I1100     ,lognot(J1010     )),J1110     )
  etst(logorc2(I0011_1100,lognot(J0101_1010)),J0111_1110)
  etst(logorc2(I0011     ,lognot(J0101_1010)),intcons(M0111,J1010))
  etst(logorc2(I0101_1010,lognot(J0011     )),J0111     )
  etst(logorc2(J0011     ,lognot(I0101     )),J0111     )
  etst(logorc2(J1100     ,lognot(I1010     )),J1110     )
  etst(logorc2(J0011_1100,lognot(I0101_1010)),J0111_1110)
  etst(logorc2(J0011     ,lognot(I0101_1010)),J0111     )
  etst(logorc2(J0101_1010,lognot(I0011     )),intcons(M0111,J1010))
  etst(logorc2(J0011     ,lognot(J0101     )),J0111     )
  etst(logorc2(J1100     ,lognot(J1010     )),J1110     )
  etst(logorc2(J0011_1100,lognot(J0101_1010)),J0111_1110)
  etst(logorc2(J0011     ,lognot(J0101_1010)),J0111     )
  etst(logorc2(J0101_1010,lognot(J0011     )),J0111     )
  etst(logorc2(EE,I0),J1)
  etst(logorc2(EE,J1),EE)
  etst(logorc2(I0,EE),J2)
  etst(logorc2(J1,EE),J1)

  xtst(lognor(TT,TT))
  xtst(lognor(TT,FF))
  xtst(lognor(TT,I0))
  xtst(lognor(TT,P0))
  xtst(lognor(TT,M0))
  xtst(lognor(TT,B0))
  xtst(lognor(FF,TT))
  xtst(lognor(FF,FF))
  xtst(lognor(FF,I0))
  xtst(lognor(FF,P0))
  xtst(lognor(FF,M0))
  xtst(lognor(FF,B0))
  xtst(lognor(I0,TT))
  xtst(lognor(I0,FF))
  xtst(lognor(I0,P0))
  xtst(lognor(I0,M0))
  xtst(lognor(I0,B0))
  xtst(lognor(P0,TT))
  xtst(lognor(P0,FF))
  xtst(lognor(P0,I0))
  xtst(lognor(P0,P0))
  xtst(lognor(P0,M0))
  xtst(lognor(P0,B0))
  xtst(lognor(M0,TT))
  xtst(lognor(M0,FF))
  xtst(lognor(M0,I0))
  xtst(lognor(M0,P0))
  xtst(lognor(M0,M0))
  xtst(lognor(M0,B0))
  xtst(lognor(B0,TT))
  xtst(lognor(B0,FF))
  xtst(lognor(B0,I0))
  xtst(lognor(B0,P0))
  xtst(lognor(B0,M0))
  xtst(lognor(B0,B0))
  etst(lognor(I0011     ,I0101     ),lognot(I0111     ))
  etst(lognor(I1100     ,I1010     ),lognot(I1110     ))
  etst(lognor(I0011_1100,I0101_1010),lognot(I0111_1110))
  etst(lognor(I0011     ,I0101_1010),lognot(intcons(Z0111,I1010)))
  etst(lognor(I0101_1010,I0011     ),lognot(intcons(Z0111,I1010)))
  etst(lognor(I0011     ,J0101     ),lognot(J0111     ))
  etst(lognor(I1100     ,J1010     ),lognot(J1110     ))
  etst(lognor(I0011_1100,J0101_1010),lognot(J0111_1110))
  etst(lognor(I0011     ,J0101_1010),lognot(intcons(M0111,J1010)))
  etst(lognor(I0101_1010,J0011     ),lognot(J0111     ))
  etst(lognor(J0011     ,I0101     ),lognot(J0111     ))
  etst(lognor(J1100     ,I1010     ),lognot(J1110     ))
  etst(lognor(J0011_1100,I0101_1010),lognot(J0111_1110))
  etst(lognor(J0011     ,I0101_1010),lognot(J0111     ))
  etst(lognor(J0101_1010,I0011     ),lognot(intcons(M0111,J1010)))
  etst(lognor(J0011     ,J0101     ),lognot(J0111     ))
  etst(lognor(J1100     ,J1010     ),lognot(J1110     ))
  etst(lognor(J0011_1100,J0101_1010),lognot(J0111_1110))
  etst(lognor(J0011     ,J0101_1010),lognot(J0111     ))
  etst(lognor(J0101_1010,J0011     ),lognot(J0111     ))
  etst(lognor(EE,I0),J2)
  etst(lognor(EE,J1),I0)
  etst(lognor(I0,EE),J2)
  etst(lognor(J1,EE),I0)

  xtst(logeqv(TT,TT))
  xtst(logeqv(TT,FF))
  xtst(logeqv(TT,I0))
  xtst(logeqv(TT,P0))
  xtst(logeqv(TT,M0))
  xtst(logeqv(TT,B0))
  xtst(logeqv(FF,TT))
  xtst(logeqv(FF,FF))
  xtst(logeqv(FF,I0))
  xtst(logeqv(FF,P0))
  xtst(logeqv(FF,M0))
  xtst(logeqv(FF,B0))
  xtst(logeqv(I0,TT))
  xtst(logeqv(I0,FF))
  xtst(logeqv(I0,P0))
  xtst(logeqv(I0,M0))
  xtst(logeqv(I0,B0))
  xtst(logeqv(P0,TT))
  xtst(logeqv(P0,FF))
  xtst(logeqv(P0,I0))
  xtst(logeqv(P0,P0))
  xtst(logeqv(P0,M0))
  xtst(logeqv(P0,B0))
  xtst(logeqv(M0,TT))
  xtst(logeqv(M0,FF))
  xtst(logeqv(M0,I0))
  xtst(logeqv(M0,P0))
  xtst(logeqv(M0,M0))
  xtst(logeqv(M0,B0))
  xtst(logeqv(B0,TT))
  xtst(logeqv(B0,FF))
  xtst(logeqv(B0,I0))
  xtst(logeqv(B0,P0))
  xtst(logeqv(B0,M0))
  xtst(logeqv(B0,B0))
  etst(logeqv(lognot(I0011     ),I0101     ),I0110     )
  etst(logeqv(lognot(I1100     ),I1010     ),I0110     )
  etst(logeqv(lognot(I0011_1100),I0101_1010),I0110_0110)
  etst(logeqv(lognot(I0011     ),I0101_1010),intcons(Z0110,I1010))
  etst(logeqv(lognot(I0101_1010),I0011     ),intcons(Z0110,I1010))
  etst(logeqv(lognot(I0011     ),J0101     ),J0110     )
  etst(logeqv(lognot(I1100     ),J1010     ),J0110     )
  etst(logeqv(lognot(I0011_1100),J0101_1010),J0110_0110)
  etst(logeqv(lognot(I0011     ),J0101_1010),intcons(M0110,J1010))
  etst(logeqv(lognot(I0101_1010),J0011     ),intcons(M0110,J0101))
  etst(logeqv(lognot(J0011     ),I0101     ),J0110     )
  etst(logeqv(lognot(J1100     ),I1010     ),J0110     )
  etst(logeqv(lognot(J0011_1100),I0101_1010),J0110_0110)
  etst(logeqv(lognot(J0011     ),I0101_1010),intcons(M0110,J0101))
  etst(logeqv(lognot(J0101_1010),I0011     ),intcons(M0110,J1010))
  etst(logeqv(lognot(J0011     ),J0101     ),I0110     )
  etst(logeqv(lognot(J1100     ),J1010     ),I0110     )
  etst(logeqv(lognot(J0011_1100),J0101_1010),I0110_0110)
  etst(logeqv(lognot(J0011     ),J0101_1010),intcons(Z0110,I0101))
  etst(logeqv(lognot(J0101_1010),J0011     ),intcons(Z0110,I0101))
  etst(logeqv(EE,I0),J2)
  etst(logeqv(EE,J1),EE)
  etst(logeqv(I0,EE),J2)
  etst(logeqv(J1,EE),EE)

  xtst(logtest(TT,TT))
  xtst(logtest(TT,FF))
  xtst(logtest(TT,I0))
  xtst(logtest(TT,P0))
  xtst(logtest(TT,M0))
  xtst(logtest(TT,B0))
  xtst(logtest(FF,TT))
  xtst(logtest(FF,FF))
  xtst(logtest(FF,I0))
  xtst(logtest(FF,P0))
  xtst(logtest(FF,M0))
  xtst(logtest(FF,B0))
  xtst(logtest(I0,TT))
  xtst(logtest(I0,FF))
  xtst(logtest(I0,P0))
  xtst(logtest(I0,M0))
  xtst(logtest(I0,B0))
  xtst(logtest(P0,TT))
  xtst(logtest(P0,FF))
  xtst(logtest(P0,I0))
  xtst(logtest(P0,P0))
  xtst(logtest(P0,M0))
  xtst(logtest(P0,B0))
  xtst(logtest(M0,TT))
  xtst(logtest(M0,FF))
  xtst(logtest(M0,I0))
  xtst(logtest(M0,P0))
  xtst(logtest(M0,M0))
  xtst(logtest(M0,B0))
  xtst(logtest(B0,TT))
  xtst(logtest(B0,FF))
  xtst(logtest(B0,I0))
  xtst(logtest(B0,P0))
  xtst(logtest(B0,M0))
  xtst(logtest(B0,B0))
  ttst(logtest(I0011     ,I0101     ))
  ttst(logtest(I1100     ,I1010     ))
  ttst(logtest(I0011_1100,I0101_1010))
  ttst(logtest(I0011     ,I0101_1010))
  ttst(logtest(I0101_1010,I0011     ))
  ttst(logtest(I0011     ,J0101     ))
  ttst(logtest(I1100     ,J1010     ))
  ttst(logtest(I0011_1100,J0101_1010))
  ttst(logtest(I0011     ,J0101_1010))
  ttst(logtest(I0101_1010,J0011     ))
  ttst(logtest(J0011     ,I0101     ))
  ttst(logtest(J1100     ,I1010     ))
  ttst(logtest(J0011_1100,I0101_1010))
  ttst(logtest(J0011     ,I0101_1010))
  ttst(logtest(J0101_1010,I0011     ))
  ttst(logtest(J0011     ,J0101     ))
  ttst(logtest(J1100     ,J1010     ))
  ttst(logtest(J0011_1100,J0101_1010))
  ttst(logtest(J0011     ,J0101_1010))
  ttst(logtest(J0101_1010,J0011     ))

  ftst(logtest(I0011     ,I1100     ))
  ftst(logtest(I1100     ,I0001     ))
  ftst(logtest(I0011_1100,I1000_0001))
  ftst(logtest(I0011     ,I1100_0011))
  ftst(logtest(I0101_1010,I0010     ))
  ftst(logtest(I0011     ,J1000     ))
  ftst(logtest(I1100     ,J0011     ))
  ftst(logtest(I0011_1100,J1000_0001))
  ftst(logtest(I0011     ,J1100_0011))
  ftst(logtest(J0011     ,I0100     ))
  ftst(logtest(J1100     ,I0010     ))
  ftst(logtest(J0011_1100,I0100_0010))
  ftst(logtest(J0101_1010,I1010     ))
  ftst(logtest(EE,I0))
  ttst(logtest(EE,J1))
  ftst(logtest(I0,EE))
  ttst(logtest(J1,EE))
  ftst(logtest(EE,J1010_0101))
  ttst(logtest(EE,I0011_1100))
  ftst(logtest(J1010_0101,EE))
  ttst(logtest(I0011_1100,EE))

  xtst(lt(TT,TT))
  xtst(lt(TT,FF))
  xtst(lt(TT,I0))
  xtst(lt(TT,P0))
  xtst(lt(TT,M0))
  xtst(lt(TT,B0))
  xtst(lt(FF,TT))
  xtst(lt(FF,FF))
  xtst(lt(FF,I0))
  xtst(lt(FF,P0))
  xtst(lt(FF,M0))
  xtst(lt(FF,B0))
  xtst(lt(I0,TT))
  xtst(lt(I0,FF))
  xtst(lt(I0,P0))
  xtst(lt(I0,M0))
  xtst(lt(I0,B0))
  xtst(lt(P0,TT))
  xtst(lt(P0,FF))
  xtst(lt(P0,I0))
  xtst(lt(P0,P0))
  xtst(lt(P0,M0))
  xtst(lt(P0,B0))
  xtst(lt(M0,TT))
  xtst(lt(M0,FF))
  xtst(lt(M0,I0))
  xtst(lt(M0,P0))
  xtst(lt(M0,M0))
  xtst(lt(M0,B0))
  xtst(lt(B0,TT))
  xtst(lt(B0,FF))
  xtst(lt(B0,I0))
  xtst(lt(B0,P0))
  xtst(lt(B0,M0))
  xtst(lt(B0,B0))
  ftst(lt(I0011     ,I0011     ))
  ttst(lt(I0011     ,I0101     ))
  ftst(lt(I1100     ,I1010     ))
  ftst(lt(I0011_1100,I0011_1100))
  ftst(lt(I0011_1100,I0101_1010))
  ttst(lt(I1100_0011,I1010_0101))
  ttst(lt(I0011     ,I0101_1010))
  ftst(lt(I0101_1010,I0011     ))
  ftst(lt(I0011     ,J0101     ))
  ftst(lt(I1100     ,J1010     ))
  ftst(lt(I0011_1100,J0101_1010))
  ftst(lt(I0011     ,J0101_1010))
  ftst(lt(I0101_1010,J0011     ))
  ttst(lt(J0011     ,I0101     ))
  ttst(lt(J1100     ,I1010     ))
  ttst(lt(J0011_1100,I0101_1010))
  ttst(lt(J0011     ,I0101_1010))
  ttst(lt(J0101_1010,I0011     ))
  ftst(lt(J0011     ,J0011     ))
  ttst(lt(J0011     ,J0101     ))
  ftst(lt(J1100     ,J1010     ))
  ftst(lt(J0011_1100,J0011_1100))
  ftst(lt(J0011_1100,J0101_1010))
  ttst(lt(J1100_0011,J1010_0101))
  ftst(lt(J0011     ,J0101_1010))
  ttst(lt(J0101_1010,J0011     ))

  xtst(le(TT,TT))
  xtst(le(TT,FF))
  xtst(le(TT,I0))
  xtst(le(TT,P0))
  xtst(le(TT,M0))
  xtst(le(TT,B0))
  xtst(le(FF,TT))
  xtst(le(FF,FF))
  xtst(le(FF,I0))
  xtst(le(FF,P0))
  xtst(le(FF,M0))
  xtst(le(FF,B0))
  xtst(le(I0,TT))
  xtst(le(I0,FF))
  xtst(le(I0,P0))
  xtst(le(I0,M0))
  xtst(le(I0,B0))
  xtst(le(P0,TT))
  xtst(le(P0,FF))
  xtst(le(P0,I0))
  xtst(le(P0,P0))
  xtst(le(P0,M0))
  xtst(le(P0,B0))
  xtst(le(M0,TT))
  xtst(le(M0,FF))
  xtst(le(M0,I0))
  xtst(le(M0,P0))
  xtst(le(M0,M0))
  xtst(le(M0,B0))
  xtst(le(B0,TT))
  xtst(le(B0,FF))
  xtst(le(B0,I0))
  xtst(le(B0,P0))
  xtst(le(B0,M0))
  xtst(le(B0,B0))
  ttst(le(I0011     ,I0011     ))
  ttst(le(I0011     ,I0101     ))
  ftst(le(I1100     ,I1010     ))
  ttst(le(I0011_1100,I0011_1100))
  ftst(le(I0011_1100,I0101_1010))
  ttst(le(I1100_0011,I1010_0101))
  ttst(le(I0011     ,I0101_1010))
  ftst(le(I0101_1010,I0011     ))
  ftst(le(I0011     ,J0101     ))
  ftst(le(I1100     ,J1010     ))
  ftst(le(I0011_1100,J0101_1010))
  ftst(le(I0011     ,J0101_1010))
  ftst(le(I0101_1010,J0011     ))
  ttst(le(J0011     ,I0101     ))
  ttst(le(J1100     ,I1010     ))
  ttst(le(J0011_1100,I0101_1010))
  ttst(le(J0011     ,I0101_1010))
  ttst(le(J0101_1010,I0011     ))
  ttst(le(J0011     ,J0011     ))
  ttst(le(J0011     ,J0101     ))
  ftst(le(J1100     ,J1010     ))
  ttst(le(J0011_1100,J0011_1100))
  ftst(le(J0011_1100,J0101_1010))
  ttst(le(J1100_0011,J1010_0101))
  ftst(le(J0011     ,J0101_1010))
  ttst(le(J0101_1010,J0011     ))

  xtst(gt(TT,TT))
  xtst(gt(TT,FF))
  xtst(gt(TT,I0))
  xtst(gt(TT,P0))
  xtst(gt(TT,M0))
  xtst(gt(TT,B0))
  xtst(gt(FF,TT))
  xtst(gt(FF,FF))
  xtst(gt(FF,I0))
  xtst(gt(FF,P0))
  xtst(gt(FF,M0))
  xtst(gt(FF,B0))
  xtst(gt(I0,TT))
  xtst(gt(I0,FF))
  xtst(gt(I0,P0))
  xtst(gt(I0,M0))
  xtst(gt(I0,B0))
  xtst(gt(P0,TT))
  xtst(gt(P0,FF))
  xtst(gt(P0,I0))
  xtst(gt(P0,P0))
  xtst(gt(P0,M0))
  xtst(gt(P0,B0))
  xtst(gt(M0,TT))
  xtst(gt(M0,FF))
  xtst(gt(M0,I0))
  xtst(gt(M0,P0))
  xtst(gt(M0,M0))
  xtst(gt(M0,B0))
  xtst(gt(B0,TT))
  xtst(gt(B0,FF))
  xtst(gt(B0,I0))
  xtst(gt(B0,P0))
  xtst(gt(B0,M0))
  xtst(gt(B0,B0))
  ftst(gt(I0011     ,I0011     ))
  ftst(gt(I0011     ,I0101     ))
  ttst(gt(I1100     ,I1010     ))
  ftst(gt(I0011_1100,I0011_1100))
  ttst(gt(I0011_1100,I0101_1010))
  ftst(gt(I1100_0011,I1010_0101))
  ftst(gt(I0011     ,I0101_1010))
  ttst(gt(I0101_1010,I0011     ))
  ttst(gt(I0011     ,J0101     ))
  ttst(gt(I1100     ,J1010     ))
  ttst(gt(I0011_1100,J0101_1010))
  ttst(gt(I0011     ,J0101_1010))
  ttst(gt(I0101_1010,J0011     ))
  ftst(gt(J0011     ,I0101     ))
  ftst(gt(J1100     ,I1010     ))
  ftst(gt(J0011_1100,I0101_1010))
  ftst(gt(J0011     ,I0101_1010))
  ftst(gt(J0101_1010,I0011     ))
  ftst(gt(J0011     ,J0011     ))
  ftst(gt(J0011     ,J0101     ))
  ttst(gt(J1100     ,J1010     ))
  ftst(gt(J0011_1100,J0011_1100))
  ttst(gt(J0011_1100,J0101_1010))
  ftst(gt(J1100_0011,J1010_0101))
  ttst(gt(J0011     ,J0101_1010))
  ftst(gt(J0101_1010,J0011     ))

  xtst(ge(TT,TT))
  xtst(ge(TT,FF))
  xtst(ge(TT,I0))
  xtst(ge(TT,P0))
  xtst(ge(TT,M0))
  xtst(ge(TT,B0))
  xtst(ge(FF,TT))
  xtst(ge(FF,FF))
  xtst(ge(FF,I0))
  xtst(ge(FF,P0))
  xtst(ge(FF,M0))
  xtst(ge(FF,B0))
  xtst(ge(I0,TT))
  xtst(ge(I0,FF))
  xtst(ge(I0,P0))
  xtst(ge(I0,M0))
  xtst(ge(I0,B0))
  xtst(ge(P0,TT))
  xtst(ge(P0,FF))
  xtst(ge(P0,I0))
  xtst(ge(P0,P0))
  xtst(ge(P0,M0))
  xtst(ge(P0,B0))
  xtst(ge(M0,TT))
  xtst(ge(M0,FF))
  xtst(ge(M0,I0))
  xtst(ge(M0,P0))
  xtst(ge(M0,M0))
  xtst(ge(M0,B0))
  xtst(ge(B0,TT))
  xtst(ge(B0,FF))
  xtst(ge(B0,I0))
  xtst(ge(B0,P0))
  xtst(ge(B0,M0))
  xtst(ge(B0,B0))
  ttst(ge(I0011     ,I0011     ))
  ftst(ge(I0011     ,I0101     ))
  ttst(ge(I1100     ,I1010     ))
  ttst(ge(I0011_1100,I0011_1100))
  ttst(ge(I0011_1100,I0101_1010))
  ftst(ge(I1100_0011,I1010_0101))
  ftst(ge(I0011     ,I0101_1010))
  ttst(ge(I0101_1010,I0011     ))
  ttst(ge(I0011     ,J0101     ))
  ttst(ge(I1100     ,J1010     ))
  ttst(ge(I0011_1100,J0101_1010))
  ttst(ge(I0011     ,J0101_1010))
  ttst(ge(I0101_1010,J0011     ))
  ftst(ge(J0011     ,I0101     ))
  ftst(ge(J1100     ,I1010     ))
  ftst(ge(J0011_1100,I0101_1010))
  ftst(ge(J0011     ,I0101_1010))
  ftst(ge(J0101_1010,I0011     ))
  ttst(ge(J0011     ,J0011     ))
  ftst(ge(J0011     ,J0101     ))
  ttst(ge(J1100     ,J1010     ))
  ttst(ge(J0011_1100,J0011_1100))
  ttst(ge(J0011_1100,J0101_1010))
  ftst(ge(J1100_0011,J1010_0101))
  ttst(ge(J0011     ,J0101_1010))
  ftst(ge(J0101_1010,J0011     ))

  xtst(evenp(TT))
  xtst(evenp(FF))
  xtst(evenp(P0))
  xtst(evenp(M0))
  xtst(evenp(B0))
  ttst(evenp(Z))
  ftst(evenp(M))
  ftst(evenp(I0011))
  ttst(evenp(J1010_0101))

  xtst(half(TT))
  xtst(half(FF))
  xtst(half(P0))
  xtst(half(M0))
  xtst(half(B0))
  etst(half(Z),Z)
  etst(half(M),M)
  etst(half(I[1]),I[0])
  etst(half(I[2]),I[1])
  etst(half(I[3]),I[1])
  etst(half(I[4]),I[2])
  etst(half(I[5]),I[2])
  etst(half(I[6]),I[3])
  etst(half(JN(-2)),M)
  etst(half(JN(-3)),JN(-2))
  etst(half(JN(-4)),JN(-2))
  etst(half(JN(-5)),JN(-3))
  etst(half(JN(-6)),JN(-3))
  etst(half(JN(-7)),JN(-4))

  xtst(small(TT))
  xtst(small(FF))
  xtst(small(P0))
  xtst(small(M0))
  xtst(small(B0))
  ttst(small(Z))
  ttst(small(M))
  ftst(small(I0011))
  ftst(small(J1010_0101))

  etst(base(),I[10])

  etst(zero(),I[0])

  xtst(plus(TT,TT))
  xtst(plus(TT,FF))
  xtst(plus(TT,I0))
  xtst(plus(TT,P0))
  xtst(plus(TT,M0))
  xtst(plus(TT,B0))
  xtst(plus(FF,TT))
  xtst(plus(FF,FF))
  xtst(plus(FF,I0))
  xtst(plus(FF,P0))
  xtst(plus(FF,M0))
  xtst(plus(FF,B0))
  xtst(plus(I0,TT))
  xtst(plus(I0,FF))
  xtst(plus(I0,P0))
  xtst(plus(I0,M0))
  xtst(plus(I0,B0))
  xtst(plus(P0,TT))
  xtst(plus(P0,FF))
  xtst(plus(P0,I0))
  xtst(plus(P0,P0))
  xtst(plus(P0,M0))
  xtst(plus(P0,B0))
  xtst(plus(M0,TT))
  xtst(plus(M0,FF))
  xtst(plus(M0,I0))
  xtst(plus(M0,P0))
  xtst(plus(M0,M0))
  xtst(plus(M0,B0))
  xtst(plus(B0,TT))
  xtst(plus(B0,FF))
  xtst(plus(B0,I0))
  xtst(plus(B0,P0))
  xtst(plus(B0,M0))
  xtst(plus(B0,B0))
  etst(plus(JN(-3),JN(-3)),JN(-6))
  etst(plus(JN(-3),JN(-2)),JN(-5))
  etst(plus(JN(-3),JN(-1)),JN(-4))
  etst(plus(JN(-3),JN( 0)),JN(-3))
  etst(plus(JN(-3),JN( 1)),JN(-2))
  etst(plus(JN(-3),JN( 2)),JN(-1))
  etst(plus(JN(-2),JN(-3)),JN(-5))
  etst(plus(JN(-2),JN(-2)),JN(-4))
  etst(plus(JN(-2),JN(-1)),JN(-3))
  etst(plus(JN(-2),JN( 0)),JN(-2))
  etst(plus(JN(-2),JN( 1)),JN(-1))
  etst(plus(JN(-2),JN( 2)),JN( 0))
  etst(plus(JN(-1),JN(-3)),JN(-4))
  etst(plus(JN(-1),JN(-2)),JN(-3))
  etst(plus(JN(-1),JN(-1)),JN(-2))
  etst(plus(JN(-1),JN( 0)),JN(-1))
  etst(plus(JN(-1),JN( 1)),JN( 0))
  etst(plus(JN(-1),JN( 2)),JN( 1))
  etst(plus(JN( 0),JN(-3)),JN(-3))
  etst(plus(JN( 0),JN(-2)),JN(-2))
  etst(plus(JN( 0),JN(-1)),JN(-1))
  etst(plus(JN( 0),JN( 0)),JN( 0))
  etst(plus(JN( 0),JN( 1)),JN( 1))
  etst(plus(JN( 0),JN( 2)),JN( 2))
  etst(plus(JN( 1),JN(-3)),JN(-2))
  etst(plus(JN( 1),JN(-2)),JN(-1))
  etst(plus(JN( 1),JN(-1)),JN( 0))
  etst(plus(JN( 1),JN( 0)),JN( 1))
  etst(plus(JN( 1),JN( 1)),JN( 2))
  etst(plus(JN( 1),JN( 2)),JN( 3))
  etst(plus(JN( 2),JN(-3)),JN(-1))
  etst(plus(JN( 2),JN(-2)),JN( 0))
  etst(plus(JN( 2),JN(-1)),JN( 1))
  etst(plus(JN( 2),JN( 0)),JN( 2))
  etst(plus(JN( 2),JN( 1)),JN( 3))
  etst(plus(JN( 2),JN( 2)),JN( 4))
  etst(plus(K2( 0, 0),K2( 0, 0)),K2( 0, 0))
  etst(plus(K2( 0, 0),K2( 0,-1)),K2( 0,-1))
  etst(plus(K2( 0, 0),K2(-1, 0)),K2(-1, 0))
  etst(plus(K2( 0, 0),K2(-1,-1)),K2(-1,-1))
  etst(plus(K2( 0,-1),K2( 0, 0)),K2( 0,-1))
  etst(plus(K2( 0,-1),K2( 0,-1)),K2( 0,-2))
  etst(plus(K2( 0,-1),K2(-1, 0)),K2(-1,-1))
  etst(plus(K2( 0,-1),K2(-1,-1)),K2(-1,-2))
  etst(plus(K2(-1, 0),K2( 0, 0)),K2(-1, 0))
  etst(plus(K2(-1, 0),K2( 0,-1)),K2(-1,-1))
  etst(plus(K2(-1, 0),K2(-1, 0)),K2(-2, 1))
  etst(plus(K2(-1, 0),K2(-1,-1)),K2(-2, 0))
  etst(plus(K2(-1,-1),K2( 0, 0)),K2(-1,-1))
  etst(plus(K2(-1,-1),K2( 0,-1)),K2(-1,-2))
  etst(plus(K2(-1,-1),K2(-1, 0)),K2(-2, 0))
  etst(plus(K2(-1,-1),K2(-1,-1)),K2(-2,-1))

  etst(plus(N(-1000000000000),N(-1000000000000)),N(-2000000000000))
  etst(plus(N(-1000000000000),N(   -1000000000)),N(-1001000000000))
  etst(plus(N(-1000000000000),N(      -1000000)),N(-1000001000000))
  etst(plus(N(-1000000000000),N(         -1000)),N(-1000000001000))
  etst(plus(N(-1000000000000),N(          -100)),N(-1000000000100))
  etst(plus(N(-1000000000000),N(           -10)),N(-1000000000010))
  etst(plus(N(-1000000000000),N(            -1)),N(-1000000000001))
  etst(plus(N(-1000000000000),N(             0)),N(-1000000000000))
  etst(plus(N(-1000000000000),N(             1)),N(-0999999999999))
  etst(plus(N(-1000000000000),N(            10)),N(-0999999999990))
  etst(plus(N(-1000000000000),N(           100)),N(-0999999999900))
  etst(plus(N(-1000000000000),N(          1000)),N(-0999999999000))
  etst(plus(N(-1000000000000),N(       1000000)),N(-0999999000000))
  etst(plus(N(-1000000000000),N(    1000000000)),N(-0999000000000))
  etst(plus(N(-1000000000000),N( 1000000000000)),N(-0000000000000))

  etst(plus(N(   -1000000000),N(-1000000000000)),N(-1001000000000))
  etst(plus(N(   -1000000000),N(   -1000000000)),N(-0002000000000))
  etst(plus(N(   -1000000000),N(      -1000000)),N(-0001001000000))
  etst(plus(N(   -1000000000),N(         -1000)),N(-0001000001000))
  etst(plus(N(   -1000000000),N(          -100)),N(-0001000000100))
  etst(plus(N(   -1000000000),N(           -10)),N(-0001000000010))
  etst(plus(N(   -1000000000),N(            -1)),N(-0001000000001))
  etst(plus(N(   -1000000000),N(             0)),N(-0001000000000))
  etst(plus(N(   -1000000000),N(             1)),N(-0000999999999))
  etst(plus(N(   -1000000000),N(            10)),N(-0000999999990))
  etst(plus(N(   -1000000000),N(           100)),N(-0000999999900))
  etst(plus(N(   -1000000000),N(          1000)),N(-0000999999000))
  etst(plus(N(   -1000000000),N(       1000000)),N(-0000999000000))
  etst(plus(N(   -1000000000),N(    1000000000)),N(-0000000000000))
  etst(plus(N(   -1000000000),N( 1000000000000)),N( 0999000000000))

  etst(plus(N(      -1000000),N(-1000000000000)),N(-1000001000000))
  etst(plus(N(      -1000000),N(   -1000000000)),N(-0001001000000))
  etst(plus(N(      -1000000),N(      -1000000)),N(-0000002000000))
  etst(plus(N(      -1000000),N(         -1000)),N(-0000001001000))
  etst(plus(N(      -1000000),N(          -100)),N(-0000001000100))
  etst(plus(N(      -1000000),N(           -10)),N(-0000001000010))
  etst(plus(N(      -1000000),N(            -1)),N(-0000001000001))
  etst(plus(N(      -1000000),N(             0)),N(-0000001000000))
  etst(plus(N(      -1000000),N(             1)),N(-0000000999999))
  etst(plus(N(      -1000000),N(            10)),N(-0000000999990))
  etst(plus(N(      -1000000),N(           100)),N(-0000000999900))
  etst(plus(N(      -1000000),N(          1000)),N(-0000000999000))
  etst(plus(N(      -1000000),N(       1000000)),N(-0000000000000))
  etst(plus(N(      -1000000),N(    1000000000)),N( 0000999000000))
  etst(plus(N(      -1000000),N( 1000000000000)),N( 0999999000000))

  etst(plus(N(         -1000),N(-1000000000000)),N(-1000000001000))
  etst(plus(N(         -1000),N(   -1000000000)),N(-0001000001000))
  etst(plus(N(         -1000),N(      -1000000)),N(-0000001001000))
  etst(plus(N(         -1000),N(         -1000)),N(-0000000002000))
  etst(plus(N(         -1000),N(          -100)),N(-0000000001100))
  etst(plus(N(         -1000),N(           -10)),N(-0000000001010))
  etst(plus(N(         -1000),N(            -1)),N(-0000000001001))
  etst(plus(N(         -1000),N(             0)),N(-0000000001000))
  etst(plus(N(         -1000),N(             1)),N(-0000000000999))
  etst(plus(N(         -1000),N(            10)),N(-0000000000990))
  etst(plus(N(         -1000),N(           100)),N(-0000000000900))
  etst(plus(N(         -1000),N(          1000)),N( 0000000000000))
  etst(plus(N(         -1000),N(       1000000)),N( 0000000999000))
  etst(plus(N(         -1000),N(    1000000000)),N( 0000999999000))
  etst(plus(N(         -1000),N( 1000000000000)),N( 0999999999000))

  etst(plus(N(          -100),N(-1000000000000)),N(-1000000000100))
  etst(plus(N(          -100),N(   -1000000000)),N(-0001000000100))
  etst(plus(N(          -100),N(      -1000000)),N(-0000001000100))
  etst(plus(N(          -100),N(         -1000)),N(-0000000001100))
  etst(plus(N(          -100),N(          -100)),N(-0000000000200))
  etst(plus(N(          -100),N(           -10)),N(-0000000000110))
  etst(plus(N(          -100),N(            -1)),N(-0000000000101))
  etst(plus(N(          -100),N(             0)),N(-0000000000100))
  etst(plus(N(          -100),N(             1)),N(-0000000000099))
  etst(plus(N(          -100),N(            10)),N(-0000000000090))
  etst(plus(N(          -100),N(           100)),N( 0000000000000))
  etst(plus(N(          -100),N(          1000)),N( 0000000000900))
  etst(plus(N(          -100),N(       1000000)),N( 0000000999900))
  etst(plus(N(          -100),N(    1000000000)),N( 0000999999900))
  etst(plus(N(          -100),N( 1000000000000)),N( 0999999999900))

  etst(plus(N(           -10),N(-1000000000000)),N(-1000000000010))
  etst(plus(N(           -10),N(   -1000000000)),N(-0001000000010))
  etst(plus(N(           -10),N(      -1000000)),N(-0000001000010))
  etst(plus(N(           -10),N(         -1000)),N(-0000000001010))
  etst(plus(N(           -10),N(          -100)),N(-0000000000110))
  etst(plus(N(           -10),N(           -10)),N(-0000000000020))
  etst(plus(N(           -10),N(            -1)),N(-0000000000011))
  etst(plus(N(           -10),N(             0)),N(-0000000000010))
  etst(plus(N(           -10),N(             1)),N(-0000000000009))
  etst(plus(N(           -10),N(            10)),N( 0000000000000))
  etst(plus(N(           -10),N(           100)),N( 0000000000090))
  etst(plus(N(           -10),N(          1000)),N( 0000000000990))
  etst(plus(N(           -10),N(       1000000)),N( 0000000999990))
  etst(plus(N(           -10),N(    1000000000)),N( 0000999999990))
  etst(plus(N(           -10),N( 1000000000000)),N( 0999999999990))

  etst(plus(N(            -1),N(-1000000000000)),N(-1000000000001))
  etst(plus(N(            -1),N(   -1000000000)),N(-0001000000001))
  etst(plus(N(            -1),N(      -1000000)),N(-0000001000001))
  etst(plus(N(            -1),N(         -1000)),N(-0000000001001))
  etst(plus(N(            -1),N(          -100)),N(-0000000000101))
  etst(plus(N(            -1),N(           -10)),N(-0000000000011))
  etst(plus(N(            -1),N(            -1)),N(-0000000000002))
  etst(plus(N(            -1),N(             0)),N(-0000000000001))
  etst(plus(N(            -1),N(             1)),N( 0000000000000))
  etst(plus(N(            -1),N(            10)),N( 0000000000009))
  etst(plus(N(            -1),N(           100)),N( 0000000000099))
  etst(plus(N(            -1),N(          1000)),N( 0000000000999))
  etst(plus(N(            -1),N(       1000000)),N( 0000000999999))
  etst(plus(N(            -1),N(    1000000000)),N( 0000999999999))
  etst(plus(N(            -1),N( 1000000000000)),N( 0999999999999))

  etst(plus(N(             0),N(-1000000000000)),N(-1000000000000))
  etst(plus(N(             0),N(   -1000000000)),N(-0001000000000))
  etst(plus(N(             0),N(      -1000000)),N(-0000001000000))
  etst(plus(N(             0),N(         -1000)),N(-0000000001000))
  etst(plus(N(             0),N(          -100)),N(-0000000000100))
  etst(plus(N(             0),N(           -10)),N(-0000000000010))
  etst(plus(N(             0),N(            -1)),N(-0000000000001))
  etst(plus(N(             0),N(             0)),N( 0000000000000))
  etst(plus(N(             0),N(             1)),N( 0000000000001))
  etst(plus(N(             0),N(            10)),N( 0000000000010))
  etst(plus(N(             0),N(           100)),N( 0000000000100))
  etst(plus(N(             0),N(          1000)),N( 0000000001000))
  etst(plus(N(             0),N(       1000000)),N( 0000001000000))
  etst(plus(N(             0),N(    1000000000)),N( 0001000000000))
  etst(plus(N(             0),N( 1000000000000)),N( 1000000000000))

  etst(plus(N( 1000000000000),N( 1000000000000)),N( 2000000000000))
  etst(plus(N( 1000000000000),N(    1000000000)),N( 1001000000000))
  etst(plus(N( 1000000000000),N(       1000000)),N( 1000001000000))
  etst(plus(N( 1000000000000),N(          1000)),N( 1000000001000))
  etst(plus(N( 1000000000000),N(           100)),N( 1000000000100))
  etst(plus(N( 1000000000000),N(            10)),N( 1000000000010))
  etst(plus(N( 1000000000000),N(             1)),N( 1000000000001))
  etst(plus(N( 1000000000000),N(             0)),N( 1000000000000))
  etst(plus(N( 1000000000000),N(            -1)),N( 0999999999999))
  etst(plus(N( 1000000000000),N(           -10)),N( 0999999999990))
  etst(plus(N( 1000000000000),N(          -100)),N( 0999999999900))
  etst(plus(N( 1000000000000),N(         -1000)),N( 0999999999000))
  etst(plus(N( 1000000000000),N(      -1000000)),N( 0999999000000))
  etst(plus(N( 1000000000000),N(   -1000000000)),N( 0999000000000))
  etst(plus(N( 1000000000000),N(-1000000000000)),N( 0000000000000))

  etst(plus(N(    1000000000),N( 1000000000000)),N( 1001000000000))
  etst(plus(N(    1000000000),N(    1000000000)),N( 0002000000000))
  etst(plus(N(    1000000000),N(       1000000)),N( 0001001000000))
  etst(plus(N(    1000000000),N(          1000)),N( 0001000001000))
  etst(plus(N(    1000000000),N(           100)),N( 0001000000100))
  etst(plus(N(    1000000000),N(            10)),N( 0001000000010))
  etst(plus(N(    1000000000),N(             1)),N( 0001000000001))
  etst(plus(N(    1000000000),N(             0)),N( 0001000000000))
  etst(plus(N(    1000000000),N(            -1)),N( 0000999999999))
  etst(plus(N(    1000000000),N(           -10)),N( 0000999999990))
  etst(plus(N(    1000000000),N(          -100)),N( 0000999999900))
  etst(plus(N(    1000000000),N(         -1000)),N( 0000999999000))
  etst(plus(N(    1000000000),N(      -1000000)),N( 0000999000000))
  etst(plus(N(    1000000000),N(   -1000000000)),N( 0000000000000))
  etst(plus(N(    1000000000),N(-1000000000000)),N(-0999000000000))

  etst(plus(N(       1000000),N( 1000000000000)),N( 1000001000000))
  etst(plus(N(       1000000),N(    1000000000)),N( 0001001000000))
  etst(plus(N(       1000000),N(       1000000)),N( 0000002000000))
  etst(plus(N(       1000000),N(          1000)),N( 0000001001000))
  etst(plus(N(       1000000),N(           100)),N( 0000001000100))
  etst(plus(N(       1000000),N(            10)),N( 0000001000010))
  etst(plus(N(       1000000),N(             1)),N( 0000001000001))
  etst(plus(N(       1000000),N(             0)),N( 0000001000000))
  etst(plus(N(       1000000),N(            -1)),N( 0000000999999))
  etst(plus(N(       1000000),N(           -10)),N( 0000000999990))
  etst(plus(N(       1000000),N(          -100)),N( 0000000999900))
  etst(plus(N(       1000000),N(         -1000)),N( 0000000999000))
  etst(plus(N(       1000000),N(      -1000000)),N( 0000000000000))
  etst(plus(N(       1000000),N(   -1000000000)),N(-0000999000000))
  etst(plus(N(       1000000),N(-1000000000000)),N(-0999999000000))

  etst(plus(N(          1000),N( 1000000000000)),N( 1000000001000))
  etst(plus(N(          1000),N(    1000000000)),N( 0001000001000))
  etst(plus(N(          1000),N(       1000000)),N( 0000001001000))
  etst(plus(N(          1000),N(          1000)),N( 0000000002000))
  etst(plus(N(          1000),N(           100)),N( 0000000001100))
  etst(plus(N(          1000),N(            10)),N( 0000000001010))
  etst(plus(N(          1000),N(             1)),N( 0000000001001))
  etst(plus(N(          1000),N(             0)),N( 0000000001000))
  etst(plus(N(          1000),N(            -1)),N( 0000000000999))
  etst(plus(N(          1000),N(           -10)),N( 0000000000990))
  etst(plus(N(          1000),N(          -100)),N( 0000000000900))
  etst(plus(N(          1000),N(         -1000)),N(-0000000000000))
  etst(plus(N(          1000),N(      -1000000)),N(-0000000999000))
  etst(plus(N(          1000),N(   -1000000000)),N(-0000999999000))
  etst(plus(N(          1000),N(-1000000000000)),N(-0999999999000))

  etst(plus(N(           100),N( 1000000000000)),N( 1000000000100))
  etst(plus(N(           100),N(    1000000000)),N( 0001000000100))
  etst(plus(N(           100),N(       1000000)),N( 0000001000100))
  etst(plus(N(           100),N(          1000)),N( 0000000001100))
  etst(plus(N(           100),N(           100)),N( 0000000000200))
  etst(plus(N(           100),N(            10)),N( 0000000000110))
  etst(plus(N(           100),N(             1)),N( 0000000000101))
  etst(plus(N(           100),N(             0)),N( 0000000000100))
  etst(plus(N(           100),N(            -1)),N( 0000000000099))
  etst(plus(N(           100),N(           -10)),N( 0000000000090))
  etst(plus(N(           100),N(          -100)),N(-0000000000000))
  etst(plus(N(           100),N(         -1000)),N(-0000000000900))
  etst(plus(N(           100),N(      -1000000)),N(-0000000999900))
  etst(plus(N(           100),N(   -1000000000)),N(-0000999999900))
  etst(plus(N(           100),N(-1000000000000)),N(-0999999999900))

  etst(plus(N(            10),N( 1000000000000)),N( 1000000000010))
  etst(plus(N(            10),N(    1000000000)),N( 0001000000010))
  etst(plus(N(            10),N(       1000000)),N( 0000001000010))
  etst(plus(N(            10),N(          1000)),N( 0000000001010))
  etst(plus(N(            10),N(           100)),N( 0000000000110))
  etst(plus(N(            10),N(            10)),N( 0000000000020))
  etst(plus(N(            10),N(             1)),N( 0000000000011))
  etst(plus(N(            10),N(             0)),N( 0000000000010))
  etst(plus(N(            10),N(            -1)),N( 0000000000009))
  etst(plus(N(            10),N(           -10)),N(-0000000000000))
  etst(plus(N(            10),N(          -100)),N(-0000000000090))
  etst(plus(N(            10),N(         -1000)),N(-0000000000990))
  etst(plus(N(            10),N(      -1000000)),N(-0000000999990))
  etst(plus(N(            10),N(   -1000000000)),N(-0000999999990))
  etst(plus(N(            10),N(-1000000000000)),N(-0999999999990))

  etst(plus(N(             1),N( 1000000000000)),N( 1000000000001))
  etst(plus(N(             1),N(    1000000000)),N( 0001000000001))
  etst(plus(N(             1),N(       1000000)),N( 0000001000001))
  etst(plus(N(             1),N(          1000)),N( 0000000001001))
  etst(plus(N(             1),N(           100)),N( 0000000000101))
  etst(plus(N(             1),N(            10)),N( 0000000000011))
  etst(plus(N(             1),N(             1)),N( 0000000000002))
  etst(plus(N(             1),N(             0)),N( 0000000000001))
  etst(plus(N(             1),N(            -1)),N(-0000000000000))
  etst(plus(N(             1),N(           -10)),N(-0000000000009))
  etst(plus(N(             1),N(          -100)),N(-0000000000099))
  etst(plus(N(             1),N(         -1000)),N(-0000000000999))
  etst(plus(N(             1),N(      -1000000)),N(-0000000999999))
  etst(plus(N(             1),N(   -1000000000)),N(-0000999999999))
  etst(plus(N(             1),N(-1000000000000)),N(-0999999999999))

  xtst(minus(TT,TT))
  xtst(minus(TT,FF))
  xtst(minus(TT,I0))
  xtst(minus(TT,P0))
  xtst(minus(TT,M0))
  xtst(minus(TT,B0))
  xtst(minus(FF,TT))
  xtst(minus(FF,FF))
  xtst(minus(FF,I0))
  xtst(minus(FF,P0))
  xtst(minus(FF,M0))
  xtst(minus(FF,B0))
  xtst(minus(I0,TT))
  xtst(minus(I0,FF))
  xtst(minus(I0,P0))
  xtst(minus(I0,M0))
  xtst(minus(I0,B0))
  xtst(minus(P0,TT))
  xtst(minus(P0,FF))
  xtst(minus(P0,I0))
  xtst(minus(P0,P0))
  xtst(minus(P0,M0))
  xtst(minus(P0,B0))
  xtst(minus(M0,TT))
  xtst(minus(M0,FF))
  xtst(minus(M0,I0))
  xtst(minus(M0,P0))
  xtst(minus(M0,M0))
  xtst(minus(M0,B0))
  xtst(minus(B0,TT))
  xtst(minus(B0,FF))
  xtst(minus(B0,I0))
  xtst(minus(B0,P0))
  xtst(minus(B0,M0))
  xtst(minus(B0,B0))
  etst(minus(JN(-3),JN(-3)),JN( 0))
  etst(minus(JN(-3),JN(-2)),JN(-1))
  etst(minus(JN(-3),JN(-1)),JN(-2))
  etst(minus(JN(-3),JN( 0)),JN(-3))
  etst(minus(JN(-3),JN( 1)),JN(-4))
  etst(minus(JN(-3),JN( 2)),JN(-5))
  etst(minus(JN(-2),JN(-3)),JN( 1))
  etst(minus(JN(-2),JN(-2)),JN( 0))
  etst(minus(JN(-2),JN(-1)),JN(-1))
  etst(minus(JN(-2),JN( 0)),JN(-2))
  etst(minus(JN(-2),JN( 1)),JN(-3))
  etst(minus(JN(-2),JN( 2)),JN(-4))
  etst(minus(JN(-1),JN(-3)),JN( 2))
  etst(minus(JN(-1),JN(-2)),JN( 1))
  etst(minus(JN(-1),JN(-1)),JN( 0))
  etst(minus(JN(-1),JN( 0)),JN(-1))
  etst(minus(JN(-1),JN( 1)),JN(-2))
  etst(minus(JN(-1),JN( 2)),JN(-3))
  etst(minus(JN( 0),JN(-3)),JN( 3))
  etst(minus(JN( 0),JN(-2)),JN( 2))
  etst(minus(JN( 0),JN(-1)),JN( 1))
  etst(minus(JN( 0),JN( 0)),JN( 0))
  etst(minus(JN( 0),JN( 1)),JN(-1))
  etst(minus(JN( 0),JN( 2)),JN(-2))
  etst(minus(JN( 1),JN(-3)),JN( 4))
  etst(minus(JN( 1),JN(-2)),JN( 3))
  etst(minus(JN( 1),JN(-1)),JN( 2))
  etst(minus(JN( 1),JN( 0)),JN( 1))
  etst(minus(JN( 1),JN( 1)),JN( 0))
  etst(minus(JN( 1),JN( 2)),JN(-1))
  etst(minus(JN( 2),JN(-3)),JN( 5))
  etst(minus(JN( 2),JN(-2)),JN( 4))
  etst(minus(JN( 2),JN(-1)),JN( 3))
  etst(minus(JN( 2),JN( 0)),JN( 2))
  etst(minus(JN( 2),JN( 1)),JN( 1))
  etst(minus(JN( 2),JN( 2)),JN( 0))
  etst(minus(K2( 0, 0),K2( 0, 0)),K2( 0, 0))
  etst(minus(K2( 0, 0),K2( 0, 1)),K2( 0,-1))
  etst(minus(K2( 0, 0),K2( 1,-1)),K2(-1, 0))
  etst(minus(K2( 0, 0),K2( 1, 0)),K2(-1,-1))
  etst(minus(K2( 0,-1),K2( 0, 0)),K2( 0,-1))
  etst(minus(K2( 0,-1),K2( 0, 1)),K2( 0,-2))
  etst(minus(K2( 0,-1),K2( 1,-1)),K2(-1,-1))
  etst(minus(K2( 0,-1),K2( 1, 0)),K2(-1,-2))
  etst(minus(K2(-1, 0),K2( 0, 0)),K2(-1, 0))
  etst(minus(K2(-1, 0),K2( 0, 1)),K2(-1,-1))
  etst(minus(K2(-1, 0),K2( 1,-1)),K2(-2, 1))
  etst(minus(K2(-1, 0),K2( 1, 0)),K2(-2, 0))
  etst(minus(K2(-1,-1),K2( 0, 0)),K2(-1,-1))
  etst(minus(K2(-1,-1),K2( 0, 1)),K2(-1,-2))
  etst(minus(K2(-1,-1),K2( 1,-1)),K2(-2, 0))
  etst(minus(K2(-1,-1),K2( 1, 0)),K2(-2,-1))

  xtst(unaryPlus(TT))
  xtst(unaryPlus(FF))
  xtst(unaryPlus(P0))
  xtst(unaryPlus(M0))
  xtst(unaryPlus(B0))
  etst(unaryPlus(Z),Z)
  etst(unaryPlus(M),M)
  etst(unaryPlus(I0011),I0011)
  etst(unaryPlus(J1010_0101),J1010_0101)

  xtst(unaryMinus(TT))
  xtst(unaryMinus(FF))
  xtst(unaryMinus(P0))
  xtst(unaryMinus(M0))
  xtst(unaryMinus(B0))
  etst(unaryMinus(JN(-2)),JN( 2))
  etst(unaryMinus(JN(-1)),JN( 1))
  etst(unaryMinus(JN( 0)),JN( 0))
  etst(unaryMinus(JN( 1)),JN(-1))
  etst(unaryMinus(JN( 2)),JN(-2))
  etst(unaryMinus(K2(-1,-1)),K2( 1, 0))
  etst(unaryMinus(K2(-1, 0)),K2( 1,-1))
  etst(unaryMinus(K2(-1, 1)),K2( 1,-2))
  etst(unaryMinus(K2( 0,-1)),K2( 0, 1))
  etst(unaryMinus(K2( 0, 0)),K2( 0, 0))
  etst(unaryMinus(K2( 0, 1)),K2( 0,-1))
  etst(unaryMinus(K2( 1,-1)),K2(-1, 0))
  etst(unaryMinus(K2( 1, 0)),K2(-1,-1))
  etst(unaryMinus(K2( 1, 1)),K2(-1,-2))

  xtst(integerLength(TT))
  xtst(integerLength(FF))
  xtst(integerLength(P0))
  xtst(integerLength(M0))
  xtst(integerLength(B0))
  etst(integerLength(Z),Z)
  etst(integerLength(M),Z)
  etst(integerLength(JN(1)),JN(1))
  etst(integerLength(JN(2)),JN(2))
  etst(integerLength(JN(3)),JN(2))
  etst(integerLength(JN(4)),JN(3))
  etst(integerLength(JN(5)),JN(3))
  etst(integerLength(JN(6)),JN(3))
  etst(integerLength(JN(7)),JN(3))
  etst(integerLength(JN(8)),JN(4))
  etst(integerLength(JN(-2)),JN(1))
  etst(integerLength(JN(-3)),JN(2))
  etst(integerLength(JN(-4)),JN(2))
  etst(integerLength(JN(-5)),JN(3))
  etst(integerLength(JN(-6)),JN(3))
  etst(integerLength(JN(-7)),JN(3))
  etst(integerLength(JN(-8)),JN(3))
  etst(integerLength(JN(-9)),JN(4))
  etst(integerLength(I0011_1100),JN(4+1*pntsize))
  etst(integerLength(J1011_1101),JN(2+1*pntsize))
  etst(integerLength(intcons(0,I0011_1100)),JN(4+2*pntsize))
  etst(integerLength(intcons(0,J1011_1101)),JN(2+2*pntsize))
  etst(integerLength(K2(0,-1)),JN(0+1*pntsize))
  etst(integerLength(K2(-1,0)),JN(0+1*pntsize))

  xtst(logcount(TT))
  xtst(logcount(FF))
  xtst(logcount(P0))
  xtst(logcount(M0))
  xtst(logcount(B0))
  etst(logcount(Z),Z)
  etst(logcount(M),Z)
  etst(logcount(JN(1)),JN(1))
  etst(logcount(JN(2)),JN(1))
  etst(logcount(JN(3)),JN(2))
  etst(logcount(JN(4)),JN(1))
  etst(logcount(JN(5)),JN(2))
  etst(logcount(JN(6)),JN(2))
  etst(logcount(JN(7)),JN(3))
  etst(logcount(JN(8)),JN(1))
  etst(logcount(JN(-2)),JN(1))
  etst(logcount(JN(-3)),JN(1))
  etst(logcount(JN(-4)),JN(2))
  etst(logcount(JN(-5)),JN(1))
  etst(logcount(JN(-6)),JN(2))
  etst(logcount(JN(-7)),JN(2))
  etst(logcount(JN(-8)),JN(3))
  etst(logcount(JN(-9)),JN(1))
  etst(logcount(I0011_1100),JN(4))
  etst(logcount(J1011_1101),JN(2))
  etst(logcount(intcons(0,I0011_1100)),JN(4))
  etst(logcount(intcons(0,J1011_1101)),JN(2+1*pntsize))
  etst(logcount(K2(0,-1)),JN(0+1*pntsize))
  etst(logcount(K2(-1,0)),JN(0+1*pntsize))

  xtst(digit0(TT))
  xtst(digit0(FF))
  xtst(digit0(P0))
  xtst(digit0(M0))
  xtst(digit0(B0))
  etst(digit0(JN( 0)),JN(  0))
  etst(digit0(JN( 1)),JN( 10))
  etst(digit0(JN( 2)),JN( 20))
  etst(digit0(JN( 3)),JN( 30))
  etst(digit0(JN( 4)),JN( 40))
  etst(digit0(JN( 5)),JN( 50))
  etst(digit0(JN( 6)),JN( 60))
  etst(digit0(JN( 7)),JN( 70))
  etst(digit0(JN( 8)),JN( 80))
  etst(digit0(JN( 9)),JN( 90))
  etst(digit0(JN(-1)),JN(-10))
  etst(digit0(JN(-2)),JN(-20))
  etst(digit0(JN(-3)),JN(-30))
  etst(digit0(JN(-4)),JN(-40))
  etst(digit0(JN(-5)),JN(-50))
  etst(digit0(JN(-6)),JN(-60))
  etst(digit0(JN(-7)),JN(-70))
  etst(digit0(JN(-8)),JN(-80))
  etst(digit0(JN(-9)),JN(-90))

  xtst(digit1(TT))
  xtst(digit1(FF))
  xtst(digit1(P0))
  xtst(digit1(M0))
  xtst(digit1(B0))
  etst(digit1(JN( 0)),JN(  1))
  etst(digit1(JN( 1)),JN( 11))
  etst(digit1(JN( 2)),JN( 21))
  etst(digit1(JN( 3)),JN( 31))
  etst(digit1(JN( 4)),JN( 41))
  etst(digit1(JN( 5)),JN( 51))
  etst(digit1(JN( 6)),JN( 61))
  etst(digit1(JN( 7)),JN( 71))
  etst(digit1(JN( 8)),JN( 81))
  etst(digit1(JN( 9)),JN( 91))
  etst(digit1(JN(-1)),JN(- 9))
  etst(digit1(JN(-2)),JN(-19))
  etst(digit1(JN(-3)),JN(-29))
  etst(digit1(JN(-4)),JN(-39))
  etst(digit1(JN(-5)),JN(-49))
  etst(digit1(JN(-6)),JN(-59))
  etst(digit1(JN(-7)),JN(-69))
  etst(digit1(JN(-8)),JN(-79))
  etst(digit1(JN(-9)),JN(-89))

  xtst(digit2(TT))
  xtst(digit2(FF))
  xtst(digit2(P0))
  xtst(digit2(M0))
  xtst(digit2(B0))
  etst(digit2(JN( 0)),JN(  2))
  etst(digit2(JN( 1)),JN( 12))
  etst(digit2(JN( 2)),JN( 22))
  etst(digit2(JN( 3)),JN( 32))
  etst(digit2(JN( 4)),JN( 42))
  etst(digit2(JN( 5)),JN( 52))
  etst(digit2(JN( 6)),JN( 62))
  etst(digit2(JN( 7)),JN( 72))
  etst(digit2(JN( 8)),JN( 82))
  etst(digit2(JN( 9)),JN( 92))
  etst(digit2(JN(-1)),JN(- 8))
  etst(digit2(JN(-2)),JN(-18))
  etst(digit2(JN(-3)),JN(-28))
  etst(digit2(JN(-4)),JN(-38))
  etst(digit2(JN(-5)),JN(-48))
  etst(digit2(JN(-6)),JN(-58))
  etst(digit2(JN(-7)),JN(-68))
  etst(digit2(JN(-8)),JN(-78))
  etst(digit2(JN(-9)),JN(-88))

  xtst(digit3(TT))
  xtst(digit3(FF))
  xtst(digit3(P0))
  xtst(digit3(M0))
  xtst(digit3(B0))
  etst(digit3(JN( 0)),JN(  3))
  etst(digit3(JN( 1)),JN( 13))
  etst(digit3(JN( 2)),JN( 23))
  etst(digit3(JN( 3)),JN( 33))
  etst(digit3(JN( 4)),JN( 43))
  etst(digit3(JN( 5)),JN( 53))
  etst(digit3(JN( 6)),JN( 63))
  etst(digit3(JN( 7)),JN( 73))
  etst(digit3(JN( 8)),JN( 83))
  etst(digit3(JN( 9)),JN( 93))
  etst(digit3(JN(-1)),JN(- 7))
  etst(digit3(JN(-2)),JN(-17))
  etst(digit3(JN(-3)),JN(-27))
  etst(digit3(JN(-4)),JN(-37))
  etst(digit3(JN(-5)),JN(-47))
  etst(digit3(JN(-6)),JN(-57))
  etst(digit3(JN(-7)),JN(-67))
  etst(digit3(JN(-8)),JN(-77))
  etst(digit3(JN(-9)),JN(-87))

  xtst(digit4(TT))
  xtst(digit4(FF))
  xtst(digit4(P0))
  xtst(digit4(M0))
  xtst(digit4(B0))
  etst(digit4(JN( 0)),JN(  4))
  etst(digit4(JN( 1)),JN( 14))
  etst(digit4(JN( 2)),JN( 24))
  etst(digit4(JN( 3)),JN( 34))
  etst(digit4(JN( 4)),JN( 44))
  etst(digit4(JN( 5)),JN( 54))
  etst(digit4(JN( 6)),JN( 64))
  etst(digit4(JN( 7)),JN( 74))
  etst(digit4(JN( 8)),JN( 84))
  etst(digit4(JN( 9)),JN( 94))
  etst(digit4(JN(-1)),JN(- 6))
  etst(digit4(JN(-2)),JN(-16))
  etst(digit4(JN(-3)),JN(-26))
  etst(digit4(JN(-4)),JN(-36))
  etst(digit4(JN(-5)),JN(-46))
  etst(digit4(JN(-6)),JN(-56))
  etst(digit4(JN(-7)),JN(-66))
  etst(digit4(JN(-8)),JN(-76))
  etst(digit4(JN(-9)),JN(-86))

  xtst(digit5(TT))
  xtst(digit5(FF))
  xtst(digit5(P0))
  xtst(digit5(M0))
  xtst(digit5(B0))
  etst(digit5(JN( 0)),JN(  5))
  etst(digit5(JN( 1)),JN( 15))
  etst(digit5(JN( 2)),JN( 25))
  etst(digit5(JN( 3)),JN( 35))
  etst(digit5(JN( 4)),JN( 45))
  etst(digit5(JN( 5)),JN( 55))
  etst(digit5(JN( 6)),JN( 65))
  etst(digit5(JN( 7)),JN( 75))
  etst(digit5(JN( 8)),JN( 85))
  etst(digit5(JN( 9)),JN( 95))
  etst(digit5(JN(-1)),JN(- 5))
  etst(digit5(JN(-2)),JN(-15))
  etst(digit5(JN(-3)),JN(-25))
  etst(digit5(JN(-4)),JN(-35))
  etst(digit5(JN(-5)),JN(-45))
  etst(digit5(JN(-6)),JN(-55))
  etst(digit5(JN(-7)),JN(-65))
  etst(digit5(JN(-8)),JN(-75))
  etst(digit5(JN(-9)),JN(-85))

  xtst(digit6(TT))
  xtst(digit6(FF))
  xtst(digit6(P0))
  xtst(digit6(M0))
  xtst(digit6(B0))
  etst(digit6(JN( 0)),JN(  6))
  etst(digit6(JN( 1)),JN( 16))
  etst(digit6(JN( 2)),JN( 26))
  etst(digit6(JN( 3)),JN( 36))
  etst(digit6(JN( 4)),JN( 46))
  etst(digit6(JN( 5)),JN( 56))
  etst(digit6(JN( 6)),JN( 66))
  etst(digit6(JN( 7)),JN( 76))
  etst(digit6(JN( 8)),JN( 86))
  etst(digit6(JN( 9)),JN( 96))
  etst(digit6(JN(-1)),JN(- 4))
  etst(digit6(JN(-2)),JN(-14))
  etst(digit6(JN(-3)),JN(-24))
  etst(digit6(JN(-4)),JN(-34))
  etst(digit6(JN(-5)),JN(-44))
  etst(digit6(JN(-6)),JN(-54))
  etst(digit6(JN(-7)),JN(-64))
  etst(digit6(JN(-8)),JN(-74))
  etst(digit6(JN(-9)),JN(-84))

  xtst(digit7(TT))
  xtst(digit7(FF))
  xtst(digit7(P0))
  xtst(digit7(M0))
  xtst(digit7(B0))
  etst(digit7(JN( 0)),JN(  7))
  etst(digit7(JN( 1)),JN( 17))
  etst(digit7(JN( 2)),JN( 27))
  etst(digit7(JN( 3)),JN( 37))
  etst(digit7(JN( 4)),JN( 47))
  etst(digit7(JN( 5)),JN( 57))
  etst(digit7(JN( 6)),JN( 67))
  etst(digit7(JN( 7)),JN( 77))
  etst(digit7(JN( 8)),JN( 87))
  etst(digit7(JN( 9)),JN( 97))
  etst(digit7(JN(-1)),JN(- 3))
  etst(digit7(JN(-2)),JN(-13))
  etst(digit7(JN(-3)),JN(-23))
  etst(digit7(JN(-4)),JN(-33))
  etst(digit7(JN(-5)),JN(-43))
  etst(digit7(JN(-6)),JN(-53))
  etst(digit7(JN(-7)),JN(-63))
  etst(digit7(JN(-8)),JN(-73))
  etst(digit7(JN(-9)),JN(-83))

  xtst(digit8(TT))
  xtst(digit8(FF))
  xtst(digit8(P0))
  xtst(digit8(M0))
  xtst(digit8(B0))
  etst(digit8(JN( 0)),JN(  8))
  etst(digit8(JN( 1)),JN( 18))
  etst(digit8(JN( 2)),JN( 28))
  etst(digit8(JN( 3)),JN( 38))
  etst(digit8(JN( 4)),JN( 48))
  etst(digit8(JN( 5)),JN( 58))
  etst(digit8(JN( 6)),JN( 68))
  etst(digit8(JN( 7)),JN( 78))
  etst(digit8(JN( 8)),JN( 88))
  etst(digit8(JN( 9)),JN( 98))
  etst(digit8(JN(-1)),JN(- 2))
  etst(digit8(JN(-2)),JN(-12))
  etst(digit8(JN(-3)),JN(-22))
  etst(digit8(JN(-4)),JN(-32))
  etst(digit8(JN(-5)),JN(-42))
  etst(digit8(JN(-6)),JN(-52))
  etst(digit8(JN(-7)),JN(-62))
  etst(digit8(JN(-8)),JN(-72))
  etst(digit8(JN(-9)),JN(-82))

  xtst(digit9(TT))
  xtst(digit9(FF))
  xtst(digit9(P0))
  xtst(digit9(M0))
  xtst(digit9(B0))
  etst(digit9(JN( 0)),JN(  9))
  etst(digit9(JN( 1)),JN( 19))
  etst(digit9(JN( 2)),JN( 29))
  etst(digit9(JN( 3)),JN( 39))
  etst(digit9(JN( 4)),JN( 49))
  etst(digit9(JN( 5)),JN( 59))
  etst(digit9(JN( 6)),JN( 69))
  etst(digit9(JN( 7)),JN( 79))
  etst(digit9(JN( 8)),JN( 89))
  etst(digit9(JN( 9)),JN( 99))
  etst(digit9(JN(-1)),JN(- 1))
  etst(digit9(JN(-2)),JN(-11))
  etst(digit9(JN(-3)),JN(-21))
  etst(digit9(JN(-4)),JN(-31))
  etst(digit9(JN(-5)),JN(-41))
  etst(digit9(JN(-6)),JN(-51))
  etst(digit9(JN(-7)),JN(-61))
  etst(digit9(JN(-8)),JN(-71))
  etst(digit9(JN(-9)),JN(-81))

  etst(N( 0),JN( 0))
  etst(N( 1),JN( 1))
  etst(N( 2),JN( 2))
  etst(N( 3),JN( 3))
  etst(N( 4),JN( 4))
  etst(N( 5),JN( 5))
  etst(N( 6),JN( 6))
  etst(N( 7),JN( 7))
  etst(N( 8),JN( 8))
  etst(N( 9),JN( 9))
  etst(N(-1),JN(-1))
  etst(N(-2),JN(-2))
  etst(N(-3),JN(-3))
  etst(N(-4),JN(-4))
  etst(N(-5),JN(-5))
  etst(N(-6),JN(-6))
  etst(N(-7),JN(-7))
  etst(N(-8),JN(-8))
  etst(N(-9),JN(-9))

  xtst(times(TT,TT))
  xtst(times(TT,FF))
  xtst(times(TT,I0))
  xtst(times(TT,P0))
  xtst(times(TT,M0))
  xtst(times(TT,B0))
  xtst(times(FF,TT))
  xtst(times(FF,FF))
  xtst(times(FF,I0))
  xtst(times(FF,P0))
  xtst(times(FF,M0))
  xtst(times(FF,B0))
  xtst(times(I0,TT))
  xtst(times(I0,FF))
  xtst(times(I0,P0))
  xtst(times(I0,M0))
  xtst(times(I0,B0))
  xtst(times(P0,TT))
  xtst(times(P0,FF))
  xtst(times(P0,I0))
  xtst(times(P0,P0))
  xtst(times(P0,M0))
  xtst(times(P0,B0))
  xtst(times(M0,TT))
  xtst(times(M0,FF))
  xtst(times(M0,I0))
  xtst(times(M0,P0))
  xtst(times(M0,M0))
  xtst(times(M0,B0))
  xtst(times(B0,TT))
  xtst(times(B0,FF))
  xtst(times(B0,I0))
  xtst(times(B0,P0))
  xtst(times(B0,M0))
  xtst(times(B0,B0))
  etst(times(JN(-3),JN(-3)),JN( 9))
  etst(times(JN(-3),JN(-2)),JN( 6))
  etst(times(JN(-3),JN(-1)),JN( 3))
  etst(times(JN(-3),JN( 0)),JN( 0))
  etst(times(JN(-3),JN( 1)),JN(-3))
  etst(times(JN(-3),JN( 2)),JN(-6))
  etst(times(JN(-2),JN(-3)),JN( 6))
  etst(times(JN(-2),JN(-2)),JN( 4))
  etst(times(JN(-2),JN(-1)),JN( 2))
  etst(times(JN(-2),JN( 0)),JN( 0))
  etst(times(JN(-2),JN( 1)),JN(-2))
  etst(times(JN(-2),JN( 2)),JN(-4))
  etst(times(JN(-1),JN(-3)),JN( 3))
  etst(times(JN(-1),JN(-2)),JN( 2))
  etst(times(JN(-1),JN(-1)),JN( 1))
  etst(times(JN(-1),JN( 0)),JN( 0))
  etst(times(JN(-1),JN( 1)),JN(-1))
  etst(times(JN(-1),JN( 2)),JN(-2))
  etst(times(JN( 0),JN(-3)),JN( 0))
  etst(times(JN( 0),JN(-2)),JN( 0))
  etst(times(JN( 0),JN(-1)),JN( 0))
  etst(times(JN( 0),JN( 0)),JN( 0))
  etst(times(JN( 0),JN( 1)),JN( 0))
  etst(times(JN( 0),JN( 2)),JN( 0))
  etst(times(JN( 1),JN(-3)),JN(-3))
  etst(times(JN( 1),JN(-2)),JN(-2))
  etst(times(JN( 1),JN(-1)),JN(-1))
  etst(times(JN( 1),JN( 0)),JN( 0))
  etst(times(JN( 1),JN( 1)),JN( 1))
  etst(times(JN( 1),JN( 2)),JN( 2))
  etst(times(JN( 2),JN(-3)),JN(-6))
  etst(times(JN( 2),JN(-2)),JN(-4))
  etst(times(JN( 2),JN(-1)),JN(-2))
  etst(times(JN( 2),JN( 0)),JN( 0))
  etst(times(JN( 2),JN( 1)),JN( 2))
  etst(times(JN( 2),JN( 2)),JN( 4))

  etst(times(N(-1000000000000),N(-1000000000000)),N( 1000000000000000000000000))
  etst(times(N(-1000000000000),N(   -1000000000)),N(    1000000000000000000000))
  etst(times(N(-1000000000000),N(      -1000000)),N(       1000000000000000000))
  etst(times(N(-1000000000000),N(         -1000)),N(          1000000000000000))
  etst(times(N(-1000000000000),N(          -100)),N(           100000000000000))
  etst(times(N(-1000000000000),N(           -10)),N(            10000000000000))
  etst(times(N(-1000000000000),N(            -1)),N(             1000000000000))
  etst(times(N(-1000000000000),N(             0)),N(                         0))
  etst(times(N(-1000000000000),N(             1)),N(            -1000000000000))
  etst(times(N(-1000000000000),N(            10)),N(           -10000000000000))
  etst(times(N(-1000000000000),N(           100)),N(          -100000000000000))
  etst(times(N(-1000000000000),N(          1000)),N(         -1000000000000000))
  etst(times(N(-1000000000000),N(       1000000)),N(      -1000000000000000000))
  etst(times(N(-1000000000000),N(    1000000000)),N(   -1000000000000000000000))
  etst(times(N(-1000000000000),N( 1000000000000)),N(-1000000000000000000000000))

  etst(times(N(-1000000000),N(-1000000000000)),N( 1000000000000000000000))
  etst(times(N(-1000000000),N(   -1000000000)),N(    1000000000000000000))
  etst(times(N(-1000000000),N(      -1000000)),N(       1000000000000000))
  etst(times(N(-1000000000),N(         -1000)),N(          1000000000000))
  etst(times(N(-1000000000),N(          -100)),N(           100000000000))
  etst(times(N(-1000000000),N(           -10)),N(            10000000000))
  etst(times(N(-1000000000),N(            -1)),N(             1000000000))
  etst(times(N(-1000000000),N(             0)),N(                      0))
  etst(times(N(-1000000000),N(             1)),N(            -1000000000))
  etst(times(N(-1000000000),N(            10)),N(           -10000000000))
  etst(times(N(-1000000000),N(           100)),N(          -100000000000))
  etst(times(N(-1000000000),N(          1000)),N(         -1000000000000))
  etst(times(N(-1000000000),N(       1000000)),N(      -1000000000000000))
  etst(times(N(-1000000000),N(    1000000000)),N(   -1000000000000000000))
  etst(times(N(-1000000000),N( 1000000000000)),N(-1000000000000000000000))

  etst(times(N(-1000000),N(-1000000000000)),N( 1000000000000000000))
  etst(times(N(-1000000),N(   -1000000000)),N(    1000000000000000))
  etst(times(N(-1000000),N(      -1000000)),N(       1000000000000))
  etst(times(N(-1000000),N(         -1000)),N(          1000000000))
  etst(times(N(-1000000),N(          -100)),N(           100000000))
  etst(times(N(-1000000),N(           -10)),N(            10000000))
  etst(times(N(-1000000),N(            -1)),N(             1000000))
  etst(times(N(-1000000),N(             0)),N(                   0))
  etst(times(N(-1000000),N(             1)),N(            -1000000))
  etst(times(N(-1000000),N(            10)),N(           -10000000))
  etst(times(N(-1000000),N(           100)),N(          -100000000))
  etst(times(N(-1000000),N(          1000)),N(         -1000000000))
  etst(times(N(-1000000),N(       1000000)),N(      -1000000000000))
  etst(times(N(-1000000),N(    1000000000)),N(   -1000000000000000))
  etst(times(N(-1000000),N( 1000000000000)),N(-1000000000000000000))

  etst(times(N(-1000),N(-1000000000000)),N( 1000000000000000))
  etst(times(N(-1000),N(   -1000000000)),N(    1000000000000))
  etst(times(N(-1000),N(      -1000000)),N(       1000000000))
  etst(times(N(-1000),N(         -1000)),N(          1000000))
  etst(times(N(-1000),N(          -100)),N(           100000))
  etst(times(N(-1000),N(           -10)),N(            10000))
  etst(times(N(-1000),N(            -1)),N(             1000))
  etst(times(N(-1000),N(             0)),N(                0))
  etst(times(N(-1000),N(             1)),N(            -1000))
  etst(times(N(-1000),N(            10)),N(           -10000))
  etst(times(N(-1000),N(           100)),N(          -100000))
  etst(times(N(-1000),N(          1000)),N(         -1000000))
  etst(times(N(-1000),N(       1000000)),N(      -1000000000))
  etst(times(N(-1000),N(    1000000000)),N(   -1000000000000))
  etst(times(N(-1000),N( 1000000000000)),N(-1000000000000000))

  etst(times(N(-100),N(-1000000000000)),N( 100000000000000))
  etst(times(N(-100),N(   -1000000000)),N(    100000000000))
  etst(times(N(-100),N(      -1000000)),N(       100000000))
  etst(times(N(-100),N(         -1000)),N(          100000))
  etst(times(N(-100),N(          -100)),N(           10000))
  etst(times(N(-100),N(           -10)),N(            1000))
  etst(times(N(-100),N(            -1)),N(             100))
  etst(times(N(-100),N(             0)),N(               0))
  etst(times(N(-100),N(             1)),N(            -100))
  etst(times(N(-100),N(            10)),N(           -1000))
  etst(times(N(-100),N(           100)),N(          -10000))
  etst(times(N(-100),N(          1000)),N(         -100000))
  etst(times(N(-100),N(       1000000)),N(      -100000000))
  etst(times(N(-100),N(    1000000000)),N(   -100000000000))
  etst(times(N(-100),N( 1000000000000)),N(-100000000000000))

  etst(times(N(-10),N(-1000000000000)),N( 10000000000000))
  etst(times(N(-10),N(   -1000000000)),N(    10000000000))
  etst(times(N(-10),N(      -1000000)),N(       10000000))
  etst(times(N(-10),N(         -1000)),N(          10000))
  etst(times(N(-10),N(          -100)),N(           1000))
  etst(times(N(-10),N(           -10)),N(            100))
  etst(times(N(-10),N(            -1)),N(             10))
  etst(times(N(-10),N(             0)),N(              0))
  etst(times(N(-10),N(             1)),N(            -10))
  etst(times(N(-10),N(            10)),N(           -100))
  etst(times(N(-10),N(           100)),N(          -1000))
  etst(times(N(-10),N(          1000)),N(         -10000))
  etst(times(N(-10),N(       1000000)),N(      -10000000))
  etst(times(N(-10),N(    1000000000)),N(   -10000000000))
  etst(times(N(-10),N( 1000000000000)),N(-10000000000000))

  etst(times(N(-1),N(-1000000000000)),N( 1000000000000))
  etst(times(N(-1),N(   -1000000000)),N(    1000000000))
  etst(times(N(-1),N(      -1000000)),N(       1000000))
  etst(times(N(-1),N(         -1000)),N(          1000))
  etst(times(N(-1),N(          -100)),N(           100))
  etst(times(N(-1),N(           -10)),N(            10))
  etst(times(N(-1),N(            -1)),N(             1))
  etst(times(N(-1),N(             0)),N(             0))
  etst(times(N(-1),N(             1)),N(            -1))
  etst(times(N(-1),N(            10)),N(           -10))
  etst(times(N(-1),N(           100)),N(          -100))
  etst(times(N(-1),N(          1000)),N(         -1000))
  etst(times(N(-1),N(       1000000)),N(      -1000000))
  etst(times(N(-1),N(    1000000000)),N(   -1000000000))
  etst(times(N(-1),N( 1000000000000)),N(-1000000000000))

  etst(times(N(0),N(-1000000000000)),N(0))
  etst(times(N(0),N(   -1000000000)),N(0))
  etst(times(N(0),N(      -1000000)),N(0))
  etst(times(N(0),N(         -1000)),N(0))
  etst(times(N(0),N(          -100)),N(0))
  etst(times(N(0),N(           -10)),N(0))
  etst(times(N(0),N(            -1)),N(0))
  etst(times(N(0),N(             0)),N(0))
  etst(times(N(0),N(             1)),N(0))
  etst(times(N(0),N(            10)),N(0))
  etst(times(N(0),N(           100)),N(0))
  etst(times(N(0),N(          1000)),N(0))
  etst(times(N(0),N(       1000000)),N(0))
  etst(times(N(0),N(    1000000000)),N(0))
  etst(times(N(0),N( 1000000000000)),N(0))

  etst(times(N(1000000000000),N( 1000000000000)),N( 1000000000000000000000000))
  etst(times(N(1000000000000),N(    1000000000)),N(    1000000000000000000000))
  etst(times(N(1000000000000),N(       1000000)),N(       1000000000000000000))
  etst(times(N(1000000000000),N(          1000)),N(          1000000000000000))
  etst(times(N(1000000000000),N(           100)),N(           100000000000000))
  etst(times(N(1000000000000),N(            10)),N(            10000000000000))
  etst(times(N(1000000000000),N(             1)),N(             1000000000000))
  etst(times(N(1000000000000),N(             0)),N(                         0))
  etst(times(N(1000000000000),N(            -1)),N(            -1000000000000))
  etst(times(N(1000000000000),N(           -10)),N(           -10000000000000))
  etst(times(N(1000000000000),N(          -100)),N(          -100000000000000))
  etst(times(N(1000000000000),N(         -1000)),N(         -1000000000000000))
  etst(times(N(1000000000000),N(      -1000000)),N(      -1000000000000000000))
  etst(times(N(1000000000000),N(   -1000000000)),N(   -1000000000000000000000))
  etst(times(N(1000000000000),N(-1000000000000)),N(-1000000000000000000000000))

  etst(times(N(1000000000),N( 1000000000000)),N( 1000000000000000000000))
  etst(times(N(1000000000),N(    1000000000)),N(    1000000000000000000))
  etst(times(N(1000000000),N(       1000000)),N(       1000000000000000))
  etst(times(N(1000000000),N(          1000)),N(          1000000000000))
  etst(times(N(1000000000),N(           100)),N(           100000000000))
  etst(times(N(1000000000),N(            10)),N(            10000000000))
  etst(times(N(1000000000),N(             1)),N(             1000000000))
  etst(times(N(1000000000),N(             0)),N(                      0))
  etst(times(N(1000000000),N(            -1)),N(            -1000000000))
  etst(times(N(1000000000),N(           -10)),N(           -10000000000))
  etst(times(N(1000000000),N(          -100)),N(          -100000000000))
  etst(times(N(1000000000),N(         -1000)),N(         -1000000000000))
  etst(times(N(1000000000),N(      -1000000)),N(      -1000000000000000))
  etst(times(N(1000000000),N(   -1000000000)),N(   -1000000000000000000))
  etst(times(N(1000000000),N(-1000000000000)),N(-1000000000000000000000))

  etst(times(N(1000000),N( 1000000000000)),N( 1000000000000000000))
  etst(times(N(1000000),N(    1000000000)),N(    1000000000000000))
  etst(times(N(1000000),N(       1000000)),N(       1000000000000))
  etst(times(N(1000000),N(          1000)),N(          1000000000))
  etst(times(N(1000000),N(           100)),N(           100000000))
  etst(times(N(1000000),N(            10)),N(            10000000))
  etst(times(N(1000000),N(             1)),N(             1000000))
  etst(times(N(1000000),N(             0)),N(                   0))
  etst(times(N(1000000),N(            -1)),N(            -1000000))
  etst(times(N(1000000),N(           -10)),N(           -10000000))
  etst(times(N(1000000),N(          -100)),N(          -100000000))
  etst(times(N(1000000),N(         -1000)),N(         -1000000000))
  etst(times(N(1000000),N(      -1000000)),N(      -1000000000000))
  etst(times(N(1000000),N(   -1000000000)),N(   -1000000000000000))
  etst(times(N(1000000),N(-1000000000000)),N(-1000000000000000000))

  etst(times(N(1000),N( 1000000000000)),N( 1000000000000000))
  etst(times(N(1000),N(    1000000000)),N(    1000000000000))
  etst(times(N(1000),N(       1000000)),N(       1000000000))
  etst(times(N(1000),N(          1000)),N(          1000000))
  etst(times(N(1000),N(           100)),N(           100000))
  etst(times(N(1000),N(            10)),N(            10000))
  etst(times(N(1000),N(             1)),N(             1000))
  etst(times(N(1000),N(             0)),N(                0))
  etst(times(N(1000),N(            -1)),N(            -1000))
  etst(times(N(1000),N(           -10)),N(           -10000))
  etst(times(N(1000),N(          -100)),N(          -100000))
  etst(times(N(1000),N(         -1000)),N(         -1000000))
  etst(times(N(1000),N(      -1000000)),N(      -1000000000))
  etst(times(N(1000),N(   -1000000000)),N(   -1000000000000))
  etst(times(N(1000),N(-1000000000000)),N(-1000000000000000))

  etst(times(N(100),N( 1000000000000)),N( 100000000000000))
  etst(times(N(100),N(    1000000000)),N(    100000000000))
  etst(times(N(100),N(       1000000)),N(       100000000))
  etst(times(N(100),N(          1000)),N(          100000))
  etst(times(N(100),N(           100)),N(           10000))
  etst(times(N(100),N(            10)),N(            1000))
  etst(times(N(100),N(             1)),N(             100))
  etst(times(N(100),N(             0)),N(               0))
  etst(times(N(100),N(            -1)),N(            -100))
  etst(times(N(100),N(           -10)),N(           -1000))
  etst(times(N(100),N(          -100)),N(          -10000))
  etst(times(N(100),N(         -1000)),N(         -100000))
  etst(times(N(100),N(      -1000000)),N(      -100000000))
  etst(times(N(100),N(   -1000000000)),N(   -100000000000))
  etst(times(N(100),N(-1000000000000)),N(-100000000000000))

  etst(times(N(10),N( 1000000000000)),N( 10000000000000))
  etst(times(N(10),N(    1000000000)),N(    10000000000))
  etst(times(N(10),N(       1000000)),N(       10000000))
  etst(times(N(10),N(          1000)),N(          10000))
  etst(times(N(10),N(           100)),N(           1000))
  etst(times(N(10),N(            10)),N(            100))
  etst(times(N(10),N(             1)),N(             10))
  etst(times(N(10),N(             0)),N(              0))
  etst(times(N(10),N(            -1)),N(            -10))
  etst(times(N(10),N(           -10)),N(           -100))
  etst(times(N(10),N(          -100)),N(          -1000))
  etst(times(N(10),N(         -1000)),N(         -10000))
  etst(times(N(10),N(      -1000000)),N(      -10000000))
  etst(times(N(10),N(   -1000000000)),N(   -10000000000))
  etst(times(N(10),N(-1000000000000)),N(-10000000000000))

  etst(times(N(1),N( 1000000000000)),N( 1000000000000))
  etst(times(N(1),N(    1000000000)),N(    1000000000))
  etst(times(N(1),N(       1000000)),N(       1000000))
  etst(times(N(1),N(          1000)),N(          1000))
  etst(times(N(1),N(           100)),N(           100))
  etst(times(N(1),N(            10)),N(            10))
  etst(times(N(1),N(             1)),N(             1))
  etst(times(N(1),N(             0)),N(             0))
  etst(times(N(1),N(            -1)),N(            -1))
  etst(times(N(1),N(           -10)),N(           -10))
  etst(times(N(1),N(          -100)),N(          -100))
  etst(times(N(1),N(         -1000)),N(         -1000))
  etst(times(N(1),N(      -1000000)),N(      -1000000))
  etst(times(N(1),N(   -1000000000)),N(   -1000000000))
  etst(times(N(1),N(-1000000000000)),N(-1000000000000))

  xtst(logbitp(TT,TT))
  xtst(logbitp(TT,FF))
  xtst(logbitp(TT,I0))
  xtst(logbitp(TT,P0))
  xtst(logbitp(TT,M0))
  xtst(logbitp(TT,B0))
  xtst(logbitp(FF,TT))
  xtst(logbitp(FF,FF))
  xtst(logbitp(FF,I0))
  xtst(logbitp(FF,P0))
  xtst(logbitp(FF,M0))
  xtst(logbitp(FF,B0))
  xtst(logbitp(I0,TT))
  xtst(logbitp(I0,FF))
  xtst(logbitp(I0,P0))
  xtst(logbitp(I0,M0))
  xtst(logbitp(I0,B0))
  xtst(logbitp(P0,TT))
  xtst(logbitp(P0,FF))
  xtst(logbitp(P0,I0))
  xtst(logbitp(P0,P0))
  xtst(logbitp(P0,M0))
  xtst(logbitp(P0,B0))
  xtst(logbitp(M0,TT))
  xtst(logbitp(M0,FF))
  xtst(logbitp(M0,I0))
  xtst(logbitp(M0,P0))
  xtst(logbitp(M0,M0))
  xtst(logbitp(M0,B0))
  xtst(logbitp(B0,TT))
  xtst(logbitp(B0,FF))
  xtst(logbitp(B0,I0))
  xtst(logbitp(B0,P0))
  xtst(logbitp(B0,M0))
  xtst(logbitp(B0,B0))
  xtst(logbitp(JN(-3),JN(-3)))
  xtst(logbitp(JN(-3),JN(-2)))
  xtst(logbitp(JN(-3),JN(-1)))
  xtst(logbitp(JN(-3),JN( 0)))
  xtst(logbitp(JN(-3),JN( 1)))
  xtst(logbitp(JN(-3),JN( 2)))
  xtst(logbitp(JN(-2),JN(-3)))
  xtst(logbitp(JN(-2),JN(-2)))
  xtst(logbitp(JN(-2),JN(-1)))
  xtst(logbitp(JN(-2),JN( 0)))
  xtst(logbitp(JN(-2),JN( 1)))
  xtst(logbitp(JN(-2),JN( 2)))
  xtst(logbitp(JN(-1),JN(-3)))
  xtst(logbitp(JN(-1),JN(-2)))
  xtst(logbitp(JN(-1),JN(-1)))
  xtst(logbitp(JN(-1),JN( 0)))
  xtst(logbitp(JN(-1),JN( 1)))
  xtst(logbitp(JN(-1),JN( 2)))
  ttst(logbitp(JN( 0),JN(-3)))
  ftst(logbitp(JN( 0),JN(-2)))
  ttst(logbitp(JN( 0),JN(-1)))
  ftst(logbitp(JN( 0),JN( 0)))
  ttst(logbitp(JN( 0),JN( 1)))
  ftst(logbitp(JN( 0),JN( 2)))
  ftst(logbitp(JN( 1),JN(-3)))
  ttst(logbitp(JN( 1),JN(-2)))
  ttst(logbitp(JN( 1),JN(-1)))
  ftst(logbitp(JN( 1),JN( 0)))
  ftst(logbitp(JN( 1),JN( 1)))
  ttst(logbitp(JN( 1),JN( 2)))
  ttst(logbitp(JN( 2),JN(-3)))
  ttst(logbitp(JN( 2),JN(-2)))
  ttst(logbitp(JN( 2),JN(-1)))
  ftst(logbitp(JN( 2),JN( 0)))
  ftst(logbitp(JN( 2),JN( 1)))
  ftst(logbitp(JN( 2),JN( 2)))
  ttst(logbitp(JN( 0+0*pntsize),I0011_1100))
  ttst(logbitp(JN( 1+0*pntsize),I0011_1100))
  ftst(logbitp(JN( 2+0*pntsize),I0011_1100))
  ftst(logbitp(JN( 3+0*pntsize),I0011_1100))
  ftst(logbitp(JN( 4+0*pntsize),I0011_1100))
  ftst(logbitp(JN( 0+1*pntsize),I0011_1100))
  ftst(logbitp(JN( 1+1*pntsize),I0011_1100))
  ttst(logbitp(JN( 2+1*pntsize),I0011_1100))
  ttst(logbitp(JN( 3+1*pntsize),I0011_1100))
  ftst(logbitp(JN( 4+1*pntsize),I0011_1100))
  ftst(logbitp(JN( 0+2*pntsize),I0011_1100))
  ftst(logbitp(JN( 1+2*pntsize),I0011_1100))
  ftst(logbitp(JN( 2+2*pntsize),I0011_1100))
  ftst(logbitp(JN( 3+2*pntsize),I0011_1100))
  ftst(logbitp(JN( 4+2*pntsize),I0011_1100))
  ttst(logbitp(JN( 0+0*pntsize),J1011_1101))
  ttst(logbitp(JN( 1+0*pntsize),J1011_1101))
  ftst(logbitp(JN( 2+0*pntsize),J1011_1101))
  ttst(logbitp(JN( 3+0*pntsize),J1011_1101))
  ttst(logbitp(JN( 4+0*pntsize),J1011_1101))
  ttst(logbitp(JN( 0+1*pntsize),J1011_1101))
  ftst(logbitp(JN( 1+1*pntsize),J1011_1101))
  ttst(logbitp(JN( 2+1*pntsize),J1011_1101))
  ttst(logbitp(JN( 3+1*pntsize),J1011_1101))
  ttst(logbitp(JN( 4+1*pntsize),J1011_1101))
  ttst(logbitp(JN( 0+2*pntsize),J1011_1101))
  ttst(logbitp(JN( 1+2*pntsize),J1011_1101))
  ttst(logbitp(JN( 2+2*pntsize),J1011_1101))
  ttst(logbitp(JN( 3+2*pntsize),J1011_1101))
  ttst(logbitp(JN( 4+2*pntsize),J1011_1101))

  xtst(ash(TT,TT))
  xtst(ash(TT,FF))
  xtst(ash(TT,I0))
  xtst(ash(TT,P0))
  xtst(ash(TT,M0))
  xtst(ash(TT,B0))
  xtst(ash(FF,TT))
  xtst(ash(FF,FF))
  xtst(ash(FF,I0))
  xtst(ash(FF,P0))
  xtst(ash(FF,M0))
  xtst(ash(FF,B0))
  xtst(ash(I0,TT))
  xtst(ash(I0,FF))
  xtst(ash(I0,P0))
  xtst(ash(I0,M0))
  xtst(ash(I0,B0))
  xtst(ash(P0,TT))
  xtst(ash(P0,FF))
  xtst(ash(P0,I0))
  xtst(ash(P0,P0))
  xtst(ash(P0,M0))
  xtst(ash(P0,B0))
  xtst(ash(M0,TT))
  xtst(ash(M0,FF))
  xtst(ash(M0,I0))
  xtst(ash(M0,P0))
  xtst(ash(M0,M0))
  xtst(ash(M0,B0))
  xtst(ash(B0,TT))
  xtst(ash(B0,FF))
  xtst(ash(B0,I0))
  xtst(ash(B0,P0))
  xtst(ash(B0,M0))
  xtst(ash(B0,B0))

  etst(ash(N(-1000000000000),N(-65)),N(-1))
  etst(ash(N(   -1000000000),N(-65)),N(-1))
  etst(ash(N(      -1000000),N(-65)),N(-1))
  etst(ash(N(         -1000),N(-65)),N(-1))
  etst(ash(N(          -100),N(-65)),N(-1))
  etst(ash(N(           -10),N(-65)),N(-1))
  etst(ash(N(            -1),N(-65)),N(-1))
  etst(ash(N(             0),N(-65)),N( 0))
  etst(ash(N(             1),N(-65)),N( 0))
  etst(ash(N(            10),N(-65)),N( 0))
  etst(ash(N(           100),N(-65)),N( 0))
  etst(ash(N(          1000),N(-65)),N( 0))
  etst(ash(N(       1000000),N(-65)),N( 0))
  etst(ash(N(    1000000000),N(-65)),N( 0))
  etst(ash(N( 1000000000000),N(-65)),N( 0))

  etst(ash(N(-1000000000000),N(-64)),N(-1))
  etst(ash(N(   -1000000000),N(-64)),N(-1))
  etst(ash(N(      -1000000),N(-64)),N(-1))
  etst(ash(N(         -1000),N(-64)),N(-1))
  etst(ash(N(          -100),N(-64)),N(-1))
  etst(ash(N(           -10),N(-64)),N(-1))
  etst(ash(N(            -1),N(-64)),N(-1))
  etst(ash(N(             0),N(-64)),N( 0))
  etst(ash(N(             1),N(-64)),N( 0))
  etst(ash(N(            10),N(-64)),N( 0))
  etst(ash(N(           100),N(-64)),N( 0))
  etst(ash(N(          1000),N(-64)),N( 0))
  etst(ash(N(       1000000),N(-64)),N( 0))
  etst(ash(N(    1000000000),N(-64)),N( 0))
  etst(ash(N( 1000000000000),N(-64)),N( 0))

  etst(ash(N(-1000000000000),N(-63)),N(-1))
  etst(ash(N(   -1000000000),N(-63)),N(-1))
  etst(ash(N(      -1000000),N(-63)),N(-1))
  etst(ash(N(         -1000),N(-63)),N(-1))
  etst(ash(N(          -100),N(-63)),N(-1))
  etst(ash(N(           -10),N(-63)),N(-1))
  etst(ash(N(            -1),N(-63)),N(-1))
  etst(ash(N(             0),N(-63)),N( 0))
  etst(ash(N(             1),N(-63)),N( 0))
  etst(ash(N(            10),N(-63)),N( 0))
  etst(ash(N(           100),N(-63)),N( 0))
  etst(ash(N(          1000),N(-63)),N( 0))
  etst(ash(N(       1000000),N(-63)),N( 0))
  etst(ash(N(    1000000000),N(-63)),N( 0))
  etst(ash(N( 1000000000000),N(-63)),N( 0))

  etst(ash(N(-1000000000000),N( -4)),N(-62500000000))
  etst(ash(N(   -1000000000),N( -4)),N(   -62500000))
  etst(ash(N(      -1000000),N( -4)),N(      -62500))
  etst(ash(N(         -1000),N( -4)),N(         -63))
  etst(ash(N(          -100),N( -4)),N(          -7))
  etst(ash(N(           -10),N( -4)),N(          -1))
  etst(ash(N(            -1),N( -4)),N(          -1))
  etst(ash(N(             0),N( -4)),N(           0))
  etst(ash(N(             1),N( -4)),N(           0))
  etst(ash(N(            10),N( -4)),N(           0))
  etst(ash(N(           100),N( -4)),N(           6))
  etst(ash(N(          1000),N( -4)),N(          62))
  etst(ash(N(       1000000),N( -4)),N(       62500))
  etst(ash(N(    1000000000),N( -4)),N(    62500000))
  etst(ash(N( 1000000000000),N( -4)),N( 62500000000))

  etst(ash(N(-1000000000000),N( -3)),N(-125000000000))
  etst(ash(N(   -1000000000),N( -3)),N(   -125000000))
  etst(ash(N(      -1000000),N( -3)),N(      -125000))
  etst(ash(N(         -1000),N( -3)),N(         -125))
  etst(ash(N(          -100),N( -3)),N(          -13))
  etst(ash(N(           -10),N( -3)),N(           -2))
  etst(ash(N(            -1),N( -3)),N(           -1))
  etst(ash(N(             0),N( -3)),N(            0))
  etst(ash(N(             1),N( -3)),N(            0))
  etst(ash(N(            10),N( -3)),N(            1))
  etst(ash(N(           100),N( -3)),N(           12))
  etst(ash(N(          1000),N( -3)),N(          125))
  etst(ash(N(       1000000),N( -3)),N(       125000))
  etst(ash(N(    1000000000),N( -3)),N(    125000000))
  etst(ash(N( 1000000000000),N( -3)),N( 125000000000))

  etst(ash(N(-1000000000000),N( -2)),N(-250000000000))
  etst(ash(N(   -1000000000),N( -2)),N(   -250000000))
  etst(ash(N(      -1000000),N( -2)),N(      -250000))
  etst(ash(N(         -1000),N( -2)),N(         -250))
  etst(ash(N(          -100),N( -2)),N(          -25))
  etst(ash(N(           -10),N( -2)),N(           -3))
  etst(ash(N(            -1),N( -2)),N(           -1))
  etst(ash(N(             0),N( -2)),N(            0))
  etst(ash(N(             1),N( -2)),N(            0))
  etst(ash(N(            10),N( -2)),N(            2))
  etst(ash(N(           100),N( -2)),N(           25))
  etst(ash(N(          1000),N( -2)),N(          250))
  etst(ash(N(       1000000),N( -2)),N(       250000))
  etst(ash(N(    1000000000),N( -2)),N(    250000000))
  etst(ash(N( 1000000000000),N( -2)),N( 250000000000))

  etst(ash(N(-1000000000000),N( -1)),N(-500000000000))
  etst(ash(N(   -1000000000),N( -1)),N(   -500000000))
  etst(ash(N(      -1000000),N( -1)),N(      -500000))
  etst(ash(N(         -1000),N( -1)),N(         -500))
  etst(ash(N(          -100),N( -1)),N(          -50))
  etst(ash(N(           -10),N( -1)),N(           -5))
  etst(ash(N(            -1),N( -1)),N(           -1))
  etst(ash(N(             0),N( -1)),N(            0))
  etst(ash(N(             1),N( -1)),N(            0))
  etst(ash(N(            10),N( -1)),N(            5))
  etst(ash(N(           100),N( -1)),N(           50))
  etst(ash(N(          1000),N( -1)),N(          500))
  etst(ash(N(       1000000),N( -1)),N(       500000))
  etst(ash(N(    1000000000),N( -1)),N(    500000000))
  etst(ash(N( 1000000000000),N( -1)),N( 500000000000))

  etst(ash(N(-1000000000000),N(  0)),N(-1000000000000))
  etst(ash(N(   -1000000000),N(  0)),N(   -1000000000))
  etst(ash(N(      -1000000),N(  0)),N(      -1000000))
  etst(ash(N(         -1000),N(  0)),N(         -1000))
  etst(ash(N(          -100),N(  0)),N(          -100))
  etst(ash(N(           -10),N(  0)),N(           -10))
  etst(ash(N(            -1),N(  0)),N(            -1))
  etst(ash(N(             0),N(  0)),N(             0))
  etst(ash(N(             1),N(  0)),N(             1))
  etst(ash(N(            10),N(  0)),N(            10))
  etst(ash(N(           100),N(  0)),N(           100))
  etst(ash(N(          1000),N(  0)),N(          1000))
  etst(ash(N(       1000000),N(  0)),N(       1000000))
  etst(ash(N(    1000000000),N(  0)),N(    1000000000))
  etst(ash(N( 1000000000000),N(  0)),N( 1000000000000))

  etst(ash(N( 1000000000000),N( 65)),N( 36893488147419103232000000000000))
  etst(ash(N(    1000000000),N( 65)),N(    36893488147419103232000000000))
  etst(ash(N(       1000000),N( 65)),N(       36893488147419103232000000))
  etst(ash(N(          1000),N( 65)),N(          36893488147419103232000))
  etst(ash(N(           100),N( 65)),N(           3689348814741910323200))
  etst(ash(N(            10),N( 65)),N(            368934881474191032320))
  etst(ash(N(             1),N( 65)),N(             36893488147419103232))
  etst(ash(N(             0),N( 65)),N(                                0))
  etst(ash(N(            -1),N( 65)),N(            -36893488147419103232))
  etst(ash(N(           -10),N( 65)),N(           -368934881474191032320))
  etst(ash(N(          -100),N( 65)),N(          -3689348814741910323200))
  etst(ash(N(         -1000),N( 65)),N(         -36893488147419103232000))
  etst(ash(N(      -1000000),N( 65)),N(      -36893488147419103232000000))
  etst(ash(N(   -1000000000),N( 65)),N(   -36893488147419103232000000000))
  etst(ash(N(-1000000000000),N( 65)),N(-36893488147419103232000000000000))

  etst(ash(N( 1000000000000),N( 64)),N( 18446744073709551616000000000000))
  etst(ash(N(    1000000000),N( 64)),N(    18446744073709551616000000000))
  etst(ash(N(       1000000),N( 64)),N(       18446744073709551616000000))
  etst(ash(N(          1000),N( 64)),N(          18446744073709551616000))
  etst(ash(N(           100),N( 64)),N(           1844674407370955161600))
  etst(ash(N(            10),N( 64)),N(            184467440737095516160))
  etst(ash(N(             1),N( 64)),N(             18446744073709551616))
  etst(ash(N(             0),N( 64)),N(                                0))
  etst(ash(N(            -1),N( 64)),N(            -18446744073709551616))
  etst(ash(N(           -10),N( 64)),N(           -184467440737095516160))
  etst(ash(N(          -100),N( 64)),N(          -1844674407370955161600))
  etst(ash(N(         -1000),N( 64)),N(         -18446744073709551616000))
  etst(ash(N(      -1000000),N( 64)),N(      -18446744073709551616000000))
  etst(ash(N(   -1000000000),N( 64)),N(   -18446744073709551616000000000))
  etst(ash(N(-1000000000000),N( 64)),N(-18446744073709551616000000000000))

  etst(ash(N( 1000000000000),N( 63)),N( 9223372036854775808000000000000))
  etst(ash(N(    1000000000),N( 63)),N(    9223372036854775808000000000))
  etst(ash(N(       1000000),N( 63)),N(       9223372036854775808000000))
  etst(ash(N(          1000),N( 63)),N(          9223372036854775808000))
  etst(ash(N(           100),N( 63)),N(           922337203685477580800))
  etst(ash(N(            10),N( 63)),N(            92233720368547758080))
  etst(ash(N(             1),N( 63)),N(             9223372036854775808))
  etst(ash(N(             0),N( 63)),N(                               0))
  etst(ash(N(            -1),N( 63)),N(            -9223372036854775808))
  etst(ash(N(           -10),N( 63)),N(           -92233720368547758080))
  etst(ash(N(          -100),N( 63)),N(          -922337203685477580800))
  etst(ash(N(         -1000),N( 63)),N(         -9223372036854775808000))
  etst(ash(N(      -1000000),N( 63)),N(      -9223372036854775808000000))
  etst(ash(N(   -1000000000),N( 63)),N(   -9223372036854775808000000000))
  etst(ash(N(-1000000000000),N( 63)),N(-9223372036854775808000000000000))

  etst(ash(N( 1000000000000),N(  4)),N( 16000000000000))
  etst(ash(N(    1000000000),N(  4)),N(    16000000000))
  etst(ash(N(       1000000),N(  4)),N(       16000000))
  etst(ash(N(          1000),N(  4)),N(          16000))
  etst(ash(N(           100),N(  4)),N(           1600))
  etst(ash(N(            10),N(  4)),N(            160))
  etst(ash(N(             1),N(  4)),N(             16))
  etst(ash(N(             0),N(  4)),N(              0))
  etst(ash(N(            -1),N(  4)),N(            -16))
  etst(ash(N(           -10),N(  4)),N(           -160))
  etst(ash(N(          -100),N(  4)),N(          -1600))
  etst(ash(N(         -1000),N(  4)),N(         -16000))
  etst(ash(N(      -1000000),N(  4)),N(      -16000000))
  etst(ash(N(   -1000000000),N(  4)),N(   -16000000000))
  etst(ash(N(-1000000000000),N(  4)),N(-16000000000000))

  etst(ash(N( 1000000000000),N(  3)),N( 8000000000000))
  etst(ash(N(    1000000000),N(  3)),N(    8000000000))
  etst(ash(N(       1000000),N(  3)),N(       8000000))
  etst(ash(N(          1000),N(  3)),N(          8000))
  etst(ash(N(           100),N(  3)),N(           800))
  etst(ash(N(            10),N(  3)),N(            80))
  etst(ash(N(             1),N(  3)),N(             8))
  etst(ash(N(             0),N(  3)),N(             0))
  etst(ash(N(            -1),N(  3)),N(            -8))
  etst(ash(N(           -10),N(  3)),N(           -80))
  etst(ash(N(          -100),N(  3)),N(          -800))
  etst(ash(N(         -1000),N(  3)),N(         -8000))
  etst(ash(N(      -1000000),N(  3)),N(      -8000000))
  etst(ash(N(   -1000000000),N(  3)),N(   -8000000000))
  etst(ash(N(-1000000000000),N(  3)),N(-8000000000000))

  etst(ash(N( 1000000000000),N(  2)),N( 4000000000000))
  etst(ash(N(    1000000000),N(  2)),N(    4000000000))
  etst(ash(N(       1000000),N(  2)),N(       4000000))
  etst(ash(N(          1000),N(  2)),N(          4000))
  etst(ash(N(           100),N(  2)),N(           400))
  etst(ash(N(            10),N(  2)),N(            40))
  etst(ash(N(             1),N(  2)),N(             4))
  etst(ash(N(             0),N(  2)),N(             0))
  etst(ash(N(            -1),N(  2)),N(            -4))
  etst(ash(N(           -10),N(  2)),N(           -40))
  etst(ash(N(          -100),N(  2)),N(          -400))
  etst(ash(N(         -1000),N(  2)),N(         -4000))
  etst(ash(N(      -1000000),N(  2)),N(      -4000000))
  etst(ash(N(   -1000000000),N(  2)),N(   -4000000000))
  etst(ash(N(-1000000000000),N(  2)),N(-4000000000000))

  etst(ash(N( 1000000000000),N(  1)),N( 2000000000000))
  etst(ash(N(    1000000000),N(  1)),N(    2000000000))
  etst(ash(N(       1000000),N(  1)),N(       2000000))
  etst(ash(N(          1000),N(  1)),N(          2000))
  etst(ash(N(           100),N(  1)),N(           200))
  etst(ash(N(            10),N(  1)),N(            20))
  etst(ash(N(             1),N(  1)),N(             2))
  etst(ash(N(             0),N(  1)),N(             0))
  etst(ash(N(            -1),N(  1)),N(            -2))
  etst(ash(N(           -10),N(  1)),N(           -20))
  etst(ash(N(          -100),N(  1)),N(          -200))
  etst(ash(N(         -1000),N(  1)),N(         -2000))
  etst(ash(N(      -1000000),N(  1)),N(      -2000000))
  etst(ash(N(   -1000000000),N(  1)),N(   -2000000000))
  etst(ash(N(-1000000000000),N(  1)),N(-2000000000000))

  etst(ash(N( 36893488147419103232000000000000),N(-65)),N( 1000000000000))
  etst(ash(N(    36893488147419103232000000000),N(-65)),N(    1000000000))
  etst(ash(N(       36893488147419103232000000),N(-65)),N(       1000000))
  etst(ash(N(          36893488147419103232000),N(-65)),N(          1000))
  etst(ash(N(           3689348814741910323200),N(-65)),N(           100))
  etst(ash(N(            368934881474191032320),N(-65)),N(            10))
  etst(ash(N(             36893488147419103232),N(-65)),N(             1))
  etst(ash(N(                                0),N(-65)),N(             0))
  etst(ash(N(            -36893488147419103232),N(-65)),N(            -1))
  etst(ash(N(           -368934881474191032320),N(-65)),N(           -10))
  etst(ash(N(          -3689348814741910323200),N(-65)),N(          -100))
  etst(ash(N(         -36893488147419103232000),N(-65)),N(         -1000))
  etst(ash(N(      -36893488147419103232000000),N(-65)),N(      -1000000))
  etst(ash(N(   -36893488147419103232000000000),N(-65)),N(   -1000000000))
  etst(ash(N(-36893488147419103232000000000000),N(-65)),N(-1000000000000))

  etst(ash(N( 18446744073709551616000000000000),N(-64)),N( 1000000000000))
  etst(ash(N(    18446744073709551616000000000),N(-64)),N(    1000000000))
  etst(ash(N(       18446744073709551616000000),N(-64)),N(       1000000))
  etst(ash(N(          18446744073709551616000),N(-64)),N(          1000))
  etst(ash(N(           1844674407370955161600),N(-64)),N(           100))
  etst(ash(N(            184467440737095516160),N(-64)),N(            10))
  etst(ash(N(             18446744073709551616),N(-64)),N(             1))
  etst(ash(N(                                0),N(-64)),N(             0))
  etst(ash(N(            -18446744073709551616),N(-64)),N(            -1))
  etst(ash(N(           -184467440737095516160),N(-64)),N(           -10))
  etst(ash(N(          -1844674407370955161600),N(-64)),N(          -100))
  etst(ash(N(         -18446744073709551616000),N(-64)),N(         -1000))
  etst(ash(N(      -18446744073709551616000000),N(-64)),N(      -1000000))
  etst(ash(N(   -18446744073709551616000000000),N(-64)),N(   -1000000000))
  etst(ash(N(-18446744073709551616000000000000),N(-64)),N(-1000000000000))

  etst(ash(N(  9223372036854775808000000000000),N(-63)),N( 1000000000000))
  etst(ash(N(     9223372036854775808000000000),N(-63)),N(    1000000000))
  etst(ash(N(        9223372036854775808000000),N(-63)),N(       1000000))
  etst(ash(N(           9223372036854775808000),N(-63)),N(          1000))
  etst(ash(N(            922337203685477580800),N(-63)),N(           100))
  etst(ash(N(             92233720368547758080),N(-63)),N(            10))
  etst(ash(N(              9223372036854775808),N(-63)),N(             1))
  etst(ash(N(                                0),N(-63)),N(             0))
  etst(ash(N(             -9223372036854775808),N(-63)),N(            -1))
  etst(ash(N(            -92233720368547758080),N(-63)),N(           -10))
  etst(ash(N(           -922337203685477580800),N(-63)),N(          -100))
  etst(ash(N(          -9223372036854775808000),N(-63)),N(         -1000))
  etst(ash(N(       -9223372036854775808000000),N(-63)),N(      -1000000))
  etst(ash(N(    -9223372036854775808000000000),N(-63)),N(   -1000000000))
  etst(ash(N( -9223372036854775808000000000000),N(-63)),N(-1000000000000))

  etst(twice0(N(-2)),N(-4))
  etst(twice0(N(-1)),N(-2))
  etst(twice0(N( 0)),N( 0))
  etst(twice0(N( 1)),N( 2))
  etst(twice0(N( 2)),N( 4))

  etst(twice1(N(-2)),N(-3))
  etst(twice1(N(-1)),N(-1))
  etst(twice1(N( 0)),N( 1))
  etst(twice1(N( 1)),N( 3))
  etst(twice1(N( 2)),N( 5))

  etst(pairUnaryMinus(PAIR(N(-1),N(-1))),PAIR(N( 1),N( 1)))
  etst(pairUnaryMinus(PAIR(N(-1),N( 0))),PAIR(N( 1),N( 0)))
  etst(pairUnaryMinus(PAIR(N(-1),N( 1))),PAIR(N( 1),N(-1)))
  etst(pairUnaryMinus(PAIR(N( 0),N(-1))),PAIR(N( 0),N( 1)))
  etst(pairUnaryMinus(PAIR(N( 0),N( 0))),PAIR(N( 0),N( 0)))
  etst(pairUnaryMinus(PAIR(N( 0),N( 1))),PAIR(N( 0),N(-1)))
  etst(pairUnaryMinus(PAIR(N( 1),N(-1))),PAIR(N(-1),N( 1)))
  etst(pairUnaryMinus(PAIR(N( 1),N( 0))),PAIR(N(-1),N( 0)))
  etst(pairUnaryMinus(PAIR(N( 1),N( 1))),PAIR(N(-1),N(-1)))

  etst(divAdjust(PAIR(N(3),N(7)),N(10)),PAIR(N(4),N(-3)))

  etst(floor0(N( 0),N(1)),PAIR(N( 0),N(0)))
  etst(floor0(N( 1),N(1)),PAIR(N( 1),N(0)))
  etst(floor0(N( 2),N(1)),PAIR(N( 2),N(0)))
  etst(floor0(N( 3),N(1)),PAIR(N( 3),N(0)))
  etst(floor0(N( 4),N(1)),PAIR(N( 4),N(0)))
  etst(floor0(N( 5),N(1)),PAIR(N( 5),N(0)))
  etst(floor0(N( 6),N(1)),PAIR(N( 6),N(0)))
  etst(floor0(N( 7),N(1)),PAIR(N( 7),N(0)))
  etst(floor0(N( 8),N(1)),PAIR(N( 8),N(0)))
  etst(floor0(N( 9),N(1)),PAIR(N( 9),N(0)))

  etst(floor0(N( 0),N(2)),PAIR(N( 0),N(0)))
  etst(floor0(N( 1),N(2)),PAIR(N( 0),N(1)))
  etst(floor0(N( 2),N(2)),PAIR(N( 1),N(0)))
  etst(floor0(N( 3),N(2)),PAIR(N( 1),N(1)))
  etst(floor0(N( 4),N(2)),PAIR(N( 2),N(0)))
  etst(floor0(N( 5),N(2)),PAIR(N( 2),N(1)))
  etst(floor0(N( 6),N(2)),PAIR(N( 3),N(0)))
  etst(floor0(N( 7),N(2)),PAIR(N( 3),N(1)))
  etst(floor0(N( 8),N(2)),PAIR(N( 4),N(0)))
  etst(floor0(N( 9),N(2)),PAIR(N( 4),N(1)))

  etst(floor0(N( 0),N(3)),PAIR(N( 0),N(0)))
  etst(floor0(N( 1),N(3)),PAIR(N( 0),N(1)))
  etst(floor0(N( 2),N(3)),PAIR(N( 0),N(2)))
  etst(floor0(N( 3),N(3)),PAIR(N( 1),N(0)))
  etst(floor0(N( 4),N(3)),PAIR(N( 1),N(1)))
  etst(floor0(N( 5),N(3)),PAIR(N( 1),N(2)))
  etst(floor0(N( 6),N(3)),PAIR(N( 2),N(0)))
  etst(floor0(N( 7),N(3)),PAIR(N( 2),N(1)))
  etst(floor0(N( 8),N(3)),PAIR(N( 2),N(2)))
  etst(floor0(N( 9),N(3)),PAIR(N( 3),N(0)))

  etst(floor0(N( 0),N(4)),PAIR(N( 0),N(0)))
  etst(floor0(N( 1),N(4)),PAIR(N( 0),N(1)))
  etst(floor0(N( 2),N(4)),PAIR(N( 0),N(2)))
  etst(floor0(N( 3),N(4)),PAIR(N( 0),N(3)))
  etst(floor0(N( 4),N(4)),PAIR(N( 1),N(0)))
  etst(floor0(N( 5),N(4)),PAIR(N( 1),N(1)))
  etst(floor0(N( 6),N(4)),PAIR(N( 1),N(2)))
  etst(floor0(N( 7),N(4)),PAIR(N( 1),N(3)))
  etst(floor0(N( 8),N(4)),PAIR(N( 2),N(0)))
  etst(floor0(N( 9),N(4)),PAIR(N( 2),N(1)))

  etst(floor0(N( 0),N(5)),PAIR(N( 0),N(0)))
  etst(floor0(N( 1),N(5)),PAIR(N( 0),N(1)))
  etst(floor0(N( 2),N(5)),PAIR(N( 0),N(2)))
  etst(floor0(N( 3),N(5)),PAIR(N( 0),N(3)))
  etst(floor0(N( 4),N(5)),PAIR(N( 0),N(4)))
  etst(floor0(N( 5),N(5)),PAIR(N( 1),N(0)))
  etst(floor0(N( 6),N(5)),PAIR(N( 1),N(1)))
  etst(floor0(N( 7),N(5)),PAIR(N( 1),N(2)))
  etst(floor0(N( 8),N(5)),PAIR(N( 1),N(3)))
  etst(floor0(N( 9),N(5)),PAIR(N( 1),N(4)))

  etst(floor0(N( 0),N(6)),PAIR(N( 0),N(0)))
  etst(floor0(N( 1),N(6)),PAIR(N( 0),N(1)))
  etst(floor0(N( 2),N(6)),PAIR(N( 0),N(2)))
  etst(floor0(N( 3),N(6)),PAIR(N( 0),N(3)))
  etst(floor0(N( 4),N(6)),PAIR(N( 0),N(4)))
  etst(floor0(N( 5),N(6)),PAIR(N( 0),N(5)))
  etst(floor0(N( 6),N(6)),PAIR(N( 1),N(0)))
  etst(floor0(N( 7),N(6)),PAIR(N( 1),N(1)))
  etst(floor0(N( 8),N(6)),PAIR(N( 1),N(2)))
  etst(floor0(N( 9),N(6)),PAIR(N( 1),N(3)))

  etst(floor0(N( 0),N(7)),PAIR(N( 0),N(0)))
  etst(floor0(N( 1),N(7)),PAIR(N( 0),N(1)))
  etst(floor0(N( 2),N(7)),PAIR(N( 0),N(2)))
  etst(floor0(N( 3),N(7)),PAIR(N( 0),N(3)))
  etst(floor0(N( 4),N(7)),PAIR(N( 0),N(4)))
  etst(floor0(N( 5),N(7)),PAIR(N( 0),N(5)))
  etst(floor0(N( 6),N(7)),PAIR(N( 0),N(6)))
  etst(floor0(N( 7),N(7)),PAIR(N( 1),N(0)))
  etst(floor0(N( 8),N(7)),PAIR(N( 1),N(1)))
  etst(floor0(N( 9),N(7)),PAIR(N( 1),N(2)))

  etst(floor0(N( 0),N(8)),PAIR(N( 0),N(0)))
  etst(floor0(N( 1),N(8)),PAIR(N( 0),N(1)))
  etst(floor0(N( 2),N(8)),PAIR(N( 0),N(2)))
  etst(floor0(N( 3),N(8)),PAIR(N( 0),N(3)))
  etst(floor0(N( 4),N(8)),PAIR(N( 0),N(4)))
  etst(floor0(N( 5),N(8)),PAIR(N( 0),N(5)))
  etst(floor0(N( 6),N(8)),PAIR(N( 0),N(6)))
  etst(floor0(N( 7),N(8)),PAIR(N( 0),N(7)))
  etst(floor0(N( 8),N(8)),PAIR(N( 1),N(0)))
  etst(floor0(N( 9),N(8)),PAIR(N( 1),N(1)))

  etst(floor0(N( 0),N(9)),PAIR(N( 0),N(0)))
  etst(floor0(N( 1),N(9)),PAIR(N( 0),N(1)))
  etst(floor0(N( 2),N(9)),PAIR(N( 0),N(2)))
  etst(floor0(N( 3),N(9)),PAIR(N( 0),N(3)))
  etst(floor0(N( 4),N(9)),PAIR(N( 0),N(4)))
  etst(floor0(N( 5),N(9)),PAIR(N( 0),N(5)))
  etst(floor0(N( 6),N(9)),PAIR(N( 0),N(6)))
  etst(floor0(N( 7),N(9)),PAIR(N( 0),N(7)))
  etst(floor0(N( 8),N(9)),PAIR(N( 0),N(8)))
  etst(floor0(N( 9),N(9)),PAIR(N( 1),N(0)))

  etst(ceiling0(N( 9),N(1)),PAIR(N( 9),N( 0)))
  etst(ceiling0(N( 8),N(1)),PAIR(N( 8),N( 0)))
  etst(ceiling0(N( 7),N(1)),PAIR(N( 7),N( 0)))
  etst(ceiling0(N( 6),N(1)),PAIR(N( 6),N( 0)))
  etst(ceiling0(N( 5),N(1)),PAIR(N( 5),N( 0)))
  etst(ceiling0(N( 4),N(1)),PAIR(N( 4),N( 0)))
  etst(ceiling0(N( 3),N(1)),PAIR(N( 3),N( 0)))
  etst(ceiling0(N( 2),N(1)),PAIR(N( 2),N( 0)))
  etst(ceiling0(N( 1),N(1)),PAIR(N( 1),N( 0)))
  etst(ceiling0(N( 0),N(1)),PAIR(N( 0),N( 0)))

  etst(ceiling0(N( 9),N(2)),PAIR(N( 5),N(-1)))
  etst(ceiling0(N( 8),N(2)),PAIR(N( 4),N( 0)))
  etst(ceiling0(N( 7),N(2)),PAIR(N( 4),N(-1)))
  etst(ceiling0(N( 6),N(2)),PAIR(N( 3),N( 0)))
  etst(ceiling0(N( 5),N(2)),PAIR(N( 3),N(-1)))
  etst(ceiling0(N( 4),N(2)),PAIR(N( 2),N( 0)))
  etst(ceiling0(N( 3),N(2)),PAIR(N( 2),N(-1)))
  etst(ceiling0(N( 2),N(2)),PAIR(N( 1),N( 0)))
  etst(ceiling0(N( 1),N(2)),PAIR(N( 1),N(-1)))
  etst(ceiling0(N( 0),N(2)),PAIR(N( 0),N( 0)))

  etst(ceiling0(N( 9),N(3)),PAIR(N( 3),N( 0)))
  etst(ceiling0(N( 8),N(3)),PAIR(N( 3),N(-1)))
  etst(ceiling0(N( 7),N(3)),PAIR(N( 3),N(-2)))
  etst(ceiling0(N( 6),N(3)),PAIR(N( 2),N( 0)))
  etst(ceiling0(N( 5),N(3)),PAIR(N( 2),N(-1)))
  etst(ceiling0(N( 4),N(3)),PAIR(N( 2),N(-2)))
  etst(ceiling0(N( 3),N(3)),PAIR(N( 1),N( 0)))
  etst(ceiling0(N( 2),N(3)),PAIR(N( 1),N(-1)))
  etst(ceiling0(N( 1),N(3)),PAIR(N( 1),N(-2)))
  etst(ceiling0(N( 0),N(3)),PAIR(N( 0),N( 0)))

  etst(ceiling0(N( 9),N(4)),PAIR(N( 3),N(-3)))
  etst(ceiling0(N( 8),N(4)),PAIR(N( 2),N( 0)))
  etst(ceiling0(N( 7),N(4)),PAIR(N( 2),N(-1)))
  etst(ceiling0(N( 6),N(4)),PAIR(N( 2),N(-2)))
  etst(ceiling0(N( 5),N(4)),PAIR(N( 2),N(-3)))
  etst(ceiling0(N( 4),N(4)),PAIR(N( 1),N( 0)))
  etst(ceiling0(N( 3),N(4)),PAIR(N( 1),N(-1)))
  etst(ceiling0(N( 2),N(4)),PAIR(N( 1),N(-2)))
  etst(ceiling0(N( 1),N(4)),PAIR(N( 1),N(-3)))
  etst(ceiling0(N( 0),N(4)),PAIR(N( 0),N( 0)))

  etst(ceiling0(N( 9),N(5)),PAIR(N( 2),N(-1)))
  etst(ceiling0(N( 8),N(5)),PAIR(N( 2),N(-2)))
  etst(ceiling0(N( 7),N(5)),PAIR(N( 2),N(-3)))
  etst(ceiling0(N( 6),N(5)),PAIR(N( 2),N(-4)))
  etst(ceiling0(N( 5),N(5)),PAIR(N( 1),N( 0)))
  etst(ceiling0(N( 4),N(5)),PAIR(N( 1),N(-1)))
  etst(ceiling0(N( 3),N(5)),PAIR(N( 1),N(-2)))
  etst(ceiling0(N( 2),N(5)),PAIR(N( 1),N(-3)))
  etst(ceiling0(N( 1),N(5)),PAIR(N( 1),N(-4)))
  etst(ceiling0(N( 0),N(5)),PAIR(N( 0),N( 0)))

  etst(ceiling0(N( 9),N(6)),PAIR(N( 2),N(-3)))
  etst(ceiling0(N( 8),N(6)),PAIR(N( 2),N(-4)))
  etst(ceiling0(N( 7),N(6)),PAIR(N( 2),N(-5)))
  etst(ceiling0(N( 6),N(6)),PAIR(N( 1),N( 0)))
  etst(ceiling0(N( 5),N(6)),PAIR(N( 1),N(-1)))
  etst(ceiling0(N( 4),N(6)),PAIR(N( 1),N(-2)))
  etst(ceiling0(N( 3),N(6)),PAIR(N( 1),N(-3)))
  etst(ceiling0(N( 2),N(6)),PAIR(N( 1),N(-4)))
  etst(ceiling0(N( 1),N(6)),PAIR(N( 1),N(-5)))
  etst(ceiling0(N( 0),N(6)),PAIR(N( 0),N( 0)))

  etst(ceiling0(N( 9),N(7)),PAIR(N( 2),N(-5)))
  etst(ceiling0(N( 8),N(7)),PAIR(N( 2),N(-6)))
  etst(ceiling0(N( 7),N(7)),PAIR(N( 1),N( 0)))
  etst(ceiling0(N( 6),N(7)),PAIR(N( 1),N(-1)))
  etst(ceiling0(N( 5),N(7)),PAIR(N( 1),N(-2)))
  etst(ceiling0(N( 4),N(7)),PAIR(N( 1),N(-3)))
  etst(ceiling0(N( 3),N(7)),PAIR(N( 1),N(-4)))
  etst(ceiling0(N( 2),N(7)),PAIR(N( 1),N(-5)))
  etst(ceiling0(N( 1),N(7)),PAIR(N( 1),N(-6)))
  etst(ceiling0(N( 0),N(7)),PAIR(N( 0),N( 0)))

  etst(ceiling0(N( 9),N(8)),PAIR(N( 2),N(-7)))
  etst(ceiling0(N( 8),N(8)),PAIR(N( 1),N( 0)))
  etst(ceiling0(N( 7),N(8)),PAIR(N( 1),N(-1)))
  etst(ceiling0(N( 6),N(8)),PAIR(N( 1),N(-2)))
  etst(ceiling0(N( 5),N(8)),PAIR(N( 1),N(-3)))
  etst(ceiling0(N( 4),N(8)),PAIR(N( 1),N(-4)))
  etst(ceiling0(N( 3),N(8)),PAIR(N( 1),N(-5)))
  etst(ceiling0(N( 2),N(8)),PAIR(N( 1),N(-6)))
  etst(ceiling0(N( 1),N(8)),PAIR(N( 1),N(-7)))
  etst(ceiling0(N( 0),N(8)),PAIR(N( 0),N( 0)))

  etst(ceiling0(N( 9),N(9)),PAIR(N( 1),N( 0)))
  etst(ceiling0(N( 8),N(9)),PAIR(N( 1),N(-1)))
  etst(ceiling0(N( 7),N(9)),PAIR(N( 1),N(-2)))
  etst(ceiling0(N( 6),N(9)),PAIR(N( 1),N(-3)))
  etst(ceiling0(N( 5),N(9)),PAIR(N( 1),N(-4)))
  etst(ceiling0(N( 4),N(9)),PAIR(N( 1),N(-5)))
  etst(ceiling0(N( 3),N(9)),PAIR(N( 1),N(-6)))
  etst(ceiling0(N( 2),N(9)),PAIR(N( 1),N(-7)))
  etst(ceiling0(N( 1),N(9)),PAIR(N( 1),N(-8)))
  etst(ceiling0(N( 0),N(9)),PAIR(N( 0),N( 0)))

  etst(round0(N( 9),N(1)),PAIR(N( 9),N( 0)))
  etst(round0(N( 8),N(1)),PAIR(N( 8),N( 0)))
  etst(round0(N( 7),N(1)),PAIR(N( 7),N( 0)))
  etst(round0(N( 6),N(1)),PAIR(N( 6),N( 0)))
  etst(round0(N( 5),N(1)),PAIR(N( 5),N( 0)))
  etst(round0(N( 4),N(1)),PAIR(N( 4),N( 0)))
  etst(round0(N( 3),N(1)),PAIR(N( 3),N( 0)))
  etst(round0(N( 2),N(1)),PAIR(N( 2),N( 0)))
  etst(round0(N( 1),N(1)),PAIR(N( 1),N( 0)))
  etst(round0(N( 0),N(1)),PAIR(N( 0),N( 0)))

  etst(round0(N( 9),N(2)),PAIR(N( 4),N( 1)))
  etst(round0(N( 8),N(2)),PAIR(N( 4),N( 0)))
  etst(round0(N( 7),N(2)),PAIR(N( 4),N(-1)))
  etst(round0(N( 6),N(2)),PAIR(N( 3),N( 0)))
  etst(round0(N( 5),N(2)),PAIR(N( 2),N( 1)))
  etst(round0(N( 4),N(2)),PAIR(N( 2),N( 0)))
  etst(round0(N( 3),N(2)),PAIR(N( 2),N(-1)))
  etst(round0(N( 2),N(2)),PAIR(N( 1),N( 0)))
  etst(round0(N( 1),N(2)),PAIR(N( 0),N( 1)))
  etst(round0(N( 0),N(2)),PAIR(N( 0),N( 0)))

  etst(round0(N( 9),N(3)),PAIR(N( 3),N( 0)))
  etst(round0(N( 8),N(3)),PAIR(N( 3),N(-1)))
  etst(round0(N( 7),N(3)),PAIR(N( 2),N( 1)))
  etst(round0(N( 6),N(3)),PAIR(N( 2),N( 0)))
  etst(round0(N( 5),N(3)),PAIR(N( 2),N(-1)))
  etst(round0(N( 4),N(3)),PAIR(N( 1),N( 1)))
  etst(round0(N( 3),N(3)),PAIR(N( 1),N( 0)))
  etst(round0(N( 2),N(3)),PAIR(N( 1),N(-1)))
  etst(round0(N( 1),N(3)),PAIR(N( 0),N( 1)))
  etst(round0(N( 0),N(3)),PAIR(N( 0),N( 0)))

  etst(round0(N( 9),N(4)),PAIR(N( 2),N( 1)))
  etst(round0(N( 8),N(4)),PAIR(N( 2),N( 0)))
  etst(round0(N( 7),N(4)),PAIR(N( 2),N(-1)))
  etst(round0(N( 6),N(4)),PAIR(N( 2),N(-2)))
  etst(round0(N( 5),N(4)),PAIR(N( 1),N( 1)))
  etst(round0(N( 4),N(4)),PAIR(N( 1),N( 0)))
  etst(round0(N( 3),N(4)),PAIR(N( 1),N(-1)))
  etst(round0(N( 2),N(4)),PAIR(N( 0),N( 2)))
  etst(round0(N( 1),N(4)),PAIR(N( 0),N( 1)))
  etst(round0(N( 0),N(4)),PAIR(N( 0),N( 0)))

  etst(round0(N( 9),N(5)),PAIR(N( 2),N(-1)))
  etst(round0(N( 8),N(5)),PAIR(N( 2),N(-2)))
  etst(round0(N( 7),N(5)),PAIR(N( 1),N( 2)))
  etst(round0(N( 6),N(5)),PAIR(N( 1),N( 1)))
  etst(round0(N( 5),N(5)),PAIR(N( 1),N( 0)))
  etst(round0(N( 4),N(5)),PAIR(N( 1),N(-1)))
  etst(round0(N( 3),N(5)),PAIR(N( 1),N(-2)))
  etst(round0(N( 2),N(5)),PAIR(N( 0),N( 2)))
  etst(round0(N( 1),N(5)),PAIR(N( 0),N( 1)))
  etst(round0(N( 0),N(5)),PAIR(N( 0),N( 0)))

  etst(round0(N( 9),N(6)),PAIR(N( 2),N(-3)))
  etst(round0(N( 8),N(6)),PAIR(N( 1),N( 2)))
  etst(round0(N( 7),N(6)),PAIR(N( 1),N( 1)))
  etst(round0(N( 6),N(6)),PAIR(N( 1),N( 0)))
  etst(round0(N( 5),N(6)),PAIR(N( 1),N(-1)))
  etst(round0(N( 4),N(6)),PAIR(N( 1),N(-2)))
  etst(round0(N( 3),N(6)),PAIR(N( 0),N( 3)))
  etst(round0(N( 2),N(6)),PAIR(N( 0),N( 2)))
  etst(round0(N( 1),N(6)),PAIR(N( 0),N( 1)))
  etst(round0(N( 0),N(6)),PAIR(N( 0),N( 0)))

  etst(round0(N( 9),N(7)),PAIR(N( 1),N( 2)))
  etst(round0(N( 8),N(7)),PAIR(N( 1),N( 1)))
  etst(round0(N( 7),N(7)),PAIR(N( 1),N( 0)))
  etst(round0(N( 6),N(7)),PAIR(N( 1),N(-1)))
  etst(round0(N( 5),N(7)),PAIR(N( 1),N(-2)))
  etst(round0(N( 4),N(7)),PAIR(N( 1),N(-3)))
  etst(round0(N( 3),N(7)),PAIR(N( 0),N(3 )))
  etst(round0(N( 2),N(7)),PAIR(N( 0),N( 2)))
  etst(round0(N( 1),N(7)),PAIR(N( 0),N( 1)))
  etst(round0(N( 0),N(7)),PAIR(N( 0),N( 0)))

  etst(round0(N( 9),N(8)),PAIR(N( 1),N( 1)))
  etst(round0(N( 8),N(8)),PAIR(N( 1),N( 0)))
  etst(round0(N( 7),N(8)),PAIR(N( 1),N(-1)))
  etst(round0(N( 6),N(8)),PAIR(N( 1),N(-2)))
  etst(round0(N( 5),N(8)),PAIR(N( 1),N(-3)))
  etst(round0(N( 4),N(8)),PAIR(N( 0),N( 4)))
  etst(round0(N( 3),N(8)),PAIR(N( 0),N( 3)))
  etst(round0(N( 2),N(8)),PAIR(N( 0),N( 2)))
  etst(round0(N( 1),N(8)),PAIR(N( 0),N( 1)))
  etst(round0(N( 0),N(8)),PAIR(N( 0),N( 0)))

  etst(round0(N( 9),N(9)),PAIR(N( 1),N( 0)))
  etst(round0(N( 8),N(9)),PAIR(N( 1),N(-1)))
  etst(round0(N( 7),N(9)),PAIR(N( 1),N(-2)))
  etst(round0(N( 6),N(9)),PAIR(N( 1),N(-3)))
  etst(round0(N( 5),N(9)),PAIR(N( 1),N(-4)))
  etst(round0(N( 4),N(9)),PAIR(N( 0),N( 4)))
  etst(round0(N( 3),N(9)),PAIR(N( 0),N( 3)))
  etst(round0(N( 2),N(9)),PAIR(N( 0),N( 2)))
  etst(round0(N( 1),N(9)),PAIR(N( 0),N( 1)))
  etst(round0(N( 0),N(9)),PAIR(N( 0),N( 0)))

  xtst(floorx(TT,TT))
  xtst(floorx(TT,FF))
  xtst(floorx(TT,I0))
  xtst(floorx(TT,P0))
  xtst(floorx(TT,M0))
  xtst(floorx(TT,B0))
  xtst(floorx(FF,TT))
  xtst(floorx(FF,FF))
  xtst(floorx(FF,I0))
  xtst(floorx(FF,P0))
  xtst(floorx(FF,M0))
  xtst(floorx(FF,B0))
  xtst(floorx(I0,TT))
  xtst(floorx(I0,FF))
  xtst(floorx(I0,P0))
  xtst(floorx(I0,M0))
  xtst(floorx(I0,B0))
  xtst(floorx(P0,TT))
  xtst(floorx(P0,FF))
  xtst(floorx(P0,I0))
  xtst(floorx(P0,P0))
  xtst(floorx(P0,M0))
  xtst(floorx(P0,B0))
  xtst(floorx(M0,TT))
  xtst(floorx(M0,FF))
  xtst(floorx(M0,I0))
  xtst(floorx(M0,P0))
  xtst(floorx(M0,M0))
  xtst(floorx(M0,B0))
  xtst(floorx(B0,TT))
  xtst(floorx(B0,FF))
  xtst(floorx(B0,I0))
  xtst(floorx(B0,P0))
  xtst(floorx(B0,M0))
  xtst(floorx(B0,B0))
  xtst(floorx(I0,Z))
  xtst(floorx(I0,M))

  etst(floorx(N(-9),N(1)),PAIR(N(-9),N(0)))
  etst(floorx(N(-8),N(1)),PAIR(N(-8),N(0)))
  etst(floorx(N(-7),N(1)),PAIR(N(-7),N(0)))
  etst(floorx(N(-6),N(1)),PAIR(N(-6),N(0)))
  etst(floorx(N(-5),N(1)),PAIR(N(-5),N(0)))
  etst(floorx(N(-4),N(1)),PAIR(N(-4),N(0)))
  etst(floorx(N(-3),N(1)),PAIR(N(-3),N(0)))
  etst(floorx(N(-2),N(1)),PAIR(N(-2),N(0)))
  etst(floorx(N(-1),N(1)),PAIR(N(-1),N(0)))
  etst(floorx(N( 0),N(1)),PAIR(N( 0),N(0)))
  etst(floorx(N( 1),N(1)),PAIR(N( 1),N(0)))
  etst(floorx(N( 2),N(1)),PAIR(N( 2),N(0)))
  etst(floorx(N( 3),N(1)),PAIR(N( 3),N(0)))
  etst(floorx(N( 4),N(1)),PAIR(N( 4),N(0)))
  etst(floorx(N( 5),N(1)),PAIR(N( 5),N(0)))
  etst(floorx(N( 6),N(1)),PAIR(N( 6),N(0)))
  etst(floorx(N( 7),N(1)),PAIR(N( 7),N(0)))
  etst(floorx(N( 8),N(1)),PAIR(N( 8),N(0)))
  etst(floorx(N( 9),N(1)),PAIR(N( 9),N(0)))

  etst(floorx(N(-9),N(2)),PAIR(N(-5),N(1)))
  etst(floorx(N(-8),N(2)),PAIR(N(-4),N(0)))
  etst(floorx(N(-7),N(2)),PAIR(N(-4),N(1)))
  etst(floorx(N(-6),N(2)),PAIR(N(-3),N(0)))
  etst(floorx(N(-5),N(2)),PAIR(N(-3),N(1)))
  etst(floorx(N(-4),N(2)),PAIR(N(-2),N(0)))
  etst(floorx(N(-3),N(2)),PAIR(N(-2),N(1)))
  etst(floorx(N(-2),N(2)),PAIR(N(-1),N(0)))
  etst(floorx(N(-1),N(2)),PAIR(N(-1),N(1)))
  etst(floorx(N( 0),N(2)),PAIR(N( 0),N(0)))
  etst(floorx(N( 1),N(2)),PAIR(N( 0),N(1)))
  etst(floorx(N( 2),N(2)),PAIR(N( 1),N(0)))
  etst(floorx(N( 3),N(2)),PAIR(N( 1),N(1)))
  etst(floorx(N( 4),N(2)),PAIR(N( 2),N(0)))
  etst(floorx(N( 5),N(2)),PAIR(N( 2),N(1)))
  etst(floorx(N( 6),N(2)),PAIR(N( 3),N(0)))
  etst(floorx(N( 7),N(2)),PAIR(N( 3),N(1)))
  etst(floorx(N( 8),N(2)),PAIR(N( 4),N(0)))
  etst(floorx(N( 9),N(2)),PAIR(N( 4),N(1)))

  etst(floorx(N(-9),N(3)),PAIR(N(-3),N(0)))
  etst(floorx(N(-8),N(3)),PAIR(N(-3),N(1)))
  etst(floorx(N(-7),N(3)),PAIR(N(-3),N(2)))
  etst(floorx(N(-6),N(3)),PAIR(N(-2),N(0)))
  etst(floorx(N(-5),N(3)),PAIR(N(-2),N(1)))
  etst(floorx(N(-4),N(3)),PAIR(N(-2),N(2)))
  etst(floorx(N(-3),N(3)),PAIR(N(-1),N(0)))
  etst(floorx(N(-2),N(3)),PAIR(N(-1),N(1)))
  etst(floorx(N(-1),N(3)),PAIR(N(-1),N(2)))
  etst(floorx(N( 0),N(3)),PAIR(N( 0),N(0)))
  etst(floorx(N( 1),N(3)),PAIR(N( 0),N(1)))
  etst(floorx(N( 2),N(3)),PAIR(N( 0),N(2)))
  etst(floorx(N( 3),N(3)),PAIR(N( 1),N(0)))
  etst(floorx(N( 4),N(3)),PAIR(N( 1),N(1)))
  etst(floorx(N( 5),N(3)),PAIR(N( 1),N(2)))
  etst(floorx(N( 6),N(3)),PAIR(N( 2),N(0)))
  etst(floorx(N( 7),N(3)),PAIR(N( 2),N(1)))
  etst(floorx(N( 8),N(3)),PAIR(N( 2),N(2)))
  etst(floorx(N( 9),N(3)),PAIR(N( 3),N(0)))

  etst(floorx(N(-9),N(4)),PAIR(N(-3),N(3)))
  etst(floorx(N(-8),N(4)),PAIR(N(-2),N(0)))
  etst(floorx(N(-7),N(4)),PAIR(N(-2),N(1)))
  etst(floorx(N(-6),N(4)),PAIR(N(-2),N(2)))
  etst(floorx(N(-5),N(4)),PAIR(N(-2),N(3)))
  etst(floorx(N(-4),N(4)),PAIR(N(-1),N(0)))
  etst(floorx(N(-3),N(4)),PAIR(N(-1),N(1)))
  etst(floorx(N(-2),N(4)),PAIR(N(-1),N(2)))
  etst(floorx(N(-1),N(4)),PAIR(N(-1),N(3)))
  etst(floorx(N( 0),N(4)),PAIR(N( 0),N(0)))
  etst(floorx(N( 1),N(4)),PAIR(N( 0),N(1)))
  etst(floorx(N( 2),N(4)),PAIR(N( 0),N(2)))
  etst(floorx(N( 3),N(4)),PAIR(N( 0),N(3)))
  etst(floorx(N( 4),N(4)),PAIR(N( 1),N(0)))
  etst(floorx(N( 5),N(4)),PAIR(N( 1),N(1)))
  etst(floorx(N( 6),N(4)),PAIR(N( 1),N(2)))
  etst(floorx(N( 7),N(4)),PAIR(N( 1),N(3)))
  etst(floorx(N( 8),N(4)),PAIR(N( 2),N(0)))
  etst(floorx(N( 9),N(4)),PAIR(N( 2),N(1)))

  etst(floorx(N(-9),N(5)),PAIR(N(-2),N(1)))
  etst(floorx(N(-8),N(5)),PAIR(N(-2),N(2)))
  etst(floorx(N(-7),N(5)),PAIR(N(-2),N(3)))
  etst(floorx(N(-6),N(5)),PAIR(N(-2),N(4)))
  etst(floorx(N(-5),N(5)),PAIR(N(-1),N(0)))
  etst(floorx(N(-4),N(5)),PAIR(N(-1),N(1)))
  etst(floorx(N(-3),N(5)),PAIR(N(-1),N(2)))
  etst(floorx(N(-2),N(5)),PAIR(N(-1),N(3)))
  etst(floorx(N(-1),N(5)),PAIR(N(-1),N(4)))
  etst(floorx(N( 0),N(5)),PAIR(N( 0),N(0)))
  etst(floorx(N( 1),N(5)),PAIR(N( 0),N(1)))
  etst(floorx(N( 2),N(5)),PAIR(N( 0),N(2)))
  etst(floorx(N( 3),N(5)),PAIR(N( 0),N(3)))
  etst(floorx(N( 4),N(5)),PAIR(N( 0),N(4)))
  etst(floorx(N( 5),N(5)),PAIR(N( 1),N(0)))
  etst(floorx(N( 6),N(5)),PAIR(N( 1),N(1)))
  etst(floorx(N( 7),N(5)),PAIR(N( 1),N(2)))
  etst(floorx(N( 8),N(5)),PAIR(N( 1),N(3)))
  etst(floorx(N( 9),N(5)),PAIR(N( 1),N(4)))

  etst(floorx(N(-9),N(6)),PAIR(N(-2),N(3)))
  etst(floorx(N(-8),N(6)),PAIR(N(-2),N(4)))
  etst(floorx(N(-7),N(6)),PAIR(N(-2),N(5)))
  etst(floorx(N(-6),N(6)),PAIR(N(-1),N(0)))
  etst(floorx(N(-5),N(6)),PAIR(N(-1),N(1)))
  etst(floorx(N(-4),N(6)),PAIR(N(-1),N(2)))
  etst(floorx(N(-3),N(6)),PAIR(N(-1),N(3)))
  etst(floorx(N(-2),N(6)),PAIR(N(-1),N(4)))
  etst(floorx(N(-1),N(6)),PAIR(N(-1),N(5)))
  etst(floorx(N( 0),N(6)),PAIR(N( 0),N(0)))
  etst(floorx(N( 1),N(6)),PAIR(N( 0),N(1)))
  etst(floorx(N( 2),N(6)),PAIR(N( 0),N(2)))
  etst(floorx(N( 3),N(6)),PAIR(N( 0),N(3)))
  etst(floorx(N( 4),N(6)),PAIR(N( 0),N(4)))
  etst(floorx(N( 5),N(6)),PAIR(N( 0),N(5)))
  etst(floorx(N( 6),N(6)),PAIR(N( 1),N(0)))
  etst(floorx(N( 7),N(6)),PAIR(N( 1),N(1)))
  etst(floorx(N( 8),N(6)),PAIR(N( 1),N(2)))
  etst(floorx(N( 9),N(6)),PAIR(N( 1),N(3)))

  etst(floorx(N(-9),N(7)),PAIR(N(-2),N(5)))
  etst(floorx(N(-8),N(7)),PAIR(N(-2),N(6)))
  etst(floorx(N(-7),N(7)),PAIR(N(-1),N(0)))
  etst(floorx(N(-6),N(7)),PAIR(N(-1),N(1)))
  etst(floorx(N(-5),N(7)),PAIR(N(-1),N(2)))
  etst(floorx(N(-4),N(7)),PAIR(N(-1),N(3)))
  etst(floorx(N(-3),N(7)),PAIR(N(-1),N(4)))
  etst(floorx(N(-2),N(7)),PAIR(N(-1),N(5)))
  etst(floorx(N(-1),N(7)),PAIR(N(-1),N(6)))
  etst(floorx(N( 0),N(7)),PAIR(N( 0),N(0)))
  etst(floorx(N( 1),N(7)),PAIR(N( 0),N(1)))
  etst(floorx(N( 2),N(7)),PAIR(N( 0),N(2)))
  etst(floorx(N( 3),N(7)),PAIR(N( 0),N(3)))
  etst(floorx(N( 4),N(7)),PAIR(N( 0),N(4)))
  etst(floorx(N( 5),N(7)),PAIR(N( 0),N(5)))
  etst(floorx(N( 6),N(7)),PAIR(N( 0),N(6)))
  etst(floorx(N( 7),N(7)),PAIR(N( 1),N(0)))
  etst(floorx(N( 8),N(7)),PAIR(N( 1),N(1)))
  etst(floorx(N( 9),N(7)),PAIR(N( 1),N(2)))

  etst(floorx(N(-9),N(8)),PAIR(N(-2),N(7)))
  etst(floorx(N(-8),N(8)),PAIR(N(-1),N(0)))
  etst(floorx(N(-7),N(8)),PAIR(N(-1),N(1)))
  etst(floorx(N(-6),N(8)),PAIR(N(-1),N(2)))
  etst(floorx(N(-5),N(8)),PAIR(N(-1),N(3)))
  etst(floorx(N(-4),N(8)),PAIR(N(-1),N(4)))
  etst(floorx(N(-3),N(8)),PAIR(N(-1),N(5)))
  etst(floorx(N(-2),N(8)),PAIR(N(-1),N(6)))
  etst(floorx(N(-1),N(8)),PAIR(N(-1),N(7)))
  etst(floorx(N( 0),N(8)),PAIR(N( 0),N(0)))
  etst(floorx(N( 1),N(8)),PAIR(N( 0),N(1)))
  etst(floorx(N( 2),N(8)),PAIR(N( 0),N(2)))
  etst(floorx(N( 3),N(8)),PAIR(N( 0),N(3)))
  etst(floorx(N( 4),N(8)),PAIR(N( 0),N(4)))
  etst(floorx(N( 5),N(8)),PAIR(N( 0),N(5)))
  etst(floorx(N( 6),N(8)),PAIR(N( 0),N(6)))
  etst(floorx(N( 7),N(8)),PAIR(N( 0),N(7)))
  etst(floorx(N( 8),N(8)),PAIR(N( 1),N(0)))
  etst(floorx(N( 9),N(8)),PAIR(N( 1),N(1)))

  etst(floorx(N(-9),N(9)),PAIR(N(-1),N(0)))
  etst(floorx(N(-8),N(9)),PAIR(N(-1),N(1)))
  etst(floorx(N(-7),N(9)),PAIR(N(-1),N(2)))
  etst(floorx(N(-6),N(9)),PAIR(N(-1),N(3)))
  etst(floorx(N(-5),N(9)),PAIR(N(-1),N(4)))
  etst(floorx(N(-4),N(9)),PAIR(N(-1),N(5)))
  etst(floorx(N(-3),N(9)),PAIR(N(-1),N(6)))
  etst(floorx(N(-2),N(9)),PAIR(N(-1),N(7)))
  etst(floorx(N(-1),N(9)),PAIR(N(-1),N(8)))
  etst(floorx(N( 0),N(9)),PAIR(N( 0),N(0)))
  etst(floorx(N( 1),N(9)),PAIR(N( 0),N(1)))
  etst(floorx(N( 2),N(9)),PAIR(N( 0),N(2)))
  etst(floorx(N( 3),N(9)),PAIR(N( 0),N(3)))
  etst(floorx(N( 4),N(9)),PAIR(N( 0),N(4)))
  etst(floorx(N( 5),N(9)),PAIR(N( 0),N(5)))
  etst(floorx(N( 6),N(9)),PAIR(N( 0),N(6)))
  etst(floorx(N( 7),N(9)),PAIR(N( 0),N(7)))
  etst(floorx(N( 8),N(9)),PAIR(N( 0),N(8)))
  etst(floorx(N( 9),N(9)),PAIR(N( 1),N(0)))

  xtst(ceilingx(TT,TT))
  xtst(ceilingx(TT,FF))
  xtst(ceilingx(TT,I0))
  xtst(ceilingx(TT,P0))
  xtst(ceilingx(TT,M0))
  xtst(ceilingx(TT,B0))
  xtst(ceilingx(FF,TT))
  xtst(ceilingx(FF,FF))
  xtst(ceilingx(FF,I0))
  xtst(ceilingx(FF,P0))
  xtst(ceilingx(FF,M0))
  xtst(ceilingx(FF,B0))
  xtst(ceilingx(I0,TT))
  xtst(ceilingx(I0,FF))
  xtst(ceilingx(I0,P0))
  xtst(ceilingx(I0,M0))
  xtst(ceilingx(I0,B0))
  xtst(ceilingx(P0,TT))
  xtst(ceilingx(P0,FF))
  xtst(ceilingx(P0,I0))
  xtst(ceilingx(P0,P0))
  xtst(ceilingx(P0,M0))
  xtst(ceilingx(P0,B0))
  xtst(ceilingx(M0,TT))
  xtst(ceilingx(M0,FF))
  xtst(ceilingx(M0,I0))
  xtst(ceilingx(M0,P0))
  xtst(ceilingx(M0,M0))
  xtst(ceilingx(M0,B0))
  xtst(ceilingx(B0,TT))
  xtst(ceilingx(B0,FF))
  xtst(ceilingx(B0,I0))
  xtst(ceilingx(B0,P0))
  xtst(ceilingx(B0,M0))
  xtst(ceilingx(B0,B0))
  xtst(ceilingx(I0,Z))
  xtst(ceilingx(I0,M))

  etst(ceilingx(N( 9),N(1)),PAIR(N( 9),N( 0)))
  etst(ceilingx(N( 8),N(1)),PAIR(N( 8),N( 0)))
  etst(ceilingx(N( 7),N(1)),PAIR(N( 7),N( 0)))
  etst(ceilingx(N( 6),N(1)),PAIR(N( 6),N( 0)))
  etst(ceilingx(N( 5),N(1)),PAIR(N( 5),N( 0)))
  etst(ceilingx(N( 4),N(1)),PAIR(N( 4),N( 0)))
  etst(ceilingx(N( 3),N(1)),PAIR(N( 3),N( 0)))
  etst(ceilingx(N( 2),N(1)),PAIR(N( 2),N( 0)))
  etst(ceilingx(N( 1),N(1)),PAIR(N( 1),N( 0)))
  etst(ceilingx(N( 0),N(1)),PAIR(N( 0),N( 0)))
  etst(ceilingx(N(-1),N(1)),PAIR(N(-1),N( 0)))
  etst(ceilingx(N(-2),N(1)),PAIR(N(-2),N( 0)))
  etst(ceilingx(N(-3),N(1)),PAIR(N(-3),N( 0)))
  etst(ceilingx(N(-4),N(1)),PAIR(N(-4),N( 0)))
  etst(ceilingx(N(-5),N(1)),PAIR(N(-5),N( 0)))
  etst(ceilingx(N(-6),N(1)),PAIR(N(-6),N( 0)))
  etst(ceilingx(N(-7),N(1)),PAIR(N(-7),N( 0)))
  etst(ceilingx(N(-8),N(1)),PAIR(N(-8),N( 0)))
  etst(ceilingx(N(-9),N(1)),PAIR(N(-9),N( 0)))

  etst(ceilingx(N( 9),N(2)),PAIR(N( 5),N(-1)))
  etst(ceilingx(N( 8),N(2)),PAIR(N( 4),N( 0)))
  etst(ceilingx(N( 7),N(2)),PAIR(N( 4),N(-1)))
  etst(ceilingx(N( 6),N(2)),PAIR(N( 3),N( 0)))
  etst(ceilingx(N( 5),N(2)),PAIR(N( 3),N(-1)))
  etst(ceilingx(N( 4),N(2)),PAIR(N( 2),N( 0)))
  etst(ceilingx(N( 3),N(2)),PAIR(N( 2),N(-1)))
  etst(ceilingx(N( 2),N(2)),PAIR(N( 1),N( 0)))
  etst(ceilingx(N( 1),N(2)),PAIR(N( 1),N(-1)))
  etst(ceilingx(N( 0),N(2)),PAIR(N( 0),N( 0)))
  etst(ceilingx(N(-1),N(2)),PAIR(N( 0),N(-1)))
  etst(ceilingx(N(-2),N(2)),PAIR(N(-1),N( 0)))
  etst(ceilingx(N(-3),N(2)),PAIR(N(-1),N(-1)))
  etst(ceilingx(N(-4),N(2)),PAIR(N(-2),N( 0)))
  etst(ceilingx(N(-5),N(2)),PAIR(N(-2),N(-1)))
  etst(ceilingx(N(-6),N(2)),PAIR(N(-3),N( 0)))
  etst(ceilingx(N(-7),N(2)),PAIR(N(-3),N(-1)))
  etst(ceilingx(N(-8),N(2)),PAIR(N(-4),N( 0)))
  etst(ceilingx(N(-9),N(2)),PAIR(N(-4),N(-1)))

  etst(ceilingx(N( 9),N(3)),PAIR(N( 3),N( 0)))
  etst(ceilingx(N( 8),N(3)),PAIR(N( 3),N(-1)))
  etst(ceilingx(N( 7),N(3)),PAIR(N( 3),N(-2)))
  etst(ceilingx(N( 6),N(3)),PAIR(N( 2),N( 0)))
  etst(ceilingx(N( 5),N(3)),PAIR(N( 2),N(-1)))
  etst(ceilingx(N( 4),N(3)),PAIR(N( 2),N(-2)))
  etst(ceilingx(N( 3),N(3)),PAIR(N( 1),N( 0)))
  etst(ceilingx(N( 2),N(3)),PAIR(N( 1),N(-1)))
  etst(ceilingx(N( 1),N(3)),PAIR(N( 1),N(-2)))
  etst(ceilingx(N( 0),N(3)),PAIR(N( 0),N( 0)))
  etst(ceilingx(N(-1),N(3)),PAIR(N( 0),N(-1)))
  etst(ceilingx(N(-2),N(3)),PAIR(N( 0),N(-2)))
  etst(ceilingx(N(-3),N(3)),PAIR(N(-1),N( 0)))
  etst(ceilingx(N(-4),N(3)),PAIR(N(-1),N(-1)))
  etst(ceilingx(N(-5),N(3)),PAIR(N(-1),N(-2)))
  etst(ceilingx(N(-6),N(3)),PAIR(N(-2),N( 0)))
  etst(ceilingx(N(-7),N(3)),PAIR(N(-2),N(-1)))
  etst(ceilingx(N(-8),N(3)),PAIR(N(-2),N(-2)))
  etst(ceilingx(N(-9),N(3)),PAIR(N(-3),N( 0)))

  etst(ceilingx(N( 9),N(4)),PAIR(N( 3),N(-3)))
  etst(ceilingx(N( 8),N(4)),PAIR(N( 2),N( 0)))
  etst(ceilingx(N( 7),N(4)),PAIR(N( 2),N(-1)))
  etst(ceilingx(N( 6),N(4)),PAIR(N( 2),N(-2)))
  etst(ceilingx(N( 5),N(4)),PAIR(N( 2),N(-3)))
  etst(ceilingx(N( 4),N(4)),PAIR(N( 1),N( 0)))
  etst(ceilingx(N( 3),N(4)),PAIR(N( 1),N(-1)))
  etst(ceilingx(N( 2),N(4)),PAIR(N( 1),N(-2)))
  etst(ceilingx(N( 1),N(4)),PAIR(N( 1),N(-3)))
  etst(ceilingx(N( 0),N(4)),PAIR(N( 0),N( 0)))
  etst(ceilingx(N(-1),N(4)),PAIR(N( 0),N(-1)))
  etst(ceilingx(N(-2),N(4)),PAIR(N( 0),N(-2)))
  etst(ceilingx(N(-3),N(4)),PAIR(N( 0),N(-3)))
  etst(ceilingx(N(-4),N(4)),PAIR(N(-1),N( 0)))
  etst(ceilingx(N(-5),N(4)),PAIR(N(-1),N(-1)))
  etst(ceilingx(N(-6),N(4)),PAIR(N(-1),N(-2)))
  etst(ceilingx(N(-7),N(4)),PAIR(N(-1),N(-3)))
  etst(ceilingx(N(-8),N(4)),PAIR(N(-2),N( 0)))
  etst(ceilingx(N(-9),N(4)),PAIR(N(-2),N(-1)))

  etst(ceilingx(N( 9),N(5)),PAIR(N( 2),N(-1)))
  etst(ceilingx(N( 8),N(5)),PAIR(N( 2),N(-2)))
  etst(ceilingx(N( 7),N(5)),PAIR(N( 2),N(-3)))
  etst(ceilingx(N( 6),N(5)),PAIR(N( 2),N(-4)))
  etst(ceilingx(N( 5),N(5)),PAIR(N( 1),N( 0)))
  etst(ceilingx(N( 4),N(5)),PAIR(N( 1),N(-1)))
  etst(ceilingx(N( 3),N(5)),PAIR(N( 1),N(-2)))
  etst(ceilingx(N( 2),N(5)),PAIR(N( 1),N(-3)))
  etst(ceilingx(N( 1),N(5)),PAIR(N( 1),N(-4)))
  etst(ceilingx(N( 0),N(5)),PAIR(N( 0),N( 0)))
  etst(ceilingx(N(-1),N(5)),PAIR(N( 0),N(-1)))
  etst(ceilingx(N(-2),N(5)),PAIR(N( 0),N(-2)))
  etst(ceilingx(N(-3),N(5)),PAIR(N( 0),N(-3)))
  etst(ceilingx(N(-4),N(5)),PAIR(N( 0),N(-4)))
  etst(ceilingx(N(-5),N(5)),PAIR(N(-1),N( 0)))
  etst(ceilingx(N(-6),N(5)),PAIR(N(-1),N(-1)))
  etst(ceilingx(N(-7),N(5)),PAIR(N(-1),N(-2)))
  etst(ceilingx(N(-8),N(5)),PAIR(N(-1),N(-3)))
  etst(ceilingx(N(-9),N(5)),PAIR(N(-1),N(-4)))

  etst(ceilingx(N( 9),N(6)),PAIR(N( 2),N(-3)))
  etst(ceilingx(N( 8),N(6)),PAIR(N( 2),N(-4)))
  etst(ceilingx(N( 7),N(6)),PAIR(N( 2),N(-5)))
  etst(ceilingx(N( 6),N(6)),PAIR(N( 1),N( 0)))
  etst(ceilingx(N( 5),N(6)),PAIR(N( 1),N(-1)))
  etst(ceilingx(N( 4),N(6)),PAIR(N( 1),N(-2)))
  etst(ceilingx(N( 3),N(6)),PAIR(N( 1),N(-3)))
  etst(ceilingx(N( 2),N(6)),PAIR(N( 1),N(-4)))
  etst(ceilingx(N( 1),N(6)),PAIR(N( 1),N(-5)))
  etst(ceilingx(N( 0),N(6)),PAIR(N( 0),N( 0)))
  etst(ceilingx(N(-1),N(6)),PAIR(N( 0),N(-1)))
  etst(ceilingx(N(-2),N(6)),PAIR(N( 0),N(-2)))
  etst(ceilingx(N(-3),N(6)),PAIR(N( 0),N(-3)))
  etst(ceilingx(N(-4),N(6)),PAIR(N( 0),N(-4)))
  etst(ceilingx(N(-5),N(6)),PAIR(N( 0),N(-5)))
  etst(ceilingx(N(-6),N(6)),PAIR(N(-1),N( 0)))
  etst(ceilingx(N(-7),N(6)),PAIR(N(-1),N(-1)))
  etst(ceilingx(N(-8),N(6)),PAIR(N(-1),N(-2)))
  etst(ceilingx(N(-9),N(6)),PAIR(N(-1),N(-3)))

  etst(ceilingx(N( 9),N(7)),PAIR(N( 2),N(-5)))
  etst(ceilingx(N( 8),N(7)),PAIR(N( 2),N(-6)))
  etst(ceilingx(N( 7),N(7)),PAIR(N( 1),N( 0)))
  etst(ceilingx(N( 6),N(7)),PAIR(N( 1),N(-1)))
  etst(ceilingx(N( 5),N(7)),PAIR(N( 1),N(-2)))
  etst(ceilingx(N( 4),N(7)),PAIR(N( 1),N(-3)))
  etst(ceilingx(N( 3),N(7)),PAIR(N( 1),N(-4)))
  etst(ceilingx(N( 2),N(7)),PAIR(N( 1),N(-5)))
  etst(ceilingx(N( 1),N(7)),PAIR(N( 1),N(-6)))
  etst(ceilingx(N( 0),N(7)),PAIR(N( 0),N( 0)))
  etst(ceilingx(N(-1),N(7)),PAIR(N( 0),N(-1)))
  etst(ceilingx(N(-2),N(7)),PAIR(N( 0),N(-2)))
  etst(ceilingx(N(-3),N(7)),PAIR(N( 0),N(-3)))
  etst(ceilingx(N(-4),N(7)),PAIR(N( 0),N(-4)))
  etst(ceilingx(N(-5),N(7)),PAIR(N( 0),N(-5)))
  etst(ceilingx(N(-6),N(7)),PAIR(N( 0),N(-6)))
  etst(ceilingx(N(-7),N(7)),PAIR(N(-1),N( 0)))
  etst(ceilingx(N(-8),N(7)),PAIR(N(-1),N(-1)))
  etst(ceilingx(N(-9),N(7)),PAIR(N(-1),N(-2)))

  etst(ceilingx(N( 9),N(8)),PAIR(N( 2),N(-7)))
  etst(ceilingx(N( 8),N(8)),PAIR(N( 1),N( 0)))
  etst(ceilingx(N( 7),N(8)),PAIR(N( 1),N(-1)))
  etst(ceilingx(N( 6),N(8)),PAIR(N( 1),N(-2)))
  etst(ceilingx(N( 5),N(8)),PAIR(N( 1),N(-3)))
  etst(ceilingx(N( 4),N(8)),PAIR(N( 1),N(-4)))
  etst(ceilingx(N( 3),N(8)),PAIR(N( 1),N(-5)))
  etst(ceilingx(N( 2),N(8)),PAIR(N( 1),N(-6)))
  etst(ceilingx(N( 1),N(8)),PAIR(N( 1),N(-7)))
  etst(ceilingx(N( 0),N(8)),PAIR(N( 0),N( 0)))
  etst(ceilingx(N(-1),N(8)),PAIR(N( 0),N(-1)))
  etst(ceilingx(N(-2),N(8)),PAIR(N( 0),N(-2)))
  etst(ceilingx(N(-3),N(8)),PAIR(N( 0),N(-3)))
  etst(ceilingx(N(-4),N(8)),PAIR(N( 0),N(-4)))
  etst(ceilingx(N(-5),N(8)),PAIR(N( 0),N(-5)))
  etst(ceilingx(N(-6),N(8)),PAIR(N( 0),N(-6)))
  etst(ceilingx(N(-7),N(8)),PAIR(N( 0),N(-7)))
  etst(ceilingx(N(-8),N(8)),PAIR(N(-1),N( 0)))
  etst(ceilingx(N(-9),N(8)),PAIR(N(-1),N(-1)))

  etst(ceilingx(N( 9),N(9)),PAIR(N( 1),N( 0)))
  etst(ceilingx(N( 8),N(9)),PAIR(N( 1),N(-1)))
  etst(ceilingx(N( 7),N(9)),PAIR(N( 1),N(-2)))
  etst(ceilingx(N( 6),N(9)),PAIR(N( 1),N(-3)))
  etst(ceilingx(N( 5),N(9)),PAIR(N( 1),N(-4)))
  etst(ceilingx(N( 4),N(9)),PAIR(N( 1),N(-5)))
  etst(ceilingx(N( 3),N(9)),PAIR(N( 1),N(-6)))
  etst(ceilingx(N( 2),N(9)),PAIR(N( 1),N(-7)))
  etst(ceilingx(N( 1),N(9)),PAIR(N( 1),N(-8)))
  etst(ceilingx(N( 0),N(9)),PAIR(N( 0),N( 0)))
  etst(ceilingx(N(-1),N(9)),PAIR(N( 0),N(-1)))
  etst(ceilingx(N(-2),N(9)),PAIR(N( 0),N(-2)))
  etst(ceilingx(N(-3),N(9)),PAIR(N( 0),N(-3)))
  etst(ceilingx(N(-4),N(9)),PAIR(N( 0),N(-4)))
  etst(ceilingx(N(-5),N(9)),PAIR(N( 0),N(-5)))
  etst(ceilingx(N(-6),N(9)),PAIR(N( 0),N(-6)))
  etst(ceilingx(N(-7),N(9)),PAIR(N( 0),N(-7)))
  etst(ceilingx(N(-8),N(9)),PAIR(N( 0),N(-8)))
  etst(ceilingx(N(-9),N(9)),PAIR(N(-1),N( 0)))

  xtst(truncatex(TT,TT))
  xtst(truncatex(TT,FF))
  xtst(truncatex(TT,I0))
  xtst(truncatex(TT,P0))
  xtst(truncatex(TT,M0))
  xtst(truncatex(TT,B0))
  xtst(truncatex(FF,TT))
  xtst(truncatex(FF,FF))
  xtst(truncatex(FF,I0))
  xtst(truncatex(FF,P0))
  xtst(truncatex(FF,M0))
  xtst(truncatex(FF,B0))
  xtst(truncatex(I0,TT))
  xtst(truncatex(I0,FF))
  xtst(truncatex(I0,P0))
  xtst(truncatex(I0,M0))
  xtst(truncatex(I0,B0))
  xtst(truncatex(P0,TT))
  xtst(truncatex(P0,FF))
  xtst(truncatex(P0,I0))
  xtst(truncatex(P0,P0))
  xtst(truncatex(P0,M0))
  xtst(truncatex(P0,B0))
  xtst(truncatex(M0,TT))
  xtst(truncatex(M0,FF))
  xtst(truncatex(M0,I0))
  xtst(truncatex(M0,P0))
  xtst(truncatex(M0,M0))
  xtst(truncatex(M0,B0))
  xtst(truncatex(B0,TT))
  xtst(truncatex(B0,FF))
  xtst(truncatex(B0,I0))
  xtst(truncatex(B0,P0))
  xtst(truncatex(B0,M0))
  xtst(truncatex(B0,B0))
  xtst(truncatex(I0,Z))
  xtst(truncatex(I0,M))

  etst(truncatex(N(-9),N(1)),PAIR(N(-9),N( 0)))
  etst(truncatex(N(-8),N(1)),PAIR(N(-8),N( 0)))
  etst(truncatex(N(-7),N(1)),PAIR(N(-7),N( 0)))
  etst(truncatex(N(-6),N(1)),PAIR(N(-6),N( 0)))
  etst(truncatex(N(-5),N(1)),PAIR(N(-5),N( 0)))
  etst(truncatex(N(-4),N(1)),PAIR(N(-4),N( 0)))
  etst(truncatex(N(-3),N(1)),PAIR(N(-3),N( 0)))
  etst(truncatex(N(-2),N(1)),PAIR(N(-2),N( 0)))
  etst(truncatex(N(-1),N(1)),PAIR(N(-1),N( 0)))
  etst(truncatex(N( 0),N(1)),PAIR(N( 0),N( 0)))
  etst(truncatex(N( 1),N(1)),PAIR(N( 1),N( 0)))
  etst(truncatex(N( 2),N(1)),PAIR(N( 2),N( 0)))
  etst(truncatex(N( 3),N(1)),PAIR(N( 3),N( 0)))
  etst(truncatex(N( 4),N(1)),PAIR(N( 4),N( 0)))
  etst(truncatex(N( 5),N(1)),PAIR(N( 5),N( 0)))
  etst(truncatex(N( 6),N(1)),PAIR(N( 6),N( 0)))
  etst(truncatex(N( 7),N(1)),PAIR(N( 7),N( 0)))
  etst(truncatex(N( 8),N(1)),PAIR(N( 8),N( 0)))
  etst(truncatex(N( 9),N(1)),PAIR(N( 9),N( 0)))

  etst(truncatex(N(-9),N(2)),PAIR(N(-4),N(-1)))
  etst(truncatex(N(-8),N(2)),PAIR(N(-4),N( 0)))
  etst(truncatex(N(-7),N(2)),PAIR(N(-3),N(-1)))
  etst(truncatex(N(-6),N(2)),PAIR(N(-3),N( 0)))
  etst(truncatex(N(-5),N(2)),PAIR(N(-2),N(-1)))
  etst(truncatex(N(-4),N(2)),PAIR(N(-2),N( 0)))
  etst(truncatex(N(-3),N(2)),PAIR(N(-1),N(-1)))
  etst(truncatex(N(-2),N(2)),PAIR(N(-1),N( 0)))
  etst(truncatex(N(-1),N(2)),PAIR(N( 0),N(-1)))
  etst(truncatex(N( 0),N(2)),PAIR(N( 0),N( 0)))
  etst(truncatex(N( 1),N(2)),PAIR(N( 0),N( 1)))
  etst(truncatex(N( 2),N(2)),PAIR(N( 1),N( 0)))
  etst(truncatex(N( 3),N(2)),PAIR(N( 1),N( 1)))
  etst(truncatex(N( 4),N(2)),PAIR(N( 2),N( 0)))
  etst(truncatex(N( 5),N(2)),PAIR(N( 2),N( 1)))
  etst(truncatex(N( 6),N(2)),PAIR(N( 3),N( 0)))
  etst(truncatex(N( 7),N(2)),PAIR(N( 3),N( 1)))
  etst(truncatex(N( 8),N(2)),PAIR(N( 4),N( 0)))
  etst(truncatex(N( 9),N(2)),PAIR(N( 4),N( 1)))

  etst(truncatex(N(-9),N(3)),PAIR(N(-3),N( 0)))
  etst(truncatex(N(-8),N(3)),PAIR(N(-2),N(-2)))
  etst(truncatex(N(-7),N(3)),PAIR(N(-2),N(-1)))
  etst(truncatex(N(-6),N(3)),PAIR(N(-2),N( 0)))
  etst(truncatex(N(-5),N(3)),PAIR(N(-1),N(-2)))
  etst(truncatex(N(-4),N(3)),PAIR(N(-1),N(-1)))
  etst(truncatex(N(-3),N(3)),PAIR(N(-1),N( 0)))
  etst(truncatex(N(-2),N(3)),PAIR(N( 0),N(-2)))
  etst(truncatex(N(-1),N(3)),PAIR(N( 0),N(-1)))
  etst(truncatex(N( 0),N(3)),PAIR(N( 0),N( 0)))
  etst(truncatex(N( 1),N(3)),PAIR(N( 0),N( 1)))
  etst(truncatex(N( 2),N(3)),PAIR(N( 0),N( 2)))
  etst(truncatex(N( 3),N(3)),PAIR(N( 1),N( 0)))
  etst(truncatex(N( 4),N(3)),PAIR(N( 1),N( 1)))
  etst(truncatex(N( 5),N(3)),PAIR(N( 1),N( 2)))
  etst(truncatex(N( 6),N(3)),PAIR(N( 2),N( 0)))
  etst(truncatex(N( 7),N(3)),PAIR(N( 2),N( 1)))
  etst(truncatex(N( 8),N(3)),PAIR(N( 2),N( 2)))
  etst(truncatex(N( 9),N(3)),PAIR(N( 3),N( 0)))

  etst(truncatex(N(-9),N(4)),PAIR(N(-2),N(-1)))
  etst(truncatex(N(-8),N(4)),PAIR(N(-2),N( 0)))
  etst(truncatex(N(-7),N(4)),PAIR(N(-1),N(-3)))
  etst(truncatex(N(-6),N(4)),PAIR(N(-1),N(-2)))
  etst(truncatex(N(-5),N(4)),PAIR(N(-1),N(-1)))
  etst(truncatex(N(-4),N(4)),PAIR(N(-1),N( 0)))
  etst(truncatex(N(-3),N(4)),PAIR(N( 0),N(-3)))
  etst(truncatex(N(-2),N(4)),PAIR(N( 0),N(-2)))
  etst(truncatex(N(-1),N(4)),PAIR(N( 0),N(-1)))
  etst(truncatex(N( 0),N(4)),PAIR(N( 0),N( 0)))
  etst(truncatex(N( 1),N(4)),PAIR(N( 0),N( 1)))
  etst(truncatex(N( 2),N(4)),PAIR(N( 0),N( 2)))
  etst(truncatex(N( 3),N(4)),PAIR(N( 0),N( 3)))
  etst(truncatex(N( 4),N(4)),PAIR(N( 1),N( 0)))
  etst(truncatex(N( 5),N(4)),PAIR(N( 1),N( 1)))
  etst(truncatex(N( 6),N(4)),PAIR(N( 1),N( 2)))
  etst(truncatex(N( 7),N(4)),PAIR(N( 1),N( 3)))
  etst(truncatex(N( 8),N(4)),PAIR(N( 2),N( 0)))
  etst(truncatex(N( 9),N(4)),PAIR(N( 2),N( 1)))

  etst(truncatex(N(-9),N(5)),PAIR(N(-1),N(-4)))
  etst(truncatex(N(-8),N(5)),PAIR(N(-1),N(-3)))
  etst(truncatex(N(-7),N(5)),PAIR(N(-1),N(-2)))
  etst(truncatex(N(-6),N(5)),PAIR(N(-1),N(-1)))
  etst(truncatex(N(-5),N(5)),PAIR(N(-1),N( 0)))
  etst(truncatex(N(-4),N(5)),PAIR(N( 0),N(-4)))
  etst(truncatex(N(-3),N(5)),PAIR(N( 0),N(-3)))
  etst(truncatex(N(-2),N(5)),PAIR(N( 0),N(-2)))
  etst(truncatex(N(-1),N(5)),PAIR(N( 0),N(-1)))
  etst(truncatex(N( 0),N(5)),PAIR(N( 0),N( 0)))
  etst(truncatex(N( 1),N(5)),PAIR(N( 0),N( 1)))
  etst(truncatex(N( 2),N(5)),PAIR(N( 0),N( 2)))
  etst(truncatex(N( 3),N(5)),PAIR(N( 0),N( 3)))
  etst(truncatex(N( 4),N(5)),PAIR(N( 0),N( 4)))
  etst(truncatex(N( 5),N(5)),PAIR(N( 1),N( 0)))
  etst(truncatex(N( 6),N(5)),PAIR(N( 1),N( 1)))
  etst(truncatex(N( 7),N(5)),PAIR(N( 1),N( 2)))
  etst(truncatex(N( 8),N(5)),PAIR(N( 1),N( 3)))
  etst(truncatex(N( 9),N(5)),PAIR(N( 1),N( 4)))

  etst(truncatex(N(-9),N(6)),PAIR(N(-1),N(-3)))
  etst(truncatex(N(-8),N(6)),PAIR(N(-1),N(-2)))
  etst(truncatex(N(-7),N(6)),PAIR(N(-1),N(-1)))
  etst(truncatex(N(-6),N(6)),PAIR(N(-1),N( 0)))
  etst(truncatex(N(-5),N(6)),PAIR(N( 0),N(-5)))
  etst(truncatex(N(-4),N(6)),PAIR(N( 0),N(-4)))
  etst(truncatex(N(-3),N(6)),PAIR(N( 0),N(-3)))
  etst(truncatex(N(-2),N(6)),PAIR(N( 0),N(-2)))
  etst(truncatex(N(-1),N(6)),PAIR(N( 0),N(-1)))
  etst(truncatex(N( 0),N(6)),PAIR(N( 0),N( 0)))
  etst(truncatex(N( 1),N(6)),PAIR(N( 0),N( 1)))
  etst(truncatex(N( 2),N(6)),PAIR(N( 0),N( 2)))
  etst(truncatex(N( 3),N(6)),PAIR(N( 0),N( 3)))
  etst(truncatex(N( 4),N(6)),PAIR(N( 0),N( 4)))
  etst(truncatex(N( 5),N(6)),PAIR(N( 0),N( 5)))
  etst(truncatex(N( 6),N(6)),PAIR(N( 1),N( 0)))
  etst(truncatex(N( 7),N(6)),PAIR(N( 1),N( 1)))
  etst(truncatex(N( 8),N(6)),PAIR(N( 1),N( 2)))
  etst(truncatex(N( 9),N(6)),PAIR(N( 1),N( 3)))

  etst(truncatex(N(-9),N(7)),PAIR(N(-1),N(-2)))
  etst(truncatex(N(-8),N(7)),PAIR(N(-1),N(-1)))
  etst(truncatex(N(-7),N(7)),PAIR(N(-1),N( 0)))
  etst(truncatex(N(-6),N(7)),PAIR(N( 0),N(-6)))
  etst(truncatex(N(-5),N(7)),PAIR(N( 0),N(-5)))
  etst(truncatex(N(-4),N(7)),PAIR(N( 0),N(-4)))
  etst(truncatex(N(-3),N(7)),PAIR(N( 0),N(-3)))
  etst(truncatex(N(-2),N(7)),PAIR(N( 0),N(-2)))
  etst(truncatex(N(-1),N(7)),PAIR(N( 0),N(-1)))
  etst(truncatex(N( 0),N(7)),PAIR(N( 0),N( 0)))
  etst(truncatex(N( 1),N(7)),PAIR(N( 0),N( 1)))
  etst(truncatex(N( 2),N(7)),PAIR(N( 0),N( 2)))
  etst(truncatex(N( 3),N(7)),PAIR(N( 0),N( 3)))
  etst(truncatex(N( 4),N(7)),PAIR(N( 0),N( 4)))
  etst(truncatex(N( 5),N(7)),PAIR(N( 0),N( 5)))
  etst(truncatex(N( 6),N(7)),PAIR(N( 0),N( 6)))
  etst(truncatex(N( 7),N(7)),PAIR(N( 1),N( 0)))
  etst(truncatex(N( 8),N(7)),PAIR(N( 1),N( 1)))
  etst(truncatex(N( 9),N(7)),PAIR(N( 1),N( 2)))

  etst(truncatex(N(-9),N(8)),PAIR(N(-1),N(-1)))
  etst(truncatex(N(-8),N(8)),PAIR(N(-1),N( 0)))
  etst(truncatex(N(-7),N(8)),PAIR(N( 0),N(-7)))
  etst(truncatex(N(-6),N(8)),PAIR(N( 0),N(-6)))
  etst(truncatex(N(-5),N(8)),PAIR(N( 0),N(-5)))
  etst(truncatex(N(-4),N(8)),PAIR(N( 0),N(-4)))
  etst(truncatex(N(-3),N(8)),PAIR(N( 0),N(-3)))
  etst(truncatex(N(-2),N(8)),PAIR(N( 0),N(-2)))
  etst(truncatex(N(-1),N(8)),PAIR(N( 0),N(-1)))
  etst(truncatex(N( 0),N(8)),PAIR(N( 0),N( 0)))
  etst(truncatex(N( 1),N(8)),PAIR(N( 0),N( 1)))
  etst(truncatex(N( 2),N(8)),PAIR(N( 0),N( 2)))
  etst(truncatex(N( 3),N(8)),PAIR(N( 0),N( 3)))
  etst(truncatex(N( 4),N(8)),PAIR(N( 0),N( 4)))
  etst(truncatex(N( 5),N(8)),PAIR(N( 0),N( 5)))
  etst(truncatex(N( 6),N(8)),PAIR(N( 0),N( 6)))
  etst(truncatex(N( 7),N(8)),PAIR(N( 0),N( 7)))
  etst(truncatex(N( 8),N(8)),PAIR(N( 1),N( 0)))
  etst(truncatex(N( 9),N(8)),PAIR(N( 1),N( 1)))

  etst(truncatex(N(-9),N(9)),PAIR(N(-1),N( 0)))
  etst(truncatex(N(-8),N(9)),PAIR(N( 0),N(-8)))
  etst(truncatex(N(-7),N(9)),PAIR(N( 0),N(-7)))
  etst(truncatex(N(-6),N(9)),PAIR(N( 0),N(-6)))
  etst(truncatex(N(-5),N(9)),PAIR(N( 0),N(-5)))
  etst(truncatex(N(-4),N(9)),PAIR(N( 0),N(-4)))
  etst(truncatex(N(-3),N(9)),PAIR(N( 0),N(-3)))
  etst(truncatex(N(-2),N(9)),PAIR(N( 0),N(-2)))
  etst(truncatex(N(-1),N(9)),PAIR(N( 0),N(-1)))
  etst(truncatex(N( 0),N(9)),PAIR(N( 0),N( 0)))
  etst(truncatex(N( 1),N(9)),PAIR(N( 0),N( 1)))
  etst(truncatex(N( 2),N(9)),PAIR(N( 0),N( 2)))
  etst(truncatex(N( 3),N(9)),PAIR(N( 0),N( 3)))
  etst(truncatex(N( 4),N(9)),PAIR(N( 0),N( 4)))
  etst(truncatex(N( 5),N(9)),PAIR(N( 0),N( 5)))
  etst(truncatex(N( 6),N(9)),PAIR(N( 0),N( 6)))
  etst(truncatex(N( 7),N(9)),PAIR(N( 0),N( 7)))
  etst(truncatex(N( 8),N(9)),PAIR(N( 0),N( 8)))
  etst(truncatex(N( 9),N(9)),PAIR(N( 1),N( 0)))

  xtst(roundx(TT,TT))
  xtst(roundx(TT,FF))
  xtst(roundx(TT,I0))
  xtst(roundx(TT,P0))
  xtst(roundx(TT,M0))
  xtst(roundx(TT,B0))
  xtst(roundx(FF,TT))
  xtst(roundx(FF,FF))
  xtst(roundx(FF,I0))
  xtst(roundx(FF,P0))
  xtst(roundx(FF,M0))
  xtst(roundx(FF,B0))
  xtst(roundx(I0,TT))
  xtst(roundx(I0,FF))
  xtst(roundx(I0,P0))
  xtst(roundx(I0,M0))
  xtst(roundx(I0,B0))
  xtst(roundx(P0,TT))
  xtst(roundx(P0,FF))
  xtst(roundx(P0,I0))
  xtst(roundx(P0,P0))
  xtst(roundx(P0,M0))
  xtst(roundx(P0,B0))
  xtst(roundx(M0,TT))
  xtst(roundx(M0,FF))
  xtst(roundx(M0,I0))
  xtst(roundx(M0,P0))
  xtst(roundx(M0,M0))
  xtst(roundx(M0,B0))
  xtst(roundx(B0,TT))
  xtst(roundx(B0,FF))
  xtst(roundx(B0,I0))
  xtst(roundx(B0,P0))
  xtst(roundx(B0,M0))
  xtst(roundx(B0,B0))
  xtst(roundx(I0,Z))
  xtst(roundx(I0,M))

  etst(roundx(N(-9),N(1)),PAIR(N(-9),N( 0)))
  etst(roundx(N(-8),N(1)),PAIR(N(-8),N( 0)))
  etst(roundx(N(-7),N(1)),PAIR(N(-7),N( 0)))
  etst(roundx(N(-6),N(1)),PAIR(N(-6),N( 0)))
  etst(roundx(N(-5),N(1)),PAIR(N(-5),N( 0)))
  etst(roundx(N(-4),N(1)),PAIR(N(-4),N( 0)))
  etst(roundx(N(-3),N(1)),PAIR(N(-3),N( 0)))
  etst(roundx(N(-2),N(1)),PAIR(N(-2),N( 0)))
  etst(roundx(N(-1),N(1)),PAIR(N(-1),N( 0)))
  etst(roundx(N( 9),N(1)),PAIR(N( 9),N( 0)))
  etst(roundx(N( 8),N(1)),PAIR(N( 8),N( 0)))
  etst(roundx(N( 7),N(1)),PAIR(N( 7),N( 0)))
  etst(roundx(N( 6),N(1)),PAIR(N( 6),N( 0)))
  etst(roundx(N( 5),N(1)),PAIR(N( 5),N( 0)))
  etst(roundx(N( 4),N(1)),PAIR(N( 4),N( 0)))
  etst(roundx(N( 3),N(1)),PAIR(N( 3),N( 0)))
  etst(roundx(N( 2),N(1)),PAIR(N( 2),N( 0)))
  etst(roundx(N( 1),N(1)),PAIR(N( 1),N( 0)))
  etst(roundx(N( 0),N(1)),PAIR(N( 0),N( 0)))

  etst(roundx(N(-9),N(2)),PAIR(N(-4),N(-1)))
  etst(roundx(N(-8),N(2)),PAIR(N(-4),N( 0)))
  etst(roundx(N(-7),N(2)),PAIR(N(-4),N( 1)))
  etst(roundx(N(-6),N(2)),PAIR(N(-3),N( 0)))
  etst(roundx(N(-5),N(2)),PAIR(N(-2),N(-1)))
  etst(roundx(N(-4),N(2)),PAIR(N(-2),N( 0)))
  etst(roundx(N(-3),N(2)),PAIR(N(-2),N( 1)))
  etst(roundx(N(-2),N(2)),PAIR(N(-1),N( 0)))
  etst(roundx(N(-1),N(2)),PAIR(N(-0),N(-1)))
  etst(roundx(N( 9),N(2)),PAIR(N( 4),N( 1)))
  etst(roundx(N( 8),N(2)),PAIR(N( 4),N( 0)))
  etst(roundx(N( 7),N(2)),PAIR(N( 4),N(-1)))
  etst(roundx(N( 6),N(2)),PAIR(N( 3),N( 0)))
  etst(roundx(N( 5),N(2)),PAIR(N( 2),N( 1)))
  etst(roundx(N( 4),N(2)),PAIR(N( 2),N( 0)))
  etst(roundx(N( 3),N(2)),PAIR(N( 2),N(-1)))
  etst(roundx(N( 2),N(2)),PAIR(N( 1),N( 0)))
  etst(roundx(N( 1),N(2)),PAIR(N( 0),N( 1)))
  etst(roundx(N( 0),N(2)),PAIR(N( 0),N( 0)))

  etst(roundx(N(-9),N(3)),PAIR(N(-3),N( 0)))
  etst(roundx(N(-8),N(3)),PAIR(N(-3),N( 1)))
  etst(roundx(N(-7),N(3)),PAIR(N(-2),N(-1)))
  etst(roundx(N(-6),N(3)),PAIR(N(-2),N( 0)))
  etst(roundx(N(-5),N(3)),PAIR(N(-2),N( 1)))
  etst(roundx(N(-4),N(3)),PAIR(N(-1),N(-1)))
  etst(roundx(N(-3),N(3)),PAIR(N(-1),N( 0)))
  etst(roundx(N(-2),N(3)),PAIR(N(-1),N( 1)))
  etst(roundx(N(-1),N(3)),PAIR(N(-0),N(-1)))
  etst(roundx(N( 9),N(3)),PAIR(N( 3),N( 0)))
  etst(roundx(N( 8),N(3)),PAIR(N( 3),N(-1)))
  etst(roundx(N( 7),N(3)),PAIR(N( 2),N( 1)))
  etst(roundx(N( 6),N(3)),PAIR(N( 2),N( 0)))
  etst(roundx(N( 5),N(3)),PAIR(N( 2),N(-1)))
  etst(roundx(N( 4),N(3)),PAIR(N( 1),N( 1)))
  etst(roundx(N( 3),N(3)),PAIR(N( 1),N( 0)))
  etst(roundx(N( 2),N(3)),PAIR(N( 1),N(-1)))
  etst(roundx(N( 1),N(3)),PAIR(N( 0),N( 1)))
  etst(roundx(N( 0),N(3)),PAIR(N( 0),N( 0)))

  etst(roundx(N(-9),N(4)),PAIR(N(-2),N(-1)))
  etst(roundx(N(-8),N(4)),PAIR(N(-2),N( 0)))
  etst(roundx(N(-7),N(4)),PAIR(N(-2),N( 1)))
  etst(roundx(N(-6),N(4)),PAIR(N(-2),N( 2)))
  etst(roundx(N(-5),N(4)),PAIR(N(-1),N(-1)))
  etst(roundx(N(-4),N(4)),PAIR(N(-1),N( 0)))
  etst(roundx(N(-3),N(4)),PAIR(N(-1),N( 1)))
  etst(roundx(N(-2),N(4)),PAIR(N(-0),N(-2)))
  etst(roundx(N(-1),N(4)),PAIR(N(-0),N(-1)))
  etst(roundx(N( 9),N(4)),PAIR(N( 2),N( 1)))
  etst(roundx(N( 8),N(4)),PAIR(N( 2),N( 0)))
  etst(roundx(N( 7),N(4)),PAIR(N( 2),N(-1)))
  etst(roundx(N( 6),N(4)),PAIR(N( 2),N(-2)))
  etst(roundx(N( 5),N(4)),PAIR(N( 1),N( 1)))
  etst(roundx(N( 4),N(4)),PAIR(N( 1),N( 0)))
  etst(roundx(N( 3),N(4)),PAIR(N( 1),N(-1)))
  etst(roundx(N( 2),N(4)),PAIR(N( 0),N( 2)))
  etst(roundx(N( 1),N(4)),PAIR(N( 0),N( 1)))
  etst(roundx(N( 0),N(4)),PAIR(N( 0),N( 0)))

  etst(roundx(N(-9),N(5)),PAIR(N(-2),N( 1)))
  etst(roundx(N(-8),N(5)),PAIR(N(-2),N( 2)))
  etst(roundx(N(-7),N(5)),PAIR(N(-1),N(-2)))
  etst(roundx(N(-6),N(5)),PAIR(N(-1),N(-1)))
  etst(roundx(N(-5),N(5)),PAIR(N(-1),N( 0)))
  etst(roundx(N(-4),N(5)),PAIR(N(-1),N( 1)))
  etst(roundx(N(-3),N(5)),PAIR(N(-1),N( 2)))
  etst(roundx(N(-2),N(5)),PAIR(N(-0),N(-2)))
  etst(roundx(N(-1),N(5)),PAIR(N(-0),N(-1)))
  etst(roundx(N( 9),N(5)),PAIR(N( 2),N(-1)))
  etst(roundx(N( 8),N(5)),PAIR(N( 2),N(-2)))
  etst(roundx(N( 7),N(5)),PAIR(N( 1),N( 2)))
  etst(roundx(N( 6),N(5)),PAIR(N( 1),N( 1)))
  etst(roundx(N( 5),N(5)),PAIR(N( 1),N( 0)))
  etst(roundx(N( 4),N(5)),PAIR(N( 1),N(-1)))
  etst(roundx(N( 3),N(5)),PAIR(N( 1),N(-2)))
  etst(roundx(N( 2),N(5)),PAIR(N( 0),N( 2)))
  etst(roundx(N( 1),N(5)),PAIR(N( 0),N( 1)))
  etst(roundx(N( 0),N(5)),PAIR(N( 0),N( 0)))

  etst(roundx(N(-9),N(6)),PAIR(N(-2),N( 3)))
  etst(roundx(N(-8),N(6)),PAIR(N(-1),N(-2)))
  etst(roundx(N(-7),N(6)),PAIR(N(-1),N(-1)))
  etst(roundx(N(-6),N(6)),PAIR(N(-1),N( 0)))
  etst(roundx(N(-5),N(6)),PAIR(N(-1),N( 1)))
  etst(roundx(N(-4),N(6)),PAIR(N(-1),N( 2)))
  etst(roundx(N(-3),N(6)),PAIR(N(-0),N(-3)))
  etst(roundx(N(-2),N(6)),PAIR(N(-0),N(-2)))
  etst(roundx(N(-1),N(6)),PAIR(N(-0),N(-1)))
  etst(roundx(N( 9),N(6)),PAIR(N( 2),N(-3)))
  etst(roundx(N( 8),N(6)),PAIR(N( 1),N( 2)))
  etst(roundx(N( 7),N(6)),PAIR(N( 1),N( 1)))
  etst(roundx(N( 6),N(6)),PAIR(N( 1),N( 0)))
  etst(roundx(N( 5),N(6)),PAIR(N( 1),N(-1)))
  etst(roundx(N( 4),N(6)),PAIR(N( 1),N(-2)))
  etst(roundx(N( 3),N(6)),PAIR(N( 0),N( 3)))
  etst(roundx(N( 2),N(6)),PAIR(N( 0),N( 2)))
  etst(roundx(N( 1),N(6)),PAIR(N( 0),N( 1)))
  etst(roundx(N( 0),N(6)),PAIR(N( 0),N( 0)))

  etst(roundx(N(-9),N(7)),PAIR(N(-1),N(-2)))
  etst(roundx(N(-8),N(7)),PAIR(N(-1),N(-1)))
  etst(roundx(N(-7),N(7)),PAIR(N(-1),N( 0)))
  etst(roundx(N(-6),N(7)),PAIR(N(-1),N( 1)))
  etst(roundx(N(-5),N(7)),PAIR(N(-1),N( 2)))
  etst(roundx(N(-4),N(7)),PAIR(N(-1),N( 3)))
  etst(roundx(N(-3),N(7)),PAIR(N(-0),N(3-)))
  etst(roundx(N(-2),N(7)),PAIR(N(-0),N(-2)))
  etst(roundx(N(-1),N(7)),PAIR(N(-0),N(-1)))
  etst(roundx(N( 9),N(7)),PAIR(N( 1),N( 2)))
  etst(roundx(N( 8),N(7)),PAIR(N( 1),N( 1)))
  etst(roundx(N( 7),N(7)),PAIR(N( 1),N( 0)))
  etst(roundx(N( 6),N(7)),PAIR(N( 1),N(-1)))
  etst(roundx(N( 5),N(7)),PAIR(N( 1),N(-2)))
  etst(roundx(N( 4),N(7)),PAIR(N( 1),N(-3)))
  etst(roundx(N( 3),N(7)),PAIR(N( 0),N(3 )))
  etst(roundx(N( 2),N(7)),PAIR(N( 0),N( 2)))
  etst(roundx(N( 1),N(7)),PAIR(N( 0),N( 1)))
  etst(roundx(N( 0),N(7)),PAIR(N( 0),N( 0)))

  etst(roundx(N(-9),N(8)),PAIR(N(-1),N(-1)))
  etst(roundx(N(-8),N(8)),PAIR(N(-1),N( 0)))
  etst(roundx(N(-7),N(8)),PAIR(N(-1),N( 1)))
  etst(roundx(N(-6),N(8)),PAIR(N(-1),N( 2)))
  etst(roundx(N(-5),N(8)),PAIR(N(-1),N( 3)))
  etst(roundx(N(-4),N(8)),PAIR(N(-0),N(-4)))
  etst(roundx(N(-3),N(8)),PAIR(N(-0),N(-3)))
  etst(roundx(N(-2),N(8)),PAIR(N(-0),N(-2)))
  etst(roundx(N(-1),N(8)),PAIR(N(-0),N(-1)))
  etst(roundx(N( 9),N(8)),PAIR(N( 1),N( 1)))
  etst(roundx(N( 8),N(8)),PAIR(N( 1),N( 0)))
  etst(roundx(N( 7),N(8)),PAIR(N( 1),N(-1)))
  etst(roundx(N( 6),N(8)),PAIR(N( 1),N(-2)))
  etst(roundx(N( 5),N(8)),PAIR(N( 1),N(-3)))
  etst(roundx(N( 4),N(8)),PAIR(N( 0),N( 4)))
  etst(roundx(N( 3),N(8)),PAIR(N( 0),N( 3)))
  etst(roundx(N( 2),N(8)),PAIR(N( 0),N( 2)))
  etst(roundx(N( 1),N(8)),PAIR(N( 0),N( 1)))
  etst(roundx(N( 0),N(8)),PAIR(N( 0),N( 0)))

  etst(roundx(N(-9),N(9)),PAIR(N(-1),N( 0)))
  etst(roundx(N(-8),N(9)),PAIR(N(-1),N( 1)))
  etst(roundx(N(-7),N(9)),PAIR(N(-1),N( 2)))
  etst(roundx(N(-6),N(9)),PAIR(N(-1),N( 3)))
  etst(roundx(N(-5),N(9)),PAIR(N(-1),N( 4)))
  etst(roundx(N(-4),N(9)),PAIR(N(-0),N(-4)))
  etst(roundx(N(-3),N(9)),PAIR(N(-0),N(-3)))
  etst(roundx(N(-2),N(9)),PAIR(N(-0),N(-2)))
  etst(roundx(N(-1),N(9)),PAIR(N(-0),N(-1)))
  etst(roundx(N( 9),N(9)),PAIR(N( 1),N( 0)))
  etst(roundx(N( 8),N(9)),PAIR(N( 1),N(-1)))
  etst(roundx(N( 7),N(9)),PAIR(N( 1),N(-2)))
  etst(roundx(N( 6),N(9)),PAIR(N( 1),N(-3)))
  etst(roundx(N( 5),N(9)),PAIR(N( 1),N(-4)))
  etst(roundx(N( 4),N(9)),PAIR(N( 0),N( 4)))
  etst(roundx(N( 3),N(9)),PAIR(N( 0),N( 3)))
  etst(roundx(N( 2),N(9)),PAIR(N( 0),N( 2)))
  etst(roundx(N( 1),N(9)),PAIR(N( 0),N( 1)))
  etst(roundx(N( 0),N(9)),PAIR(N( 0),N( 0)))

  xtst(divx(TT,TT))
  xtst(divx(TT,FF))
  xtst(divx(TT,I0))
  xtst(divx(TT,P0))
  xtst(divx(TT,M0))
  xtst(divx(TT,B0))
  xtst(divx(FF,TT))
  xtst(divx(FF,FF))
  xtst(divx(FF,I0))
  xtst(divx(FF,P0))
  xtst(divx(FF,M0))
  xtst(divx(FF,B0))
  xtst(divx(I0,TT))
  xtst(divx(I0,FF))
  xtst(divx(I0,P0))
  xtst(divx(I0,M0))
  xtst(divx(I0,B0))
  xtst(divx(P0,TT))
  xtst(divx(P0,FF))
  xtst(divx(P0,I0))
  xtst(divx(P0,P0))
  xtst(divx(P0,M0))
  xtst(divx(P0,B0))
  xtst(divx(M0,TT))
  xtst(divx(M0,FF))
  xtst(divx(M0,I0))
  xtst(divx(M0,P0))
  xtst(divx(M0,M0))
  xtst(divx(M0,B0))
  xtst(divx(B0,TT))
  xtst(divx(B0,FF))
  xtst(divx(B0,I0))
  xtst(divx(B0,P0))
  xtst(divx(B0,M0))
  xtst(divx(B0,B0))
  xtst(divx(I0,Z))
  xtst(divx(I0,M))

  etst(divx(N(-9),N(1)),N(-9))
  etst(divx(N(-8),N(1)),N(-8))
  etst(divx(N(-7),N(1)),N(-7))
  etst(divx(N(-6),N(1)),N(-6))
  etst(divx(N(-5),N(1)),N(-5))
  etst(divx(N(-4),N(1)),N(-4))
  etst(divx(N(-3),N(1)),N(-3))
  etst(divx(N(-2),N(1)),N(-2))
  etst(divx(N(-1),N(1)),N(-1))
  etst(divx(N( 0),N(1)),N( 0))
  etst(divx(N( 1),N(1)),N( 1))
  etst(divx(N( 2),N(1)),N( 2))
  etst(divx(N( 3),N(1)),N( 3))
  etst(divx(N( 4),N(1)),N( 4))
  etst(divx(N( 5),N(1)),N( 5))
  etst(divx(N( 6),N(1)),N( 6))
  etst(divx(N( 7),N(1)),N( 7))
  etst(divx(N( 8),N(1)),N( 8))
  etst(divx(N( 9),N(1)),N( 9))

  etst(divx(N(-9),N(2)),N(-5))
  etst(divx(N(-8),N(2)),N(-4))
  etst(divx(N(-7),N(2)),N(-4))
  etst(divx(N(-6),N(2)),N(-3))
  etst(divx(N(-5),N(2)),N(-3))
  etst(divx(N(-4),N(2)),N(-2))
  etst(divx(N(-3),N(2)),N(-2))
  etst(divx(N(-2),N(2)),N(-1))
  etst(divx(N(-1),N(2)),N(-1))
  etst(divx(N( 0),N(2)),N( 0))
  etst(divx(N( 1),N(2)),N( 0))
  etst(divx(N( 2),N(2)),N( 1))
  etst(divx(N( 3),N(2)),N( 1))
  etst(divx(N( 4),N(2)),N( 2))
  etst(divx(N( 5),N(2)),N( 2))
  etst(divx(N( 6),N(2)),N( 3))
  etst(divx(N( 7),N(2)),N( 3))
  etst(divx(N( 8),N(2)),N( 4))
  etst(divx(N( 9),N(2)),N( 4))

  etst(divx(N(-9),N(3)),N(-3))
  etst(divx(N(-8),N(3)),N(-3))
  etst(divx(N(-7),N(3)),N(-3))
  etst(divx(N(-6),N(3)),N(-2))
  etst(divx(N(-5),N(3)),N(-2))
  etst(divx(N(-4),N(3)),N(-2))
  etst(divx(N(-3),N(3)),N(-1))
  etst(divx(N(-2),N(3)),N(-1))
  etst(divx(N(-1),N(3)),N(-1))
  etst(divx(N( 0),N(3)),N( 0))
  etst(divx(N( 1),N(3)),N( 0))
  etst(divx(N( 2),N(3)),N( 0))
  etst(divx(N( 3),N(3)),N( 1))
  etst(divx(N( 4),N(3)),N( 1))
  etst(divx(N( 5),N(3)),N( 1))
  etst(divx(N( 6),N(3)),N( 2))
  etst(divx(N( 7),N(3)),N( 2))
  etst(divx(N( 8),N(3)),N( 2))
  etst(divx(N( 9),N(3)),N( 3))

  etst(divx(N(-9),N(4)),N(-3))
  etst(divx(N(-8),N(4)),N(-2))
  etst(divx(N(-7),N(4)),N(-2))
  etst(divx(N(-6),N(4)),N(-2))
  etst(divx(N(-5),N(4)),N(-2))
  etst(divx(N(-4),N(4)),N(-1))
  etst(divx(N(-3),N(4)),N(-1))
  etst(divx(N(-2),N(4)),N(-1))
  etst(divx(N(-1),N(4)),N(-1))
  etst(divx(N( 0),N(4)),N( 0))
  etst(divx(N( 1),N(4)),N( 0))
  etst(divx(N( 2),N(4)),N( 0))
  etst(divx(N( 3),N(4)),N( 0))
  etst(divx(N( 4),N(4)),N( 1))
  etst(divx(N( 5),N(4)),N( 1))
  etst(divx(N( 6),N(4)),N( 1))
  etst(divx(N( 7),N(4)),N( 1))
  etst(divx(N( 8),N(4)),N( 2))
  etst(divx(N( 9),N(4)),N( 2))

  etst(divx(N(-9),N(5)),N(-2))
  etst(divx(N(-8),N(5)),N(-2))
  etst(divx(N(-7),N(5)),N(-2))
  etst(divx(N(-6),N(5)),N(-2))
  etst(divx(N(-5),N(5)),N(-1))
  etst(divx(N(-4),N(5)),N(-1))
  etst(divx(N(-3),N(5)),N(-1))
  etst(divx(N(-2),N(5)),N(-1))
  etst(divx(N(-1),N(5)),N(-1))
  etst(divx(N( 0),N(5)),N( 0))
  etst(divx(N( 1),N(5)),N( 0))
  etst(divx(N( 2),N(5)),N( 0))
  etst(divx(N( 3),N(5)),N( 0))
  etst(divx(N( 4),N(5)),N( 0))
  etst(divx(N( 5),N(5)),N( 1))
  etst(divx(N( 6),N(5)),N( 1))
  etst(divx(N( 7),N(5)),N( 1))
  etst(divx(N( 8),N(5)),N( 1))
  etst(divx(N( 9),N(5)),N( 1))

  etst(divx(N(-9),N(6)),N(-2))
  etst(divx(N(-8),N(6)),N(-2))
  etst(divx(N(-7),N(6)),N(-2))
  etst(divx(N(-6),N(6)),N(-1))
  etst(divx(N(-5),N(6)),N(-1))
  etst(divx(N(-4),N(6)),N(-1))
  etst(divx(N(-3),N(6)),N(-1))
  etst(divx(N(-2),N(6)),N(-1))
  etst(divx(N(-1),N(6)),N(-1))
  etst(divx(N( 0),N(6)),N( 0))
  etst(divx(N( 1),N(6)),N( 0))
  etst(divx(N( 2),N(6)),N( 0))
  etst(divx(N( 3),N(6)),N( 0))
  etst(divx(N( 4),N(6)),N( 0))
  etst(divx(N( 5),N(6)),N( 0))
  etst(divx(N( 6),N(6)),N( 1))
  etst(divx(N( 7),N(6)),N( 1))
  etst(divx(N( 8),N(6)),N( 1))
  etst(divx(N( 9),N(6)),N( 1))

  etst(divx(N(-9),N(7)),N(-2))
  etst(divx(N(-8),N(7)),N(-2))
  etst(divx(N(-7),N(7)),N(-1))
  etst(divx(N(-6),N(7)),N(-1))
  etst(divx(N(-5),N(7)),N(-1))
  etst(divx(N(-4),N(7)),N(-1))
  etst(divx(N(-3),N(7)),N(-1))
  etst(divx(N(-2),N(7)),N(-1))
  etst(divx(N(-1),N(7)),N(-1))
  etst(divx(N( 0),N(7)),N( 0))
  etst(divx(N( 1),N(7)),N( 0))
  etst(divx(N( 2),N(7)),N( 0))
  etst(divx(N( 3),N(7)),N( 0))
  etst(divx(N( 4),N(7)),N( 0))
  etst(divx(N( 5),N(7)),N( 0))
  etst(divx(N( 6),N(7)),N( 0))
  etst(divx(N( 7),N(7)),N( 1))
  etst(divx(N( 8),N(7)),N( 1))
  etst(divx(N( 9),N(7)),N( 1))

  etst(divx(N(-9),N(8)),N(-2))
  etst(divx(N(-8),N(8)),N(-1))
  etst(divx(N(-7),N(8)),N(-1))
  etst(divx(N(-6),N(8)),N(-1))
  etst(divx(N(-5),N(8)),N(-1))
  etst(divx(N(-4),N(8)),N(-1))
  etst(divx(N(-3),N(8)),N(-1))
  etst(divx(N(-2),N(8)),N(-1))
  etst(divx(N(-1),N(8)),N(-1))
  etst(divx(N( 0),N(8)),N( 0))
  etst(divx(N( 1),N(8)),N( 0))
  etst(divx(N( 2),N(8)),N( 0))
  etst(divx(N( 3),N(8)),N( 0))
  etst(divx(N( 4),N(8)),N( 0))
  etst(divx(N( 5),N(8)),N( 0))
  etst(divx(N( 6),N(8)),N( 0))
  etst(divx(N( 7),N(8)),N( 0))
  etst(divx(N( 8),N(8)),N( 1))
  etst(divx(N( 9),N(8)),N( 1))

  etst(divx(N(-9),N(9)),N(-1))
  etst(divx(N(-8),N(9)),N(-1))
  etst(divx(N(-7),N(9)),N(-1))
  etst(divx(N(-6),N(9)),N(-1))
  etst(divx(N(-5),N(9)),N(-1))
  etst(divx(N(-4),N(9)),N(-1))
  etst(divx(N(-3),N(9)),N(-1))
  etst(divx(N(-2),N(9)),N(-1))
  etst(divx(N(-1),N(9)),N(-1))
  etst(divx(N( 0),N(9)),N( 0))
  etst(divx(N( 1),N(9)),N( 0))
  etst(divx(N( 2),N(9)),N( 0))
  etst(divx(N( 3),N(9)),N( 0))
  etst(divx(N( 4),N(9)),N( 0))
  etst(divx(N( 5),N(9)),N( 0))
  etst(divx(N( 6),N(9)),N( 0))
  etst(divx(N( 7),N(9)),N( 0))
  etst(divx(N( 8),N(9)),N( 0))
  etst(divx(N( 9),N(9)),N( 1))

  xtst(modx(TT,TT))
  xtst(modx(TT,FF))
  xtst(modx(TT,I0))
  xtst(modx(TT,P0))
  xtst(modx(TT,M0))
  xtst(modx(TT,B0))
  xtst(modx(FF,TT))
  xtst(modx(FF,FF))
  xtst(modx(FF,I0))
  xtst(modx(FF,P0))
  xtst(modx(FF,M0))
  xtst(modx(FF,B0))
  xtst(modx(I0,TT))
  xtst(modx(I0,FF))
  xtst(modx(I0,P0))
  xtst(modx(I0,M0))
  xtst(modx(I0,B0))
  xtst(modx(P0,TT))
  xtst(modx(P0,FF))
  xtst(modx(P0,I0))
  xtst(modx(P0,P0))
  xtst(modx(P0,M0))
  xtst(modx(P0,B0))
  xtst(modx(M0,TT))
  xtst(modx(M0,FF))
  xtst(modx(M0,I0))
  xtst(modx(M0,P0))
  xtst(modx(M0,M0))
  xtst(modx(M0,B0))
  xtst(modx(B0,TT))
  xtst(modx(B0,FF))
  xtst(modx(B0,I0))
  xtst(modx(B0,P0))
  xtst(modx(B0,M0))
  xtst(modx(B0,B0))
  xtst(modx(I0,Z))
  xtst(modx(I0,M))

  etst(modx(N(-9),N(1)),N(0))
  etst(modx(N(-8),N(1)),N(0))
  etst(modx(N(-7),N(1)),N(0))
  etst(modx(N(-6),N(1)),N(0))
  etst(modx(N(-5),N(1)),N(0))
  etst(modx(N(-4),N(1)),N(0))
  etst(modx(N(-3),N(1)),N(0))
  etst(modx(N(-2),N(1)),N(0))
  etst(modx(N(-1),N(1)),N(0))
  etst(modx(N( 0),N(1)),N(0))
  etst(modx(N( 1),N(1)),N(0))
  etst(modx(N( 2),N(1)),N(0))
  etst(modx(N( 3),N(1)),N(0))
  etst(modx(N( 4),N(1)),N(0))
  etst(modx(N( 5),N(1)),N(0))
  etst(modx(N( 6),N(1)),N(0))
  etst(modx(N( 7),N(1)),N(0))
  etst(modx(N( 8),N(1)),N(0))
  etst(modx(N( 9),N(1)),N(0))

  etst(modx(N(-9),N(2)),N(1))
  etst(modx(N(-8),N(2)),N(0))
  etst(modx(N(-7),N(2)),N(1))
  etst(modx(N(-6),N(2)),N(0))
  etst(modx(N(-5),N(2)),N(1))
  etst(modx(N(-4),N(2)),N(0))
  etst(modx(N(-3),N(2)),N(1))
  etst(modx(N(-2),N(2)),N(0))
  etst(modx(N(-1),N(2)),N(1))
  etst(modx(N( 0),N(2)),N(0))
  etst(modx(N( 1),N(2)),N(1))
  etst(modx(N( 2),N(2)),N(0))
  etst(modx(N( 3),N(2)),N(1))
  etst(modx(N( 4),N(2)),N(0))
  etst(modx(N( 5),N(2)),N(1))
  etst(modx(N( 6),N(2)),N(0))
  etst(modx(N( 7),N(2)),N(1))
  etst(modx(N( 8),N(2)),N(0))
  etst(modx(N( 9),N(2)),N(1))

  etst(modx(N(-9),N(3)),N(0))
  etst(modx(N(-8),N(3)),N(1))
  etst(modx(N(-7),N(3)),N(2))
  etst(modx(N(-6),N(3)),N(0))
  etst(modx(N(-5),N(3)),N(1))
  etst(modx(N(-4),N(3)),N(2))
  etst(modx(N(-3),N(3)),N(0))
  etst(modx(N(-2),N(3)),N(1))
  etst(modx(N(-1),N(3)),N(2))
  etst(modx(N( 0),N(3)),N(0))
  etst(modx(N( 1),N(3)),N(1))
  etst(modx(N( 2),N(3)),N(2))
  etst(modx(N( 3),N(3)),N(0))
  etst(modx(N( 4),N(3)),N(1))
  etst(modx(N( 5),N(3)),N(2))
  etst(modx(N( 6),N(3)),N(0))
  etst(modx(N( 7),N(3)),N(1))
  etst(modx(N( 8),N(3)),N(2))
  etst(modx(N( 9),N(3)),N(0))

  etst(modx(N(-9),N(4)),N(3))
  etst(modx(N(-8),N(4)),N(0))
  etst(modx(N(-7),N(4)),N(1))
  etst(modx(N(-6),N(4)),N(2))
  etst(modx(N(-5),N(4)),N(3))
  etst(modx(N(-4),N(4)),N(0))
  etst(modx(N(-3),N(4)),N(1))
  etst(modx(N(-2),N(4)),N(2))
  etst(modx(N(-1),N(4)),N(3))
  etst(modx(N( 0),N(4)),N(0))
  etst(modx(N( 1),N(4)),N(1))
  etst(modx(N( 2),N(4)),N(2))
  etst(modx(N( 3),N(4)),N(3))
  etst(modx(N( 4),N(4)),N(0))
  etst(modx(N( 5),N(4)),N(1))
  etst(modx(N( 6),N(4)),N(2))
  etst(modx(N( 7),N(4)),N(3))
  etst(modx(N( 8),N(4)),N(0))
  etst(modx(N( 9),N(4)),N(1))

  etst(modx(N(-9),N(5)),N(1))
  etst(modx(N(-8),N(5)),N(2))
  etst(modx(N(-7),N(5)),N(3))
  etst(modx(N(-6),N(5)),N(4))
  etst(modx(N(-5),N(5)),N(0))
  etst(modx(N(-4),N(5)),N(1))
  etst(modx(N(-3),N(5)),N(2))
  etst(modx(N(-2),N(5)),N(3))
  etst(modx(N(-1),N(5)),N(4))
  etst(modx(N( 0),N(5)),N(0))
  etst(modx(N( 1),N(5)),N(1))
  etst(modx(N( 2),N(5)),N(2))
  etst(modx(N( 3),N(5)),N(3))
  etst(modx(N( 4),N(5)),N(4))
  etst(modx(N( 5),N(5)),N(0))
  etst(modx(N( 6),N(5)),N(1))
  etst(modx(N( 7),N(5)),N(2))
  etst(modx(N( 8),N(5)),N(3))
  etst(modx(N( 9),N(5)),N(4))

  etst(modx(N(-9),N(6)),N(3))
  etst(modx(N(-8),N(6)),N(4))
  etst(modx(N(-7),N(6)),N(5))
  etst(modx(N(-6),N(6)),N(0))
  etst(modx(N(-5),N(6)),N(1))
  etst(modx(N(-4),N(6)),N(2))
  etst(modx(N(-3),N(6)),N(3))
  etst(modx(N(-2),N(6)),N(4))
  etst(modx(N(-1),N(6)),N(5))
  etst(modx(N( 0),N(6)),N(0))
  etst(modx(N( 1),N(6)),N(1))
  etst(modx(N( 2),N(6)),N(2))
  etst(modx(N( 3),N(6)),N(3))
  etst(modx(N( 4),N(6)),N(4))
  etst(modx(N( 5),N(6)),N(5))
  etst(modx(N( 6),N(6)),N(0))
  etst(modx(N( 7),N(6)),N(1))
  etst(modx(N( 8),N(6)),N(2))
  etst(modx(N( 9),N(6)),N(3))

  etst(modx(N(-9),N(7)),N(5))
  etst(modx(N(-8),N(7)),N(6))
  etst(modx(N(-7),N(7)),N(0))
  etst(modx(N(-6),N(7)),N(1))
  etst(modx(N(-5),N(7)),N(2))
  etst(modx(N(-4),N(7)),N(3))
  etst(modx(N(-3),N(7)),N(4))
  etst(modx(N(-2),N(7)),N(5))
  etst(modx(N(-1),N(7)),N(6))
  etst(modx(N( 0),N(7)),N(0))
  etst(modx(N( 1),N(7)),N(1))
  etst(modx(N( 2),N(7)),N(2))
  etst(modx(N( 3),N(7)),N(3))
  etst(modx(N( 4),N(7)),N(4))
  etst(modx(N( 5),N(7)),N(5))
  etst(modx(N( 6),N(7)),N(6))
  etst(modx(N( 7),N(7)),N(0))
  etst(modx(N( 8),N(7)),N(1))
  etst(modx(N( 9),N(7)),N(2))

  etst(modx(N(-9),N(8)),N(7))
  etst(modx(N(-8),N(8)),N(0))
  etst(modx(N(-7),N(8)),N(1))
  etst(modx(N(-6),N(8)),N(2))
  etst(modx(N(-5),N(8)),N(3))
  etst(modx(N(-4),N(8)),N(4))
  etst(modx(N(-3),N(8)),N(5))
  etst(modx(N(-2),N(8)),N(6))
  etst(modx(N(-1),N(8)),N(7))
  etst(modx(N( 0),N(8)),N(0))
  etst(modx(N( 1),N(8)),N(1))
  etst(modx(N( 2),N(8)),N(2))
  etst(modx(N( 3),N(8)),N(3))
  etst(modx(N( 4),N(8)),N(4))
  etst(modx(N( 5),N(8)),N(5))
  etst(modx(N( 6),N(8)),N(6))
  etst(modx(N( 7),N(8)),N(7))
  etst(modx(N( 8),N(8)),N(0))
  etst(modx(N( 9),N(8)),N(1))

  etst(modx(N(-9),N(9)),N(0))
  etst(modx(N(-8),N(9)),N(1))
  etst(modx(N(-7),N(9)),N(2))
  etst(modx(N(-6),N(9)),N(3))
  etst(modx(N(-5),N(9)),N(4))
  etst(modx(N(-4),N(9)),N(5))
  etst(modx(N(-3),N(9)),N(6))
  etst(modx(N(-2),N(9)),N(7))
  etst(modx(N(-1),N(9)),N(8))
  etst(modx(N( 0),N(9)),N(0))
  etst(modx(N( 1),N(9)),N(1))
  etst(modx(N( 2),N(9)),N(2))
  etst(modx(N( 3),N(9)),N(3))
  etst(modx(N( 4),N(9)),N(4))
  etst(modx(N( 5),N(9)),N(5))
  etst(modx(N( 6),N(9)),N(6))
  etst(modx(N( 7),N(9)),N(7))
  etst(modx(N( 8),N(9)),N(8))
  etst(modx(N( 9),N(9)),N(0))

  ttst(boolp(TT))
  ttst(boolp(FF))
  ftst(boolp(I0))
  ftst(boolp(P0))
  ftst(boolp(M0))
  ftst(boolp(B0))

  ftst(intp(TT))
  ftst(intp(FF))
  ttst(intp(I0))
  ftst(intp(P0))
  ftst(intp(M0))
  ftst(intp(B0))

  ftst(pairp(TT))
  ftst(pairp(FF))
  ftst(pairp(I0))
  ttst(pairp(P0))
  ftst(pairp(M0))
  ftst(pairp(B0))

  ttst(eq(X,raise0(TT)))
  ftst(eq(X,raise0(FF)))
  ftst(eq(X,raise0(I0)))
  ftst(eq(X,raise0(P0)))
  ftst(eq(X,raise0(M0)))
  ftst(eq(X,raise0(B0)))
  ftst(eq(TT,raise0(FF)))
  ftst(eq(FF,raise0(FF)))
  ftst(eq(I0,raise0(FF)))
  ftst(eq(P0,raise0(FF)))
  ftst(eq(M0,raise0(FF)))
  ftst(eq(B0,raise0(FF)))
  etst(raise0(pair(TT,TT)),raise0(pair(TT,TT)))
  utst(raise0(pair(TT,TT)),raise0(pair(TT,FF)))
  utst(raise0(pair(TT,FF)),raise0(pair(TT,TT)))
  etst(raise0(pair(TT,FF)),raise0(pair(TT,FF)))

  ftst(mapp(TT))
  ftst(mapp(FF))
  ftst(mapp(I0))
  ftst(mapp(P0))
  ttst(mapp(M0))
  ftst(mapp(B0))

  ftst(objp(TT))
  ftst(objp(FF))
  ftst(objp(I0))
  ftst(objp(P0))
  ftst(objp(M0))
  ttst(objp(B0))

  xtst(object(TT))
  xtst(object(FF))
  xtst(object(I0))
  xtst(object(M0))
  xtst(object(B0))
  xtst(object(pair(TT,P0)))
  xtst(object(pair(FF,P0)))
  xtst(object(pair(I0,P0)))
  xtst(object(pair(M0,P0)))
  xtst(object(pair(B0,P0)))
  xtst(object(pair(pair(I[1],TT),P0)))
  xtst(object(pair(pair(I[1],FF),P0)))
  xtst(object(pair(pair(I[1],P0),P0)))
  xtst(object(pair(pair(I[1],M0),P0)))
  xtst(object(pair(pair(I[1],B0),P0)))
  xtst(object(pair(pair(TT,I[1]),P0)))
  xtst(object(pair(pair(FF,I[1]),P0)))
  xtst(object(pair(pair(P0,I[1]),P0)))
  xtst(object(pair(pair(M0,I[1]),P0)))
  xtst(object(pair(pair(B0,I[1]),P0)))
  xtst(object(pair(pair(M,I[1]),P0)))
  xtst(object(pair(pair(I[1],M),P0)))
  xtst(object(pair(pair(Z,I[0]),P0)))
  xtst(object(pair(pair(Z,I[1]),P0)))
  xtst(object(pair(pair(Z,I[2]),P0)))
  xtst(object(pair(pair(Z,I[3]),P0)))
  xtst(object(pair(pair(Z,I[4]),P0)))
  ttst(objp(object(pair(pair(I[0],I[5]),P0))))
  ftst(pairp(object(pair(pair(I[0],I[5]),P0))))
  ttst(objp(object(pair(pair(I[0],I[6]),P0))))
  ttst(objp(object(pair(pair(I[0],I[7]),P0))))
  ttst(objp(object(pair(pair(I[1],I[0]),P0))))
  ttst(objp(object(pair(pair(I[1],I[4]),P0))))
  ttst(objp(object(pair(pair(I[1],I[7]),TT))))
  ttst(objp(object(pair(pair(I[1],I[7]),FF))))
  ttst(objp(object(pair(pair(I[1],I[7]),I0))))
  ttst(objp(object(pair(pair(I[1],I[7]),P0))))
  ttst(objp(object(pair(pair(I[1],I[7]),M0))))
  ttst(objp(object(pair(pair(I[1],I[7]),B0))))

  xtst(destruct(TT))
  xtst(destruct(FF))
  xtst(destruct(P0))
  xtst(destruct(M0))
  xtst(destruct(Z))
  xtst(destruct(M))
  xtst(destruct(I0011))
  xtst(destruct(J1010_0101))
  etst(destruct(object(pair(pair(I[0],I[5]),P0))),pair(pair(I[0],I[5]),P0))
  etst(destruct(object(pair(pair(I[0],I[6]),P0))),pair(pair(I[0],I[6]),P0))
  etst(destruct(object(pair(pair(I[0],I[7]),P0))),pair(pair(I[0],I[7]),P0))
  etst(destruct(object(pair(pair(I[1],I[0]),P0))),pair(pair(I[1],I[0]),P0))
  etst(destruct(object(pair(pair(I[1],I[4]),P0))),pair(pair(I[1],I[4]),P0))
  etst(destruct(object(pair(pair(I[1],I[7]),TT))),pair(pair(I[1],I[7]),TT))
  etst(destruct(object(pair(pair(I[1],I[7]),FF))),pair(pair(I[1],I[7]),FF))
  etst(destruct(object(pair(pair(I[1],I[7]),I0))),pair(pair(I[1],I[7]),I0))
  etst(destruct(object(pair(pair(I[1],I[7]),P0))),pair(pair(I[1],I[7]),P0))
  etst(destruct(object(pair(pair(I[1],I[7]),M0))),pair(pair(I[1],I[7]),M0))
  etst(destruct(object(pair(pair(I[1],I[7]),B0))),pair(pair(I[1],I[7]),B0))

  etst(str2vec(""),N(1))
  etst(str2vec("a"),N(353))
  etst(str2vec("ab"),N(90721))
  etst(str2vec("abc"),N(23290465))
  etst(str2vec("abcd"),N(5979202145))
  etst(str2vec("abcdefg"),N(101162102301090401))
  etst(str2vec("abcdefgh"),N(25969838361917219425))
  etst(str2vec("abcdefghi"),N(6666797704897355801185))

  etst(V(),N(1))
  etst(V(a),N(353))
  etst(V(ab),N(90721))
  etst(V(abc),N(23290465))
  etst(V(abcd),N(5979202145))
  etst(V(abcdefg),N(101162102301090401))
  etst(V(abcdefgh),N(25969838361917219425))
  etst(V(abcdefghi),N(6666797704897355801185))

  etst(W(1,),V())
  etst(W(1,a),V(a))
  etst(W(1,ab),V(ab))
  etst(W(1,abc),V(abc))
  etst(W(1,abcd),V(abcd))
  etst(W(1,abcde),V(abcde))
  etst(W(1,abcdef),V(abcdef))
  etst(W(1,abcdefg),V(abcdefg))
  etst(W(1,abcdefgh),V(abcdefgh))
  etst(W(1,abcdefghi),V(abcdefghi))

  utst(W(2,),V())
  utst(W(2,a),V(a))
  utst(W(2,ab),V(ab))
  utst(W(2,abc),V(abc))
  utst(W(2,abcd),V(abcd))
  utst(W(2,abcde),V(abcde))
  utst(W(2,abcdef),V(abcdef))
  utst(W(2,abcdefg),V(abcdefg))
  utst(W(2,abcdefgh),V(abcdefgh))
  utst(W(2,abcdefghi),V(abcdefghi))

  xtst(vector(TT))
  xtst(vector(FF))
  xtst(vector(P0))
  xtst(vector(M0))
  xtst(vector(B0))
  etst(vector(N(0)),N(1))
  etst(vector(N(1)),N(1))
  etst(vector(N(2)),N(1))
  etst(vector(N(255)),N(1))
  etst(vector(N(256)),N(256))
  etst(vector(N(257)),N(257))
  etst(vector(N(511)),N(511))
  etst(vector(N(512)),N(256))
  etst(vector(N(513)),N(257))
  etst(vector(N(767)),N(511))
  etst(vector(intcons(123,intcons(456,I[2]))),intcons(123,intcons(456,I[1])))
  etst(vector(intcons(123,intcons(456,N(767)))),
       intcons(123,intcons(456,N(511))))
  etst(vector(W(2,)),V())
  etst(vector(W(2,a)),V(a))
  etst(vector(W(2,ab)),V(ab))
  etst(vector(W(2,abc)),V(abc))
  etst(vector(W(2,abcd)),V(abcd))
  etst(vector(W(2,abcde)),V(abcde))
  etst(vector(W(2,abcdef)),V(abcdef))
  etst(vector(W(2,abcdefg)),V(abcdefg))
  etst(vector(W(2,abcdefgh)),V(abcdefgh))
  etst(vector(W(2,abcdefghi)),V(abcdefghi))
  etst(vector(W(255,)),V())
  etst(vector(W(255,a)),V(a))
  etst(vector(W(255,ab)),V(ab))
  etst(vector(W(255,abc)),V(abc))
  etst(vector(W(255,abcd)),V(abcd))
  etst(vector(W(255,abcde)),V(abcde))
  etst(vector(W(255,abcdef)),V(abcdef))
  etst(vector(W(255,abcdefg)),V(abcdefg))
  etst(vector(W(255,abcdefgh)),V(abcdefgh))
  etst(vector(W(255,abcdefghi)),V(abcdefghi))

  xtst(vectorNorm(TT))
  xtst(vectorNorm(FF))
  xtst(vectorNorm(P0))
  xtst(vectorNorm(M0))
  xtst(vectorNorm(B0))
  etst(vectorNorm(N(0)),TT)
  etst(vectorNorm(N(1)),N(1))
  etst(vectorNorm(N(2)),TT)
  etst(vectorNorm(N(255)),TT)
  etst(vectorNorm(N(256)),N(256))
  etst(vectorNorm(N(257)),N(257))
  etst(vectorNorm(N(511)),N(511))
  etst(vectorNorm(N(512)),TT)
  etst(vectorNorm(N(513)),TT)
  etst(vectorNorm(N(767)),TT)
  ttst(vectorNorm(intcons(123,intcons(456,I[2]))))
  ttst(vectorNorm(intcons(123,intcons(456,N(767)))))
  ttst(vectorNorm(W(2,)))
  ttst(vectorNorm(W(2,a)))
  ttst(vectorNorm(W(2,ab)))
  ttst(vectorNorm(W(2,abc)))
  ttst(vectorNorm(W(2,abcd)))
  ttst(vectorNorm(W(2,abcde)))
  ttst(vectorNorm(W(2,abcdef)))
  ttst(vectorNorm(W(2,abcdefg)))
  ttst(vectorNorm(W(2,abcdefgh)))
  ttst(vectorNorm(W(2,abcdefghi)))
  ttst(vectorNorm(W(255,)))
  ttst(vectorNorm(W(255,a)))
  ttst(vectorNorm(W(255,ab)))
  ttst(vectorNorm(W(255,abc)))
  ttst(vectorNorm(W(255,abcd)))
  ttst(vectorNorm(W(255,abcde)))
  ttst(vectorNorm(W(255,abcdef)))
  ttst(vectorNorm(W(255,abcdefg)))
  ttst(vectorNorm(W(255,abcdefgh)))
  ttst(vectorNorm(W(255,abcdefghi)))

  xtst(vectorEmpty(TT))
  xtst(vectorEmpty(FF))
  xtst(vectorEmpty(P0))
  xtst(vectorEmpty(M0))
  xtst(vectorEmpty(B0))
  ttst(vectorEmpty(N(0)))
  ttst(vectorEmpty(N(1)))
  ttst(vectorEmpty(N(2)))
  ttst(vectorEmpty(N(255)))
  ftst(vectorEmpty(N(256)))
  ftst(vectorEmpty(N(257)))
  ftst(vectorEmpty(N(511)))
  ftst(vectorEmpty(N(512)))
  ftst(vectorEmpty(N(513)))
  ftst(vectorEmpty(N(767)))
  ftst(vectorEmpty(intcons(123,intcons(456,I[2]))))
  ftst(vectorEmpty(intcons(123,intcons(456,N(767)))))
  ttst(vectorEmpty(intcons(123,intcons(456,M))))

  xtst(vectorLength(TT))
  xtst(vectorLength(FF))
  xtst(vectorLength(P0))
  xtst(vectorLength(M0))
  xtst(vectorLength(B0))
  etst(N(0),vectorLength(N(0)))
  etst(N(0),vectorLength(N(1)))
  etst(N(0),vectorLength(N(2)))
  etst(N(0),vectorLength(N(255)))
  etst(N(1),vectorLength(N(256)))
  etst(N(1),vectorLength(N(257)))
  etst(N(1),vectorLength(N(511)))
  etst(N(1),vectorLength(N(512)))
  etst(N(1),vectorLength(N(513)))
  etst(N(1),vectorLength(N(767)))
  etst(N(1),vectorLength(N(65535)))
  etst(N(2),vectorLength(N(65536)))
  etst(N(3),vectorLength(N(4294967295)))
  etst(N(4),vectorLength(N(4294967296)))
  etst(N(7),vectorLength(N(18446744073709551615)))
  etst(N(8),vectorLength(N(18446744073709551616)))
  etst(N(15),vectorLength(N(340282366920938463463374607431768211455)))
  etst(N(16),vectorLength(N(340282366920938463463374607431768211456)))
  etst(N(0),vectorLength(intcons(123,intcons(456,M))))

  xtst(vectorIndex(TT,TT))
  xtst(vectorIndex(TT,FF))
  xtst(vectorIndex(TT,I0))
  xtst(vectorIndex(TT,P0))
  xtst(vectorIndex(TT,M0))
  xtst(vectorIndex(TT,B0))
  xtst(vectorIndex(FF,TT))
  xtst(vectorIndex(FF,FF))
  xtst(vectorIndex(FF,I0))
  xtst(vectorIndex(FF,P0))
  xtst(vectorIndex(FF,M0))
  xtst(vectorIndex(FF,B0))
  xtst(vectorIndex(I0,TT))
  xtst(vectorIndex(I0,FF))
  xtst(vectorIndex(I0,P0))
  xtst(vectorIndex(I0,M0))
  xtst(vectorIndex(I0,B0))
  xtst(vectorIndex(P0,TT))
  xtst(vectorIndex(P0,FF))
  xtst(vectorIndex(P0,I0))
  xtst(vectorIndex(P0,P0))
  xtst(vectorIndex(P0,M0))
  xtst(vectorIndex(P0,B0))
  xtst(vectorIndex(M0,TT))
  xtst(vectorIndex(M0,FF))
  xtst(vectorIndex(M0,I0))
  xtst(vectorIndex(M0,P0))
  xtst(vectorIndex(M0,M0))
  xtst(vectorIndex(M0,B0))
  xtst(vectorIndex(B0,TT))
  xtst(vectorIndex(B0,FF))
  xtst(vectorIndex(B0,I0))
  xtst(vectorIndex(B0,P0))
  xtst(vectorIndex(B0,M0))
  xtst(vectorIndex(B0,B0))
  xtst(vectorIndex(N(-1),N(-1)))
  xtst(vectorIndex(N(-1),N( 0)))
  xtst(vectorIndex(N(-1),N( 1)))
  xtst(vectorIndex(N(0),N(-1)))
  xtst(vectorIndex(N(0),N( 0)))
  xtst(vectorIndex(N(0),N( 1)))
  xtst(vectorIndex(N(255),N(-1)))
  xtst(vectorIndex(N(255),N( 0)))
  xtst(vectorIndex(N(255),N( 1)))
  xtst(vectorIndex(N(256),N(-1)))
  etst(vectorIndex(N(256),N( 0)),N(0))
  xtst(vectorIndex(N(256),N( 1)))
  xtst(vectorIndex(N(257),N(-1)))
  etst(vectorIndex(N(257),N( 0)),N(1))
  xtst(vectorIndex(N(257),N( 1)))
  xtst(vectorIndex(N(66049),N(-1)))
  etst(vectorIndex(N(66049),N( 0)),N(1))
  etst(vectorIndex(N(66049),N( 1)),N(2))
  xtst(vectorIndex(N(66049),N( 2)))
  xtst(vectorIndex(N(19025181769461858817),N(-1)))
  etst(vectorIndex(N(19025181769461858817),N( 0)),N(1))
  etst(vectorIndex(N(19025181769461858817),N( 1)),N(2))
  etst(vectorIndex(N(19025181769461858817),N( 2)),N(3))
  etst(vectorIndex(N(19025181769461858817),N( 3)),N(4))
  etst(vectorIndex(N(19025181769461858817),N( 4)),N(5))
  etst(vectorIndex(N(19025181769461858817),N( 5)),N(6))
  etst(vectorIndex(N(19025181769461858817),N( 6)),N(7))
  etst(vectorIndex(N(19025181769461858817),N( 7)),N(8))
  xtst(vectorIndex(N(19025181769461858817),N( 8)))
  xtst(vectorIndex(N(4888965617228783485441),N(-1)))
  etst(vectorIndex(N(4888965617228783485441),N( 0)),N(1))
  etst(vectorIndex(N(4888965617228783485441),N( 1)),N(2))
  etst(vectorIndex(N(4888965617228783485441),N( 2)),N(3))
  etst(vectorIndex(N(4888965617228783485441),N( 3)),N(4))
  etst(vectorIndex(N(4888965617228783485441),N( 4)),N(5))
  etst(vectorIndex(N(4888965617228783485441),N( 5)),N(6))
  etst(vectorIndex(N(4888965617228783485441),N( 6)),N(7))
  etst(vectorIndex(N(4888965617228783485441),N( 7)),N(8))
  etst(vectorIndex(N(4888965617228783485441),N( 8)),N(9))
  xtst(vectorIndex(N(4888965617228783485441),N( 9)))
  xtst(vectorIndex(V(abc),N(-1)))
  etst(vectorIndex(V(abc),N(0)),N(97))
  etst(vectorIndex(V(abc),N(1)),N(98))
  etst(vectorIndex(V(abc),N(2)),N(99))
  xtst(vectorIndex(V(abc),N(3)))
  xtst(vectorIndex(W(2,abc),N(-1)))
  etst(vectorIndex(W(2,abc),N(0)),N(97))
  etst(vectorIndex(W(2,abc),N(1)),N(98))
  etst(vectorIndex(W(2,abc),N(2)),N(99))
  xtst(vectorIndex(W(2,abc),N(3)))
  xtst(vectorIndex(W(255,abc),N(-1)))
  etst(vectorIndex(W(255,abc),N(0)),N(97))
  etst(vectorIndex(W(255,abc),N(1)),N(98))
  etst(vectorIndex(W(255,abc),N(2)),N(99))
  xtst(vectorIndex(W(255,abc),N(3)))
  xtst(vectorIndex(V(abcdefghi),N(-1)))
  xtst(vectorIndex(V(abcdefgh) ,N(-1)))
  xtst(vectorIndex(V(abcdefg)  ,N(-1)))
  xtst(vectorIndex(V(abcdef)   ,N(-1)))
  xtst(vectorIndex(V(abcde)    ,N(-1)))
  xtst(vectorIndex(V(abcd)     ,N(-1)))
  xtst(vectorIndex(V(abc)      ,N(-1)))
  xtst(vectorIndex(V(ab)       ,N(-1)))
  xtst(vectorIndex(V(a)        ,N(-1)))
  xtst(vectorIndex(V()         ,N(-1)))
  etst(vectorIndex(V(abcdefghi),N(0)),N(97))
  etst(vectorIndex(V(abcdefgh) ,N(0)),N(97))
  etst(vectorIndex(V(abcdefg)  ,N(0)),N(97))
  etst(vectorIndex(V(abcdef)   ,N(0)),N(97))
  etst(vectorIndex(V(abcde)    ,N(0)),N(97))
  etst(vectorIndex(V(abcd)     ,N(0)),N(97))
  etst(vectorIndex(V(abc)      ,N(0)),N(97))
  etst(vectorIndex(V(ab)       ,N(0)),N(97))
  etst(vectorIndex(V(a)        ,N(0)),N(97))
  xtst(vectorIndex(V()         ,N(0)))
  etst(vectorIndex(V(abcdefghi),N(1)),N(98))
  etst(vectorIndex(V(abcdefgh) ,N(1)),N(98))
  etst(vectorIndex(V(abcdefg)  ,N(1)),N(98))
  etst(vectorIndex(V(abcdef)   ,N(1)),N(98))
  etst(vectorIndex(V(abcde)    ,N(1)),N(98))
  etst(vectorIndex(V(abcd)     ,N(1)),N(98))
  etst(vectorIndex(V(abc)      ,N(1)),N(98))
  etst(vectorIndex(V(ab)       ,N(1)),N(98))
  xtst(vectorIndex(V(a)        ,N(1)))
  xtst(vectorIndex(V()         ,N(1)))
  etst(vectorIndex(V(abcdefghi),N(2)),N(99))
  etst(vectorIndex(V(abcdefgh) ,N(2)),N(99))
  etst(vectorIndex(V(abcdefg)  ,N(2)),N(99))
  etst(vectorIndex(V(abcdef)   ,N(2)),N(99))
  etst(vectorIndex(V(abcde)    ,N(2)),N(99))
  etst(vectorIndex(V(abcd)     ,N(2)),N(99))
  etst(vectorIndex(V(abc)      ,N(2)),N(99))
  xtst(vectorIndex(V(ab)       ,N(2)))
  xtst(vectorIndex(V(a)        ,N(2)))
  xtst(vectorIndex(V()         ,N(2)))
  etst(vectorIndex(V(abcdefghi),N(3)),N(100))
  etst(vectorIndex(V(abcdefgh) ,N(3)),N(100))
  etst(vectorIndex(V(abcdefg)  ,N(3)),N(100))
  etst(vectorIndex(V(abcdef)   ,N(3)),N(100))
  etst(vectorIndex(V(abcde)    ,N(3)),N(100))
  etst(vectorIndex(V(abcd)     ,N(3)),N(100))
  xtst(vectorIndex(V(abc)      ,N(3)))
  xtst(vectorIndex(V(ab)       ,N(3)))
  xtst(vectorIndex(V(a)        ,N(3)))
  xtst(vectorIndex(V()         ,N(3)))
  etst(vectorIndex(V(abcdefghi),N(4)),N(101))
  etst(vectorIndex(V(abcdefgh) ,N(4)),N(101))
  etst(vectorIndex(V(abcdefg)  ,N(4)),N(101))
  etst(vectorIndex(V(abcdef)   ,N(4)),N(101))
  etst(vectorIndex(V(abcde)    ,N(4)),N(101))
  xtst(vectorIndex(V(abcd)     ,N(4)))
  xtst(vectorIndex(V(abc)      ,N(4)))
  xtst(vectorIndex(V(ab)       ,N(4)))
  xtst(vectorIndex(V(a)        ,N(4)))
  xtst(vectorIndex(V()         ,N(4)))
  etst(vectorIndex(V(abcdefghi),N(5)),N(102))
  etst(vectorIndex(V(abcdefgh) ,N(5)),N(102))
  etst(vectorIndex(V(abcdefg)  ,N(5)),N(102))
  etst(vectorIndex(V(abcdef)   ,N(5)),N(102))
  xtst(vectorIndex(V(abcde)    ,N(5)))
  xtst(vectorIndex(V(abcd)     ,N(5)))
  xtst(vectorIndex(V(abc)      ,N(5)))
  xtst(vectorIndex(V(ab)       ,N(5)))
  xtst(vectorIndex(V(a)        ,N(5)))
  xtst(vectorIndex(V()         ,N(5)))
  etst(vectorIndex(V(abcdefghi),N(6)),N(103))
  etst(vectorIndex(V(abcdefgh) ,N(6)),N(103))
  etst(vectorIndex(V(abcdefg)  ,N(6)),N(103))
  xtst(vectorIndex(V(abcdef)   ,N(6)))
  xtst(vectorIndex(V(abcde)    ,N(6)))
  xtst(vectorIndex(V(abcd)     ,N(6)))
  xtst(vectorIndex(V(abc)      ,N(6)))
  xtst(vectorIndex(V(ab)       ,N(6)))
  xtst(vectorIndex(V(a)        ,N(6)))
  xtst(vectorIndex(V()         ,N(6)))
  etst(vectorIndex(V(abcdefghi),N(7)),N(104))
  etst(vectorIndex(V(abcdefgh) ,N(7)),N(104))
  xtst(vectorIndex(V(abcdefg)  ,N(7)))
  xtst(vectorIndex(V(abcdef)   ,N(7)))
  xtst(vectorIndex(V(abcde)    ,N(7)))
  xtst(vectorIndex(V(abcd)     ,N(7)))
  xtst(vectorIndex(V(abc)      ,N(7)))
  xtst(vectorIndex(V(ab)       ,N(7)))
  xtst(vectorIndex(V(a)        ,N(7)))
  xtst(vectorIndex(V()         ,N(7)))
  etst(vectorIndex(V(abcdefghi),N(8)),N(105))
  xtst(vectorIndex(V(abcdefgh) ,N(8)))
  xtst(vectorIndex(V(abcdefg)  ,N(8)))
  xtst(vectorIndex(V(abcdef)   ,N(8)))
  xtst(vectorIndex(V(abcde)    ,N(8)))
  xtst(vectorIndex(V(abcd)     ,N(8)))
  xtst(vectorIndex(V(abc)      ,N(8)))
  xtst(vectorIndex(V(ab)       ,N(8)))
  xtst(vectorIndex(V(a)        ,N(8)))
  xtst(vectorIndex(V()         ,N(8)))
  xtst(vectorIndex(V(abcdefghi),N(9)))
  xtst(vectorIndex(V(abcdefgh) ,N(9)))
  xtst(vectorIndex(V(abcdefg)  ,N(9)))
  xtst(vectorIndex(V(abcdef)   ,N(9)))
  xtst(vectorIndex(V(abcde)    ,N(9)))
  xtst(vectorIndex(V(abcd)     ,N(9)))
  xtst(vectorIndex(V(abc)      ,N(9)))
  xtst(vectorIndex(V(ab)       ,N(9)))
  xtst(vectorIndex(V(a)        ,N(9)))
  xtst(vectorIndex(V()         ,N(9)))
  xtst(vectorIndex(W(2,abcdefghi),N(-1)))
  xtst(vectorIndex(W(2,abcdefgh) ,N(-1)))
  xtst(vectorIndex(W(2,abcdefg)  ,N(-1)))
  xtst(vectorIndex(W(2,abcdef)   ,N(-1)))
  xtst(vectorIndex(W(2,abcde)    ,N(-1)))
  xtst(vectorIndex(W(2,abcd)     ,N(-1)))
  xtst(vectorIndex(W(2,abc)      ,N(-1)))
  xtst(vectorIndex(W(2,ab)       ,N(-1)))
  xtst(vectorIndex(W(2,a)        ,N(-1)))
  xtst(vectorIndex(W(2,)         ,N(-1)))
  etst(vectorIndex(W(2,abcdefghi),N(0)),N(97))
  etst(vectorIndex(W(2,abcdefgh) ,N(0)),N(97))
  etst(vectorIndex(W(2,abcdefg)  ,N(0)),N(97))
  etst(vectorIndex(W(2,abcdef)   ,N(0)),N(97))
  etst(vectorIndex(W(2,abcde)    ,N(0)),N(97))
  etst(vectorIndex(W(2,abcd)     ,N(0)),N(97))
  etst(vectorIndex(W(2,abc)      ,N(0)),N(97))
  etst(vectorIndex(W(2,ab)       ,N(0)),N(97))
  etst(vectorIndex(W(2,a)        ,N(0)),N(97))
  xtst(vectorIndex(W(2,)         ,N(0)))
  etst(vectorIndex(W(2,abcdefghi),N(1)),N(98))
  etst(vectorIndex(W(2,abcdefgh) ,N(1)),N(98))
  etst(vectorIndex(W(2,abcdefg)  ,N(1)),N(98))
  etst(vectorIndex(W(2,abcdef)   ,N(1)),N(98))
  etst(vectorIndex(W(2,abcde)    ,N(1)),N(98))
  etst(vectorIndex(W(2,abcd)     ,N(1)),N(98))
  etst(vectorIndex(W(2,abc)      ,N(1)),N(98))
  etst(vectorIndex(W(2,ab)       ,N(1)),N(98))
  xtst(vectorIndex(W(2,a)        ,N(1)))
  xtst(vectorIndex(W(2,)         ,N(1)))
  etst(vectorIndex(W(2,abcdefghi),N(2)),N(99))
  etst(vectorIndex(W(2,abcdefgh) ,N(2)),N(99))
  etst(vectorIndex(W(2,abcdefg)  ,N(2)),N(99))
  etst(vectorIndex(W(2,abcdef)   ,N(2)),N(99))
  etst(vectorIndex(W(2,abcde)    ,N(2)),N(99))
  etst(vectorIndex(W(2,abcd)     ,N(2)),N(99))
  etst(vectorIndex(W(2,abc)      ,N(2)),N(99))
  xtst(vectorIndex(W(2,ab)       ,N(2)))
  xtst(vectorIndex(W(2,a)        ,N(2)))
  xtst(vectorIndex(W(2,)         ,N(2)))
  etst(vectorIndex(W(2,abcdefghi),N(3)),N(100))
  etst(vectorIndex(W(2,abcdefgh) ,N(3)),N(100))
  etst(vectorIndex(W(2,abcdefg)  ,N(3)),N(100))
  etst(vectorIndex(W(2,abcdef)   ,N(3)),N(100))
  etst(vectorIndex(W(2,abcde)    ,N(3)),N(100))
  etst(vectorIndex(W(2,abcd)     ,N(3)),N(100))
  xtst(vectorIndex(W(2,abc)      ,N(3)))
  xtst(vectorIndex(W(2,ab)       ,N(3)))
  xtst(vectorIndex(W(2,a)        ,N(3)))
  xtst(vectorIndex(W(2,)         ,N(3)))
  etst(vectorIndex(W(2,abcdefghi),N(4)),N(101))
  etst(vectorIndex(W(2,abcdefgh) ,N(4)),N(101))
  etst(vectorIndex(W(2,abcdefg)  ,N(4)),N(101))
  etst(vectorIndex(W(2,abcdef)   ,N(4)),N(101))
  etst(vectorIndex(W(2,abcde)    ,N(4)),N(101))
  xtst(vectorIndex(W(2,abcd)     ,N(4)))
  xtst(vectorIndex(W(2,abc)      ,N(4)))
  xtst(vectorIndex(W(2,ab)       ,N(4)))
  xtst(vectorIndex(W(2,a)        ,N(4)))
  xtst(vectorIndex(W(2,)         ,N(4)))
  etst(vectorIndex(W(2,abcdefghi),N(5)),N(102))
  etst(vectorIndex(W(2,abcdefgh) ,N(5)),N(102))
  etst(vectorIndex(W(2,abcdefg)  ,N(5)),N(102))
  etst(vectorIndex(W(2,abcdef)   ,N(5)),N(102))
  xtst(vectorIndex(W(2,abcde)    ,N(5)))
  xtst(vectorIndex(W(2,abcd)     ,N(5)))
  xtst(vectorIndex(W(2,abc)      ,N(5)))
  xtst(vectorIndex(W(2,ab)       ,N(5)))
  xtst(vectorIndex(W(2,a)        ,N(5)))
  xtst(vectorIndex(W(2,)         ,N(5)))
  etst(vectorIndex(W(2,abcdefghi),N(6)),N(103))
  etst(vectorIndex(W(2,abcdefgh) ,N(6)),N(103))
  etst(vectorIndex(W(2,abcdefg)  ,N(6)),N(103))
  xtst(vectorIndex(W(2,abcdef)   ,N(6)))
  xtst(vectorIndex(W(2,abcde)    ,N(6)))
  xtst(vectorIndex(W(2,abcd)     ,N(6)))
  xtst(vectorIndex(W(2,abc)      ,N(6)))
  xtst(vectorIndex(W(2,ab)       ,N(6)))
  xtst(vectorIndex(W(2,a)        ,N(6)))
  xtst(vectorIndex(W(2,)         ,N(6)))
  etst(vectorIndex(W(2,abcdefghi),N(7)),N(104))
  etst(vectorIndex(W(2,abcdefgh) ,N(7)),N(104))
  xtst(vectorIndex(W(2,abcdefg)  ,N(7)))
  xtst(vectorIndex(W(2,abcdef)   ,N(7)))
  xtst(vectorIndex(W(2,abcde)    ,N(7)))
  xtst(vectorIndex(W(2,abcd)     ,N(7)))
  xtst(vectorIndex(W(2,abc)      ,N(7)))
  xtst(vectorIndex(W(2,ab)       ,N(7)))
  xtst(vectorIndex(W(2,a)        ,N(7)))
  xtst(vectorIndex(W(2,)         ,N(7)))
  etst(vectorIndex(W(2,abcdefghi),N(8)),N(105))
  xtst(vectorIndex(W(2,abcdefgh) ,N(8)))
  xtst(vectorIndex(W(2,abcdefg)  ,N(8)))
  xtst(vectorIndex(W(2,abcdef)   ,N(8)))
  xtst(vectorIndex(W(2,abcde)    ,N(8)))
  xtst(vectorIndex(W(2,abcd)     ,N(8)))
  xtst(vectorIndex(W(2,abc)      ,N(8)))
  xtst(vectorIndex(W(2,ab)       ,N(8)))
  xtst(vectorIndex(W(2,a)        ,N(8)))
  xtst(vectorIndex(W(2,)         ,N(8)))
  xtst(vectorIndex(W(2,abcdefghi),N(9)))
  xtst(vectorIndex(W(2,abcdefgh) ,N(9)))
  xtst(vectorIndex(W(2,abcdefg)  ,N(9)))
  xtst(vectorIndex(W(2,abcdef)   ,N(9)))
  xtst(vectorIndex(W(2,abcde)    ,N(9)))
  xtst(vectorIndex(W(2,abcd)     ,N(9)))
  xtst(vectorIndex(W(2,abc)      ,N(9)))
  xtst(vectorIndex(W(2,ab)       ,N(9)))
  xtst(vectorIndex(W(2,a)        ,N(9)))
  xtst(vectorIndex(W(2,)         ,N(9)))
  xtst(vectorIndex(W(255,abcdefghi),N(-1)))
  xtst(vectorIndex(W(255,abcdefgh) ,N(-1)))
  xtst(vectorIndex(W(255,abcdefg)  ,N(-1)))
  xtst(vectorIndex(W(255,abcdef)   ,N(-1)))
  xtst(vectorIndex(W(255,abcde)    ,N(-1)))
  xtst(vectorIndex(W(255,abcd)     ,N(-1)))
  xtst(vectorIndex(W(255,abc)      ,N(-1)))
  xtst(vectorIndex(W(255,ab)       ,N(-1)))
  xtst(vectorIndex(W(255,a)        ,N(-1)))
  xtst(vectorIndex(W(255,)         ,N(-1)))
  etst(vectorIndex(W(255,abcdefghi),N(0)),N(97))
  etst(vectorIndex(W(255,abcdefgh) ,N(0)),N(97))
  etst(vectorIndex(W(255,abcdefg)  ,N(0)),N(97))
  etst(vectorIndex(W(255,abcdef)   ,N(0)),N(97))
  etst(vectorIndex(W(255,abcde)    ,N(0)),N(97))
  etst(vectorIndex(W(255,abcd)     ,N(0)),N(97))
  etst(vectorIndex(W(255,abc)      ,N(0)),N(97))
  etst(vectorIndex(W(255,ab)       ,N(0)),N(97))
  etst(vectorIndex(W(255,a)        ,N(0)),N(97))
  xtst(vectorIndex(W(255,)         ,N(0)))
  etst(vectorIndex(W(255,abcdefghi),N(1)),N(98))
  etst(vectorIndex(W(255,abcdefgh) ,N(1)),N(98))
  etst(vectorIndex(W(255,abcdefg)  ,N(1)),N(98))
  etst(vectorIndex(W(255,abcdef)   ,N(1)),N(98))
  etst(vectorIndex(W(255,abcde)    ,N(1)),N(98))
  etst(vectorIndex(W(255,abcd)     ,N(1)),N(98))
  etst(vectorIndex(W(255,abc)      ,N(1)),N(98))
  etst(vectorIndex(W(255,ab)       ,N(1)),N(98))
  xtst(vectorIndex(W(255,a)        ,N(1)))
  xtst(vectorIndex(W(255,)         ,N(1)))
  etst(vectorIndex(W(255,abcdefghi),N(2)),N(99))
  etst(vectorIndex(W(255,abcdefgh) ,N(2)),N(99))
  etst(vectorIndex(W(255,abcdefg)  ,N(2)),N(99))
  etst(vectorIndex(W(255,abcdef)   ,N(2)),N(99))
  etst(vectorIndex(W(255,abcde)    ,N(2)),N(99))
  etst(vectorIndex(W(255,abcd)     ,N(2)),N(99))
  etst(vectorIndex(W(255,abc)      ,N(2)),N(99))
  xtst(vectorIndex(W(255,ab)       ,N(2)))
  xtst(vectorIndex(W(255,a)        ,N(2)))
  xtst(vectorIndex(W(255,)         ,N(2)))
  etst(vectorIndex(W(255,abcdefghi),N(3)),N(100))
  etst(vectorIndex(W(255,abcdefgh) ,N(3)),N(100))
  etst(vectorIndex(W(255,abcdefg)  ,N(3)),N(100))
  etst(vectorIndex(W(255,abcdef)   ,N(3)),N(100))
  etst(vectorIndex(W(255,abcde)    ,N(3)),N(100))
  etst(vectorIndex(W(255,abcd)     ,N(3)),N(100))
  xtst(vectorIndex(W(255,abc)      ,N(3)))
  xtst(vectorIndex(W(255,ab)       ,N(3)))
  xtst(vectorIndex(W(255,a)        ,N(3)))
  xtst(vectorIndex(W(255,)         ,N(3)))
  etst(vectorIndex(W(255,abcdefghi),N(4)),N(101))
  etst(vectorIndex(W(255,abcdefgh) ,N(4)),N(101))
  etst(vectorIndex(W(255,abcdefg)  ,N(4)),N(101))
  etst(vectorIndex(W(255,abcdef)   ,N(4)),N(101))
  etst(vectorIndex(W(255,abcde)    ,N(4)),N(101))
  xtst(vectorIndex(W(255,abcd)     ,N(4)))
  xtst(vectorIndex(W(255,abc)      ,N(4)))
  xtst(vectorIndex(W(255,ab)       ,N(4)))
  xtst(vectorIndex(W(255,a)        ,N(4)))
  xtst(vectorIndex(W(255,)         ,N(4)))
  etst(vectorIndex(W(255,abcdefghi),N(5)),N(102))
  etst(vectorIndex(W(255,abcdefgh) ,N(5)),N(102))
  etst(vectorIndex(W(255,abcdefg)  ,N(5)),N(102))
  etst(vectorIndex(W(255,abcdef)   ,N(5)),N(102))
  xtst(vectorIndex(W(255,abcde)    ,N(5)))
  xtst(vectorIndex(W(255,abcd)     ,N(5)))
  xtst(vectorIndex(W(255,abc)      ,N(5)))
  xtst(vectorIndex(W(255,ab)       ,N(5)))
  xtst(vectorIndex(W(255,a)        ,N(5)))
  xtst(vectorIndex(W(255,)         ,N(5)))
  etst(vectorIndex(W(255,abcdefghi),N(6)),N(103))
  etst(vectorIndex(W(255,abcdefgh) ,N(6)),N(103))
  etst(vectorIndex(W(255,abcdefg)  ,N(6)),N(103))
  xtst(vectorIndex(W(255,abcdef)   ,N(6)))
  xtst(vectorIndex(W(255,abcde)    ,N(6)))
  xtst(vectorIndex(W(255,abcd)     ,N(6)))
  xtst(vectorIndex(W(255,abc)      ,N(6)))
  xtst(vectorIndex(W(255,ab)       ,N(6)))
  xtst(vectorIndex(W(255,a)        ,N(6)))
  xtst(vectorIndex(W(255,)         ,N(6)))
  etst(vectorIndex(W(255,abcdefghi),N(7)),N(104))
  etst(vectorIndex(W(255,abcdefgh) ,N(7)),N(104))
  xtst(vectorIndex(W(255,abcdefg)  ,N(7)))
  xtst(vectorIndex(W(255,abcdef)   ,N(7)))
  xtst(vectorIndex(W(255,abcde)    ,N(7)))
  xtst(vectorIndex(W(255,abcd)     ,N(7)))
  xtst(vectorIndex(W(255,abc)      ,N(7)))
  xtst(vectorIndex(W(255,ab)       ,N(7)))
  xtst(vectorIndex(W(255,a)        ,N(7)))
  xtst(vectorIndex(W(255,)         ,N(7)))
  etst(vectorIndex(W(255,abcdefghi),N(8)),N(105))
  xtst(vectorIndex(W(255,abcdefgh) ,N(8)))
  xtst(vectorIndex(W(255,abcdefg)  ,N(8)))
  xtst(vectorIndex(W(255,abcdef)   ,N(8)))
  xtst(vectorIndex(W(255,abcde)    ,N(8)))
  xtst(vectorIndex(W(255,abcd)     ,N(8)))
  xtst(vectorIndex(W(255,abc)      ,N(8)))
  xtst(vectorIndex(W(255,ab)       ,N(8)))
  xtst(vectorIndex(W(255,a)        ,N(8)))
  xtst(vectorIndex(W(255,)         ,N(8)))
  xtst(vectorIndex(W(255,abcdefghi),N(9)))
  xtst(vectorIndex(W(255,abcdefgh) ,N(9)))
  xtst(vectorIndex(W(255,abcdefg)  ,N(9)))
  xtst(vectorIndex(W(255,abcdef)   ,N(9)))
  xtst(vectorIndex(W(255,abcde)    ,N(9)))
  xtst(vectorIndex(W(255,abcd)     ,N(9)))
  xtst(vectorIndex(W(255,abc)      ,N(9)))
  xtst(vectorIndex(W(255,ab)       ,N(9)))
  xtst(vectorIndex(W(255,a)        ,N(9)))
  xtst(vectorIndex(W(255,)         ,N(9)))

  xtst(vectorSuffix(TT,TT))
  xtst(vectorSuffix(TT,FF))
  xtst(vectorSuffix(TT,I0))
  xtst(vectorSuffix(TT,P0))
  xtst(vectorSuffix(TT,M0))
  xtst(vectorSuffix(TT,B0))
  xtst(vectorSuffix(FF,TT))
  xtst(vectorSuffix(FF,FF))
  xtst(vectorSuffix(FF,I0))
  xtst(vectorSuffix(FF,P0))
  xtst(vectorSuffix(FF,M0))
  xtst(vectorSuffix(FF,B0))
  xtst(vectorSuffix(I0,TT))
  xtst(vectorSuffix(I0,FF))
  xtst(vectorSuffix(I0,P0))
  xtst(vectorSuffix(I0,M0))
  xtst(vectorSuffix(I0,B0))
  xtst(vectorSuffix(P0,TT))
  xtst(vectorSuffix(P0,FF))
  xtst(vectorSuffix(P0,I0))
  xtst(vectorSuffix(P0,P0))
  xtst(vectorSuffix(P0,M0))
  xtst(vectorSuffix(P0,B0))
  xtst(vectorSuffix(M0,TT))
  xtst(vectorSuffix(M0,FF))
  xtst(vectorSuffix(M0,I0))
  xtst(vectorSuffix(M0,P0))
  xtst(vectorSuffix(M0,M0))
  xtst(vectorSuffix(M0,B0))
  xtst(vectorSuffix(B0,TT))
  xtst(vectorSuffix(B0,FF))
  xtst(vectorSuffix(B0,I0))
  xtst(vectorSuffix(B0,P0))
  xtst(vectorSuffix(B0,M0))
  xtst(vectorSuffix(B0,B0))
  etst(vectorSuffix(V(abc),N(-1)),V(abc))
  etst(vectorSuffix(V(abc),N( 0)),V(abc))
  etst(vectorSuffix(V(abc),N( 1)),V(bc))
  etst(vectorSuffix(V(abc),N( 2)),V(c))
  etst(vectorSuffix(V(abc),N( 3)),E)
  etst(vectorSuffix(V(abc),N( 4)),E)
  etst(vectorSuffix(V(abcdefg),N(-1)),V(abcdefg))
  etst(vectorSuffix(V(abcdefg),N( 0)),V(abcdefg))
  etst(vectorSuffix(V(abcdefg),N( 1)),V(bcdefg))
  etst(vectorSuffix(V(abcdefg),N( 2)),V(cdefg))
  etst(vectorSuffix(V(abcdefg),N( 3)),V(defg))
  etst(vectorSuffix(V(abcdefg),N( 4)),V(efg))
  etst(vectorSuffix(V(abcdefg),N( 5)),V(fg))
  etst(vectorSuffix(V(abcdefg),N( 6)),V(g))
  etst(vectorSuffix(V(abcdefg),N( 7)),V())
  etst(vectorSuffix(V(abcdefg),N( 8)),V())
  etst(vectorSuffix(V(abcdefg),N( 9)),V())
  etst(vectorSuffix(V(abcdefg),N(10)),V())
  etst(vectorSuffix(V(abcdefgh),N(-1)),V(abcdefgh))
  etst(vectorSuffix(V(abcdefgh),N( 0)),V(abcdefgh))
  etst(vectorSuffix(V(abcdefgh),N( 1)),V(bcdefgh))
  etst(vectorSuffix(V(abcdefgh),N( 2)),V(cdefgh))
  etst(vectorSuffix(V(abcdefgh),N( 3)),V(defgh))
  etst(vectorSuffix(V(abcdefgh),N( 4)),V(efgh))
  etst(vectorSuffix(V(abcdefgh),N( 5)),V(fgh))
  etst(vectorSuffix(V(abcdefgh),N( 6)),V(gh))
  etst(vectorSuffix(V(abcdefgh),N( 7)),V(h))
  etst(vectorSuffix(V(abcdefgh),N( 8)),V())
  etst(vectorSuffix(V(abcdefgh),N( 9)),V())
  etst(vectorSuffix(V(abcdefgh),N(10)),V())
  etst(vectorSuffix(V(abcdefghi),N(-1)),V(abcdefghi))
  etst(vectorSuffix(V(abcdefghi),N( 0)),V(abcdefghi))
  etst(vectorSuffix(V(abcdefghi),N( 1)),V(bcdefghi))
  etst(vectorSuffix(V(abcdefghi),N( 2)),V(cdefghi))
  etst(vectorSuffix(V(abcdefghi),N( 3)),V(defghi))
  etst(vectorSuffix(V(abcdefghi),N( 4)),V(efghi))
  etst(vectorSuffix(V(abcdefghi),N( 5)),V(fghi))
  etst(vectorSuffix(V(abcdefghi),N( 6)),V(ghi))
  etst(vectorSuffix(V(abcdefghi),N( 7)),V(hi))
  etst(vectorSuffix(V(abcdefghi),N( 8)),V(i))
  etst(vectorSuffix(V(abcdefghi),N( 9)),V())
  etst(vectorSuffix(V(abcdefghi),N(10)),V())
  etst(vectorSuffix(N(256),N(-1)),N(256))
  etst(vectorSuffix(N(256),N( 0)),N(256))
  etst(vectorSuffix(N(256),N( 1)),E)
  etst(vectorSuffix(N(256),N( 2)),E)
  etst(vectorSuffix(N(512),N(-1)),N(256))
  etst(vectorSuffix(N(512),N( 0)),N(256))
  etst(vectorSuffix(N(512),N( 1)),E)
  etst(vectorSuffix(N(512),N( 2)),E)
  etst(vectorSuffix(W(2,abcdefg),N(-1)),V(abcdefg))
  etst(vectorSuffix(W(2,abcdefg),N( 0)),V(abcdefg))
  etst(vectorSuffix(W(2,abcdefg),N( 1)),V(bcdefg))
  etst(vectorSuffix(W(2,abcdefg),N( 2)),V(cdefg))
  etst(vectorSuffix(W(2,abcdefg),N( 3)),V(defg))
  etst(vectorSuffix(W(2,abcdefg),N( 4)),V(efg))
  etst(vectorSuffix(W(2,abcdefg),N( 5)),V(fg))
  etst(vectorSuffix(W(2,abcdefg),N( 6)),V(g))
  etst(vectorSuffix(W(2,abcdefg),N( 7)),V())
  etst(vectorSuffix(W(2,abcdefg),N( 8)),V())
  etst(vectorSuffix(W(2,abcdefg),N( 9)),V())
  etst(vectorSuffix(W(2,abcdefg),N(10)),V())
  etst(vectorSuffix(W(2,abcdefgh),N(-1)),V(abcdefgh))
  etst(vectorSuffix(W(2,abcdefgh),N( 0)),V(abcdefgh))
  etst(vectorSuffix(W(2,abcdefgh),N( 1)),V(bcdefgh))
  etst(vectorSuffix(W(2,abcdefgh),N( 2)),V(cdefgh))
  etst(vectorSuffix(W(2,abcdefgh),N( 3)),V(defgh))
  etst(vectorSuffix(W(2,abcdefgh),N( 4)),V(efgh))
  etst(vectorSuffix(W(2,abcdefgh),N( 5)),V(fgh))
  etst(vectorSuffix(W(2,abcdefgh),N( 6)),V(gh))
  etst(vectorSuffix(W(2,abcdefgh),N( 7)),V(h))
  etst(vectorSuffix(W(2,abcdefgh),N( 8)),V())
  etst(vectorSuffix(W(2,abcdefgh),N( 9)),V())
  etst(vectorSuffix(W(2,abcdefgh),N(10)),V())
  etst(vectorSuffix(W(2,abcdefghi),N(-1)),V(abcdefghi))
  etst(vectorSuffix(W(2,abcdefghi),N( 0)),V(abcdefghi))
  etst(vectorSuffix(W(2,abcdefghi),N( 1)),V(bcdefghi))
  etst(vectorSuffix(W(2,abcdefghi),N( 2)),V(cdefghi))
  etst(vectorSuffix(W(2,abcdefghi),N( 3)),V(defghi))
  etst(vectorSuffix(W(2,abcdefghi),N( 4)),V(efghi))
  etst(vectorSuffix(W(2,abcdefghi),N( 5)),V(fghi))
  etst(vectorSuffix(W(2,abcdefghi),N( 6)),V(ghi))
  etst(vectorSuffix(W(2,abcdefghi),N( 7)),V(hi))
  etst(vectorSuffix(W(2,abcdefghi),N( 8)),V(i))
  etst(vectorSuffix(W(2,abcdefghi),N( 9)),V())
  etst(vectorSuffix(W(2,abcdefghi),N(10)),V())
  etst(vectorSuffix(W(255,abcdefg),N(-1)),V(abcdefg))
  etst(vectorSuffix(W(255,abcdefg),N( 0)),V(abcdefg))
  etst(vectorSuffix(W(255,abcdefg),N( 1)),V(bcdefg))
  etst(vectorSuffix(W(255,abcdefg),N( 2)),V(cdefg))
  etst(vectorSuffix(W(255,abcdefg),N( 3)),V(defg))
  etst(vectorSuffix(W(255,abcdefg),N( 4)),V(efg))
  etst(vectorSuffix(W(255,abcdefg),N( 5)),V(fg))
  etst(vectorSuffix(W(255,abcdefg),N( 6)),V(g))
  etst(vectorSuffix(W(255,abcdefg),N( 7)),V())
  etst(vectorSuffix(W(255,abcdefg),N( 8)),V())
  etst(vectorSuffix(W(255,abcdefg),N( 9)),V())
  etst(vectorSuffix(W(255,abcdefg),N(10)),V())
  etst(vectorSuffix(W(255,abcdefgh),N(-1)),V(abcdefgh))
  etst(vectorSuffix(W(255,abcdefgh),N( 0)),V(abcdefgh))
  etst(vectorSuffix(W(255,abcdefgh),N( 1)),V(bcdefgh))
  etst(vectorSuffix(W(255,abcdefgh),N( 2)),V(cdefgh))
  etst(vectorSuffix(W(255,abcdefgh),N( 3)),V(defgh))
  etst(vectorSuffix(W(255,abcdefgh),N( 4)),V(efgh))
  etst(vectorSuffix(W(255,abcdefgh),N( 5)),V(fgh))
  etst(vectorSuffix(W(255,abcdefgh),N( 6)),V(gh))
  etst(vectorSuffix(W(255,abcdefgh),N( 7)),V(h))
  etst(vectorSuffix(W(255,abcdefgh),N( 8)),V())
  etst(vectorSuffix(W(255,abcdefgh),N( 9)),V())
  etst(vectorSuffix(W(255,abcdefgh),N(10)),V())
  etst(vectorSuffix(W(255,abcdefghi),N(-1)),V(abcdefghi))
  etst(vectorSuffix(W(255,abcdefghi),N( 0)),V(abcdefghi))
  etst(vectorSuffix(W(255,abcdefghi),N( 1)),V(bcdefghi))
  etst(vectorSuffix(W(255,abcdefghi),N( 2)),V(cdefghi))
  etst(vectorSuffix(W(255,abcdefghi),N( 3)),V(defghi))
  etst(vectorSuffix(W(255,abcdefghi),N( 4)),V(efghi))
  etst(vectorSuffix(W(255,abcdefghi),N( 5)),V(fghi))
  etst(vectorSuffix(W(255,abcdefghi),N( 6)),V(ghi))
  etst(vectorSuffix(W(255,abcdefghi),N( 7)),V(hi))
  etst(vectorSuffix(W(255,abcdefghi),N( 8)),V(i))
  etst(vectorSuffix(W(255,abcdefghi),N( 9)),V())
  etst(vectorSuffix(W(255,abcdefghi),N(10)),V())

  xtst(vectorPrefix(TT,TT))
  xtst(vectorPrefix(TT,FF))
  xtst(vectorPrefix(TT,I0))
  xtst(vectorPrefix(TT,P0))
  xtst(vectorPrefix(TT,M0))
  xtst(vectorPrefix(TT,B0))
  xtst(vectorPrefix(FF,TT))
  xtst(vectorPrefix(FF,FF))
  xtst(vectorPrefix(FF,I0))
  xtst(vectorPrefix(FF,P0))
  xtst(vectorPrefix(FF,M0))
  xtst(vectorPrefix(FF,B0))
  xtst(vectorPrefix(I0,TT))
  xtst(vectorPrefix(I0,FF))
  xtst(vectorPrefix(I0,P0))
  xtst(vectorPrefix(I0,M0))
  xtst(vectorPrefix(I0,B0))
  xtst(vectorPrefix(P0,TT))
  xtst(vectorPrefix(P0,FF))
  xtst(vectorPrefix(P0,I0))
  xtst(vectorPrefix(P0,P0))
  xtst(vectorPrefix(P0,M0))
  xtst(vectorPrefix(P0,B0))
  xtst(vectorPrefix(M0,TT))
  xtst(vectorPrefix(M0,FF))
  xtst(vectorPrefix(M0,I0))
  xtst(vectorPrefix(M0,P0))
  xtst(vectorPrefix(M0,M0))
  xtst(vectorPrefix(M0,B0))
  xtst(vectorPrefix(B0,TT))
  xtst(vectorPrefix(B0,FF))
  xtst(vectorPrefix(B0,I0))
  xtst(vectorPrefix(B0,P0))
  xtst(vectorPrefix(B0,M0))
  xtst(vectorPrefix(B0,B0))
  etst(vectorPrefix(V(abc),N(-1)),E)
  etst(vectorPrefix(V(abc),N( 0)),E)
  etst(vectorPrefix(V(abc),N( 1)),V(a))
  etst(vectorPrefix(V(abc),N( 2)),V(ab))
  etst(vectorPrefix(V(abc),N( 3)),V(abc))
  etst(vectorPrefix(V(abc),N( 4)),V(abc))
  etst(vectorPrefix(V(abcdefg),N(-1)),E)
  etst(vectorPrefix(V(abcdefg),N( 0)),E)
  etst(vectorPrefix(V(abcdefg),N( 1)),V(a))
  etst(vectorPrefix(V(abcdefg),N( 2)),V(ab))
  etst(vectorPrefix(V(abcdefg),N( 3)),V(abc))
  etst(vectorPrefix(V(abcdefg),N( 4)),V(abcd))
  etst(vectorPrefix(V(abcdefg),N( 5)),V(abcde))
  etst(vectorPrefix(V(abcdefg),N( 6)),V(abcdef))
  etst(vectorPrefix(V(abcdefg),N( 7)),V(abcdefg))
  etst(vectorPrefix(V(abcdefg),N( 8)),V(abcdefg))
  etst(vectorPrefix(V(abcdefg),N( 9)),V(abcdefg))
  etst(vectorPrefix(V(abcdefg),N(10)),V(abcdefg))
  etst(vectorPrefix(V(abcdefgh),N(-1)),E)
  etst(vectorPrefix(V(abcdefgh),N( 0)),E)
  etst(vectorPrefix(V(abcdefgh),N( 1)),V(a))
  etst(vectorPrefix(V(abcdefgh),N( 2)),V(ab))
  etst(vectorPrefix(V(abcdefgh),N( 3)),V(abc))
  etst(vectorPrefix(V(abcdefgh),N( 4)),V(abcd))
  etst(vectorPrefix(V(abcdefgh),N( 5)),V(abcde))
  etst(vectorPrefix(V(abcdefgh),N( 6)),V(abcdef))
  etst(vectorPrefix(V(abcdefgh),N( 7)),V(abcdefg))
  etst(vectorPrefix(V(abcdefgh),N( 8)),V(abcdefgh))
  etst(vectorPrefix(V(abcdefgh),N( 9)),V(abcdefgh))
  etst(vectorPrefix(V(abcdefgh),N(10)),V(abcdefgh))
  etst(vectorPrefix(V(abcdefghi),N(-1)),E)
  etst(vectorPrefix(V(abcdefghi),N( 0)),E)
  etst(vectorPrefix(V(abcdefghi),N( 1)),V(a))
  etst(vectorPrefix(V(abcdefghi),N( 2)),V(ab))
  etst(vectorPrefix(V(abcdefghi),N( 3)),V(abc))
  etst(vectorPrefix(V(abcdefghi),N( 4)),V(abcd))
  etst(vectorPrefix(V(abcdefghi),N( 5)),V(abcde))
  etst(vectorPrefix(V(abcdefghi),N( 6)),V(abcdef))
  etst(vectorPrefix(V(abcdefghi),N( 7)),V(abcdefg))
  etst(vectorPrefix(V(abcdefghi),N( 8)),V(abcdefgh))
  etst(vectorPrefix(V(abcdefghi),N( 9)),V(abcdefghi))
  etst(vectorPrefix(V(abcdefghi),N(10)),V(abcdefghi))
  etst(vectorPrefix(N(256),N(-1)),E)
  etst(vectorPrefix(N(256),N( 0)),E)
  etst(vectorPrefix(N(256),N( 1)),N(256))
  etst(vectorPrefix(N(256),N( 2)),N(256))
  etst(vectorPrefix(N(512),N(-1)),E)
  etst(vectorPrefix(N(512),N( 0)),E)
  etst(vectorPrefix(N(512),N( 1)),N(256))
  etst(vectorPrefix(N(512),N( 2)),N(256))

  etst(vectorPrefix(W(2,abc),N(-1)),E)
  etst(vectorPrefix(W(2,abc),N( 0)),E)
  etst(vectorPrefix(W(2,abc),N( 1)),V(a))
  etst(vectorPrefix(W(2,abc),N( 2)),V(ab))
  etst(vectorPrefix(W(2,abc),N( 3)),V(abc))
  etst(vectorPrefix(W(2,abc),N( 4)),V(abc))
  etst(vectorPrefix(W(2,abcdefg),N(-1)),E)
  etst(vectorPrefix(W(2,abcdefg),N( 0)),E)
  etst(vectorPrefix(W(2,abcdefg),N( 1)),V(a))
  etst(vectorPrefix(W(2,abcdefg),N( 2)),V(ab))
  etst(vectorPrefix(W(2,abcdefg),N( 3)),V(abc))
  etst(vectorPrefix(W(2,abcdefg),N( 4)),V(abcd))
  etst(vectorPrefix(W(2,abcdefg),N( 5)),V(abcde))
  etst(vectorPrefix(W(2,abcdefg),N( 6)),V(abcdef))
  etst(vectorPrefix(W(2,abcdefg),N( 7)),V(abcdefg))
  etst(vectorPrefix(W(2,abcdefg),N( 8)),V(abcdefg))
  etst(vectorPrefix(W(2,abcdefg),N( 9)),V(abcdefg))
  etst(vectorPrefix(W(2,abcdefg),N(10)),V(abcdefg))
  etst(vectorPrefix(W(2,abcdefgh),N(-1)),E)
  etst(vectorPrefix(W(2,abcdefgh),N( 0)),E)
  etst(vectorPrefix(W(2,abcdefgh),N( 1)),V(a))
  etst(vectorPrefix(W(2,abcdefgh),N( 2)),V(ab))
  etst(vectorPrefix(W(2,abcdefgh),N( 3)),V(abc))
  etst(vectorPrefix(W(2,abcdefgh),N( 4)),V(abcd))
  etst(vectorPrefix(W(2,abcdefgh),N( 5)),V(abcde))
  etst(vectorPrefix(W(2,abcdefgh),N( 6)),V(abcdef))
  etst(vectorPrefix(W(2,abcdefgh),N( 7)),V(abcdefg))
  etst(vectorPrefix(W(2,abcdefgh),N( 8)),V(abcdefgh))
  etst(vectorPrefix(W(2,abcdefgh),N( 9)),V(abcdefgh))
  etst(vectorPrefix(W(2,abcdefgh),N(10)),V(abcdefgh))
  etst(vectorPrefix(W(2,abcdefghi),N(-1)),E)
  etst(vectorPrefix(W(2,abcdefghi),N( 0)),E)
  etst(vectorPrefix(W(2,abcdefghi),N( 1)),V(a))
  etst(vectorPrefix(W(2,abcdefghi),N( 2)),V(ab))
  etst(vectorPrefix(W(2,abcdefghi),N( 3)),V(abc))
  etst(vectorPrefix(W(2,abcdefghi),N( 4)),V(abcd))
  etst(vectorPrefix(W(2,abcdefghi),N( 5)),V(abcde))
  etst(vectorPrefix(W(2,abcdefghi),N( 6)),V(abcdef))
  etst(vectorPrefix(W(2,abcdefghi),N( 7)),V(abcdefg))
  etst(vectorPrefix(W(2,abcdefghi),N( 8)),V(abcdefgh))
  etst(vectorPrefix(W(2,abcdefghi),N( 9)),V(abcdefghi))
  etst(vectorPrefix(W(2,abcdefghi),N(10)),V(abcdefghi))

  etst(vectorPrefix(W(255,abc),N(-1)),E)
  etst(vectorPrefix(W(255,abc),N( 0)),E)
  etst(vectorPrefix(W(255,abc),N( 1)),V(a))
  etst(vectorPrefix(W(255,abc),N( 2)),V(ab))
  etst(vectorPrefix(W(255,abc),N( 3)),V(abc))
  etst(vectorPrefix(W(255,abc),N( 4)),V(abc))
  etst(vectorPrefix(W(255,abcdefg),N(-1)),E)
  etst(vectorPrefix(W(255,abcdefg),N( 0)),E)
  etst(vectorPrefix(W(255,abcdefg),N( 1)),V(a))
  etst(vectorPrefix(W(255,abcdefg),N( 2)),V(ab))
  etst(vectorPrefix(W(255,abcdefg),N( 3)),V(abc))
  etst(vectorPrefix(W(255,abcdefg),N( 4)),V(abcd))
  etst(vectorPrefix(W(255,abcdefg),N( 5)),V(abcde))
  etst(vectorPrefix(W(255,abcdefg),N( 6)),V(abcdef))
  etst(vectorPrefix(W(255,abcdefg),N( 7)),V(abcdefg))
  etst(vectorPrefix(W(255,abcdefg),N( 8)),V(abcdefg))
  etst(vectorPrefix(W(255,abcdefg),N( 9)),V(abcdefg))
  etst(vectorPrefix(W(255,abcdefg),N(10)),V(abcdefg))
  etst(vectorPrefix(W(255,abcdefgh),N(-1)),E)
  etst(vectorPrefix(W(255,abcdefgh),N( 0)),E)
  etst(vectorPrefix(W(255,abcdefgh),N( 1)),V(a))
  etst(vectorPrefix(W(255,abcdefgh),N( 2)),V(ab))
  etst(vectorPrefix(W(255,abcdefgh),N( 3)),V(abc))
  etst(vectorPrefix(W(255,abcdefgh),N( 4)),V(abcd))
  etst(vectorPrefix(W(255,abcdefgh),N( 5)),V(abcde))
  etst(vectorPrefix(W(255,abcdefgh),N( 6)),V(abcdef))
  etst(vectorPrefix(W(255,abcdefgh),N( 7)),V(abcdefg))
  etst(vectorPrefix(W(255,abcdefgh),N( 8)),V(abcdefgh))
  etst(vectorPrefix(W(255,abcdefgh),N( 9)),V(abcdefgh))
  etst(vectorPrefix(W(255,abcdefgh),N(10)),V(abcdefgh))
  etst(vectorPrefix(W(255,abcdefghi),N(-1)),E)
  etst(vectorPrefix(W(255,abcdefghi),N( 0)),E)
  etst(vectorPrefix(W(255,abcdefghi),N( 1)),V(a))
  etst(vectorPrefix(W(255,abcdefghi),N( 2)),V(ab))
  etst(vectorPrefix(W(255,abcdefghi),N( 3)),V(abc))
  etst(vectorPrefix(W(255,abcdefghi),N( 4)),V(abcd))
  etst(vectorPrefix(W(255,abcdefghi),N( 5)),V(abcde))
  etst(vectorPrefix(W(255,abcdefghi),N( 6)),V(abcdef))
  etst(vectorPrefix(W(255,abcdefghi),N( 7)),V(abcdefg))
  etst(vectorPrefix(W(255,abcdefghi),N( 8)),V(abcdefgh))
  etst(vectorPrefix(W(255,abcdefghi),N( 9)),V(abcdefghi))
  etst(vectorPrefix(W(255,abcdefghi),N(10)),V(abcdefghi))

  xtst(vectorSubseq(V(abc),TT,TT))
  xtst(vectorSubseq(V(abc),TT,FF))
  xtst(vectorSubseq(V(abc),TT,I0))
  xtst(vectorSubseq(V(abc),TT,P0))
  xtst(vectorSubseq(V(abc),TT,M0))
  xtst(vectorSubseq(V(abc),TT,B0))
  xtst(vectorSubseq(V(abc),FF,TT))
  xtst(vectorSubseq(V(abc),FF,FF))
  xtst(vectorSubseq(V(abc),FF,I0))
  xtst(vectorSubseq(V(abc),FF,P0))
  xtst(vectorSubseq(V(abc),FF,M0))
  xtst(vectorSubseq(V(abc),FF,B0))
  xtst(vectorSubseq(V(abc),I0,TT))
  xtst(vectorSubseq(V(abc),I0,FF))
  xtst(vectorSubseq(V(abc),I0,P0))
  xtst(vectorSubseq(V(abc),I0,M0))
  xtst(vectorSubseq(V(abc),I0,B0))
  xtst(vectorSubseq(V(abc),P0,TT))
  xtst(vectorSubseq(V(abc),P0,FF))
  xtst(vectorSubseq(V(abc),P0,I0))
  xtst(vectorSubseq(V(abc),P0,P0))
  xtst(vectorSubseq(V(abc),P0,M0))
  xtst(vectorSubseq(V(abc),P0,B0))
  xtst(vectorSubseq(V(abc),M0,TT))
  xtst(vectorSubseq(V(abc),M0,FF))
  xtst(vectorSubseq(V(abc),M0,I0))
  xtst(vectorSubseq(V(abc),M0,P0))
  xtst(vectorSubseq(V(abc),M0,M0))
  xtst(vectorSubseq(V(abc),M0,B0))
  xtst(vectorSubseq(V(abc),B0,TT))
  xtst(vectorSubseq(V(abc),B0,FF))
  xtst(vectorSubseq(V(abc),B0,I0))
  xtst(vectorSubseq(V(abc),B0,P0))
  xtst(vectorSubseq(V(abc),B0,M0))
  xtst(vectorSubseq(V(abc),B0,B0))
  xtst(vectorSubseq(TT,N(0),TT))
  xtst(vectorSubseq(TT,N(0),FF))
  xtst(vectorSubseq(TT,N(0),I0))
  xtst(vectorSubseq(TT,N(0),P0))
  xtst(vectorSubseq(TT,N(0),M0))
  xtst(vectorSubseq(TT,N(0),B0))
  xtst(vectorSubseq(FF,N(0),TT))
  xtst(vectorSubseq(FF,N(0),FF))
  xtst(vectorSubseq(FF,N(0),I0))
  xtst(vectorSubseq(FF,N(0),P0))
  xtst(vectorSubseq(FF,N(0),M0))
  xtst(vectorSubseq(FF,N(0),B0))
  xtst(vectorSubseq(I0,N(0),TT))
  xtst(vectorSubseq(I0,N(0),FF))
  xtst(vectorSubseq(I0,N(0),P0))
  xtst(vectorSubseq(I0,N(0),M0))
  xtst(vectorSubseq(I0,N(0),B0))
  xtst(vectorSubseq(P0,N(0),TT))
  xtst(vectorSubseq(P0,N(0),FF))
  xtst(vectorSubseq(P0,N(0),I0))
  xtst(vectorSubseq(P0,N(0),P0))
  xtst(vectorSubseq(P0,N(0),M0))
  xtst(vectorSubseq(P0,N(0),B0))
  xtst(vectorSubseq(M0,N(0),TT))
  xtst(vectorSubseq(M0,N(0),FF))
  xtst(vectorSubseq(M0,N(0),I0))
  xtst(vectorSubseq(M0,N(0),P0))
  xtst(vectorSubseq(M0,N(0),M0))
  xtst(vectorSubseq(M0,N(0),B0))
  xtst(vectorSubseq(B0,N(0),TT))
  xtst(vectorSubseq(B0,N(0),FF))
  xtst(vectorSubseq(B0,N(0),I0))
  xtst(vectorSubseq(B0,N(0),P0))
  xtst(vectorSubseq(B0,N(0),M0))
  xtst(vectorSubseq(B0,N(0),B0))
  xtst(vectorSubseq(TT,TT,N(2)))
  xtst(vectorSubseq(TT,FF,N(2)))
  xtst(vectorSubseq(TT,I0,N(2)))
  xtst(vectorSubseq(TT,P0,N(2)))
  xtst(vectorSubseq(TT,M0,N(2)))
  xtst(vectorSubseq(TT,B0,N(2)))
  xtst(vectorSubseq(FF,TT,N(2)))
  xtst(vectorSubseq(FF,FF,N(2)))
  xtst(vectorSubseq(FF,I0,N(2)))
  xtst(vectorSubseq(FF,P0,N(2)))
  xtst(vectorSubseq(FF,M0,N(2)))
  xtst(vectorSubseq(FF,B0,N(2)))
  xtst(vectorSubseq(I0,TT,N(2)))
  xtst(vectorSubseq(I0,FF,N(2)))
  xtst(vectorSubseq(I0,P0,N(2)))
  xtst(vectorSubseq(I0,M0,N(2)))
  xtst(vectorSubseq(I0,B0,N(2)))
  xtst(vectorSubseq(P0,TT,N(2)))
  xtst(vectorSubseq(P0,FF,N(2)))
  xtst(vectorSubseq(P0,I0,N(2)))
  xtst(vectorSubseq(P0,P0,N(2)))
  xtst(vectorSubseq(P0,M0,N(2)))
  xtst(vectorSubseq(P0,B0,N(2)))
  xtst(vectorSubseq(M0,TT,N(2)))
  xtst(vectorSubseq(M0,FF,N(2)))
  xtst(vectorSubseq(M0,I0,N(2)))
  xtst(vectorSubseq(M0,P0,N(2)))
  xtst(vectorSubseq(M0,M0,N(2)))
  xtst(vectorSubseq(M0,B0,N(2)))
  xtst(vectorSubseq(B0,TT,N(2)))
  xtst(vectorSubseq(B0,FF,N(2)))
  xtst(vectorSubseq(B0,I0,N(2)))
  xtst(vectorSubseq(B0,P0,N(2)))
  xtst(vectorSubseq(B0,M0,N(2)))
  xtst(vectorSubseq(B0,B0,N(2)))

  etst(vectorSubseq(V(),N(-1),N(-1)),V())
  etst(vectorSubseq(V(),N(-1),N( 0)),V())
  etst(vectorSubseq(V(),N(-1),N( 1)),V())
  etst(vectorSubseq(V(),N(-1),N( 2)),V())
  etst(vectorSubseq(V(),N(-1),N( 3)),V())
  etst(vectorSubseq(V(),N(-1),N( 4)),V())
  etst(vectorSubseq(V(),N(-1),N( 5)),V())
  etst(vectorSubseq(V(),N(-1),N( 6)),V())
  etst(vectorSubseq(V(),N(-1),N( 7)),V())
  etst(vectorSubseq(V(),N(-1),N( 8)),V())
  etst(vectorSubseq(V(),N(-1),N( 9)),V())
  etst(vectorSubseq(V(),N(-1),N(10)),V())

  etst(vectorSubseq(V(),N( 0),N(-1)),V())
  etst(vectorSubseq(V(),N( 0),N( 0)),V())
  etst(vectorSubseq(V(),N( 0),N( 1)),V())
  etst(vectorSubseq(V(),N( 0),N( 2)),V())
  etst(vectorSubseq(V(),N( 0),N( 3)),V())
  etst(vectorSubseq(V(),N( 0),N( 4)),V())
  etst(vectorSubseq(V(),N( 0),N( 5)),V())
  etst(vectorSubseq(V(),N( 0),N( 6)),V())
  etst(vectorSubseq(V(),N( 0),N( 7)),V())
  etst(vectorSubseq(V(),N( 0),N( 8)),V())
  etst(vectorSubseq(V(),N( 0),N( 9)),V())
  etst(vectorSubseq(V(),N( 0),N(10)),V())

  etst(vectorSubseq(V(),N( 1),N(-1)),V())
  etst(vectorSubseq(V(),N( 1),N( 0)),V())
  etst(vectorSubseq(V(),N( 1),N( 1)),V())
  etst(vectorSubseq(V(),N( 1),N( 2)),V())
  etst(vectorSubseq(V(),N( 1),N( 3)),V())
  etst(vectorSubseq(V(),N( 1),N( 4)),V())
  etst(vectorSubseq(V(),N( 1),N( 5)),V())
  etst(vectorSubseq(V(),N( 1),N( 6)),V())
  etst(vectorSubseq(V(),N( 1),N( 7)),V())
  etst(vectorSubseq(V(),N( 1),N( 8)),V())
  etst(vectorSubseq(V(),N( 1),N( 9)),V())
  etst(vectorSubseq(V(),N( 1),N(10)),V())

  etst(vectorSubseq(V(),N( 2),N(-1)),V())
  etst(vectorSubseq(V(),N( 2),N( 0)),V())
  etst(vectorSubseq(V(),N( 2),N( 1)),V())
  etst(vectorSubseq(V(),N( 2),N( 2)),V())
  etst(vectorSubseq(V(),N( 2),N( 3)),V())
  etst(vectorSubseq(V(),N( 2),N( 4)),V())
  etst(vectorSubseq(V(),N( 2),N( 5)),V())
  etst(vectorSubseq(V(),N( 2),N( 6)),V())
  etst(vectorSubseq(V(),N( 2),N( 7)),V())
  etst(vectorSubseq(V(),N( 2),N( 8)),V())
  etst(vectorSubseq(V(),N( 2),N( 9)),V())
  etst(vectorSubseq(V(),N( 2),N(10)),V())

  etst(vectorSubseq(V(),N( 3),N(-1)),V())
  etst(vectorSubseq(V(),N( 3),N( 0)),V())
  etst(vectorSubseq(V(),N( 3),N( 1)),V())
  etst(vectorSubseq(V(),N( 3),N( 2)),V())
  etst(vectorSubseq(V(),N( 3),N( 3)),V())
  etst(vectorSubseq(V(),N( 3),N( 4)),V())
  etst(vectorSubseq(V(),N( 3),N( 5)),V())
  etst(vectorSubseq(V(),N( 3),N( 6)),V())
  etst(vectorSubseq(V(),N( 3),N( 7)),V())
  etst(vectorSubseq(V(),N( 3),N( 8)),V())
  etst(vectorSubseq(V(),N( 3),N( 9)),V())
  etst(vectorSubseq(V(),N( 3),N(10)),V())

  etst(vectorSubseq(V(),N( 4),N(-1)),V())
  etst(vectorSubseq(V(),N( 4),N( 0)),V())
  etst(vectorSubseq(V(),N( 4),N( 1)),V())
  etst(vectorSubseq(V(),N( 4),N( 2)),V())
  etst(vectorSubseq(V(),N( 4),N( 3)),V())
  etst(vectorSubseq(V(),N( 4),N( 4)),V())
  etst(vectorSubseq(V(),N( 4),N( 5)),V())
  etst(vectorSubseq(V(),N( 4),N( 6)),V())
  etst(vectorSubseq(V(),N( 4),N( 7)),V())
  etst(vectorSubseq(V(),N( 4),N( 8)),V())
  etst(vectorSubseq(V(),N( 4),N( 9)),V())
  etst(vectorSubseq(V(),N( 4),N(10)),V())

  etst(vectorSubseq(V(),N( 5),N(-1)),V())
  etst(vectorSubseq(V(),N( 5),N( 0)),V())
  etst(vectorSubseq(V(),N( 5),N( 1)),V())
  etst(vectorSubseq(V(),N( 5),N( 2)),V())
  etst(vectorSubseq(V(),N( 5),N( 3)),V())
  etst(vectorSubseq(V(),N( 5),N( 4)),V())
  etst(vectorSubseq(V(),N( 5),N( 5)),V())
  etst(vectorSubseq(V(),N( 5),N( 6)),V())
  etst(vectorSubseq(V(),N( 5),N( 7)),V())
  etst(vectorSubseq(V(),N( 5),N( 8)),V())
  etst(vectorSubseq(V(),N( 5),N( 9)),V())
  etst(vectorSubseq(V(),N( 5),N(10)),V())

  etst(vectorSubseq(V(),N( 6),N(-1)),V())
  etst(vectorSubseq(V(),N( 6),N( 0)),V())
  etst(vectorSubseq(V(),N( 6),N( 1)),V())
  etst(vectorSubseq(V(),N( 6),N( 2)),V())
  etst(vectorSubseq(V(),N( 6),N( 3)),V())
  etst(vectorSubseq(V(),N( 6),N( 4)),V())
  etst(vectorSubseq(V(),N( 6),N( 5)),V())
  etst(vectorSubseq(V(),N( 6),N( 6)),V())
  etst(vectorSubseq(V(),N( 6),N( 7)),V())
  etst(vectorSubseq(V(),N( 6),N( 8)),V())
  etst(vectorSubseq(V(),N( 6),N( 9)),V())
  etst(vectorSubseq(V(),N( 6),N(10)),V())

  etst(vectorSubseq(V(),N( 7),N(-1)),V())
  etst(vectorSubseq(V(),N( 7),N( 0)),V())
  etst(vectorSubseq(V(),N( 7),N( 1)),V())
  etst(vectorSubseq(V(),N( 7),N( 2)),V())
  etst(vectorSubseq(V(),N( 7),N( 3)),V())
  etst(vectorSubseq(V(),N( 7),N( 4)),V())
  etst(vectorSubseq(V(),N( 7),N( 5)),V())
  etst(vectorSubseq(V(),N( 7),N( 6)),V())
  etst(vectorSubseq(V(),N( 7),N( 7)),V())
  etst(vectorSubseq(V(),N( 7),N( 8)),V())
  etst(vectorSubseq(V(),N( 7),N( 9)),V())
  etst(vectorSubseq(V(),N( 7),N(10)),V())

  etst(vectorSubseq(V(),N( 8),N(-1)),V())
  etst(vectorSubseq(V(),N( 8),N( 0)),V())
  etst(vectorSubseq(V(),N( 8),N( 1)),V())
  etst(vectorSubseq(V(),N( 8),N( 2)),V())
  etst(vectorSubseq(V(),N( 8),N( 3)),V())
  etst(vectorSubseq(V(),N( 8),N( 4)),V())
  etst(vectorSubseq(V(),N( 8),N( 5)),V())
  etst(vectorSubseq(V(),N( 8),N( 6)),V())
  etst(vectorSubseq(V(),N( 8),N( 7)),V())
  etst(vectorSubseq(V(),N( 8),N( 8)),V())
  etst(vectorSubseq(V(),N( 8),N( 9)),V())
  etst(vectorSubseq(V(),N( 8),N(10)),V())

  etst(vectorSubseq(V(),N( 9),N(-1)),V())
  etst(vectorSubseq(V(),N( 9),N( 0)),V())
  etst(vectorSubseq(V(),N( 9),N( 1)),V())
  etst(vectorSubseq(V(),N( 9),N( 2)),V())
  etst(vectorSubseq(V(),N( 9),N( 3)),V())
  etst(vectorSubseq(V(),N( 9),N( 4)),V())
  etst(vectorSubseq(V(),N( 9),N( 5)),V())
  etst(vectorSubseq(V(),N( 9),N( 6)),V())
  etst(vectorSubseq(V(),N( 9),N( 7)),V())
  etst(vectorSubseq(V(),N( 9),N( 8)),V())
  etst(vectorSubseq(V(),N( 9),N( 9)),V())
  etst(vectorSubseq(V(),N( 9),N(10)),V())

  etst(vectorSubseq(V(),N(10),N(-1)),V())
  etst(vectorSubseq(V(),N(10),N( 0)),V())
  etst(vectorSubseq(V(),N(10),N( 1)),V())
  etst(vectorSubseq(V(),N(10),N( 2)),V())
  etst(vectorSubseq(V(),N(10),N( 3)),V())
  etst(vectorSubseq(V(),N(10),N( 4)),V())
  etst(vectorSubseq(V(),N(10),N( 5)),V())
  etst(vectorSubseq(V(),N(10),N( 6)),V())
  etst(vectorSubseq(V(),N(10),N( 7)),V())
  etst(vectorSubseq(V(),N(10),N( 8)),V())
  etst(vectorSubseq(V(),N(10),N( 9)),V())
  etst(vectorSubseq(V(),N(10),N(10)),V())

  etst(vectorSubseq(V(abcdefghi),N(-1),N(-1)),V())
  etst(vectorSubseq(V(abcdefghi),N(-1),N( 0)),V())
  etst(vectorSubseq(V(abcdefghi),N(-1),N( 1)),V(a))
  etst(vectorSubseq(V(abcdefghi),N(-1),N( 2)),V(ab))
  etst(vectorSubseq(V(abcdefghi),N(-1),N( 3)),V(abc))
  etst(vectorSubseq(V(abcdefghi),N(-1),N( 4)),V(abcd))
  etst(vectorSubseq(V(abcdefghi),N(-1),N( 5)),V(abcde))
  etst(vectorSubseq(V(abcdefghi),N(-1),N( 6)),V(abcdef))
  etst(vectorSubseq(V(abcdefghi),N(-1),N( 7)),V(abcdefg))
  etst(vectorSubseq(V(abcdefghi),N(-1),N( 8)),V(abcdefgh))
  etst(vectorSubseq(V(abcdefghi),N(-1),N( 9)),V(abcdefghi))
  etst(vectorSubseq(V(abcdefghi),N(-1),N(10)),V(abcdefghi))
  etst(vectorSubseq(V(abcdefghi),N( 0),N(-1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 0),N( 0)),V())
  etst(vectorSubseq(V(abcdefghi),N( 0),N( 1)),V(a))
  etst(vectorSubseq(V(abcdefghi),N( 0),N( 2)),V(ab))
  etst(vectorSubseq(V(abcdefghi),N( 0),N( 3)),V(abc))
  etst(vectorSubseq(V(abcdefghi),N( 0),N( 4)),V(abcd))
  etst(vectorSubseq(V(abcdefghi),N( 0),N( 5)),V(abcde))
  etst(vectorSubseq(V(abcdefghi),N( 0),N( 6)),V(abcdef))
  etst(vectorSubseq(V(abcdefghi),N( 0),N( 7)),V(abcdefg))
  etst(vectorSubseq(V(abcdefghi),N( 0),N( 8)),V(abcdefgh))
  etst(vectorSubseq(V(abcdefghi),N( 0),N( 9)),V(abcdefghi))
  etst(vectorSubseq(V(abcdefghi),N( 0),N(10)),V(abcdefghi))
  etst(vectorSubseq(V(abcdefghi),N( 1),N(-1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 1),N( 0)),V())
  etst(vectorSubseq(V(abcdefghi),N( 1),N( 1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 1),N( 2)),V(b))
  etst(vectorSubseq(V(abcdefghi),N( 1),N( 3)),V(bc))
  etst(vectorSubseq(V(abcdefghi),N( 1),N( 4)),V(bcd))
  etst(vectorSubseq(V(abcdefghi),N( 1),N( 5)),V(bcde))
  etst(vectorSubseq(V(abcdefghi),N( 1),N( 6)),V(bcdef))
  etst(vectorSubseq(V(abcdefghi),N( 1),N( 7)),V(bcdefg))
  etst(vectorSubseq(V(abcdefghi),N( 1),N( 8)),V(bcdefgh))
  etst(vectorSubseq(V(abcdefghi),N( 1),N( 9)),V(bcdefghi))
  etst(vectorSubseq(V(abcdefghi),N( 1),N(10)),V(bcdefghi))
  etst(vectorSubseq(V(abcdefghi),N( 2),N(-1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 2),N( 0)),V())
  etst(vectorSubseq(V(abcdefghi),N( 2),N( 1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 2),N( 2)),V())
  etst(vectorSubseq(V(abcdefghi),N( 2),N( 3)),V(c))
  etst(vectorSubseq(V(abcdefghi),N( 2),N( 4)),V(cd))
  etst(vectorSubseq(V(abcdefghi),N( 2),N( 5)),V(cde))
  etst(vectorSubseq(V(abcdefghi),N( 2),N( 6)),V(cdef))
  etst(vectorSubseq(V(abcdefghi),N( 2),N( 7)),V(cdefg))
  etst(vectorSubseq(V(abcdefghi),N( 2),N( 8)),V(cdefgh))
  etst(vectorSubseq(V(abcdefghi),N( 2),N( 9)),V(cdefghi))
  etst(vectorSubseq(V(abcdefghi),N( 2),N(10)),V(cdefghi))
  etst(vectorSubseq(V(abcdefghi),N( 3),N(-1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 3),N( 0)),V())
  etst(vectorSubseq(V(abcdefghi),N( 3),N( 1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 3),N( 2)),V())
  etst(vectorSubseq(V(abcdefghi),N( 3),N( 3)),V())
  etst(vectorSubseq(V(abcdefghi),N( 3),N( 4)),V(d))
  etst(vectorSubseq(V(abcdefghi),N( 3),N( 5)),V(de))
  etst(vectorSubseq(V(abcdefghi),N( 3),N( 6)),V(def))
  etst(vectorSubseq(V(abcdefghi),N( 3),N( 7)),V(defg))
  etst(vectorSubseq(V(abcdefghi),N( 3),N( 8)),V(defgh))
  etst(vectorSubseq(V(abcdefghi),N( 3),N( 9)),V(defghi))
  etst(vectorSubseq(V(abcdefghi),N( 3),N(10)),V(defghi))
  etst(vectorSubseq(V(abcdefghi),N( 4),N(-1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 4),N( 0)),V())
  etst(vectorSubseq(V(abcdefghi),N( 4),N( 1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 4),N( 2)),V())
  etst(vectorSubseq(V(abcdefghi),N( 4),N( 3)),V())
  etst(vectorSubseq(V(abcdefghi),N( 4),N( 4)),V())
  etst(vectorSubseq(V(abcdefghi),N( 4),N( 5)),V(e))
  etst(vectorSubseq(V(abcdefghi),N( 4),N( 6)),V(ef))
  etst(vectorSubseq(V(abcdefghi),N( 4),N( 7)),V(efg))
  etst(vectorSubseq(V(abcdefghi),N( 4),N( 8)),V(efgh))
  etst(vectorSubseq(V(abcdefghi),N( 4),N( 9)),V(efghi))
  etst(vectorSubseq(V(abcdefghi),N( 4),N(10)),V(efghi))
  etst(vectorSubseq(V(abcdefghi),N( 5),N(-1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 5),N( 0)),V())
  etst(vectorSubseq(V(abcdefghi),N( 5),N( 1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 5),N( 2)),V())
  etst(vectorSubseq(V(abcdefghi),N( 5),N( 3)),V())
  etst(vectorSubseq(V(abcdefghi),N( 5),N( 4)),V())
  etst(vectorSubseq(V(abcdefghi),N( 5),N( 5)),V())
  etst(vectorSubseq(V(abcdefghi),N( 5),N( 6)),V(f))
  etst(vectorSubseq(V(abcdefghi),N( 5),N( 7)),V(fg))
  etst(vectorSubseq(V(abcdefghi),N( 5),N( 8)),V(fgh))
  etst(vectorSubseq(V(abcdefghi),N( 5),N( 9)),V(fghi))
  etst(vectorSubseq(V(abcdefghi),N( 5),N(10)),V(fghi))
  etst(vectorSubseq(V(abcdefghi),N( 6),N(-1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 6),N( 0)),V())
  etst(vectorSubseq(V(abcdefghi),N( 6),N( 1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 6),N( 2)),V())
  etst(vectorSubseq(V(abcdefghi),N( 6),N( 3)),V())
  etst(vectorSubseq(V(abcdefghi),N( 6),N( 4)),V())
  etst(vectorSubseq(V(abcdefghi),N( 6),N( 5)),V())
  etst(vectorSubseq(V(abcdefghi),N( 6),N( 6)),V())
  etst(vectorSubseq(V(abcdefghi),N( 6),N( 7)),V(g))
  etst(vectorSubseq(V(abcdefghi),N( 6),N( 8)),V(gh))
  etst(vectorSubseq(V(abcdefghi),N( 6),N( 9)),V(ghi))
  etst(vectorSubseq(V(abcdefghi),N( 6),N(10)),V(ghi))
  etst(vectorSubseq(V(abcdefghi),N( 7),N(-1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 7),N( 0)),V())
  etst(vectorSubseq(V(abcdefghi),N( 7),N( 1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 7),N( 2)),V())
  etst(vectorSubseq(V(abcdefghi),N( 7),N( 3)),V())
  etst(vectorSubseq(V(abcdefghi),N( 7),N( 4)),V())
  etst(vectorSubseq(V(abcdefghi),N( 7),N( 5)),V())
  etst(vectorSubseq(V(abcdefghi),N( 7),N( 6)),V())
  etst(vectorSubseq(V(abcdefghi),N( 7),N( 7)),V())
  etst(vectorSubseq(V(abcdefghi),N( 7),N( 8)),V(h))
  etst(vectorSubseq(V(abcdefghi),N( 7),N( 9)),V(hi))
  etst(vectorSubseq(V(abcdefghi),N( 7),N(10)),V(hi))
  etst(vectorSubseq(V(abcdefghi),N( 8),N(-1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 8),N( 0)),V())
  etst(vectorSubseq(V(abcdefghi),N( 8),N( 1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 8),N( 2)),V())
  etst(vectorSubseq(V(abcdefghi),N( 8),N( 3)),V())
  etst(vectorSubseq(V(abcdefghi),N( 8),N( 4)),V())
  etst(vectorSubseq(V(abcdefghi),N( 8),N( 5)),V())
  etst(vectorSubseq(V(abcdefghi),N( 8),N( 6)),V())
  etst(vectorSubseq(V(abcdefghi),N( 8),N( 7)),V())
  etst(vectorSubseq(V(abcdefghi),N( 8),N( 8)),V())
  etst(vectorSubseq(V(abcdefghi),N( 8),N( 9)),V(i))
  etst(vectorSubseq(V(abcdefghi),N( 8),N(10)),V(i))
  etst(vectorSubseq(V(abcdefghi),N( 9),N(-1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 9),N( 0)),V())
  etst(vectorSubseq(V(abcdefghi),N( 9),N( 1)),V())
  etst(vectorSubseq(V(abcdefghi),N( 9),N( 2)),V())
  etst(vectorSubseq(V(abcdefghi),N( 9),N( 3)),V())
  etst(vectorSubseq(V(abcdefghi),N( 9),N( 4)),V())
  etst(vectorSubseq(V(abcdefghi),N( 9),N( 5)),V())
  etst(vectorSubseq(V(abcdefghi),N( 9),N( 6)),V())
  etst(vectorSubseq(V(abcdefghi),N( 9),N( 7)),V())
  etst(vectorSubseq(V(abcdefghi),N( 9),N( 8)),V())
  etst(vectorSubseq(V(abcdefghi),N( 9),N( 9)),V())
  etst(vectorSubseq(V(abcdefghi),N( 9),N(10)),V())
  etst(vectorSubseq(V(abcdefghi),N(10),N(-1)),V())
  etst(vectorSubseq(V(abcdefghi),N(10),N( 0)),V())
  etst(vectorSubseq(V(abcdefghi),N(10),N( 1)),V())
  etst(vectorSubseq(V(abcdefghi),N(10),N( 2)),V())
  etst(vectorSubseq(V(abcdefghi),N(10),N( 3)),V())
  etst(vectorSubseq(V(abcdefghi),N(10),N( 4)),V())
  etst(vectorSubseq(V(abcdefghi),N(10),N( 5)),V())
  etst(vectorSubseq(V(abcdefghi),N(10),N( 6)),V())
  etst(vectorSubseq(V(abcdefghi),N(10),N( 7)),V())
  etst(vectorSubseq(V(abcdefghi),N(10),N( 8)),V())
  etst(vectorSubseq(V(abcdefghi),N(10),N( 9)),V())
  etst(vectorSubseq(V(abcdefghi),N(10),N(10)),V())
  etst(vectorSubseq(V(abcdefgh),N(-1),N(-1)),V())
  etst(vectorSubseq(V(abcdefgh),N(-1),N( 0)),V())
  etst(vectorSubseq(V(abcdefgh),N(-1),N( 1)),V(a))
  etst(vectorSubseq(V(abcdefgh),N(-1),N( 2)),V(ab))
  etst(vectorSubseq(V(abcdefgh),N(-1),N( 3)),V(abc))
  etst(vectorSubseq(V(abcdefgh),N(-1),N( 4)),V(abcd))
  etst(vectorSubseq(V(abcdefgh),N(-1),N( 5)),V(abcde))
  etst(vectorSubseq(V(abcdefgh),N(-1),N( 6)),V(abcdef))
  etst(vectorSubseq(V(abcdefgh),N(-1),N( 7)),V(abcdefg))
  etst(vectorSubseq(V(abcdefgh),N(-1),N( 8)),V(abcdefgh))
  etst(vectorSubseq(V(abcdefgh),N(-1),N( 9)),V(abcdefgh))
  etst(vectorSubseq(V(abcdefgh),N(-1),N(10)),V(abcdefgh))
  etst(vectorSubseq(V(abcdefgh),N( 0),N(-1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 0),N( 0)),V())
  etst(vectorSubseq(V(abcdefgh),N( 0),N( 1)),V(a))
  etst(vectorSubseq(V(abcdefgh),N( 0),N( 2)),V(ab))
  etst(vectorSubseq(V(abcdefgh),N( 0),N( 3)),V(abc))
  etst(vectorSubseq(V(abcdefgh),N( 0),N( 4)),V(abcd))
  etst(vectorSubseq(V(abcdefgh),N( 0),N( 5)),V(abcde))
  etst(vectorSubseq(V(abcdefgh),N( 0),N( 6)),V(abcdef))
  etst(vectorSubseq(V(abcdefgh),N( 0),N( 7)),V(abcdefg))
  etst(vectorSubseq(V(abcdefgh),N( 0),N( 8)),V(abcdefgh))
  etst(vectorSubseq(V(abcdefgh),N( 0),N( 9)),V(abcdefgh))
  etst(vectorSubseq(V(abcdefgh),N( 0),N(10)),V(abcdefgh))
  etst(vectorSubseq(V(abcdefgh),N( 1),N(-1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 1),N( 0)),V())
  etst(vectorSubseq(V(abcdefgh),N( 1),N( 1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 1),N( 2)),V(b))
  etst(vectorSubseq(V(abcdefgh),N( 1),N( 3)),V(bc))
  etst(vectorSubseq(V(abcdefgh),N( 1),N( 4)),V(bcd))
  etst(vectorSubseq(V(abcdefgh),N( 1),N( 5)),V(bcde))
  etst(vectorSubseq(V(abcdefgh),N( 1),N( 6)),V(bcdef))
  etst(vectorSubseq(V(abcdefgh),N( 1),N( 7)),V(bcdefg))
  etst(vectorSubseq(V(abcdefgh),N( 1),N( 8)),V(bcdefgh))
  etst(vectorSubseq(V(abcdefgh),N( 1),N( 9)),V(bcdefgh))
  etst(vectorSubseq(V(abcdefgh),N( 1),N(10)),V(bcdefgh))
  etst(vectorSubseq(V(abcdefgh),N( 2),N(-1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 2),N( 0)),V())
  etst(vectorSubseq(V(abcdefgh),N( 2),N( 1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 2),N( 2)),V())
  etst(vectorSubseq(V(abcdefgh),N( 2),N( 3)),V(c))
  etst(vectorSubseq(V(abcdefgh),N( 2),N( 4)),V(cd))
  etst(vectorSubseq(V(abcdefgh),N( 2),N( 5)),V(cde))
  etst(vectorSubseq(V(abcdefgh),N( 2),N( 6)),V(cdef))
  etst(vectorSubseq(V(abcdefgh),N( 2),N( 7)),V(cdefg))
  etst(vectorSubseq(V(abcdefgh),N( 2),N( 8)),V(cdefgh))
  etst(vectorSubseq(V(abcdefgh),N( 2),N( 9)),V(cdefgh))
  etst(vectorSubseq(V(abcdefgh),N( 2),N(10)),V(cdefgh))
  etst(vectorSubseq(V(abcdefgh),N( 3),N(-1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 3),N( 0)),V())
  etst(vectorSubseq(V(abcdefgh),N( 3),N( 1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 3),N( 2)),V())
  etst(vectorSubseq(V(abcdefgh),N( 3),N( 3)),V())
  etst(vectorSubseq(V(abcdefgh),N( 3),N( 4)),V(d))
  etst(vectorSubseq(V(abcdefgh),N( 3),N( 5)),V(de))
  etst(vectorSubseq(V(abcdefgh),N( 3),N( 6)),V(def))
  etst(vectorSubseq(V(abcdefgh),N( 3),N( 7)),V(defg))
  etst(vectorSubseq(V(abcdefgh),N( 3),N( 8)),V(defgh))
  etst(vectorSubseq(V(abcdefgh),N( 3),N( 9)),V(defgh))
  etst(vectorSubseq(V(abcdefgh),N( 3),N(10)),V(defgh))
  etst(vectorSubseq(V(abcdefgh),N( 4),N(-1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 4),N( 0)),V())
  etst(vectorSubseq(V(abcdefgh),N( 4),N( 1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 4),N( 2)),V())
  etst(vectorSubseq(V(abcdefgh),N( 4),N( 3)),V())
  etst(vectorSubseq(V(abcdefgh),N( 4),N( 4)),V())
  etst(vectorSubseq(V(abcdefgh),N( 4),N( 5)),V(e))
  etst(vectorSubseq(V(abcdefgh),N( 4),N( 6)),V(ef))
  etst(vectorSubseq(V(abcdefgh),N( 4),N( 7)),V(efg))
  etst(vectorSubseq(V(abcdefgh),N( 4),N( 8)),V(efgh))
  etst(vectorSubseq(V(abcdefgh),N( 4),N( 9)),V(efgh))
  etst(vectorSubseq(V(abcdefgh),N( 4),N(10)),V(efgh))
  etst(vectorSubseq(V(abcdefgh),N( 5),N(-1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 5),N( 0)),V())
  etst(vectorSubseq(V(abcdefgh),N( 5),N( 1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 5),N( 2)),V())
  etst(vectorSubseq(V(abcdefgh),N( 5),N( 3)),V())
  etst(vectorSubseq(V(abcdefgh),N( 5),N( 4)),V())
  etst(vectorSubseq(V(abcdefgh),N( 5),N( 5)),V())
  etst(vectorSubseq(V(abcdefgh),N( 5),N( 6)),V(f))
  etst(vectorSubseq(V(abcdefgh),N( 5),N( 7)),V(fg))
  etst(vectorSubseq(V(abcdefgh),N( 5),N( 8)),V(fgh))
  etst(vectorSubseq(V(abcdefgh),N( 5),N( 9)),V(fgh))
  etst(vectorSubseq(V(abcdefgh),N( 5),N(10)),V(fgh))
  etst(vectorSubseq(V(abcdefgh),N( 6),N(-1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 6),N( 0)),V())
  etst(vectorSubseq(V(abcdefgh),N( 6),N( 1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 6),N( 2)),V())
  etst(vectorSubseq(V(abcdefgh),N( 6),N( 3)),V())
  etst(vectorSubseq(V(abcdefgh),N( 6),N( 4)),V())
  etst(vectorSubseq(V(abcdefgh),N( 6),N( 5)),V())
  etst(vectorSubseq(V(abcdefgh),N( 6),N( 6)),V())
  etst(vectorSubseq(V(abcdefgh),N( 6),N( 7)),V(g))
  etst(vectorSubseq(V(abcdefgh),N( 6),N( 8)),V(gh))
  etst(vectorSubseq(V(abcdefgh),N( 6),N( 9)),V(gh))
  etst(vectorSubseq(V(abcdefgh),N( 6),N(10)),V(gh))
  etst(vectorSubseq(V(abcdefgh),N( 7),N(-1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 7),N( 0)),V())
  etst(vectorSubseq(V(abcdefgh),N( 7),N( 1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 7),N( 2)),V())
  etst(vectorSubseq(V(abcdefgh),N( 7),N( 3)),V())
  etst(vectorSubseq(V(abcdefgh),N( 7),N( 4)),V())
  etst(vectorSubseq(V(abcdefgh),N( 7),N( 5)),V())
  etst(vectorSubseq(V(abcdefgh),N( 7),N( 6)),V())
  etst(vectorSubseq(V(abcdefgh),N( 7),N( 7)),V())
  etst(vectorSubseq(V(abcdefgh),N( 7),N( 8)),V(h))
  etst(vectorSubseq(V(abcdefgh),N( 7),N( 9)),V(h))
  etst(vectorSubseq(V(abcdefgh),N( 7),N(10)),V(h))
  etst(vectorSubseq(V(abcdefgh),N( 8),N(-1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 8),N( 0)),V())
  etst(vectorSubseq(V(abcdefgh),N( 8),N( 1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 8),N( 2)),V())
  etst(vectorSubseq(V(abcdefgh),N( 8),N( 3)),V())
  etst(vectorSubseq(V(abcdefgh),N( 8),N( 4)),V())
  etst(vectorSubseq(V(abcdefgh),N( 8),N( 5)),V())
  etst(vectorSubseq(V(abcdefgh),N( 8),N( 6)),V())
  etst(vectorSubseq(V(abcdefgh),N( 8),N( 7)),V())
  etst(vectorSubseq(V(abcdefgh),N( 8),N( 8)),V())
  etst(vectorSubseq(V(abcdefgh),N( 8),N( 9)),V())
  etst(vectorSubseq(V(abcdefgh),N( 8),N(10)),V())
  etst(vectorSubseq(V(abcdefgh),N( 9),N(-1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 9),N( 0)),V())
  etst(vectorSubseq(V(abcdefgh),N( 9),N( 1)),V())
  etst(vectorSubseq(V(abcdefgh),N( 9),N( 2)),V())
  etst(vectorSubseq(V(abcdefgh),N( 9),N( 3)),V())
  etst(vectorSubseq(V(abcdefgh),N( 9),N( 4)),V())
  etst(vectorSubseq(V(abcdefgh),N( 9),N( 5)),V())
  etst(vectorSubseq(V(abcdefgh),N( 9),N( 6)),V())
  etst(vectorSubseq(V(abcdefgh),N( 9),N( 7)),V())
  etst(vectorSubseq(V(abcdefgh),N( 9),N( 8)),V())
  etst(vectorSubseq(V(abcdefgh),N( 9),N( 9)),V())
  etst(vectorSubseq(V(abcdefgh),N( 9),N(10)),V())
  etst(vectorSubseq(V(abcdefgh),N(10),N(-1)),V())
  etst(vectorSubseq(V(abcdefgh),N(10),N( 0)),V())
  etst(vectorSubseq(V(abcdefgh),N(10),N( 1)),V())
  etst(vectorSubseq(V(abcdefgh),N(10),N( 2)),V())
  etst(vectorSubseq(V(abcdefgh),N(10),N( 3)),V())
  etst(vectorSubseq(V(abcdefgh),N(10),N( 4)),V())
  etst(vectorSubseq(V(abcdefgh),N(10),N( 5)),V())
  etst(vectorSubseq(V(abcdefgh),N(10),N( 6)),V())
  etst(vectorSubseq(V(abcdefgh),N(10),N( 7)),V())
  etst(vectorSubseq(V(abcdefgh),N(10),N( 8)),V())
  etst(vectorSubseq(V(abcdefgh),N(10),N( 9)),V())
  etst(vectorSubseq(V(abcdefgh),N(10),N(10)),V())
  etst(vectorSubseq(V(abcdefg),N(-1),N(-1)),V())
  etst(vectorSubseq(V(abcdefg),N(-1),N( 0)),V())
  etst(vectorSubseq(V(abcdefg),N(-1),N( 1)),V(a))
  etst(vectorSubseq(V(abcdefg),N(-1),N( 2)),V(ab))
  etst(vectorSubseq(V(abcdefg),N(-1),N( 3)),V(abc))
  etst(vectorSubseq(V(abcdefg),N(-1),N( 4)),V(abcd))
  etst(vectorSubseq(V(abcdefg),N(-1),N( 5)),V(abcde))
  etst(vectorSubseq(V(abcdefg),N(-1),N( 6)),V(abcdef))
  etst(vectorSubseq(V(abcdefg),N(-1),N( 7)),V(abcdefg))
  etst(vectorSubseq(V(abcdefg),N(-1),N( 8)),V(abcdefg))
  etst(vectorSubseq(V(abcdefg),N(-1),N( 9)),V(abcdefg))
  etst(vectorSubseq(V(abcdefg),N(-1),N(10)),V(abcdefg))
  etst(vectorSubseq(V(abcdefg),N( 0),N(-1)),V())
  etst(vectorSubseq(V(abcdefg),N( 0),N( 0)),V())
  etst(vectorSubseq(V(abcdefg),N( 0),N( 1)),V(a))
  etst(vectorSubseq(V(abcdefg),N( 0),N( 2)),V(ab))
  etst(vectorSubseq(V(abcdefg),N( 0),N( 3)),V(abc))
  etst(vectorSubseq(V(abcdefg),N( 0),N( 4)),V(abcd))
  etst(vectorSubseq(V(abcdefg),N( 0),N( 5)),V(abcde))
  etst(vectorSubseq(V(abcdefg),N( 0),N( 6)),V(abcdef))
  etst(vectorSubseq(V(abcdefg),N( 0),N( 7)),V(abcdefg))
  etst(vectorSubseq(V(abcdefg),N( 0),N( 8)),V(abcdefg))
  etst(vectorSubseq(V(abcdefg),N( 0),N( 9)),V(abcdefg))
  etst(vectorSubseq(V(abcdefg),N( 0),N(10)),V(abcdefg))
  etst(vectorSubseq(V(abcdefg),N( 1),N(-1)),V())
  etst(vectorSubseq(V(abcdefg),N( 1),N( 0)),V())
  etst(vectorSubseq(V(abcdefg),N( 1),N( 1)),V())
  etst(vectorSubseq(V(abcdefg),N( 1),N( 2)),V(b))
  etst(vectorSubseq(V(abcdefg),N( 1),N( 3)),V(bc))
  etst(vectorSubseq(V(abcdefg),N( 1),N( 4)),V(bcd))
  etst(vectorSubseq(V(abcdefg),N( 1),N( 5)),V(bcde))
  etst(vectorSubseq(V(abcdefg),N( 1),N( 6)),V(bcdef))
  etst(vectorSubseq(V(abcdefg),N( 1),N( 7)),V(bcdefg))
  etst(vectorSubseq(V(abcdefg),N( 1),N( 8)),V(bcdefg))
  etst(vectorSubseq(V(abcdefg),N( 1),N( 9)),V(bcdefg))
  etst(vectorSubseq(V(abcdefg),N( 1),N(10)),V(bcdefg))
  etst(vectorSubseq(V(abcdefg),N( 2),N(-1)),V())
  etst(vectorSubseq(V(abcdefg),N( 2),N( 0)),V())
  etst(vectorSubseq(V(abcdefg),N( 2),N( 1)),V())
  etst(vectorSubseq(V(abcdefg),N( 2),N( 2)),V())
  etst(vectorSubseq(V(abcdefg),N( 2),N( 3)),V(c))
  etst(vectorSubseq(V(abcdefg),N( 2),N( 4)),V(cd))
  etst(vectorSubseq(V(abcdefg),N( 2),N( 5)),V(cde))
  etst(vectorSubseq(V(abcdefg),N( 2),N( 6)),V(cdef))
  etst(vectorSubseq(V(abcdefg),N( 2),N( 7)),V(cdefg))
  etst(vectorSubseq(V(abcdefg),N( 2),N( 8)),V(cdefg))
  etst(vectorSubseq(V(abcdefg),N( 2),N( 9)),V(cdefg))
  etst(vectorSubseq(V(abcdefg),N( 2),N(10)),V(cdefg))
  etst(vectorSubseq(V(abcdefg),N( 3),N(-1)),V())
  etst(vectorSubseq(V(abcdefg),N( 3),N( 0)),V())
  etst(vectorSubseq(V(abcdefg),N( 3),N( 1)),V())
  etst(vectorSubseq(V(abcdefg),N( 3),N( 2)),V())
  etst(vectorSubseq(V(abcdefg),N( 3),N( 3)),V())
  etst(vectorSubseq(V(abcdefg),N( 3),N( 4)),V(d))
  etst(vectorSubseq(V(abcdefg),N( 3),N( 5)),V(de))
  etst(vectorSubseq(V(abcdefg),N( 3),N( 6)),V(def))
  etst(vectorSubseq(V(abcdefg),N( 3),N( 7)),V(defg))
  etst(vectorSubseq(V(abcdefg),N( 3),N( 8)),V(defg))
  etst(vectorSubseq(V(abcdefg),N( 3),N( 9)),V(defg))
  etst(vectorSubseq(V(abcdefg),N( 3),N(10)),V(defg))
  etst(vectorSubseq(V(abcdefg),N( 4),N(-1)),V())
  etst(vectorSubseq(V(abcdefg),N( 4),N( 0)),V())
  etst(vectorSubseq(V(abcdefg),N( 4),N( 1)),V())
  etst(vectorSubseq(V(abcdefg),N( 4),N( 2)),V())
  etst(vectorSubseq(V(abcdefg),N( 4),N( 3)),V())
  etst(vectorSubseq(V(abcdefg),N( 4),N( 4)),V())
  etst(vectorSubseq(V(abcdefg),N( 4),N( 5)),V(e))
  etst(vectorSubseq(V(abcdefg),N( 4),N( 6)),V(ef))
  etst(vectorSubseq(V(abcdefg),N( 4),N( 7)),V(efg))
  etst(vectorSubseq(V(abcdefg),N( 4),N( 8)),V(efg))
  etst(vectorSubseq(V(abcdefg),N( 4),N( 9)),V(efg))
  etst(vectorSubseq(V(abcdefg),N( 4),N(10)),V(efg))
  etst(vectorSubseq(V(abcdefg),N( 5),N(-1)),V())
  etst(vectorSubseq(V(abcdefg),N( 5),N( 0)),V())
  etst(vectorSubseq(V(abcdefg),N( 5),N( 1)),V())
  etst(vectorSubseq(V(abcdefg),N( 5),N( 2)),V())
  etst(vectorSubseq(V(abcdefg),N( 5),N( 3)),V())
  etst(vectorSubseq(V(abcdefg),N( 5),N( 4)),V())
  etst(vectorSubseq(V(abcdefg),N( 5),N( 5)),V())
  etst(vectorSubseq(V(abcdefg),N( 5),N( 6)),V(f))
  etst(vectorSubseq(V(abcdefg),N( 5),N( 7)),V(fg))
  etst(vectorSubseq(V(abcdefg),N( 5),N( 8)),V(fg))
  etst(vectorSubseq(V(abcdefg),N( 5),N( 9)),V(fg))
  etst(vectorSubseq(V(abcdefg),N( 5),N(10)),V(fg))
  etst(vectorSubseq(V(abcdefg),N( 6),N(-1)),V())
  etst(vectorSubseq(V(abcdefg),N( 6),N( 0)),V())
  etst(vectorSubseq(V(abcdefg),N( 6),N( 1)),V())
  etst(vectorSubseq(V(abcdefg),N( 6),N( 2)),V())
  etst(vectorSubseq(V(abcdefg),N( 6),N( 3)),V())
  etst(vectorSubseq(V(abcdefg),N( 6),N( 4)),V())
  etst(vectorSubseq(V(abcdefg),N( 6),N( 5)),V())
  etst(vectorSubseq(V(abcdefg),N( 6),N( 6)),V())
  etst(vectorSubseq(V(abcdefg),N( 6),N( 7)),V(g))
  etst(vectorSubseq(V(abcdefg),N( 6),N( 8)),V(g))
  etst(vectorSubseq(V(abcdefg),N( 6),N( 9)),V(g))
  etst(vectorSubseq(V(abcdefg),N( 6),N(10)),V(g))
  etst(vectorSubseq(V(abcdefg),N( 7),N(-1)),V())
  etst(vectorSubseq(V(abcdefg),N( 7),N( 0)),V())
  etst(vectorSubseq(V(abcdefg),N( 7),N( 1)),V())
  etst(vectorSubseq(V(abcdefg),N( 7),N( 2)),V())
  etst(vectorSubseq(V(abcdefg),N( 7),N( 3)),V())
  etst(vectorSubseq(V(abcdefg),N( 7),N( 4)),V())
  etst(vectorSubseq(V(abcdefg),N( 7),N( 5)),V())
  etst(vectorSubseq(V(abcdefg),N( 7),N( 6)),V())
  etst(vectorSubseq(V(abcdefg),N( 7),N( 7)),V())
  etst(vectorSubseq(V(abcdefg),N( 7),N( 8)),V())
  etst(vectorSubseq(V(abcdefg),N( 7),N( 9)),V())
  etst(vectorSubseq(V(abcdefg),N( 7),N(10)),V())
  etst(vectorSubseq(V(abcdefg),N( 8),N(-1)),V())
  etst(vectorSubseq(V(abcdefg),N( 8),N( 0)),V())
  etst(vectorSubseq(V(abcdefg),N( 8),N( 1)),V())
  etst(vectorSubseq(V(abcdefg),N( 8),N( 2)),V())
  etst(vectorSubseq(V(abcdefg),N( 8),N( 3)),V())
  etst(vectorSubseq(V(abcdefg),N( 8),N( 4)),V())
  etst(vectorSubseq(V(abcdefg),N( 8),N( 5)),V())
  etst(vectorSubseq(V(abcdefg),N( 8),N( 6)),V())
  etst(vectorSubseq(V(abcdefg),N( 8),N( 7)),V())
  etst(vectorSubseq(V(abcdefg),N( 8),N( 8)),V())
  etst(vectorSubseq(V(abcdefg),N( 8),N( 9)),V())
  etst(vectorSubseq(V(abcdefg),N( 8),N(10)),V())
  etst(vectorSubseq(V(abcdefg),N( 9),N(-1)),V())
  etst(vectorSubseq(V(abcdefg),N( 9),N( 0)),V())
  etst(vectorSubseq(V(abcdefg),N( 9),N( 1)),V())
  etst(vectorSubseq(V(abcdefg),N( 9),N( 2)),V())
  etst(vectorSubseq(V(abcdefg),N( 9),N( 3)),V())
  etst(vectorSubseq(V(abcdefg),N( 9),N( 4)),V())
  etst(vectorSubseq(V(abcdefg),N( 9),N( 5)),V())
  etst(vectorSubseq(V(abcdefg),N( 9),N( 6)),V())
  etst(vectorSubseq(V(abcdefg),N( 9),N( 7)),V())
  etst(vectorSubseq(V(abcdefg),N( 9),N( 8)),V())
  etst(vectorSubseq(V(abcdefg),N( 9),N( 9)),V())
  etst(vectorSubseq(V(abcdefg),N( 9),N(10)),V())
  etst(vectorSubseq(V(abcdefg),N(10),N(-1)),V())
  etst(vectorSubseq(V(abcdefg),N(10),N( 0)),V())
  etst(vectorSubseq(V(abcdefg),N(10),N( 1)),V())
  etst(vectorSubseq(V(abcdefg),N(10),N( 2)),V())
  etst(vectorSubseq(V(abcdefg),N(10),N( 3)),V())
  etst(vectorSubseq(V(abcdefg),N(10),N( 4)),V())
  etst(vectorSubseq(V(abcdefg),N(10),N( 5)),V())
  etst(vectorSubseq(V(abcdefg),N(10),N( 6)),V())
  etst(vectorSubseq(V(abcdefg),N(10),N( 7)),V())
  etst(vectorSubseq(V(abcdefg),N(10),N( 8)),V())
  etst(vectorSubseq(V(abcdefg),N(10),N( 9)),V())
  etst(vectorSubseq(V(abcdefg),N(10),N(10)),V())

  etst(vectorSubseq(W(2,abcdefghi),N(-1),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(-1),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(-1),N( 1)),V(a))
  etst(vectorSubseq(W(2,abcdefghi),N(-1),N( 2)),V(ab))
  etst(vectorSubseq(W(2,abcdefghi),N(-1),N( 3)),V(abc))
  etst(vectorSubseq(W(2,abcdefghi),N(-1),N( 4)),V(abcd))
  etst(vectorSubseq(W(2,abcdefghi),N(-1),N( 5)),V(abcde))
  etst(vectorSubseq(W(2,abcdefghi),N(-1),N( 6)),V(abcdef))
  etst(vectorSubseq(W(2,abcdefghi),N(-1),N( 7)),V(abcdefg))
  etst(vectorSubseq(W(2,abcdefghi),N(-1),N( 8)),V(abcdefgh))
  etst(vectorSubseq(W(2,abcdefghi),N(-1),N( 9)),V(abcdefghi))
  etst(vectorSubseq(W(2,abcdefghi),N(-1),N(10)),V(abcdefghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 0),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 0),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 0),N( 1)),V(a))
  etst(vectorSubseq(W(2,abcdefghi),N( 0),N( 2)),V(ab))
  etst(vectorSubseq(W(2,abcdefghi),N( 0),N( 3)),V(abc))
  etst(vectorSubseq(W(2,abcdefghi),N( 0),N( 4)),V(abcd))
  etst(vectorSubseq(W(2,abcdefghi),N( 0),N( 5)),V(abcde))
  etst(vectorSubseq(W(2,abcdefghi),N( 0),N( 6)),V(abcdef))
  etst(vectorSubseq(W(2,abcdefghi),N( 0),N( 7)),V(abcdefg))
  etst(vectorSubseq(W(2,abcdefghi),N( 0),N( 8)),V(abcdefgh))
  etst(vectorSubseq(W(2,abcdefghi),N( 0),N( 9)),V(abcdefghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 0),N(10)),V(abcdefghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 1),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 1),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 1),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 1),N( 2)),V(b))
  etst(vectorSubseq(W(2,abcdefghi),N( 1),N( 3)),V(bc))
  etst(vectorSubseq(W(2,abcdefghi),N( 1),N( 4)),V(bcd))
  etst(vectorSubseq(W(2,abcdefghi),N( 1),N( 5)),V(bcde))
  etst(vectorSubseq(W(2,abcdefghi),N( 1),N( 6)),V(bcdef))
  etst(vectorSubseq(W(2,abcdefghi),N( 1),N( 7)),V(bcdefg))
  etst(vectorSubseq(W(2,abcdefghi),N( 1),N( 8)),V(bcdefgh))
  etst(vectorSubseq(W(2,abcdefghi),N( 1),N( 9)),V(bcdefghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 1),N(10)),V(bcdefghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 2),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 2),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 2),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 2),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 2),N( 3)),V(c))
  etst(vectorSubseq(W(2,abcdefghi),N( 2),N( 4)),V(cd))
  etst(vectorSubseq(W(2,abcdefghi),N( 2),N( 5)),V(cde))
  etst(vectorSubseq(W(2,abcdefghi),N( 2),N( 6)),V(cdef))
  etst(vectorSubseq(W(2,abcdefghi),N( 2),N( 7)),V(cdefg))
  etst(vectorSubseq(W(2,abcdefghi),N( 2),N( 8)),V(cdefgh))
  etst(vectorSubseq(W(2,abcdefghi),N( 2),N( 9)),V(cdefghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 2),N(10)),V(cdefghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 3),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 3),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 3),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 3),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 3),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 3),N( 4)),V(d))
  etst(vectorSubseq(W(2,abcdefghi),N( 3),N( 5)),V(de))
  etst(vectorSubseq(W(2,abcdefghi),N( 3),N( 6)),V(def))
  etst(vectorSubseq(W(2,abcdefghi),N( 3),N( 7)),V(defg))
  etst(vectorSubseq(W(2,abcdefghi),N( 3),N( 8)),V(defgh))
  etst(vectorSubseq(W(2,abcdefghi),N( 3),N( 9)),V(defghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 3),N(10)),V(defghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 4),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 4),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 4),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 4),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 4),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 4),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 4),N( 5)),V(e))
  etst(vectorSubseq(W(2,abcdefghi),N( 4),N( 6)),V(ef))
  etst(vectorSubseq(W(2,abcdefghi),N( 4),N( 7)),V(efg))
  etst(vectorSubseq(W(2,abcdefghi),N( 4),N( 8)),V(efgh))
  etst(vectorSubseq(W(2,abcdefghi),N( 4),N( 9)),V(efghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 4),N(10)),V(efghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 5),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 5),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 5),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 5),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 5),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 5),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 5),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 5),N( 6)),V(f))
  etst(vectorSubseq(W(2,abcdefghi),N( 5),N( 7)),V(fg))
  etst(vectorSubseq(W(2,abcdefghi),N( 5),N( 8)),V(fgh))
  etst(vectorSubseq(W(2,abcdefghi),N( 5),N( 9)),V(fghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 5),N(10)),V(fghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 6),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 6),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 6),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 6),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 6),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 6),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 6),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 6),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 6),N( 7)),V(g))
  etst(vectorSubseq(W(2,abcdefghi),N( 6),N( 8)),V(gh))
  etst(vectorSubseq(W(2,abcdefghi),N( 6),N( 9)),V(ghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 6),N(10)),V(ghi))
  etst(vectorSubseq(W(2,abcdefghi),N( 7),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 7),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 7),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 7),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 7),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 7),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 7),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 7),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 7),N( 7)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 7),N( 8)),V(h))
  etst(vectorSubseq(W(2,abcdefghi),N( 7),N( 9)),V(hi))
  etst(vectorSubseq(W(2,abcdefghi),N( 7),N(10)),V(hi))
  etst(vectorSubseq(W(2,abcdefghi),N( 8),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 8),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 8),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 8),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 8),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 8),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 8),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 8),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 8),N( 7)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 8),N( 8)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 8),N( 9)),V(i))
  etst(vectorSubseq(W(2,abcdefghi),N( 8),N(10)),V(i))
  etst(vectorSubseq(W(2,abcdefghi),N( 9),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 9),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 9),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 9),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 9),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 9),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 9),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 9),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 9),N( 7)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 9),N( 8)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 9),N( 9)),V())
  etst(vectorSubseq(W(2,abcdefghi),N( 9),N(10)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(10),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(10),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(10),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(10),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(10),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(10),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(10),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(10),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(10),N( 7)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(10),N( 8)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(10),N( 9)),V())
  etst(vectorSubseq(W(2,abcdefghi),N(10),N(10)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(-1),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(-1),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(-1),N( 1)),V(a))
  etst(vectorSubseq(W(2,abcdefgh),N(-1),N( 2)),V(ab))
  etst(vectorSubseq(W(2,abcdefgh),N(-1),N( 3)),V(abc))
  etst(vectorSubseq(W(2,abcdefgh),N(-1),N( 4)),V(abcd))
  etst(vectorSubseq(W(2,abcdefgh),N(-1),N( 5)),V(abcde))
  etst(vectorSubseq(W(2,abcdefgh),N(-1),N( 6)),V(abcdef))
  etst(vectorSubseq(W(2,abcdefgh),N(-1),N( 7)),V(abcdefg))
  etst(vectorSubseq(W(2,abcdefgh),N(-1),N( 8)),V(abcdefgh))
  etst(vectorSubseq(W(2,abcdefgh),N(-1),N( 9)),V(abcdefgh))
  etst(vectorSubseq(W(2,abcdefgh),N(-1),N(10)),V(abcdefgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 0),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 0),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 0),N( 1)),V(a))
  etst(vectorSubseq(W(2,abcdefgh),N( 0),N( 2)),V(ab))
  etst(vectorSubseq(W(2,abcdefgh),N( 0),N( 3)),V(abc))
  etst(vectorSubseq(W(2,abcdefgh),N( 0),N( 4)),V(abcd))
  etst(vectorSubseq(W(2,abcdefgh),N( 0),N( 5)),V(abcde))
  etst(vectorSubseq(W(2,abcdefgh),N( 0),N( 6)),V(abcdef))
  etst(vectorSubseq(W(2,abcdefgh),N( 0),N( 7)),V(abcdefg))
  etst(vectorSubseq(W(2,abcdefgh),N( 0),N( 8)),V(abcdefgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 0),N( 9)),V(abcdefgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 0),N(10)),V(abcdefgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 1),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 1),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 1),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 1),N( 2)),V(b))
  etst(vectorSubseq(W(2,abcdefgh),N( 1),N( 3)),V(bc))
  etst(vectorSubseq(W(2,abcdefgh),N( 1),N( 4)),V(bcd))
  etst(vectorSubseq(W(2,abcdefgh),N( 1),N( 5)),V(bcde))
  etst(vectorSubseq(W(2,abcdefgh),N( 1),N( 6)),V(bcdef))
  etst(vectorSubseq(W(2,abcdefgh),N( 1),N( 7)),V(bcdefg))
  etst(vectorSubseq(W(2,abcdefgh),N( 1),N( 8)),V(bcdefgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 1),N( 9)),V(bcdefgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 1),N(10)),V(bcdefgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 2),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 2),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 2),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 2),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 2),N( 3)),V(c))
  etst(vectorSubseq(W(2,abcdefgh),N( 2),N( 4)),V(cd))
  etst(vectorSubseq(W(2,abcdefgh),N( 2),N( 5)),V(cde))
  etst(vectorSubseq(W(2,abcdefgh),N( 2),N( 6)),V(cdef))
  etst(vectorSubseq(W(2,abcdefgh),N( 2),N( 7)),V(cdefg))
  etst(vectorSubseq(W(2,abcdefgh),N( 2),N( 8)),V(cdefgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 2),N( 9)),V(cdefgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 2),N(10)),V(cdefgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 3),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 3),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 3),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 3),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 3),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 3),N( 4)),V(d))
  etst(vectorSubseq(W(2,abcdefgh),N( 3),N( 5)),V(de))
  etst(vectorSubseq(W(2,abcdefgh),N( 3),N( 6)),V(def))
  etst(vectorSubseq(W(2,abcdefgh),N( 3),N( 7)),V(defg))
  etst(vectorSubseq(W(2,abcdefgh),N( 3),N( 8)),V(defgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 3),N( 9)),V(defgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 3),N(10)),V(defgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 4),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 4),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 4),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 4),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 4),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 4),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 4),N( 5)),V(e))
  etst(vectorSubseq(W(2,abcdefgh),N( 4),N( 6)),V(ef))
  etst(vectorSubseq(W(2,abcdefgh),N( 4),N( 7)),V(efg))
  etst(vectorSubseq(W(2,abcdefgh),N( 4),N( 8)),V(efgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 4),N( 9)),V(efgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 4),N(10)),V(efgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 5),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 5),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 5),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 5),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 5),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 5),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 5),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 5),N( 6)),V(f))
  etst(vectorSubseq(W(2,abcdefgh),N( 5),N( 7)),V(fg))
  etst(vectorSubseq(W(2,abcdefgh),N( 5),N( 8)),V(fgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 5),N( 9)),V(fgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 5),N(10)),V(fgh))
  etst(vectorSubseq(W(2,abcdefgh),N( 6),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 6),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 6),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 6),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 6),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 6),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 6),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 6),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 6),N( 7)),V(g))
  etst(vectorSubseq(W(2,abcdefgh),N( 6),N( 8)),V(gh))
  etst(vectorSubseq(W(2,abcdefgh),N( 6),N( 9)),V(gh))
  etst(vectorSubseq(W(2,abcdefgh),N( 6),N(10)),V(gh))
  etst(vectorSubseq(W(2,abcdefgh),N( 7),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 7),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 7),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 7),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 7),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 7),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 7),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 7),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 7),N( 7)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 7),N( 8)),V(h))
  etst(vectorSubseq(W(2,abcdefgh),N( 7),N( 9)),V(h))
  etst(vectorSubseq(W(2,abcdefgh),N( 7),N(10)),V(h))
  etst(vectorSubseq(W(2,abcdefgh),N( 8),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 8),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 8),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 8),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 8),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 8),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 8),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 8),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 8),N( 7)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 8),N( 8)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 8),N( 9)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 8),N(10)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 9),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 9),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 9),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 9),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 9),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 9),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 9),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 9),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 9),N( 7)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 9),N( 8)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 9),N( 9)),V())
  etst(vectorSubseq(W(2,abcdefgh),N( 9),N(10)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(10),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(10),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(10),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(10),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(10),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(10),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(10),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(10),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(10),N( 7)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(10),N( 8)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(10),N( 9)),V())
  etst(vectorSubseq(W(2,abcdefgh),N(10),N(10)),V())
  etst(vectorSubseq(W(2,abcdefg),N(-1),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefg),N(-1),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefg),N(-1),N( 1)),V(a))
  etst(vectorSubseq(W(2,abcdefg),N(-1),N( 2)),V(ab))
  etst(vectorSubseq(W(2,abcdefg),N(-1),N( 3)),V(abc))
  etst(vectorSubseq(W(2,abcdefg),N(-1),N( 4)),V(abcd))
  etst(vectorSubseq(W(2,abcdefg),N(-1),N( 5)),V(abcde))
  etst(vectorSubseq(W(2,abcdefg),N(-1),N( 6)),V(abcdef))
  etst(vectorSubseq(W(2,abcdefg),N(-1),N( 7)),V(abcdefg))
  etst(vectorSubseq(W(2,abcdefg),N(-1),N( 8)),V(abcdefg))
  etst(vectorSubseq(W(2,abcdefg),N(-1),N( 9)),V(abcdefg))
  etst(vectorSubseq(W(2,abcdefg),N(-1),N(10)),V(abcdefg))
  etst(vectorSubseq(W(2,abcdefg),N( 0),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 0),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 0),N( 1)),V(a))
  etst(vectorSubseq(W(2,abcdefg),N( 0),N( 2)),V(ab))
  etst(vectorSubseq(W(2,abcdefg),N( 0),N( 3)),V(abc))
  etst(vectorSubseq(W(2,abcdefg),N( 0),N( 4)),V(abcd))
  etst(vectorSubseq(W(2,abcdefg),N( 0),N( 5)),V(abcde))
  etst(vectorSubseq(W(2,abcdefg),N( 0),N( 6)),V(abcdef))
  etst(vectorSubseq(W(2,abcdefg),N( 0),N( 7)),V(abcdefg))
  etst(vectorSubseq(W(2,abcdefg),N( 0),N( 8)),V(abcdefg))
  etst(vectorSubseq(W(2,abcdefg),N( 0),N( 9)),V(abcdefg))
  etst(vectorSubseq(W(2,abcdefg),N( 0),N(10)),V(abcdefg))
  etst(vectorSubseq(W(2,abcdefg),N( 1),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 1),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 1),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 1),N( 2)),V(b))
  etst(vectorSubseq(W(2,abcdefg),N( 1),N( 3)),V(bc))
  etst(vectorSubseq(W(2,abcdefg),N( 1),N( 4)),V(bcd))
  etst(vectorSubseq(W(2,abcdefg),N( 1),N( 5)),V(bcde))
  etst(vectorSubseq(W(2,abcdefg),N( 1),N( 6)),V(bcdef))
  etst(vectorSubseq(W(2,abcdefg),N( 1),N( 7)),V(bcdefg))
  etst(vectorSubseq(W(2,abcdefg),N( 1),N( 8)),V(bcdefg))
  etst(vectorSubseq(W(2,abcdefg),N( 1),N( 9)),V(bcdefg))
  etst(vectorSubseq(W(2,abcdefg),N( 1),N(10)),V(bcdefg))
  etst(vectorSubseq(W(2,abcdefg),N( 2),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 2),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 2),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 2),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 2),N( 3)),V(c))
  etst(vectorSubseq(W(2,abcdefg),N( 2),N( 4)),V(cd))
  etst(vectorSubseq(W(2,abcdefg),N( 2),N( 5)),V(cde))
  etst(vectorSubseq(W(2,abcdefg),N( 2),N( 6)),V(cdef))
  etst(vectorSubseq(W(2,abcdefg),N( 2),N( 7)),V(cdefg))
  etst(vectorSubseq(W(2,abcdefg),N( 2),N( 8)),V(cdefg))
  etst(vectorSubseq(W(2,abcdefg),N( 2),N( 9)),V(cdefg))
  etst(vectorSubseq(W(2,abcdefg),N( 2),N(10)),V(cdefg))
  etst(vectorSubseq(W(2,abcdefg),N( 3),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 3),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 3),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 3),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 3),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 3),N( 4)),V(d))
  etst(vectorSubseq(W(2,abcdefg),N( 3),N( 5)),V(de))
  etst(vectorSubseq(W(2,abcdefg),N( 3),N( 6)),V(def))
  etst(vectorSubseq(W(2,abcdefg),N( 3),N( 7)),V(defg))
  etst(vectorSubseq(W(2,abcdefg),N( 3),N( 8)),V(defg))
  etst(vectorSubseq(W(2,abcdefg),N( 3),N( 9)),V(defg))
  etst(vectorSubseq(W(2,abcdefg),N( 3),N(10)),V(defg))
  etst(vectorSubseq(W(2,abcdefg),N( 4),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 4),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 4),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 4),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 4),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 4),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 4),N( 5)),V(e))
  etst(vectorSubseq(W(2,abcdefg),N( 4),N( 6)),V(ef))
  etst(vectorSubseq(W(2,abcdefg),N( 4),N( 7)),V(efg))
  etst(vectorSubseq(W(2,abcdefg),N( 4),N( 8)),V(efg))
  etst(vectorSubseq(W(2,abcdefg),N( 4),N( 9)),V(efg))
  etst(vectorSubseq(W(2,abcdefg),N( 4),N(10)),V(efg))
  etst(vectorSubseq(W(2,abcdefg),N( 5),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 5),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 5),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 5),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 5),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 5),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 5),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 5),N( 6)),V(f))
  etst(vectorSubseq(W(2,abcdefg),N( 5),N( 7)),V(fg))
  etst(vectorSubseq(W(2,abcdefg),N( 5),N( 8)),V(fg))
  etst(vectorSubseq(W(2,abcdefg),N( 5),N( 9)),V(fg))
  etst(vectorSubseq(W(2,abcdefg),N( 5),N(10)),V(fg))
  etst(vectorSubseq(W(2,abcdefg),N( 6),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 6),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 6),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 6),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 6),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 6),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 6),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 6),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 6),N( 7)),V(g))
  etst(vectorSubseq(W(2,abcdefg),N( 6),N( 8)),V(g))
  etst(vectorSubseq(W(2,abcdefg),N( 6),N( 9)),V(g))
  etst(vectorSubseq(W(2,abcdefg),N( 6),N(10)),V(g))
  etst(vectorSubseq(W(2,abcdefg),N( 7),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 7),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 7),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 7),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 7),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 7),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 7),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 7),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 7),N( 7)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 7),N( 8)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 7),N( 9)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 7),N(10)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 8),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 8),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 8),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 8),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 8),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 8),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 8),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 8),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 8),N( 7)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 8),N( 8)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 8),N( 9)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 8),N(10)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 9),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 9),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 9),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 9),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 9),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 9),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 9),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 9),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 9),N( 7)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 9),N( 8)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 9),N( 9)),V())
  etst(vectorSubseq(W(2,abcdefg),N( 9),N(10)),V())
  etst(vectorSubseq(W(2,abcdefg),N(10),N(-1)),V())
  etst(vectorSubseq(W(2,abcdefg),N(10),N( 0)),V())
  etst(vectorSubseq(W(2,abcdefg),N(10),N( 1)),V())
  etst(vectorSubseq(W(2,abcdefg),N(10),N( 2)),V())
  etst(vectorSubseq(W(2,abcdefg),N(10),N( 3)),V())
  etst(vectorSubseq(W(2,abcdefg),N(10),N( 4)),V())
  etst(vectorSubseq(W(2,abcdefg),N(10),N( 5)),V())
  etst(vectorSubseq(W(2,abcdefg),N(10),N( 6)),V())
  etst(vectorSubseq(W(2,abcdefg),N(10),N( 7)),V())
  etst(vectorSubseq(W(2,abcdefg),N(10),N( 8)),V())
  etst(vectorSubseq(W(2,abcdefg),N(10),N( 9)),V())
  etst(vectorSubseq(W(2,abcdefg),N(10),N(10)),V())

  etst(vectorSubseq(W(255,abcdefghi),N(-1),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(-1),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(-1),N( 1)),V(a))
  etst(vectorSubseq(W(255,abcdefghi),N(-1),N( 2)),V(ab))
  etst(vectorSubseq(W(255,abcdefghi),N(-1),N( 3)),V(abc))
  etst(vectorSubseq(W(255,abcdefghi),N(-1),N( 4)),V(abcd))
  etst(vectorSubseq(W(255,abcdefghi),N(-1),N( 5)),V(abcde))
  etst(vectorSubseq(W(255,abcdefghi),N(-1),N( 6)),V(abcdef))
  etst(vectorSubseq(W(255,abcdefghi),N(-1),N( 7)),V(abcdefg))
  etst(vectorSubseq(W(255,abcdefghi),N(-1),N( 8)),V(abcdefgh))
  etst(vectorSubseq(W(255,abcdefghi),N(-1),N( 9)),V(abcdefghi))
  etst(vectorSubseq(W(255,abcdefghi),N(-1),N(10)),V(abcdefghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 0),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 0),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 0),N( 1)),V(a))
  etst(vectorSubseq(W(255,abcdefghi),N( 0),N( 2)),V(ab))
  etst(vectorSubseq(W(255,abcdefghi),N( 0),N( 3)),V(abc))
  etst(vectorSubseq(W(255,abcdefghi),N( 0),N( 4)),V(abcd))
  etst(vectorSubseq(W(255,abcdefghi),N( 0),N( 5)),V(abcde))
  etst(vectorSubseq(W(255,abcdefghi),N( 0),N( 6)),V(abcdef))
  etst(vectorSubseq(W(255,abcdefghi),N( 0),N( 7)),V(abcdefg))
  etst(vectorSubseq(W(255,abcdefghi),N( 0),N( 8)),V(abcdefgh))
  etst(vectorSubseq(W(255,abcdefghi),N( 0),N( 9)),V(abcdefghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 0),N(10)),V(abcdefghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 1),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 1),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 1),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 1),N( 2)),V(b))
  etst(vectorSubseq(W(255,abcdefghi),N( 1),N( 3)),V(bc))
  etst(vectorSubseq(W(255,abcdefghi),N( 1),N( 4)),V(bcd))
  etst(vectorSubseq(W(255,abcdefghi),N( 1),N( 5)),V(bcde))
  etst(vectorSubseq(W(255,abcdefghi),N( 1),N( 6)),V(bcdef))
  etst(vectorSubseq(W(255,abcdefghi),N( 1),N( 7)),V(bcdefg))
  etst(vectorSubseq(W(255,abcdefghi),N( 1),N( 8)),V(bcdefgh))
  etst(vectorSubseq(W(255,abcdefghi),N( 1),N( 9)),V(bcdefghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 1),N(10)),V(bcdefghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 2),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 2),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 2),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 2),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 2),N( 3)),V(c))
  etst(vectorSubseq(W(255,abcdefghi),N( 2),N( 4)),V(cd))
  etst(vectorSubseq(W(255,abcdefghi),N( 2),N( 5)),V(cde))
  etst(vectorSubseq(W(255,abcdefghi),N( 2),N( 6)),V(cdef))
  etst(vectorSubseq(W(255,abcdefghi),N( 2),N( 7)),V(cdefg))
  etst(vectorSubseq(W(255,abcdefghi),N( 2),N( 8)),V(cdefgh))
  etst(vectorSubseq(W(255,abcdefghi),N( 2),N( 9)),V(cdefghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 2),N(10)),V(cdefghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 3),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 3),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 3),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 3),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 3),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 3),N( 4)),V(d))
  etst(vectorSubseq(W(255,abcdefghi),N( 3),N( 5)),V(de))
  etst(vectorSubseq(W(255,abcdefghi),N( 3),N( 6)),V(def))
  etst(vectorSubseq(W(255,abcdefghi),N( 3),N( 7)),V(defg))
  etst(vectorSubseq(W(255,abcdefghi),N( 3),N( 8)),V(defgh))
  etst(vectorSubseq(W(255,abcdefghi),N( 3),N( 9)),V(defghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 3),N(10)),V(defghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 4),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 4),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 4),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 4),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 4),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 4),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 4),N( 5)),V(e))
  etst(vectorSubseq(W(255,abcdefghi),N( 4),N( 6)),V(ef))
  etst(vectorSubseq(W(255,abcdefghi),N( 4),N( 7)),V(efg))
  etst(vectorSubseq(W(255,abcdefghi),N( 4),N( 8)),V(efgh))
  etst(vectorSubseq(W(255,abcdefghi),N( 4),N( 9)),V(efghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 4),N(10)),V(efghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 5),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 5),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 5),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 5),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 5),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 5),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 5),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 5),N( 6)),V(f))
  etst(vectorSubseq(W(255,abcdefghi),N( 5),N( 7)),V(fg))
  etst(vectorSubseq(W(255,abcdefghi),N( 5),N( 8)),V(fgh))
  etst(vectorSubseq(W(255,abcdefghi),N( 5),N( 9)),V(fghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 5),N(10)),V(fghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 6),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 6),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 6),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 6),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 6),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 6),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 6),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 6),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 6),N( 7)),V(g))
  etst(vectorSubseq(W(255,abcdefghi),N( 6),N( 8)),V(gh))
  etst(vectorSubseq(W(255,abcdefghi),N( 6),N( 9)),V(ghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 6),N(10)),V(ghi))
  etst(vectorSubseq(W(255,abcdefghi),N( 7),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 7),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 7),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 7),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 7),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 7),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 7),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 7),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 7),N( 7)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 7),N( 8)),V(h))
  etst(vectorSubseq(W(255,abcdefghi),N( 7),N( 9)),V(hi))
  etst(vectorSubseq(W(255,abcdefghi),N( 7),N(10)),V(hi))
  etst(vectorSubseq(W(255,abcdefghi),N( 8),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 8),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 8),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 8),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 8),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 8),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 8),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 8),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 8),N( 7)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 8),N( 8)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 8),N( 9)),V(i))
  etst(vectorSubseq(W(255,abcdefghi),N( 8),N(10)),V(i))
  etst(vectorSubseq(W(255,abcdefghi),N( 9),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 9),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 9),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 9),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 9),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 9),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 9),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 9),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 9),N( 7)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 9),N( 8)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 9),N( 9)),V())
  etst(vectorSubseq(W(255,abcdefghi),N( 9),N(10)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(10),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(10),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(10),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(10),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(10),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(10),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(10),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(10),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(10),N( 7)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(10),N( 8)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(10),N( 9)),V())
  etst(vectorSubseq(W(255,abcdefghi),N(10),N(10)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(-1),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(-1),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(-1),N( 1)),V(a))
  etst(vectorSubseq(W(255,abcdefgh),N(-1),N( 2)),V(ab))
  etst(vectorSubseq(W(255,abcdefgh),N(-1),N( 3)),V(abc))
  etst(vectorSubseq(W(255,abcdefgh),N(-1),N( 4)),V(abcd))
  etst(vectorSubseq(W(255,abcdefgh),N(-1),N( 5)),V(abcde))
  etst(vectorSubseq(W(255,abcdefgh),N(-1),N( 6)),V(abcdef))
  etst(vectorSubseq(W(255,abcdefgh),N(-1),N( 7)),V(abcdefg))
  etst(vectorSubseq(W(255,abcdefgh),N(-1),N( 8)),V(abcdefgh))
  etst(vectorSubseq(W(255,abcdefgh),N(-1),N( 9)),V(abcdefgh))
  etst(vectorSubseq(W(255,abcdefgh),N(-1),N(10)),V(abcdefgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 0),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 0),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 0),N( 1)),V(a))
  etst(vectorSubseq(W(255,abcdefgh),N( 0),N( 2)),V(ab))
  etst(vectorSubseq(W(255,abcdefgh),N( 0),N( 3)),V(abc))
  etst(vectorSubseq(W(255,abcdefgh),N( 0),N( 4)),V(abcd))
  etst(vectorSubseq(W(255,abcdefgh),N( 0),N( 5)),V(abcde))
  etst(vectorSubseq(W(255,abcdefgh),N( 0),N( 6)),V(abcdef))
  etst(vectorSubseq(W(255,abcdefgh),N( 0),N( 7)),V(abcdefg))
  etst(vectorSubseq(W(255,abcdefgh),N( 0),N( 8)),V(abcdefgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 0),N( 9)),V(abcdefgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 0),N(10)),V(abcdefgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 1),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 1),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 1),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 1),N( 2)),V(b))
  etst(vectorSubseq(W(255,abcdefgh),N( 1),N( 3)),V(bc))
  etst(vectorSubseq(W(255,abcdefgh),N( 1),N( 4)),V(bcd))
  etst(vectorSubseq(W(255,abcdefgh),N( 1),N( 5)),V(bcde))
  etst(vectorSubseq(W(255,abcdefgh),N( 1),N( 6)),V(bcdef))
  etst(vectorSubseq(W(255,abcdefgh),N( 1),N( 7)),V(bcdefg))
  etst(vectorSubseq(W(255,abcdefgh),N( 1),N( 8)),V(bcdefgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 1),N( 9)),V(bcdefgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 1),N(10)),V(bcdefgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 2),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 2),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 2),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 2),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 2),N( 3)),V(c))
  etst(vectorSubseq(W(255,abcdefgh),N( 2),N( 4)),V(cd))
  etst(vectorSubseq(W(255,abcdefgh),N( 2),N( 5)),V(cde))
  etst(vectorSubseq(W(255,abcdefgh),N( 2),N( 6)),V(cdef))
  etst(vectorSubseq(W(255,abcdefgh),N( 2),N( 7)),V(cdefg))
  etst(vectorSubseq(W(255,abcdefgh),N( 2),N( 8)),V(cdefgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 2),N( 9)),V(cdefgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 2),N(10)),V(cdefgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 3),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 3),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 3),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 3),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 3),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 3),N( 4)),V(d))
  etst(vectorSubseq(W(255,abcdefgh),N( 3),N( 5)),V(de))
  etst(vectorSubseq(W(255,abcdefgh),N( 3),N( 6)),V(def))
  etst(vectorSubseq(W(255,abcdefgh),N( 3),N( 7)),V(defg))
  etst(vectorSubseq(W(255,abcdefgh),N( 3),N( 8)),V(defgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 3),N( 9)),V(defgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 3),N(10)),V(defgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 4),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 4),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 4),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 4),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 4),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 4),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 4),N( 5)),V(e))
  etst(vectorSubseq(W(255,abcdefgh),N( 4),N( 6)),V(ef))
  etst(vectorSubseq(W(255,abcdefgh),N( 4),N( 7)),V(efg))
  etst(vectorSubseq(W(255,abcdefgh),N( 4),N( 8)),V(efgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 4),N( 9)),V(efgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 4),N(10)),V(efgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 5),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 5),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 5),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 5),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 5),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 5),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 5),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 5),N( 6)),V(f))
  etst(vectorSubseq(W(255,abcdefgh),N( 5),N( 7)),V(fg))
  etst(vectorSubseq(W(255,abcdefgh),N( 5),N( 8)),V(fgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 5),N( 9)),V(fgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 5),N(10)),V(fgh))
  etst(vectorSubseq(W(255,abcdefgh),N( 6),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 6),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 6),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 6),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 6),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 6),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 6),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 6),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 6),N( 7)),V(g))
  etst(vectorSubseq(W(255,abcdefgh),N( 6),N( 8)),V(gh))
  etst(vectorSubseq(W(255,abcdefgh),N( 6),N( 9)),V(gh))
  etst(vectorSubseq(W(255,abcdefgh),N( 6),N(10)),V(gh))
  etst(vectorSubseq(W(255,abcdefgh),N( 7),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 7),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 7),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 7),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 7),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 7),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 7),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 7),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 7),N( 7)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 7),N( 8)),V(h))
  etst(vectorSubseq(W(255,abcdefgh),N( 7),N( 9)),V(h))
  etst(vectorSubseq(W(255,abcdefgh),N( 7),N(10)),V(h))
  etst(vectorSubseq(W(255,abcdefgh),N( 8),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 8),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 8),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 8),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 8),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 8),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 8),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 8),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 8),N( 7)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 8),N( 8)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 8),N( 9)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 8),N(10)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 9),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 9),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 9),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 9),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 9),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 9),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 9),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 9),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 9),N( 7)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 9),N( 8)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 9),N( 9)),V())
  etst(vectorSubseq(W(255,abcdefgh),N( 9),N(10)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(10),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(10),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(10),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(10),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(10),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(10),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(10),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(10),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(10),N( 7)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(10),N( 8)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(10),N( 9)),V())
  etst(vectorSubseq(W(255,abcdefgh),N(10),N(10)),V())
  etst(vectorSubseq(W(255,abcdefg),N(-1),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefg),N(-1),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefg),N(-1),N( 1)),V(a))
  etst(vectorSubseq(W(255,abcdefg),N(-1),N( 2)),V(ab))
  etst(vectorSubseq(W(255,abcdefg),N(-1),N( 3)),V(abc))
  etst(vectorSubseq(W(255,abcdefg),N(-1),N( 4)),V(abcd))
  etst(vectorSubseq(W(255,abcdefg),N(-1),N( 5)),V(abcde))
  etst(vectorSubseq(W(255,abcdefg),N(-1),N( 6)),V(abcdef))
  etst(vectorSubseq(W(255,abcdefg),N(-1),N( 7)),V(abcdefg))
  etst(vectorSubseq(W(255,abcdefg),N(-1),N( 8)),V(abcdefg))
  etst(vectorSubseq(W(255,abcdefg),N(-1),N( 9)),V(abcdefg))
  etst(vectorSubseq(W(255,abcdefg),N(-1),N(10)),V(abcdefg))
  etst(vectorSubseq(W(255,abcdefg),N( 0),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 0),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 0),N( 1)),V(a))
  etst(vectorSubseq(W(255,abcdefg),N( 0),N( 2)),V(ab))
  etst(vectorSubseq(W(255,abcdefg),N( 0),N( 3)),V(abc))
  etst(vectorSubseq(W(255,abcdefg),N( 0),N( 4)),V(abcd))
  etst(vectorSubseq(W(255,abcdefg),N( 0),N( 5)),V(abcde))
  etst(vectorSubseq(W(255,abcdefg),N( 0),N( 6)),V(abcdef))
  etst(vectorSubseq(W(255,abcdefg),N( 0),N( 7)),V(abcdefg))
  etst(vectorSubseq(W(255,abcdefg),N( 0),N( 8)),V(abcdefg))
  etst(vectorSubseq(W(255,abcdefg),N( 0),N( 9)),V(abcdefg))
  etst(vectorSubseq(W(255,abcdefg),N( 0),N(10)),V(abcdefg))
  etst(vectorSubseq(W(255,abcdefg),N( 1),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 1),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 1),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 1),N( 2)),V(b))
  etst(vectorSubseq(W(255,abcdefg),N( 1),N( 3)),V(bc))
  etst(vectorSubseq(W(255,abcdefg),N( 1),N( 4)),V(bcd))
  etst(vectorSubseq(W(255,abcdefg),N( 1),N( 5)),V(bcde))
  etst(vectorSubseq(W(255,abcdefg),N( 1),N( 6)),V(bcdef))
  etst(vectorSubseq(W(255,abcdefg),N( 1),N( 7)),V(bcdefg))
  etst(vectorSubseq(W(255,abcdefg),N( 1),N( 8)),V(bcdefg))
  etst(vectorSubseq(W(255,abcdefg),N( 1),N( 9)),V(bcdefg))
  etst(vectorSubseq(W(255,abcdefg),N( 1),N(10)),V(bcdefg))
  etst(vectorSubseq(W(255,abcdefg),N( 2),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 2),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 2),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 2),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 2),N( 3)),V(c))
  etst(vectorSubseq(W(255,abcdefg),N( 2),N( 4)),V(cd))
  etst(vectorSubseq(W(255,abcdefg),N( 2),N( 5)),V(cde))
  etst(vectorSubseq(W(255,abcdefg),N( 2),N( 6)),V(cdef))
  etst(vectorSubseq(W(255,abcdefg),N( 2),N( 7)),V(cdefg))
  etst(vectorSubseq(W(255,abcdefg),N( 2),N( 8)),V(cdefg))
  etst(vectorSubseq(W(255,abcdefg),N( 2),N( 9)),V(cdefg))
  etst(vectorSubseq(W(255,abcdefg),N( 2),N(10)),V(cdefg))
  etst(vectorSubseq(W(255,abcdefg),N( 3),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 3),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 3),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 3),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 3),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 3),N( 4)),V(d))
  etst(vectorSubseq(W(255,abcdefg),N( 3),N( 5)),V(de))
  etst(vectorSubseq(W(255,abcdefg),N( 3),N( 6)),V(def))
  etst(vectorSubseq(W(255,abcdefg),N( 3),N( 7)),V(defg))
  etst(vectorSubseq(W(255,abcdefg),N( 3),N( 8)),V(defg))
  etst(vectorSubseq(W(255,abcdefg),N( 3),N( 9)),V(defg))
  etst(vectorSubseq(W(255,abcdefg),N( 3),N(10)),V(defg))
  etst(vectorSubseq(W(255,abcdefg),N( 4),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 4),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 4),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 4),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 4),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 4),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 4),N( 5)),V(e))
  etst(vectorSubseq(W(255,abcdefg),N( 4),N( 6)),V(ef))
  etst(vectorSubseq(W(255,abcdefg),N( 4),N( 7)),V(efg))
  etst(vectorSubseq(W(255,abcdefg),N( 4),N( 8)),V(efg))
  etst(vectorSubseq(W(255,abcdefg),N( 4),N( 9)),V(efg))
  etst(vectorSubseq(W(255,abcdefg),N( 4),N(10)),V(efg))
  etst(vectorSubseq(W(255,abcdefg),N( 5),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 5),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 5),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 5),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 5),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 5),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 5),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 5),N( 6)),V(f))
  etst(vectorSubseq(W(255,abcdefg),N( 5),N( 7)),V(fg))
  etst(vectorSubseq(W(255,abcdefg),N( 5),N( 8)),V(fg))
  etst(vectorSubseq(W(255,abcdefg),N( 5),N( 9)),V(fg))
  etst(vectorSubseq(W(255,abcdefg),N( 5),N(10)),V(fg))
  etst(vectorSubseq(W(255,abcdefg),N( 6),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 6),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 6),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 6),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 6),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 6),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 6),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 6),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 6),N( 7)),V(g))
  etst(vectorSubseq(W(255,abcdefg),N( 6),N( 8)),V(g))
  etst(vectorSubseq(W(255,abcdefg),N( 6),N( 9)),V(g))
  etst(vectorSubseq(W(255,abcdefg),N( 6),N(10)),V(g))
  etst(vectorSubseq(W(255,abcdefg),N( 7),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 7),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 7),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 7),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 7),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 7),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 7),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 7),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 7),N( 7)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 7),N( 8)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 7),N( 9)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 7),N(10)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 8),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 8),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 8),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 8),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 8),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 8),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 8),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 8),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 8),N( 7)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 8),N( 8)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 8),N( 9)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 8),N(10)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 9),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 9),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 9),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 9),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 9),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 9),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 9),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 9),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 9),N( 7)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 9),N( 8)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 9),N( 9)),V())
  etst(vectorSubseq(W(255,abcdefg),N( 9),N(10)),V())
  etst(vectorSubseq(W(255,abcdefg),N(10),N(-1)),V())
  etst(vectorSubseq(W(255,abcdefg),N(10),N( 0)),V())
  etst(vectorSubseq(W(255,abcdefg),N(10),N( 1)),V())
  etst(vectorSubseq(W(255,abcdefg),N(10),N( 2)),V())
  etst(vectorSubseq(W(255,abcdefg),N(10),N( 3)),V())
  etst(vectorSubseq(W(255,abcdefg),N(10),N( 4)),V())
  etst(vectorSubseq(W(255,abcdefg),N(10),N( 5)),V())
  etst(vectorSubseq(W(255,abcdefg),N(10),N( 6)),V())
  etst(vectorSubseq(W(255,abcdefg),N(10),N( 7)),V())
  etst(vectorSubseq(W(255,abcdefg),N(10),N( 8)),V())
  etst(vectorSubseq(W(255,abcdefg),N(10),N( 9)),V())
  etst(vectorSubseq(W(255,abcdefg),N(10),N(10)),V())

  xtst(vector2bytes(TT))
  xtst(vector2bytes(FF))
  xtst(vector2bytes(P0))
  xtst(vector2bytes(M0))
  xtst(vector2bytes(B0))
  etst(vector2bytes(M),T)
  etst(vector2bytes(N(-1000000000000000000000000)),T)
  etst(vector2bytes(N(0)),T)
  etst(vector2bytes(N(1)),T)
  etst(vector2bytes(N(255)),T)
  etst(vector2bytes(N(256)),PAIR(N(0),T))
  etst(vector2bytes(N(65537)),PAIR(N(1),PAIR(N(0),T)))
  etst(vector2bytes(V(ABCDEFGHIJ)),
  PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),PAIR(N(69),
  PAIR(N(70),PAIR(N(71),PAIR(N(72),PAIR(N(73),PAIR(N(74),T)))))))))))
  etst(vector2bytes(V(ABCDEFGHI)),
  PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),PAIR(N(69),
  PAIR(N(70),PAIR(N(71),PAIR(N(72),PAIR(N(73),T))))))))))
  etst(vector2bytes(V(ABCDEFGH)),
  PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),PAIR(N(69),
  PAIR(N(70),PAIR(N(71),PAIR(N(72),T)))))))))
  etst(vector2bytes(V(ABCDEFG)),
  PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),PAIR(N(69),
  PAIR(N(70),PAIR(N(71),T))))))))
  etst(vector2bytes(V(ABCDEF)),
  PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),PAIR(N(69),PAIR(N(70),T)))))))
  etst(vector2bytes(V(ABCDE)),
  PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),PAIR(N(69),T))))))
  etst(vector2bytes(V(ABCD)),
  PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),T)))))
  etst(vector2bytes(V(ABC)),PAIR(N(65),PAIR(N(66),PAIR(N(67),T))))
  etst(vector2bytes(V(AB)),PAIR(N(65),PAIR(N(66),T)))
  etst(vector2bytes(V(A)),PAIR(N(65),T))
  etst(vector2bytes(V()),T)

  xtst(vector2vects(TT))
  xtst(vector2vects(FF))
  xtst(vector2vects(P0))
  xtst(vector2vects(M0))
  xtst(vector2vects(B0))
  etst(vector2vects(M),T)
  etst(vector2vects(N(-1000000000000000000000000)),T)
  etst(vector2vects(N(0)),T)
  etst(vector2vects(N(1)),T)
  etst(vector2vects(N(255)),T)
  etst(vector2vects(N(321)),PAIR(V(A),T))
  etst(vector2vects(N(82497)),PAIR(V(A),PAIR(V(B),T)))
  etst(vector2vects(V(ABCDEFGHIJ)),
  PAIR(V(A),PAIR(V(B),PAIR(V(C),PAIR(V(D),PAIR(V(E),
  PAIR(V(F),PAIR(V(G),PAIR(V(H),PAIR(V(I),PAIR(V(J),T)))))))))))
  etst(vector2vects(V(ABCDEFGHI)),
  PAIR(V(A),PAIR(V(B),PAIR(V(C),PAIR(V(D),PAIR(V(E),
  PAIR(V(F),PAIR(V(G),PAIR(V(H),PAIR(V(I),T))))))))))
  etst(vector2vects(V(ABCDEFGH)),
  PAIR(V(A),PAIR(V(B),PAIR(V(C),PAIR(V(D),PAIR(V(E),
  PAIR(V(F),PAIR(V(G),PAIR(V(H),T)))))))))
  etst(vector2vects(V(ABCDEFG)),
  PAIR(V(A),PAIR(V(B),PAIR(V(C),PAIR(V(D),PAIR(V(E),
  PAIR(V(F),PAIR(V(G),T))))))))
  etst(vector2vects(V(ABCDEF)),
  PAIR(V(A),PAIR(V(B),PAIR(V(C),PAIR(V(D),PAIR(V(E),PAIR(V(F),T)))))))
  etst(vector2vects(V(ABCDE)),
  PAIR(V(A),PAIR(V(B),PAIR(V(C),PAIR(V(D),PAIR(V(E),T))))))
  etst(vector2vects(V(ABCD)),PAIR(V(A),PAIR(V(B),PAIR(V(C),PAIR(V(D),T)))))
  etst(vector2vects(V(ABC)),PAIR(V(A),PAIR(V(B),PAIR(V(C),T))))
  etst(vector2vects(V(AB)),PAIR(V(A),PAIR(V(B),T)))
  etst(vector2vects(V(A)),PAIR(V(A),T))

  etst(bt2bytes(T),T)
  etst(bt2bytes(F),T)
  etst(bt2bytes(M0),T)
  etst(bt2bytes(B0),T)
  etst(bt2bytes(N(-1)),T)
  etst(bt2bytes(N(0)),PAIR(N(0),T))
  etst(bt2bytes(N(255)),PAIR(N(255),T))
  etst(bt2bytes(N(256)),T)
  etst(bt2bytes(V()),PAIR(N(1),T))
  etst(bt2bytes(PAIR(N(1),N(2))),PAIR(N(1),PAIR(N(2),T)))
  etst(bt2bytes(PAIR(PAIR(N(1),N(-1)),PAIR(B0,N(100)))),
    PAIR(N(1),PAIR(N(100),T)))

  etst(bt2vects(T),T)
  etst(bt2vects(F),T)
  etst(bt2vects(M0),T)
  etst(bt2vects(B0),T)
  etst(bt2vects(N(-1)),T)
  etst(bt2vects(N(0)),PAIR(bytevect[0],T))
  etst(bt2vects(N(255)),PAIR(bytevect[255],T))
  etst(bt2vects(N(256)),T)
  etst(bt2vects(V()),PAIR(bytevect[1],T))
  etst(bt2vects(PAIR(N(65),N(66))),PAIR(V(A),PAIR(V(B),T)))
  etst(bt2vects(PAIR(PAIR(N(65),N(-1)),PAIR(B0,N(66)))),
    PAIR(V(A),PAIR(V(B),T)))

  etst(vt2bytes(T),T)
  etst(vt2bytes(F),T)
  etst(vt2bytes(M0),T)
  etst(vt2bytes(B0),T)
  etst(vt2bytes(N(-1)),T)
  etst(vt2bytes(N(0)),T)
  etst(vt2bytes(N(1)),T)
  etst(vt2bytes(N(255)),T)
  etst(vt2bytes(N(256)),PAIR(N(0),T))
  etst(vt2bytes(N(513)),PAIR(N(1),T))
  etst(vt2bytes(V()),T)
  etst(vt2bytes(V(abc)),PAIR(N(97),PAIR(N(98),PAIR(N(99),T))))
  etst(vt2bytes(PAIR(V(ab),V(c))),PAIR(N(97),PAIR(N(98),PAIR(N(99),T))))
  etst(vt2bytes(PAIR(PAIR(V(a),N(-1)),PAIR(B0,V(bc)))),
    PAIR(N(97),PAIR(N(98),PAIR(N(99),T))))

  etst(vt2vects(T),T)
  etst(vt2vects(F),T)
  etst(vt2vects(M0),T)
  etst(vt2vects(B0),T)
  etst(vt2vects(N(-1)),T)
  etst(vt2vects(N(0)),T)
  etst(vt2vects(N(1)),T)
  etst(vt2vects(N(255)),T)
  etst(vt2vects(N(256)),PAIR(bytevect[0],T))
  etst(vt2vects(N(513)),PAIR(bytevect[1],T))
  etst(vt2vects(V()),T)
  etst(vt2vects(V(abc)),PAIR(V(a),PAIR(V(b),PAIR(V(c),T))))
  etst(vt2vects(PAIR(V(ab),V(c))),PAIR(V(a),PAIR(V(b),PAIR(V(c),T))))
  etst(vt2vects(PAIR(PAIR(V(a),N(-1)),PAIR(B0,V(bc)))),
    PAIR(V(a),PAIR(V(b),PAIR(V(c),T))))

  etst(bytes2vector(1,vector2bytes(V())),V())
  etst(bytes2vector(1,vector2bytes(V(a))),V(a))
  etst(bytes2vector(1,vector2bytes(V(ab))),V(ab))
  etst(bytes2vector(1,vector2bytes(V(abc))),V(abc))
  etst(bytes2vector(1,vector2bytes(V(abcd))),V(abcd))
  etst(bytes2vector(1,vector2bytes(V(abcde))),V(abcde))
  etst(bytes2vector(1,vector2bytes(V(abcdef))),V(abcdef))
  etst(bytes2vector(1,vector2bytes(V(abcdefg))),V(abcdefg))
  etst(bytes2vector(1,vector2bytes(V(abcdefgh))),V(abcdefgh))
  etst(bytes2vector(1,vector2bytes(V(abcdefghi))),V(abcdefghi))

  etst(bt2vector(T),E)
  etst(bt2vector(F),E)
  etst(bt2vector(M0),E)
  etst(bt2vector(B0),E)
  etst(bt2vector(N(-1)),E)
  etst(bt2vector(N(0)),N(256))
  etst(bt2vector(N(255)),N(511))
  etst(bt2vector(N(256)),E)
  etst(bt2vector(V()),N(257))
  etst(bt2vector(PAIR(N(97),N(98))),V(ab))
  etst(bt2vector(PAIR(PAIR(N(97),N(-1)),PAIR(B0,N(98)))),V(ab))

  etst(vt2vector(T),E)
  etst(vt2vector(F),E)
  etst(vt2vector(M0),E)
  etst(vt2vector(B0),E)
  etst(vt2vector(N(-1)),E)
  etst(vt2vector(N(0)),E)
  etst(vt2vector(N(1)),E)
  etst(vt2vector(N(255)),E)
  etst(vt2vector(N(256)),N(256))
  etst(vt2vector(N(513)),N(257))
  etst(vt2vector(V()),E)
  etst(vt2vector(V(abc)),V(abc))
  etst(vt2vector(PAIR(V(ab),V(c))),V(abc))
  etst(vt2vector(PAIR(PAIR(V(a),N(-1)),PAIR(B0,V(bc)))),V(abc))

  etst(spy(T),V(spy))
  etst(trace(T),V(trace))
  etst(print(T),V(print))
  etst(timer(T),V(timer))

/*
---------------------------------------------
Test of truth and falsehood
---------------------------------------------
*/

  ntst(term2rnf(term_t,NIL))
  ptst(term2rnf(term_lambda(term_t),NIL))

  ntst(term_eval(term_t))
  ptst(term_eval(term_lambda(term_t)))
  ptst(term_eval(term_f))

/*
---------------------------------------------
Test of variables
---------------------------------------------
*/

  ntst(term_eval2(var_0,term_t,term_t))
  ntst(term_eval2(var_0,term_t,term_f))
  ptst(term_eval2(var_0,term_f,term_t))
  ptst(term_eval2(var_0,term_f,term_f))

  ntst(term_eval2(var_1,term_t,term_t))
  ptst(term_eval2(var_1,term_t,term_f))
  ntst(term_eval2(var_1,term_f,term_t))
  ptst(term_eval2(var_1,term_f,term_f))

  ntst(term_eval2(var_2,term_t,term_t))
  ntst(term_eval2(var_2,term_t,term_f))
  ntst(term_eval2(var_2,term_f,term_t))
  ntst(term_eval2(var_2,term_f,term_f))

/*
---------------------------------------------
Test of selection
---------------------------------------------
*/

  ntst(term_eval(term_if(term_t,term_t,term_t)))
  ntst(term_eval(term_if(term_t,term_t,term_f)))
  ptst(term_eval(term_if(term_t,term_f,term_t)))
  ptst(term_eval(term_if(term_t,term_f,term_f)))
  ntst(term_eval(term_if(term_f,term_t,term_t)))
  ptst(term_eval(term_if(term_f,term_t,term_f)))
  ntst(term_eval(term_if(term_f,term_f,term_t)))
  ptst(term_eval(term_if(term_f,term_f,term_f)))

/*
---------------------------------------------
Test of selection/variables
---------------------------------------------
*/

  ntst(term_eval3(term_if(var_0,var_1,var_2),term_t,term_t,term_t))
  ntst(term_eval3(term_if(var_0,var_1,var_2),term_t,term_t,term_f))
  ptst(term_eval3(term_if(var_0,var_1,var_2),term_t,term_f,term_t))
  ptst(term_eval3(term_if(var_0,var_1,var_2),term_t,term_f,term_f))
  ntst(term_eval3(term_if(var_0,var_1,var_2),term_f,term_t,term_t))
  ptst(term_eval3(term_if(var_0,var_1,var_2),term_f,term_t,term_f))
  ntst(term_eval3(term_if(var_0,var_1,var_2),term_f,term_f,term_t))
  ptst(term_eval3(term_if(var_0,var_1,var_2),term_f,term_f,term_f))

/*
---------------------------------------------
Test of apply/lambda
---------------------------------------------
*/

  ntst(term_eval1(term_apply(term_lambda(var_0),term_t),term_t))
  ntst(term_eval1(term_apply(term_lambda(var_0),term_t),term_f))
  ptst(term_eval1(term_apply(term_lambda(var_0),term_f),term_t))
  ptst(term_eval1(term_apply(term_lambda(var_0),term_f),term_f))

  ntst(term_eval1(term_apply(term_lambda(term_t),term_t),term_t))
  ntst(term_eval1(term_apply(term_lambda(term_t),term_t),term_f))
  ntst(term_eval1(term_apply(term_lambda(term_t),term_f),term_t))
  ntst(term_eval1(term_apply(term_lambda(term_t),term_f),term_f))

  ptst(term_eval1(term_apply(term_lambda(term_f),term_t),term_t))
  ptst(term_eval1(term_apply(term_lambda(term_f),term_t),term_f))
  ptst(term_eval1(term_apply(term_lambda(term_f),term_f),term_t))
  ptst(term_eval1(term_apply(term_lambda(term_f),term_f),term_f))

  ntst(term_eval1(term_apply(term_lambda(var_1),term_t),term_t))
  ptst(term_eval1(term_apply(term_lambda(var_1),term_t),term_f))
  ntst(term_eval1(term_apply(term_lambda(var_1),term_f),term_t))
  ptst(term_eval1(term_apply(term_lambda(var_1),term_f),term_f))

/*
---------------------------------------------
Test of Curried functions
---------------------------------------------
*/

  ntst(term_eval(term_apply(term_apply(term_k,term_t),term_t)))
  ntst(term_eval(term_apply(term_apply(term_k,term_t),term_f)))
  ptst(term_eval(term_apply(term_apply(term_k,term_f),term_t)))
  ptst(term_eval(term_apply(term_apply(term_k,term_f),term_f)))

  ntst(term_eval(term_apply(term_apply(term_ki,term_t),term_t)))
  ptst(term_eval(term_apply(term_apply(term_ki,term_t),term_f)))
  ntst(term_eval(term_apply(term_apply(term_ki,term_f),term_t)))
  ptst(term_eval(term_apply(term_apply(term_ki,term_f),term_f)))

/*
---------------------------------------------
Test of unoptimized pair function
---------------------------------------------
*/

  ntst(term_eval(term_hd(term_p(term_t,term_t))))
  ntst(term_eval(term_hd(term_p(term_t,term_f))))
  ptst(term_eval(term_hd(term_p(term_f,term_t))))
  ptst(term_eval(term_hd(term_p(term_f,term_f))))
  ntst(term_eval(term_tl(term_p(term_t,term_t))))
  ptst(term_eval(term_tl(term_p(term_t,term_f))))
  ntst(term_eval(term_tl(term_p(term_f,term_t))))
  ptst(term_eval(term_tl(term_p(term_f,term_f))))

/*
---------------------------------------------
Test of fixed point combinator
---------------------------------------------
*/

  ntst(term_eval(term_apply(term_y,term_t)))
  ntst(term_eval(term_apply(term_y,term_f)))
  ptst(term_eval(term_yk))
  ptst(term_eval(term_apply(term_yk,term_t)))
  ptst(term_eval(term_apply(term_yk,term_f)))
  ptst(term_eval(term_apply(term_apply(term_yk,term_t),term_t)))
  ptst(term_eval(term_apply(term_apply(term_yk,term_f),term_f)))
  ptst(term_eval(term_yki))
  ntst(term_eval(term_apply(term_yki,term_t)))
  ptst(term_eval(term_apply(term_yki,term_f)))

/*
---------------------------------------------
Test of delay_apply
---------------------------------------------
*/

  ntst(map2rnf(delay_apply(map_pair(map_t,map_t),map_t)))
  ntst(map2rnf(delay_apply(map_pair(map_t,map_t),map_f)))
  ntst(map2rnf(delay_apply(map_pair(map_t,map_f),map_t)))
  ptst(map2rnf(delay_apply(map_pair(map_t,map_f),map_f)))
  ptst(map2rnf(delay_apply(map_pair(map_f,map_t),map_t)))
  ntst(map2rnf(delay_apply(map_pair(map_f,map_t),map_f)))
  ptst(map2rnf(delay_apply(map_pair(map_f,map_f),map_t)))
  ptst(map2rnf(delay_apply(map_pair(map_f,map_f),map_f)))

  ntst(map2rnf(delay_apply(map_pair(UT,UB),UT)))
  ptst(map2rnf(delay_apply(map_pair(UF,UB),UT)))
  ntst(map2rnf(delay_apply(map_pair(UB,UT),UF)))
  ptst(map2rnf(delay_apply(map_pair(UB,UF),UF)))

/* The following tests loop indefinitely:
  ntst(map2rnf(delay_apply(map_pair(UB,UT),UT)))
  ntst(map2rnf(delay_apply(map_pair(UT,UB),UF)))
*/

/*
---------------------------------------------
Test of call
---------------------------------------------
*/

  ntst(term_eval(term_call(var_0,pair(term_t,pair(term_t,NIL)))))
  ptst(term_eval(term_call(var_0,pair(term_f,pair(term_t,NIL)))))
  ntst(term_eval(term_call(var_0,pair(term_t,pair(term_f,NIL)))))
  ptst(term_eval(term_call(var_0,pair(term_f,pair(term_f,NIL)))))

  ntst(term_eval(term_call(var_1,pair(term_t,pair(term_t,NIL)))))
  ntst(term_eval(term_call(var_1,pair(term_f,pair(term_t,NIL)))))
  ptst(term_eval(term_call(var_1,pair(term_t,pair(term_f,NIL)))))
  ptst(term_eval(term_call(var_1,pair(term_f,pair(term_f,NIL)))))

  ntst(term_eval(term_call(var_2,pair(term_t,pair(term_t,NIL)))))
  ntst(term_eval(term_call(var_2,pair(term_f,pair(term_t,NIL)))))
  ntst(term_eval(term_call(var_2,pair(term_t,pair(term_f,NIL)))))
  ntst(term_eval(term_call(var_2,pair(term_f,pair(term_f,NIL)))))

  ntst(term_eval(term_call(call_if,pair(term_t,pair(term_t,pair(term_t,NIL))))))
  ntst(term_eval(term_call(call_if,pair(term_t,pair(term_t,pair(term_f,NIL))))))
  ptst(term_eval(term_call(call_if,pair(term_t,pair(term_f,pair(term_t,NIL))))))
  ptst(term_eval(term_call(call_if,pair(term_t,pair(term_f,pair(term_f,NIL))))))
  ntst(term_eval(term_call(call_if,pair(term_f,pair(term_t,pair(term_t,NIL))))))
  ptst(term_eval(term_call(call_if,pair(term_f,pair(term_t,pair(term_f,NIL))))))
  ntst(term_eval(term_call(call_if,pair(term_f,pair(term_f,pair(term_t,NIL))))))
  ptst(term_eval(term_call(call_if,pair(term_f,pair(term_f,pair(term_f,NIL))))))

  ntst(map_apply(map_pair(map_t,map_t),map_t))
  ntst(map_apply(map_pair(map_t,map_t),map_f))
  ntst(map_apply(map_pair(map_t,map_f),map_t))
  ptst(map_apply(map_pair(map_t,map_f),map_f))
  ptst(map_apply(map_pair(map_f,map_t),map_t))
  ntst(map_apply(map_pair(map_f,map_t),map_f))
  ptst(map_apply(map_pair(map_f,map_f),map_t))
  ptst(map_apply(map_pair(map_f,map_f),map_f))

  ntst(term_eval(CALL_HEAD(term_const(map_pair(map_t,map_t)))))
  ntst(term_eval(CALL_HEAD(term_const(map_pair(map_t,map_f)))))
  ptst(term_eval(CALL_HEAD(term_const(map_pair(map_f,map_t)))))
  ptst(term_eval(CALL_HEAD(term_const(map_pair(map_f,map_f)))))

  ntst(map_apply(lambda_head,map_pair(map_t,map_t)))
  ntst(map_apply(lambda_head,map_pair(map_t,map_f)))
  ptst(map_apply(lambda_head,map_pair(map_f,map_t)))
  ptst(map_apply(lambda_head,map_pair(map_f,map_f)))

/*
---------------------------------------------
Test of ecall
---------------------------------------------
*/

  ttst(term_eval(term_ecall2(V0,term_t,term_t)))
  ftst(term_eval(term_ecall2(V0,term_f,term_t)))
  ttst(term_eval(term_ecall2(V0,term_t,term_f)))
  ftst(term_eval(term_ecall2(V0,term_f,term_f)))

  ttst(term_eval(term_ecall2(V1,term_t,term_t)))
  ttst(term_eval(term_ecall2(V1,term_f,term_t)))
  ftst(term_eval(term_ecall2(V1,term_t,term_f)))
  ftst(term_eval(term_ecall2(V1,term_f,term_f)))

  etst(F,term_eval(term_ecall2(V0,term_pair(term_f,term_f),term_t)))

  etst(Z1,term_eval(term_ecall2(E3(if,TT,Z1,Z2),term_t,term_f)))
  etst(Z2,term_eval(term_ecall2(E3(if,FF,Z1,Z2),term_t,term_f)))
  etst(Z1,term_eval(term_ecall2(E3(if,V0,Z1,Z2),term_t,term_f)))
  etst(Z2,term_eval(term_ecall2(E3(if,V1,Z1,Z2),term_t,term_f)))

/*
---------------------------------------------
Test of fct
---------------------------------------------
*/

  ftst(root(fct_if))
  ftst(root(apply(fct_if,MT)))
  ftst(root(apply(apply(fct_if,MT),MT)))
  ttst(root(apply(apply(apply(fct_if,MT),MT),MT)))
  ttst(root(apply(apply(apply(fct_if,MT),MT),MF)))
  ftst(root(apply(apply(apply(fct_if,MT),MF),MT)))
  ftst(root(apply(apply(apply(fct_if,MT),MF),MF)))
  ttst(root(apply(apply(apply(fct_if,MF),MT),MT)))
  ftst(root(apply(apply(apply(fct_if,MF),MT),MF)))
  ttst(root(apply(apply(apply(fct_if,MF),MF),MT)))
  ftst(root(apply(apply(apply(fct_if,MF),MF),MF)))

  ftst(root(fct_apply))
  ftst(root(apply(fct_apply,MKI)))
  ttst(root(apply(apply(fct_apply,MT),MT)))
  ttst(root(apply(apply(fct_apply,MT),MF)))
  ttst(root(apply(apply(fct_apply,MT),MI)))
  ttst(root(apply(apply(fct_apply,MF),MT)))
  ttst(root(apply(apply(fct_apply,MF),MF)))
  ttst(root(apply(apply(fct_apply,MF),MI)))
  ttst(root(apply(apply(fct_apply,MI),MT)))
  ftst(root(apply(apply(fct_apply,MI),MF)))
  ftst(root(apply(apply(fct_apply,MI),MI)))
  ftst(root(apply(apply(fct_apply,MK),MT)))
  ftst(root(apply(apply(fct_apply,MK),MF)))
  ftst(root(apply(apply(fct_apply,MK),MI)))

/*
---------------------------------------------
Test of tagged map operations
---------------------------------------------
*/

  xtst(root(TT))
  xtst(root(FF))
  xtst(root(N(117)))
  xtst(root(P0))
  xtst(root(B0))
  ttst(root(MT))
  ftst(root(MF))

  ttst(root(apply(MY,MT)))
  ttst(root(apply(MY,MF)))
  ftst(root(MYK))
  ftst(root(apply(MYK,MT)))
  ftst(root(apply(MYK,MF)))
  ftst(root(apply(apply(MYK,MT),MT)))
  ftst(root(apply(apply(MYK,MF),MF)))
  ftst(root(MYKI))
  ttst(root(apply(MYKI,MT)))
  ftst(root(apply(MYKI,MF)))
  ftst(root(MPTX))
  ftst(root(apply(MPTX,MF)))
  ttst(root(apply(apply(MPTX,MF),MT)))
  ftst(root(apply(apply(MPTX,MF),MF)))
  ftst(root(MPXT))
  ftst(root(apply(MPXT,MF)))
  ftst(root(apply(apply(MPXT,MF),MT)))
  ttst(root(apply(apply(MPXT,MF),MF)))

  ftst(root(MPXY))
  ftst(root(apply(MPXY,MT)))
  ftst(root(apply(apply(MPXY,MT),MT)))
  ttst(root(apply(apply(apply(MPXY,MT),MT),MT)))
  ftst(root(apply(apply(apply(MPXY,MF),MT),MT)))
  ttst(root(apply(apply(apply(MPXY,MT),MF),MT)))
  ftst(root(apply(apply(apply(MPXY,MF),MF),MT)))
  ttst(root(apply(apply(apply(MPXY,MT),MT),MF)))
  ttst(root(apply(apply(apply(MPXY,MF),MT),MF)))
  ftst(root(apply(apply(apply(MPXY,MT),MF),MF)))
  ftst(root(apply(apply(apply(MPXY,MF),MF),MF)))

  ftst(root(MPYX))
  ftst(root(apply(MPYX,MT)))
  ftst(root(apply(apply(MPYX,MT),MT)))
  ttst(root(apply(apply(apply(MPYX,MT),MT),MT)))
  ttst(root(apply(apply(apply(MPYX,MF),MT),MT)))
  ftst(root(apply(apply(apply(MPYX,MT),MF),MT)))
  ftst(root(apply(apply(apply(MPYX,MF),MF),MT)))
  ttst(root(apply(apply(apply(MPYX,MT),MT),MF)))
  ftst(root(apply(apply(apply(MPYX,MF),MT),MF)))
  ttst(root(apply(apply(apply(MPYX,MT),MF),MF)))
  ftst(root(apply(apply(apply(MPYX,MF),MF),MF)))

  etst(N(0),rnf2card(T));
  etst(N(1),rnf2card(map_pair(F,T)));
  etst(N(0),rnf2card(map_pair(T,map_pair(T,T))));
  etst(N(1),rnf2card(map_pair(F,map_pair(T,T))));
  etst(N(2),rnf2card(map_pair(T,map_pair(F,T))));
  etst(N(3),rnf2card(map_pair(F,map_pair(F,T))));
  etst(N(0),rnf2card(map_pair(T,map_pair(T,map_pair(T,T)))));
  etst(N(1),rnf2card(map_pair(F,map_pair(T,map_pair(T,T)))));
  etst(N(2),rnf2card(map_pair(T,map_pair(F,map_pair(T,T)))));
  etst(N(3),rnf2card(map_pair(F,map_pair(F,map_pair(T,T)))));
  etst(N(4),rnf2card(map_pair(T,map_pair(T,map_pair(F,T)))));
  etst(N(5),rnf2card(map_pair(F,map_pair(T,map_pair(F,T)))));
  etst(N(6),rnf2card(map_pair(T,map_pair(F,map_pair(F,T)))));
  etst(N(7),rnf2card(map_pair(F,map_pair(F,map_pair(F,T)))));
  etst(N(0),rnf2card(map_tail(map_tail(N(0)))))
  etst(N(1),rnf2card(map_tail(map_tail(N(1)))))
  etst(N(2),rnf2card(map_tail(map_tail(N(2)))))
  etst(N(3),rnf2card(map_tail(map_tail(N(3)))))
  etst(N(4),rnf2card(map_tail(map_tail(N(4)))))
  etst(N(5),rnf2card(map_tail(map_tail(N(5)))))
  etst(N(6),rnf2card(map_tail(map_tail(N(6)))))
  etst(N(7),rnf2card(map_tail(map_tail(N(7)))))
  etst(N(729578423057820578023754809),
   rnf2card(map_tail(map_tail(N(729578423057820578023754809)))))
  etst(N(109865398174380174830165017480165098217401),
   rnf2card(map_tail(map_tail(N(109865398174380174830165017480165098217401)))))

  etst(N( 0),rnf2int(T));
  etst(N( 0),rnf2int(map_pair(F,T)));
  etst(N( 0),rnf2int(map_pair(T,map_pair(T,T))));
  etst(N( 0),rnf2int(map_pair(F,map_pair(T,T))));
  etst(N( 1),rnf2int(map_pair(T,map_pair(F,T))));
  etst(N(-1),rnf2int(map_pair(F,map_pair(F,T))));
  etst(N( 0),rnf2int(map_pair(T,map_pair(T,map_pair(T,T)))));
  etst(N( 0),rnf2int(map_pair(F,map_pair(T,map_pair(T,T)))));
  etst(N( 1),rnf2int(map_pair(T,map_pair(F,map_pair(T,T)))));
  etst(N(-1),rnf2int(map_pair(F,map_pair(F,map_pair(T,T)))));
  etst(N( 2),rnf2int(map_pair(T,map_pair(T,map_pair(F,T)))));
  etst(N(-2),rnf2int(map_pair(F,map_pair(T,map_pair(F,T)))));
  etst(N( 3),rnf2int(map_pair(T,map_pair(F,map_pair(F,T)))));
  etst(N(-3),rnf2int(map_pair(F,map_pair(F,map_pair(F,T)))));
  etst(N( 0),rnf2int(map_tail(N( 0))))
  etst(N( 1),rnf2int(map_tail(N( 1))))
  etst(N( 2),rnf2int(map_tail(N( 2))))
  etst(N( 3),rnf2int(map_tail(N( 3))))
  etst(N( 4),rnf2int(map_tail(N( 4))))
  etst(N( 5),rnf2int(map_tail(N( 5))))
  etst(N( 6),rnf2int(map_tail(N( 6))))
  etst(N( 7),rnf2int(map_tail(N( 7))))
  etst(N(-1),rnf2int(map_tail(N(-1))))
  etst(N(-2),rnf2int(map_tail(N(-2))))
  etst(N(-3),rnf2int(map_tail(N(-3))))
  etst(N(-4),rnf2int(map_tail(N(-4))))
  etst(N(-5),rnf2int(map_tail(N(-5))))
  etst(N(-6),rnf2int(map_tail(N(-6))))
  etst(N(-7),rnf2int(map_tail(N(-7))))
  etst(N(729578423057820578023754809),
   rnf2int(map_tail(N(729578423057820578023754809))))
  etst(N(109865398174380174830165017480165098217401),
   rnf2int(map_tail(N(109865398174380174830165017480165098217401))))
  etst(N(-729578423057820578023754809),
   rnf2int(map_tail(N(-729578423057820578023754809))))
  etst(N(-109865398174380174830165017480165098217401),
   rnf2int(map_tail(N(-109865398174380174830165017480165098217401))))

  ttst(root(maptag(UT)))
  ftst(root(maptag(UF)))
  ftst(root(maptag(UI)))
  ttst(root(apply(maptag(UI),MT)))
  ftst(root(apply(maptag(UI),MF)))
  ttst(root(apply(MI,apply(maptag(UI),MT))))
  ftst(root(apply(MI,apply(maptag(UI),MF))))

  etst(untag1(UT      ),T)
  etst(untag1(UF      ),F)
  etst(untag1(UPTT    ),F)
  etst(untag1(UPTF    ),F)
  etst(untag1(UPFT    ),F)
  etst(untag1(UPFF    ),F)
  etst(untag1(UPPTTPTT),F)
  etst(untag1(UPPTTPTF),F)
  etst(untag1(UPPTTPFT),F)
  etst(untag1(UPPTTPFF),F)
  etst(untag1(UPPTFPTT),F)
  etst(untag1(UPPTFPTF),F)
  etst(untag1(UPPTFPFT),F)
  etst(untag1(UPPTFPFF),F)
  etst(untag1(UPPFTPTT),F)
  etst(untag1(UPPFTPTF),F)
  etst(untag1(UPPFTPFT),F)
  etst(untag1(UPPFTPFF),F)
  etst(untag1(UPPFFPTT),F)
  etst(untag1(UPPFFPTF),F)
  etst(untag1(UPPFFPFT),F)
  etst(untag1(UPPFFPFF),F)

  etst(N(0),rnf2card(UCARD(0)))
  etst(N(1),rnf2card(UCARD(1)))
  etst(N(2),rnf2card(UCARD(2)))
  etst(N(3),rnf2card(UCARD(3)))
  etst(N(4),rnf2card(UCARD(4)))
  etst(N(5),rnf2card(UCARD(5)))
  etst(N(6),rnf2card(UCARD(6)))
  etst(N(7),rnf2card(UCARD(7)))
  etst(N(729578423057820578023754809),
   rnf2card(UCARD(729578423057820578023754809)))
  etst(N(109865398174380174830165017480165098217401),
   rnf2card(UCARD(109865398174380174830165017480165098217401)))

  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(0)),UCARD(0))),F)
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(0))),N( 0))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(1))),N( 0))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(2))),N( 1))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(3))),N(-1))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(4))),N( 2))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(5))),N(-2))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(6))),N( 3))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(7))),N(-3))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(2)),MAP_PAIR(UT,UT))),pair(T,T))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(2)),MAP_PAIR(UT,UF))),pair(T,F))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(2)),MAP_PAIR(UF,UT))),pair(F,T))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(2)),MAP_PAIR(UF,UF))),pair(F,F))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(3)),UT)),raise0(T))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(3)),UF)),raise0(F))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(3)),UPPTTPTT)),raise0(F))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(3)),N(117))),raise0(N(117)))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(5)),UT)),objcons(N(0),N(5),T))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(5)),UF)),objcons(N(0),N(5),F))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(5)),UPPTTPTT)),
   objcons(N(0),N(5),F))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(5)),N(117))),
   objcons(N(0),N(5),N(117)))

  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(0)),UT)),objcons(N(1),N(0),T))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(1)),UF)),objcons(N(1),N(1),F))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(2)),UPPTTPTT)),
   objcons(N(1),N(2),F))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(3)),N(117))),
   objcons(N(1),N(3),N(117)))

  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(4)),UT)),objcons(N(1),N(4),T))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(5)),UF)),objcons(N(1),N(5),F))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(6)),UPPTTPTT)),
   objcons(N(1),N(6),F))
  etst(untag1(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(7)),N(117))),
   objcons(N(1),N(7),N(117)))

  etst(untag1(T),T)
  etst(untag1(F),F)
  etst(untag1(N(117)),N(117))
  etst(untag1(PAIR(T,F)),PAIR(T,F))
  etst(untag1(raise0(F)),raise0(F))
  etst(untag1(MI),MI)
  ttst(root(apply(untag1(MI),MT)))
  ftst(root(apply(untag1(MI),MF)))

  ttst(root(apply(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UY)),MT)))
  ttst(root(apply(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UY)),MF)))
  ftst(root(MYK))
  ftst(root(apply(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UYK)),MT)))
  ftst(root(apply(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UYK)),MF)))
  ftst(root(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UYK)),MT),MT)))
  ftst(root(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UYK)),MF),MF)))
  ftst(root(MYKI))
  ttst(root(apply(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UYKI)),MT)))
  ftst(root(apply(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UYKI)),MF)))
  ftst(root(MPTX))
  ftst(root(apply(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPTX)),MF)))
  ttst(root(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPTX)),MF),MT)))
  ftst(root(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPTX)),MF),MF)))
  ftst(root(MPXT))
  ftst(root(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPXT)),MF)))
  ftst(root(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPXT)),MF),MT)))
  ttst(root(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPXT)),MF),MF)))

  ftst(root(untag1(MPXY)))
  ftst(root(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPXY)),MT)))
  ftst(root(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPXY)),MT),MT)))
  ttst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPXY)),MT),MT),MT)))
  ftst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPXY)),MF),MT),MT)))
  ttst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPXY)),MT),MF),MT)))
  ftst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPXY)),MF),MF),MT)))
  ttst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPXY)),MT),MT),MF)))
  ttst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPXY)),MF),MT),MF)))
  ftst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPXY)),MT),MF),MF)))
  ftst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPXY)),MF),MF),MF)))

  ftst(root(untag1(MPYX)))
  ftst(root(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPYX)),MT)))
  ftst(root(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPYX)),MT),MT)))
  ttst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPYX)),MT),MT),MT)))
  ttst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPYX)),MF),MT),MT)))
  ftst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPYX)),MT),MF),MT)))
  ftst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPYX)),MF),MF),MT)))
  ttst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPYX)),MT),MT),MF)))
  ftst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPYX)),MF),MT),MF)))
  ttst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPYX)),MT),MF),MF)))
  ftst(root(apply(apply(apply(
   untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UPYX)),MF),MF),MF)))

  ptst(map_pair(UT,UB))
  ptst(map_pair(UB,UT))
  ntst(map_head(map_pair(UT,UB)))
  ptst(map_head(map_pair(UF,UB)))
  ntst(map_tail(map_pair(UB,UT)))
  ptst(map_tail(map_pair(UB,UF)))

  ntst(mapp(MB))
  ttst(mapp(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UB))))

/*Destructive tests which loop forever:
  ttst(root(MB))
  ntst(map_tail(map_pair(UT,UB)))
  ntst(map_head(map_pair(UB,UT)))
  ttst(root(untag1(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(4)),UB))))
*/

  etst(untag(mapcons(UT      )),T)
  etst(untag(mapcons(UF      )),F)
  etst(untag(mapcons(UPTT    )),F)
  etst(untag(mapcons(UPTF    )),F)
  etst(untag(mapcons(UPFT    )),F)
  etst(untag(mapcons(UPFF    )),F)
  etst(untag(mapcons(UPPTTPTT)),F)
  etst(untag(mapcons(UPPTTPTF)),F)
  etst(untag(mapcons(UPPTTPFT)),F)
  etst(untag(mapcons(UPPTTPFF)),F)
  etst(untag(mapcons(UPPTFPTT)),F)
  etst(untag(mapcons(UPPTFPTF)),F)
  etst(untag(mapcons(UPPTFPFT)),F)
  etst(untag(mapcons(UPPTFPFF)),F)
  etst(untag(mapcons(UPPFTPTT)),F)
  etst(untag(mapcons(UPPFTPTF)),F)
  etst(untag(mapcons(UPPFTPFT)),F)
  etst(untag(mapcons(UPPFTPFF)),F)
  etst(untag(mapcons(UPPFFPTT)),F)
  etst(untag(mapcons(UPPFFPTF)),F)
  etst(untag(mapcons(UPPFFPFT)),F)
  etst(untag(mapcons(UPPFFPFF)),F)

  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(0)),UCARD(0)))),F)
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(0)))),N( 0))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(1)))),N( 0))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(2)))),N( 1))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(3)))),N(-1))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(4)))),N( 2))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(5)))),N(-2))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(6)))),N( 3))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(1)),UCARD(7)))),N(-3))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(2)),MAP_PAIR(UT,UT)))),
   pair(T,T))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(2)),MAP_PAIR(UT,UF)))),
   pair(T,F))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(2)),MAP_PAIR(UF,UT)))),
   pair(F,T))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(2)),MAP_PAIR(UF,UF)))),
   pair(F,F))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(3)),UT))),raise0(T))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(3)),UF))),raise0(F))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(3)),UPPTTPTT))),raise0(F))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(3)),N(117)))),
   raise0(N(117)))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(5)),UT))),
   objcons(N(0),N(5),T))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(5)),UF))),
   objcons(N(0),N(5),F))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(5)),UPPTTPTT))),
   objcons(N(0),N(5),F))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(0),UCARD(5)),N(117)))),
   objcons(N(0),N(5),N(117)))

  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(0)),UT))),
   objcons(N(1),N(0),T))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(1)),UF))),
   objcons(N(1),N(1),F))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(2)),UPPTTPTT))),
   objcons(N(1),N(2),F))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(3)),N(117)))),
   objcons(N(1),N(3),N(117)))

  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(4)),UT))),
   objcons(N(1),N(4),T))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(5)),UF))),
   objcons(N(1),N(5),F))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(6)),UPPTTPTT))),
   objcons(N(1),N(6),F))
  etst(untag(mapcons(MAP_PAIR(MAP_PAIR(UCARD(1),UCARD(7)),N(117)))),
   objcons(N(1),N(7),N(117)))

  etst(untag(mapcons(T)),T)
  etst(untag(mapcons(F)),F)
  etst(untag(mapcons(N(117))),N(117))
  etst(untag(mapcons(PAIR(T,F))),PAIR(T,F))
  etst(untag(mapcons(raise0(F))),raise0(F))
  etst(untag(mapcons(MI)),MI)
  ttst(root(apply(untag(mapcons(MI)),MT)))
  ftst(root(apply(untag(mapcons(MI)),MF)))

/*
---------------------------------------------
Test of eager functions
---------------------------------------------
*/

  etst(eval2(V0,T,T),T)
  etst(eval2(V0,T,F),T)
  etst(eval2(V0,F,T),F)
  etst(eval2(V0,F,F),F)
  etst(eval2(V1,T,T),T)
  etst(eval2(V1,T,F),F)
  etst(eval2(V1,F,T),T)
  etst(eval2(V1,F,F),F)

  etst(eval2(T,T,T),T)
  etst(eval2(T,T,F),T)
  etst(eval2(T,F,T),T)
  etst(eval2(T,F,F),T)
  etst(eval2(F,T,T),F)
  etst(eval2(F,T,F),F)
  etst(eval2(F,F,T),F)
  etst(eval2(F,F,F),F)

  etst(eval0(TT),TT)
  etst(eval0(FF),FF)
  etst(eval0(Z0),Z0)
  etst(eval0(Z1),Z1)
  etst(eval0(Z2),Z2)
  etst(eval0(Z3),Z3)
  etst(eval0(P0),P0)
  etst(eval0(P1),P1)
  etst(eval0(P2),P2)
  etst(eval0(P3),P3)
  etst(eval0(X0),X0)
  etst(eval0(X1),X1)
  etst(eval0(X2),X2)
  etst(eval0(X3),X3)
  etst(eval0(M0),M0)
  etst(eval0(M1),M1)
  etst(eval0(M2),M2)
  etst(eval0(M3),M3)
  etst(eval0(B0),B0)
  etst(eval0(B1),B1)
  etst(eval0(B2),B2)
  etst(eval0(B3),B3)

  etst(eval0(E1(digit0,TT)),XX)
  etst(eval0(E1(digit0,FF)),XX)
  etst(eval0(E1(digit0,P0)),XX)
  etst(eval0(E1(digit0,P1)),XX)
  etst(eval0(E1(digit0,P2)),XX)
  etst(eval0(E1(digit0,P3)),XX)
  etst(eval0(E1(digit0,X0)),X0)
  etst(eval0(E1(digit0,X1)),X1)
  etst(eval0(E1(digit0,X2)),X2)
  etst(eval0(E1(digit0,X3)),X3)
  etst(eval0(E1(digit0,M0)),XX)
  etst(eval0(E1(digit0,M1)),XX)
  etst(eval0(E1(digit0,M2)),XX)
  etst(eval0(E1(digit0,M3)),XX)
  etst(eval0(E1(digit0,B0)),XX)
  etst(eval0(E1(digit0,B1)),XX)
  etst(eval0(E1(digit0,B2)),XX)
  etst(eval0(E1(digit0,B3)),XX)
  etst(eval0(E1(digit0,JN( 0))),JN(  0))
  etst(eval0(E1(digit0,JN( 1))),JN( 10))
  etst(eval0(E1(digit0,JN( 2))),JN( 20))
  etst(eval0(E1(digit0,JN( 3))),JN( 30))
  etst(eval0(E1(digit0,JN( 4))),JN( 40))
  etst(eval0(E1(digit0,JN( 5))),JN( 50))
  etst(eval0(E1(digit0,JN( 6))),JN( 60))
  etst(eval0(E1(digit0,JN( 7))),JN( 70))
  etst(eval0(E1(digit0,JN( 8))),JN( 80))
  etst(eval0(E1(digit0,JN( 9))),JN( 90))
  etst(eval0(E1(digit0,JN(-1))),JN(-10))
  etst(eval0(E1(digit0,JN(-2))),JN(-20))
  etst(eval0(E1(digit0,JN(-3))),JN(-30))
  etst(eval0(E1(digit0,JN(-4))),JN(-40))
  etst(eval0(E1(digit0,JN(-5))),JN(-50))
  etst(eval0(E1(digit0,JN(-6))),JN(-60))
  etst(eval0(E1(digit0,JN(-7))),JN(-70))
  etst(eval0(E1(digit0,JN(-8))),JN(-80))
  etst(eval0(E1(digit0,JN(-9))),JN(-90))

  etst(eval0(E1(digit1,TT)),XX)
  etst(eval0(E1(digit1,FF)),XX)
  etst(eval0(E1(digit1,P0)),XX)
  etst(eval0(E1(digit1,X0)),X0)
  etst(eval0(E1(digit1,M0)),XX)
  etst(eval0(E1(digit1,B0)),XX)
  etst(eval0(E1(digit1,JN( 2))),JN( 21))

  etst(eval0(E1(digit2,TT)),XX)
  etst(eval0(E1(digit2,FF)),XX)
  etst(eval0(E1(digit2,P0)),XX)
  etst(eval0(E1(digit2,X0)),X0)
  etst(eval0(E1(digit2,M0)),XX)
  etst(eval0(E1(digit2,B0)),XX)
  etst(eval0(E1(digit2,JN( 2))),JN( 22))

  etst(eval0(E1(digit3,TT)),XX)
  etst(eval0(E1(digit3,FF)),XX)
  etst(eval0(E1(digit3,P0)),XX)
  etst(eval0(E1(digit3,X0)),X0)
  etst(eval0(E1(digit3,M0)),XX)
  etst(eval0(E1(digit3,B0)),XX)
  etst(eval0(E1(digit3,JN( 2))),JN( 23))

  etst(eval0(E1(digit4,TT)),XX)
  etst(eval0(E1(digit4,FF)),XX)
  etst(eval0(E1(digit4,P0)),XX)
  etst(eval0(E1(digit4,X0)),X0)
  etst(eval0(E1(digit4,M0)),XX)
  etst(eval0(E1(digit4,B0)),XX)
  etst(eval0(E1(digit4,JN( 2))),JN( 24))

  etst(eval0(E1(digit5,TT)),XX)
  etst(eval0(E1(digit5,FF)),XX)
  etst(eval0(E1(digit5,P0)),XX)
  etst(eval0(E1(digit5,X0)),X0)
  etst(eval0(E1(digit5,M0)),XX)
  etst(eval0(E1(digit5,B0)),XX)
  etst(eval0(E1(digit5,JN( 2))),JN( 25))

  etst(eval0(E1(digit6,TT)),XX)
  etst(eval0(E1(digit6,FF)),XX)
  etst(eval0(E1(digit6,P0)),XX)
  etst(eval0(E1(digit6,X0)),X0)
  etst(eval0(E1(digit6,M0)),XX)
  etst(eval0(E1(digit6,B0)),XX)
  etst(eval0(E1(digit6,JN( 2))),JN( 26))

  etst(eval0(E1(digit7,TT)),XX)
  etst(eval0(E1(digit7,FF)),XX)
  etst(eval0(E1(digit7,P0)),XX)
  etst(eval0(E1(digit7,X0)),X0)
  etst(eval0(E1(digit7,M0)),XX)
  etst(eval0(E1(digit7,B0)),XX)
  etst(eval0(E1(digit7,JN( 2))),JN( 27))

  etst(eval0(E1(digit8,TT)),XX)
  etst(eval0(E1(digit8,FF)),XX)
  etst(eval0(E1(digit8,P0)),XX)
  etst(eval0(E1(digit8,X0)),X0)
  etst(eval0(E1(digit8,M0)),XX)
  etst(eval0(E1(digit8,B0)),XX)
  etst(eval0(E1(digit8,JN( 2))),JN( 28))

  etst(eval0(E1(digit9,TT)),XX)
  etst(eval0(E1(digit9,FF)),XX)
  etst(eval0(E1(digit9,P0)),XX)
  etst(eval0(E1(digit9,X0)),X0)
  etst(eval0(E1(digit9,M0)),XX)
  etst(eval0(E1(digit9,B0)),XX)
  etst(eval0(E1(digit9,JN( 2))),JN( 29))

  etst(eval0(E2(times,TT,TT)),XX)
  etst(eval0(E2(times,TT,FF)),XX)
  etst(eval0(E2(times,TT,Z1)),XX)
  etst(eval0(E2(times,TT,P1)),XX)
  etst(eval0(E2(times,TT,X1)),X1)
  etst(eval0(E2(times,TT,M1)),XX)
  etst(eval0(E2(times,TT,B1)),XX)
  etst(eval0(E2(times,FF,TT)),XX)
  etst(eval0(E2(times,FF,FF)),XX)
  etst(eval0(E2(times,FF,Z1)),XX)
  etst(eval0(E2(times,FF,P1)),XX)
  etst(eval0(E2(times,FF,X1)),X1)
  etst(eval0(E2(times,FF,M1)),XX)
  etst(eval0(E2(times,FF,B1)),XX)
  etst(eval0(E2(times,Z0,TT)),XX)
  etst(eval0(E2(times,Z0,FF)),XX)
  etst(eval0(E2(times,Z0,Z1)),Z0)
  etst(eval0(E2(times,Z0,P1)),XX)
  etst(eval0(E2(times,Z0,X1)),X1)
  etst(eval0(E2(times,Z0,M1)),XX)
  etst(eval0(E2(times,Z0,B1)),XX)
  etst(eval0(E2(times,P0,TT)),XX)
  etst(eval0(E2(times,P0,FF)),XX)
  etst(eval0(E2(times,P0,Z1)),XX)
  etst(eval0(E2(times,P0,P1)),XX)
  etst(eval0(E2(times,P0,X1)),X1)
  etst(eval0(E2(times,P0,M1)),XX)
  etst(eval0(E2(times,P0,B1)),XX)
  etst(eval0(E2(times,X0,TT)),X0)
  etst(eval0(E2(times,X0,FF)),X0)
  etst(eval0(E2(times,X0,Z1)),X0)
  etst(eval0(E2(times,X0,P1)),X0)
  etst(eval0(E2(times,X0,X1)),X0)
  etst(eval0(E2(times,X0,M1)),X0)
  etst(eval0(E2(times,X0,B1)),X0)
  etst(eval0(E2(times,M0,TT)),XX)
  etst(eval0(E2(times,M0,FF)),XX)
  etst(eval0(E2(times,M0,Z1)),XX)
  etst(eval0(E2(times,M0,P1)),XX)
  etst(eval0(E2(times,M0,X1)),X1)
  etst(eval0(E2(times,M0,M1)),XX)
  etst(eval0(E2(times,M0,B1)),XX)
  etst(eval0(E2(times,B0,TT)),XX)
  etst(eval0(E2(times,B0,FF)),XX)
  etst(eval0(E2(times,B0,Z1)),XX)
  etst(eval0(E2(times,B0,P1)),XX)
  etst(eval0(E2(times,B0,X1)),X1)
  etst(eval0(E2(times,B0,M1)),XX)
  etst(eval0(E2(times,B0,B1)),XX)
  etst(eval0(E2(times,N(100000000000),N(1000000000))),N(100000000000000000000))

  etst(eval0(E2(plus,TT,TT)),XX)
  etst(eval0(E2(plus,TT,Z1)),XX)
  etst(eval0(E2(plus,TT,X1)),X1)
  etst(eval0(E2(plus,Z0,TT)),XX)
  etst(eval0(E2(plus,Z0,X1)),X1)
  etst(eval0(E2(plus,X0,TT)),X0)
  etst(eval0(E2(plus,X0,Z1)),X0)
  etst(eval0(E2(plus,X0,X1)),X0)
  etst(eval0(E2(plus,N(100000000000),N(1000000000))),N(101000000000))

  etst(eval0(E2(minus,TT,TT)),XX)
  etst(eval0(E2(minus,TT,Z1)),XX)
  etst(eval0(E2(minus,TT,X1)),X1)
  etst(eval0(E2(minus,Z0,TT)),XX)
  etst(eval0(E2(minus,Z0,X1)),X1)
  etst(eval0(E2(minus,X0,TT)),X0)
  etst(eval0(E2(minus,X0,Z1)),X0)
  etst(eval0(E2(minus,X0,X1)),X0)
  etst(eval0(E2(minus,N(100000000000),N(1000000000))),N(99000000000))

  etst(eval0(E2(minus,N(7),N(6))),N(1))
  etst(eval0(E2(minus,N(7),E2(times,N(2),N(3)))),N(1))
  etst(eval0(E2(minus,N(7),E2(minus,N(8),N(2)))),N(1))
  etst(eval0(E2(minus,E2(plus,N(3),N(4)),N(6))),N(1))
  etst(eval0(E2(minus,E2(minus,N(9),N(2)),N(6))),N(1))

  etst(eval0(E2(and,TT,TT)),TT)
  etst(eval0(E2(and,TT,FF)),FF)
  etst(eval0(E2(and,TT,Z1)),Z1)
  etst(eval0(E2(and,TT,X1)),X1)
  etst(eval0(E2(and,FF,TT)),FF)
  etst(eval0(E2(and,FF,FF)),FF)
  etst(eval0(E2(and,FF,Z1)),FF)
  etst(eval0(E2(and,FF,X1)),FF)
  etst(eval0(E2(and,FF,BB)),FF)
  etst(eval0(E2(and,Z0,TT)),Z0)
  etst(eval0(E2(and,Z0,FF)),Z0)
  etst(eval0(E2(and,Z0,Z1)),Z0)
  etst(eval0(E2(and,Z0,X1)),Z0)
  etst(eval0(E2(and,X0,TT)),X0)
  etst(eval0(E2(and,X0,FF)),X0)
  etst(eval0(E2(and,X0,Z1)),X0)
  etst(eval0(E2(and,X0,X1)),X0)

  etst(eval0(E2(or,TT,TT)),TT)
  etst(eval0(E2(or,TT,FF)),TT)
  etst(eval0(E2(or,TT,Z1)),TT)
  etst(eval0(E2(or,TT,X1)),TT)
  etst(eval0(E2(or,TT,BB)),TT)
  etst(eval0(E2(or,FF,TT)),TT)
  etst(eval0(E2(or,FF,FF)),FF)
  etst(eval0(E2(or,FF,Z1)),Z1)
  etst(eval0(E2(or,FF,X1)),X1)
  etst(eval0(E2(or,Z0,TT)),TT)
  etst(eval0(E2(or,Z0,FF)),FF)
  etst(eval0(E2(or,Z0,Z1)),Z1)
  etst(eval0(E2(or,Z0,X1)),X1)
  etst(eval0(E2(or,X0,TT)),X0)
  etst(eval0(E2(or,X0,FF)),X0)
  etst(eval0(E2(or,X0,Z1)),X0)
  etst(eval0(E2(or,X0,X1)),X0)

  etst(eval0(E2(then,TT,TT)),TT)
  etst(eval0(E2(then,TT,FF)),FF)
  etst(eval0(E2(then,TT,Z1)),Z1)
  etst(eval0(E2(then,TT,X1)),X1)
  etst(eval0(E2(then,FF,TT)),TT)
  etst(eval0(E2(then,FF,FF)),FF)
  etst(eval0(E2(then,FF,Z1)),Z1)
  etst(eval0(E2(then,FF,X1)),X1)
  etst(eval0(E2(then,Z0,TT)),TT)
  etst(eval0(E2(then,Z0,FF)),FF)
  etst(eval0(E2(then,Z0,Z1)),Z1)
  etst(eval0(E2(then,Z0,X1)),X1)
  etst(eval0(E2(then,X0,TT)),X0)
  etst(eval0(E2(then,X0,FF)),X0)
  etst(eval0(E2(then,X0,Z1)),X0)
  etst(eval0(E2(then,X0,X1)),X0)
  etst(eval0(E2(then,TT,N(117))),N(117))
  etst(eval0(E2(then,FF,N(117))),N(117))
  etst(eval0(E2(then,Z0,N(117))),N(117))
  etst(eval0(E2(then,P0,N(117))),N(117))
  etst(eval0(E2(then,X0,N(117))),X0)
  etst(eval0(E2(then,M0,N(117))),N(117))
  etst(eval0(E2(then,B0,N(117))),N(117))

  etst(eval0(E2(pair,TT,TT)),pair(TT,TT))
  etst(eval0(E2(pair,TT,Z1)),pair(TT,Z1))
  etst(eval0(E2(pair,TT,X1)),X1)
  etst(eval0(E2(pair,Z0,TT)),pair(Z0,TT))
  etst(eval0(E2(pair,Z0,Z1)),pair(Z0,Z1))
  etst(eval0(E2(pair,Z0,X1)),X1)
  etst(eval0(E2(pair,X0,TT)),X0)
  etst(eval0(E2(pair,X0,Z1)),X0)
  etst(eval0(E2(pair,X0,X1)),X0)

  etst(eval0(E2(ge,TT,TT)),XX)
  etst(eval0(E2(ge,TT,Z1)),XX)
  etst(eval0(E2(ge,TT,X1)),X1)
  etst(eval0(E2(ge,Z0,TT)),XX)
  etst(eval0(E2(ge,Z0,X1)),X1)
  etst(eval0(E2(ge,X0,TT)),X0)
  etst(eval0(E2(ge,X0,Z1)),X0)
  etst(eval0(E2(ge,X0,X1)),X0)
  ttst(eval0(E2(ge,N(5),N(5))))
  ftst(eval0(E2(ge,N(5),N(8))))
  ttst(eval0(E2(ge,N(8),N(5))))

  etst(eval0(E2(gt,TT,TT)),XX)
  etst(eval0(E2(gt,TT,Z1)),XX)
  etst(eval0(E2(gt,TT,X1)),X1)
  etst(eval0(E2(gt,Z0,TT)),XX)
  etst(eval0(E2(gt,Z0,X1)),X1)
  etst(eval0(E2(gt,X0,TT)),X0)
  etst(eval0(E2(gt,X0,Z1)),X0)
  etst(eval0(E2(gt,X0,X1)),X0)
  ftst(eval0(E2(gt,N(5),N(5))))
  ftst(eval0(E2(gt,N(5),N(8))))
  ttst(eval0(E2(gt,N(8),N(5))))

  etst(eval0(E2(equal,TT,TT)),TT)
  etst(eval0(E2(equal,TT,Z1)),FF)
  etst(eval0(E2(equal,TT,X1)),X1)
  etst(eval0(E2(equal,Z0,TT)),FF)
  etst(eval0(E2(equal,Z0,X1)),X1)
  etst(eval0(E2(equal,X0,TT)),X0)
  etst(eval0(E2(equal,X0,Z1)),X0)
  etst(eval0(E2(equal,X0,X1)),X0)
  ttst(eval0(E2(equal,N(5),N(5))))
  ftst(eval0(E2(equal,N(5),N(8))))
  ftst(eval0(E2(equal,N(8),N(5))))

  etst(eval0(E2(le,TT,TT)),XX)
  etst(eval0(E2(le,TT,Z1)),XX)
  etst(eval0(E2(le,TT,X1)),X1)
  etst(eval0(E2(le,Z0,TT)),XX)
  etst(eval0(E2(le,Z0,X1)),X1)
  etst(eval0(E2(le,X0,TT)),X0)
  etst(eval0(E2(le,X0,Z1)),X0)
  etst(eval0(E2(le,X0,X1)),X0)
  ttst(eval0(E2(le,N(5),N(5))))
  ttst(eval0(E2(le,N(5),N(8))))
  ftst(eval0(E2(le,N(8),N(5))))

  etst(eval0(E2(lt,TT,TT)),XX)
  etst(eval0(E2(lt,TT,Z1)),XX)
  etst(eval0(E2(lt,TT,X1)),X1)
  etst(eval0(E2(lt,Z0,TT)),XX)
  etst(eval0(E2(lt,Z0,X1)),X1)
  etst(eval0(E2(lt,X0,TT)),X0)
  etst(eval0(E2(lt,X0,Z1)),X0)
  etst(eval0(E2(lt,X0,X1)),X0)
  ftst(eval0(E2(lt,N(5),N(5))))
  ttst(eval0(E2(lt,N(5),N(8))))
  ftst(eval0(E2(lt,N(8),N(5))))

  etst(eval0(E2(apply,TT,TT)),XX)
  etst(eval0(E2(apply,TT,M1)),XX)
  etst(eval0(E2(apply,TT,X1)),X1)
  etst(eval0(E2(apply,M0,TT)),XX)
  etst(eval0(E2(apply,M0,M1)),M0)
  etst(eval0(E2(apply,M0,X1)),X1)
  etst(eval0(E2(apply,X0,TT)),X0)
  etst(eval0(E2(apply,X0,M1)),X0)
  etst(eval0(E2(apply,X0,X1)),X0)

  ttst(root(MT))
  ftst(root(MF))
  ttst(root(M0))
  ftst(root(M1))
  ftst(root(M2))
  ftst(root(M3))
  ttst(root(apply(M0,MT)))
  ttst(root(apply(M0,MF)))
  ttst(root(apply(M1,MT)))
  ttst(root(apply(M1,MF)))
  ftst(root(apply(M2,MT)))
  ftst(root(apply(M2,MF)))
  ftst(root(apply(M3,MT)))
  ftst(root(apply(M3,MF)))
  ttst(root(apply(apply(M0,MT),MT)))
  ttst(root(apply(apply(M0,MT),MF)))
  ttst(root(apply(apply(M0,MF),MT)))
  ttst(root(apply(apply(M0,MF),MF)))
  ttst(root(apply(apply(M1,MT),MT)))
  ttst(root(apply(apply(M1,MT),MF)))
  ttst(root(apply(apply(M1,MF),MT)))
  ttst(root(apply(apply(M1,MF),MF)))
  ttst(root(apply(apply(M2,MT),MT)))
  ttst(root(apply(apply(M2,MT),MF)))
  ftst(root(apply(apply(M2,MF),MT)))
  ftst(root(apply(apply(M2,MF),MF)))
  ttst(root(apply(apply(M3,MT),MT)))
  ftst(root(apply(apply(M3,MT),MF)))
  ttst(root(apply(apply(M3,MF),MT)))
  ftst(root(apply(apply(M3,MF),MF)))

  ttst(root(eval0(E2(apply,M0,MT))))
  ttst(root(eval0(E2(apply,M0,MF))))
  ttst(root(eval0(E2(apply,M1,MT))))
  ttst(root(eval0(E2(apply,M1,MF))))
  ftst(root(eval0(E2(apply,M2,MT))))
  ftst(root(eval0(E2(apply,M2,MF))))
  ftst(root(eval0(E2(apply,M3,MT))))
  ftst(root(eval0(E2(apply,M3,MF))))
  ttst(root(eval0(E2(apply,E2(apply,M0,MT),MT))))
  ttst(root(eval0(E2(apply,E2(apply,M0,MT),MF))))
  ttst(root(eval0(E2(apply,E2(apply,M0,MF),MT))))
  ttst(root(eval0(E2(apply,E2(apply,M0,MF),MF))))
  ttst(root(eval0(E2(apply,E2(apply,M1,MT),MT))))
  ttst(root(eval0(E2(apply,E2(apply,M1,MT),MF))))
  ttst(root(eval0(E2(apply,E2(apply,M1,MF),MT))))
  ttst(root(eval0(E2(apply,E2(apply,M1,MF),MF))))
  ttst(root(eval0(E2(apply,E2(apply,M2,MT),MT))))
  ttst(root(eval0(E2(apply,E2(apply,M2,MT),MF))))
  ftst(root(eval0(E2(apply,E2(apply,M2,MF),MT))))
  ftst(root(eval0(E2(apply,E2(apply,M2,MF),MF))))
  ttst(root(eval0(E2(apply,E2(apply,M3,MT),MT))))
  ftst(root(eval0(E2(apply,E2(apply,M3,MT),MF))))
  ttst(root(eval0(E2(apply,E2(apply,M3,MF),MT))))
  ftst(root(eval0(E2(apply,E2(apply,M3,MF),MF))))

  etst(eval0(E1(boolp,TT)),TT)
  etst(eval0(E1(boolp,FF)),TT)
  etst(eval0(E1(boolp,Z0)),FF)
  etst(eval0(E1(boolp,P0)),FF)
  etst(eval0(E1(boolp,X0)),X0)
  etst(eval0(E1(boolp,M0)),FF)
  etst(eval0(E1(boolp,B0)),FF)

  etst(eval0(E1(catch,TT)),pair(FF,TT))
  etst(eval0(E1(catch,FF)),pair(FF,FF))
  etst(eval0(E1(catch,Z0)),pair(FF,Z0))
  etst(eval0(E1(catch,P0)),pair(FF,P0))
  etst(eval0(E1(catch,X0)),pair(TT,Z0))
  etst(eval0(E1(catch,M0)),pair(FF,M0))
  etst(eval0(E1(catch,B0)),pair(FF,B0))

  etst(eval0(E1(maptag1,TT)),M0)
  etst(eval0(E1(maptag1,FF)),M0)
  etst(eval0(E1(maptag1,Z0)),M0)
  etst(eval0(E1(maptag1,P0)),M0)
  etst(eval0(E1(maptag1,X0)),M0)
  etst(eval0(E1(maptag1,M0)),M0)
  etst(eval0(E1(maptag1,B0)),M0)

  etst(untag(eval0(E1(maptag1,TT))),TT)
  etst(untag(eval0(E1(maptag1,FF))),FF)
  etst(untag(eval0(E1(maptag1,Z0))),Z0)
  etst(untag(eval0(E1(maptag1,P0))),P0)
  etst(untag(eval0(E1(maptag1,X0))),X0)
  etst(untag(eval0(E1(maptag1,M0))),M0)
  etst(untag(eval0(E1(maptag1,B0))),B0)

  etst(eval0(E2(div,TT,TT)),XX)
  etst(eval0(E2(div,TT,Z1)),XX)
  etst(eval0(E2(div,TT,X1)),X1)
  etst(eval0(E2(div,Z0,TT)),XX)
  etst(eval0(E2(div,Z0,X1)),X1)
  etst(eval0(E2(div,X0,TT)),X0)
  etst(eval0(E2(div,X0,Z1)),X0)
  etst(eval0(E2(div,X0,X1)),X0)
  etst(eval0(E2(div,N( 1234),N(100))),N( 12))
  etst(eval0(E2(div,N(-1234),N(100))),N(-13))
  etst(eval0(E2(div,N( 1250),N(100))),N( 12))
  etst(eval0(E2(div,N( 1350),N(100))),N( 13))

  etst(eval0(E1(head,TT)),TT)
  etst(eval0(E1(head,FF)),FF)
  etst(eval0(E1(head,Z0)),Z0)
  etst(eval0(E1(head,X0)),X0)
  etst(eval0(E1(head,M0)),M0)
  etst(eval0(E1(head,B0)),B0)
  etst(eval0(E1(head,P0)),TT)
  etst(eval0(E1(head,P1)),TT)
  etst(eval0(E1(head,P2)),FF)
  etst(eval0(E1(head,P3)),FF)

  etst(eval0(E1(intp,TT)),FF)
  etst(eval0(E1(intp,FF)),FF)
  etst(eval0(E1(intp,Z0)),TT)
  etst(eval0(E1(intp,P0)),FF)
  etst(eval0(E1(intp,X0)),X0)
  etst(eval0(E1(intp,M0)),FF)
  etst(eval0(E1(intp,B0)),FF)

  etst(eval0(E2(boolg,TT,N(117))),N(117))
  etst(eval0(E2(boolg,FF,N(117))),N(117))
  etst(eval0(E2(boolg,Z0,N(117))),XX)
  etst(eval0(E2(boolg,P0,N(117))),XX)
  etst(eval0(E2(boolg,X0,N(117))),X0)
  etst(eval0(E2(boolg,M0,N(117))),XX)
  etst(eval0(E2(boolg,B0,N(117))),XX)

  etst(eval0(E2(intg,TT,N(117))),XX)
  etst(eval0(E2(intg,FF,N(117))),XX)
  etst(eval0(E2(intg,Z0,N(117))),N(117))
  etst(eval0(E2(intg,P0,N(117))),XX)
  etst(eval0(E2(intg,X0,N(117))),X0)
  etst(eval0(E2(intg,M0,N(117))),XX)
  etst(eval0(E2(intg,B0,N(117))),XX)

  etst(eval0(E2(mapg,TT,N(117))),XX)
  etst(eval0(E2(mapg,FF,N(117))),XX)
  etst(eval0(E2(mapg,Z0,N(117))),XX)
  etst(eval0(E2(mapg,P0,N(117))),XX)
  etst(eval0(E2(mapg,X0,N(117))),X0)
  etst(eval0(E2(mapg,M0,N(117))),N(117))
  etst(eval0(E2(mapg,B0,N(117))),XX)

  etst(eval0(E2(objg,TT,N(117))),XX)
  etst(eval0(E2(objg,FF,N(117))),XX)
  etst(eval0(E2(objg,Z0,N(117))),XX)
  etst(eval0(E2(objg,P0,N(117))),XX)
  etst(eval0(E2(objg,X0,N(117))),X0)
  etst(eval0(E2(objg,M0,N(117))),XX)
  etst(eval0(E2(objg,B0,N(117))),N(117))

  etst(eval0(E2(pairg,TT,N(117))),XX)
  etst(eval0(E2(pairg,FF,N(117))),XX)
  etst(eval0(E2(pairg,Z0,N(117))),XX)
  etst(eval0(E2(pairg,P0,N(117))),N(117))
  etst(eval0(E2(pairg,X0,N(117))),X0)
  etst(eval0(E2(pairg,M0,N(117))),XX)
  etst(eval0(E2(pairg,B0,N(117))),XX)

  etst(eval0(E1(mapp,TT)),FF)
  etst(eval0(E1(mapp,FF)),FF)
  etst(eval0(E1(mapp,Z0)),FF)
  etst(eval0(E1(mapp,P0)),FF)
  etst(eval0(E1(mapp,X0)),X0)
  etst(eval0(E1(mapp,M0)),TT)
  etst(eval0(E1(mapp,B0)),FF)

  etst(eval0(E1(maptag,TT)),M0)
  etst(eval0(E1(maptag,FF)),M0)
  etst(eval0(E1(maptag,Z0)),M0)
  etst(eval0(E1(maptag,P0)),M0)
  etst(eval0(E1(maptag,X0)),X0)
  etst(eval0(E1(maptag,M0)),M0)
  etst(eval0(E1(maptag,B0)),M0)

  etst(untag(eval0(E1(maptag,TT))),TT)
  etst(untag(eval0(E1(maptag,FF))),FF)
  etst(untag(eval0(E1(maptag,Z0))),Z0)
  etst(untag(eval0(E1(maptag,P0))),P0)
  etst(untag(eval0(E1(maptag,M0))),M0)
  etst(untag(eval0(E1(maptag,B0))),B0)

  etst(eval0(E2(mod,TT,TT)),XX)
  etst(eval0(E2(mod,TT,Z1)),XX)
  etst(eval0(E2(mod,TT,X1)),X1)
  etst(eval0(E2(mod,Z0,TT)),XX)
  etst(eval0(E2(mod,Z0,X1)),X1)
  etst(eval0(E2(mod,X0,TT)),X0)
  etst(eval0(E2(mod,X0,Z1)),X0)
  etst(eval0(E2(mod,X0,X1)),X0)
  etst(eval0(E2(mod,N( 1234),N(100))),N(34))
  etst(eval0(E2(mod,N(-1234),N(100))),N(66))
  etst(eval0(E2(mod,N( 1250),N(100))),N(50))
  etst(eval0(E2(mod,N( 1350),N(100))),N(50))

  etst(eval0(E1(norm,TT)),TT)
  etst(eval0(E1(norm,FF)),FF)
  etst(eval0(E1(norm,Z0)),Z0)
  etst(eval0(E1(norm,P0)),P0)
  etst(eval0(E1(norm,X0)),X0)
  etst(eval0(E1(norm,M0)),M0)
  etst(eval0(E1(norm,B0)),B0)

  etst(eval0(E1(objp,TT)),FF)
  etst(eval0(E1(objp,FF)),FF)
  etst(eval0(E1(objp,Z0)),FF)
  etst(eval0(E1(objp,P0)),FF)
  etst(eval0(E1(objp,X0)),X0)
  etst(eval0(E1(objp,M0)),FF)
  etst(eval0(E1(objp,B0)),TT)

  etst(eval0(E1(pairp,TT)),FF)
  etst(eval0(E1(pairp,FF)),FF)
  etst(eval0(E1(pairp,Z0)),FF)
  etst(eval0(E1(pairp,P0)),TT)
  etst(eval0(E1(pairp,X0)),X0)
  etst(eval0(E1(pairp,M0)),FF)
  etst(eval0(E1(pairp,B0)),FF)

  etst(eval0(E1(raise,TT)),XX)
  etst(eval0(E1(raise,I[0])),X0)
  etst(eval0(E1(raise,I[1])),X1)
  etst(eval0(E1(raise,I[2])),X2)
  etst(eval0(E1(raise,I[3])),X3)
  etst(eval0(E1(raise,X0)),X0)

  etst(eval0(E1(root,TT)),XX)
  etst(eval0(E1(root,FF)),XX)
  etst(eval0(E1(root,Z0)),XX)
  etst(eval0(E1(root,P0)),XX)
  etst(eval0(E1(root,X0)),X0)
  etst(eval0(E1(root,B0)),XX)
  etst(eval0(E1(root,M0)),TT)
  etst(eval0(E1(root,M1)),FF)
  etst(eval0(E1(root,M2)),FF)
  etst(eval0(E1(root,M3)),FF)
  ttst(eval0(E1(root,E2(apply,M0,MT))))
  ttst(eval0(E1(root,E2(apply,M0,MF))))
  ttst(eval0(E1(root,E2(apply,M1,MT))))
  ttst(eval0(E1(root,E2(apply,M1,MF))))
  ftst(eval0(E1(root,E2(apply,M2,MT))))
  ftst(eval0(E1(root,E2(apply,M2,MF))))
  ftst(eval0(E1(root,E2(apply,M3,MT))))
  ftst(eval0(E1(root,E2(apply,M3,MF))))
  ttst(eval0(E1(root,E2(apply,E2(apply,M0,MT),MT))))
  ttst(eval0(E1(root,E2(apply,E2(apply,M0,MT),MF))))
  ttst(eval0(E1(root,E2(apply,E2(apply,M0,MF),MT))))
  ttst(eval0(E1(root,E2(apply,E2(apply,M0,MF),MF))))
  ttst(eval0(E1(root,E2(apply,E2(apply,M1,MT),MT))))
  ttst(eval0(E1(root,E2(apply,E2(apply,M1,MT),MF))))
  ttst(eval0(E1(root,E2(apply,E2(apply,M1,MF),MT))))
  ttst(eval0(E1(root,E2(apply,E2(apply,M1,MF),MF))))
  ttst(eval0(E1(root,E2(apply,E2(apply,M2,MT),MT))))
  ttst(eval0(E1(root,E2(apply,E2(apply,M2,MT),MF))))
  ftst(eval0(E1(root,E2(apply,E2(apply,M2,MF),MT))))
  ftst(eval0(E1(root,E2(apply,E2(apply,M2,MF),MF))))
  ttst(eval0(E1(root,E2(apply,E2(apply,M3,MT),MT))))
  ftst(eval0(E1(root,E2(apply,E2(apply,M3,MT),MF))))
  ttst(eval0(E1(root,E2(apply,E2(apply,M3,MF),MT))))
  ftst(eval0(E1(root,E2(apply,E2(apply,M3,MF),MF))))

  etst(eval0(E1(tail,TT)),TT)
  etst(eval0(E1(tail,FF)),FF)
  etst(eval0(E1(tail,Z0)),Z0)
  etst(eval0(E1(tail,X0)),X0)
  etst(eval0(E1(tail,M0)),M0)
  etst(eval0(E1(tail,B0)),B0)
  etst(eval0(E1(tail,P0)),TT)
  etst(eval0(E1(tail,P1)),FF)
  etst(eval0(E1(tail,P2)),TT)
  etst(eval0(E1(tail,P3)),FF)

  etst(eval0(E1(untag,TT)),XX)
  etst(eval0(E1(untag,FF)),XX)
  etst(eval0(E1(untag,Z0)),XX)
  etst(eval0(E1(untag,P0)),XX)
  etst(eval0(E1(untag,X0)),X0)
  etst(eval0(E1(untag,B0)),XX)
  etst(eval0(E1(untag,M0)),TT)
  etst(eval0(E1(untag,E1(maptag1,TT))),TT)
  etst(eval0(E1(untag,E1(maptag1,FF))),FF)
  etst(eval0(E1(untag,E1(maptag1,Z0))),Z0)
  etst(eval0(E1(untag,E1(maptag1,P0))),P0)
  etst(eval0(E1(untag,E1(maptag1,X0))),X0)
  etst(eval0(E1(untag,E1(maptag1,M0))),M0)
  etst(eval0(E1(untag,E1(maptag1,B0))),B0)

  etst(eval0(E0(digitend)),Z)

  etst(eval0(E1(par,TT)),TT)
  etst(eval0(E1(par,FF)),FF)
  etst(eval0(E1(par,Z0)),Z0)
  etst(eval0(E1(par,P0)),P0)
  etst(eval0(E1(par,X0)),X0)
  etst(eval0(E1(par,M0)),M0)
  etst(eval0(E1(par,B0)),B0)

  etst(eval0(E1(uplus,TT)),XX)
  etst(eval0(E1(uplus,FF)),XX)
  etst(eval0(E1(uplus,N(117))),N(117))
  etst(eval0(E1(uplus,P0)),XX)
  etst(eval0(E1(uplus,X0)),X0)
  etst(eval0(E1(uplus,M0)),XX)
  etst(eval0(E1(uplus,B0)),XX)

  etst(eval0(E1(dplus,TT)),XX)
  etst(eval0(E1(dplus,FF)),XX)
  etst(eval0(E1(dplus,N(117))),N(117))
  etst(eval0(E1(dplus,P0)),XX)
  etst(eval0(E1(dplus,X0)),X0)
  etst(eval0(E1(dplus,M0)),XX)
  etst(eval0(E1(dplus,B0)),XX)

  etst(eval0(E1(uminus,TT)),XX)
  etst(eval0(E1(uminus,FF)),XX)
  etst(eval0(E1(uminus,N(117))),N(-117))
  etst(eval0(E1(uminus,P0)),XX)
  etst(eval0(E1(uminus,X0)),X0)
  etst(eval0(E1(uminus,M0)),XX)
  etst(eval0(E1(uminus,B0)),XX)

  etst(eval0(E1(dminus,TT)),XX)
  etst(eval0(E1(dminus,FF)),XX)
  etst(eval0(E1(dminus,N(117))),N(-117))
  etst(eval0(E1(dminus,P0)),XX)
  etst(eval0(E1(dminus,X0)),X0)
  etst(eval0(E1(dminus,M0)),XX)
  etst(eval0(E1(dminus,B0)),XX)

  etst(eval0(E1(not,TT)),FF)
  etst(eval0(E1(not,FF)),TT)
  etst(eval0(E1(not,Z0)),TT)
  etst(eval0(E1(not,P0)),TT)
  etst(eval0(E1(not,X0)),X0)
  etst(eval0(E1(not,M0)),TT)
  etst(eval0(E1(not,B0)),TT)

  etst(eval0(E0(Base)),N(10))

  etst(eval2(E2(LET,T,V0),T,T),T)
  etst(eval2(E2(LET,T,V0),T,F),T)
  etst(eval2(E2(LET,T,V0),F,T),T)
  etst(eval2(E2(LET,T,V0),F,F),T)
  etst(eval2(E2(LET,F,V0),T,T),F)
  etst(eval2(E2(LET,F,V0),T,F),F)
  etst(eval2(E2(LET,F,V0),F,T),F)
  etst(eval2(E2(LET,F,V0),F,F),F)
  etst(eval2(E2(LET,X1,V0),T,T),X1)
  etst(eval2(E2(LET,X1,V0),T,F),X1)
  etst(eval2(E2(LET,X1,V0),F,T),X1)
  etst(eval2(E2(LET,X1,V0),F,F),X1)

  etst(eval2(E2(LET,T,V1),T,T),T)
  etst(eval2(E2(LET,T,V1),T,F),T)
  etst(eval2(E2(LET,T,V1),F,T),F)
  etst(eval2(E2(LET,T,V1),F,F),F)
  etst(eval2(E2(LET,F,V1),T,T),T)
  etst(eval2(E2(LET,F,V1),T,F),T)
  etst(eval2(E2(LET,F,V1),F,T),F)
  etst(eval2(E2(LET,F,V1),F,F),F)
  etst(eval2(E2(LET,X1,V1),T,T),X1)
  etst(eval2(E2(LET,X1,V1),T,F),X1)
  etst(eval2(E2(LET,X1,V1),F,T),X1)
  etst(eval2(E2(LET,X1,V1),F,F),X1)

  etst(eval2(E2(LET,T,V2),T,T),T)
  etst(eval2(E2(LET,T,V2),T,F),F)
  etst(eval2(E2(LET,T,V2),F,T),T)
  etst(eval2(E2(LET,T,V2),F,F),F)
  etst(eval2(E2(LET,F,V2),T,T),T)
  etst(eval2(E2(LET,F,V2),T,F),F)
  etst(eval2(E2(LET,F,V2),F,T),T)
  etst(eval2(E2(LET,F,V2),F,F),F)
  etst(eval2(E2(LET,X1,V2),T,T),X1)
  etst(eval2(E2(LET,X1,V2),T,F),X1)
  etst(eval2(E2(LET,X1,V2),F,T),X1)
  etst(eval2(E2(LET,X1,V2),F,F),X1)

  etst(eval2(E2(LET,T,T),T,T),T)
  etst(eval2(E2(LET,T,T),T,F),T)
  etst(eval2(E2(LET,T,T),F,T),T)
  etst(eval2(E2(LET,T,T),F,F),T)
  etst(eval2(E2(LET,F,T),T,T),T)
  etst(eval2(E2(LET,F,T),T,F),T)
  etst(eval2(E2(LET,F,T),F,T),T)
  etst(eval2(E2(LET,F,T),F,F),T)
  etst(eval2(E2(LET,X1,T),T,T),X1)
  etst(eval2(E2(LET,X1,T),T,F),X1)
  etst(eval2(E2(LET,X1,T),F,T),X1)
  etst(eval2(E2(LET,X1,T),F,F),X1)

  etst(eval2(E2(LET,T,F),T,T),F)
  etst(eval2(E2(LET,T,F),T,F),F)
  etst(eval2(E2(LET,T,F),F,T),F)
  etst(eval2(E2(LET,T,F),F,F),F)
  etst(eval2(E2(LET,F,F),T,T),F)
  etst(eval2(E2(LET,F,F),T,F),F)
  etst(eval2(E2(LET,F,F),F,T),F)
  etst(eval2(E2(LET,F,F),F,F),F)
  etst(eval2(E2(LET,X1,F),T,T),X1)
  etst(eval2(E2(LET,X1,F),T,F),X1)
  etst(eval2(E2(LET,X1,F),F,T),X1)
  etst(eval2(E2(LET,X1,F),F,F),X1)

  etst(eval2(E2(LET,T,X0),T,T),X0)
  etst(eval2(E2(LET,T,X0),T,F),X0)
  etst(eval2(E2(LET,T,X0),F,T),X0)
  etst(eval2(E2(LET,T,X0),F,F),X0)
  etst(eval2(E2(LET,F,X0),T,T),X0)
  etst(eval2(E2(LET,F,X0),T,F),X0)
  etst(eval2(E2(LET,F,X0),F,T),X0)
  etst(eval2(E2(LET,F,X0),F,F),X0)
  etst(eval2(E2(LET,X1,X0),T,T),X1)
  etst(eval2(E2(LET,X1,X0),T,F),X1)
  etst(eval2(E2(LET,X1,X0),F,T),X1)
  etst(eval2(E2(LET,X1,X0),F,F),X1)

  etst(eval2(E2(LET,Z1,E2(LET,Z0,V0)),Z2,Z3),Z0)
  etst(eval2(E2(LET,Z1,E2(LET,Z0,V1)),Z2,Z3),Z1)
  etst(eval2(E2(LET,Z1,E2(LET,Z0,V2)),Z2,Z3),Z2)
  etst(eval2(E2(LET,Z1,E2(LET,Z0,V3)),Z2,Z3),Z3)
  etst(eval2(E2(LET,Z1,E2(LET,Z0,TT)),Z2,Z3),TT)
  etst(eval2(E2(LET,Z1,E2(LET,Z0,FF)),Z2,Z3),FF)
  etst(eval2(E2(LET,Z1,E2(LET,Z0,X0)),Z2,Z3),X0)
  etst(eval2(E2(LET,Z1,E2(LET,X1,X0)),Z2,Z3),X1)
  etst(eval2(E2(LET,X2,E2(LET,X1,X0)),Z2,Z3),X2)
  etst(eval2(E2(LET,Z1,E2(LET,X1,BB)),Z2,Z3),X1)
  etst(eval2(E2(LET,X2,E2(LET,BB,BB)),Z2,Z3),X2)

  etst(eval0(E2(ash,TT,TT)),XX)
  etst(eval0(E2(ash,TT,Z1)),XX)
  etst(eval0(E2(ash,TT,X1)),X1)
  etst(eval0(E2(ash,Z0,TT)),XX)
  etst(eval0(E2(ash,Z0,X1)),X1)
  etst(eval0(E2(ash,X0,TT)),X0)
  etst(eval0(E2(ash,X0,Z1)),X0)
  etst(eval0(E2(ash,X0,X1)),X0)
  etst(eval0(E2(ash,X0,BB)),X0)
  etst(eval0(E2(ash,N(1),N(10))),N(1024))

/*
  Tests that bottom does not return a value:
  ttst(eval0(E0(bottom)))
  ttst(eval0(B))
  ttst(eval0(BB))
*/

  etst(eval0(E1(bt2bytes,PAIR(PAIR(N(1),N(-1)),PAIR(B0,N(100))))),
    PAIR(N(1),PAIR(N(100),T)))
  etst(eval0(E1(bt2bytes,X1)),X1)

  etst(eval0(E1(bt2vector,PAIR(PAIR(N(97),N(-1)),PAIR(B0,N(98))))),V(ab))
  etst(eval0(E1(bt2vector,X1)),X1)

  etst(eval0(E2(ceiling,TT,TT)),XX)
  etst(eval0(E2(ceiling,TT,Z1)),XX)
  etst(eval0(E2(ceiling,TT,X1)),X1)
  etst(eval0(E2(ceiling,Z0,TT)),XX)
  etst(eval0(E2(ceiling,Z0,X1)),X1)
  etst(eval0(E2(ceiling,X0,TT)),X0)
  etst(eval0(E2(ceiling,X0,Z1)),X0)
  etst(eval0(E2(ceiling,X0,X1)),X0)
  etst(eval0(E2(ceiling,X0,BB)),X0)
  etst(eval0(E2(ceiling,N( 1234),N(100))),PAIR(N( 13),N(-66)))
  etst(eval0(E2(ceiling,N(-1234),N(100))),PAIR(N(-12),N(-34)))
  etst(eval0(E2(ceiling,N( 1250),N(100))),PAIR(N( 13),N(-50)))
  etst(eval0(E2(ceiling,N( 1350),N(100))),PAIR(N( 14),N(-50)))

  etst(eval0(E1(destruct,TT)),XX)
  etst(eval0(E1(destruct,FF)),XX)
  etst(eval0(E1(destruct,Z0)),XX)
  etst(eval0(E1(destruct,P0)),XX)
  etst(eval0(E1(destruct,X0)),X0)
  etst(eval0(E1(destruct,M0)),XX)
  etst(eval0(E1(destruct,B0)),PAIR(PAIR(N(1),N(2)),N(3)))
  etst(eval0(E1(destruct,B1)),PAIR(PAIR(N(9),N(2)),N(3)))
  etst(eval0(E1(destruct,B2)),PAIR(PAIR(N(1),N(9)),N(3)))
  etst(eval0(E1(destruct,B3)),PAIR(PAIR(N(1),N(2)),N(9)))

  etst(eval0(E1(evenp,TT)),XX)
  etst(eval0(E1(evenp,FF)),XX)
  etst(eval0(E1(evenp,N(116))),TT)
  etst(eval0(E1(evenp,N(117))),FF)
  etst(eval0(E1(evenp,P0)),XX)
  etst(eval0(E1(evenp,X0)),X0)
  etst(eval0(E1(evenp,M0)),XX)
  etst(eval0(E1(evenp,B0)),XX)

  etst(eval0(E0(exception)),X)

  etst(eval0(E0(false)),F)

  etst(eval0(E2(floor,TT,TT)),XX)
  etst(eval0(E2(floor,TT,Z1)),XX)
  etst(eval0(E2(floor,TT,X1)),X1)
  etst(eval0(E2(floor,Z0,TT)),XX)
  etst(eval0(E2(floor,Z0,X1)),X1)
  etst(eval0(E2(floor,X0,TT)),X0)
  etst(eval0(E2(floor,X0,Z1)),X0)
  etst(eval0(E2(floor,X0,X1)),X0)
  etst(eval0(E2(floor,X0,BB)),X0)
  etst(eval0(E2(floor,N( 1234),N(100))),PAIR(N( 12),N(34)))
  etst(eval0(E2(floor,N(-1234),N(100))),PAIR(N(-13),N(66)))
  etst(eval0(E2(floor,N( 1250),N(100))),PAIR(N( 12),N(50)))
  etst(eval0(E2(floor,N( 1350),N(100))),PAIR(N( 13),N(50)))

  etst(eval0(E1(half,TT)),XX)
  etst(eval0(E1(half,FF)),XX)
  etst(eval0(E1(half,N(116))),N(58))
  etst(eval0(E1(half,N(117))),N(58))
  etst(eval0(E1(half,P0)),XX)
  etst(eval0(E1(half,X0)),X0)
  etst(eval0(E1(half,M0)),XX)
  etst(eval0(E1(half,B0)),XX)

  etst(eval0(E3(if,TT,Z1,Z2)),Z1)
  etst(eval0(E3(if,TT,Z1,X0)),Z1)
  etst(eval0(E3(if,TT,Z1,BB)),Z1)
  etst(eval0(E3(if,FF,Z1,Z2)),Z2)
  etst(eval0(E3(if,FF,X0,Z2)),Z2)
  etst(eval0(E3(if,FF,BB,Z2)),Z2)
  etst(eval0(E3(if,Z0,Z1,Z2)),Z2)
  etst(eval0(E3(if,Z0,X0,Z2)),Z2)
  etst(eval0(E3(if,Z0,BB,Z2)),Z2)
  etst(eval0(E3(if,P0,Z1,Z2)),Z2)
  etst(eval0(E3(if,P0,X0,Z2)),Z2)
  etst(eval0(E3(if,P0,BB,Z2)),Z2)
  etst(eval0(E3(if,M0,Z1,Z2)),Z2)
  etst(eval0(E3(if,M0,X0,Z2)),Z2)
  etst(eval0(E3(if,M0,BB,Z2)),Z2)
  etst(eval0(E3(if,B0,Z1,Z2)),Z2)
  etst(eval0(E3(if,B0,X0,Z2)),Z2)
  etst(eval0(E3(if,B0,BB,Z2)),Z2)
  etst(eval0(E3(if,X0,Z1,Z2)),X0)
  etst(eval0(E3(if,X0,BB,BB)),X0)

  etst(eval0(E1(intlength,TT)),XX)
  etst(eval0(E1(intlength,FF)),XX)
  etst(eval0(E1(intlength,N(117))),N(7))
  etst(eval0(E1(intlength,P0)),XX)
  etst(eval0(E1(intlength,X0)),X0)
  etst(eval0(E1(intlength,M0)),XX)
  etst(eval0(E1(intlength,B0)),XX)

  etst(eval0(E2(logand,TT,TT)),XX)
  etst(eval0(E2(logand,TT,Z1)),XX)
  etst(eval0(E2(logand,TT,X1)),X1)
  etst(eval0(E2(logand,Z0,TT)),XX)
  etst(eval0(E2(logand,Z0,X1)),X1)
  etst(eval0(E2(logand,X0,TT)),X0)
  etst(eval0(E2(logand,X0,Z1)),X0)
  etst(eval0(E2(logand,X0,X1)),X0)
  etst(eval0(E2(logand,X0,BB)),X0)
  etst(eval0(E2(logand,I0011,I0101)),I0001)

  etst(eval0(E2(logandc1,TT,TT)),XX)
  etst(eval0(E2(logandc1,TT,Z1)),XX)
  etst(eval0(E2(logandc1,TT,X1)),X1)
  etst(eval0(E2(logandc1,Z0,TT)),XX)
  etst(eval0(E2(logandc1,Z0,X1)),X1)
  etst(eval0(E2(logandc1,X0,TT)),X0)
  etst(eval0(E2(logandc1,X0,Z1)),X0)
  etst(eval0(E2(logandc1,X0,X1)),X0)
  etst(eval0(E2(logandc1,X0,BB)),X0)
  etst(eval0(E2(logandc1,lognot(I0011),I0101)),I0001)

  etst(eval0(E2(logandc2,TT,TT)),XX)
  etst(eval0(E2(logandc2,TT,Z1)),XX)
  etst(eval0(E2(logandc2,TT,X1)),X1)
  etst(eval0(E2(logandc2,Z0,TT)),XX)
  etst(eval0(E2(logandc2,Z0,X1)),X1)
  etst(eval0(E2(logandc2,X0,TT)),X0)
  etst(eval0(E2(logandc2,X0,Z1)),X0)
  etst(eval0(E2(logandc2,X0,X1)),X0)
  etst(eval0(E2(logandc2,X0,BB)),X0)
  etst(eval0(E2(logandc2,I0011,lognot(I0101))),I0001)

  etst(eval0(E2(logbitp,TT,TT)),XX)
  etst(eval0(E2(logbitp,TT,Z1)),XX)
  etst(eval0(E2(logbitp,TT,X1)),X1)
  etst(eval0(E2(logbitp,Z0,TT)),XX)
  etst(eval0(E2(logbitp,Z0,X1)),X1)
  etst(eval0(E2(logbitp,X0,TT)),X0)
  etst(eval0(E2(logbitp,X0,Z1)),X0)
  etst(eval0(E2(logbitp,X0,X1)),X0)
  etst(eval0(E2(logbitp,X0,BB)),X0)
  ttst(eval0(E2(logbitp,N(0),I0011)))
  ttst(eval0(E2(logbitp,N(1),I0011)))
  ftst(eval0(E2(logbitp,N(2),I0011)))
  ftst(eval0(E2(logbitp,N(3),I0011)))

  etst(eval0(E1(logcount,TT)),XX)
  etst(eval0(E1(logcount,FF)),XX)
  etst(eval0(E1(logcount,P0)),XX)
  etst(eval0(E1(logcount,X0)),X0)
  etst(eval0(E1(logcount,M0)),XX)
  etst(eval0(E1(logcount,B0)),XX)
  etst(eval0(E1(logcount,I0000)),N(0))
  etst(eval0(E1(logcount,I0001)),N(1))
  etst(eval0(E1(logcount,I0010)),N(1))
  etst(eval0(E1(logcount,I0011)),N(2))
  etst(eval0(E1(logcount,I0100)),N(1))
  etst(eval0(E1(logcount,I0101)),N(2))
  etst(eval0(E1(logcount,I0110)),N(2))
  etst(eval0(E1(logcount,I0111)),N(3))

  etst(eval0(E2(logeqv,TT,TT)),XX)
  etst(eval0(E2(logeqv,TT,Z1)),XX)
  etst(eval0(E2(logeqv,TT,X1)),X1)
  etst(eval0(E2(logeqv,Z0,TT)),XX)
  etst(eval0(E2(logeqv,Z0,X1)),X1)
  etst(eval0(E2(logeqv,X0,TT)),X0)
  etst(eval0(E2(logeqv,X0,Z1)),X0)
  etst(eval0(E2(logeqv,X0,X1)),X0)
  etst(eval0(E2(logeqv,X0,BB)),X0)
  etst(eval0(E2(logeqv,lognot(I0011),I0101)),I0110)

  etst(eval0(E2(logior,TT,TT)),XX)
  etst(eval0(E2(logior,TT,Z1)),XX)
  etst(eval0(E2(logior,TT,X1)),X1)
  etst(eval0(E2(logior,Z0,TT)),XX)
  etst(eval0(E2(logior,Z0,X1)),X1)
  etst(eval0(E2(logior,X0,TT)),X0)
  etst(eval0(E2(logior,X0,Z1)),X0)
  etst(eval0(E2(logior,X0,X1)),X0)
  etst(eval0(E2(logior,X0,BB)),X0)
  etst(eval0(E2(logior,I0011,I0101)),I0111)

  etst(eval0(E2(lognand,TT,TT)),XX)
  etst(eval0(E2(lognand,TT,Z1)),XX)
  etst(eval0(E2(lognand,TT,X1)),X1)
  etst(eval0(E2(lognand,Z0,TT)),XX)
  etst(eval0(E2(lognand,Z0,X1)),X1)
  etst(eval0(E2(lognand,X0,TT)),X0)
  etst(eval0(E2(lognand,X0,Z1)),X0)
  etst(eval0(E2(lognand,X0,X1)),X0)
  etst(eval0(E2(lognand,X0,BB)),X0)
  etst(eval0(E2(lognand,I0011,I0101)),lognot(I0001))

  etst(eval0(E2(lognor,TT,TT)),XX)
  etst(eval0(E2(lognor,TT,Z1)),XX)
  etst(eval0(E2(lognor,TT,X1)),X1)
  etst(eval0(E2(lognor,Z0,TT)),XX)
  etst(eval0(E2(lognor,Z0,X1)),X1)
  etst(eval0(E2(lognor,X0,TT)),X0)
  etst(eval0(E2(lognor,X0,Z1)),X0)
  etst(eval0(E2(lognor,X0,X1)),X0)
  etst(eval0(E2(lognor,X0,BB)),X0)
  etst(eval0(E2(lognor,I0011,I0101)),lognot(I0111))

  etst(eval0(E1(lognot,TT)),XX)
  etst(eval0(E1(lognot,FF)),XX)
  etst(eval0(E1(lognot,I0011)),J1100)
  etst(eval0(E1(lognot,P0)),XX)
  etst(eval0(E1(lognot,X0)),X0)
  etst(eval0(E1(lognot,M0)),XX)
  etst(eval0(E1(lognot,B0)),XX)

  etst(eval0(E2(logorc1,TT,TT)),XX)
  etst(eval0(E2(logorc1,TT,Z1)),XX)
  etst(eval0(E2(logorc1,TT,X1)),X1)
  etst(eval0(E2(logorc1,Z0,TT)),XX)
  etst(eval0(E2(logorc1,Z0,X1)),X1)
  etst(eval0(E2(logorc1,X0,TT)),X0)
  etst(eval0(E2(logorc1,X0,Z1)),X0)
  etst(eval0(E2(logorc1,X0,X1)),X0)
  etst(eval0(E2(logorc1,X0,BB)),X0)
  etst(eval0(E2(logorc1,lognot(I0011),I0101)),I0111)

  etst(eval0(E2(logorc2,TT,TT)),XX)
  etst(eval0(E2(logorc2,TT,Z1)),XX)
  etst(eval0(E2(logorc2,TT,X1)),X1)
  etst(eval0(E2(logorc2,Z0,TT)),XX)
  etst(eval0(E2(logorc2,Z0,X1)),X1)
  etst(eval0(E2(logorc2,X0,TT)),X0)
  etst(eval0(E2(logorc2,X0,Z1)),X0)
  etst(eval0(E2(logorc2,X0,X1)),X0)
  etst(eval0(E2(logorc2,X0,BB)),X0)
  etst(eval0(E2(logorc2,I0011,lognot(I0101))),I0111)

  etst(eval0(E2(logtest,TT,TT)),XX)
  etst(eval0(E2(logtest,TT,Z1)),XX)
  etst(eval0(E2(logtest,TT,X1)),X1)
  etst(eval0(E2(logtest,Z0,TT)),XX)
  etst(eval0(E2(logtest,Z0,X1)),X1)
  etst(eval0(E2(logtest,X0,TT)),X0)
  etst(eval0(E2(logtest,X0,Z1)),X0)
  etst(eval0(E2(logtest,X0,X1)),X0)
  etst(eval0(E2(logtest,X0,BB)),X0)
  ttst(eval0(E2(logtest,I0011,I0101)))
  ftst(eval0(E2(logtest,I0011,I1100)))

  etst(eval0(E2(logxor,TT,TT)),XX)
  etst(eval0(E2(logxor,TT,Z1)),XX)
  etst(eval0(E2(logxor,TT,X1)),X1)
  etst(eval0(E2(logxor,Z0,TT)),XX)
  etst(eval0(E2(logxor,Z0,X1)),X1)
  etst(eval0(E2(logxor,X0,TT)),X0)
  etst(eval0(E2(logxor,X0,Z1)),X0)
  etst(eval0(E2(logxor,X0,X1)),X0)
  etst(eval0(E2(logxor,X0,BB)),X0)
  etst(eval0(E2(logxor,I0011,I0101)),I0110)

  etst(eval0(E1(map,TT)),M0)
  etst(eval0(E1(map,FF)),M0)
  etst(eval0(E1(map,X0)),M0)
  etst(eval0(E1(map,BB)),M0)
  etst(untag(eval0(E1(map,TT))),TT)
  etst(untag(eval0(E1(map,FF))),FF)
  etst(untag(eval0(E1(map,X0))),X0)
  ftst(eval0(E1(root,E1(map,UPTT))))
  ttst(eval0(E1(root,E2(apply,E1(map,UPTT),E1(map,T)))))
  ttst(eval0(E1(root,E2(apply,E1(map,UPTF),E1(map,T)))))
  ftst(eval0(E1(root,E2(apply,E1(map,UPFT),E1(map,T)))))
  ftst(eval0(E1(root,E2(apply,E1(map,UPFF),E1(map,T)))))
  ttst(eval0(E1(root,E2(apply,E1(map,UPTT),E1(map,F)))))
  ftst(eval0(E1(root,E2(apply,E1(map,UPTF),E1(map,F)))))
  ttst(eval0(E1(root,E2(apply,E1(map,UPFT),E1(map,F)))))
  ftst(eval0(E1(root,E2(apply,E1(map,UPFF),E1(map,F)))))

  etst(eval0(E1(prenorm,TT)),TT)
  etst(eval0(E1(prenorm,FF)),FF)
  etst(eval0(E1(prenorm,Z0)),Z0)
  etst(eval0(E1(prenorm,P0)),P0)
  etst(eval0(E1(prenorm,X0)),X0)
  etst(eval0(E1(prenorm,M0)),M0)
  etst(eval0(E1(prenorm,B0)),B0)

  etst(eval0(E1(notnot,TT)),TT)
  etst(eval0(E1(notnot,FF)),FF)
  etst(eval0(E1(notnot,Z0)),FF)
  etst(eval0(E1(notnot,P0)),FF)
  etst(eval0(E1(notnot,X0)),X0)
  etst(eval0(E1(notnot,M0)),FF)
  etst(eval0(E1(notnot,B0)),FF)

  etst(eval0(E1(object,TT)),XX)
  etst(eval0(E1(object,FF)),XX)
  etst(eval0(E1(object,Z0)),XX)
  etst(eval0(E1(object,P0)),XX)
  etst(eval0(E1(object,X0)),X0)
  etst(eval0(E1(object,M0)),XX)
  etst(eval0(E1(object,B0)),XX)
  etst(eval0(E1(object,PAIR(PAIR(I[1],I[2]),I[3]))),B0)

  etst(eval0(E1(print,T)),V(print))

  etst(eval0(E2(round,TT,TT)),XX)
  etst(eval0(E2(round,TT,Z1)),XX)
  etst(eval0(E2(round,TT,X1)),X1)
  etst(eval0(E2(round,Z0,TT)),XX)
  etst(eval0(E2(round,Z0,X1)),X1)
  etst(eval0(E2(round,X0,TT)),X0)
  etst(eval0(E2(round,X0,Z1)),X0)
  etst(eval0(E2(round,X0,X1)),X0)
  etst(eval0(E2(round,X0,BB)),X0)
  etst(eval0(E2(round,N( 1234),N(100))),PAIR(N( 12),N( 34)))
  etst(eval0(E2(round,N(-1234),N(100))),PAIR(N(-12),N(-34)))
  etst(eval0(E2(round,N( 1250),N(100))),PAIR(N( 12),N( 50)))
  etst(eval0(E2(round,N( 1350),N(100))),PAIR(N( 14),N(-50)))

  etst(eval0(E1(spy,T)),V(spy))
  etst(eval0(E1(timer,T)),V(timer))
  etst(eval0(E1(trace,T)),V(trace))

  etst(eval0(E2(truncate,TT,TT)),XX)
  etst(eval0(E2(truncate,TT,Z1)),XX)
  etst(eval0(E2(truncate,TT,X1)),X1)
  etst(eval0(E2(truncate,Z0,TT)),XX)
  etst(eval0(E2(truncate,Z0,X1)),X1)
  etst(eval0(E2(truncate,X0,TT)),X0)
  etst(eval0(E2(truncate,X0,Z1)),X0)
  etst(eval0(E2(truncate,X0,X1)),X0)
  etst(eval0(E2(truncate,X0,BB)),X0)
  etst(eval0(E2(truncate,N( 1234),N(100))),PAIR(N( 12),N( 34)))
  etst(eval0(E2(truncate,N(-1234),N(100))),PAIR(N(-12),N(-34)))
  etst(eval0(E2(truncate,N( 1250),N(100))),PAIR(N( 12),N( 50)))
  etst(eval0(E2(truncate,N( 1350),N(100))),PAIR(N( 13),N( 50)))

  etst(eval0(E1(vector,TT)),XX)
  etst(eval0(E1(vector,FF)),XX)
  etst(eval0(E1(vector,P0)),XX)
  etst(eval0(E1(vector,X0)),X0)
  etst(eval0(E1(vector,M0)),XX)
  etst(eval0(E1(vector,B0)),XX)
  etst(eval0(E1(vector,N(513))),N(257))

  etst(eval0(E1(vempty,TT)),XX)
  etst(eval0(E1(vempty,FF)),XX)
  etst(eval0(E1(vempty,P0)),XX)
  etst(eval0(E1(vempty,X0)),X0)
  etst(eval0(E1(vempty,M0)),XX)
  etst(eval0(E1(vempty,B0)),XX)
  ttst(eval0(E1(vempty,N(255))))
  ftst(eval0(E1(vempty,N(513))))

  etst(eval0(E2(vindex,TT,TT)),XX)
  etst(eval0(E2(vindex,TT,Z1)),XX)
  etst(eval0(E2(vindex,TT,X1)),X1)
  etst(eval0(E2(vindex,Z0,TT)),XX)
  etst(eval0(E2(vindex,Z0,X1)),X1)
  etst(eval0(E2(vindex,X0,TT)),X0)
  etst(eval0(E2(vindex,X0,Z1)),X0)
  etst(eval0(E2(vindex,X0,X1)),X0)
  etst(eval0(E2(vindex,X0,BB)),X0)
  etst(eval0(E2(vindex,V(abc),N(0))),N(97))
  etst(eval0(E2(vindex,V(abc),N(1))),N(98))
  etst(eval0(E2(vindex,V(abc),N(2))),N(99))

  etst(eval0(E1(vlength,TT)),XX)
  etst(eval0(E1(vlength,FF)),XX)
  etst(eval0(E1(vlength,P0)),XX)
  etst(eval0(E1(vlength,X0)),X0)
  etst(eval0(E1(vlength,M0)),XX)
  etst(eval0(E1(vlength,B0)),XX)
  etst(eval0(E1(vlength,N(255))),N(0))
  etst(eval0(E1(vlength,N(513))),N(1))

  etst(eval0(E2(vprefix,TT,TT)),XX)
  etst(eval0(E2(vprefix,TT,Z1)),XX)
  etst(eval0(E2(vprefix,TT,X1)),X1)
  etst(eval0(E2(vprefix,Z0,TT)),XX)
  etst(eval0(E2(vprefix,Z0,X1)),X1)
  etst(eval0(E2(vprefix,X0,TT)),X0)
  etst(eval0(E2(vprefix,X0,Z1)),X0)
  etst(eval0(E2(vprefix,X0,X1)),X0)
  etst(eval0(E2(vprefix,X0,BB)),X0)
  etst(eval0(E2(vprefix,V(abc),N(0))),E)
  etst(eval0(E2(vprefix,V(abc),N(1))),V(a))
  etst(eval0(E2(vprefix,V(abc),N(2))),V(ab))
  etst(eval0(E2(vprefix,V(abc),N(3))),V(abc))

  etst(eval0(E3(vsubseq,TT,TT,TT)),XX)
  etst(eval0(E3(vsubseq,TT,TT,Z1)),XX)
  etst(eval0(E3(vsubseq,TT,TT,X1)),X1)
  etst(eval0(E3(vsubseq,TT,Z0,TT)),XX)
  etst(eval0(E3(vsubseq,TT,Z0,Z0)),XX)
  etst(eval0(E3(vsubseq,TT,Z0,X1)),X1)
  etst(eval0(E3(vsubseq,TT,X0,TT)),X0)
  etst(eval0(E3(vsubseq,TT,X0,Z1)),X0)
  etst(eval0(E3(vsubseq,TT,X0,X1)),X0)
  etst(eval0(E3(vsubseq,TT,X0,BB)),X0)
  etst(eval0(E3(vsubseq,Z0,TT,TT)),XX)
  etst(eval0(E3(vsubseq,Z0,TT,Z1)),XX)
  etst(eval0(E3(vsubseq,Z0,TT,X1)),X1)
  etst(eval0(E3(vsubseq,Z0,Z0,TT)),XX)
  etst(eval0(E3(vsubseq,Z0,Z0,X1)),X1)
  etst(eval0(E3(vsubseq,Z0,X0,TT)),X0)
  etst(eval0(E3(vsubseq,Z0,X0,Z1)),X0)
  etst(eval0(E3(vsubseq,Z0,X0,X1)),X0)
  etst(eval0(E3(vsubseq,Z0,X0,BB)),X0)
  etst(eval0(E3(vsubseq,X2,TT,TT)),X2)
  etst(eval0(E3(vsubseq,X2,TT,Z1)),X2)
  etst(eval0(E3(vsubseq,X2,TT,X1)),X2)
  etst(eval0(E3(vsubseq,X2,Z0,TT)),X2)
  etst(eval0(E3(vsubseq,X2,Z0,Z0)),X2)
  etst(eval0(E3(vsubseq,X2,Z0,X1)),X2)
  etst(eval0(E3(vsubseq,X2,X0,TT)),X2)
  etst(eval0(E3(vsubseq,X2,X0,Z1)),X2)
  etst(eval0(E3(vsubseq,X2,X0,X1)),X2)
  etst(eval0(E3(vsubseq,X2,X0,BB)),X2)
  etst(eval0(E3(vsubseq,X2,BB,BB)),X2)
  etst(eval0(E3(vsubseq,V(abcdefghi),N(3),N(6))),V(def))

  etst(eval0(E2(vsuffix,TT,TT)),XX)
  etst(eval0(E2(vsuffix,TT,Z1)),XX)
  etst(eval0(E2(vsuffix,TT,X1)),X1)
  etst(eval0(E2(vsuffix,Z0,TT)),XX)
  etst(eval0(E2(vsuffix,Z0,X1)),X1)
  etst(eval0(E2(vsuffix,X0,TT)),X0)
  etst(eval0(E2(vsuffix,X0,Z1)),X0)
  etst(eval0(E2(vsuffix,X0,X1)),X0)
  etst(eval0(E2(vsuffix,X0,BB)),X0)
  etst(eval0(E2(vsuffix,V(abc),N(0))),V(abc))
  etst(eval0(E2(vsuffix,V(abc),N(1))),V(bc))
  etst(eval0(E2(vsuffix,V(abc),N(2))),V(c))
  etst(eval0(E2(vsuffix,V(abc),N(3))),E)

  etst(eval0(E1(vt2bytes,PAIR(PAIR(V(a),N(-1)),PAIR(B0,V(bc))))),
    PAIR(N(97),PAIR(N(98),PAIR(N(99),T))))
  etst(eval0(E1(vt2bytes,X1)),X1)

  etst(eval0(E1(vt2v,PAIR(PAIR(V(a),N(-1)),PAIR(B0,V(bc))))),V(abc))
  etst(eval0(E1(vt2v,X1)),X1)

  etst(eval0(E2(ecall,V0,PAIR(Z0,PAIR(Z1,T)))),Z1)
  etst(eval0(E2(ecall,V1,PAIR(Z0,PAIR(Z1,T)))),Z0)
  etst(eval0(E2(ecall,Z2,PAIR(Z0,PAIR(Z1,T)))),Z2)
  etst(eval0(E2(ecall,X0,PAIR(Z0,PAIR(Z1,T)))),X0)
  etst(eval0(E2(ecall,X0,PAIR(Z0,PAIR(X1,T)))),X1)
  etst(eval0(E2(ecall,BB,PAIR(Z0,PAIR(X1,T)))),X1)
  etst(eval0(E2(ecall,X0,PAIR(X2,PAIR(X1,T)))),X2)
  etst(eval0(E2(ecall,BB,PAIR(X2,PAIR(BB,T)))),X2)
  etst(N(23),
    eval2(
      E2(ecall,E2(minus,V1,V0),PAIR(N(20),PAIR(N(-3),T))),
      N(10),N(3)))
  etst(N(23),
    eval2(
      E2(ecall,E2(minus,V1,V0),
        PAIR(E2(times,N(10),N(2)),PAIR(E1(uminus,N(3)),T))),
      N(10),N(3)))
  etst(N(23),
    eval2(
      E2(ecall,E2(minus,V1,V0),PAIR(E2(times,V0,N(2)),PAIR(E1(uminus,V1),T))),
      N(10),N(3)))

/*
The last three tests above compute 10*2-(-3).
In E2(minus,V0,V1), V0 refers to the last argument.
In E2(times,V0,I[2]), V0 refers to the first argument.
*/

/*
---------------------------------------------
Auxiliary functions for compilation
---------------------------------------------
*/

  test(varlist(0)==T)
  test(ROOT(varlist(1))==TAG_PAIR)
  test(ROOT(HEAD(varlist(1)))==TTAG_VAR)
  test(HEAD(HEAD(varlist(1)))==0)
  test(TAIL(varlist(1))==T)
  test(ROOT(varlist(3))==TAG_PAIR)
  test(ROOT(HEAD(varlist(3)))==TTAG_VAR)
  test(HEAD(HEAD(varlist(3)))==0)
  test(ROOT(TAIL(varlist(3)))==TAG_PAIR)
  test(ROOT(HEAD(TAIL(varlist(3))))==TTAG_VAR)
  test(HEAD(HEAD(TAIL(varlist(3))))==1)
  test(ROOT(TAIL(TAIL(varlist(3))))==TAG_PAIR)
  test(ROOT(HEAD(TAIL(TAIL(varlist(3)))))==TTAG_VAR)
  test(HEAD(HEAD(TAIL(TAIL(varlist(3)))))==2)
  test(TAIL(TAIL(TAIL(varlist(3))))==T)
  test(listlength(varlist(0))==0)
  test(listlength(varlist(1))==1)
  test(listlength(varlist(2))==2)
  test(listlength(varlist(3))==3)



/*
---------------------------------------------
Rack I/O
---------------------------------------------
*/

  xtst(sl2rack(FF))
  xtst(sl2rack(I0))
  xtst(sl2rack(J1))
  xtst(sl2rack(EE))
  xtst(sl2rack(M0))
  xtst(sl2rack(B0))
  xtst(sl2rack(PAIR(TT,TT)))
  xtst(sl2rack(PAIR(FF,TT)))
  xtst(sl2rack(PAIR(I0,TT)))
  xtst(sl2rack(PAIR(J1,TT)))
  xtst(sl2rack(PAIR(EE,TT)))
  xtst(sl2rack(PAIR(V(ab),TT)))
  xtst(sl2rack(PAIR(P0,TT)))
  xtst(sl2rack(PAIR(M0,TT)))
  xtst(sl2rack(PAIR(B0,TT)))
  xtst(sl2rack(PAIR(bytevect[0],FF)))
  xtst(sl2rack(PAIR(bytevect[0],I0)))
  xtst(sl2rack(PAIR(bytevect[0],J1)))
  xtst(sl2rack(PAIR(bytevect[0],M0)))
  xtst(sl2rack(PAIR(bytevect[0],B0)))

  etst(T,
    sl2rack(
      PAIR(bytevect[3],T)))
  etst(T,
    sl2rack(
      PAIR(bytevect[3],FF)))
  etst(PAIR(T,T),
    sl2rack(
      PAIR(bytevect[1],
      PAIR(bytevect[1],
      PAIR(bytevect[4],T)))))
  etst(N(7),
    sl2rack(
      PAIR(bytevect[0],
      PAIR(bytevect[7],
      PAIR(bytevect[4],T)))))
  etst(V(abc),
    sl2rack(
      PAIR(bytevect[2],
      PAIR(bytevect[3],
      PAIR(bytevect[97],
      PAIR(bytevect[98],
      PAIR(bytevect[99],
      PAIR(bytevect[4],T))))))))
  etst(PAIR(N(6),V(abc)),
    sl2rack(
      PAIR(bytevect[0],
      PAIR(bytevect[6],
      PAIR(bytevect[2],
      PAIR(bytevect[3],
      PAIR(bytevect[97],
      PAIR(bytevect[98],
      PAIR(bytevect[99],
      PAIR(bytevect[3],
      PAIR(bytevect[4],
      PAIR(bytevect[6],T))))))))))))
  etst(rack2sl(T),
    PAIR(bytevect[3],T))
  etst(rack2sl(PAIR(T,T)),
    PAIR(bytevect[1],
    PAIR(bytevect[1],
    PAIR(bytevect[4],T))))
  etst(rack2sl(N(7)),
    PAIR(bytevect[0],
    PAIR(bytevect[7],
    PAIR(bytevect[4],T))))
  etst(rack2sl(V(abc)),
    PAIR(bytevect[2],
    PAIR(bytevect[3],
    PAIR(bytevect[97],
    PAIR(bytevect[98],
    PAIR(bytevect[99],
    PAIR(bytevect[4],T)))))))
  etst(rack2sl(PAIR(N(6),V(abc))),
    PAIR(bytevect[0],
    PAIR(bytevect[6],
    PAIR(bytevect[2],
    PAIR(bytevect[3],
    PAIR(bytevect[97],
    PAIR(bytevect[98],
    PAIR(bytevect[99],
    PAIR(bytevect[3],
    PAIR(bytevect[4],
    PAIR(bytevect[6],T)))))))))))
  etst(rack2sl(PAIR(N(7),N(7))),
    PAIR(bytevect[0],
    PAIR(bytevect[7],
    PAIR(bytevect[3],
    PAIR(bytevect[3],
    PAIR(bytevect[5],T))))))
  etst(rack2sl(PAIR(PAIR(T,T),PAIR(T,T))),
    PAIR(bytevect[1],
    PAIR(bytevect[1],
    PAIR(bytevect[3],
    PAIR(bytevect[3],
    PAIR(bytevect[5],T))))))
  xtst(rack2sl(PAIR(PAIR(T,T),PAIR(T,N(-1)))))
  xtst(rack2sl(PAIR(PAIR(T,T),PAIR(T,F))))

/*
---------------------------------------------
RIPEMD-160
---------------------------------------------
*/

  test(STREQ("9C1185A5C5E9FC54612808977EE8F548B2258D31",msg2hash("")))
  test(STREQ("0BDC9D2D256B3EE9DAAE347BE6F4DC835A467FFE",msg2hash("a")))
  test(STREQ("71E0B62322D3A031C0FB621BE4D90773B7E66BB2",
    msg2hash("abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ123")))
  test(STREQ("B0E20B6E3116640286ED3A87A5713079B21F5189",
    msg2hash("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789")))

  test(vt2length(T)==0)
  test(vt2length(F)==0)
  test(vt2length(M0)==0)
  test(vt2length(B0)==0)
  test(vt2length(N(-1))==0)
  test(vt2length(N(0))==0)
  test(vt2length(N(1))==0)
  test(vt2length(N(255))==0)
  test(vt2length(N(256))==1)
  test(vt2length(N(513))==1)
  test(vt2length(V())==0)
  test(vt2length(V(abc))==3)
  test(vt2length(PAIR(V(ab),V(c)))==3)
  test(vt2length(PAIR(PAIR(V(a),N(-1)),PAIR(B0,V(bc))))==3)

  test(STREQ("9C1185A5C5E9FC54612808977EE8F548B2258D31",vt2hash(V())))
  test(STREQ("0BDC9D2D256B3EE9DAAE347BE6F4DC835A467FFE",vt2hash(V(a))))
  test(STREQ("71E0B62322D3A031C0FB621BE4D90773B7E66BB2",
    vt2hash(V(abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ123))))
  test(STREQ("B0E20B6E3116640286ED3A87A5713079B21F5189",
    vt2hash(V(ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789))))
  test(STREQ("8EB208F7E05D987A9B044A8E98C6B087F15A0BFC",
    vt2hash(V(abc))))
  test(STREQ("8EB208F7E05D987A9B044A8E98C6B087F15A0BFC",
    vt2hash(PAIR(V(ab),V(c)))))
  test(STREQ("8EB208F7E05D987A9B044A8E98C6B087F15A0BFC",
    vt2hash(PAIR(PAIR(V(a),N(-1)),PAIR(B0,V(bc))))))

  etst(ripemd(TT),
    PAIR(bytevect[0x9C],
    PAIR(bytevect[0x11],
    PAIR(bytevect[0x85],
    PAIR(bytevect[0xA5],
    PAIR(bytevect[0xC5],
    PAIR(bytevect[0xE9],
    PAIR(bytevect[0xFC],
    PAIR(bytevect[0x54],
    PAIR(bytevect[0x61],
    PAIR(bytevect[0x28],
    PAIR(bytevect[0x08],
    PAIR(bytevect[0x97],
    PAIR(bytevect[0x7E],
    PAIR(bytevect[0xE8],
    PAIR(bytevect[0xF5],
    PAIR(bytevect[0x48],
    PAIR(bytevect[0xB2],
    PAIR(bytevect[0x25],
    PAIR(bytevect[0x8D],
    PAIR(bytevect[0x31],T)))))))))))))))))))))
  etst(ripemd(PAIR(bytevect[97],TT)),
    PAIR(bytevect[0x0B],
    PAIR(bytevect[0xDC],
    PAIR(bytevect[0x9D],
    PAIR(bytevect[0x2D],
    PAIR(bytevect[0x25],
    PAIR(bytevect[0x6B],
    PAIR(bytevect[0x3E],
    PAIR(bytevect[0xE9],
    PAIR(bytevect[0xDA],
    PAIR(bytevect[0xAE],
    PAIR(bytevect[0x34],
    PAIR(bytevect[0x7B],
    PAIR(bytevect[0xE6],
    PAIR(bytevect[0xF4],
    PAIR(bytevect[0xDC],
    PAIR(bytevect[0x83],
    PAIR(bytevect[0x5A],
    PAIR(bytevect[0x46],
    PAIR(bytevect[0x7F],
    PAIR(bytevect[0xFE],T)))))))))))))))))))))
  etst(
    ripemd(
      PAIR(V(abcdefghijklmnopqrstuvwxyz),V(ABCDEFGHIJKLMNOPQRSTUVWXYZ123))),
    PAIR(bytevect[0x71],
    PAIR(bytevect[0xE0],
    PAIR(bytevect[0xB6],
    PAIR(bytevect[0x23],
    PAIR(bytevect[0x22],
    PAIR(bytevect[0xD3],
    PAIR(bytevect[0xA0],
    PAIR(bytevect[0x31],
    PAIR(bytevect[0xC0],
    PAIR(bytevect[0xFB],
    PAIR(bytevect[0x62],
    PAIR(bytevect[0x1B],
    PAIR(bytevect[0xE4],
    PAIR(bytevect[0xD9],
    PAIR(bytevect[0x07],
    PAIR(bytevect[0x73],
    PAIR(bytevect[0xB7],
    PAIR(bytevect[0xE6],
    PAIR(bytevect[0x6B],
    PAIR(bytevect[0xB2],T)))))))))))))))))))))
  etst(ripemd(PAIR(PAIR(V(a),N(-1)),PAIR(B0,V(bc)))),
    PAIR(bytevect[0x8E],
    PAIR(bytevect[0xB2],
    PAIR(bytevect[0x08],
    PAIR(bytevect[0xF7],
    PAIR(bytevect[0xE0],
    PAIR(bytevect[0x5D],
    PAIR(bytevect[0x98],
    PAIR(bytevect[0x7A],
    PAIR(bytevect[0x9B],
    PAIR(bytevect[0x04],
    PAIR(bytevect[0x4A],
    PAIR(bytevect[0x8E],
    PAIR(bytevect[0x98],
    PAIR(bytevect[0xC6],
    PAIR(bytevect[0xB0],
    PAIR(bytevect[0x87],
    PAIR(bytevect[0xF1],
    PAIR(bytevect[0x5A],
    PAIR(bytevect[0x0B],
    PAIR(bytevect[0xFC],T)))))))))))))))))))))

/*
---------------------------------------------
Test of loaded pages
---------------------------------------------
*/

#define U_true U_0(code_true)
#define M_true M_0(code_true)
#define M_If(x,y,z) M_3(code_If,x,y,z)
#define M_ap(x,y) M_2(code_ap,x,y)
#define U_Head(x) U_1(code_Head,x)
#define M_Head(x) M_1(code_Head,x)
#define U_Tail(x) U_1(code_Tail,x)
#define M_Tail(x) M_1(code_Tail,x)
#define U_LazyPair(x,y) U_2(code_LazyPair,x,y)
#define M_LazyPair(x,y) M_2(code_LazyPair,x,y)
#define M_false M_0(code_false)
#define U_Equal(x,y) U_2(code_Equal,x,y)
#define M_Equal(x,y) M_2(code_Equal,x,y)
#define U_norm(x) U_1(code_norm,x)
#define M_norm(x) M_1(code_norm,x)
#define U_equal(x,y) U_2(code_equal,x,y)
#define M_equal(x,y) M_2(code_equal,x,y)

#define Ttst(x) ntst(U_Equal(x,T))
#define Ftst(x) ntst(U_Equal(x,F))
#define Xtst(x) ntst(U_Equal(x,X))
#define Etst(x,y) ntst(U_Equal(x,y))
#define Utst(x,y) ptst(U_Equal(x,y))

#define Digit0(x)        U_1(code_digit0    ,x    )
#define Digit1(x)        U_1(code_digit1    ,x    )
#define Digit2(x)        U_1(code_digit2    ,x    )
#define Digit3(x)        U_1(code_digit3    ,x    )
#define Digit4(x)        U_1(code_digit4    ,x    )
#define Digit5(x)        U_1(code_digit5    ,x    )
#define Digit6(x)        U_1(code_digit6    ,x    )
#define Digit7(x)        U_1(code_digit7    ,x    )
#define Digit8(x)        U_1(code_digit8    ,x    )
#define Digit9(x)        U_1(code_digit9    ,x    )
#define Times(x,y)       U_2(code_times     ,x,y  )
#define Plus(x,y)        U_2(code_plus      ,x,y  )
#define Minus(x,y)       U_2(code_minus     ,x,y  )
#define And(x,y)         U_2(code_and       ,x,y  )
#define Or(x,y)          U_2(code_or        ,x,y  )
#define Then(x,y)        U_2(code_then      ,x,y  )
#define Pair(x,y)        U_2(code_pair      ,x,y  )
#define Lt(x,y)          U_2(code_lt        ,x,y  )
#define Le(x,y)          U_2(code_le        ,x,y  )
#define Eq(x,y)          U_2(code_equal     ,x,y  )
#define Ge(x,y)          U_2(code_ge        ,x,y  )
#define Gt(x,y)          U_2(code_gt        ,x,y  )
#define Apply(x,y)       U_2(code_apply     ,x,y  )
#define Boolp(x)         U_1(code_boolp     ,x    )
#define Catch(x)         U_1(code_catch     ,x    )
#define Maptag1(x)       U_1(code_maptag1   ,x    )
#define Div(x,y)         U_2(code_div       ,x,y  )
#define Head(x)          U_1(code_head      ,x    )
#define Intp(x)          U_1(code_intp      ,x    )
#define Boolg(x,y)       U_2(code_boolg     ,x,y  )
#define Intg(x,y)        U_2(code_intg      ,x,y  )
#define Mapg(x,y)        U_2(code_mapg      ,x,y  )
#define Objg(x,y)        U_2(code_objg      ,x,y  )
#define Pairg(x,y)       U_2(code_pairg     ,x,y  )
#define Mapp(x)          U_1(code_mapp      ,x    )
#define Maptag(x)        U_1(code_maptag    ,x    )
#define Mod(x,y)         U_2(code_mod       ,x,y  )
#define Norm(x)          U_1(code_norm      ,x    )
#define Objp(x)          U_1(code_objp      ,x    )
#define Pairp(x)         U_1(code_pairp     ,x    )
#define Raise(x)         U_1(code_raise     ,x    )
#define Root(x)          U_1(code_root      ,x    )
#define Tail(x)          U_1(code_tail      ,x    )
#define Untag(x)         U_1(code_untag     ,x    )
#define Digitend         U_0(code_digitend        )
#define Par(x)           U_1(code_par       ,x    )
#define Uplus(x)         U_1(code_uplus     ,x    )
#define Dplus(x)         U_1(code_dplus     ,x    )
#define Uminus(x)        U_1(code_uminus    ,x    )
#define Dminus(x)        U_1(code_dminus    ,x    )
#define Not(x)           U_1(code_not       ,x    )
#define Base             U_0(code_Base            )
#define LET(x,y)         U_2(code_LET       ,x,y  )
#define Ash(x,y)         U_2(code_ash       ,x,y  )
#define Bottom           U_0(code_bottom          )
#define Bt2bytes(x)      U_1(code_bt2bytes  ,x    )
#define Bt2vects(x)      U_1(code_bt2vects  ,x    )
#define Bt2vector(x)     U_1(code_bt2vector ,x    )
#define Ceiling(x,y)     U_2(code_ceiling   ,x,y  )
#define Destruct(x)      U_1(code_destruct  ,x    )
#define Evenp(x)         U_1(code_evenp     ,x    )
#define Exception        U_0(code_exception       )
#define False            U_0(code_false           )
#define Floor(x,y)       U_2(code_floor     ,x,y  )
#define Half(x)          U_1(code_half      ,x    )
#define If(x,y,z)        U_3(code_if        ,x,y,z)
#define Intlength(x)     U_1(code_intlength ,x    )
#define Logand(x,y)      U_2(code_logand    ,x,y  )
#define Logandc1(x,y)    U_2(code_logandc1  ,x,y  )
#define Logandc2(x,y)    U_2(code_logandc2  ,x,y  )
#define Logbitp(x,y)     U_2(code_logbitp   ,x,y  )
#define Logcount(x)      U_1(code_logcount  ,x    )
#define Logeqv(x,y)      U_2(code_logeqv    ,x,y  )
#define Logior(x,y)      U_2(code_logior    ,x,y  )
#define Lognand(x,y)     U_2(code_lognand   ,x,y  )
#define Lognor(x,y)      U_2(code_lognor    ,x,y  )
#define Lognot(x)        U_1(code_lognot    ,x    )
#define Logorc1(x,y)     U_2(code_logorc1   ,x,y  )
#define Logorc2(x,y)     U_2(code_logorc2   ,x,y  )
#define Logtest(x,y)     U_2(code_logtest   ,x,y  )
#define Logxor(x,y)      U_2(code_logxor    ,x,y  )
#define Map(x)           U_1(code_map       ,x    )
#define Prenorm(x)       U_1(code_prenorm   ,x    )
#define Notnot(x)        U_1(code_notnot    ,x    )
#define Object(x)        U_1(code_object    ,x    )
#define Print(x)         U_1(code_print     ,x    )
#define Round(x,y)       U_2(code_round     ,x,y  )
#define Spy(x)           U_1(code_spy       ,x    )
#define Timer(x)         U_1(code_timer     ,x    )
#define Trace(x)         U_1(code_trace     ,x    )
#define Truncate(x,y)    U_2(code_truncate  ,x,y  )
#define Vector(x)        U_1(code_vector    ,x    )
#define Vempty(x)        U_1(code_vempty    ,x    )
#define Vindex(x,y)      U_2(code_vindex    ,x,y  )
#define Vlength(x)       U_1(code_vlength   ,x    )
#define Vprefix(x,y)     U_2(code_vprefix   ,x,y  )
#define Vsubseq(x,y,z)   U_3(code_vsubseq   ,x,y,z)
#define Vsuffix(x,y)     U_2(code_vsuffix   ,x,y  )
#define V2bytes(x)       U_1(code_v2bytes   ,x    )
#define V2vects(x)       U_1(code_v2vects   ,x    )
#define Vt2vects(x)      U_1(code_vt2vects  ,x    )
#define Vt2bytes(x)      U_1(code_vt2bytes  ,x    )
#define Vt2v(x)          U_1(code_vt2v      ,x    )
#define YY               U_0(code_yy              )
#define Factorial(x)     U_1(code_factorial ,x    )
#define Emptylist        U_0(code_emptylist       )
#define Zeroth(x)        U_1(code_zeroth    ,x    )
#define First(x)         U_1(code_first     ,x    )
#define Second(x)        U_1(code_second    ,x    )
#define Revappend(x,y)   U_2(code_revappend ,x,y  )
#define Reverse(x)       U_1(code_reverse   ,x    )
#define Hello            U_0(code_hello           )
#define Echo             U_0(code_echo            )
#define Echo1(x)         U_1(code_echo1     ,x    )
#define Eecho            U_0(code_eecho           )
#define Eecho1(x)        U_1(code_eecho1    ,x    )
#define Quit(x)          U_1(code_quit      ,x    )
#define Write(x)         U_1(code_write     ,x    )
#define Readreq          U_0(code_readreq         )
#define Exec(x,y)        U_2(code_exec      ,x,y  )
#define Extend(x,y)      U_2(code_extend    ,x,y  )
#define Boot(x,y,z,v)    U_4(code_boot    ,x,y,z,v)
#define Read(x)          U_1(code_read      ,x    )
#define Int(x,y)         U_2(code_int       ,x,y  )
#define Extended(x)      U_1(code_extended  ,x    )
#define Compile(x)       U_1(code_compile   ,x    )
#define Ripemd(x)        U_1(code_ripemd    ,x    )
#define Sl2rack(x)       U_1(code_sl2rack   ,x    )
#define Rack2sl(x)       U_1(code_rack2sl   ,x    )
#define Eval(x,y,z)      U_3(code_eval      ,x,y,z)

#define Truea            U_0(code_true            )
#define Ifa(x,y,z)       U_3(code_If        ,x,y,z)
#define Apa(x,y)         U_2(code_ap        ,x,y  )
#define Heada(x)         U_1(code_Head      ,x    )
#define Taila(x)         U_1(code_Tail      ,x    )
#define LazyPaira(x,y)   U_2(code_LazyPair  ,x,y  )
#define Equala(x,y)      U_2(code_Equal     ,x,y  )
#define Digitenda        U_0(code_digitend        )
#define Basea            U_0(code_Base            )
#define Falsea           U_0(code_false           )
#define Testfunc1a(x)    U_1(code_testfunc1a,x    )
#define Testfunc2a(x,y)  U_2(code_testfunc2a,x,y  )
#define Testfunc5a(x)    U_1(code_testfunc5a,x    )
#define Testfunc6a(x)    U_1(code_testfunc6a,x    )
#define Testfunc7a(x)    U_1(code_testfunc7a,x    )
#define Testfunc8a(x,y)  U_2(code_testfunc8a,x,y  )

#define Trueb            U_0(code_trueb           )
#define Ifb(x,y,z)       U_3(code_Ifb       ,x,y,z)
#define Apb(x,y)         U_2(code_apb       ,x,y  )
#define Headb(x)         U_1(code_Headb     ,x    )
#define Tailb(x)         U_1(code_Tailb     ,x    )
#define LazyPairb(x,y)   U_2(code_LazyPairb ,x,y  )
#define Equalb(x,y)      U_2(code_Equalb    ,x,y  )
#define Digitendb        U_0(code_digitend        )
#define Baseb            U_0(code_Base            )
#define Falseb           U_0(code_false           )
#define Testfunc1b(x)    U_1(code_testfunc1b,x    )
#define Testfunc2b(x,y)  U_2(code_testfunc2b,x,y  )
#define Testfunc5b(x)    U_1(code_testfunc5b,x    )
#define Testfunc6b(x)    U_1(code_testfunc6b,x    )
#define Testfunc7b(x)    U_1(code_testfunc7b,x    )
#define Testfunc8b(x,y)  U_2(code_testfunc8b,x,y  )

  if(name2sym("base")!=T){

    DEFC(code_true     ,name2code("true"));
    DEFC(code_If       ,name2code("If \" then \" else \""));
    DEFC(code_ap       ,name2code("\" ' \""));
    DEFC(code_Head     ,name2code("\" Head"));
    DEFC(code_Tail     ,name2code("\" Tail"));
    DEFC(code_LazyPair ,name2code("\" LazyPair \""));
    DEFC(code_Equal    ,name2code("\" Equal \""));

    DEFC(code_digit0   ,name2code("\" %0"));
    DEFC(code_digit1   ,name2code("\" %1"));
    DEFC(code_digit2   ,name2code("\" %2"));
    DEFC(code_digit3   ,name2code("\" %3"));
    DEFC(code_digit4   ,name2code("\" %4"));
    DEFC(code_digit5   ,name2code("\" %5"));
    DEFC(code_digit6   ,name2code("\" %6"));
    DEFC(code_digit7   ,name2code("\" %7"));
    DEFC(code_digit8   ,name2code("\" %8"));
    DEFC(code_digit9   ,name2code("\" %9"));
    DEFC(code_times    ,name2code("\" * \""));
    DEFC(code_plus     ,name2code("\" + \""));
    DEFC(code_minus    ,name2code("\" - \""));
    DEFC(code_and      ,name2code("\" .and. \""));
    DEFC(code_or       ,name2code("\" .or. \""));
    DEFC(code_then     ,name2code("\" .then. \""));
    DEFC(code_pair     ,name2code("\" :: \""));
    DEFC(code_lt       ,name2code("\" < \""));
    DEFC(code_le       ,name2code("\" <= \""));
    DEFC(code_equal    ,name2code("\" = \""));
    DEFC(code_gt       ,name2code("\" > \""));
    DEFC(code_ge       ,name2code("\" >= \""));
    DEFC(code_apply    ,name2code("\" apply \""));
    DEFC(code_boolp    ,name2code("\" boolp"));
    DEFC(code_catch    ,name2code("\" catch"));
    DEFC(code_maptag1  ,name2code("\" catching maptag"));
    DEFC(code_div      ,name2code("\" div \""));
    DEFC(code_head     ,name2code("\" head"));
    DEFC(code_intp     ,name2code("\" intp"));
    DEFC(code_boolg    ,name2code("\" is bool : \""));
    DEFC(code_intg     ,name2code("\" is int : \""));
    DEFC(code_mapg     ,name2code("\" is map : \""));
    DEFC(code_objg     ,name2code("\" is object : \""));
    DEFC(code_pairg    ,name2code("\" is pair : \""));
    DEFC(code_mapp     ,name2code("\" mapp"));
    DEFC(code_maptag   ,name2code("\" maptag"));
    DEFC(code_mod      ,name2code("\" mod \""));
    DEFC(code_norm     ,name2code("\" norm"));
    DEFC(code_objp     ,name2code("\" objectp"));
    DEFC(code_pairp    ,name2code("\" pairp"));
    DEFC(code_raise    ,name2code("\" raise"));
    DEFC(code_root     ,name2code("\" root"));
    DEFC(code_tail     ,name2code("\" tail"));
    DEFC(code_untag    ,name2code("\" untag"));
    DEFC(code_digitend ,name2code("%%"));
    DEFC(code_par      ,name2code("( \" )"));
    DEFC(code_uplus    ,name2code("+ \""));
    DEFC(code_dplus    ,name2code("+\""));
    DEFC(code_uminus   ,name2code("- \""));
    DEFC(code_dminus   ,name2code("-\""));
    DEFC(code_not      ,name2code(".not. \""));
    DEFC(code_Base     ,name2code("Base"));
    DEFC(code_LET      ,name2code("LET \" BE \""));
    DEFC(code_ash      ,name2code("ash ( \" , \" )"));
/*  DEFC(code_bottom   ,name2code("bottom")); Unused */
    DEFC(code_bt2bytes ,name2code("bt2byte* ( \" )"));
/*  DEFC(code_bt2vects ,name2code("bt2vector* ( \" )")); Unused */
    DEFC(code_bt2vector,name2code("bt2vector ( \" )"));
    DEFC(code_ceiling  ,name2code("ceiling ( \" , \" )"));
    DEFC(code_destruct ,name2code("destruct ( \" )"));
    DEFC(code_evenp    ,name2code("evenp ( \" )"));
    DEFC(code_exception,name2code("exception"));
    DEFC(code_false    ,name2code("false"));
    DEFC(code_floor    ,name2code("floor ( \" , \" )"));
    DEFC(code_half     ,name2code("half ( \" )"));
    DEFC(code_if       ,name2code("if \" then \" else \""));
    DEFC(code_intlength,name2code("integer-length ( \" )"));
    DEFC(code_logand   ,name2code("logand ( \" , \" )"));
    DEFC(code_logandc1 ,name2code("logandc1 ( \" , \" )"));
    DEFC(code_logandc2 ,name2code("logandc2 ( \" , \" )"));
    DEFC(code_logbitp  ,name2code("logbitp ( \" , \" )"));
    DEFC(code_logcount ,name2code("logcount ( \" )"));
    DEFC(code_logeqv   ,name2code("logeqv ( \" , \" )"));
    DEFC(code_logior   ,name2code("logior ( \" , \" )"));
    DEFC(code_lognand  ,name2code("lognand ( \" , \" )"));
    DEFC(code_lognor   ,name2code("lognor ( \" , \" )"));
    DEFC(code_lognot   ,name2code("lognot ( \" )"));
    DEFC(code_logorc1  ,name2code("logorc1 ( \" , \" )"));
    DEFC(code_logorc2  ,name2code("logorc2 ( \" , \" )"));
    DEFC(code_logtest  ,name2code("logtest ( \" , \" )"));
    DEFC(code_logxor   ,name2code("logxor ( \" , \" )"));
    DEFC(code_map      ,name2code("map ( \" )"));
    DEFC(code_prenorm  ,name2code("norm \""));
    DEFC(code_notnot   ,name2code("notnot \""));
    DEFC(code_object   ,name2code("object ( \" )"));
    DEFC(code_print    ,name2code("print ( \" )"));
    DEFC(code_round    ,name2code("round ( \" , \" )"));
    DEFC(code_spy      ,name2code("spy ( \" )"));
    DEFC(code_timer    ,name2code("timer ( \" )"));
    DEFC(code_trace    ,name2code("trace ( \" )"));
    DEFC(code_truncate ,name2code("truncate ( \" , \" )"));
    DEFC(code_vector   ,name2code("vector ( \" )"));
    DEFC(code_vempty   ,name2code("vector-empty ( \" )"));
    DEFC(code_vindex   ,name2code("vector-index ( \" , \" )"));
    DEFC(code_vlength  ,name2code("vector-length ( \" )"));
    DEFC(code_vprefix  ,name2code("vector-prefix ( \" , \" )"));
    DEFC(code_vsubseq  ,name2code("vector-subseq ( \" , \" , \" )"));
    DEFC(code_vsuffix  ,name2code("vector-suffix ( \" , \" )"));
    DEFC(code_v2bytes  ,name2code("vector2byte* ( \" )"));
/*  DEFC(code_v2vects  ,name2code("vector2vector* ( \" )")); Unused */
    DEFC(code_vt2bytes ,name2code("vt2byte* ( \" )"));
/*  DEFC(code_vt2vects ,name2code("vt2vector* ( \" )")); Unused */
    DEFC(code_vt2v     ,name2code("vt2vector ( \" )"));
    DEFC(code_yy       ,name2code("YY"));
    DEFC(code_factorial,name2code("\" factorial"));
    DEFC(code_emptylist,name2code("<<>>"));
    DEFC(code_zeroth   ,name2code("\" zeroth"));
    DEFC(code_first    ,name2code("\" first"));
    DEFC(code_second   ,name2code("\" second"));
    DEFC(code_revappend,name2code("revappend ( \" , \" )"));
    DEFC(code_reverse  ,name2code("reverse ( \" )"));
    DEFC(code_hello    ,name2code("Hello World"));
    DEFC(code_echo     ,name2code("Echo"));
    DEFC(code_echo1    ,name2code("Echo1 ( \" )"));
    DEFC(code_eecho    ,name2code("Eecho"));
    DEFC(code_eecho1   ,name2code("Eecho1 ( \" )"));
    DEFC(code_quit     ,name2code("quit request ( \" )"));
    DEFC(code_write    ,name2code("write request ( \" )"));
    DEFC(code_readreq  ,name2code("read request"));
    DEFC(code_exec     ,name2code("exec request ( \" , \" )"));
    DEFC(code_extend   ,name2code("extend request ( \" , \" )"));
    DEFC(code_boot     ,name2code("boot event ( \" , \" , \" , \" )"));
    DEFC(code_read     ,name2code("read reply ( \" )"));
    DEFC(code_int      ,name2code("exec reply ( \" , \" )"));
    DEFC(code_extended ,name2code("extend reply ( \" )"));
    DEFC(code_compile  ,name2code("compile ( \" )"));
    DEFC(code_ripemd   ,name2code("ripemd ( \" )"));
    DEFC(code_sl2rack  ,name2code("sl2rack ( \" )"));
    DEFC(code_rack2sl  ,name2code("rack2sl ( \" )"));
    DEFC(code_eval     ,name2code("eval ( \" , \" , \" )"));

    DEFC(fixterm1      ,term_lambda(term_pair(term_const(N(2)),var_0)));
    DEFC(fixtermp      ,cons(ETAG_print,NIL,V(Only printed once)));
    DEFC(fixterm2      ,term_lambda(term_pair(term_ecall(fixtermp,T),var_0)));
    DEFC(fixmap1       ,map_apply(YY,term_eval(fixterm1)));
    DEFC(fixmap2       ,map_apply(YY,term_eval(fixterm2)));

    DEFC(baseref       ,name2ref("base"));
    DEFC(basecache     ,aget(state,baseref));
    DEFC(baserack      ,aget(basecache,baseref));
    DEFC(basecodea     ,aget(baserack,const_code));
    DEFC(basediagnose  ,aget(baserack,const_diagnose));
    DEFC(baseexpansion ,aget(baserack,const_expansion));
    DEFC(basecodex     ,aget(baserack,const_codex));
    DEFC(basemacro     ,def2rhs(mget4(basecodex,baseref,Z,Z,const_macro)));
    DEFC(basecache1    ,mput2(basecache,T,baseref,const_code));
    DEFC(basecache2    ,mput2(basecache1,T,baseref,const_diagnose));

    DEFC(basecacheb    ,Compile(basecache2));
    DEFC(baserackb     ,aget(basecacheb,baseref));
    DEFC(basecodeb     ,aget(baserackb,const_code));
    DEFC(basediagnoseb ,aget(baserackb,const_diagnose));

    DEFC(code_trueb    ,name2code1(basecodeb,("true")));
    DEFC(code_Ifb      ,name2code1(basecodeb,("If \" then \" else \"")));
/*  DEFC(code_apb      ,name2code1(basecodeb,("\" ' \""))); Unused */
    DEFC(code_Headb    ,name2code1(basecodeb,("\" Head")));
    DEFC(code_Tailb    ,name2code1(basecodeb,("\" Tail")));
    DEFC(code_LazyPairb,name2code1(basecodeb,("\" LazyPair \"")));
    DEFC(code_Equalb   ,name2code1(basecodeb,("\" Equal \"")));
/*  DEFC(code_digitendb,name2code1(basecodeb,("\" Equal \""))); Unused */
/*  DEFC(code_Baseb    ,name2code1(basecodeb,("\" Equal \""))); Unused */
/*  DEFC(code_falseb   ,name2code1(basecodeb,("\" Equal \""))); Unused */

    DEFC(code_testfunc1a,name2code1(basecodea,"testfunc1 ( \" )"));
    DEFC(code_testfunc1b,name2code1(basecodeb,"testfunc1 ( \" )"));
    DEFC(code_testfunc2a,name2code1(basecodea,"testfunc2 ( \" , \" )"));
    DEFC(code_testfunc2b,name2code1(basecodeb,"testfunc2 ( \" , \" )"));
    DEFC(code_testfunc5a,name2code1(basecodea,"testfunc5 ( \" )"));
    DEFC(code_testfunc5b,name2code1(basecodeb,"testfunc5 ( \" )"));
    DEFC(code_testfunc6a,name2code1(basecodea,"testfunc6 ( \" )"));
    DEFC(code_testfunc6b,name2code1(basecodeb,"testfunc6 ( \" )"));
    DEFC(code_testfunc7a,name2code1(basecodea,"testfunc7 ( \" )"));
    DEFC(code_testfunc7b,name2code1(basecodeb,"testfunc7 ( \" )"));
    DEFC(code_testfunc8a,name2code1(basecodea,"testfunc8 ( \" , \" )"));
    DEFC(code_testfunc8b,name2code1(basecodeb,"testfunc8 ( \" , \" )"));

    ntst(map2rnf(U_true))
    ttst(root(M_true))

    ftst(root(code_If))
    ftst(root(apply(code_If,MT)))
    ftst(root(apply(apply(code_If,MT),MT)))
    ttst(root(M_If(MT,MT,MT)))
    ttst(root(M_If(MT,MT,MF)))
    ftst(root(M_If(MT,MF,MT)))
    ftst(root(M_If(MT,MF,MF)))
    ttst(root(M_If(MF,MT,MT)))
    ftst(root(M_If(MF,MT,MF)))
    ttst(root(M_If(MF,MF,MT)))
    ftst(root(M_If(MF,MF,MF)))

    ftst(root(code_ap))
    ftst(root(apply(code_ap,MKI)))
    ttst(root(M_ap(MT,MT)))
    ttst(root(M_ap(MT,MF)))
    ttst(root(M_ap(MT,MI)))
    ttst(root(M_ap(MF,MT)))
    ttst(root(M_ap(MF,MF)))
    ttst(root(M_ap(MF,MI)))
    ttst(root(M_ap(MI,MT)))
    ftst(root(M_ap(MI,MF)))
    ftst(root(M_ap(MI,MI)))
    ftst(root(M_ap(MK,MT)))
    ftst(root(M_ap(MK,MF)))
    ftst(root(M_ap(MK,MI)))

    ntst(U_Head(UT))
    ntst(U_Head(UF))
    ntst(U_Head(UPTT))
    ntst(U_Head(UPTF))
    ptst(U_Head(UPFT))
    ptst(U_Head(UPFF))
    ttst(root(M_Head(MT)))
    ttst(root(M_Head(MF)))
    ttst(root(M_Head(MPTT)))
    ttst(root(M_Head(MPTF)))
    ftst(root(M_Head(MPFT)))
    ftst(root(M_Head(MPFF)))

    ntst(U_Tail(UT))
    ntst(U_Tail(UF))
    ntst(U_Tail(UPTT))
    ptst(U_Tail(UPTF))
    ntst(U_Tail(UPFT))
    ptst(U_Tail(UPFF))
    ttst(root(M_Tail(MT)))
    ttst(root(M_Tail(MF)))
    ttst(root(M_Tail(MPTT)))
    ftst(root(M_Tail(MPTF)))
    ttst(root(M_Tail(MPFT)))
    ftst(root(M_Tail(MPFF)))

    ptst(U_LazyPair(UT,UT))
    ptst(U_LazyPair(UT,UF))
    ptst(U_LazyPair(UF,UT))
    ptst(U_LazyPair(UF,UF))
    ntst(U_Head(U_LazyPair(UT,UT)))
    ntst(U_Head(U_LazyPair(UT,UF)))
    ptst(U_Head(U_LazyPair(UF,UT)))
    ptst(U_Head(U_LazyPair(UF,UF)))
    ntst(U_Tail(U_LazyPair(UT,UT)))
    ptst(U_Tail(U_LazyPair(UT,UF)))
    ntst(U_Tail(U_LazyPair(UF,UT)))
    ptst(U_Tail(U_LazyPair(UF,UF)))
    ftst(root(M_LazyPair(MT,MT)))
    ftst(root(M_LazyPair(MT,MF)))
    ftst(root(M_LazyPair(MF,MT)))
    ftst(root(M_LazyPair(MF,MF)))
    ttst(root(M_Head(M_LazyPair(MT,MT))))
    ttst(root(M_Head(M_LazyPair(MT,MF))))
    ftst(root(M_Head(M_LazyPair(MF,MT))))
    ftst(root(M_Head(M_LazyPair(MF,MF))))
    ttst(root(M_Tail(M_LazyPair(MT,MT))))
    ftst(root(M_Tail(M_LazyPair(MT,MF))))
    ttst(root(M_Tail(M_LazyPair(MF,MT))))
    ftst(root(M_Tail(M_LazyPair(MF,MF))))

    ftst(root(code_false))
    ttst(root(apply(M_false,MT)))
    ttst(root(apply(M_false,MF)))

    ptst(I[6]) /* (ref :: idx) :: val */
    ptst(U_Head(I[6])) /* (ref :: idx) */
    ntst(U_Head(U_Head(I[6]))) /* ref = 0 */
    ptst(U_Tail(U_Head(I[6]))) /* idx != 0 */
    ptst(U_Head(U_Tail(U_Head(I[6])))) /* idx bit 0 = 1 */
    ntst(U_Head(U_Head(U_Tail(U_Head(I[6])))))
    ntst(U_Tail(U_Head(U_Tail(U_Head(I[6])))))
    ntst(U_Tail(U_Tail(U_Head(I[6])))) /* idx no more bits */
    ptst(U_Tail(I[6])) /* val */
    ntst(U_Head(U_Tail(I[6]))) /* sign */
    ptst(U_Tail(U_Tail(I[6])))
    ntst(U_Head(U_Tail(U_Tail(I[6])))) /* val bit 0 */
    ptst(U_Tail(U_Tail(U_Tail(I[6]))))
    ptst(U_Head(U_Tail(U_Tail(U_Tail(I[6]))))) /* val bit 1 */
    ntst(U_Head(U_Head(U_Tail(U_Tail(U_Tail(I[6]))))))
    ntst(U_Tail(U_Head(U_Tail(U_Tail(U_Tail(I[6]))))))
    ptst(U_Tail(U_Tail(U_Tail(U_Tail(I[6])))))
    ptst(U_Head(U_Tail(U_Tail(U_Tail(U_Tail(I[6])))))) /* val bit 2 */
    ntst(U_Head(U_Head(U_Tail(U_Tail(U_Tail(U_Tail(I[6])))))))
    ntst(U_Tail(U_Head(U_Tail(U_Tail(U_Tail(U_Tail(I[6])))))))
    ntst(U_Tail(U_Tail(U_Tail(U_Tail(U_Tail(I[6]))))))

    ttst(root(M_Equal(M_true,M_true)))
    ftst(root(M_Equal(M_true,M_false)))
    ftst(root(M_Equal(M_false,M_true)))
    ttst(root(M_Equal(M_false,M_false)))

    ntst(U_Equal(UT,UT))
    ptst(U_Equal(UT,UF))
    ptst(U_Equal(UF,UT))
    ntst(U_Equal(UF,UF))

    ttst(root(M_Equal(MT      ,MT)))
    ftst(root(M_Equal(MF      ,MT)))
    ftst(root(M_Equal(MPTT    ,MT)))
    ftst(root(M_Equal(MPTF    ,MT)))
    ftst(root(M_Equal(MPFT    ,MT)))
    ftst(root(M_Equal(MPFF    ,MT)))
    ftst(root(M_Equal(MPPTTPTT,MT)))
    ftst(root(M_Equal(MPPTTPTF,MT)))
    ftst(root(M_Equal(MPPTTPFT,MT)))
    ftst(root(M_Equal(MPPTTPFF,MT)))
    ftst(root(M_Equal(MPPTFPTT,MT)))
    ftst(root(M_Equal(MPPTFPTF,MT)))
    ftst(root(M_Equal(MPPTFPFT,MT)))
    ftst(root(M_Equal(MPPTFPFF,MT)))
    ftst(root(M_Equal(MPPFTPTT,MT)))
    ftst(root(M_Equal(MPPFTPTF,MT)))
    ftst(root(M_Equal(MPPFTPFT,MT)))
    ftst(root(M_Equal(MPPFTPFF,MT)))
    ftst(root(M_Equal(MPPFFPTT,MT)))
    ftst(root(M_Equal(MPPFFPTF,MT)))
    ftst(root(M_Equal(MPPFFPFT,MT)))
    ftst(root(M_Equal(MPPFFPFF,MT)))

    ftst(root(M_Equal(MT      ,MF)))
    ttst(root(M_Equal(MF      ,MF)))
    ttst(root(M_Equal(MPTT    ,MF)))
    ftst(root(M_Equal(MPTF    ,MF)))
    ftst(root(M_Equal(MPFT    ,MF)))
    ftst(root(M_Equal(MPFF    ,MF)))
    ftst(root(M_Equal(MPPTTPTT,MF)))
    ftst(root(M_Equal(MPPTTPTF,MF)))
    ftst(root(M_Equal(MPPTTPFT,MF)))
    ftst(root(M_Equal(MPPTTPFF,MF)))
    ftst(root(M_Equal(MPPTFPTT,MF)))
    ftst(root(M_Equal(MPPTFPTF,MF)))
    ftst(root(M_Equal(MPPTFPFT,MF)))
    ftst(root(M_Equal(MPPTFPFF,MF)))
    ftst(root(M_Equal(MPPFTPTT,MF)))
    ftst(root(M_Equal(MPPFTPTF,MF)))
    ftst(root(M_Equal(MPPFTPFT,MF)))
    ftst(root(M_Equal(MPPFTPFF,MF)))
    ftst(root(M_Equal(MPPFFPTT,MF)))
    ftst(root(M_Equal(MPPFFPTF,MF)))
    ftst(root(M_Equal(MPPFFPFT,MF)))
    ftst(root(M_Equal(MPPFFPFF,MF)))

    ftst(root(M_Equal(MT      ,MPTT)))
    ttst(root(M_Equal(MF      ,MPTT)))
    ttst(root(M_Equal(MPTT    ,MPTT)))
    ftst(root(M_Equal(MPTF    ,MPTT)))
    ftst(root(M_Equal(MPFT    ,MPTT)))
    ftst(root(M_Equal(MPFF    ,MPTT)))
    ftst(root(M_Equal(MPPTTPTT,MPTT)))
    ftst(root(M_Equal(MPPTTPTF,MPTT)))
    ftst(root(M_Equal(MPPTTPFT,MPTT)))
    ftst(root(M_Equal(MPPTTPFF,MPTT)))
    ftst(root(M_Equal(MPPTFPTT,MPTT)))
    ftst(root(M_Equal(MPPTFPTF,MPTT)))
    ftst(root(M_Equal(MPPTFPFT,MPTT)))
    ftst(root(M_Equal(MPPTFPFF,MPTT)))
    ftst(root(M_Equal(MPPFTPTT,MPTT)))
    ftst(root(M_Equal(MPPFTPTF,MPTT)))
    ftst(root(M_Equal(MPPFTPFT,MPTT)))
    ftst(root(M_Equal(MPPFTPFF,MPTT)))
    ftst(root(M_Equal(MPPFFPTT,MPTT)))
    ftst(root(M_Equal(MPPFFPTF,MPTT)))
    ftst(root(M_Equal(MPPFFPFT,MPTT)))
    ftst(root(M_Equal(MPPFFPFF,MPTT)))

    ftst(root(M_Equal(MT      ,MPTF)))
    ftst(root(M_Equal(MF      ,MPTF)))
    ftst(root(M_Equal(MPTT    ,MPTF)))
    ttst(root(M_Equal(MPTF    ,MPTF)))
    ftst(root(M_Equal(MPFT    ,MPTF)))
    ftst(root(M_Equal(MPFF    ,MPTF)))
    ftst(root(M_Equal(MPPTTPTT,MPTF)))
    ftst(root(M_Equal(MPPTTPTF,MPTF)))
    ftst(root(M_Equal(MPPTTPFT,MPTF)))
    ftst(root(M_Equal(MPPTTPFF,MPTF)))
    ftst(root(M_Equal(MPPTFPTT,MPTF)))
    ftst(root(M_Equal(MPPTFPTF,MPTF)))
    ftst(root(M_Equal(MPPTFPFT,MPTF)))
    ftst(root(M_Equal(MPPTFPFF,MPTF)))
    ftst(root(M_Equal(MPPFTPTT,MPTF)))
    ftst(root(M_Equal(MPPFTPTF,MPTF)))
    ftst(root(M_Equal(MPPFTPFT,MPTF)))
    ftst(root(M_Equal(MPPFTPFF,MPTF)))
    ftst(root(M_Equal(MPPFFPTT,MPTF)))
    ftst(root(M_Equal(MPPFFPTF,MPTF)))
    ftst(root(M_Equal(MPPFFPFT,MPTF)))
    ftst(root(M_Equal(MPPFFPFF,MPTF)))

    ftst(root(M_Equal(MT      ,MPFT)))
    ftst(root(M_Equal(MF      ,MPFT)))
    ftst(root(M_Equal(MPTT    ,MPFT)))
    ftst(root(M_Equal(MPTF    ,MPFT)))
    ttst(root(M_Equal(MPFT    ,MPFT)))
    ftst(root(M_Equal(MPFF    ,MPFT)))
    ftst(root(M_Equal(MPPTTPTT,MPFT)))
    ftst(root(M_Equal(MPPTTPTF,MPFT)))
    ftst(root(M_Equal(MPPTTPFT,MPFT)))
    ftst(root(M_Equal(MPPTTPFF,MPFT)))
    ftst(root(M_Equal(MPPTFPTT,MPFT)))
    ftst(root(M_Equal(MPPTFPTF,MPFT)))
    ftst(root(M_Equal(MPPTFPFT,MPFT)))
    ftst(root(M_Equal(MPPTFPFF,MPFT)))
    ftst(root(M_Equal(MPPFTPTT,MPFT)))
    ftst(root(M_Equal(MPPFTPTF,MPFT)))
    ftst(root(M_Equal(MPPFTPFT,MPFT)))
    ftst(root(M_Equal(MPPFTPFF,MPFT)))
    ftst(root(M_Equal(MPPFFPTT,MPFT)))
    ftst(root(M_Equal(MPPFFPTF,MPFT)))
    ftst(root(M_Equal(MPPFFPFT,MPFT)))
    ftst(root(M_Equal(MPPFFPFF,MPFT)))

    ftst(root(M_Equal(MT      ,MPFF)))
    ftst(root(M_Equal(MF      ,MPFF)))
    ftst(root(M_Equal(MPTT    ,MPFF)))
    ftst(root(M_Equal(MPTF    ,MPFF)))
    ftst(root(M_Equal(MPFT    ,MPFF)))
    ttst(root(M_Equal(MPFF    ,MPFF)))
    ttst(root(M_Equal(MPPTTPTT,MPFF)))
    ftst(root(M_Equal(MPPTTPTF,MPFF)))
    ftst(root(M_Equal(MPPTTPFT,MPFF)))
    ftst(root(M_Equal(MPPTTPFF,MPFF)))
    ftst(root(M_Equal(MPPTFPTT,MPFF)))
    ftst(root(M_Equal(MPPTFPTF,MPFF)))
    ftst(root(M_Equal(MPPTFPFT,MPFF)))
    ftst(root(M_Equal(MPPTFPFF,MPFF)))
    ftst(root(M_Equal(MPPFTPTT,MPFF)))
    ftst(root(M_Equal(MPPFTPTF,MPFF)))
    ftst(root(M_Equal(MPPFTPFT,MPFF)))
    ftst(root(M_Equal(MPPFTPFF,MPFF)))
    ftst(root(M_Equal(MPPFFPTT,MPFF)))
    ftst(root(M_Equal(MPPFFPTF,MPFF)))
    ftst(root(M_Equal(MPPFFPFT,MPFF)))
    ftst(root(M_Equal(MPPFFPFF,MPFF)))

    ftst(root(M_Equal(MT      ,MPPTTPTT)))
    ftst(root(M_Equal(MF      ,MPPTTPTT)))
    ftst(root(M_Equal(MPTT    ,MPPTTPTT)))
    ftst(root(M_Equal(MPTF    ,MPPTTPTT)))
    ftst(root(M_Equal(MPFT    ,MPPTTPTT)))
    ttst(root(M_Equal(MPFF    ,MPPTTPTT)))
    ttst(root(M_Equal(MPPTTPTT,MPPTTPTT)))
    ftst(root(M_Equal(MPPTTPTF,MPPTTPTT)))
    ftst(root(M_Equal(MPPTTPFT,MPPTTPTT)))
    ftst(root(M_Equal(MPPTTPFF,MPPTTPTT)))
    ftst(root(M_Equal(MPPTFPTT,MPPTTPTT)))
    ftst(root(M_Equal(MPPTFPTF,MPPTTPTT)))
    ftst(root(M_Equal(MPPTFPFT,MPPTTPTT)))
    ftst(root(M_Equal(MPPTFPFF,MPPTTPTT)))
    ftst(root(M_Equal(MPPFTPTT,MPPTTPTT)))
    ftst(root(M_Equal(MPPFTPTF,MPPTTPTT)))
    ftst(root(M_Equal(MPPFTPFT,MPPTTPTT)))
    ftst(root(M_Equal(MPPFTPFF,MPPTTPTT)))
    ftst(root(M_Equal(MPPFFPTT,MPPTTPTT)))
    ftst(root(M_Equal(MPPFFPTF,MPPTTPTT)))
    ftst(root(M_Equal(MPPFFPFT,MPPTTPTT)))
    ftst(root(M_Equal(MPPFFPFF,MPPTTPTT)))

    ftst(root(M_Equal(MT      ,MPPTTPTF)))
    ftst(root(M_Equal(MF      ,MPPTTPTF)))
    ftst(root(M_Equal(MPTT    ,MPPTTPTF)))
    ftst(root(M_Equal(MPTF    ,MPPTTPTF)))
    ftst(root(M_Equal(MPFT    ,MPPTTPTF)))
    ftst(root(M_Equal(MPFF    ,MPPTTPTF)))
    ftst(root(M_Equal(MPPTTPTT,MPPTTPTF)))
    ttst(root(M_Equal(MPPTTPTF,MPPTTPTF)))
    ftst(root(M_Equal(MPPTTPFT,MPPTTPTF)))
    ftst(root(M_Equal(MPPTTPFF,MPPTTPTF)))
    ftst(root(M_Equal(MPPTFPTT,MPPTTPTF)))
    ftst(root(M_Equal(MPPTFPTF,MPPTTPTF)))
    ftst(root(M_Equal(MPPTFPFT,MPPTTPTF)))
    ftst(root(M_Equal(MPPTFPFF,MPPTTPTF)))
    ftst(root(M_Equal(MPPFTPTT,MPPTTPTF)))
    ftst(root(M_Equal(MPPFTPTF,MPPTTPTF)))
    ftst(root(M_Equal(MPPFTPFT,MPPTTPTF)))
    ftst(root(M_Equal(MPPFTPFF,MPPTTPTF)))
    ftst(root(M_Equal(MPPFFPTT,MPPTTPTF)))
    ftst(root(M_Equal(MPPFFPTF,MPPTTPTF)))
    ftst(root(M_Equal(MPPFFPFT,MPPTTPTF)))
    ftst(root(M_Equal(MPPFFPFF,MPPTTPTF)))

    ftst(root(M_Equal(MT      ,MPPTTPFT)))
    ftst(root(M_Equal(MF      ,MPPTTPFT)))
    ftst(root(M_Equal(MPTT    ,MPPTTPFT)))
    ftst(root(M_Equal(MPTF    ,MPPTTPFT)))
    ftst(root(M_Equal(MPFT    ,MPPTTPFT)))
    ftst(root(M_Equal(MPFF    ,MPPTTPFT)))
    ftst(root(M_Equal(MPPTTPTT,MPPTTPFT)))
    ftst(root(M_Equal(MPPTTPTF,MPPTTPFT)))
    ttst(root(M_Equal(MPPTTPFT,MPPTTPFT)))
    ftst(root(M_Equal(MPPTTPFF,MPPTTPFT)))
    ftst(root(M_Equal(MPPTFPTT,MPPTTPFT)))
    ftst(root(M_Equal(MPPTFPTF,MPPTTPFT)))
    ftst(root(M_Equal(MPPTFPFT,MPPTTPFT)))
    ftst(root(M_Equal(MPPTFPFF,MPPTTPFT)))
    ftst(root(M_Equal(MPPFTPTT,MPPTTPFT)))
    ftst(root(M_Equal(MPPFTPTF,MPPTTPFT)))
    ftst(root(M_Equal(MPPFTPFT,MPPTTPFT)))
    ftst(root(M_Equal(MPPFTPFF,MPPTTPFT)))
    ftst(root(M_Equal(MPPFFPTT,MPPTTPFT)))
    ftst(root(M_Equal(MPPFFPTF,MPPTTPFT)))
    ftst(root(M_Equal(MPPFFPFT,MPPTTPFT)))
    ftst(root(M_Equal(MPPFFPFF,MPPTTPFT)))

    ftst(root(M_Equal(MT      ,MPPTTPFF)))
    ftst(root(M_Equal(MF      ,MPPTTPFF)))
    ftst(root(M_Equal(MPTT    ,MPPTTPFF)))
    ftst(root(M_Equal(MPTF    ,MPPTTPFF)))
    ftst(root(M_Equal(MPFT    ,MPPTTPFF)))
    ftst(root(M_Equal(MPFF    ,MPPTTPFF)))
    ftst(root(M_Equal(MPPTTPTT,MPPTTPFF)))
    ftst(root(M_Equal(MPPTTPTF,MPPTTPFF)))
    ftst(root(M_Equal(MPPTTPFT,MPPTTPFF)))
    ttst(root(M_Equal(MPPTTPFF,MPPTTPFF)))
    ftst(root(M_Equal(MPPTFPTT,MPPTTPFF)))
    ftst(root(M_Equal(MPPTFPTF,MPPTTPFF)))
    ftst(root(M_Equal(MPPTFPFT,MPPTTPFF)))
    ftst(root(M_Equal(MPPTFPFF,MPPTTPFF)))
    ftst(root(M_Equal(MPPFTPTT,MPPTTPFF)))
    ftst(root(M_Equal(MPPFTPTF,MPPTTPFF)))
    ftst(root(M_Equal(MPPFTPFT,MPPTTPFF)))
    ftst(root(M_Equal(MPPFTPFF,MPPTTPFF)))
    ftst(root(M_Equal(MPPFFPTT,MPPTTPFF)))
    ftst(root(M_Equal(MPPFFPTF,MPPTTPFF)))
    ftst(root(M_Equal(MPPFFPFT,MPPTTPFF)))
    ftst(root(M_Equal(MPPFFPFF,MPPTTPFF)))

    ftst(root(M_Equal(MT      ,MPPTFPTT)))
    ftst(root(M_Equal(MF      ,MPPTFPTT)))
    ftst(root(M_Equal(MPTT    ,MPPTFPTT)))
    ftst(root(M_Equal(MPTF    ,MPPTFPTT)))
    ftst(root(M_Equal(MPFT    ,MPPTFPTT)))
    ftst(root(M_Equal(MPFF    ,MPPTFPTT)))
    ftst(root(M_Equal(MPPTTPTT,MPPTFPTT)))
    ftst(root(M_Equal(MPPTTPTF,MPPTFPTT)))
    ftst(root(M_Equal(MPPTTPFT,MPPTFPTT)))
    ftst(root(M_Equal(MPPTTPFF,MPPTFPTT)))
    ttst(root(M_Equal(MPPTFPTT,MPPTFPTT)))
    ftst(root(M_Equal(MPPTFPTF,MPPTFPTT)))
    ftst(root(M_Equal(MPPTFPFT,MPPTFPTT)))
    ftst(root(M_Equal(MPPTFPFF,MPPTFPTT)))
    ftst(root(M_Equal(MPPFTPTT,MPPTFPTT)))
    ftst(root(M_Equal(MPPFTPTF,MPPTFPTT)))
    ftst(root(M_Equal(MPPFTPFT,MPPTFPTT)))
    ftst(root(M_Equal(MPPFTPFF,MPPTFPTT)))
    ftst(root(M_Equal(MPPFFPTT,MPPTFPTT)))
    ftst(root(M_Equal(MPPFFPTF,MPPTFPTT)))
    ftst(root(M_Equal(MPPFFPFT,MPPTFPTT)))
    ftst(root(M_Equal(MPPFFPFF,MPPTFPTT)))

    ftst(root(M_Equal(MT      ,MPPTFPTF)))
    ftst(root(M_Equal(MF      ,MPPTFPTF)))
    ftst(root(M_Equal(MPTT    ,MPPTFPTF)))
    ftst(root(M_Equal(MPTF    ,MPPTFPTF)))
    ftst(root(M_Equal(MPFT    ,MPPTFPTF)))
    ftst(root(M_Equal(MPFF    ,MPPTFPTF)))
    ftst(root(M_Equal(MPPTTPTT,MPPTFPTF)))
    ftst(root(M_Equal(MPPTTPTF,MPPTFPTF)))
    ftst(root(M_Equal(MPPTTPFT,MPPTFPTF)))
    ftst(root(M_Equal(MPPTTPFF,MPPTFPTF)))
    ftst(root(M_Equal(MPPTFPTT,MPPTFPTF)))
    ttst(root(M_Equal(MPPTFPTF,MPPTFPTF)))
    ftst(root(M_Equal(MPPTFPFT,MPPTFPTF)))
    ftst(root(M_Equal(MPPTFPFF,MPPTFPTF)))
    ftst(root(M_Equal(MPPFTPTT,MPPTFPTF)))
    ftst(root(M_Equal(MPPFTPTF,MPPTFPTF)))
    ftst(root(M_Equal(MPPFTPFT,MPPTFPTF)))
    ftst(root(M_Equal(MPPFTPFF,MPPTFPTF)))
    ftst(root(M_Equal(MPPFFPTT,MPPTFPTF)))
    ftst(root(M_Equal(MPPFFPTF,MPPTFPTF)))
    ftst(root(M_Equal(MPPFFPFT,MPPTFPTF)))
    ftst(root(M_Equal(MPPFFPFF,MPPTFPTF)))

    ftst(root(M_Equal(MT      ,MPPTFPFT)))
    ftst(root(M_Equal(MF      ,MPPTFPFT)))
    ftst(root(M_Equal(MPTT    ,MPPTFPFT)))
    ftst(root(M_Equal(MPTF    ,MPPTFPFT)))
    ftst(root(M_Equal(MPFT    ,MPPTFPFT)))
    ftst(root(M_Equal(MPFF    ,MPPTFPFT)))
    ftst(root(M_Equal(MPPTTPTT,MPPTFPFT)))
    ftst(root(M_Equal(MPPTTPTF,MPPTFPFT)))
    ftst(root(M_Equal(MPPTTPFT,MPPTFPFT)))
    ftst(root(M_Equal(MPPTTPFF,MPPTFPFT)))
    ftst(root(M_Equal(MPPTFPTT,MPPTFPFT)))
    ftst(root(M_Equal(MPPTFPTF,MPPTFPFT)))
    ttst(root(M_Equal(MPPTFPFT,MPPTFPFT)))
    ftst(root(M_Equal(MPPTFPFF,MPPTFPFT)))
    ftst(root(M_Equal(MPPFTPTT,MPPTFPFT)))
    ftst(root(M_Equal(MPPFTPTF,MPPTFPFT)))
    ftst(root(M_Equal(MPPFTPFT,MPPTFPFT)))
    ftst(root(M_Equal(MPPFTPFF,MPPTFPFT)))
    ftst(root(M_Equal(MPPFFPTT,MPPTFPFT)))
    ftst(root(M_Equal(MPPFFPTF,MPPTFPFT)))
    ftst(root(M_Equal(MPPFFPFT,MPPTFPFT)))
    ftst(root(M_Equal(MPPFFPFF,MPPTFPFT)))

    ftst(root(M_Equal(MT      ,MPPTFPFF)))
    ftst(root(M_Equal(MF      ,MPPTFPFF)))
    ftst(root(M_Equal(MPTT    ,MPPTFPFF)))
    ftst(root(M_Equal(MPTF    ,MPPTFPFF)))
    ftst(root(M_Equal(MPFT    ,MPPTFPFF)))
    ftst(root(M_Equal(MPFF    ,MPPTFPFF)))
    ftst(root(M_Equal(MPPTTPTT,MPPTFPFF)))
    ftst(root(M_Equal(MPPTTPTF,MPPTFPFF)))
    ftst(root(M_Equal(MPPTTPFT,MPPTFPFF)))
    ftst(root(M_Equal(MPPTTPFF,MPPTFPFF)))
    ftst(root(M_Equal(MPPTFPTT,MPPTFPFF)))
    ftst(root(M_Equal(MPPTFPTF,MPPTFPFF)))
    ftst(root(M_Equal(MPPTFPFT,MPPTFPFF)))
    ttst(root(M_Equal(MPPTFPFF,MPPTFPFF)))
    ftst(root(M_Equal(MPPFTPTT,MPPTFPFF)))
    ftst(root(M_Equal(MPPFTPTF,MPPTFPFF)))
    ftst(root(M_Equal(MPPFTPFT,MPPTFPFF)))
    ftst(root(M_Equal(MPPFTPFF,MPPTFPFF)))
    ftst(root(M_Equal(MPPFFPTT,MPPTFPFF)))
    ftst(root(M_Equal(MPPFFPTF,MPPTFPFF)))
    ftst(root(M_Equal(MPPFFPFT,MPPTFPFF)))
    ftst(root(M_Equal(MPPFFPFF,MPPTFPFF)))

    ftst(root(M_Equal(MT      ,MPPFTPTT)))
    ftst(root(M_Equal(MF      ,MPPFTPTT)))
    ftst(root(M_Equal(MPTT    ,MPPFTPTT)))
    ftst(root(M_Equal(MPTF    ,MPPFTPTT)))
    ftst(root(M_Equal(MPFT    ,MPPFTPTT)))
    ftst(root(M_Equal(MPFF    ,MPPFTPTT)))
    ftst(root(M_Equal(MPPTTPTT,MPPFTPTT)))
    ftst(root(M_Equal(MPPTTPTF,MPPFTPTT)))
    ftst(root(M_Equal(MPPTTPFT,MPPFTPTT)))
    ftst(root(M_Equal(MPPTTPFF,MPPFTPTT)))
    ftst(root(M_Equal(MPPTFPTT,MPPFTPTT)))
    ftst(root(M_Equal(MPPTFPTF,MPPFTPTT)))
    ftst(root(M_Equal(MPPTFPFT,MPPFTPTT)))
    ftst(root(M_Equal(MPPTFPFF,MPPFTPTT)))
    ttst(root(M_Equal(MPPFTPTT,MPPFTPTT)))
    ftst(root(M_Equal(MPPFTPTF,MPPFTPTT)))
    ftst(root(M_Equal(MPPFTPFT,MPPFTPTT)))
    ftst(root(M_Equal(MPPFTPFF,MPPFTPTT)))
    ftst(root(M_Equal(MPPFFPTT,MPPFTPTT)))
    ftst(root(M_Equal(MPPFFPTF,MPPFTPTT)))
    ftst(root(M_Equal(MPPFFPFT,MPPFTPTT)))
    ftst(root(M_Equal(MPPFFPFF,MPPFTPTT)))

    ftst(root(M_Equal(MT      ,MPPFTPTF)))
    ftst(root(M_Equal(MF      ,MPPFTPTF)))
    ftst(root(M_Equal(MPTT    ,MPPFTPTF)))
    ftst(root(M_Equal(MPTF    ,MPPFTPTF)))
    ftst(root(M_Equal(MPFT    ,MPPFTPTF)))
    ftst(root(M_Equal(MPFF    ,MPPFTPTF)))
    ftst(root(M_Equal(MPPTTPTT,MPPFTPTF)))
    ftst(root(M_Equal(MPPTTPTF,MPPFTPTF)))
    ftst(root(M_Equal(MPPTTPFT,MPPFTPTF)))
    ftst(root(M_Equal(MPPTTPFF,MPPFTPTF)))
    ftst(root(M_Equal(MPPTFPTT,MPPFTPTF)))
    ftst(root(M_Equal(MPPTFPTF,MPPFTPTF)))
    ftst(root(M_Equal(MPPTFPFT,MPPFTPTF)))
    ftst(root(M_Equal(MPPTFPFF,MPPFTPTF)))
    ftst(root(M_Equal(MPPFTPTT,MPPFTPTF)))
    ttst(root(M_Equal(MPPFTPTF,MPPFTPTF)))
    ftst(root(M_Equal(MPPFTPFT,MPPFTPTF)))
    ftst(root(M_Equal(MPPFTPFF,MPPFTPTF)))
    ftst(root(M_Equal(MPPFFPTT,MPPFTPTF)))
    ftst(root(M_Equal(MPPFFPTF,MPPFTPTF)))
    ftst(root(M_Equal(MPPFFPFT,MPPFTPTF)))
    ftst(root(M_Equal(MPPFFPFF,MPPFTPTF)))

    ftst(root(M_Equal(MT      ,MPPFTPFT)))
    ftst(root(M_Equal(MF      ,MPPFTPFT)))
    ftst(root(M_Equal(MPTT    ,MPPFTPFT)))
    ftst(root(M_Equal(MPTF    ,MPPFTPFT)))
    ftst(root(M_Equal(MPFT    ,MPPFTPFT)))
    ftst(root(M_Equal(MPFF    ,MPPFTPFT)))
    ftst(root(M_Equal(MPPTTPTT,MPPFTPFT)))
    ftst(root(M_Equal(MPPTTPTF,MPPFTPFT)))
    ftst(root(M_Equal(MPPTTPFT,MPPFTPFT)))
    ftst(root(M_Equal(MPPTTPFF,MPPFTPFT)))
    ftst(root(M_Equal(MPPTFPTT,MPPFTPFT)))
    ftst(root(M_Equal(MPPTFPTF,MPPFTPFT)))
    ftst(root(M_Equal(MPPTFPFT,MPPFTPFT)))
    ftst(root(M_Equal(MPPTFPFF,MPPFTPFT)))
    ftst(root(M_Equal(MPPFTPTT,MPPFTPFT)))
    ftst(root(M_Equal(MPPFTPTF,MPPFTPFT)))
    ttst(root(M_Equal(MPPFTPFT,MPPFTPFT)))
    ftst(root(M_Equal(MPPFTPFF,MPPFTPFT)))
    ftst(root(M_Equal(MPPFFPTT,MPPFTPFT)))
    ftst(root(M_Equal(MPPFFPTF,MPPFTPFT)))
    ftst(root(M_Equal(MPPFFPFT,MPPFTPFT)))
    ftst(root(M_Equal(MPPFFPFF,MPPFTPFT)))

    ftst(root(M_Equal(MT      ,MPPFTPFF)))
    ftst(root(M_Equal(MF      ,MPPFTPFF)))
    ftst(root(M_Equal(MPTT    ,MPPFTPFF)))
    ftst(root(M_Equal(MPTF    ,MPPFTPFF)))
    ftst(root(M_Equal(MPFT    ,MPPFTPFF)))
    ftst(root(M_Equal(MPFF    ,MPPFTPFF)))
    ftst(root(M_Equal(MPPTTPTT,MPPFTPFF)))
    ftst(root(M_Equal(MPPTTPTF,MPPFTPFF)))
    ftst(root(M_Equal(MPPTTPFT,MPPFTPFF)))
    ftst(root(M_Equal(MPPTTPFF,MPPFTPFF)))
    ftst(root(M_Equal(MPPTFPTT,MPPFTPFF)))
    ftst(root(M_Equal(MPPTFPTF,MPPFTPFF)))
    ftst(root(M_Equal(MPPTFPFT,MPPFTPFF)))
    ftst(root(M_Equal(MPPTFPFF,MPPFTPFF)))
    ftst(root(M_Equal(MPPFTPTT,MPPFTPFF)))
    ftst(root(M_Equal(MPPFTPTF,MPPFTPFF)))
    ftst(root(M_Equal(MPPFTPFT,MPPFTPFF)))
    ttst(root(M_Equal(MPPFTPFF,MPPFTPFF)))
    ftst(root(M_Equal(MPPFFPTT,MPPFTPFF)))
    ftst(root(M_Equal(MPPFFPTF,MPPFTPFF)))
    ftst(root(M_Equal(MPPFFPFT,MPPFTPFF)))
    ftst(root(M_Equal(MPPFFPFF,MPPFTPFF)))

    ftst(root(M_Equal(MT      ,MPPFFPTT)))
    ftst(root(M_Equal(MF      ,MPPFFPTT)))
    ftst(root(M_Equal(MPTT    ,MPPFFPTT)))
    ftst(root(M_Equal(MPTF    ,MPPFFPTT)))
    ftst(root(M_Equal(MPFT    ,MPPFFPTT)))
    ftst(root(M_Equal(MPFF    ,MPPFFPTT)))
    ftst(root(M_Equal(MPPTTPTT,MPPFFPTT)))
    ftst(root(M_Equal(MPPTTPTF,MPPFFPTT)))
    ftst(root(M_Equal(MPPTTPFT,MPPFFPTT)))
    ftst(root(M_Equal(MPPTTPFF,MPPFFPTT)))
    ftst(root(M_Equal(MPPTFPTT,MPPFFPTT)))
    ftst(root(M_Equal(MPPTFPTF,MPPFFPTT)))
    ftst(root(M_Equal(MPPTFPFT,MPPFFPTT)))
    ftst(root(M_Equal(MPPTFPFF,MPPFFPTT)))
    ftst(root(M_Equal(MPPFTPTT,MPPFFPTT)))
    ftst(root(M_Equal(MPPFTPTF,MPPFFPTT)))
    ftst(root(M_Equal(MPPFTPFT,MPPFFPTT)))
    ftst(root(M_Equal(MPPFTPFF,MPPFFPTT)))
    ttst(root(M_Equal(MPPFFPTT,MPPFFPTT)))
    ftst(root(M_Equal(MPPFFPTF,MPPFFPTT)))
    ftst(root(M_Equal(MPPFFPFT,MPPFFPTT)))
    ftst(root(M_Equal(MPPFFPFF,MPPFFPTT)))

    ftst(root(M_Equal(MT      ,MPPFFPTF)))
    ftst(root(M_Equal(MF      ,MPPFFPTF)))
    ftst(root(M_Equal(MPTT    ,MPPFFPTF)))
    ftst(root(M_Equal(MPTF    ,MPPFFPTF)))
    ftst(root(M_Equal(MPFT    ,MPPFFPTF)))
    ftst(root(M_Equal(MPFF    ,MPPFFPTF)))
    ftst(root(M_Equal(MPPTTPTT,MPPFFPTF)))
    ftst(root(M_Equal(MPPTTPTF,MPPFFPTF)))
    ftst(root(M_Equal(MPPTTPFT,MPPFFPTF)))
    ftst(root(M_Equal(MPPTTPFF,MPPFFPTF)))
    ftst(root(M_Equal(MPPTFPTT,MPPFFPTF)))
    ftst(root(M_Equal(MPPTFPTF,MPPFFPTF)))
    ftst(root(M_Equal(MPPTFPFT,MPPFFPTF)))
    ftst(root(M_Equal(MPPTFPFF,MPPFFPTF)))
    ftst(root(M_Equal(MPPFTPTT,MPPFFPTF)))
    ftst(root(M_Equal(MPPFTPTF,MPPFFPTF)))
    ftst(root(M_Equal(MPPFTPFT,MPPFFPTF)))
    ftst(root(M_Equal(MPPFTPFF,MPPFFPTF)))
    ftst(root(M_Equal(MPPFFPTT,MPPFFPTF)))
    ttst(root(M_Equal(MPPFFPTF,MPPFFPTF)))
    ftst(root(M_Equal(MPPFFPFT,MPPFFPTF)))
    ftst(root(M_Equal(MPPFFPFF,MPPFFPTF)))

    ftst(root(M_Equal(MT      ,MPPFFPFT)))
    ftst(root(M_Equal(MF      ,MPPFFPFT)))
    ftst(root(M_Equal(MPTT    ,MPPFFPFT)))
    ftst(root(M_Equal(MPTF    ,MPPFFPFT)))
    ftst(root(M_Equal(MPFT    ,MPPFFPFT)))
    ftst(root(M_Equal(MPFF    ,MPPFFPFT)))
    ftst(root(M_Equal(MPPTTPTT,MPPFFPFT)))
    ftst(root(M_Equal(MPPTTPTF,MPPFFPFT)))
    ftst(root(M_Equal(MPPTTPFT,MPPFFPFT)))
    ftst(root(M_Equal(MPPTTPFF,MPPFFPFT)))
    ftst(root(M_Equal(MPPTFPTT,MPPFFPFT)))
    ftst(root(M_Equal(MPPTFPTF,MPPFFPFT)))
    ftst(root(M_Equal(MPPTFPFT,MPPFFPFT)))
    ftst(root(M_Equal(MPPTFPFF,MPPFFPFT)))
    ftst(root(M_Equal(MPPFTPTT,MPPFFPFT)))
    ftst(root(M_Equal(MPPFTPTF,MPPFFPFT)))
    ftst(root(M_Equal(MPPFTPFT,MPPFFPFT)))
    ftst(root(M_Equal(MPPFTPFF,MPPFFPFT)))
    ftst(root(M_Equal(MPPFFPTT,MPPFFPFT)))
    ftst(root(M_Equal(MPPFFPTF,MPPFFPFT)))
    ttst(root(M_Equal(MPPFFPFT,MPPFFPFT)))
    ftst(root(M_Equal(MPPFFPFF,MPPFFPFT)))

    ftst(root(M_Equal(MT      ,MPPFFPFF)))
    ftst(root(M_Equal(MF      ,MPPFFPFF)))
    ftst(root(M_Equal(MPTT    ,MPPFFPFF)))
    ftst(root(M_Equal(MPTF    ,MPPFFPFF)))
    ftst(root(M_Equal(MPFT    ,MPPFFPFF)))
    ftst(root(M_Equal(MPFF    ,MPPFFPFF)))
    ftst(root(M_Equal(MPPTTPTT,MPPFFPFF)))
    ftst(root(M_Equal(MPPTTPTF,MPPFFPFF)))
    ftst(root(M_Equal(MPPTTPFT,MPPFFPFF)))
    ftst(root(M_Equal(MPPTTPFF,MPPFFPFF)))
    ftst(root(M_Equal(MPPTFPTT,MPPFFPFF)))
    ftst(root(M_Equal(MPPTFPTF,MPPFFPFF)))
    ftst(root(M_Equal(MPPTFPFT,MPPFFPFF)))
    ftst(root(M_Equal(MPPTFPFF,MPPFFPFF)))
    ftst(root(M_Equal(MPPFTPTT,MPPFFPFF)))
    ftst(root(M_Equal(MPPFTPTF,MPPFFPFF)))
    ftst(root(M_Equal(MPPFTPFT,MPPFFPFF)))
    ftst(root(M_Equal(MPPFTPFF,MPPFFPFF)))
    ftst(root(M_Equal(MPPFFPTT,MPPFFPFF)))
    ftst(root(M_Equal(MPPFFPTF,MPPFFPFF)))
    ftst(root(M_Equal(MPPFFPFT,MPPFFPFF)))
    ttst(root(M_Equal(MPPFFPFF,MPPFFPFF)))

    ntst(U_Equal(TT,TT))
    ptst(U_Equal(TT,FF))
    ptst(U_Equal(TT,I0))
    ptst(U_Equal(TT,EE))
    ptst(U_Equal(TT,P0))
    ptst(U_Equal(TT,XX))
    ptst(U_Equal(TT,M0))
    ptst(U_Equal(TT,B0))
    ptst(U_Equal(FF,TT))
    ntst(U_Equal(FF,FF))
    ptst(U_Equal(FF,I0))
    ptst(U_Equal(FF,EE))
    ptst(U_Equal(FF,P0))
    ptst(U_Equal(FF,XX))
    ptst(U_Equal(FF,M0))
    ptst(U_Equal(FF,B0))
    ptst(U_Equal(I0,TT))
    ptst(U_Equal(I0,FF))
    ntst(U_Equal(I0,I0))
    ptst(U_Equal(I0,EE))
    ptst(U_Equal(I0,P0))
    ptst(U_Equal(I0,XX))
    ptst(U_Equal(I0,M0))
    ptst(U_Equal(I0,B0))
    ptst(U_Equal(EE,TT))
    ptst(U_Equal(EE,FF))
    ptst(U_Equal(EE,I0))
    ntst(U_Equal(EE,EE))
    ptst(U_Equal(EE,P0))
    ptst(U_Equal(EE,XX))
    ptst(U_Equal(EE,M0))
    ptst(U_Equal(EE,B0))
    ptst(U_Equal(P0,TT))
    ptst(U_Equal(P0,FF))
    ptst(U_Equal(P0,I0))
    ptst(U_Equal(P0,EE))
    ntst(U_Equal(P0,P0))
    ptst(U_Equal(P0,XX))
    ptst(U_Equal(P0,M0))
    ptst(U_Equal(P0,B0))
    ptst(U_Equal(XX,TT))
    ptst(U_Equal(XX,FF))
    ptst(U_Equal(XX,I0))
    ptst(U_Equal(XX,EE))
    ptst(U_Equal(XX,P0))
    ntst(U_Equal(XX,XX))
    ptst(U_Equal(XX,M0))
    ptst(U_Equal(XX,B0))
    ptst(U_Equal(M0,TT))
    ptst(U_Equal(M0,FF))
    ptst(U_Equal(M0,I0))
    ptst(U_Equal(M0,EE))
    ptst(U_Equal(M0,P0))
    ptst(U_Equal(M0,XX))
    ntst(U_Equal(M0,M0))
    ptst(U_Equal(M0,B0))
    ptst(U_Equal(B0,TT))
    ptst(U_Equal(B0,FF))
    ptst(U_Equal(B0,I0))
    ptst(U_Equal(B0,EE))
    ptst(U_Equal(B0,P0))
    ptst(U_Equal(B0,XX))
    ptst(U_Equal(B0,M0))
    ntst(U_Equal(B0,B0))

    ntst(U_Equal(TT,U_norm(TT)))
    ptst(U_Equal(TT,U_norm(FF)))

    ptst(U_Equal(TT,U_norm(I0)))
    ptst(U_Equal(TT,U_norm(EE)))
    ptst(U_Equal(TT,U_norm(P0)))
    ptst(U_Equal(TT,U_norm(M0)))
    ptst(U_Equal(TT,U_norm(B0)))
    ptst(U_Equal(FF,U_norm(TT)))
    ntst(U_Equal(FF,U_norm(FF)))
    ptst(U_Equal(FF,U_norm(I0)))
    ptst(U_Equal(FF,U_norm(EE)))
    ptst(U_Equal(FF,U_norm(P0)))
    ptst(U_Equal(FF,U_norm(M0)))
    ptst(U_Equal(FF,U_norm(B0)))
    ptst(U_Equal(I0,U_norm(TT)))
    ptst(U_Equal(I0,U_norm(FF)))
    ntst(U_Equal(I0,U_norm(I0)))
    ptst(U_Equal(I0,U_norm(EE)))
    ptst(U_Equal(I0,U_norm(P0)))
    ptst(U_Equal(I0,U_norm(M0)))
    ptst(U_Equal(I0,U_norm(B0)))
    ptst(U_Equal(EE,U_norm(TT)))
    ptst(U_Equal(EE,U_norm(FF)))
    ptst(U_Equal(EE,U_norm(I0)))
    ntst(U_Equal(EE,U_norm(EE)))
    ptst(U_Equal(EE,U_norm(P0)))
    ptst(U_Equal(EE,U_norm(M0)))
    ptst(U_Equal(EE,U_norm(B0)))
    ptst(U_Equal(P0,U_norm(TT)))
    ptst(U_Equal(P0,U_norm(FF)))
    ptst(U_Equal(P0,U_norm(I0)))
    ptst(U_Equal(P0,U_norm(EE)))
    ntst(U_Equal(P0,U_norm(P0)))
    ptst(U_Equal(P0,U_norm(M0)))
    ptst(U_Equal(P0,U_norm(B0)))
    ptst(U_Equal(M0,U_norm(TT)))
    ptst(U_Equal(M0,U_norm(FF)))
    ptst(U_Equal(M0,U_norm(I0)))
    ptst(U_Equal(M0,U_norm(EE)))
    ptst(U_Equal(M0,U_norm(P0)))
    ntst(U_Equal(M0,U_norm(M0)))
    ptst(U_Equal(M0,U_norm(B0)))
    ptst(U_Equal(B0,U_norm(TT)))
    ptst(U_Equal(B0,U_norm(FF)))
    ptst(U_Equal(B0,U_norm(I0)))
    ptst(U_Equal(B0,U_norm(EE)))
    ptst(U_Equal(B0,U_norm(P0)))
    ptst(U_Equal(B0,U_norm(M0)))
    ntst(U_Equal(B0,U_norm(B0)))

    ttst(root(M_equal(M_true,M_true)))
    ftst(root(M_equal(M_true,M_false)))
    ftst(root(M_equal(M_false,M_true)))
    ttst(root(M_equal(M_false,M_false)))

    ntst(U_equal(UT,UT))
    ptst(U_equal(UT,UF))
    ptst(U_equal(UF,UT))
    ntst(U_equal(UF,UF))

    ntst(U_equal(TT,TT))
    ptst(U_equal(TT,FF))
    ptst(U_equal(TT,I0))
    ptst(U_equal(TT,EE))
    ptst(U_equal(TT,P0))
    ptst(U_equal(TT,M0))
    ptst(U_equal(TT,B0))

    ptst(U_equal(FF,TT))
    ntst(U_equal(FF,FF))
    ptst(U_equal(FF,I0))
    ptst(U_equal(FF,EE))
    ptst(U_equal(FF,P0))
    ptst(U_equal(FF,M0))
    ptst(U_equal(FF,B0))

    ptst(U_equal(I0,TT))
    ptst(U_equal(I0,FF))
    ntst(U_equal(I0,I0))
    ptst(U_equal(I0,EE))
    ptst(U_equal(I0,P0))
    ptst(U_equal(I0,M0))
    ptst(U_equal(I0,B0))

    ptst(U_equal(EE,TT))
    ptst(U_equal(EE,FF))
    ptst(U_equal(EE,I0))
    ntst(U_equal(EE,EE))
    ptst(U_equal(EE,P0))
    ptst(U_equal(EE,M0))
    ptst(U_equal(EE,B0))

    ptst(U_equal(P0,TT))
    ptst(U_equal(P0,FF))
    ptst(U_equal(P0,I0))
    ptst(U_equal(P0,EE))
    ntst(U_equal(P0,P0))
    ptst(U_equal(P0,M0))
    ptst(U_equal(P0,B0))

    ptst(U_equal(M0,TT))
    ptst(U_equal(M0,FF))
    ptst(U_equal(M0,I0))
    ptst(U_equal(M0,EE))
    ptst(U_equal(M0,P0))
    ntst(U_equal(M0,M0))
    ptst(U_equal(M0,B0))

    ptst(U_equal(B0,TT))
    ptst(U_equal(B0,FF))
    ptst(U_equal(B0,I0))
    ptst(U_equal(B0,EE))
    ptst(U_equal(B0,P0))
    ptst(U_equal(B0,M0))
    ntst(U_equal(B0,B0))

/*
---------------------------------------------
Test of optimized constructs on base page
---------------------------------------------
*/

    Ftst(False)

    Ttst(Eq(TT,TT))
    Ftst(Eq(TT,FF))
    Ftst(Eq(TT,I0))
    Ftst(Eq(TT,EE))
    Ftst(Eq(TT,P0))
    Ftst(Eq(TT,M0))
    Ftst(Eq(TT,B0))
    Ftst(Eq(FF,TT))
    Ttst(Eq(FF,FF))
    Ftst(Eq(FF,I0))
    Ftst(Eq(FF,EE))
    Ftst(Eq(FF,P0))
    Ftst(Eq(FF,M0))
    Ftst(Eq(FF,B0))
    Ftst(Eq(I0,TT))
    Ftst(Eq(I0,FF))
    Ttst(Eq(I0,I0))
    Ftst(Eq(I0,EE))
    Ftst(Eq(I0,P0))
    Ftst(Eq(I0,M0))
    Ftst(Eq(I0,B0))
    Ftst(Eq(EE,TT))
    Ftst(Eq(EE,FF))
    Ftst(Eq(EE,I0))
    Ttst(Eq(EE,EE))
    Ftst(Eq(EE,P0))
    Ftst(Eq(EE,M0))
    Ftst(Eq(EE,B0))
    Ftst(Eq(P0,TT))
    Ftst(Eq(P0,FF))
    Ftst(Eq(P0,I0))
    Ftst(Eq(P0,EE))
    Ttst(Eq(P0,P0))
    Ftst(Eq(P0,M0))
    Ftst(Eq(P0,B0))
    Ftst(Eq(M0,TT))
    Ftst(Eq(M0,FF))
    Ftst(Eq(M0,I0))
    Ftst(Eq(M0,EE))
    Ftst(Eq(M0,P0))
    Ttst(Eq(M0,M0))
    Ftst(Eq(M0,B0))
    Ftst(Eq(B0,TT))
    Ftst(Eq(B0,FF))
    Ftst(Eq(B0,I0))
    Ftst(Eq(B0,EE))
    Ftst(Eq(B0,P0))
    Ftst(Eq(B0,M0))
    Ttst(Eq(B0,B0))

    Ttst(Eq(I12 ,I12 ))
    Ftst(Eq(I12 ,I123))
    Ftst(Eq(I12 ,I923))
    Ftst(Eq(I12 ,I129))
    Ftst(Eq(I123,I12 ))
    Ttst(Eq(I123,I123))
    Ftst(Eq(I123,I923))
    Ftst(Eq(I123,I129))
    Ftst(Eq(I923,I12 ))
    Ftst(Eq(I923,I123))
    Ttst(Eq(I923,I923))
    Ftst(Eq(I923,I129))
    Ftst(Eq(I129,I12 ))
    Ftst(Eq(I129,I123))
    Ftst(Eq(I129,I923))
    Ttst(Eq(I129,I129))

    Ttst(Eq(J12 ,J12 ))
    Ftst(Eq(J12 ,J123))
    Ftst(Eq(J12 ,J923))
    Ftst(Eq(J12 ,J129))
    Ftst(Eq(J123,J12 ))
    Ttst(Eq(J123,J123))
    Ftst(Eq(J123,J923))
    Ftst(Eq(J123,J129))
    Ftst(Eq(J923,J12 ))
    Ftst(Eq(J923,J123))
    Ttst(Eq(J923,J923))
    Ftst(Eq(J923,J129))
    Ftst(Eq(J129,J12 ))
    Ftst(Eq(J129,J123))
    Ftst(Eq(J129,J923))
    Ttst(Eq(J129,J129))

    Ttst(Eq(EE,I[1]))

    Ttst(Eq(P0,P0))
    Ftst(Eq(P0,P1))
    Ftst(Eq(P0,P2))
    Ftst(Eq(P0,P3))
    Ftst(Eq(P1,P0))
    Ttst(Eq(P1,P1))
    Ftst(Eq(P1,P2))
    Ftst(Eq(P1,P3))
    Ftst(Eq(P2,P0))
    Ftst(Eq(P2,P1))
    Ttst(Eq(P2,P2))
    Ftst(Eq(P2,P3))
    Ftst(Eq(P3,P0))
    Ftst(Eq(P3,P1))
    Ftst(Eq(P3,P2))
    Ttst(Eq(P3,P3))

    Ttst(Eq(B0,B0))
    Ftst(Eq(B0,B1))
    Ftst(Eq(B0,B2))
    Ftst(Eq(B0,B3))
    Ftst(Eq(B1,B0))
    Ttst(Eq(B1,B1))
    Ftst(Eq(B1,B2))
    Ftst(Eq(B1,B3))
    Ftst(Eq(B2,B0))
    Ftst(Eq(B2,B1))
    Ttst(Eq(B2,B2))
    Ftst(Eq(B2,B3))
    Ftst(Eq(B3,B0))
    Ftst(Eq(B3,B1))
    Ftst(Eq(B3,B2))
    Ttst(Eq(B3,B3))

    Xtst(Lognot(TT))
    Xtst(Lognot(FF))
    Xtst(Lognot(P0))
    Xtst(Lognot(M0))
    Xtst(Lognot(B0))
    Etst(Lognot(Z),M)
    Etst(Lognot(M),Z)
    Etst(Lognot(I0011),J1100)
    Etst(Lognot(J1010_0101),I0101_1010)

    Xtst(Logand(TT,TT))
    Xtst(Logand(TT,FF))
    Xtst(Logand(TT,I0))
    Xtst(Logand(TT,P0))
    Xtst(Logand(TT,M0))
    Xtst(Logand(TT,B0))
    Xtst(Logand(FF,TT))
    Xtst(Logand(FF,FF))
    Xtst(Logand(FF,I0))
    Xtst(Logand(FF,P0))
    Xtst(Logand(FF,M0))
    Xtst(Logand(FF,B0))
    Xtst(Logand(I0,TT))
    Xtst(Logand(I0,FF))
    Xtst(Logand(I0,P0))
    Xtst(Logand(I0,M0))
    Xtst(Logand(I0,B0))
    Xtst(Logand(P0,TT))
    Xtst(Logand(P0,FF))
    Xtst(Logand(P0,I0))
    Xtst(Logand(P0,P0))
    Xtst(Logand(P0,M0))
    Xtst(Logand(P0,B0))
    Xtst(Logand(M0,TT))
    Xtst(Logand(M0,FF))
    Xtst(Logand(M0,I0))
    Xtst(Logand(M0,P0))
    Xtst(Logand(M0,M0))
    Xtst(Logand(M0,B0))
    Xtst(Logand(B0,TT))
    Xtst(Logand(B0,FF))
    Xtst(Logand(B0,I0))
    Xtst(Logand(B0,P0))
    Xtst(Logand(B0,M0))
    Xtst(Logand(B0,B0))
    Etst(Logand(I0011     ,I0101     ),I0001     )
    Etst(Logand(I1100     ,I1010     ),I1000     )
    Etst(Logand(I0011_1100,I0101_1010),I0001_1000)
    Etst(Logand(I0011     ,I0101_1010),I0001     )
    Etst(Logand(I0101_1010,I0011     ),I0001     )
    Etst(Logand(I0011     ,J0101     ),I0001     )
    Etst(Logand(I1100     ,J1010     ),I1000     )
    Etst(Logand(I0011_1100,J0101_1010),I0001_1000)
    Etst(Logand(I0011     ,J0101_1010),I0001     )
    Etst(Logand(I0101_1010,J0011     ),intcons(Z0001,I1010))
    Etst(Logand(J0011     ,I0101     ),I0001     )
    Etst(Logand(J1100     ,I1010     ),I1000     )
    Etst(Logand(J0011_1100,I0101_1010),I0001_1000)
    Etst(Logand(J0011     ,I0101_1010),intcons(Z0001,I1010))
    Etst(Logand(J0101_1010,I0011     ),I0001     )
    Etst(Logand(J0011     ,J0101     ),J0001     )
    Etst(Logand(J1100     ,J1010     ),J1000     )
    Etst(Logand(J0011_1100,J0101_1010),J0001_1000)
    Etst(Logand(J0011     ,J0101_1010),intcons(M0001,J1010))
    Etst(Logand(J0101_1010,J0011     ),intcons(M0001,J1010))
    Etst(Logand(I0011_1100,I1000_0001),Z)
    Etst(Logand(EE,I0),I0)
    Etst(Logand(EE,J1),EE)
    Etst(Logand(I0,EE),I0)
    Etst(Logand(J1,EE),EE)

    Xtst(Logior(TT,TT))
    Xtst(Logior(TT,FF))
    Xtst(Logior(TT,I0))
    Xtst(Logior(TT,P0))
    Xtst(Logior(TT,M0))
    Xtst(Logior(TT,B0))
    Xtst(Logior(FF,TT))
    Xtst(Logior(FF,FF))
    Xtst(Logior(FF,I0))
    Xtst(Logior(FF,P0))
    Xtst(Logior(FF,M0))
    Xtst(Logior(FF,B0))
    Xtst(Logior(I0,TT))
    Xtst(Logior(I0,FF))
    Xtst(Logior(I0,P0))
    Xtst(Logior(I0,M0))
    Xtst(Logior(I0,B0))
    Xtst(Logior(P0,TT))
    Xtst(Logior(P0,FF))
    Xtst(Logior(P0,I0))
    Xtst(Logior(P0,P0))
    Xtst(Logior(P0,M0))
    Xtst(Logior(P0,B0))
    Xtst(Logior(M0,TT))
    Xtst(Logior(M0,FF))
    Xtst(Logior(M0,I0))
    Xtst(Logior(M0,P0))
    Xtst(Logior(M0,M0))
    Xtst(Logior(M0,B0))
    Xtst(Logior(B0,TT))
    Xtst(Logior(B0,FF))
    Xtst(Logior(B0,I0))
    Xtst(Logior(B0,P0))
    Xtst(Logior(B0,M0))
    Xtst(Logior(B0,B0))
    Etst(Logior(I0011     ,I0101     ),I0111     )
    Etst(Logior(I1100     ,I1010     ),I1110     )
    Etst(Logior(I0011_1100,I0101_1010),I0111_1110)
    Etst(Logior(I0011     ,I0101_1010),intcons(Z0111,I1010))
    Etst(Logior(I0101_1010,I0011     ),intcons(Z0111,I1010))
    Etst(Logior(I0011     ,J0101     ),J0111     )
    Etst(Logior(I1100     ,J1010     ),J1110     )
    Etst(Logior(I0011_1100,J0101_1010),J0111_1110)
    Etst(Logior(I0011     ,J0101_1010),intcons(M0111,J1010))
    Etst(Logior(I0101_1010,J0011     ),J0111     )
    Etst(Logior(J0011     ,I0101     ),J0111     )
    Etst(Logior(J1100     ,I1010     ),J1110     )
    Etst(Logior(J0011_1100,I0101_1010),J0111_1110)
    Etst(Logior(J0011     ,I0101_1010),J0111     )
    Etst(Logior(J0101_1010,I0011     ),intcons(M0111,J1010))
    Etst(Logior(J0011     ,J0101     ),J0111     )
    Etst(Logior(J1100     ,J1010     ),J1110     )
    Etst(Logior(J0011_1100,J0101_1010),J0111_1110)
    Etst(Logior(J0011     ,J0101_1010),J0111     )
    Etst(Logior(J0101_1010,J0011     ),J0111     )
    Etst(Logior(J0011_1100,J1110_0111),M)
    Etst(Logior(EE,I0),EE)
    Etst(Logior(EE,J1),J1)
    Etst(Logior(I0,EE),EE)
    Etst(Logior(J1,EE),J1)

    Xtst(Logxor(TT,TT))
    Xtst(Logxor(TT,FF))
    Xtst(Logxor(TT,I0))
    Xtst(Logxor(TT,P0))
    Xtst(Logxor(TT,M0))
    Xtst(Logxor(TT,B0))
    Xtst(Logxor(FF,TT))
    Xtst(Logxor(FF,FF))
    Xtst(Logxor(FF,I0))
    Xtst(Logxor(FF,P0))
    Xtst(Logxor(FF,M0))
    Xtst(Logxor(FF,B0))
    Xtst(Logxor(I0,TT))
    Xtst(Logxor(I0,FF))
    Xtst(Logxor(I0,P0))
    Xtst(Logxor(I0,M0))
    Xtst(Logxor(I0,B0))
    Xtst(Logxor(P0,TT))
    Xtst(Logxor(P0,FF))
    Xtst(Logxor(P0,I0))
    Xtst(Logxor(P0,P0))
    Xtst(Logxor(P0,M0))
    Xtst(Logxor(P0,B0))
    Xtst(Logxor(M0,TT))
    Xtst(Logxor(M0,FF))
    Xtst(Logxor(M0,I0))
    Xtst(Logxor(M0,P0))
    Xtst(Logxor(M0,M0))
    Xtst(Logxor(M0,B0))
    Xtst(Logxor(B0,TT))
    Xtst(Logxor(B0,FF))
    Xtst(Logxor(B0,I0))
    Xtst(Logxor(B0,P0))
    Xtst(Logxor(B0,M0))
    Xtst(Logxor(B0,B0))
    Etst(Logxor(I0011     ,I0101     ),I0110     )
    Etst(Logxor(I1100     ,I1010     ),I0110     )
    Etst(Logxor(I0011_1100,I0101_1010),I0110_0110)
    Etst(Logxor(I0011     ,I0101_1010),intcons(Z0110,I1010))
    Etst(Logxor(I0101_1010,I0011     ),intcons(Z0110,I1010))
    Etst(Logxor(I0011     ,J0101     ),J0110     )
    Etst(Logxor(I1100     ,J1010     ),J0110     )
    Etst(Logxor(I0011_1100,J0101_1010),J0110_0110)
    Etst(Logxor(I0011     ,J0101_1010),intcons(M0110,J1010))
    Etst(Logxor(I0101_1010,J0011     ),intcons(M0110,J0101))
    Etst(Logxor(J0011     ,I0101     ),J0110     )
    Etst(Logxor(J1100     ,I1010     ),J0110     )
    Etst(Logxor(J0011_1100,I0101_1010),J0110_0110)
    Etst(Logxor(J0011     ,I0101_1010),intcons(M0110,J0101))
    Etst(Logxor(J0101_1010,I0011     ),intcons(M0110,J1010))
    Etst(Logxor(J0011     ,J0101     ),I0110     )
    Etst(Logxor(J1100     ,J1010     ),I0110     )
    Etst(Logxor(J0011_1100,J0101_1010),I0110_0110)
    Etst(Logxor(J0011     ,J0101_1010),intcons(Z0110,I0101))
    Etst(Logxor(J0101_1010,J0011     ),intcons(Z0110,I0101))
    Etst(Logxor(J0101_1010,J0101_1010),Z)
    Etst(Logxor(J0101_1010,I1010_0101),M)
    Etst(Logxor(EE,I0),EE)
    Etst(Logxor(EE,J1),J2)
    Etst(Logxor(I0,EE),EE)
    Etst(Logxor(J1,EE),J2)

    Xtst(Logandc1(TT,TT))
    Xtst(Logandc1(TT,FF))
    Xtst(Logandc1(TT,I0))
    Xtst(Logandc1(TT,P0))
    Xtst(Logandc1(TT,M0))
    Xtst(Logandc1(TT,B0))
    Xtst(Logandc1(FF,TT))
    Xtst(Logandc1(FF,FF))
    Xtst(Logandc1(FF,I0))
    Xtst(Logandc1(FF,P0))
    Xtst(Logandc1(FF,M0))
    Xtst(Logandc1(FF,B0))
    Xtst(Logandc1(I0,TT))
    Xtst(Logandc1(I0,FF))
    Xtst(Logandc1(I0,P0))
    Xtst(Logandc1(I0,M0))
    Xtst(Logandc1(I0,B0))
    Xtst(Logandc1(P0,TT))
    Xtst(Logandc1(P0,FF))
    Xtst(Logandc1(P0,I0))
    Xtst(Logandc1(P0,P0))
    Xtst(Logandc1(P0,M0))
    Xtst(Logandc1(P0,B0))
    Xtst(Logandc1(M0,TT))
    Xtst(Logandc1(M0,FF))
    Xtst(Logandc1(M0,I0))
    Xtst(Logandc1(M0,P0))
    Xtst(Logandc1(M0,M0))
    Xtst(Logandc1(M0,B0))
    Xtst(Logandc1(B0,TT))
    Xtst(Logandc1(B0,FF))
    Xtst(Logandc1(B0,I0))
    Xtst(Logandc1(B0,P0))
    Xtst(Logandc1(B0,M0))
    Xtst(Logandc1(B0,B0))
    Etst(Logandc1(lognot(I0011     ),I0101     ),I0001     )
    Etst(Logandc1(lognot(I1100     ),I1010     ),I1000     )
    Etst(Logandc1(lognot(I0011_1100),I0101_1010),I0001_1000)
    Etst(Logandc1(lognot(I0011     ),I0101_1010),I0001     )
    Etst(Logandc1(lognot(I0101_1010),I0011     ),I0001     )
    Etst(Logandc1(lognot(I0011     ),J0101     ),I0001     )
    Etst(Logandc1(lognot(I1100     ),J1010     ),I1000     )
    Etst(Logandc1(lognot(I0011_1100),J0101_1010),I0001_1000)
    Etst(Logandc1(lognot(I0011     ),J0101_1010),I0001     )
    Etst(Logandc1(lognot(I0101_1010),J0011     ),intcons(Z0001,I1010))
    Etst(Logandc1(lognot(J0011     ),I0101     ),I0001     )
    Etst(Logandc1(lognot(J1100     ),I1010     ),I1000     )
    Etst(Logandc1(lognot(J0011_1100),I0101_1010),I0001_1000)
    Etst(Logandc1(lognot(J0011     ),I0101_1010),intcons(Z0001,I1010))
    Etst(Logandc1(lognot(J0101_1010),I0011     ),I0001     )
    Etst(Logandc1(lognot(J0011     ),J0101     ),J0001     )
    Etst(Logandc1(lognot(J1100     ),J1010     ),J1000     )
    Etst(Logandc1(lognot(J0011_1100),J0101_1010),J0001_1000)
    Etst(Logandc1(lognot(J0011     ),J0101_1010),intcons(M0001,J1010))
    Etst(Logandc1(lognot(J0101_1010),J0011     ),intcons(M0001,J1010))
    Etst(Logandc1(EE,I0),I0)
    Etst(Logandc1(EE,J1),J2)
    Etst(Logandc1(I0,EE),EE)
    Etst(Logandc1(J1,EE),I0)

    Xtst(Logandc2(TT,TT))
    Xtst(Logandc2(TT,FF))
    Xtst(Logandc2(TT,I0))
    Xtst(Logandc2(TT,P0))
    Xtst(Logandc2(TT,M0))
    Xtst(Logandc2(TT,B0))
    Xtst(Logandc2(FF,TT))
    Xtst(Logandc2(FF,FF))
    Xtst(Logandc2(FF,I0))
    Xtst(Logandc2(FF,P0))
    Xtst(Logandc2(FF,M0))
    Xtst(Logandc2(FF,B0))
    Xtst(Logandc2(I0,TT))
    Xtst(Logandc2(I0,FF))
    Xtst(Logandc2(I0,P0))
    Xtst(Logandc2(I0,M0))
    Xtst(Logandc2(I0,B0))
    Xtst(Logandc2(P0,TT))
    Xtst(Logandc2(P0,FF))
    Xtst(Logandc2(P0,I0))
    Xtst(Logandc2(P0,P0))
    Xtst(Logandc2(P0,M0))
    Xtst(Logandc2(P0,B0))
    Xtst(Logandc2(M0,TT))
    Xtst(Logandc2(M0,FF))
    Xtst(Logandc2(M0,I0))
    Xtst(Logandc2(M0,P0))
    Xtst(Logandc2(M0,M0))
    Xtst(Logandc2(M0,B0))
    Xtst(Logandc2(B0,TT))
    Xtst(Logandc2(B0,FF))
    Xtst(Logandc2(B0,I0))
    Xtst(Logandc2(B0,P0))
    Xtst(Logandc2(B0,M0))
    Xtst(Logandc2(B0,B0))
    Etst(Logandc2(I0011     ,lognot(I0101     )),I0001     )
    Etst(Logandc2(I1100     ,lognot(I1010     )),I1000     )
    Etst(Logandc2(I0011_1100,lognot(I0101_1010)),I0001_1000)
    Etst(Logandc2(I0011     ,lognot(I0101_1010)),I0001     )
    Etst(Logandc2(I0101_1010,lognot(I0011     )),I0001     )
    Etst(Logandc2(I0011     ,lognot(J0101     )),I0001     )
    Etst(Logandc2(I1100     ,lognot(J1010     )),I1000     )
    Etst(Logandc2(I0011_1100,lognot(J0101_1010)),I0001_1000)
    Etst(Logandc2(I0011     ,lognot(J0101_1010)),I0001     )
    Etst(Logandc2(I0101_1010,lognot(J0011     )),intcons(Z0001,I1010))
    Etst(Logandc2(J0011     ,lognot(I0101     )),I0001     )
    Etst(Logandc2(J1100     ,lognot(I1010     )),I1000     )
    Etst(Logandc2(J0011_1100,lognot(I0101_1010)),I0001_1000)
    Etst(Logandc2(J0011     ,lognot(I0101_1010)),intcons(Z0001,I1010))
    Etst(Logandc2(J0101_1010,lognot(I0011     )),I0001     )
    Etst(Logandc2(J0011     ,lognot(J0101     )),J0001     )
    Etst(Logandc2(J1100     ,lognot(J1010     )),J1000     )
    Etst(Logandc2(J0011_1100,lognot(J0101_1010)),J0001_1000)
    Etst(Logandc2(J0011     ,lognot(J0101_1010)),intcons(M0001,J1010))
    Etst(Logandc2(J0101_1010,lognot(J0011     )),intcons(M0001,J1010))
    Etst(Logandc2(EE,I0),EE)
    Etst(Logandc2(EE,J1),I0)
    Etst(Logandc2(I0,EE),I0)
    Etst(Logandc2(J1,EE),J2)

    Xtst(Lognand(TT,TT))
    Xtst(Lognand(TT,FF))
    Xtst(Lognand(TT,I0))
    Xtst(Lognand(TT,P0))
    Xtst(Lognand(TT,M0))
    Xtst(Lognand(TT,B0))
    Xtst(Lognand(FF,TT))
    Xtst(Lognand(FF,FF))
    Xtst(Lognand(FF,I0))
    Xtst(Lognand(FF,P0))
    Xtst(Lognand(FF,M0))
    Xtst(Lognand(FF,B0))
    Xtst(Lognand(I0,TT))
    Xtst(Lognand(I0,FF))
    Xtst(Lognand(I0,P0))
    Xtst(Lognand(I0,M0))
    Xtst(Lognand(I0,B0))
    Xtst(Lognand(P0,TT))
    Xtst(Lognand(P0,FF))
    Xtst(Lognand(P0,I0))
    Xtst(Lognand(P0,P0))
    Xtst(Lognand(P0,M0))
    Xtst(Lognand(P0,B0))
    Xtst(Lognand(M0,TT))
    Xtst(Lognand(M0,FF))
    Xtst(Lognand(M0,I0))
    Xtst(Lognand(M0,P0))
    Xtst(Lognand(M0,M0))
    Xtst(Lognand(M0,B0))
    Xtst(Lognand(B0,TT))
    Xtst(Lognand(B0,FF))
    Xtst(Lognand(B0,I0))
    Xtst(Lognand(B0,P0))
    Xtst(Lognand(B0,M0))
    Xtst(Lognand(B0,B0))
    Etst(Lognand(I0011     ,I0101     ),lognot(I0001     ))
    Etst(Lognand(I1100     ,I1010     ),lognot(I1000     ))
    Etst(Lognand(I0011_1100,I0101_1010),lognot(I0001_1000))
    Etst(Lognand(I0011     ,I0101_1010),lognot(I0001     ))
    Etst(Lognand(I0101_1010,I0011     ),lognot(I0001     ))
    Etst(Lognand(I0011     ,J0101     ),lognot(I0001     ))
    Etst(Lognand(I1100     ,J1010     ),lognot(I1000     ))
    Etst(Lognand(I0011_1100,J0101_1010),lognot(I0001_1000))
    Etst(Lognand(I0011     ,J0101_1010),lognot(I0001     ))
    Etst(Lognand(I0101_1010,J0011     ),lognot(intcons(Z0001,I1010)))
    Etst(Lognand(J0011     ,I0101     ),lognot(I0001     ))
    Etst(Lognand(J1100     ,I1010     ),lognot(I1000     ))
    Etst(Lognand(J0011_1100,I0101_1010),lognot(I0001_1000))
    Etst(Lognand(J0011     ,I0101_1010),lognot(intcons(Z0001,I1010)))
    Etst(Lognand(J0101_1010,I0011     ),lognot(I0001     ))
    Etst(Lognand(J0011     ,J0101     ),lognot(J0001     ))
    Etst(Lognand(J1100     ,J1010     ),lognot(J1000     ))
    Etst(Lognand(J0011_1100,J0101_1010),lognot(J0001_1000))
    Etst(Lognand(J0011     ,J0101_1010),lognot(intcons(M0001,J1010)))
    Etst(Lognand(J0101_1010,J0011     ),lognot(intcons(M0001,J1010)))
    Etst(Lognand(EE,I0),J1)
    Etst(Lognand(EE,J1),J2)
    Etst(Lognand(I0,EE),J1)
    Etst(Lognand(J1,EE),J2)

    Xtst(Logorc1(TT,TT))
    Xtst(Logorc1(TT,FF))
    Xtst(Logorc1(TT,I0))
    Xtst(Logorc1(TT,P0))
    Xtst(Logorc1(TT,M0))
    Xtst(Logorc1(TT,B0))
    Xtst(Logorc1(FF,TT))
    Xtst(Logorc1(FF,FF))
    Xtst(Logorc1(FF,I0))
    Xtst(Logorc1(FF,P0))
    Xtst(Logorc1(FF,M0))
    Xtst(Logorc1(FF,B0))
    Xtst(Logorc1(I0,TT))
    Xtst(Logorc1(I0,FF))
    Xtst(Logorc1(I0,P0))
    Xtst(Logorc1(I0,M0))
    Xtst(Logorc1(I0,B0))
    Xtst(Logorc1(P0,TT))
    Xtst(Logorc1(P0,FF))
    Xtst(Logorc1(P0,I0))
    Xtst(Logorc1(P0,P0))
    Xtst(Logorc1(P0,M0))
    Xtst(Logorc1(P0,B0))
    Xtst(Logorc1(M0,TT))
    Xtst(Logorc1(M0,FF))
    Xtst(Logorc1(M0,I0))
    Xtst(Logorc1(M0,P0))
    Xtst(Logorc1(M0,M0))
    Xtst(Logorc1(M0,B0))
    Xtst(Logorc1(B0,TT))
    Xtst(Logorc1(B0,FF))
    Xtst(Logorc1(B0,I0))
    Xtst(Logorc1(B0,P0))
    Xtst(Logorc1(B0,M0))
    Xtst(Logorc1(B0,B0))
    Etst(Logorc1(lognot(I0011     ),I0101     ),I0111     )
    Etst(Logorc1(lognot(I1100     ),I1010     ),I1110     )
    Etst(Logorc1(lognot(I0011_1100),I0101_1010),I0111_1110)
    Etst(Logorc1(lognot(I0011     ),I0101_1010),intcons(Z0111,I1010))
    Etst(Logorc1(lognot(I0101_1010),I0011     ),intcons(Z0111,I1010))
    Etst(Logorc1(lognot(I0011     ),J0101     ),J0111     )
    Etst(Logorc1(lognot(I1100     ),J1010     ),J1110     )
    Etst(Logorc1(lognot(I0011_1100),J0101_1010),J0111_1110)
    Etst(Logorc1(lognot(I0011     ),J0101_1010),intcons(M0111,J1010))
    Etst(Logorc1(lognot(I0101_1010),J0011     ),J0111     )
    Etst(Logorc1(lognot(J0011     ),I0101     ),J0111     )
    Etst(Logorc1(lognot(J1100     ),I1010     ),J1110     )
    Etst(Logorc1(lognot(J0011_1100),I0101_1010),J0111_1110)
    Etst(Logorc1(lognot(J0011     ),I0101_1010),J0111     )
    Etst(Logorc1(lognot(J0101_1010),I0011     ),intcons(M0111,J1010))
    Etst(Logorc1(lognot(J0011     ),J0101     ),J0111     )
    Etst(Logorc1(lognot(J1100     ),J1010     ),J1110     )
    Etst(Logorc1(lognot(J0011_1100),J0101_1010),J0111_1110)
    Etst(Logorc1(lognot(J0011     ),J0101_1010),J0111     )
    Etst(Logorc1(lognot(J0101_1010),J0011     ),J0111     )
    Etst(Logorc1(EE,I0),J2)
    Etst(Logorc1(EE,J1),J1)
    Etst(Logorc1(I0,EE),J1)
    Etst(Logorc1(J1,EE),EE)

    Xtst(Logorc2(TT,TT))
    Xtst(Logorc2(TT,FF))
    Xtst(Logorc2(TT,I0))
    Xtst(Logorc2(TT,P0))
    Xtst(Logorc2(TT,M0))
    Xtst(Logorc2(TT,B0))
    Xtst(Logorc2(FF,TT))
    Xtst(Logorc2(FF,FF))
    Xtst(Logorc2(FF,I0))
    Xtst(Logorc2(FF,P0))
    Xtst(Logorc2(FF,M0))
    Xtst(Logorc2(FF,B0))
    Xtst(Logorc2(I0,TT))
    Xtst(Logorc2(I0,FF))
    Xtst(Logorc2(I0,P0))
    Xtst(Logorc2(I0,M0))
    Xtst(Logorc2(I0,B0))
    Xtst(Logorc2(P0,TT))
    Xtst(Logorc2(P0,FF))
    Xtst(Logorc2(P0,I0))
    Xtst(Logorc2(P0,P0))
    Xtst(Logorc2(P0,M0))
    Xtst(Logorc2(P0,B0))
    Xtst(Logorc2(M0,TT))
    Xtst(Logorc2(M0,FF))
    Xtst(Logorc2(M0,I0))
    Xtst(Logorc2(M0,P0))
    Xtst(Logorc2(M0,M0))
    Xtst(Logorc2(M0,B0))
    Xtst(Logorc2(B0,TT))
    Xtst(Logorc2(B0,FF))
    Xtst(Logorc2(B0,I0))
    Xtst(Logorc2(B0,P0))
    Xtst(Logorc2(B0,M0))
    Xtst(Logorc2(B0,B0))
    Etst(Logorc2(I0011     ,lognot(I0101     )),I0111     )
    Etst(Logorc2(I1100     ,lognot(I1010     )),I1110     )
    Etst(Logorc2(I0011_1100,lognot(I0101_1010)),I0111_1110)
    Etst(Logorc2(I0011     ,lognot(I0101_1010)),intcons(Z0111,I1010))
    Etst(Logorc2(I0101_1010,lognot(I0011     )),intcons(Z0111,I1010))
    Etst(Logorc2(I0011     ,lognot(J0101     )),J0111     )
    Etst(Logorc2(I1100     ,lognot(J1010     )),J1110     )
    Etst(Logorc2(I0011_1100,lognot(J0101_1010)),J0111_1110)
    Etst(Logorc2(I0011     ,lognot(J0101_1010)),intcons(M0111,J1010))
    Etst(Logorc2(I0101_1010,lognot(J0011     )),J0111     )
    Etst(Logorc2(J0011     ,lognot(I0101     )),J0111     )
    Etst(Logorc2(J1100     ,lognot(I1010     )),J1110     )
    Etst(Logorc2(J0011_1100,lognot(I0101_1010)),J0111_1110)
    Etst(Logorc2(J0011     ,lognot(I0101_1010)),J0111     )
    Etst(Logorc2(J0101_1010,lognot(I0011     )),intcons(M0111,J1010))
    Etst(Logorc2(J0011     ,lognot(J0101     )),J0111     )
    Etst(Logorc2(J1100     ,lognot(J1010     )),J1110     )
    Etst(Logorc2(J0011_1100,lognot(J0101_1010)),J0111_1110)
    Etst(Logorc2(J0011     ,lognot(J0101_1010)),J0111     )
    Etst(Logorc2(J0101_1010,lognot(J0011     )),J0111     )
    Etst(Logorc2(EE,I0),J1)
    Etst(Logorc2(EE,J1),EE)
    Etst(Logorc2(I0,EE),J2)
    Etst(Logorc2(J1,EE),J1)

    Xtst(Lognor(TT,TT))
    Xtst(Lognor(TT,FF))
    Xtst(Lognor(TT,I0))
    Xtst(Lognor(TT,P0))
    Xtst(Lognor(TT,M0))
    Xtst(Lognor(TT,B0))
    Xtst(Lognor(FF,TT))
    Xtst(Lognor(FF,FF))
    Xtst(Lognor(FF,I0))
    Xtst(Lognor(FF,P0))
    Xtst(Lognor(FF,M0))
    Xtst(Lognor(FF,B0))
    Xtst(Lognor(I0,TT))
    Xtst(Lognor(I0,FF))
    Xtst(Lognor(I0,P0))
    Xtst(Lognor(I0,M0))
    Xtst(Lognor(I0,B0))
    Xtst(Lognor(P0,TT))
    Xtst(Lognor(P0,FF))
    Xtst(Lognor(P0,I0))
    Xtst(Lognor(P0,P0))
    Xtst(Lognor(P0,M0))
    Xtst(Lognor(P0,B0))
    Xtst(Lognor(M0,TT))
    Xtst(Lognor(M0,FF))
    Xtst(Lognor(M0,I0))
    Xtst(Lognor(M0,P0))
    Xtst(Lognor(M0,M0))
    Xtst(Lognor(M0,B0))
    Xtst(Lognor(B0,TT))
    Xtst(Lognor(B0,FF))
    Xtst(Lognor(B0,I0))
    Xtst(Lognor(B0,P0))
    Xtst(Lognor(B0,M0))
    Xtst(Lognor(B0,B0))
    Etst(Lognor(I0011     ,I0101     ),lognot(I0111     ))
    Etst(Lognor(I1100     ,I1010     ),lognot(I1110     ))
    Etst(Lognor(I0011_1100,I0101_1010),lognot(I0111_1110))
    Etst(Lognor(I0011     ,I0101_1010),lognot(intcons(Z0111,I1010)))
    Etst(Lognor(I0101_1010,I0011     ),lognot(intcons(Z0111,I1010)))
    Etst(Lognor(I0011     ,J0101     ),lognot(J0111     ))
    Etst(Lognor(I1100     ,J1010     ),lognot(J1110     ))
    Etst(Lognor(I0011_1100,J0101_1010),lognot(J0111_1110))
    Etst(Lognor(I0011     ,J0101_1010),lognot(intcons(M0111,J1010)))
    Etst(Lognor(I0101_1010,J0011     ),lognot(J0111     ))
    Etst(Lognor(J0011     ,I0101     ),lognot(J0111     ))
    Etst(Lognor(J1100     ,I1010     ),lognot(J1110     ))
    Etst(Lognor(J0011_1100,I0101_1010),lognot(J0111_1110))
    Etst(Lognor(J0011     ,I0101_1010),lognot(J0111     ))
    Etst(Lognor(J0101_1010,I0011     ),lognot(intcons(M0111,J1010)))
    Etst(Lognor(J0011     ,J0101     ),lognot(J0111     ))
    Etst(Lognor(J1100     ,J1010     ),lognot(J1110     ))
    Etst(Lognor(J0011_1100,J0101_1010),lognot(J0111_1110))
    Etst(Lognor(J0011     ,J0101_1010),lognot(J0111     ))
    Etst(Lognor(J0101_1010,J0011     ),lognot(J0111     ))
    Etst(Lognor(EE,I0),J2)
    Etst(Lognor(EE,J1),I0)
    Etst(Lognor(I0,EE),J2)
    Etst(Lognor(J1,EE),I0)

    Xtst(Logeqv(TT,TT))
    Xtst(Logeqv(TT,FF))
    Xtst(Logeqv(TT,I0))
    Xtst(Logeqv(TT,P0))
    Xtst(Logeqv(TT,M0))
    Xtst(Logeqv(TT,B0))
    Xtst(Logeqv(FF,TT))
    Xtst(Logeqv(FF,FF))
    Xtst(Logeqv(FF,I0))
    Xtst(Logeqv(FF,P0))
    Xtst(Logeqv(FF,M0))
    Xtst(Logeqv(FF,B0))
    Xtst(Logeqv(I0,TT))
    Xtst(Logeqv(I0,FF))
    Xtst(Logeqv(I0,P0))
    Xtst(Logeqv(I0,M0))
    Xtst(Logeqv(I0,B0))
    Xtst(Logeqv(P0,TT))
    Xtst(Logeqv(P0,FF))
    Xtst(Logeqv(P0,I0))
    Xtst(Logeqv(P0,P0))
    Xtst(Logeqv(P0,M0))
    Xtst(Logeqv(P0,B0))
    Xtst(Logeqv(M0,TT))
    Xtst(Logeqv(M0,FF))
    Xtst(Logeqv(M0,I0))
    Xtst(Logeqv(M0,P0))
    Xtst(Logeqv(M0,M0))
    Xtst(Logeqv(M0,B0))
    Xtst(Logeqv(B0,TT))
    Xtst(Logeqv(B0,FF))
    Xtst(Logeqv(B0,I0))
    Xtst(Logeqv(B0,P0))
    Xtst(Logeqv(B0,M0))
    Xtst(Logeqv(B0,B0))
    Etst(Logeqv(lognot(I0011     ),I0101     ),I0110     )
    Etst(Logeqv(lognot(I1100     ),I1010     ),I0110     )
    Etst(Logeqv(lognot(I0011_1100),I0101_1010),I0110_0110)
    Etst(Logeqv(lognot(I0011     ),I0101_1010),intcons(Z0110,I1010))
    Etst(Logeqv(lognot(I0101_1010),I0011     ),intcons(Z0110,I1010))
    Etst(Logeqv(lognot(I0011     ),J0101     ),J0110     )
    Etst(Logeqv(lognot(I1100     ),J1010     ),J0110     )
    Etst(Logeqv(lognot(I0011_1100),J0101_1010),J0110_0110)
    Etst(Logeqv(lognot(I0011     ),J0101_1010),intcons(M0110,J1010))
    Etst(Logeqv(lognot(I0101_1010),J0011     ),intcons(M0110,J0101))
    Etst(Logeqv(lognot(J0011     ),I0101     ),J0110     )
    Etst(Logeqv(lognot(J1100     ),I1010     ),J0110     )
    Etst(Logeqv(lognot(J0011_1100),I0101_1010),J0110_0110)
    Etst(Logeqv(lognot(J0011     ),I0101_1010),intcons(M0110,J0101))
    Etst(Logeqv(lognot(J0101_1010),I0011     ),intcons(M0110,J1010))
    Etst(Logeqv(lognot(J0011     ),J0101     ),I0110     )
    Etst(Logeqv(lognot(J1100     ),J1010     ),I0110     )
    Etst(Logeqv(lognot(J0011_1100),J0101_1010),I0110_0110)
    Etst(Logeqv(lognot(J0011     ),J0101_1010),intcons(Z0110,I0101))
    Etst(Logeqv(lognot(J0101_1010),J0011     ),intcons(Z0110,I0101))
    Etst(Logeqv(EE,I0),J2)
    Etst(Logeqv(EE,J1),EE)
    Etst(Logeqv(I0,EE),J2)
    Etst(Logeqv(J1,EE),EE)

    Xtst(Logtest(TT,TT))
    Xtst(Logtest(TT,FF))
    Xtst(Logtest(TT,I0))
    Xtst(Logtest(TT,P0))
    Xtst(Logtest(TT,M0))
    Xtst(Logtest(TT,B0))
    Xtst(Logtest(FF,TT))
    Xtst(Logtest(FF,FF))
    Xtst(Logtest(FF,I0))
    Xtst(Logtest(FF,P0))
    Xtst(Logtest(FF,M0))
    Xtst(Logtest(FF,B0))
    Xtst(Logtest(I0,TT))
    Xtst(Logtest(I0,FF))
    Xtst(Logtest(I0,P0))
    Xtst(Logtest(I0,M0))
    Xtst(Logtest(I0,B0))
    Xtst(Logtest(P0,TT))
    Xtst(Logtest(P0,FF))
    Xtst(Logtest(P0,I0))
    Xtst(Logtest(P0,P0))
    Xtst(Logtest(P0,M0))
    Xtst(Logtest(P0,B0))
    Xtst(Logtest(M0,TT))
    Xtst(Logtest(M0,FF))
    Xtst(Logtest(M0,I0))
    Xtst(Logtest(M0,P0))
    Xtst(Logtest(M0,M0))
    Xtst(Logtest(M0,B0))
    Xtst(Logtest(B0,TT))
    Xtst(Logtest(B0,FF))
    Xtst(Logtest(B0,I0))
    Xtst(Logtest(B0,P0))
    Xtst(Logtest(B0,M0))
    Xtst(Logtest(B0,B0))
    Ttst(Logtest(I0011     ,I0101     ))
    Ttst(Logtest(I1100     ,I1010     ))
    Ttst(Logtest(I0011_1100,I0101_1010))
    Ttst(Logtest(I0011     ,I0101_1010))
    Ttst(Logtest(I0101_1010,I0011     ))
    Ttst(Logtest(I0011     ,J0101     ))
    Ttst(Logtest(I1100     ,J1010     ))
    Ttst(Logtest(I0011_1100,J0101_1010))
    Ttst(Logtest(I0011     ,J0101_1010))
    Ttst(Logtest(I0101_1010,J0011     ))
    Ttst(Logtest(J0011     ,I0101     ))
    Ttst(Logtest(J1100     ,I1010     ))
    Ttst(Logtest(J0011_1100,I0101_1010))
    Ttst(Logtest(J0011     ,I0101_1010))
    Ttst(Logtest(J0101_1010,I0011     ))
    Ttst(Logtest(J0011     ,J0101     ))
    Ttst(Logtest(J1100     ,J1010     ))
    Ttst(Logtest(J0011_1100,J0101_1010))
    Ttst(Logtest(J0011     ,J0101_1010))
    Ttst(Logtest(J0101_1010,J0011     ))

    Ftst(Logtest(I0011     ,I1100     ))
    Ftst(Logtest(I1100     ,I0001     ))
    Ftst(Logtest(I0011_1100,I1000_0001))
    Ftst(Logtest(I0011     ,I1100_0011))
    Ftst(Logtest(I0101_1010,I0010     ))
    Ftst(Logtest(I0011     ,J1000     ))
    Ftst(Logtest(I1100     ,J0011     ))
    Ftst(Logtest(I0011_1100,J1000_0001))
    Ftst(Logtest(I0011     ,J1100_0011))
    Ftst(Logtest(J0011     ,I0100     ))
    Ftst(Logtest(J1100     ,I0010     ))
    Ftst(Logtest(J0011_1100,I0100_0010))
    Ftst(Logtest(J0101_1010,I1010     ))
    Ftst(Logtest(EE,I0))
    Ttst(Logtest(EE,J1))
    Ftst(Logtest(I0,EE))
    Ttst(Logtest(J1,EE))
    Ftst(Logtest(EE,J1010_0101))
    Ttst(Logtest(EE,I0011_1100))
    Ftst(Logtest(J1010_0101,EE))
    Ttst(Logtest(I0011_1100,EE))

    Xtst(Lt(TT,TT))
    Xtst(Lt(TT,FF))
    Xtst(Lt(TT,I0))
    Xtst(Lt(TT,P0))
    Xtst(Lt(TT,M0))
    Xtst(Lt(TT,B0))
    Xtst(Lt(FF,TT))
    Xtst(Lt(FF,FF))
    Xtst(Lt(FF,I0))
    Xtst(Lt(FF,P0))
    Xtst(Lt(FF,M0))
    Xtst(Lt(FF,B0))
    Xtst(Lt(I0,TT))
    Xtst(Lt(I0,FF))
    Xtst(Lt(I0,P0))
    Xtst(Lt(I0,M0))
    Xtst(Lt(I0,B0))
    Xtst(Lt(P0,TT))
    Xtst(Lt(P0,FF))
    Xtst(Lt(P0,I0))
    Xtst(Lt(P0,P0))
    Xtst(Lt(P0,M0))
    Xtst(Lt(P0,B0))
    Xtst(Lt(M0,TT))
    Xtst(Lt(M0,FF))
    Xtst(Lt(M0,I0))
    Xtst(Lt(M0,P0))
    Xtst(Lt(M0,M0))
    Xtst(Lt(M0,B0))
    Xtst(Lt(B0,TT))
    Xtst(Lt(B0,FF))
    Xtst(Lt(B0,I0))
    Xtst(Lt(B0,P0))
    Xtst(Lt(B0,M0))
    Xtst(Lt(B0,B0))
    Ftst(Lt(I0011     ,I0011     ))
    Ttst(Lt(I0011     ,I0101     ))
    Ftst(Lt(I1100     ,I1010     ))
    Ftst(Lt(I0011_1100,I0011_1100))
    Ftst(Lt(I0011_1100,I0101_1010))
    Ttst(Lt(I1100_0011,I1010_0101))
    Ttst(Lt(I0011     ,I0101_1010))
    Ftst(Lt(I0101_1010,I0011     ))
    Ftst(Lt(I0011     ,J0101     ))
    Ftst(Lt(I1100     ,J1010     ))
    Ftst(Lt(I0011_1100,J0101_1010))
    Ftst(Lt(I0011     ,J0101_1010))
    Ftst(Lt(I0101_1010,J0011     ))
    Ttst(Lt(J0011     ,I0101     ))
    Ttst(Lt(J1100     ,I1010     ))
    Ttst(Lt(J0011_1100,I0101_1010))
    Ttst(Lt(J0011     ,I0101_1010))
    Ttst(Lt(J0101_1010,I0011     ))
    Ftst(Lt(J0011     ,J0011     ))
    Ttst(Lt(J0011     ,J0101     ))
    Ftst(Lt(J1100     ,J1010     ))
    Ftst(Lt(J0011_1100,J0011_1100))
    Ftst(Lt(J0011_1100,J0101_1010))
    Ttst(Lt(J1100_0011,J1010_0101))
    Ftst(Lt(J0011     ,J0101_1010))
    Ttst(Lt(J0101_1010,J0011     ))

    Xtst(Le(TT,TT))
    Xtst(Le(TT,FF))
    Xtst(Le(TT,I0))
    Xtst(Le(TT,P0))
    Xtst(Le(TT,M0))
    Xtst(Le(TT,B0))
    Xtst(Le(FF,TT))
    Xtst(Le(FF,FF))
    Xtst(Le(FF,I0))
    Xtst(Le(FF,P0))
    Xtst(Le(FF,M0))
    Xtst(Le(FF,B0))
    Xtst(Le(I0,TT))
    Xtst(Le(I0,FF))
    Xtst(Le(I0,P0))
    Xtst(Le(I0,M0))
    Xtst(Le(I0,B0))
    Xtst(Le(P0,TT))
    Xtst(Le(P0,FF))
    Xtst(Le(P0,I0))
    Xtst(Le(P0,P0))
    Xtst(Le(P0,M0))
    Xtst(Le(P0,B0))
    Xtst(Le(M0,TT))
    Xtst(Le(M0,FF))
    Xtst(Le(M0,I0))
    Xtst(Le(M0,P0))
    Xtst(Le(M0,M0))
    Xtst(Le(M0,B0))
    Xtst(Le(B0,TT))
    Xtst(Le(B0,FF))
    Xtst(Le(B0,I0))
    Xtst(Le(B0,P0))
    Xtst(Le(B0,M0))
    Xtst(Le(B0,B0))
    Ttst(Le(I0011     ,I0011     ))
    Ttst(Le(I0011     ,I0101     ))
    Ftst(Le(I1100     ,I1010     ))
    Ttst(Le(I0011_1100,I0011_1100))
    Ftst(Le(I0011_1100,I0101_1010))
    Ttst(Le(I1100_0011,I1010_0101))
    Ttst(Le(I0011     ,I0101_1010))
    Ftst(Le(I0101_1010,I0011     ))
    Ftst(Le(I0011     ,J0101     ))
    Ftst(Le(I1100     ,J1010     ))
    Ftst(Le(I0011_1100,J0101_1010))
    Ftst(Le(I0011     ,J0101_1010))
    Ftst(Le(I0101_1010,J0011     ))
    Ttst(Le(J0011     ,I0101     ))
    Ttst(Le(J1100     ,I1010     ))
    Ttst(Le(J0011_1100,I0101_1010))
    Ttst(Le(J0011     ,I0101_1010))
    Ttst(Le(J0101_1010,I0011     ))
    Ttst(Le(J0011     ,J0011     ))
    Ttst(Le(J0011     ,J0101     ))
    Ftst(Le(J1100     ,J1010     ))
    Ttst(Le(J0011_1100,J0011_1100))
    Ftst(Le(J0011_1100,J0101_1010))
    Ttst(Le(J1100_0011,J1010_0101))
    Ftst(Le(J0011     ,J0101_1010))
    Ttst(Le(J0101_1010,J0011     ))

    Xtst(Gt(TT,TT))
    Xtst(Gt(TT,FF))
    Xtst(Gt(TT,I0))
    Xtst(Gt(TT,P0))
    Xtst(Gt(TT,M0))
    Xtst(Gt(TT,B0))
    Xtst(Gt(FF,TT))
    Xtst(Gt(FF,FF))
    Xtst(Gt(FF,I0))
    Xtst(Gt(FF,P0))
    Xtst(Gt(FF,M0))
    Xtst(Gt(FF,B0))
    Xtst(Gt(I0,TT))
    Xtst(Gt(I0,FF))
    Xtst(Gt(I0,P0))
    Xtst(Gt(I0,M0))
    Xtst(Gt(I0,B0))
    Xtst(Gt(P0,TT))
    Xtst(Gt(P0,FF))
    Xtst(Gt(P0,I0))
    Xtst(Gt(P0,P0))
    Xtst(Gt(P0,M0))
    Xtst(Gt(P0,B0))
    Xtst(Gt(M0,TT))
    Xtst(Gt(M0,FF))
    Xtst(Gt(M0,I0))
    Xtst(Gt(M0,P0))
    Xtst(Gt(M0,M0))
    Xtst(Gt(M0,B0))
    Xtst(Gt(B0,TT))
    Xtst(Gt(B0,FF))
    Xtst(Gt(B0,I0))
    Xtst(Gt(B0,P0))
    Xtst(Gt(B0,M0))
    Xtst(Gt(B0,B0))
    Ftst(Gt(I0011     ,I0011     ))
    Ftst(Gt(I0011     ,I0101     ))
    Ttst(Gt(I1100     ,I1010     ))
    Ftst(Gt(I0011_1100,I0011_1100))
    Ttst(Gt(I0011_1100,I0101_1010))
    Ftst(Gt(I1100_0011,I1010_0101))
    Ftst(Gt(I0011     ,I0101_1010))
    Ttst(Gt(I0101_1010,I0011     ))
    Ttst(Gt(I0011     ,J0101     ))
    Ttst(Gt(I1100     ,J1010     ))
    Ttst(Gt(I0011_1100,J0101_1010))
    Ttst(Gt(I0011     ,J0101_1010))
    Ttst(Gt(I0101_1010,J0011     ))
    Ftst(Gt(J0011     ,I0101     ))
    Ftst(Gt(J1100     ,I1010     ))
    Ftst(Gt(J0011_1100,I0101_1010))
    Ftst(Gt(J0011     ,I0101_1010))
    Ftst(Gt(J0101_1010,I0011     ))
    Ftst(Gt(J0011     ,J0011     ))
    Ftst(Gt(J0011     ,J0101     ))
    Ttst(Gt(J1100     ,J1010     ))
    Ftst(Gt(J0011_1100,J0011_1100))
    Ttst(Gt(J0011_1100,J0101_1010))
    Ftst(Gt(J1100_0011,J1010_0101))
    Ttst(Gt(J0011     ,J0101_1010))
    Ftst(Gt(J0101_1010,J0011     ))

    Xtst(Ge(TT,TT))
    Xtst(Ge(TT,FF))
    Xtst(Ge(TT,I0))
    Xtst(Ge(TT,P0))
    Xtst(Ge(TT,M0))
    Xtst(Ge(TT,B0))
    Xtst(Ge(FF,TT))
    Xtst(Ge(FF,FF))
    Xtst(Ge(FF,I0))
    Xtst(Ge(FF,P0))
    Xtst(Ge(FF,M0))
    Xtst(Ge(FF,B0))
    Xtst(Ge(I0,TT))
    Xtst(Ge(I0,FF))
    Xtst(Ge(I0,P0))
    Xtst(Ge(I0,M0))
    Xtst(Ge(I0,B0))
    Xtst(Ge(P0,TT))
    Xtst(Ge(P0,FF))
    Xtst(Ge(P0,I0))
    Xtst(Ge(P0,P0))
    Xtst(Ge(P0,M0))
    Xtst(Ge(P0,B0))
    Xtst(Ge(M0,TT))
    Xtst(Ge(M0,FF))
    Xtst(Ge(M0,I0))
    Xtst(Ge(M0,P0))
    Xtst(Ge(M0,M0))
    Xtst(Ge(M0,B0))
    Xtst(Ge(B0,TT))
    Xtst(Ge(B0,FF))
    Xtst(Ge(B0,I0))
    Xtst(Ge(B0,P0))
    Xtst(Ge(B0,M0))
    Xtst(Ge(B0,B0))
    Ttst(Ge(I0011     ,I0011     ))
    Ftst(Ge(I0011     ,I0101     ))
    Ttst(Ge(I1100     ,I1010     ))
    Ttst(Ge(I0011_1100,I0011_1100))
    Ttst(Ge(I0011_1100,I0101_1010))
    Ftst(Ge(I1100_0011,I1010_0101))
    Ftst(Ge(I0011     ,I0101_1010))
    Ttst(Ge(I0101_1010,I0011     ))
    Ttst(Ge(I0011     ,J0101     ))
    Ttst(Ge(I1100     ,J1010     ))
    Ttst(Ge(I0011_1100,J0101_1010))
    Ttst(Ge(I0011     ,J0101_1010))
    Ttst(Ge(I0101_1010,J0011     ))
    Ftst(Ge(J0011     ,I0101     ))
    Ftst(Ge(J1100     ,I1010     ))
    Ftst(Ge(J0011_1100,I0101_1010))
    Ftst(Ge(J0011     ,I0101_1010))
    Ftst(Ge(J0101_1010,I0011     ))
    Ttst(Ge(J0011     ,J0011     ))
    Ftst(Ge(J0011     ,J0101     ))
    Ttst(Ge(J1100     ,J1010     ))
    Ttst(Ge(J0011_1100,J0011_1100))
    Ttst(Ge(J0011_1100,J0101_1010))
    Ftst(Ge(J1100_0011,J1010_0101))
    Ttst(Ge(J0011     ,J0101_1010))
    Ftst(Ge(J0101_1010,J0011     ))

    Xtst(Evenp(TT))
    Xtst(Evenp(FF))
    Xtst(Evenp(P0))
    Xtst(Evenp(M0))
    Xtst(Evenp(B0))
    Ttst(Evenp(Z))
    Ftst(Evenp(M))
    Ftst(Evenp(I0011))
    Ttst(Evenp(J1010_0101))

    Xtst(Half(TT))
    Xtst(Half(FF))
    Xtst(Half(P0))
    Xtst(Half(M0))
    Xtst(Half(B0))
    Etst(Half(Z),Z)
    Etst(Half(M),M)
    Etst(Half(I[1]),I[0])
    Etst(Half(I[2]),I[1])
    Etst(Half(I[3]),I[1])
    Etst(Half(I[4]),I[2])
    Etst(Half(I[5]),I[2])
    Etst(Half(I[6]),I[3])
    Etst(Half(JN(-2)),M)
    Etst(Half(JN(-3)),JN(-2))
    Etst(Half(JN(-4)),JN(-2))
    Etst(Half(JN(-5)),JN(-3))
    Etst(Half(JN(-6)),JN(-3))
    Etst(Half(JN(-7)),JN(-4))

    Etst(Base,I[10])

    Etst(Digitend,I[0])

    Xtst(Plus(TT,TT))
    Xtst(Plus(TT,FF))
    Xtst(Plus(TT,I0))
    Xtst(Plus(TT,P0))
    Xtst(Plus(TT,M0))
    Xtst(Plus(TT,B0))
    Xtst(Plus(FF,TT))
    Xtst(Plus(FF,FF))
    Xtst(Plus(FF,I0))
    Xtst(Plus(FF,P0))
    Xtst(Plus(FF,M0))
    Xtst(Plus(FF,B0))
    Xtst(Plus(I0,TT))
    Xtst(Plus(I0,FF))
    Xtst(Plus(I0,P0))
    Xtst(Plus(I0,M0))
    Xtst(Plus(I0,B0))
    Xtst(Plus(P0,TT))
    Xtst(Plus(P0,FF))
    Xtst(Plus(P0,I0))
    Xtst(Plus(P0,P0))
    Xtst(Plus(P0,M0))
    Xtst(Plus(P0,B0))
    Xtst(Plus(M0,TT))
    Xtst(Plus(M0,FF))
    Xtst(Plus(M0,I0))
    Xtst(Plus(M0,P0))
    Xtst(Plus(M0,M0))
    Xtst(Plus(M0,B0))
    Xtst(Plus(B0,TT))
    Xtst(Plus(B0,FF))
    Xtst(Plus(B0,I0))
    Xtst(Plus(B0,P0))
    Xtst(Plus(B0,M0))
    Xtst(Plus(B0,B0))
    Etst(Plus(JN(-3),JN(-3)),JN(-6))
    Etst(Plus(JN(-3),JN(-2)),JN(-5))
    Etst(Plus(JN(-3),JN(-1)),JN(-4))
    Etst(Plus(JN(-3),JN( 0)),JN(-3))
    Etst(Plus(JN(-3),JN( 1)),JN(-2))
    Etst(Plus(JN(-3),JN( 2)),JN(-1))
    Etst(Plus(JN(-2),JN(-3)),JN(-5))
    Etst(Plus(JN(-2),JN(-2)),JN(-4))
    Etst(Plus(JN(-2),JN(-1)),JN(-3))
    Etst(Plus(JN(-2),JN( 0)),JN(-2))
    Etst(Plus(JN(-2),JN( 1)),JN(-1))
    Etst(Plus(JN(-2),JN( 2)),JN( 0))
    Etst(Plus(JN(-1),JN(-3)),JN(-4))
    Etst(Plus(JN(-1),JN(-2)),JN(-3))
    Etst(Plus(JN(-1),JN(-1)),JN(-2))
    Etst(Plus(JN(-1),JN( 0)),JN(-1))
    Etst(Plus(JN(-1),JN( 1)),JN( 0))
    Etst(Plus(JN(-1),JN( 2)),JN( 1))
    Etst(Plus(JN( 0),JN(-3)),JN(-3))
    Etst(Plus(JN( 0),JN(-2)),JN(-2))
    Etst(Plus(JN( 0),JN(-1)),JN(-1))
    Etst(Plus(JN( 0),JN( 0)),JN( 0))
    Etst(Plus(JN( 0),JN( 1)),JN( 1))
    Etst(Plus(JN( 0),JN( 2)),JN( 2))
    Etst(Plus(JN( 1),JN(-3)),JN(-2))
    Etst(Plus(JN( 1),JN(-2)),JN(-1))
    Etst(Plus(JN( 1),JN(-1)),JN( 0))
    Etst(Plus(JN( 1),JN( 0)),JN( 1))
    Etst(Plus(JN( 1),JN( 1)),JN( 2))
    Etst(Plus(JN( 1),JN( 2)),JN( 3))
    Etst(Plus(JN( 2),JN(-3)),JN(-1))
    Etst(Plus(JN( 2),JN(-2)),JN( 0))
    Etst(Plus(JN( 2),JN(-1)),JN( 1))
    Etst(Plus(JN( 2),JN( 0)),JN( 2))
    Etst(Plus(JN( 2),JN( 1)),JN( 3))
    Etst(Plus(JN( 2),JN( 2)),JN( 4))
    Etst(Plus(K2( 0, 0),K2( 0, 0)),K2( 0, 0))
    Etst(Plus(K2( 0, 0),K2( 0,-1)),K2( 0,-1))
    Etst(Plus(K2( 0, 0),K2(-1, 0)),K2(-1, 0))
    Etst(Plus(K2( 0, 0),K2(-1,-1)),K2(-1,-1))
    Etst(Plus(K2( 0,-1),K2( 0, 0)),K2( 0,-1))
    Etst(Plus(K2( 0,-1),K2( 0,-1)),K2( 0,-2))
    Etst(Plus(K2( 0,-1),K2(-1, 0)),K2(-1,-1))
    Etst(Plus(K2( 0,-1),K2(-1,-1)),K2(-1,-2))
    Etst(Plus(K2(-1, 0),K2( 0, 0)),K2(-1, 0))
    Etst(Plus(K2(-1, 0),K2( 0,-1)),K2(-1,-1))
    Etst(Plus(K2(-1, 0),K2(-1, 0)),K2(-2, 1))
    Etst(Plus(K2(-1, 0),K2(-1,-1)),K2(-2, 0))
    Etst(Plus(K2(-1,-1),K2( 0, 0)),K2(-1,-1))
    Etst(Plus(K2(-1,-1),K2( 0,-1)),K2(-1,-2))
    Etst(Plus(K2(-1,-1),K2(-1, 0)),K2(-2, 0))
    Etst(Plus(K2(-1,-1),K2(-1,-1)),K2(-2,-1))

    Etst(Plus(N(-1000000000000),N(-1000000000000)),N(-2000000000000))
    Etst(Plus(N(-1000000000000),N(   -1000000000)),N(-1001000000000))
    Etst(Plus(N(-1000000000000),N(      -1000000)),N(-1000001000000))
    Etst(Plus(N(-1000000000000),N(         -1000)),N(-1000000001000))
    Etst(Plus(N(-1000000000000),N(          -100)),N(-1000000000100))
    Etst(Plus(N(-1000000000000),N(           -10)),N(-1000000000010))
    Etst(Plus(N(-1000000000000),N(            -1)),N(-1000000000001))
    Etst(Plus(N(-1000000000000),N(             0)),N(-1000000000000))
    Etst(Plus(N(-1000000000000),N(             1)),N(-0999999999999))
    Etst(Plus(N(-1000000000000),N(            10)),N(-0999999999990))
    Etst(Plus(N(-1000000000000),N(           100)),N(-0999999999900))
    Etst(Plus(N(-1000000000000),N(          1000)),N(-0999999999000))
    Etst(Plus(N(-1000000000000),N(       1000000)),N(-0999999000000))
    Etst(Plus(N(-1000000000000),N(    1000000000)),N(-0999000000000))
    Etst(Plus(N(-1000000000000),N( 1000000000000)),N(-0000000000000))

    Etst(Plus(N(   -1000000000),N(-1000000000000)),N(-1001000000000))
    Etst(Plus(N(   -1000000000),N(   -1000000000)),N(-0002000000000))
    Etst(Plus(N(   -1000000000),N(      -1000000)),N(-0001001000000))
    Etst(Plus(N(   -1000000000),N(         -1000)),N(-0001000001000))
    Etst(Plus(N(   -1000000000),N(          -100)),N(-0001000000100))
    Etst(Plus(N(   -1000000000),N(           -10)),N(-0001000000010))
    Etst(Plus(N(   -1000000000),N(            -1)),N(-0001000000001))
    Etst(Plus(N(   -1000000000),N(             0)),N(-0001000000000))
    Etst(Plus(N(   -1000000000),N(             1)),N(-0000999999999))
    Etst(Plus(N(   -1000000000),N(            10)),N(-0000999999990))
    Etst(Plus(N(   -1000000000),N(           100)),N(-0000999999900))
    Etst(Plus(N(   -1000000000),N(          1000)),N(-0000999999000))
    Etst(Plus(N(   -1000000000),N(       1000000)),N(-0000999000000))
    Etst(Plus(N(   -1000000000),N(    1000000000)),N(-0000000000000))
    Etst(Plus(N(   -1000000000),N( 1000000000000)),N( 0999000000000))

    Etst(Plus(N(      -1000000),N(-1000000000000)),N(-1000001000000))
    Etst(Plus(N(      -1000000),N(   -1000000000)),N(-0001001000000))
    Etst(Plus(N(      -1000000),N(      -1000000)),N(-0000002000000))
    Etst(Plus(N(      -1000000),N(         -1000)),N(-0000001001000))
    Etst(Plus(N(      -1000000),N(          -100)),N(-0000001000100))
    Etst(Plus(N(      -1000000),N(           -10)),N(-0000001000010))
    Etst(Plus(N(      -1000000),N(            -1)),N(-0000001000001))
    Etst(Plus(N(      -1000000),N(             0)),N(-0000001000000))
    Etst(Plus(N(      -1000000),N(             1)),N(-0000000999999))
    Etst(Plus(N(      -1000000),N(            10)),N(-0000000999990))
    Etst(Plus(N(      -1000000),N(           100)),N(-0000000999900))
    Etst(Plus(N(      -1000000),N(          1000)),N(-0000000999000))
    Etst(Plus(N(      -1000000),N(       1000000)),N(-0000000000000))
    Etst(Plus(N(      -1000000),N(    1000000000)),N( 0000999000000))
    Etst(Plus(N(      -1000000),N( 1000000000000)),N( 0999999000000))

    Etst(Plus(N(         -1000),N(-1000000000000)),N(-1000000001000))
    Etst(Plus(N(         -1000),N(   -1000000000)),N(-0001000001000))
    Etst(Plus(N(         -1000),N(      -1000000)),N(-0000001001000))
    Etst(Plus(N(         -1000),N(         -1000)),N(-0000000002000))
    Etst(Plus(N(         -1000),N(          -100)),N(-0000000001100))
    Etst(Plus(N(         -1000),N(           -10)),N(-0000000001010))
    Etst(Plus(N(         -1000),N(            -1)),N(-0000000001001))
    Etst(Plus(N(         -1000),N(             0)),N(-0000000001000))
    Etst(Plus(N(         -1000),N(             1)),N(-0000000000999))
    Etst(Plus(N(         -1000),N(            10)),N(-0000000000990))
    Etst(Plus(N(         -1000),N(           100)),N(-0000000000900))
    Etst(Plus(N(         -1000),N(          1000)),N( 0000000000000))
    Etst(Plus(N(         -1000),N(       1000000)),N( 0000000999000))
    Etst(Plus(N(         -1000),N(    1000000000)),N( 0000999999000))
    Etst(Plus(N(         -1000),N( 1000000000000)),N( 0999999999000))

    Etst(Plus(N(          -100),N(-1000000000000)),N(-1000000000100))
    Etst(Plus(N(          -100),N(   -1000000000)),N(-0001000000100))
    Etst(Plus(N(          -100),N(      -1000000)),N(-0000001000100))
    Etst(Plus(N(          -100),N(         -1000)),N(-0000000001100))
    Etst(Plus(N(          -100),N(          -100)),N(-0000000000200))
    Etst(Plus(N(          -100),N(           -10)),N(-0000000000110))
    Etst(Plus(N(          -100),N(            -1)),N(-0000000000101))
    Etst(Plus(N(          -100),N(             0)),N(-0000000000100))
    Etst(Plus(N(          -100),N(             1)),N(-0000000000099))
    Etst(Plus(N(          -100),N(            10)),N(-0000000000090))
    Etst(Plus(N(          -100),N(           100)),N( 0000000000000))
    Etst(Plus(N(          -100),N(          1000)),N( 0000000000900))
    Etst(Plus(N(          -100),N(       1000000)),N( 0000000999900))
    Etst(Plus(N(          -100),N(    1000000000)),N( 0000999999900))
    Etst(Plus(N(          -100),N( 1000000000000)),N( 0999999999900))

    Etst(Plus(N(           -10),N(-1000000000000)),N(-1000000000010))
    Etst(Plus(N(           -10),N(   -1000000000)),N(-0001000000010))
    Etst(Plus(N(           -10),N(      -1000000)),N(-0000001000010))
    Etst(Plus(N(           -10),N(         -1000)),N(-0000000001010))
    Etst(Plus(N(           -10),N(          -100)),N(-0000000000110))
    Etst(Plus(N(           -10),N(           -10)),N(-0000000000020))
    Etst(Plus(N(           -10),N(            -1)),N(-0000000000011))
    Etst(Plus(N(           -10),N(             0)),N(-0000000000010))
    Etst(Plus(N(           -10),N(             1)),N(-0000000000009))
    Etst(Plus(N(           -10),N(            10)),N( 0000000000000))
    Etst(Plus(N(           -10),N(           100)),N( 0000000000090))
    Etst(Plus(N(           -10),N(          1000)),N( 0000000000990))
    Etst(Plus(N(           -10),N(       1000000)),N( 0000000999990))
    Etst(Plus(N(           -10),N(    1000000000)),N( 0000999999990))
    Etst(Plus(N(           -10),N( 1000000000000)),N( 0999999999990))

    Etst(Plus(N(            -1),N(-1000000000000)),N(-1000000000001))
    Etst(Plus(N(            -1),N(   -1000000000)),N(-0001000000001))
    Etst(Plus(N(            -1),N(      -1000000)),N(-0000001000001))
    Etst(Plus(N(            -1),N(         -1000)),N(-0000000001001))
    Etst(Plus(N(            -1),N(          -100)),N(-0000000000101))
    Etst(Plus(N(            -1),N(           -10)),N(-0000000000011))
    Etst(Plus(N(            -1),N(            -1)),N(-0000000000002))
    Etst(Plus(N(            -1),N(             0)),N(-0000000000001))
    Etst(Plus(N(            -1),N(             1)),N( 0000000000000))
    Etst(Plus(N(            -1),N(            10)),N( 0000000000009))
    Etst(Plus(N(            -1),N(           100)),N( 0000000000099))
    Etst(Plus(N(            -1),N(          1000)),N( 0000000000999))
    Etst(Plus(N(            -1),N(       1000000)),N( 0000000999999))
    Etst(Plus(N(            -1),N(    1000000000)),N( 0000999999999))
    Etst(Plus(N(            -1),N( 1000000000000)),N( 0999999999999))

    Etst(Plus(N(             0),N(-1000000000000)),N(-1000000000000))
    Etst(Plus(N(             0),N(   -1000000000)),N(-0001000000000))
    Etst(Plus(N(             0),N(      -1000000)),N(-0000001000000))
    Etst(Plus(N(             0),N(         -1000)),N(-0000000001000))
    Etst(Plus(N(             0),N(          -100)),N(-0000000000100))
    Etst(Plus(N(             0),N(           -10)),N(-0000000000010))
    Etst(Plus(N(             0),N(            -1)),N(-0000000000001))
    Etst(Plus(N(             0),N(             0)),N( 0000000000000))
    Etst(Plus(N(             0),N(             1)),N( 0000000000001))
    Etst(Plus(N(             0),N(            10)),N( 0000000000010))
    Etst(Plus(N(             0),N(           100)),N( 0000000000100))
    Etst(Plus(N(             0),N(          1000)),N( 0000000001000))
    Etst(Plus(N(             0),N(       1000000)),N( 0000001000000))
    Etst(Plus(N(             0),N(    1000000000)),N( 0001000000000))
    Etst(Plus(N(             0),N( 1000000000000)),N( 1000000000000))

    Etst(Plus(N( 1000000000000),N( 1000000000000)),N( 2000000000000))
    Etst(Plus(N( 1000000000000),N(    1000000000)),N( 1001000000000))
    Etst(Plus(N( 1000000000000),N(       1000000)),N( 1000001000000))
    Etst(Plus(N( 1000000000000),N(          1000)),N( 1000000001000))
    Etst(Plus(N( 1000000000000),N(           100)),N( 1000000000100))
    Etst(Plus(N( 1000000000000),N(            10)),N( 1000000000010))
    Etst(Plus(N( 1000000000000),N(             1)),N( 1000000000001))
    Etst(Plus(N( 1000000000000),N(             0)),N( 1000000000000))
    Etst(Plus(N( 1000000000000),N(            -1)),N( 0999999999999))
    Etst(Plus(N( 1000000000000),N(           -10)),N( 0999999999990))
    Etst(Plus(N( 1000000000000),N(          -100)),N( 0999999999900))
    Etst(Plus(N( 1000000000000),N(         -1000)),N( 0999999999000))
    Etst(Plus(N( 1000000000000),N(      -1000000)),N( 0999999000000))
    Etst(Plus(N( 1000000000000),N(   -1000000000)),N( 0999000000000))
    Etst(Plus(N( 1000000000000),N(-1000000000000)),N( 0000000000000))

    Etst(Plus(N(    1000000000),N( 1000000000000)),N( 1001000000000))
    Etst(Plus(N(    1000000000),N(    1000000000)),N( 0002000000000))
    Etst(Plus(N(    1000000000),N(       1000000)),N( 0001001000000))
    Etst(Plus(N(    1000000000),N(          1000)),N( 0001000001000))
    Etst(Plus(N(    1000000000),N(           100)),N( 0001000000100))
    Etst(Plus(N(    1000000000),N(            10)),N( 0001000000010))
    Etst(Plus(N(    1000000000),N(             1)),N( 0001000000001))
    Etst(Plus(N(    1000000000),N(             0)),N( 0001000000000))
    Etst(Plus(N(    1000000000),N(            -1)),N( 0000999999999))
    Etst(Plus(N(    1000000000),N(           -10)),N( 0000999999990))
    Etst(Plus(N(    1000000000),N(          -100)),N( 0000999999900))
    Etst(Plus(N(    1000000000),N(         -1000)),N( 0000999999000))
    Etst(Plus(N(    1000000000),N(      -1000000)),N( 0000999000000))
    Etst(Plus(N(    1000000000),N(   -1000000000)),N( 0000000000000))
    Etst(Plus(N(    1000000000),N(-1000000000000)),N(-0999000000000))

    Etst(Plus(N(       1000000),N( 1000000000000)),N( 1000001000000))
    Etst(Plus(N(       1000000),N(    1000000000)),N( 0001001000000))
    Etst(Plus(N(       1000000),N(       1000000)),N( 0000002000000))
    Etst(Plus(N(       1000000),N(          1000)),N( 0000001001000))
    Etst(Plus(N(       1000000),N(           100)),N( 0000001000100))
    Etst(Plus(N(       1000000),N(            10)),N( 0000001000010))
    Etst(Plus(N(       1000000),N(             1)),N( 0000001000001))
    Etst(Plus(N(       1000000),N(             0)),N( 0000001000000))
    Etst(Plus(N(       1000000),N(            -1)),N( 0000000999999))
    Etst(Plus(N(       1000000),N(           -10)),N( 0000000999990))
    Etst(Plus(N(       1000000),N(          -100)),N( 0000000999900))
    Etst(Plus(N(       1000000),N(         -1000)),N( 0000000999000))
    Etst(Plus(N(       1000000),N(      -1000000)),N( 0000000000000))
    Etst(Plus(N(       1000000),N(   -1000000000)),N(-0000999000000))
    Etst(Plus(N(       1000000),N(-1000000000000)),N(-0999999000000))

    Etst(Plus(N(          1000),N( 1000000000000)),N( 1000000001000))
    Etst(Plus(N(          1000),N(    1000000000)),N( 0001000001000))
    Etst(Plus(N(          1000),N(       1000000)),N( 0000001001000))
    Etst(Plus(N(          1000),N(          1000)),N( 0000000002000))
    Etst(Plus(N(          1000),N(           100)),N( 0000000001100))
    Etst(Plus(N(          1000),N(            10)),N( 0000000001010))
    Etst(Plus(N(          1000),N(             1)),N( 0000000001001))
    Etst(Plus(N(          1000),N(             0)),N( 0000000001000))
    Etst(Plus(N(          1000),N(            -1)),N( 0000000000999))
    Etst(Plus(N(          1000),N(           -10)),N( 0000000000990))
    Etst(Plus(N(          1000),N(          -100)),N( 0000000000900))
    Etst(Plus(N(          1000),N(         -1000)),N(-0000000000000))
    Etst(Plus(N(          1000),N(      -1000000)),N(-0000000999000))
    Etst(Plus(N(          1000),N(   -1000000000)),N(-0000999999000))
    Etst(Plus(N(          1000),N(-1000000000000)),N(-0999999999000))

    Etst(Plus(N(           100),N( 1000000000000)),N( 1000000000100))
    Etst(Plus(N(           100),N(    1000000000)),N( 0001000000100))
    Etst(Plus(N(           100),N(       1000000)),N( 0000001000100))
    Etst(Plus(N(           100),N(          1000)),N( 0000000001100))
    Etst(Plus(N(           100),N(           100)),N( 0000000000200))
    Etst(Plus(N(           100),N(            10)),N( 0000000000110))
    Etst(Plus(N(           100),N(             1)),N( 0000000000101))
    Etst(Plus(N(           100),N(             0)),N( 0000000000100))
    Etst(Plus(N(           100),N(            -1)),N( 0000000000099))
    Etst(Plus(N(           100),N(           -10)),N( 0000000000090))
    Etst(Plus(N(           100),N(          -100)),N(-0000000000000))
    Etst(Plus(N(           100),N(         -1000)),N(-0000000000900))
    Etst(Plus(N(           100),N(      -1000000)),N(-0000000999900))
    Etst(Plus(N(           100),N(   -1000000000)),N(-0000999999900))
    Etst(Plus(N(           100),N(-1000000000000)),N(-0999999999900))

    Etst(Plus(N(            10),N( 1000000000000)),N( 1000000000010))
    Etst(Plus(N(            10),N(    1000000000)),N( 0001000000010))
    Etst(Plus(N(            10),N(       1000000)),N( 0000001000010))
    Etst(Plus(N(            10),N(          1000)),N( 0000000001010))
    Etst(Plus(N(            10),N(           100)),N( 0000000000110))
    Etst(Plus(N(            10),N(            10)),N( 0000000000020))
    Etst(Plus(N(            10),N(             1)),N( 0000000000011))
    Etst(Plus(N(            10),N(             0)),N( 0000000000010))
    Etst(Plus(N(            10),N(            -1)),N( 0000000000009))
    Etst(Plus(N(            10),N(           -10)),N(-0000000000000))
    Etst(Plus(N(            10),N(          -100)),N(-0000000000090))
    Etst(Plus(N(            10),N(         -1000)),N(-0000000000990))
    Etst(Plus(N(            10),N(      -1000000)),N(-0000000999990))
    Etst(Plus(N(            10),N(   -1000000000)),N(-0000999999990))
    Etst(Plus(N(            10),N(-1000000000000)),N(-0999999999990))

    Etst(Plus(N(             1),N( 1000000000000)),N( 1000000000001))
    Etst(Plus(N(             1),N(    1000000000)),N( 0001000000001))
    Etst(Plus(N(             1),N(       1000000)),N( 0000001000001))
    Etst(Plus(N(             1),N(          1000)),N( 0000000001001))
    Etst(Plus(N(             1),N(           100)),N( 0000000000101))
    Etst(Plus(N(             1),N(            10)),N( 0000000000011))
    Etst(Plus(N(             1),N(             1)),N( 0000000000002))
    Etst(Plus(N(             1),N(             0)),N( 0000000000001))
    Etst(Plus(N(             1),N(            -1)),N(-0000000000000))
    Etst(Plus(N(             1),N(           -10)),N(-0000000000009))
    Etst(Plus(N(             1),N(          -100)),N(-0000000000099))
    Etst(Plus(N(             1),N(         -1000)),N(-0000000000999))
    Etst(Plus(N(             1),N(      -1000000)),N(-0000000999999))
    Etst(Plus(N(             1),N(   -1000000000)),N(-0000999999999))
    Etst(Plus(N(             1),N(-1000000000000)),N(-0999999999999))

    Xtst(Minus(TT,TT))
    Xtst(Minus(TT,FF))
    Xtst(Minus(TT,I0))
    Xtst(Minus(TT,P0))
    Xtst(Minus(TT,M0))
    Xtst(Minus(TT,B0))
    Xtst(Minus(FF,TT))
    Xtst(Minus(FF,FF))
    Xtst(Minus(FF,I0))
    Xtst(Minus(FF,P0))
    Xtst(Minus(FF,M0))
    Xtst(Minus(FF,B0))
    Xtst(Minus(I0,TT))
    Xtst(Minus(I0,FF))
    Xtst(Minus(I0,P0))
    Xtst(Minus(I0,M0))
    Xtst(Minus(I0,B0))
    Xtst(Minus(P0,TT))
    Xtst(Minus(P0,FF))
    Xtst(Minus(P0,I0))
    Xtst(Minus(P0,P0))
    Xtst(Minus(P0,M0))
    Xtst(Minus(P0,B0))
    Xtst(Minus(M0,TT))
    Xtst(Minus(M0,FF))
    Xtst(Minus(M0,I0))
    Xtst(Minus(M0,P0))
    Xtst(Minus(M0,M0))
    Xtst(Minus(M0,B0))
    Xtst(Minus(B0,TT))
    Xtst(Minus(B0,FF))
    Xtst(Minus(B0,I0))
    Xtst(Minus(B0,P0))
    Xtst(Minus(B0,M0))
    Xtst(Minus(B0,B0))
    Etst(Minus(JN(-3),JN(-3)),JN( 0))
    Etst(Minus(JN(-3),JN(-2)),JN(-1))
    Etst(Minus(JN(-3),JN(-1)),JN(-2))
    Etst(Minus(JN(-3),JN( 0)),JN(-3))
    Etst(Minus(JN(-3),JN( 1)),JN(-4))
    Etst(Minus(JN(-3),JN( 2)),JN(-5))
    Etst(Minus(JN(-2),JN(-3)),JN( 1))
    Etst(Minus(JN(-2),JN(-2)),JN( 0))
    Etst(Minus(JN(-2),JN(-1)),JN(-1))
    Etst(Minus(JN(-2),JN( 0)),JN(-2))
    Etst(Minus(JN(-2),JN( 1)),JN(-3))
    Etst(Minus(JN(-2),JN( 2)),JN(-4))
    Etst(Minus(JN(-1),JN(-3)),JN( 2))
    Etst(Minus(JN(-1),JN(-2)),JN( 1))
    Etst(Minus(JN(-1),JN(-1)),JN( 0))
    Etst(Minus(JN(-1),JN( 0)),JN(-1))
    Etst(Minus(JN(-1),JN( 1)),JN(-2))
    Etst(Minus(JN(-1),JN( 2)),JN(-3))
    Etst(Minus(JN( 0),JN(-3)),JN( 3))
    Etst(Minus(JN( 0),JN(-2)),JN( 2))
    Etst(Minus(JN( 0),JN(-1)),JN( 1))
    Etst(Minus(JN( 0),JN( 0)),JN( 0))
    Etst(Minus(JN( 0),JN( 1)),JN(-1))
    Etst(Minus(JN( 0),JN( 2)),JN(-2))
    Etst(Minus(JN( 1),JN(-3)),JN( 4))
    Etst(Minus(JN( 1),JN(-2)),JN( 3))
    Etst(Minus(JN( 1),JN(-1)),JN( 2))
    Etst(Minus(JN( 1),JN( 0)),JN( 1))
    Etst(Minus(JN( 1),JN( 1)),JN( 0))
    Etst(Minus(JN( 1),JN( 2)),JN(-1))
    Etst(Minus(JN( 2),JN(-3)),JN( 5))
    Etst(Minus(JN( 2),JN(-2)),JN( 4))
    Etst(Minus(JN( 2),JN(-1)),JN( 3))
    Etst(Minus(JN( 2),JN( 0)),JN( 2))
    Etst(Minus(JN( 2),JN( 1)),JN( 1))
    Etst(Minus(JN( 2),JN( 2)),JN( 0))
    Etst(Minus(K2( 0, 0),K2( 0, 0)),K2( 0, 0))
    Etst(Minus(K2( 0, 0),K2( 0, 1)),K2( 0,-1))
    Etst(Minus(K2( 0, 0),K2( 1,-1)),K2(-1, 0))
    Etst(Minus(K2( 0, 0),K2( 1, 0)),K2(-1,-1))
    Etst(Minus(K2( 0,-1),K2( 0, 0)),K2( 0,-1))
    Etst(Minus(K2( 0,-1),K2( 0, 1)),K2( 0,-2))
    Etst(Minus(K2( 0,-1),K2( 1,-1)),K2(-1,-1))
    Etst(Minus(K2( 0,-1),K2( 1, 0)),K2(-1,-2))
    Etst(Minus(K2(-1, 0),K2( 0, 0)),K2(-1, 0))
    Etst(Minus(K2(-1, 0),K2( 0, 1)),K2(-1,-1))
    Etst(Minus(K2(-1, 0),K2( 1,-1)),K2(-2, 1))
    Etst(Minus(K2(-1, 0),K2( 1, 0)),K2(-2, 0))
    Etst(Minus(K2(-1,-1),K2( 0, 0)),K2(-1,-1))
    Etst(Minus(K2(-1,-1),K2( 0, 1)),K2(-1,-2))
    Etst(Minus(K2(-1,-1),K2( 1,-1)),K2(-2, 0))
    Etst(Minus(K2(-1,-1),K2( 1, 0)),K2(-2,-1))

    Xtst(Uplus(TT))
    Xtst(Uplus(FF))
    Xtst(Uplus(P0))
    Xtst(Uplus(M0))
    Xtst(Uplus(B0))
    Etst(Uplus(Z),Z)
    Etst(Uplus(M),M)
    Etst(Uplus(I0011),I0011)
    Etst(Uplus(J1010_0101),J1010_0101)

    Xtst(Dplus(TT))
    Xtst(Dplus(FF))
    Xtst(Dplus(P0))
    Xtst(Dplus(M0))
    Xtst(Dplus(B0))
    Etst(Dplus(Z),Z)
    Etst(Dplus(M),M)
    Etst(Dplus(I0011),I0011)
    Etst(Dplus(J1010_0101),J1010_0101)

    Xtst(Uminus(TT))
    Xtst(Uminus(FF))
    Xtst(Uminus(P0))
    Xtst(Uminus(M0))
    Xtst(Uminus(B0))
    Etst(Uminus(JN(-2)),JN( 2))
    Etst(Uminus(JN(-1)),JN( 1))
    Etst(Uminus(JN( 0)),JN( 0))
    Etst(Uminus(JN( 1)),JN(-1))
    Etst(Uminus(JN( 2)),JN(-2))
    Etst(Uminus(K2(-1,-1)),K2( 1, 0))
    Etst(Uminus(K2(-1, 0)),K2( 1,-1))
    Etst(Uminus(K2(-1, 1)),K2( 1,-2))
    Etst(Uminus(K2( 0,-1)),K2( 0, 1))
    Etst(Uminus(K2( 0, 0)),K2( 0, 0))
    Etst(Uminus(K2( 0, 1)),K2( 0,-1))
    Etst(Uminus(K2( 1,-1)),K2(-1, 0))
    Etst(Uminus(K2( 1, 0)),K2(-1,-1))
    Etst(Uminus(K2( 1, 1)),K2(-1,-2))

    Xtst(Dminus(TT))
    Xtst(Dminus(FF))
    Xtst(Dminus(P0))
    Xtst(Dminus(M0))
    Xtst(Dminus(B0))
    Etst(Dminus(JN(-2)),JN( 2))
    Etst(Dminus(JN(-1)),JN( 1))
    Etst(Dminus(JN( 0)),JN( 0))
    Etst(Dminus(JN( 1)),JN(-1))
    Etst(Dminus(JN( 2)),JN(-2))
    Etst(Dminus(K2(-1,-1)),K2( 1, 0))
    Etst(Dminus(K2(-1, 0)),K2( 1,-1))
    Etst(Dminus(K2(-1, 1)),K2( 1,-2))
    Etst(Dminus(K2( 0,-1)),K2( 0, 1))
    Etst(Dminus(K2( 0, 0)),K2( 0, 0))
    Etst(Dminus(K2( 0, 1)),K2( 0,-1))
    Etst(Dminus(K2( 1,-1)),K2(-1, 0))
    Etst(Dminus(K2( 1, 0)),K2(-1,-1))
    Etst(Dminus(K2( 1, 1)),K2(-1,-2))

    Xtst(Intlength(TT))
    Xtst(Intlength(FF))
    Xtst(Intlength(P0))
    Xtst(Intlength(M0))
    Xtst(Intlength(B0))
    Etst(Intlength(Z),Z)
    Etst(Intlength(M),Z)
    Etst(Intlength(JN(1)),JN(1))
    Etst(Intlength(JN(2)),JN(2))
    Etst(Intlength(JN(3)),JN(2))
    Etst(Intlength(JN(4)),JN(3))
    Etst(Intlength(JN(5)),JN(3))
    Etst(Intlength(JN(6)),JN(3))
    Etst(Intlength(JN(7)),JN(3))
    Etst(Intlength(JN(8)),JN(4))
    Etst(Intlength(JN(-2)),JN(1))
    Etst(Intlength(JN(-3)),JN(2))
    Etst(Intlength(JN(-4)),JN(2))
    Etst(Intlength(JN(-5)),JN(3))
    Etst(Intlength(JN(-6)),JN(3))
    Etst(Intlength(JN(-7)),JN(3))
    Etst(Intlength(JN(-8)),JN(3))
    Etst(Intlength(JN(-9)),JN(4))
    Etst(Intlength(I0011_1100),JN(4+1*pntsize))
    Etst(Intlength(J1011_1101),JN(2+1*pntsize))
    Etst(Intlength(intcons(0,I0011_1100)),JN(4+2*pntsize))
    Etst(Intlength(intcons(0,J1011_1101)),JN(2+2*pntsize))
    Etst(Intlength(K2(0,-1)),JN(0+1*pntsize))
    Etst(Intlength(K2(-1,0)),JN(0+1*pntsize))

    Xtst(Logcount(TT))
    Xtst(Logcount(FF))
    Xtst(Logcount(P0))
    Xtst(Logcount(M0))
    Xtst(Logcount(B0))
    Etst(Logcount(Z),Z)
    Etst(Logcount(M),Z)
    Etst(Logcount(JN(1)),JN(1))
    Etst(Logcount(JN(2)),JN(1))
    Etst(Logcount(JN(3)),JN(2))
    Etst(Logcount(JN(4)),JN(1))
    Etst(Logcount(JN(5)),JN(2))
    Etst(Logcount(JN(6)),JN(2))
    Etst(Logcount(JN(7)),JN(3))
    Etst(Logcount(JN(8)),JN(1))
    Etst(Logcount(JN(-2)),JN(1))
    Etst(Logcount(JN(-3)),JN(1))
    Etst(Logcount(JN(-4)),JN(2))
    Etst(Logcount(JN(-5)),JN(1))
    Etst(Logcount(JN(-6)),JN(2))
    Etst(Logcount(JN(-7)),JN(2))
    Etst(Logcount(JN(-8)),JN(3))
    Etst(Logcount(JN(-9)),JN(1))
    Etst(Logcount(I0011_1100),JN(4))
    Etst(Logcount(J1011_1101),JN(2))
    Etst(Logcount(intcons(0,I0011_1100)),JN(4))
    Etst(Logcount(intcons(0,J1011_1101)),JN(2+1*pntsize))
    Etst(Logcount(K2(0,-1)),JN(0+1*pntsize))
    Etst(Logcount(K2(-1,0)),JN(0+1*pntsize))

    Xtst(Digit0(TT))
    Xtst(Digit0(FF))
    Xtst(Digit0(P0))
    Xtst(Digit0(M0))
    Xtst(Digit0(B0))
    Etst(Digit0(JN( 0)),JN(  0))
    Etst(Digit0(JN( 1)),JN( 10))
    Etst(Digit0(JN( 2)),JN( 20))
    Etst(Digit0(JN( 3)),JN( 30))
    Etst(Digit0(JN( 4)),JN( 40))
    Etst(Digit0(JN( 5)),JN( 50))
    Etst(Digit0(JN( 6)),JN( 60))
    Etst(Digit0(JN( 7)),JN( 70))
    Etst(Digit0(JN( 8)),JN( 80))
    Etst(Digit0(JN( 9)),JN( 90))
    Etst(Digit0(JN(-1)),JN(-10))
    Etst(Digit0(JN(-2)),JN(-20))
    Etst(Digit0(JN(-3)),JN(-30))
    Etst(Digit0(JN(-4)),JN(-40))
    Etst(Digit0(JN(-5)),JN(-50))
    Etst(Digit0(JN(-6)),JN(-60))
    Etst(Digit0(JN(-7)),JN(-70))
    Etst(Digit0(JN(-8)),JN(-80))
    Etst(Digit0(JN(-9)),JN(-90))

    Xtst(Digit1(TT))
    Xtst(Digit1(FF))
    Xtst(Digit1(P0))
    Xtst(Digit1(M0))
    Xtst(Digit1(B0))
    Etst(Digit1(JN( 0)),JN(  1))
    Etst(Digit1(JN( 1)),JN( 11))
    Etst(Digit1(JN( 2)),JN( 21))
    Etst(Digit1(JN( 3)),JN( 31))
    Etst(Digit1(JN( 4)),JN( 41))
    Etst(Digit1(JN( 5)),JN( 51))
    Etst(Digit1(JN( 6)),JN( 61))
    Etst(Digit1(JN( 7)),JN( 71))
    Etst(Digit1(JN( 8)),JN( 81))
    Etst(Digit1(JN( 9)),JN( 91))
    Etst(Digit1(JN(-1)),JN(- 9))
    Etst(Digit1(JN(-2)),JN(-19))
    Etst(Digit1(JN(-3)),JN(-29))
    Etst(Digit1(JN(-4)),JN(-39))
    Etst(Digit1(JN(-5)),JN(-49))
    Etst(Digit1(JN(-6)),JN(-59))
    Etst(Digit1(JN(-7)),JN(-69))
    Etst(Digit1(JN(-8)),JN(-79))
    Etst(Digit1(JN(-9)),JN(-89))

    Xtst(Digit2(TT))
    Xtst(Digit2(FF))
    Xtst(Digit2(P0))
    Xtst(Digit2(M0))
    Xtst(Digit2(B0))
    Etst(Digit2(JN( 0)),JN(  2))
    Etst(Digit2(JN( 1)),JN( 12))
    Etst(Digit2(JN( 2)),JN( 22))
    Etst(Digit2(JN( 3)),JN( 32))
    Etst(Digit2(JN( 4)),JN( 42))
    Etst(Digit2(JN( 5)),JN( 52))
    Etst(Digit2(JN( 6)),JN( 62))
    Etst(Digit2(JN( 7)),JN( 72))
    Etst(Digit2(JN( 8)),JN( 82))
    Etst(Digit2(JN( 9)),JN( 92))
    Etst(Digit2(JN(-1)),JN(- 8))
    Etst(Digit2(JN(-2)),JN(-18))
    Etst(Digit2(JN(-3)),JN(-28))
    Etst(Digit2(JN(-4)),JN(-38))
    Etst(Digit2(JN(-5)),JN(-48))
    Etst(Digit2(JN(-6)),JN(-58))
    Etst(Digit2(JN(-7)),JN(-68))
    Etst(Digit2(JN(-8)),JN(-78))
    Etst(Digit2(JN(-9)),JN(-88))

    Xtst(Digit3(TT))
    Xtst(Digit3(FF))
    Xtst(Digit3(P0))
    Xtst(Digit3(M0))
    Xtst(Digit3(B0))
    Etst(Digit3(JN( 0)),JN(  3))
    Etst(Digit3(JN( 1)),JN( 13))
    Etst(Digit3(JN( 2)),JN( 23))
    Etst(Digit3(JN( 3)),JN( 33))
    Etst(Digit3(JN( 4)),JN( 43))
    Etst(Digit3(JN( 5)),JN( 53))
    Etst(Digit3(JN( 6)),JN( 63))
    Etst(Digit3(JN( 7)),JN( 73))
    Etst(Digit3(JN( 8)),JN( 83))
    Etst(Digit3(JN( 9)),JN( 93))
    Etst(Digit3(JN(-1)),JN(- 7))
    Etst(Digit3(JN(-2)),JN(-17))
    Etst(Digit3(JN(-3)),JN(-27))
    Etst(Digit3(JN(-4)),JN(-37))
    Etst(Digit3(JN(-5)),JN(-47))
    Etst(Digit3(JN(-6)),JN(-57))
    Etst(Digit3(JN(-7)),JN(-67))
    Etst(Digit3(JN(-8)),JN(-77))
    Etst(Digit3(JN(-9)),JN(-87))

    Xtst(Digit4(TT))
    Xtst(Digit4(FF))
    Xtst(Digit4(P0))
    Xtst(Digit4(M0))
    Xtst(Digit4(B0))
    Etst(Digit4(JN( 0)),JN(  4))
    Etst(Digit4(JN( 1)),JN( 14))
    Etst(Digit4(JN( 2)),JN( 24))
    Etst(Digit4(JN( 3)),JN( 34))
    Etst(Digit4(JN( 4)),JN( 44))
    Etst(Digit4(JN( 5)),JN( 54))
    Etst(Digit4(JN( 6)),JN( 64))
    Etst(Digit4(JN( 7)),JN( 74))
    Etst(Digit4(JN( 8)),JN( 84))
    Etst(Digit4(JN( 9)),JN( 94))
    Etst(Digit4(JN(-1)),JN(- 6))
    Etst(Digit4(JN(-2)),JN(-16))
    Etst(Digit4(JN(-3)),JN(-26))
    Etst(Digit4(JN(-4)),JN(-36))
    Etst(Digit4(JN(-5)),JN(-46))
    Etst(Digit4(JN(-6)),JN(-56))
    Etst(Digit4(JN(-7)),JN(-66))
    Etst(Digit4(JN(-8)),JN(-76))
    Etst(Digit4(JN(-9)),JN(-86))

    Xtst(Digit5(TT))
    Xtst(Digit5(FF))
    Xtst(Digit5(P0))
    Xtst(Digit5(M0))
    Xtst(Digit5(B0))
    Etst(Digit5(JN( 0)),JN(  5))
    Etst(Digit5(JN( 1)),JN( 15))
    Etst(Digit5(JN( 2)),JN( 25))
    Etst(Digit5(JN( 3)),JN( 35))
    Etst(Digit5(JN( 4)),JN( 45))
    Etst(Digit5(JN( 5)),JN( 55))
    Etst(Digit5(JN( 6)),JN( 65))
    Etst(Digit5(JN( 7)),JN( 75))
    Etst(Digit5(JN( 8)),JN( 85))
    Etst(Digit5(JN( 9)),JN( 95))
    Etst(Digit5(JN(-1)),JN(- 5))
    Etst(Digit5(JN(-2)),JN(-15))
    Etst(Digit5(JN(-3)),JN(-25))
    Etst(Digit5(JN(-4)),JN(-35))
    Etst(Digit5(JN(-5)),JN(-45))
    Etst(Digit5(JN(-6)),JN(-55))
    Etst(Digit5(JN(-7)),JN(-65))
    Etst(Digit5(JN(-8)),JN(-75))
    Etst(Digit5(JN(-9)),JN(-85))

    Xtst(Digit6(TT))
    Xtst(Digit6(FF))
    Xtst(Digit6(P0))
    Xtst(Digit6(M0))
    Xtst(Digit6(B0))
    Etst(Digit6(JN( 0)),JN(  6))
    Etst(Digit6(JN( 1)),JN( 16))
    Etst(Digit6(JN( 2)),JN( 26))
    Etst(Digit6(JN( 3)),JN( 36))
    Etst(Digit6(JN( 4)),JN( 46))
    Etst(Digit6(JN( 5)),JN( 56))
    Etst(Digit6(JN( 6)),JN( 66))
    Etst(Digit6(JN( 7)),JN( 76))
    Etst(Digit6(JN( 8)),JN( 86))
    Etst(Digit6(JN( 9)),JN( 96))
    Etst(Digit6(JN(-1)),JN(- 4))
    Etst(Digit6(JN(-2)),JN(-14))
    Etst(Digit6(JN(-3)),JN(-24))
    Etst(Digit6(JN(-4)),JN(-34))
    Etst(Digit6(JN(-5)),JN(-44))
    Etst(Digit6(JN(-6)),JN(-54))
    Etst(Digit6(JN(-7)),JN(-64))
    Etst(Digit6(JN(-8)),JN(-74))
    Etst(Digit6(JN(-9)),JN(-84))

    Xtst(Digit7(TT))
    Xtst(Digit7(FF))
    Xtst(Digit7(P0))
    Xtst(Digit7(M0))
    Xtst(Digit7(B0))
    Etst(Digit7(JN( 0)),JN(  7))
    Etst(Digit7(JN( 1)),JN( 17))
    Etst(Digit7(JN( 2)),JN( 27))
    Etst(Digit7(JN( 3)),JN( 37))
    Etst(Digit7(JN( 4)),JN( 47))
    Etst(Digit7(JN( 5)),JN( 57))
    Etst(Digit7(JN( 6)),JN( 67))
    Etst(Digit7(JN( 7)),JN( 77))
    Etst(Digit7(JN( 8)),JN( 87))
    Etst(Digit7(JN( 9)),JN( 97))
    Etst(Digit7(JN(-1)),JN(- 3))
    Etst(Digit7(JN(-2)),JN(-13))
    Etst(Digit7(JN(-3)),JN(-23))
    Etst(Digit7(JN(-4)),JN(-33))
    Etst(Digit7(JN(-5)),JN(-43))
    Etst(Digit7(JN(-6)),JN(-53))
    Etst(Digit7(JN(-7)),JN(-63))
    Etst(Digit7(JN(-8)),JN(-73))
    Etst(Digit7(JN(-9)),JN(-83))

    Xtst(Digit8(TT))
    Xtst(Digit8(FF))
    Xtst(Digit8(P0))
    Xtst(Digit8(M0))
    Xtst(Digit8(B0))
    Etst(Digit8(JN( 0)),JN(  8))
    Etst(Digit8(JN( 1)),JN( 18))
    Etst(Digit8(JN( 2)),JN( 28))
    Etst(Digit8(JN( 3)),JN( 38))
    Etst(Digit8(JN( 4)),JN( 48))
    Etst(Digit8(JN( 5)),JN( 58))
    Etst(Digit8(JN( 6)),JN( 68))
    Etst(Digit8(JN( 7)),JN( 78))
    Etst(Digit8(JN( 8)),JN( 88))
    Etst(Digit8(JN( 9)),JN( 98))
    Etst(Digit8(JN(-1)),JN(- 2))
    Etst(Digit8(JN(-2)),JN(-12))
    Etst(Digit8(JN(-3)),JN(-22))
    Etst(Digit8(JN(-4)),JN(-32))
    Etst(Digit8(JN(-5)),JN(-42))
    Etst(Digit8(JN(-6)),JN(-52))
    Etst(Digit8(JN(-7)),JN(-62))
    Etst(Digit8(JN(-8)),JN(-72))
    Etst(Digit8(JN(-9)),JN(-82))

    Xtst(Digit9(TT))
    Xtst(Digit9(FF))
    Xtst(Digit9(P0))
    Xtst(Digit9(M0))
    Xtst(Digit9(B0))
    Etst(Digit9(JN( 0)),JN(  9))
    Etst(Digit9(JN( 1)),JN( 19))
    Etst(Digit9(JN( 2)),JN( 29))
    Etst(Digit9(JN( 3)),JN( 39))
    Etst(Digit9(JN( 4)),JN( 49))
    Etst(Digit9(JN( 5)),JN( 59))
    Etst(Digit9(JN( 6)),JN( 69))
    Etst(Digit9(JN( 7)),JN( 79))
    Etst(Digit9(JN( 8)),JN( 89))
    Etst(Digit9(JN( 9)),JN( 99))
    Etst(Digit9(JN(-1)),JN(- 1))
    Etst(Digit9(JN(-2)),JN(-11))
    Etst(Digit9(JN(-3)),JN(-21))
    Etst(Digit9(JN(-4)),JN(-31))
    Etst(Digit9(JN(-5)),JN(-41))
    Etst(Digit9(JN(-6)),JN(-51))
    Etst(Digit9(JN(-7)),JN(-61))
    Etst(Digit9(JN(-8)),JN(-71))
    Etst(Digit9(JN(-9)),JN(-81))

    Xtst(Times(TT,TT))
    Xtst(Times(TT,FF))
    Xtst(Times(TT,I0))
    Xtst(Times(TT,P0))
    Xtst(Times(TT,M0))
    Xtst(Times(TT,B0))
    Xtst(Times(FF,TT))
    Xtst(Times(FF,FF))
    Xtst(Times(FF,I0))
    Xtst(Times(FF,P0))
    Xtst(Times(FF,M0))
    Xtst(Times(FF,B0))
    Xtst(Times(I0,TT))
    Xtst(Times(I0,FF))
    Xtst(Times(I0,P0))
    Xtst(Times(I0,M0))
    Xtst(Times(I0,B0))
    Xtst(Times(P0,TT))
    Xtst(Times(P0,FF))
    Xtst(Times(P0,I0))
    Xtst(Times(P0,P0))
    Xtst(Times(P0,M0))
    Xtst(Times(P0,B0))
    Xtst(Times(M0,TT))
    Xtst(Times(M0,FF))
    Xtst(Times(M0,I0))
    Xtst(Times(M0,P0))
    Xtst(Times(M0,M0))
    Xtst(Times(M0,B0))
    Xtst(Times(B0,TT))
    Xtst(Times(B0,FF))
    Xtst(Times(B0,I0))
    Xtst(Times(B0,P0))
    Xtst(Times(B0,M0))
    Xtst(Times(B0,B0))
    Etst(Times(JN(-3),JN(-3)),JN( 9))
    Etst(Times(JN(-3),JN(-2)),JN( 6))
    Etst(Times(JN(-3),JN(-1)),JN( 3))
    Etst(Times(JN(-3),JN( 0)),JN( 0))
    Etst(Times(JN(-3),JN( 1)),JN(-3))
    Etst(Times(JN(-3),JN( 2)),JN(-6))
    Etst(Times(JN(-2),JN(-3)),JN( 6))
    Etst(Times(JN(-2),JN(-2)),JN( 4))
    Etst(Times(JN(-2),JN(-1)),JN( 2))
    Etst(Times(JN(-2),JN( 0)),JN( 0))
    Etst(Times(JN(-2),JN( 1)),JN(-2))
    Etst(Times(JN(-2),JN( 2)),JN(-4))
    Etst(Times(JN(-1),JN(-3)),JN( 3))
    Etst(Times(JN(-1),JN(-2)),JN( 2))
    Etst(Times(JN(-1),JN(-1)),JN( 1))
    Etst(Times(JN(-1),JN( 0)),JN( 0))
    Etst(Times(JN(-1),JN( 1)),JN(-1))
    Etst(Times(JN(-1),JN( 2)),JN(-2))
    Etst(Times(JN( 0),JN(-3)),JN( 0))
    Etst(Times(JN( 0),JN(-2)),JN( 0))
    Etst(Times(JN( 0),JN(-1)),JN( 0))
    Etst(Times(JN( 0),JN( 0)),JN( 0))
    Etst(Times(JN( 0),JN( 1)),JN( 0))
    Etst(Times(JN( 0),JN( 2)),JN( 0))
    Etst(Times(JN( 1),JN(-3)),JN(-3))
    Etst(Times(JN( 1),JN(-2)),JN(-2))
    Etst(Times(JN( 1),JN(-1)),JN(-1))
    Etst(Times(JN( 1),JN( 0)),JN( 0))
    Etst(Times(JN( 1),JN( 1)),JN( 1))
    Etst(Times(JN( 1),JN( 2)),JN( 2))
    Etst(Times(JN( 2),JN(-3)),JN(-6))
    Etst(Times(JN( 2),JN(-2)),JN(-4))
    Etst(Times(JN( 2),JN(-1)),JN(-2))
    Etst(Times(JN( 2),JN( 0)),JN( 0))
    Etst(Times(JN( 2),JN( 1)),JN( 2))
    Etst(Times(JN( 2),JN( 2)),JN( 4))

    Etst(Times(N(-1000000000000),N(-1000000000000)),N( 1000000000000000000000000))
    Etst(Times(N(-1000000000000),N(   -1000000000)),N(    1000000000000000000000))
    Etst(Times(N(-1000000000000),N(      -1000000)),N(       1000000000000000000))
    Etst(Times(N(-1000000000000),N(         -1000)),N(          1000000000000000))
    Etst(Times(N(-1000000000000),N(          -100)),N(           100000000000000))
    Etst(Times(N(-1000000000000),N(           -10)),N(            10000000000000))
    Etst(Times(N(-1000000000000),N(            -1)),N(             1000000000000))
    Etst(Times(N(-1000000000000),N(             0)),N(                         0))
    Etst(Times(N(-1000000000000),N(             1)),N(            -1000000000000))
    Etst(Times(N(-1000000000000),N(            10)),N(           -10000000000000))
    Etst(Times(N(-1000000000000),N(           100)),N(          -100000000000000))
    Etst(Times(N(-1000000000000),N(          1000)),N(         -1000000000000000))
    Etst(Times(N(-1000000000000),N(       1000000)),N(      -1000000000000000000))
    Etst(Times(N(-1000000000000),N(    1000000000)),N(   -1000000000000000000000))
    Etst(Times(N(-1000000000000),N( 1000000000000)),N(-1000000000000000000000000))

    Etst(Times(N(-1000000000),N(-1000000000000)),N( 1000000000000000000000))
    Etst(Times(N(-1000000000),N(   -1000000000)),N(    1000000000000000000))
    Etst(Times(N(-1000000000),N(      -1000000)),N(       1000000000000000))
    Etst(Times(N(-1000000000),N(         -1000)),N(          1000000000000))
    Etst(Times(N(-1000000000),N(          -100)),N(           100000000000))
    Etst(Times(N(-1000000000),N(           -10)),N(            10000000000))
    Etst(Times(N(-1000000000),N(            -1)),N(             1000000000))
    Etst(Times(N(-1000000000),N(             0)),N(                      0))
    Etst(Times(N(-1000000000),N(             1)),N(            -1000000000))
    Etst(Times(N(-1000000000),N(            10)),N(           -10000000000))
    Etst(Times(N(-1000000000),N(           100)),N(          -100000000000))
    Etst(Times(N(-1000000000),N(          1000)),N(         -1000000000000))
    Etst(Times(N(-1000000000),N(       1000000)),N(      -1000000000000000))
    Etst(Times(N(-1000000000),N(    1000000000)),N(   -1000000000000000000))
    Etst(Times(N(-1000000000),N( 1000000000000)),N(-1000000000000000000000))

    Etst(Times(N(-1000000),N(-1000000000000)),N( 1000000000000000000))
    Etst(Times(N(-1000000),N(   -1000000000)),N(    1000000000000000))
    Etst(Times(N(-1000000),N(      -1000000)),N(       1000000000000))
    Etst(Times(N(-1000000),N(         -1000)),N(          1000000000))
    Etst(Times(N(-1000000),N(          -100)),N(           100000000))
    Etst(Times(N(-1000000),N(           -10)),N(            10000000))
    Etst(Times(N(-1000000),N(            -1)),N(             1000000))
    Etst(Times(N(-1000000),N(             0)),N(                   0))
    Etst(Times(N(-1000000),N(             1)),N(            -1000000))
    Etst(Times(N(-1000000),N(            10)),N(           -10000000))
    Etst(Times(N(-1000000),N(           100)),N(          -100000000))
    Etst(Times(N(-1000000),N(          1000)),N(         -1000000000))
    Etst(Times(N(-1000000),N(       1000000)),N(      -1000000000000))
    Etst(Times(N(-1000000),N(    1000000000)),N(   -1000000000000000))
    Etst(Times(N(-1000000),N( 1000000000000)),N(-1000000000000000000))

    Etst(Times(N(-1000),N(-1000000000000)),N( 1000000000000000))
    Etst(Times(N(-1000),N(   -1000000000)),N(    1000000000000))
    Etst(Times(N(-1000),N(      -1000000)),N(       1000000000))
    Etst(Times(N(-1000),N(         -1000)),N(          1000000))
    Etst(Times(N(-1000),N(          -100)),N(           100000))
    Etst(Times(N(-1000),N(           -10)),N(            10000))
    Etst(Times(N(-1000),N(            -1)),N(             1000))
    Etst(Times(N(-1000),N(             0)),N(                0))
    Etst(Times(N(-1000),N(             1)),N(            -1000))
    Etst(Times(N(-1000),N(            10)),N(           -10000))
    Etst(Times(N(-1000),N(           100)),N(          -100000))
    Etst(Times(N(-1000),N(          1000)),N(         -1000000))
    Etst(Times(N(-1000),N(       1000000)),N(      -1000000000))
    Etst(Times(N(-1000),N(    1000000000)),N(   -1000000000000))
    Etst(Times(N(-1000),N( 1000000000000)),N(-1000000000000000))

    Etst(Times(N(-100),N(-1000000000000)),N( 100000000000000))
    Etst(Times(N(-100),N(   -1000000000)),N(    100000000000))
    Etst(Times(N(-100),N(      -1000000)),N(       100000000))
    Etst(Times(N(-100),N(         -1000)),N(          100000))
    Etst(Times(N(-100),N(          -100)),N(           10000))
    Etst(Times(N(-100),N(           -10)),N(            1000))
    Etst(Times(N(-100),N(            -1)),N(             100))
    Etst(Times(N(-100),N(             0)),N(               0))
    Etst(Times(N(-100),N(             1)),N(            -100))
    Etst(Times(N(-100),N(            10)),N(           -1000))
    Etst(Times(N(-100),N(           100)),N(          -10000))
    Etst(Times(N(-100),N(          1000)),N(         -100000))
    Etst(Times(N(-100),N(       1000000)),N(      -100000000))
    Etst(Times(N(-100),N(    1000000000)),N(   -100000000000))
    Etst(Times(N(-100),N( 1000000000000)),N(-100000000000000))

    Etst(Times(N(-10),N(-1000000000000)),N( 10000000000000))
    Etst(Times(N(-10),N(   -1000000000)),N(    10000000000))
    Etst(Times(N(-10),N(      -1000000)),N(       10000000))
    Etst(Times(N(-10),N(         -1000)),N(          10000))
    Etst(Times(N(-10),N(          -100)),N(           1000))
    Etst(Times(N(-10),N(           -10)),N(            100))
    Etst(Times(N(-10),N(            -1)),N(             10))
    Etst(Times(N(-10),N(             0)),N(              0))
    Etst(Times(N(-10),N(             1)),N(            -10))
    Etst(Times(N(-10),N(            10)),N(           -100))
    Etst(Times(N(-10),N(           100)),N(          -1000))
    Etst(Times(N(-10),N(          1000)),N(         -10000))
    Etst(Times(N(-10),N(       1000000)),N(      -10000000))
    Etst(Times(N(-10),N(    1000000000)),N(   -10000000000))
    Etst(Times(N(-10),N( 1000000000000)),N(-10000000000000))

    Etst(Times(N(-1),N(-1000000000000)),N( 1000000000000))
    Etst(Times(N(-1),N(   -1000000000)),N(    1000000000))
    Etst(Times(N(-1),N(      -1000000)),N(       1000000))
    Etst(Times(N(-1),N(         -1000)),N(          1000))
    Etst(Times(N(-1),N(          -100)),N(           100))
    Etst(Times(N(-1),N(           -10)),N(            10))
    Etst(Times(N(-1),N(            -1)),N(             1))
    Etst(Times(N(-1),N(             0)),N(             0))
    Etst(Times(N(-1),N(             1)),N(            -1))
    Etst(Times(N(-1),N(            10)),N(           -10))
    Etst(Times(N(-1),N(           100)),N(          -100))
    Etst(Times(N(-1),N(          1000)),N(         -1000))
    Etst(Times(N(-1),N(       1000000)),N(      -1000000))
    Etst(Times(N(-1),N(    1000000000)),N(   -1000000000))
    Etst(Times(N(-1),N( 1000000000000)),N(-1000000000000))

    Etst(Times(N(0),N(-1000000000000)),N(0))
    Etst(Times(N(0),N(   -1000000000)),N(0))
    Etst(Times(N(0),N(      -1000000)),N(0))
    Etst(Times(N(0),N(         -1000)),N(0))
    Etst(Times(N(0),N(          -100)),N(0))
    Etst(Times(N(0),N(           -10)),N(0))
    Etst(Times(N(0),N(            -1)),N(0))
    Etst(Times(N(0),N(             0)),N(0))
    Etst(Times(N(0),N(             1)),N(0))
    Etst(Times(N(0),N(            10)),N(0))
    Etst(Times(N(0),N(           100)),N(0))
    Etst(Times(N(0),N(          1000)),N(0))
    Etst(Times(N(0),N(       1000000)),N(0))
    Etst(Times(N(0),N(    1000000000)),N(0))
    Etst(Times(N(0),N( 1000000000000)),N(0))

    Etst(Times(N(1000000000000),N( 1000000000000)),N( 1000000000000000000000000))
    Etst(Times(N(1000000000000),N(    1000000000)),N(    1000000000000000000000))
    Etst(Times(N(1000000000000),N(       1000000)),N(       1000000000000000000))
    Etst(Times(N(1000000000000),N(          1000)),N(          1000000000000000))
    Etst(Times(N(1000000000000),N(           100)),N(           100000000000000))
    Etst(Times(N(1000000000000),N(            10)),N(            10000000000000))
    Etst(Times(N(1000000000000),N(             1)),N(             1000000000000))
    Etst(Times(N(1000000000000),N(             0)),N(                         0))
    Etst(Times(N(1000000000000),N(            -1)),N(            -1000000000000))
    Etst(Times(N(1000000000000),N(           -10)),N(           -10000000000000))
    Etst(Times(N(1000000000000),N(          -100)),N(          -100000000000000))
    Etst(Times(N(1000000000000),N(         -1000)),N(         -1000000000000000))
    Etst(Times(N(1000000000000),N(      -1000000)),N(      -1000000000000000000))
    Etst(Times(N(1000000000000),N(   -1000000000)),N(   -1000000000000000000000))
    Etst(Times(N(1000000000000),N(-1000000000000)),N(-1000000000000000000000000))

    Etst(Times(N(1000000000),N( 1000000000000)),N( 1000000000000000000000))
    Etst(Times(N(1000000000),N(    1000000000)),N(    1000000000000000000))
    Etst(Times(N(1000000000),N(       1000000)),N(       1000000000000000))
    Etst(Times(N(1000000000),N(          1000)),N(          1000000000000))
    Etst(Times(N(1000000000),N(           100)),N(           100000000000))
    Etst(Times(N(1000000000),N(            10)),N(            10000000000))
    Etst(Times(N(1000000000),N(             1)),N(             1000000000))
    Etst(Times(N(1000000000),N(             0)),N(                      0))
    Etst(Times(N(1000000000),N(            -1)),N(            -1000000000))
    Etst(Times(N(1000000000),N(           -10)),N(           -10000000000))
    Etst(Times(N(1000000000),N(          -100)),N(          -100000000000))
    Etst(Times(N(1000000000),N(         -1000)),N(         -1000000000000))
    Etst(Times(N(1000000000),N(      -1000000)),N(      -1000000000000000))
    Etst(Times(N(1000000000),N(   -1000000000)),N(   -1000000000000000000))
    Etst(Times(N(1000000000),N(-1000000000000)),N(-1000000000000000000000))

    Etst(Times(N(1000000),N( 1000000000000)),N( 1000000000000000000))
    Etst(Times(N(1000000),N(    1000000000)),N(    1000000000000000))
    Etst(Times(N(1000000),N(       1000000)),N(       1000000000000))
    Etst(Times(N(1000000),N(          1000)),N(          1000000000))
    Etst(Times(N(1000000),N(           100)),N(           100000000))
    Etst(Times(N(1000000),N(            10)),N(            10000000))
    Etst(Times(N(1000000),N(             1)),N(             1000000))
    Etst(Times(N(1000000),N(             0)),N(                   0))
    Etst(Times(N(1000000),N(            -1)),N(            -1000000))
    Etst(Times(N(1000000),N(           -10)),N(           -10000000))
    Etst(Times(N(1000000),N(          -100)),N(          -100000000))
    Etst(Times(N(1000000),N(         -1000)),N(         -1000000000))
    Etst(Times(N(1000000),N(      -1000000)),N(      -1000000000000))
    Etst(Times(N(1000000),N(   -1000000000)),N(   -1000000000000000))
    Etst(Times(N(1000000),N(-1000000000000)),N(-1000000000000000000))

    Etst(Times(N(1000),N( 1000000000000)),N( 1000000000000000))
    Etst(Times(N(1000),N(    1000000000)),N(    1000000000000))
    Etst(Times(N(1000),N(       1000000)),N(       1000000000))
    Etst(Times(N(1000),N(          1000)),N(          1000000))
    Etst(Times(N(1000),N(           100)),N(           100000))
    Etst(Times(N(1000),N(            10)),N(            10000))
    Etst(Times(N(1000),N(             1)),N(             1000))
    Etst(Times(N(1000),N(             0)),N(                0))
    Etst(Times(N(1000),N(            -1)),N(            -1000))
    Etst(Times(N(1000),N(           -10)),N(           -10000))
    Etst(Times(N(1000),N(          -100)),N(          -100000))
    Etst(Times(N(1000),N(         -1000)),N(         -1000000))
    Etst(Times(N(1000),N(      -1000000)),N(      -1000000000))
    Etst(Times(N(1000),N(   -1000000000)),N(   -1000000000000))
    Etst(Times(N(1000),N(-1000000000000)),N(-1000000000000000))

    Etst(Times(N(100),N( 1000000000000)),N( 100000000000000))
    Etst(Times(N(100),N(    1000000000)),N(    100000000000))
    Etst(Times(N(100),N(       1000000)),N(       100000000))
    Etst(Times(N(100),N(          1000)),N(          100000))
    Etst(Times(N(100),N(           100)),N(           10000))
    Etst(Times(N(100),N(            10)),N(            1000))
    Etst(Times(N(100),N(             1)),N(             100))
    Etst(Times(N(100),N(             0)),N(               0))
    Etst(Times(N(100),N(            -1)),N(            -100))
    Etst(Times(N(100),N(           -10)),N(           -1000))
    Etst(Times(N(100),N(          -100)),N(          -10000))
    Etst(Times(N(100),N(         -1000)),N(         -100000))
    Etst(Times(N(100),N(      -1000000)),N(      -100000000))
    Etst(Times(N(100),N(   -1000000000)),N(   -100000000000))
    Etst(Times(N(100),N(-1000000000000)),N(-100000000000000))

    Etst(Times(N(10),N( 1000000000000)),N( 10000000000000))
    Etst(Times(N(10),N(    1000000000)),N(    10000000000))
    Etst(Times(N(10),N(       1000000)),N(       10000000))
    Etst(Times(N(10),N(          1000)),N(          10000))
    Etst(Times(N(10),N(           100)),N(           1000))
    Etst(Times(N(10),N(            10)),N(            100))
    Etst(Times(N(10),N(             1)),N(             10))
    Etst(Times(N(10),N(             0)),N(              0))
    Etst(Times(N(10),N(            -1)),N(            -10))
    Etst(Times(N(10),N(           -10)),N(           -100))
    Etst(Times(N(10),N(          -100)),N(          -1000))
    Etst(Times(N(10),N(         -1000)),N(         -10000))
    Etst(Times(N(10),N(      -1000000)),N(      -10000000))
    Etst(Times(N(10),N(   -1000000000)),N(   -10000000000))
    Etst(Times(N(10),N(-1000000000000)),N(-10000000000000))

    Etst(Times(N(1),N( 1000000000000)),N( 1000000000000))
    Etst(Times(N(1),N(    1000000000)),N(    1000000000))
    Etst(Times(N(1),N(       1000000)),N(       1000000))
    Etst(Times(N(1),N(          1000)),N(          1000))
    Etst(Times(N(1),N(           100)),N(           100))
    Etst(Times(N(1),N(            10)),N(            10))
    Etst(Times(N(1),N(             1)),N(             1))
    Etst(Times(N(1),N(             0)),N(             0))
    Etst(Times(N(1),N(            -1)),N(            -1))
    Etst(Times(N(1),N(           -10)),N(           -10))
    Etst(Times(N(1),N(          -100)),N(          -100))
    Etst(Times(N(1),N(         -1000)),N(         -1000))
    Etst(Times(N(1),N(      -1000000)),N(      -1000000))
    Etst(Times(N(1),N(   -1000000000)),N(   -1000000000))
    Etst(Times(N(1),N(-1000000000000)),N(-1000000000000))

    Xtst(Logbitp(TT,TT))
    Xtst(Logbitp(TT,FF))
    Xtst(Logbitp(TT,I0))
    Xtst(Logbitp(TT,P0))
    Xtst(Logbitp(TT,M0))
    Xtst(Logbitp(TT,B0))
    Xtst(Logbitp(FF,TT))
    Xtst(Logbitp(FF,FF))
    Xtst(Logbitp(FF,I0))
    Xtst(Logbitp(FF,P0))
    Xtst(Logbitp(FF,M0))
    Xtst(Logbitp(FF,B0))
    Xtst(Logbitp(I0,TT))
    Xtst(Logbitp(I0,FF))
    Xtst(Logbitp(I0,P0))
    Xtst(Logbitp(I0,M0))
    Xtst(Logbitp(I0,B0))
    Xtst(Logbitp(P0,TT))
    Xtst(Logbitp(P0,FF))
    Xtst(Logbitp(P0,I0))
    Xtst(Logbitp(P0,P0))
    Xtst(Logbitp(P0,M0))
    Xtst(Logbitp(P0,B0))
    Xtst(Logbitp(M0,TT))
    Xtst(Logbitp(M0,FF))
    Xtst(Logbitp(M0,I0))
    Xtst(Logbitp(M0,P0))
    Xtst(Logbitp(M0,M0))
    Xtst(Logbitp(M0,B0))
    Xtst(Logbitp(B0,TT))
    Xtst(Logbitp(B0,FF))
    Xtst(Logbitp(B0,I0))
    Xtst(Logbitp(B0,P0))
    Xtst(Logbitp(B0,M0))
    Xtst(Logbitp(B0,B0))
    Xtst(Logbitp(JN(-3),JN(-3)))
    Xtst(Logbitp(JN(-3),JN(-2)))
    Xtst(Logbitp(JN(-3),JN(-1)))
    Xtst(Logbitp(JN(-3),JN( 0)))
    Xtst(Logbitp(JN(-3),JN( 1)))
    Xtst(Logbitp(JN(-3),JN( 2)))
    Xtst(Logbitp(JN(-2),JN(-3)))
    Xtst(Logbitp(JN(-2),JN(-2)))
    Xtst(Logbitp(JN(-2),JN(-1)))
    Xtst(Logbitp(JN(-2),JN( 0)))
    Xtst(Logbitp(JN(-2),JN( 1)))
    Xtst(Logbitp(JN(-2),JN( 2)))
    Xtst(Logbitp(JN(-1),JN(-3)))
    Xtst(Logbitp(JN(-1),JN(-2)))
    Xtst(Logbitp(JN(-1),JN(-1)))
    Xtst(Logbitp(JN(-1),JN( 0)))
    Xtst(Logbitp(JN(-1),JN( 1)))
    Xtst(Logbitp(JN(-1),JN( 2)))
    Ttst(Logbitp(JN( 0),JN(-3)))
    Ftst(Logbitp(JN( 0),JN(-2)))
    Ttst(Logbitp(JN( 0),JN(-1)))
    Ftst(Logbitp(JN( 0),JN( 0)))
    Ttst(Logbitp(JN( 0),JN( 1)))
    Ftst(Logbitp(JN( 0),JN( 2)))
    Ftst(Logbitp(JN( 1),JN(-3)))
    Ttst(Logbitp(JN( 1),JN(-2)))
    Ttst(Logbitp(JN( 1),JN(-1)))
    Ftst(Logbitp(JN( 1),JN( 0)))
    Ftst(Logbitp(JN( 1),JN( 1)))
    Ttst(Logbitp(JN( 1),JN( 2)))
    Ttst(Logbitp(JN( 2),JN(-3)))
    Ttst(Logbitp(JN( 2),JN(-2)))
    Ttst(Logbitp(JN( 2),JN(-1)))
    Ftst(Logbitp(JN( 2),JN( 0)))
    Ftst(Logbitp(JN( 2),JN( 1)))
    Ftst(Logbitp(JN( 2),JN( 2)))
    Ttst(Logbitp(JN( 0+0*pntsize),I0011_1100))
    Ttst(Logbitp(JN( 1+0*pntsize),I0011_1100))
    Ftst(Logbitp(JN( 2+0*pntsize),I0011_1100))
    Ftst(Logbitp(JN( 3+0*pntsize),I0011_1100))
    Ftst(Logbitp(JN( 4+0*pntsize),I0011_1100))
    Ftst(Logbitp(JN( 0+1*pntsize),I0011_1100))
    Ftst(Logbitp(JN( 1+1*pntsize),I0011_1100))
    Ttst(Logbitp(JN( 2+1*pntsize),I0011_1100))
    Ttst(Logbitp(JN( 3+1*pntsize),I0011_1100))
    Ftst(Logbitp(JN( 4+1*pntsize),I0011_1100))
    Ftst(Logbitp(JN( 0+2*pntsize),I0011_1100))
    Ftst(Logbitp(JN( 1+2*pntsize),I0011_1100))
    Ftst(Logbitp(JN( 2+2*pntsize),I0011_1100))
    Ftst(Logbitp(JN( 3+2*pntsize),I0011_1100))
    Ftst(Logbitp(JN( 4+2*pntsize),I0011_1100))
    Ttst(Logbitp(JN( 0+0*pntsize),J1011_1101))
    Ttst(Logbitp(JN( 1+0*pntsize),J1011_1101))
    Ftst(Logbitp(JN( 2+0*pntsize),J1011_1101))
    Ttst(Logbitp(JN( 3+0*pntsize),J1011_1101))
    Ttst(Logbitp(JN( 4+0*pntsize),J1011_1101))
    Ttst(Logbitp(JN( 0+1*pntsize),J1011_1101))
    Ftst(Logbitp(JN( 1+1*pntsize),J1011_1101))
    Ttst(Logbitp(JN( 2+1*pntsize),J1011_1101))
    Ttst(Logbitp(JN( 3+1*pntsize),J1011_1101))
    Ttst(Logbitp(JN( 4+1*pntsize),J1011_1101))
    Ttst(Logbitp(JN( 0+2*pntsize),J1011_1101))
    Ttst(Logbitp(JN( 1+2*pntsize),J1011_1101))
    Ttst(Logbitp(JN( 2+2*pntsize),J1011_1101))
    Ttst(Logbitp(JN( 3+2*pntsize),J1011_1101))
    Ttst(Logbitp(JN( 4+2*pntsize),J1011_1101))

    Xtst(Ash(TT,TT))
    Xtst(Ash(TT,FF))
    Xtst(Ash(TT,I0))
    Xtst(Ash(TT,P0))
    Xtst(Ash(TT,M0))
    Xtst(Ash(TT,B0))
    Xtst(Ash(FF,TT))
    Xtst(Ash(FF,FF))
    Xtst(Ash(FF,I0))
    Xtst(Ash(FF,P0))
    Xtst(Ash(FF,M0))
    Xtst(Ash(FF,B0))
    Xtst(Ash(I0,TT))
    Xtst(Ash(I0,FF))
    Xtst(Ash(I0,P0))
    Xtst(Ash(I0,M0))
    Xtst(Ash(I0,B0))
    Xtst(Ash(P0,TT))
    Xtst(Ash(P0,FF))
    Xtst(Ash(P0,I0))
    Xtst(Ash(P0,P0))
    Xtst(Ash(P0,M0))
    Xtst(Ash(P0,B0))
    Xtst(Ash(M0,TT))
    Xtst(Ash(M0,FF))
    Xtst(Ash(M0,I0))
    Xtst(Ash(M0,P0))
    Xtst(Ash(M0,M0))
    Xtst(Ash(M0,B0))
    Xtst(Ash(B0,TT))
    Xtst(Ash(B0,FF))
    Xtst(Ash(B0,I0))
    Xtst(Ash(B0,P0))
    Xtst(Ash(B0,M0))
    Xtst(Ash(B0,B0))

    Etst(Ash(N(-1000000000000),N(-65)),N(-1))
    Etst(Ash(N(   -1000000000),N(-65)),N(-1))
    Etst(Ash(N(      -1000000),N(-65)),N(-1))
    Etst(Ash(N(         -1000),N(-65)),N(-1))
    Etst(Ash(N(          -100),N(-65)),N(-1))
    Etst(Ash(N(           -10),N(-65)),N(-1))
    Etst(Ash(N(            -1),N(-65)),N(-1))
    Etst(Ash(N(             0),N(-65)),N( 0))
    Etst(Ash(N(             1),N(-65)),N( 0))
    Etst(Ash(N(            10),N(-65)),N( 0))
    Etst(Ash(N(           100),N(-65)),N( 0))
    Etst(Ash(N(          1000),N(-65)),N( 0))
    Etst(Ash(N(       1000000),N(-65)),N( 0))
    Etst(Ash(N(    1000000000),N(-65)),N( 0))
    Etst(Ash(N( 1000000000000),N(-65)),N( 0))

    Etst(Ash(N(-1000000000000),N(-64)),N(-1))
    Etst(Ash(N(   -1000000000),N(-64)),N(-1))
    Etst(Ash(N(      -1000000),N(-64)),N(-1))
    Etst(Ash(N(         -1000),N(-64)),N(-1))
    Etst(Ash(N(          -100),N(-64)),N(-1))
    Etst(Ash(N(           -10),N(-64)),N(-1))
    Etst(Ash(N(            -1),N(-64)),N(-1))
    Etst(Ash(N(             0),N(-64)),N( 0))
    Etst(Ash(N(             1),N(-64)),N( 0))
    Etst(Ash(N(            10),N(-64)),N( 0))
    Etst(Ash(N(           100),N(-64)),N( 0))
    Etst(Ash(N(          1000),N(-64)),N( 0))
    Etst(Ash(N(       1000000),N(-64)),N( 0))
    Etst(Ash(N(    1000000000),N(-64)),N( 0))
    Etst(Ash(N( 1000000000000),N(-64)),N( 0))

    Etst(Ash(N(-1000000000000),N(-63)),N(-1))
    Etst(Ash(N(   -1000000000),N(-63)),N(-1))
    Etst(Ash(N(      -1000000),N(-63)),N(-1))
    Etst(Ash(N(         -1000),N(-63)),N(-1))
    Etst(Ash(N(          -100),N(-63)),N(-1))
    Etst(Ash(N(           -10),N(-63)),N(-1))
    Etst(Ash(N(            -1),N(-63)),N(-1))
    Etst(Ash(N(             0),N(-63)),N( 0))
    Etst(Ash(N(             1),N(-63)),N( 0))
    Etst(Ash(N(            10),N(-63)),N( 0))
    Etst(Ash(N(           100),N(-63)),N( 0))
    Etst(Ash(N(          1000),N(-63)),N( 0))
    Etst(Ash(N(       1000000),N(-63)),N( 0))
    Etst(Ash(N(    1000000000),N(-63)),N( 0))
    Etst(Ash(N( 1000000000000),N(-63)),N( 0))

    Etst(Ash(N(-1000000000000),N( -4)),N(-62500000000))
    Etst(Ash(N(   -1000000000),N( -4)),N(   -62500000))
    Etst(Ash(N(      -1000000),N( -4)),N(      -62500))
    Etst(Ash(N(         -1000),N( -4)),N(         -63))
    Etst(Ash(N(          -100),N( -4)),N(          -7))
    Etst(Ash(N(           -10),N( -4)),N(          -1))
    Etst(Ash(N(            -1),N( -4)),N(          -1))
    Etst(Ash(N(             0),N( -4)),N(           0))
    Etst(Ash(N(             1),N( -4)),N(           0))
    Etst(Ash(N(            10),N( -4)),N(           0))
    Etst(Ash(N(           100),N( -4)),N(           6))
    Etst(Ash(N(          1000),N( -4)),N(          62))
    Etst(Ash(N(       1000000),N( -4)),N(       62500))
    Etst(Ash(N(    1000000000),N( -4)),N(    62500000))
    Etst(Ash(N( 1000000000000),N( -4)),N( 62500000000))

    Etst(Ash(N(-1000000000000),N( -3)),N(-125000000000))
    Etst(Ash(N(   -1000000000),N( -3)),N(   -125000000))
    Etst(Ash(N(      -1000000),N( -3)),N(      -125000))
    Etst(Ash(N(         -1000),N( -3)),N(         -125))
    Etst(Ash(N(          -100),N( -3)),N(          -13))
    Etst(Ash(N(           -10),N( -3)),N(           -2))
    Etst(Ash(N(            -1),N( -3)),N(           -1))
    Etst(Ash(N(             0),N( -3)),N(            0))
    Etst(Ash(N(             1),N( -3)),N(            0))
    Etst(Ash(N(            10),N( -3)),N(            1))
    Etst(Ash(N(           100),N( -3)),N(           12))
    Etst(Ash(N(          1000),N( -3)),N(          125))
    Etst(Ash(N(       1000000),N( -3)),N(       125000))
    Etst(Ash(N(    1000000000),N( -3)),N(    125000000))
    Etst(Ash(N( 1000000000000),N( -3)),N( 125000000000))

    Etst(Ash(N(-1000000000000),N( -2)),N(-250000000000))
    Etst(Ash(N(   -1000000000),N( -2)),N(   -250000000))
    Etst(Ash(N(      -1000000),N( -2)),N(      -250000))
    Etst(Ash(N(         -1000),N( -2)),N(         -250))
    Etst(Ash(N(          -100),N( -2)),N(          -25))
    Etst(Ash(N(           -10),N( -2)),N(           -3))
    Etst(Ash(N(            -1),N( -2)),N(           -1))
    Etst(Ash(N(             0),N( -2)),N(            0))
    Etst(Ash(N(             1),N( -2)),N(            0))
    Etst(Ash(N(            10),N( -2)),N(            2))
    Etst(Ash(N(           100),N( -2)),N(           25))
    Etst(Ash(N(          1000),N( -2)),N(          250))
    Etst(Ash(N(       1000000),N( -2)),N(       250000))
    Etst(Ash(N(    1000000000),N( -2)),N(    250000000))
    Etst(Ash(N( 1000000000000),N( -2)),N( 250000000000))

    Etst(Ash(N(-1000000000000),N( -1)),N(-500000000000))
    Etst(Ash(N(   -1000000000),N( -1)),N(   -500000000))
    Etst(Ash(N(      -1000000),N( -1)),N(      -500000))
    Etst(Ash(N(         -1000),N( -1)),N(         -500))
    Etst(Ash(N(          -100),N( -1)),N(          -50))
    Etst(Ash(N(           -10),N( -1)),N(           -5))
    Etst(Ash(N(            -1),N( -1)),N(           -1))
    Etst(Ash(N(             0),N( -1)),N(            0))
    Etst(Ash(N(             1),N( -1)),N(            0))
    Etst(Ash(N(            10),N( -1)),N(            5))
    Etst(Ash(N(           100),N( -1)),N(           50))
    Etst(Ash(N(          1000),N( -1)),N(          500))
    Etst(Ash(N(       1000000),N( -1)),N(       500000))
    Etst(Ash(N(    1000000000),N( -1)),N(    500000000))
    Etst(Ash(N( 1000000000000),N( -1)),N( 500000000000))

    Etst(Ash(N(-1000000000000),N(  0)),N(-1000000000000))
    Etst(Ash(N(   -1000000000),N(  0)),N(   -1000000000))
    Etst(Ash(N(      -1000000),N(  0)),N(      -1000000))
    Etst(Ash(N(         -1000),N(  0)),N(         -1000))
    Etst(Ash(N(          -100),N(  0)),N(          -100))
    Etst(Ash(N(           -10),N(  0)),N(           -10))
    Etst(Ash(N(            -1),N(  0)),N(            -1))
    Etst(Ash(N(             0),N(  0)),N(             0))
    Etst(Ash(N(             1),N(  0)),N(             1))
    Etst(Ash(N(            10),N(  0)),N(            10))
    Etst(Ash(N(           100),N(  0)),N(           100))
    Etst(Ash(N(          1000),N(  0)),N(          1000))
    Etst(Ash(N(       1000000),N(  0)),N(       1000000))
    Etst(Ash(N(    1000000000),N(  0)),N(    1000000000))
    Etst(Ash(N( 1000000000000),N(  0)),N( 1000000000000))

    Etst(Ash(N( 1000000000000),N( 65)),N( 36893488147419103232000000000000))
    Etst(Ash(N(    1000000000),N( 65)),N(    36893488147419103232000000000))
    Etst(Ash(N(       1000000),N( 65)),N(       36893488147419103232000000))
    Etst(Ash(N(          1000),N( 65)),N(          36893488147419103232000))
    Etst(Ash(N(           100),N( 65)),N(           3689348814741910323200))
    Etst(Ash(N(            10),N( 65)),N(            368934881474191032320))
    Etst(Ash(N(             1),N( 65)),N(             36893488147419103232))
    Etst(Ash(N(             0),N( 65)),N(                                0))
    Etst(Ash(N(            -1),N( 65)),N(            -36893488147419103232))
    Etst(Ash(N(           -10),N( 65)),N(           -368934881474191032320))
    Etst(Ash(N(          -100),N( 65)),N(          -3689348814741910323200))
    Etst(Ash(N(         -1000),N( 65)),N(         -36893488147419103232000))
    Etst(Ash(N(      -1000000),N( 65)),N(      -36893488147419103232000000))
    Etst(Ash(N(   -1000000000),N( 65)),N(   -36893488147419103232000000000))
    Etst(Ash(N(-1000000000000),N( 65)),N(-36893488147419103232000000000000))

    Etst(Ash(N( 1000000000000),N( 64)),N( 18446744073709551616000000000000))
    Etst(Ash(N(    1000000000),N( 64)),N(    18446744073709551616000000000))
    Etst(Ash(N(       1000000),N( 64)),N(       18446744073709551616000000))
    Etst(Ash(N(          1000),N( 64)),N(          18446744073709551616000))
    Etst(Ash(N(           100),N( 64)),N(           1844674407370955161600))
    Etst(Ash(N(            10),N( 64)),N(            184467440737095516160))
    Etst(Ash(N(             1),N( 64)),N(             18446744073709551616))
    Etst(Ash(N(             0),N( 64)),N(                                0))
    Etst(Ash(N(            -1),N( 64)),N(            -18446744073709551616))
    Etst(Ash(N(           -10),N( 64)),N(           -184467440737095516160))
    Etst(Ash(N(          -100),N( 64)),N(          -1844674407370955161600))
    Etst(Ash(N(         -1000),N( 64)),N(         -18446744073709551616000))
    Etst(Ash(N(      -1000000),N( 64)),N(      -18446744073709551616000000))
    Etst(Ash(N(   -1000000000),N( 64)),N(   -18446744073709551616000000000))
    Etst(Ash(N(-1000000000000),N( 64)),N(-18446744073709551616000000000000))

    Etst(Ash(N( 1000000000000),N( 63)),N( 9223372036854775808000000000000))
    Etst(Ash(N(    1000000000),N( 63)),N(    9223372036854775808000000000))
    Etst(Ash(N(       1000000),N( 63)),N(       9223372036854775808000000))
    Etst(Ash(N(          1000),N( 63)),N(          9223372036854775808000))
    Etst(Ash(N(           100),N( 63)),N(           922337203685477580800))
    Etst(Ash(N(            10),N( 63)),N(            92233720368547758080))
    Etst(Ash(N(             1),N( 63)),N(             9223372036854775808))
    Etst(Ash(N(             0),N( 63)),N(                               0))
    Etst(Ash(N(            -1),N( 63)),N(            -9223372036854775808))
    Etst(Ash(N(           -10),N( 63)),N(           -92233720368547758080))
    Etst(Ash(N(          -100),N( 63)),N(          -922337203685477580800))
    Etst(Ash(N(         -1000),N( 63)),N(         -9223372036854775808000))
    Etst(Ash(N(      -1000000),N( 63)),N(      -9223372036854775808000000))
    Etst(Ash(N(   -1000000000),N( 63)),N(   -9223372036854775808000000000))
    Etst(Ash(N(-1000000000000),N( 63)),N(-9223372036854775808000000000000))

    Etst(Ash(N( 1000000000000),N(  4)),N( 16000000000000))
    Etst(Ash(N(    1000000000),N(  4)),N(    16000000000))
    Etst(Ash(N(       1000000),N(  4)),N(       16000000))
    Etst(Ash(N(          1000),N(  4)),N(          16000))
    Etst(Ash(N(           100),N(  4)),N(           1600))
    Etst(Ash(N(            10),N(  4)),N(            160))
    Etst(Ash(N(             1),N(  4)),N(             16))
    Etst(Ash(N(             0),N(  4)),N(              0))
    Etst(Ash(N(            -1),N(  4)),N(            -16))
    Etst(Ash(N(           -10),N(  4)),N(           -160))
    Etst(Ash(N(          -100),N(  4)),N(          -1600))
    Etst(Ash(N(         -1000),N(  4)),N(         -16000))
    Etst(Ash(N(      -1000000),N(  4)),N(      -16000000))
    Etst(Ash(N(   -1000000000),N(  4)),N(   -16000000000))
    Etst(Ash(N(-1000000000000),N(  4)),N(-16000000000000))

    Etst(Ash(N( 1000000000000),N(  3)),N( 8000000000000))
    Etst(Ash(N(    1000000000),N(  3)),N(    8000000000))
    Etst(Ash(N(       1000000),N(  3)),N(       8000000))
    Etst(Ash(N(          1000),N(  3)),N(          8000))
    Etst(Ash(N(           100),N(  3)),N(           800))
    Etst(Ash(N(            10),N(  3)),N(            80))
    Etst(Ash(N(             1),N(  3)),N(             8))
    Etst(Ash(N(             0),N(  3)),N(             0))
    Etst(Ash(N(            -1),N(  3)),N(            -8))
    Etst(Ash(N(           -10),N(  3)),N(           -80))
    Etst(Ash(N(          -100),N(  3)),N(          -800))
    Etst(Ash(N(         -1000),N(  3)),N(         -8000))
    Etst(Ash(N(      -1000000),N(  3)),N(      -8000000))
    Etst(Ash(N(   -1000000000),N(  3)),N(   -8000000000))
    Etst(Ash(N(-1000000000000),N(  3)),N(-8000000000000))

    Etst(Ash(N( 1000000000000),N(  2)),N( 4000000000000))
    Etst(Ash(N(    1000000000),N(  2)),N(    4000000000))
    Etst(Ash(N(       1000000),N(  2)),N(       4000000))
    Etst(Ash(N(          1000),N(  2)),N(          4000))
    Etst(Ash(N(           100),N(  2)),N(           400))
    Etst(Ash(N(            10),N(  2)),N(            40))
    Etst(Ash(N(             1),N(  2)),N(             4))
    Etst(Ash(N(             0),N(  2)),N(             0))
    Etst(Ash(N(            -1),N(  2)),N(            -4))
    Etst(Ash(N(           -10),N(  2)),N(           -40))
    Etst(Ash(N(          -100),N(  2)),N(          -400))
    Etst(Ash(N(         -1000),N(  2)),N(         -4000))
    Etst(Ash(N(      -1000000),N(  2)),N(      -4000000))
    Etst(Ash(N(   -1000000000),N(  2)),N(   -4000000000))
    Etst(Ash(N(-1000000000000),N(  2)),N(-4000000000000))

    Etst(Ash(N( 1000000000000),N(  1)),N( 2000000000000))
    Etst(Ash(N(    1000000000),N(  1)),N(    2000000000))
    Etst(Ash(N(       1000000),N(  1)),N(       2000000))
    Etst(Ash(N(          1000),N(  1)),N(          2000))
    Etst(Ash(N(           100),N(  1)),N(           200))
    Etst(Ash(N(            10),N(  1)),N(            20))
    Etst(Ash(N(             1),N(  1)),N(             2))
    Etst(Ash(N(             0),N(  1)),N(             0))
    Etst(Ash(N(            -1),N(  1)),N(            -2))
    Etst(Ash(N(           -10),N(  1)),N(           -20))
    Etst(Ash(N(          -100),N(  1)),N(          -200))
    Etst(Ash(N(         -1000),N(  1)),N(         -2000))
    Etst(Ash(N(      -1000000),N(  1)),N(      -2000000))
    Etst(Ash(N(   -1000000000),N(  1)),N(   -2000000000))
    Etst(Ash(N(-1000000000000),N(  1)),N(-2000000000000))

    Etst(Ash(N( 36893488147419103232000000000000),N(-65)),N( 1000000000000))
    Etst(Ash(N(    36893488147419103232000000000),N(-65)),N(    1000000000))
    Etst(Ash(N(       36893488147419103232000000),N(-65)),N(       1000000))
    Etst(Ash(N(          36893488147419103232000),N(-65)),N(          1000))
    Etst(Ash(N(           3689348814741910323200),N(-65)),N(           100))
    Etst(Ash(N(            368934881474191032320),N(-65)),N(            10))
    Etst(Ash(N(             36893488147419103232),N(-65)),N(             1))
    Etst(Ash(N(                                0),N(-65)),N(             0))
    Etst(Ash(N(            -36893488147419103232),N(-65)),N(            -1))
    Etst(Ash(N(           -368934881474191032320),N(-65)),N(           -10))
    Etst(Ash(N(          -3689348814741910323200),N(-65)),N(          -100))
    Etst(Ash(N(         -36893488147419103232000),N(-65)),N(         -1000))
    Etst(Ash(N(      -36893488147419103232000000),N(-65)),N(      -1000000))
    Etst(Ash(N(   -36893488147419103232000000000),N(-65)),N(   -1000000000))
    Etst(Ash(N(-36893488147419103232000000000000),N(-65)),N(-1000000000000))

    Etst(Ash(N( 18446744073709551616000000000000),N(-64)),N( 1000000000000))
    Etst(Ash(N(    18446744073709551616000000000),N(-64)),N(    1000000000))
    Etst(Ash(N(       18446744073709551616000000),N(-64)),N(       1000000))
    Etst(Ash(N(          18446744073709551616000),N(-64)),N(          1000))
    Etst(Ash(N(           1844674407370955161600),N(-64)),N(           100))
    Etst(Ash(N(            184467440737095516160),N(-64)),N(            10))
    Etst(Ash(N(             18446744073709551616),N(-64)),N(             1))
    Etst(Ash(N(                                0),N(-64)),N(             0))
    Etst(Ash(N(            -18446744073709551616),N(-64)),N(            -1))
    Etst(Ash(N(           -184467440737095516160),N(-64)),N(           -10))
    Etst(Ash(N(          -1844674407370955161600),N(-64)),N(          -100))
    Etst(Ash(N(         -18446744073709551616000),N(-64)),N(         -1000))
    Etst(Ash(N(      -18446744073709551616000000),N(-64)),N(      -1000000))
    Etst(Ash(N(   -18446744073709551616000000000),N(-64)),N(   -1000000000))
    Etst(Ash(N(-18446744073709551616000000000000),N(-64)),N(-1000000000000))

    Etst(Ash(N(  9223372036854775808000000000000),N(-63)),N( 1000000000000))
    Etst(Ash(N(     9223372036854775808000000000),N(-63)),N(    1000000000))
    Etst(Ash(N(        9223372036854775808000000),N(-63)),N(       1000000))
    Etst(Ash(N(           9223372036854775808000),N(-63)),N(          1000))
    Etst(Ash(N(            922337203685477580800),N(-63)),N(           100))
    Etst(Ash(N(             92233720368547758080),N(-63)),N(            10))
    Etst(Ash(N(              9223372036854775808),N(-63)),N(             1))
    Etst(Ash(N(                                0),N(-63)),N(             0))
    Etst(Ash(N(             -9223372036854775808),N(-63)),N(            -1))
    Etst(Ash(N(            -92233720368547758080),N(-63)),N(           -10))
    Etst(Ash(N(           -922337203685477580800),N(-63)),N(          -100))
    Etst(Ash(N(          -9223372036854775808000),N(-63)),N(         -1000))
    Etst(Ash(N(       -9223372036854775808000000),N(-63)),N(      -1000000))
    Etst(Ash(N(    -9223372036854775808000000000),N(-63)),N(   -1000000000))
    Etst(Ash(N( -9223372036854775808000000000000),N(-63)),N(-1000000000000))

    Ttst(Boolp(TT))
    Ttst(Boolp(FF))
    Ftst(Boolp(I0))
    Ftst(Boolp(P0))
    Ftst(Boolp(M0))
    Ftst(Boolp(B0))

    Ftst(Intp(TT))
    Ftst(Intp(FF))
    Ttst(Intp(I0))
    Ftst(Intp(P0))
    Ftst(Intp(M0))
    Ftst(Intp(B0))

    Ftst(Pairp(TT))
    Ftst(Pairp(FF))
    Ftst(Pairp(I0))
    Ttst(Pairp(P0))
    Ftst(Pairp(M0))
    Ftst(Pairp(B0))

    etst(X,untag1(Raise(TT)))
    utst(X,untag1(Raise(FF)))
    utst(X,untag1(Raise(I0)))
    utst(X,untag1(Raise(P0)))
    utst(X,untag1(Raise(M0)))
    utst(X,untag1(Raise(B0)))
    utst(TT,untag1(Raise(FF)))
    utst(FF,untag1(Raise(FF)))
    utst(I0,untag1(Raise(FF)))
    utst(P0,untag1(Raise(FF)))
    utst(M0,untag1(Raise(FF)))
    utst(B0,untag1(Raise(FF)))
    etst(untag1(Raise(pair(TT,TT))),untag1(Raise(pair(TT,TT))))
    utst(untag1(Raise(pair(TT,TT))),untag1(Raise(pair(TT,FF))))
    utst(untag1(Raise(pair(TT,FF))),untag1(Raise(pair(TT,TT))))
    etst(untag1(Raise(pair(TT,FF))),untag1(Raise(pair(TT,FF))))

    Ftst(Mapp(TT))
    Ftst(Mapp(FF))
    Ftst(Mapp(I0))
    Ftst(Mapp(P0))
    Ttst(Mapp(M0))
    Ftst(Mapp(B0))

    Ftst(Objp(TT))
    Ftst(Objp(FF))
    Ftst(Objp(I0))
    Ftst(Objp(P0))
    Ftst(Objp(M0))
    Ttst(Objp(B0))

    Xtst(Object(TT))
    Xtst(Object(FF))
    Xtst(Object(I0))
    Xtst(Object(M0))
    Xtst(Object(B0))
    Xtst(Object(Pair(TT,P0)))
    Xtst(Object(Pair(FF,P0)))
    Xtst(Object(Pair(I0,P0)))
    Xtst(Object(Pair(M0,P0)))
    Xtst(Object(Pair(B0,P0)))
    Xtst(Object(Pair(Pair(I[1],TT),P0)))
    Xtst(Object(Pair(Pair(I[1],FF),P0)))
    Xtst(Object(Pair(Pair(I[1],P0),P0)))
    Xtst(Object(Pair(Pair(I[1],M0),P0)))
    Xtst(Object(Pair(Pair(I[1],B0),P0)))
    Xtst(Object(Pair(Pair(TT,I[1]),P0)))
    Xtst(Object(Pair(Pair(FF,I[1]),P0)))
    Xtst(Object(Pair(Pair(P0,I[1]),P0)))
    Xtst(Object(Pair(Pair(M0,I[1]),P0)))
    Xtst(Object(Pair(Pair(B0,I[1]),P0)))
    Xtst(Object(Pair(Pair(M,I[1]),P0)))
    Xtst(Object(Pair(Pair(I[1],M),P0)))
    Xtst(Object(Pair(Pair(Z,I[0]),P0)))
    Xtst(Object(Pair(Pair(Z,I[1]),P0)))
    Xtst(Object(Pair(Pair(Z,I[2]),P0)))
    Xtst(Object(Pair(Pair(Z,I[3]),P0)))
    Xtst(Object(Pair(Pair(Z,I[4]),P0)))
    Ttst(Objp(Object(Pair(Pair(I[0],I[5]),P0))))
    Ftst(Pairp(Object(Pair(Pair(I[0],I[5]),P0))))
    Ttst(Objp(Object(Pair(Pair(I[0],I[6]),P0))))
    Ttst(Objp(Object(Pair(Pair(I[0],I[7]),P0))))
    Ttst(Objp(Object(Pair(Pair(I[1],I[0]),P0))))
    Ttst(Objp(Object(Pair(Pair(I[1],I[4]),P0))))
    Ttst(Objp(Object(Pair(Pair(I[1],I[7]),TT))))
    Ttst(Objp(Object(Pair(Pair(I[1],I[7]),FF))))
    Ttst(Objp(Object(Pair(Pair(I[1],I[7]),I0))))
    Ttst(Objp(Object(Pair(Pair(I[1],I[7]),P0))))
    Ttst(Objp(Object(Pair(Pair(I[1],I[7]),M0))))
    Ttst(Objp(Object(Pair(Pair(I[1],I[7]),B0))))

    Xtst(Destruct(TT))
    Xtst(Destruct(FF))
    Xtst(Destruct(P0))
    Xtst(Destruct(M0))
    Xtst(Destruct(Z))
    Xtst(Destruct(M))
    Xtst(Destruct(I0011))
    Xtst(Destruct(J1010_0101))
    Etst(Destruct(Object(Pair(Pair(I[0],I[5]),P0))),Pair(Pair(I[0],I[5]),P0))
    Etst(Destruct(Object(Pair(Pair(I[0],I[6]),P0))),Pair(Pair(I[0],I[6]),P0))
    Etst(Destruct(Object(Pair(Pair(I[0],I[7]),P0))),Pair(Pair(I[0],I[7]),P0))
    Etst(Destruct(Object(Pair(Pair(I[1],I[0]),P0))),Pair(Pair(I[1],I[0]),P0))
    Etst(Destruct(Object(Pair(Pair(I[1],I[4]),P0))),Pair(Pair(I[1],I[4]),P0))
    Etst(Destruct(Object(Pair(Pair(I[1],I[7]),TT))),Pair(Pair(I[1],I[7]),TT))
    Etst(Destruct(Object(Pair(Pair(I[1],I[7]),FF))),Pair(Pair(I[1],I[7]),FF))
    Etst(Destruct(Object(Pair(Pair(I[1],I[7]),I0))),Pair(Pair(I[1],I[7]),I0))
    Etst(Destruct(Object(Pair(Pair(I[1],I[7]),P0))),Pair(Pair(I[1],I[7]),P0))
    Etst(Destruct(Object(Pair(Pair(I[1],I[7]),M0))),Pair(Pair(I[1],I[7]),M0))
    Etst(Destruct(Object(Pair(Pair(I[1],I[7]),B0))),Pair(Pair(I[1],I[7]),B0))

    Xtst(Vector(TT))
    Xtst(Vector(FF))
    Xtst(Vector(P0))
    Xtst(Vector(M0))
    Xtst(Vector(B0))

    Xtst(Vempty(TT))
    Xtst(Vempty(FF))
    Xtst(Vempty(P0))
    Xtst(Vempty(M0))
    Xtst(Vempty(B0))
    Ttst(Vempty(N(0)))
    Ttst(Vempty(N(1)))
    Ttst(Vempty(N(2)))
    Ttst(Vempty(N(255)))
    Ftst(Vempty(N(256)))
    Ftst(Vempty(N(257)))
    Ftst(Vempty(N(511)))
    Ftst(Vempty(N(512)))
    Ftst(Vempty(N(513)))
    Ftst(Vempty(N(767)))
    Ftst(Vempty(intcons(123,intcons(456,I[2]))))
    Ftst(Vempty(intcons(123,intcons(456,N(767)))))
    Ttst(Vempty(intcons(123,intcons(456,M))))

    Etst(Vector(N(0)),N(1))
    Etst(Vector(N(1)),N(1))
    Etst(Vector(N(2)),N(1))
    Etst(Vector(N(255)),N(1))
    Etst(Vector(N(256)),N(256))
    Etst(Vector(N(257)),N(257))
    Etst(Vector(N(511)),N(511))
    Etst(Vector(N(512)),N(256))
    Etst(Vector(N(513)),N(257))
    Etst(Vector(N(767)),N(511))
    Etst(Vector(intcons(123,intcons(456,I[2]))),intcons(123,intcons(456,I[1])))
    Etst(Vector(intcons(123,intcons(456,N(767)))),
         intcons(123,intcons(456,N(511))))

    Xtst(Vlength(TT))
    Xtst(Vlength(FF))
    Xtst(Vlength(P0))
    Xtst(Vlength(M0))
    Xtst(Vlength(B0))
    Etst(N(0),Vlength(N(0)))
    Etst(N(0),Vlength(N(1)))
    Etst(N(0),Vlength(N(2)))
    Etst(N(0),Vlength(N(255)))
    Etst(N(1),Vlength(N(256)))
    Etst(N(1),Vlength(N(257)))
    Etst(N(1),Vlength(N(511)))
    Etst(N(1),Vlength(N(512)))
    Etst(N(1),Vlength(N(513)))
    Etst(N(1),Vlength(N(767)))
    Etst(N(1),Vlength(N(65535)))
    Etst(N(2),Vlength(N(65536)))
    Etst(N(3),Vlength(N(4294967295)))
    Etst(N(4),Vlength(N(4294967296)))
    Etst(N(7),Vlength(N(18446744073709551615)))
    Etst(N(8),Vlength(N(18446744073709551616)))
    Etst(N(15),Vlength(N(340282366920938463463374607431768211455)))
    Etst(N(16),Vlength(N(340282366920938463463374607431768211456)))
    Etst(N(0),Vlength(intcons(123,intcons(456,M))))

    Xtst(Vindex(TT,TT))
    Xtst(Vindex(TT,FF))
    Xtst(Vindex(TT,I0))
    Xtst(Vindex(TT,P0))
    Xtst(Vindex(TT,M0))
    Xtst(Vindex(TT,B0))
    Xtst(Vindex(FF,TT))
    Xtst(Vindex(FF,FF))
    Xtst(Vindex(FF,I0))
    Xtst(Vindex(FF,P0))
    Xtst(Vindex(FF,M0))
    Xtst(Vindex(FF,B0))
    Xtst(Vindex(I0,TT))
    Xtst(Vindex(I0,FF))
    Xtst(Vindex(I0,P0))
    Xtst(Vindex(I0,M0))
    Xtst(Vindex(I0,B0))
    Xtst(Vindex(P0,TT))
    Xtst(Vindex(P0,FF))
    Xtst(Vindex(P0,I0))
    Xtst(Vindex(P0,P0))
    Xtst(Vindex(P0,M0))
    Xtst(Vindex(P0,B0))
    Xtst(Vindex(M0,TT))
    Xtst(Vindex(M0,FF))
    Xtst(Vindex(M0,I0))
    Xtst(Vindex(M0,P0))
    Xtst(Vindex(M0,M0))
    Xtst(Vindex(M0,B0))
    Xtst(Vindex(B0,TT))
    Xtst(Vindex(B0,FF))
    Xtst(Vindex(B0,I0))
    Xtst(Vindex(B0,P0))
    Xtst(Vindex(B0,M0))
    Xtst(Vindex(B0,B0))
    Xtst(Vindex(N(-1),N(-1)))
    Xtst(Vindex(N(-1),N( 0)))
    Xtst(Vindex(N(-1),N( 1)))
    Xtst(Vindex(N(0),N(-1)))
    Xtst(Vindex(N(0),N( 0)))
    Xtst(Vindex(N(0),N( 1)))
    Xtst(Vindex(N(255),N(-1)))
    Xtst(Vindex(N(255),N( 0)))
    Xtst(Vindex(N(255),N( 1)))
    Xtst(Vindex(N(256),N(-1)))
    Etst(Vindex(N(256),N( 0)),N(0))
    Xtst(Vindex(N(256),N( 1)))
    Xtst(Vindex(N(257),N(-1)))
    Etst(Vindex(N(257),N( 0)),N(1))
    Xtst(Vindex(N(257),N( 1)))
    Xtst(Vindex(N(66049),N(-1)))
    Etst(Vindex(N(66049),N( 0)),N(1))
    Etst(Vindex(N(66049),N( 1)),N(2))
    Xtst(Vindex(N(66049),N( 2)))
    Xtst(Vindex(N(19025181769461858817),N(-1)))
    Etst(Vindex(N(19025181769461858817),N( 0)),N(1))
    Etst(Vindex(N(19025181769461858817),N( 1)),N(2))
    Etst(Vindex(N(19025181769461858817),N( 2)),N(3))
    Etst(Vindex(N(19025181769461858817),N( 3)),N(4))
    Etst(Vindex(N(19025181769461858817),N( 4)),N(5))
    Etst(Vindex(N(19025181769461858817),N( 5)),N(6))
    Etst(Vindex(N(19025181769461858817),N( 6)),N(7))
    Etst(Vindex(N(19025181769461858817),N( 7)),N(8))
    Xtst(Vindex(N(19025181769461858817),N( 8)))
    Xtst(Vindex(N(4888965617228783485441),N(-1)))
    Etst(Vindex(N(4888965617228783485441),N( 0)),N(1))
    Etst(Vindex(N(4888965617228783485441),N( 1)),N(2))
    Etst(Vindex(N(4888965617228783485441),N( 2)),N(3))
    Etst(Vindex(N(4888965617228783485441),N( 3)),N(4))
    Etst(Vindex(N(4888965617228783485441),N( 4)),N(5))
    Etst(Vindex(N(4888965617228783485441),N( 5)),N(6))
    Etst(Vindex(N(4888965617228783485441),N( 6)),N(7))
    Etst(Vindex(N(4888965617228783485441),N( 7)),N(8))
    Etst(Vindex(N(4888965617228783485441),N( 8)),N(9))
    Xtst(Vindex(N(4888965617228783485441),N( 9)))
    Etst(Vindex(V(abc),N(0)),N(97))
    Etst(Vindex(V(abc),N(1)),N(98))
    Etst(Vindex(V(abc),N(2)),N(99))

    Xtst(Vsuffix(TT,TT))
    Xtst(Vsuffix(TT,FF))
    Xtst(Vsuffix(TT,I0))
    Xtst(Vsuffix(TT,P0))
    Xtst(Vsuffix(TT,M0))
    Xtst(Vsuffix(TT,B0))
    Xtst(Vsuffix(FF,TT))
    Xtst(Vsuffix(FF,FF))
    Xtst(Vsuffix(FF,I0))
    Xtst(Vsuffix(FF,P0))
    Xtst(Vsuffix(FF,M0))
    Xtst(Vsuffix(FF,B0))
    Xtst(Vsuffix(I0,TT))
    Xtst(Vsuffix(I0,FF))
    Xtst(Vsuffix(I0,P0))
    Xtst(Vsuffix(I0,M0))
    Xtst(Vsuffix(I0,B0))
    Xtst(Vsuffix(P0,TT))
    Xtst(Vsuffix(P0,FF))
    Xtst(Vsuffix(P0,I0))
    Xtst(Vsuffix(P0,P0))
    Xtst(Vsuffix(P0,M0))
    Xtst(Vsuffix(P0,B0))
    Xtst(Vsuffix(M0,TT))
    Xtst(Vsuffix(M0,FF))
    Xtst(Vsuffix(M0,I0))
    Xtst(Vsuffix(M0,P0))
    Xtst(Vsuffix(M0,M0))
    Xtst(Vsuffix(M0,B0))
    Xtst(Vsuffix(B0,TT))
    Xtst(Vsuffix(B0,FF))
    Xtst(Vsuffix(B0,I0))
    Xtst(Vsuffix(B0,P0))
    Xtst(Vsuffix(B0,M0))
    Xtst(Vsuffix(B0,B0))
    Etst(Vsuffix(V(abc),N(-1)),V(abc))
    Etst(Vsuffix(V(abc),N( 0)),V(abc))
    Etst(Vsuffix(V(abc),N( 1)),V(bc))
    Etst(Vsuffix(V(abc),N( 2)),V(c))
    Etst(Vsuffix(V(abc),N( 3)),E)
    Etst(Vsuffix(V(abc),N( 4)),E)
    Etst(Vsuffix(V(abcdefg),N(-1)),V(abcdefg))
    Etst(Vsuffix(V(abcdefg),N( 0)),V(abcdefg))
    Etst(Vsuffix(V(abcdefg),N( 1)),V(bcdefg))
    Etst(Vsuffix(V(abcdefg),N( 2)),V(cdefg))
    Etst(Vsuffix(V(abcdefg),N( 3)),V(defg))
    Etst(Vsuffix(V(abcdefg),N( 4)),V(efg))
    Etst(Vsuffix(V(abcdefg),N( 5)),V(fg))
    Etst(Vsuffix(V(abcdefg),N( 6)),V(g))
    Etst(Vsuffix(V(abcdefg),N( 7)),V())
    Etst(Vsuffix(V(abcdefg),N( 8)),V())
    Etst(Vsuffix(V(abcdefg),N( 9)),V())
    Etst(Vsuffix(V(abcdefg),N(10)),V())
    Etst(Vsuffix(V(abcdefgh),N(-1)),V(abcdefgh))
    Etst(Vsuffix(V(abcdefgh),N( 0)),V(abcdefgh))
    Etst(Vsuffix(V(abcdefgh),N( 1)),V(bcdefgh))
    Etst(Vsuffix(V(abcdefgh),N( 2)),V(cdefgh))
    Etst(Vsuffix(V(abcdefgh),N( 3)),V(defgh))
    Etst(Vsuffix(V(abcdefgh),N( 4)),V(efgh))
    Etst(Vsuffix(V(abcdefgh),N( 5)),V(fgh))
    Etst(Vsuffix(V(abcdefgh),N( 6)),V(gh))
    Etst(Vsuffix(V(abcdefgh),N( 7)),V(h))
    Etst(Vsuffix(V(abcdefgh),N( 8)),V())
    Etst(Vsuffix(V(abcdefgh),N( 9)),V())
    Etst(Vsuffix(V(abcdefgh),N(10)),V())
    Etst(Vsuffix(V(abcdefghi),N(-1)),V(abcdefghi))
    Etst(Vsuffix(V(abcdefghi),N( 0)),V(abcdefghi))
    Etst(Vsuffix(V(abcdefghi),N( 1)),V(bcdefghi))
    Etst(Vsuffix(V(abcdefghi),N( 2)),V(cdefghi))
    Etst(Vsuffix(V(abcdefghi),N( 3)),V(defghi))
    Etst(Vsuffix(V(abcdefghi),N( 4)),V(efghi))
    Etst(Vsuffix(V(abcdefghi),N( 5)),V(fghi))
    Etst(Vsuffix(V(abcdefghi),N( 6)),V(ghi))
    Etst(Vsuffix(V(abcdefghi),N( 7)),V(hi))
    Etst(Vsuffix(V(abcdefghi),N( 8)),V(i))
    Etst(Vsuffix(V(abcdefghi),N( 9)),V())
    Etst(Vsuffix(V(abcdefghi),N(10)),V())
    Etst(Vsuffix(N(256),N(-1)),N(256))
    Etst(Vsuffix(N(256),N( 0)),N(256))
    Etst(Vsuffix(N(256),N( 1)),E)
    Etst(Vsuffix(N(256),N( 2)),E)
    Etst(Vsuffix(N(512),N(-1)),N(256))
    Etst(Vsuffix(N(512),N( 0)),N(256))
    Etst(Vsuffix(N(512),N( 1)),E)
    Etst(Vsuffix(N(512),N( 2)),E)

    Xtst(Vprefix(TT,TT))
    Xtst(Vprefix(TT,FF))
    Xtst(Vprefix(TT,I0))
    Xtst(Vprefix(TT,P0))
    Xtst(Vprefix(TT,M0))
    Xtst(Vprefix(TT,B0))
    Xtst(Vprefix(FF,TT))
    Xtst(Vprefix(FF,FF))
    Xtst(Vprefix(FF,I0))
    Xtst(Vprefix(FF,P0))
    Xtst(Vprefix(FF,M0))
    Xtst(Vprefix(FF,B0))
    Xtst(Vprefix(I0,TT))
    Xtst(Vprefix(I0,FF))
    Xtst(Vprefix(I0,P0))
    Xtst(Vprefix(I0,M0))
    Xtst(Vprefix(I0,B0))
    Xtst(Vprefix(P0,TT))
    Xtst(Vprefix(P0,FF))
    Xtst(Vprefix(P0,I0))
    Xtst(Vprefix(P0,P0))
    Xtst(Vprefix(P0,M0))
    Xtst(Vprefix(P0,B0))
    Xtst(Vprefix(M0,TT))
    Xtst(Vprefix(M0,FF))
    Xtst(Vprefix(M0,I0))
    Xtst(Vprefix(M0,P0))
    Xtst(Vprefix(M0,M0))
    Xtst(Vprefix(M0,B0))
    Xtst(Vprefix(B0,TT))
    Xtst(Vprefix(B0,FF))
    Xtst(Vprefix(B0,I0))
    Xtst(Vprefix(B0,P0))
    Xtst(Vprefix(B0,M0))
    Xtst(Vprefix(B0,B0))
    Etst(Vprefix(V(abc),N(-1)),E)
    Etst(Vprefix(V(abc),N( 0)),E)
    Etst(Vprefix(V(abc),N( 1)),V(a))
    Etst(Vprefix(V(abc),N( 2)),V(ab))
    Etst(Vprefix(V(abc),N( 3)),V(abc))
    Etst(Vprefix(V(abc),N( 4)),V(abc))
    Etst(Vprefix(V(abcdefg),N(-1)),E)
    Etst(Vprefix(V(abcdefg),N( 0)),E)
    Etst(Vprefix(V(abcdefg),N( 1)),V(a))
    Etst(Vprefix(V(abcdefg),N( 2)),V(ab))
    Etst(Vprefix(V(abcdefg),N( 3)),V(abc))
    Etst(Vprefix(V(abcdefg),N( 4)),V(abcd))
    Etst(Vprefix(V(abcdefg),N( 5)),V(abcde))
    Etst(Vprefix(V(abcdefg),N( 6)),V(abcdef))
    Etst(Vprefix(V(abcdefg),N( 7)),V(abcdefg))
    Etst(Vprefix(V(abcdefg),N( 8)),V(abcdefg))
    Etst(Vprefix(V(abcdefg),N( 9)),V(abcdefg))
    Etst(Vprefix(V(abcdefg),N(10)),V(abcdefg))
    Etst(Vprefix(V(abcdefgh),N(-1)),E)
    Etst(Vprefix(V(abcdefgh),N( 0)),E)
    Etst(Vprefix(V(abcdefgh),N( 1)),V(a))
    Etst(Vprefix(V(abcdefgh),N( 2)),V(ab))
    Etst(Vprefix(V(abcdefgh),N( 3)),V(abc))
    Etst(Vprefix(V(abcdefgh),N( 4)),V(abcd))
    Etst(Vprefix(V(abcdefgh),N( 5)),V(abcde))
    Etst(Vprefix(V(abcdefgh),N( 6)),V(abcdef))
    Etst(Vprefix(V(abcdefgh),N( 7)),V(abcdefg))
    Etst(Vprefix(V(abcdefgh),N( 8)),V(abcdefgh))
    Etst(Vprefix(V(abcdefgh),N( 9)),V(abcdefgh))
    Etst(Vprefix(V(abcdefgh),N(10)),V(abcdefgh))
    Etst(Vprefix(V(abcdefghi),N(-1)),E)
    Etst(Vprefix(V(abcdefghi),N( 0)),E)
    Etst(Vprefix(V(abcdefghi),N( 1)),V(a))
    Etst(Vprefix(V(abcdefghi),N( 2)),V(ab))
    Etst(Vprefix(V(abcdefghi),N( 3)),V(abc))
    Etst(Vprefix(V(abcdefghi),N( 4)),V(abcd))
    Etst(Vprefix(V(abcdefghi),N( 5)),V(abcde))
    Etst(Vprefix(V(abcdefghi),N( 6)),V(abcdef))
    Etst(Vprefix(V(abcdefghi),N( 7)),V(abcdefg))
    Etst(Vprefix(V(abcdefghi),N( 8)),V(abcdefgh))
    Etst(Vprefix(V(abcdefghi),N( 9)),V(abcdefghi))
    Etst(Vprefix(V(abcdefghi),N(10)),V(abcdefghi))
    Etst(Vprefix(N(256),N(-1)),E)
    Etst(Vprefix(N(256),N( 0)),E)
    Etst(Vprefix(N(256),N( 1)),N(256))
    Etst(Vprefix(N(256),N( 2)),N(256))
    Etst(Vprefix(N(512),N(-1)),E)
    Etst(Vprefix(N(512),N( 0)),E)
    Etst(Vprefix(N(512),N( 1)),N(256))
    Etst(Vprefix(N(512),N( 2)),N(256))

    Xtst(Vsubseq(V(abc),TT,TT))
    Xtst(Vsubseq(V(abc),TT,FF))
    Xtst(Vsubseq(V(abc),TT,I0))
    Xtst(Vsubseq(V(abc),TT,P0))
    Xtst(Vsubseq(V(abc),TT,M0))
    Xtst(Vsubseq(V(abc),TT,B0))
    Xtst(Vsubseq(V(abc),FF,TT))
    Xtst(Vsubseq(V(abc),FF,FF))
    Xtst(Vsubseq(V(abc),FF,I0))
    Xtst(Vsubseq(V(abc),FF,P0))
    Xtst(Vsubseq(V(abc),FF,M0))
    Xtst(Vsubseq(V(abc),FF,B0))
    Xtst(Vsubseq(V(abc),I0,TT))
    Xtst(Vsubseq(V(abc),I0,FF))
    Xtst(Vsubseq(V(abc),I0,P0))
    Xtst(Vsubseq(V(abc),I0,M0))
    Xtst(Vsubseq(V(abc),I0,B0))
    Xtst(Vsubseq(V(abc),P0,TT))
    Xtst(Vsubseq(V(abc),P0,FF))
    Xtst(Vsubseq(V(abc),P0,I0))
    Xtst(Vsubseq(V(abc),P0,P0))
    Xtst(Vsubseq(V(abc),P0,M0))
    Xtst(Vsubseq(V(abc),P0,B0))
    Xtst(Vsubseq(V(abc),M0,TT))
    Xtst(Vsubseq(V(abc),M0,FF))
    Xtst(Vsubseq(V(abc),M0,I0))
    Xtst(Vsubseq(V(abc),M0,P0))
    Xtst(Vsubseq(V(abc),M0,M0))
    Xtst(Vsubseq(V(abc),M0,B0))
    Xtst(Vsubseq(V(abc),B0,TT))
    Xtst(Vsubseq(V(abc),B0,FF))
    Xtst(Vsubseq(V(abc),B0,I0))
    Xtst(Vsubseq(V(abc),B0,P0))
    Xtst(Vsubseq(V(abc),B0,M0))
    Xtst(Vsubseq(V(abc),B0,B0))
    Xtst(Vsubseq(TT,N(0),TT))
    Xtst(Vsubseq(TT,N(0),FF))
    Xtst(Vsubseq(TT,N(0),I0))
    Xtst(Vsubseq(TT,N(0),P0))
    Xtst(Vsubseq(TT,N(0),M0))
    Xtst(Vsubseq(TT,N(0),B0))
    Xtst(Vsubseq(FF,N(0),TT))
    Xtst(Vsubseq(FF,N(0),FF))
    Xtst(Vsubseq(FF,N(0),I0))
    Xtst(Vsubseq(FF,N(0),P0))
    Xtst(Vsubseq(FF,N(0),M0))
    Xtst(Vsubseq(FF,N(0),B0))
    Xtst(Vsubseq(I0,N(0),TT))
    Xtst(Vsubseq(I0,N(0),FF))
    Xtst(Vsubseq(I0,N(0),P0))
    Xtst(Vsubseq(I0,N(0),M0))
    Xtst(Vsubseq(I0,N(0),B0))
    Xtst(Vsubseq(P0,N(0),TT))
    Xtst(Vsubseq(P0,N(0),FF))
    Xtst(Vsubseq(P0,N(0),I0))
    Xtst(Vsubseq(P0,N(0),P0))
    Xtst(Vsubseq(P0,N(0),M0))
    Xtst(Vsubseq(P0,N(0),B0))
    Xtst(Vsubseq(M0,N(0),TT))
    Xtst(Vsubseq(M0,N(0),FF))
    Xtst(Vsubseq(M0,N(0),I0))
    Xtst(Vsubseq(M0,N(0),P0))
    Xtst(Vsubseq(M0,N(0),M0))
    Xtst(Vsubseq(M0,N(0),B0))
    Xtst(Vsubseq(B0,N(0),TT))
    Xtst(Vsubseq(B0,N(0),FF))
    Xtst(Vsubseq(B0,N(0),I0))
    Xtst(Vsubseq(B0,N(0),P0))
    Xtst(Vsubseq(B0,N(0),M0))
    Xtst(Vsubseq(B0,N(0),B0))
    Xtst(Vsubseq(TT,TT,N(2)))
    Xtst(Vsubseq(TT,FF,N(2)))
    Xtst(Vsubseq(TT,I0,N(2)))
    Xtst(Vsubseq(TT,P0,N(2)))
    Xtst(Vsubseq(TT,M0,N(2)))
    Xtst(Vsubseq(TT,B0,N(2)))
    Xtst(Vsubseq(FF,TT,N(2)))
    Xtst(Vsubseq(FF,FF,N(2)))
    Xtst(Vsubseq(FF,I0,N(2)))
    Xtst(Vsubseq(FF,P0,N(2)))
    Xtst(Vsubseq(FF,M0,N(2)))
    Xtst(Vsubseq(FF,B0,N(2)))
    Xtst(Vsubseq(I0,TT,N(2)))
    Xtst(Vsubseq(I0,FF,N(2)))
    Xtst(Vsubseq(I0,P0,N(2)))
    Xtst(Vsubseq(I0,M0,N(2)))
    Xtst(Vsubseq(I0,B0,N(2)))
    Xtst(Vsubseq(P0,TT,N(2)))
    Xtst(Vsubseq(P0,FF,N(2)))
    Xtst(Vsubseq(P0,I0,N(2)))
    Xtst(Vsubseq(P0,P0,N(2)))
    Xtst(Vsubseq(P0,M0,N(2)))
    Xtst(Vsubseq(P0,B0,N(2)))
    Xtst(Vsubseq(M0,TT,N(2)))
    Xtst(Vsubseq(M0,FF,N(2)))
    Xtst(Vsubseq(M0,I0,N(2)))
    Xtst(Vsubseq(M0,P0,N(2)))
    Xtst(Vsubseq(M0,M0,N(2)))
    Xtst(Vsubseq(M0,B0,N(2)))
    Xtst(Vsubseq(B0,TT,N(2)))
    Xtst(Vsubseq(B0,FF,N(2)))
    Xtst(Vsubseq(B0,I0,N(2)))
    Xtst(Vsubseq(B0,P0,N(2)))
    Xtst(Vsubseq(B0,M0,N(2)))
    Xtst(Vsubseq(B0,B0,N(2)))

    Etst(Vsubseq(V(),N(-1),N(-1)),V())
    Etst(Vsubseq(V(),N(-1),N( 0)),V())
    Etst(Vsubseq(V(),N(-1),N( 1)),V())
    Etst(Vsubseq(V(),N(-1),N( 2)),V())
    Etst(Vsubseq(V(),N(-1),N( 3)),V())
    Etst(Vsubseq(V(),N(-1),N( 4)),V())
    Etst(Vsubseq(V(),N(-1),N( 5)),V())
    Etst(Vsubseq(V(),N(-1),N( 6)),V())
    Etst(Vsubseq(V(),N(-1),N( 7)),V())
    Etst(Vsubseq(V(),N(-1),N( 8)),V())
    Etst(Vsubseq(V(),N(-1),N( 9)),V())
    Etst(Vsubseq(V(),N(-1),N(10)),V())

    Etst(Vsubseq(V(),N( 0),N(-1)),V())
    Etst(Vsubseq(V(),N( 0),N( 0)),V())
    Etst(Vsubseq(V(),N( 0),N( 1)),V())
    Etst(Vsubseq(V(),N( 0),N( 2)),V())
    Etst(Vsubseq(V(),N( 0),N( 3)),V())
    Etst(Vsubseq(V(),N( 0),N( 4)),V())
    Etst(Vsubseq(V(),N( 0),N( 5)),V())
    Etst(Vsubseq(V(),N( 0),N( 6)),V())
    Etst(Vsubseq(V(),N( 0),N( 7)),V())
    Etst(Vsubseq(V(),N( 0),N( 8)),V())
    Etst(Vsubseq(V(),N( 0),N( 9)),V())
    Etst(Vsubseq(V(),N( 0),N(10)),V())

    Etst(Vsubseq(V(),N( 1),N(-1)),V())
    Etst(Vsubseq(V(),N( 1),N( 0)),V())
    Etst(Vsubseq(V(),N( 1),N( 1)),V())
    Etst(Vsubseq(V(),N( 1),N( 2)),V())
    Etst(Vsubseq(V(),N( 1),N( 3)),V())
    Etst(Vsubseq(V(),N( 1),N( 4)),V())
    Etst(Vsubseq(V(),N( 1),N( 5)),V())
    Etst(Vsubseq(V(),N( 1),N( 6)),V())
    Etst(Vsubseq(V(),N( 1),N( 7)),V())
    Etst(Vsubseq(V(),N( 1),N( 8)),V())
    Etst(Vsubseq(V(),N( 1),N( 9)),V())
    Etst(Vsubseq(V(),N( 1),N(10)),V())

    Etst(Vsubseq(V(),N( 2),N(-1)),V())
    Etst(Vsubseq(V(),N( 2),N( 0)),V())
    Etst(Vsubseq(V(),N( 2),N( 1)),V())
    Etst(Vsubseq(V(),N( 2),N( 2)),V())
    Etst(Vsubseq(V(),N( 2),N( 3)),V())
    Etst(Vsubseq(V(),N( 2),N( 4)),V())
    Etst(Vsubseq(V(),N( 2),N( 5)),V())
    Etst(Vsubseq(V(),N( 2),N( 6)),V())
    Etst(Vsubseq(V(),N( 2),N( 7)),V())
    Etst(Vsubseq(V(),N( 2),N( 8)),V())
    Etst(Vsubseq(V(),N( 2),N( 9)),V())
    Etst(Vsubseq(V(),N( 2),N(10)),V())

    Etst(Vsubseq(V(),N( 3),N(-1)),V())
    Etst(Vsubseq(V(),N( 3),N( 0)),V())
    Etst(Vsubseq(V(),N( 3),N( 1)),V())
    Etst(Vsubseq(V(),N( 3),N( 2)),V())
    Etst(Vsubseq(V(),N( 3),N( 3)),V())
    Etst(Vsubseq(V(),N( 3),N( 4)),V())
    Etst(Vsubseq(V(),N( 3),N( 5)),V())
    Etst(Vsubseq(V(),N( 3),N( 6)),V())
    Etst(Vsubseq(V(),N( 3),N( 7)),V())
    Etst(Vsubseq(V(),N( 3),N( 8)),V())
    Etst(Vsubseq(V(),N( 3),N( 9)),V())
    Etst(Vsubseq(V(),N( 3),N(10)),V())

    Etst(Vsubseq(V(),N( 4),N(-1)),V())
    Etst(Vsubseq(V(),N( 4),N( 0)),V())
    Etst(Vsubseq(V(),N( 4),N( 1)),V())
    Etst(Vsubseq(V(),N( 4),N( 2)),V())
    Etst(Vsubseq(V(),N( 4),N( 3)),V())
    Etst(Vsubseq(V(),N( 4),N( 4)),V())
    Etst(Vsubseq(V(),N( 4),N( 5)),V())
    Etst(Vsubseq(V(),N( 4),N( 6)),V())
    Etst(Vsubseq(V(),N( 4),N( 7)),V())
    Etst(Vsubseq(V(),N( 4),N( 8)),V())
    Etst(Vsubseq(V(),N( 4),N( 9)),V())
    Etst(Vsubseq(V(),N( 4),N(10)),V())

    Etst(Vsubseq(V(),N( 5),N(-1)),V())
    Etst(Vsubseq(V(),N( 5),N( 0)),V())
    Etst(Vsubseq(V(),N( 5),N( 1)),V())
    Etst(Vsubseq(V(),N( 5),N( 2)),V())
    Etst(Vsubseq(V(),N( 5),N( 3)),V())
    Etst(Vsubseq(V(),N( 5),N( 4)),V())
    Etst(Vsubseq(V(),N( 5),N( 5)),V())
    Etst(Vsubseq(V(),N( 5),N( 6)),V())
    Etst(Vsubseq(V(),N( 5),N( 7)),V())
    Etst(Vsubseq(V(),N( 5),N( 8)),V())
    Etst(Vsubseq(V(),N( 5),N( 9)),V())
    Etst(Vsubseq(V(),N( 5),N(10)),V())

    Etst(Vsubseq(V(),N( 6),N(-1)),V())
    Etst(Vsubseq(V(),N( 6),N( 0)),V())
    Etst(Vsubseq(V(),N( 6),N( 1)),V())
    Etst(Vsubseq(V(),N( 6),N( 2)),V())
    Etst(Vsubseq(V(),N( 6),N( 3)),V())
    Etst(Vsubseq(V(),N( 6),N( 4)),V())
    Etst(Vsubseq(V(),N( 6),N( 5)),V())
    Etst(Vsubseq(V(),N( 6),N( 6)),V())
    Etst(Vsubseq(V(),N( 6),N( 7)),V())
    Etst(Vsubseq(V(),N( 6),N( 8)),V())
    Etst(Vsubseq(V(),N( 6),N( 9)),V())
    Etst(Vsubseq(V(),N( 6),N(10)),V())

    Etst(Vsubseq(V(),N( 7),N(-1)),V())
    Etst(Vsubseq(V(),N( 7),N( 0)),V())
    Etst(Vsubseq(V(),N( 7),N( 1)),V())
    Etst(Vsubseq(V(),N( 7),N( 2)),V())
    Etst(Vsubseq(V(),N( 7),N( 3)),V())
    Etst(Vsubseq(V(),N( 7),N( 4)),V())
    Etst(Vsubseq(V(),N( 7),N( 5)),V())
    Etst(Vsubseq(V(),N( 7),N( 6)),V())
    Etst(Vsubseq(V(),N( 7),N( 7)),V())
    Etst(Vsubseq(V(),N( 7),N( 8)),V())
    Etst(Vsubseq(V(),N( 7),N( 9)),V())
    Etst(Vsubseq(V(),N( 7),N(10)),V())

    Etst(Vsubseq(V(),N( 8),N(-1)),V())
    Etst(Vsubseq(V(),N( 8),N( 0)),V())
    Etst(Vsubseq(V(),N( 8),N( 1)),V())
    Etst(Vsubseq(V(),N( 8),N( 2)),V())
    Etst(Vsubseq(V(),N( 8),N( 3)),V())
    Etst(Vsubseq(V(),N( 8),N( 4)),V())
    Etst(Vsubseq(V(),N( 8),N( 5)),V())
    Etst(Vsubseq(V(),N( 8),N( 6)),V())
    Etst(Vsubseq(V(),N( 8),N( 7)),V())
    Etst(Vsubseq(V(),N( 8),N( 8)),V())
    Etst(Vsubseq(V(),N( 8),N( 9)),V())
    Etst(Vsubseq(V(),N( 8),N(10)),V())

    Etst(Vsubseq(V(),N( 9),N(-1)),V())
    Etst(Vsubseq(V(),N( 9),N( 0)),V())
    Etst(Vsubseq(V(),N( 9),N( 1)),V())
    Etst(Vsubseq(V(),N( 9),N( 2)),V())
    Etst(Vsubseq(V(),N( 9),N( 3)),V())
    Etst(Vsubseq(V(),N( 9),N( 4)),V())
    Etst(Vsubseq(V(),N( 9),N( 5)),V())
    Etst(Vsubseq(V(),N( 9),N( 6)),V())
    Etst(Vsubseq(V(),N( 9),N( 7)),V())
    Etst(Vsubseq(V(),N( 9),N( 8)),V())
    Etst(Vsubseq(V(),N( 9),N( 9)),V())
    Etst(Vsubseq(V(),N( 9),N(10)),V())

    Etst(Vsubseq(V(),N(10),N(-1)),V())
    Etst(Vsubseq(V(),N(10),N( 0)),V())
    Etst(Vsubseq(V(),N(10),N( 1)),V())
    Etst(Vsubseq(V(),N(10),N( 2)),V())
    Etst(Vsubseq(V(),N(10),N( 3)),V())
    Etst(Vsubseq(V(),N(10),N( 4)),V())
    Etst(Vsubseq(V(),N(10),N( 5)),V())
    Etst(Vsubseq(V(),N(10),N( 6)),V())
    Etst(Vsubseq(V(),N(10),N( 7)),V())
    Etst(Vsubseq(V(),N(10),N( 8)),V())
    Etst(Vsubseq(V(),N(10),N( 9)),V())
    Etst(Vsubseq(V(),N(10),N(10)),V())

    Etst(Vsubseq(V(abcdefghi),N(-1),N(-1)),V())
    Etst(Vsubseq(V(abcdefghi),N(-1),N( 0)),V())
    Etst(Vsubseq(V(abcdefghi),N(-1),N( 1)),V(a))
    Etst(Vsubseq(V(abcdefghi),N(-1),N( 2)),V(ab))
    Etst(Vsubseq(V(abcdefghi),N(-1),N( 3)),V(abc))
    Etst(Vsubseq(V(abcdefghi),N(-1),N( 4)),V(abcd))
    Etst(Vsubseq(V(abcdefghi),N(-1),N( 5)),V(abcde))
    Etst(Vsubseq(V(abcdefghi),N(-1),N( 6)),V(abcdef))
    Etst(Vsubseq(V(abcdefghi),N(-1),N( 7)),V(abcdefg))
    Etst(Vsubseq(V(abcdefghi),N(-1),N( 8)),V(abcdefgh))
    Etst(Vsubseq(V(abcdefghi),N(-1),N( 9)),V(abcdefghi))
    Etst(Vsubseq(V(abcdefghi),N(-1),N(10)),V(abcdefghi))

    Etst(Vsubseq(V(abcdefghi),N( 0),N(-1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 0),N( 0)),V())
    Etst(Vsubseq(V(abcdefghi),N( 0),N( 1)),V(a))
    Etst(Vsubseq(V(abcdefghi),N( 0),N( 2)),V(ab))
    Etst(Vsubseq(V(abcdefghi),N( 0),N( 3)),V(abc))
    Etst(Vsubseq(V(abcdefghi),N( 0),N( 4)),V(abcd))
    Etst(Vsubseq(V(abcdefghi),N( 0),N( 5)),V(abcde))
    Etst(Vsubseq(V(abcdefghi),N( 0),N( 6)),V(abcdef))
    Etst(Vsubseq(V(abcdefghi),N( 0),N( 7)),V(abcdefg))
    Etst(Vsubseq(V(abcdefghi),N( 0),N( 8)),V(abcdefgh))
    Etst(Vsubseq(V(abcdefghi),N( 0),N( 9)),V(abcdefghi))
    Etst(Vsubseq(V(abcdefghi),N( 0),N(10)),V(abcdefghi))

    Etst(Vsubseq(V(abcdefghi),N( 1),N(-1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 1),N( 0)),V())
    Etst(Vsubseq(V(abcdefghi),N( 1),N( 1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 1),N( 2)),V(b))
    Etst(Vsubseq(V(abcdefghi),N( 1),N( 3)),V(bc))
    Etst(Vsubseq(V(abcdefghi),N( 1),N( 4)),V(bcd))
    Etst(Vsubseq(V(abcdefghi),N( 1),N( 5)),V(bcde))
    Etst(Vsubseq(V(abcdefghi),N( 1),N( 6)),V(bcdef))
    Etst(Vsubseq(V(abcdefghi),N( 1),N( 7)),V(bcdefg))
    Etst(Vsubseq(V(abcdefghi),N( 1),N( 8)),V(bcdefgh))
    Etst(Vsubseq(V(abcdefghi),N( 1),N( 9)),V(bcdefghi))
    Etst(Vsubseq(V(abcdefghi),N( 1),N(10)),V(bcdefghi))

    Etst(Vsubseq(V(abcdefghi),N( 2),N(-1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 2),N( 0)),V())
    Etst(Vsubseq(V(abcdefghi),N( 2),N( 1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 2),N( 2)),V())
    Etst(Vsubseq(V(abcdefghi),N( 2),N( 3)),V(c))
    Etst(Vsubseq(V(abcdefghi),N( 2),N( 4)),V(cd))
    Etst(Vsubseq(V(abcdefghi),N( 2),N( 5)),V(cde))
    Etst(Vsubseq(V(abcdefghi),N( 2),N( 6)),V(cdef))
    Etst(Vsubseq(V(abcdefghi),N( 2),N( 7)),V(cdefg))
    Etst(Vsubseq(V(abcdefghi),N( 2),N( 8)),V(cdefgh))
    Etst(Vsubseq(V(abcdefghi),N( 2),N( 9)),V(cdefghi))
    Etst(Vsubseq(V(abcdefghi),N( 2),N(10)),V(cdefghi))

    Etst(Vsubseq(V(abcdefghi),N( 3),N(-1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 3),N( 0)),V())
    Etst(Vsubseq(V(abcdefghi),N( 3),N( 1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 3),N( 2)),V())
    Etst(Vsubseq(V(abcdefghi),N( 3),N( 3)),V())
    Etst(Vsubseq(V(abcdefghi),N( 3),N( 4)),V(d))
    Etst(Vsubseq(V(abcdefghi),N( 3),N( 5)),V(de))
    Etst(Vsubseq(V(abcdefghi),N( 3),N( 6)),V(def))
    Etst(Vsubseq(V(abcdefghi),N( 3),N( 7)),V(defg))
    Etst(Vsubseq(V(abcdefghi),N( 3),N( 8)),V(defgh))
    Etst(Vsubseq(V(abcdefghi),N( 3),N( 9)),V(defghi))
    Etst(Vsubseq(V(abcdefghi),N( 3),N(10)),V(defghi))

    Etst(Vsubseq(V(abcdefghi),N( 4),N(-1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 4),N( 0)),V())
    Etst(Vsubseq(V(abcdefghi),N( 4),N( 1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 4),N( 2)),V())
    Etst(Vsubseq(V(abcdefghi),N( 4),N( 3)),V())
    Etst(Vsubseq(V(abcdefghi),N( 4),N( 4)),V())
    Etst(Vsubseq(V(abcdefghi),N( 4),N( 5)),V(e))
    Etst(Vsubseq(V(abcdefghi),N( 4),N( 6)),V(ef))
    Etst(Vsubseq(V(abcdefghi),N( 4),N( 7)),V(efg))
    Etst(Vsubseq(V(abcdefghi),N( 4),N( 8)),V(efgh))
    Etst(Vsubseq(V(abcdefghi),N( 4),N( 9)),V(efghi))
    Etst(Vsubseq(V(abcdefghi),N( 4),N(10)),V(efghi))

    Etst(Vsubseq(V(abcdefghi),N( 5),N(-1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 5),N( 0)),V())
    Etst(Vsubseq(V(abcdefghi),N( 5),N( 1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 5),N( 2)),V())
    Etst(Vsubseq(V(abcdefghi),N( 5),N( 3)),V())
    Etst(Vsubseq(V(abcdefghi),N( 5),N( 4)),V())
    Etst(Vsubseq(V(abcdefghi),N( 5),N( 5)),V())
    Etst(Vsubseq(V(abcdefghi),N( 5),N( 6)),V(f))
    Etst(Vsubseq(V(abcdefghi),N( 5),N( 7)),V(fg))
    Etst(Vsubseq(V(abcdefghi),N( 5),N( 8)),V(fgh))
    Etst(Vsubseq(V(abcdefghi),N( 5),N( 9)),V(fghi))
    Etst(Vsubseq(V(abcdefghi),N( 5),N(10)),V(fghi))

    Etst(Vsubseq(V(abcdefghi),N( 6),N(-1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 6),N( 0)),V())
    Etst(Vsubseq(V(abcdefghi),N( 6),N( 1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 6),N( 2)),V())
    Etst(Vsubseq(V(abcdefghi),N( 6),N( 3)),V())
    Etst(Vsubseq(V(abcdefghi),N( 6),N( 4)),V())
    Etst(Vsubseq(V(abcdefghi),N( 6),N( 5)),V())
    Etst(Vsubseq(V(abcdefghi),N( 6),N( 6)),V())
    Etst(Vsubseq(V(abcdefghi),N( 6),N( 7)),V(g))
    Etst(Vsubseq(V(abcdefghi),N( 6),N( 8)),V(gh))
    Etst(Vsubseq(V(abcdefghi),N( 6),N( 9)),V(ghi))
    Etst(Vsubseq(V(abcdefghi),N( 6),N(10)),V(ghi))

    Etst(Vsubseq(V(abcdefghi),N( 7),N(-1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 7),N( 0)),V())
    Etst(Vsubseq(V(abcdefghi),N( 7),N( 1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 7),N( 2)),V())
    Etst(Vsubseq(V(abcdefghi),N( 7),N( 3)),V())
    Etst(Vsubseq(V(abcdefghi),N( 7),N( 4)),V())
    Etst(Vsubseq(V(abcdefghi),N( 7),N( 5)),V())
    Etst(Vsubseq(V(abcdefghi),N( 7),N( 6)),V())
    Etst(Vsubseq(V(abcdefghi),N( 7),N( 7)),V())
    Etst(Vsubseq(V(abcdefghi),N( 7),N( 8)),V(h))
    Etst(Vsubseq(V(abcdefghi),N( 7),N( 9)),V(hi))
    Etst(Vsubseq(V(abcdefghi),N( 7),N(10)),V(hi))

    Etst(Vsubseq(V(abcdefghi),N( 8),N(-1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 8),N( 0)),V())
    Etst(Vsubseq(V(abcdefghi),N( 8),N( 1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 8),N( 2)),V())
    Etst(Vsubseq(V(abcdefghi),N( 8),N( 3)),V())
    Etst(Vsubseq(V(abcdefghi),N( 8),N( 4)),V())
    Etst(Vsubseq(V(abcdefghi),N( 8),N( 5)),V())
    Etst(Vsubseq(V(abcdefghi),N( 8),N( 6)),V())
    Etst(Vsubseq(V(abcdefghi),N( 8),N( 7)),V())
    Etst(Vsubseq(V(abcdefghi),N( 8),N( 8)),V())
    Etst(Vsubseq(V(abcdefghi),N( 8),N( 9)),V(i))
    Etst(Vsubseq(V(abcdefghi),N( 8),N(10)),V(i))

    Etst(Vsubseq(V(abcdefghi),N( 9),N(-1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 9),N( 0)),V())
    Etst(Vsubseq(V(abcdefghi),N( 9),N( 1)),V())
    Etst(Vsubseq(V(abcdefghi),N( 9),N( 2)),V())
    Etst(Vsubseq(V(abcdefghi),N( 9),N( 3)),V())
    Etst(Vsubseq(V(abcdefghi),N( 9),N( 4)),V())
    Etst(Vsubseq(V(abcdefghi),N( 9),N( 5)),V())
    Etst(Vsubseq(V(abcdefghi),N( 9),N( 6)),V())
    Etst(Vsubseq(V(abcdefghi),N( 9),N( 7)),V())
    Etst(Vsubseq(V(abcdefghi),N( 9),N( 8)),V())
    Etst(Vsubseq(V(abcdefghi),N( 9),N( 9)),V())
    Etst(Vsubseq(V(abcdefghi),N( 9),N(10)),V())

    Etst(Vsubseq(V(abcdefghi),N(10),N(-1)),V())
    Etst(Vsubseq(V(abcdefghi),N(10),N( 0)),V())
    Etst(Vsubseq(V(abcdefghi),N(10),N( 1)),V())
    Etst(Vsubseq(V(abcdefghi),N(10),N( 2)),V())
    Etst(Vsubseq(V(abcdefghi),N(10),N( 3)),V())
    Etst(Vsubseq(V(abcdefghi),N(10),N( 4)),V())
    Etst(Vsubseq(V(abcdefghi),N(10),N( 5)),V())
    Etst(Vsubseq(V(abcdefghi),N(10),N( 6)),V())
    Etst(Vsubseq(V(abcdefghi),N(10),N( 7)),V())
    Etst(Vsubseq(V(abcdefghi),N(10),N( 8)),V())
    Etst(Vsubseq(V(abcdefghi),N(10),N( 9)),V())
    Etst(Vsubseq(V(abcdefghi),N(10),N(10)),V())

    Xtst(V2bytes(TT))
    Xtst(V2bytes(FF))
    Xtst(V2bytes(P0))
    Xtst(V2bytes(M0))
    Xtst(V2bytes(B0))
    Etst(V2bytes(M),T)
    Etst(V2bytes(N(-1000000000000000000000000)),T)
    Etst(V2bytes(N(0)),T)
    Etst(V2bytes(N(1)),T)
    Etst(V2bytes(N(255)),T)
    Etst(V2bytes(N(256)),PAIR(N(0),T))
    Etst(V2bytes(N(65537)),PAIR(N(1),PAIR(N(0),T)))
    Etst(V2bytes(V(ABCDEFGHIJ)),
    PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),PAIR(N(69),
    PAIR(N(70),PAIR(N(71),PAIR(N(72),PAIR(N(73),PAIR(N(74),T)))))))))))
    Etst(V2bytes(V(ABCDEFGHI)),
    PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),PAIR(N(69),
    PAIR(N(70),PAIR(N(71),PAIR(N(72),PAIR(N(73),T))))))))))
    Etst(V2bytes(V(ABCDEFGH)),
    PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),PAIR(N(69),
    PAIR(N(70),PAIR(N(71),PAIR(N(72),T)))))))))
    Etst(V2bytes(V(ABCDEFG)),
    PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),PAIR(N(69),
    PAIR(N(70),PAIR(N(71),T))))))))
    Etst(V2bytes(V(ABCDEF)),
    PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),PAIR(N(69),PAIR(N(70),T)))))))
    Etst(V2bytes(V(ABCDE)),
    PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),PAIR(N(69),T))))))
    Etst(V2bytes(V(ABCD)),
    PAIR(N(65),PAIR(N(66),PAIR(N(67),PAIR(N(68),T)))))
    Etst(V2bytes(V(ABC)),PAIR(N(65),PAIR(N(66),PAIR(N(67),T))))
    Etst(V2bytes(V(AB)),PAIR(N(65),PAIR(N(66),T)))
    Etst(V2bytes(V(A)),PAIR(N(65),T))
    Etst(V2bytes(V()),T)

    Etst(Bt2bytes(T),T)
    Etst(Bt2bytes(F),T)
    Etst(Bt2bytes(M0),T)
    Etst(Bt2bytes(B0),T)
    Etst(Bt2bytes(N(-1)),T)
    Etst(Bt2bytes(N(0)),PAIR(N(0),T))
    Etst(Bt2bytes(N(255)),PAIR(N(255),T))
    Etst(Bt2bytes(N(256)),T)
    Etst(Bt2bytes(V()),PAIR(N(1),T))
    Etst(Bt2bytes(PAIR(N(1),N(2))),PAIR(N(1),PAIR(N(2),T)))
    Etst(Bt2bytes(PAIR(PAIR(N(1),N(-1)),PAIR(B0,N(100)))),
      PAIR(N(1),PAIR(N(100),T)))

    Etst(Vt2bytes(T),T)
    Etst(Vt2bytes(F),T)
    Etst(Vt2bytes(M0),T)
    Etst(Vt2bytes(B0),T)
    Etst(Vt2bytes(N(-1)),T)
    Etst(Vt2bytes(N(0)),T)
    Etst(Vt2bytes(N(1)),T)
    Etst(Vt2bytes(N(255)),T)
    Etst(Vt2bytes(N(256)),PAIR(N(0),T))
    Etst(Vt2bytes(N(513)),PAIR(N(1),T))
    Etst(Vt2bytes(V()),T)
    Etst(Vt2bytes(V(abc)),PAIR(N(97),PAIR(N(98),PAIR(N(99),T))))
    Etst(Vt2bytes(PAIR(V(ab),V(c))),PAIR(N(97),PAIR(N(98),PAIR(N(99),T))))
    Etst(Vt2bytes(PAIR(PAIR(V(a),N(-1)),PAIR(B0,V(bc)))),
      PAIR(N(97),PAIR(N(98),PAIR(N(99),T))))

    Etst(Bt2vector(T),E)
    Etst(Bt2vector(F),E)
    Etst(Bt2vector(M0),E)
    Etst(Bt2vector(B0),E)
    Etst(Bt2vector(N(-1)),E)
    Etst(Bt2vector(N(0)),N(256))
    Etst(Bt2vector(N(255)),N(511))
    Etst(Bt2vector(N(256)),E)
    Etst(Bt2vector(V()),N(257))
    Etst(Bt2vector(PAIR(N(97),N(98))),V(ab))
    Etst(Bt2vector(PAIR(PAIR(N(97),N(-1)),PAIR(B0,N(98)))),V(ab))

    Etst(Vt2v(T),E)
    Etst(Vt2v(F),E)
    Etst(Vt2v(M0),E)
    Etst(Vt2v(B0),E)
    Etst(Vt2v(N(-1)),E)
    Etst(Vt2v(N(0)),E)
    Etst(Vt2v(N(1)),E)
    Etst(Vt2v(N(255)),E)
    Etst(Vt2v(N(256)),N(256))
    Etst(Vt2v(N(513)),N(257))
    Etst(Vt2v(V()),E)
    Etst(Vt2v(V(abc)),V(abc))
    Etst(Vt2v(PAIR(V(ab),V(c))),V(abc))
    Etst(Vt2v(PAIR(PAIR(V(a),N(-1)),PAIR(B0,V(bc)))),V(abc))

    Etst(Spy(T),V(spy))
    Etst(Trace(T),V(trace))
    Etst(Print(T),V(print))
    Etst(Timer(T),V(timer))

    Etst(And(T,T),T)
    Etst(And(T,F),F)
    Etst(And(T,Z),Z)
    Etst(And(T,X),X)
    Etst(And(F,T),F)
    Etst(And(F,F),F)
    Etst(And(F,Z),F)
    Etst(And(F,X),F)
    Etst(And(F,B),F)
    Etst(And(Z,T),Z)
    Etst(And(Z,F),Z)
    Etst(And(Z,Z),Z)
    Etst(And(Z,X),Z)
    Etst(And(Z,B),Z)
    Etst(And(X,T),X)
    Etst(And(X,F),X)
    Etst(And(X,Z),X)
    Etst(And(X,X),X)
    Etst(And(X,B),X)

    Etst(Or(T,T),T)
    Etst(Or(T,F),T)
    Etst(Or(T,Z),T)
    Etst(Or(T,X),T)
    Etst(Or(T,B),T)
    Etst(Or(F,T),T)
    Etst(Or(F,F),F)
    Etst(Or(F,Z),Z)
    Etst(Or(F,X),X)
    Etst(Or(Z,T),T)
    Etst(Or(Z,F),F)
    Etst(Or(Z,Z),Z)
    Etst(Or(Z,X),X)
    Etst(Or(X,T),X)
    Etst(Or(X,F),X)
    Etst(Or(X,Z),X)
    Etst(Or(X,X),X)
    Etst(Or(X,B),X)

    Etst(Then(T,T),T)
    Etst(Then(T,F),F)
    Etst(Then(T,Z),Z)
    Etst(Then(T,X),X)
    Etst(Then(F,T),T)
    Etst(Then(F,F),F)
    Etst(Then(F,Z),Z)
    Etst(Then(F,X),X)
    Etst(Then(Z,T),T)
    Etst(Then(Z,F),F)
    Etst(Then(Z,Z),Z)
    Etst(Then(Z,X),X)
    Etst(Then(X,T),X)
    Etst(Then(X,F),X)
    Etst(Then(X,Z),X)
    Etst(Then(X,X),X)
    Etst(Then(X,B),X)

    Ttst(Root(M_true))

    Ftst(Root(code_If))
    Ftst(Root(Apply(code_If,MT)))
    Ftst(Root(Apply(Apply(code_If,MT),MT)))
    Ttst(Root(M_If(MT,MT,MT)))
    Ttst(Root(M_If(MT,MT,MF)))
    Ftst(Root(M_If(MT,MF,MT)))
    Ftst(Root(M_If(MT,MF,MF)))
    Ttst(Root(M_If(MF,MT,MT)))
    Ftst(Root(M_If(MF,MT,MF)))
    Ttst(Root(M_If(MF,MF,MT)))
    Ftst(Root(M_If(MF,MF,MF)))

    Ftst(Root(code_ap))
    Ftst(Root(Apply(code_ap,MKI)))
    Ttst(Root(M_ap(MT,MT)))
    Ttst(Root(M_ap(MT,MF)))
    Ttst(Root(M_ap(MT,MI)))
    Ttst(Root(M_ap(MF,MT)))
    Ttst(Root(M_ap(MF,MF)))
    Ttst(Root(M_ap(MF,MI)))
    Ttst(Root(M_ap(MI,MT)))
    Ftst(Root(M_ap(MI,MF)))
    Ftst(Root(M_ap(MI,MI)))
    Ftst(Root(M_ap(MK,MT)))
    Ftst(Root(M_ap(MK,MF)))
    Ftst(Root(M_ap(MK,MI)))

    Etst(Pair(Z0,Z1),Pair(Z0,Z1))
    Utst(Pair(Z0,Z1),Pair(Z3,Z1))
    Utst(Pair(Z0,Z1),Pair(Z0,Z3))
    Etst(Pair(X0,Z1),X0)
    Etst(Pair(Z0,X1),X1)
    Etst(Pair(X0,X1),X0)

    Etst(Head(TT),TT)
    Etst(Head(FF),FF)
    Etst(Head(Z0),Z0)
    Etst(Head(X0),X0)
    Etst(Head(M0),M0)
    Etst(Head(B0),B0)
    Etst(Head(Pair(Z0,Z1)),Z0)

    Etst(Tail(TT),TT)
    Etst(Tail(FF),FF)
    Etst(Tail(Z0),Z0)
    Etst(Tail(X0),X0)
    Etst(Tail(M0),M0)
    Etst(Tail(B0),B0)
    Etst(Tail(Pair(Z0,Z1)),Z1)

    Etst(Boolp(TT),TT)
    Etst(Boolp(FF),TT)
    Etst(Boolp(Z0),FF)
    Etst(Boolp(P0),FF)
    Etst(Boolp(X0),X0)
    Etst(Boolp(M0),FF)
    Etst(Boolp(B0),FF)

    Etst(Notnot(TT),TT)
    Etst(Notnot(FF),FF)
    Etst(Notnot(Z0),FF)
    Etst(Notnot(P0),FF)
    Etst(Notnot(X0),X0)
    Etst(Notnot(M0),FF)
    Etst(Notnot(B0),FF)

    Etst(If(TT,Z1,Z2),Z1)
    Etst(If(FF,Z1,Z2),Z2)
    Etst(If(Z0,Z1,Z2),Z2)
    Etst(If(P0,Z1,Z2),Z2)
    Etst(If(X0,Z1,Z2),X0)
    Etst(If(M0,Z1,Z2),Z2)
    Etst(If(B0,Z1,Z2),Z2)

    Etst(If(TT,X1,Z2),X1)
    Etst(If(FF,X1,Z2),Z2)
    Etst(If(Z0,X1,Z2),Z2)
    Etst(If(P0,X1,Z2),Z2)
    Etst(If(X0,X1,Z2),X0)
    Etst(If(M0,X1,Z2),Z2)
    Etst(If(B0,X1,Z2),Z2)

    Etst(If(TT,Z1,X2),Z1)
    Etst(If(FF,Z1,X2),X2)
    Etst(If(Z0,Z1,X2),X2)
    Etst(If(P0,Z1,X2),X2)
    Etst(If(X0,Z1,X2),X0)
    Etst(If(M0,Z1,X2),X2)
    Etst(If(B0,Z1,X2),X2)

    Etst(If(TT,Z1,BB),Z1)
    Etst(If(FF,BB,Z2),Z2)
    Etst(If(Z0,BB,Z2),Z2)
    Etst(If(P0,BB,Z2),Z2)
    Etst(If(X0,BB,Z2),X0)
    Etst(If(M0,BB,Z2),Z2)
    Etst(If(B0,BB,Z2),Z2)

    Etst(Catch(TT),Pair(FF,TT))
    Etst(Catch(FF),Pair(FF,FF))
    Etst(Catch(Z0),Pair(FF,Z0))
    Etst(Catch(P0),Pair(FF,P0))
    Etst(Catch(X0),Pair(TT,I[0]))
    Etst(Catch(X1),Pair(TT,I[1]))
    Etst(Catch(X2),Pair(TT,I[2]))
    Etst(Catch(X3),Pair(TT,I[3]))
    Etst(Catch(M0),Pair(FF,M0))
    Etst(Catch(B0),Pair(FF,B0))

    Etst(Catch(Raise(TT)),Pair(TT,TT))
    Etst(Catch(Raise(FF)),Pair(TT,FF))
    Etst(Catch(Raise(Z0)),Pair(TT,Z0))
    Etst(Catch(Raise(EE)),Pair(TT,EE))
    Etst(Catch(Raise(P0)),Pair(TT,P0))
    Etst(Catch(Raise(M0)),Pair(TT,M0))
    Etst(Catch(Raise(B0)),Pair(TT,B0))
    Etst(Catch(Raise(Raise(TT))),Pair(TT,TT))
    Etst(Catch(Raise(Raise(FF))),Pair(TT,FF))
    Etst(Catch(Raise(Raise(Z0))),Pair(TT,Z0))
    Etst(Catch(Raise(Raise(EE))),Pair(TT,EE))
    Etst(Catch(Raise(Raise(P0))),Pair(TT,P0))
    Etst(Catch(Raise(Raise(M0))),Pair(TT,M0))
    Etst(Catch(Raise(Raise(B0))),Pair(TT,B0))

    Xtst(Floor(TT,TT))
    Xtst(Floor(TT,FF))
    Xtst(Floor(TT,I0))
    Xtst(Floor(TT,P0))
    Xtst(Floor(TT,M0))
    Xtst(Floor(TT,B0))
    Xtst(Floor(FF,TT))
    Xtst(Floor(FF,FF))
    Xtst(Floor(FF,I0))
    Xtst(Floor(FF,P0))
    Xtst(Floor(FF,M0))
    Xtst(Floor(FF,B0))
    Xtst(Floor(I0,TT))
    Xtst(Floor(I0,FF))
    Xtst(Floor(I0,P0))
    Xtst(Floor(I0,M0))
    Xtst(Floor(I0,B0))
    Xtst(Floor(P0,TT))
    Xtst(Floor(P0,FF))
    Xtst(Floor(P0,I0))
    Xtst(Floor(P0,P0))
    Xtst(Floor(P0,M0))
    Xtst(Floor(P0,B0))
    Xtst(Floor(M0,TT))
    Xtst(Floor(M0,FF))
    Xtst(Floor(M0,I0))
    Xtst(Floor(M0,P0))
    Xtst(Floor(M0,M0))
    Xtst(Floor(M0,B0))
    Xtst(Floor(B0,TT))
    Xtst(Floor(B0,FF))
    Xtst(Floor(B0,I0))
    Xtst(Floor(B0,P0))
    Xtst(Floor(B0,M0))
    Xtst(Floor(B0,B0))
    Xtst(Floor(I0,Z))
    Xtst(Floor(I0,M))

    Etst(Floor(N(-9),N(1)),PAIR(N(-9),N(0)))
    Etst(Floor(N(-8),N(1)),PAIR(N(-8),N(0)))
    Etst(Floor(N(-7),N(1)),PAIR(N(-7),N(0)))
    Etst(Floor(N(-6),N(1)),PAIR(N(-6),N(0)))
    Etst(Floor(N(-5),N(1)),PAIR(N(-5),N(0)))
    Etst(Floor(N(-4),N(1)),PAIR(N(-4),N(0)))
    Etst(Floor(N(-3),N(1)),PAIR(N(-3),N(0)))
    Etst(Floor(N(-2),N(1)),PAIR(N(-2),N(0)))
    Etst(Floor(N(-1),N(1)),PAIR(N(-1),N(0)))
    Etst(Floor(N( 0),N(1)),PAIR(N( 0),N(0)))
    Etst(Floor(N( 1),N(1)),PAIR(N( 1),N(0)))
    Etst(Floor(N( 2),N(1)),PAIR(N( 2),N(0)))
    Etst(Floor(N( 3),N(1)),PAIR(N( 3),N(0)))
    Etst(Floor(N( 4),N(1)),PAIR(N( 4),N(0)))
    Etst(Floor(N( 5),N(1)),PAIR(N( 5),N(0)))
    Etst(Floor(N( 6),N(1)),PAIR(N( 6),N(0)))
    Etst(Floor(N( 7),N(1)),PAIR(N( 7),N(0)))
    Etst(Floor(N( 8),N(1)),PAIR(N( 8),N(0)))
    Etst(Floor(N( 9),N(1)),PAIR(N( 9),N(0)))

    Etst(Floor(N(-9),N(2)),PAIR(N(-5),N(1)))
    Etst(Floor(N(-8),N(2)),PAIR(N(-4),N(0)))
    Etst(Floor(N(-7),N(2)),PAIR(N(-4),N(1)))
    Etst(Floor(N(-6),N(2)),PAIR(N(-3),N(0)))
    Etst(Floor(N(-5),N(2)),PAIR(N(-3),N(1)))
    Etst(Floor(N(-4),N(2)),PAIR(N(-2),N(0)))
    Etst(Floor(N(-3),N(2)),PAIR(N(-2),N(1)))
    Etst(Floor(N(-2),N(2)),PAIR(N(-1),N(0)))
    Etst(Floor(N(-1),N(2)),PAIR(N(-1),N(1)))
    Etst(Floor(N( 0),N(2)),PAIR(N( 0),N(0)))
    Etst(Floor(N( 1),N(2)),PAIR(N( 0),N(1)))
    Etst(Floor(N( 2),N(2)),PAIR(N( 1),N(0)))
    Etst(Floor(N( 3),N(2)),PAIR(N( 1),N(1)))
    Etst(Floor(N( 4),N(2)),PAIR(N( 2),N(0)))
    Etst(Floor(N( 5),N(2)),PAIR(N( 2),N(1)))
    Etst(Floor(N( 6),N(2)),PAIR(N( 3),N(0)))
    Etst(Floor(N( 7),N(2)),PAIR(N( 3),N(1)))
    Etst(Floor(N( 8),N(2)),PAIR(N( 4),N(0)))
    Etst(Floor(N( 9),N(2)),PAIR(N( 4),N(1)))

    Etst(Floor(N(-9),N(3)),PAIR(N(-3),N(0)))
    Etst(Floor(N(-8),N(3)),PAIR(N(-3),N(1)))
    Etst(Floor(N(-7),N(3)),PAIR(N(-3),N(2)))
    Etst(Floor(N(-6),N(3)),PAIR(N(-2),N(0)))
    Etst(Floor(N(-5),N(3)),PAIR(N(-2),N(1)))
    Etst(Floor(N(-4),N(3)),PAIR(N(-2),N(2)))
    Etst(Floor(N(-3),N(3)),PAIR(N(-1),N(0)))
    Etst(Floor(N(-2),N(3)),PAIR(N(-1),N(1)))
    Etst(Floor(N(-1),N(3)),PAIR(N(-1),N(2)))
    Etst(Floor(N( 0),N(3)),PAIR(N( 0),N(0)))
    Etst(Floor(N( 1),N(3)),PAIR(N( 0),N(1)))
    Etst(Floor(N( 2),N(3)),PAIR(N( 0),N(2)))
    Etst(Floor(N( 3),N(3)),PAIR(N( 1),N(0)))
    Etst(Floor(N( 4),N(3)),PAIR(N( 1),N(1)))
    Etst(Floor(N( 5),N(3)),PAIR(N( 1),N(2)))
    Etst(Floor(N( 6),N(3)),PAIR(N( 2),N(0)))
    Etst(Floor(N( 7),N(3)),PAIR(N( 2),N(1)))
    Etst(Floor(N( 8),N(3)),PAIR(N( 2),N(2)))
    Etst(Floor(N( 9),N(3)),PAIR(N( 3),N(0)))

    Etst(Floor(N(-9),N(4)),PAIR(N(-3),N(3)))
    Etst(Floor(N(-8),N(4)),PAIR(N(-2),N(0)))
    Etst(Floor(N(-7),N(4)),PAIR(N(-2),N(1)))
    Etst(Floor(N(-6),N(4)),PAIR(N(-2),N(2)))
    Etst(Floor(N(-5),N(4)),PAIR(N(-2),N(3)))
    Etst(Floor(N(-4),N(4)),PAIR(N(-1),N(0)))
    Etst(Floor(N(-3),N(4)),PAIR(N(-1),N(1)))
    Etst(Floor(N(-2),N(4)),PAIR(N(-1),N(2)))
    Etst(Floor(N(-1),N(4)),PAIR(N(-1),N(3)))
    Etst(Floor(N( 0),N(4)),PAIR(N( 0),N(0)))
    Etst(Floor(N( 1),N(4)),PAIR(N( 0),N(1)))
    Etst(Floor(N( 2),N(4)),PAIR(N( 0),N(2)))
    Etst(Floor(N( 3),N(4)),PAIR(N( 0),N(3)))
    Etst(Floor(N( 4),N(4)),PAIR(N( 1),N(0)))
    Etst(Floor(N( 5),N(4)),PAIR(N( 1),N(1)))
    Etst(Floor(N( 6),N(4)),PAIR(N( 1),N(2)))
    Etst(Floor(N( 7),N(4)),PAIR(N( 1),N(3)))
    Etst(Floor(N( 8),N(4)),PAIR(N( 2),N(0)))
    Etst(Floor(N( 9),N(4)),PAIR(N( 2),N(1)))

    Etst(Floor(N(-9),N(5)),PAIR(N(-2),N(1)))
    Etst(Floor(N(-8),N(5)),PAIR(N(-2),N(2)))
    Etst(Floor(N(-7),N(5)),PAIR(N(-2),N(3)))
    Etst(Floor(N(-6),N(5)),PAIR(N(-2),N(4)))
    Etst(Floor(N(-5),N(5)),PAIR(N(-1),N(0)))
    Etst(Floor(N(-4),N(5)),PAIR(N(-1),N(1)))
    Etst(Floor(N(-3),N(5)),PAIR(N(-1),N(2)))
    Etst(Floor(N(-2),N(5)),PAIR(N(-1),N(3)))
    Etst(Floor(N(-1),N(5)),PAIR(N(-1),N(4)))
    Etst(Floor(N( 0),N(5)),PAIR(N( 0),N(0)))
    Etst(Floor(N( 1),N(5)),PAIR(N( 0),N(1)))
    Etst(Floor(N( 2),N(5)),PAIR(N( 0),N(2)))
    Etst(Floor(N( 3),N(5)),PAIR(N( 0),N(3)))
    Etst(Floor(N( 4),N(5)),PAIR(N( 0),N(4)))
    Etst(Floor(N( 5),N(5)),PAIR(N( 1),N(0)))
    Etst(Floor(N( 6),N(5)),PAIR(N( 1),N(1)))
    Etst(Floor(N( 7),N(5)),PAIR(N( 1),N(2)))
    Etst(Floor(N( 8),N(5)),PAIR(N( 1),N(3)))
    Etst(Floor(N( 9),N(5)),PAIR(N( 1),N(4)))

    Etst(Floor(N(-9),N(6)),PAIR(N(-2),N(3)))
    Etst(Floor(N(-8),N(6)),PAIR(N(-2),N(4)))
    Etst(Floor(N(-7),N(6)),PAIR(N(-2),N(5)))
    Etst(Floor(N(-6),N(6)),PAIR(N(-1),N(0)))
    Etst(Floor(N(-5),N(6)),PAIR(N(-1),N(1)))
    Etst(Floor(N(-4),N(6)),PAIR(N(-1),N(2)))
    Etst(Floor(N(-3),N(6)),PAIR(N(-1),N(3)))
    Etst(Floor(N(-2),N(6)),PAIR(N(-1),N(4)))
    Etst(Floor(N(-1),N(6)),PAIR(N(-1),N(5)))
    Etst(Floor(N( 0),N(6)),PAIR(N( 0),N(0)))
    Etst(Floor(N( 1),N(6)),PAIR(N( 0),N(1)))
    Etst(Floor(N( 2),N(6)),PAIR(N( 0),N(2)))
    Etst(Floor(N( 3),N(6)),PAIR(N( 0),N(3)))
    Etst(Floor(N( 4),N(6)),PAIR(N( 0),N(4)))
    Etst(Floor(N( 5),N(6)),PAIR(N( 0),N(5)))
    Etst(Floor(N( 6),N(6)),PAIR(N( 1),N(0)))
    Etst(Floor(N( 7),N(6)),PAIR(N( 1),N(1)))
    Etst(Floor(N( 8),N(6)),PAIR(N( 1),N(2)))
    Etst(Floor(N( 9),N(6)),PAIR(N( 1),N(3)))

    Etst(Floor(N(-9),N(7)),PAIR(N(-2),N(5)))
    Etst(Floor(N(-8),N(7)),PAIR(N(-2),N(6)))
    Etst(Floor(N(-7),N(7)),PAIR(N(-1),N(0)))
    Etst(Floor(N(-6),N(7)),PAIR(N(-1),N(1)))
    Etst(Floor(N(-5),N(7)),PAIR(N(-1),N(2)))
    Etst(Floor(N(-4),N(7)),PAIR(N(-1),N(3)))
    Etst(Floor(N(-3),N(7)),PAIR(N(-1),N(4)))
    Etst(Floor(N(-2),N(7)),PAIR(N(-1),N(5)))
    Etst(Floor(N(-1),N(7)),PAIR(N(-1),N(6)))
    Etst(Floor(N( 0),N(7)),PAIR(N( 0),N(0)))
    Etst(Floor(N( 1),N(7)),PAIR(N( 0),N(1)))
    Etst(Floor(N( 2),N(7)),PAIR(N( 0),N(2)))
    Etst(Floor(N( 3),N(7)),PAIR(N( 0),N(3)))
    Etst(Floor(N( 4),N(7)),PAIR(N( 0),N(4)))
    Etst(Floor(N( 5),N(7)),PAIR(N( 0),N(5)))
    Etst(Floor(N( 6),N(7)),PAIR(N( 0),N(6)))
    Etst(Floor(N( 7),N(7)),PAIR(N( 1),N(0)))
    Etst(Floor(N( 8),N(7)),PAIR(N( 1),N(1)))
    Etst(Floor(N( 9),N(7)),PAIR(N( 1),N(2)))

    Etst(Floor(N(-9),N(8)),PAIR(N(-2),N(7)))
    Etst(Floor(N(-8),N(8)),PAIR(N(-1),N(0)))
    Etst(Floor(N(-7),N(8)),PAIR(N(-1),N(1)))
    Etst(Floor(N(-6),N(8)),PAIR(N(-1),N(2)))
    Etst(Floor(N(-5),N(8)),PAIR(N(-1),N(3)))
    Etst(Floor(N(-4),N(8)),PAIR(N(-1),N(4)))
    Etst(Floor(N(-3),N(8)),PAIR(N(-1),N(5)))
    Etst(Floor(N(-2),N(8)),PAIR(N(-1),N(6)))
    Etst(Floor(N(-1),N(8)),PAIR(N(-1),N(7)))
    Etst(Floor(N( 0),N(8)),PAIR(N( 0),N(0)))
    Etst(Floor(N( 1),N(8)),PAIR(N( 0),N(1)))
    Etst(Floor(N( 2),N(8)),PAIR(N( 0),N(2)))
    Etst(Floor(N( 3),N(8)),PAIR(N( 0),N(3)))
    Etst(Floor(N( 4),N(8)),PAIR(N( 0),N(4)))
    Etst(Floor(N( 5),N(8)),PAIR(N( 0),N(5)))
    Etst(Floor(N( 6),N(8)),PAIR(N( 0),N(6)))
    Etst(Floor(N( 7),N(8)),PAIR(N( 0),N(7)))
    Etst(Floor(N( 8),N(8)),PAIR(N( 1),N(0)))
    Etst(Floor(N( 9),N(8)),PAIR(N( 1),N(1)))

    Etst(Floor(N(-9),N(9)),PAIR(N(-1),N(0)))
    Etst(Floor(N(-8),N(9)),PAIR(N(-1),N(1)))
    Etst(Floor(N(-7),N(9)),PAIR(N(-1),N(2)))
    Etst(Floor(N(-6),N(9)),PAIR(N(-1),N(3)))
    Etst(Floor(N(-5),N(9)),PAIR(N(-1),N(4)))
    Etst(Floor(N(-4),N(9)),PAIR(N(-1),N(5)))
    Etst(Floor(N(-3),N(9)),PAIR(N(-1),N(6)))
    Etst(Floor(N(-2),N(9)),PAIR(N(-1),N(7)))
    Etst(Floor(N(-1),N(9)),PAIR(N(-1),N(8)))
    Etst(Floor(N( 0),N(9)),PAIR(N( 0),N(0)))
    Etst(Floor(N( 1),N(9)),PAIR(N( 0),N(1)))
    Etst(Floor(N( 2),N(9)),PAIR(N( 0),N(2)))
    Etst(Floor(N( 3),N(9)),PAIR(N( 0),N(3)))
    Etst(Floor(N( 4),N(9)),PAIR(N( 0),N(4)))
    Etst(Floor(N( 5),N(9)),PAIR(N( 0),N(5)))
    Etst(Floor(N( 6),N(9)),PAIR(N( 0),N(6)))
    Etst(Floor(N( 7),N(9)),PAIR(N( 0),N(7)))
    Etst(Floor(N( 8),N(9)),PAIR(N( 0),N(8)))
    Etst(Floor(N( 9),N(9)),PAIR(N( 1),N(0)))

    Xtst(Ceiling(TT,TT))
    Xtst(Ceiling(TT,FF))
    Xtst(Ceiling(TT,I0))
    Xtst(Ceiling(TT,P0))
    Xtst(Ceiling(TT,M0))
    Xtst(Ceiling(TT,B0))
    Xtst(Ceiling(FF,TT))
    Xtst(Ceiling(FF,FF))
    Xtst(Ceiling(FF,I0))
    Xtst(Ceiling(FF,P0))
    Xtst(Ceiling(FF,M0))
    Xtst(Ceiling(FF,B0))
    Xtst(Ceiling(I0,TT))
    Xtst(Ceiling(I0,FF))
    Xtst(Ceiling(I0,P0))
    Xtst(Ceiling(I0,M0))
    Xtst(Ceiling(I0,B0))
    Xtst(Ceiling(P0,TT))
    Xtst(Ceiling(P0,FF))
    Xtst(Ceiling(P0,I0))
    Xtst(Ceiling(P0,P0))
    Xtst(Ceiling(P0,M0))
    Xtst(Ceiling(P0,B0))
    Xtst(Ceiling(M0,TT))
    Xtst(Ceiling(M0,FF))
    Xtst(Ceiling(M0,I0))
    Xtst(Ceiling(M0,P0))
    Xtst(Ceiling(M0,M0))
    Xtst(Ceiling(M0,B0))
    Xtst(Ceiling(B0,TT))
    Xtst(Ceiling(B0,FF))
    Xtst(Ceiling(B0,I0))
    Xtst(Ceiling(B0,P0))
    Xtst(Ceiling(B0,M0))
    Xtst(Ceiling(B0,B0))
    Xtst(Ceiling(I0,Z))
    Xtst(Ceiling(I0,M))

    Etst(Ceiling(N( 9),N(1)),PAIR(N( 9),N( 0)))
    Etst(Ceiling(N( 8),N(1)),PAIR(N( 8),N( 0)))
    Etst(Ceiling(N( 7),N(1)),PAIR(N( 7),N( 0)))
    Etst(Ceiling(N( 6),N(1)),PAIR(N( 6),N( 0)))
    Etst(Ceiling(N( 5),N(1)),PAIR(N( 5),N( 0)))
    Etst(Ceiling(N( 4),N(1)),PAIR(N( 4),N( 0)))
    Etst(Ceiling(N( 3),N(1)),PAIR(N( 3),N( 0)))
    Etst(Ceiling(N( 2),N(1)),PAIR(N( 2),N( 0)))
    Etst(Ceiling(N( 1),N(1)),PAIR(N( 1),N( 0)))
    Etst(Ceiling(N( 0),N(1)),PAIR(N( 0),N( 0)))
    Etst(Ceiling(N(-1),N(1)),PAIR(N(-1),N( 0)))
    Etst(Ceiling(N(-2),N(1)),PAIR(N(-2),N( 0)))
    Etst(Ceiling(N(-3),N(1)),PAIR(N(-3),N( 0)))
    Etst(Ceiling(N(-4),N(1)),PAIR(N(-4),N( 0)))
    Etst(Ceiling(N(-5),N(1)),PAIR(N(-5),N( 0)))
    Etst(Ceiling(N(-6),N(1)),PAIR(N(-6),N( 0)))
    Etst(Ceiling(N(-7),N(1)),PAIR(N(-7),N( 0)))
    Etst(Ceiling(N(-8),N(1)),PAIR(N(-8),N( 0)))
    Etst(Ceiling(N(-9),N(1)),PAIR(N(-9),N( 0)))

    Etst(Ceiling(N( 9),N(2)),PAIR(N( 5),N(-1)))
    Etst(Ceiling(N( 8),N(2)),PAIR(N( 4),N( 0)))
    Etst(Ceiling(N( 7),N(2)),PAIR(N( 4),N(-1)))
    Etst(Ceiling(N( 6),N(2)),PAIR(N( 3),N( 0)))
    Etst(Ceiling(N( 5),N(2)),PAIR(N( 3),N(-1)))
    Etst(Ceiling(N( 4),N(2)),PAIR(N( 2),N( 0)))
    Etst(Ceiling(N( 3),N(2)),PAIR(N( 2),N(-1)))
    Etst(Ceiling(N( 2),N(2)),PAIR(N( 1),N( 0)))
    Etst(Ceiling(N( 1),N(2)),PAIR(N( 1),N(-1)))
    Etst(Ceiling(N( 0),N(2)),PAIR(N( 0),N( 0)))
    Etst(Ceiling(N(-1),N(2)),PAIR(N( 0),N(-1)))
    Etst(Ceiling(N(-2),N(2)),PAIR(N(-1),N( 0)))
    Etst(Ceiling(N(-3),N(2)),PAIR(N(-1),N(-1)))
    Etst(Ceiling(N(-4),N(2)),PAIR(N(-2),N( 0)))
    Etst(Ceiling(N(-5),N(2)),PAIR(N(-2),N(-1)))
    Etst(Ceiling(N(-6),N(2)),PAIR(N(-3),N( 0)))
    Etst(Ceiling(N(-7),N(2)),PAIR(N(-3),N(-1)))
    Etst(Ceiling(N(-8),N(2)),PAIR(N(-4),N( 0)))
    Etst(Ceiling(N(-9),N(2)),PAIR(N(-4),N(-1)))

    Etst(Ceiling(N( 9),N(3)),PAIR(N( 3),N( 0)))
    Etst(Ceiling(N( 8),N(3)),PAIR(N( 3),N(-1)))
    Etst(Ceiling(N( 7),N(3)),PAIR(N( 3),N(-2)))
    Etst(Ceiling(N( 6),N(3)),PAIR(N( 2),N( 0)))
    Etst(Ceiling(N( 5),N(3)),PAIR(N( 2),N(-1)))
    Etst(Ceiling(N( 4),N(3)),PAIR(N( 2),N(-2)))
    Etst(Ceiling(N( 3),N(3)),PAIR(N( 1),N( 0)))
    Etst(Ceiling(N( 2),N(3)),PAIR(N( 1),N(-1)))
    Etst(Ceiling(N( 1),N(3)),PAIR(N( 1),N(-2)))
    Etst(Ceiling(N( 0),N(3)),PAIR(N( 0),N( 0)))
    Etst(Ceiling(N(-1),N(3)),PAIR(N( 0),N(-1)))
    Etst(Ceiling(N(-2),N(3)),PAIR(N( 0),N(-2)))
    Etst(Ceiling(N(-3),N(3)),PAIR(N(-1),N( 0)))
    Etst(Ceiling(N(-4),N(3)),PAIR(N(-1),N(-1)))
    Etst(Ceiling(N(-5),N(3)),PAIR(N(-1),N(-2)))
    Etst(Ceiling(N(-6),N(3)),PAIR(N(-2),N( 0)))
    Etst(Ceiling(N(-7),N(3)),PAIR(N(-2),N(-1)))
    Etst(Ceiling(N(-8),N(3)),PAIR(N(-2),N(-2)))
    Etst(Ceiling(N(-9),N(3)),PAIR(N(-3),N( 0)))

    Etst(Ceiling(N( 9),N(4)),PAIR(N( 3),N(-3)))
    Etst(Ceiling(N( 8),N(4)),PAIR(N( 2),N( 0)))
    Etst(Ceiling(N( 7),N(4)),PAIR(N( 2),N(-1)))
    Etst(Ceiling(N( 6),N(4)),PAIR(N( 2),N(-2)))
    Etst(Ceiling(N( 5),N(4)),PAIR(N( 2),N(-3)))
    Etst(Ceiling(N( 4),N(4)),PAIR(N( 1),N( 0)))
    Etst(Ceiling(N( 3),N(4)),PAIR(N( 1),N(-1)))
    Etst(Ceiling(N( 2),N(4)),PAIR(N( 1),N(-2)))
    Etst(Ceiling(N( 1),N(4)),PAIR(N( 1),N(-3)))
    Etst(Ceiling(N( 0),N(4)),PAIR(N( 0),N( 0)))
    Etst(Ceiling(N(-1),N(4)),PAIR(N( 0),N(-1)))
    Etst(Ceiling(N(-2),N(4)),PAIR(N( 0),N(-2)))
    Etst(Ceiling(N(-3),N(4)),PAIR(N( 0),N(-3)))
    Etst(Ceiling(N(-4),N(4)),PAIR(N(-1),N( 0)))
    Etst(Ceiling(N(-5),N(4)),PAIR(N(-1),N(-1)))
    Etst(Ceiling(N(-6),N(4)),PAIR(N(-1),N(-2)))
    Etst(Ceiling(N(-7),N(4)),PAIR(N(-1),N(-3)))
    Etst(Ceiling(N(-8),N(4)),PAIR(N(-2),N( 0)))
    Etst(Ceiling(N(-9),N(4)),PAIR(N(-2),N(-1)))

    Etst(Ceiling(N( 9),N(5)),PAIR(N( 2),N(-1)))
    Etst(Ceiling(N( 8),N(5)),PAIR(N( 2),N(-2)))
    Etst(Ceiling(N( 7),N(5)),PAIR(N( 2),N(-3)))
    Etst(Ceiling(N( 6),N(5)),PAIR(N( 2),N(-4)))
    Etst(Ceiling(N( 5),N(5)),PAIR(N( 1),N( 0)))
    Etst(Ceiling(N( 4),N(5)),PAIR(N( 1),N(-1)))
    Etst(Ceiling(N( 3),N(5)),PAIR(N( 1),N(-2)))
    Etst(Ceiling(N( 2),N(5)),PAIR(N( 1),N(-3)))
    Etst(Ceiling(N( 1),N(5)),PAIR(N( 1),N(-4)))
    Etst(Ceiling(N( 0),N(5)),PAIR(N( 0),N( 0)))
    Etst(Ceiling(N(-1),N(5)),PAIR(N( 0),N(-1)))
    Etst(Ceiling(N(-2),N(5)),PAIR(N( 0),N(-2)))
    Etst(Ceiling(N(-3),N(5)),PAIR(N( 0),N(-3)))
    Etst(Ceiling(N(-4),N(5)),PAIR(N( 0),N(-4)))
    Etst(Ceiling(N(-5),N(5)),PAIR(N(-1),N( 0)))
    Etst(Ceiling(N(-6),N(5)),PAIR(N(-1),N(-1)))
    Etst(Ceiling(N(-7),N(5)),PAIR(N(-1),N(-2)))
    Etst(Ceiling(N(-8),N(5)),PAIR(N(-1),N(-3)))
    Etst(Ceiling(N(-9),N(5)),PAIR(N(-1),N(-4)))

    Etst(Ceiling(N( 9),N(6)),PAIR(N( 2),N(-3)))
    Etst(Ceiling(N( 8),N(6)),PAIR(N( 2),N(-4)))
    Etst(Ceiling(N( 7),N(6)),PAIR(N( 2),N(-5)))
    Etst(Ceiling(N( 6),N(6)),PAIR(N( 1),N( 0)))
    Etst(Ceiling(N( 5),N(6)),PAIR(N( 1),N(-1)))
    Etst(Ceiling(N( 4),N(6)),PAIR(N( 1),N(-2)))
    Etst(Ceiling(N( 3),N(6)),PAIR(N( 1),N(-3)))
    Etst(Ceiling(N( 2),N(6)),PAIR(N( 1),N(-4)))
    Etst(Ceiling(N( 1),N(6)),PAIR(N( 1),N(-5)))
    Etst(Ceiling(N( 0),N(6)),PAIR(N( 0),N( 0)))
    Etst(Ceiling(N(-1),N(6)),PAIR(N( 0),N(-1)))
    Etst(Ceiling(N(-2),N(6)),PAIR(N( 0),N(-2)))
    Etst(Ceiling(N(-3),N(6)),PAIR(N( 0),N(-3)))
    Etst(Ceiling(N(-4),N(6)),PAIR(N( 0),N(-4)))
    Etst(Ceiling(N(-5),N(6)),PAIR(N( 0),N(-5)))
    Etst(Ceiling(N(-6),N(6)),PAIR(N(-1),N( 0)))
    Etst(Ceiling(N(-7),N(6)),PAIR(N(-1),N(-1)))
    Etst(Ceiling(N(-8),N(6)),PAIR(N(-1),N(-2)))
    Etst(Ceiling(N(-9),N(6)),PAIR(N(-1),N(-3)))

    Etst(Ceiling(N( 9),N(7)),PAIR(N( 2),N(-5)))
    Etst(Ceiling(N( 8),N(7)),PAIR(N( 2),N(-6)))
    Etst(Ceiling(N( 7),N(7)),PAIR(N( 1),N( 0)))
    Etst(Ceiling(N( 6),N(7)),PAIR(N( 1),N(-1)))
    Etst(Ceiling(N( 5),N(7)),PAIR(N( 1),N(-2)))
    Etst(Ceiling(N( 4),N(7)),PAIR(N( 1),N(-3)))
    Etst(Ceiling(N( 3),N(7)),PAIR(N( 1),N(-4)))
    Etst(Ceiling(N( 2),N(7)),PAIR(N( 1),N(-5)))
    Etst(Ceiling(N( 1),N(7)),PAIR(N( 1),N(-6)))
    Etst(Ceiling(N( 0),N(7)),PAIR(N( 0),N( 0)))
    Etst(Ceiling(N(-1),N(7)),PAIR(N( 0),N(-1)))
    Etst(Ceiling(N(-2),N(7)),PAIR(N( 0),N(-2)))
    Etst(Ceiling(N(-3),N(7)),PAIR(N( 0),N(-3)))
    Etst(Ceiling(N(-4),N(7)),PAIR(N( 0),N(-4)))
    Etst(Ceiling(N(-5),N(7)),PAIR(N( 0),N(-5)))
    Etst(Ceiling(N(-6),N(7)),PAIR(N( 0),N(-6)))
    Etst(Ceiling(N(-7),N(7)),PAIR(N(-1),N( 0)))
    Etst(Ceiling(N(-8),N(7)),PAIR(N(-1),N(-1)))
    Etst(Ceiling(N(-9),N(7)),PAIR(N(-1),N(-2)))

    Etst(Ceiling(N( 9),N(8)),PAIR(N( 2),N(-7)))
    Etst(Ceiling(N( 8),N(8)),PAIR(N( 1),N( 0)))
    Etst(Ceiling(N( 7),N(8)),PAIR(N( 1),N(-1)))
    Etst(Ceiling(N( 6),N(8)),PAIR(N( 1),N(-2)))
    Etst(Ceiling(N( 5),N(8)),PAIR(N( 1),N(-3)))
    Etst(Ceiling(N( 4),N(8)),PAIR(N( 1),N(-4)))
    Etst(Ceiling(N( 3),N(8)),PAIR(N( 1),N(-5)))
    Etst(Ceiling(N( 2),N(8)),PAIR(N( 1),N(-6)))
    Etst(Ceiling(N( 1),N(8)),PAIR(N( 1),N(-7)))
    Etst(Ceiling(N( 0),N(8)),PAIR(N( 0),N( 0)))
    Etst(Ceiling(N(-1),N(8)),PAIR(N( 0),N(-1)))
    Etst(Ceiling(N(-2),N(8)),PAIR(N( 0),N(-2)))
    Etst(Ceiling(N(-3),N(8)),PAIR(N( 0),N(-3)))
    Etst(Ceiling(N(-4),N(8)),PAIR(N( 0),N(-4)))
    Etst(Ceiling(N(-5),N(8)),PAIR(N( 0),N(-5)))
    Etst(Ceiling(N(-6),N(8)),PAIR(N( 0),N(-6)))
    Etst(Ceiling(N(-7),N(8)),PAIR(N( 0),N(-7)))
    Etst(Ceiling(N(-8),N(8)),PAIR(N(-1),N( 0)))
    Etst(Ceiling(N(-9),N(8)),PAIR(N(-1),N(-1)))

    Etst(Ceiling(N( 9),N(9)),PAIR(N( 1),N( 0)))
    Etst(Ceiling(N( 8),N(9)),PAIR(N( 1),N(-1)))
    Etst(Ceiling(N( 7),N(9)),PAIR(N( 1),N(-2)))
    Etst(Ceiling(N( 6),N(9)),PAIR(N( 1),N(-3)))
    Etst(Ceiling(N( 5),N(9)),PAIR(N( 1),N(-4)))
    Etst(Ceiling(N( 4),N(9)),PAIR(N( 1),N(-5)))
    Etst(Ceiling(N( 3),N(9)),PAIR(N( 1),N(-6)))
    Etst(Ceiling(N( 2),N(9)),PAIR(N( 1),N(-7)))
    Etst(Ceiling(N( 1),N(9)),PAIR(N( 1),N(-8)))
    Etst(Ceiling(N( 0),N(9)),PAIR(N( 0),N( 0)))
    Etst(Ceiling(N(-1),N(9)),PAIR(N( 0),N(-1)))
    Etst(Ceiling(N(-2),N(9)),PAIR(N( 0),N(-2)))
    Etst(Ceiling(N(-3),N(9)),PAIR(N( 0),N(-3)))
    Etst(Ceiling(N(-4),N(9)),PAIR(N( 0),N(-4)))
    Etst(Ceiling(N(-5),N(9)),PAIR(N( 0),N(-5)))
    Etst(Ceiling(N(-6),N(9)),PAIR(N( 0),N(-6)))
    Etst(Ceiling(N(-7),N(9)),PAIR(N( 0),N(-7)))
    Etst(Ceiling(N(-8),N(9)),PAIR(N( 0),N(-8)))
    Etst(Ceiling(N(-9),N(9)),PAIR(N(-1),N( 0)))

    Xtst(Truncate(TT,TT))
    Xtst(Truncate(TT,FF))
    Xtst(Truncate(TT,I0))
    Xtst(Truncate(TT,P0))
    Xtst(Truncate(TT,M0))
    Xtst(Truncate(TT,B0))
    Xtst(Truncate(FF,TT))
    Xtst(Truncate(FF,FF))
    Xtst(Truncate(FF,I0))
    Xtst(Truncate(FF,P0))
    Xtst(Truncate(FF,M0))
    Xtst(Truncate(FF,B0))
    Xtst(Truncate(I0,TT))
    Xtst(Truncate(I0,FF))
    Xtst(Truncate(I0,P0))
    Xtst(Truncate(I0,M0))
    Xtst(Truncate(I0,B0))
    Xtst(Truncate(P0,TT))
    Xtst(Truncate(P0,FF))
    Xtst(Truncate(P0,I0))
    Xtst(Truncate(P0,P0))
    Xtst(Truncate(P0,M0))
    Xtst(Truncate(P0,B0))
    Xtst(Truncate(M0,TT))
    Xtst(Truncate(M0,FF))
    Xtst(Truncate(M0,I0))
    Xtst(Truncate(M0,P0))
    Xtst(Truncate(M0,M0))
    Xtst(Truncate(M0,B0))
    Xtst(Truncate(B0,TT))
    Xtst(Truncate(B0,FF))
    Xtst(Truncate(B0,I0))
    Xtst(Truncate(B0,P0))
    Xtst(Truncate(B0,M0))
    Xtst(Truncate(B0,B0))
    Xtst(Truncate(I0,Z))
    Xtst(Truncate(I0,M))

    Etst(Truncate(N(-9),N(1)),PAIR(N(-9),N( 0)))
    Etst(Truncate(N(-8),N(1)),PAIR(N(-8),N( 0)))
    Etst(Truncate(N(-7),N(1)),PAIR(N(-7),N( 0)))
    Etst(Truncate(N(-6),N(1)),PAIR(N(-6),N( 0)))
    Etst(Truncate(N(-5),N(1)),PAIR(N(-5),N( 0)))
    Etst(Truncate(N(-4),N(1)),PAIR(N(-4),N( 0)))
    Etst(Truncate(N(-3),N(1)),PAIR(N(-3),N( 0)))
    Etst(Truncate(N(-2),N(1)),PAIR(N(-2),N( 0)))
    Etst(Truncate(N(-1),N(1)),PAIR(N(-1),N( 0)))
    Etst(Truncate(N( 0),N(1)),PAIR(N( 0),N( 0)))
    Etst(Truncate(N( 1),N(1)),PAIR(N( 1),N( 0)))
    Etst(Truncate(N( 2),N(1)),PAIR(N( 2),N( 0)))
    Etst(Truncate(N( 3),N(1)),PAIR(N( 3),N( 0)))
    Etst(Truncate(N( 4),N(1)),PAIR(N( 4),N( 0)))
    Etst(Truncate(N( 5),N(1)),PAIR(N( 5),N( 0)))
    Etst(Truncate(N( 6),N(1)),PAIR(N( 6),N( 0)))
    Etst(Truncate(N( 7),N(1)),PAIR(N( 7),N( 0)))
    Etst(Truncate(N( 8),N(1)),PAIR(N( 8),N( 0)))
    Etst(Truncate(N( 9),N(1)),PAIR(N( 9),N( 0)))

    Etst(Truncate(N(-9),N(2)),PAIR(N(-4),N(-1)))
    Etst(Truncate(N(-8),N(2)),PAIR(N(-4),N( 0)))
    Etst(Truncate(N(-7),N(2)),PAIR(N(-3),N(-1)))
    Etst(Truncate(N(-6),N(2)),PAIR(N(-3),N( 0)))
    Etst(Truncate(N(-5),N(2)),PAIR(N(-2),N(-1)))
    Etst(Truncate(N(-4),N(2)),PAIR(N(-2),N( 0)))
    Etst(Truncate(N(-3),N(2)),PAIR(N(-1),N(-1)))
    Etst(Truncate(N(-2),N(2)),PAIR(N(-1),N( 0)))
    Etst(Truncate(N(-1),N(2)),PAIR(N( 0),N(-1)))
    Etst(Truncate(N( 0),N(2)),PAIR(N( 0),N( 0)))
    Etst(Truncate(N( 1),N(2)),PAIR(N( 0),N( 1)))
    Etst(Truncate(N( 2),N(2)),PAIR(N( 1),N( 0)))
    Etst(Truncate(N( 3),N(2)),PAIR(N( 1),N( 1)))
    Etst(Truncate(N( 4),N(2)),PAIR(N( 2),N( 0)))
    Etst(Truncate(N( 5),N(2)),PAIR(N( 2),N( 1)))
    Etst(Truncate(N( 6),N(2)),PAIR(N( 3),N( 0)))
    Etst(Truncate(N( 7),N(2)),PAIR(N( 3),N( 1)))
    Etst(Truncate(N( 8),N(2)),PAIR(N( 4),N( 0)))
    Etst(Truncate(N( 9),N(2)),PAIR(N( 4),N( 1)))

    Etst(Truncate(N(-9),N(3)),PAIR(N(-3),N( 0)))
    Etst(Truncate(N(-8),N(3)),PAIR(N(-2),N(-2)))
    Etst(Truncate(N(-7),N(3)),PAIR(N(-2),N(-1)))
    Etst(Truncate(N(-6),N(3)),PAIR(N(-2),N( 0)))
    Etst(Truncate(N(-5),N(3)),PAIR(N(-1),N(-2)))
    Etst(Truncate(N(-4),N(3)),PAIR(N(-1),N(-1)))
    Etst(Truncate(N(-3),N(3)),PAIR(N(-1),N( 0)))
    Etst(Truncate(N(-2),N(3)),PAIR(N( 0),N(-2)))
    Etst(Truncate(N(-1),N(3)),PAIR(N( 0),N(-1)))
    Etst(Truncate(N( 0),N(3)),PAIR(N( 0),N( 0)))
    Etst(Truncate(N( 1),N(3)),PAIR(N( 0),N( 1)))
    Etst(Truncate(N( 2),N(3)),PAIR(N( 0),N( 2)))
    Etst(Truncate(N( 3),N(3)),PAIR(N( 1),N( 0)))
    Etst(Truncate(N( 4),N(3)),PAIR(N( 1),N( 1)))
    Etst(Truncate(N( 5),N(3)),PAIR(N( 1),N( 2)))
    Etst(Truncate(N( 6),N(3)),PAIR(N( 2),N( 0)))
    Etst(Truncate(N( 7),N(3)),PAIR(N( 2),N( 1)))
    Etst(Truncate(N( 8),N(3)),PAIR(N( 2),N( 2)))
    Etst(Truncate(N( 9),N(3)),PAIR(N( 3),N( 0)))

    Etst(Truncate(N(-9),N(4)),PAIR(N(-2),N(-1)))
    Etst(Truncate(N(-8),N(4)),PAIR(N(-2),N( 0)))
    Etst(Truncate(N(-7),N(4)),PAIR(N(-1),N(-3)))
    Etst(Truncate(N(-6),N(4)),PAIR(N(-1),N(-2)))
    Etst(Truncate(N(-5),N(4)),PAIR(N(-1),N(-1)))
    Etst(Truncate(N(-4),N(4)),PAIR(N(-1),N( 0)))
    Etst(Truncate(N(-3),N(4)),PAIR(N( 0),N(-3)))
    Etst(Truncate(N(-2),N(4)),PAIR(N( 0),N(-2)))
    Etst(Truncate(N(-1),N(4)),PAIR(N( 0),N(-1)))
    Etst(Truncate(N( 0),N(4)),PAIR(N( 0),N( 0)))
    Etst(Truncate(N( 1),N(4)),PAIR(N( 0),N( 1)))
    Etst(Truncate(N( 2),N(4)),PAIR(N( 0),N( 2)))
    Etst(Truncate(N( 3),N(4)),PAIR(N( 0),N( 3)))
    Etst(Truncate(N( 4),N(4)),PAIR(N( 1),N( 0)))
    Etst(Truncate(N( 5),N(4)),PAIR(N( 1),N( 1)))
    Etst(Truncate(N( 6),N(4)),PAIR(N( 1),N( 2)))
    Etst(Truncate(N( 7),N(4)),PAIR(N( 1),N( 3)))
    Etst(Truncate(N( 8),N(4)),PAIR(N( 2),N( 0)))
    Etst(Truncate(N( 9),N(4)),PAIR(N( 2),N( 1)))

    Etst(Truncate(N(-9),N(5)),PAIR(N(-1),N(-4)))
    Etst(Truncate(N(-8),N(5)),PAIR(N(-1),N(-3)))
    Etst(Truncate(N(-7),N(5)),PAIR(N(-1),N(-2)))
    Etst(Truncate(N(-6),N(5)),PAIR(N(-1),N(-1)))
    Etst(Truncate(N(-5),N(5)),PAIR(N(-1),N( 0)))
    Etst(Truncate(N(-4),N(5)),PAIR(N( 0),N(-4)))
    Etst(Truncate(N(-3),N(5)),PAIR(N( 0),N(-3)))
    Etst(Truncate(N(-2),N(5)),PAIR(N( 0),N(-2)))
    Etst(Truncate(N(-1),N(5)),PAIR(N( 0),N(-1)))
    Etst(Truncate(N( 0),N(5)),PAIR(N( 0),N( 0)))
    Etst(Truncate(N( 1),N(5)),PAIR(N( 0),N( 1)))
    Etst(Truncate(N( 2),N(5)),PAIR(N( 0),N( 2)))
    Etst(Truncate(N( 3),N(5)),PAIR(N( 0),N( 3)))
    Etst(Truncate(N( 4),N(5)),PAIR(N( 0),N( 4)))
    Etst(Truncate(N( 5),N(5)),PAIR(N( 1),N( 0)))
    Etst(Truncate(N( 6),N(5)),PAIR(N( 1),N( 1)))
    Etst(Truncate(N( 7),N(5)),PAIR(N( 1),N( 2)))
    Etst(Truncate(N( 8),N(5)),PAIR(N( 1),N( 3)))
    Etst(Truncate(N( 9),N(5)),PAIR(N( 1),N( 4)))

    Etst(Truncate(N(-9),N(6)),PAIR(N(-1),N(-3)))
    Etst(Truncate(N(-8),N(6)),PAIR(N(-1),N(-2)))
    Etst(Truncate(N(-7),N(6)),PAIR(N(-1),N(-1)))
    Etst(Truncate(N(-6),N(6)),PAIR(N(-1),N( 0)))
    Etst(Truncate(N(-5),N(6)),PAIR(N( 0),N(-5)))
    Etst(Truncate(N(-4),N(6)),PAIR(N( 0),N(-4)))
    Etst(Truncate(N(-3),N(6)),PAIR(N( 0),N(-3)))
    Etst(Truncate(N(-2),N(6)),PAIR(N( 0),N(-2)))
    Etst(Truncate(N(-1),N(6)),PAIR(N( 0),N(-1)))
    Etst(Truncate(N( 0),N(6)),PAIR(N( 0),N( 0)))
    Etst(Truncate(N( 1),N(6)),PAIR(N( 0),N( 1)))
    Etst(Truncate(N( 2),N(6)),PAIR(N( 0),N( 2)))
    Etst(Truncate(N( 3),N(6)),PAIR(N( 0),N( 3)))
    Etst(Truncate(N( 4),N(6)),PAIR(N( 0),N( 4)))
    Etst(Truncate(N( 5),N(6)),PAIR(N( 0),N( 5)))
    Etst(Truncate(N( 6),N(6)),PAIR(N( 1),N( 0)))
    Etst(Truncate(N( 7),N(6)),PAIR(N( 1),N( 1)))
    Etst(Truncate(N( 8),N(6)),PAIR(N( 1),N( 2)))
    Etst(Truncate(N( 9),N(6)),PAIR(N( 1),N( 3)))

    Etst(Truncate(N(-9),N(7)),PAIR(N(-1),N(-2)))
    Etst(Truncate(N(-8),N(7)),PAIR(N(-1),N(-1)))
    Etst(Truncate(N(-7),N(7)),PAIR(N(-1),N( 0)))
    Etst(Truncate(N(-6),N(7)),PAIR(N( 0),N(-6)))
    Etst(Truncate(N(-5),N(7)),PAIR(N( 0),N(-5)))
    Etst(Truncate(N(-4),N(7)),PAIR(N( 0),N(-4)))
    Etst(Truncate(N(-3),N(7)),PAIR(N( 0),N(-3)))
    Etst(Truncate(N(-2),N(7)),PAIR(N( 0),N(-2)))
    Etst(Truncate(N(-1),N(7)),PAIR(N( 0),N(-1)))
    Etst(Truncate(N( 0),N(7)),PAIR(N( 0),N( 0)))
    Etst(Truncate(N( 1),N(7)),PAIR(N( 0),N( 1)))
    Etst(Truncate(N( 2),N(7)),PAIR(N( 0),N( 2)))
    Etst(Truncate(N( 3),N(7)),PAIR(N( 0),N( 3)))
    Etst(Truncate(N( 4),N(7)),PAIR(N( 0),N( 4)))
    Etst(Truncate(N( 5),N(7)),PAIR(N( 0),N( 5)))
    Etst(Truncate(N( 6),N(7)),PAIR(N( 0),N( 6)))
    Etst(Truncate(N( 7),N(7)),PAIR(N( 1),N( 0)))
    Etst(Truncate(N( 8),N(7)),PAIR(N( 1),N( 1)))
    Etst(Truncate(N( 9),N(7)),PAIR(N( 1),N( 2)))

    Etst(Truncate(N(-9),N(8)),PAIR(N(-1),N(-1)))
    Etst(Truncate(N(-8),N(8)),PAIR(N(-1),N( 0)))
    Etst(Truncate(N(-7),N(8)),PAIR(N( 0),N(-7)))
    Etst(Truncate(N(-6),N(8)),PAIR(N( 0),N(-6)))
    Etst(Truncate(N(-5),N(8)),PAIR(N( 0),N(-5)))
    Etst(Truncate(N(-4),N(8)),PAIR(N( 0),N(-4)))
    Etst(Truncate(N(-3),N(8)),PAIR(N( 0),N(-3)))
    Etst(Truncate(N(-2),N(8)),PAIR(N( 0),N(-2)))
    Etst(Truncate(N(-1),N(8)),PAIR(N( 0),N(-1)))
    Etst(Truncate(N( 0),N(8)),PAIR(N( 0),N( 0)))
    Etst(Truncate(N( 1),N(8)),PAIR(N( 0),N( 1)))
    Etst(Truncate(N( 2),N(8)),PAIR(N( 0),N( 2)))
    Etst(Truncate(N( 3),N(8)),PAIR(N( 0),N( 3)))
    Etst(Truncate(N( 4),N(8)),PAIR(N( 0),N( 4)))
    Etst(Truncate(N( 5),N(8)),PAIR(N( 0),N( 5)))
    Etst(Truncate(N( 6),N(8)),PAIR(N( 0),N( 6)))
    Etst(Truncate(N( 7),N(8)),PAIR(N( 0),N( 7)))
    Etst(Truncate(N( 8),N(8)),PAIR(N( 1),N( 0)))
    Etst(Truncate(N( 9),N(8)),PAIR(N( 1),N( 1)))

    Etst(Truncate(N(-9),N(9)),PAIR(N(-1),N( 0)))
    Etst(Truncate(N(-8),N(9)),PAIR(N( 0),N(-8)))
    Etst(Truncate(N(-7),N(9)),PAIR(N( 0),N(-7)))
    Etst(Truncate(N(-6),N(9)),PAIR(N( 0),N(-6)))
    Etst(Truncate(N(-5),N(9)),PAIR(N( 0),N(-5)))
    Etst(Truncate(N(-4),N(9)),PAIR(N( 0),N(-4)))
    Etst(Truncate(N(-3),N(9)),PAIR(N( 0),N(-3)))
    Etst(Truncate(N(-2),N(9)),PAIR(N( 0),N(-2)))
    Etst(Truncate(N(-1),N(9)),PAIR(N( 0),N(-1)))
    Etst(Truncate(N( 0),N(9)),PAIR(N( 0),N( 0)))
    Etst(Truncate(N( 1),N(9)),PAIR(N( 0),N( 1)))
    Etst(Truncate(N( 2),N(9)),PAIR(N( 0),N( 2)))
    Etst(Truncate(N( 3),N(9)),PAIR(N( 0),N( 3)))
    Etst(Truncate(N( 4),N(9)),PAIR(N( 0),N( 4)))
    Etst(Truncate(N( 5),N(9)),PAIR(N( 0),N( 5)))
    Etst(Truncate(N( 6),N(9)),PAIR(N( 0),N( 6)))
    Etst(Truncate(N( 7),N(9)),PAIR(N( 0),N( 7)))
    Etst(Truncate(N( 8),N(9)),PAIR(N( 0),N( 8)))
    Etst(Truncate(N( 9),N(9)),PAIR(N( 1),N( 0)))

    Xtst(Round(TT,TT))
    Xtst(Round(TT,FF))
    Xtst(Round(TT,I0))
    Xtst(Round(TT,P0))
    Xtst(Round(TT,M0))
    Xtst(Round(TT,B0))
    Xtst(Round(FF,TT))
    Xtst(Round(FF,FF))
    Xtst(Round(FF,I0))
    Xtst(Round(FF,P0))
    Xtst(Round(FF,M0))
    Xtst(Round(FF,B0))
    Xtst(Round(I0,TT))
    Xtst(Round(I0,FF))
    Xtst(Round(I0,P0))
    Xtst(Round(I0,M0))
    Xtst(Round(I0,B0))
    Xtst(Round(P0,TT))
    Xtst(Round(P0,FF))
    Xtst(Round(P0,I0))
    Xtst(Round(P0,P0))
    Xtst(Round(P0,M0))
    Xtst(Round(P0,B0))
    Xtst(Round(M0,TT))
    Xtst(Round(M0,FF))
    Xtst(Round(M0,I0))
    Xtst(Round(M0,P0))
    Xtst(Round(M0,M0))
    Xtst(Round(M0,B0))
    Xtst(Round(B0,TT))
    Xtst(Round(B0,FF))
    Xtst(Round(B0,I0))
    Xtst(Round(B0,P0))
    Xtst(Round(B0,M0))
    Xtst(Round(B0,B0))
    Xtst(Round(I0,Z))
    Xtst(Round(I0,M))

    Etst(Round(N(-9),N(1)),PAIR(N(-9),N( 0)))
    Etst(Round(N(-8),N(1)),PAIR(N(-8),N( 0)))
    Etst(Round(N(-7),N(1)),PAIR(N(-7),N( 0)))
    Etst(Round(N(-6),N(1)),PAIR(N(-6),N( 0)))
    Etst(Round(N(-5),N(1)),PAIR(N(-5),N( 0)))
    Etst(Round(N(-4),N(1)),PAIR(N(-4),N( 0)))
    Etst(Round(N(-3),N(1)),PAIR(N(-3),N( 0)))
    Etst(Round(N(-2),N(1)),PAIR(N(-2),N( 0)))
    Etst(Round(N(-1),N(1)),PAIR(N(-1),N( 0)))
    Etst(Round(N( 9),N(1)),PAIR(N( 9),N( 0)))
    Etst(Round(N( 8),N(1)),PAIR(N( 8),N( 0)))
    Etst(Round(N( 7),N(1)),PAIR(N( 7),N( 0)))
    Etst(Round(N( 6),N(1)),PAIR(N( 6),N( 0)))
    Etst(Round(N( 5),N(1)),PAIR(N( 5),N( 0)))
    Etst(Round(N( 4),N(1)),PAIR(N( 4),N( 0)))
    Etst(Round(N( 3),N(1)),PAIR(N( 3),N( 0)))
    Etst(Round(N( 2),N(1)),PAIR(N( 2),N( 0)))
    Etst(Round(N( 1),N(1)),PAIR(N( 1),N( 0)))
    Etst(Round(N( 0),N(1)),PAIR(N( 0),N( 0)))

    Etst(Round(N(-9),N(2)),PAIR(N(-4),N(-1)))
    Etst(Round(N(-8),N(2)),PAIR(N(-4),N( 0)))
    Etst(Round(N(-7),N(2)),PAIR(N(-4),N( 1)))
    Etst(Round(N(-6),N(2)),PAIR(N(-3),N( 0)))
    Etst(Round(N(-5),N(2)),PAIR(N(-2),N(-1)))
    Etst(Round(N(-4),N(2)),PAIR(N(-2),N( 0)))
    Etst(Round(N(-3),N(2)),PAIR(N(-2),N( 1)))
    Etst(Round(N(-2),N(2)),PAIR(N(-1),N( 0)))
    Etst(Round(N(-1),N(2)),PAIR(N(-0),N(-1)))
    Etst(Round(N( 9),N(2)),PAIR(N( 4),N( 1)))
    Etst(Round(N( 8),N(2)),PAIR(N( 4),N( 0)))
    Etst(Round(N( 7),N(2)),PAIR(N( 4),N(-1)))
    Etst(Round(N( 6),N(2)),PAIR(N( 3),N( 0)))
    Etst(Round(N( 5),N(2)),PAIR(N( 2),N( 1)))
    Etst(Round(N( 4),N(2)),PAIR(N( 2),N( 0)))
    Etst(Round(N( 3),N(2)),PAIR(N( 2),N(-1)))
    Etst(Round(N( 2),N(2)),PAIR(N( 1),N( 0)))
    Etst(Round(N( 1),N(2)),PAIR(N( 0),N( 1)))
    Etst(Round(N( 0),N(2)),PAIR(N( 0),N( 0)))

    Etst(Round(N(-9),N(3)),PAIR(N(-3),N( 0)))
    Etst(Round(N(-8),N(3)),PAIR(N(-3),N( 1)))
    Etst(Round(N(-7),N(3)),PAIR(N(-2),N(-1)))
    Etst(Round(N(-6),N(3)),PAIR(N(-2),N( 0)))
    Etst(Round(N(-5),N(3)),PAIR(N(-2),N( 1)))
    Etst(Round(N(-4),N(3)),PAIR(N(-1),N(-1)))
    Etst(Round(N(-3),N(3)),PAIR(N(-1),N( 0)))
    Etst(Round(N(-2),N(3)),PAIR(N(-1),N( 1)))
    Etst(Round(N(-1),N(3)),PAIR(N(-0),N(-1)))
    Etst(Round(N( 9),N(3)),PAIR(N( 3),N( 0)))
    Etst(Round(N( 8),N(3)),PAIR(N( 3),N(-1)))
    Etst(Round(N( 7),N(3)),PAIR(N( 2),N( 1)))
    Etst(Round(N( 6),N(3)),PAIR(N( 2),N( 0)))
    Etst(Round(N( 5),N(3)),PAIR(N( 2),N(-1)))
    Etst(Round(N( 4),N(3)),PAIR(N( 1),N( 1)))
    Etst(Round(N( 3),N(3)),PAIR(N( 1),N( 0)))
    Etst(Round(N( 2),N(3)),PAIR(N( 1),N(-1)))
    Etst(Round(N( 1),N(3)),PAIR(N( 0),N( 1)))
    Etst(Round(N( 0),N(3)),PAIR(N( 0),N( 0)))

    Etst(Round(N(-9),N(4)),PAIR(N(-2),N(-1)))
    Etst(Round(N(-8),N(4)),PAIR(N(-2),N( 0)))
    Etst(Round(N(-7),N(4)),PAIR(N(-2),N( 1)))
    Etst(Round(N(-6),N(4)),PAIR(N(-2),N( 2)))
    Etst(Round(N(-5),N(4)),PAIR(N(-1),N(-1)))
    Etst(Round(N(-4),N(4)),PAIR(N(-1),N( 0)))
    Etst(Round(N(-3),N(4)),PAIR(N(-1),N( 1)))
    Etst(Round(N(-2),N(4)),PAIR(N(-0),N(-2)))
    Etst(Round(N(-1),N(4)),PAIR(N(-0),N(-1)))
    Etst(Round(N( 9),N(4)),PAIR(N( 2),N( 1)))
    Etst(Round(N( 8),N(4)),PAIR(N( 2),N( 0)))
    Etst(Round(N( 7),N(4)),PAIR(N( 2),N(-1)))
    Etst(Round(N( 6),N(4)),PAIR(N( 2),N(-2)))
    Etst(Round(N( 5),N(4)),PAIR(N( 1),N( 1)))
    Etst(Round(N( 4),N(4)),PAIR(N( 1),N( 0)))
    Etst(Round(N( 3),N(4)),PAIR(N( 1),N(-1)))
    Etst(Round(N( 2),N(4)),PAIR(N( 0),N( 2)))
    Etst(Round(N( 1),N(4)),PAIR(N( 0),N( 1)))
    Etst(Round(N( 0),N(4)),PAIR(N( 0),N( 0)))

    Etst(Round(N(-9),N(5)),PAIR(N(-2),N( 1)))
    Etst(Round(N(-8),N(5)),PAIR(N(-2),N( 2)))
    Etst(Round(N(-7),N(5)),PAIR(N(-1),N(-2)))
    Etst(Round(N(-6),N(5)),PAIR(N(-1),N(-1)))
    Etst(Round(N(-5),N(5)),PAIR(N(-1),N( 0)))
    Etst(Round(N(-4),N(5)),PAIR(N(-1),N( 1)))
    Etst(Round(N(-3),N(5)),PAIR(N(-1),N( 2)))
    Etst(Round(N(-2),N(5)),PAIR(N(-0),N(-2)))
    Etst(Round(N(-1),N(5)),PAIR(N(-0),N(-1)))
    Etst(Round(N( 9),N(5)),PAIR(N( 2),N(-1)))
    Etst(Round(N( 8),N(5)),PAIR(N( 2),N(-2)))
    Etst(Round(N( 7),N(5)),PAIR(N( 1),N( 2)))
    Etst(Round(N( 6),N(5)),PAIR(N( 1),N( 1)))
    Etst(Round(N( 5),N(5)),PAIR(N( 1),N( 0)))
    Etst(Round(N( 4),N(5)),PAIR(N( 1),N(-1)))
    Etst(Round(N( 3),N(5)),PAIR(N( 1),N(-2)))
    Etst(Round(N( 2),N(5)),PAIR(N( 0),N( 2)))
    Etst(Round(N( 1),N(5)),PAIR(N( 0),N( 1)))
    Etst(Round(N( 0),N(5)),PAIR(N( 0),N( 0)))

    Etst(Round(N(-9),N(6)),PAIR(N(-2),N( 3)))
    Etst(Round(N(-8),N(6)),PAIR(N(-1),N(-2)))
    Etst(Round(N(-7),N(6)),PAIR(N(-1),N(-1)))
    Etst(Round(N(-6),N(6)),PAIR(N(-1),N( 0)))
    Etst(Round(N(-5),N(6)),PAIR(N(-1),N( 1)))
    Etst(Round(N(-4),N(6)),PAIR(N(-1),N( 2)))
    Etst(Round(N(-3),N(6)),PAIR(N(-0),N(-3)))
    Etst(Round(N(-2),N(6)),PAIR(N(-0),N(-2)))
    Etst(Round(N(-1),N(6)),PAIR(N(-0),N(-1)))
    Etst(Round(N( 9),N(6)),PAIR(N( 2),N(-3)))
    Etst(Round(N( 8),N(6)),PAIR(N( 1),N( 2)))
    Etst(Round(N( 7),N(6)),PAIR(N( 1),N( 1)))
    Etst(Round(N( 6),N(6)),PAIR(N( 1),N( 0)))
    Etst(Round(N( 5),N(6)),PAIR(N( 1),N(-1)))
    Etst(Round(N( 4),N(6)),PAIR(N( 1),N(-2)))
    Etst(Round(N( 3),N(6)),PAIR(N( 0),N( 3)))
    Etst(Round(N( 2),N(6)),PAIR(N( 0),N( 2)))
    Etst(Round(N( 1),N(6)),PAIR(N( 0),N( 1)))
    Etst(Round(N( 0),N(6)),PAIR(N( 0),N( 0)))

    Etst(Round(N(-9),N(7)),PAIR(N(-1),N(-2)))
    Etst(Round(N(-8),N(7)),PAIR(N(-1),N(-1)))
    Etst(Round(N(-7),N(7)),PAIR(N(-1),N( 0)))
    Etst(Round(N(-6),N(7)),PAIR(N(-1),N( 1)))
    Etst(Round(N(-5),N(7)),PAIR(N(-1),N( 2)))
    Etst(Round(N(-4),N(7)),PAIR(N(-1),N( 3)))
    Etst(Round(N(-3),N(7)),PAIR(N(-0),N(3-)))
    Etst(Round(N(-2),N(7)),PAIR(N(-0),N(-2)))
    Etst(Round(N(-1),N(7)),PAIR(N(-0),N(-1)))
    Etst(Round(N( 9),N(7)),PAIR(N( 1),N( 2)))
    Etst(Round(N( 8),N(7)),PAIR(N( 1),N( 1)))
    Etst(Round(N( 7),N(7)),PAIR(N( 1),N( 0)))
    Etst(Round(N( 6),N(7)),PAIR(N( 1),N(-1)))
    Etst(Round(N( 5),N(7)),PAIR(N( 1),N(-2)))
    Etst(Round(N( 4),N(7)),PAIR(N( 1),N(-3)))
    Etst(Round(N( 3),N(7)),PAIR(N( 0),N(3 )))
    Etst(Round(N( 2),N(7)),PAIR(N( 0),N( 2)))
    Etst(Round(N( 1),N(7)),PAIR(N( 0),N( 1)))
    Etst(Round(N( 0),N(7)),PAIR(N( 0),N( 0)))

    Etst(Round(N(-9),N(8)),PAIR(N(-1),N(-1)))
    Etst(Round(N(-8),N(8)),PAIR(N(-1),N( 0)))
    Etst(Round(N(-7),N(8)),PAIR(N(-1),N( 1)))
    Etst(Round(N(-6),N(8)),PAIR(N(-1),N( 2)))
    Etst(Round(N(-5),N(8)),PAIR(N(-1),N( 3)))
    Etst(Round(N(-4),N(8)),PAIR(N(-0),N(-4)))
    Etst(Round(N(-3),N(8)),PAIR(N(-0),N(-3)))
    Etst(Round(N(-2),N(8)),PAIR(N(-0),N(-2)))
    Etst(Round(N(-1),N(8)),PAIR(N(-0),N(-1)))
    Etst(Round(N( 9),N(8)),PAIR(N( 1),N( 1)))
    Etst(Round(N( 8),N(8)),PAIR(N( 1),N( 0)))
    Etst(Round(N( 7),N(8)),PAIR(N( 1),N(-1)))
    Etst(Round(N( 6),N(8)),PAIR(N( 1),N(-2)))
    Etst(Round(N( 5),N(8)),PAIR(N( 1),N(-3)))
    Etst(Round(N( 4),N(8)),PAIR(N( 0),N( 4)))
    Etst(Round(N( 3),N(8)),PAIR(N( 0),N( 3)))
    Etst(Round(N( 2),N(8)),PAIR(N( 0),N( 2)))
    Etst(Round(N( 1),N(8)),PAIR(N( 0),N( 1)))
    Etst(Round(N( 0),N(8)),PAIR(N( 0),N( 0)))

    Etst(Round(N(-9),N(9)),PAIR(N(-1),N( 0)))
    Etst(Round(N(-8),N(9)),PAIR(N(-1),N( 1)))
    Etst(Round(N(-7),N(9)),PAIR(N(-1),N( 2)))
    Etst(Round(N(-6),N(9)),PAIR(N(-1),N( 3)))
    Etst(Round(N(-5),N(9)),PAIR(N(-1),N( 4)))
    Etst(Round(N(-4),N(9)),PAIR(N(-0),N(-4)))
    Etst(Round(N(-3),N(9)),PAIR(N(-0),N(-3)))
    Etst(Round(N(-2),N(9)),PAIR(N(-0),N(-2)))
    Etst(Round(N(-1),N(9)),PAIR(N(-0),N(-1)))
    Etst(Round(N( 9),N(9)),PAIR(N( 1),N( 0)))
    Etst(Round(N( 8),N(9)),PAIR(N( 1),N(-1)))
    Etst(Round(N( 7),N(9)),PAIR(N( 1),N(-2)))
    Etst(Round(N( 6),N(9)),PAIR(N( 1),N(-3)))
    Etst(Round(N( 5),N(9)),PAIR(N( 1),N(-4)))
    Etst(Round(N( 4),N(9)),PAIR(N( 0),N( 4)))
    Etst(Round(N( 3),N(9)),PAIR(N( 0),N( 3)))
    Etst(Round(N( 2),N(9)),PAIR(N( 0),N( 2)))
    Etst(Round(N( 1),N(9)),PAIR(N( 0),N( 1)))
    Etst(Round(N( 0),N(9)),PAIR(N( 0),N( 0)))

    Xtst(Div(TT,TT))
    Xtst(Div(TT,FF))
    Xtst(Div(TT,I0))
    Xtst(Div(TT,P0))
    Xtst(Div(TT,M0))
    Xtst(Div(TT,B0))
    Xtst(Div(FF,TT))
    Xtst(Div(FF,FF))
    Xtst(Div(FF,I0))
    Xtst(Div(FF,P0))
    Xtst(Div(FF,M0))
    Xtst(Div(FF,B0))
    Xtst(Div(I0,TT))
    Xtst(Div(I0,FF))
    Xtst(Div(I0,P0))
    Xtst(Div(I0,M0))
    Xtst(Div(I0,B0))
    Xtst(Div(P0,TT))
    Xtst(Div(P0,FF))
    Xtst(Div(P0,I0))
    Xtst(Div(P0,P0))
    Xtst(Div(P0,M0))
    Xtst(Div(P0,B0))
    Xtst(Div(M0,TT))
    Xtst(Div(M0,FF))
    Xtst(Div(M0,I0))
    Xtst(Div(M0,P0))
    Xtst(Div(M0,M0))
    Xtst(Div(M0,B0))
    Xtst(Div(B0,TT))
    Xtst(Div(B0,FF))
    Xtst(Div(B0,I0))
    Xtst(Div(B0,P0))
    Xtst(Div(B0,M0))
    Xtst(Div(B0,B0))
    Xtst(Div(I0,Z))
    Xtst(Div(I0,M))

    Etst(Div(N(-9),N(1)),N(-9))
    Etst(Div(N(-8),N(1)),N(-8))
    Etst(Div(N(-7),N(1)),N(-7))
    Etst(Div(N(-6),N(1)),N(-6))
    Etst(Div(N(-5),N(1)),N(-5))
    Etst(Div(N(-4),N(1)),N(-4))
    Etst(Div(N(-3),N(1)),N(-3))
    Etst(Div(N(-2),N(1)),N(-2))
    Etst(Div(N(-1),N(1)),N(-1))
    Etst(Div(N( 0),N(1)),N( 0))
    Etst(Div(N( 1),N(1)),N( 1))
    Etst(Div(N( 2),N(1)),N( 2))
    Etst(Div(N( 3),N(1)),N( 3))
    Etst(Div(N( 4),N(1)),N( 4))
    Etst(Div(N( 5),N(1)),N( 5))
    Etst(Div(N( 6),N(1)),N( 6))
    Etst(Div(N( 7),N(1)),N( 7))
    Etst(Div(N( 8),N(1)),N( 8))
    Etst(Div(N( 9),N(1)),N( 9))

    Etst(Div(N(-9),N(2)),N(-5))
    Etst(Div(N(-8),N(2)),N(-4))
    Etst(Div(N(-7),N(2)),N(-4))
    Etst(Div(N(-6),N(2)),N(-3))
    Etst(Div(N(-5),N(2)),N(-3))
    Etst(Div(N(-4),N(2)),N(-2))
    Etst(Div(N(-3),N(2)),N(-2))
    Etst(Div(N(-2),N(2)),N(-1))
    Etst(Div(N(-1),N(2)),N(-1))
    Etst(Div(N( 0),N(2)),N( 0))
    Etst(Div(N( 1),N(2)),N( 0))
    Etst(Div(N( 2),N(2)),N( 1))
    Etst(Div(N( 3),N(2)),N( 1))
    Etst(Div(N( 4),N(2)),N( 2))
    Etst(Div(N( 5),N(2)),N( 2))
    Etst(Div(N( 6),N(2)),N( 3))
    Etst(Div(N( 7),N(2)),N( 3))
    Etst(Div(N( 8),N(2)),N( 4))
    Etst(Div(N( 9),N(2)),N( 4))

    Etst(Div(N(-9),N(3)),N(-3))
    Etst(Div(N(-8),N(3)),N(-3))
    Etst(Div(N(-7),N(3)),N(-3))
    Etst(Div(N(-6),N(3)),N(-2))
    Etst(Div(N(-5),N(3)),N(-2))
    Etst(Div(N(-4),N(3)),N(-2))
    Etst(Div(N(-3),N(3)),N(-1))
    Etst(Div(N(-2),N(3)),N(-1))
    Etst(Div(N(-1),N(3)),N(-1))
    Etst(Div(N( 0),N(3)),N( 0))
    Etst(Div(N( 1),N(3)),N( 0))
    Etst(Div(N( 2),N(3)),N( 0))
    Etst(Div(N( 3),N(3)),N( 1))
    Etst(Div(N( 4),N(3)),N( 1))
    Etst(Div(N( 5),N(3)),N( 1))
    Etst(Div(N( 6),N(3)),N( 2))
    Etst(Div(N( 7),N(3)),N( 2))
    Etst(Div(N( 8),N(3)),N( 2))
    Etst(Div(N( 9),N(3)),N( 3))

    Etst(Div(N(-9),N(4)),N(-3))
    Etst(Div(N(-8),N(4)),N(-2))
    Etst(Div(N(-7),N(4)),N(-2))
    Etst(Div(N(-6),N(4)),N(-2))
    Etst(Div(N(-5),N(4)),N(-2))
    Etst(Div(N(-4),N(4)),N(-1))
    Etst(Div(N(-3),N(4)),N(-1))
    Etst(Div(N(-2),N(4)),N(-1))
    Etst(Div(N(-1),N(4)),N(-1))
    Etst(Div(N( 0),N(4)),N( 0))
    Etst(Div(N( 1),N(4)),N( 0))
    Etst(Div(N( 2),N(4)),N( 0))
    Etst(Div(N( 3),N(4)),N( 0))
    Etst(Div(N( 4),N(4)),N( 1))
    Etst(Div(N( 5),N(4)),N( 1))
    Etst(Div(N( 6),N(4)),N( 1))
    Etst(Div(N( 7),N(4)),N( 1))
    Etst(Div(N( 8),N(4)),N( 2))
    Etst(Div(N( 9),N(4)),N( 2))

    Etst(Div(N(-9),N(5)),N(-2))
    Etst(Div(N(-8),N(5)),N(-2))
    Etst(Div(N(-7),N(5)),N(-2))
    Etst(Div(N(-6),N(5)),N(-2))
    Etst(Div(N(-5),N(5)),N(-1))
    Etst(Div(N(-4),N(5)),N(-1))
    Etst(Div(N(-3),N(5)),N(-1))
    Etst(Div(N(-2),N(5)),N(-1))
    Etst(Div(N(-1),N(5)),N(-1))
    Etst(Div(N( 0),N(5)),N( 0))
    Etst(Div(N( 1),N(5)),N( 0))
    Etst(Div(N( 2),N(5)),N( 0))
    Etst(Div(N( 3),N(5)),N( 0))
    Etst(Div(N( 4),N(5)),N( 0))
    Etst(Div(N( 5),N(5)),N( 1))
    Etst(Div(N( 6),N(5)),N( 1))
    Etst(Div(N( 7),N(5)),N( 1))
    Etst(Div(N( 8),N(5)),N( 1))
    Etst(Div(N( 9),N(5)),N( 1))

    Etst(Div(N(-9),N(6)),N(-2))
    Etst(Div(N(-8),N(6)),N(-2))
    Etst(Div(N(-7),N(6)),N(-2))
    Etst(Div(N(-6),N(6)),N(-1))
    Etst(Div(N(-5),N(6)),N(-1))
    Etst(Div(N(-4),N(6)),N(-1))
    Etst(Div(N(-3),N(6)),N(-1))
    Etst(Div(N(-2),N(6)),N(-1))
    Etst(Div(N(-1),N(6)),N(-1))
    Etst(Div(N( 0),N(6)),N( 0))
    Etst(Div(N( 1),N(6)),N( 0))
    Etst(Div(N( 2),N(6)),N( 0))
    Etst(Div(N( 3),N(6)),N( 0))
    Etst(Div(N( 4),N(6)),N( 0))
    Etst(Div(N( 5),N(6)),N( 0))
    Etst(Div(N( 6),N(6)),N( 1))
    Etst(Div(N( 7),N(6)),N( 1))
    Etst(Div(N( 8),N(6)),N( 1))
    Etst(Div(N( 9),N(6)),N( 1))

    Etst(Div(N(-9),N(7)),N(-2))
    Etst(Div(N(-8),N(7)),N(-2))
    Etst(Div(N(-7),N(7)),N(-1))
    Etst(Div(N(-6),N(7)),N(-1))
    Etst(Div(N(-5),N(7)),N(-1))
    Etst(Div(N(-4),N(7)),N(-1))
    Etst(Div(N(-3),N(7)),N(-1))
    Etst(Div(N(-2),N(7)),N(-1))
    Etst(Div(N(-1),N(7)),N(-1))
    Etst(Div(N( 0),N(7)),N( 0))
    Etst(Div(N( 1),N(7)),N( 0))
    Etst(Div(N( 2),N(7)),N( 0))
    Etst(Div(N( 3),N(7)),N( 0))
    Etst(Div(N( 4),N(7)),N( 0))
    Etst(Div(N( 5),N(7)),N( 0))
    Etst(Div(N( 6),N(7)),N( 0))
    Etst(Div(N( 7),N(7)),N( 1))
    Etst(Div(N( 8),N(7)),N( 1))
    Etst(Div(N( 9),N(7)),N( 1))

    Etst(Div(N(-9),N(8)),N(-2))
    Etst(Div(N(-8),N(8)),N(-1))
    Etst(Div(N(-7),N(8)),N(-1))
    Etst(Div(N(-6),N(8)),N(-1))
    Etst(Div(N(-5),N(8)),N(-1))
    Etst(Div(N(-4),N(8)),N(-1))
    Etst(Div(N(-3),N(8)),N(-1))
    Etst(Div(N(-2),N(8)),N(-1))
    Etst(Div(N(-1),N(8)),N(-1))
    Etst(Div(N( 0),N(8)),N( 0))
    Etst(Div(N( 1),N(8)),N( 0))
    Etst(Div(N( 2),N(8)),N( 0))
    Etst(Div(N( 3),N(8)),N( 0))
    Etst(Div(N( 4),N(8)),N( 0))
    Etst(Div(N( 5),N(8)),N( 0))
    Etst(Div(N( 6),N(8)),N( 0))
    Etst(Div(N( 7),N(8)),N( 0))
    Etst(Div(N( 8),N(8)),N( 1))
    Etst(Div(N( 9),N(8)),N( 1))

    Etst(Div(N(-9),N(9)),N(-1))
    Etst(Div(N(-8),N(9)),N(-1))
    Etst(Div(N(-7),N(9)),N(-1))
    Etst(Div(N(-6),N(9)),N(-1))
    Etst(Div(N(-5),N(9)),N(-1))
    Etst(Div(N(-4),N(9)),N(-1))
    Etst(Div(N(-3),N(9)),N(-1))
    Etst(Div(N(-2),N(9)),N(-1))
    Etst(Div(N(-1),N(9)),N(-1))
    Etst(Div(N( 0),N(9)),N( 0))
    Etst(Div(N( 1),N(9)),N( 0))
    Etst(Div(N( 2),N(9)),N( 0))
    Etst(Div(N( 3),N(9)),N( 0))
    Etst(Div(N( 4),N(9)),N( 0))
    Etst(Div(N( 5),N(9)),N( 0))
    Etst(Div(N( 6),N(9)),N( 0))
    Etst(Div(N( 7),N(9)),N( 0))
    Etst(Div(N( 8),N(9)),N( 0))
    Etst(Div(N( 9),N(9)),N( 1))

    Xtst(Mod(TT,TT))
    Xtst(Mod(TT,FF))
    Xtst(Mod(TT,I0))
    Xtst(Mod(TT,P0))
    Xtst(Mod(TT,M0))
    Xtst(Mod(TT,B0))
    Xtst(Mod(FF,TT))
    Xtst(Mod(FF,FF))
    Xtst(Mod(FF,I0))
    Xtst(Mod(FF,P0))
    Xtst(Mod(FF,M0))
    Xtst(Mod(FF,B0))
    Xtst(Mod(I0,TT))
    Xtst(Mod(I0,FF))
    Xtst(Mod(I0,P0))
    Xtst(Mod(I0,M0))
    Xtst(Mod(I0,B0))
    Xtst(Mod(P0,TT))
    Xtst(Mod(P0,FF))
    Xtst(Mod(P0,I0))
    Xtst(Mod(P0,P0))
    Xtst(Mod(P0,M0))
    Xtst(Mod(P0,B0))
    Xtst(Mod(M0,TT))
    Xtst(Mod(M0,FF))
    Xtst(Mod(M0,I0))
    Xtst(Mod(M0,P0))
    Xtst(Mod(M0,M0))
    Xtst(Mod(M0,B0))
    Xtst(Mod(B0,TT))
    Xtst(Mod(B0,FF))
    Xtst(Mod(B0,I0))
    Xtst(Mod(B0,P0))
    Xtst(Mod(B0,M0))
    Xtst(Mod(B0,B0))
    Xtst(Mod(I0,Z))
    Xtst(Mod(I0,M))

    Etst(Mod(N(-9),N(1)),N(0))
    Etst(Mod(N(-8),N(1)),N(0))
    Etst(Mod(N(-7),N(1)),N(0))
    Etst(Mod(N(-6),N(1)),N(0))
    Etst(Mod(N(-5),N(1)),N(0))
    Etst(Mod(N(-4),N(1)),N(0))
    Etst(Mod(N(-3),N(1)),N(0))
    Etst(Mod(N(-2),N(1)),N(0))
    Etst(Mod(N(-1),N(1)),N(0))
    Etst(Mod(N( 0),N(1)),N(0))
    Etst(Mod(N( 1),N(1)),N(0))
    Etst(Mod(N( 2),N(1)),N(0))
    Etst(Mod(N( 3),N(1)),N(0))
    Etst(Mod(N( 4),N(1)),N(0))
    Etst(Mod(N( 5),N(1)),N(0))
    Etst(Mod(N( 6),N(1)),N(0))
    Etst(Mod(N( 7),N(1)),N(0))
    Etst(Mod(N( 8),N(1)),N(0))
    Etst(Mod(N( 9),N(1)),N(0))

    Etst(Mod(N(-9),N(2)),N(1))
    Etst(Mod(N(-8),N(2)),N(0))
    Etst(Mod(N(-7),N(2)),N(1))
    Etst(Mod(N(-6),N(2)),N(0))
    Etst(Mod(N(-5),N(2)),N(1))
    Etst(Mod(N(-4),N(2)),N(0))
    Etst(Mod(N(-3),N(2)),N(1))
    Etst(Mod(N(-2),N(2)),N(0))
    Etst(Mod(N(-1),N(2)),N(1))
    Etst(Mod(N( 0),N(2)),N(0))
    Etst(Mod(N( 1),N(2)),N(1))
    Etst(Mod(N( 2),N(2)),N(0))
    Etst(Mod(N( 3),N(2)),N(1))
    Etst(Mod(N( 4),N(2)),N(0))
    Etst(Mod(N( 5),N(2)),N(1))
    Etst(Mod(N( 6),N(2)),N(0))
    Etst(Mod(N( 7),N(2)),N(1))
    Etst(Mod(N( 8),N(2)),N(0))
    Etst(Mod(N( 9),N(2)),N(1))

    Etst(Mod(N(-9),N(3)),N(0))
    Etst(Mod(N(-8),N(3)),N(1))
    Etst(Mod(N(-7),N(3)),N(2))
    Etst(Mod(N(-6),N(3)),N(0))
    Etst(Mod(N(-5),N(3)),N(1))
    Etst(Mod(N(-4),N(3)),N(2))
    Etst(Mod(N(-3),N(3)),N(0))
    Etst(Mod(N(-2),N(3)),N(1))
    Etst(Mod(N(-1),N(3)),N(2))
    Etst(Mod(N( 0),N(3)),N(0))
    Etst(Mod(N( 1),N(3)),N(1))
    Etst(Mod(N( 2),N(3)),N(2))
    Etst(Mod(N( 3),N(3)),N(0))
    Etst(Mod(N( 4),N(3)),N(1))
    Etst(Mod(N( 5),N(3)),N(2))
    Etst(Mod(N( 6),N(3)),N(0))
    Etst(Mod(N( 7),N(3)),N(1))
    Etst(Mod(N( 8),N(3)),N(2))
    Etst(Mod(N( 9),N(3)),N(0))

    Etst(Mod(N(-9),N(4)),N(3))
    Etst(Mod(N(-8),N(4)),N(0))
    Etst(Mod(N(-7),N(4)),N(1))
    Etst(Mod(N(-6),N(4)),N(2))
    Etst(Mod(N(-5),N(4)),N(3))
    Etst(Mod(N(-4),N(4)),N(0))
    Etst(Mod(N(-3),N(4)),N(1))
    Etst(Mod(N(-2),N(4)),N(2))
    Etst(Mod(N(-1),N(4)),N(3))
    Etst(Mod(N( 0),N(4)),N(0))
    Etst(Mod(N( 1),N(4)),N(1))
    Etst(Mod(N( 2),N(4)),N(2))
    Etst(Mod(N( 3),N(4)),N(3))
    Etst(Mod(N( 4),N(4)),N(0))
    Etst(Mod(N( 5),N(4)),N(1))
    Etst(Mod(N( 6),N(4)),N(2))
    Etst(Mod(N( 7),N(4)),N(3))
    Etst(Mod(N( 8),N(4)),N(0))
    Etst(Mod(N( 9),N(4)),N(1))

    Etst(Mod(N(-9),N(5)),N(1))
    Etst(Mod(N(-8),N(5)),N(2))
    Etst(Mod(N(-7),N(5)),N(3))
    Etst(Mod(N(-6),N(5)),N(4))
    Etst(Mod(N(-5),N(5)),N(0))
    Etst(Mod(N(-4),N(5)),N(1))
    Etst(Mod(N(-3),N(5)),N(2))
    Etst(Mod(N(-2),N(5)),N(3))
    Etst(Mod(N(-1),N(5)),N(4))
    Etst(Mod(N( 0),N(5)),N(0))
    Etst(Mod(N( 1),N(5)),N(1))
    Etst(Mod(N( 2),N(5)),N(2))
    Etst(Mod(N( 3),N(5)),N(3))
    Etst(Mod(N( 4),N(5)),N(4))
    Etst(Mod(N( 5),N(5)),N(0))
    Etst(Mod(N( 6),N(5)),N(1))
    Etst(Mod(N( 7),N(5)),N(2))
    Etst(Mod(N( 8),N(5)),N(3))
    Etst(Mod(N( 9),N(5)),N(4))

    Etst(Mod(N(-9),N(6)),N(3))
    Etst(Mod(N(-8),N(6)),N(4))
    Etst(Mod(N(-7),N(6)),N(5))
    Etst(Mod(N(-6),N(6)),N(0))
    Etst(Mod(N(-5),N(6)),N(1))
    Etst(Mod(N(-4),N(6)),N(2))
    Etst(Mod(N(-3),N(6)),N(3))
    Etst(Mod(N(-2),N(6)),N(4))
    Etst(Mod(N(-1),N(6)),N(5))
    Etst(Mod(N( 0),N(6)),N(0))
    Etst(Mod(N( 1),N(6)),N(1))
    Etst(Mod(N( 2),N(6)),N(2))
    Etst(Mod(N( 3),N(6)),N(3))
    Etst(Mod(N( 4),N(6)),N(4))
    Etst(Mod(N( 5),N(6)),N(5))
    Etst(Mod(N( 6),N(6)),N(0))
    Etst(Mod(N( 7),N(6)),N(1))
    Etst(Mod(N( 8),N(6)),N(2))
    Etst(Mod(N( 9),N(6)),N(3))

    Etst(Mod(N(-9),N(7)),N(5))
    Etst(Mod(N(-8),N(7)),N(6))
    Etst(Mod(N(-7),N(7)),N(0))
    Etst(Mod(N(-6),N(7)),N(1))
    Etst(Mod(N(-5),N(7)),N(2))
    Etst(Mod(N(-4),N(7)),N(3))
    Etst(Mod(N(-3),N(7)),N(4))
    Etst(Mod(N(-2),N(7)),N(5))
    Etst(Mod(N(-1),N(7)),N(6))
    Etst(Mod(N( 0),N(7)),N(0))
    Etst(Mod(N( 1),N(7)),N(1))
    Etst(Mod(N( 2),N(7)),N(2))
    Etst(Mod(N( 3),N(7)),N(3))
    Etst(Mod(N( 4),N(7)),N(4))
    Etst(Mod(N( 5),N(7)),N(5))
    Etst(Mod(N( 6),N(7)),N(6))
    Etst(Mod(N( 7),N(7)),N(0))
    Etst(Mod(N( 8),N(7)),N(1))
    Etst(Mod(N( 9),N(7)),N(2))

    Etst(Mod(N(-9),N(8)),N(7))
    Etst(Mod(N(-8),N(8)),N(0))
    Etst(Mod(N(-7),N(8)),N(1))
    Etst(Mod(N(-6),N(8)),N(2))
    Etst(Mod(N(-5),N(8)),N(3))
    Etst(Mod(N(-4),N(8)),N(4))
    Etst(Mod(N(-3),N(8)),N(5))
    Etst(Mod(N(-2),N(8)),N(6))
    Etst(Mod(N(-1),N(8)),N(7))
    Etst(Mod(N( 0),N(8)),N(0))
    Etst(Mod(N( 1),N(8)),N(1))
    Etst(Mod(N( 2),N(8)),N(2))
    Etst(Mod(N( 3),N(8)),N(3))
    Etst(Mod(N( 4),N(8)),N(4))
    Etst(Mod(N( 5),N(8)),N(5))
    Etst(Mod(N( 6),N(8)),N(6))
    Etst(Mod(N( 7),N(8)),N(7))
    Etst(Mod(N( 8),N(8)),N(0))
    Etst(Mod(N( 9),N(8)),N(1))

    Etst(Mod(N(-9),N(9)),N(0))
    Etst(Mod(N(-8),N(9)),N(1))
    Etst(Mod(N(-7),N(9)),N(2))
    Etst(Mod(N(-6),N(9)),N(3))
    Etst(Mod(N(-5),N(9)),N(4))
    Etst(Mod(N(-4),N(9)),N(5))
    Etst(Mod(N(-3),N(9)),N(6))
    Etst(Mod(N(-2),N(9)),N(7))
    Etst(Mod(N(-1),N(9)),N(8))
    Etst(Mod(N( 0),N(9)),N(0))
    Etst(Mod(N( 1),N(9)),N(1))
    Etst(Mod(N( 2),N(9)),N(2))
    Etst(Mod(N( 3),N(9)),N(3))
    Etst(Mod(N( 4),N(9)),N(4))
    Etst(Mod(N( 5),N(9)),N(5))
    Etst(Mod(N( 6),N(9)),N(6))
    Etst(Mod(N( 7),N(9)),N(7))
    Etst(Mod(N( 8),N(9)),N(8))
    Etst(Mod(N( 9),N(9)),N(0))

    Etst(Digitend ,N(0))

    Etst(Exception,X)

    Etst(Intg(TT,Z1),XX)
    Etst(Intg(FF,Z1),XX)
    Etst(Intg(Z0,Z1),Z1)
    Etst(Intg(X0,Z1),X0)
    Etst(Intg(X1,Z1),X1)
    Etst(Intg(P0,Z1),XX)
    Etst(Intg(M0,Z1),XX)
    Etst(Intg(B0,Z1),XX)

    Etst(Boolg(TT,Z1),Z1)
    Etst(Boolg(FF,Z1),Z1)
    Etst(Boolg(Z0,Z1),XX)
    Etst(Boolg(X0,Z1),X0)
    Etst(Boolg(X1,Z1),X1)
    Etst(Boolg(P0,Z1),XX)
    Etst(Boolg(M0,Z1),XX)
    Etst(Boolg(B0,Z1),XX)

    Etst(Mapg(TT,Z1),XX)
    Etst(Mapg(FF,Z1),XX)
    Etst(Mapg(Z0,Z1),XX)
    Etst(Mapg(X0,Z1),X0)
    Etst(Mapg(X1,Z1),X1)
    Etst(Mapg(P0,Z1),XX)
    Etst(Mapg(M0,Z1),Z1)
    Etst(Mapg(B0,Z1),XX)

    Etst(Objg(TT,Z1),XX)
    Etst(Objg(FF,Z1),XX)
    Etst(Objg(Z0,Z1),XX)
    Etst(Objg(X0,Z1),X0)
    Etst(Objg(X1,Z1),X1)
    Etst(Objg(P0,Z1),XX)
    Etst(Objg(M0,Z1),XX)
    Etst(Objg(B0,Z1),Z1)

    Etst(Pairg(TT,Z1),XX)
    Etst(Pairg(FF,Z1),XX)
    Etst(Pairg(Z0,Z1),XX)
    Etst(Pairg(X0,Z1),X0)
    Etst(Pairg(X1,Z1),X1)
    Etst(Pairg(P0,Z1),Z1)
    Etst(Pairg(M0,Z1),XX)
    Etst(Pairg(B0,Z1),XX)

    Etst(Par(TT),TT)
    Etst(Par(FF),FF)
    Etst(Par(I0),I0)
    Etst(Par(X0),X0)
    Etst(Par(P0),P0)
    Etst(Par(M0),M0)
    Etst(Par(B0),B0)
    Etst(Par(UT      ),UT      )
    Etst(Par(UF      ),UF      )
    Etst(Par(UPTT    ),UPTT    )
    Etst(Par(UPTF    ),UPTF    )
    Etst(Par(UPFT    ),UPFT    )
    Etst(Par(UPFF    ),UPFF    )
    Etst(Par(UPPTTPTT),UPPTTPTT)
    Etst(Par(UPPTTPTF),UPPTTPTF)
    Etst(Par(UPPTTPFT),UPPTTPFT)
    Etst(Par(UPPTTPFF),UPPTTPFF)
    Etst(Par(UPPTFPTT),UPPTFPTT)
    Etst(Par(UPPTFPTF),UPPTFPTF)
    Etst(Par(UPPTFPFT),UPPTFPFT)
    Etst(Par(UPPTFPFF),UPPTFPFF)
    Etst(Par(UPPFTPTT),UPPFTPTT)
    Etst(Par(UPPFTPTF),UPPFTPTF)
    Etst(Par(UPPFTPFT),UPPFTPFT)
    Etst(Par(UPPFTPFF),UPPFTPFF)
    Etst(Par(UPPFFPTT),UPPFFPTT)
    Etst(Par(UPPFFPTF),UPPFFPTF)
    Etst(Par(UPPFFPFT),UPPFFPFT)
    Etst(Par(UPPFFPFF),UPPFFPFF)

    Etst(Norm(TT),TT)
    Etst(Norm(FF),FF)
    Etst(Norm(I0),I0)
    Etst(Norm(X0),X0)
    Etst(Norm(P0),P0)
    Etst(Norm(M0),M0)
    Etst(Norm(B0),B0)
    Etst(Norm(UT      ),T)
    Etst(Norm(UF      ),F)
    Etst(Norm(UPTT    ),F)
    Etst(Norm(UPTF    ),F)
    Etst(Norm(UPFT    ),F)
    Etst(Norm(UPFF    ),F)
    Etst(Norm(UPPTTPTT),F)
    Etst(Norm(UPPTTPTF),F)
    Etst(Norm(UPPTTPFT),F)
    Etst(Norm(UPPTTPFF),F)
    Etst(Norm(UPPTFPTT),F)
    Etst(Norm(UPPTFPTF),F)
    Etst(Norm(UPPTFPFT),F)
    Etst(Norm(UPPTFPFF),F)
    Etst(Norm(UPPFTPTT),F)
    Etst(Norm(UPPFTPTF),F)
    Etst(Norm(UPPFTPFT),F)
    Etst(Norm(UPPFTPFF),F)
    Etst(Norm(UPPFFPTT),F)
    Etst(Norm(UPPFFPTF),F)
    Etst(Norm(UPPFFPFT),F)
    Etst(Norm(UPPFFPFF),F)

    Etst(Prenorm(TT),TT)
    Etst(Prenorm(FF),FF)
    Etst(Prenorm(I0),I0)
    Etst(Prenorm(X0),X0)
    Etst(Prenorm(P0),P0)
    Etst(Prenorm(M0),M0)
    Etst(Prenorm(B0),B0)
    Etst(Prenorm(UT      ),T)
    Etst(Prenorm(UF      ),F)
    Etst(Prenorm(UPTT    ),F)
    Etst(Prenorm(UPTF    ),F)
    Etst(Prenorm(UPFT    ),F)
    Etst(Prenorm(UPFF    ),F)
    Etst(Prenorm(UPPTTPTT),F)
    Etst(Prenorm(UPPTTPTF),F)
    Etst(Prenorm(UPPTTPFT),F)
    Etst(Prenorm(UPPTTPFF),F)
    Etst(Prenorm(UPPTFPTT),F)
    Etst(Prenorm(UPPTFPTF),F)
    Etst(Prenorm(UPPTFPFT),F)
    Etst(Prenorm(UPPTFPFF),F)
    Etst(Prenorm(UPPFTPTT),F)
    Etst(Prenorm(UPPFTPTF),F)
    Etst(Prenorm(UPPFTPFT),F)
    Etst(Prenorm(UPPFTPFF),F)
    Etst(Prenorm(UPPFFPTT),F)
    Etst(Prenorm(UPPFFPTF),F)
    Etst(Prenorm(UPPFFPFT),F)
    Etst(Prenorm(UPPFFPFF),F)

    Etst(Not(TT),FF)
    Etst(Not(FF),TT)
    Etst(Not(Z0),TT)
    Etst(Not(P0),TT)
    Etst(Not(X0),X0)
    Etst(Not(M0),TT)
    Etst(Not(B0),TT)

    Etst(Untag(TT),XX)
    Etst(Untag(FF),XX)
    Etst(Untag(Z0),XX)
    Etst(Untag(P0),XX)
    Etst(Untag(X0),X0)
    Etst(Untag(M0),TT)
    Etst(Untag(B0),XX)

    Ttst(Mapp(Map(TT)))
    Ttst(Mapp(Map(FF)))
    Ttst(Mapp(Map(Z0)))
    Ttst(Mapp(Map(P0)))
    Ttst(Mapp(Map(X0)))
    Ttst(Mapp(Map(M0)))
    Ttst(Mapp(Map(B0)))
    Etst(Untag(Map(TT)),TT)
    Etst(Untag(Map(FF)),FF)
    Etst(Untag(Map(Z0)),Z0)
    Etst(Untag(Map(P0)),P0)
    Etst(Untag(Map(X0)),X0)
    Etst(Untag(Map(M0)),M0)
    Etst(Untag(Map(B0)),B0)
    Etst(Untag(Map(UPPTTPTT)),FF)

    Ttst(Mapp(Maptag(TT)))
    Ttst(Mapp(Maptag(FF)))
    Ttst(Mapp(Maptag(Z0)))
    Ttst(Mapp(Maptag(P0)))
    Etst(Maptag(X0),X0)
    Ttst(Mapp(Maptag(M0)))
    Ttst(Mapp(Maptag(B0)))
    Etst(Untag(Maptag(TT)),TT)
    Etst(Untag(Maptag(FF)),FF)
    Etst(Untag(Maptag(Z0)),Z0)
    Etst(Untag(Maptag(P0)),P0)
    Etst(Untag(Maptag(X0)),X0)
    Etst(Untag(Maptag(M0)),M0)
    Etst(Untag(Maptag(B0)),B0)

    Ttst(Mapp(Maptag1(TT)))
    Ttst(Mapp(Maptag1(FF)))
    Ttst(Mapp(Maptag1(Z0)))
    Ttst(Mapp(Maptag1(P0)))
    Ttst(Mapp(Maptag1(X0)))
    Ttst(Mapp(Maptag1(M0)))
    Ttst(Mapp(Maptag1(B0)))
    Etst(Untag(Maptag1(TT)),TT)
    Etst(Untag(Maptag1(FF)),FF)
    Etst(Untag(Maptag1(Z0)),Z0)
    Etst(Untag(Maptag1(P0)),P0)
    Etst(Untag(Maptag1(X0)),X0)
    Etst(Untag(Maptag1(M0)),M0)
    Etst(Untag(Maptag1(B0)),B0)

    ntst(LET(map_t,map_pair(map_t,map_t)))
    ntst(LET(map_f,map_pair(map_t,map_t)))
    ntst(LET(map_t,map_pair(map_t,map_f)))
    ptst(LET(map_f,map_pair(map_t,map_f)))
    ptst(LET(map_t,map_pair(map_f,map_t)))
    ntst(LET(map_f,map_pair(map_f,map_t)))
    ptst(LET(map_t,map_pair(map_f,map_f)))
    ptst(LET(map_f,map_pair(map_f,map_f)))

    Xtst(Plus(TT,TT))
    Xtst(Plus(TT,FF))
    Xtst(Plus(TT,I0))
    Xtst(Plus(TT,P0))
    Xtst(Plus(TT,M0))
    Xtst(Plus(TT,B0))
    Xtst(Plus(FF,TT))
    Xtst(Plus(FF,FF))
    Xtst(Plus(FF,I0))
    Xtst(Plus(FF,P0))
    Xtst(Plus(FF,M0))
    Xtst(Plus(FF,B0))
    Xtst(Plus(I0,TT))
    Xtst(Plus(I0,FF))
    Xtst(Plus(I0,P0))
    Xtst(Plus(I0,M0))
    Xtst(Plus(I0,B0))
    Xtst(Plus(P0,TT))
    Xtst(Plus(P0,FF))
    Xtst(Plus(P0,I0))
    Xtst(Plus(P0,P0))
    Xtst(Plus(P0,M0))
    Xtst(Plus(P0,B0))
    Xtst(Plus(M0,TT))
    Xtst(Plus(M0,FF))
    Xtst(Plus(M0,I0))
    Xtst(Plus(M0,P0))
    Xtst(Plus(M0,M0))
    Xtst(Plus(M0,B0))
    Xtst(Plus(B0,TT))
    Xtst(Plus(B0,FF))
    Xtst(Plus(B0,I0))
    Xtst(Plus(B0,P0))
    Xtst(Plus(B0,M0))
    Xtst(Plus(B0,B0))
    Etst(Plus(JN(-3),JN(-3)),JN(-6))
    Etst(Plus(JN(-3),JN(-2)),JN(-5))
    Etst(Plus(JN(-3),JN(-1)),JN(-4))
    Etst(Plus(JN(-3),JN( 0)),JN(-3))
    Etst(Plus(JN(-3),JN( 1)),JN(-2))
    Etst(Plus(JN(-3),JN( 2)),JN(-1))
    Etst(Plus(JN(-2),JN(-3)),JN(-5))
    Etst(Plus(JN(-2),JN(-2)),JN(-4))
    Etst(Plus(JN(-2),JN(-1)),JN(-3))
    Etst(Plus(JN(-2),JN( 0)),JN(-2))
    Etst(Plus(JN(-2),JN( 1)),JN(-1))
    Etst(Plus(JN(-2),JN( 2)),JN( 0))
    Etst(Plus(JN(-1),JN(-3)),JN(-4))
    Etst(Plus(JN(-1),JN(-2)),JN(-3))
    Etst(Plus(JN(-1),JN(-1)),JN(-2))
    Etst(Plus(JN(-1),JN( 0)),JN(-1))
    Etst(Plus(JN(-1),JN( 1)),JN( 0))
    Etst(Plus(JN(-1),JN( 2)),JN( 1))
    Etst(Plus(JN( 0),JN(-3)),JN(-3))
    Etst(Plus(JN( 0),JN(-2)),JN(-2))
    Etst(Plus(JN( 0),JN(-1)),JN(-1))
    Etst(Plus(JN( 0),JN( 0)),JN( 0))
    Etst(Plus(JN( 0),JN( 1)),JN( 1))
    Etst(Plus(JN( 0),JN( 2)),JN( 2))
    Etst(Plus(JN( 1),JN(-3)),JN(-2))
    Etst(Plus(JN( 1),JN(-2)),JN(-1))
    Etst(Plus(JN( 1),JN(-1)),JN( 0))
    Etst(Plus(JN( 1),JN( 0)),JN( 1))
    Etst(Plus(JN( 1),JN( 1)),JN( 2))
    Etst(Plus(JN( 1),JN( 2)),JN( 3))
    Etst(Plus(JN( 2),JN(-3)),JN(-1))
    Etst(Plus(JN( 2),JN(-2)),JN( 0))
    Etst(Plus(JN( 2),JN(-1)),JN( 1))
    Etst(Plus(JN( 2),JN( 0)),JN( 2))
    Etst(Plus(JN( 2),JN( 1)),JN( 3))
    Etst(Plus(JN( 2),JN( 2)),JN( 4))
    Etst(Plus(K2( 0, 0),K2( 0, 0)),K2( 0, 0))
    Etst(Plus(K2( 0, 0),K2( 0,-1)),K2( 0,-1))
    Etst(Plus(K2( 0, 0),K2(-1, 0)),K2(-1, 0))
    Etst(Plus(K2( 0, 0),K2(-1,-1)),K2(-1,-1))
    Etst(Plus(K2( 0,-1),K2( 0, 0)),K2( 0,-1))
    Etst(Plus(K2( 0,-1),K2( 0,-1)),K2( 0,-2))
    Etst(Plus(K2( 0,-1),K2(-1, 0)),K2(-1,-1))
    Etst(Plus(K2( 0,-1),K2(-1,-1)),K2(-1,-2))
    Etst(Plus(K2(-1, 0),K2( 0, 0)),K2(-1, 0))
    Etst(Plus(K2(-1, 0),K2( 0,-1)),K2(-1,-1))
    Etst(Plus(K2(-1, 0),K2(-1, 0)),K2(-2, 1))
    Etst(Plus(K2(-1, 0),K2(-1,-1)),K2(-2, 0))
    Etst(Plus(K2(-1,-1),K2( 0, 0)),K2(-1,-1))
    Etst(Plus(K2(-1,-1),K2( 0,-1)),K2(-1,-2))
    Etst(Plus(K2(-1,-1),K2(-1, 0)),K2(-2, 0))
    Etst(Plus(K2(-1,-1),K2(-1,-1)),K2(-2,-1))

    Etst(Plus(N(-1000000000000),N(-1000000000000)),N(-2000000000000))
    Etst(Plus(N(-1000000000000),N(   -1000000000)),N(-1001000000000))
    Etst(Plus(N(-1000000000000),N(      -1000000)),N(-1000001000000))
    Etst(Plus(N(-1000000000000),N(         -1000)),N(-1000000001000))
    Etst(Plus(N(-1000000000000),N(          -100)),N(-1000000000100))
    Etst(Plus(N(-1000000000000),N(           -10)),N(-1000000000010))
    Etst(Plus(N(-1000000000000),N(            -1)),N(-1000000000001))
    Etst(Plus(N(-1000000000000),N(             0)),N(-1000000000000))
    Etst(Plus(N(-1000000000000),N(             1)),N(-0999999999999))
    Etst(Plus(N(-1000000000000),N(            10)),N(-0999999999990))
    Etst(Plus(N(-1000000000000),N(           100)),N(-0999999999900))
    Etst(Plus(N(-1000000000000),N(          1000)),N(-0999999999000))
    Etst(Plus(N(-1000000000000),N(       1000000)),N(-0999999000000))
    Etst(Plus(N(-1000000000000),N(    1000000000)),N(-0999000000000))
    Etst(Plus(N(-1000000000000),N( 1000000000000)),N(-0000000000000))

    Etst(Plus(N(   -1000000000),N(-1000000000000)),N(-1001000000000))
    Etst(Plus(N(   -1000000000),N(   -1000000000)),N(-0002000000000))
    Etst(Plus(N(   -1000000000),N(      -1000000)),N(-0001001000000))
    Etst(Plus(N(   -1000000000),N(         -1000)),N(-0001000001000))
    Etst(Plus(N(   -1000000000),N(          -100)),N(-0001000000100))
    Etst(Plus(N(   -1000000000),N(           -10)),N(-0001000000010))
    Etst(Plus(N(   -1000000000),N(            -1)),N(-0001000000001))
    Etst(Plus(N(   -1000000000),N(             0)),N(-0001000000000))
    Etst(Plus(N(   -1000000000),N(             1)),N(-0000999999999))
    Etst(Plus(N(   -1000000000),N(            10)),N(-0000999999990))
    Etst(Plus(N(   -1000000000),N(           100)),N(-0000999999900))
    Etst(Plus(N(   -1000000000),N(          1000)),N(-0000999999000))
    Etst(Plus(N(   -1000000000),N(       1000000)),N(-0000999000000))
    Etst(Plus(N(   -1000000000),N(    1000000000)),N(-0000000000000))
    Etst(Plus(N(   -1000000000),N( 1000000000000)),N( 0999000000000))

    Etst(Plus(N(      -1000000),N(-1000000000000)),N(-1000001000000))
    Etst(Plus(N(      -1000000),N(   -1000000000)),N(-0001001000000))
    Etst(Plus(N(      -1000000),N(      -1000000)),N(-0000002000000))
    Etst(Plus(N(      -1000000),N(         -1000)),N(-0000001001000))
    Etst(Plus(N(      -1000000),N(          -100)),N(-0000001000100))
    Etst(Plus(N(      -1000000),N(           -10)),N(-0000001000010))
    Etst(Plus(N(      -1000000),N(            -1)),N(-0000001000001))
    Etst(Plus(N(      -1000000),N(             0)),N(-0000001000000))
    Etst(Plus(N(      -1000000),N(             1)),N(-0000000999999))
    Etst(Plus(N(      -1000000),N(            10)),N(-0000000999990))
    Etst(Plus(N(      -1000000),N(           100)),N(-0000000999900))
    Etst(Plus(N(      -1000000),N(          1000)),N(-0000000999000))
    Etst(Plus(N(      -1000000),N(       1000000)),N(-0000000000000))
    Etst(Plus(N(      -1000000),N(    1000000000)),N( 0000999000000))
    Etst(Plus(N(      -1000000),N( 1000000000000)),N( 0999999000000))

    Etst(Plus(N(         -1000),N(-1000000000000)),N(-1000000001000))
    Etst(Plus(N(         -1000),N(   -1000000000)),N(-0001000001000))
    Etst(Plus(N(         -1000),N(      -1000000)),N(-0000001001000))
    Etst(Plus(N(         -1000),N(         -1000)),N(-0000000002000))
    Etst(Plus(N(         -1000),N(          -100)),N(-0000000001100))
    Etst(Plus(N(         -1000),N(           -10)),N(-0000000001010))
    Etst(Plus(N(         -1000),N(            -1)),N(-0000000001001))
    Etst(Plus(N(         -1000),N(             0)),N(-0000000001000))
    Etst(Plus(N(         -1000),N(             1)),N(-0000000000999))
    Etst(Plus(N(         -1000),N(            10)),N(-0000000000990))
    Etst(Plus(N(         -1000),N(           100)),N(-0000000000900))
    Etst(Plus(N(         -1000),N(          1000)),N( 0000000000000))
    Etst(Plus(N(         -1000),N(       1000000)),N( 0000000999000))
    Etst(Plus(N(         -1000),N(    1000000000)),N( 0000999999000))
    Etst(Plus(N(         -1000),N( 1000000000000)),N( 0999999999000))

    Etst(Plus(N(          -100),N(-1000000000000)),N(-1000000000100))
    Etst(Plus(N(          -100),N(   -1000000000)),N(-0001000000100))
    Etst(Plus(N(          -100),N(      -1000000)),N(-0000001000100))
    Etst(Plus(N(          -100),N(         -1000)),N(-0000000001100))
    Etst(Plus(N(          -100),N(          -100)),N(-0000000000200))
    Etst(Plus(N(          -100),N(           -10)),N(-0000000000110))
    Etst(Plus(N(          -100),N(            -1)),N(-0000000000101))
    Etst(Plus(N(          -100),N(             0)),N(-0000000000100))
    Etst(Plus(N(          -100),N(             1)),N(-0000000000099))
    Etst(Plus(N(          -100),N(            10)),N(-0000000000090))
    Etst(Plus(N(          -100),N(           100)),N( 0000000000000))
    Etst(Plus(N(          -100),N(          1000)),N( 0000000000900))
    Etst(Plus(N(          -100),N(       1000000)),N( 0000000999900))
    Etst(Plus(N(          -100),N(    1000000000)),N( 0000999999900))
    Etst(Plus(N(          -100),N( 1000000000000)),N( 0999999999900))

    Etst(Plus(N(           -10),N(-1000000000000)),N(-1000000000010))
    Etst(Plus(N(           -10),N(   -1000000000)),N(-0001000000010))
    Etst(Plus(N(           -10),N(      -1000000)),N(-0000001000010))
    Etst(Plus(N(           -10),N(         -1000)),N(-0000000001010))
    Etst(Plus(N(           -10),N(          -100)),N(-0000000000110))
    Etst(Plus(N(           -10),N(           -10)),N(-0000000000020))
    Etst(Plus(N(           -10),N(            -1)),N(-0000000000011))
    Etst(Plus(N(           -10),N(             0)),N(-0000000000010))
    Etst(Plus(N(           -10),N(             1)),N(-0000000000009))
    Etst(Plus(N(           -10),N(            10)),N( 0000000000000))
    Etst(Plus(N(           -10),N(           100)),N( 0000000000090))
    Etst(Plus(N(           -10),N(          1000)),N( 0000000000990))
    Etst(Plus(N(           -10),N(       1000000)),N( 0000000999990))
    Etst(Plus(N(           -10),N(    1000000000)),N( 0000999999990))
    Etst(Plus(N(           -10),N( 1000000000000)),N( 0999999999990))

    Etst(Plus(N(            -1),N(-1000000000000)),N(-1000000000001))
    Etst(Plus(N(            -1),N(   -1000000000)),N(-0001000000001))
    Etst(Plus(N(            -1),N(      -1000000)),N(-0000001000001))
    Etst(Plus(N(            -1),N(         -1000)),N(-0000000001001))
    Etst(Plus(N(            -1),N(          -100)),N(-0000000000101))
    Etst(Plus(N(            -1),N(           -10)),N(-0000000000011))
    Etst(Plus(N(            -1),N(            -1)),N(-0000000000002))
    Etst(Plus(N(            -1),N(             0)),N(-0000000000001))
    Etst(Plus(N(            -1),N(             1)),N( 0000000000000))
    Etst(Plus(N(            -1),N(            10)),N( 0000000000009))
    Etst(Plus(N(            -1),N(           100)),N( 0000000000099))
    Etst(Plus(N(            -1),N(          1000)),N( 0000000000999))
    Etst(Plus(N(            -1),N(       1000000)),N( 0000000999999))
    Etst(Plus(N(            -1),N(    1000000000)),N( 0000999999999))
    Etst(Plus(N(            -1),N( 1000000000000)),N( 0999999999999))

    Etst(Plus(N(             0),N(-1000000000000)),N(-1000000000000))
    Etst(Plus(N(             0),N(   -1000000000)),N(-0001000000000))
    Etst(Plus(N(             0),N(      -1000000)),N(-0000001000000))
    Etst(Plus(N(             0),N(         -1000)),N(-0000000001000))
    Etst(Plus(N(             0),N(          -100)),N(-0000000000100))
    Etst(Plus(N(             0),N(           -10)),N(-0000000000010))
    Etst(Plus(N(             0),N(            -1)),N(-0000000000001))
    Etst(Plus(N(             0),N(             0)),N( 0000000000000))
    Etst(Plus(N(             0),N(             1)),N( 0000000000001))
    Etst(Plus(N(             0),N(            10)),N( 0000000000010))
    Etst(Plus(N(             0),N(           100)),N( 0000000000100))
    Etst(Plus(N(             0),N(          1000)),N( 0000000001000))
    Etst(Plus(N(             0),N(       1000000)),N( 0000001000000))
    Etst(Plus(N(             0),N(    1000000000)),N( 0001000000000))
    Etst(Plus(N(             0),N( 1000000000000)),N( 1000000000000))

    Etst(Plus(N( 1000000000000),N( 1000000000000)),N( 2000000000000))
    Etst(Plus(N( 1000000000000),N(    1000000000)),N( 1001000000000))
    Etst(Plus(N( 1000000000000),N(       1000000)),N( 1000001000000))
    Etst(Plus(N( 1000000000000),N(          1000)),N( 1000000001000))
    Etst(Plus(N( 1000000000000),N(           100)),N( 1000000000100))
    Etst(Plus(N( 1000000000000),N(            10)),N( 1000000000010))
    Etst(Plus(N( 1000000000000),N(             1)),N( 1000000000001))
    Etst(Plus(N( 1000000000000),N(             0)),N( 1000000000000))
    Etst(Plus(N( 1000000000000),N(            -1)),N( 0999999999999))
    Etst(Plus(N( 1000000000000),N(           -10)),N( 0999999999990))
    Etst(Plus(N( 1000000000000),N(          -100)),N( 0999999999900))
    Etst(Plus(N( 1000000000000),N(         -1000)),N( 0999999999000))
    Etst(Plus(N( 1000000000000),N(      -1000000)),N( 0999999000000))
    Etst(Plus(N( 1000000000000),N(   -1000000000)),N( 0999000000000))
    Etst(Plus(N( 1000000000000),N(-1000000000000)),N( 0000000000000))

    Etst(Plus(N(    1000000000),N( 1000000000000)),N( 1001000000000))
    Etst(Plus(N(    1000000000),N(    1000000000)),N( 0002000000000))
    Etst(Plus(N(    1000000000),N(       1000000)),N( 0001001000000))
    Etst(Plus(N(    1000000000),N(          1000)),N( 0001000001000))
    Etst(Plus(N(    1000000000),N(           100)),N( 0001000000100))
    Etst(Plus(N(    1000000000),N(            10)),N( 0001000000010))
    Etst(Plus(N(    1000000000),N(             1)),N( 0001000000001))
    Etst(Plus(N(    1000000000),N(             0)),N( 0001000000000))
    Etst(Plus(N(    1000000000),N(            -1)),N( 0000999999999))
    Etst(Plus(N(    1000000000),N(           -10)),N( 0000999999990))
    Etst(Plus(N(    1000000000),N(          -100)),N( 0000999999900))
    Etst(Plus(N(    1000000000),N(         -1000)),N( 0000999999000))
    Etst(Plus(N(    1000000000),N(      -1000000)),N( 0000999000000))
    Etst(Plus(N(    1000000000),N(   -1000000000)),N( 0000000000000))
    Etst(Plus(N(    1000000000),N(-1000000000000)),N(-0999000000000))

    Etst(Plus(N(       1000000),N( 1000000000000)),N( 1000001000000))
    Etst(Plus(N(       1000000),N(    1000000000)),N( 0001001000000))
    Etst(Plus(N(       1000000),N(       1000000)),N( 0000002000000))
    Etst(Plus(N(       1000000),N(          1000)),N( 0000001001000))
    Etst(Plus(N(       1000000),N(           100)),N( 0000001000100))
    Etst(Plus(N(       1000000),N(            10)),N( 0000001000010))
    Etst(Plus(N(       1000000),N(             1)),N( 0000001000001))
    Etst(Plus(N(       1000000),N(             0)),N( 0000001000000))
    Etst(Plus(N(       1000000),N(            -1)),N( 0000000999999))
    Etst(Plus(N(       1000000),N(           -10)),N( 0000000999990))
    Etst(Plus(N(       1000000),N(          -100)),N( 0000000999900))
    Etst(Plus(N(       1000000),N(         -1000)),N( 0000000999000))
    Etst(Plus(N(       1000000),N(      -1000000)),N( 0000000000000))
    Etst(Plus(N(       1000000),N(   -1000000000)),N(-0000999000000))
    Etst(Plus(N(       1000000),N(-1000000000000)),N(-0999999000000))

    Etst(Plus(N(          1000),N( 1000000000000)),N( 1000000001000))
    Etst(Plus(N(          1000),N(    1000000000)),N( 0001000001000))
    Etst(Plus(N(          1000),N(       1000000)),N( 0000001001000))
    Etst(Plus(N(          1000),N(          1000)),N( 0000000002000))
    Etst(Plus(N(          1000),N(           100)),N( 0000000001100))
    Etst(Plus(N(          1000),N(            10)),N( 0000000001010))
    Etst(Plus(N(          1000),N(             1)),N( 0000000001001))
    Etst(Plus(N(          1000),N(             0)),N( 0000000001000))
    Etst(Plus(N(          1000),N(            -1)),N( 0000000000999))
    Etst(Plus(N(          1000),N(           -10)),N( 0000000000990))
    Etst(Plus(N(          1000),N(          -100)),N( 0000000000900))
    Etst(Plus(N(          1000),N(         -1000)),N(-0000000000000))
    Etst(Plus(N(          1000),N(      -1000000)),N(-0000000999000))
    Etst(Plus(N(          1000),N(   -1000000000)),N(-0000999999000))
    Etst(Plus(N(          1000),N(-1000000000000)),N(-0999999999000))

    Etst(Plus(N(           100),N( 1000000000000)),N( 1000000000100))
    Etst(Plus(N(           100),N(    1000000000)),N( 0001000000100))
    Etst(Plus(N(           100),N(       1000000)),N( 0000001000100))
    Etst(Plus(N(           100),N(          1000)),N( 0000000001100))
    Etst(Plus(N(           100),N(           100)),N( 0000000000200))
    Etst(Plus(N(           100),N(            10)),N( 0000000000110))
    Etst(Plus(N(           100),N(             1)),N( 0000000000101))
    Etst(Plus(N(           100),N(             0)),N( 0000000000100))
    Etst(Plus(N(           100),N(            -1)),N( 0000000000099))
    Etst(Plus(N(           100),N(           -10)),N( 0000000000090))
    Etst(Plus(N(           100),N(          -100)),N(-0000000000000))
    Etst(Plus(N(           100),N(         -1000)),N(-0000000000900))
    Etst(Plus(N(           100),N(      -1000000)),N(-0000000999900))
    Etst(Plus(N(           100),N(   -1000000000)),N(-0000999999900))
    Etst(Plus(N(           100),N(-1000000000000)),N(-0999999999900))

    Etst(Plus(N(            10),N( 1000000000000)),N( 1000000000010))
    Etst(Plus(N(            10),N(    1000000000)),N( 0001000000010))
    Etst(Plus(N(            10),N(       1000000)),N( 0000001000010))
    Etst(Plus(N(            10),N(          1000)),N( 0000000001010))
    Etst(Plus(N(            10),N(           100)),N( 0000000000110))
    Etst(Plus(N(            10),N(            10)),N( 0000000000020))
    Etst(Plus(N(            10),N(             1)),N( 0000000000011))
    Etst(Plus(N(            10),N(             0)),N( 0000000000010))
    Etst(Plus(N(            10),N(            -1)),N( 0000000000009))
    Etst(Plus(N(            10),N(           -10)),N(-0000000000000))
    Etst(Plus(N(            10),N(          -100)),N(-0000000000090))
    Etst(Plus(N(            10),N(         -1000)),N(-0000000000990))
    Etst(Plus(N(            10),N(      -1000000)),N(-0000000999990))
    Etst(Plus(N(            10),N(   -1000000000)),N(-0000999999990))
    Etst(Plus(N(            10),N(-1000000000000)),N(-0999999999990))

    Etst(Plus(N(             1),N( 1000000000000)),N( 1000000000001))
    Etst(Plus(N(             1),N(    1000000000)),N( 0001000000001))
    Etst(Plus(N(             1),N(       1000000)),N( 0000001000001))
    Etst(Plus(N(             1),N(          1000)),N( 0000000001001))
    Etst(Plus(N(             1),N(           100)),N( 0000000000101))
    Etst(Plus(N(             1),N(            10)),N( 0000000000011))
    Etst(Plus(N(             1),N(             1)),N( 0000000000002))
    Etst(Plus(N(             1),N(             0)),N( 0000000000001))
    Etst(Plus(N(             1),N(            -1)),N(-0000000000000))
    Etst(Plus(N(             1),N(           -10)),N(-0000000000009))
    Etst(Plus(N(             1),N(          -100)),N(-0000000000099))
    Etst(Plus(N(             1),N(         -1000)),N(-0000000000999))
    Etst(Plus(N(             1),N(      -1000000)),N(-0000000999999))
    Etst(Plus(N(             1),N(   -1000000000)),N(-0000999999999))
    Etst(Plus(N(             1),N(-1000000000000)),N(-0999999999999))

    Xtst(Minus(TT,TT))
    Xtst(Minus(TT,FF))
    Xtst(Minus(TT,I0))
    Xtst(Minus(TT,P0))
    Xtst(Minus(TT,M0))
    Xtst(Minus(TT,B0))
    Xtst(Minus(FF,TT))
    Xtst(Minus(FF,FF))
    Xtst(Minus(FF,I0))
    Xtst(Minus(FF,P0))
    Xtst(Minus(FF,M0))
    Xtst(Minus(FF,B0))
    Xtst(Minus(I0,TT))
    Xtst(Minus(I0,FF))
    Xtst(Minus(I0,P0))
    Xtst(Minus(I0,M0))
    Xtst(Minus(I0,B0))
    Xtst(Minus(P0,TT))
    Xtst(Minus(P0,FF))
    Xtst(Minus(P0,I0))
    Xtst(Minus(P0,P0))
    Xtst(Minus(P0,M0))
    Xtst(Minus(P0,B0))
    Xtst(Minus(M0,TT))
    Xtst(Minus(M0,FF))
    Xtst(Minus(M0,I0))
    Xtst(Minus(M0,P0))
    Xtst(Minus(M0,M0))
    Xtst(Minus(M0,B0))
    Xtst(Minus(B0,TT))
    Xtst(Minus(B0,FF))
    Xtst(Minus(B0,I0))
    Xtst(Minus(B0,P0))
    Xtst(Minus(B0,M0))
    Xtst(Minus(B0,B0))
    Etst(Minus(JN(-3),JN(-3)),JN( 0))
    Etst(Minus(JN(-3),JN(-2)),JN(-1))
    Etst(Minus(JN(-3),JN(-1)),JN(-2))
    Etst(Minus(JN(-3),JN( 0)),JN(-3))
    Etst(Minus(JN(-3),JN( 1)),JN(-4))
    Etst(Minus(JN(-3),JN( 2)),JN(-5))
    Etst(Minus(JN(-2),JN(-3)),JN( 1))
    Etst(Minus(JN(-2),JN(-2)),JN( 0))
    Etst(Minus(JN(-2),JN(-1)),JN(-1))
    Etst(Minus(JN(-2),JN( 0)),JN(-2))
    Etst(Minus(JN(-2),JN( 1)),JN(-3))
    Etst(Minus(JN(-2),JN( 2)),JN(-4))
    Etst(Minus(JN(-1),JN(-3)),JN( 2))
    Etst(Minus(JN(-1),JN(-2)),JN( 1))
    Etst(Minus(JN(-1),JN(-1)),JN( 0))
    Etst(Minus(JN(-1),JN( 0)),JN(-1))
    Etst(Minus(JN(-1),JN( 1)),JN(-2))
    Etst(Minus(JN(-1),JN( 2)),JN(-3))
    Etst(Minus(JN( 0),JN(-3)),JN( 3))
    Etst(Minus(JN( 0),JN(-2)),JN( 2))
    Etst(Minus(JN( 0),JN(-1)),JN( 1))
    Etst(Minus(JN( 0),JN( 0)),JN( 0))
    Etst(Minus(JN( 0),JN( 1)),JN(-1))
    Etst(Minus(JN( 0),JN( 2)),JN(-2))
    Etst(Minus(JN( 1),JN(-3)),JN( 4))
    Etst(Minus(JN( 1),JN(-2)),JN( 3))
    Etst(Minus(JN( 1),JN(-1)),JN( 2))
    Etst(Minus(JN( 1),JN( 0)),JN( 1))
    Etst(Minus(JN( 1),JN( 1)),JN( 0))
    Etst(Minus(JN( 1),JN( 2)),JN(-1))
    Etst(Minus(JN( 2),JN(-3)),JN( 5))
    Etst(Minus(JN( 2),JN(-2)),JN( 4))
    Etst(Minus(JN( 2),JN(-1)),JN( 3))
    Etst(Minus(JN( 2),JN( 0)),JN( 2))
    Etst(Minus(JN( 2),JN( 1)),JN( 1))
    Etst(Minus(JN( 2),JN( 2)),JN( 0))
    Etst(Minus(K2( 0, 0),K2( 0, 0)),K2( 0, 0))
    Etst(Minus(K2( 0, 0),K2( 0, 1)),K2( 0,-1))
    Etst(Minus(K2( 0, 0),K2( 1,-1)),K2(-1, 0))
    Etst(Minus(K2( 0, 0),K2( 1, 0)),K2(-1,-1))
    Etst(Minus(K2( 0,-1),K2( 0, 0)),K2( 0,-1))
    Etst(Minus(K2( 0,-1),K2( 0, 1)),K2( 0,-2))
    Etst(Minus(K2( 0,-1),K2( 1,-1)),K2(-1,-1))
    Etst(Minus(K2( 0,-1),K2( 1, 0)),K2(-1,-2))
    Etst(Minus(K2(-1, 0),K2( 0, 0)),K2(-1, 0))
    Etst(Minus(K2(-1, 0),K2( 0, 1)),K2(-1,-1))
    Etst(Minus(K2(-1, 0),K2( 1,-1)),K2(-2, 1))
    Etst(Minus(K2(-1, 0),K2( 1, 0)),K2(-2, 0))
    Etst(Minus(K2(-1,-1),K2( 0, 0)),K2(-1,-1))
    Etst(Minus(K2(-1,-1),K2( 0, 1)),K2(-1,-2))
    Etst(Minus(K2(-1,-1),K2( 1,-1)),K2(-2, 0))
    Etst(Minus(K2(-1,-1),K2( 1, 0)),K2(-2,-1))

    Xtst(Uplus(TT))
    Xtst(Uplus(FF))
    Xtst(Uplus(P0))
    Xtst(Uplus(M0))
    Xtst(Uplus(B0))
    Etst(Uplus(Z),Z)
    Etst(Uplus(M),M)
    Etst(Uplus(I0011),I0011)
    Etst(Uplus(J1010_0101),J1010_0101)

    Xtst(Dplus(TT))
    Xtst(Dplus(FF))
    Xtst(Dplus(P0))
    Xtst(Dplus(M0))
    Xtst(Dplus(B0))
    Etst(Dplus(Z),Z)
    Etst(Dplus(M),M)
    Etst(Dplus(I0011),I0011)
    Etst(Dplus(J1010_0101),J1010_0101)

    Xtst(Uminus(TT))
    Xtst(Uminus(FF))
    Xtst(Uminus(P0))
    Xtst(Uminus(M0))
    Xtst(Uminus(B0))
    Etst(Uminus(JN(-2)),JN( 2))
    Etst(Uminus(JN(-1)),JN( 1))
    Etst(Uminus(JN( 0)),JN( 0))
    Etst(Uminus(JN( 1)),JN(-1))
    Etst(Uminus(JN( 2)),JN(-2))
    Etst(Uminus(K2(-1,-1)),K2( 1, 0))
    Etst(Uminus(K2(-1, 0)),K2( 1,-1))
    Etst(Uminus(K2(-1, 1)),K2( 1,-2))
    Etst(Uminus(K2( 0,-1)),K2( 0, 1))
    Etst(Uminus(K2( 0, 0)),K2( 0, 0))
    Etst(Uminus(K2( 0, 1)),K2( 0,-1))
    Etst(Uminus(K2( 1,-1)),K2(-1, 0))
    Etst(Uminus(K2( 1, 0)),K2(-1,-1))
    Etst(Uminus(K2( 1, 1)),K2(-1,-2))

    Xtst(Dminus(TT))
    Xtst(Dminus(FF))
    Xtst(Dminus(P0))
    Xtst(Dminus(M0))
    Xtst(Dminus(B0))
    Etst(Dminus(JN(-2)),JN( 2))
    Etst(Dminus(JN(-1)),JN( 1))
    Etst(Dminus(JN( 0)),JN( 0))
    Etst(Dminus(JN( 1)),JN(-1))
    Etst(Dminus(JN( 2)),JN(-2))
    Etst(Dminus(K2(-1,-1)),K2( 1, 0))
    Etst(Dminus(K2(-1, 0)),K2( 1,-1))
    Etst(Dminus(K2(-1, 1)),K2( 1,-2))
    Etst(Dminus(K2( 0,-1)),K2( 0, 1))
    Etst(Dminus(K2( 0, 0)),K2( 0, 0))
    Etst(Dminus(K2( 0, 1)),K2( 0,-1))
    Etst(Dminus(K2( 1,-1)),K2(-1, 0))
    Etst(Dminus(K2( 1, 0)),K2(-1,-1))
    Etst(Dminus(K2( 1, 1)),K2(-1,-2))

    Xtst(Intlength(TT))
    Xtst(Intlength(FF))
    Xtst(Intlength(P0))
    Xtst(Intlength(M0))
    Xtst(Intlength(B0))
    Etst(Intlength(Z),Z)
    Etst(Intlength(M),Z)
    Etst(Intlength(JN(1)),JN(1))
    Etst(Intlength(JN(2)),JN(2))
    Etst(Intlength(JN(3)),JN(2))
    Etst(Intlength(JN(4)),JN(3))
    Etst(Intlength(JN(5)),JN(3))
    Etst(Intlength(JN(6)),JN(3))
    Etst(Intlength(JN(7)),JN(3))
    Etst(Intlength(JN(8)),JN(4))
    Etst(Intlength(JN(-2)),JN(1))
    Etst(Intlength(JN(-3)),JN(2))
    Etst(Intlength(JN(-4)),JN(2))
    Etst(Intlength(JN(-5)),JN(3))
    Etst(Intlength(JN(-6)),JN(3))
    Etst(Intlength(JN(-7)),JN(3))
    Etst(Intlength(JN(-8)),JN(3))
    Etst(Intlength(JN(-9)),JN(4))
    Etst(Intlength(I0011_1100),JN(4+1*pntsize))
    Etst(Intlength(J1011_1101),JN(2+1*pntsize))
    Etst(Intlength(intcons(0,I0011_1100)),JN(4+2*pntsize))
    Etst(Intlength(intcons(0,J1011_1101)),JN(2+2*pntsize))
    Etst(Intlength(K2(0,-1)),JN(0+1*pntsize))
    Etst(Intlength(K2(-1,0)),JN(0+1*pntsize))

    Xtst(Logcount(TT))
    Xtst(Logcount(FF))
    Xtst(Logcount(P0))
    Xtst(Logcount(M0))
    Xtst(Logcount(B0))
    Etst(Logcount(Z),Z)
    Etst(Logcount(M),Z)
    Etst(Logcount(JN(1)),JN(1))
    Etst(Logcount(JN(2)),JN(1))
    Etst(Logcount(JN(3)),JN(2))
    Etst(Logcount(JN(4)),JN(1))
    Etst(Logcount(JN(5)),JN(2))
    Etst(Logcount(JN(6)),JN(2))
    Etst(Logcount(JN(7)),JN(3))
    Etst(Logcount(JN(8)),JN(1))
    Etst(Logcount(JN(-2)),JN(1))
    Etst(Logcount(JN(-3)),JN(1))
    Etst(Logcount(JN(-4)),JN(2))
    Etst(Logcount(JN(-5)),JN(1))
    Etst(Logcount(JN(-6)),JN(2))
    Etst(Logcount(JN(-7)),JN(2))
    Etst(Logcount(JN(-8)),JN(3))
    Etst(Logcount(JN(-9)),JN(1))
    Etst(Logcount(I0011_1100),JN(4))
    Etst(Logcount(J1011_1101),JN(2))
    Etst(Logcount(intcons(0,I0011_1100)),JN(4))
    Etst(Logcount(intcons(0,J1011_1101)),JN(2+1*pntsize))
    Etst(Logcount(K2(0,-1)),JN(0+1*pntsize))
    Etst(Logcount(K2(-1,0)),JN(0+1*pntsize))

    Xtst(Lognot(TT))
    Xtst(Lognot(FF))
    Xtst(Lognot(P0))
    Xtst(Lognot(M0))
    Xtst(Lognot(B0))
    Etst(Lognot(Z),M)
    Etst(Lognot(M),Z)
    Etst(Lognot(I0011),J1100)
    Etst(Lognot(J1010_0101),I0101_1010)

    Xtst(Logand(TT,TT))
    Xtst(Logand(TT,FF))
    Xtst(Logand(TT,I0))
    Xtst(Logand(TT,P0))
    Xtst(Logand(TT,M0))
    Xtst(Logand(TT,B0))
    Xtst(Logand(FF,TT))
    Xtst(Logand(FF,FF))
    Xtst(Logand(FF,I0))
    Xtst(Logand(FF,P0))
    Xtst(Logand(FF,M0))
    Xtst(Logand(FF,B0))
    Xtst(Logand(I0,TT))
    Xtst(Logand(I0,FF))
    Xtst(Logand(I0,P0))
    Xtst(Logand(I0,M0))
    Xtst(Logand(I0,B0))
    Xtst(Logand(P0,TT))
    Xtst(Logand(P0,FF))
    Xtst(Logand(P0,I0))
    Xtst(Logand(P0,P0))
    Xtst(Logand(P0,M0))
    Xtst(Logand(P0,B0))
    Xtst(Logand(M0,TT))
    Xtst(Logand(M0,FF))
    Xtst(Logand(M0,I0))
    Xtst(Logand(M0,P0))
    Xtst(Logand(M0,M0))
    Xtst(Logand(M0,B0))
    Xtst(Logand(B0,TT))
    Xtst(Logand(B0,FF))
    Xtst(Logand(B0,I0))
    Xtst(Logand(B0,P0))
    Xtst(Logand(B0,M0))
    Xtst(Logand(B0,B0))
    Etst(Logand(I0011     ,I0101     ),I0001     )
    Etst(Logand(I1100     ,I1010     ),I1000     )
    Etst(Logand(I0011_1100,I0101_1010),I0001_1000)
    Etst(Logand(I0011     ,I0101_1010),I0001     )
    Etst(Logand(I0101_1010,I0011     ),I0001     )
    Etst(Logand(I0011     ,J0101     ),I0001     )
    Etst(Logand(I1100     ,J1010     ),I1000     )
    Etst(Logand(I0011_1100,J0101_1010),I0001_1000)
    Etst(Logand(I0011     ,J0101_1010),I0001     )
    Etst(Logand(I0101_1010,J0011     ),intcons(Z0001,I1010))
    Etst(Logand(J0011     ,I0101     ),I0001     )
    Etst(Logand(J1100     ,I1010     ),I1000     )
    Etst(Logand(J0011_1100,I0101_1010),I0001_1000)
    Etst(Logand(J0011     ,I0101_1010),intcons(Z0001,I1010))
    Etst(Logand(J0101_1010,I0011     ),I0001     )
    Etst(Logand(J0011     ,J0101     ),J0001     )
    Etst(Logand(J1100     ,J1010     ),J1000     )
    Etst(Logand(J0011_1100,J0101_1010),J0001_1000)
    Etst(Logand(J0011     ,J0101_1010),intcons(M0001,J1010))
    Etst(Logand(J0101_1010,J0011     ),intcons(M0001,J1010))
    Etst(Logand(I0011_1100,I1000_0001),Z)
    Etst(Logand(EE,I0),I0)
    Etst(Logand(EE,J1),EE)
    Etst(Logand(I0,EE),I0)
    Etst(Logand(J1,EE),EE)

    Xtst(Logior(TT,TT))
    Xtst(Logior(TT,FF))
    Xtst(Logior(TT,I0))
    Xtst(Logior(TT,P0))
    Xtst(Logior(TT,M0))
    Xtst(Logior(TT,B0))
    Xtst(Logior(FF,TT))
    Xtst(Logior(FF,FF))
    Xtst(Logior(FF,I0))
    Xtst(Logior(FF,P0))
    Xtst(Logior(FF,M0))
    Xtst(Logior(FF,B0))
    Xtst(Logior(I0,TT))
    Xtst(Logior(I0,FF))
    Xtst(Logior(I0,P0))
    Xtst(Logior(I0,M0))
    Xtst(Logior(I0,B0))
    Xtst(Logior(P0,TT))
    Xtst(Logior(P0,FF))
    Xtst(Logior(P0,I0))
    Xtst(Logior(P0,P0))
    Xtst(Logior(P0,M0))
    Xtst(Logior(P0,B0))
    Xtst(Logior(M0,TT))
    Xtst(Logior(M0,FF))
    Xtst(Logior(M0,I0))
    Xtst(Logior(M0,P0))
    Xtst(Logior(M0,M0))
    Xtst(Logior(M0,B0))
    Xtst(Logior(B0,TT))
    Xtst(Logior(B0,FF))
    Xtst(Logior(B0,I0))
    Xtst(Logior(B0,P0))
    Xtst(Logior(B0,M0))
    Xtst(Logior(B0,B0))
    Etst(Logior(I0011     ,I0101     ),I0111     )
    Etst(Logior(I1100     ,I1010     ),I1110     )
    Etst(Logior(I0011_1100,I0101_1010),I0111_1110)
    Etst(Logior(I0011     ,I0101_1010),intcons(Z0111,I1010))
    Etst(Logior(I0101_1010,I0011     ),intcons(Z0111,I1010))
    Etst(Logior(I0011     ,J0101     ),J0111     )
    Etst(Logior(I1100     ,J1010     ),J1110     )
    Etst(Logior(I0011_1100,J0101_1010),J0111_1110)
    Etst(Logior(I0011     ,J0101_1010),intcons(M0111,J1010))
    Etst(Logior(I0101_1010,J0011     ),J0111     )
    Etst(Logior(J0011     ,I0101     ),J0111     )
    Etst(Logior(J1100     ,I1010     ),J1110     )
    Etst(Logior(J0011_1100,I0101_1010),J0111_1110)
    Etst(Logior(J0011     ,I0101_1010),J0111     )
    Etst(Logior(J0101_1010,I0011     ),intcons(M0111,J1010))
    Etst(Logior(J0011     ,J0101     ),J0111     )
    Etst(Logior(J1100     ,J1010     ),J1110     )
    Etst(Logior(J0011_1100,J0101_1010),J0111_1110)
    Etst(Logior(J0011     ,J0101_1010),J0111     )
    Etst(Logior(J0101_1010,J0011     ),J0111     )
    Etst(Logior(J0011_1100,J1110_0111),M)
    Etst(Logior(EE,I0),EE)
    Etst(Logior(EE,J1),J1)
    Etst(Logior(I0,EE),EE)
    Etst(Logior(J1,EE),J1)

    Xtst(Logxor(TT,TT))
    Xtst(Logxor(TT,FF))
    Xtst(Logxor(TT,I0))
    Xtst(Logxor(TT,P0))
    Xtst(Logxor(TT,M0))
    Xtst(Logxor(TT,B0))
    Xtst(Logxor(FF,TT))
    Xtst(Logxor(FF,FF))
    Xtst(Logxor(FF,I0))
    Xtst(Logxor(FF,P0))
    Xtst(Logxor(FF,M0))
    Xtst(Logxor(FF,B0))
    Xtst(Logxor(I0,TT))
    Xtst(Logxor(I0,FF))
    Xtst(Logxor(I0,P0))
    Xtst(Logxor(I0,M0))
    Xtst(Logxor(I0,B0))
    Xtst(Logxor(P0,TT))
    Xtst(Logxor(P0,FF))
    Xtst(Logxor(P0,I0))
    Xtst(Logxor(P0,P0))
    Xtst(Logxor(P0,M0))
    Xtst(Logxor(P0,B0))
    Xtst(Logxor(M0,TT))
    Xtst(Logxor(M0,FF))
    Xtst(Logxor(M0,I0))
    Xtst(Logxor(M0,P0))
    Xtst(Logxor(M0,M0))
    Xtst(Logxor(M0,B0))
    Xtst(Logxor(B0,TT))
    Xtst(Logxor(B0,FF))
    Xtst(Logxor(B0,I0))
    Xtst(Logxor(B0,P0))
    Xtst(Logxor(B0,M0))
    Xtst(Logxor(B0,B0))
    Etst(Logxor(I0011     ,I0101     ),I0110     )
    Etst(Logxor(I1100     ,I1010     ),I0110     )
    Etst(Logxor(I0011_1100,I0101_1010),I0110_0110)
    Etst(Logxor(I0011     ,I0101_1010),intcons(Z0110,I1010))
    Etst(Logxor(I0101_1010,I0011     ),intcons(Z0110,I1010))
    Etst(Logxor(I0011     ,J0101     ),J0110     )
    Etst(Logxor(I1100     ,J1010     ),J0110     )
    Etst(Logxor(I0011_1100,J0101_1010),J0110_0110)
    Etst(Logxor(I0011     ,J0101_1010),intcons(M0110,J1010))
    Etst(Logxor(I0101_1010,J0011     ),intcons(M0110,J0101))
    Etst(Logxor(J0011     ,I0101     ),J0110     )
    Etst(Logxor(J1100     ,I1010     ),J0110     )
    Etst(Logxor(J0011_1100,I0101_1010),J0110_0110)
    Etst(Logxor(J0011     ,I0101_1010),intcons(M0110,J0101))
    Etst(Logxor(J0101_1010,I0011     ),intcons(M0110,J1010))
    Etst(Logxor(J0011     ,J0101     ),I0110     )
    Etst(Logxor(J1100     ,J1010     ),I0110     )
    Etst(Logxor(J0011_1100,J0101_1010),I0110_0110)
    Etst(Logxor(J0011     ,J0101_1010),intcons(Z0110,I0101))
    Etst(Logxor(J0101_1010,J0011     ),intcons(Z0110,I0101))
    Etst(Logxor(J0101_1010,J0101_1010),Z)
    Etst(Logxor(J0101_1010,I1010_0101),M)
    Etst(Logxor(EE,I0),EE)
    Etst(Logxor(EE,J1),J2)
    Etst(Logxor(I0,EE),EE)
    Etst(Logxor(J1,EE),J2)

    Xtst(Logandc1(TT,TT))
    Xtst(Logandc1(TT,FF))
    Xtst(Logandc1(TT,I0))
    Xtst(Logandc1(TT,P0))
    Xtst(Logandc1(TT,M0))
    Xtst(Logandc1(TT,B0))
    Xtst(Logandc1(FF,TT))
    Xtst(Logandc1(FF,FF))
    Xtst(Logandc1(FF,I0))
    Xtst(Logandc1(FF,P0))
    Xtst(Logandc1(FF,M0))
    Xtst(Logandc1(FF,B0))
    Xtst(Logandc1(I0,TT))
    Xtst(Logandc1(I0,FF))
    Xtst(Logandc1(I0,P0))
    Xtst(Logandc1(I0,M0))
    Xtst(Logandc1(I0,B0))
    Xtst(Logandc1(P0,TT))
    Xtst(Logandc1(P0,FF))
    Xtst(Logandc1(P0,I0))
    Xtst(Logandc1(P0,P0))
    Xtst(Logandc1(P0,M0))
    Xtst(Logandc1(P0,B0))
    Xtst(Logandc1(M0,TT))
    Xtst(Logandc1(M0,FF))
    Xtst(Logandc1(M0,I0))
    Xtst(Logandc1(M0,P0))
    Xtst(Logandc1(M0,M0))
    Xtst(Logandc1(M0,B0))
    Xtst(Logandc1(B0,TT))
    Xtst(Logandc1(B0,FF))
    Xtst(Logandc1(B0,I0))
    Xtst(Logandc1(B0,P0))
    Xtst(Logandc1(B0,M0))
    Xtst(Logandc1(B0,B0))
    Etst(Logandc1(lognot(I0011     ),I0101     ),I0001     )
    Etst(Logandc1(lognot(I1100     ),I1010     ),I1000     )
    Etst(Logandc1(lognot(I0011_1100),I0101_1010),I0001_1000)
    Etst(Logandc1(lognot(I0011     ),I0101_1010),I0001     )
    Etst(Logandc1(lognot(I0101_1010),I0011     ),I0001     )
    Etst(Logandc1(lognot(I0011     ),J0101     ),I0001     )
    Etst(Logandc1(lognot(I1100     ),J1010     ),I1000     )
    Etst(Logandc1(lognot(I0011_1100),J0101_1010),I0001_1000)
    Etst(Logandc1(lognot(I0011     ),J0101_1010),I0001     )
    Etst(Logandc1(lognot(I0101_1010),J0011     ),intcons(Z0001,I1010))
    Etst(Logandc1(lognot(J0011     ),I0101     ),I0001     )
    Etst(Logandc1(lognot(J1100     ),I1010     ),I1000     )
    Etst(Logandc1(lognot(J0011_1100),I0101_1010),I0001_1000)
    Etst(Logandc1(lognot(J0011     ),I0101_1010),intcons(Z0001,I1010))
    Etst(Logandc1(lognot(J0101_1010),I0011     ),I0001     )
    Etst(Logandc1(lognot(J0011     ),J0101     ),J0001     )
    Etst(Logandc1(lognot(J1100     ),J1010     ),J1000     )
    Etst(Logandc1(lognot(J0011_1100),J0101_1010),J0001_1000)
    Etst(Logandc1(lognot(J0011     ),J0101_1010),intcons(M0001,J1010))
    Etst(Logandc1(lognot(J0101_1010),J0011     ),intcons(M0001,J1010))
    Etst(Logandc1(EE,I0),I0)
    Etst(Logandc1(EE,J1),J2)
    Etst(Logandc1(I0,EE),EE)
    Etst(Logandc1(J1,EE),I0)

    Xtst(Logandc2(TT,TT))
    Xtst(Logandc2(TT,FF))
    Xtst(Logandc2(TT,I0))
    Xtst(Logandc2(TT,P0))
    Xtst(Logandc2(TT,M0))
    Xtst(Logandc2(TT,B0))
    Xtst(Logandc2(FF,TT))
    Xtst(Logandc2(FF,FF))
    Xtst(Logandc2(FF,I0))
    Xtst(Logandc2(FF,P0))
    Xtst(Logandc2(FF,M0))
    Xtst(Logandc2(FF,B0))
    Xtst(Logandc2(I0,TT))
    Xtst(Logandc2(I0,FF))
    Xtst(Logandc2(I0,P0))
    Xtst(Logandc2(I0,M0))
    Xtst(Logandc2(I0,B0))
    Xtst(Logandc2(P0,TT))
    Xtst(Logandc2(P0,FF))
    Xtst(Logandc2(P0,I0))
    Xtst(Logandc2(P0,P0))
    Xtst(Logandc2(P0,M0))
    Xtst(Logandc2(P0,B0))
    Xtst(Logandc2(M0,TT))
    Xtst(Logandc2(M0,FF))
    Xtst(Logandc2(M0,I0))
    Xtst(Logandc2(M0,P0))
    Xtst(Logandc2(M0,M0))
    Xtst(Logandc2(M0,B0))
    Xtst(Logandc2(B0,TT))
    Xtst(Logandc2(B0,FF))
    Xtst(Logandc2(B0,I0))
    Xtst(Logandc2(B0,P0))
    Xtst(Logandc2(B0,M0))
    Xtst(Logandc2(B0,B0))
    Etst(Logandc2(I0011     ,lognot(I0101     )),I0001     )
    Etst(Logandc2(I1100     ,lognot(I1010     )),I1000     )
    Etst(Logandc2(I0011_1100,lognot(I0101_1010)),I0001_1000)
    Etst(Logandc2(I0011     ,lognot(I0101_1010)),I0001     )
    Etst(Logandc2(I0101_1010,lognot(I0011     )),I0001     )
    Etst(Logandc2(I0011     ,lognot(J0101     )),I0001     )
    Etst(Logandc2(I1100     ,lognot(J1010     )),I1000     )
    Etst(Logandc2(I0011_1100,lognot(J0101_1010)),I0001_1000)
    Etst(Logandc2(I0011     ,lognot(J0101_1010)),I0001     )
    Etst(Logandc2(I0101_1010,lognot(J0011     )),intcons(Z0001,I1010))
    Etst(Logandc2(J0011     ,lognot(I0101     )),I0001     )
    Etst(Logandc2(J1100     ,lognot(I1010     )),I1000     )
    Etst(Logandc2(J0011_1100,lognot(I0101_1010)),I0001_1000)
    Etst(Logandc2(J0011     ,lognot(I0101_1010)),intcons(Z0001,I1010))
    Etst(Logandc2(J0101_1010,lognot(I0011     )),I0001     )
    Etst(Logandc2(J0011     ,lognot(J0101     )),J0001     )
    Etst(Logandc2(J1100     ,lognot(J1010     )),J1000     )
    Etst(Logandc2(J0011_1100,lognot(J0101_1010)),J0001_1000)
    Etst(Logandc2(J0011     ,lognot(J0101_1010)),intcons(M0001,J1010))
    Etst(Logandc2(J0101_1010,lognot(J0011     )),intcons(M0001,J1010))
    Etst(Logandc2(EE,I0),EE)
    Etst(Logandc2(EE,J1),I0)
    Etst(Logandc2(I0,EE),I0)
    Etst(Logandc2(J1,EE),J2)

    Xtst(Lognand(TT,TT))
    Xtst(Lognand(TT,FF))
    Xtst(Lognand(TT,I0))
    Xtst(Lognand(TT,P0))
    Xtst(Lognand(TT,M0))
    Xtst(Lognand(TT,B0))
    Xtst(Lognand(FF,TT))
    Xtst(Lognand(FF,FF))
    Xtst(Lognand(FF,I0))
    Xtst(Lognand(FF,P0))
    Xtst(Lognand(FF,M0))
    Xtst(Lognand(FF,B0))
    Xtst(Lognand(I0,TT))
    Xtst(Lognand(I0,FF))
    Xtst(Lognand(I0,P0))
    Xtst(Lognand(I0,M0))
    Xtst(Lognand(I0,B0))
    Xtst(Lognand(P0,TT))
    Xtst(Lognand(P0,FF))
    Xtst(Lognand(P0,I0))
    Xtst(Lognand(P0,P0))
    Xtst(Lognand(P0,M0))
    Xtst(Lognand(P0,B0))
    Xtst(Lognand(M0,TT))
    Xtst(Lognand(M0,FF))
    Xtst(Lognand(M0,I0))
    Xtst(Lognand(M0,P0))
    Xtst(Lognand(M0,M0))
    Xtst(Lognand(M0,B0))
    Xtst(Lognand(B0,TT))
    Xtst(Lognand(B0,FF))
    Xtst(Lognand(B0,I0))
    Xtst(Lognand(B0,P0))
    Xtst(Lognand(B0,M0))
    Xtst(Lognand(B0,B0))
    Etst(Lognand(I0011     ,I0101     ),lognot(I0001     ))
    Etst(Lognand(I1100     ,I1010     ),lognot(I1000     ))
    Etst(Lognand(I0011_1100,I0101_1010),lognot(I0001_1000))
    Etst(Lognand(I0011     ,I0101_1010),lognot(I0001     ))
    Etst(Lognand(I0101_1010,I0011     ),lognot(I0001     ))
    Etst(Lognand(I0011     ,J0101     ),lognot(I0001     ))
    Etst(Lognand(I1100     ,J1010     ),lognot(I1000     ))
    Etst(Lognand(I0011_1100,J0101_1010),lognot(I0001_1000))
    Etst(Lognand(I0011     ,J0101_1010),lognot(I0001     ))
    Etst(Lognand(I0101_1010,J0011     ),lognot(intcons(Z0001,I1010)))
    Etst(Lognand(J0011     ,I0101     ),lognot(I0001     ))
    Etst(Lognand(J1100     ,I1010     ),lognot(I1000     ))
    Etst(Lognand(J0011_1100,I0101_1010),lognot(I0001_1000))
    Etst(Lognand(J0011     ,I0101_1010),lognot(intcons(Z0001,I1010)))
    Etst(Lognand(J0101_1010,I0011     ),lognot(I0001     ))
    Etst(Lognand(J0011     ,J0101     ),lognot(J0001     ))
    Etst(Lognand(J1100     ,J1010     ),lognot(J1000     ))
    Etst(Lognand(J0011_1100,J0101_1010),lognot(J0001_1000))
    Etst(Lognand(J0011     ,J0101_1010),lognot(intcons(M0001,J1010)))
    Etst(Lognand(J0101_1010,J0011     ),lognot(intcons(M0001,J1010)))
    Etst(Lognand(EE,I0),J1)
    Etst(Lognand(EE,J1),J2)
    Etst(Lognand(I0,EE),J1)
    Etst(Lognand(J1,EE),J2)

    Xtst(Logorc1(TT,TT))
    Xtst(Logorc1(TT,FF))
    Xtst(Logorc1(TT,I0))
    Xtst(Logorc1(TT,P0))
    Xtst(Logorc1(TT,M0))
    Xtst(Logorc1(TT,B0))
    Xtst(Logorc1(FF,TT))
    Xtst(Logorc1(FF,FF))
    Xtst(Logorc1(FF,I0))
    Xtst(Logorc1(FF,P0))
    Xtst(Logorc1(FF,M0))
    Xtst(Logorc1(FF,B0))
    Xtst(Logorc1(I0,TT))
    Xtst(Logorc1(I0,FF))
    Xtst(Logorc1(I0,P0))
    Xtst(Logorc1(I0,M0))
    Xtst(Logorc1(I0,B0))
    Xtst(Logorc1(P0,TT))
    Xtst(Logorc1(P0,FF))
    Xtst(Logorc1(P0,I0))
    Xtst(Logorc1(P0,P0))
    Xtst(Logorc1(P0,M0))
    Xtst(Logorc1(P0,B0))
    Xtst(Logorc1(M0,TT))
    Xtst(Logorc1(M0,FF))
    Xtst(Logorc1(M0,I0))
    Xtst(Logorc1(M0,P0))
    Xtst(Logorc1(M0,M0))
    Xtst(Logorc1(M0,B0))
    Xtst(Logorc1(B0,TT))
    Xtst(Logorc1(B0,FF))
    Xtst(Logorc1(B0,I0))
    Xtst(Logorc1(B0,P0))
    Xtst(Logorc1(B0,M0))
    Xtst(Logorc1(B0,B0))
    Etst(Logorc1(lognot(I0011     ),I0101     ),I0111     )
    Etst(Logorc1(lognot(I1100     ),I1010     ),I1110     )
    Etst(Logorc1(lognot(I0011_1100),I0101_1010),I0111_1110)
    Etst(Logorc1(lognot(I0011     ),I0101_1010),intcons(Z0111,I1010))
    Etst(Logorc1(lognot(I0101_1010),I0011     ),intcons(Z0111,I1010))
    Etst(Logorc1(lognot(I0011     ),J0101     ),J0111     )
    Etst(Logorc1(lognot(I1100     ),J1010     ),J1110     )
    Etst(Logorc1(lognot(I0011_1100),J0101_1010),J0111_1110)
    Etst(Logorc1(lognot(I0011     ),J0101_1010),intcons(M0111,J1010))
    Etst(Logorc1(lognot(I0101_1010),J0011     ),J0111     )
    Etst(Logorc1(lognot(J0011     ),I0101     ),J0111     )
    Etst(Logorc1(lognot(J1100     ),I1010     ),J1110     )
    Etst(Logorc1(lognot(J0011_1100),I0101_1010),J0111_1110)
    Etst(Logorc1(lognot(J0011     ),I0101_1010),J0111     )
    Etst(Logorc1(lognot(J0101_1010),I0011     ),intcons(M0111,J1010))
    Etst(Logorc1(lognot(J0011     ),J0101     ),J0111     )
    Etst(Logorc1(lognot(J1100     ),J1010     ),J1110     )
    Etst(Logorc1(lognot(J0011_1100),J0101_1010),J0111_1110)
    Etst(Logorc1(lognot(J0011     ),J0101_1010),J0111     )
    Etst(Logorc1(lognot(J0101_1010),J0011     ),J0111     )
    Etst(Logorc1(EE,I0),J2)
    Etst(Logorc1(EE,J1),J1)
    Etst(Logorc1(I0,EE),J1)
    Etst(Logorc1(J1,EE),EE)

    Xtst(Logorc2(TT,TT))
    Xtst(Logorc2(TT,FF))
    Xtst(Logorc2(TT,I0))
    Xtst(Logorc2(TT,P0))
    Xtst(Logorc2(TT,M0))
    Xtst(Logorc2(TT,B0))
    Xtst(Logorc2(FF,TT))
    Xtst(Logorc2(FF,FF))
    Xtst(Logorc2(FF,I0))
    Xtst(Logorc2(FF,P0))
    Xtst(Logorc2(FF,M0))
    Xtst(Logorc2(FF,B0))
    Xtst(Logorc2(I0,TT))
    Xtst(Logorc2(I0,FF))
    Xtst(Logorc2(I0,P0))
    Xtst(Logorc2(I0,M0))
    Xtst(Logorc2(I0,B0))
    Xtst(Logorc2(P0,TT))
    Xtst(Logorc2(P0,FF))
    Xtst(Logorc2(P0,I0))
    Xtst(Logorc2(P0,P0))
    Xtst(Logorc2(P0,M0))
    Xtst(Logorc2(P0,B0))
    Xtst(Logorc2(M0,TT))
    Xtst(Logorc2(M0,FF))
    Xtst(Logorc2(M0,I0))
    Xtst(Logorc2(M0,P0))
    Xtst(Logorc2(M0,M0))
    Xtst(Logorc2(M0,B0))
    Xtst(Logorc2(B0,TT))
    Xtst(Logorc2(B0,FF))
    Xtst(Logorc2(B0,I0))
    Xtst(Logorc2(B0,P0))
    Xtst(Logorc2(B0,M0))
    Xtst(Logorc2(B0,B0))
    Etst(Logorc2(I0011     ,lognot(I0101     )),I0111     )
    Etst(Logorc2(I1100     ,lognot(I1010     )),I1110     )
    Etst(Logorc2(I0011_1100,lognot(I0101_1010)),I0111_1110)
    Etst(Logorc2(I0011     ,lognot(I0101_1010)),intcons(Z0111,I1010))
    Etst(Logorc2(I0101_1010,lognot(I0011     )),intcons(Z0111,I1010))
    Etst(Logorc2(I0011     ,lognot(J0101     )),J0111     )
    Etst(Logorc2(I1100     ,lognot(J1010     )),J1110     )
    Etst(Logorc2(I0011_1100,lognot(J0101_1010)),J0111_1110)
    Etst(Logorc2(I0011     ,lognot(J0101_1010)),intcons(M0111,J1010))
    Etst(Logorc2(I0101_1010,lognot(J0011     )),J0111     )
    Etst(Logorc2(J0011     ,lognot(I0101     )),J0111     )
    Etst(Logorc2(J1100     ,lognot(I1010     )),J1110     )
    Etst(Logorc2(J0011_1100,lognot(I0101_1010)),J0111_1110)
    Etst(Logorc2(J0011     ,lognot(I0101_1010)),J0111     )
    Etst(Logorc2(J0101_1010,lognot(I0011     )),intcons(M0111,J1010))
    Etst(Logorc2(J0011     ,lognot(J0101     )),J0111     )
    Etst(Logorc2(J1100     ,lognot(J1010     )),J1110     )
    Etst(Logorc2(J0011_1100,lognot(J0101_1010)),J0111_1110)
    Etst(Logorc2(J0011     ,lognot(J0101_1010)),J0111     )
    Etst(Logorc2(J0101_1010,lognot(J0011     )),J0111     )
    Etst(Logorc2(EE,I0),J1)
    Etst(Logorc2(EE,J1),EE)
    Etst(Logorc2(I0,EE),J2)
    Etst(Logorc2(J1,EE),J1)

    Xtst(Lognor(TT,TT))
    Xtst(Lognor(TT,FF))
    Xtst(Lognor(TT,I0))
    Xtst(Lognor(TT,P0))
    Xtst(Lognor(TT,M0))
    Xtst(Lognor(TT,B0))
    Xtst(Lognor(FF,TT))
    Xtst(Lognor(FF,FF))
    Xtst(Lognor(FF,I0))
    Xtst(Lognor(FF,P0))
    Xtst(Lognor(FF,M0))
    Xtst(Lognor(FF,B0))
    Xtst(Lognor(I0,TT))
    Xtst(Lognor(I0,FF))
    Xtst(Lognor(I0,P0))
    Xtst(Lognor(I0,M0))
    Xtst(Lognor(I0,B0))
    Xtst(Lognor(P0,TT))
    Xtst(Lognor(P0,FF))
    Xtst(Lognor(P0,I0))
    Xtst(Lognor(P0,P0))
    Xtst(Lognor(P0,M0))
    Xtst(Lognor(P0,B0))
    Xtst(Lognor(M0,TT))
    Xtst(Lognor(M0,FF))
    Xtst(Lognor(M0,I0))
    Xtst(Lognor(M0,P0))
    Xtst(Lognor(M0,M0))
    Xtst(Lognor(M0,B0))
    Xtst(Lognor(B0,TT))
    Xtst(Lognor(B0,FF))
    Xtst(Lognor(B0,I0))
    Xtst(Lognor(B0,P0))
    Xtst(Lognor(B0,M0))
    Xtst(Lognor(B0,B0))
    Etst(Lognor(I0011     ,I0101     ),lognot(I0111     ))
    Etst(Lognor(I1100     ,I1010     ),lognot(I1110     ))
    Etst(Lognor(I0011_1100,I0101_1010),lognot(I0111_1110))
    Etst(Lognor(I0011     ,I0101_1010),lognot(intcons(Z0111,I1010)))
    Etst(Lognor(I0101_1010,I0011     ),lognot(intcons(Z0111,I1010)))
    Etst(Lognor(I0011     ,J0101     ),lognot(J0111     ))
    Etst(Lognor(I1100     ,J1010     ),lognot(J1110     ))
    Etst(Lognor(I0011_1100,J0101_1010),lognot(J0111_1110))
    Etst(Lognor(I0011     ,J0101_1010),lognot(intcons(M0111,J1010)))
    Etst(Lognor(I0101_1010,J0011     ),lognot(J0111     ))
    Etst(Lognor(J0011     ,I0101     ),lognot(J0111     ))
    Etst(Lognor(J1100     ,I1010     ),lognot(J1110     ))
    Etst(Lognor(J0011_1100,I0101_1010),lognot(J0111_1110))
    Etst(Lognor(J0011     ,I0101_1010),lognot(J0111     ))
    Etst(Lognor(J0101_1010,I0011     ),lognot(intcons(M0111,J1010)))
    Etst(Lognor(J0011     ,J0101     ),lognot(J0111     ))
    Etst(Lognor(J1100     ,J1010     ),lognot(J1110     ))
    Etst(Lognor(J0011_1100,J0101_1010),lognot(J0111_1110))
    Etst(Lognor(J0011     ,J0101_1010),lognot(J0111     ))
    Etst(Lognor(J0101_1010,J0011     ),lognot(J0111     ))
    Etst(Lognor(EE,I0),J2)
    Etst(Lognor(EE,J1),I0)
    Etst(Lognor(I0,EE),J2)
    Etst(Lognor(J1,EE),I0)

    Xtst(Logeqv(TT,TT))
    Xtst(Logeqv(TT,FF))
    Xtst(Logeqv(TT,I0))
    Xtst(Logeqv(TT,P0))
    Xtst(Logeqv(TT,M0))
    Xtst(Logeqv(TT,B0))
    Xtst(Logeqv(FF,TT))
    Xtst(Logeqv(FF,FF))
    Xtst(Logeqv(FF,I0))
    Xtst(Logeqv(FF,P0))
    Xtst(Logeqv(FF,M0))
    Xtst(Logeqv(FF,B0))
    Xtst(Logeqv(I0,TT))
    Xtst(Logeqv(I0,FF))
    Xtst(Logeqv(I0,P0))
    Xtst(Logeqv(I0,M0))
    Xtst(Logeqv(I0,B0))
    Xtst(Logeqv(P0,TT))
    Xtst(Logeqv(P0,FF))
    Xtst(Logeqv(P0,I0))
    Xtst(Logeqv(P0,P0))
    Xtst(Logeqv(P0,M0))
    Xtst(Logeqv(P0,B0))
    Xtst(Logeqv(M0,TT))
    Xtst(Logeqv(M0,FF))
    Xtst(Logeqv(M0,I0))
    Xtst(Logeqv(M0,P0))
    Xtst(Logeqv(M0,M0))
    Xtst(Logeqv(M0,B0))
    Xtst(Logeqv(B0,TT))
    Xtst(Logeqv(B0,FF))
    Xtst(Logeqv(B0,I0))
    Xtst(Logeqv(B0,P0))
    Xtst(Logeqv(B0,M0))
    Xtst(Logeqv(B0,B0))
    Etst(Logeqv(lognot(I0011     ),I0101     ),I0110     )
    Etst(Logeqv(lognot(I1100     ),I1010     ),I0110     )
    Etst(Logeqv(lognot(I0011_1100),I0101_1010),I0110_0110)
    Etst(Logeqv(lognot(I0011     ),I0101_1010),intcons(Z0110,I1010))
    Etst(Logeqv(lognot(I0101_1010),I0011     ),intcons(Z0110,I1010))
    Etst(Logeqv(lognot(I0011     ),J0101     ),J0110     )
    Etst(Logeqv(lognot(I1100     ),J1010     ),J0110     )
    Etst(Logeqv(lognot(I0011_1100),J0101_1010),J0110_0110)
    Etst(Logeqv(lognot(I0011     ),J0101_1010),intcons(M0110,J1010))
    Etst(Logeqv(lognot(I0101_1010),J0011     ),intcons(M0110,J0101))
    Etst(Logeqv(lognot(J0011     ),I0101     ),J0110     )
    Etst(Logeqv(lognot(J1100     ),I1010     ),J0110     )
    Etst(Logeqv(lognot(J0011_1100),I0101_1010),J0110_0110)
    Etst(Logeqv(lognot(J0011     ),I0101_1010),intcons(M0110,J0101))
    Etst(Logeqv(lognot(J0101_1010),I0011     ),intcons(M0110,J1010))
    Etst(Logeqv(lognot(J0011     ),J0101     ),I0110     )
    Etst(Logeqv(lognot(J1100     ),J1010     ),I0110     )
    Etst(Logeqv(lognot(J0011_1100),J0101_1010),I0110_0110)
    Etst(Logeqv(lognot(J0011     ),J0101_1010),intcons(Z0110,I0101))
    Etst(Logeqv(lognot(J0101_1010),J0011     ),intcons(Z0110,I0101))
    Etst(Logeqv(EE,I0),J2)
    Etst(Logeqv(EE,J1),EE)
    Etst(Logeqv(I0,EE),J2)
    Etst(Logeqv(J1,EE),EE)

    Xtst(Logtest(TT,TT))
    Xtst(Logtest(TT,FF))
    Xtst(Logtest(TT,I0))
    Xtst(Logtest(TT,P0))
    Xtst(Logtest(TT,M0))
    Xtst(Logtest(TT,B0))
    Xtst(Logtest(FF,TT))
    Xtst(Logtest(FF,FF))
    Xtst(Logtest(FF,I0))
    Xtst(Logtest(FF,P0))
    Xtst(Logtest(FF,M0))
    Xtst(Logtest(FF,B0))
    Xtst(Logtest(I0,TT))
    Xtst(Logtest(I0,FF))
    Xtst(Logtest(I0,P0))
    Xtst(Logtest(I0,M0))
    Xtst(Logtest(I0,B0))
    Xtst(Logtest(P0,TT))
    Xtst(Logtest(P0,FF))
    Xtst(Logtest(P0,I0))
    Xtst(Logtest(P0,P0))
    Xtst(Logtest(P0,M0))
    Xtst(Logtest(P0,B0))
    Xtst(Logtest(M0,TT))
    Xtst(Logtest(M0,FF))
    Xtst(Logtest(M0,I0))
    Xtst(Logtest(M0,P0))
    Xtst(Logtest(M0,M0))
    Xtst(Logtest(M0,B0))
    Xtst(Logtest(B0,TT))
    Xtst(Logtest(B0,FF))
    Xtst(Logtest(B0,I0))
    Xtst(Logtest(B0,P0))
    Xtst(Logtest(B0,M0))
    Xtst(Logtest(B0,B0))
    Ttst(Logtest(I0011     ,I0101     ))
    Ttst(Logtest(I1100     ,I1010     ))
    Ttst(Logtest(I0011_1100,I0101_1010))
    Ttst(Logtest(I0011     ,I0101_1010))
    Ttst(Logtest(I0101_1010,I0011     ))
    Ttst(Logtest(I0011     ,J0101     ))
    Ttst(Logtest(I1100     ,J1010     ))
    Ttst(Logtest(I0011_1100,J0101_1010))
    Ttst(Logtest(I0011     ,J0101_1010))
    Ttst(Logtest(I0101_1010,J0011     ))
    Ttst(Logtest(J0011     ,I0101     ))
    Ttst(Logtest(J1100     ,I1010     ))
    Ttst(Logtest(J0011_1100,I0101_1010))
    Ttst(Logtest(J0011     ,I0101_1010))
    Ttst(Logtest(J0101_1010,I0011     ))
    Ttst(Logtest(J0011     ,J0101     ))
    Ttst(Logtest(J1100     ,J1010     ))
    Ttst(Logtest(J0011_1100,J0101_1010))
    Ttst(Logtest(J0011     ,J0101_1010))
    Ttst(Logtest(J0101_1010,J0011     ))

    Ftst(Logtest(I0011     ,I1100     ))
    Ftst(Logtest(I1100     ,I0001     ))
    Ftst(Logtest(I0011_1100,I1000_0001))
    Ftst(Logtest(I0011     ,I1100_0011))
    Ftst(Logtest(I0101_1010,I0010     ))
    Ftst(Logtest(I0011     ,J1000     ))
    Ftst(Logtest(I1100     ,J0011     ))
    Ftst(Logtest(I0011_1100,J1000_0001))
    Ftst(Logtest(I0011     ,J1100_0011))
    Ftst(Logtest(J0011     ,I0100     ))
    Ftst(Logtest(J1100     ,I0010     ))
    Ftst(Logtest(J0011_1100,I0100_0010))
    Ftst(Logtest(J0101_1010,I1010     ))
    Ftst(Logtest(EE,I0))
    Ttst(Logtest(EE,J1))
    Ftst(Logtest(I0,EE))
    Ttst(Logtest(J1,EE))
    Ftst(Logtest(EE,J1010_0101))
    Ttst(Logtest(EE,I0011_1100))
    Ftst(Logtest(J1010_0101,EE))
    Ttst(Logtest(I0011_1100,EE))

    Xtst(Lt(TT,TT))
    Xtst(Lt(TT,FF))
    Xtst(Lt(TT,I0))
    Xtst(Lt(TT,P0))
    Xtst(Lt(TT,M0))
    Xtst(Lt(TT,B0))
    Xtst(Lt(FF,TT))
    Xtst(Lt(FF,FF))
    Xtst(Lt(FF,I0))
    Xtst(Lt(FF,P0))
    Xtst(Lt(FF,M0))
    Xtst(Lt(FF,B0))
    Xtst(Lt(I0,TT))
    Xtst(Lt(I0,FF))
    Xtst(Lt(I0,P0))
    Xtst(Lt(I0,M0))
    Xtst(Lt(I0,B0))
    Xtst(Lt(P0,TT))
    Xtst(Lt(P0,FF))
    Xtst(Lt(P0,I0))
    Xtst(Lt(P0,P0))
    Xtst(Lt(P0,M0))
    Xtst(Lt(P0,B0))
    Xtst(Lt(M0,TT))
    Xtst(Lt(M0,FF))
    Xtst(Lt(M0,I0))
    Xtst(Lt(M0,P0))
    Xtst(Lt(M0,M0))
    Xtst(Lt(M0,B0))
    Xtst(Lt(B0,TT))
    Xtst(Lt(B0,FF))
    Xtst(Lt(B0,I0))
    Xtst(Lt(B0,P0))
    Xtst(Lt(B0,M0))
    Xtst(Lt(B0,B0))
    Ftst(Lt(I0011     ,I0011     ))
    Ttst(Lt(I0011     ,I0101     ))
    Ftst(Lt(I1100     ,I1010     ))
    Ftst(Lt(I0011_1100,I0011_1100))
    Ftst(Lt(I0011_1100,I0101_1010))
    Ttst(Lt(I1100_0011,I1010_0101))
    Ttst(Lt(I0011     ,I0101_1010))
    Ftst(Lt(I0101_1010,I0011     ))
    Ftst(Lt(I0011     ,J0101     ))
    Ftst(Lt(I1100     ,J1010     ))
    Ftst(Lt(I0011_1100,J0101_1010))
    Ftst(Lt(I0011     ,J0101_1010))
    Ftst(Lt(I0101_1010,J0011     ))
    Ttst(Lt(J0011     ,I0101     ))
    Ttst(Lt(J1100     ,I1010     ))
    Ttst(Lt(J0011_1100,I0101_1010))
    Ttst(Lt(J0011     ,I0101_1010))
    Ttst(Lt(J0101_1010,I0011     ))
    Ftst(Lt(J0011     ,J0011     ))
    Ttst(Lt(J0011     ,J0101     ))
    Ftst(Lt(J1100     ,J1010     ))
    Ftst(Lt(J0011_1100,J0011_1100))
    Ftst(Lt(J0011_1100,J0101_1010))
    Ttst(Lt(J1100_0011,J1010_0101))
    Ftst(Lt(J0011     ,J0101_1010))
    Ttst(Lt(J0101_1010,J0011     ))

    Xtst(Le(TT,TT))
    Xtst(Le(TT,FF))
    Xtst(Le(TT,I0))
    Xtst(Le(TT,P0))
    Xtst(Le(TT,M0))
    Xtst(Le(TT,B0))
    Xtst(Le(FF,TT))
    Xtst(Le(FF,FF))
    Xtst(Le(FF,I0))
    Xtst(Le(FF,P0))
    Xtst(Le(FF,M0))
    Xtst(Le(FF,B0))
    Xtst(Le(I0,TT))
    Xtst(Le(I0,FF))
    Xtst(Le(I0,P0))
    Xtst(Le(I0,M0))
    Xtst(Le(I0,B0))
    Xtst(Le(P0,TT))
    Xtst(Le(P0,FF))
    Xtst(Le(P0,I0))
    Xtst(Le(P0,P0))
    Xtst(Le(P0,M0))
    Xtst(Le(P0,B0))
    Xtst(Le(M0,TT))
    Xtst(Le(M0,FF))
    Xtst(Le(M0,I0))
    Xtst(Le(M0,P0))
    Xtst(Le(M0,M0))
    Xtst(Le(M0,B0))
    Xtst(Le(B0,TT))
    Xtst(Le(B0,FF))
    Xtst(Le(B0,I0))
    Xtst(Le(B0,P0))
    Xtst(Le(B0,M0))
    Xtst(Le(B0,B0))
    Ttst(Le(I0011     ,I0011     ))
    Ttst(Le(I0011     ,I0101     ))
    Ftst(Le(I1100     ,I1010     ))
    Ttst(Le(I0011_1100,I0011_1100))
    Ftst(Le(I0011_1100,I0101_1010))
    Ttst(Le(I1100_0011,I1010_0101))
    Ttst(Le(I0011     ,I0101_1010))
    Ftst(Le(I0101_1010,I0011     ))
    Ftst(Le(I0011     ,J0101     ))
    Ftst(Le(I1100     ,J1010     ))
    Ftst(Le(I0011_1100,J0101_1010))
    Ftst(Le(I0011     ,J0101_1010))
    Ftst(Le(I0101_1010,J0011     ))
    Ttst(Le(J0011     ,I0101     ))
    Ttst(Le(J1100     ,I1010     ))
    Ttst(Le(J0011_1100,I0101_1010))
    Ttst(Le(J0011     ,I0101_1010))
    Ttst(Le(J0101_1010,I0011     ))
    Ttst(Le(J0011     ,J0011     ))
    Ttst(Le(J0011     ,J0101     ))
    Ftst(Le(J1100     ,J1010     ))
    Ttst(Le(J0011_1100,J0011_1100))
    Ftst(Le(J0011_1100,J0101_1010))
    Ttst(Le(J1100_0011,J1010_0101))
    Ftst(Le(J0011     ,J0101_1010))
    Ttst(Le(J0101_1010,J0011     ))

    Xtst(Gt(TT,TT))
    Xtst(Gt(TT,FF))
    Xtst(Gt(TT,I0))
    Xtst(Gt(TT,P0))
    Xtst(Gt(TT,M0))
    Xtst(Gt(TT,B0))
    Xtst(Gt(FF,TT))
    Xtst(Gt(FF,FF))
    Xtst(Gt(FF,I0))
    Xtst(Gt(FF,P0))
    Xtst(Gt(FF,M0))
    Xtst(Gt(FF,B0))
    Xtst(Gt(I0,TT))
    Xtst(Gt(I0,FF))
    Xtst(Gt(I0,P0))
    Xtst(Gt(I0,M0))
    Xtst(Gt(I0,B0))
    Xtst(Gt(P0,TT))
    Xtst(Gt(P0,FF))
    Xtst(Gt(P0,I0))
    Xtst(Gt(P0,P0))
    Xtst(Gt(P0,M0))
    Xtst(Gt(P0,B0))
    Xtst(Gt(M0,TT))
    Xtst(Gt(M0,FF))
    Xtst(Gt(M0,I0))
    Xtst(Gt(M0,P0))
    Xtst(Gt(M0,M0))
    Xtst(Gt(M0,B0))
    Xtst(Gt(B0,TT))
    Xtst(Gt(B0,FF))
    Xtst(Gt(B0,I0))
    Xtst(Gt(B0,P0))
    Xtst(Gt(B0,M0))
    Xtst(Gt(B0,B0))
    Ftst(Gt(I0011     ,I0011     ))
    Ftst(Gt(I0011     ,I0101     ))
    Ttst(Gt(I1100     ,I1010     ))
    Ftst(Gt(I0011_1100,I0011_1100))
    Ttst(Gt(I0011_1100,I0101_1010))
    Ftst(Gt(I1100_0011,I1010_0101))
    Ftst(Gt(I0011     ,I0101_1010))
    Ttst(Gt(I0101_1010,I0011     ))
    Ttst(Gt(I0011     ,J0101     ))
    Ttst(Gt(I1100     ,J1010     ))
    Ttst(Gt(I0011_1100,J0101_1010))
    Ttst(Gt(I0011     ,J0101_1010))
    Ttst(Gt(I0101_1010,J0011     ))
    Ftst(Gt(J0011     ,I0101     ))
    Ftst(Gt(J1100     ,I1010     ))
    Ftst(Gt(J0011_1100,I0101_1010))
    Ftst(Gt(J0011     ,I0101_1010))
    Ftst(Gt(J0101_1010,I0011     ))
    Ftst(Gt(J0011     ,J0011     ))
    Ftst(Gt(J0011     ,J0101     ))
    Ttst(Gt(J1100     ,J1010     ))
    Ftst(Gt(J0011_1100,J0011_1100))
    Ttst(Gt(J0011_1100,J0101_1010))
    Ftst(Gt(J1100_0011,J1010_0101))
    Ttst(Gt(J0011     ,J0101_1010))
    Ftst(Gt(J0101_1010,J0011     ))

    Xtst(Ge(TT,TT))
    Xtst(Ge(TT,FF))
    Xtst(Ge(TT,I0))
    Xtst(Ge(TT,P0))
    Xtst(Ge(TT,M0))
    Xtst(Ge(TT,B0))
    Xtst(Ge(FF,TT))
    Xtst(Ge(FF,FF))
    Xtst(Ge(FF,I0))
    Xtst(Ge(FF,P0))
    Xtst(Ge(FF,M0))
    Xtst(Ge(FF,B0))
    Xtst(Ge(I0,TT))
    Xtst(Ge(I0,FF))
    Xtst(Ge(I0,P0))
    Xtst(Ge(I0,M0))
    Xtst(Ge(I0,B0))
    Xtst(Ge(P0,TT))
    Xtst(Ge(P0,FF))
    Xtst(Ge(P0,I0))
    Xtst(Ge(P0,P0))
    Xtst(Ge(P0,M0))
    Xtst(Ge(P0,B0))
    Xtst(Ge(M0,TT))
    Xtst(Ge(M0,FF))
    Xtst(Ge(M0,I0))
    Xtst(Ge(M0,P0))
    Xtst(Ge(M0,M0))
    Xtst(Ge(M0,B0))
    Xtst(Ge(B0,TT))
    Xtst(Ge(B0,FF))
    Xtst(Ge(B0,I0))
    Xtst(Ge(B0,P0))
    Xtst(Ge(B0,M0))
    Xtst(Ge(B0,B0))
    Ttst(Ge(I0011     ,I0011     ))
    Ftst(Ge(I0011     ,I0101     ))
    Ttst(Ge(I1100     ,I1010     ))
    Ttst(Ge(I0011_1100,I0011_1100))
    Ttst(Ge(I0011_1100,I0101_1010))
    Ftst(Ge(I1100_0011,I1010_0101))
    Ftst(Ge(I0011     ,I0101_1010))
    Ttst(Ge(I0101_1010,I0011     ))
    Ttst(Ge(I0011     ,J0101     ))
    Ttst(Ge(I1100     ,J1010     ))
    Ttst(Ge(I0011_1100,J0101_1010))
    Ttst(Ge(I0011     ,J0101_1010))
    Ttst(Ge(I0101_1010,J0011     ))
    Ftst(Ge(J0011     ,I0101     ))
    Ftst(Ge(J1100     ,I1010     ))
    Ftst(Ge(J0011_1100,I0101_1010))
    Ftst(Ge(J0011     ,I0101_1010))
    Ftst(Ge(J0101_1010,I0011     ))
    Ttst(Ge(J0011     ,J0011     ))
    Ftst(Ge(J0011     ,J0101     ))
    Ttst(Ge(J1100     ,J1010     ))
    Ttst(Ge(J0011_1100,J0011_1100))
    Ttst(Ge(J0011_1100,J0101_1010))
    Ftst(Ge(J1100_0011,J1010_0101))
    Ttst(Ge(J0011     ,J0101_1010))
    Ftst(Ge(J0101_1010,J0011     ))

    Etst(False,F)

    Etst(Base,N(10))

    Xtst(Evenp(TT))
    Xtst(Evenp(FF))
    Xtst(Evenp(P0))
    Xtst(Evenp(M0))
    Xtst(Evenp(B0))
    Ttst(Evenp(Z))
    Ftst(Evenp(M))
    Ftst(Evenp(I0011))
    Ttst(Evenp(J1010_0101))

    Xtst(Half(TT))
    Xtst(Half(FF))
    Xtst(Half(P0))
    Xtst(Half(M0))
    Xtst(Half(B0))
    Etst(Half(Z),Z)
    Etst(Half(M),M)
    Etst(Half(I[1]),I[0])
    Etst(Half(I[2]),I[1])
    Etst(Half(I[3]),I[1])
    Etst(Half(I[4]),I[2])
    Etst(Half(I[5]),I[2])
    Etst(Half(I[6]),I[3])
    Etst(Half(JN(-2)),M)
    Etst(Half(JN(-3)),JN(-2))
    Etst(Half(JN(-4)),JN(-2))
    Etst(Half(JN(-5)),JN(-3))
    Etst(Half(JN(-6)),JN(-3))
    Etst(Half(JN(-7)),JN(-4))

    Ttst(Eq(TT,TT))
    Ftst(Eq(TT,FF))
    Ftst(Eq(TT,I0))
    Ftst(Eq(TT,EE))
    Ftst(Eq(TT,P0))
    Ftst(Eq(TT,M0))
    Ftst(Eq(TT,B0))
    Ftst(Eq(FF,TT))
    Ttst(Eq(FF,FF))
    Ftst(Eq(FF,I0))
    Ftst(Eq(FF,EE))
    Ftst(Eq(FF,P0))
    Ftst(Eq(FF,M0))
    Ftst(Eq(FF,B0))
    Ftst(Eq(I0,TT))
    Ftst(Eq(I0,FF))
    Ttst(Eq(I0,I0))
    Ftst(Eq(I0,EE))
    Ftst(Eq(I0,P0))
    Ftst(Eq(I0,M0))
    Ftst(Eq(I0,B0))
    Ftst(Eq(EE,TT))
    Ftst(Eq(EE,FF))
    Ftst(Eq(EE,I0))
    Ttst(Eq(EE,EE))
    Ftst(Eq(EE,P0))
    Ftst(Eq(EE,M0))
    Ftst(Eq(EE,B0))
    Ftst(Eq(P0,TT))
    Ftst(Eq(P0,FF))
    Ftst(Eq(P0,I0))
    Ftst(Eq(P0,EE))
    Ttst(Eq(P0,P0))
    Ftst(Eq(P0,M0))
    Ftst(Eq(P0,B0))
    Ftst(Eq(M0,TT))
    Ftst(Eq(M0,FF))
    Ftst(Eq(M0,I0))
    Ftst(Eq(M0,EE))
    Ftst(Eq(M0,P0))
    Ttst(Eq(M0,M0))
    Ftst(Eq(M0,B0))
    Ftst(Eq(B0,TT))
    Ftst(Eq(B0,FF))
    Ftst(Eq(B0,I0))
    Ftst(Eq(B0,EE))
    Ftst(Eq(B0,P0))
    Ftst(Eq(B0,M0))
    Ttst(Eq(B0,B0))

    Ttst(Eq(I12 ,I12 ))
    Ftst(Eq(I12 ,I123))
    Ftst(Eq(I12 ,I923))
    Ftst(Eq(I12 ,I129))
    Ftst(Eq(I123,I12 ))
    Ttst(Eq(I123,I123))
    Ftst(Eq(I123,I923))
    Ftst(Eq(I123,I129))
    Ftst(Eq(I923,I12 ))
    Ftst(Eq(I923,I123))
    Ttst(Eq(I923,I923))
    Ftst(Eq(I923,I129))
    Ftst(Eq(I129,I12 ))
    Ftst(Eq(I129,I123))
    Ftst(Eq(I129,I923))
    Ttst(Eq(I129,I129))

    Ttst(Eq(J12 ,J12 ))
    Ftst(Eq(J12 ,J123))
    Ftst(Eq(J12 ,J923))
    Ftst(Eq(J12 ,J129))
    Ftst(Eq(J123,J12 ))
    Ttst(Eq(J123,J123))
    Ftst(Eq(J123,J923))
    Ftst(Eq(J123,J129))
    Ftst(Eq(J923,J12 ))
    Ftst(Eq(J923,J123))
    Ttst(Eq(J923,J923))
    Ftst(Eq(J923,J129))
    Ftst(Eq(J129,J12 ))
    Ftst(Eq(J129,J123))
    Ftst(Eq(J129,J923))
    Ttst(Eq(J129,J129))

    Ttst(Eq(EE,I[1]))

    Ttst(Eq(P0,P0))
    Ftst(Eq(P0,P1))
    Ftst(Eq(P0,P2))
    Ftst(Eq(P0,P3))
    Ftst(Eq(P1,P0))
    Ttst(Eq(P1,P1))
    Ftst(Eq(P1,P2))
    Ftst(Eq(P1,P3))
    Ftst(Eq(P2,P0))
    Ftst(Eq(P2,P1))
    Ttst(Eq(P2,P2))
    Ftst(Eq(P2,P3))
    Ftst(Eq(P3,P0))
    Ftst(Eq(P3,P1))
    Ftst(Eq(P3,P2))
    Ttst(Eq(P3,P3))

    Ttst(Eq(B0,B0))
    Ftst(Eq(B0,B1))
    Ftst(Eq(B0,B2))
    Ftst(Eq(B0,B3))
    Ftst(Eq(B1,B0))
    Ttst(Eq(B1,B1))
    Ftst(Eq(B1,B2))
    Ftst(Eq(B1,B3))
    Ftst(Eq(B2,B0))
    Ftst(Eq(B2,B1))
    Ttst(Eq(B2,B2))
    Ftst(Eq(B2,B3))
    Ftst(Eq(B3,B0))
    Ftst(Eq(B3,B1))
    Ftst(Eq(B3,B2))
    Ttst(Eq(B3,B3))

    ftst(Root(fct_if))
    ftst(Root(Apply(fct_if,MT)))
    ftst(Root(Apply(Apply(fct_if,MT),MT)))
    ttst(Root(Apply(Apply(Apply(fct_if,MT),MT),MT)))
    ttst(Root(Apply(Apply(Apply(fct_if,MT),MT),MF)))
    ftst(Root(Apply(Apply(Apply(fct_if,MT),MF),MT)))
    ftst(Root(Apply(Apply(Apply(fct_if,MT),MF),MF)))
    ttst(Root(Apply(Apply(Apply(fct_if,MF),MT),MT)))
    ftst(Root(Apply(Apply(Apply(fct_if,MF),MT),MF)))
    ttst(Root(Apply(Apply(Apply(fct_if,MF),MF),MT)))
    ftst(Root(Apply(Apply(Apply(fct_if,MF),MF),MF)))

    Xtst(Sl2rack(FF))
    Xtst(Sl2rack(I0))
    Xtst(Sl2rack(J1))
    Xtst(Sl2rack(EE))
    Xtst(Sl2rack(M0))
    Xtst(Sl2rack(B0))
    Xtst(Sl2rack(PAIR(TT,TT)))
    Xtst(Sl2rack(PAIR(FF,TT)))
    Xtst(Sl2rack(PAIR(I0,TT)))
    Xtst(Sl2rack(PAIR(J1,TT)))
    Xtst(Sl2rack(PAIR(EE,TT)))
    Xtst(Sl2rack(PAIR(V(ab),TT)))
    Xtst(Sl2rack(PAIR(P0,TT)))
    Xtst(Sl2rack(PAIR(M0,TT)))
    Xtst(Sl2rack(PAIR(B0,TT)))
    Xtst(Sl2rack(PAIR(bytevect[0],FF)))
    Xtst(Sl2rack(PAIR(bytevect[0],I0)))
    Xtst(Sl2rack(PAIR(bytevect[0],J1)))
    Xtst(Sl2rack(PAIR(bytevect[0],M0)))
    Xtst(Sl2rack(PAIR(bytevect[0],B0)))

    Etst(T,
      Sl2rack(
        PAIR(bytevect[3],T)))
    Etst(T,
      Sl2rack(
        PAIR(bytevect[3],FF)))
    Etst(PAIR(T,T),
      Sl2rack(
        PAIR(bytevect[1],
        PAIR(bytevect[1],
        PAIR(bytevect[4],T)))))
    Etst(N(7),
      Sl2rack(
        PAIR(bytevect[0],
        PAIR(bytevect[7],
        PAIR(bytevect[4],T)))))
    Etst(V(abc),
      Sl2rack(
        PAIR(bytevect[2],
        PAIR(bytevect[3],
        PAIR(bytevect[97],
        PAIR(bytevect[98],
        PAIR(bytevect[99],
        PAIR(bytevect[4],T))))))))
    Etst(PAIR(N(6),V(abc)),
      Sl2rack(
        PAIR(bytevect[0],
        PAIR(bytevect[6],
        PAIR(bytevect[2],
        PAIR(bytevect[3],
        PAIR(bytevect[97],
        PAIR(bytevect[98],
        PAIR(bytevect[99],
        PAIR(bytevect[3],
        PAIR(bytevect[4],
        PAIR(bytevect[6],T))))))))))))

    Etst(Rack2sl(T),
      PAIR(bytevect[3],T))
    Etst(Rack2sl(PAIR(T,T)),
      PAIR(bytevect[1],
      PAIR(bytevect[1],
      PAIR(bytevect[4],T))))
    Etst(Rack2sl(N(7)),
      PAIR(bytevect[0],
      PAIR(bytevect[7],
      PAIR(bytevect[4],T))))
    Etst(Rack2sl(V(abc)),
      PAIR(bytevect[2],
      PAIR(bytevect[3],
      PAIR(bytevect[97],
      PAIR(bytevect[98],
      PAIR(bytevect[99],
      PAIR(bytevect[4],T)))))))
    Etst(Rack2sl(PAIR(N(6),V(abc))),
      PAIR(bytevect[0],
      PAIR(bytevect[6],
      PAIR(bytevect[2],
      PAIR(bytevect[3],
      PAIR(bytevect[97],
      PAIR(bytevect[98],
      PAIR(bytevect[99],
      PAIR(bytevect[3],
      PAIR(bytevect[4],
      PAIR(bytevect[6],T)))))))))))
    Etst(Rack2sl(PAIR(N(7),N(7))),
      PAIR(bytevect[0],
      PAIR(bytevect[7],
      PAIR(bytevect[3],
      PAIR(bytevect[3],
      PAIR(bytevect[5],T))))))
    Etst(Rack2sl(PAIR(PAIR(T,T),PAIR(T,T))),
      PAIR(bytevect[1],
      PAIR(bytevect[1],
      PAIR(bytevect[3],
      PAIR(bytevect[3],
      PAIR(bytevect[5],T))))))
    Xtst(Rack2sl(PAIR(PAIR(T,T),PAIR(T,N(-1)))))
    Xtst(Rack2sl(PAIR(PAIR(T,T),PAIR(T,F))))

    Etst(Ripemd(TT),
      PAIR(bytevect[0x9C],
      PAIR(bytevect[0x11],
      PAIR(bytevect[0x85],
      PAIR(bytevect[0xA5],
      PAIR(bytevect[0xC5],
      PAIR(bytevect[0xE9],
      PAIR(bytevect[0xFC],
      PAIR(bytevect[0x54],
      PAIR(bytevect[0x61],
      PAIR(bytevect[0x28],
      PAIR(bytevect[0x08],
      PAIR(bytevect[0x97],
      PAIR(bytevect[0x7E],
      PAIR(bytevect[0xE8],
      PAIR(bytevect[0xF5],
      PAIR(bytevect[0x48],
      PAIR(bytevect[0xB2],
      PAIR(bytevect[0x25],
      PAIR(bytevect[0x8D],
      PAIR(bytevect[0x31],T)))))))))))))))))))))
    Etst(Ripemd(PAIR(bytevect[97],TT)),
      PAIR(bytevect[0x0B],
      PAIR(bytevect[0xDC],
      PAIR(bytevect[0x9D],
      PAIR(bytevect[0x2D],
      PAIR(bytevect[0x25],
      PAIR(bytevect[0x6B],
      PAIR(bytevect[0x3E],
      PAIR(bytevect[0xE9],
      PAIR(bytevect[0xDA],
      PAIR(bytevect[0xAE],
      PAIR(bytevect[0x34],
      PAIR(bytevect[0x7B],
      PAIR(bytevect[0xE6],
      PAIR(bytevect[0xF4],
      PAIR(bytevect[0xDC],
      PAIR(bytevect[0x83],
      PAIR(bytevect[0x5A],
      PAIR(bytevect[0x46],
      PAIR(bytevect[0x7F],
      PAIR(bytevect[0xFE],T)))))))))))))))))))))
    Etst(
      Ripemd(
        PAIR(V(abcdefghijklmnopqrstuvwxyz),V(ABCDEFGHIJKLMNOPQRSTUVWXYZ123))),
      PAIR(bytevect[0x71],
      PAIR(bytevect[0xE0],
      PAIR(bytevect[0xB6],
      PAIR(bytevect[0x23],
      PAIR(bytevect[0x22],
      PAIR(bytevect[0xD3],
      PAIR(bytevect[0xA0],
      PAIR(bytevect[0x31],
      PAIR(bytevect[0xC0],
      PAIR(bytevect[0xFB],
      PAIR(bytevect[0x62],
      PAIR(bytevect[0x1B],
      PAIR(bytevect[0xE4],
      PAIR(bytevect[0xD9],
      PAIR(bytevect[0x07],
      PAIR(bytevect[0x73],
      PAIR(bytevect[0xB7],
      PAIR(bytevect[0xE6],
      PAIR(bytevect[0x6B],
      PAIR(bytevect[0xB2],T)))))))))))))))))))))
    Etst(Ripemd(PAIR(PAIR(V(a),N(-1)),PAIR(B0,V(bc)))),
      PAIR(bytevect[0x8E],
      PAIR(bytevect[0xB2],
      PAIR(bytevect[0x08],
      PAIR(bytevect[0xF7],
      PAIR(bytevect[0xE0],
      PAIR(bytevect[0x5D],
      PAIR(bytevect[0x98],
      PAIR(bytevect[0x7A],
      PAIR(bytevect[0x9B],
      PAIR(bytevect[0x04],
      PAIR(bytevect[0x4A],
      PAIR(bytevect[0x8E],
      PAIR(bytevect[0x98],
      PAIR(bytevect[0xC6],
      PAIR(bytevect[0xB0],
      PAIR(bytevect[0x87],
      PAIR(bytevect[0xF1],
      PAIR(bytevect[0x5A],
      PAIR(bytevect[0x0B],
      PAIR(bytevect[0xFC],T)))))))))))))))))))))

/*
---------------------------------------------
Test of eager functions
---------------------------------------------
*/

    Etst(Emptylist,T)

    Etst(Zeroth(PAIR(N(1),PAIR(N(2),PAIR(N(3),T)))),N(1))
    Etst(First (PAIR(N(1),PAIR(N(2),PAIR(N(3),T)))),N(2))
    Etst(Second(PAIR(N(1),PAIR(N(2),PAIR(N(3),T)))),N(3))
    Etst(Zeroth(X1),X1)
    Etst(First (X1),X1)
    Etst(Second(X1),X1)
    Etst(Revappend(T,PAIR(N(3),PAIR(N(4),T))),PAIR(N(3),PAIR(N(4),T)))
    Etst(Revappend(PAIR(N(2),T),PAIR(N(3),PAIR(N(4),T))),
      PAIR(N(2),PAIR(N(3),PAIR(N(4),T))))
    Etst(Revappend(PAIR(N(2),PAIR(N(1),T)),PAIR(N(3),PAIR(N(4),T))),
      PAIR(N(1),PAIR(N(2),PAIR(N(3),PAIR(N(4),T)))))
    Etst(Revappend(X1,PAIR(N(3),PAIR(N(4),T))),X1)
    Etst(Revappend(PAIR(N(2),PAIR(N(1),T)),X2),X2)
    Etst(Revappend(X1,X2),X1)
    Etst(Reverse(PAIR(N(1),PAIR(N(2),PAIR(N(3),T)))),
      PAIR(N(3),PAIR(N(2),PAIR(N(1),T))))

/*
---------------------------------------------
Test of call from lazy to eager function
---------------------------------------------
*/

    Xtst(Factorial(TT))
    Xtst(Factorial(FF))
    Xtst(Factorial(P0))
    Xtst(Factorial(M0))
    Xtst(Factorial(B0))
    Etst(Factorial(N(0)),N(1))
    Etst(Factorial(N(1)),N(1))
    Etst(Factorial(N(2)),N(2))
    Etst(Factorial(N(3)),N(6))
    Etst(Factorial(N(4)),N(24))
    Etst(Factorial(N(10)),N(3628800))
    Etst(Factorial(N(100)),
N(93326215443944152681699238856266700490715968264381621468592963895217599993229915608941463976156518286253697920827223758251185210916864000000000000000000000000))

/*
---------------------------------------------
Test of handlers
---------------------------------------------
*/

    Etst(Quit(N(3)),PAIR(PAIR(N(0),PAIR(V(quit),T)),PAIR(N(3),T)))
    Etst(Write(V(abc)),PAIR(PAIR(N(0),PAIR(V(write),T)),PAIR(V(abc),T)))
    Etst(Readreq,PAIR(PAIR(N(0),PAIR(V(read),T)),T))
    Etst(Exec(M2,M3),PAIR(PAIR(N(0),PAIR(V(exec),T)),PAIR(M2,PAIR(M3,T))))
    Etst(Extend(V(a),V(bc)),
      PAIR(PAIR(N(0),PAIR(V(extend),T)),PAIR(V(a),PAIR(V(bc),T))))
    Etst(Boot(Z0,Z1,Z2,Z3),
      PAIR(PAIR(N(0),PAIR(V(boot),T)),PAIR(Z0,PAIR(Z1,PAIR(Z2,PAIR(Z3,T))))))
    Etst(Read(V(abc)),PAIR(PAIR(N(0),PAIR(V(read),T)),PAIR(V(abc),T)))
    Etst(Int(N(3),T),PAIR(PAIR(N(0),PAIR(V(exec),T)),PAIR(N(3),PAIR(T,T))))
    Etst(Extended(N(3)),PAIR(PAIR(N(0),PAIR(V(extend),T)),PAIR(N(3),T)))

    etst(code_hello,M0)
    etst(code_echo,M0)
    etst(code_eecho,M0)
    etst(tail(Echo1(T)),PAIR(Exec(T,M0),T))
    etst(Echo1(T),PAIR(map2rnf(Readreq),PAIR(Exec(T,M0),T)))
    etst(Echo1(PAIR(Read(V(q)),T)),
      PAIR(Write(PAIR(T,N(266))),PAIR(Quit(N(0)),T)))
    etst(Echo1(PAIR(Read(V(r)),T)),
      PAIR(Write(V(r)),PAIR(map2rnf(Readreq),PAIR(Exec(T,M0),T))))
    etst(Eecho1(T),PAIR(map2rnf(Readreq),PAIR(Exec(T,M0),T)))
    etst(Eecho1(PAIR(Read(V(q)),T)),
      PAIR(Write(PAIR(T,N(266))),PAIR(Quit(N(0)),T)))
    etst(Eecho1(PAIR(Read(V(r)),T)),
      PAIR(Write(PAIR(V(r),V(r))),PAIR(map2rnf(Readreq),PAIR(Exec(T,M0),T))))
    etst(untag(APPLY(untag1(Hello),mapcons(T))),
      PAIR(Write(PAIR(V(Hello World),N(266))),T))
    etst(untag(APPLY(untag(code_hello),mapcons(T))),
      PAIR(Write(PAIR(V(Hello World),N(266))),T))
    etst(untag(APPLY(untag(code_echo),mapcons(T))),
      PAIR(map2rnf(Readreq),PAIR(Exec(T,M0),T)))
    etst(untag(APPLY(untag(code_echo),mapcons(PAIR(Read(V(q)),T)))),
      PAIR(Write(PAIR(T,N(266))),PAIR(Quit(N(0)),T)))
    etst(untag(APPLY(untag(code_echo),mapcons(PAIR(Read(V(r)),T)))),
      PAIR(Write(V(r)),PAIR(map2rnf(Readreq),PAIR(Exec(T,M0),T))))
    etst(untag(APPLY(untag(code_eecho),mapcons(T))),
      PAIR(map2rnf(Readreq),PAIR(Exec(T,M0),T)))
    etst(untag(APPLY(untag(code_eecho),mapcons(PAIR(Read(V(q)),T)))),
      PAIR(Write(PAIR(T,N(266))),PAIR(Quit(N(0)),T)))
    etst(untag(APPLY(untag(code_eecho),mapcons(PAIR(Read(V(r)),T)))),
      PAIR(Write(PAIR(V(r),V(r))),PAIR(map2rnf(Readreq),PAIR(Exec(T,M0),T))))

/*
---------------------------------------------
Test of compilation
---------------------------------------------
*/

/*  The optimized Y-combinator is introduced for the sake of compilation
    so we test it here. Note that UY is a defined Y-combinator whereas
    YY is the optimized one. */

    ntst(map2rnf(map_apply(UY,UT)))
    ntst(map2rnf(map_apply(UY,UF)))
    ptst(map2rnf(UYK))
    ptst(map2rnf(map_apply(UYK,UT)))
    ptst(map2rnf(map_apply(UYK,UF)))
    ptst(map2rnf(map_apply(map_apply(UYK,UT),UT)))
    ptst(map2rnf(map_apply(map_apply(UYK,UF),UF)))
    ptst(map2rnf(UYKI))
    ntst(map2rnf(map_apply(UYKI,UT)))
    ptst(map2rnf(map_apply(UYKI,UF)))

    test(ROOT(YY)==MTAG_FIX)
    ntst(map2rnf(map_apply(YY,UT)))
    ntst(map2rnf(map_apply(YY,UF)))
    ptst(map2rnf(map_apply(YY,UK)))
    ptst(map2rnf(map_apply(map_apply(YY,UK),UT)))
    ptst(map2rnf(map_apply(map_apply(YY,UK),UF)))
    ptst(map2rnf(map_apply(map_apply(map_apply(YY,UK),UT),UT)))
    ptst(map2rnf(map_apply(map_apply(map_apply(YY,UK),UF),UF)))
    ptst(map2rnf(map_apply(YY,UKI)))
    ntst(map2rnf(map_apply(map_apply(YY,UKI),UT)))
    ptst(map2rnf(map_apply(map_apply(YY,UKI),UF)))
    test(ROOT(fixmap1)==MTAG_PAIR)
    Etst(map_head(fixmap1),N(2))
    test(ROOT(map_tail(fixmap1))==MTAG_PAIR)
    Etst(map_head(map_tail(fixmap1)),N(2))
    test(ROOT(map_tail(map_tail(fixmap1)))==MTAG_PAIR)
    Etst(map_head(map_tail(map_tail(fixmap1))),N(2))
    test(ROOT(fixmap2)==MTAG_PAIR)
    Etst(map_head(fixmap2),V(print))
    test(ROOT(map_tail(fixmap2))==MTAG_PAIR)
    Etst(map_head(map_tail(fixmap2)),V(print))
    test(ROOT(map_tail(map_tail(fixmap2)))==MTAG_PAIR)
    Etst(map_head(map_tail(map_tail(fixmap2))),V(print))

/*  Test the eagerly constructed part of the output from the compiler */

    etst(mget1(basecache,Z),baseref)
    etst(mget1(basecacheb,Z),baseref)
    etst(mget1(baserack,const_bib),PAIR(baseref,T))
    etst(mget1(baserackb,const_bib),PAIR(baseref,T))
    etst(mget1(baserack,const_bib      ),mget1(baserackb,const_bib      ))
    etst(basediagnose,maptag(T))
    etst(basediagnoseb,maptag(T))
    etst(basediagnose,basediagnoseb)
    etst(mget1(basecodea,name2idx("testfunc1 ( \" )")),maptag(T))
    etst(mget1(basecodeb,name2idx("testfunc1 ( \" )")),maptag(T))
    etst(mget1(basecodea,name2idx("testfunc1 ( \" )")),
         mget1(basecodeb,name2idx("testfunc1 ( \" )")))
    etst(basecodea,basecodeb)
/*  etst(basecache,basecacheb)*/

/*  Test the lazily constructed part of the output from the compiler */

    Etst(Truea,T)
    Etst(Trueb,T)
    Etst(Falsea,F)
    Etst(Falseb,F)
    Etst(Digitenda,Z)
    Etst(Digitendb,Z)
    Etst(Basea,N(10))
    Etst(Baseb,N(10))
    Etst(Ifa(Truea,Digitenda,Basea),N(0))
    Etst(Ifb(Trueb,Digitendb,Baseb),N(0))
    Etst(LazyPaira(Truea,Truea),F)
    Etst(LazyPairb(Trueb,Trueb),F)
    Etst(Heada(LazyPaira(Digitenda,Basea)),N(0))
    Etst(Headb(LazyPairb(Digitendb,Baseb)),N(0))
    Etst(Taila(LazyPaira(Digitenda,Basea)),N(10))
    Etst(Tailb(LazyPairb(Digitendb,Baseb)),N(10))
    Etst(Equala(Digitenda,Digitenda),T)
    Etst(Equalb(Digitendb,Digitendb),T)
    Etst(Equala(Digitenda,Basea),F)
    Etst(Equalb(Digitendb,Baseb),F)
    Etst(Equala(Basea,Digitenda),F)
    Etst(Equalb(Baseb,Digitendb),F)
    Etst(Equala(Basea,Basea),T)
    Etst(Equalb(Baseb,Baseb),T)

/*  testfunc1 to testfunc8 tests various peculiarities */

    Etst(Testfunc1a(N(2)),N(5))
    Etst(Testfunc1b(N(2)),N(5))
    Etst(Testfunc2a(N(2),N(3)),N(2))
    Etst(Testfunc2b(N(2),N(3)),N(2))
    Etst(Testfunc5a(N(2)),T)
    Etst(Testfunc5b(N(2)),T)
    Etst(Testfunc6a(N(3)),N(2))
    Etst(Testfunc6b(N(3)),N(2))
    Etst(Testfunc7a(N(2)),N(5))
    Etst(Testfunc7b(N(2)),N(5))
    Etst(Testfunc8a(N(2),N(3)),N(5))
    Etst(Testfunc8b(N(2),N(3)),N(5))

/*
---------------------------------------------
Test of verification and macro expansion
---------------------------------------------
*/

    ttst(untag(basediagnose))
    test(ROOT(basediagnoseb)==TAG_MAP);
    test(ROOT(HEAD(basediagnoseb))==MTAG_CLOSURE);
    ttst(untag(basediagnoseb))
    test(ROOT(basediagnoseb)==TAG_MAP);
    test(ROOT(HEAD(basediagnoseb))==MTAG_INDIR);
    etst(Eval(basemacro,T,basecache),M0)
    etst(map_apply(HEAD(Eval(basemacro,T,basecache)),basecache),
      baseexpansion)
    etst(Eval(basemacro,T,basecacheb),M0)
    etst(map_apply(HEAD(Eval(basemacro,T,basecacheb)),basecacheb),
      baseexpansion)

/*
---------------------------------------------
Test of verification/expansion of check page
---------------------------------------------
*/

    if(name2sym("check")!=T){
      DEFC(checkref       ,name2ref("check"));
      DEFC(checkcache     ,aget(state,checkref));
      DEFC(checkrack      ,aget(checkcache,checkref));
      DEFC(checkdiagnose  ,aget(checkrack,const_diagnose));
/*    DEFC(checkbody      ,aget(checkrack,const_body)); Unused */
      DEFC(checkexpansion ,aget(checkrack,const_expansion));
      DEFC(checkcodex     ,aget(checkrack,const_codex));
      DEFC(checkmacro     ,def2rhs(mget4(checkcodex,checkref,Z,Z,const_macro)));

      DEFC(checkcacheb    ,Compile(checkcache));
      DEFC(checkrackb     ,aget(checkcacheb,checkref));
      DEFC(checkdiagnoseb ,aget(checkrackb,const_diagnose));

      ttst(untag(checkdiagnose))
      test(ROOT(checkdiagnoseb)==TAG_MAP);
      test(ROOT(HEAD(checkdiagnoseb))==MTAG_CLOSURE);
      ttst(untag(checkdiagnoseb))
      test(ROOT(checkdiagnoseb)==TAG_MAP);
      test(ROOT(HEAD(checkdiagnoseb))==MTAG_INDIR);

      etst(Eval(checkmacro,T,checkcache),M0)
      etst(map_apply(HEAD(Eval(checkmacro,T,checkcache)),checkcache),
        checkexpansion)
      etst(Eval(checkmacro,T,checkcacheb),M0)
      etst(map_apply(HEAD(Eval(checkmacro,T,checkcacheb)),checkcacheb),
        checkexpansion)
    }

/*
---------------------------------------------
Test of verification of Peano page
---------------------------------------------
*/

    if(name2sym("Peano")!=T){
      DEFC(Peanoref       ,name2ref("Peano"));
      DEFC(Peanocache     ,aget(state,Peanoref));
      DEFC(Peanorack      ,aget(Peanocache,Peanoref));
      DEFC(Peanodiagnose  ,aget(Peanorack,const_diagnose));

      DEFC(Peanocacheb    ,Compile(Peanocache));
      DEFC(Peanorackb     ,aget(Peanocacheb,Peanoref));
      DEFC(Peanodiagnoseb ,aget(Peanorackb,const_diagnose));

      ttst(untag(Peanodiagnose))
      test(ROOT(Peanodiagnoseb)==TAG_MAP);
      test(ROOT(HEAD(Peanodiagnoseb))==MTAG_CLOSURE);
      ttst(untag(Peanodiagnoseb))
      test(ROOT(Peanodiagnoseb)==TAG_MAP);
      test(ROOT(HEAD(Peanodiagnoseb))==MTAG_INDIR);
    }

/*
---------------------------------------------
Test of verification of testsuite page
---------------------------------------------
*/

    if(name2sym("test")!=T){
      DEFC(testref       ,name2ref("test"));
      DEFC(testcache     ,aget(state,testref));
      DEFC(testrack      ,aget(testcache,testref));
      DEFC(testdiagnose  ,aget(testrack,const_diagnose));

      DEFC(testcacheb    ,Compile(testcache));
      DEFC(testrackb     ,aget(testcacheb,testref));
      DEFC(testdiagnoseb ,aget(testrackb,const_diagnose));
      ttst(untag(testdiagnose))
      test(ROOT(testdiagnoseb)==TAG_MAP);
      test(ROOT(HEAD(testdiagnoseb))==MTAG_CLOSURE);
      ttst(untag(testdiagnoseb))
      test(ROOT(testdiagnoseb)==TAG_MAP);
      test(ROOT(HEAD(testdiagnoseb))==MTAG_INDIR);
    }

  }

  printf("%s tests succeeded.\n",ptoa(testcnt));}
#endif /* FULL_TESTSUITE */


/*****************
 * MAIN FUNCTION *
 *****************/

void usage(){
  printf("\n");
  printf("  Usage:\n");
  printf("\n");
  printf("  lgwam script FILE ARG1 ... ARGn\n");
  printf("    Load page and execute function mentioned in FILE\n");
  printf("  lgwam test\n");
  printf("    Execute basic built in testsuite\n");
  printf("  lgwam testrun FILE\n");
  printf("    Load page mentioned in FILE and execute extended testsuite\n");
  printf("  lgwam dry\n");
  printf("    Do dry run: initialize memory and exit\n");
  printf("  lgwam { -h | help | --help }\n");
  printf("    Print this message\n");
  printf("  lgwam { -v | version | --version }\n");
  printf("    Print version and license information\n");
  printf("  lgwam { pages | --pages }\n");
  printf("    List built in pages\n");
  printf("\n");
  exit(0);}

#define _QUOTEME(x) #x
#define QUOTEME(x) _QUOTEME(x)
void version(){
  printf("\n");
  printf("  lgwam (LoGiWeb Abstract Machine) Version %s\n",QUOTEME(VERSION));
  printf("  Copyright (C) 2009-2010 Klaus Grue\n");
  printf("  Available under GNU GPL\n");
  printf("  lgwam comes with ABSOLUTELY NO WARRANTY\n");
  printf("\n");
  exit(0);}

void listPages(){
  preload(TRUE);
  exit(0);}

int main1(int argc,char *argv[],char *env[]){
  pnt i;
  int result=0;
  catchSigsegv();
  if(argc<=1) usage();
  test=STREQ(argv[1],"test")||STREQ(argv[1],"testrun");
  pntsize=findsize();
  halfsize=pntsize>>1;
  bytesize=pntsize>>3;
  if(8*bytesize!=pntsize){
    printf("Pointers have %s bits\n",ptoa(pntsize));
    printf("The number of bits is required to be a multiple of 8.\n");
    printf("Goodbye.\n");
    return 0;}
  if(bytesize<sizeof(Trigger)){
    printf("Cannot store function pointers in pointers\n");
    printf("Pointers have %s bytes\n",ptoa(bytesize));
    printf("Function pointers have %s bytes\n",ptoa((pnt)sizeof(Trigger)));
    return 0;}
  signmask=(pnt)(1)<<(pntsize-1);
  markmask=(pnt)(1)<<(pntsize-2);
  halfmask=((pnt)(1)<<halfsize)-1;
  septetmask=(pnt)(-1)<<(pntsize-7);
  heapinit();
  for(i=0;i<256;i++) SETQ(I[i],int2int(i));
  for(i=0;i<256;i++) SETQ(bytevect[i],keep(consvec(i+256,NIL)));
  SETQ(F,cons(TAG_FALSE,NIL,NIL));
  SETQ(X,raise0(T));
  SETQ(Z,I[0]);
  SETQ(M,int2int(-1));
  SETQ(E,consvec(1,0));
  SETQ(term_t,term_const(NIL));
  SETQ(term_f,term_lambda(term_t));
  SETQ(var_0,term_var(0));
  SETQ(var_1,term_var(1));
  SETQ(var_2,term_var(2));
  SETQ(varlist_1,pair(var_0,T));
  SETQ(varlist_2,pair(var_0,pair(var_1,T)));
  SETQ(varlist_3,pair(var_0,pair(var_1,pair(var_2,T))));
  SETQ(term_fix,term_apply(var_1,var_0));
  SETQ(map_fix,cons(MTAG_FIX,NIL,NIL));
  SETQ(V0,cons2(ETAG_var,0,T));
  SETQ(V1,cons2(ETAG_var,1,T));
  SETQ(V2,cons2(ETAG_var,2,T));
  SETQ(V3,cons2(ETAG_var,3,T));
  SETQ(term_delay,term_apply(var_0,var_1));
  SETQ(map_t,term2rnf(term_t,NIL));
  SETQ(map_f,term2rnf(term_f,NIL));
  SETQ(maptag_int,map_pair(T,small2rnf(1)));
  SETQ(maptag_pair,map_pair(T,small2rnf(2)));
  SETQ(maptag_ex,map_pair(T,small2rnf(3)));
  SETQ(maptag_map,map_pair(T,small2rnf(4)));
  SETQ(fct_lambda,Z);
  SETQ(fct_apply,mapcons2(CTAG_APPLY,term2fct(2,term_apply(var_1,var_0))));
  SETQ(fct_if,
    mapcons2(CTAG_IF,term2fct(3,term_apply(term_pair(var_1,var_0),var_2))));
  SETQ(fct_true,mapcons2(CTAG_TRUE,NIL));
  SETQ(fct_quote,I[1]);
  const_bib=          vec2int(V(bibliography));
  const_dict=         vec2int(V(dictionary));
  const_body=         vec2int(V(body));
  const_vector=       vec2int(V(vector));
  const_expansion=    vec2int(V(expansion));
  const_diagnose=     vec2int(V(diagnose));
  const_cache=        vec2int(V(cluster)); 
  const_code=         vec2int(V(code));  
  const_codex=        vec2int(V(codex)); 
  const_definition=   vec2int(V(definition));
  const_name=         vec2int(V(name));
  const_value=        vec2int(V(value));
  const_unpack=       vec2int(V(unpack));
  const_macro=        vec2int(V(macro));
  const_claim=        vec2int(V(claim));
  const_render=       vec2int(V(render));
  const_execute=      vec2int(V(execute));
  const_executables=  vec2int(V(executables));
  const_lambda=       vec2int(V(lambda));
  const_apply=        vec2int(V(apply));
  const_if=           vec2int(V(if));
  const_true=         vec2int(V(true));
  const_quote=        vec2int(V(quote));
  const_spy=          V(spy);
  const_trace=        V(trace);
  const_print=        V(print);
  const_timer=        V(timer);
  const_quit=         V(quit);
  const_write=        V(write);
  const_exec=         V(exec);
  const_read=         V(read);
  const_extend=       V(extend);
  sym_exec=           SYM(exec);
  sym_extend=         SYM(extend);
  sym_boot=           SYM(boot);
  sym_read=           SYM(read);
  tree_introduce=     TREE(introduce);
  tree_define=        TREE(define);
  tree_exit=          TREE(exit);
  tree_time=          TREE(time);
  tree_memory=        TREE(memory);
  state=T;
  init_names();
  catchSigint();
  if(STREQ(argv[1],"-v")) version(); else
  if(STREQ(argv[1],"version")) version(); else
  if(STREQ(argv[1],"--version")) version(); else
  if(STREQ(argv[1],"pages")) listPages(); else
  if(STREQ(argv[1],"--pages")) listPages(); else
  if(STREQ(argv[1],"script")) {} else
  if(STREQ(argv[1],"testrun")) {} else
  if(STREQ(argv[1],"test")) {} else
  if(STREQ(argv[1],"dry")) {} else
  usage();
  if(STREQ(argv[1],"script")||STREQ(argv[1],"testrun"))
  result=runscript(argc,argv,env);
  else if(test) testsuite();
  deathReport();
  if(STACKSIZE-sp) printf("Warning. Stack residue: %s\n",ptoa(STACKSIZE-sp));
  return result;}

#ifndef __CYGWIN__
int main(int argc,char **argv,char **env){return main1(argc,argv,env);}
#else
void catch(int error,char *msg){
  if(error==0) return;
  printf("%s: %s\n",msg,strerror(error));
  exit(1);}

int global_argc;
char **global_argv;
char **global_env;
int global_result;

void *main0(void *ignored){
  global_result=main1(global_argc,global_argv,global_env);
  return NULL;}

int main(int argc,char **argv,char **env){
  pthread_t thread;
  pthread_attr_t attr;
  void *ignored;
  size_t stacksize=200100100;
  global_argc=argc;
  global_argv=argv;
  global_env=env;
  catch(pthread_attr_init(&attr),"pthread_attr_init");
  catch(pthread_attr_setstacksize(&attr,stacksize),"pthread_attr_setstacksize");
  catch(pthread_create(&thread,&attr,&main0,NULL),"pthread_create");
  catch(pthread_join(thread,&ignored),"pthread_join");
  return global_result;}
#endif /* __CYGWIN__ */
