
/* 
 *  Unix SMB/Netbios implementation.
 *  Version 1.9.
 *  RPC Pipe client / server routines
 *  Copyright (C) Andrew Tridgell              1992-1997,
 *  Copyright (C) Luke Kenneth Casson Leighton 1996-1997,
 *  Copyright (C) Paul Ashton                       1997.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "../includes.h"

extern int DEBUGLEVEL;


/*******************************************************************
 makes a SH_INFO_1_STR structure
********************************************************************/
void make_srv_share_info1_str(SH_INFO_1_STR *sh1, char *net_name, char *remark)
{
	if (sh1 == NULL) return;

	DEBUG(5,("make_srv_share_info1_str\n"));

	make_unistr2(&(sh1->uni_netname), net_name, strlen(net_name)+1);
	make_unistr2(&(sh1->uni_remark ), remark  , strlen(remark  )+1);
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_share_info1_str(char *desc, BOOL io, SH_INFO_1_STR *sh1, struct mem_buffer *buf, int *q,  int depth)
{
	if (sh1 == NULL) return;

	DEBUG(5,("%s%06x srv_io_share_info1_str%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_unistr2("", io, &(sh1->uni_netname), True, buf, q, depth); 
	smb_io_unistr2("", io, &(sh1->uni_remark ), True, buf, q, depth); 
}

/*******************************************************************
 makes a SH_INFO_1 structure
********************************************************************/
void make_srv_share_info1(SH_INFO_1 *sh1, char *net_name, uint32 type, char *remark)
{
	if (sh1 == NULL) return;

	DEBUG(5,("make_srv_share_info1: %s %8x %s\n", net_name, type, remark));

	sh1->ptr_netname = net_name != NULL ? 1 : 0;
	sh1->type        = type;
	sh1->ptr_remark  = remark   != NULL ? 1 : 0;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_share_info1(char *desc, BOOL io, SH_INFO_1 *sh1, struct mem_buffer *buf, int *q,  int depth)
{
	if (sh1 == NULL) return;

	DEBUG(5,("%s%06x srv_io_share_info1%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr_netname", depth, buf, q, io, &(sh1->ptr_netname));
	buf_uint32("type       ", depth, buf, q, io, &(sh1->type       ));
	buf_uint32("ptr_remark ", depth, buf, q, io, &(sh1->ptr_remark));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_srv_share_info_1(char *desc, BOOL io, SRV_SHARE_INFO_1 *ctr, struct mem_buffer *buf, int *q,  int depth)
{
	if (ctr == NULL) return;

	DEBUG(5,("%s%06x srv_io_share_1_ctr%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("num_entries_read", depth, buf, q, io, &(ctr->num_entries_read));
	buf_uint32("ptr_share_info", depth, buf, q, io, &(ctr->ptr_share_info));

	if (ctr->ptr_share_info != 0)
	{
		int i;
		int num_entries = ctr->num_entries_read;
		if (num_entries > MAX_SHARE_ENTRIES)
		{
			num_entries = MAX_SHARE_ENTRIES; /* report this! */
		}

		buf_uint32("num_entries_read2", depth, buf, q, io, &(ctr->num_entries_read2));

		for (i = 0; i < num_entries; i++)
		{
			buf_grow(buf, *q);
			srv_io_share_info1("", io, &(ctr->info_1[i]), buf, q, depth); 
		}

		for (i = 0; i < num_entries; i++)
		{
			buf_grow(buf, *q);
			srv_io_share_info1_str("", io, &(ctr->info_1_str[i]), buf, q, depth); 
		}

		buf_align(buf, q);
	}
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_srv_share_ctr(char *desc, BOOL io, SRV_SHARE_INFO_CTR *ctr, struct mem_buffer *buf, int *q,  int depth)
{
	if (ctr == NULL) return;

	DEBUG(5,("%s%06x srv_io_srv_share_ctr%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("switch_value", depth, buf, q, io, &(ctr->switch_value));
	buf_uint32("ptr_share_ctr", depth, buf, q, io, &(ctr->ptr_share_ctr));

	if (ctr->ptr_share_ctr != 0)
	{
		switch (ctr->switch_value)
		{
			case 1:
			{
				srv_io_srv_share_info_1("", io, &(ctr->share.info1), buf, q, depth); 
				break;
			}
			default:
			{
				DEBUG(5,("%s no share info at switch_value %d\n",
				         tab_depth(depth), ctr->switch_value));
				break;
			}
		}
	}
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void make_srv_q_net_share_enum(SRV_Q_NET_SHARE_ENUM *q_n, 
				char *srv_name, 
				uint32 share_level, SRV_SHARE_INFO_CTR *ctr,
				uint32 preferred_len,
				ENUM_HND *hnd)
{
	if (q_n == NULL || ctr == NULL || hnd == NULL) return;

	q_n->ctr = ctr;

	DEBUG(5,("make_q_net_share_enum\n"));

	make_buf_unistr2(&(q_n->uni_srv_name), &(q_n->ptr_srv_name), srv_name);

	q_n->share_level    = share_level;
	q_n->preferred_len = preferred_len;

	memcpy(&(q_n->enum_hnd), hnd, sizeof(*hnd));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_q_net_share_enum(char *desc, BOOL io, SRV_Q_NET_SHARE_ENUM *q_n, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_n == NULL) return;

	DEBUG(5,("%s%06x srv_io_q_net_share_enum%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr_srv_name", depth, buf, q, io, &(q_n->ptr_srv_name));
	smb_io_unistr2("", io, &(q_n->uni_srv_name), True, buf, q, depth); 

	buf_align(buf, q);

	buf_uint32("share_level", depth, buf, q, io, &(q_n->share_level  ));

	if (q_n->share_level != -1)
	{
		srv_io_srv_share_ctr("share_ctr", io, q_n->ctr, buf, q, depth);
	}

	buf_uint32("preferred_len", depth, buf, q, io, &(q_n->preferred_len));

	smb_io_enum_hnd("enum_hnd", io, &(q_n->enum_hnd), buf, q, depth); 
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_r_net_share_enum(char *desc, BOOL io, SRV_R_NET_SHARE_ENUM *r_n, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_n == NULL) return;

	DEBUG(5,("%s%06x srv_io_r_net_share_enum%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("share_level", depth, buf, q, io, &(r_n->share_level));

	if (r_n->share_level != 0)
	{
		srv_io_srv_share_ctr("share_ctr", io, r_n->ctr, buf, q, depth);
	}

	buf_uint32("total_entries", depth, buf, q, io, &(r_n->total_entries));
	smb_io_enum_hnd("enum_hnd", io, &(r_n->enum_hnd), buf, q, depth); 
	buf_uint32("status     ", depth, buf, q, io, &(r_n->status));
}

/*******************************************************************
 makes a SESS_INFO_0_STR structure
********************************************************************/
void make_srv_sess_info0_str(SESS_INFO_0_STR *ss0, char *name)
{
	if (ss0 == NULL) return;

	DEBUG(5,("make_srv_sess_info0_str\n"));

	make_unistr2(&(ss0->uni_name), name, strlen(name));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_sess_info0_str(char *desc, BOOL io, SESS_INFO_0_STR *ss0, struct mem_buffer *buf, int *q,  int depth)
{
	if (ss0 == NULL) return;

	DEBUG(5,("%s%06x srv_io_sess_info0_str%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_unistr2("", io, &(ss0->uni_name), True, buf, q, depth); 
}

/*******************************************************************
 makes a SESS_INFO_0 structure
********************************************************************/
void make_srv_sess_info0(SESS_INFO_0 *ss0, char *name)
{
	if (ss0 == NULL) return;

	DEBUG(5,("make_srv_sess_info0: %s\n", name));

	ss0->ptr_name = name != NULL ? 1 : 0;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_sess_info0(char *desc, BOOL io, SESS_INFO_0 *ss0, struct mem_buffer *buf, int *q,  int depth)
{
	if (ss0 == NULL) return;

	DEBUG(5,("%s%06x srv_io_sess_info0%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr_name", depth, buf, q, io, &(ss0->ptr_name));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_srv_sess_info_0(char *desc, BOOL io, SRV_SESS_INFO_0 *ss0, struct mem_buffer *buf, int *q,  int depth)
{
	if (ss0 == NULL) return;

	DEBUG(5,("%s%06x srv_io_srv_sess_info_0%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("num_entries_read", depth, buf, q, io, &(ss0->num_entries_read));
	buf_uint32("ptr_sess_info", depth, buf, q, io, &(ss0->ptr_sess_info));

	if (ss0->ptr_sess_info != 0)
	{
		int i;
		int num_entries = ss0->num_entries_read;
		if (num_entries > MAX_SESS_ENTRIES)
		{
			num_entries = MAX_SESS_ENTRIES; /* report this! */
		}

		buf_uint32("num_entries_read2", depth, buf, q, io, &(ss0->num_entries_read2));

		for (i = 0; i < num_entries; i++)
		{
			buf_grow(buf, *q);
			srv_io_sess_info0("", io, &(ss0->info_0[i]), buf, q, depth); 
		}

		for (i = 0; i < num_entries; i++)
		{
			buf_grow(buf, *q);
			srv_io_sess_info0_str("", io, &(ss0->info_0_str[i]), buf, q, depth); 
		}

		buf_align(buf, q);
	}
}

/*******************************************************************
 makes a SESS_INFO_1_STR structure
********************************************************************/
void make_srv_sess_info1_str(SESS_INFO_1_STR *ss1, char *name, char *user)
{
	if (ss1 == NULL) return;

	DEBUG(5,("make_srv_sess_info1_str\n"));

	make_unistr2(&(ss1->uni_name), name, strlen(name));
	make_unistr2(&(ss1->uni_user), name, strlen(user));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_sess_info1_str(char *desc, BOOL io, SESS_INFO_1_STR *ss1, struct mem_buffer *buf, int *q,  int depth)
{
	if (ss1 == NULL) return;

	DEBUG(5,("%s%06x srv_io_sess_info1_str%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_unistr2("", io, &(ss1->uni_name), True, buf, q, depth); 
	smb_io_unistr2("", io, &(ss1->uni_user), True, buf, q, depth); 
}

/*******************************************************************
 makes a SESS_INFO_1 structure
********************************************************************/
void make_srv_sess_info1(SESS_INFO_1 *ss1, 
				char *name, char *user,
				uint32 num_opens, uint32 open_time, uint32 idle_time,
				uint32 user_flags)
{
	if (ss1 == NULL) return;

	DEBUG(5,("make_srv_sess_info1: %s\n", name));

	ss1->ptr_name = name != NULL ? 1 : 0;
	ss1->ptr_user = user != NULL ? 1 : 0;

	ss1->num_opens  = num_opens;
	ss1->open_time  = open_time;
	ss1->idle_time  = idle_time;
	ss1->user_flags = user_flags;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_sess_info1(char *desc, BOOL io, SESS_INFO_1 *ss1, struct mem_buffer *buf, int *q,  int depth)
{
	if (ss1 == NULL) return;

	DEBUG(5,("%s%06x srv_io_sess_info1%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr_name  ", depth, buf, q, io, &(ss1->ptr_name  ));
	buf_uint32("ptr_user  ", depth, buf, q, io, &(ss1->ptr_user  ));

	buf_uint32("num_opens ", depth, buf, q, io, &(ss1->num_opens ));
	buf_uint32("open_time ", depth, buf, q, io, &(ss1->open_time ));
	buf_uint32("idle_time ", depth, buf, q, io, &(ss1->idle_time ));
	buf_uint32("user_flags", depth, buf, q, io, &(ss1->user_flags));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_srv_sess_info_1(char *desc, BOOL io, SRV_SESS_INFO_1 *ss1, struct mem_buffer *buf, int *q,  int depth)
{
	if (ss1 == NULL) return;

	DEBUG(5,("%s%06x srv_io_srv_sess_info_1%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("num_entries_read", depth, buf, q, io, &(ss1->num_entries_read));
	buf_uint32("ptr_sess_info", depth, buf, q, io, &(ss1->ptr_sess_info));

	if (ss1->ptr_sess_info != 0)
	{
		int i;
		int num_entries = ss1->num_entries_read;
		if (num_entries > MAX_SESS_ENTRIES)
		{
			num_entries = MAX_SESS_ENTRIES; /* report this! */
		}

		buf_uint32("num_entries_read2", depth, buf, q, io, &(ss1->num_entries_read2));

		for (i = 0; i < num_entries; i++)
		{
			buf_grow(buf, *q);
			srv_io_sess_info1("", io, &(ss1->info_1[i]), buf, q, depth); 
		}

		for (i = 0; i < num_entries; i++)
		{
			buf_grow(buf, *q);
			srv_io_sess_info1_str("", io, &(ss1->info_1_str[i]), buf, q, depth); 
		}

		buf_align(buf, q);
	}
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_srv_sess_ctr(char *desc, BOOL io, SRV_SESS_INFO_CTR *ctr, struct mem_buffer *buf, int *q,  int depth)
{
	if (ctr == NULL) return;

	DEBUG(5,("%s%06x srv_io_srv_sess_ctr%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("switch_value", depth, buf, q, io, &(ctr->switch_value));
	buf_uint32("ptr_sess_ctr", depth, buf, q, io, &(ctr->ptr_sess_ctr));

	if (ctr->ptr_sess_ctr != 0)
	{
		switch (ctr->switch_value)
		{
			case 0:
			{
				srv_io_srv_sess_info_0("", io, &(ctr->sess.info0), buf, q, depth); 
				break;
			}
			case 1:
			{
				srv_io_srv_sess_info_1("", io, &(ctr->sess.info1), buf, q, depth); 
				break;
			}
			default:
			{
				DEBUG(5,("%s no session info at switch_value %d\n",
				         tab_depth(depth), ctr->switch_value));
				break;
			}
		}
	}
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void make_srv_q_net_sess_enum(SRV_Q_NET_SESS_ENUM *q_n, 
				char *srv_name, char *qual_name,
				uint32 sess_level, SRV_SESS_INFO_CTR *ctr,
				uint32 preferred_len,
				ENUM_HND *hnd)
{
	if (q_n == NULL || ctr == NULL || hnd == NULL) return;

	q_n->ctr = ctr;

	DEBUG(5,("make_q_net_sess_enum\n"));

	make_buf_unistr2(&(q_n->uni_srv_name), &(q_n->ptr_srv_name), srv_name);
	make_buf_unistr2(&(q_n->uni_qual_name), &(q_n->ptr_qual_name), qual_name);

	q_n->sess_level    = sess_level;
	q_n->preferred_len = preferred_len;

	memcpy(&(q_n->enum_hnd), hnd, sizeof(*hnd));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_q_net_sess_enum(char *desc, BOOL io, SRV_Q_NET_SESS_ENUM *q_n, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_n == NULL) return;

	DEBUG(5,("%s%06x srv_io_q_net_sess_enum%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr_srv_name", depth, buf, q, io, &(q_n->ptr_srv_name));
	smb_io_unistr2("", io, &(q_n->uni_srv_name), True, buf, q, depth); 

	buf_align(buf, q);

	buf_uint32("ptr_qual_name", depth, buf, q, io, &(q_n->ptr_qual_name));
	smb_io_unistr2("", io, &(q_n->uni_qual_name), True, buf, q, depth); 

	buf_align(buf, q);

	buf_uint32("sess_level", depth, buf, q, io, &(q_n->sess_level  ));
	
	if (q_n->sess_level != -1)
	{
		srv_io_srv_sess_ctr("sess_ctr", io, q_n->ctr, buf, q, depth);
	}

	buf_uint32("preferred_len", depth, buf, q, io, &(q_n->preferred_len));

	smb_io_enum_hnd("enum_hnd", io, &(q_n->enum_hnd), buf, q, depth); 
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_r_net_sess_enum(char *desc, BOOL io, SRV_R_NET_SESS_ENUM *r_n, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_n == NULL) return;

	DEBUG(5,("%s%06x srv_io_r_net_sess_enum%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("sess_level", depth, buf, q, io, &(r_n->sess_level));

	if (r_n->sess_level != -1)
	{
		srv_io_srv_sess_ctr("sess_ctr", io, r_n->ctr, buf, q, depth);
	}

	buf_uint32("total_entries", depth, buf, q, io, &(r_n->total_entries));
	smb_io_enum_hnd("enum_hnd", io, &(r_n->enum_hnd), buf, q, depth); 
	buf_uint32("status     ", depth, buf, q, io, &(r_n->status));
}

/*******************************************************************
 makes a CONN_INFO_0 structure
********************************************************************/
void make_srv_conn_info0(CONN_INFO_0 *ss0, uint32 id)
{
	if (ss0 == NULL) return;

	DEBUG(5,("make_srv_conn_info0\n"));

	ss0->id = id;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_conn_info0(char *desc, BOOL io, CONN_INFO_0 *ss0, struct mem_buffer *buf, int *q,  int depth)
{
	if (ss0 == NULL) return;

	DEBUG(5,("%s%06x srv_io_conn_info0%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("id", depth, buf, q, io, &(ss0->id));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_srv_conn_info_0(char *desc, BOOL io, SRV_CONN_INFO_0 *ss0, struct mem_buffer *buf, int *q,  int depth)
{
	if (ss0 == NULL) return;

	DEBUG(5,("%s%06x srv_io_srv_conn_info_0%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("num_entries_read", depth, buf, q, io, &(ss0->num_entries_read));
	buf_uint32("ptr_conn_info", depth, buf, q, io, &(ss0->ptr_conn_info));

	if (ss0->ptr_conn_info != 0)
	{
		int i;
		int num_entries = ss0->num_entries_read;
		if (num_entries > MAX_CONN_ENTRIES)
		{
			num_entries = MAX_CONN_ENTRIES; /* report this! */
		}

		buf_uint32("num_entries_read2", depth, buf, q, io, &(ss0->num_entries_read2));

		for (i = 0; i < num_entries; i++)
		{
			buf_grow(buf, *q);
			srv_io_conn_info0("", io, &(ss0->info_0[i]), buf, q, depth); 
		}

		buf_align(buf, q);
	}
}

/*******************************************************************
 makes a CONN_INFO_1_STR structure
********************************************************************/
void make_srv_conn_info1_str(CONN_INFO_1_STR *ss1, char *usr_name, char *net_name)
{
	if (ss1 == NULL) return;

	DEBUG(5,("make_srv_conn_info1_str\n"));

	make_unistr2(&(ss1->uni_usr_name), usr_name, strlen(usr_name));
	make_unistr2(&(ss1->uni_net_name), net_name, strlen(net_name));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_conn_info1_str(char *desc, BOOL io, CONN_INFO_1_STR *ss1, struct mem_buffer *buf, int *q,  int depth)
{
	if (ss1 == NULL) return;

	DEBUG(5,("%s%06x srv_io_conn_info1_str%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_unistr2("", io, &(ss1->uni_usr_name), True, buf, q, depth); 
	smb_io_unistr2("", io, &(ss1->uni_net_name), True, buf, q, depth); 
}

/*******************************************************************
 makes a CONN_INFO_1 structure
********************************************************************/
void make_srv_conn_info1(CONN_INFO_1 *ss1, 
				uint32 id, uint32 type,
				uint32 num_opens, uint32 num_users, uint32 open_time,
				char *usr_name, char *net_name)
{
	if (ss1 == NULL) return;

	DEBUG(5,("make_srv_conn_info1: %s %s\n", usr_name, net_name));

	ss1->id        = id       ;
	ss1->type      = type     ;
	ss1->num_opens = num_opens ;
	ss1->num_users = num_users;
	ss1->open_time = open_time;

	ss1->ptr_usr_name = usr_name != NULL ? 1 : 0;
	ss1->ptr_net_name = net_name != NULL ? 1 : 0;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_conn_info1(char *desc, BOOL io, CONN_INFO_1 *ss1, struct mem_buffer *buf, int *q,  int depth)
{
	if (ss1 == NULL) return;

	DEBUG(5,("%s%06x srv_io_conn_info1%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("id          ", depth, buf, q, io, &(ss1->id        ));
	buf_uint32("type        ", depth, buf, q, io, &(ss1->type      ));
	buf_uint32("num_opens   ", depth, buf, q, io, &(ss1->num_opens ));
	buf_uint32("num_users   ", depth, buf, q, io, &(ss1->num_users ));
	buf_uint32("open_time   ", depth, buf, q, io, &(ss1->open_time ));

	buf_uint32("ptr_usr_name", depth, buf, q, io, &(ss1->ptr_usr_name));
	buf_uint32("ptr_net_name", depth, buf, q, io, &(ss1->ptr_net_name));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_srv_conn_info_1(char *desc, BOOL io, SRV_CONN_INFO_1 *ss1, struct mem_buffer *buf, int *q,  int depth)
{
	if (ss1 == NULL) return;

	DEBUG(5,("%s%06x srv_io_srv_conn_info_1%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("num_entries_read", depth, buf, q, io, &(ss1->num_entries_read));
	buf_uint32("ptr_conn_info", depth, buf, q, io, &(ss1->ptr_conn_info));

	if (ss1->ptr_conn_info != 0)
	{
		int i;
		int num_entries = ss1->num_entries_read;
		if (num_entries > MAX_CONN_ENTRIES)
		{
			num_entries = MAX_CONN_ENTRIES; /* report this! */
		}

		buf_uint32("num_entries_read2", depth, buf, q, io, &(ss1->num_entries_read2));

		for (i = 0; i < num_entries; i++)
		{
			buf_grow(buf, *q);
			srv_io_conn_info1("", io, &(ss1->info_1[i]), buf, q, depth); 
		}

		for (i = 0; i < num_entries; i++)
		{
			buf_grow(buf, *q);
			srv_io_conn_info1_str("", io, &(ss1->info_1_str[i]), buf, q, depth); 
		}

		buf_align(buf, q);
	}
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_srv_conn_ctr(char *desc, BOOL io, SRV_CONN_INFO_CTR *ctr, struct mem_buffer *buf, int *q,  int depth)
{
	if (ctr == NULL) return;

	DEBUG(5,("%s%06x srv_io_srv_conn_ctr%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("switch_value", depth, buf, q, io, &(ctr->switch_value));
	buf_uint32("ptr_conn_ctr", depth, buf, q, io, &(ctr->ptr_conn_ctr));

	if (ctr->ptr_conn_ctr != 0)
	{
		switch (ctr->switch_value)
		{
			case 0:
			{
				srv_io_srv_conn_info_0("", io, &(ctr->conn.info0), buf, q, depth); 
				break;
			}
			case 1:
			{
				srv_io_srv_conn_info_1("", io, &(ctr->conn.info1), buf, q, depth); 
				break;
			}
			default:
			{
				DEBUG(5,("%s no connion info at switch_value %d\n",
				         tab_depth(depth), ctr->switch_value));
				break;
			}
		}
	}
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void make_srv_q_net_conn_enum(SRV_Q_NET_CONN_ENUM *q_n, 
				char *srv_name, char *qual_name,
				uint32 conn_level, SRV_CONN_INFO_CTR *ctr,
				uint32 preferred_len,
				ENUM_HND *hnd)
{
	if (q_n == NULL || ctr == NULL || hnd == NULL) return;

	q_n->ctr = ctr;

	DEBUG(5,("make_q_net_conn_enum\n"));

	make_buf_unistr2(&(q_n->uni_srv_name ), &(q_n->ptr_srv_name ), srv_name );
	make_buf_unistr2(&(q_n->uni_qual_name), &(q_n->ptr_qual_name), qual_name);

	q_n->conn_level    = conn_level;
	q_n->preferred_len = preferred_len;

	memcpy(&(q_n->enum_hnd), hnd, sizeof(*hnd));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_q_net_conn_enum(char *desc, BOOL io, SRV_Q_NET_CONN_ENUM *q_n, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_n == NULL) return;

	DEBUG(5,("%s%06x srv_io_q_net_conn_enum%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr_srv_name ", depth, buf, q, io, &(q_n->ptr_srv_name));
	smb_io_unistr2("", io, &(q_n->uni_srv_name), q_n->ptr_srv_name, buf, q, depth); 

	buf_align(buf, q);

	buf_uint32("ptr_qual_name", depth, buf, q, io, &(q_n->ptr_qual_name));
	smb_io_unistr2("", io, &(q_n->uni_qual_name), q_n->ptr_qual_name, buf, q, depth); 

	buf_align(buf, q);

	buf_uint32("conn_level", depth, buf, q, io, &(q_n->conn_level  ));
	
	if (q_n->conn_level != -1)
	{
		srv_io_srv_conn_ctr("conn_ctr", io, q_n->ctr, buf, q, depth);
	}

	buf_uint32("preferred_len", depth, buf, q, io, &(q_n->preferred_len));

	smb_io_enum_hnd("enum_hnd", io, &(q_n->enum_hnd), buf, q, depth); 
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_r_net_conn_enum(char *desc, BOOL io, SRV_R_NET_CONN_ENUM *r_n, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_n == NULL) return;

	DEBUG(5,("%s%06x srv_io_r_net_conn_enum%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("conn_level", depth, buf, q, io, &(r_n->conn_level));

	if (r_n->conn_level != -1)
	{
		srv_io_srv_conn_ctr("conn_ctr", io, r_n->ctr, buf, q, depth);
	}

	buf_uint32("total_entries", depth, buf, q, io, &(r_n->total_entries));
	smb_io_enum_hnd("enum_hnd", io, &(r_n->enum_hnd), buf, q, depth); 
	buf_uint32("status     ", depth, buf, q, io, &(r_n->status));
}

/*******************************************************************
 makes a FILE_INFO_3_STR structure
********************************************************************/
void make_srv_file_info3_str(FILE_INFO_3_STR *fi3, char *user_name, char *path_name)
{
	if (fi3 == NULL) return;

	DEBUG(5,("make_srv_file_info3_str\n"));

	make_unistr2(&(fi3->uni_path_name), path_name, strlen(path_name)+1);
	make_unistr2(&(fi3->uni_user_name), user_name, strlen(user_name)+1);
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_file_info3_str(char *desc, BOOL io, FILE_INFO_3_STR *sh1, struct mem_buffer *buf, int *q,  int depth)
{
	if (sh1 == NULL) return;

	DEBUG(5,("%s%06x srv_io_file_info3_str%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_unistr2("", io, &(sh1->uni_path_name), True, buf, q, depth); 
	smb_io_unistr2("", io, &(sh1->uni_user_name), True, buf, q, depth); 
}

/*******************************************************************
 makes a FILE_INFO_3 structure
********************************************************************/
void make_srv_file_info3(FILE_INFO_3 *fl3,
				uint32 id, uint32 perms, uint32 num_locks,
				char *path_name, char *user_name)
{
	if (fl3 == NULL) return;

	DEBUG(5,("make_srv_file_info3: %s %s\n", path_name, user_name));

	fl3->id        = id;	
	fl3->perms     = perms;
	fl3->num_locks = num_locks;

	fl3->ptr_path_name = path_name != NULL ? 1 : 0;
	fl3->ptr_user_name = user_name != NULL ? 1 : 0;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_file_info3(char *desc, BOOL io, FILE_INFO_3 *fl3, struct mem_buffer *buf, int *q,  int depth)
{
	if (fl3 == NULL) return;

	DEBUG(5,("%s%06x srv_io_file_info3%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("id           ", depth, buf, q, io, &(fl3->id           ));
	buf_uint32("perms        ", depth, buf, q, io, &(fl3->perms        ));
	buf_uint32("num_locks    ", depth, buf, q, io, &(fl3->num_locks    ));
	buf_uint32("ptr_path_name", depth, buf, q, io, &(fl3->ptr_path_name));
	buf_uint32("ptr_user_name", depth, buf, q, io, &(fl3->ptr_user_name));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_srv_file_info_3(char *desc, BOOL io, SRV_FILE_INFO_3 *fl3, struct mem_buffer *buf, int *q,  int depth)
{
	if (fl3 == NULL) return;

	DEBUG(5,("%s%06x srv_io_file_3_fl3%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("num_entries_read", depth, buf, q, io, &(fl3->num_entries_read));
	buf_uint32("ptr_file_fl3", depth, buf, q, io, &(fl3->ptr_file_info));
	if (fl3->ptr_file_info != 0)
	{
		int i;
		int num_entries = fl3->num_entries_read;
		if (num_entries > MAX_FILE_ENTRIES)
		{
			num_entries = MAX_FILE_ENTRIES; /* report this! */
		}

		buf_uint32("num_entries_read2", depth, buf, q, io, &(fl3->num_entries_read2));

		for (i = 0; i < num_entries; i++)
		{
			buf_grow(buf, *q);
			srv_io_file_info3("", io, &(fl3->info_3[i]), buf, q, depth); 
		}

		for (i = 0; i < num_entries; i++)
		{
			buf_grow(buf, *q);
			srv_io_file_info3_str("", io, &(fl3->info_3_str[i]), buf, q, depth); 
		}

		buf_align(buf, q);
	}
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_srv_file_ctr(char *desc, BOOL io, SRV_FILE_INFO_CTR *ctr, struct mem_buffer *buf, int *q,  int depth)
{
	if (ctr == NULL) return;

	DEBUG(5,("%s%06x srv_io_srv_file_ctr%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("switch_value", depth, buf, q, io, &(ctr->switch_value));
	buf_uint32("ptr_file_ctr", depth, buf, q, io, &(ctr->ptr_file_ctr));

	if (ctr->ptr_file_ctr != 0)
	{
		switch (ctr->switch_value)
		{
			case 3:
			{
				srv_io_srv_file_info_3("", io, &(ctr->file.info3), buf, q, depth); 
				break;
			}
			default:
			{
				DEBUG(5,("%s no file info at switch_value %d\n",
				         tab_depth(depth), ctr->switch_value));
				break;
			}
		}
	}
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void make_srv_q_net_file_enum(SRV_Q_NET_FILE_ENUM *q_n, 
				char *srv_name, char *qual_name,
				uint32 file_level, SRV_FILE_INFO_CTR *ctr,
				uint32 preferred_len,
				ENUM_HND *hnd)
{
	if (q_n == NULL || ctr == NULL || hnd == NULL) return;

	q_n->ctr = ctr;

	DEBUG(5,("make_q_net_file_enum\n"));

	make_buf_unistr2(&(q_n->uni_srv_name), &(q_n->ptr_srv_name), srv_name);
	make_buf_unistr2(&(q_n->uni_qual_name), &(q_n->ptr_qual_name), qual_name);

	q_n->file_level    = file_level;
	q_n->preferred_len = preferred_len;

	memcpy(&(q_n->enum_hnd), hnd, sizeof(*hnd));
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_q_net_file_enum(char *desc, BOOL io, SRV_Q_NET_FILE_ENUM *q_n, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_n == NULL) return;

	DEBUG(5,("%s%06x srv_io_q_net_file_enum%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr_srv_name", depth, buf, q, io, &(q_n->ptr_srv_name));
	smb_io_unistr2("", io, &(q_n->uni_srv_name), True, buf, q, depth); 

	buf_align(buf, q);

	buf_uint32("ptr_qual_name", depth, buf, q, io, &(q_n->ptr_qual_name));
	smb_io_unistr2("", io, &(q_n->uni_qual_name), True, buf, q, depth); 

	buf_align(buf, q);

	buf_uint32("file_level", depth, buf, q, io, &(q_n->file_level  ));

	if (q_n->file_level != -1)
	{
		srv_io_srv_file_ctr("file_ctr", io, q_n->ctr, buf, q, depth);
	}

	buf_uint32("preferred_len", depth, buf, q, io, &(q_n->preferred_len));

	smb_io_enum_hnd("enum_hnd", io, &(q_n->enum_hnd), buf, q, depth); 
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_r_net_file_enum(char *desc, BOOL io, SRV_R_NET_FILE_ENUM *r_n, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_n == NULL) return;

	DEBUG(5,("%s%06x srv_io_r_net_file_enum%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("file_level", depth, buf, q, io, &(r_n->file_level));

	if (r_n->file_level != 0)
	{
		srv_io_srv_file_ctr("file_ctr", io, r_n->ctr, buf, q, depth);
	}

	buf_uint32("total_entries", depth, buf, q, io, &(r_n->total_entries));
	smb_io_enum_hnd("enum_hnd", io, &(r_n->enum_hnd), buf, q, depth); 
	buf_uint32("status     ", depth, buf, q, io, &(r_n->status));
}

/*******************************************************************
 makes a SRV_INFO_101 structure.
 ********************************************************************/
void make_srv_info_101(SRV_INFO_101 *sv101, uint32 platform_id, char *name,
				uint32 ver_major, uint32 ver_minor,
				uint32 srv_type, char *comment)
{
	if (sv101 == NULL) return;

	DEBUG(5,("make_srv_info_101\n"));

	sv101->platform_id  = platform_id;
	make_buf_unistr2(&(sv101->uni_name    ), &(sv101->ptr_name   ) , name    );
	sv101->ver_major    = ver_major;
	sv101->ver_minor    = ver_minor;
	sv101->srv_type     = srv_type;
	make_buf_unistr2(&(sv101->uni_comment ), &(sv101->ptr_comment) , comment );
}


/*******************************************************************
 reads or writes a SRV_INFO_101 structure.
 ********************************************************************/
void srv_io_info_101(char *desc, BOOL io, SRV_INFO_101 *sv101, struct mem_buffer *buf, int *q,  int depth)
{
	if (sv101 == NULL) return;

	DEBUG(5,("%s%06x srv_io_info_101%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("platform_id ", depth, buf, q, io, &(sv101->platform_id ));
	buf_uint32("ptr_name    ", depth, buf, q, io, &(sv101->ptr_name    ));
	buf_uint32("ver_major   ", depth, buf, q, io, &(sv101->ver_major   ));
	buf_uint32("ver_minor   ", depth, buf, q, io, &(sv101->ver_minor   ));
	buf_uint32("srv_type    ", depth, buf, q, io, &(sv101->srv_type    ));
	buf_uint32("ptr_comment ", depth, buf, q, io, &(sv101->ptr_comment ));

	buf_align(buf, q);

	smb_io_unistr2("uni_name    ", io, &(sv101->uni_name    ), True, buf, q, depth); 
	smb_io_unistr2("uni_comment ", io, &(sv101->uni_comment ), True, buf, q, depth); 
}

/*******************************************************************
 makes a SRV_INFO_102 structure.
 ********************************************************************/
void make_srv_info_102(SRV_INFO_102 *sv102, uint32 platform_id, char *name,
				char *comment, uint32 ver_major, uint32 ver_minor,
				uint32 srv_type, uint32 users, uint32 disc, uint32 hidden,
				uint32 announce, uint32 ann_delta, uint32 licenses,
				char *usr_path)
{
	if (sv102 == NULL) return;

	DEBUG(5,("make_srv_info_102\n"));

	sv102->platform_id  = platform_id;
	make_buf_unistr2(&(sv102->uni_name    ), &(sv102->ptr_name    ), name    );
	sv102->ver_major    = ver_major;
	sv102->ver_minor    = ver_minor;
	sv102->srv_type     = srv_type;
	make_buf_unistr2(&(sv102->uni_comment ), &(sv102->ptr_comment ), comment );

	/* same as 101 up to here */

	sv102->users        = users;
	sv102->disc         = disc;
	sv102->hidden       = hidden;
	sv102->announce     = announce;
	sv102->ann_delta    =ann_delta;
	sv102->licenses     = licenses;
	make_buf_unistr2(&(sv102->uni_usr_path), &(sv102->ptr_usr_path), usr_path);
}


/*******************************************************************
 reads or writes a SRV_INFO_102 structure.
 ********************************************************************/
void srv_io_info_102(char *desc, BOOL io, SRV_INFO_102 *sv102, struct mem_buffer *buf, int *q,  int depth)
{
	if (sv102 == NULL) return;

	DEBUG(5,("%s%06x srv_io_info102%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("platform_id ", depth, buf, q, io, &(sv102->platform_id ));
	buf_uint32("ptr_name    ", depth, buf, q, io, &(sv102->ptr_name    ));
	buf_uint32("ver_major   ", depth, buf, q, io, &(sv102->ver_major   ));
	buf_uint32("ver_minor   ", depth, buf, q, io, &(sv102->ver_minor   ));
	buf_uint32("srv_type    ", depth, buf, q, io, &(sv102->srv_type    ));
	buf_uint32("ptr_comment ", depth, buf, q, io, &(sv102->ptr_comment ));

	/* same as 101 up to here */

	buf_uint32("users       ", depth, buf, q, io, &(sv102->users       ));
	buf_uint32("disc        ", depth, buf, q, io, &(sv102->disc        ));
	buf_uint32("hidden      ", depth, buf, q, io, &(sv102->hidden      ));
	buf_uint32("announce    ", depth, buf, q, io, &(sv102->announce    ));
	buf_uint32("ann_delta   ", depth, buf, q, io, &(sv102->ann_delta   ));
	buf_uint32("licenses    ", depth, buf, q, io, &(sv102->licenses    ));
	buf_uint32("ptr_usr_path", depth, buf, q, io, &(sv102->ptr_usr_path));

	smb_io_unistr2("uni_name    ", io, &(sv102->uni_name    ), True, buf, q, depth); 
	buf_align(buf, q);
	smb_io_unistr2("uni_comment ", io, &(sv102->uni_comment ), True, buf, q, depth); 
	buf_align(buf, q);
	smb_io_unistr2("uni_usr_path", io, &(sv102->uni_usr_path), True, buf, q, depth); 
}

/*******************************************************************
 reads or writes a SRV_INFO_102 structure.
 ********************************************************************/
void srv_io_info_ctr(char *desc, BOOL io, SRV_INFO_CTR *ctr, struct mem_buffer *buf, int *q,  int depth)
{
	if (ctr == NULL) return;

	DEBUG(5,("%s%06x srv_io_info_ctr%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("switch_value", depth, buf, q, io, &(ctr->switch_value));
	buf_uint32("ptr_srv_ctr ", depth, buf, q, io, &(ctr->ptr_srv_ctr ));

	if (ctr->ptr_srv_ctr != 0 && ctr->switch_value != 0 && ctr != NULL)
	{
		switch (ctr->switch_value)
		{
			case 101:
			{
				srv_io_info_101("sv101", io, &(ctr->srv.sv101), buf, q, depth); 
				break;
			}
			case 102:
			{
				srv_io_info_102("sv102", io, &(ctr->srv.sv102), buf, q, depth); 
				break;
			}
			default:
			{
				DEBUG(5,("%s% no server info at switch_value %d\n",
						 tab_depth(depth), ctr->switch_value));
				break;
			}
		}
	}
}

/*******************************************************************
 makes a SRV_Q_NET_SRV_GET_INFO structure.
 ********************************************************************/
void make_srv_q_net_srv_get_info(SRV_Q_NET_SRV_GET_INFO *srv,
				char *server_name, uint32 switch_value)
{
	if (srv == NULL) return;

	DEBUG(5,("make_srv_q_net_srv_get_info\n"));

	make_buf_unistr2(&(srv->uni_srv_name), &(srv->ptr_srv_name), server_name);

	srv->switch_value = switch_value;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_q_net_srv_get_info(char *desc, BOOL io, SRV_Q_NET_SRV_GET_INFO *q_n, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_n == NULL) return;

	DEBUG(5,("%s%06x srv_io_q_net_srv_get_info%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr_srv_name  ", depth, buf, q, io, &(q_n->ptr_srv_name));
	smb_io_unistr2("", io, &(q_n->uni_srv_name), True, buf, q, depth); 

	buf_align(buf, q);

	buf_uint32("switch_value  ", depth, buf, q, io, &(q_n->switch_value));
}

/*******************************************************************
 makes a SRV_R_NET_SRV_GET_INFO structure.
 ********************************************************************/
void make_srv_r_net_srv_get_info(SRV_R_NET_SRV_GET_INFO *srv,
				uint32 switch_value, SRV_INFO_CTR *ctr, uint32 status)
{
	if (srv == NULL) return;

	DEBUG(5,("make_srv_r_net_srv_get_info\n"));

	srv->ctr = ctr;

	if (status == 0x0)
	{
		srv->ctr->switch_value = switch_value;
		srv->ctr->ptr_srv_ctr  = 1;
	}
	else
	{
		srv->ctr->switch_value = 0;
		srv->ctr->ptr_srv_ctr  = 0;
	}

	srv->status = status;
}

/*******************************************************************
 reads or writes a structure.
 ********************************************************************/
void srv_io_r_net_srv_get_info(char *desc, BOOL io, SRV_R_NET_SRV_GET_INFO *r_n, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_n == NULL) return;

	DEBUG(5,("%s%06x srv_io_r_net_srv_get_info%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	srv_io_info_ctr("ctr", io, r_n->ctr, buf, q, depth); 

	buf_uint32("status      ", depth, buf, q, io, &(r_n->status      ));
}


/*******************************************************************
 makes a SRV_Q_NET_SRV_SET_INFO structure.
 ********************************************************************/
void make_srv_q_net_srv_set_info(SRV_Q_NET_SRV_SET_INFO *srv,
				uint32 switch_value, SRV_INFO_CTR *ctr)
{
	if (srv == NULL) return;

	DEBUG(5,("make_srv_r_net_srv_set_info\n"));

	srv->ctr    = ctr;

	srv->ctr->switch_value = switch_value;
	srv->ctr->ptr_srv_ctr  = 1;
}

/*******************************************************************
reads or writes a structure.
********************************************************************/
void srv_io_q_net_srv_set_info(char *desc, BOOL io, SRV_Q_NET_SRV_SET_INFO *q_n, struct mem_buffer *buf, int *q,  int depth)
{
	if (q_n == NULL) return;

	DEBUG(5,("%s%06x srv_io_q_net_srv_set_info%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("ptr_srv_name  ", depth, buf, q, io, &(q_n->ptr_srv_name));
	smb_io_unistr2("", io, &(q_n->uni_srv_name), True, buf, q, depth); 

	buf_align(buf, q);

	srv_io_info_ctr("ctr", io, q_n->ctr, buf, q, depth); 
}

/*******************************************************************
 makes a SRV_R_NET_SRV_SET_INFO structure.
 ********************************************************************/
void make_srv_r_net_srv_set_info(SRV_R_NET_SRV_SET_INFO *srv,
				uint32 switch_value, SRV_INFO_CTR *ctr, uint32 status)
{
	if (srv == NULL) return;

	DEBUG(5,("make_srv_r_net_srv_set_info\n"));

	if (status == 0x0)
	{
		srv->switch_value = switch_value;
	}
	else
	{
		srv->switch_value = 0;
	}

	srv->status = status;
}

/*******************************************************************
 reads or writes a structure.
 ********************************************************************/
void srv_io_r_net_srv_set_info(char *desc, BOOL io, SRV_R_NET_SRV_SET_INFO *r_n, struct mem_buffer *buf, int *q,  int depth)
{
	if (r_n == NULL) return;

	DEBUG(5,("%s%06x srv_io_q_net_srv_set_info%s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("switch_value", depth, buf, q, io, &(r_n->switch_value));
	buf_uint32("status      ", depth, buf, q, io, &(r_n->status      ));
}


