
/* 
 *  Unix SMB/Netbios implementation.
 *  Version 1.9.
 *  RPC Pipe client / server routines
 *  Copyright (C) Andrew Tridgell              1992-1997,
 *  Copyright (C) Luke Kenneth Casson Leighton 1996-1997,
 *  Copyright (C) Paul Ashton                       1997.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "../includes.h"

extern int DEBUGLEVEL;



/*******************************************************************
reads or writes a UTIME type.
********************************************************************/
void smb_io_utime(char *desc, BOOL io, UTIME *t, struct mem_buffer *buf, int *q, int depth)
{
	if (t == NULL) return;

	DEBUG(5,("%s%06x smb_io_utime %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32 ("time", depth, buf, q, io, &(t->time));
}

/*******************************************************************
reads or writes an NTTIME structure.
********************************************************************/
void smb_io_time(char *desc, BOOL io, NTTIME *nttime, struct mem_buffer *buf, int *q, int depth)
{
	if (nttime == NULL) return;

	DEBUG(5,("%s%06x smb_io_time %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("low ", depth, buf, q, io, &(nttime->low )); /* low part */
	buf_uint32("high", depth, buf, q, io, &(nttime->high)); /* high part */
}

/*******************************************************************
gets an enumeration handle from an ENUM_HND structure.
********************************************************************/
uint32 get_enum_hnd(ENUM_HND *enh)
{
	return (enh && enh->ptr_hnd != 0) ? enh->handle : 0;
}

/*******************************************************************
makes an ENUM_HND structure.
********************************************************************/
void make_enum_hnd(ENUM_HND *enh, uint32 hnd)
{
	if (enh == NULL) return;

	DEBUG(5,("smb_io_enum_hnd\n"));

	enh->ptr_hnd = (hnd != 0) ? 1 : 0;
	enh->handle = hnd;
}

/*******************************************************************
reads or writes an ENUM_HND structure.
********************************************************************/
void smb_io_enum_hnd(char *desc, BOOL io, ENUM_HND *hnd, struct mem_buffer *buf, int *q, int depth)
{
	if (hnd == NULL) return;

	DEBUG(5,("%s%06x smb_io_enum_hnd %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("ptr_hnd", depth, buf, q, io, &(hnd->ptr_hnd)); /* pointer */
	if (hnd->ptr_hnd != 0)
	{
		buf_uint32("handle ", depth, buf, q, io, &(hnd->handle )); /* enum handle */
	}
}

/*******************************************************************
creates a DOM_SID structure.

BIG NOTE: this function only does SIDS where the identauth is not >= 2^32 
identauth >= 2^32 can be detected because it will be specified in hex

********************************************************************/
void make_dom_sid(DOM_SID *sid, char *str_sid)
{
	pstring domsid;
	int identauth;
	char *p;

	if (sid == NULL) return;

	if (domsid == NULL)
	{
		DEBUG(4,("netlogon domain SID: none\n"));
		sid->sid_rev_num = 0;
		sid->num_auths = 0;
		return;
	}
		
	pstrcpy(domsid, str_sid);

	DEBUG(4,("make_dom_sid %d SID:  %s\n", __LINE__, domsid));

	/* assume, but should check, that domsid starts "S-" */
	p = strtok(domsid+2,"-");
	sid->sid_rev_num = atoi(p);

	/* identauth in decimal should be <  2^32 */
	/* identauth in hex     should be >= 2^32 */
	identauth = atoi(strtok(0,"-"));

	DEBUG(4,("netlogon rev %d\n", sid->sid_rev_num));
	DEBUG(4,("netlogon %s ia %d\n", p, identauth));

	sid->id_auth[0] = 0;
	sid->id_auth[1] = 0;
	sid->id_auth[2] = (identauth & 0xff000000) >> 24;
	sid->id_auth[3] = (identauth & 0x00ff0000) >> 16;
	sid->id_auth[4] = (identauth & 0x0000ff00) >> 8;
	sid->id_auth[5] = (identauth & 0x000000ff);

	sid->num_auths = 0;

	while ((p = strtok(0, "-")) != NULL && sid->num_auths < MAXSUBAUTHS)
	{
		sid->sub_auths[sid->num_auths++] = atoi(p);
	}

	DEBUG(4,("make_dom_sid: %d SID:  %s\n", __LINE__, domsid));
}

/*******************************************************************
reads or writes a DOM_SID structure.
********************************************************************/
void smb_io_dom_sid(char *desc, BOOL io, DOM_SID *sid, struct mem_buffer *buf, int *q, int depth)
{
	int i;

	if (sid == NULL) return;

	DEBUG(5,("%s%06x smb_io_dom_sid %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint8 ("sid_rev_num", depth, buf, q, io, &(sid->sid_rev_num)); 
	buf_uint8 ("num_auths  ", depth, buf, q, io, &(sid->num_auths));

	for (i = 0; i < 6; i++)
	{
		fstring tmp;
		sprintf(tmp, "id_auth[%d] ", i);
		buf_uint8 (tmp, depth, buf, q, io, &(sid->id_auth[i]));
	}

	/* oops! XXXX should really issue a warning here... */
	if (sid->num_auths > MAXSUBAUTHS) sid->num_auths = MAXSUBAUTHS;

	buf_uint32s(False, "sub_auths ", depth, buf, q, io, sid->sub_auths, sid->num_auths);
}

/*******************************************************************
creates a DOM_SID2 structure.
********************************************************************/
void make_dom_sid2(DOM_SID2 *sid, char *str_sid)
{
	make_dom_sid(&(sid->sid), str_sid);
	sid->num_auths = sid->sid.num_auths;
}

/*******************************************************************
reads or writes a DOM_SID2 structure.
********************************************************************/
void smb_io_dom_sid2(char *desc, BOOL io, DOM_SID2 *sid, struct mem_buffer *buf, int *q, int depth)
{
	if (sid == NULL) return;

	DEBUG(5,("%s%06x smb_io_dom_sid2 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("num_auths", depth, buf, q, io, &(sid->num_auths));

	smb_io_dom_sid("sid", io, &(sid->sid), buf, q, depth);
}

/*******************************************************************
creates a UNIHDR structure.
********************************************************************/
void make_uni_hdr(UNIHDR *hdr, int max_len, int len, uint32 buffer)
{
	hdr->uni_max_len = 2 * max_len;
	hdr->uni_str_len = 2 * len;
	hdr->buffer      = buffer;
}

/*******************************************************************
reads or writes a UNIHDR structure.
********************************************************************/
void smb_io_unihdr(char *desc, BOOL io, UNIHDR *hdr, struct mem_buffer *buf, int *q, int depth)
{
	if (hdr == NULL) return;

	DEBUG(5,("%s%06x smb_io_unihdr %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint16("uni_str_len", depth, buf, q, io, &(hdr->uni_str_len));
	buf_uint16("uni_max_len", depth, buf, q, io, &(hdr->uni_max_len));
	buf_uint32("buffer     ", depth, buf, q, io, &(hdr->buffer     ));

	/* oops! XXXX maybe issue a warning that this is happening... */
	if (hdr->uni_max_len > MAX_UNISTRLEN) hdr->uni_max_len = MAX_UNISTRLEN;
	if (hdr->uni_str_len > MAX_UNISTRLEN) hdr->uni_str_len = MAX_UNISTRLEN;
}

/*******************************************************************
creates a UNIHDR2 structure.
********************************************************************/
void make_uni_hdr2(UNIHDR2 *hdr, int max_len, int len, uint16 terminate)
{
	make_uni_hdr(&(hdr->unihdr), max_len, len, terminate);
	hdr->buffer = len > 0 ? 1 : 0;
}

/*******************************************************************
reads or writes a UNIHDR2 structure.
********************************************************************/
void smb_io_unihdr2(char *desc, BOOL io, UNIHDR2 *hdr2, struct mem_buffer *buf, int *q, int depth)
{
	if (hdr2 == NULL) return;

	DEBUG(5,("%s%06x smb_io_unihdr2 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_unihdr("hdr", io, &(hdr2->unihdr), buf, q, depth);
	buf_uint32("buffer", depth, buf, q, io, &(hdr2->buffer));
}

/*******************************************************************
creates a UNISTR structure.
********************************************************************/
void make_unistr(UNISTR *str, char *buf)
{
	/* store the string (null-terminated copy) */
	struni2(str->buffer, buf);
}

/*******************************************************************
reads or writes a UNISTR structure.
XXXX NOTE: UNISTR structures NEED to be null-terminated.
********************************************************************/
void smb_io_unistr(char *desc, BOOL io, UNISTR *uni, struct mem_buffer *buf, int *q, int depth)
{
	if (uni == NULL) return;

	DEBUG(5,("%s%06x smb_io_unistr: ",  tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	buf_unistr("unistr", depth, buf, q, io, uni);
}

/*******************************************************************
creates a UNINOTSTR2 structure.
********************************************************************/
void make_uninotstr2(UNINOTSTR2 *str, char *buf, int len)
{
	/* set up string lengths. add one if string is not null-terminated */
	str->uni_max_len = (len+1)*2;
	str->undoc       = 0;
	str->uni_buf_len = (len+1)*2;

	/* store the string (null-terminated copy) */
	struni2(str->buffer, buf);
}

/*******************************************************************
reads or writes a UNINOTSTR2 structure.
XXXX NOTE: UNISTR2 structures need NOT be null-terminated.
     the uni_str_len member tells you how long the string is;
     the uni_max_len member tells you how large the buffer is.
********************************************************************/
void smb_io_uninotstr2(char *desc, BOOL io, UNINOTSTR2 *uni2, uint32 buffer, struct mem_buffer *buf, int *q, int depth)
{
	if (uni2 == NULL) return;

	if (buffer)
	{
		DEBUG(5,("%s%06x smb_io_uninotstr2 %s\n", tab_depth(depth), *q, desc));
		depth++;

		buf_align(buf, q);
		
		buf_uint32("uni_max_len", depth, buf, q, io, &(uni2->uni_max_len));
		buf_uint32("undoc      ", depth, buf, q, io, &(uni2->undoc      ));
		buf_uint32("uni_buf_len", depth, buf, q, io, &(uni2->uni_buf_len));

		/* oops! XXXX maybe issue a warning that this is happening... */
		if (uni2->uni_max_len > MAX_UNISTRLEN) uni2->uni_max_len = MAX_UNISTRLEN;
		if (uni2->uni_buf_len > MAX_UNISTRLEN) uni2->uni_buf_len = MAX_UNISTRLEN;

		/* buffer advanced by indicated length of string
		   NOT by searching for null-termination */
		buf_uninotstr2(True, "buffer     ", depth, buf, q, io, uni2);
	}
	else
	{
		DEBUG(5,("%s%06x smb_io_uninotstr2 - NULL %s\n", tab_depth(depth), *q, desc));
		bzero(uni2, sizeof(*uni2));
	}
}

/*******************************************************************
creates a UNISTR2 structure: sets up the buffer, too
********************************************************************/
void make_buf_unistr2(UNISTR2 *str, uint32 *ptr, char *buf)
{
	if (buf != NULL)
	{
		*ptr = 1;
		make_unistr2(str, buf, strlen(buf));
	}
	else
	{
		*ptr = 0;
		make_unistr2(str, "", 0);
	}
}

/*******************************************************************
creates a UNISTR2 structure.
********************************************************************/
void make_unistr2(UNISTR2 *str, char *buf, int len)
{
	/* set up string lengths. add one if string is not null-terminated */
	str->uni_max_len = len+1;
	str->undoc       = 0;
	str->uni_str_len = len+1;

	/* store the string (null-terminated copy) */
	struni2(str->buffer, buf);
}

/*******************************************************************
reads or writes a UNISTR2 structure.
XXXX NOTE: UNISTR2 structures need NOT be null-terminated.
     the uni_str_len member tells you how long the string is;
     the uni_max_len member tells you how large the buffer is.
********************************************************************/
void smb_io_unistr2(char *desc, BOOL io, UNISTR2 *uni2, uint32 buffer, struct mem_buffer *buf, int *q, int depth)
{
	if (uni2 == NULL) return;

	if (buffer)
	{
		DEBUG(5,("%s%06x smb_io_unistr2 %s\n", tab_depth(depth), *q, desc));
		depth++;

		buf_align(buf, q);
		
		buf_uint32("uni_max_len", depth, buf, q, io, &(uni2->uni_max_len));
		buf_uint32("undoc      ", depth, buf, q, io, &(uni2->undoc      ));
		buf_uint32("uni_str_len", depth, buf, q, io, &(uni2->uni_str_len));

		/* oops! XXXX maybe issue a warning that this is happening... */
		if (uni2->uni_max_len > MAX_UNISTRLEN) uni2->uni_max_len = MAX_UNISTRLEN;
		if (uni2->uni_str_len > MAX_UNISTRLEN) uni2->uni_str_len = MAX_UNISTRLEN;

		/* buffer advanced by indicated length of string
		   NOT by searching for null-termination */
		buf_unistr2(True, "buffer     ", depth, buf, q, io, uni2);
	}
	else
	{
		DEBUG(5,("%s%06x smb_io_unistr2 - NULL %s\n", tab_depth(depth), *q, desc));
		bzero(uni2, sizeof(*uni2));
	}
}

/*******************************************************************
creates a DOM_SID2 structure.
********************************************************************/
void make_dom_str_sid(DOM_STR_SID *sid, char *sid_str)
{
	int len_sid_str = strlen(sid_str);

	sid->type = 0x5;
	sid->undoc = 0;
	make_uni_hdr2(&(sid->hdr), len_sid_str, len_sid_str, 0);
	make_unistr  (&(sid->str), sid_str);
}

/*******************************************************************
reads or writes a DOM_STR_SID structure.
********************************************************************/
void smb_io_dom_str_sid(char *desc, BOOL io, DOM_STR_SID *sid, struct mem_buffer *buf, int *q, int depth)
{
	if (sid == NULL) return;

	DEBUG(5,("%s%06x smb_io_dom_str_sid %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	/* should be value 5, so enforce it */
	sid->type = 5;

	/* should be value 0, so enforce it */
	sid->undoc = 0;

	buf_uint32("type ", depth, buf, q, io, &(sid->type ));
	buf_uint32("undoc", depth, buf, q, io, &(sid->undoc));

	smb_io_unihdr2("", io, &(sid->hdr), buf, q, depth);
	smb_io_unistr ("", io, &(sid->str), buf, q, depth);
}

/*******************************************************************
creates a DOM_RID2 structure.
********************************************************************/
void make_dom_rid2(DOM_RID2 *rid2, uint32 rid)
{
	rid2->type    = 0x5;
	rid2->undoc   = 0x5;
	rid2->rid     = rid;
	rid2->rid_idx = 0;
}

/*******************************************************************
reads or writes a DOM_RID2 structure.
********************************************************************/
void smb_io_dom_rid2(char *desc, BOOL io, DOM_RID2 *rid2, struct mem_buffer *buf, int *q, int depth)
{
	if (rid2 == NULL) return;

	DEBUG(5,("%s%06x smb_io_dom_rid2 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	/* should be value 5, so enforce it */
	rid2->type = 5;

	/* should be value 5, so enforce it */
	rid2->undoc = 5;

	buf_uint32("type   ", depth, buf, q, io, &(rid2->type));
	buf_uint32("undoc  ", depth, buf, q, io, &(rid2->undoc   ));
	buf_uint32("rid    ", depth, buf, q, io, &(rid2->rid     ));
	buf_uint32("rid_idx", depth, buf, q, io, &(rid2->rid_idx ));
}

/*******************************************************************
creates a DOM_RID3 structure.
********************************************************************/
void make_dom_rid3(DOM_RID3 *rid3, uint32 rid)
{
	rid3->rid      = rid;
	rid3->type1    = 0x1;
	rid3->ptr_type = 0x1; /* non-zero, basically. */
	rid3->type2    = 0x1;
}

/*******************************************************************
reads or writes a DOM_RID3 structure.
********************************************************************/
void smb_io_dom_rid3(char *desc, BOOL io, DOM_RID3 *rid3, struct mem_buffer *buf, int *q, int depth)
{
	if (rid3 == NULL) return;

	DEBUG(5,("%s%06x smb_io_dom_rid3 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("rid     ", depth, buf, q, io, &(rid3->rid     ));
	buf_uint32("type1   ", depth, buf, q, io, &(rid3->type1   ));
	buf_uint32("ptr_type", depth, buf, q, io, &(rid3->ptr_type));
	buf_uint32("type2   ", depth, buf, q, io, &(rid3->type2   ));
}

/*******************************************************************
creates a DOM_RID4 structure.
********************************************************************/
void make_dom_rid4(DOM_RID4 *rid4, uint16 unknown, uint16 attr, uint32 rid)
{
	rid4->unknown = unknown;
	rid4->attr    = attr;
	rid4->rid     = rid;
}

/*******************************************************************
reads or writes a DOM_RID4 structure.
********************************************************************/
void smb_io_dom_rid4(char *desc, BOOL io, DOM_RID4 *rid4, struct mem_buffer *buf, int *q, int depth)
{
	if (rid4 == NULL) return;

	DEBUG(5,("%s%06x smb_io_dom_rid4. XXXX !check size of unknown! XXXX %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("unknown", depth, buf, q, io, &(rid4->unknown));
	buf_uint16("attr   ", depth, buf, q, io, &(rid4->attr   ));
	buf_uint32("rid    ", depth, buf, q, io, &(rid4->rid    ));
}

/*******************************************************************
makes a SAM_STR1 structure.
********************************************************************/
void make_sam_str1(SAM_STR1 *sam, char *sam_acct, char *sam_name, char *sam_desc)
{
	int len_sam_acct = sam_acct != NULL ? strlen(sam_acct) : 0;
	int len_sam_name = sam_name != NULL ? strlen(sam_name) : 0;
	int len_sam_desc = sam_desc != NULL ? strlen(sam_desc) : 0;

	if (sam == NULL) return;

	DEBUG(5,("make_sam_str1: %d\n", __LINE__));

	make_unistr2(&(sam->uni_acct_name), sam_acct, len_sam_acct);
	make_unistr2(&(sam->uni_full_name), sam_name, len_sam_name);
	make_unistr2(&(sam->uni_acct_desc), sam_desc, len_sam_desc);
}

/*******************************************************************
reads or writes a SAM_STR1 structure.
********************************************************************/
void smb_io_sam_str1(char *desc, BOOL io, SAM_STR1 *sam, uint32 acct_buf, uint32 name_buf, uint32 desc_buf, struct mem_buffer *buf, int *q, int depth)
{
	if (sam == NULL) return;

	DEBUG(5,("%s%06x smb_io_sam_str1 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_unistr2("unistr2", io, &(sam->uni_acct_name), acct_buf, buf, q, depth); /* account name unicode string */
	smb_io_unistr2("unistr2", io, &(sam->uni_full_name), name_buf, buf, q, depth); /* full name unicode string */
	smb_io_unistr2("unistr2", io, &(sam->uni_acct_desc), desc_buf, buf, q, depth); /* account description unicode string */
}

/*******************************************************************
makes a SAM_ENTRY1 structure.
********************************************************************/
void make_sam_entry1(SAM_ENTRY1 *sam, uint32 user_idx, struct smb_passwd *pass,
				char *sam_full, char *sam_desc)
{
	char *sam_name = pass->smb_name;

	int len_sam_name = sam_name ? strlen(sam_name) : 0;
	int len_sam_full = sam_full ? strlen(sam_full) : 0;
	int len_sam_desc = sam_desc ? strlen(sam_desc) : 0;

	if (sam == NULL) return;

	DEBUG(5,("make_sam_entry2: %d\n", __LINE__));

	sam->user_idx = user_idx;
	sam->rid_user = pass->smb_userid;
	sam->acb_info = pass->acct_ctrl;
	sam->pad      = 0;

	make_uni_hdr(&(sam->hdr_acct_name), len_sam_name, len_sam_name, sam_name != NULL);
	make_uni_hdr(&(sam->hdr_user_name), len_sam_full, len_sam_full, sam_full != NULL);
	make_uni_hdr(&(sam->hdr_user_desc), len_sam_desc, len_sam_desc, sam_desc != NULL);
}

/*******************************************************************
reads or writes a SAM_ENTRY1 structure.
********************************************************************/
void smb_io_sam_entry1(char *desc, BOOL io, SAM_ENTRY1 *sam, struct mem_buffer *buf, int *q, int depth)
{
	if (sam == NULL) return;

	DEBUG(5,("%s%06x smb_io_sam_entry2 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("user_idx ", depth, buf, q, io, &(sam->user_idx ));

	buf_uint32("rid_user ", depth, buf, q, io, &(sam->rid_user ));
	buf_uint16("acb_info ", depth, buf, q, io, &(sam->acb_info ));
	buf_uint16("pad      ", depth, buf, q, io, &(sam->pad      ));

	smb_io_unihdr("unihdr", io, &(sam->hdr_acct_name), buf, q, depth); /* account name unicode string header */
	smb_io_unihdr("unihdr", io, &(sam->hdr_user_name), buf, q, depth); /* account name unicode string header */
	smb_io_unihdr("unihdr", io, &(sam->hdr_user_desc), buf, q, depth); /* account name unicode string header */
}

/*******************************************************************
makes a SAM_STR2 structure.
********************************************************************/
void make_sam_str2(SAM_STR2 *sam, char *sam_acct, char *sam_desc)
{
	int len_sam_acct = sam_acct != NULL ? strlen(sam_acct) : 0;
	int len_sam_desc = sam_desc != NULL ? strlen(sam_desc) : 0;

	if (sam == NULL) return;

	DEBUG(5,("make_sam_str2: %d\n", __LINE__));

	make_unistr2(&(sam->uni_srv_name), sam_acct, len_sam_acct);
	make_unistr2(&(sam->uni_srv_desc), sam_desc, len_sam_desc);
}

/*******************************************************************
reads or writes a SAM_STR2 structure.
********************************************************************/
void smb_io_sam_str2(char *desc, BOOL io, SAM_STR2 *sam, uint32 acct_buf, uint32 desc_buf, struct mem_buffer *buf, int *q, int depth)
{
	if (sam == NULL) return;

	DEBUG(5,("%s%06x smb_io_sam_str2 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_unistr2("unistr2", io, &(sam->uni_srv_name), acct_buf, buf, q, depth); /* account name unicode string */
	smb_io_unistr2("unistr2", io, &(sam->uni_srv_desc), desc_buf, buf, q, depth); /* account description unicode string */
}

/*******************************************************************
makes a SAM_ENTRY2 structure.
********************************************************************/
void make_sam_entry2(SAM_ENTRY2 *sam, uint32 user_idx, struct smb_passwd *pass,
				char *sam_desc)
{
	char *sam_name = pass->smb_name;

	int len_sam_name = sam_name ? strlen(sam_name) : 0;
	int len_sam_desc = sam_desc ? strlen(sam_desc) : 0;

	if (sam == NULL) return;

	DEBUG(5,("make_sam_entry2: %d\n", __LINE__));

	sam->user_idx = user_idx;
	sam->rid_user = pass->smb_userid;
	sam->acb_info = pass->acct_ctrl;
	sam->pad      = 0;

	make_uni_hdr(&(sam->hdr_srv_name), len_sam_name, len_sam_name, sam_name != NULL);
	make_uni_hdr(&(sam->hdr_srv_desc), len_sam_desc, len_sam_desc, sam_desc != NULL);
}

/*******************************************************************
reads or writes a SAM_ENTRY2 structure.
********************************************************************/
void smb_io_sam_entry2(char *desc, BOOL io, SAM_ENTRY2 *sam, struct mem_buffer *buf, int *q, int depth)
{
	if (sam == NULL) return;

	DEBUG(5,("%s%06x smb_io_sam_entry2 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("user_idx ", depth, buf, q, io, &(sam->user_idx ));

	buf_uint32("rid_user ", depth, buf, q, io, &(sam->rid_user ));
	buf_uint16("acb_info ", depth, buf, q, io, &(sam->acb_info ));
	buf_uint16("pad      ", depth, buf, q, io, &(sam->pad      ));

	smb_io_unihdr("unihdr", io, &(sam->hdr_srv_name), buf, q, depth); /* account name unicode string header */
	smb_io_unihdr("unihdr", io, &(sam->hdr_srv_desc), buf, q, depth); /* account name unicode string header */
}

/*******************************************************************
makes a SAM_STR3 structure.
********************************************************************/
void make_sam_str3(SAM_STR3 *sam, char *grp_acct, char *grp_desc)
{
	int len_grp_acct = strlen(grp_acct);
	int len_grp_desc = strlen(grp_desc);

	if (sam == NULL) return;

	DEBUG(5,("make_sam_str3: %d\n", __LINE__));

	make_unistr2(&(sam->uni_grp_name), grp_acct, len_grp_acct);
	make_unistr2(&(sam->uni_grp_desc), grp_desc, len_grp_desc);
}

/*******************************************************************
reads or writes a SAM_STR3 structure.
********************************************************************/
void smb_io_sam_str3(char *desc, BOOL io, SAM_STR3 *sam, uint32 acct_buf, uint32 desc_buf, struct mem_buffer *buf, int *q, int depth)
{
	if (sam == NULL) return;

	DEBUG(5,("%s%06x smb_io_sam_str3 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	smb_io_unistr2("unistr2", io, &(sam->uni_grp_name), acct_buf, buf, q, depth); /* account name unicode string */
	smb_io_unistr2("unistr2", io, &(sam->uni_grp_desc), desc_buf, buf, q, depth); /* account description unicode string */
}

/*******************************************************************
makes a SAM_ENTRY3 structure.
********************************************************************/
void make_sam_entry3(SAM_ENTRY3 *sam, uint32 grp_idx, struct smb_passwd *pass,
				char *grp_desc)
{
	char *grp_name = pass->smb_name;

	int len_grp_name = grp_name ? strlen(grp_name) : 0;
	int len_grp_desc = grp_desc ? strlen(grp_desc) : 0;

	if (sam == NULL) return;

	DEBUG(5,("make_sam_entry3: %d\n", __LINE__));

	sam->grp_idx = grp_idx;
	sam->rid_grp = pass->smb_userid;
	sam->attr    = 0x07; /* group rid attributes - gets ignored by nt 4.0 */

	make_uni_hdr(&(sam->hdr_grp_name), len_grp_name, len_grp_name, grp_name != NULL);
	make_uni_hdr(&(sam->hdr_grp_desc), len_grp_desc, len_grp_desc, grp_desc != NULL);
}

/*******************************************************************
reads or writes a SAM_ENTRY3 structure.
********************************************************************/
void smb_io_sam_entry3(char *desc, BOOL io, SAM_ENTRY3 *sam, struct mem_buffer *buf, int *q, int depth)
{
	if (sam == NULL) return;

	DEBUG(5,("%s%06x smb_io_sam_entry3 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint32("grp_idx", depth, buf, q, io, &(sam->grp_idx));

	buf_uint32("rid_grp", depth, buf, q, io, &(sam->rid_grp));
	buf_uint32("attr   ", depth, buf, q, io, &(sam->attr   ));

	smb_io_unihdr("unihdr", io, &(sam->hdr_grp_name), buf, q, depth); /* account name unicode string header */
	smb_io_unihdr("unihdr", io, &(sam->hdr_grp_desc), buf, q, depth); /* account name unicode string header */
}

/*******************************************************************
makes a SAM_ENTRY structure.
********************************************************************/
void make_sam_entry(SAM_ENTRY *sam, char *sam_name, uint32 rid)
{
	int len_sam_name = strlen(sam_name);

	if (sam == NULL) return;

	DEBUG(5,("make_sam_entry: %d\n", __LINE__));

	sam->rid = rid;
	make_uni_hdr(&(sam->hdr_name), len_sam_name, len_sam_name, sam_name != NULL);
}

/*******************************************************************
reads or writes a SAM_ENTRY structure.
********************************************************************/
void smb_io_sam_entry(char *desc, BOOL io, SAM_ENTRY *sam, struct mem_buffer *buf, int *q, int depth)
{
	if (sam == NULL) return;

	DEBUG(5,("%s%06x smb_io_sam_entry %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	buf_uint32("rid", depth, buf, q, io, &(sam->rid ));
	smb_io_unihdr("unihdr", io, &(sam->hdr_name), buf, q, depth); /* account name unicode string header */
}

/*******************************************************************
makes a DOM_CLNT_SRV structure.
********************************************************************/
void make_clnt_srv(DOM_CLNT_SRV *log, char *logon_srv, char *comp_name)
{
	if (log == NULL) return;

	DEBUG(5,("make_clnt_srv: %d\n", __LINE__));

	if (logon_srv != NULL)
	{
		log->undoc_buffer = 1;
		make_unistr2(&(log->uni_logon_srv), logon_srv, strlen(logon_srv));
	}
	else
	{
		log->undoc_buffer = 0;
	}

	if (comp_name != NULL)
	{
		log->undoc_buffer2 = 1;
		make_unistr2(&(log->uni_comp_name), comp_name, strlen(comp_name));
	}
	else
	{
		log->undoc_buffer2 = 0;
	}
}

/*******************************************************************
reads or writes a DOM_CLNT_SRV structure.
********************************************************************/
void smb_io_clnt_srv(char *desc, BOOL io, DOM_CLNT_SRV *log, struct mem_buffer *buf, int *q, int depth)
{
	if (log == NULL) return;

	DEBUG(5,("%s%06x smb_io_clnt_srv %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("undoc_buffer ", depth, buf, q, io, &(log->undoc_buffer ));
	if (log->undoc_buffer != 0)
	{
		smb_io_unistr2("unistr2", io, &(log->uni_logon_srv), log->undoc_buffer, buf, q, depth);
	}

	buf_align(buf, q);

	buf_uint32("undoc_buffer2", depth, buf, q, io, &(log->undoc_buffer2));
	if (log->undoc_buffer2 != 0)
	{
		smb_io_unistr2("unistr2", io, &(log->uni_comp_name), log->undoc_buffer2, buf, q, depth);
	}
}

/*******************************************************************
makes a DOM_LOG_INFO structure.
********************************************************************/
void make_log_info(DOM_LOG_INFO *log, char *logon_srv, char *acct_name,
		uint16 sec_chan, char *comp_name)
{
	if (log == NULL) return;

	DEBUG(5,("make_log_info %d\n", __LINE__));

	log->undoc_buffer = 1;

	make_unistr2(&(log->uni_logon_srv), logon_srv, strlen(logon_srv));
	make_unistr2(&(log->uni_acct_name), acct_name, strlen(acct_name));

	log->sec_chan = sec_chan;

	make_unistr2(&(log->uni_comp_name), comp_name, strlen(comp_name));
}

/*******************************************************************
reads or writes a DOM_LOG_INFO structure.
********************************************************************/
void smb_io_log_info(char *desc, BOOL io, DOM_LOG_INFO *log, struct mem_buffer *buf, int *q, int depth)
{
	if (log == NULL) return;

	DEBUG(5,("%s%06x smb_io_log_info %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("undoc_buffer", depth, buf, q, io, &(log->undoc_buffer));

	smb_io_unistr2("unistr2", io, &(log->uni_logon_srv), True, buf, q, depth);
	smb_io_unistr2("unistr2", io, &(log->uni_acct_name), True, buf, q, depth);

	buf_uint16("sec_chan", depth, buf, q, io, &(log->sec_chan));

	smb_io_unistr2("unistr2", io, &(log->uni_comp_name), True, buf, q, depth);
}

/*******************************************************************
reads or writes a DOM_CHAL structure.
********************************************************************/
void smb_io_chal(char *desc, BOOL io, DOM_CHAL *chal, struct mem_buffer *buf, int *q, int depth)
{
	if (chal == NULL) return;

	DEBUG(5,("%s%06x smb_io_chal %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint8s (False, "data", depth, buf, q, io, chal->data, 8);
}

/*******************************************************************
reads or writes a DOM_CRED structure.
********************************************************************/
void smb_io_cred(char *desc, BOOL io, DOM_CRED *cred, struct mem_buffer *buf, int *q, int depth)
{
	if (cred == NULL) return;

	DEBUG(5,("%s%06x smb_io_cred %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	smb_io_chal ("", io, &(cred->challenge), buf, q, depth);
	smb_io_utime("", io, &(cred->timestamp), buf, q, depth);
}

/*******************************************************************
makes a DOM_CLNT_INFO2 structure.
********************************************************************/
void make_clnt_info2(DOM_CLNT_INFO2 *clnt,
				char *logon_srv, char *comp_name,
				DOM_CRED *clnt_cred)
{
	if (clnt == NULL) return;

	DEBUG(5,("make_clnt_info: %d\n", __LINE__));

	make_clnt_srv(&(clnt->login), logon_srv, comp_name);

	if (clnt_cred != NULL)
	{
		clnt->ptr_cred = 1;
		memcpy(&(clnt->cred), clnt_cred, sizeof(clnt->cred));
	}
	else
	{
		clnt->ptr_cred = 0;
	}
}

/*******************************************************************
reads or writes a DOM_CLNT_INFO2 structure.
********************************************************************/
void smb_io_clnt_info2(char *desc, BOOL io, DOM_CLNT_INFO2 *clnt, struct mem_buffer *buf, int *q, int depth)
{
	if (clnt == NULL) return;

	DEBUG(5,("%s%06x smb_io_clnt_info2 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	smb_io_clnt_srv("", io, &(clnt->login), buf, q, depth);

	buf_align(buf, q);
	
	buf_uint32("ptr_cred", depth, buf, q, io, &(clnt->ptr_cred));
	smb_io_cred    ("", io, &(clnt->cred ), buf, q, depth);
}

/*******************************************************************
makes a DOM_CLNT_INFO structure.
********************************************************************/
void make_clnt_info(DOM_CLNT_INFO *clnt,
		char *logon_srv, char *acct_name,
		uint16 sec_chan, char *comp_name,
				DOM_CRED *cred)
{
	if (clnt == NULL || cred == NULL) return;

	DEBUG(5,("make_clnt_info\n"));

	make_log_info(&(clnt->login), logon_srv, acct_name, sec_chan, comp_name);
	memcpy(&(clnt->cred), cred, sizeof(clnt->cred));
}

/*******************************************************************
reads or writes a DOM_CLNT_INFO structure.
********************************************************************/
void smb_io_clnt_info(char *desc, BOOL io, DOM_CLNT_INFO *clnt, struct mem_buffer *buf, int *q, int depth)
{
	if (clnt == NULL) return;

	DEBUG(5,("%s%06x smb_io_clnt_info %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	smb_io_log_info("", io, &(clnt->login), buf, q, depth);
	smb_io_cred    ("", io, &(clnt->cred ), buf, q, depth);
}

/*******************************************************************
makes a DOM_LOGON_ID structure.
********************************************************************/
void make_logon_id(DOM_LOGON_ID *log, uint32 log_id_low, uint32 log_id_high)
{
	if (log == NULL) return;

	DEBUG(5,("make_logon_id: %d\n", __LINE__));

	log->low  = log_id_low;
	log->high = log_id_high;
}

/*******************************************************************
reads or writes a DOM_LOGON_ID structure.
********************************************************************/
void smb_io_logon_id(char *desc, BOOL io, DOM_LOGON_ID *log, struct mem_buffer *buf, int *q, int depth)
{
	if (log == NULL) return;

	DEBUG(5,("%s%06x smb_io_logon_id %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("low ", depth, buf, q, io, &(log->low ));
	buf_uint32("high", depth, buf, q, io, &(log->high));
}

/*******************************************************************
makes an ARC4_OWF structure.
********************************************************************/
void make_arc4_owf(ARC4_OWF *hash, uint8 data[16])
{
	if (hash == NULL) return;

	DEBUG(5,("make_arc4_owf: %d\n", __LINE__));
	
	if (data != NULL)
	{
		memcpy(hash->data, data, sizeof(hash->data));
	}
	else
	{
		bzero(hash->data, sizeof(hash->data));
	}
}

/*******************************************************************
reads or writes an ARC4_OWF structure.
********************************************************************/
void smb_io_arc4_owf(char *desc, BOOL io, ARC4_OWF *hash, struct mem_buffer *buf, int *q, int depth)
{
	if (hash == NULL) return;

	DEBUG(5,("%s%06x smb_io_arc4_owf %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint8s (False, "data", depth, buf, q, io, hash->data, 16);
}

/*******************************************************************
makes a DOM_ID_INFO_1 structure.
********************************************************************/
void make_id_info1(DOM_ID_INFO_1 *id, char *domain_name,
				uint32 param_ctrl, uint32 log_id_low, uint32 log_id_high,
				char *user_name, char *wksta_name,
				char sess_key[16],
				unsigned char lm_cypher[16], unsigned char nt_cypher[16])
{
	int len_domain_name = strlen(domain_name);
	int len_user_name   = strlen(user_name  );
	int len_wksta_name  = strlen(wksta_name );

	unsigned char arc4_lm_owf[16];
	unsigned char arc4_nt_owf[16];

	if (id == NULL) return;

	DEBUG(5,("make_id_info1: %d\n", __LINE__));

	id->ptr_id_info1 = 1;

	make_uni_hdr(&(id->hdr_domain_name), len_domain_name, len_domain_name, 4);

	id->param_ctrl = param_ctrl;
	make_logon_id(&(id->logon_id), log_id_low, log_id_high);

	make_uni_hdr(&(id->hdr_user_name  ), len_user_name  , len_user_name  , 4);
	make_uni_hdr(&(id->hdr_wksta_name ), len_wksta_name , len_wksta_name , 4);

#ifdef USE_ARCFOUR

	if (lm_cypher && nt_cypher)
	{
		void arcfour(uint8 key[16], uint8 out[16], uint8 in[16]);
		unsigned char arc4_key[16];
#ifdef DEBUG_PASSWORD
		DEBUG(100,("lm cypher:"));
		dump_data(100, lm_cypher, 16);

		DEBUG(100,("nt cypher:"));
		dump_data(100, nt_cypher, 16);
#endif

		memset(arc4_key, 0, 16);
		memcpy(arc4_key, sess_key, 16);

		arcfour(arc4_key, arc4_lm_owf, lm_cypher);
		arcfour(arc4_key, arc4_nt_owf, nt_cypher);

#ifdef DEBUG_PASSWORD
		DEBUG(100,("arcfour encrypt of lm owf password:"));
		dump_data(100, arc4_lm_owf, 16);

		DEBUG(100,("arcfour encrypt of nt owf password:"));
		dump_data(100, arc4_nt_owf, 16);
#endif
		/* set up pointers to cypher blocks */
		lm_cypher = arc4_lm_owf;
		nt_cypher = arc4_nt_owf;
	}

#else

	if (lm_cypher)
	{
		/* oops.  can only send what-ever-it-is direct */
		memcpy(arc4_lm_owf, lm_cypher, 16);
		lm_cypher = arc4_lm_owf;
	}
	if (nt_cypher)
	{
		/* oops.  can only send what-ever-it-is direct */
		memcpy(arc4_nt_owf, nt_cypher, 16);
		nt_cypher = arc4_nt_owf;
	}

#endif

	make_arc4_owf(&(id->arc4_lm_owf), lm_cypher);
	make_arc4_owf(&(id->arc4_nt_owf), nt_cypher);

	make_unistr2(&(id->uni_domain_name), domain_name, len_domain_name);
	make_unistr2(&(id->uni_user_name  ), user_name  , len_user_name  );
	make_unistr2(&(id->uni_wksta_name ), wksta_name , len_wksta_name );
}

/*******************************************************************
reads or writes an DOM_ID_INFO_1 structure.
********************************************************************/
void smb_io_id_info1(char *desc, BOOL io, DOM_ID_INFO_1 *id, struct mem_buffer *buf, int *q, int depth)
{
	if (id == NULL) return;

	DEBUG(5,("%s%06x smb_io_id_info1 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("ptr_id_info1", depth, buf, q, io, &(id->ptr_id_info1));

	if (id->ptr_id_info1 != 0)
	{
		smb_io_unihdr("unihdr", io, &(id->hdr_domain_name), buf, q, depth);

		buf_uint32("param_ctrl", depth, buf, q, io, &(id->param_ctrl));
		smb_io_logon_id("", io, &(id->logon_id), buf, q, depth);

		smb_io_unihdr("unihdr", io, &(id->hdr_user_name  ), buf, q, depth);
		smb_io_unihdr("unihdr", io, &(id->hdr_wksta_name ), buf, q, depth);

		smb_io_arc4_owf("", io, &(id->arc4_lm_owf), buf, q, depth);
		smb_io_arc4_owf("", io, &(id->arc4_nt_owf), buf, q, depth);

		smb_io_unistr2("unistr2", io, &(id->uni_domain_name), id->hdr_domain_name.buffer, buf, q, depth);
		smb_io_unistr2("unistr2", io, &(id->uni_user_name  ), id->hdr_user_name.buffer, buf, q, depth);
		smb_io_unistr2("unistr2", io, &(id->uni_wksta_name ), id->hdr_wksta_name.buffer, buf, q, depth);
	}
}

/*******************************************************************
makes a DOM_SAM_INFO structure.
********************************************************************/
void make_sam_info(DOM_SAM_INFO *sam,
				char *logon_srv, char *comp_name, DOM_CRED *clnt_cred,
				DOM_CRED *rtn_cred, uint16 logon_level, uint16 switch_value,
				DOM_ID_INFO_1 *id1, uint16 switch_value2)
{
	if (sam == NULL) return;

	DEBUG(5,("make_sam_info: %d\n", __LINE__));

	make_clnt_info2(&(sam->client), logon_srv, comp_name, clnt_cred);

	if (rtn_cred != NULL)
	{
		sam->ptr_rtn_cred = 1;
		memcpy(&(sam->rtn_cred), rtn_cred, sizeof(sam->rtn_cred));
	}
	else
	{
		sam->ptr_rtn_cred = 0;
	}

	sam->logon_level  = logon_level;
	sam->switch_value = switch_value;

	switch (sam->switch_value)
	{
		case 1:
		{
			sam->auth.id1 = id1;
			break;
		}
		default:
		{
			/* PANIC! */
			DEBUG(4,("make_sam_info: unknown switch_value!\n"));
			break;
		}
	}

	sam->switch_value2 = switch_value2;
}

/*******************************************************************
reads or writes a DOM_SAM_INFO structure.
********************************************************************/
void smb_io_sam_info(char *desc, BOOL io, DOM_SAM_INFO *sam, struct mem_buffer *buf, int *q, int depth)
{
	if (sam == NULL) return;

	DEBUG(5,("%s%06x smb_io_sam_info %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	smb_io_clnt_info2("", io, &(sam->client  ), buf, q, depth);

	buf_uint32("ptr_rtn_cred ", depth, buf, q, io, &(sam->ptr_rtn_cred));
	smb_io_cred      ("", io, &(sam->rtn_cred), buf, q, depth);

	buf_uint16("logon_level  ", depth, buf, q, io, &(sam->logon_level ));
	buf_uint16("switch_value ", depth, buf, q, io, &(sam->switch_value));

	switch (sam->switch_value)
	{
		case 1:
		{
			smb_io_id_info1("", io, sam->auth.id1, buf, q, depth);
			break;
		}
		default:
		{
			/* PANIC! */
			DEBUG(4,("smb_io_sam_info: unknown switch_value!\n"));
			break;
		}
	}

	buf_uint16("switch_value2", depth, buf, q, io, &(sam->switch_value2));
}

/*******************************************************************
reads or writes a DOM_GID structure.
********************************************************************/
void smb_io_gid(char *desc, BOOL io, DOM_GID *gid, struct mem_buffer *buf, int *q, int depth)
{
	if (gid == NULL) return;

	DEBUG(5,("%s%06x smb_io_gid %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("g_rid", depth, buf, q, io, &(gid->g_rid));
	buf_uint32("attr ", depth, buf, q, io, &(gid->attr ));
}

/*******************************************************************
creates an RPC_HDR structure.
********************************************************************/
void make_rpc_hdr(RPC_HDR *hdr, enum RPC_PKT_TYPE pkt_type, uint8 frag,
				uint32 call_id, int data_len)
{
	if (hdr == NULL) return;

	hdr->major        = 5;               /* RPC version 5 */
	hdr->minor        = 0;               /* minor version 0 */
	hdr->pkt_type     = pkt_type;        /* RPC packet type */
	hdr->frag         = frag;            /* first frag + last frag */
	hdr->pack_type    = 0x10;            /* packed data representation */
	hdr->frag_len     = data_len;        /* fragment length, fill in later */
	hdr->auth_len     = 0;               /* authentication length */
	hdr->call_id      = call_id;         /* call identifier - match incoming RPC */
}

/*******************************************************************
reads or writes an RPC_HDR structure.
********************************************************************/
void smb_io_rpc_hdr(char *desc, BOOL io, RPC_HDR *rpc, struct mem_buffer *buf, int *q, int depth)
{
	if (rpc == NULL) return;

	DEBUG(5,("%s%06x smb_io_rpc_hdr %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_uint8 ("major     ", depth, buf, q, io, &(rpc->major));
	buf_uint8 ("minor     ", depth, buf, q, io, &(rpc->minor));
	buf_uint8 ("pkt_type  ", depth, buf, q, io, &(rpc->pkt_type));
	buf_uint8 ("frag      ", depth, buf, q, io, &(rpc->frag));
	buf_uint32("pack_type ", depth, buf, q, io, &(rpc->pack_type));
	buf_uint16("frag_len  ", depth, buf, q, io, &(rpc->frag_len));
	buf_uint16("auth_len  ", depth, buf, q, io, &(rpc->auth_len));
	buf_uint32("call_id   ", depth, buf, q, io, &(rpc->call_id));
}

/*******************************************************************
creates an RPC_IFACE structure.
********************************************************************/
void make_rpc_iface(RPC_IFACE *ifc, char data[16], uint32 version)
{
	if (ifc == NULL || data == NULL) return;

	memcpy(ifc->data, data, sizeof(ifc->data)); /* 16 bytes of number */
	ifc->version = version; /* the interface number */
}

/*******************************************************************
reads or writes an RPC_IFACE structure.
********************************************************************/
void smb_io_rpc_iface(char *desc, BOOL io, RPC_IFACE *ifc, struct mem_buffer *buf, int *q, int depth)
{
	if (ifc == NULL) return;

	DEBUG(5,("%s%06x smb_io_rpc_iface %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint8s (False, "data   ", depth, buf, q, io, ifc->data, sizeof(ifc->data));
	buf_uint32 (       "version", depth, buf, q, io, &(ifc->version));
}

/*******************************************************************
creates an RPC_ADDR_STR structure.
********************************************************************/
void make_rpc_addr_str(RPC_ADDR_STR *str, char *name)
{
	if (str == NULL || name == NULL) return;

	str->len = strlen(name) + 1;
	fstrcpy(str->str, name);
}

/*******************************************************************
reads or writes an RPC_ADDR_STR structure.
********************************************************************/
void smb_io_rpc_addr_str(char *desc, BOOL io, RPC_ADDR_STR *str, struct mem_buffer *buf, int *q, int depth)
{
	if (str == NULL) return;

	DEBUG(5,("%s%06x smb_io_rpc_addr_str %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);

	buf_uint16 (      "len", depth, buf, q, io, &(str->len));
	buf_uint8s (True, "str", depth, buf, q, io, (uchar*)str->str, str->len);
}

/*******************************************************************
creates an RPC_HDR_BBA structure.
********************************************************************/
void make_rpc_hdr_bba(RPC_HDR_BBA *bba, uint16 max_tsize, uint16 max_rsize, uint32 assoc_gid)
{
	if (bba == NULL) return;

	bba->max_tsize = max_tsize; /* maximum transmission fragment size (0x1630) */
	bba->max_rsize = max_rsize; /* max receive fragment size (0x1630) */   
	bba->assoc_gid = assoc_gid; /* associated group id (0x0) */ 
}

/*******************************************************************
reads or writes an RPC_HDR_BBA structure.
********************************************************************/
void smb_io_rpc_hdr_bba(char *desc, BOOL io, RPC_HDR_BBA *rpc, struct mem_buffer *buf, int *q, int depth)
{
	if (rpc == NULL) return;

	DEBUG(5,("%s%06x smb_io_rpc_hdr_bba %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_uint16("max_tsize", depth, buf, q, io, &(rpc->max_tsize));
	buf_uint16("max_rsize", depth, buf, q, io, &(rpc->max_rsize));
	buf_uint32("assoc_gid", depth, buf, q, io, &(rpc->assoc_gid));
}

/*******************************************************************
creates an RPC_HDR_RB structure.
********************************************************************/
void make_rpc_hdr_rb(RPC_HDR_RB *rpc, 
				uint16 max_tsize, uint16 max_rsize, uint32 assoc_gid,
				uint32 num_elements, uint16 context_id, uint8 num_syntaxes,
				RPC_IFACE *abstract, RPC_IFACE *transfer)
{
	if (rpc == NULL) return;

	make_rpc_hdr_bba(&(rpc->bba), max_tsize, max_rsize, assoc_gid);

	rpc->num_elements = num_elements ; /* the number of elements (0x1) */
	rpc->context_id   = context_id   ; /* presentation context identifier (0x0) */
	rpc->num_syntaxes = num_syntaxes ; /* the number of syntaxes (has always been 1?)(0x1) */

	/* num and vers. of interface client is using */
	memcpy(&(rpc->abstract), abstract, sizeof(rpc->abstract));

	/* num and vers. of interface to use for replies */
	memcpy(&(rpc->transfer), transfer, sizeof(rpc->transfer));
}

/*******************************************************************
reads or writes an RPC_HDR_RB structure.
********************************************************************/
void smb_io_rpc_hdr_rb(char *desc, BOOL io, RPC_HDR_RB *rpc, struct mem_buffer *buf, int *q, int depth)
{
	if (rpc == NULL) return;

	DEBUG(5,("%s%06x smb_io_rpc_hdr_bba %s\n", tab_depth(depth), *q, desc));
	depth++;

	smb_io_rpc_hdr_bba("", io, &(rpc->bba), buf, q, depth);

	buf_uint32("num_elements", depth, buf, q, io, &(rpc->num_elements));
	buf_uint16("context_id  ", depth, buf, q, io, &(rpc->context_id  ));
	buf_uint8 ("num_syntaxes", depth, buf, q, io, &(rpc->num_syntaxes));

	smb_io_rpc_iface("", io, &(rpc->abstract), buf, q, depth);
	smb_io_rpc_iface("", io, &(rpc->transfer), buf, q, depth);
}

/*******************************************************************
creates an RPC_RESULTS structure.

lkclXXXX only one reason at the moment!

********************************************************************/
void make_rpc_results(RPC_RESULTS *res, 
				uint8 num_results, uint16 result, uint16 reason)
{
	if (res == NULL) return;

	res->num_results = num_results; /* the number of results (0x01) */
	res->result      = result     ;  /* result (0x00 = accept) */
	res->reason      = reason     ;  /* reason (0x00 = no reason specified) */
}

/*******************************************************************
reads or writes an RPC_RESULTS structure.

lkclXXXX only one reason at the moment!

********************************************************************/
void smb_io_rpc_results(char *desc, BOOL io, RPC_RESULTS *res, struct mem_buffer *buf, int *q, int depth)
{
	if (res == NULL) return;

	DEBUG(5,("%s%06x smb_io_rpc_results %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint8 ("num_results", depth, buf, q, io, &(res->num_results));

	buf_align(buf, q);
	
	buf_uint16("result     ", depth, buf, q, io, &(res->result     ));
	buf_uint16("reason     ", depth, buf, q, io, &(res->reason     ));
}

/*******************************************************************
creates an RPC_HDR_BA structure.

lkclXXXX only one reason at the moment!

********************************************************************/
void make_rpc_hdr_ba(RPC_HDR_BA *rpc, 
				uint16 max_tsize, uint16 max_rsize, uint32 assoc_gid,
				char *pipe_addr,
				uint8 num_results, uint16 result, uint16 reason,
				RPC_IFACE *transfer)
{
	if (rpc == NULL || transfer == NULL || pipe_addr == NULL) return;

	make_rpc_hdr_bba (&(rpc->bba ), max_tsize, max_rsize, assoc_gid);
	make_rpc_addr_str(&(rpc->addr), pipe_addr);
	make_rpc_results (&(rpc->res ), num_results, result, reason);

	/* the transfer syntax from the request */
	memcpy(&(rpc->transfer), transfer, sizeof(rpc->transfer));
}

/*******************************************************************
reads or writes an RPC_HDR_BA structure.
********************************************************************/
void smb_io_rpc_hdr_ba(char *desc, BOOL io, RPC_HDR_BA *rpc, struct mem_buffer *buf, int *q, int depth)
{
	if (rpc == NULL) return;

	DEBUG(5,("%s%06x smb_io_rpc_hdr_ba %s\n", tab_depth(depth), *q, desc));
	depth++;

	smb_io_rpc_hdr_bba ("", io, &(rpc->bba)     , buf, q, depth);
	smb_io_rpc_addr_str("", io, &(rpc->addr)    , buf, q, depth);
	smb_io_rpc_results ("", io, &(rpc->res)     , buf, q, depth);
	smb_io_rpc_iface   ("", io, &(rpc->transfer), buf, q, depth);
}

/*******************************************************************
makes an LSA_OBJ_ATTR structure.
********************************************************************/
void make_obj_attr(LSA_OBJ_ATTR *attr, uint32 attributes, uint32 sec_qos)
{
	if (attr == NULL) return;

	DEBUG(5,("make_obj_attr\n"));

	attr->len = 0x18; /* length of object attribute block, in bytes */
	attr->ptr_root_dir = 0;
	attr->ptr_obj_name = 0;
	attr->attributes = attributes;
	attr->ptr_sec_desc = 0;
	attr->sec_qos = sec_qos;
}

/*******************************************************************
reads or writes an LSA_OBJ_ATTR structure.
********************************************************************/
void smb_io_obj_attr(char *desc, BOOL io, LSA_OBJ_ATTR *attr, struct mem_buffer *buf, int *q, int depth)
{
	int start;

	if (attr == NULL) return;

	DEBUG(5,("%s%06x smb_io_obj_attr %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	start = *q;

	/* these pointers had _better_ be zero, because we don't know
	   what they point to!
	 */
	buf_uint32("len"         , depth, buf, q, io, &(attr->len         )); /* 0x18 - length (in bytes) inc. the length field. */
	buf_uint32("ptr_root_dir", depth, buf, q, io, &(attr->ptr_root_dir)); /* 0 - root directory (pointer) */
	buf_uint32("ptr_obj_name", depth, buf, q, io, &(attr->ptr_obj_name)); /* 0 - object name (pointer) */
	buf_uint32("attributes"  , depth, buf, q, io, &(attr->attributes  )); /* 0 - attributes (undocumented) */
	buf_uint32("ptr_sec_desc", depth, buf, q, io, &(attr->ptr_sec_desc)); /* 0 - security descriptior (pointer) */
	buf_uint32("sec_qos"     , depth, buf, q, io, &(attr->sec_qos     )); /* 0 - security quality of service */

	if (attr->len != *q - start)
	{
		DEBUG(3,("smb_io_obj_attr: length %lx does not match size %lx\n",
		         attr->len, *q - start));
	}
}

/*******************************************************************
creates an RPC_HDR_RR structure.
********************************************************************/
void make_rpc_hdr_rr(RPC_HDR_RR *hdr, enum RPC_PKT_TYPE pkt_type,
				uint32 call_id, int data_len, uint8 opnum)
{
	if (hdr == NULL) return;

	/* frag is FIRST_FRAG | LAST_FRAG.  lkclXXXX must define these */
	make_rpc_hdr(&(hdr->hdr), pkt_type, 0x03, call_id, data_len);

	hdr->alloc_hint   = data_len - 0x18; /* allocation hint */
	hdr->context_id   = 0;               /* presentation context identifier */
	hdr->cancel_count = 0;               /* cancel count */
	hdr->opnum        = opnum;           /* opnum */
	hdr->reserved     = 0;               /* 0 - reserved */
}

/*******************************************************************
reads or writes an RPC_HDR_RR structure.
********************************************************************/
void smb_io_rpc_hdr_rr(char *desc, BOOL io, RPC_HDR_RR *rpc, struct mem_buffer *buf, int *q, int depth)
{
	if (rpc == NULL) return;

	DEBUG(5,("%s%06x smb_io_rpc_hdr_rr %s\n", tab_depth(depth), *q, desc));
	depth++;

	smb_io_rpc_hdr("", io, &(rpc->hdr), buf, q, depth);

	buf_uint32("alloc_hint", depth, buf, q, io, &(rpc->alloc_hint));
	buf_uint8 ("context_id", depth, buf, q, io, &(rpc->context_id));
	buf_uint8 ("cancel_ct ", depth, buf, q, io, &(rpc->cancel_count));
	buf_uint8 ("opnum     ", depth, buf, q, io, &(rpc->opnum));
	buf_uint8 ("reserved  ", depth, buf, q, io, &(rpc->reserved));
}

/*******************************************************************
reads or writes an POLICY_HND structure.
********************************************************************/
void smb_io_pol_hnd(char *desc, BOOL io, POLICY_HND *pol, struct mem_buffer *buf, int *q, int depth)
{
	if (pol == NULL) return;

	DEBUG(5,("%s%06x smb_io_pol_hnd %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint8s (False, "data", depth, buf, q, io, pol->data, POL_HND_SIZE);
}

/*******************************************************************
reads or writes a dom query structure.
********************************************************************/
void smb_io_dom_query_3(char *desc, BOOL io, DOM_QUERY_3 *d_q, struct mem_buffer *buf, int *q, int depth)
{
	smb_io_dom_query("", io, d_q, buf, q, depth);
}

/*******************************************************************
reads or writes a dom query structure.
********************************************************************/
void smb_io_dom_query_5(char *desc, BOOL io, DOM_QUERY_3 *d_q, struct mem_buffer *buf, int *q, int depth)
{
	smb_io_dom_query("", io, d_q, buf, q, depth);
}

/*******************************************************************
reads or writes a dom query structure.
********************************************************************/
void smb_io_dom_query(char *desc, BOOL io, DOM_QUERY *d_q, struct mem_buffer *buf, int *q, int depth)
{
	if (d_q == NULL) return;

	DEBUG(5,("%s%06x smb_io_dom_query %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint16("uni_dom_max_len", depth, buf, q, io, &(d_q->uni_dom_max_len)); /* domain name string length * 2 */
	buf_uint16("uni_dom_str_len", depth, buf, q, io, &(d_q->uni_dom_str_len)); /* domain name string length * 2 */

	buf_uint32("buffer_dom_name", depth, buf, q, io, &(d_q->buffer_dom_name)); /* undocumented domain name string buffer pointer */
	buf_uint32("buffer_dom_sid ", depth, buf, q, io, &(d_q->buffer_dom_sid )); /* undocumented domain SID string buffer pointer */

	smb_io_unistr2("unistr2", io, &(d_q->uni_domain_name), d_q->buffer_dom_name, buf, q, depth); /* domain name (unicode string) */

	if (d_q->buffer_dom_sid != 0)
	{
		smb_io_dom_sid2("", io, &(d_q->dom_sid), buf, q, depth); /* domain SID */
	}
	else
	{
		bzero(&(d_q->dom_sid), sizeof(d_q->dom_sid));
	}
}

/*******************************************************************
reads or writes a DOM_R_REF structure.
********************************************************************/
void smb_io_dom_r_ref(char *desc, BOOL io, DOM_R_REF *r_r, struct mem_buffer *buf, int *q, int depth)
{
	int i;

	DEBUG(5,("%s%06x smb_io_dom_r_ref %s\n", tab_depth(depth), *q, desc));
	depth++;

	if (r_r == NULL) return;

	buf_align(buf, q);
	
	buf_uint32("undoc_buffer   ", depth, buf, q, io, &(r_r->undoc_buffer)); /* undocumented buffer pointer. */
	buf_uint32("num_ref_doms_1 ", depth, buf, q, io, &(r_r->num_ref_doms_1)); /* num referenced domains? */
	buf_uint32("buffer_dom_name", depth, buf, q, io, &(r_r->buffer_dom_name)); /* undocumented domain name buffer pointer. */
	buf_uint32("max_entries    ", depth, buf, q, io, &(r_r->max_entries)); /* 32 - max number of entries */
	buf_uint32("num_ref_doms_2 ", depth, buf, q, io, &(r_r->num_ref_doms_2)); /* 4 - num referenced domains? */

	smb_io_unihdr2("", io, &(r_r->hdr_dom_name), buf, q, depth); /* domain name unicode string header */

	for (i = 0; i < r_r->num_ref_doms_1-1; i++)
	{
		smb_io_unihdr2("", io, &(r_r->hdr_ref_dom[i]), buf, q, depth);
	}

	smb_io_unistr("", io, &(r_r->uni_dom_name), buf, q, depth); /* domain name unicode string */

	for (i = 0; i < r_r->num_ref_doms_2; i++)
	{
		smb_io_dom_sid2("", io, &(r_r->ref_dom[i]), buf, q, depth); /* referenced domain SIDs */
	}
}

/*******************************************************************
reads or writes a DOM_NAME structure.
********************************************************************/
void smb_io_dom_name(char *desc, BOOL io, DOM_NAME *name, struct mem_buffer *buf, int *q, int depth)
{
	if (name == NULL) return;

	DEBUG(5,("%s%06x smb_io_dom_name %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("uni_str_len", depth, buf, q, io, &(name->uni_str_len));

	/* don't know if len is specified by uni_str_len member... */
	/* assume unicode string is unicode-null-terminated, instead */

	smb_io_unistr("", io, &(name->str), buf, q, depth);
}


/*******************************************************************
reads or writes a structure.
********************************************************************/
void smb_io_neg_flags(char *desc, BOOL io, NEG_FLAGS *neg, struct mem_buffer *buf, int *q, int depth)
{
	if (neg == NULL) return;

	DEBUG(5,("%s%06x smb_io_neg_flags %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("neg_flags", depth, buf, q, io, &(neg->neg_flags));
}

/*******************************************************************
creates a NETLOGON_INFO_3 structure.
********************************************************************/
void make_netinfo_3(NETLOGON_INFO_3 *info, uint32 flags, uint32 logon_attempts)
{
	info->flags          = flags;
	info->logon_attempts = logon_attempts;
	info->reserved_1     = 0x0;
	info->reserved_2     = 0x0;
	info->reserved_3     = 0x0;
	info->reserved_4     = 0x0;
	info->reserved_5     = 0x0;
}

/*******************************************************************
reads or writes a NETLOGON_INFO_3 structure.
********************************************************************/
void smb_io_netinfo_3(char *desc, BOOL io, NETLOGON_INFO_3 *info, struct mem_buffer *buf, int *q, int depth)
{
	if (info == NULL) return;

	DEBUG(5,("%s%06x smb_io_netinfo_3 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("flags         ", depth, buf, q, io, &(info->flags         ));
	buf_uint32("logon_attempts", depth, buf, q, io, &(info->logon_attempts));
	buf_uint32("reserved_1    ", depth, buf, q, io, &(info->reserved_1    ));
	buf_uint32("reserved_2    ", depth, buf, q, io, &(info->reserved_2    ));
	buf_uint32("reserved_3    ", depth, buf, q, io, &(info->reserved_3    ));
	buf_uint32("reserved_4    ", depth, buf, q, io, &(info->reserved_4    ));
	buf_uint32("reserved_5    ", depth, buf, q, io, &(info->reserved_5    ));
}


/*******************************************************************
creates a NETLOGON_INFO_1 structure.
********************************************************************/
void make_netinfo_1(NETLOGON_INFO_1 *info, uint32 flags, uint32 pdc_status)
{
	info->flags      = flags;
	info->pdc_status = pdc_status;
}

/*******************************************************************
reads or writes a NETLOGON_INFO_1 structure.
********************************************************************/
void smb_io_netinfo_1(char *desc, BOOL io, NETLOGON_INFO_1 *info, struct mem_buffer *buf, int *q, int depth)
{
	if (info == NULL) return;

	DEBUG(5,("%s%06x smb_io_netinfo_1 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("flags     ", depth, buf, q, io, &(info->flags     ));
	buf_uint32("pdc_status", depth, buf, q, io, &(info->pdc_status));
}

/*******************************************************************
creates a NETLOGON_INFO_2 structure.
********************************************************************/
void make_netinfo_2(NETLOGON_INFO_2 *info, uint32 flags, uint32 pdc_status,
				uint32 tc_status, char *trusted_dc_name)
{
	int len_dc_name = strlen(trusted_dc_name);
	info->flags      = flags;
	info->pdc_status = pdc_status;
	info->ptr_trusted_dc_name = 1;
	info->tc_status  = tc_status;

	if (trusted_dc_name != NULL)
	{
		make_unistr2(&(info->uni_trusted_dc_name), trusted_dc_name, len_dc_name);
	}
	else
	{
		make_unistr2(&(info->uni_trusted_dc_name), "", 1);
	}
}

/*******************************************************************
reads or writes a NETLOGON_INFO_2 structure.
********************************************************************/
void smb_io_netinfo_2(char *desc, BOOL io, NETLOGON_INFO_2 *info, struct mem_buffer *buf, int *q, int depth)
{
	if (info == NULL) return;

	DEBUG(5,("%s%06x smb_io_netinfo_2 %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32("flags              ", depth, buf, q, io, &(info->flags              ));
	buf_uint32("pdc_status         ", depth, buf, q, io, &(info->pdc_status         ));
	buf_uint32("ptr_trusted_dc_name", depth, buf, q, io, &(info->ptr_trusted_dc_name));
	buf_uint32("tc_status          ", depth, buf, q, io, &(info->tc_status          ));

	if (info->ptr_trusted_dc_name != 0)
	{
		smb_io_unistr2("unistr2", io, &(info->uni_trusted_dc_name), info->ptr_trusted_dc_name, buf, q, depth);
	}

	buf_align(buf, q);
}

/*******************************************************************
reads or writes a LOGON_HRS structure.
********************************************************************/
void smb_io_logon_hrs(char *desc, BOOL io, LOGON_HRS *hrs, struct mem_buffer *buf, int *q, int depth)
{
	if (hrs == NULL) return;

	DEBUG(5,("%s%06x smb_io_logon_hrs %s\n", tab_depth(depth), *q, desc));
	depth++;

	buf_align(buf, q);
	
	buf_uint32 (       "len  ", depth, buf, q, io, &(hrs->len ));

	if (hrs->len > 64)
	{
		DEBUG(5,("smb_io_logon_hrs: truncating length\n"));
		hrs->len = 64;
	}

	buf_uint8s (False, "hours", depth, buf, q, io, hrs->hours, hrs->len);
}

