
/* 
 *  Unix SMB/Netbios implementation.
 *  Version 1.9.
 *  RPC Pipe client / server routines
 *  Copyright (C) Andrew Tridgell              1992-1997,
 *  Copyright (C) Luke Kenneth Casson Leighton 1996-1997,
 *  Copyright (C) Paul Ashton                       1997.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "../includes.h"
#include "../trans2.h"
#include "../nterr.h"

extern int DEBUGLEVEL;


/***************************************************************************
lsa_reply_open_policy
 ***************************************************************************/
static void lsa_reply_open_policy(struct mem_buffer *rdata, int *rdata_off)
{
	int i;
	LSA_R_OPEN_POL r_o;

	/* set up the LSA QUERY INFO response */
	bzero(&(r_o.pol.data), POL_HND_SIZE);
	for (i = 4; i < POL_HND_SIZE; i++)
	{
		r_o.pol.data[i] = i;
	}
	r_o.status = 0x0;

	/* store the response in the SMB stream */
	lsa_io_r_open_pol("", False, &r_o, rdata, rdata_off, 0);
}

/***************************************************************************
make_dom_query
 ***************************************************************************/
static void make_dom_query(DOM_QUERY *d_q, char *dom_name, char *dom_sid)
{
	int domlen = strlen(dom_name);

	d_q->uni_dom_max_len = domlen * 2;
	d_q->uni_dom_str_len = domlen * 2;

	d_q->buffer_dom_name = 4; /* domain buffer pointer */
	d_q->buffer_dom_sid  = 2; /* domain sid pointer */

	/* this string is supposed to be character short */
	make_unistr2(&(d_q->uni_domain_name), dom_name, domlen);

	make_dom_sid2(&(d_q->dom_sid), dom_sid);
}

/***************************************************************************
lsa_reply_query_info
 ***************************************************************************/
static void lsa_reply_enum_trust_dom(LSA_Q_ENUM_TRUST_DOM *q_e,
				struct mem_buffer *rdata, int *rdata_off,
				uint32 enum_context, char *dom_name, char *dom_sid)
{
	LSA_R_ENUM_TRUST_DOM r_e;

	/* set up the LSA QUERY INFO response */
	make_r_enum_trust_dom(&r_e, enum_context, dom_name, dom_sid,
	      dom_name != NULL ? 0x0 : 0x80000000 | NT_STATUS_UNABLE_TO_FREE_VM);

	/* store the response in the SMB stream */
	lsa_io_r_enum_trust_dom("", False, &r_e, rdata, rdata_off, 0);
}

/***************************************************************************
lsa_reply_query_info
 ***************************************************************************/
static void lsa_reply_query_info(LSA_Q_QUERY_INFO *q_q, struct mem_buffer *rdata, int *rdata_off,
				char *dom_name, char *dom_sid)
{
	LSA_R_QUERY_INFO r_q;

	/* set up the LSA QUERY INFO response */

	r_q.undoc_buffer = 0x22000000; /* bizarre */
	r_q.info_class = q_q->info_class;

	make_dom_query(&r_q.dom.id5, dom_name, dom_sid);

	r_q.status = 0x0;

	/* store the response in the SMB stream */
	lsa_io_r_query("", False, &r_q, rdata, rdata_off, 0);
}

/***************************************************************************
make_dom_ref

 pretty much hard-coded choice of "other" sids, unfortunately...

 ***************************************************************************/
static void make_dom_ref(DOM_R_REF *ref,
				char *dom_name, char *dom_sid,
				char *other_sid1, char *other_sid2, char *other_sid3)
{
	int len_dom_name   = strlen(dom_name);
	int len_other_sid1 = strlen(other_sid1);
	int len_other_sid2 = strlen(other_sid2);
	int len_other_sid3 = strlen(other_sid3);

	ref->undoc_buffer = 1;
	ref->num_ref_doms_1 = 4;
	ref->buffer_dom_name = 1;
	ref->max_entries = 32;
	ref->num_ref_doms_2 = 4;

	make_uni_hdr2(&(ref->hdr_dom_name  ), len_dom_name  , len_dom_name  , 0);
	make_uni_hdr2(&(ref->hdr_ref_dom[0]), len_other_sid1, len_other_sid1, 0);
	make_uni_hdr2(&(ref->hdr_ref_dom[1]), len_other_sid2, len_other_sid2, 0);
	make_uni_hdr2(&(ref->hdr_ref_dom[2]), len_other_sid3, len_other_sid3, 0);

	if (dom_name != NULL)
	{
		make_unistr(&(ref->uni_dom_name), dom_name);
	}

	make_dom_sid2(&(ref->ref_dom[0]), dom_sid   );
	make_dom_sid2(&(ref->ref_dom[1]), other_sid1);
	make_dom_sid2(&(ref->ref_dom[2]), other_sid2);
	make_dom_sid2(&(ref->ref_dom[3]), other_sid3);
}

/***************************************************************************
make_reply_lookup_rids
 ***************************************************************************/
static void make_reply_lookup_rids(LSA_R_LOOKUP_RIDS *r_l,
				int num_entries, uint32 dom_rids[MAX_LOOKUP_SIDS],
				char *dom_name, char *dom_sid,
				char *other_sid1, char *other_sid2, char *other_sid3)
{
	int i;

	make_dom_ref(&(r_l->dom_ref), dom_name, dom_sid,
	             other_sid1, other_sid2, other_sid3);

	r_l->num_entries = num_entries;
	r_l->undoc_buffer = 1;
	r_l->num_entries2 = num_entries;

	for (i = 0; i < num_entries; i++)
	{
		make_dom_rid2(&(r_l->dom_rid[i]), dom_rids[i]);
	}

	r_l->num_entries3 = num_entries;
}

/***************************************************************************
make_reply_lookup_sids
 ***************************************************************************/
static void make_reply_lookup_sids(LSA_R_LOOKUP_SIDS *r_l,
				int num_entries, fstring dom_sids[MAX_LOOKUP_SIDS],
				char *dom_name, char *dom_sid,
				char *other_sid1, char *other_sid2, char *other_sid3)
{
	int i;

	make_dom_ref(&(r_l->dom_ref), dom_name, dom_sid,
	             other_sid1, other_sid2, other_sid3);

	r_l->num_entries = num_entries;
	r_l->undoc_buffer = 1;
	r_l->num_entries2 = num_entries;

	for (i = 0; i < num_entries; i++)
	{
		make_dom_str_sid(&(r_l->str_sid[i]), dom_sids[i]);
	}

	r_l->num_entries3 = num_entries;
}

/***************************************************************************
lsa_reply_lookup_sids
 ***************************************************************************/
static void lsa_reply_lookup_sids(struct mem_buffer *rdata, int *rdata_off,
				int num_entries, fstring dom_sids[MAX_LOOKUP_SIDS],
				char *dom_name, char *dom_sid,
				char *other_sid1, char *other_sid2, char *other_sid3)
{
	LSA_R_LOOKUP_SIDS r_l;

	/* set up the LSA Lookup SIDs response */
	make_reply_lookup_sids(&r_l, num_entries, dom_sids,
				dom_name, dom_sid, other_sid1, other_sid2, other_sid3);
	r_l.status = 0x0;

	/* store the response in the SMB stream */
	lsa_io_r_lookup_sids("", False, &r_l, rdata, rdata_off, 0);
}

/***************************************************************************
lsa_reply_lookup_rids
 ***************************************************************************/
static void lsa_reply_lookup_rids(struct mem_buffer *rdata, int *rdata_off,
				int num_entries, uint32 dom_rids[MAX_LOOKUP_SIDS],
				char *dom_name, char *dom_sid,
				char *other_sid1, char *other_sid2, char *other_sid3)
{
	LSA_R_LOOKUP_RIDS r_l;

	/* set up the LSA Lookup RIDs response */
	make_reply_lookup_rids(&r_l, num_entries, dom_rids,
				dom_name, dom_sid, other_sid1, other_sid2, other_sid3);
	r_l.status = 0x0;

	/* store the response in the SMB stream */
	lsa_io_r_lookup_rids("", False, &r_l, rdata, rdata_off, 0);
}

/***************************************************************************
api_lsa_open_policy
 ***************************************************************************/
static void api_lsa_open_policy( int uid, struct mem_buffer *data, int *data_off,
                             struct mem_buffer *rdata, int *rdata_off )
{
	LSA_Q_OPEN_POL q_o;

	/* grab the server, object attributes and desired access flag...*/
	lsa_io_q_open_pol("", True, &q_o, data, data_off, 0);

	/* lkclXXXX having decoded it, ignore all fields in the open policy! */

	/* return a 20 byte policy handle */
	lsa_reply_open_policy(rdata, rdata_off);
}

/***************************************************************************
api_lsa_enum_trust_dom
 ***************************************************************************/
static void api_lsa_enum_trust_dom( int uid, struct mem_buffer *data, int *data_off,
                                    struct mem_buffer *rdata, int *rdata_off )
{
	LSA_Q_ENUM_TRUST_DOM q_e;

	/* grab the enum trust domain context etc. */
	lsa_io_q_enum_trust_dom("", True, &q_e, data, data_off, 0);

	/* construct reply.  return status is always 0x0 */
	lsa_reply_enum_trust_dom(&q_e, rdata, rdata_off,
	                                      0, NULL, NULL);
}

/***************************************************************************
api_lsa_query_info
 ***************************************************************************/
static void api_lsa_query_info( int uid, struct mem_buffer *data, int *data_off,
                                struct mem_buffer *rdata, int *rdata_off )
{
	LSA_Q_QUERY_INFO q_i;
	pstring dom_name;
	pstring dom_sid;

	/* grab the info class and policy handle */
	lsa_io_q_query("", True, &q_i, data, data_off, 0);

	pstrcpy(dom_name, lp_workgroup());
	pstrcpy(dom_sid , lp_domain_sid());

	/* construct reply.  return status is always 0x0 */
	lsa_reply_query_info(&q_i, rdata, rdata_off, 
									 dom_name, dom_sid);
}

/***************************************************************************
api_lsa_lookup_sids
 ***************************************************************************/
static void api_lsa_lookup_sids( int uid, struct mem_buffer *data, int *data_off,
                                 struct mem_buffer *rdata, int *rdata_off )
{
	int i;
	LSA_Q_LOOKUP_SIDS q_l;
	pstring dom_name;
	pstring dom_sid;
	fstring dom_sids[MAX_LOOKUP_SIDS];

	/* grab the info class and policy handle */
	lsa_io_q_lookup_sids("", True, &q_l, data, data_off, 0);

	pstrcpy(dom_name, lp_workgroup());
	pstrcpy(dom_sid , lp_domain_sid());

	/* convert received SIDs to strings, so we can do them. */
	for (i = 0; i < q_l.num_entries; i++)
	{
		fstrcpy(dom_sids[i], dom_sid_to_string(&(q_l.dom_sids[i].sid)));
	}

	/* construct reply.  return status is always 0x0 */
	lsa_reply_lookup_sids(rdata, rdata_off,
	            q_l.num_entries, dom_sids, /* text-converted SIDs */
				dom_name, dom_sid, /* domain name, domain SID */
				"S-1-1", "S-1-3", "S-1-5"); /* the three other SIDs */
}

/***************************************************************************
api_lsa_lookup_names
 ***************************************************************************/
static void api_lsa_lookup_names( int uid, struct mem_buffer *data, int *data_off,
                                  struct mem_buffer *rdata, int *rdata_off )
{
	int i;
	LSA_Q_LOOKUP_RIDS q_l;
	pstring dom_name;
	pstring dom_sid;
	uint32 dom_rids[MAX_LOOKUP_SIDS];
	uint32 dummy_g_rid;

	/* grab the info class and policy handle */
	lsa_io_q_lookup_rids("", True, &q_l, data, data_off, 0);

	pstrcpy(dom_name, lp_workgroup());
	pstrcpy(dom_sid , lp_domain_sid());

	/* convert received RIDs to strings, so we can do them. */
	for (i = 0; i < q_l.num_entries; i++)
	{
		char *user_name = unistr2(q_l.lookup_name[i].str.buffer);
		if (!name_to_rid(user_name, &dom_rids[i], &dummy_g_rid))
		{
			/* WHOOPS!  we should really do something about this... */
			dom_rids[i] = 0;
		}
	}

	/* construct reply.  return status is always 0x0 */
	lsa_reply_lookup_rids(rdata, rdata_off,
	            q_l.num_entries, dom_rids, /* text-converted SIDs */
				dom_name, dom_sid, /* domain name, domain SID */
				"S-1-1", "S-1-3", "S-1-5"); /* the three other SIDs */
}

/***************************************************************************
 api_lsa_close
 ***************************************************************************/
static void api_lsa_close( int uid, struct mem_buffer *data, int *data_off,
                                  struct mem_buffer *rdata, int *rdata_off)
{
	char *q = &(rdata->data[*rdata_off]);

	SIVAL(q, 0, 0); q += 4;
	SIVAL(q, 0, 0); q += 4;
	SIVAL(q, 0, 0); q += 4;
	SIVAL(q, 0, 0); q += 4;
	SIVAL(q, 0, 0); q += 4;
	SIVAL(q, 0, 0); q += 4;

	(*rdata_off) += 24;
}

/***************************************************************************
 api_lsa_open_secret
 ***************************************************************************/
static void api_lsa_open_secret( int uid, struct mem_buffer *data, int *data_off,
                                  struct mem_buffer *rdata, int *rdata_off)
{
	char *q = &(rdata->data[*rdata_off]);

	SIVAL(q, 0, 0); q += 4;
	SIVAL(q, 0, 0); q += 4;
	SIVAL(q, 0, 0); q += 4;
	SIVAL(q, 0, 0); q += 4;
	SIVAL(q, 0, 0); q += 4;
	SIVAL(q, 0, 0xC0000000 | NT_STATUS_OBJECT_NAME_NOT_FOUND); q += 4;
	
	(*rdata_off) += 24;
}

/***************************************************************************
 \PIPE\ntlsa commands
 ***************************************************************************/
static struct api_struct api_lsa_cmds[] =
{
	{ "LSA_OPENPOLICY"      , LSA_OPENPOLICY      , api_lsa_open_policy    },
	{ "LSA_QUERYINFOPOLICY" , LSA_QUERYINFOPOLICY , api_lsa_query_info     },
	{ "LSA_ENUMTRUSTDOM"    , LSA_ENUMTRUSTDOM    , api_lsa_enum_trust_dom },
	{ "LSA_CLOSE"           , LSA_CLOSE           , api_lsa_close          },
	{ "LSA_OPENSECRET"      , LSA_OPENSECRET      , api_lsa_open_secret    },
	{ "LSA_LOOKUPSIDS"      , LSA_LOOKUPSIDS      , api_lsa_lookup_sids    },
	{ "LSA_LOOKUPNAMES"     , LSA_LOOKUPNAMES     , api_lsa_lookup_names   },
	{ NULL                  , 0                   , NULL                   }
};

/***************************************************************************
 api_ntLsarpcTNP
 ***************************************************************************/
BOOL api_ntlsa_rpc(int cnum,int uid,
				struct mem_buffer *data, struct mem_buffer *rdata)
{
	return api_rpcTNP("api_ntlsa_rpc", api_lsa_cmds, cnum, uid, data, rdata);
}

