/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   NT Domain Authentication SMB / MSRPC client
   Copyright (C) Andrew Tridgell 1994-1997
   Copyright (C) Luke Kenneth Casson Leighton 1996-1997
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/



#ifdef SYSLOG
#undef SYSLOG
#endif

#include "../includes.h"
#include "../nterr.h"

extern int DEBUGLEVEL;



/****************************************************************************
experimental samr tests
****************************************************************************/
BOOL do_samr_session_open(struct cli_state *cli, int t_idx, struct client_info *info)
{
	RPC_IFACE abstract;
	RPC_IFACE transfer;

	/******************** initialise ********************************/

	DEBUG(1,("do_samr_session_open: %d\n", __LINE__));

	/******************* open the \PIPE\SAMR file *****************/

	if ((info->dom.samr_fnum = cli_open(cli, t_idx, PIPE_SAMR, O_CREAT, DENY_NONE,
	                         NULL, NULL, NULL)) == 0xffff)
	{
		DEBUG(1,("do_samr_session_open: cli_open failed\n"));
		return False;
	}

	/**************** Set Named Pipe State ***************/
	if (!rpc_pipe_set_hnd_state(cli, t_idx, PIPE_SAMR, info->dom.samr_fnum, 0x4300))
	{
		DEBUG(1,("do_samr_session_open: pipe hnd state failed\n"));
		return False;
	}

	/******************* bind request on \PIPE\SAMR *****************/

	if (!rpc_pipe_bind(cli, t_idx, PIPE_SAMR, info->dom.samr_fnum, &abstract, &transfer))
	{
		DEBUG(1,("do_samr_session_open: rpc bind failed\n"));
		return False;
	}

	return True;
}


/****************************************************************************
close the \PIPE\samr session
****************************************************************************/
void do_samr_session_close(struct cli_state *cli, int t_idx, struct client_info *info)
{
	if (info->dom.samr_fnum != 0xffff)
	{
		cli_close(cli, t_idx, info->dom.samr_fnum, 0);
	}
}


/****************************************************************************
do a SAMR query user groups
****************************************************************************/
BOOL get_samr_query_usergroups(struct cli_state *cli, int t_idx, uint16 fnum, 
				POLICY_HND *pol_open_domain, uint32 user_rid,
				uint32 *num_groups, DOM_GID *gid)
{
	POLICY_HND pol_open_user;
	if (pol_open_domain == NULL || num_groups == NULL || gid == NULL) return False;

	/* send open domain (on user sid) */
	if (!do_samr_open_user(cli, t_idx, fnum,
				pol_open_domain,
				0x02011b, user_rid,
				&pol_open_user))
	{
		return False;
	}

	/* send user info query, level 0x15 */
	if (!do_samr_query_usergroups(cli, t_idx, fnum,
				&pol_open_user,
				num_groups, gid))
	{
		DEBUG(5,("do_samr_query_usergroups: error in query user groups\n"));
	}

	return do_samr_close(cli, t_idx, fnum, &pol_open_user);
}

/****************************************************************************
do a SAMR query user info, level 0x15
****************************************************************************/
BOOL get_samr_query_userinfo_15(struct cli_state *cli, int t_idx, uint16 fnum, 
				POLICY_HND *pol_open_domain, uint32 user_rid, SAM_USER_INFO_15 *usr)
{
	POLICY_HND pol_open_user;
	if (pol_open_domain == NULL || usr == NULL) return False;

	bzero(usr, sizeof(*usr));

	/* send open domain (on user sid) */
	if (!do_samr_open_user(cli, t_idx, fnum,
				pol_open_domain,
				0x02011b, user_rid,
				&pol_open_user))
	{
		return False;
	}

	/* send user info query, level 0x15 */
	if (!do_samr_query_userinfo(cli, t_idx, fnum,
				&pol_open_user,
				0x15, (void*)usr))
	{
		DEBUG(5,("do_samr_query_userinfo_15: error in query user info, level 0x15\n"));
	}

	return do_samr_close(cli, t_idx, fnum, &pol_open_user);
}

/****************************************************************************
do a SAMR enumerate users
****************************************************************************/
BOOL do_samr_enum_dom_users(struct cli_state *cli, int t_idx, uint16 fnum, 
				POLICY_HND *pol, uint16 num_entries, uint16 unk_0,
				uint16 acb_mask, uint16 unk_1, uint32 size,
				struct acct_info sam[MAX_SAM_ENTRIES],
				int *num_sam_users)
{
	struct mem_buffer data;
	struct mem_buffer rdata;
	struct mem_buffer rparams;
	int p = 0, r = 0;

	SAMR_Q_ENUM_DOM_USERS q_e;
    BOOL valid_pol = False;

	/* create and send a MSRPC command with api SAMR_ENUM_DOM_USERS */

	buf_init(&data   , 4, SAFETY_MARGIN);
	buf_init(&rdata  , 4, SAFETY_MARGIN);
	buf_init(&rparams, 4, SAFETY_MARGIN);

	buf_alloc(&data , 1024);

	DEBUG(4,("SAMR Enum SAM DB max size:%x\n", size));

	if (pol == NULL || sam == NULL || num_sam_users == NULL || fnum == 0xffff) return False;

	/* store the parameters */
	make_samr_q_enum_dom_users(&q_e, pol,
	                           num_entries, unk_0,
	                           acb_mask, unk_1, size);

	/* turn parameters into data stream */
	p = 0x18;
	samr_io_q_enum_dom_users("", False, &q_e, &data, &p, 0);

	data.data_used = p;

	/* create the request RPC_HDR_RR _after_ the main data: length is now known */
	create_rpc_request(&data, &r, get_rpc_call_id(), SAMR_ENUM_DOM_USERS, p);

	/* send the data on \PIPE\ */
	if (rpc_api_pipe(cli, t_idx, 0x0026, fnum, NULL, &data, &rparams, &(rdata)))
	{
		SAMR_R_ENUM_DOM_USERS r_e;
		RPC_HDR_RR hdr;
		int hdr_len;
		int pkt_len;

		DEBUG(5, ("rpc_api_pipe: return OK\n"));

		p = 0;

		smb_io_rpc_hdr_rr   ("", True, &hdr, &(rdata), &p, 0);
		if (p) buf_align(&(rdata), &p); /* oh, what a surprise */

		hdr_len = p;

		if (p && hdr_len != hdr.hdr.frag_len - hdr.alloc_hint)
		{
			/* header length not same as calculated header length */
			DEBUG(2,("samr_r_enum_dom_users: hdr_len %x != frag_len-alloc_hint %x\n",
			          hdr_len, hdr.hdr.frag_len - hdr.alloc_hint));
			p = 0;
		}

		if (p) samr_io_r_enum_dom_users("", True, &r_e, &(rdata), &p, 0);
		
		pkt_len = p;

		if (p && pkt_len != hdr.hdr.frag_len)
		{
			/* packet data size not same as reported fragment length */
			DEBUG(2,("do_samr_enum_dom_users: pkt_len %x != frag_len \n",
			                           pkt_len, hdr.hdr.frag_len));
			p = 0;
		}

		if (p && r_e.status != 0)
		{
			/* report error code */
			DEBUG(0,("SAMR_R_ENUM_DOM_USERS: %s\n", get_nt_error_msg(r_e.status)));
			p = 0;
		}

		if (p)
		{
			int i;
			int name_idx = 0;

			*num_sam_users = r_e.num_entries2;
			if (*num_sam_users > MAX_SAM_ENTRIES)
			{
				*num_sam_users = MAX_SAM_ENTRIES;
				DEBUG(2,("do_samr_enum_dom_users: sam user entries limited to %d\n",
				          *num_sam_users));
			}
			for (i = 0; i < *num_sam_users; i++)
			{
				sam[i].smb_userid = r_e.sam[i].rid;
				if (r_e.sam[i].hdr_name.buffer)
				{
					char *acct_name = unistrn2(r_e.uni_acct_name[name_idx].buffer,
					                           r_e.uni_acct_name[name_idx].uni_str_len);
					fstrcpy(sam[i].acct_name, acct_name);
					name_idx++;
				}
				else
				{
					bzero(sam[i].acct_name, sizeof(sam[i].acct_name));
				}
				DEBUG(5,("do_samr_enum_dom_users: idx: %4d rid: %8x acct: %s\n",
				          i, sam[i].smb_userid, sam[i].acct_name));
			}
			valid_pol = True;
		}
	}

	buf_free(&data   );
	buf_free(&rdata  );
	buf_free(&rparams);

	return valid_pol;
}

/****************************************************************************
do a SAMR Connect
****************************************************************************/
BOOL do_samr_connect(struct cli_state *cli, int t_idx, uint16 fnum, 
				char *srv_name, uint32 unknown_0,
				POLICY_HND *rtn_pol)
{
	struct mem_buffer data;
	struct mem_buffer rdata;
	struct mem_buffer rparams;
	int p = 0, r = 0;

	SAMR_Q_CONNECT q_o;
    BOOL valid_pol = False;

	/* create and send a MSRPC command with api SAMR_CONNECT */

	buf_init(&data   , 4, SAFETY_MARGIN);
	buf_init(&rdata  , 4, SAFETY_MARGIN);
	buf_init(&rparams, 4, SAFETY_MARGIN);

	buf_alloc(&data , 1024);

	DEBUG(4,("SAMR Open Policy server:%s undoc value:%x\n",
				srv_name, unknown_0));

	if (srv_name == NULL || rtn_pol == NULL || fnum == 0xffff) return False;

	/* store the parameters */
	make_samr_q_connect(&q_o, srv_name, unknown_0);

	/* turn parameters into data stream */
	p = 0x18;
	samr_io_q_connect("", False, &q_o,  &data, &p, 0);

	data.data_used = p;

	/* create the request RPC_HDR_RR _after_ the main data: length is now known */
	create_rpc_request(&data, &r, get_rpc_call_id(), SAMR_CONNECT, p);

	/* send the data on \PIPE\ */
	if (rpc_api_pipe(cli, t_idx, 0x0026, fnum, NULL, &data, &rparams, &(rdata)))
	{
		SAMR_R_CONNECT r_o;
		RPC_HDR_RR hdr;
		int hdr_len;
		int pkt_len;

		DEBUG(5, ("cli_api_pipe: return OK\n"));

		p = 0;

		smb_io_rpc_hdr_rr   ("", True, &hdr, &(rdata), &p, 0);
		if (p) buf_align(&(rdata), &p); /* oh, what a surprise */

		hdr_len = p;

		if (p && hdr_len != hdr.hdr.frag_len - hdr.alloc_hint)
		{
			/* header length not same as calculated header length */
			DEBUG(2,("samr_r_connect: hdr_len %x != frag_len-alloc_hint %x\n",
			          hdr_len, hdr.hdr.frag_len - hdr.alloc_hint));
			p = 0;
		}

		if (p) samr_io_r_connect("", True, &r_o, &(rdata), &p, 0);
		
		pkt_len = p;

		if (p && pkt_len != hdr.hdr.frag_len)
		{
			/* packet data size not same as reported fragment length */
			DEBUG(2,("do_samr_connect: pkt_len %x != frag_len \n",
			                           pkt_len, hdr.hdr.frag_len));
			p = 0;
		}

		if (p && r_o.status != 0)
		{
			/* report error code */
			DEBUG(0,("SAMR_R_CONNECT: %s\n", get_nt_error_msg(r_o.status)));
			p = 0;
		}

		if (p)
		{
			memcpy(rtn_pol, &r_o.pol, sizeof(r_o.pol));
			valid_pol = True;
		}
	}

	buf_free(&data   );
	buf_free(&rdata  );
	buf_free(&rparams);

	return valid_pol;
}

/****************************************************************************
do a SAMR Open User
****************************************************************************/
BOOL do_samr_open_user(struct cli_state *cli, int t_idx, uint16 fnum, 
				POLICY_HND *pol, uint32 unk_0, uint32 rid, 
				POLICY_HND *user_pol)
{
	struct mem_buffer data;
	struct mem_buffer rdata;
	struct mem_buffer rparams;
	int p = 0, r = 0;

	SAMR_Q_OPEN_USER q_o;
    BOOL valid_pol = False;

	/* create and send a MSRPC command with api SAMR_OPEN_USER */

	buf_init(&data   , 4, SAFETY_MARGIN);
	buf_init(&rdata  , 4, SAFETY_MARGIN);
	buf_init(&rparams, 4, SAFETY_MARGIN);

	buf_alloc(&data , 1024);

	DEBUG(4,("SAMR Open User.  unk_0: %08x RID:%x\n",
	          unk_0, rid));

	if (pol == NULL || user_pol == NULL || fnum == 0xffff) return False;

	/* store the parameters */
	make_samr_q_open_user(&q_o, pol, unk_0, rid);

	/* turn parameters into data stream */
	p = 0x18;
	samr_io_q_open_user("", False, &q_o,  &data, &p, 0);

	data.data_used = p;

	/* create the request RPC_HDR_RR _after_ the main data: length is now known */
	create_rpc_request(&data, &r, get_rpc_call_id(), SAMR_OPEN_USER, p);

	/* send the data on \PIPE\ */
	if (rpc_api_pipe(cli, t_idx, 0x0026, fnum, NULL, &data, &rparams, &(rdata)))
	{
		SAMR_R_OPEN_USER r_o;
		RPC_HDR_RR hdr;
		int hdr_len;
		int pkt_len;

		DEBUG(5, ("cli_api_pipe: return OK\n"));

		p = 0;

		smb_io_rpc_hdr_rr   ("", True, &hdr, &(rdata), &p, 0);
		if (p) buf_align(&(rdata), &p); /* oh, what a surprise */

		hdr_len = p;

		if (p && hdr_len != hdr.hdr.frag_len - hdr.alloc_hint)
		{
			/* header length not same as calculated header length */
			DEBUG(2,("samr_r_open_user: hdr_len %x != frag_len-alloc_hint %x\n",
			          hdr_len, hdr.hdr.frag_len - hdr.alloc_hint));
			p = 0;
		}

		if (p) samr_io_r_open_user("", True, &r_o, &(rdata), &p, 0);
		
		pkt_len = p;

		if (p && pkt_len != hdr.hdr.frag_len)
		{
			/* packet data size not same as reported fragment length */
			DEBUG(2,("do_samr_open_user: pkt_len %x != frag_len \n",
			                           pkt_len, hdr.hdr.frag_len));
			p = 0;
		}

		if (p && r_o.status != 0)
		{
			/* report error code */
			DEBUG(0,("SAMR_R_OPEN_USER: %s\n", get_nt_error_msg(r_o.status)));
			p = 0;
		}

		if (p)
		{
			memcpy(user_pol, &r_o.user_pol, sizeof(r_o.user_pol));
			valid_pol = True;
		}
	}

	buf_free(&data   );
	buf_free(&rdata  );
	buf_free(&rparams);

	return valid_pol;
}

/****************************************************************************
do a SAMR Open Domain
****************************************************************************/
BOOL do_samr_open_domain(struct cli_state *cli, int t_idx, uint16 fnum, 
				POLICY_HND *pol, uint32 rid, char *sid,
				POLICY_HND *rtn_pol)
{
	struct mem_buffer data;
	struct mem_buffer rdata;
	struct mem_buffer rparams;
	int p = 0, r = 0;

	SAMR_Q_OPEN_DOMAIN q_o;
    BOOL valid_pol = False;

	/* create and send a MSRPC command with api SAMR_OPEN_DOMAIN */

	buf_init(&data   , 4, SAFETY_MARGIN);
	buf_init(&rdata  , 4, SAFETY_MARGIN);
	buf_init(&rparams, 4, SAFETY_MARGIN);

	buf_alloc(&data , 1024);

	DEBUG(4,("SAMR Open Domain.  SID:%s RID:%x\n", sid, rid));

	if (pol == NULL || sid == NULL || rtn_pol == NULL || fnum == 0xffff) return False;

	/* store the parameters */
	make_samr_q_open_domain(&q_o, pol, rid, sid);

	/* turn parameters into data stream */
	p = 0x18;
	samr_io_q_open_domain("", False, &q_o,  &data, &p, 0);

	data.data_used = p;

	/* create the request RPC_HDR_RR _after_ the main data: length is now known */
	create_rpc_request(&data, &r, get_rpc_call_id(), SAMR_OPEN_DOMAIN, p);

	/* send the data on \PIPE\ */
	if (rpc_api_pipe(cli, t_idx, 0x0026, fnum, NULL, &data, &rparams, &(rdata)))
	{
		SAMR_R_OPEN_DOMAIN r_o;
		RPC_HDR_RR hdr;
		int hdr_len;
		int pkt_len;

		DEBUG(5, ("cli_api_pipe: return OK\n"));

		p = 0;

		smb_io_rpc_hdr_rr   ("", True, &hdr, &(rdata), &p, 0);
		if (p) buf_align(&(rdata), &p); /* oh, what a surprise */

		hdr_len = p;

		if (p && hdr_len != hdr.hdr.frag_len - hdr.alloc_hint)
		{
			/* header length not same as calculated header length */
			DEBUG(2,("samr_r_open_domain: hdr_len %x != frag_len-alloc_hint %x\n",
			          hdr_len, hdr.hdr.frag_len - hdr.alloc_hint));
			p = 0;
		}

		if (p) samr_io_r_open_domain("", True, &r_o, &(rdata), &p, 0);
		
		pkt_len = p;

		if (p && pkt_len != hdr.hdr.frag_len)
		{
			/* packet data size not same as reported fragment length */
			DEBUG(2,("do_samr_open_domain: pkt_len %x != frag_len \n",
			                           pkt_len, hdr.hdr.frag_len));
			p = 0;
		}

		if (p && r_o.status != 0)
		{
			/* report error code */
			DEBUG(0,("SAMR_R_OPEN_DOMAIN: %s\n", get_nt_error_msg(r_o.status)));
			p = 0;
		}

		if (p)
		{
			memcpy(rtn_pol, &r_o.pol, sizeof(r_o.pol));
			valid_pol = True;
		}
	}

	buf_free(&data   );
	buf_free(&rdata  );
	buf_free(&rparams);

	return valid_pol;
}

/****************************************************************************
do a SAMR Query Unknown 12
****************************************************************************/
BOOL do_samr_query_unknown_12(struct cli_state *cli, int t_idx, uint16 fnum, 
				POLICY_HND *pol, uint32 rid, uint32 num_gids, uint32 *gids,
				uint32 *num_aliases,
				fstring als_names    [MAX_LOOKUP_SIDS],
				uint32  num_als_users[MAX_LOOKUP_SIDS])
{
	struct mem_buffer data;
	struct mem_buffer rdata;
	struct mem_buffer rparams;
	int p = 0, r = 0;

	SAMR_Q_UNKNOWN_12 q_o;
    BOOL valid_query = False;

	/* create and send a MSRPC command with api SAMR_UNKNOWN_12 */

	buf_init(&data   , 4, SAFETY_MARGIN);
	buf_init(&rdata  , 4, SAFETY_MARGIN);
	buf_init(&rparams, 4, SAFETY_MARGIN);

	buf_alloc(&data , 1024);

	DEBUG(4,("SAMR Query Unknown 12.\n"));

	if (pol == NULL || rid == 0 || num_gids == 0 || gids == NULL ||
	    num_aliases == NULL || als_names == NULL || num_als_users == NULL ||
	    fnum == 0xffff) return False;

	/* store the parameters */
	make_samr_q_unknown_12(&q_o, pol, rid, num_gids, gids);

	/* turn parameters into data stream */
	p = 0x18;
	samr_io_q_unknown_12("", False, &q_o,  &data, &p, 0);

	data.data_used = p;

	/* create the request RPC_HDR_RR _after_ the main data: length is now known */
	create_rpc_request(&data, &r, get_rpc_call_id(), SAMR_UNKNOWN_12, p);

	/* send the data on \PIPE\ */
	if (rpc_api_pipe(cli, t_idx, 0x0026, fnum, NULL, &data, &rparams, &(rdata)))
	{
		SAMR_R_UNKNOWN_12 r_o;
		RPC_HDR_RR hdr;
		int hdr_len;
		int pkt_len;

		DEBUG(5, ("cli_api_pipe: return OK\n"));

		p = 0;

		smb_io_rpc_hdr_rr   ("", True, &hdr, &(rdata), &p, 0);
		if (p) buf_align(&(rdata), &p); /* oh, what a surprise */

		hdr_len = p;

		if (p && hdr_len != hdr.hdr.frag_len - hdr.alloc_hint)
		{
			/* header length not same as calculated header length */
			DEBUG(2,("samr_r_query_useraliases: hdr_len %x != frag_len-alloc_hint %x\n",
			          hdr_len, hdr.hdr.frag_len - hdr.alloc_hint));
			p = 0;
		}

		if (p) samr_io_r_unknown_12("", True, &r_o, &(rdata), &p, 0);
		
		pkt_len = p;

		if (p && pkt_len != hdr.hdr.frag_len)
		{
			/* packet data size not same as reported fragment length */
			DEBUG(2,("do_samr_unknown_12: pkt_len %x != frag_len \n",
			                           pkt_len, hdr.hdr.frag_len));
			p = 0;
		}

		if (p && r_o.status != 0)
		{
			/* report error code */
			DEBUG(0,("SAMR_R_UNKNOWN_12: %s\n", get_nt_error_msg(r_o.status)));
			p = 0;
		}

		if (p)
		{
			if (r_o.ptr_aliases != 0 && r_o.ptr_als_usrs != 0 &&
			    r_o.num_als_usrs1 == r_o.num_aliases1)
			{
				int i;

				valid_query = True;
				*num_aliases = r_o.num_aliases1;

				for (i = 0; i < r_o.num_aliases1; i++)
				{
					fstrcpy(als_names[i], unistrn2(r_o.uni_als_name[i].buffer, r_o.uni_als_name[i].uni_str_len));
				}
				for (i = 0; i < r_o.num_als_usrs1; i++)
				{
					num_als_users[i] = r_o.num_als_usrs[i];
				}
			}
			else if (r_o.ptr_aliases == 0 && r_o.ptr_als_usrs == 0)
			{
				valid_query = True;
				*num_aliases = 0;
			}
			else
			{
				p = 0;
			}
		}
	}

	buf_free(&data   );
	buf_free(&rdata  );
	buf_free(&rparams);

	return valid_query;
}

/****************************************************************************
do a SAMR Query User Groups
****************************************************************************/
BOOL do_samr_query_usergroups(struct cli_state *cli, int t_idx, uint16 fnum, 
				POLICY_HND *pol, uint32 *num_groups, DOM_GID *gid)
{
	struct mem_buffer data;
	struct mem_buffer rdata;
	struct mem_buffer rparams;
	int p = 0, r = 0;

	SAMR_Q_QUERY_USERGROUPS q_o;
    BOOL valid_query = False;

	/* create and send a MSRPC command with api SAMR_QUERY_USERGROUPS */

	buf_init(&data   , 4, SAFETY_MARGIN);
	buf_init(&rdata  , 4, SAFETY_MARGIN);
	buf_init(&rparams, 4, SAFETY_MARGIN);

	buf_alloc(&data , 1024);

	DEBUG(4,("SAMR Query User Groups.\n"));

	if (pol == NULL || gid == NULL || num_groups == 0|| fnum == 0xffff) return False;

	/* store the parameters */
	make_samr_q_query_usergroups(&q_o, pol);

	/* turn parameters into data stream */
	p = 0x18;
	samr_io_q_query_usergroups("", False, &q_o,  &data, &p, 0);

	data.data_used = p;

	/* create the request RPC_HDR_RR _after_ the main data: length is now known */
	create_rpc_request(&data, &r, get_rpc_call_id(), SAMR_QUERY_USERGROUPS, p);

	/* send the data on \PIPE\ */
	if (rpc_api_pipe(cli, t_idx, 0x0026, fnum, NULL, &data, &rparams, &(rdata)))
	{
		SAMR_R_QUERY_USERGROUPS r_o;
		RPC_HDR_RR hdr;
		int hdr_len;
		int pkt_len;

		/* get user info */
		r_o.gid = gid;

		DEBUG(5, ("cli_api_pipe: return OK\n"));

		p = 0;

		smb_io_rpc_hdr_rr   ("", True, &hdr, &(rdata), &p, 0);
		if (p) buf_align(&(rdata), &p); /* oh, what a surprise */

		hdr_len = p;

		if (p && hdr_len != hdr.hdr.frag_len - hdr.alloc_hint)
		{
			/* header length not same as calculated header length */
			DEBUG(2,("samr_r_query_usergroups: hdr_len %x != frag_len-alloc_hint %x\n",
			          hdr_len, hdr.hdr.frag_len - hdr.alloc_hint));
			p = 0;
		}

		if (p) samr_io_r_query_usergroups("", True, &r_o, &(rdata), &p, 0);
		
		pkt_len = p;

		if (p && pkt_len != hdr.hdr.frag_len)
		{
			/* packet data size not same as reported fragment length */
			DEBUG(2,("do_samr_query_usergroups: pkt_len %x != frag_len \n",
			                           pkt_len, hdr.hdr.frag_len));
			p = 0;
		}

		if (p && r_o.status != 0)
		{
			/* report error code */
			DEBUG(0,("SAMR_R_QUERY_USERGROUPS: %s\n", get_nt_error_msg(r_o.status)));
			p = 0;
		}

		if (p && r_o.ptr_0 != 0)
		{
			valid_query = True;
			*num_groups = r_o.num_entries;
		}

	}

	buf_free(&data   );
	buf_free(&rdata  );
	buf_free(&rparams);

	return valid_query;
}

/****************************************************************************
do a SAMR Query User Info
****************************************************************************/
BOOL do_samr_query_userinfo(struct cli_state *cli, int t_idx, uint16 fnum, 
				POLICY_HND *pol, uint16 switch_value, void* usr)
{
	struct mem_buffer data;
	struct mem_buffer rdata;
	struct mem_buffer rparams;
	int p = 0, r = 0;

	SAMR_Q_QUERY_USERINFO q_o;
    BOOL valid_query = False;

	/* create and send a MSRPC command with api SAMR_QUERY_USERINFO */

	buf_init(&data   , 4, SAFETY_MARGIN);
	buf_init(&rdata  , 4, SAFETY_MARGIN);
	buf_init(&rparams, 4, SAFETY_MARGIN);

	buf_alloc(&data , 1024);

	DEBUG(4,("SAMR Query User Info.  level: %d\n", switch_value));

	if (pol == NULL || usr == NULL || switch_value == 0|| fnum == 0xffff) return False;

	/* store the parameters */
	make_samr_q_query_userinfo(&q_o, pol, switch_value);

	/* turn parameters into data stream */
	p = 0x18;
	samr_io_q_query_userinfo("", False, &q_o,  &data, &p, 0);

	data.data_used = p;

	/* create the request RPC_HDR_RR _after_ the main data: length is now known */
	create_rpc_request(&data, &r, get_rpc_call_id(), SAMR_QUERY_USERINFO, p);

	/* send the data on \PIPE\ */
	if (rpc_api_pipe(cli, t_idx, 0x0026, fnum, NULL, &data, &rparams, &(rdata)))
	{
		SAMR_R_QUERY_USERINFO r_o;
		RPC_HDR_RR hdr;
		int hdr_len;
		int pkt_len;

		/* get user info */
		r_o.info.id = usr;

		DEBUG(5, ("cli_api_pipe: return OK\n"));

		p = 0;

		smb_io_rpc_hdr_rr   ("", True, &hdr, &(rdata), &p, 0);
		if (p) buf_align(&(rdata), &p); /* oh, what a surprise */

		hdr_len = p;

		if (p && hdr_len != hdr.hdr.frag_len - hdr.alloc_hint)
		{
			/* header length not same as calculated header length */
			DEBUG(2,("samr_r_query_userinfo: hdr_len %x != frag_len-alloc_hint %x\n",
			          hdr_len, hdr.hdr.frag_len - hdr.alloc_hint));
			p = 0;
		}

		if (p) samr_io_r_query_userinfo("", True, &r_o, &(rdata), &p, 0);
		
		pkt_len = p;

		if (p && pkt_len != hdr.hdr.frag_len)
		{
			/* packet data size not same as reported fragment length */
			DEBUG(2,("do_samr_query_userinfo: pkt_len %x != frag_len \n",
			                           pkt_len, hdr.hdr.frag_len));
			p = 0;
		}

		if (p && r_o.status != 0)
		{
			/* report error code */
			DEBUG(0,("SAMR_R_QUERY_USERINFO: %s\n", get_nt_error_msg(r_o.status)));
			p = 0;
		}

		if (p && r_o.switch_value != switch_value)
		{
			DEBUG(0,("SAMR_R_QUERY_USERINFO: received incorrect level %d\n",
			          r_o.switch_value));
		}

		if (p && r_o.ptr != 0)
		{
			valid_query = True;
		}
	}

	buf_free(&data   );
	buf_free(&rdata  );
	buf_free(&rparams);

	return valid_query;
}

/****************************************************************************
do a SAMR Close
****************************************************************************/
BOOL do_samr_close(struct cli_state *cli, int t_idx, uint16 fnum, POLICY_HND *hnd)
{
	struct mem_buffer data;
	struct mem_buffer rdata;
	struct mem_buffer rparams;
	int p = 0, r = 0;

	SAMR_Q_CLOSE_HND q_c;
    BOOL valid_close = False;

	buf_init(&data   , 4, SAFETY_MARGIN);
	buf_init(&rdata  , 4, SAFETY_MARGIN);
	buf_init(&rparams, 4, SAFETY_MARGIN);

	buf_alloc(&data , 1024);

	if (hnd == NULL) return False;

	/* create and send a MSRPC command with api SAMR_OPENPOLICY */

	DEBUG(4,("SAMR Close\n"));

	/* store the parameters */
	make_samr_q_close_hnd(&q_c, hnd);

	/* turn parameters into data stream */
	p = 0x18;
	samr_io_q_close_hnd("", False, &q_c,  &data, &p, 0);

	data.data_used = p;

	/* create the request RPC_HDR_RR with no data */
	create_rpc_request(&data, &r, get_rpc_call_id(), SAMR_CLOSE_HND, p);

	/* send the data on \PIPE\ */
	if (rpc_api_pipe(cli, t_idx, 0x0026, fnum, NULL, &data, &rparams, &(rdata)))
	{
		SAMR_R_CLOSE_HND r_c;
		RPC_HDR_RR hdr;
		int hdr_len;
		int pkt_len;

		DEBUG(5, ("cli_api_pipe: return OK\n"));

		p = 0;

		smb_io_rpc_hdr_rr   ("", True, &hdr, &(rdata), &p, 0);
		if (p) buf_align(&(rdata), &p); /* oh, what a surprise */

		hdr_len = p;

		if (p && hdr_len != hdr.hdr.frag_len - hdr.alloc_hint)
		{
			/* header length not same as calculated header length */
			DEBUG(2,("do_samr_close: hdr_len %x != frag_len-alloc_hint %x\n",
			          hdr_len, hdr.hdr.frag_len - hdr.alloc_hint));
#ifdef CATCH_RPC_FRAG_LEN_ERRORS
			p = 0;
#endif
		}

		if (p) samr_io_r_close_hnd("", True, &r_c, &(rdata), &p, 0);
		
		pkt_len = p;

		if (p && pkt_len != hdr.hdr.frag_len)
		{
			/* packet data size not same as reported fragment length */
			DEBUG(2,("do_samr_close: pkt_len %x != frag_len \n",
			                           pkt_len, hdr.hdr.frag_len));
#ifdef CATCH_RPC_FRAG_LEN_ERRORS
			p = 0;
#endif
		}

		if (p && r_c.status != 0)
		{
			/* report error code */
			DEBUG(0,("SAMR_CLOSE_HND: %s\n", get_nt_error_msg(r_c.status)));
			p = 0;
		}

		if (p)
		{
			/* check that the returned policy handle is all zeros */
			int i;
			valid_close = True;

			for (i = 0; i < sizeof(r_c.pol.data); i++)
			{
				if (r_c.pol.data[i] != 0)
				{
					valid_close = False;
					break;
				}
			}	
			if (!valid_close)
			{
				DEBUG(0,("SAMR_CLOSE_HND: non-zero handle returned\n"));
			}
		}
	}

	buf_free(&data   );
	buf_free(&rdata  );
	buf_free(&rparams);

	return valid_close;
}

