/* 
   Unix SMB/CIFS implementation.

   Validate the krb5 pac generation routines
   
   Copyright (C) Andrew Bartlett <abartlet@samba.org> 2005

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include "system/kerberos.h"
#include "auth/auth.h"
#include "auth/kerberos/kerberos.h"
#include "librpc/gen_ndr/ndr_krb5pac.h"
#include "samba3/samba3.h"
#include "libcli/security/security.h"
#include "torture/torture.h"

static BOOL torture_pac_self_check(struct torture_context *test, 
								   const void *_data) 
{
	NTSTATUS nt_status;
	DATA_BLOB tmp_blob;
	struct PAC_DATA *pac_data;
	struct PAC_LOGON_INFO *logon_info;
	union netr_Validation validation;

	/* Generate a nice, arbitary keyblock */
	uint8_t server_bytes[16];
	uint8_t krbtgt_bytes[16];
	krb5_keyblock server_keyblock;
	krb5_keyblock krbtgt_keyblock;
	
	krb5_error_code ret;

	struct smb_krb5_context *smb_krb5_context;

	struct auth_serversupplied_info *server_info;
	struct auth_serversupplied_info *server_info_out;

	krb5_principal client_principal;
	time_t logon_time = time(NULL);

	torture_assert(test, 0 == smb_krb5_init_context(test, &smb_krb5_context), 
				 "smb_krb5_init_context");

	generate_random_buffer(server_bytes, 16);
	generate_random_buffer(krbtgt_bytes, 16);

	ret = krb5_keyblock_init(smb_krb5_context->krb5_context,
				 ENCTYPE_ARCFOUR_HMAC,
				 server_bytes, sizeof(server_bytes),
				 &server_keyblock);
	if (ret) {
		torture_fail(test, "(self test) Server Keyblock encoding failed: %s", 
		       smb_get_krb5_error_message(smb_krb5_context->krb5_context, 
						  ret, test));
		
		return False;
	}

	ret = krb5_keyblock_init(smb_krb5_context->krb5_context,
				 ENCTYPE_ARCFOUR_HMAC,
				 krbtgt_bytes, sizeof(krbtgt_bytes),
				 &krbtgt_keyblock);
	if (ret) {
		torture_fail(test, "(self test) KRBTGT Keyblock encoding failed: %s", 
		       smb_get_krb5_error_message(smb_krb5_context->krb5_context, 
						  ret, test));
	
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		return False;
	}

	/* We need an input, and this one requires no underlying database */
	nt_status = auth_anonymous_server_info(test, &server_info);

	if (!NT_STATUS_IS_OK(nt_status)) {
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		return False;
	}

	ret = krb5_parse_name_norealm(smb_krb5_context->krb5_context, 
				      server_info->account_name, &client_principal);
	if (ret) {
		torture_fail(test, "krb5_parse_name_norealm");
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		return False;
	}

	/* OK, go ahead and make a PAC */
	ret = kerberos_create_pac(test, server_info, 
				  smb_krb5_context->krb5_context,  
				  &krbtgt_keyblock,
				  &server_keyblock,
				  client_principal,
				  logon_time,
				  &tmp_blob);
	
	if (ret) {
		torture_fail(test, "(self test) PAC encoding failed: %s", 
		       smb_get_krb5_error_message(smb_krb5_context->krb5_context, 
						  ret, test));

		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, 
				    client_principal);
		return False;
	}

	dump_data(10,tmp_blob.data,tmp_blob.length);

	/* Now check that we can read it back */
	nt_status = kerberos_decode_pac(test, &pac_data,
					tmp_blob,
					smb_krb5_context->krb5_context,
					&krbtgt_keyblock,
					&server_keyblock,
					client_principal, 
					logon_time, NULL);

	if (!NT_STATUS_IS_OK(nt_status)) {
		torture_fail(test, "(self test) PAC decoding failed: %s", 
			  nt_errstr(nt_status));

		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, 
				    client_principal);

		return False;
	}

	/* Now check that we can read it back */
	nt_status = kerberos_pac_logon_info(test, &logon_info,
					    tmp_blob,
					    smb_krb5_context->krb5_context,
					    &krbtgt_keyblock,
					    &server_keyblock,
					    client_principal, 
					    logon_time, 
					    NULL);
	
	if (!NT_STATUS_IS_OK(nt_status)) {
		torture_fail(test, 
			"(self test) PAC decoding (for logon info) failed: %s", 
		    nt_errstr(nt_status));

		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, 
				    client_principal);
		
		return False;
	}
	
	krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
				    &krbtgt_keyblock);
	krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
				    &server_keyblock);
	krb5_free_principal(smb_krb5_context->krb5_context, 
			    client_principal);

	validation.sam3 = &logon_info->info3;
	nt_status = make_server_info_netlogon_validation(test,
							 "",
							 3, &validation,
							 &server_info_out); 
	if (!NT_STATUS_IS_OK(nt_status)) {
		torture_fail(test, 
			"(self test) PAC decoding (make server info) failed: %s", 
		    nt_errstr(nt_status));
		
		return False;
	}
	
	if (!dom_sid_equal(server_info->account_sid, 
			   server_info_out->account_sid)) {
		torture_fail(test, 
			"(self test) PAC Decode resulted in *different* domain SID: %s != %s",
		       dom_sid_string(test, server_info->account_sid), 
		       dom_sid_string(test, server_info_out->account_sid));
		return False;
	}
	
	return True;
}


/* This is the PAC generated on my test network, by my test Win2k3 server.
   -- abartlet 2005-07-04
 */

static const uint8_t saved_pac[] = {
	0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xd8, 0x01, 0x00, 0x00, 
	0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
	0x20, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
	0x40, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
	0x58, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc, 0xcc, 0xcc,
	0xc8, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x30, 0xdf, 0xa6, 0xcb, 
	0x4f, 0x7d, 0xc5, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 
	0xff, 0xff, 0xff, 0x7f, 0xc0, 0x3c, 0x4e, 0x59, 0x62, 0x73, 0xc5, 0x01, 0xc0, 0x3c, 0x4e, 0x59,
	0x62, 0x73, 0xc5, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x16, 0x00, 0x16, 0x00,
	0x04, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x0c, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x14, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x02, 0x00, 0x65, 0x00, 0x00, 0x00, 
	0xed, 0x03, 0x00, 0x00, 0x04, 0x02, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x02, 0x00,
	0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x16, 0x00, 0x20, 0x00, 0x02, 0x00, 0x16, 0x00, 0x18, 0x00,
	0x24, 0x00, 0x02, 0x00, 0x28, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x21, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x01, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
	0x57, 0x00, 0x32, 0x00, 0x30, 0x00, 0x30, 0x00, 0x33, 0x00, 0x46, 0x00, 0x49, 0x00, 0x4e, 0x00,
	0x41, 0x00, 0x4c, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x02, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
	0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x57, 0x00, 0x32, 0x00,
	0x30, 0x00, 0x30, 0x00, 0x33, 0x00, 0x46, 0x00, 0x49, 0x00, 0x4e, 0x00, 0x41, 0x00, 0x4c, 0x00,
	0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x57, 0x00, 0x49, 0x00,
	0x4e, 0x00, 0x32, 0x00, 0x4b, 0x00, 0x33, 0x00, 0x54, 0x00, 0x48, 0x00, 0x49, 0x00, 0x4e, 0x00,
	0x4b, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
	0x15, 0x00, 0x00, 0x00, 0x11, 0x2f, 0xaf, 0xb5, 0x90, 0x04, 0x1b, 0xec, 0x50, 0x3b, 0xec, 0xdc,
	0x01, 0x00, 0x00, 0x00, 0x30, 0x00, 0x02, 0x00, 0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
	0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x80, 0x66, 0x28, 0xea, 0x37, 0x80, 0xc5, 0x01, 0x16, 0x00, 0x77, 0x00, 0x32, 0x00, 0x30, 0x00,
	0x30, 0x00, 0x33, 0x00, 0x66, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x61, 0x00, 0x6c, 0x00, 0x24, 0x00,
	0x76, 0xff, 0xff, 0xff, 0x37, 0xd5, 0xb0, 0xf7, 0x24, 0xf0, 0xd6, 0xd4, 0xec, 0x09, 0x86, 0x5a,
	0xa0, 0xe8, 0xc3, 0xa9, 0x00, 0x00, 0x00, 0x00, 0x76, 0xff, 0xff, 0xff, 0xb4, 0xd8, 0xb8, 0xfe,
	0x83, 0xb3, 0x13, 0x3f, 0xfc, 0x5c, 0x41, 0xad, 0xe2, 0x64, 0x83, 0xe0, 0x00, 0x00, 0x00, 0x00
};

/* Check with a known 'well formed' PAC, from my test server */
static BOOL torture_pac_saved_check(struct torture_context *test,
									const void *_data) 
{
	NTSTATUS nt_status;
	DATA_BLOB tmp_blob, validate_blob;
	struct PAC_DATA *pac_data, pac_data2;
	struct PAC_LOGON_INFO *logon_info;
	union netr_Validation validation;
	const char *pac_file, *pac_kdc_key, *pac_member_key;
	struct auth_serversupplied_info *server_info_out;

	krb5_keyblock server_keyblock;
	krb5_keyblock krbtgt_keyblock;
	struct samr_Password *krbtgt_bytes, *krbsrv_bytes;
	
	krb5_error_code ret;
	struct smb_krb5_context *smb_krb5_context;

	const char *principal_string;
	krb5_principal client_principal;
	const char *authtime_string;
	time_t authtime;

	torture_assert(test, 0 == smb_krb5_init_context(test, &smb_krb5_context),
				   "smb_krb5_init_context");

	pac_kdc_key = lp_parm_string(-1,"torture","pac_kdc_key");
	if (pac_kdc_key == NULL) {
		pac_kdc_key = "B286757148AF7FD252C53603A150B7E7";
	}

	pac_member_key = lp_parm_string(-1,"torture","pac_member_key");
	if (pac_member_key == NULL) {
		pac_member_key = "D217FAEAE5E6B5F95CCC94077AB8A5FC";
	}

	torture_comment(test, "Using pac_kdc_key '%s'", pac_kdc_key);
	torture_comment(test, "Using pac_member_key '%s'", pac_member_key);

	/* The krbtgt key in use when the above PAC was generated.
	 * This is an arcfour-hmac-md5 key, extracted with our 'net
	 * samdump' tool. */
	krbtgt_bytes = smbpasswd_gethexpwd(test, pac_kdc_key);
	if (!krbtgt_bytes) {
		torture_fail(test, "(saved test) Could not interpret krbtgt key");
		return False;
	}

	krbsrv_bytes = smbpasswd_gethexpwd(test, pac_member_key);
	if (!krbsrv_bytes) {
		torture_fail(test, "(saved test) Could not interpret krbsrv key");
		return False;
	}

	ret = krb5_keyblock_init(smb_krb5_context->krb5_context,
				 ENCTYPE_ARCFOUR_HMAC,
				 krbsrv_bytes->hash, sizeof(krbsrv_bytes->hash),
				 &server_keyblock);
	if (ret) {
		torture_fail(test, 
					 "(saved test) Server Keyblock encoding failed: %s", 
			  smb_get_krb5_error_message(smb_krb5_context->krb5_context, 
						     ret, test));

		return False;
	}

	ret = krb5_keyblock_init(smb_krb5_context->krb5_context,
				 ENCTYPE_ARCFOUR_HMAC,
				 krbtgt_bytes->hash, sizeof(krbtgt_bytes->hash),
				 &krbtgt_keyblock);
	if (ret) {
		torture_fail(test, 
			  "(saved test) Server Keyblock encoding failed: %s", 
			  smb_get_krb5_error_message(smb_krb5_context->krb5_context, 
						     ret, test));

		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		return False;
	}

	pac_file = lp_parm_string(-1,"torture","pac_file");
	if (pac_file) {
		tmp_blob.data = (uint8_t *)file_load(pac_file, &tmp_blob.length, test);
		torture_comment(test, "(saved test) Loaded pac of size %ld from %s", (long)tmp_blob.length, pac_file);
	} else {
		tmp_blob = data_blob_talloc(test, saved_pac, sizeof(saved_pac));
	}
	
	dump_data(10,tmp_blob.data,tmp_blob.length);

	principal_string = lp_parm_string(-1,"torture","pac_client_principal");
	if (!principal_string) {
		principal_string = "w2003final$@WIN2K3.THINKER.LOCAL";
	}

	authtime_string = lp_parm_string(-1,"torture","pac_authtime");
	if (!authtime_string) {
		authtime = 1120440609;
	} else {
		authtime = strtoull(authtime_string, NULL, 0);
	}

	ret = krb5_parse_name(smb_krb5_context->krb5_context, principal_string, 
			      &client_principal);
	if (ret) {
		torture_fail(test, 
			  "(saved test) parsing of client principal [%s] failed: %s", 
			  principal_string, 
			  smb_get_krb5_error_message(smb_krb5_context->krb5_context, ret, test));

		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		return False;
	}

	/* Decode and verify the signaure on the PAC */
	nt_status = kerberos_decode_pac(test, &pac_data,
					tmp_blob,
					smb_krb5_context->krb5_context,
					&krbtgt_keyblock,
					&server_keyblock, 
					client_principal, authtime, NULL);
	if (!NT_STATUS_IS_OK(nt_status)) {
		torture_fail(test, "(saved test) PAC decoding failed: %s", 
			  nt_errstr(nt_status));

		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, client_principal);
		
		return False;
	}

	/* Parse the PAC again, for the logon info this time */
	nt_status = kerberos_pac_logon_info(test, &logon_info,
					    tmp_blob,
					    smb_krb5_context->krb5_context,
					    &krbtgt_keyblock,
					    &server_keyblock,
					    client_principal, authtime, NULL);

	if (!NT_STATUS_IS_OK(nt_status)) {
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, client_principal);
	
		torture_fail(test, 
			"(saved test) PAC decoding (for logon info) failed: %s", 
			  nt_errstr(nt_status));

		return False;
	}

	validation.sam3 = &logon_info->info3;
	nt_status = make_server_info_netlogon_validation(test,
							 "",
							 3, &validation,
							 &server_info_out); 
	if (!NT_STATUS_IS_OK(nt_status)) {
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
				    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, client_principal);

		torture_fail(test, 
				"(saved test) PAC decoding (make server info) failed: %s", 
		       nt_errstr(nt_status));
		
		return False;
	}

	if (!pac_file &&
	    !dom_sid_equal(dom_sid_parse_talloc(test, 
			   "S-1-5-21-3048156945-3961193616-3706469200-1005"), 
			   server_info_out->account_sid)) {
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, client_principal);

		torture_fail(test, 
			"(saved test) PAC Decode resulted in *different* domain SID: %s != %s",
		    "S-1-5-21-3048156945-3961193616-3706469200-1005", 
		       dom_sid_string(test, server_info_out->account_sid));
		return False;
	}

	ret = kerberos_encode_pac(test, 
				  pac_data,
				  smb_krb5_context->krb5_context,
				  &krbtgt_keyblock,
				  &server_keyblock,
				  &validate_blob);

	if (ret != 0) {
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, client_principal);

		torture_fail(test, "(saved test) PAC push failed");
		return False;
	}

	dump_data(10, validate_blob.data, validate_blob.length);

	/* compare both the length and the data bytes after a
	 * pull/push cycle.  This ensures we use the exact same
	 * pointer, padding etc algorithms as win2k3.
	 */
	if (tmp_blob.length != validate_blob.length) {
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, client_principal);

		torture_fail(test, 
			"(saved test) PAC push failed: original buffer length[%u] != created buffer length[%u]",
			(unsigned)tmp_blob.length, (unsigned)validate_blob.length);
		return False;
	}

	if (memcmp(tmp_blob.data, validate_blob.data, tmp_blob.length) != 0) {
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, client_principal);

		torture_fail(test, "(saved test) PAC push failed: length[%u] matches, but data does not", (unsigned)tmp_blob.length);
		DEBUG(0, ("tmp_data:\n"));
		dump_data(0, tmp_blob.data, tmp_blob.length);
		DEBUG(0, ("validate_blob:\n"));
		dump_data(0, validate_blob.data, validate_blob.length);

		return False;
	}

	ret = kerberos_create_pac(test, 
				  server_info_out,
				  smb_krb5_context->krb5_context,
				  &krbtgt_keyblock,
				  &server_keyblock,
				  client_principal, authtime,
				  &validate_blob);

	if (ret != 0) {
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, client_principal);

		torture_fail(test, "(saved test) regnerated PAC create failed");
		return False;
	}

	dump_data(10,validate_blob.data,validate_blob.length);

	/* compare both the length and the data bytes after a
	 * pull/push cycle.  This ensures we use the exact same
	 * pointer, padding etc algorithms as win2k3.
	 */
	if (tmp_blob.length != validate_blob.length) {
		nt_status = ndr_pull_struct_blob(&validate_blob, test, &pac_data2,
					      (ndr_pull_flags_fn_t)ndr_pull_PAC_DATA);
		torture_assert_ntstatus_ok(test, nt_status, "can't parse the PAC");
		
		NDR_PRINT_DEBUG(PAC_DATA, pac_data);

		NDR_PRINT_DEBUG(PAC_DATA, &pac_data2);

		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, client_principal);

		torture_fail(test, "(saved test) PAC regenerate failed: original buffer length[%u] != created buffer length[%u]",
				(unsigned)tmp_blob.length, (unsigned)validate_blob.length);
		return False;
	}

	if (memcmp(tmp_blob.data, validate_blob.data, tmp_blob.length) != 0) {
		nt_status = ndr_pull_struct_blob(&validate_blob, test, &pac_data2,
					      (ndr_pull_flags_fn_t)ndr_pull_PAC_DATA);
		torture_assert_ntstatus_ok(test, nt_status, "can't parse the PAC");
		
		NDR_PRINT_DEBUG(PAC_DATA, pac_data);

		NDR_PRINT_DEBUG(PAC_DATA, &pac_data2);

		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, client_principal);

		torture_fail(test, "(saved test) PAC regenerate failed: length[%u] matches, but data does not", (unsigned)tmp_blob.length);
		DEBUG(0, ("tmp_data:\n"));
		dump_data(0, tmp_blob.data, tmp_blob.length);
		DEBUG(0, ("validate_blob:\n"));
		dump_data(0, validate_blob.data, validate_blob.length);

		return False;
	}

	/* Break the auth time, to ensure we check this vital detail (not setting this caused all the pain in the first place... */
	nt_status = kerberos_decode_pac(test, &pac_data,
					tmp_blob,
					smb_krb5_context->krb5_context,
					&krbtgt_keyblock,
					&server_keyblock,
					client_principal, 
					authtime + 1, NULL);
	if (NT_STATUS_IS_OK(nt_status)) {
		torture_fail(test, "(saved test) PAC decoding DID NOT fail on broken auth time (time + 1)");

		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		krb5_free_principal(smb_krb5_context->krb5_context, client_principal);
		return False;
	}

	/* Break the client principal */
	krb5_free_principal(smb_krb5_context->krb5_context, client_principal);

	ret = krb5_parse_name(smb_krb5_context->krb5_context,
			     "not the right principal", &client_principal);
	if (ret) {
		torture_fail(test, "(saved test) parsing of bogus client principal failed: %s", 
			  smb_get_krb5_error_message(smb_krb5_context->krb5_context, ret, test));

		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		return False;
	}

	nt_status = kerberos_decode_pac(test, &pac_data,
					tmp_blob,
					smb_krb5_context->krb5_context,
					&krbtgt_keyblock,
					&server_keyblock,
					client_principal, 
					authtime, NULL);
	if (NT_STATUS_IS_OK(nt_status)) {
		torture_fail(test, "(saved test) PAC decoding DID NOT fail on modified principal");

		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		return False;
	}

	/* Finally...  Bugger up the signature, and check we fail the checksum */
	tmp_blob.data[tmp_blob.length - 2]++;

	nt_status = kerberos_decode_pac(test, &pac_data,
					tmp_blob,
					smb_krb5_context->krb5_context,
					&krbtgt_keyblock,
					&server_keyblock,
					client_principal, 
					authtime, NULL);
	if (NT_STATUS_IS_OK(nt_status)) {
		torture_fail(test, "(saved test) PAC decoding DID NOT fail on broken checksum");

		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &krbtgt_keyblock);
		krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
					    &server_keyblock);
		return False;
	}

	krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
				    &krbtgt_keyblock);
	krb5_free_keyblock_contents(smb_krb5_context->krb5_context, 
				    &server_keyblock);

	return True;
}

struct torture_suite *torture_pac(TALLOC_CTX *mem_ctx)
{
	struct torture_suite *suite = torture_suite_create(mem_ctx, "LOCAL-PAC");

	torture_suite_add_simple_tcase(suite, "self check", 
								   torture_pac_self_check, NULL);

	torture_suite_add_simple_tcase(suite, "saved check",
								   torture_pac_saved_check, NULL);

	return suite;
}
