/* 
   ldb database library

   Copyright (C) Simo Sorce  2005
   Copyright (C) Stefa Metzmacher <metze@samba.org> 2007

     ** NOTE! The following LGPL license applies to the ldb
     ** library. This does NOT imply that all of Samba is released
     ** under the LGPL
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, see <http://www.gnu.org/licenses/>.
*/

/*
 *  Name: ldb
 *
 *  Component: ldb deleted objects control module
 *
 *  Description: this module hides deleted objects, and returns them if the right control is there
 *
 *  Author: Stefan Metzmacher
 */

#include "includes.h"
#include "ldb/include/ldb.h"
#include "ldb/include/ldb_errors.h"
#include "ldb/include/ldb_private.h"
#include "dsdb/samdb/samdb.h"

/* search */
struct show_deleted_search_request {

	struct ldb_module *module;
	struct ldb_request *req;
};

static int show_deleted_search_callback(struct ldb_request *req,
					struct ldb_reply *ares)
{
	struct show_deleted_search_request *ar;

	ar = talloc_get_type(req->context, struct show_deleted_search_request);

	if (!ares) {
		return ldb_module_done(ar->req, NULL, NULL,
					LDB_ERR_OPERATIONS_ERROR);
	}
	if (ares->error != LDB_SUCCESS) {
		return ldb_module_done(ar->req, ares->controls,
					ares->response, ares->error);
	}

	switch (ares->type) {
	case LDB_REPLY_ENTRY:

		return ldb_module_send_entry(ar->req, ares->message, ares->controls);

	case LDB_REPLY_REFERRAL:
		return ldb_module_send_referral(ar->req, ares->referral);

	case LDB_REPLY_DONE:
		return ldb_module_done(ar->req, ares->controls,
					ares->response, LDB_SUCCESS);

	}
	return LDB_SUCCESS;
}

static int show_deleted_search(struct ldb_module *module, struct ldb_request *req)
{
	struct ldb_control *control;
	struct ldb_control **saved_controls;
	struct show_deleted_search_request *ar;
	struct ldb_request *down_req;
	char *old_filter;
	char *new_filter;
	int ret;

	ar = talloc_zero(req, struct show_deleted_search_request);
	if (ar == NULL) {
		return LDB_ERR_OPERATIONS_ERROR;
	}
	ar->module = module;
	ar->req = req;

	/* check if there's a show deleted control */
	control = ldb_request_get_control(req, LDB_CONTROL_SHOW_DELETED_OID);

	if ( ! control) {
		old_filter = ldb_filter_from_tree(ar, req->op.search.tree);
		new_filter = talloc_asprintf(ar, "(&(!(isDeleted=TRUE))%s)",
						 old_filter);

		ret = ldb_build_search_req(&down_req, module->ldb, ar,
					   req->op.search.base,
					   req->op.search.scope,
					   new_filter,
					   req->op.search.attrs,
					   req->controls,
					   ar, show_deleted_search_callback,
					   req);

	} else {
		ret = ldb_build_search_req_ex(&down_req, module->ldb, ar,
					      req->op.search.base,
					      req->op.search.scope,
					      req->op.search.tree,
					      req->op.search.attrs,
					      req->controls,
					      ar, show_deleted_search_callback,
					      req);
	}
	if (ret != LDB_SUCCESS) {
		return ret;
	}

	/* if a control is there remove if from the modified request */
	if (control && !save_controls(control, down_req, &saved_controls)) {
		return LDB_ERR_OPERATIONS_ERROR;
	}

	/* perform the search */
	return ldb_next_request(module, down_req);
}

static int show_deleted_init(struct ldb_module *module)
{
	int ret;

	ret = ldb_mod_register_control(module, LDB_CONTROL_SHOW_DELETED_OID);
	if (ret != LDB_SUCCESS) {
		ldb_debug(module->ldb, LDB_DEBUG_ERROR,
			"extended_dn: Unable to register control with rootdse!\n");
		return LDB_ERR_OPERATIONS_ERROR;
	}

	return ldb_next_init(module);
}

_PUBLIC_ const struct ldb_module_ops ldb_show_deleted_module_ops = {
	.name		   = "show_deleted",
	.search            = show_deleted_search,
	.init_context	   = show_deleted_init
};
