/* cache.c -- pam_smb cache management code 
   
   Copyright (c) Dave Airlie 2000
   airlied@samba.org

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   */

#include "config.h"

#include <stdlib.h>
#include <fcntl.h>
#include <syslog.h>
#include <errno.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include "constants.h"
#include "tdb.h"
#include "cache.h"
#include "pamsmbd.h"
#include "pam_smb_queue.h"

#ifdef CRYPT
#include <crypt.h>
#else
#include "md5driver.h"
#endif

TDB_CONTEXT *cache_db;

/*
 * Initalise the cache database
 */
int serv_initcachedb()
{
  cache_db = tdb_open(NULL, 0, TDB_INTERNAL, 0, 0);

  if (cache_db==NULL)
    {
      perror("tdb_open failed");
      exit(-1);
    }
  return 0;
}

/*
 * Close the cache database
 */
int serv_closecachedb()
{
  tdb_close(cache_db);
  return 0;
}

/*
 * Add an entry to the cache
 */
int add_entry_to_cache(usercache_data *entry)
{
  int res;
  TDB_DATA key;
  TDB_DATA value;
  time_t curtime;

  key.dptr = entry->ux_user;
  key.dsize = strlen(entry->ux_user)+1;

  curtime=time(NULL);
  entry->usertimeout=curtime;

  value.dptr = (char *)entry;
  value.dsize = sizeof(usercache_data);
  
  tdb_lockall(cache_db);
  res=tdb_store(cache_db, key, value, TDB_REPLACE);
  tdb_unlockall(cache_db);
  if (res==0)
    {
      cur_cache_size++;
      if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_add: inserted entry\n");
      return 0;
    }

  syslog(LOG_AUTHPRIV | LOG_ERR, "cache_add: error on db insert\n");
  perror("Error on tdb_store\n");
  sigterm_handler(0);
  return 0;
}

    
/*
 * Check if an entry is in the cache
 */
int check_entry_in_cache(usercache_data *entry, int timeout, int failtimeout)
{
  TDB_DATA key;
  TDB_DATA value;
  usercache_data *check;
  time_t curtime;

  if (timeout==SMB_NO_CACHE && failtimeout == SMB_NO_CACHE)
	return CACHE_ENOENT;

  curtime = time(NULL);
  key.dptr = entry->ux_user;
  key.dsize = strlen(entry->ux_user)+1;

  value=tdb_fetch(cache_db, key);
  if (value.dptr!=NULL)
    {
  
      /* Check if password is correct */
      check=(usercache_data *)value.dptr;
      if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_check: found entry checking passwords");
   
#ifdef CRYPT
    if (!(strncmp(crypt( (const char *) entry->password, (const char *)check->password), check->password, 13)))
#else

	if (!(memcmp( (void *) md5string( ( void *) entry->password ), (void *) check->password, MD5_LENGTH)))
#endif
	{
	  if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_check: found entry\n");
	  if ((timeout != SMB_NO_CACHE) && ((timeout < 0) || (curtime > ((timeout*60)+check->usertimeout))))
	    {
	      if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_check: account valid\n");
	      free(value.dptr);
	      return CACHE_OK;
	    }
	  else if (failtimeout != SMB_NO_CACHE && ((failtimeout < 1) || (curtime <= (failtimeout * 60)+check->usertimeout)))
	    {
	      if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_check: account valid in failover cache\n");
	      free(value.dptr);
	      return CACHE_OK_FAILOVER;
	    }
	  else
            {
	      if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_check: account cache entry expired\n");
	      free(value.dptr);
	      return CACHE_ETIME;
            }
	}
      else
	{
	  if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_check: password wrong\n");
	  free(value.dptr);
	  return CACHE_ENOENT;
	}
    }
  else
    {
      if (debug)   syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_check: no entry\n");
      return CACHE_ENOENT;
    }
  syslog(LOG_AUTHPRIV | LOG_ERR, "cache_check: error on db fetch\n");

  return CACHE_ENOENT;  
}


/*
 * Handler for SIGUSR1 signal, does full cache cleanout
 * (perhaps we should just close and re-init the cache...it would probably be faster..)
 */
void sigusr1_handler(int signum)
{
  if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_clean: dropping %d entries, creating new user cache",cur_cache_size);
  serv_closecachedb();
  cur_cache_size = 0;
  serv_initcachedb();
  return;
}
