/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   NBT netbios routines and daemon - version 2
   Copyright (C) Andrew Tridgell 1994-1995
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
*/

#include "includes.h"
#include "nameserv.h"


static void queue_packet(struct packet_struct *packet);
void process(void);
static void dump_names(void);

extern int DEBUGLEVEL;

extern pstring debugf;

extern pstring scope;

extern BOOL CanRecurse;

extern struct in_addr myip;
extern struct in_addr bcast_ip;
extern struct in_addr Netmask;
extern pstring myhostname;
static pstring host_file;
static pstring myname="";

static int ClientNMB=-1;
static int ClientDGRAM=-1;

static struct name_record *namelist = NULL;
static struct domain_record *domainlist = NULL;

/* are we running as a daemon ? */
static BOOL is_daemon = False;

/* machine comment for host announcements */
static pstring comment="";

extern pstring user_socket_options;

static BOOL got_bcast = False;
static BOOL got_myip = False;
static BOOL got_nmask = False;


/****************************************************************************
catch a sighup
****************************************************************************/
static int sig_hup()
{
  BlockSignals(True);

  DEBUG(0,("Got SIGHUP - not implemented\n"));
  dump_names();
  if (!is_daemon)
    exit(1);

  BlockSignals(False);
#ifndef DONT_REINSTALL_SIG
  signal(SIGHUP,SIGNAL_CAST sig_hup);
#endif
  return(0);
}

/****************************************************************************
catch a sigpipe
****************************************************************************/
static int sig_pipe()
{
  BlockSignals(True);

  DEBUG(0,("Got SIGPIPE\n"));
  if (!is_daemon)
    exit(1);
  BlockSignals(False);
  return(0);
}

/****************************************************************************
possibly continue after a fault
****************************************************************************/
static void fault_continue(void)
{
  static int errcount=0;

  errcount++;

  if (is_daemon && errcount<100)
    process();

  exit(1);
}

/****************************************************************************
  true if two netbios names are equal
****************************************************************************/
static BOOL name_equal(struct nmb_name *n1,struct nmb_name *n2)
{
  if (n1->name_type != n2->name_type) return(False);

  return(strequal(n1->name,n2->name) && strequal(n1->scope,n2->scope));
}

/****************************************************************************
  add a netbios name into the namelist
  **************************************************************************/
static void add_name(struct name_record *n)
{
  struct name_record *n2;

  if (!namelist) {
    namelist = n;
    n->prev = NULL;
    n->next = NULL;
    return;
  }

  for (n2 = namelist; n2->next; n2 = n2->next) ;

  n2->next = n;
  n->next = NULL;
  n->prev = n2;
}

/****************************************************************************
  add a domain into the list
  **************************************************************************/
static void add_domain(struct domain_record *d)
{
  struct domain_record *d2;

  if (!domainlist) {
    domainlist = d;
    d->prev = NULL;
    d->next = NULL;
    return;
  }

  for (d2 = domainlist; d2->next; d2 = d2->next) ;

  d2->next = d;
  d->next = NULL;
  d->prev = d2;
}

/****************************************************************************
  remove a name from the namelist. The pointer must be an element just 
  retrieved
  **************************************************************************/
static void remove_name(struct name_record *n)
{
  struct name_record *nlist = namelist;
  while (nlist && nlist != n) nlist = nlist->next;
  if (nlist) {
    if (nlist->next) nlist->next->prev = nlist->prev;
    if (nlist->prev) nlist->prev->next = nlist->next;
    free(nlist);
  }
}

/****************************************************************************
  find a name in the namelist matching some criterion
  **************************************************************************/
static struct name_record *find_name(struct nmb_name *n)
{
  struct name_record *ret;
  for (ret = namelist; ret; ret = ret->next)
    if (name_equal(&ret->name,n)) return(ret);

  return(NULL);
}

/****************************************************************************
  dump a copy of the name table
  **************************************************************************/
static void dump_names(void)
{
  time_t t = time(NULL);
  struct name_record *n;
  struct domain_record *d;

  DEBUG(3,("Dump of local name table:\n"));

  for (n = namelist; n; n = n->next) {
    DEBUG(3,("%s(%x) %s TTL=%d Unique=%s\n",
	     n->name.name,n->name.name_type,
	     inet_ntoa(n->ip),
	     n->death_time?n->death_time-t:0,
	     BOOLSTR(n->unique)));
    }

  DEBUG(3,("\nDump of domain list:\n"));
  for (d = domainlist; d; d = d->next)
    DEBUG(3,("%s %s\n",d->name,inet_ntoa(d->bcast_ip)));
}


/****************************************************************************
  add a permanent host entry to the name list
  ****************************************************************************/
static struct name_record *add_host_entry(char *name,int type,BOOL unique,int ttl,
					  enum name_source source,
					  struct in_addr ip)
{
  struct name_record *n = (struct name_record *)malloc(sizeof(*n));
  struct name_record *n2=NULL;

  if (!n) return(NULL);

  bzero((char *)n,sizeof(*n));

  make_nmb_name(&n->name,name,type,scope);
  if ((n2=find_name(&n->name))) {
    free(n);
    n = n2;
  }

  if (ttl) n->death_time = time(NULL)+ttl;
  n->ip = ip;
  n->unique = unique;
  n->source = source;
  
  if (!n2) add_name(n);

  DEBUG(3,("Added host entry %s(%x) at %s ttl=%d unique=%s\n",
	   name,type,inet_ntoa(ip),ttl,BOOLSTR(unique)));

  return(n);
}


/****************************************************************************
  add a domain entry
  ****************************************************************************/
static struct domain_record *add_domain_entry(char *name,struct in_addr ip)
{
  struct domain_record *d = (struct domain_record *)malloc(sizeof(*d));

  if (!d) return(NULL);

  bzero((char *)d,sizeof(*d));

  StrnCpy(d->name,name,sizeof(d->name)-1);
  d->bcast_ip = ip;
  
  add_domain(d);

  ip = *interpret_addr2("255.255.255.255");
  if (name[0] != '*') add_host_entry(name,0x1e,False,0,SELF,ip);	  

  DEBUG(3,("Added domain entry %s at %s\n",
	   name,inet_ntoa(ip)));

  return(d);
}


/****************************************************************************
  add the magic samba names, useful for finding samba servers
  **************************************************************************/
static void add_my_names(void)
{
  struct in_addr ip = *interpret_addr2("0.0.0.0");

  add_host_entry("__SAMBA__",0x20,True,0,SELF,ip);
  add_host_entry("__SAMBA__",0x0,True,0,SELF,ip);
  add_host_entry(myname,0x20,True,0,SELF,ip);
  add_host_entry(myname,0x0,True,0,SELF,ip);
  add_host_entry(myname,0x3,True,0,SELF,ip);
  if (!domainlist)
    add_domain_entry(WORKGROUP,bcast_ip);
}


/*******************************************************************
  delete old names from the namelist
  ******************************************************************/
static void housekeeping(void)
{
  time_t lastrun=0;
  time_t t = time(NULL);
  struct name_record *n;
  struct name_record *next;

  if (!lastrun) lastrun = t;
  if (t < lastrun + 5*60) return;

  lastrun = t;

  for (n = namelist; n; n = next) {
    if (n->death_time && n->death_time < t) {
      DEBUG(3,("Removing dead name %s(%x)\n",
	       n->name.name,n->name.name_type));
      next = n->next;
      if (n->prev) n->prev->next = n->next;
      if (n->next) n->next->prev = n->prev;
      if (namelist == n) namelist = n->next; 
      free(n);
    } else {
      next = n->next;
    }
  }
}

/****************************************************************************
load a netbios hosts file
****************************************************************************/
static void load_hosts_file(char *fname)
{
  FILE *f = fopen(fname,"r");
  pstring line;
  if (!f) {
    DEBUG(2,("Can't open lmhosts file %s\n",fname));
    return;
  }

  while (!feof(f))
    {
      if (!fgets_slash(line,sizeof(pstring),f)) continue;
      
      if (*line == '#') continue;

      {
	BOOL group=False;
	string ip="",name="",flags="",extra="";
	char *ptr;
	int count = 0;
	struct in_addr ipaddr;
	enum name_source source = LMHOSTS;

	ptr = line;

	if (next_token(&ptr,ip,NULL)) ++count;
	if (next_token(&ptr,name,NULL)) ++count;
	if (next_token(&ptr,flags,NULL)) ++count;
	if (next_token(&ptr,extra,NULL)) ++count;

	if (count <= 0) continue;

	if (count > 0 && count < 2)
	  {
	    DEBUG(0,("Ill formed hosts line [%s]\n",line));	    
	    continue;
	  }

	if (strchr(flags,'G') || strchr(flags,'S'))
	  group = True;

	if (strchr(flags,'M') && !group) {
	  source = SELF;
	  strcpy(myname,name);
	}

	ipaddr = *interpret_addr2(ip);

	if (group) {
	  add_domain_entry(name,ipaddr);
	} else {
	  add_host_entry(name,0x20,True,0,source,ipaddr);
	}
      }
    }

  fclose(f);
}

/*******************************************************************
  check if 2 IPs are on the same net
  we will assume the local netmask, although this could be wrong XXXX
  ******************************************************************/
static BOOL same_net(struct in_addr ip1,struct in_addr ip2)
{
  unsigned int net1,net2,nmask;

  putip((char *)&nmask,(char *)&Netmask);
  putip((char *)&net1,(char *)&ip1);
  putip((char *)&net2,(char *)&ip2);
	    
  return((net1 & nmask) == (net2 & nmask));
}


/****************************************************************************
  construct a host announcement unicast
  **************************************************************************/
static void announce_host(struct domain_record *d,char *my_name,char *Comment)
{
  time_t t = time(NULL);
  pstring outbuf;
  char *p;

  /* announce every minute at first then progress to every 15 mins */
  if (d->lastannounce_time && 
      (t - d->lastannounce_time) < d->announce_interval)
    return;

  if (d->announce_interval < 15*60) d->announce_interval += 60;
  d->lastannounce_time = t;

  DEBUG(2,("Sending host announcement to %s for workgroup %s\n",
	   inet_ntoa(d->bcast_ip),d->name));	   

  if (!*Comment) Comment = "NoComment";
  if (!*my_name) my_name = "NoName";

  if (strlen(Comment) > 43) Comment[43] = 0;  

  bzero(outbuf,sizeof(outbuf));
  p = outbuf;
  CVAL(p,0) = 1; /* host announce */
  SSVAL(p,1,0x6006); /* update count?? */
  CVAL(p,3) = 0xEA; /* res1 */ 
  SSVAL(p,4,d->announce_interval);
  p += 6;
  StrnCpy(p,my_name,16);
  strupper(p);
  p += 16;
  CVAL(p,0) = 1; /* major version (was 1) */
  CVAL(p,1) = 0x33; /* minor version (was 51) */
  SIVAL(p,2,0xB03); /* server and w'station + unix + printq + domain member*/
  SSVAL(p,6,0x30B); /* election version */
  SSVAL(p,8,0xAA55); /* browse constant */
  p += 10;
  strcpy(p,Comment);
  p += strlen(p)+1;

  send_udp_dgram(ClientDGRAM,outbuf,PTR_DIFF(p,outbuf),
		 my_name,d->name,0,0x1d,d->bcast_ip,myip);
}


/****************************************************************************
process a browse frame
****************************************************************************/
static void process_browse_packet(struct packet_struct *p,char *buf,int len)
{
  /* don't actually do anything with these at the moment */
#if 0
  int command = CVAL(buf,0);
  switch (command) {
  case 0xc: /* domain announcement */
    {
      fstring group;
      StrnCpy(group,buf+6,15);
      DEBUG(2,("Got workgroup announce for %s (%s)\n",
	       group,inet_ntoa(lastip)));
      process_workgroup_announce(p,group);
      break;
    }
  }
#endif
}


/****************************************************************************
process udp 138 datagrams
****************************************************************************/
static void process_dgram(struct packet_struct *p)
{
  char *buf;
  int len;
  struct dgram_packet *dgram = &p->packet.dgram;

  if (dgram->header.msg_type != 0x10 &&
      dgram->header.msg_type != 0x11 &&
      dgram->header.msg_type != 0x12) {
    /* don't process error packets etc yet */
    return;
  }

  buf = &dgram->data[0];
  buf -= 4; /* XXXX for the pseudo tcp length - 
	       someday I need to get rid of this */

  if (CVAL(buf,smb_com) != SMBtrans) return;
  if (!strequal(smb_buf(buf),"\\MAILSLOT\\BROWSE")) return;
  len = SVAL(buf,smb_vwv11);
  buf = smb_base(buf) + SVAL(buf,smb_vwv12);
  if (len <= 0) return;
  process_browse_packet(p,buf,len);
}

/*******************************************************************
  find a workgroup using the specified broadcast
  ******************************************************************/
static BOOL find_workgroup(char *name,struct in_addr ip)
{
  fstring name1;
  BOOL ret;
  struct in_addr ipout;

  strcpy(name1,"\001\002__MSBROWSE__\002");

  ret = name_query(ClientNMB,name1,0x1,True,False,ip,&ipout,queue_packet);
  if (!ret) return(False);

  name_status(ClientNMB,name1,0x1,False,ipout,name,queue_packet);

  if (name[0] != '*') {
    DEBUG(2,("Found workgroup %s on broadcast %s\n",name,inet_ntoa(ip)));
  } else {
    DEBUG(3,("Failed to find workgroup %s on broadcast %s\n",name,inet_ntoa(ip)));
  }
  return(name[0] != '*');
}


/****************************************************************************
  a hook for browsing handling - called every minute
  **************************************************************************/
static void do_browse_hook(void)
{
  struct domain_record *d;

  for (d = domainlist; d; d = d->next) {
    /* if the ip address is 0 then set to the broadcast */
    if (zero_ip(d->bcast_ip)) d->bcast_ip = bcast_ip;

    /* if the workgroup is '*' then find a workgroup to be part of */
    if (d->name[0] == '*') {
      if (!find_workgroup(d->name,d->bcast_ip)) continue;
      add_host_entry(d->name,0x1e,False,0,SELF,*interpret_addr2("255.255.255.255"));
    }

    announce_host(d,myname,comment);
  }
}



/****************************************************************************
reply to a name release
****************************************************************************/
static void reply_name_release(struct packet_struct *p)
{
  struct nmb_packet *nmb = &p->packet.nmb;
  struct packet_struct p2;
  struct nmb_packet *nmb2;
  struct res_rec answer_rec;
  struct in_addr ip;
  int rcode=0;
  int nb_flags = nmb->additional->rdata[0];

  putip((char *)&ip,&nmb->additional->rdata[2]);  

  {
    struct name_record *n = find_name(&nmb->question.question_name);
    if (n && n->unique && n->source == REGISTER &&
	ip_equal(ip,n->ip)) {
      remove_name(n); n = NULL;
    }

    /* XXXX under what conditions should we reject the removal?? */
  }

  DEBUG(3,("Name release on name %s(%x) rcode=%d\n",
	   nmb->question.question_name.name,
	   nmb->question.question_name.name_type,rcode));


  /* Send a NAME RELEASE RESPONSE */
  p2 = *p;
  nmb2 = &p2.packet.nmb;

  nmb2->header.response = True;
  nmb2->header.nm_flags.bcast = False;
  nmb2->header.nm_flags.recursion_available = CanRecurse;
  nmb2->header.nm_flags.trunc = False;
  nmb2->header.nm_flags.authoritative = True; 
  nmb2->header.qdcount = 0;
  nmb2->header.ancount = 1;
  nmb2->header.nscount = 0;
  nmb2->header.arcount = 0;
  nmb2->header.rcode = rcode;

  nmb2->answers = &answer_rec;
  bzero((char *)nmb2->answers,sizeof(*nmb2->answers));
  
  nmb2->answers->rr_name = nmb->question.question_name;
  nmb2->answers->rr_type = nmb->question.question_type;
  nmb2->answers->rr_class = nmb->question.question_class;
  nmb2->answers->ttl = 0; 
  nmb2->answers->rdlength = 6;
  nmb2->answers->rdata[0] = nb_flags;
  putip(&nmb2->answers->rdata[2],(char *)&ip);

  send_packet(&p2);
}



/****************************************************************************
  reply to a reg request
  **************************************************************************/
static void reply_name_reg(struct packet_struct *p)
{
  struct nmb_packet *nmb = &p->packet.nmb;
  char *qname = nmb->question.question_name.name;
  BOOL wildcard = (qname[0] == '*'); 
  BOOL bcast = nmb->header.nm_flags.bcast;
  int ttl = nmb->additional->ttl;
  int name_type = nmb->question.question_name.name_type;
  int nb_flags = nmb->additional->rdata[0];
  struct packet_struct p2;
  struct nmb_packet *nmb2;
  struct res_rec answer_rec;
  struct in_addr ip;
  BOOL group = (nb_flags&0x80)?True:False;
  int rcode = 0;  

  if (wildcard) return;

  putip((char *)&ip,&nmb->additional->rdata[2]);

  if (group) {
    /* apparently we should return 255.255.255.255 for group queries (email from MS) */
    ip = *interpret_addr2("255.255.255.255");
  }

  {
    struct name_record *n = find_name(&nmb->question.question_name);

    if (n) {
      if (!group && !ip_equal(ip,n->ip)) {
	/* check if the previous owner still wants it, 
	   if so reject the registration, otherwise change the owner 
	   and refresh */
	/* XXXXXX not done yet */
	rcode = 5;
      } else {
	/* refresh the name */
	n->death_time = ttl?p->timestamp + ttl:0;
      }
    } else {
      /* add the name to our database */
      n = add_host_entry(qname,name_type,!group,ttl,REGISTER,ip);
    }
  }

  if (bcast) return;

  DEBUG(3,("Name registration for name %s(%x) at %s rcode=%d\n",
	   nmb->question.question_name.name,
	   nmb->question.question_name.name_type,
	   inet_ntoa(ip),rcode));

  /* Send a NAME REGISTRATION RESPONSE */
  /* a lot of fields get copied from the query. This gives us the IP
     and port the reply will be sent to etc */
  p2 = *p;
  nmb2 = &p2.packet.nmb;

  nmb2->header.opcode = 5; 
  nmb2->header.response = True;
  nmb2->header.nm_flags.bcast = False;
  nmb2->header.nm_flags.recursion_available = CanRecurse;
  nmb2->header.nm_flags.trunc = False;
  nmb2->header.nm_flags.authoritative = True; 
  nmb2->header.qdcount = 0;
  nmb2->header.ancount = 1;
  nmb2->header.nscount = 0;
  nmb2->header.arcount = 0;
  nmb2->header.rcode = rcode;

  nmb2->answers = &answer_rec;
  bzero((char *)nmb2->answers,sizeof(*nmb2->answers));
  
  nmb2->answers->rr_name = nmb->question.question_name;
  nmb2->answers->rr_type = nmb->question.question_type;
  nmb2->answers->rr_class = nmb->question.question_class;

  nmb2->answers->ttl = ttl; 
  nmb2->answers->rdlength = 6;
  nmb2->answers->rdata[0] = nb_flags;
  putip(&nmb2->answers->rdata[2],(char *)&ip);

  send_packet(&p2);  
}


/****************************************************************************
reply to a name status query
****************************************************************************/
static void reply_name_status(struct packet_struct *p)
{
  struct nmb_packet *nmb = &p->packet.nmb;
  char *qname = nmb->question.question_name.name;
  BOOL wildcard = (qname[0] == '*'); 
  struct packet_struct p2;
  struct nmb_packet *nmb2;
  struct res_rec answer_rec;
  char *buf;
  int count;
  int rcode = 0;
  struct name_record *n = find_name(&nmb->question.question_name);

  if (!wildcard && (!n || n->source != SELF)) 
    return;


  DEBUG(3,("Name status for name %s(%x) rcode=%d\n",
	   nmb->question.question_name.name,
	   nmb->question.question_name.name_type,
	   rcode));

  /* Send a POSITIVE NAME STATUS RESPONSE */
  /* a lot of fields get copied from the query. This gives us the IP
     and port the reply will be sent to etc */
  p2 = *p;
  nmb2 = &p2.packet.nmb;

  nmb2->header.response = True;
  nmb2->header.nm_flags.bcast = False;
  nmb2->header.nm_flags.recursion_available = CanRecurse;
  nmb2->header.nm_flags.trunc = False;
  nmb2->header.nm_flags.authoritative = True; /* WfWg ignores 
						 non-authoritative answers */
  nmb2->header.qdcount = 0;
  nmb2->header.ancount = 1;
  nmb2->header.nscount = 0;
  nmb2->header.arcount = 0;
  nmb2->header.rcode = rcode;

  nmb2->answers = &answer_rec;
  bzero((char *)nmb2->answers,sizeof(*nmb2->answers));
  

  nmb2->answers->rr_name = nmb->question.question_name;
  nmb2->answers->rr_type = nmb->question.question_type;
  nmb2->answers->rr_class = nmb->question.question_class;
  nmb2->answers->ttl = 0; /* XXX what ttl to answer with? */

  for (count=0, n = namelist ; n; n = n->next) count++;

  count = MIN(count,400/18); /* XXXX hack, we should calculate exactly
				how many will fit */

  
  buf = &nmb2->answers->rdata[0];
  SCVAL(buf,0,count);
  buf += 1;

  for (n = namelist ; n; n = n->next) 
    {
      bzero(buf,18);
      strcpy(buf,n->name.name);
      strupper(buf);
      buf[15] = n->name.name_type;
      buf += 16;
      buf[0] = 0x4; /* active */
      if (strequal(n->name.name,myname)) buf[0] |= 0x2; /* permanent */
      if (!n->unique) buf[0] |= 0x80; /* group */
      buf += 2;
      count--;
    }

  /* we should fill in more fields of the statistics structure */
  bzero(buf,46);
  putip(buf,(char *)&myip);
  {
    extern int num_good_sends,num_good_receives;
    SIVAL(buf,20,num_good_sends);
    SIVAL(buf,24,num_good_receives);
  }

  buf += 46;

  nmb2->answers->rdlength = PTR_DIFF(buf,&nmb2->answers->rdata[0]);

  send_packet(&p2);
}



/****************************************************************************
reply to a name query
****************************************************************************/
static void reply_name_query(struct packet_struct *p)
{
  struct nmb_packet *nmb = &p->packet.nmb;
  char *qname = nmb->question.question_name.name;
  BOOL wildcard = (qname[0] == '*'); 
  BOOL bcast = nmb->header.nm_flags.bcast;
  struct in_addr retip;
  int name_type = nmb->question.question_name.name_type;
  struct packet_struct p2;
  struct nmb_packet *nmb2;
  struct res_rec answer_rec;
  int ttl=0;
  int rcode=0;
  BOOL unique = True;

  if (wildcard)
    retip = myip;

  if (!wildcard) {
    struct name_record *n = find_name(&nmb->question.question_name);

    if (!n) {
      struct in_addr ip;
      unsigned long a;

      /* only do DNS lookups if the query is for type 0x20 or type 0x0 */
      if (name_type != 0x20 && name_type != 0) return;

      /* look it up with DNS */      
      a = interpret_addr(qname);

      if (!a) {
	/* no luck with DNS. We could possibly recurse here XXXX */
	return;
      }

      /* add it to our cache of names. give it 2 hours in the cache */
      putip((char *)&ip,(char *)&a);
      n = add_host_entry(qname,name_type,True,2*60*60,DNS,ip);

      /* failed to add it? yikes! */
      if (!n) return;
    }

    /* don't respond to bcast queries for group names */
    if (bcast && !n->unique) return;

    /* don't respond to bcast queries for addresses on the same net as the machine 
       doing the querying unless its our IP (or something from LMHOSTS) */
    if (bcast && 
	n->source != SELF && n->source != LMHOSTS && 
	same_net(n->ip,p->ip)) return;

    /* is our entry already dead? */
    if (n->death_time) {
      if (n->death_time < p->timestamp) return;
      ttl = n->death_time - p->timestamp;
    }

    retip = n->ip;
    unique = n->unique;
  } 

  /* if the IP is 0 then substitute my IP - we should see which one is on the 
     right interface for the caller to do this right */
  if (zero_ip(retip)) retip = myip;
  

  DEBUG(3,("Name query for name %s(%x) at %s rcode=%d\n",
	   nmb->question.question_name.name,
	   nmb->question.question_name.name_type,
	   inet_ntoa(retip),rcode));


  /* a lot of fields get copied from the query. This gives us the IP
     and port the reply will be sent to etc */
  p2 = *p;
  nmb2 = &p2.packet.nmb;

  nmb2->header.response = True;
  nmb2->header.nm_flags.bcast = False;
  nmb2->header.nm_flags.recursion_available = CanRecurse;
  nmb2->header.nm_flags.trunc = False;
  nmb2->header.nm_flags.authoritative = True; /* WfWg ignores 
						 non-authoritative answers */
  nmb2->header.qdcount = 0;
  nmb2->header.ancount = 1;
  nmb2->header.nscount = 0;
  nmb2->header.arcount = 0;
  nmb2->header.rcode = rcode;

  nmb2->answers = &answer_rec;
  bzero((char *)nmb2->answers,sizeof(*nmb2->answers));

  nmb2->answers->rr_name = nmb->question.question_name;
  nmb2->answers->rr_type = nmb->question.question_type;
  nmb2->answers->rr_class = nmb->question.question_class;
  nmb2->answers->ttl = ttl; /* XXX what ttl to answer with? */
  nmb2->answers->rdlength = 6;
  nmb2->answers->rdata[0] = unique?0:0x80; 
  nmb2->answers->rdata[1] = 0; 
  putip(&nmb2->answers->rdata[2],(char *)&retip);

  send_packet(&p2);
}



/* the global packet linked-list. incoming entries are added to the
   end of this list.  it is supposed to remain fairly short so we
   won't bother with an end pointer. */
static struct packet_struct *packet_queue = NULL;


/*******************************************************************
  queue a packet into the packet queue
  ******************************************************************/
static void queue_packet(struct packet_struct *packet)
{
  struct packet_struct *p;
  if (!packet_queue) {
    packet->prev = NULL;
    packet->next = NULL;
    packet_queue = packet;
    return;
  }
  
  /* find the bottom */
  for (p=packet_queue;p->next;p=p->next) ;

  p->next = packet;
  packet->next = NULL;
  packet->prev = p;
}

/****************************************************************************
  process a nmb packet
  ****************************************************************************/
static void process_nmb(struct packet_struct *p)
{
  struct nmb_packet *nmb = &p->packet.nmb;

  /* if this is a response then ignore it */
  if (nmb->header.response) return;
  
  if (!nmb->header.nm_flags.bcast && 
      nmb->header.opcode == 0x5 && 
      nmb->header.qdcount==1 && 
      nmb->header.arcount==1)
    {
      reply_name_reg(p);
      return;
    }

  if (nmb->header.opcode==0 && 
      nmb->header.qdcount==1)
    {
      switch (nmb->question.question_type)
	{
	case 0x20:
	  reply_name_query(p);
	  break;

	case 0x21:
	  reply_name_status(p);
	  break;
	}
      return;
    }

  if (!nmb->header.nm_flags.bcast && 
      nmb->header.opcode == 6 && 
      nmb->header.qdcount==1 && 
      nmb->header.arcount==1)
    {
      reply_name_release(p);
      return;
    }

  if (!nmb->header.nm_flags.bcast && 
      ((nmb->header.opcode == 8) || (nmb->header.opcode == 9)) && 
      nmb->header.qdcount==1 && 
      nmb->header.arcount==1)
    {
      reply_name_reg(p);
      return;
    }
}



/*******************************************************************
  run elements off the packet queue till its empty
  ******************************************************************/
static void run_packet_queue(void)
{
  struct packet_struct *p;

  while ((p=packet_queue)) {
    switch (p->packet_type)
      {
      case NMB_PACKET:
	process_nmb(p);
	break;

      case DGRAM_PACKET:
	process_dgram(p);
	break;
      }

    packet_queue = packet_queue->next;
    if (packet_queue) packet_queue->prev = NULL;
    free_packet(p);
  }
}


/****************************************************************************
  The main select loop, listen for packets and respond
  ***************************************************************************/
void process(void)
{

  while (True)
    {
      fd_set fds;
      int selrtn;
      struct timeval timeout;

      FD_ZERO(&fds);
      FD_SET(ClientNMB,&fds);
      FD_SET(ClientDGRAM,&fds);
      timeout.tv_sec = NMBD_SELECT_LOOP;
      timeout.tv_usec = 0;

      selrtn = sys_select(&fds,&timeout);

      if (FD_ISSET(ClientNMB,&fds)) {
	struct packet_struct *packet = read_packet(ClientNMB,NMB_PACKET);
	if (packet) queue_packet(packet);
      }

      if (FD_ISSET(ClientDGRAM,&fds)) {
	struct packet_struct *packet = read_packet(ClientDGRAM,DGRAM_PACKET);
	if (packet) queue_packet(packet);
      }

      run_packet_queue();

      do_browse_hook();

      housekeeping();
    }
}


/****************************************************************************
  open the socket communication
****************************************************************************/
static BOOL open_sockets(BOOL isdaemon,int port)
{
  struct hostent *hp;
 
  /* get host info */
  if ((hp = Get_Hostbyname(myhostname)) == 0) 
    {
      DEBUG(0,( "Get_Hostbyname: Unknown host. %s\n",myhostname));
      return False;
    }   

  if (isdaemon)
    ClientNMB = open_socket_in(SOCK_DGRAM, port,0);
  else
    ClientNMB = 0;

  ClientDGRAM = open_socket_in(SOCK_DGRAM,DGRAM_PORT,3);

  if (ClientNMB == -1)
    return(False);

  signal(SIGPIPE, SIGNAL_CAST sig_pipe);

  set_socket_options(ClientNMB,"SO_BROADCAST");
  set_socket_options(ClientDGRAM,"SO_BROADCAST");
  set_socket_options(ClientNMB,user_socket_options);

  DEBUG(3, ("Socket opened.\n"));
  return True;
};


/****************************************************************************
  initialise connect, service and file structs
****************************************************************************/
static BOOL init_structs(void )
{
  if (!get_myname(myhostname,got_myip?NULL:&myip))
    return(False);

  /* Read the broadcast address from the interface */
  {
    struct in_addr ip0,ip1,ip2;

    ip0 = myip;

    if (!(got_bcast && got_nmask))
      {
	get_broadcast(&ip0,&ip1,&ip2);

	if (!got_myip)
	  myip = ip0;
    
	if (!got_bcast)
	  bcast_ip = ip1;
    
	if (!got_nmask)
	  Netmask = ip2;   
      } 

    DEBUG(1,("Using IP %s  ",inet_ntoa(myip))); /* core dump reported 
						   doing this. Why?? XXXXX  */
    DEBUG(1,("broadcast %s  ",inet_ntoa(bcast_ip)));
    DEBUG(1,("netmask %s\n",inet_ntoa(Netmask)));    

  }

  if (! *myname) {
    char *p;
    strcpy(myname,myhostname);
    p = strchr(myname,'.');
    if (p) *p = 0;
  }

  return True;
}

/****************************************************************************
usage on the program
****************************************************************************/
static void usage(char *pname)
{
  DEBUG(0,("Incorrect program usage - is the command line correct?\n"));

  printf("Usage: %s [-n name] [-B bcast address] [-D] [-p port] [-d debuglevel] [-l log basename]\n",pname);
  printf("Version %s\n",VERSION);
  printf("\t-D                    become a daemon\n");
  printf("\t-p port               listen on the specified port\n");
  printf("\t-d debuglevel         set the debuglevel\n");
  printf("\t-l log basename.      Basename for log/debug files\n");
  printf("\t-n netbiosname.       the netbios name to advertise for this host\n");
  printf("\t-B broadcast address  the address to use for broadcasts\n");
  printf("\t-N netmask           the netmask to use for subnet determination\n");
  printf("\t-H hosts file        load a netbios hosts file\n");
  printf("\t-G group name        add a group name to be part of\n");
  printf("\t-C comment           sets the machine comment that appears in browse lists\n");
  printf("\n");
}


/****************************************************************************
  main program
  **************************************************************************/
int main(int argc,char *argv[])
{
  int port = NMB_PORT;
  int opt;
  extern FILE *dbf;
  extern char *optarg;

  *host_file = 0;

  TimeInit();

  charset_initialise();

  strcpy(debugf,NMBLOGFILE);

#ifdef LMHOSTSFILE
  strcpy(host_file,LMHOSTSFILE);
#endif

  /* this is for people who can't start the program correctly */
  while (argc > 1 && (*argv[1] != '-'))
    {
      argv++;
      argc--;
    }

  fault_setup(fault_continue);

  signal(SIGHUP,SIGNAL_CAST sig_hup);

  bcast_ip = *interpret_addr2("0.0.0.0");
  myip = *interpret_addr2("0.0.0.0");

  while ((opt = getopt (argc, argv, "T:O:I:C:bAi:B:N:Rn:l:d:Dp:hSH:G:")) != EOF)
    switch (opt)
      {
      case 'O':
	strcpy(user_socket_options,optarg);
	break;
      case 'C':
	strcpy(comment,optarg);
	break;
      case 'G':
	add_domain_entry(optarg,bcast_ip);
	break;
      case 'H':
	strcpy(host_file,optarg);
	break;
      case 'I':
	myip = *interpret_addr2(optarg);
	got_myip = True;
	break;
      case 'B':
	bcast_ip = *interpret_addr2(optarg);
	got_bcast = True;
	break;
      case 'N':
	Netmask = *interpret_addr2(optarg);
	got_nmask = True;
	break;
      case 'n':
	strcpy(myname,optarg);
	break;
      case 'l':
	sprintf(debugf,"%s.nmb",optarg);
	break;
      case 'i':
	strcpy(scope,optarg);
	strupper(scope);
	break;
      case 'D':
	is_daemon = True;
	break;
      case 'd':
	DEBUGLEVEL = atoi(optarg);
	break;
      case 'p':
	port = atoi(optarg);
	break;
      case 'h':
	usage(argv[0]);
	exit(0);
	break;
      default:
	usage(argv[0]);
	exit(1);
      }

  
  if (DEBUGLEVEL > 10)
    {
      extern FILE *login,*logout;
      pstring fname;
      sprintf(fname,"%s.in",debugf);
      login = fopen(fname,"w"); 
      sprintf(fname,"%s.out",debugf);
      logout = fopen(fname,"w");
    }
  
  DEBUG(1,("%s netbios nameserver version %s started\n",timestring(),VERSION));
  DEBUG(1,("Copyright Andrew Tridgell 1994\n"));

  if (*host_file)
    {
      load_hosts_file(host_file);
      DEBUG(3,("Loaded hosts file\n"));
    }

  init_structs();

  if (!*comment)
    strcpy(comment,"Samba %v");
  string_sub(comment,"%v",VERSION);
  string_sub(comment,"%h",myhostname);

  add_my_names();

  DEBUG(3,("Checked names\n"));
  
  dump_names();

  DEBUG(3,("Dumped names\n"));

  if (!is_daemon && !is_a_socket(0)) {
    DEBUG(0,("standard input is not a socket, assuming -D option\n"));
    is_daemon = True;
  }
  

  if (is_daemon) {
    DEBUG(2,("%s becoming a daemon\n",timestring()));
    become_daemon();
  }


  DEBUG(3,("Opening sockets\n"));

  if (open_sockets(is_daemon,port))
    {
      process();
      close_sockets();
    }

  if (dbf)
    fclose(dbf);
  return(0);
}
