/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   status reporting
   Copyright (C) Andrew Tridgell 1994-1995
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * This program reports current SMB connections
 */

#include "includes.h"
#include "loadparm.h"

struct connect_record crec;
extern int DEBUGLEVEL;
extern FILE *dbf;

int main(int argc, char *argv[])
{
  FILE *f;
  pstring fname="";
  int uid, c, n;
  static pstring servicesf = CONFIGFILE;
  extern char *optarg;
  int verbose = 0;
  void *dir;
  char *s;
  BOOL firstopen=True;
  BOOL processes_only=False;

  charset_initialise();

  DEBUGLEVEL = 0;
  dbf = fopen("/dev/null","w");

  if (getuid() != geteuid()) {
    printf("smbstatus should not be run setuid\n");
    return(1);
  }

  while ((c = getopt(argc, argv, "pds:")) != EOF) {
    switch (c) {
    case 'd':
      verbose = 1;
      break;
    case 'p':
      processes_only = 1;
      break;
    case 's':
      strcpy(servicesf, optarg);
      break;
    default:
      fprintf(stderr, "Usage: %s [-p] [-d] [-s configfile]\n", *argv);
      return (-1);
    }
  }

  if (!lp_load(servicesf)) {
    fprintf(stderr, "Can't load %s - run testparm to debug it\n", servicesf);
    return (-1);
  }

  if (verbose) {
    printf("using configfile = %s\n", servicesf);
    printf("lockdir = %s\n", *lp_lockdir() ? lp_lockdir() : "NULL");
  }

  strcpy(fname,lp_lockdir());
  standard_sub_basic(fname);
  trim_string(fname,"","/");
  strcat(fname,"/STATUS..LCK");

  f = fopen(fname,"r");
  if (!f) {
    printf("Couldn't open status file %s\n",fname);
    if (!lp_status(-1))
      printf("You need to have status=yes in your smb config file\n");
    return(0);
  }

  uid = getuid();

  if (!processes_only) {
    printf("\nSamba version %s\n",VERSION);
    
    printf("Service      uid      gid      pid     machine\n");
    printf("----------------------------------------------\n");
  }

  while (!feof(f))
    {
      static int last_pid = 0;
      if (fread(&crec,sizeof(crec),1,f) != 1)
	break;
      if (crec.magic == 0x280267 && process_exists(crec.pid)) {
	if (processes_only) {
	  if (last_pid != crec.pid)
	    printf("%d\n",crec.pid);
	  last_pid = crec.pid; /* XXXX we can still get repeats, have to
				  add a sort at some time */
	}
	else
	  printf("%-10.10s   %-8s %-8s %5d   %-8s (%s) %s",
		 crec.name,uidtoname(crec.uid),gidtoname(crec.gid),crec.pid,
		 crec.machine,crec.addr,
		 asctime(LocalTime(&crec.start,GMT_TO_LOCAL)));
      }
    }
  fclose(f);

  if (processes_only) exit(0);

  printf("\n");

  dir = opendir(lp_lockdir());
  if (!dir) return(0);
  while ((s=readdirname(dir))) {
    char buf[16];
    int pid,mode;
    time_t t;
    int fd;
    pstring lname;
    int dev,inode;

    if (sscanf(s,"share.%d.%d",&dev,&inode)!=2) continue;

    strcpy(lname,lp_lockdir());
    trim_string(lname,NULL,"/");
    strcat(lname,"/");
    strcat(lname,s);

    fd = open(lname,O_RDONLY,0);
    if (fd < 0) continue;
    if (read(fd,buf,16) != 16) continue;
    n = read(fd,fname,sizeof(fname));
    fname[MAX(n,0)]=0;
    close(fd);

    t = IVAL(buf,0);
    mode = IVAL(buf,4);
    pid = IVAL(buf,8);

    if (IVAL(buf,12) != LOCKING_VERSION || !process_exists(pid)) {
      if (unlink(lname)==0)
	printf("Deleted stale share file %s\n",s);
      continue;
    }

    fname[sizeof(fname)-1] = 0;

    if (firstopen) {
      firstopen=False;
      printf("Locked files:\n");
      printf("Pid    DenyMode   R/W     Name\n");
      printf("------------------------------\n");
    }


    printf("%-5d  ",pid);
    switch ((mode>>4)&0xF)
      {
      case DENY_NONE: printf("DENY_NONE  "); break;
      case DENY_ALL:  printf("DENY_ALL   "); break;
      case DENY_DOS:  printf("DENY_DOS   "); break;
      case DENY_READ: printf("DENY_READ  "); break;
      case DENY_WRITE:printf("DENY_WRITE "); break;
      }
    switch (mode&0xF) 
      {
      case 0: printf("RDONLY "); break;
      case 1: printf("WRONLY "); break;
      case 2: printf("RDWR   "); break;
      }
    printf(" %s   %s",fname,asctime(LocalTime(&t,GMT_TO_LOCAL)));
  }
  closedir(dir);

  if (firstopen)
    printf("No locked files\n");

  return (0);
}

