/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   Copyright (C) Andrew Tridgell 1992,1993,1994
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include "loadparm.h"

extern int DEBUGLEVEL;
extern connection_struct Connections[];

/*
   This module implements directory related functions for Samba.
*/



static struct dptr_struct
{
  int key;
  int pid;
  int cnum;
  void *ptr;
  BOOL valid;
  BOOL finished;
  BOOL expect_close;
  char *wcard; /* Field only used for lanman2 trans2_findfirst/next searches */
  uint16 attr; /* Field only used for lanman2 trans2_findfirst/next searches */
  pstring path;
}
dirptrs[MAXDIR];

static int next_key = 1;


/****************************************************************************
initialise the dir array
****************************************************************************/
void init_dptrs(void)
{
  static BOOL dptrs_init=False;
  int i;

  if (dptrs_init) return;
  for (i=0;i<MAXDIR;i++)    
    {
      dirptrs[i].valid = False;
      dirptrs[i].wcard = NULL;
      dirptrs[i].ptr = NULL;
    }
  dptrs_init = True;
}

/****************************************************************************
get the dir ptr for a dir index
****************************************************************************/
void *dptr_get(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      return(dirptrs[i].ptr);
  return(NULL);
}

/****************************************************************************
get the dir path for a dir index
****************************************************************************/
char *dptr_path(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      return(dirptrs[i].path);
  return(NULL);
}

/****************************************************************************
get the dir wcard for a dir index (lanman2 specific)
****************************************************************************/
char *dptr_wcard(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      return(dirptrs[i].wcard);
  return(NULL);
}

/****************************************************************************
set the dir wcard for a dir index (lanman2 specific)
Returns 0 on ok, 1 on fail.
****************************************************************************/
int dptr_set_wcard(int key, char *wcard)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      {
	dirptrs[i].wcard = wcard;
	return 0;
      }
  return 1;
}

/****************************************************************************
set the dir attrib for a dir index (lanman2 specific)
Returns 0 on ok, 1 on fail.
****************************************************************************/
int dptr_set_attr(int key, uint16 attr)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      {
	dirptrs[i].attr = attr;
	return 0;
      }
  return 1;
}

/****************************************************************************
get the dir attrib for a dir index (lanman2 specific)
****************************************************************************/
uint16 dptr_attr(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      return(dirptrs[i].attr);
  return(0);
}



/****************************************************************************
check a key
****************************************************************************/
BOOL key_ok(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      return(True);
  return(False);
}

/****************************************************************************
prompte a dptr (to make it recently used)
****************************************************************************/
void dptr_promote(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      {
	struct dptr_struct d;
	int j;
	d = dirptrs[i];

	for (j=i;j>0;j--)
	  dirptrs[j] = dirptrs[j-1];
	dirptrs[0] = d;
	return;
      }
}


/****************************************************************************
demote a dptr (to make it likely to get closed)
****************************************************************************/
void dptr_demote(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      {
	struct dptr_struct d;
	int j;
	d = dirptrs[i];
	for (j=i;j<(MAXDIR-1) && dirptrs[j].valid;j++)
	  dirptrs[j] = dirptrs[j+1];
	dirptrs[j] = d;
	return;
      }
}


/****************************************************************************
find a free key
****************************************************************************/
int find_key(int start)
{
  int key;

  for (key=start;key<256;key++)
    if (!key_ok(key))
      return(key);

  for (key=1;key<start;key++)
    if (!key_ok(key))
      return(key);

  DEBUG(0,("ERROR: Out of dirptr keys!\n"));
  return 1;
}       

/****************************************************************************
set the dptr finished flag
****************************************************************************/
void dptr_finished(unsigned int key,BOOL finished)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      {
	dirptrs[i].finished = finished;
	break;
      }
}
  

/****************************************************************************
close a dptr
****************************************************************************/
void dptr_close(int key)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].key == key)
      {
	DEBUG(6,("closing dptr_num %d\n",key));
	if (dirptrs[i].ptr)
	  CloseDir(dirptrs[i].ptr);
	/* Lanman 2 specific code */
	if (dirptrs[i].wcard)
	  free(dirptrs[i].wcard);
	dirptrs[i].valid = False;
	next_key = key+1;
	if (next_key > 255) next_key = 1;
	return;
      }
}

/****************************************************************************
close all dptrs for a cnum
****************************************************************************/
void dptr_closecnum(int cnum)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && dirptrs[i].cnum == cnum)
      {
	DEBUG(6,("closing dptr_num %d\n",dirptrs[i].key));
	if (dirptrs[i].ptr)
	  CloseDir(dirptrs[i].ptr);
	/* Lanman 2 specific code */
	if (dirptrs[i].wcard)
	  free(dirptrs[i].wcard);
	dirptrs[i].valid = False;
      }
}

/****************************************************************************
close a dptr that matches a given path, only if it matches the pid also
****************************************************************************/
void dptr_closepath(char *path,int pid)
{
  int i;
  for (i=0;i<MAXDIR;i++)
    if (dirptrs[i].valid && pid == dirptrs[i].pid)
      {      
	DEBUG(5,("closepath [%s] [%s]\n",path,dirptrs[i].path));
	if (strequal(dirptrs[i].path,path))
	  {
	    DEBUG(6,("closing dptr_num %d\n",dirptrs[i].key));
	    if (dirptrs[i].ptr)
	      CloseDir(dirptrs[i].ptr);
	    if (dirptrs[i].wcard)
	      free(dirptrs[i].wcard);
	    dirptrs[i].valid = False;
	    next_key = dirptrs[i].key+1;
	    if (next_key > 255) next_key = 1;
	  }
      }
}

/****************************************************************************
create a new dir ptr
****************************************************************************/
int dptr_create(int cnum,char *path, BOOL expect_close,int pid)
{
  int i,key;
  void *p = Connections[cnum].dirptr;

  for (i=0;i<MAXDIR;i++)
    if (!dirptrs[i].valid)
      break;

  /* as a second option, grab one not marked for expect_close but marked 
     as finished */
  if (i == MAXDIR)
    for (i=MAXDIR-1;i>=0;i--)
      if (dirptrs[i].finished && !dirptrs[i].expect_close)
	{
	  dptr_close(dirptrs[i].key);	
	  break;
	}

  /* as a third option, grab one not marked for expect_close */
  if (i < 0)
    for (i=MAXDIR-1;i>=0;i--)
      if (!dirptrs[i].expect_close)
	{
	  dptr_close(dirptrs[i].key);	
	  break;
	}

  if (i < 0)
    {
      /* All dirptrs are used by uncloseable SMBffirsts */
      DEBUG(0,("Error - all dirptrs in use by SMBffirsts.\n"));
      return(-1);
    }

  dirptrs[i].ptr = p;
  strcpy(dirptrs[i].path,path);
  key = find_key(next_key);
  dirptrs[i].key = key;
  dirptrs[i].valid = True;
  dirptrs[i].finished = False;
  dirptrs[i].cnum = cnum;
  dirptrs[i].pid = pid;
  dirptrs[i].expect_close = expect_close;
  dirptrs[i].wcard = NULL; /* Only used in lanman2 searches */
  dirptrs[i].attr = 0; /* Only used in lanman2 searches */

  dptr_promote(key);

  DEBUG(3,("creating new dirptr %d (0x%x) for path %s, expect_close = %d\n",key,p,path,
	expect_close));  

  return(key);
}

/****************************************************************************
fill the 5 byte server reserved dptr field
****************************************************************************/
BOOL dptr_fill(char *buf,unsigned int key)
{
  void *p = dptr_get(key);
  int offset;
  if (!p)
    {
      DEBUG(1,("filling null dirptr %d\n",key));
      return(False);
    }
  offset = TellDir(p);
  DEBUG(6,("fill on dirptr 0x%x now at %d\n",p,offset));
  buf[0] = key;
  SIVAL(buf,1,offset);
  DEBUG(6,("filled dirptr %d at offset %d\n",key,offset));
  return(True);
}


/****************************************************************************
return True is the offset is at zero
****************************************************************************/
BOOL dptr_zero(char *buf)
{
  return(IVAL(buf,0) == 0);
}

/****************************************************************************
fetch the dir ptr and seek it given the 5 byte server field
****************************************************************************/
void *dptr_fetch(char *buf,int *num)
{
  unsigned int key = *(unsigned char *)buf;
  void *p = dptr_get(key);
  int offset;
  if (!p)
    {
      DEBUG(3,("fetched null dirptr %d\n",key));
      return(NULL);
    }
  *num = key;
  offset = IVAL(buf,1);
  SeekDir(p,offset);
  dptr_promote(key);
  DEBUG(3,("fetching dirptr %d for path %s\n",key,dptr_path(key)));
  return(p);
}

/****************************************************************************
fetch the dir ptr and seek it given the lanman2 parameter block
****************************************************************************/
void *dptr_fetch_lanman2(char *params,int dptr_num)
{
  void *p = dptr_get(dptr_num);
  uint32 resume_key = SVAL(params,6);
  BOOL uses_resume_key = BITSETW(params+10,2);
  BOOL continue_bit = BITSETW(params+10,3);

  if (!p)
    {
      DEBUG(3,("fetched null dirptr %d\n",dptr_num));
      return(NULL);
    }
  if(uses_resume_key && !continue_bit)
    SeekDir(p,resume_key);
  dptr_promote(dptr_num);
  DEBUG(3,("fetching dirptr %d for path %s\n",dptr_num,dptr_path(dptr_num)));
  return(p);
}

/****************************************************************************
  start a directory listing
****************************************************************************/
BOOL start_dir(int cnum,char *directory)
{
  DEBUG(5,("start_dir cnum=%d dir=%s\n",cnum,directory));

  if (!check_name(directory,cnum))
    return(False);
  
  if (! *directory)
    directory = ".";

  Connections[cnum].dirptr = OpenDir(directory);

  string_set(&Connections[cnum].dirpath,directory);
  
  return(Connections[cnum].dirptr != NULL);
}

/****************************************************************************
  get a directory entry
****************************************************************************/
BOOL get_dir_entry(int cnum,char *mask,int dirtype,char *fname,int *size,int *mode,time_t *date,BOOL check_descend)
{
  char *dname;
  BOOL do_mangling;
  BOOL found = False;
  struct stat sbuf;
  pstring path;
  pstring pathreal;
  BOOL isrootdir;
  pstring filename;
  BOOL matched;
  BOOL has_wild = (strchr(mask,'?') || strchr(mask,'*'));

  *path = *pathreal = *filename = 0;

  isrootdir = (strequal(Connections[cnum].dirpath,"./") ||
	       strequal(Connections[cnum].dirpath,".") ||
	       strequal(Connections[cnum].dirpath,"/"));
  
  if (!Connections[cnum].dirptr)
    return(False);
  
  while (!found)
    {
      extern char magic_char;
      dname = ReadDirName(Connections[cnum].dirptr);

      DEBUG(6,("readdir on dirptr 0x%x now at offset %d\n",
	    Connections[cnum].dirptr,TellDir(Connections[cnum].dirptr)));
      
      if (dname == NULL) 
	return(False);
      
      matched = False;

      do_mangling = ((has_wild || strchr(mask,magic_char)) && 
		     lp_manglednames(SNUM(cnum)));

      strcpy(filename,dname);      

      if ((strcmp(filename,mask) == 0) ||
	  (name_convert(filename,dname,do_mangling,lp_mangled_map(SNUM(cnum))) &&
	   mask_match(filename,mask,False,False)))
	{
	  if (isrootdir && (strequal(filename,"..") || strequal(filename,".")))
	    continue;

	  strcpy(fname,filename);
	  *path = 0;
	  strcpy(path,Connections[cnum].dirpath);
	  strcat(path,"/");
	  strcpy(pathreal,path);
	  strcat(path,fname);
	  strcat(pathreal,dname);
	  if (stat(pathreal,&sbuf) != 0) 
	    {
	      DEBUG(5,("Couldn't stat 1 [%s]\n",path));
	      continue;
	    }

	  if (check_descend &&
	      !strequal(fname,".") && !strequal(fname,".."))
	    continue;
	  
	  *mode = dos_mode(cnum,pathreal,&sbuf);

	  if (((*mode & ~dirtype) & (aHIDDEN | aSYSTEM | aDIR)) != 0)
	    {	      
	      DEBUG(5,("[%s] attribs didn't match %x\n",filename,dirtype));
	      continue;
	    }
	  *size = sbuf.st_size;
	  *date = sbuf.st_mtime;

	  DEBUG(5,("get_dir_entry found %s fname=%s\n",pathreal,fname));
	  
	  found = True;
	}
    }

  return(found);
}



typedef struct
{
  int pos;
  void *dirptr;
} Dir;

/*******************************************************************
open a directory
********************************************************************/
void *OpenDir(char *name)
{
  Dir *dirp;
  void *p = opendir(name);
  if (!p) return(NULL);
  dirp = (Dir *)malloc(sizeof(Dir));
  if (!dirp)
    {
      closedir(p);
      return(NULL);
    }
  dirp->pos = 0;
  dirp->dirptr = p;
  return((void *)dirp);
}


/*******************************************************************
close a directory
********************************************************************/
void CloseDir(void *p)
{
  Dir *dirp = (Dir *)p;
  if (!dirp) return;    
  closedir(dirp->dirptr);
  free(dirp);
}

#ifdef USE_DIRECT
#define DIRECT direct
#else
#define DIRECT dirent
#endif

/*******************************************************************
read from a directory
********************************************************************/
char *ReadDirName(void *p)
{
  Dir *dirp = (Dir *)p;
  struct DIRECT *ptr;

  if (!dirp) return(NULL);
  
  ptr = (struct DIRECT *)readdir(dirp->dirptr);
  if (!ptr) return(NULL);

#ifdef NEXT2
  if (telldir(dirp->dirptr) < 0)
    {
      DEBUG(3,("ReadDir encountered null - returning NULL\n"));
      return(NULL);
    }
#endif

  dirp->pos++;

#ifdef SOLARIS
  /* this handles a broken compiler setup, causing a mixture
   of BSD and SYSV headers and libraries */
  {
    static BOOL broken_readdir = False;
    if (!broken_readdir && !(*ptr->d_name) && strequal("..",ptr->d_name-2))
      {
	DEBUG(0,("Your readdir() is broken. You have somehow mixed SYSV and BSD headers and libraries\n"));
	broken_readdir = True;
      }
    if (broken_readdir)
      return(ptr->d_name-2);
  }
#endif

  return(ptr->d_name);
}


/*******************************************************************
seek a dir
********************************************************************/
BOOL SeekDir(void *p,int pos)
{
  Dir *dirp = (Dir *)p;

  if (!dirp) return(False);

  if (pos < dirp->pos)
    {
      rewinddir(dirp->dirptr);
      dirp->pos = 0;
    }

  while (dirp->pos < pos)
    {
      if (!readdir(dirp->dirptr)) return(False);
      dirp->pos++;
    }

  return(True);
}

/*******************************************************************
tell a dir position
********************************************************************/
int TellDir(void *p)
{
  Dir *dirp = (Dir *)p;

  if (!dirp) return(-1);
  
  return(dirp->pos);
}




#ifdef REPLACE_GETWD
/* This is getcwd.c from bash.  It is needed in Interactive UNIX.  To
 * add support for another OS you need to determine which of the
 * conditional compilation macros you need to define.  All the options
 * are defined for Interactive UNIX.
 */
#ifdef ISC
#define HAVE_UNISTD_H
#define USGr3
#define USG
#endif

#if defined (HAVE_UNISTD_H)
#  include <unistd.h>
#endif

#if defined (__STDC__)
#  define CONST const
#  define PTR void *
#else /* !__STDC__ */
#  define CONST
#  define PTR char *
#endif /* !__STDC__ */

#if !defined (PATH_MAX)
#  if defined (MAXPATHLEN)
#    define PATH_MAX MAXPATHLEN
#  else /* !MAXPATHLEN */
#    define PATH_MAX 1024
#  endif /* !MAXPATHLEN */
#endif /* !PATH_MAX */

#if defined (_POSIX_VERSION) || defined (USGr3) || defined (HAVE_DIRENT_H)
#  if !defined (HAVE_DIRENT)
#    define HAVE_DIRENT
#  endif /* !HAVE_DIRENT */
#endif /* _POSIX_VERSION || USGr3 || HAVE_DIRENT_H */

#if defined (HAVE_DIRENT)
#  define D_NAMLEN(d)	(strlen ((d)->d_name))
#else
#  define D_NAMLEN(d)	((d)->d_namlen)
#endif /* ! (_POSIX_VERSION || USGr3) */

#if defined (USG) || defined (USGr3)
#  define d_fileno d_ino
#endif

#if !defined (alloca)
extern char *alloca ();
#endif /* alloca */

/* Get the pathname of the current working directory,
   and put it in SIZE bytes of BUF.  Returns NULL if the
   directory couldn't be determined or SIZE was too small.
   If successful, returns BUF.  In GNU, if BUF is NULL,
   an array is allocated with `malloc'; the array is SIZE
   bytes long, unless SIZE <= 0, in which case it is as
   big as necessary.  */
#if defined (__STDC__)
char *
getcwd (char *buf, size_t size)
#else /* !__STDC__ */
char *
getcwd (buf, size)
     char *buf;
     int size;
#endif /* !__STDC__ */
{
  static CONST char dots[]
    = "../../../../../../../../../../../../../../../../../../../../../../../\
../../../../../../../../../../../../../../../../../../../../../../../../../../\
../../../../../../../../../../../../../../../../../../../../../../../../../..";
  CONST char *dotp, *dotlist;
  size_t dotsize;
  dev_t rootdev, thisdev;
  ino_t rootino, thisino;
  char path[PATH_MAX + 1];
  register char *pathp;
  char *pathbuf;
  size_t pathsize;
  struct stat st;

  if (buf != NULL && size == 0)
    {
      errno = EINVAL;
      return ((char *)NULL);
    }

  pathsize = sizeof (path);
  pathp = &path[pathsize];
  *--pathp = '\0';
  pathbuf = path;

  if (stat (".", &st) < 0)
    return ((char *)NULL);
  thisdev = st.st_dev;
  thisino = st.st_ino;

  if (stat ("/", &st) < 0)
    return ((char *)NULL);
  rootdev = st.st_dev;
  rootino = st.st_ino;

  dotsize = sizeof (dots) - 1;
  dotp = &dots[sizeof (dots)];
  dotlist = dots;
  while (!(thisdev == rootdev && thisino == rootino))
    {
      register DIR *dirstream;
      register struct dirent *d;
      dev_t dotdev;
      ino_t dotino;
      char mount_point;
      int namlen;

      /* Look at the parent directory.  */
      if (dotp == dotlist)
	{
	  /* My, what a deep directory tree you have, Grandma.  */
	  char *new;
	  if (dotlist == dots)
	    {
	      new = malloc (dotsize * 2 + 1);
	      if (new == NULL)
		goto lose;
	      memcpy (new, dots, dotsize);
	    }
	  else
	    {
	      new = realloc ((PTR) dotlist, dotsize * 2 + 1);
	      if (new == NULL)
		goto lose;
	    }
	  memcpy (&new[dotsize], new, dotsize);
	  dotp = &new[dotsize];
	  dotsize *= 2;
	  new[dotsize] = '\0';
	  dotlist = new;
	}

      dotp -= 3;

      /* Figure out if this directory is a mount point.  */
      if (stat (dotp, &st) < 0)
	goto lose;
      dotdev = st.st_dev;
      dotino = st.st_ino;
      mount_point = dotdev != thisdev;

      /* Search for the last directory.  */
      dirstream = opendir(dotp);
      if (dirstream == NULL)
	goto lose;
      while ((d = (struct dirent *)readdir(dirstream)) != NULL)
	{
	  if (d->d_name[0] == '.' &&
	      (d->d_name[1] == '\0' ||
		(d->d_name[1] == '.' && d->d_name[2] == '\0')))
	    continue;
	  if (mount_point || d->d_fileno == thisino)
	    {
	      char *name;

	      namlen = D_NAMLEN(d);
	      name = (char *)
		alloca (dotlist + dotsize - dotp + 1 + namlen + 1);
	      memcpy (name, dotp, dotlist + dotsize - dotp);
	      name[dotlist + dotsize - dotp] = '/';
	      memcpy (&name[dotlist + dotsize - dotp + 1],
		      d->d_name, namlen + 1);
	      if (lstat (name, &st) < 0)
		{
		  int save = errno;
		  closedir(dirstream);
		  errno = save;
		  goto lose;
		}
	      if (st.st_dev == thisdev && st.st_ino == thisino)
		break;
	    }
	}
      if (d == NULL)
	{
	  int save = errno;
	  closedir(dirstream);
	  errno = save;
	  goto lose;
	}
      else
	{
	  size_t space;

	  while ((space = pathp - pathbuf) <= namlen)
	    {
	      char *new;

	      if (pathbuf == path)
		{
		  new = malloc (pathsize * 2);
		  if (!new)
		    goto lose;
		}
	      else
		{
		  new = realloc ((PTR) pathbuf, (pathsize * 2));
		  if (!new)
		    goto lose;
		  pathp = new + space;
		}
	      (void) memcpy (new + pathsize + space, pathp, pathsize - space);
	      pathp = new + pathsize + space;
	      pathbuf = new;
	      pathsize *= 2;
	    }

	  pathp -= namlen;
	  (void) memcpy (pathp, d->d_name, namlen);
	  *--pathp = '/';
	  closedir(dirstream);
	}

      thisdev = dotdev;
      thisino = dotino;
    }

  if (pathp == &path[sizeof(path) - 1])
    *--pathp = '/';

  if (dotlist != dots)
    free ((PTR) dotlist);

  {
    size_t len = pathbuf + pathsize - pathp;
    if (buf == NULL)
      {
	if (len < (size_t) size)
	  len = size;
	buf = (char *) malloc (len);
	if (buf == NULL)
	  goto lose2;
      }
    else if ((size_t) size < len)
      {
	errno = ERANGE;
	goto lose2;
      }
    (void) memcpy((PTR) buf, (PTR) pathp, len);
  }

  if (pathbuf != path)
    free (pathbuf);

  return (buf);

 lose:
  if ((dotlist != dots) && dotlist)
    {
      int e = errno;
      free ((PTR) dotlist);
      errno = e;
    }

 lose2:
  if ((pathbuf != path) && pathbuf)
    {
      int e = errno;
      free ((PTR) pathbuf);
      errno = e;
    }
  return ((char *)NULL);
}
#endif
