/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   Copyright (C) Jeremy Allison 1994

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include "loadparm.h"
#include "trans2.h"

extern int DEBUGLEVEL;
extern int Protocol;
extern connection_struct Connections[];
extern files_struct Files[];
extern BOOL case_sensitive;


/*********************************************************
* Routine to match a given string with a regexp - uses
* simplified regexp that takes * and ? only. Case can be
* significant or not.
*********************************************************/
BOOL lanman2_match( char *str, char *regexp, int case_sig)
{
  char *p;
  pstring p1, p2;
  BOOL matched;

  /* Make local copies of str and regexp */
  StrnCpy(p1,regexp,sizeof(pstring)-1);
  StrnCpy(p2,str,sizeof(pstring)-1);

  if (strchr(p1,'.'))
    {
      string_sub(p1,"*.*","*");
      string_sub(p1,".*","*");
    }

  /* Remove any *? and ** as they are meaningless */
  for(p = p1; *p; p++)
    while( *p == '*' && (p[1] == '?' ||p[1] == '*'))
      (void)strcpy( &p[1], &p[2]);

  if (strequal(p1,"*")) return(True);

  DEBUG(5,("lanman2_match str=<%s> regexp=<%s>, case_sig = %d\n", p2, p1, case_sig));

  matched = do_match(p2,p1,case_sig);

  DEBUG(5,("lanman2_match returning %d\n", matched));

  return matched;
}

/****************************************************************************
  Send the required number of replies back.
  We assume all fields other than the data fields are
  set correctly for the type of call.
  HACK ! Always assumes smb_setup field is zero.
****************************************************************************/
static int send_trans2_replies(char *outbuf, int bufsize, char *params, 
			 int paramsize, char *pdata, int datasize)
{
  /* As we are using a protocol > LANMAN1 then the maxxmit
     variable must have been set in the sessetupX call.
     This takes precedence over the max_xmit field in the
     global struct. These different max_xmit variables should
     be merged as this is now too confusing */

  extern int maxxmit;
  int data_to_send = datasize;
  int params_to_send = paramsize;
  int useable_space;
  char *pp = params;
  char *pd = pdata;
  int params_sent_thistime, data_sent_thistime, total_sent_thistime;

  /* Initially set the wcnt area to be 10 - this is true for all
     trans2 replies */
  set_message(outbuf,10,0,True);

  /* If there genuinely are no parameters or data to send just send
     the empty packet */
  if(params_to_send == 0 && data_to_send == 0)
    {
      send_smb(outbuf);
      return 0;
    }

  /* Space is bufsize minus Netbios over TCP header minus SMB header */
  useable_space = bufsize - (smb_buf(outbuf) - outbuf);
  useable_space = MIN(useable_space, maxxmit);

  while( params_to_send || data_to_send)
    {
      /* Calculate whether we will totally or partially fill this packet */
      total_sent_thistime = params_to_send + data_to_send;
      total_sent_thistime = MIN(total_sent_thistime, useable_space);

      set_message(outbuf, 10, total_sent_thistime, True);

      /* Set total params and data to be sent */
      SSVAL(outbuf,smb_tprcnt,paramsize);
      SSVAL(outbuf,smb_tdrcnt,datasize);

      /* Calculate how many parameters and data we can fit into
	 this packet. Parameters get precedence */

      params_sent_thistime = MIN(params_to_send,useable_space);
      data_sent_thistime = useable_space - params_sent_thistime;
      data_sent_thistime = MIN(data_sent_thistime,data_to_send);

      SSVAL(outbuf,smb_prcnt, params_sent_thistime);
      if(params_sent_thistime == 0)
	{
	  SSVAL(outbuf,smb_proff,0);
	  SSVAL(outbuf,smb_prdisp,0);
	} else {
	  /* smb_proff is the offset from the start of the SMB header to the
	     parameter bytes, however the first 4 bytes of outbuf are
	     the Netbios over TCP header. Thus use smb_base() to subtract
	     them from the calculation */
	  SSVAL(outbuf,smb_proff,(smb_buf(outbuf) - smb_base(outbuf)));
	  /* Absolute displacement of param bytes sent in this packet */
	  SSVAL(outbuf,smb_prdisp,pp - params);
	}

      SSVAL(outbuf,smb_drcnt, data_sent_thistime);
      if(data_sent_thistime == 0)
	{
	  SSVAL(outbuf,smb_droff,0);
	  SSVAL(outbuf,smb_drdisp, 0);
	} else {
	  /* The offset of the data bytes is the offset of the
	     parameter bytes plus the number of parameters being sent this time */
	  SSVAL(outbuf,smb_droff,(smb_buf(outbuf) - smb_base(outbuf)) + params_sent_thistime);
	  SSVAL(outbuf,smb_drdisp, pd - pdata);
	}

      /* Copy the param bytes into the packet */
      if(params_sent_thistime)
	memcpy(smb_buf(outbuf),pp,params_sent_thistime);
      /* Copy in the data bytes */
      if(data_sent_thistime)
	memcpy(smb_buf(outbuf)+params_sent_thistime,pd,data_sent_thistime);

      DEBUG(9,("t2_rep: params_sent_thistime = %d, data_sent_thistime = %d, useable_space = %d\n",
	       params_sent_thistime, data_sent_thistime, useable_space));
      DEBUG(9,("t2_rep: params_to_send = %d, data_to_send = %d, paramsize = %d, datasize = %d\n",
	       params_to_send, data_to_send, paramsize, datasize));

      /* Send the packet */
      send_smb(outbuf);

      pp += params_sent_thistime;
      pd += data_sent_thistime;

      params_to_send -= params_sent_thistime;
      data_to_send -= data_sent_thistime;

      /* Sanity check */
      if(params_to_send < 0 || data_to_send < 0)
	{
	  DEBUG(2,("send_trans2_replies failed sanity check pts = %d, dts = %d\n!!!",
		   params_to_send, data_to_send));
	  return -1;
	}
    }

  return 0;
}


/****************************************************************************
  reply to a TRANSACT2_OPEN
****************************************************************************/

int call_trans2open(char *inbuf, char *outbuf, int bufsize, int cnum, 
		    char **pparams, char **ppdata)
{
  char *params = *pparams;
  int16 open_mode = SVAL(params, 2);
  int16 open_attr = SVAL(params,6);
#if 0
  BOOL return_additional_info = BITSETW(params,0);
  int16 open_sattr = SVAL(params, 4);
  int32 open_time = IVAL(params,8);
#endif
  int16 open_ofun = SVAL(params,12);
  int32 open_size = IVAL(params,14);
  char *pname = &params[28];
  int16 name_len = strlen(pname)+1;

  pstring fname;
  int fnum = -1;
  int openmode = 0;
  BOOL file_existed = False;
  int unixmode;
  int size=0,fmode=0,mtime=0,rmode;
  int32 inode = 0;
  struct stat sbuf;

  StrnCpy(fname,pname,name_len);

  DEBUG(3,("call_trans2open %s cnum = %d open_mode=%d open_attr = %d open_ofun = %d open_size = %d\n",
	 fname,cnum,open_mode, open_attr, open_ofun, open_size));

  /* XXXX we need to handle passed times, sattr and flags */

  unix_convert(fname,cnum);
  
  rmode = open_mode & 0x7;

  switch (open_mode & 0x7)
    {
    case 0:       
      openmode = O_RDONLY; 
      break;
    case 1: 
      openmode = O_WRONLY; 
      break;
    case 0x7: 
    case 2: 
      rmode = 2;
      openmode = O_RDWR; 
      break;
    case 3:  /* map execute to read */
      openmode = O_RDONLY; 
      break;
    default:
      rmode = 0;
      openmode = O_RDONLY;
      break;
    }
  
  /* now add create and trunc bits */
  if (open_ofun & 0x10)
    openmode |= O_CREAT;
  if ((open_ofun & 0x3) == 2)
    openmode |= O_TRUNC;
  
  fnum = find_free_file();
  if (fnum < 0)
    return(ERROR(ERRSRV,ERRnofids));

  if (!check_name(fname,cnum))
    return(UNIXERROR(ERRDOS,ERRnoaccess));

  unixmode = unix_mode(cnum,open_attr);
      
  file_existed = file_exist(fname);
      
  if ((open_ofun & 0x3) == 0 && file_existed)
    return(ERROR(ERRDOS,ERRbadfile));
      
  open_file(fnum,cnum,fname,openmode,unixmode);
      
  if (!Files[fnum].open)
    return(UNIXERROR(ERRDOS,ERRnoaccess));

  if (fstat(Files[fnum].fd,&sbuf) != 0)
    {
      close_file(fnum);
      return(ERROR(ERRDOS,ERRnoaccess));
    }
    
  size = sbuf.st_size;
  fmode = dos_mode(cnum,fname,&sbuf);
  mtime = sbuf.st_mtime;
  inode = sbuf.st_ino;
  if (fmode & aDIR)
    {
      close_file(fnum);
      return(ERROR(ERRDOS,ERRnoaccess));
    }

  /* Realloc the size of parameters and data we will return */
  params = *pparams = Realloc(*pparams, 28);
  if(params == NULL)
    return(ERROR(ERRDOS,ERRnomem));

  memset(params,'\0',28);
  SSVAL(params,0,fnum);
  SSVAL(params,2,fmode);
  put_dos_date2(params,4, mtime);
  SIVAL(params,8, size);
  SSVAL(params,12,rmode);
  {
    int smb_action = 0;
    
    if (file_existed && !(openmode & O_TRUNC)) smb_action = 1;
    if (!file_existed) smb_action = 2;
    if (file_existed && (openmode & O_TRUNC)) smb_action = 3;

    /* if they requested a opportunistic lock and they have it read-only
       then tell them they have it - this is wrong but it might work */
    if ((CVAL(inbuf,smb_flg) & 0x20) && openmode == O_RDONLY)
      smb_action |= 0x8000;
    
    SSVAL(params,18,smb_action);
  }
  SIVAL(params,20,inode);
 
  DEBUG(3,("%s TRANS2_OPEN %s fd=%d fnum=%d cnum=%d mode=%d omode=%d (numopen=%d)\n",
	 timestring(),fname,Files[fnum].fd,fnum,cnum,
	 open_mode,openmode,Connections[cnum].num_files_open));

  /* Send the required number of replies */
  send_trans2_replies(outbuf, bufsize, params, 28, *ppdata, 0);

  return -1;
}

/****************************************************************************
  get a level dependent lanman2 dir entry.
****************************************************************************/
static int get_lanman2_dir_entry(int cnum,char *path_mask,int dirtype,int info_level,
				 int requires_resume_key,
				 BOOL dont_descend,char **ppdata, 
				 char *base_data, int space_remaining, 
				 BOOL *out_of_space,
				 int *last_name_off)
{
  char *dname;
  BOOL found = False;
  struct stat sbuf;
  pstring mask;
  pstring pathreal;
  pstring fname;
  BOOL matched;
  char *p, *pdata = *ppdata;
  int reskey=0, prev_dirpos;
  int mode=0;
  uint32 size=0;
  uint32 mdate=0, adate=0, cdate=0;
  int name_offset;
  int space_for_this_entry;

  *fname = 0;
  *out_of_space = False;

  if (!Connections[cnum].dirptr)
    return(False);

  p = strrchr(path_mask,'/');
  if(p != NULL)
    {
      if(p[1] == '\0')
	strcpy(mask,"*.*");
      else
	strcpy(mask, p+1);
    }
  else
    strcpy(mask, path_mask);

  while (!found)
    {
      /* Needed if we run out of space */
      prev_dirpos = TellDir(Connections[cnum].dirptr);
      dname = ReadDirName(Connections[cnum].dirptr);

      reskey = TellDir(Connections[cnum].dirptr);

      DEBUG(6,("get_lanman2_dir_entry:readdir on dirptr 0x%x now at offset %d\n",
	       Connections[cnum].dirptr,TellDir(Connections[cnum].dirptr)));
      
      if (!dname) 
	return(False);

      matched = False;

      strcpy(fname,dname);      

      if(lanman2_match(fname, mask, case_sensitive))
	{
	  if (dont_descend &&
	      !strequal(fname,".") && !strequal(fname,".."))
	    continue;

	  strcpy(pathreal,Connections[cnum].dirpath);
	  strcat(pathreal,"/");
	  strcat(pathreal,fname);
	  if (stat(pathreal,&sbuf) != 0) 
	    {
	      DEBUG(5,("get_lanman2_dir_entry:Couldn't stat [%s] (%s)\n",pathreal,strerror(errno)));
	      continue;
	    }

	  /* Work out if we have enough space to include
	     this entry */
	  space_for_this_entry = strlen(fname) + 1;
	  space_for_this_entry += (requires_resume_key ? 4 : 0);
	  space_for_this_entry += ((info_level == 1) ? l1_achName : l2_achName);
	  if(space_for_this_entry > space_remaining)
	    {
	      /* Move the dirptr back to prev_dirpos */
	      SeekDir(Connections[cnum].dirptr, prev_dirpos);
	      *out_of_space = True;
	      DEBUG(9,("get_lanman2_dir_entry: space_for_this_entry = %d, space_remaining = %d\n",
		    space_for_this_entry, space_remaining));
	      return False; /* Not finished - just out of space */
	    }

	  mode = dos_mode(cnum,pathreal,&sbuf);

	  if (((mode & ~dirtype) & (aHIDDEN | aSYSTEM | aDIR)) != 0)
	    {	      
	      DEBUG(5,("[%s] attribs didn't match %x\n",fname,dirtype));
	      continue;
	    }
	  size = sbuf.st_size;
	  mdate = sbuf.st_mtime;
	  adate = sbuf.st_atime;
	  cdate = sbuf.st_ctime;
	  if(mode & aDIR)
	    size = 0;

	  DEBUG(5,("get_lanman2_dir_entry found %s fname=%s\n",pathreal,fname));
	  
	  found = True;
	}
    }

  /* Set up the level 1 or level 2 data block */
  if(requires_resume_key)
    {
      SIVAL(pdata,0,reskey);
      pdata += 4;
    }

#ifdef KANJI
  kj_dos_format (fname, True);
#endif /* KANJI */
  if(info_level == 1)
    {
      put_dos_date2(pdata,l1_fdateCreation,cdate);
      put_dos_date2(pdata,l1_fdateLastAccess,adate);
      put_dos_date2(pdata,l1_fdateLastWrite,mdate);
      SIVAL(pdata,l1_cbFile,size);
      SIVAL(pdata,l1_cbFileAlloc,1024); /* Make 1024 for now */
      SSVAL(pdata,l1_attrFile,mode);
      SCVAL(pdata,l1_cchName,strlen(fname));
      strcpy(pdata + l1_achName, fname);
      name_offset = l1_achName;
    }
  else 
    { /* info_level 2 */
      put_dos_date2(pdata,l2_fdateCreation,cdate);
      put_dos_date2(pdata,l2_fdateLastAccess,adate);
      put_dos_date2(pdata,l2_fdateLastWrite,mdate);
      SIVAL(pdata,l2_cbFile,size);
      SIVAL(pdata,l2_cbFileAlloc,1024); /* Make 1024 for now */
      SSVAL(pdata,l2_attrFile,mode);
      SIVAL(pdata,l2_cbList,0); /* No extended attributes */
      SCVAL(pdata,l2_cchName,strlen(fname));
      strcpy(pdata + l2_achName, fname);
      name_offset = l2_achName;
    }
  /* Setup the last_filename pointer, as an offset from base_data */
  *last_name_off = (pdata - base_data) + name_offset;
  /* Advance the data pointer to the next slot */
  *ppdata = pdata + name_offset + strlen(pdata + name_offset) + 1;
  return(found);
}
  
/****************************************************************************
  returns maximum size of buffer needed to return a FINDFIRST/NEXT entry
****************************************************************************/
int level_info_size(int level, BOOL requires_resume_key)
{
  int size = ( level == 1 ? l1_achName : l2_achName);
  
  size += sizeof(pstring);
  size += ( requires_resume_key ? 4 : 0);
  return size;
}

/****************************************************************************
  checks info level for validity.
****************************************************************************/
int check_info_level(char *inbuf, char *outbuf, int info_level)
{
  /* We only support info levels 1 and 2 */
  if(info_level == 3)
    return(ERROR(ERRDOS,ERROR_EAS_NOT_SUPPORTED));

  if(info_level != 1 && info_level != 2)
    return(ERROR(ERRSRV,ERRnosupport));

  return 0;
}

/****************************************************************************
  reply to a TRANS2_FINDFIRST
****************************************************************************/
int call_trans2findfirst(char *inbuf, char *outbuf, int bufsize, int cnum, 
			 char **pparams, char **ppdata)
{
  /* We must be careful here that we don't return more than the
     allowed number of data bytes. If this means returning fewer than
     maxentries then so be it. We assume that the redirector has
     enough room for the fixed number of parameter bytes it has
     requested. */
  uint32 max_data_bytes = SVAL(inbuf, smb_mdrcnt);
  char *params = *pparams;
  char *pdata = *ppdata;
  int dirtype = SVAL(params,0);
  int maxentries = SVAL(params,2);
  BOOL close_after_first = BITSETW(params+4,0);
  BOOL close_if_end = BITSETW(params+4,1);
  BOOL requires_resume_key = BITSETW(params+4,2);
  int info_level = SVAL(params,6);
  pstring directory;
  pstring mask;
  char *p, *wcard;
  int last_name_off;
  int dptr_num = -1;
  int return_data_size;
  int numentries = 0;
  int i, outsize;
  BOOL finished = False;
  BOOL dont_descend = False;
  BOOL out_of_space = False;
  int space_remaining;

  *directory = *mask = 0;

  DEBUG(3,("call_trans2findfirst: dirtype = %d, maxentries = %d, close_after_first=%d, close_if_end = %d requires_resume_key = %d level = %d, max_data_bytes = %d\n",
	   dirtype, maxentries, close_after_first, close_if_end, requires_resume_key,
	   info_level, max_data_bytes));

  if((outsize = check_info_level(inbuf, outbuf, /* Needed for error return */ info_level)))
    return outsize;

  strcpy(directory, params + 12); /* Complete directory path with wildcard mask appended */

  DEBUG(5,("path=%s\n",directory));

  unix_convert(directory,cnum);
  if(!check_name(directory,cnum)) {
    return(ERROR(ERRDOS,ERRbadpath));
  }
  p = strrchr(directory,'/');
  if(p == NULL) {
    strcpy(mask,directory);
    strcpy(directory,"./");
  } else {
    strcpy(mask,p+1);
    *p = 0;
  }

  DEBUG(5,("dir=%s, mask = %s\n",directory, mask));

  /* Realloc the data to point to the required buffer size */
  if((return_data_size = level_info_size(info_level, requires_resume_key))!=-1)
    {
      return_data_size *= maxentries;
      if(return_data_size > max_data_bytes)
	return_data_size = max_data_bytes;
      *ppdata = Realloc( *ppdata, return_data_size );
      if(*ppdata == NULL)
	return(ERROR(ERRDOS,ERRnomem));
      pdata = *ppdata;
    }
  else
    return(ERROR(ERRSRV,ERRerror));

  /* Realloc the params space */
  params = *pparams = Realloc(*pparams, 10);
  if(params == NULL)
    return(ERROR(ERRDOS,ERRnomem));

  if(!start_dir(cnum,directory))
    return(ERROR(ERRDOS,ERRbadpath));
	
  dptr_num = dptr_create(cnum,directory, True ,SVAL(inbuf,smb_pid));
  if (dptr_num < 0)
    {
      if(Connections[cnum].dirptr != NULL)
	CloseDir(Connections[cnum].dirptr);
      return(ERROR(ERRDOS,ERRnofids));
    }

  /* Save the wildcard match and attribs we are using on this directory - needed as
     lanman2 assumes these are being saved between calls */

  if(!(wcard = strdup(mask)))
    {
      dptr_demote(dptr_num);
      dptr_close(dptr_num);
      return(ERROR(ERRDOS,ERRnomem));
    }

  dptr_set_wcard(dptr_num, wcard);
  dptr_set_attr(dptr_num, dirtype);

  DEBUG(4,("dptr_num is %d, wcard = %s, attr = %d\n",dptr_num, wcard, dirtype));

  /* We don't need to check for VOL here as this is returned by 
     a different TRANS2 call. */

  DEBUG(8,("dirpath=<%s> dontdescend=<%s>\n",Connections[cnum].dirpath,lp_dontdescend(SNUM(cnum))));
  if (in_list(Connections[cnum].dirpath,lp_dontdescend(SNUM(cnum)),case_sensitive))
    dont_descend = True;
    
  p = pdata;
  space_remaining = max_data_bytes;
  out_of_space = False;

  for (i=0;(i<maxentries) && !finished && !out_of_space;i++)
    {

      /* this is a heuristic to avoid seeking the dirptr except when 
	 absolutely necessary. It allows for a filename of about 40 chars */
      if (space_remaining < DIRLEN_GUESS && numentries > 0)
	{
	  out_of_space = True;
	  finished = False;
	}
      else
	{
	  finished = 
	    !get_lanman2_dir_entry(cnum,mask,dirtype,info_level,
				   requires_resume_key,dont_descend,
				   &p,pdata,space_remaining, &out_of_space,
				   &last_name_off);
	}

      if (finished && out_of_space)
	finished = False;

      if (!finished && !out_of_space)
	numentries++;
      space_remaining = max_data_bytes - (p - pdata);
    }
  
  /* Check if we can close the dirptr */
  if(close_after_first || (finished && close_if_end))
    {
      dptr_demote(dptr_num);
      dptr_close(dptr_num);
      DEBUG(5,("call_trans2findfirst - (2) closing dptr_num %d\n", dptr_num));
      dptr_num = -1;
    }

  if(finished && (numentries == 0))
    {
      return(ERROR(ERRDOS,ERRnofiles));
    }

  /* At this point pdata points to numentries directory entries. */

  /* Set up the return parameter block */
  SSVAL(params,0,dptr_num);
  SSVAL(params,2,numentries);
  SSVAL(params,4,finished);
  SSVAL(params,6,0); /* Never an EA error */
  SSVAL(params,8,last_name_off);

  send_trans2_replies( outbuf, bufsize, params, 10, pdata, p - pdata);

  if ((! *directory) && dptr_path(dptr_num))
    sprintf(directory,"(%s)",dptr_path(dptr_num));

  DEBUG(4,("%s %s mask=%s directory=%s cnum=%d dirtype=%d numentries=%d\n",
	timestring(),
	smb_fn_name(CVAL(inbuf,smb_com)), 
	mask,directory,cnum,dirtype,numentries));

  return(-1);
}

/****************************************************************************
  reply to a TRANS2_FINDNEXT
****************************************************************************/

int call_trans2findnext(char *inbuf, char *outbuf, int length, int bufsize,
			int cnum, char **pparams, char **ppdata)
{
  /* We must be careful here that we don't return more than the
     allowed number of data bytes. If this means returning fewer than
     maxentries then so be it. We assume that the redirector has
     enough room for the fixed number of parameter bytes it has
     requested. */
  int max_data_bytes = SVAL(inbuf, smb_mdrcnt);
  char *params = *pparams;
  char *pdata = *ppdata;
  int16 dptr_num = SVAL(params,0);
  int maxentries = SVAL(params,2);
  uint16 info_level = SVAL(params,4);
  uint32 resume_key = IVAL(params,6);
  BOOL close_after_request = BITSETW(params+10,0);
  BOOL close_if_end = BITSETW(params+10,1);
  BOOL requires_resume_key = BITSETW(params+10,2);
  BOOL continue_bit = BITSETW(params+10,3);
  pstring mask;
  pstring directory;
  char *p;
  uint16 dirtype;
  int return_data_size;
  int numentries = 0;
  int i, last_name_off, outsize;
  BOOL finished = False;
  BOOL dont_descend = False;
  BOOL out_of_space = False;
  int space_remaining;

  *mask = *directory = 0;

  DEBUG(3,("call_trans2findnext: dirhandle = %d, max_data_bytes = %d, maxentries = %d, close_after_request=%d, close_if_end = %d requires_resume_key = %d resume_key = %d continue=%d level = %d\n",
	   dptr_num, max_data_bytes, maxentries, close_after_request, close_if_end, 
	   requires_resume_key, resume_key, continue_bit, info_level));

  if((outsize = check_info_level(inbuf, outbuf, /* Needed for error return */ info_level)))
    return outsize;

  /* Realloc the data to point to the required buffer size */
  if((return_data_size = level_info_size(info_level, requires_resume_key))!=-1)
    {
      return_data_size *= maxentries;
      if(return_data_size > max_data_bytes)
	return_data_size = max_data_bytes;
      *ppdata = Realloc( *ppdata, return_data_size );

      if(*ppdata == NULL)
	return(ERROR(ERRDOS,ERRnomem));
      pdata = *ppdata;
    }
  else
    return(ERROR(ERRSRV,ERRerror));

  /* Realloc the params space */
  params = *pparams = Realloc(*pparams, 6*sizeof(WORD));
  if(params == NULL)
    return(ERROR(ERRDOS,ERRnomem));

  /* Check that the dptr is valid */

  if(!(Connections[cnum].dirptr = dptr_fetch_lanman2(params, dptr_num)))
    return(ERROR(ERRDOS,ERRnofiles));

  string_set(&Connections[cnum].dirpath,dptr_path(dptr_num));

  /* Get the wildcard mask from the dptr */
  if((p = dptr_wcard(dptr_num))== NULL)
    {
      DEBUG(2,("dptr_num %d has no wildcard\n", dptr_num));
      return (ERROR(ERRDOS,ERRnofiles));
    }
  strcpy(mask, p);
  strcpy(directory,Connections[cnum].dirpath);

  /* Get the attr mask from the dptr */
  dirtype = dptr_attr(dptr_num);

  DEBUG(3,("dptr_num is %d, mask = %s, attr = %x, dirptr=(0x%X,%d)\n",
	   dptr_num, mask, dirtype, 
	   Connections[cnum].dirptr,
	   TellDir(Connections[cnum].dirptr)));

  /* We don't need to check for VOL here as this is returned by 
     a different TRANS2 call. */

  DEBUG(8,("dirpath=<%s> dontdescend=<%s>\n",Connections[cnum].dirpath,lp_dontdescend(SNUM(cnum))));
  if (in_list(Connections[cnum].dirpath,lp_dontdescend(SNUM(cnum)),case_sensitive))
    dont_descend = True;
    
  p = pdata;
  space_remaining = max_data_bytes;
  out_of_space = False;

  /* If we have a resume key - seek to the correct position. */
  if(requires_resume_key && !continue_bit)
    SeekDir(Connections[cnum].dirptr, resume_key);

  for (i=0;(i<(int)maxentries) && !finished && !out_of_space ;i++)
    {
      /* this is a heuristic to avoid seeking the dirptr except when 
	 absolutely necessary. It allows for a filename of about 40 chars */
      if (space_remaining < DIRLEN_GUESS && numentries > 0)
	{
	  out_of_space = True;
	  finished = False;
	}
      else
	{
	  finished = 
	    !get_lanman2_dir_entry(cnum,mask,dirtype,info_level,
				   requires_resume_key,dont_descend,
				   &p,pdata,space_remaining, &out_of_space,
				   &last_name_off);
	}

      if (finished && out_of_space)
	finished = False;

      if (!finished && !out_of_space)
	numentries++;
      space_remaining = max_data_bytes - (p - pdata);
    }
  
  /* Check if we can close the dirptr */
  if(close_after_request || (finished && close_if_end))
    {
      dptr_demote(dptr_num);
      dptr_close(dptr_num); /* This frees up the saved mask */
      DEBUG(5,("call_trans2findnext: closing dptr_num = %d\n", dptr_num));
      dptr_num = -1;
    }

  /* At this point pdata points to numentries directory entries. */
  if(finished && (numentries == 0))
    {
      return(ERROR(ERRDOS,ERRnofiles));
    }

  /* Set up the return parameter block */
  SSVAL(params,0,numentries);
  SSVAL(params,2,finished);
  SSVAL(params,4,0); /* Never an EA error */
  SSVAL(params,6,last_name_off);

  send_trans2_replies( outbuf, bufsize, params, 12, pdata, p - pdata);

  if ((! *directory) && dptr_path(dptr_num))
    sprintf(directory,"(%s)",dptr_path(dptr_num));

  DEBUG(3,("%s %s mask=%s directory=%s cnum=%d dirtype=%d numentries=%d\n",
	   timestring(),
	   smb_fn_name(CVAL(inbuf,smb_com)), 
	   mask,directory,cnum,dirtype,numentries));

  return(-1);
}

/****************************************************************************
  reply to a TRANS2_QFSINFO (query filesystem info)
****************************************************************************/

int call_trans2qfsinfo(char *inbuf, char *outbuf, int length, int bufsize,
			int cnum, char **pparams, char **ppdata)
{
  char *pdata = *ppdata;
  char *params = *pparams;
  uint16 info_level = SVAL(params,0);
  int outsize = 0;
  pstring service_name;
  int data_len;
  struct stat st;
  
  *service_name = 0;
  
  DEBUG(3,("call_trans2qfsinfo: cnum = %d, level = %d\n", cnum, info_level));

  if((outsize = check_info_level(inbuf, outbuf, /* Needed for error return */ info_level)))
    return outsize;

  /* Realloc the data space (big enough for level 1 and level 2 answers). */
  pdata = *ppdata = Realloc(*ppdata, 9*sizeof(WORD));
  if(pdata == NULL)
    return(ERROR(ERRDOS,ERRnomem));

  strcpy(service_name,SERVICE(SNUM(cnum)));

  if(stat(".",&st)!=0)
    {
      DEBUG(2,("call_trans2qfsinfo: stat of . failed (%s)\n", strerror(errno)));
      return (ERROR(ERRSRV,ERRinvdevice));
    }

  if(info_level == 1)
    {
      int dfree,dsize,bsize;

      disk_free(".",&bsize,&dfree,&dsize);

      SIVAL(pdata,l1_idFileSystem,st.st_dev);
      SIVAL(pdata,l1_cSectorUnit,bsize/512);
      SIVAL(pdata,l1_cUnit,dsize);
      SIVAL(pdata,l1_cUnitAvail,dfree);
      SSVAL(pdata,l1_cbSector,512);
      data_len = 18;
      DEBUG(5,("call_trans2qfsinfo : bsize = %d, idFileSystem = %x, cSectorUnit = %d, cUnit = %d, cUnitAvail = %d, cbSector = %d\n", bsize, st.st_dev, bsize/512, dsize, dfree, 512));
    }
  else
    { 
      /* Return volume name */
      int volname_len = strlen(service_name);
      if(volname_len > 11)
	volname_len = 11;

      put_dos_date2(pdata,l2_vol_fdateCreation,st.st_ctime);
      SCVAL(pdata,l2_vol_cch,volname_len);
      StrnCpy(pdata+l2_vol_szVolLabel,service_name,volname_len);
      *(pdata+l2_vol_szVolLabel+volname_len) = '\0';
      data_len = l2_vol_szVolLabel + volname_len + 1;
      DEBUG(5,("call_trans2qfsinfo : time = %x, namelen = %d, name = %s\n",st.st_ctime,volname_len,
	       pdata+l2_vol_szVolLabel));
    }

  send_trans2_replies( outbuf, bufsize, params, 0, pdata, data_len);

  DEBUG(4,("%s %s info_level =%d\n",timestring(),smb_fn_name(CVAL(inbuf,smb_com)), info_level));

  return -1;
}

/****************************************************************************
  reply to a TRANS2_SETFSINFO (set filesystem info)
****************************************************************************/

int call_trans2setfsinfo(char *inbuf, char *outbuf, int length, int bufsize,
			int cnum, char **pparams, char **ppdata)
{
  /* Just say yes we did it - there is nothing that
     can be set here so it doesn't matter. */
  int outsize;
  DEBUG(3,("call_trans2setfsinfo\n"));

  if (!CAN_WRITE(cnum))
    return(ERROR(ERRSRV,ERRaccess));

  outsize = set_message(outbuf,10,0,True);

  return outsize;
}

/****************************************************************************
  reply to a TRANS2_QPATHINFO (query file info by path)
****************************************************************************/

int call_trans2qpathinfo(char *inbuf, char *outbuf, int length, int bufsize,
			int cnum, char **pparams, char **ppdata)
{
  char *params = *pparams;
  char *pdata = *ppdata;
  uint16 info_level = SVAL(params,0);
  int mode=0;
  int size=0;
  pstring fname;
  unsigned int data_size;
  int outsize;
  struct stat sbuf;
  BOOL ok = False;

  DEBUG(3,("call_trans2qpathinfo info_level = %d\n", info_level));

  if((outsize = check_info_level(inbuf, outbuf, /* Needed for error return */ info_level)))
    return outsize;

  strcpy(fname,&params[6]);

  unix_convert(fname,cnum);
  if (check_name(fname,cnum))
    {
      if (stat(fname,&sbuf) == 0)
	{
	  mode = dos_mode(cnum,fname,&sbuf);
	  size = sbuf.st_size;
	  if (mode & aDIR)
	    size = 0;
	  ok = True;
	}
      else
	DEBUG(3,("stat of %s failed (%s)\n",fname,strerror(errno)));
    }
  
  if (!ok)
    return(UNIXERROR(ERRDOS,ERRbadfile));
  
  /* Realloc the parameter and data sizes */
  params = *pparams = Realloc(*pparams,2);
  if(params == NULL)
    return(ERROR(ERRDOS,ERRnomem));

  data_size = (info_level == 1 ? 22 : 26);

  if(data_size > SVAL(inbuf, smb_mdrcnt))
    {
      DEBUG(3,("trans2qpathinfo : max_data = %d, data_size = %d\n",
	    SVAL(inbuf,smb_mdrcnt), data_size));
    }
  pdata = *ppdata = Realloc(*ppdata, data_size);
  if(pdata == NULL)
    return(ERROR(ERRDOS,ERRnomem));

  /* No extended attribute errors */
  SSVAL(params,0,0);

  put_dos_date2(pdata,l1_fdateCreation,sbuf.st_ctime);
  put_dos_date2(pdata,l1_fdateLastAccess,sbuf.st_atime);
  put_dos_date2(pdata,l1_fdateLastWrite,sbuf.st_mtime);
  SIVAL(pdata,l1_cbFile,size);
  SIVAL(pdata,l1_cbFileAlloc,1024); /* Make 1024 for now */
  SSVAL(pdata,l1_attrFile,mode);
  if(info_level == 2)
    SIVAL(pdata,l2_cbList,0);

  send_trans2_replies( outbuf, bufsize, params, 2, pdata, data_size);

  DEBUG(3,("%s trans2qpathinfo name=%s mode=%d size=%d\n",timestring(),fname,mode,size));
  
  return(-1);
}

/****************************************************************************
  reply to a TRANS2_SETPATHINFO (set file info by path)
****************************************************************************/

int call_trans2setpathinfo(char *inbuf, char *outbuf, int length, int bufsize,
			int cnum, char **pparams, char **ppdata)
{
  char *params = *pparams;
  char *pdata = *ppdata;
  uint16 info_level = SVAL(params,0);
  int mode=0;
  int size=0;
  struct utimbuf tvs;
  int fd;
  int outsize;
  BOOL is_directory = False;
  struct stat st;
  pstring fname;

  DEBUG(3,("call_trans2setpathinfo info_level = %d\n", info_level));

  if((outsize = check_info_level(inbuf, outbuf, /* Needed for error return */ info_level)))
    return outsize;

  /* extract the filename */
  strcpy(fname,&params[6]);

  /* Realloc the parameter and data sizes */
  params = *pparams = Realloc(*pparams,2);
  if(params == NULL)
    return(ERROR(ERRDOS,ERRnomem));

  unix_convert(fname,cnum);
  if(!check_name(fname, cnum))
    return(ERROR(ERRDOS,ERRbadpath));

  if (!CAN_WRITE(cnum))
    return(ERROR(ERRSRV,ERRaccess));

  if(stat(fname,&st)!=0)
    {
      DEBUG(3,("stat of %s failed (%s)\n", fname, strerror(errno)));
      return(ERROR(ERRDOS,ERRbadpath));
    }

  is_directory = ( st.st_mode & S_IFDIR ) ? True : False;

  tvs.actime = make_unix_date2(pdata+l1_fdateLastAccess);
  tvs.modtime = make_unix_date2(pdata+l1_fdateLastWrite);
  mode = unix_mode(cnum,SVAL(pdata,l1_attrFile));
  size = IVAL(pdata,l1_cbFile);

  /* Try and set the times, size and mode of this file - if they are different 
   from the current values */
  if(!is_directory && (st.st_mtime != tvs.modtime || st.st_atime != tvs.actime))
    {
      if(utime(fname, &tvs)!=0)
	{
	  DEBUG(3,("utime of %s failed (%s)\n", fname, strerror(errno)));
	  if(errno == ENOENT)
	    return(ERROR(ERRDOS,ERRbadpath));
	  return(ERROR(ERRDOS,ERRnoaccess));
	}
    }
  if(mode != st.st_mode)
    {
      if(chmod(fname,mode)!=0)
	{
	  DEBUG(2,("chmod of %s failed (%s)\n", fname, strerror(errno)));
	  return(ERROR(ERRDOS,ERRnoaccess));
	}
    }

  /* Only try and change size on non-directories */
  if(!is_directory && (size != st.st_size))
    {
      if((fd = open(fname, O_RDWR))!= -1)
	{
	  if(set_filelen(fd, size)!=0)
	    {
	      close(fd);
	      return(ERROR(ERRDOS,ERRnoaccess));
	    }
	  close(fd);
	}
      else
	{
	  DEBUG(2,("open of %s failed (%s)\n", fname, strerror(errno)));
	  return(ERROR(ERRDOS,ERRnoaccess));
	}
    }

  SSVAL(params,0,0);

  send_trans2_replies(outbuf, bufsize, params, 2, pdata, 0);
  
  return(-1);
}

/****************************************************************************
  reply to a TRANS2_QFILEINFO (query file info by fileid)
****************************************************************************/

int call_trans2qfileinfo(char *inbuf, char *outbuf, int length, int bufsize,
			int cnum, char **pparams, char **ppdata)
{
  char *params = *pparams;
  char *pdata = *ppdata;
  uint16 info_level = SVAL(params,2);
  int mode=0;
  int size=0;
  int16 fnum = SVAL(params,0);
  unsigned int data_size;
  int outsize;
  struct stat sbuf;
  char *fname;
  BOOL ok = False;

  DEBUG(3,("call_trans2qfileinfo fnum = %d, info_level = %d\n", fnum, info_level));

  if((outsize = check_info_level(inbuf, outbuf, /* Needed for error return */ info_level)))
    return outsize;

  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));
  if (Protocol >= PROTOCOL_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }

  fname = Files[fnum].name;

  if (fstat(Files[fnum].fd,&sbuf) == 0)
    {
      mode = dos_mode(cnum,fname,&sbuf);
      size = sbuf.st_size;
      if (mode & aDIR)
	size = 0;
      ok = True;
    }
  else
    DEBUG(3,("fstat of fnum %d failed (%s)\n",fnum, strerror(errno)));
  
  if (!ok)
    return(UNIXERROR(ERRDOS,ERRbadfid));
  
  /* Realloc the parameter and data sizes */
  params = *pparams = Realloc(*pparams,2);
  if(params == NULL)
    return(ERROR(ERRDOS,ERRnomem));

  data_size = (info_level == 1 ? 22 : 26);

  if(data_size > SVAL(inbuf, smb_mdrcnt))
    {
      DEBUG(3,("trans2qpathinfo : max_data = %d, data_size = %d\n",
	    SVAL(inbuf,smb_mdrcnt), data_size));
    }
  pdata = *ppdata = Realloc(*ppdata, data_size);
  if(pdata == NULL)
    return(ERROR(ERRDOS,ERRnomem));

  /* No extended attribute errors */
  SSVAL(params,0,0);

  put_dos_date2(pdata,l1_fdateCreation,sbuf.st_ctime);
  put_dos_date2(pdata,l1_fdateLastAccess,sbuf.st_atime);
  put_dos_date2(pdata,l1_fdateLastWrite,sbuf.st_mtime);
  SIVAL(pdata,l1_cbFile,size);
  SIVAL(pdata,l1_cbFileAlloc,1024); /* Make 1024 for now */
  SSVAL(pdata,l1_attrFile,mode);
  if(info_level == 2)
    SIVAL(pdata,l2_cbList,0);

  send_trans2_replies( outbuf, bufsize, params, 2, pdata, data_size);

  DEBUG(3,("%s trans2qfileinfo fname = %s fnum=%d mode=%d size=%d\n",timestring(),fname,fnum,mode,size));
  
  return(-1);
}

/****************************************************************************
  reply to a TRANS2_SETFILEINFO (set file info by fileid)
****************************************************************************/

int call_trans2setfileinfo(char *inbuf, char *outbuf, int length, int bufsize,
			int cnum, char **pparams, char **ppdata)
{
  char *params = *pparams;
  char *pdata = *ppdata;
  uint16 info_level = SVAL(params,2);
  int mode=0;
  int size=0;
  struct utimbuf tvs;
  struct stat st;
  int outsize;
  int16 fnum = SVAL(params,0);
  int fd;
  char *fname;

  DEBUG(3,("call_trans2setfileinfo fnum = %d, info_level = %d\n", fnum, info_level));

  if((outsize = check_info_level(inbuf, outbuf, /* Needed for error return */ info_level)))
    return outsize;

  if (!OPEN_FNUM(fnum))
    return(ERROR(ERRDOS,ERRbadfid));

  if (!CAN_WRITE(cnum))
    return(ERROR(ERRSRV,ERRaccess));

  if (Protocol >= PROTOCOL_LANMAN1)
    {
      if(HAS_CACHED_ERROR(fnum))
	return (CACHED_ERROR(fnum));
    }

  /* Realloc the parameter and data sizes */
  params = *pparams = Realloc(*pparams,2);
  if(params == NULL)
    return(ERROR(ERRDOS,ERRnomem));

  fd = Files[fnum].fd;
  fname = Files[fnum].name;

  if(fstat(fd,&st)!=0)
    {
      DEBUG(3,("fstat of fnum %d failed (%s)\n", fnum, strerror(errno)));
      return(ERROR(ERRDOS,ERRbadpath));
    }

  tvs.actime = make_unix_date2(pdata+l1_fdateLastAccess);
  tvs.modtime = make_unix_date2(pdata+l1_fdateLastWrite);
  mode = unix_mode(cnum,SVAL(pdata,l1_attrFile));
  size = IVAL(pdata,l1_cbFile);

  /* Try and set the times, size and mode of this file - if they are different 
   from the current values */
  if(st.st_mtime != tvs.modtime || st.st_atime != tvs.actime)
    {
      if(utime(fname, &tvs)!=0)
	{
	  DEBUG(2,("utime of %s (fnum %d) failed (%s)\n", fname, fnum, strerror(errno)));
	  if(errno == ENOENT)
	    return(ERROR(ERRDOS,ERRbadpath));
	  return(ERROR(ERRDOS,ERRnoaccess));
	}
    }
  if(mode != st.st_mode)
    {
      if(chmod(fname,mode)!=0)
	{
	  DEBUG(2,("chmod of %s (fnum %d) failed (%s)\n", fname, fnum, strerror(errno)));
	  return(ERROR(ERRDOS,ERRnoaccess));
	}
    }

  if(size != st.st_size)
    {
      /* lan server seems to use size=0 as "don't set it" so we won't */
      if(size && set_filelen(fd, size)!=0)
	return(ERROR(ERRDOS,ERRnoaccess));
    }

  SSVAL(params,0,0);

  send_trans2_replies(outbuf, bufsize, params, 2, *ppdata, 0);
  
  return(-1);
}

/****************************************************************************
  reply to a TRANS2_MKDIR (make directory with extended attributes).
****************************************************************************/

int call_trans2mkdir(char *inbuf, char *outbuf, int length, int bufsize,
			int cnum, char **pparams, char **ppdata)
{
  char *params = *pparams;
  pstring directory;
  int ret = -1;

  if (!CAN_WRITE(cnum))
    return(ERROR(ERRSRV,ERRaccess));

  strcpy(directory, &params[4]);

  DEBUG(3,("call_trans2mkdir : name = %s\n", directory));

  unix_convert(directory,cnum);
  if (check_name(directory,cnum))
    ret = mkdir(directory,unix_mode(cnum,aDIR));
  
  if(ret < 0)
    {
      DEBUG(5,("call_trans2mkdir error (%s)\n", strerror(errno)));
      return(UNIXERROR(ERRDOS,ERRnoaccess));
    }

  /* Realloc the parameter and data sizes */
  params = *pparams = Realloc(*pparams,2);
  if(params == NULL)
    return(ERROR(ERRDOS,ERRnomem));

  SSVAL(params,0,0);

  send_trans2_replies(outbuf, bufsize, params, 2, *ppdata, 0);
  
  return(-1);
}

/****************************************************************************
  reply to a TRANS2_FINDNOTIFYFIRST (start monitoring a directory for changes)
  We don't actually do this - we just send a null response.
****************************************************************************/

int call_trans2findnotifyfirst(char *inbuf, char *outbuf, int length, int bufsize,
			int cnum, char **pparams, char **ppdata)
{
  static uint16 fnf_handle = 257;
  char *params = *pparams;
  int outsize;
  uint16 info_level = SVAL(params,4);

  DEBUG(3,("call_trans2findnotifyfirst - info_level %d\n", info_level));

  if((outsize = check_info_level(inbuf, outbuf, /* Needed for error return */ info_level)))
    return outsize;

  /* Realloc the parameter and data sizes */
  params = *pparams = Realloc(*pparams,6);
  if(params == NULL)
    return(ERROR(ERRDOS,ERRnomem));

  SSVAL(params,0,fnf_handle);
  SSVAL(params,2,0); /* No changes */
  SSVAL(params,4,0); /* No EA errors */

  fnf_handle++;

  if(fnf_handle == 0)
    fnf_handle = 257;

  send_trans2_replies(outbuf, bufsize, params, 6, *ppdata, 0);
  
  return(-1);
}

/****************************************************************************
  reply to a TRANS2_FINDNOTIFYNEXT (continue monitoring a directory for 
  changes). Currently this does nothing.
****************************************************************************/
int call_trans2findnotifynext(char *inbuf, char *outbuf, int length, int bufsize,
			int cnum, char **pparams, char **ppdata)
{
  char *params = *pparams;

  DEBUG(3,("call_trans2findnotifynext\n"));

  /* Realloc the parameter and data sizes */
  params = *pparams = Realloc(*pparams,4);
  if(params == NULL)
    return(ERROR(ERRDOS,ERRnomem));

  SSVAL(params,0,0); /* No changes */
  SSVAL(params,2,0); /* No EA errors */

  send_trans2_replies(outbuf, bufsize, params, 4, *ppdata, 0);
  
  return(-1);
}

/****************************************************************************
  reply to a SMBfindclose (stop trans2 directory search)
****************************************************************************/
int reply_findclose(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum;
  int outsize = 0;
  uint16 dptr_num=SVAL(inbuf,smb_vwv0);
  void *dptr;

  cnum = SVAL(inbuf,smb_tid);

  DEBUG(3,("reply_findclose, cnum = %d, dptr_num = %d\n", cnum, dptr_num));

  if((dptr = dptr_get(dptr_num))==NULL)
    {
      /* Invalid dptr_num */
      DEBUG(5,("reply_findclose - dptr_num is bad\n"));

      return (ERROR(ERRDOS,ERRbadfid));
    }

  dptr_demote(dptr_num);
  dptr_close(dptr_num);

  outsize = set_message(outbuf,0,0,True);

  DEBUG(3,("%s SMBfindclose cnum=%d, dptr_num = %d\n",timestring(),cnum,dptr_num));

  return(outsize);
}

/****************************************************************************
  reply to a SMBfindnclose (stop FINDNOTIFYFIRST directory search)
****************************************************************************/
int reply_findnclose(char *inbuf,char *outbuf,int length,int bufsize)
{
  int cnum;
  int outsize = 0;
  int dptr_num=-1;

  cnum = SVAL(inbuf,smb_tid);
  dptr_num = SVAL(inbuf,smb_vwv0);

  DEBUG(3,("reply_findnclose, cnum = %d, dptr_num = %d\n", cnum, dptr_num));

  /* We never give out valid handles for a 
     findnotifyfirst - so any dptr_num is ok here. 
     Just ignore it. */

  outsize = set_message(outbuf,0,0,True);

  DEBUG(3,("%s SMB_findnclose cnum=%d, dptr_num = %d\n",timestring(),cnum,dptr_num));

  return(outsize);
}


/****************************************************************************
  reply to a SMBtrans2
****************************************************************************/
int reply_trans2(char *inbuf,char *outbuf,int length,int bufsize)
{
  int outsize = 0;
  int cnum = SVAL(inbuf,smb_tid);
  unsigned int total_params = SVAL(inbuf, smb_tpscnt);
  unsigned int total_data =SVAL(inbuf, smb_tdscnt);
#if 0
  unsigned int max_param_reply = SVAL(inbuf, smb_mprcnt);
  unsigned int max_data_reply = SVAL(inbuf, smb_mdrcnt);
  unsigned int max_setup_fields = SVAL(inbuf, smb_msrcnt);
  BOOL close_tid = BITSETW(inbuf+smb_flags,0);
  BOOL no_final_response = BITSETW(inbuf+smb_flags,1);
  int32 timeout = IVALS(inbuf,smb_timeout);
#endif
  unsigned int suwcnt = SVAL(inbuf, smb_suwcnt);
  unsigned int tran_call = SVAL(inbuf, smb_setup0);
  char *params = NULL, *data = NULL;
  int num_params, num_params_sofar, num_data, num_data_sofar;

  outsize = set_message(outbuf,0,0,True);

  /* All trans2 messages we handle have smb_sucnt == 1 - ensure this
     is so as a sanity check */
  if(suwcnt != 1 )
    {
      DEBUG(2,("Invalid smb_sucnt in trans2 call\n"));
      return(ERROR(ERRSRV,ERRerror));
    }
    
  /* Allocate the space for the maximum needed parameters and data */
  if (total_params > 0)
    params = (char *)malloc(total_params);
  if (total_data > 0)
    data = (char *)malloc(total_data);
  
  if ((total_params && !params)  || (total_data && !data))
    {
      DEBUG(2,("Out of memory in reply_trans2\n"));
      return(ERROR(ERRDOS,ERRnomem));
    }

  /* Copy the param and data bytes sent with this request into
     the params buffer */
  num_params = num_params_sofar = SVAL(inbuf,smb_pscnt);
  num_data = num_data_sofar = SVAL(inbuf, smb_dscnt);

  memcpy( params, smb_base(inbuf) + SVAL(inbuf, smb_psoff), num_params);
  memcpy( data, smb_base(inbuf) + SVAL(inbuf, smb_dsoff), num_data);

  if(num_data_sofar < total_data || num_params_sofar < total_params)
    {
    /* We need to send an interim response then receive the rest
       of the parameter/data bytes */
      outsize = set_message(outbuf,0,0,True);
      send_smb(outbuf);

      while( num_data_sofar < total_data || num_params_sofar < total_params)
	{
	  receive_smb(inbuf, 0, False);
	  
	  /* Ensure this is still a trans2 packet (sanity check) */
	  if(CVAL(inbuf, smb_com) != SMBtrans2)
	    {
	      outsize = set_message(outbuf,0,0,True);
	      DEBUG(2,("Invalid secondary trans2 packet\n"));
	      free(params);
	      free(data);
	      return(ERROR(ERRSRV,ERRerror));
	    }
      
	  /* Revise total_params and total_data in case they have changed downwards */
	  total_params = SVAL(inbuf, smb_tpscnt);
	  total_data = SVAL(inbuf, smb_tdscnt);
	  num_params_sofar += (num_params = SVAL(inbuf,smb_spscnt));
	  num_data_sofar += ( num_data = SVAL(inbuf, smb_sdscnt));
	  memcpy( &params[ SVAL(inbuf, smb_spsdisp)], 
		 smb_base(inbuf) + SVAL(inbuf, smb_spsoff), num_params);
	  memcpy( &data[SVAL(inbuf, smb_sdsdisp)],
		 smb_base(inbuf)+ SVAL(inbuf, smb_sdsoff), num_data);
	}
    }

  /* Now we must call the relevant TRANS2 function */
  switch( tran_call ) 
    {
    case TRANSACT2_OPEN:
      outsize = call_trans2open(inbuf, outbuf, bufsize, cnum, &params, &data);
      break;
    case TRANSACT2_FINDFIRST:
      outsize = call_trans2findfirst(inbuf, outbuf, bufsize, cnum, &params, &data);
      break;
    case TRANSACT2_FINDNEXT:
      outsize = call_trans2findnext(inbuf, outbuf, length, bufsize, cnum, &params, &data);
      break;
    case TRANSACT2_QFSINFO:
      outsize = call_trans2qfsinfo(inbuf, outbuf, length, bufsize, cnum, &params, &data);
      break;
    case TRANSACT2_SETFSINFO:
      outsize = call_trans2setfsinfo(inbuf, outbuf, length, bufsize, cnum, &params, &data);
      break;
    case TRANSACT2_QPATHINFO:
      outsize = call_trans2qpathinfo(inbuf, outbuf, length, bufsize, cnum, &params, &data);
      break;
    case TRANSACT2_SETPATHINFO:
      outsize = call_trans2setpathinfo(inbuf, outbuf, length, bufsize, cnum, &params, &data);
      break;
    case TRANSACT2_QFILEINFO:
      outsize = call_trans2qfileinfo(inbuf, outbuf, length, bufsize, cnum, &params, &data);
      break;
    case TRANSACT2_SETFILEINFO:
      outsize = call_trans2setfileinfo(inbuf, outbuf, length, bufsize, cnum, &params, &data);
      break;
    case TRANSACT2_FINDNOTIFYFIRST:
      outsize = call_trans2findnotifyfirst(inbuf, outbuf, length, bufsize, cnum, &params, &data);
      break;
    case TRANSACT2_FINDNOTIFYNEXT:
      outsize = call_trans2findnotifynext(inbuf, outbuf, length, bufsize, cnum, &params, &data);
      break;
    case TRANSACT2_MKDIR:
      outsize = call_trans2mkdir(inbuf, outbuf, length, bufsize, cnum, &params, &data);
      break;
    default:
      /* Error in request */
      DEBUG(2,("%s Unknown request %d in trans2 call\n",timestring(), tran_call));
      if(params)
	free(params);
      if(data)
	free(data);
      return (ERROR(ERRSRV,ERRerror));
    }

  /* As we do not know how many data packets will need to be
     returned here the various call_trans2xxxx calls
     must send their own. Thus a call_trans2xxx routine only
     returns a value other than -1 when it wants to send
     an error packet. 
  */

  if(params)
    free(params);
  if(data)
    free(data);
  return outsize; /* If a correct response was needed the call_trans2xxx 
		     calls have already sent it. If outsize != -1 then it is
		     returning an error packet. */
}
