/* 
   Unix SMB/Netbios implementation.
   Version 0.1
   WINS server routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Module name: nameservreply.c

   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   04 jul 96: lkcl@pires.co.uk
   created module nameservreply containing NetBIOS reply functions

*/

#include "includes.h"

extern int ClientNMB[];
extern int DEBUGLEVEL;

extern struct name_record *namelist;
extern time_t names_last_modified;

extern struct in_addr ipgrp;


/* a name query challenge structure */
struct nmb_query_chllng
{
	/* see rfc1002.txt 5.1.4.1 p62 */
	struct in_addr reply_ip;
	uint16 response_id;
	BOOL multi;
	struct nmb_ip nb;
	int port;
};

static void response_name_register(time_t timestamp, struct packet_struct *p,
                                  struct response_record *n);
static void response_name_challenge(time_t timestamp, struct packet_struct *p,
                                    struct response_record *n);

/****************************************************************************
reply to a reg request
**************************************************************************/
void reply_name_reg(struct packet_struct *p)
{
  struct nmb_packet *nmb = &p->packet.nmb;
  struct nmb_name *question = &nmb->question.question_name;
  
  struct nmb_name *reply_name = question;

  BOOL bcast = nmb->header.nm_flags.bcast;
  BOOL multi = nmb->header.opcode == NMB_REG_MULTI ||
               question->name_type == 0x1c;
  
  BOOL group  = False;
  BOOL unique = False;

  struct name_record *n = NULL;

  BOOL success = True;
  BOOL error = False;
  BOOL secured_redirect = False;

  struct nmb_ip register_ip[26];
  int num_reg_ips = 0;
  int rdata_idx = 0;

  int idx;
  
  if (!nmb->header.nm_flags.recursion_desired)
  {
    /* packet is not for name server. */
    proxy_forward_packet(p->timestamp,p);
    return;
  }

  DEBUG(3,("Name registration for name %s at %s",
	           namestr(question),inet_ntoa(p->ip)));
  
  if (bcast)
  {
    DEBUG(3,("\n"));
    DEBUG(3,("wins server received broadcast name registration. ignoring\n"));
    return;
  }

  while (rdata_idx < nmb->additional->rdlength)
  {
    BOOL gp;
    putip(&register_ip[num_reg_ips].ip,&nmb->additional->rdata[rdata_idx+2]);
    register_ip[num_reg_ips].nb_flags = nmb->additional->rdata[rdata_idx];
    register_ip[num_reg_ips].death_time = GET_TTL(nmb->additional->ttl);
    register_ip[num_reg_ips].source = REGISTER;

    gp = NAME_GROUP(register_ip[num_reg_ips].nb_flags);

    group  |= gp;
    unique |= !gp;

    DEBUG(3,("%s %2x ",inet_ntoa(register_ip[num_reg_ips].ip),
                      (unsigned char)register_ip[num_reg_ips].nb_flags));
    num_reg_ips++;
    rdata_idx += 6;
  
  }

  DEBUG(3,("\n"));

  if (group && unique)
  {
    success = False;
    error = True;
  }

  if (!error && group && question->name_type != 0x1c)
  {
    /* group names registered with a WINS server should be
       registered under ip 255.255.255.255 - except internet
       group names, and multi-homed names. maybe.
     */

     if (num_reg_ips > 1)
     {
       error = True;
       success = False;
     }
     else
     {
       multi = False;
       putip(&register_ip[0].ip,&ipgrp);
     }
  }

  if (!error && success)
  {
    /* see if the name already exists */
    n = find_name(namelist, question, 0);
  }
  
  if (n)
  {
    idx = find_name_idx(n, register_ip[0].ip, 0);
    if (idx == -1) idx = 0;

    if (!group) /* unique names */
	{
	  if (NAME_GROUP(n->ip_flgs[idx].nb_flags))
	  {
	      /* no-one can register a name that's a group name as a unique name */
	      success = False;
	  }
	  else if (!ip_equal(register_ip[0].ip, n->ip_flgs[idx].ip))
	  {
	      /* XXXX rfc1001.txt says:
	       * if we are doing secured WINS, we must send a Wait-Acknowledge
	       * packet (WACK) to the person who wants the name, then do a
	       * name query on the person who currently owns the unique name.
	       * if the current owner still says they own it, the person who wants
		   * the name can't have it. if they do not, or are not alive, they can.
	       */

          secured_redirect = True;

	      reply_name = &n->name;
	  }
	  else
	  {
	      DEBUG(3,("%s owner: %s\n",namestr(&n->name),
									inet_ntoa(n->ip_flgs[0].ip)));
	  }
	}
    else
	{
	  if (!NAME_GROUP(n->ip_flgs[idx].nb_flags))
      {
        /* no-one can register a name that's a unique name as a group name */
        success = False;
      }
    }
  }
  
  /* do we need also to recursively register this name with a top-level
     wins server before granting the name?
   */
  if ((!error) && (secured_redirect || *lp_wins_register()))
  {
    struct in_addr wins_ip, send_ip;
    struct nmb_query_chllng *nmb_data;

	nmb_data = (struct nmb_query_chllng*)(malloc(sizeof(*nmb_data)));

    if (*lp_wins_register() && !secured_redirect)
    {
      send_ip = *interpret_addr2(lp_wins_register());
      if (zero_ip(send_ip))
      {
        DEBUG(1,("could not find ip address of recursive WINS server\n"));
      }
      if (ismyip(send_ip))
      {
        DEBUG(0,("ERROR: recursive name registration with ourself!\n"));
        return;
      }
    }

    if (nmb_data)
    {
	  uint16 id = nmb->header.name_trn_id;
      int fd;

	  send_wait_ack(p, id, 15*1000, register_ip[0].nb_flags);

      /* rfc1002.txt 5.1.4.1 p62 - save the id, ip and port number */
      putip(&nmb_data->reply_ip, &p->ip);
      memcpy(&nmb_data->nb, &register_ip[0], sizeof(nmb_data->nb));
      nmb_data->port = p->port;
      nmb_data->response_id = id;
      nmb_data->multi = multi;

      /* initiate some enquiries to the current owner. */
      if (secured_redirect)
      {
        /* END-NODE CHALLENGE REGISTRATION RESPONSE see rfc1002.txt 4.2.7 */
        putip(&send_ip, &n->ip_flgs[idx].ip);
        fd = ClientNMB[iface_idx(send_ip)];
	    netbios_name_query(p->timestamp,fd,response_name_challenge,
                           (void*)nmb_data,reply_name,
                           False,False,send_ip);
      }
      else
      {
        /* RECURSIVE NAME REGISTRATION */
        putip(&send_ip, &wins_ip);
        fd = ClientNMB[iface_idx(send_ip)];
	    netbios_name_register(p->timestamp,fd,nmb->header.opcode,
                              response_name_register,
                              (void*)nmb_data,reply_name,
                              &register_ip[0], 1,
                              False,False,send_ip);
      }

    }
  }
  else
  {
    /* Send a NAME REGISTRATION RESPONSE (pos/neg) see rfc1002.txt 4.2.13-14 */
  	add_name_respond(p, False, nmb->header.name_trn_id,
                        reply_name, 
                        &register_ip[0],1,
	                    success,False,
                        register_ip[0].ip,p->ip);
  }
}


/****************************************************************************
  RECURSIVE name registration. winsd will register names with another
  wins server, and then send a positive or negative name registration
  back to the original sender.

  response or no response for a name register received. 
  **************************************************************************/
static void response_name_register(time_t timestamp, struct packet_struct *p,
                                  struct response_record *n)
{
	BOOL success = False;
	struct nmb_name *ans_name = &n->name;
	struct nmb_ip reg_ip;
	struct nmb_query_chllng *nb_data = (struct nmb_query_chllng*)n->nmb_data;

	memcpy(&reg_ip, &n->nb, sizeof(reg_ip));
	reg_ip.source = REGISTER;

	if (!p)
	{
		if (n->bcast)
		{
			/* implicit registration */
			success = True;
		}
		else
		{
			DEBUG(1,("WINS server %s did not respond to name registration!\n",
			          inet_ntoa(n->send_ip)));
			/* XXXX whoops. we have problems. must deal with this */
			success = False;
		}
	}
	else
	{
		struct nmb_packet *nmb = &p->packet.nmb;
		int rcode = nmb->header.rcode;
		char *rdata = nmb->answers->rdata;

		reg_ip.death_time = nmb->answers->ttl;
		ans_name = &nmb->question.question_name;

		if (rcode == 0 && rdata)
		{
			/* copy the netbios flags and the ip address out of reply data */
			reg_ip.nb_flags = rdata[0];
			putip((char*)&reg_ip.ip,&rdata[2]);
			
			success = True;
		}

		if (n->num_msgs > 1)
		{
			DEBUG(1,("more than one name register response received!\n"));
			return;
		}
	}
	if (success)
	{
		DEBUG(3,("name registration for %s accepted by %s\n",
		            namestr(ans_name), inet_ntoa(n->send_ip)));
		add_netbios_entry(timestamp,&namelist, &names_last_modified, ans_name,
		                  &reg_ip, n->bcast, True);
	}
	else
	{
		DEBUG(3,("name registration for %s rejected by %s!\n",
				namestr(ans_name), inet_ntoa(n->send_ip)));
		remove_netbios_name(timestamp,&namelist, &names_last_modified,
				            ans_name, reg_ip.ip, True);
	}

	/* report results of registration to original name registerer */
 	p->port = nb_data->port;
	p->timestamp = timestamp;
	p->ip = nb_data->reply_ip;
	p->fd = ClientNMB[iface_idx(p->ip)];

	send_name_response(p, nb_data->response_id,
	                      nb_data->multi ? NMB_REG_MULTI : NMB_REG,
	                      success ? RCODE_REG_OK : RCODE_REG_ACT_ERR,
	                      True, True, ans_name,
	                      reg_ip.nb_flags, reg_ip.death_time,
	                      nb_data->reply_ip);
}

/****************************************************************************
  response from a name query for secured WINS registration. a state of
  NAME_REGISTER_CHALLENGE is dealt with here.
  ****************************************************************************/
static void response_name_challenge(time_t timestamp, struct packet_struct *p,
                                    struct response_record *n)
{
	struct nmb_name *ans_name = &n->name;
	struct nmb_query_chllng *nb_data = (struct nmb_query_chllng*)n->nmb_data;
	BOOL new_owner = False;

	struct nmb_ip found;
	struct packet_struct p2;

    memcpy(&found, &nb_data->nb, sizeof(found));

	if (!nb_data) return;

	if (!p)
	{
        /* name challenge: no reply. we can reply to the person that
           wanted the unique name and tell them that they can have it
         */

		putip(&found.ip, &nb_data->reply_ip);
		new_owner = True;

		DEBUG(4,("owner %s %s did not respond to name challenge\n",
					namestr(ans_name), inet_ntoa(n->send_ip)));
	}
	else
	{
		struct nmb_packet *nmb = &p->packet.nmb;
		char *rdata = nmb->answers->rdata;
 		BOOL success = False;

		int rcode = nmb->header.rcode;

		ans_name = &nmb->answers->rr_name;

		if (rcode == 0 && rdata)
		{
			/* copy the netbios flags and the ip address out of reply data */
			found.nb_flags = rdata[0];
			putip((char*)&found.ip,&rdata[2]);
			success = True;
		}

		DEBUG(4, ("Name query at %s ip %s - ",
			  namestr(&n->name), inet_ntoa(n->send_ip)));

		if (!name_equal(&n->name, ans_name))
		{
			/* someone gave us the wrong name as a reply. oops. */
			/* XXXX should say to them 'oi! release that name!' */

			DEBUG(4,("unexpected name received: %s\n", namestr(ans_name)));
			success = False;
		}

		if (n->num_msgs > 1)
		{
			if ((!NAME_GROUP(found.nb_flags)) && !ip_equal(n->from_ip, p->ip))
			{
				DEBUG(3,("Unique Name conflict detected!\n"));

				p2.ip = p->ip;
				p2.port = p->port;
				p2.fd = p->fd;
				p2.timestamp = p->timestamp;
				p2.packet_type = NMB_PACKET;

				/* note that the ip and nb_flags should be sent here - 
				   see rfc1002.txt 4.2.8
				 */
				reply_netbios_packet(&p2,nmb->header.name_trn_id,
									 RCODE_QUERY_CFT_ERR,NMB_QUERY,0,True,True,
									 ans_name,
									 RR_TYPE_NULL, RR_CLASS_IP,
									 0, NULL, 0);
				success = False;
			}
		}
		else
		{
			/* record who this packet is from, so that if another packet
			   is received, and it's a unique name, we can detect a conflict
			 */
			putip(&n->from_ip, &p->ip);
		}

		if (success)
		{
			/* we had sent out a name query to the current owner
			   of a name because someone else wanted it. now they
			   have responded saying that they still want the name,
			   so the other host can't have it.
			 */

			/* first check all the details are correct */
			if (!ip_equal(n->send_ip, found.ip))
			{
				/* someone gave us the wrong ip as a reply. oops. */
				/* XXXX should say to them 'oi! release that name!' */

				DEBUG(4,("expected ip: %s\n", inet_ntoa(n->send_ip)));
				DEBUG(4,("unexpected ip: %s\n", inet_ntoa(found.ip)));
				return;
			}

			DEBUG(4, (" OK: %s\n", inet_ntoa(found.ip)));

			/* fine: now tell the other host they can't have the name */
			new_owner = False;
		}
		else
		{
			DEBUG(4, (" NEGATIVE RESPONSE!\n"));

			/* the owner didn't want the name: the other host can have it */
			found.ip = nb_data->reply_ip;
			new_owner = True;
		}
	}

	p2.ip = nb_data->reply_ip;
	p2.port = nb_data->port;
	p2.fd = ClientNMB[iface_idx(nb_data->reply_ip)];
	p2.timestamp = timestamp;
	p2.packet_type = NMB_PACKET;

	/* register the old or the new owners' ip, removing the old
	   name as well as adding the new one.
	 */
	add_name_respond(&p2, False, nb_data->response_id, ans_name,
					&found, 1,
					new_owner, new_owner, n->send_ip, nb_data->reply_ip);
}


