/* 
   Unix SMB/Netbios implementation.
   Version 3.0
   NBT netbios routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

*/

#include "includes.h"

extern int ClientNMB;
extern int ServerNMB;
extern int ClientDGRAM;

extern int DEBUGLEVEL;

extern int num_response_packets;

BOOL CanRecurse = True;
extern struct in_addr ipgrp;


/***************************************************************************
  updates the unique transaction identifier
  **************************************************************************/
void debug_browse_data(char *outbuf, int len)
{
    int i,j;
    for (i = 0; i < len; i+= 16)
      {
    DEBUG(4, ("%3x char ", i));
    
    for (j = 0; j < 16; j++)
      {
        unsigned char x = outbuf[i+j];
        if (x < 32 || x > 127) x = '.';
        
        if (i+j >= len) break;
        DEBUG(4, ("%c", x));
      }
    
    DEBUG(4, (" hex ", i));
    
    for (j = 0; j < 16; j++)
      {
        if (i+j >= len) break;
        DEBUG(4, (" %02x", (unsigned char)outbuf[i+j]));
      }
    
    DEBUG(4, ("\n"));
      }
    
}


/****************************************************************************
  determine if a packet is for us. Note that to have any chance of
  being efficient we need to drop as many packets as possible at this
  stage as subsequent processing is expensive. 

  We also must make absolutely sure we don't tread on another machines
  property by answering a packet that is not for us.
  ****************************************************************************/
static BOOL listening(struct packet_struct *p,struct nmb_name *n)
{
  struct subnet_record *d;
  struct name_record *n1;
  int i;

  d = find_subnet(p->ip);
  
  n1 = find_name_search(&d,n,FIND_LOCAL|FIND_WINS|FIND_SELF, p->ip, &i);

  return (n1 != NULL);
}


/****************************************************************************
  process udp 138 datagrams
  ****************************************************************************/
static void process_dgram(struct packet_struct *p)
{
  char *buf;
  char *buf2;
  int len;
  struct dgram_packet *dgram = &p->packet.dgram;

  /* if we aren't listening to the destination name then ignore the packet */
  if (!listening(p,&dgram->dest_name))
  {
    DEBUG(4,("not listening to %s\n",namestr(&dgram->dest_name)));
    return;
  }


  if (dgram->header.msg_type != 0x10 &&
      dgram->header.msg_type != 0x11 &&
      dgram->header.msg_type != 0x12) {
    /* don't process error packets etc yet */
    DEBUG(4,("msg type %x ignored\n",dgram->header.msg_type));
    return;
  }

  buf = &dgram->data[0];
  buf -= 4; /* XXXX for the pseudo tcp length - 
           someday I need to get rid of this */

  if (CVAL(buf,smb_com) != SMBtrans) return;

  len = SVAL(buf,smb_vwv11);
  buf2 = smb_base(buf) + SVAL(buf,smb_vwv12);

  DEBUG(4,("datagram from %s to %s for %s of type %d len=%d\n",
       namestr(&dgram->source_name),namestr(&dgram->dest_name),
       smb_buf(buf),CVAL(buf2,0),len));
 
  if (len <= 0) return;

   /* datagram packet received for the browser mailslot */
   if (strequal(smb_buf(buf),BROWSE_MAILSLOT)) {
     process_browse_packet(p,buf2,len);
     return;
   }

   /* datagram packet received for the domain log on mailslot */
   if (strequal(smb_buf(buf),NET_LOGON_MAILSLOT)) {
     process_logon_packet(p,buf2,len);
     return;
   }
}

/****************************************************************************
  listens for NMB or DGRAM packets, and queues them
  ***************************************************************************/
void listen_for_packets(BOOL run_election)
{
	fd_set fds;
	int selrtn;
	struct timeval timeout;

	FD_ZERO(&fds);
	FD_SET(ClientNMB,&fds);
	FD_SET(ServerNMB,&fds);
	FD_SET(ClientDGRAM,&fds);

	/* during elections and when expecting a netbios response packet we
	   need to send election packets at tighter intervals 

	   ideally it needs to be the interval (in ms) between time now and
	   the time we are expecting the next netbios packet
     */

	timeout.tv_sec = (run_election||num_response_packets)?1:NMBD_SELECT_LOOP;
	timeout.tv_usec = 0;

	selrtn = sys_select(&fds,&timeout);

	if (FD_ISSET(ServerNMB,&fds))
	{
		struct packet_struct *packet = read_packet(ServerNMB, NMB_PACKET);
		if (packet)
		{
			if (ismyip(packet->ip) && packet->port == NMB_PORT &&
			    packet->packet.nmb.header.nm_flags.bcast)
			{
				debug_nmb_packet(packet);
				DEBUG(7,("discard nmb bcast non-response packet from %s:%d\n",
						inet_ntoa(packet->ip),packet->port));	  
			}
			else
			{
				process_nmb(packet);
			}
			free_packet(packet);
		}
	}

	if (FD_ISSET(ClientNMB,&fds))
	{
		struct packet_struct *packet = read_packet(ClientNMB, NMB_PACKET);
		if (packet)
		{
			if (ismyip(packet->ip) && packet->port == NMB_PORT &&
			    packet->packet.nmb.header.nm_flags.bcast)
			{
				debug_nmb_packet(packet);
				DEBUG(7,("discard nmb bcast non-response packet from %s:%d\n",
						inet_ntoa(packet->ip),packet->port));	  
			}
			else
			{
				process_nmb(packet);
			}
			free_packet(packet);
		}
	}

	if (FD_ISSET(ClientDGRAM,&fds))
	{
		struct packet_struct *packet = read_packet(ClientDGRAM, DGRAM_PACKET);
		if (packet)
		{
			if (ismyip(packet->ip) && packet->port == DGRAM_PORT)
			{
				DEBUG(7,("discarding own packet from %s:%d\n",
						inet_ntoa(packet->ip),packet->port));	  
			}
			else
			{
				process_dgram(packet);
			}
			free_packet(packet);
		}
	}
}



/****************************************************************************
  construct and send a netbios DGRAM

  Note that this currently sends all answers to port 138. thats the
  wrong things to do! I should send to the requestors port. XXX
  **************************************************************************/
BOOL send_mailslot_reply(char *mailslot,int fd,
				char *buf,int len,
				struct nmb_name *srcname, struct in_addr src_ip,
				struct nmb_name *dstname, struct in_addr dest_ip)
{
  struct packet_struct p;
  struct dgram_packet *dgram = &p.packet.dgram;
  struct in_addr wins_ip = ipgrp;
  char *ptr,*p2;
  char tmp[4];

  /* ha ha. no. do NOT send packets to 255.255.255.255: it's a pseudo address */
  if (ip_equal(wins_ip, dest_ip)) return False;

  bzero((char *)&p,sizeof(p));

  dgram->header.msg_type = 0x11; /* DIRECT GROUP DATAGRAM */
  dgram->header.flags.node_type = M_NODE;
  dgram->header.flags.first = True;
  dgram->header.flags.more = False;
  dgram->header.dgm_id = update_name_trn_id();
  dgram->header.source_ip = src_ip;
  dgram->header.source_port = DGRAM_PORT;
  dgram->header.dgm_length = 0; /* let build_dgram() handle this */
  dgram->header.packet_offset = 0;
  
  memcpy(&dgram->source_name, srcname, sizeof(*srcname));
  memcpy(&dgram->dest_name  , dstname, sizeof(*dstname));

  ptr = &dgram->data[0];

  /* now setup the smb part */
  ptr -= 4; /* XXX ugliness because of handling of tcp SMB length */
  memcpy(tmp,ptr,4);
  set_message(ptr,17,17 + len,True);
  memcpy(ptr,tmp,4);

  CVAL(ptr,smb_com) = SMBtrans;
  SSVAL(ptr,smb_vwv1,len);
  SSVAL(ptr,smb_vwv11,len);
  SSVAL(ptr,smb_vwv12,70 + strlen(mailslot));
  SSVAL(ptr,smb_vwv13,3);
  SSVAL(ptr,smb_vwv14,1);
  SSVAL(ptr,smb_vwv15,1);
  SSVAL(ptr,smb_vwv16,2);
  p2 = smb_buf(ptr);
  strcpy(p2,mailslot);
  p2 = skip_string(p2,1);

  memcpy(p2,buf,len);
  p2 += len;

  dgram->datasize = PTR_DIFF(p2,ptr+4); /* +4 for tcp length */

  p.ip = dest_ip;
  p.port = DGRAM_PORT;
  p.fd = ClientDGRAM;
  p.timestamp = time(NULL);
  p.packet_type = DGRAM_PACKET;

  DEBUG(4,("send mailslot %s from %s %s", mailslot,
                    inet_ntoa(src_ip),namestr(&dgram->source_name)));
  DEBUG(4,("to %s %s\n", inet_ntoa(dest_ip),namestr(&dgram->dest_name)));

  return(send_packet(&p));
}
