/* 
   Unix SMB/Netbios implementation.
   Version 3.0
   NBT netbios routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   04 jul 96: lkcl@pires.co.uk
   created module namedbsubnet containing subnet database functions

   30 July 96: David.Chappell@mail.trincoll.edu
   Expanded multiple workgroup domain master browser support.

*/

#include "includes.h"
#include "smb.h"


extern int DEBUGLEVEL;

extern struct in_addr ipgrp;
extern struct in_addr ipzero;

extern pstring myname;

BOOL updatedlists = True;
int updatecount = 0;

/* local interfaces structure */
extern struct interface *local_interfaces;

/* this is our subnet/workgroup/server database */
struct subnet_record *subnetlist = NULL;

static int num_subnets = 0;


/****************************************************************************
  add a domain into the list
  **************************************************************************/
int get_num_subnets(void)
{
    return num_subnets;
}


/****************************************************************************
  add a subnet into the list
  **************************************************************************/
static void add_subnet(struct subnet_record *d)
{
  struct subnet_record *d2;

  num_subnets++;

  if (!subnetlist)
  {
    subnetlist = d;
    d->prev = NULL;
    d->next = NULL;
    return;
  }

  for (d2 = subnetlist; d2->next; d2 = d2->next);

  d2->next = d;
  d->next = NULL;
  d->prev = d2;
}


/****************************************************************************
  find a subnet in the subnetlist 
  **************************************************************************/
struct subnet_record *find_subnet(struct in_addr bcast_ip)
{   
  struct subnet_record *d;
  
  /* search through subnet list for broadcast/netmask that matches
     the source ip address. a subnet 255.255.255.255 represents the
     WINS list. */
  
  for (d = subnetlist; d; d = d->next)
    {
        if (ip_equal(bcast_ip, ipgrp))
        {
           if (ip_equal(bcast_ip, d->bcast_ip))
           {
               return d;
           }
        }
        else if (same_net(bcast_ip, d->bcast_ip, d->mask_ip))
        {
          return(d);
        }
    }
  
  return (NULL);
}


/****************************************************************************
  finds the appropriate subnet structure. directed packets (non-bcast) are
  assumed to come from a point-to-point (P or M node), and so the subnet we
  return in this instance is the WINS 'pseudo-subnet' with ip 255.255.255.255
  ****************************************************************************/
struct subnet_record *find_req_subnet(struct in_addr ip, BOOL wins)
{
  if (wins)
  {
    /* find the subnet under the pseudo-ip of 255.255.255.255 */
    struct subnet_record *d = find_subnet(ipgrp);
    if (d) return d;
  }

  /* identify the subnet the broadcast request came from */
  return find_subnet(*iface_bcast(ip));
}


/****************************************************************************
  create a subnet entry
  ****************************************************************************/
struct subnet_record *make_subnet(struct in_addr bcast_ip,
                                  struct in_addr mask_ip)
{
  struct subnet_record *d;
  d = (struct subnet_record *)malloc(sizeof(*d));
  
  if (!d) return(NULL);
  
  bzero((char *)d,sizeof(*d));
  
  DEBUG(4, ("making subnet %s ", inet_ntoa(bcast_ip)));
  DEBUG(4, ("%s\n", inet_ntoa(mask_ip)));
  
  d->bcast_ip = bcast_ip;
  d->mask_ip  = mask_ip;
  d->myip     = *iface_ip(bcast_ip);

  d->workgrouplist = NULL;
  d->namelist = NULL;
  d->names_last_modified = 0;
  
  add_subnet(d);
  
  return d;
}


/****************************************************************************
  add the remote interfaces from lp_interfaces()
  to the netbios subnet database.
  ****************************************************************************/
void add_subnet_interfaces(BOOL add_wins)
{
	struct interface *i;

	/* loop on all local interfaces */
	for (i = local_interfaces; i; i = i->next)
	{
		/* add the interface into our subnet database */
		if (!find_subnet(i->bcast))
		{
		  make_subnet(i->bcast,i->nmask);
		}
	}

    /* add the pseudo-ip interface for WINS: 255.255.255.255 */
    if (add_wins && *lp_wins_server())
    {
        make_subnet(ipgrp, ipzero);
    }
    else
    {
        DEBUG(0,("not adding wins pseudo-subnet\n"));
    }
}

