/* 
   Unix SMB/Netbios implementation.
   Version 3.0
   NBT netbios routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Module name: namedbresp.c

*/

#include "includes.h"

extern int DEBUGLEVEL;

int num_response_packets = 0;
static struct response_record *responselist;
extern struct in_addr ipgrp;
extern struct in_addr ipzero;

extern char *source_description[];


/***************************************************************************
  add an expected response record into the list
  **************************************************************************/
void add_response_record(struct response_record *n)
{
  struct response_record *n2;

  num_response_packets++; /* count of total number of packets still around */

  DEBUG(4,("adding response record id:%d num_records:%d\n",
                   n->response_id, num_response_packets));

  if (!responselist)
  {
      responselist = n;
      n->prev = NULL;
      n->next = NULL;
  }
  else
  {
    for (n2 = responselist; n2->next; n2 = n2->next) ;
  
    n2->next = n;
    n->next = NULL;
    n->prev = n2;
  }
}


/***************************************************************************
  remove an expected response record from the list
  **************************************************************************/
void remove_response_record(struct response_record *n)
{
	/* remove global record of expected response */
	delete_response_record(n->response_id, False);

	if (n->prev) n->prev->next = n->next;
	if (n->next) n->next->prev = n->prev;

	if (responselist == n) responselist = n->next; 

	free(n);
	if (n->nmb_data) free(n->nmb_data);

	num_response_packets--; /* count of total number of packets still around */
}


/****************************************************************************
  create a name query response record
  **************************************************************************/
static struct response_record *make_response_queue_record(time_t time_now,
            void (*fn)(time_t,struct packet_struct *,struct response_record *),
			int reply_port, uint16 fd, int quest_type,
			struct nmb_name *name, void *nmb_data,
            struct nmb_ip *nb, int num_ips,
			BOOL bcast,BOOL recurse,
			struct in_addr send_ip)
{
  struct response_record *n;

  if (!fn) return NULL;

  if (!(n = (struct response_record *)malloc(sizeof(*n)))) 
    return(NULL);

  DEBUG(4,("make_response_record: "));
  DEBUG(4,("%s ", inet_ntoa(send_ip)));

  n->quest_type = quest_type; /* type of packet to be sent */

  /* when a response comes in, we record who it came from, so that
     if another response is received for a unique name, we can
     prove that there's a conflict
   */
  putip(&n->from_ip,&ipzero);

  n->num_msgs = 0;

  n->repeat_interval = 1; /* XXXX should be in ms */
  n->repeat_count = 3; /* 3 retries */
  n->repeat_time = time_now; 

  n->reply_port = reply_port;
  n->fd = fd;

  /* get an appropriate transaction id - one that isn't being used already */
  n->response_id = save_response_record(time_now, n->reply_port,
									send_ip, fd, n->repeat_time + 2, False);

  n->repeat_time += n->repeat_interval; /* initial retry time */

  if (nb)
  {
    memcpy(&n->nb, nb, sizeof(n->nb));
  }
  n->num_ips = num_ips;

  n->send_ip = send_ip;

  n->bcast = bcast;
  n->recurse = recurse;

  if (name)
  {
    memcpy(&n->name, name, sizeof(*name));
  }

  n->fn = fn;
  n->nmb_data = nmb_data;
  
  return n;
}


/****************************************************************************
  find a response in a subnet's name query response list. 
  **************************************************************************/
struct response_record *find_response_record(uint16 id)
{  
  struct response_record *n;

  if (!responselist) return NULL;

  for (n = responselist; n; n = n->next)
  {
    if (n->response_id == id) return n;
  }

  return NULL;
}


/****************************************************************************
  initiate a netbios name query to find someone's or someones' IP
  this is intended to be used (not exclusively) for broadcasting to
  master browsers (WORKGROUP(1d or 1b) or __MSBROWSE__(1)) to get
  complete lists across a wide area network

  a broadcast message will be over-ridden if the packet it sent to an
  ip of zero, and the packet will be sent to the WINS name server instead,
  if it exists. if samba is configured to be a WINS server, then the
  packet will be dealt with _as if_ it had been sent on the network,
  but in fact the function process_nmb() is called.
  ****************************************************************************/
static void queue_netbios_packet(time_t time_now, int fd, int quest_type,
            void (*fn)(time_t,struct packet_struct *, struct response_record *),
			void *nmb_data,
			struct nmb_name *name, struct nmb_ip *nb, int num_ips,
            BOOL bcast,BOOL recurse,
            struct in_addr send_ip)
{
  struct response_record *n;

  /* XXXX note: please see rfc1001.txt section 10 for details on this
     function: it is currently inappropriate to use this - it will do
     for now - once there is a clarification of B, M and P nodes and
     which one samba is supposed to be
   */

  DEBUG(4,("queue nb pkt: %s\n", inet_ntoa(send_ip)));

  if (ip_equal(send_ip, ipgrp) && (*lp_wins_server()))
  {
      struct in_addr wins_ip;

      if (bcast)
      {
        DEBUG(4,("error in queue_netbios_packet: bcast to wins server\n"));
        return;
      }

      /* we are using a WINS server */
      wins_ip = *interpret_addr2(lp_wins_server());
      
      if (!zero_ip(wins_ip))
      {
          putip(&send_ip,&wins_ip);
      }
      else
      {
          /* oops. wins server parameter MUST be a host_name or ip_address */
          DEBUG(0,("invalid smb.conf parameter 'wins server'\n"));
      }
  }
  
  /* queue the response expected because if we do a query with an ip
     of zero, we are expecting to hear from ourself immediately */
  if ((n = make_response_queue_record(time_now, fn, 
                        NMB_PORT, fd,
                        quest_type, name, nmb_data, nb, num_ips,
                        bcast, recurse, send_ip)))
  { 
    add_response_record(n);
  }

  if (!initiate_netbios_packet(time_now, n))
  {
    /* packet wasn't sent - not expecting a response */
    DEBUG(4,("did not initiate netbios packet: %s\n", inet_ntoa(send_ip)));
    remove_response_record(n);
    return;
  }
}


/****************************************************************************
  initiate netbios name query
  ****************************************************************************/
void netbios_name_query(time_t time_now, int fd,
            void (*fn)(time_t,struct packet_struct *, struct response_record *),
			void *nmb_data,
			struct nmb_name *name,
            BOOL bcast,BOOL recurse, struct in_addr send_ip)
{
	queue_netbios_packet(time_now, fd, NMB_QUERY, fn, 
	                     nmb_data, name, NULL, 0,
	                     bcast, recurse, send_ip);
}


/****************************************************************************
  initiate netbios name status
  ****************************************************************************/
void netbios_name_status(time_t time_now, int fd,
            void (*fn)(time_t,struct packet_struct *, struct response_record *),
			void *nmb_data,
			struct nmb_name *name,
            BOOL bcast,BOOL recurse, struct in_addr send_ip)
{
	queue_netbios_packet(time_now, fd, NMB_STATUS, fn, 
	                     nmb_data, name, NULL, 0,
	                     bcast, recurse, send_ip);
}


/****************************************************************************
  initiate netbios name registration
  ****************************************************************************/
void netbios_name_register(time_t time_now, int fd, int quest_type,
            void (*fn)(time_t,struct packet_struct *, struct response_record *),
			void *nmb_data,
			struct nmb_name *name, struct nmb_ip *nb, int num_ips,
            BOOL bcast,BOOL recurse, struct in_addr send_ip)
{
	queue_netbios_packet(time_now, fd, quest_type, fn, 
	                     nmb_data, name, nb, num_ips,
	                     bcast, recurse, send_ip);
}


/****************************************************************************
  initiate netbios name release
  ****************************************************************************/
void netbios_name_release(time_t time_now, int fd,
            void (*fn)(time_t,struct packet_struct *, struct response_record *),
			void *nmb_data,
			struct nmb_name *name, struct nmb_ip *nb, int num_ips,
            BOOL bcast,BOOL recurse, struct in_addr send_ip)
{
	queue_netbios_packet(time_now, fd, NMB_REL, fn, 
	                     nmb_data, name, nb, num_ips,
	                     bcast, recurse,
	                     send_ip);
}


/*******************************************************************
  remove old name response entries

  XXXX retry code needs to be added, including a retry wait period and a count
       see name_query() and name_status() for suggested implementation.

  ******************************************************************/
void expire_netbios_response_entries(time_t time_now)
{
	struct response_record *n, *nextn;

	for (n = responselist; n; n = nextn)
	{
		nextn = n->next;

		if (time_now >= n->repeat_time)
		{
			if (n->repeat_count > 0)
			{
				/* resend the entry */
				update_response_record(time_now+2,n->response_id,False);
				initiate_netbios_packet(time_now, n);

				n->repeat_time += n->repeat_interval; /* XXXX ms needed */
				n->repeat_count--;
			}
			else
			{
				DEBUG(4,("timeout %d response %d for %s %s\n",
				       n->repeat_time, n->response_id, namestr(&n->name),
				       inet_ntoa(n->send_ip)));

				if (n->num_msgs == 0 && n->fn)
				{
					n->fn(time_now, NULL, n); /* process the non-response */
				}
				remove_response_record(n); /* remove the non-response */
			}
		}
	}
}
