/*
 *   fs/cifs/misc.c
 *
 *   Copyright (c) International Business Machines  Corp., 2002
 *   Author(s): Steve French (sfrench@us.ibm.com)
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation; either version 2.1 of the License, or
 *   (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <linux/slab.h>
#include <linux/ctype.h>
#include "cifspdu.h"
#include "cifsglob.h"
#include "cifsproto.h"
#include "cifs_debug.h"

static DECLARE_MUTEX(GlobalMid_Sem);	/* also protects XID globals */
__u16 GlobalMid;		/* multiplex id - rotating counter */

/* The xid serves as a useful identifier for each incoming vfs request, 
   in a similar way to the mid which is useful to track each sent smb, 
   and CurrentXid can also provide a running counter (although it 
   will eventually wrap past zero) of the total vfs operations handled 
   since the cifs fs was mounted */

unsigned int _GetXid(void)
{
	unsigned int xid;

	down(&GlobalMid_Sem);
	GlobalTotalActiveXid++;
	if (GlobalTotalActiveXid > GlobalMaxActiveXid)
		GlobalMaxActiveXid = GlobalTotalActiveXid;	/* keep high water mark for number of simultaneous vfs ops in our filesystem */
	xid = GlobalCurrentXid++;
	up(&GlobalMid_Sem);
	return xid;
}

void _FreeXid(unsigned int xid)
{
	down(&GlobalMid_Sem);
	GlobalTotalActiveXid--;
	up(&GlobalMid_Sem);
}

struct cifsSesInfo *sesInfoAlloc(void)
{
	struct cifsSesInfo *ret_buf;

	ret_buf =
	    (struct cifsSesInfo *) kmalloc(sizeof(struct cifsSesInfo),
					   GFP_KERNEL);
	if (ret_buf) {
		memset(ret_buf, 0, sizeof(struct cifsSesInfo));
		atomic_inc(&sesInfoAllocCount);
		list_add(&ret_buf->cifsSessionList, &GlobalSMBSessionList);
		init_MUTEX(&ret_buf->sesSem);
	}
	return ret_buf;
}

void sesInfoFree(struct cifsSesInfo *buf_to_free)
{
    if(buf_to_free == NULL) {
        cFYI(1,("\nNull buffer passed to sesInfoFree"));
        return;
    }
     
	atomic_dec(&sesInfoAllocCount);
	list_del(&buf_to_free->cifsSessionList);
	if (buf_to_free->serverOS)
		kfree(buf_to_free->serverOS);
	if (buf_to_free->serverDomain)
		kfree(buf_to_free->serverDomain);
	if (buf_to_free->serverNOS)
		kfree(buf_to_free->serverNOS);
	kfree(buf_to_free);
}

struct cifsTconInfo *tconInfoAlloc(void)
{
	struct cifsTconInfo *ret_buf;
	ret_buf =
	    (struct cifsTconInfo *) kmalloc(sizeof(struct cifsTconInfo),
					    GFP_KERNEL);
	if (ret_buf) {
		memset(ret_buf, 0, sizeof(struct cifsTconInfo));
		atomic_inc(&tconInfoAllocCount);
		list_add(&ret_buf->cifsConnectionList,
			 &GlobalTreeConnectionList);
		init_MUTEX(&ret_buf->tconSem);
	}
	return ret_buf;
}

void tconInfoFree(struct cifsTconInfo *buf_to_free)
{
    if(buf_to_free == NULL)  {
        cFYI(1,("\nNull buffer passed to tconInfoFree"));
        return;
    }
      
	atomic_dec(&tconInfoAllocCount);
	list_del(&buf_to_free->cifsConnectionList);
	if (buf_to_free->nativeFileSystem)
		kfree(buf_to_free->nativeFileSystem);
	kfree(buf_to_free);
}

void *kcalloc(size_t size, int type)
{
	void *addr;
	addr = kmalloc(size, type);
	if (addr)
		memset(addr, 0, size);
	return addr;
}


struct smb_hdr *buf_get(void)
{
	struct smb_hdr *ret_buf;

/*  ret_buf = vmalloc(CIFS_MAX_MSGSIZE+MAX_CIFS_HDR_SIZE);*//* BB use negotiated size instead of max_msgsize */
	ret_buf =
	    (struct smb_hdr *) kmalloc(CIFS_MAX_MSGSIZE +
				       MAX_CIFS_HDR_SIZE, GFP_KERNEL);
	/* clear the first few header bytes */
	if (ret_buf) {
		memset(ret_buf, 0, sizeof(struct smb_hdr));
		atomic_inc(&bufAllocCount);
	}
	/* BB For better performance put these buffers in a pool */
	return ret_buf;
}

void buf_release(void *buf_to_free)
{
	/* BB release the memory if threshold exceeded else put back in pool */
/*  kfree(buf_to_free); */
    if(buf_to_free == NULL)
    {
        cFYI(1,("\nNull buffer passed to buf_release"));
        return;
    }
    kfree(buf_to_free);
	atomic_dec(&bufAllocCount);
	return;
}

void header_assemble(struct smb_hdr *buffer,
		     char smb_command /* command */ ,
		     const struct cifsTconInfo *treeCon, int word_count
		     /* length of fixed section (word count) in two byte units  */
    )
{
	int i;
	__u32 tmp;
	char *temp = (char *) buffer;

	for (i = 0; i < MAX_CIFS_HDR_SIZE; i++) {
		temp[i] = 0;	/* BB is this needed ?? */
	}

	buffer->smb_buf_length =
	    (2 * word_count) + sizeof(struct smb_hdr) -
	    4 /*  RFC 1001 length field does not count */  +
	    2 /* for bcc field itself */ ;
	/* Note that this is the only network field that has to be converted to big endian and it is done just before we send it */

	buffer->Protocol[0] = 0xFF;
	buffer->Protocol[1] = 'S';
	buffer->Protocol[2] = 'M';
	buffer->Protocol[3] = 'B';
	buffer->Command = smb_command;
	buffer->Flags = 0x00;	/* case sensitive */
	buffer->Flags2 = SMBFLG2_KNOWS_LONG_NAMES;
	tmp = cpu_to_le32(current->pid);
	buffer->Pid = tmp & 0xFFFF;
	tmp >>= 16;
	buffer->PidHigh = tmp & 0xFFFF;
	down(&GlobalMid_Sem);
	GlobalMid++;
	buffer->Mid = GlobalMid;
	if (treeCon) {
		buffer->Tid = treeCon->tid;
		if (treeCon->ses) {	/* BB fix endianness here */
			if (treeCon->ses->capabilities & CAP_UNICODE)
				buffer->Flags2 |= SMBFLG2_UNICODE;
			if (treeCon->ses->capabilities & CAP_STATUS32) {
				buffer->Flags2 |= SMBFLG2_ERR_STATUS;
			}
			buffer->Uid = treeCon->ses->Suid;	/* always in LE format */
		}
		if (treeCon->Flags & SMB_SHARE_IS_IN_DFS)	/* BB fix endianness here */
			buffer->Flags2 |= SMBFLG2_DFS;
	}
/*  buffer->Flags2 = cpu_to_le16(buffer->Flags2); *//* now moved - done just before sending */
	up(&GlobalMid_Sem);
	buffer->WordCount = (char) word_count;	/* converted later */
	return;
}

int checkSMBhdr(struct smb_hdr *smb, __u16 mid)
{
	/* Make sure that this really is an SMB, that it is a response, and that the message ids match */
	if ((*(unsigned int *) smb->Protocol == cpu_to_le32(0x424d53ff)) && (smb->Flags & SMBFLG_RESPONSE) && (mid == smb->Mid)) {	/* BB add check for length of SMB */
		return 0;
	} else {
		if (*(unsigned int *) smb->Protocol != cpu_to_le32(0x424d53ff))
			cERROR(1,
			       ("\n Bad protocol string signature header is %x ",
				*(unsigned int *) smb->Protocol));
		if (!(smb->Flags & SMBFLG_RESPONSE))
			cERROR(1, ("\n Not response "));
		if (mid != smb->Mid)
			cERROR(1, ("\n Mids do not match \n"));
		cERROR(1, ("CIFS: bad smb detected. The Mid=%d\n",
			   smb->Mid));
		return 1;
	}
}

int checkSMB(struct smb_hdr *smb, __u16 mid, int length)
{
	cFYI(0,
	     ("\nEntering checkSMB with Length: %x, smb_buf_length: %x ",
	      length, ntohl(smb->smb_buf_length)));
	if ((length < 2 + sizeof(struct smb_hdr))
	    || (4 + ntohl(smb->smb_buf_length) >
		CIFS_MAX_MSGSIZE + MAX_CIFS_HDR_SIZE)) {
		if (length < 2 + sizeof(struct smb_hdr)) {
			cERROR(1,
			       ("\n Length less than 2 + sizeof smb_hdr "));
			if ((length >= sizeof(struct smb_hdr) - 1)
			    && (smb->Status.CifsError != 0))
				return 0;	/* this is ok - some error cases do not return wct and bcc */

		}
		if (4 + ntohl(smb->smb_buf_length) >
		    CIFS_MAX_MSGSIZE + MAX_CIFS_HDR_SIZE)
			cERROR(1,
			       ("\n smb_buf_length greater than CIFS_MAX_MSGSIZE ... "));
		cERROR(1,
		       ("CIFS: bad smb detected. Illegal length. The mid=%d\n",
			smb->Mid));
		return 1;
	}

	if (checkSMBhdr(smb, mid))
		return 1;

	if ((4 + ntohl(smb->smb_buf_length) != smbCalcSize(smb))
	    || (4 + ntohl(smb->smb_buf_length) != length)) {
		return 0;
	} else {
		cERROR(1, ("\nCIFS: smbCalcSize %x ", smbCalcSize(smb)));
		cERROR(1,
		       ("CIFS: bad smb size detected. The Mid=%d\n",
			smb->Mid));
		return 1;
	}
}

void dump_smb(struct smb_hdr *smb_buf, int smb_buf_length)
{
	int i, j;
	char debug_line[17];
	unsigned char *buffer;

	if (traceSMB == 0)
		return;

	buffer = (unsigned char *) smb_buf;
	for (i = 0, j = 0; i < smb_buf_length; i++, j++) {
		if (i % 8 == 0) {	/* we have reached the beginning of line  */
			printk("\n| ");
			j = 0;
		}
		printk("%0#4x ", buffer[i]);
		debug_line[2 * j] = ' ';
		if (isprint(buffer[i]))
			debug_line[1 + (2 * j)] = buffer[i];
		else
			debug_line[1 + (2 * j)] = '_';

		if (i % 8 == 7) {	/* we have reached end of line, time to print ascii */
			debug_line[16] = 0;
			printk(" | %s", debug_line);
		}
	}
	for (; j < 8; j++) {
		printk("     ");
		debug_line[2 * j] = ' ';
		debug_line[1 + (2 * j)] = ' ';
	}
	printk(" | %s\n", debug_line);
	return;
}
