/*
 *   fs/cifs/connect.c
 *
 *   Copyright (c) International Business Machines  Corp., 2002
 *   Author(s): Steve French (sfrench@us.ibm.com)
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation; either version 2.1 of the License, or
 *   (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */
#include <linux/fs.h>
#include <linux/net.h>
#include <linux/string.h>
#include <linux/list.h>
#include <linux/wait.h>
#include <linux/version.h>
#include <linux/ipv6.h>
#include <asm/uaccess.h>
#include <asm/processor.h>
#include "cifspdu.h"
#include "cifsglob.h"
#include "cifsproto.h"
#include "cifs_unicode.h"
#include "cifs_debug.h"
#include "cifs_fs_sb.h"

#define CIFS_PORT 445
#define RFC1001_PORT 139

extern void SMBencrypt(unsigned char *passwd, unsigned char *c8,
		       unsigned char *p24);
extern void SMBNTencrypt(unsigned char *passwd, unsigned char *c8,
			 unsigned char *p24);
extern int inet_addr(char *);

struct smb_vol {
	char *username;
	char *password;
	char *domainname;
	char *UNC;
	char *UNCip;
	int id_of_kernel_thread_for_volume;	/* BB move this */
};

int cifs_demultiplex_thread(struct TCP_Server_Info *server)
{
	int length, total_read, i;
	unsigned int pdu_length;
	struct smb_hdr *smb_buffer = NULL;
	struct msghdr smb_msg;
	mm_segment_t temp_fs;
	struct iovec iov;
	struct socket *csocket = server->ssocket;
	struct list_head *tmp;
	struct smbSesInfo *ses;
	struct task_struct *task_to_wake = NULL;
	struct mid_q_entry *mid_entry;
	char *temp;

	daemonize();

	server->tsk = current;	/* save off process info so we can be woken at shutdown */
	cFYI(1, ("\nDemultiplex PID: %d", current->pid));

	temp_fs = get_fs();	/* we must turn off socket api parm checking */
	set_fs(get_ds());

	for (i = 0; server->fsStatus != CifsDying; i++) {
		if (smb_buffer == NULL)
			smb_buffer = buf_get();
		else
			memset(smb_buffer, 0, sizeof(struct smb_hdr));

		if (smb_buffer == NULL) {
			cERROR(1,
			       ("\n Error - can not get mem for SMB response buffer "));
			return ENOMEM;
		}
		iov.iov_base = smb_buffer;
		iov.iov_len = sizeof(struct smb_hdr) - 1;	/* 1 byte less since wct is not always returned in error cases */
		smb_msg.msg_iov = &iov;
		smb_msg.msg_iovlen = 1;
		smb_msg.msg_control = NULL;
		smb_msg.msg_controllen = 0;

		length =
		    sock_recvmsg(csocket, &smb_msg,
				 sizeof(struct smb_hdr) -
				 1 /* RFC1001 header and SMB header */ ,
				 MSG_PEEK /* flags see socket.h */ );
		if (length < 0) {
			cERROR(1,
			       ("\nReceived error on sock_recvmsg( peek) with length = %d\n",
				length));
			break;
		}
		if (length == 0) {
			cFYI(1,
			     ("\nZero length peek received - dead session?? "));
			/* schedule_timeout(HZ/4); 
			   continue; */
			break;
		}
		pdu_length = 4 + ntohl(smb_buffer->smb_buf_length);
		cFYI(1, ("\nPeek length rcvd: %d with smb length: %d on pass %d", length, pdu_length, i));	/* BB */

		temp = (char *) smb_buffer;
		if (length > 3) {
			if (temp[0] == (char) 0x85) {
				iov.iov_base = smb_buffer;
				iov.iov_len = 4;
				length = sock_recvmsg(csocket, &smb_msg, 4, 0);	/* BB verify length of keep alive packet BB */
				cFYI(0,
				     ("\nReceived 4 byte keep alive packet "));
			} else if ((temp[0] == (char) 0x83)
				   && (length == 5)) {
				/* we get this from Windows 98 instead of error on SMB negprot response */
				cERROR(1,
				       ("\nNegative RFC 1002 Session response. Error = 0x%x",
					temp[4]));
				break;

			} else if (temp[0] != (char) 0) {
				cERROR(1,
				       ("\nUnknown RFC 1001 frame received not 0x00 nor 0x85"));
				dump_mem(" Received Data is: ", temp,
					 length);
				break;
			} else {
				if ((length != sizeof(struct smb_hdr) - 1)
				    || (pdu_length >
					CIFS_MAX_MSGSIZE +
					MAX_CIFS_HDR_SIZE)
				    || (pdu_length <
					sizeof(struct smb_hdr) - 1)
				    ||
				    (checkSMBhdr
				     (smb_buffer, smb_buffer->Mid))) {
					cERROR(1,
					       (KERN_ERR
						"\nInvalid size or format for SMB found with length %d and pdu_lenght %d\n",
						length, pdu_length));
					/* BB fix by finding next smb signature - and reading off data until next smb ? BB */
					break;
				} else {	/* length ok */

					length = 0;
					iov.iov_base = smb_buffer;
					iov.iov_len = pdu_length;
					for (total_read = 0; total_read < pdu_length; total_read += length) {	/* BB check for buffer overflow with bad pdu_length */
						/*  iov.iov_base = smb_buffer+total_read;
						   iov.iov_len =  pdu_length-total_read; */
						length =
						    sock_recvmsg(csocket,
								 &smb_msg,
								 pdu_length
								 -
								 total_read
								 /* buflen */
								 , 0
								 /* see socket.h */
						    );
						cFYI(1,
						     ("\nFor iovlen %d Length received: %d with total read %d on pass %d",
						      iov.iov_len, length,
						      total_read, i));
						if (length == 0) {
							cERROR(1,
							       ("\nZero length receive when expecting %d ",
								pdu_length
								-
								total_read));
							break;
						}
					}
				}

				dump_smb(smb_buffer, length);
				if (checkSMB
				    (smb_buffer, smb_buffer->Mid,
				     length)) {
					cERROR(1,
					       ("\n Bad SMB Received "));
					continue;
				}

				list_for_each(tmp, &server->pending_mid_q) {
					mid_entry = list_entry(tmp, struct
							       mid_q_entry,
							       qhead);

					if (mid_entry->mid ==
					    smb_buffer->Mid) {
						cFYI(1,
						     (" Mid matched - waking up\n "));
						task_to_wake =
						    mid_entry->tsk;
						mid_entry->resp_buf =
						    smb_buffer;
					}
				}

				if (task_to_wake) {
					smb_buffer = NULL;	/* will be freed by users thread after he is done */
					wake_up_process(task_to_wake);
				} else
					cERROR(1, ("\n No task to wake - mid_entry->tsk is empty or oplock break received - add code for it !\n"));	/* BB add code to parse async oplock breaks coming from the server */
			}
		} else {
			cFYI(0,
			     ("\nFrame less than four bytes received  %d bytes long.",
			      length));
			if (length > 0) {
				length = sock_recvmsg(csocket, &smb_msg, length, 0);	/* throw away junk frame */
				cFYI(1,
				     (" with junk  0x%x in it\n",
				      *(__u32 *) smb_buffer));
			}
		}
	}			/* end for loop */
	/* BB add code to lock SMB sessions while releasing */
	sock_release(csocket);
	server->ssocket = NULL;
	set_fs(temp_fs);
	if (smb_buffer)
		buf_release(smb_buffer);	/* buffer usually freed in free_mid - but need to free it on error cases or exit case */
	if (list_empty(&server->pending_mid_q)) {
		/* loop through server session structures attached to this and mark them dead */
		list_for_each(tmp, &GlobalSMBSessionList) {
			ses =
			    list_entry(tmp, struct smbSesInfo,
				       smbSessionList);
			if (ses->server == server) {
				ses->failureFlag = FailDead;
				ses->server = NULL;
			}
		}
		kfree(server);
	} else			/* BB need to more gracefully handle the rare negative session response case because response will be still outstanding */
		cERROR(1, ("\nThere are still active MIDs in queue and we are exiting but we can not delete mid_q_entries or TCP_Server_Info structure due to pending requests MEMORY LEAK!!\n "));	/* BB wake up waitors, and/or wait and/or free stale mids and try again? BB */
/* BB Need to fix bug in error path above - perhaps wait until smb requests time out and then free the tcp per server struct BB */

/* sock_release(csocket); *//* now done in cifs_umount */
	cFYI(1, ("\nAbout to exit from demultiplex thread\n"));

	/* BB: Do we need to add reconnect logic here? */

	return 0;
}

int parse_mount_options(char *options, struct smb_vol *vol)
{
	char *value;
	char *data;
	vol->username = NULL;
	vol->password = NULL;
	vol->domainname = NULL;
	vol->UNC = NULL;
	vol->id_of_kernel_thread_for_volume = 0;

	if (!options)
		return 1;

	while ((data = strsep(&options, ",")) != NULL) {
		if (!*data)
			continue;
		if ((value = strchr(data, '=')) != NULL)
			*value++ = '\0';
		if (strnicmp(data, "user", 4) == 0) {
			if (!value || !*value) {
				printk(KERN_ERR
				       "CIFS: invalid or missing username");
				return 1;	/* needs_arg; */
			}
			if (strnlen(value, 200) < 200) {
				vol->username = value;
			} else {
				printk(KERN_ERR "CIFS: username too long");
				return 1;
			}
		} else if (strnicmp(data, "pass", 4) == 0) {
			if (!value || !*value) {
				vol->password = NULL;
			} else if (strnlen(value, 17) < 17) {
				vol->password = value;
			} else {
				printk(KERN_ERR "CIFS: password too long");
				return 1;
			}
		} else if ((strnicmp(data, "unc", 3) == 0)
			   || (strnicmp(data, "target", 6) == 0)
			   || (strnicmp(data, "path", 4) == 0)) {
			if (!value || !*value) {
				printk(KERN_ERR
				       "CIFS: invalid path to network resource");
				return 1;	/* needs_arg; */
			}
			if (strnlen(value, 300) < 300) {
				vol->UNC = value;
				if (strncmp(vol->UNC, "//", 2) == 0) {
					vol->UNC[0] = '\\';
					vol->UNC[1] = '\\';
				} else if (strncmp(vol->UNC, "\\\\", 2) != 0) {	/* target does not begin with // or \\ */
					printk(KERN_ERR
					       "CIFS: UNC Path does not begin with // or \\\\");
					return 1;
				}
				vol->UNCip = &vol->UNC[2];
			} else {
				printk(KERN_ERR "CIFS: UNC name too long");
				return 1;
			}
		} else if (strnicmp(data, "domain", 3) == 0) {
			if (!value || !*value) {
				printk(KERN_ERR
				       "CIFS: invalid domain name");
				return 1;	/* needs_arg; */
			}
			if (strnlen(value, 65) < 65) {
				vol->domainname = value;
				cFYI(1, ("\nDomain name set"));
			} else {
				printk(KERN_ERR
				       "CIFS: domain name too long");
				return 1;
			}

		} else
			printk(KERN_ERR
			       "CIFS: Unrecognized mount keyword %s",
			       value);
	}
	return 0;
}

struct smbSesInfo *find_tcp_session(__u32 new_target_ip_addr,
				    char *userName,
				    struct TCP_Server_Info **psrvTcp)
{
	struct list_head *tmp;
	struct smbSesInfo *ses;

	*psrvTcp = NULL;

	list_for_each(tmp, &GlobalSMBSessionList) {
		ses = list_entry(tmp, struct smbSesInfo, smbSessionList);
		if (ses->server->sockAddr.sin_addr.s_addr ==
		    new_target_ip_addr) {
			/* BB lock server and tcp session and increment use count here?? */
			*psrvTcp = ses->server;	/* found a match on the TCP session */
			/* BB check if reconnection needed */
			if (strncmp
			    (ses->userName, userName,
			     MAX_USERNAME_SIZE) == 0)
				return ses;	/* found an exact match on both SMB session and tcp session */
		}
	}
	return NULL;
}

struct smbTconInfo *find_unc(__u32 new_target_ip_addr, char *uncName)
{
	struct list_head *tmp;
	struct smbTconInfo *tcon;

	list_for_each(tmp, &GlobalTreeConnectionList) {
		tcon =
		    list_entry(tmp, struct smbTconInfo, smbConnectionList);
		if (tcon->ses) {
			if (tcon->ses->server) {
				if (tcon->ses->server->sockAddr.sin_addr.
				    s_addr == new_target_ip_addr) {
					/* BB lock tcon and server and tcp session and increment use count here?? */
					/* found a match on the TCP session */
					/* BB check if reconnection needed */
					if (strncmp
					    (tcon->treeName, uncName,
					     MAX_TREE_SIZE) == 0) {
						return tcon;	/* found an exact match on both SMB session and tcp session */
					}
				}
			}
		}
	}
	return NULL;
}


int cifs_mount(struct super_block *sb, struct cifs_sb_info *cifs_sb,
	       char *mount_data)
{
	int rc = 0;
	int xid;
	struct socket *csocket;
	struct sockaddr_in sin_server;
	struct sockaddr_in6 sin_server6;
	struct sockaddr_in sin;
	struct sockaddr_in6 sin6;
	struct smb_vol volume_info;
	struct smbSesInfo *pSesInfo = NULL;
	struct smbSesInfo *existingSes = NULL;
	struct smbTconInfo *tcon = NULL;
	struct TCP_Server_Info *srvTcp;
	char cryptKey[CIFS_CRYPTO_KEY_SIZE];
	char session_key[CIFS_SESSION_KEY_SIZE];
	char session_key2[CIFS_SESSION_KEY_SIZE];
	char password_with_pad[CIFS_ENCPWD_SIZE];

	xid = GetXid();
	cFYI(0,
	     ("\nEntering cifs_mount. Xid: %d with: %s\n", xid,
	      mount_data));

	parse_mount_options(mount_data, &volume_info);

	if (volume_info.username) {
		cFYI(1, ("\nUsername: %s ", volume_info.username));
	} else {
		cERROR(1, ("\nNo username specified "));
	}

	if (volume_info.UNC) {
		sin_server.sin_addr.s_addr = inet_addr(volume_info.UNCip);
		cFYI(1, ("\nUNC: %s", volume_info.UNC));
	} else {
		cERROR(1,
		       ("\nCIFS mount error: No UNC path (e.g. -o unc=//192.168.1.100/public) specified  "));
		FreeXid(xid);
		return -ENODEV;
	}
	/* BB add support to use the multiuser_mount flag BB */
	existingSes =
	    find_tcp_session(sin_server.sin_addr.s_addr,
			     volume_info.username, &srvTcp);
	if (srvTcp) {
		cFYI(1, ("\nExisting tcp session with server found "));	/* BB fix and add logic here */
	} else {		/* create socket */

		/* BB Move this ipv4 code to new helper tcp connect routine - and add new one to support ipv6: PF_INET6 vs. PF_INET etc BB */
		rc = sock_create(PF_INET, SOCK_STREAM, IPPROTO_TCP,
				 &csocket);
		if (rc < 0) {
			cERROR(1,
			       ("Error creating socket. Aborting operation\n"));
			FreeXid(xid);
			return rc;
		}
		sin.sin_family = AF_INET;
		sin.sin_addr.s_addr = INADDR_ANY;
/*   local_port = 1025; 
   do {
       sin.sin_port = htons(--local_port);
       rc = csocket->ops->bind(csocket, (struct sockaddr *) &sin, sizeof(sin));
       } while (rc && (local_port > 445)); 
   if(rc) {
        printk("\nError binding to port - someone else listening on a lot of ports %d\n", rc);
        FreeXid(xid); 
        return 0;
	}   *//* Bind unneeded */

		sin_server.sin_family = AF_INET;
		sin_server.sin_port = htons(CIFS_PORT);

		rc = csocket->ops->connect(csocket,
					   (struct sockaddr *) &sin_server,
					   sizeof(sin_server), 0
					   /* BB fix the timeout to be shorter - and find out what are all of the values for these flags? */
		    );
		if (rc < 0) {
			sin_server.sin_port = htons(RFC1001_PORT);
			rc = csocket->ops->connect(csocket,
						   (struct sockaddr *)
						   &sin_server,
						   sizeof(sin), 0);
			if (rc < 0) {
				cFYI(1,
				     ("CIFSfs: Error connecting to socket. %d\n",
				      rc));
				sock_release(csocket);
				FreeXid(xid);
				return rc;
			}
		}

		srvTcp =
		    kmalloc(sizeof(struct TCP_Server_Info), GFP_KERNEL);
		if (srvTcp == NULL)
			rc = ENOMEM;
		else {
			memset(srvTcp, 0, sizeof(struct TCP_Server_Info));
			memcpy(&srvTcp->sockAddr, &sin_server,
			       sizeof(struct sockaddr_in));
			srvTcp->ssocket = csocket;
			init_waitqueue_head(&srvTcp->response_q);
			INIT_LIST_HEAD(&srvTcp->pending_mid_q);
		}
	}

	if (existingSes) {
		pSesInfo = existingSes;
		cFYI(1, ("\nExisting smb sess found "));
	} else if (!rc) {
		cFYI(1, ("\nExisting smb sess not found "));
		pSesInfo = sesInfoAlloc();
		if (pSesInfo == NULL)
			rc = ENOMEM;
		else {
			pSesInfo->server = srvTcp;
			if (volume_info.id_of_kernel_thread_for_volume ==
			    0)
				kernel_thread((void *) (void *)
					      cifs_demultiplex_thread,
					      srvTcp,
					      CLONE_FS | CLONE_FILES |
					      CLONE_VM);
			sprintf(pSesInfo->serverName, "%u.%u.%u.%u",
				NIPQUAD(sin_server.sin_addr.s_addr));
		}

		/* send negotiate protocol smb */
		if (!rc)
			rc = CIFSSMBNegotiate(xid, pSesInfo, cryptKey);
		if (!rc) {
			cFYI(1,
			     ("\nSecurity Mode : %x", pSesInfo->secMode));
			cFYI(1,
			     (" Server Capabilities: %x",
			      pSesInfo->capabilities));
			cFYI(1,
			     (" Time Zone: 0x%x %d\n", pSesInfo->timeZone,
			      pSesInfo->timeZone));

			memset(password_with_pad, 0, CIFS_ENCPWD_SIZE);
			if (volume_info.password)
				strcpy(password_with_pad,
				       volume_info.password);

			SMBNTencrypt(password_with_pad, cryptKey,
				     session_key2);
			/* dump_mem("\nCIFS (Samba NT encrypt): ",session_key2, CIFS_SESSION_KEY_SIZE); */
			toUpper(cifs_sb->local_nls, password_with_pad);
			SMBencrypt(password_with_pad, cryptKey,
				   session_key);
			/* dump_mem("\nCIFS (Samba encrypt): ", session_key,
			   CIFS_SESSION_KEY_SIZE);             */

			rc = CIFSSMBSessSetup(xid, pSesInfo,
					      volume_info.username
					      /* BB fix getting user name from alternate locations */
					      , volume_info.domainname,
					      session_key, session_key2,
					      cifs_sb->local_nls);
			if (rc) {
				cERROR(1,
				       ("\nSend error in SessSetup = %d\n",
					rc));
			} else {
				cFYI(1,
				     ("CIFS Session Established successfully "));
				strncpy(pSesInfo->userName,
					volume_info.username,
					MAX_USERNAME_SIZE);
			}
		}
	}

	/* BB search for existing tcon to this server share */

	if (!rc) {
		tcon = tconInfoAlloc();
		if (tcon == NULL)
			rc = ENOMEM;
	}


/* BB Add find tree routine here BB */
	if (find_unc(sin_server.sin_addr.s_addr, volume_info.UNC))
		cFYI(1, ("\nFound match on UNC path "));

/* send tree connection smb */
	if (!rc) {
		cFYI(1, ("\nAbout to tree connect "));

		rc = CIFSSMBTCon(xid, pSesInfo, volume_info.UNC, tcon,
				 cifs_sb->local_nls);
		cFYI(1, ("\nCIFS Tcon rc = %d\n", rc));
	}
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,2))
	if (pSesInfo->capabilities & CAP_LARGE_FILES) {
		cFYI(1, ("\nLarge files supported "));
		sb->s_maxbytes = (u64) 1 << 63;
	} else
		sb->s_maxbytes = (u64) 1 << 31;	/* 2 GB */
#endif

/* on error free sesinfo and tcon struct if needed */
	if (rc) {
		if (tcon)
			tconInfoFree(tcon);	/* BB other cleanup to add? */
		if (existingSes == 0) {
			if (pSesInfo) {
				if (pSesInfo->server) {
					if (pSesInfo->server->ssocket) {	/* BB add lock on srvTcp here */
						sock_release(pSesInfo->
							     server->
							     ssocket);
					}
					kfree(pSesInfo->server);
					srvTcp = NULL;
				}
				sesInfoFree(pSesInfo);
				pSesInfo = NULL;
			}
		}
	} else {
		atomic_inc(&pSesInfo->inUse);
		atomic_inc(&tcon->useCount);
		cifs_sb->tcon = tcon;
		tcon->ses = pSesInfo;

		/* we do not care if the following two calls succeed - they are informational only */
		CIFSSMBQFSDeviceInfo(xid, tcon, cifs_sb->local_nls);
		CIFSSMBQFSAttributeInfo(xid, tcon, cifs_sb->local_nls);
		if (tcon->ses->capabilities & CAP_UNIX)
			CIFSSMBQFSUnixInfo(xid, tcon, cifs_sb->local_nls);
	}

	FreeXid(xid);

	return rc;
}

int CIFSSMBSessSetup(unsigned int xid, struct smbSesInfo *ses, char *user,
		     char *domain, char session_key[CIFS_SESSION_KEY_SIZE],
		     char session_key2[CIFS_SESSION_KEY_SIZE],
		     const struct nls_table *nls_codepage)
{
	struct smb_hdr *smb_buffer;
	struct smb_hdr *smb_buffer_response;
	SESSION_SETUP_ANDX *pSMB;
	SESSION_SETUP_ANDX *pSMBr;
	char *bcc_ptr;
	int rc = 0;
	int remaining_words = 0;
	int bytes_returned = 0;
	int len;

	cFYI(1, ("\nIn sesssetup "));

	smb_buffer = buf_get();
	if (smb_buffer == 0) {
		return ENOMEM;
	}
	smb_buffer_response = buf_get();
	if (smb_buffer_response == 0) {
		buf_release(smb_buffer);
		return ENOMEM;
	}
	pSMB = (SESSION_SETUP_ANDX *) smb_buffer;
	pSMBr = (SESSION_SETUP_ANDX *) smb_buffer_response;

	/* send SMBsessionSetup here */
	header_assemble(smb_buffer, SMB_COM_SESSION_SETUP_ANDX,
			0 /* no tCon exists yet */ , 13 /* wct */ );

	pSMB->req_no_secext.AndXCommand = 0xFF;
	pSMB->req_no_secext.MaxBufferSize = ses->maxBuf;
	cFYI(1, (" maxbuf is %d ", ses->maxBuf));
	pSMB->req_no_secext.MaxMpxCount = ses->maxReq;
	pSMB->req_no_secext.Capabilities =
	    CAP_LARGE_FILES | CAP_NT_SMBS | CAP_LEVEL_II_OPLOCKS;
	if (ses->capabilities & CAP_UNICODE) {
		smb_buffer->Flags2 |= SMBFLG2_UNICODE;
		pSMB->req_no_secext.Capabilities |= CAP_UNICODE;
	}
	if (ses->capabilities & CAP_STATUS32) {
		smb_buffer->Flags2 |= SMBFLG2_ERR_STATUS;
		pSMB->req_no_secext.Capabilities |= CAP_STATUS32;
	}
	if (ses->capabilities & CAP_DFS) {
		smb_buffer->Flags2 |= SMBFLG2_DFS;
		pSMB->req_no_secext.Capabilities |= CAP_DFS;
	}
	pSMB->req_no_secext.CaseInsensitivePasswordLength =
	    CIFS_SESSION_KEY_SIZE;
	pSMB->req_no_secext.CaseSensitivePasswordLength =
	    CIFS_SESSION_KEY_SIZE;
	bcc_ptr = pByteArea(smb_buffer);
	memcpy(bcc_ptr, (char *) session_key, CIFS_SESSION_KEY_SIZE);
	bcc_ptr += CIFS_SESSION_KEY_SIZE;
	memcpy(bcc_ptr, (char *) session_key2, CIFS_SESSION_KEY_SIZE);
	bcc_ptr += CIFS_SESSION_KEY_SIZE;

	if (ses->capabilities & CAP_UNICODE) {
		if ((int) bcc_ptr % 2) {	/* must be word aligned for Unicode strings */
			*bcc_ptr = 0;
			bcc_ptr++;
		}
		bytes_returned =
		    cifs_strtoUCS((UniChar *) bcc_ptr, user, 100,
				  nls_codepage);
		bcc_ptr += 2 * bytes_returned;	/* convert num of 16 bit words to bytes */
		bcc_ptr += 2;	/* trailing null */
		if (domain == NULL)
			bytes_returned =
			    cifs_strtoUCS((UniChar *) bcc_ptr,
					  "CIFS_LINUX_DOM", 32,
					  nls_codepage);
		else
			bytes_returned =
			    cifs_strtoUCS((UniChar *) bcc_ptr, domain, 64,
					  nls_codepage);
		bcc_ptr += 2 * bytes_returned;
		bcc_ptr += 2;
		bytes_returned =
		    cifs_strtoUCS((UniChar *) bcc_ptr, "Linux version ",
				  32, nls_codepage);
		bcc_ptr += 2 * bytes_returned;
		bytes_returned =
		    cifs_strtoUCS((UniChar *) bcc_ptr, UTS_RELEASE, 32,
				  nls_codepage);
		bcc_ptr += 2 * bytes_returned;
		bcc_ptr += 2;
		bytes_returned =
		    cifs_strtoUCS((UniChar *) bcc_ptr, CIFS_NETWORK_OPSYS,
				  64, nls_codepage);
		bcc_ptr += 2 * bytes_returned;
		bcc_ptr += 2;
	} else {
		strncpy(bcc_ptr, user, 200);
		bcc_ptr += strnlen(user, 200);
		*bcc_ptr = 0;
		bcc_ptr++;
		if (domain == NULL) {
			strcpy(bcc_ptr, "CIFS_LINUX_DOM");
			bcc_ptr += strlen("CIFS_LINUX_DOM") + 1;
		} else {
			strncpy(bcc_ptr, domain, 64);
			bcc_ptr += strnlen(domain, 64);
			*bcc_ptr = 0;
			bcc_ptr++;
		}
		strcpy(bcc_ptr, "Linux version ");
		bcc_ptr += strlen("Linux version ");
		strcpy(bcc_ptr, UTS_RELEASE);
		bcc_ptr += strlen(UTS_RELEASE) + 1;
		strcpy(bcc_ptr, CIFS_NETWORK_OPSYS);
		bcc_ptr += strlen(CIFS_NETWORK_OPSYS) + 1;
	}
	BCC(smb_buffer) = (int) bcc_ptr - (int) pByteArea(smb_buffer);
	smb_buffer->smb_buf_length += BCC(smb_buffer);

	rc = SendReceive(xid, ses, smb_buffer, smb_buffer_response,
			 &bytes_returned, 1);
	/* dump_mem("\nSessSetup response is: ", smb_buffer_response, 92);  */
	if (rc) {
/*    rc = map_smb_to_linux_error(smb_buffer_response);  *//* done in SendReceive now */
	} else if ((smb_buffer_response->WordCount == 3)
		   || (smb_buffer_response->WordCount == 4)) {
		if (pSMBr->resp.Action & GUEST_LOGIN)
			cFYI(1, (" Guest login"));	/* BB do we want to set anything in SesInfo struct ? */
		if (ses) {
			ses->Suid = smb_buffer_response->Uid;	/* UID left in wire format (le) */
			cFYI(1, ("UID = %d ", ses->Suid));
			bcc_ptr = pByteArea(smb_buffer_response);	/* response can have either 3 or 4 word count - Samba sends 3 */
			if ((pSMBr->resp.hdr.WordCount == 3)
			    || ((pSMBr->resp.hdr.WordCount == 4)
				&& (pSMBr->resp.SecurityBlobLength <
				    pSMBr->resp.ByteCount))) {
				if (pSMBr->resp.hdr.WordCount == 4)
					bcc_ptr +=
					    pSMBr->resp.SecurityBlobLength;

				if (smb_buffer->Flags2 &= SMBFLG2_UNICODE) {
					if ((int) (bcc_ptr) % 2) {
						remaining_words =
						    (BCC
						     (smb_buffer_response)
						     - 1) / 2;
						bcc_ptr++;	/* Unicode strings must be word aligned */
					} else {
						remaining_words =
						    BCC
						    (smb_buffer_response) /
						    2;
					}
					len =
					    UniStrnlen((UniChar *) bcc_ptr,
						       remaining_words -
						       1);
/* We look for obvious messed up bcc or strings in response so we do not go off the end since (at least) WIN2K 
   and Windows XP have a major bug in not null terminating last Unicode string in response  */
					ses->serverOS =
					    kcalloc(2 * (len + 1),
						    GFP_KERNEL);
					cifs_strfromUCS_le(ses->serverOS,
							   (UniChar *)
							   bcc_ptr, len,
							   nls_codepage);
					bcc_ptr += 2 * (len + 1);
					remaining_words -= len + 1;
					ses->serverOS[2 * len] = 0;
					ses->serverOS[1 + (2 * len)] = 0;
					if (remaining_words > 0) {
						len =
						    UniStrnlen((UniChar *)
							       bcc_ptr,
							       remaining_words
							       - 1);
						ses->serverNOS =
						    kcalloc(2 * (len + 1),
							    GFP_KERNEL);
						cifs_strfromUCS_le(ses->
								   serverNOS,
								   (UniChar
								    *)
								   bcc_ptr,
								   len,
								   nls_codepage);
						bcc_ptr += 2 * (len + 1);
						ses->serverNOS[2 * len] =
						    0;
						ses->serverNOS[1 +
							       (2 * len)] =
						    0;
						remaining_words -= len + 1;
						if (remaining_words > 0) {
							len = UniStrnlen((UniChar *) bcc_ptr, remaining_words);	/* last string is not always null terminated (for e.g. for Windows XP & 2000) */
							ses->serverDomain =
							    kcalloc(2 *
								    (len +
								     1),
								    GFP_KERNEL);
							cifs_strfromUCS_le
							    (ses->
							     serverDomain,
							     (UniChar *)
							     bcc_ptr, len,
							     nls_codepage);
							bcc_ptr +=
							    2 * (len + 1);
							ses->
							    serverDomain[2
									 *
									 len]
							    = 0;
							ses->
							    serverDomain[1
									 +
									 (2
									  *
									  len)]
							    = 0;
						} /* else no more room so create dummy domain string */
						else
							ses->serverDomain =
							    kcalloc(2,
								    GFP_KERNEL);
					} else {	/* no room so create dummy domain and NOS string */
						ses->serverDomain =
						    kcalloc(2, GFP_KERNEL);
						ses->serverNOS =
						    kcalloc(2, GFP_KERNEL);
					}
				} else {	/* ASCII */

					len = strnlen(bcc_ptr, 1024);
					if (((int) bcc_ptr + len) - (int)
					    pByteArea(smb_buffer_response)
					    <= BCC(smb_buffer_response)) {
						ses->serverOS =
						    kcalloc(len + 1,
							    GFP_KERNEL);
						strncpy(ses->serverOS,
							bcc_ptr, len);

						bcc_ptr += len;
						bcc_ptr[0] = 0;	/* null terminate the string */
						bcc_ptr++;

						len =
						    strnlen(bcc_ptr, 1024);
						ses->serverNOS =
						    kcalloc(len + 1,
							    GFP_KERNEL);
						strncpy(ses->serverNOS, bcc_ptr, len);	/* BB watch for buffer overflow */
						bcc_ptr += len;
						bcc_ptr[0] = 0;
						bcc_ptr++;

						len =
						    strnlen(bcc_ptr, 1024);
						ses->serverDomain =
						    kcalloc(len + 1,
							    GFP_KERNEL);
						strncpy(ses->serverDomain, bcc_ptr, len);	/* BB watch for buffer overflow */
						bcc_ptr += len;
						bcc_ptr[0] = 0;
						bcc_ptr++;
					} else
						cFYI(1,
						     ("Variable field of length %d extends beyond end of smb ",
						      len));
				}
			} else {
				cERROR(1,
				       (" Security Blob Length extends beyond end of SMB"));
			}
		} else {
			cERROR(1, ("No session structure passed in."));
		}
	} else {
		cERROR(1,
		       (" Invalid Word count %d: ",
			smb_buffer_response->WordCount));
		rc = -EIO;
	}
	/* BB check wct of 17 and sanity check cryptkeylen before copying */
	if (smb_buffer)
		buf_release(smb_buffer);
	if (smb_buffer_response)
		buf_release(smb_buffer_response);

	return rc;
}

int CIFSSMBTCon(unsigned int xid, struct smbSesInfo *ses, const char *tree,
		struct smbTconInfo *tcon,
		const struct nls_table *nls_codepage)
{
	struct smb_hdr *smb_buffer;
	struct smb_hdr *smb_buffer_response;
	TCONX_REQ *pSMB;
	TCONX_RSP *pSMBr;
	char *bcc_ptr;
	int rc = 0;
	int length;

	if (ses == NULL)
		return EIO;

	smb_buffer = buf_get();
	if (smb_buffer == 0) {
		return ENOMEM;
	}
	smb_buffer_response = buf_get();
	if (smb_buffer_response == 0) {
		buf_release(smb_buffer);
		return ENOMEM;
	}

	header_assemble(smb_buffer, SMB_COM_TREE_CONNECT_ANDX,
			0 /*no tid */ , 4 /*wct */ );
	smb_buffer->Uid = ses->Suid;
	pSMB = (TCONX_REQ *) smb_buffer;
	pSMBr = (TCONX_RSP *) smb_buffer_response;

	pSMB->AndXCommand = 0xFF;
	pSMB->PasswordLength = 1;	/* minimum */
	bcc_ptr = &(pSMB->Password[0]);
	bcc_ptr++;		/* skip password */

	if (ses->capabilities & CAP_STATUS32) {
		smb_buffer->Flags2 |= SMBFLG2_ERR_STATUS;
	}
	if (ses->capabilities & CAP_DFS) {
		smb_buffer->Flags2 |= SMBFLG2_DFS;
	}
	if (ses->capabilities & CAP_UNICODE) {
		smb_buffer->Flags2 |= SMBFLG2_UNICODE;
		length =
		    cifs_strtoUCS((UniChar *) bcc_ptr, tree, 100,
				  nls_codepage);
		bcc_ptr += 2 * length;	/* convert num of 16 bit words to bytes */
		bcc_ptr += 2;	/* skip trailing null */
	} else {		/* ASCII */

		strcpy(bcc_ptr, tree);
		bcc_ptr += strlen(tree) + 1;
	}
	strcpy(bcc_ptr, "?????");
	bcc_ptr += strlen("?????");
	bcc_ptr += 1;
	BCC(smb_buffer) = (int) bcc_ptr - (int) pByteArea(smb_buffer);
	smb_buffer->smb_buf_length += BCC(smb_buffer);

	rc = SendReceive(xid, ses, smb_buffer, smb_buffer_response,
			 &length, 0);

	/* if (rc) rc = map_smb_to_linux_error(smb_buffer_response); not needed now done in SendReceive */
	if (rc == 0) {
		tcon->tid = smb_buffer_response->Tid;
		bcc_ptr = pByteArea(smb_buffer_response);
		length = strnlen(bcc_ptr, BCC(smb_buffer_response) - 2);
		bcc_ptr += length + 1;	/* skip service field (NB: this field is always ASCII) */
		strncpy(tcon->treeName, tree, MAX_TREE_SIZE);
		if (smb_buffer->Flags2 &= SMBFLG2_UNICODE) {
			length = UniStrnlen((UniChar *) bcc_ptr, 512);
			if (((int) bcc_ptr + (2 * length)) -
			    (int) pByteArea(smb_buffer_response) <=
			    BCC(smb_buffer_response)) {
				tcon->nativeFileSystem =
				    kcalloc(length + 1, GFP_KERNEL);
				cifs_strfromUCS_le(tcon->nativeFileSystem,
						   (UniChar *) bcc_ptr,
						   length, nls_codepage);
				bcc_ptr += 2 * length;
				bcc_ptr[0] = 0;	/* null terminate the string */
				bcc_ptr[1] = 0;
				bcc_ptr += 2;
			}
			/* else do not bother copying these informational fields */
		} else {
			length = strnlen(bcc_ptr, 1024);
			if (((int) bcc_ptr + length) -
			    (int) pByteArea(smb_buffer_response) <=
			    BCC(smb_buffer_response)) {
				tcon->nativeFileSystem =
				    kmalloc(length + 1, GFP_KERNEL);
				strncpy(tcon->nativeFileSystem, bcc_ptr,
					length);
			}
			/* else do not bother copying these informational fields */
		}
		tcon->Flags = pSMBr->OptionalSupport;
		cFYI(1, ("\nTcon flags: 0x%x ", tcon->Flags));
	}

	if (smb_buffer)
		buf_release(smb_buffer);
	if (smb_buffer_response)
		buf_release(smb_buffer_response);
	return rc;
}

int cifs_umount(struct super_block *sb, struct cifs_sb_info *cifs_sb)
{
	int rc = 0;
	int xid;
	struct smbSesInfo *ses = NULL;

	xid = GetXid();
	if (cifs_sb->tcon) {
		ses = cifs_sb->tcon->ses;	/* have to save ptr to ses struct before we delete tcon! */
		rc = CIFSSMBTDis(xid, cifs_sb->tcon);
		if (rc == -EBUSY) {
			FreeXid(xid);
			return 0;
		}
		tconInfoFree(cifs_sb->tcon);
		if ((ses) && (ses->server)) {	/* BB only if ses use count is now zero - fix count logic - move into Tdis */
			cFYI(1, ("\nAbout to do SMBLogoff "));
			rc = CIFSSMBLogoff(xid, ses);
			if (rc == -EBUSY) {
				FreeXid(xid);
				return 0;
			}
			ses->server->fsStatus = CifsDying;
			wake_up_process(ses->server->tsk);
			schedule_timeout(HZ / 4);	/* give captive thread time to exit */
		} else
			cFYI(1, ("\nNo session or bad tcon"));
	}
	/* BB future check active count of tcon and then free if needed BB */
	cifs_sb->tcon = NULL;
	if (ses) {
		schedule_timeout(HZ / 2);
		if (ses->server)
			kfree(ses->server);
	}
	if (ses)
		sesInfoFree(ses);

	FreeXid(xid);
	return rc;		/* BB check if we should always return zero here */
}
