/*
 *   fs/cifs/file.c
 *
 *   vfs operations that deal with files
 * 
 *   Copyright (c) International Business Machines  Corp., 2002
 *   Author(s): Steve French (sfrench@us.ibm.com)
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation; either version 2.1 of the License, or
 *   (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <linux/fs.h>
#include <linux/stat.h>
#include <linux/fcntl.h>
#include <asm/div64.h>
#include "cifsfs.h"
#include "cifspdu.h"
#include "cifsglob.h"
#include "cifsproto.h"
#include "cifs_unicode.h"
#include "cifs_debug.h"
#include "cifs_fs_sb.h"

int cifs_open(struct inode *inode, struct file *file)
{
	int rc = -EACCES;
	int xid;
	struct cifs_sb_info *cifs_sb;
	struct smbTconInfo *pTcon;
	char *full_path = NULL;
	int desiredAccess = 0x20197;
	int disposition = FILE_OPEN;
	__u16 netfid;

	xid = GetXid();

	cFYI(1, (" inode = 0x%p file flags are %x", inode, file->f_flags));
	cifs_sb = (struct cifs_sb_info *) (&(inode->i_sb->u));
	pTcon = cifs_sb->tcon;

	full_path = build_path_from_dentry(file->f_dentry);

	if ((file->f_flags & O_ACCMODE) == O_RDONLY)
		desiredAccess = GENERIC_READ;
	else if ((file->f_flags & O_ACCMODE) == O_WRONLY)
		desiredAccess = GENERIC_WRITE;
	else if ((file->f_flags & O_ACCMODE) == O_RDWR)
		desiredAccess = GENERIC_ALL;

/*	if ((file->f_flags & O_ACCMODE) == O_RDONLY)   
        desiredAccess =
		    FILE_READ_DATA | FILE_READ_ATTRIBUTES | FILE_EXECUTE |
		    READ_CONTROL;
	else if ((file->f_flags & O_ACCMODE) == O_WRONLY)
		desiredAccess =
		    FILE_WRITE_DATA | FILE_WRITE_ATTRIBUTES |
		    FILE_APPEND_DATA;
	else if ((file->f_flags & O_ACCMODE) == O_RDWR)
		desiredAccess =
		    FILE_WRITE_DATA | FILE_READ_DATA |
		    FILE_WRITE_ATTRIBUTES | FILE_READ_ATTRIBUTES |
		    FILE_APPEND_DATA | FILE_EXECUTE | READ_CONTROL;  */

/* BB go through these flags carefully to find equivalent NTCreateX flags */
/*
#define O_ACCMODE	   0003
#define O_RDONLY	     00
#define O_WRONLY	     01
#define O_RDWR		     02
#define O_CREAT		   0100	
#define O_EXCL		   0200	
#define O_NOCTTY	   0400	
#define O_TRUNC		  01000	
#define O_APPEND	  02000
#define O_NONBLOCK	  04000
#define O_NDELAY	O_NONBLOCK
#define O_SYNC		 010000
#define FASYNC		 020000	
#define O_DIRECT	 040000	
#define O_LARGEFILE	0100000
#define O_DIRECTORY	0200000	
#define O_NOFOLLOW	0400000
#define O_ATOMICLOOKUP	01000000 */

	if (file->f_flags & O_CREAT)
		disposition = FILE_OPEN_IF;	/* open if exists, create if does not exist */

	/* BB pass O_SYNC flag through on file attributes .. BB */
	rc = CIFSSMBOpen(xid, pTcon, full_path, disposition, desiredAccess,
			 0x3FF
			 /* i.e. 777 dummy value, ignored for time being */
			 , &netfid, cifs_sb->local_nls);
	if (rc) {
		cFYI(1, ("\ncifs_open returned 0x%x ", rc));
	} else {
		file->private_data =
		    kmalloc(sizeof(struct smbFileInfo), GFP_KERNEL);
		if (file->private_data) {
			memset(file->private_data, 0,
			       sizeof(struct smbFileInfo));
			((struct smbFileInfo *) file->private_data)->
			    netfid = netfid;
		}
	}

	if (full_path)
		kfree(full_path);
	FreeXid(xid);
	return rc;
}

int cifs_close(struct inode *inode, struct file *file)
{
	int rc = 0;
	int xid;
	struct cifs_sb_info *cifs_sb;
	struct smbTconInfo *pTcon;
	struct smbFileInfo *pSMBFile =
	    (struct smbFileInfo *) file->private_data;

	cFYI(1, ("\n  inode = 0x%p with ", inode));

	xid = GetXid();

	cifs_sb = (struct cifs_sb_info *) (&(inode->i_sb->u));
	pTcon = cifs_sb->tcon;

	if (pSMBFile) {
		rc = CIFSSMBClose(xid, pTcon, pSMBFile->netfid);
		kfree(file->private_data);
		file->private_data = NULL;
	} else
		rc = -EBADF;

	FreeXid(xid);
	return rc;

}

int cifs_closedir(struct inode *inode, struct file *file)
{
	int rc = 0;
	int xid;
	struct smbFileInfo *pSMBFileStruct =
	    (struct smbFileInfo *) file->private_data;

	cFYI(1, ("\n  inode = 0x%p with ", inode));

	xid = GetXid();

	if (pSMBFileStruct) {
		kfree(file->private_data);
		file->private_data = NULL;
	}
	FreeXid(xid);
	return rc;

}


int cifs_lock(struct file *file, int cmd, struct file_lock *pfLock)
{
	int rc, xid;
	__u32 lockType = LOCKING_ANDX_LARGE_FILES;
	__u32 numLock = 0;
	__u32 numUnlock = 0;
	__u64 length;
	struct cifs_sb_info *cifs_sb;
	struct smbTconInfo *pTcon;
	length = 1 + pfLock->fl_end - pfLock->fl_start;

	rc = -EACCES;

	xid = GetXid();

	cFYI(1,
	     ("\nLock parm: 0x%x flockflags: 0x%x flocktype: 0x%x start: %lld end: %lld",
	      cmd, pfLock->fl_flags, pfLock->fl_type, pfLock->fl_start,
	      pfLock->fl_end));

	if (pfLock->fl_flags & FL_POSIX)
		cFYI(1, ("\nPosix "));
	if (pfLock->fl_flags & FL_FLOCK)
		cFYI(1, ("\nFlock "));
	if (pfLock->fl_flags & FL_BROKEN)
		cFYI(1, ("\nBroken flock emulation "));
	if (pfLock->fl_flags & FL_ACCESS)
		cFYI(1, ("\nProcess suspended by mandatory locking "));
	if (pfLock->fl_flags & FL_LEASE)
		cFYI(1, ("\nLease on file "));
	if (pfLock->fl_flags & 0xFFD0)
		cFYI(1, ("\n Unknown lock flags "));

	if (pfLock->fl_type == F_WRLCK) {
		cFYI(1, ("\nF_WRLCK "));
		numLock = 1;
	} else if (pfLock->fl_type == F_UNLCK) {
		cFYI(1, ("\nF_UNLCK "));
		numUnlock = 1;
	} else if (pfLock->fl_type == F_RDLCK) {
		cFYI(1, ("\nF_RDLCK "));
		lockType |= LOCKING_ANDX_SHARED_LOCK;
		numLock = 1;
	} else if (pfLock->fl_type == F_EXLCK) {
		cFYI(1, ("\nF_EXLCK "));
		numLock = 1;
	} else if (pfLock->fl_type == F_SHLCK) {
		cFYI(1, ("\nF_SHLCK "));
		lockType |= LOCKING_ANDX_SHARED_LOCK;
		numLock = 1;
	} else
		cFYI(1, ("\nUnknown type of lock "));

	cifs_sb = (struct cifs_sb_info *) (&(file->f_dentry->d_sb->u));
	pTcon = cifs_sb->tcon;


	if (file->private_data == NULL) {
		FreeXid(xid);
		return -EBADF;
	}

	if (IS_GETLK(cmd)) {
		rc = CIFSSMBLock(xid, pTcon,
				 ((struct smbFileInfo *) file->
				  private_data)->netfid,
				 length,
				 pfLock->fl_start, 0, 1, lockType,
				 0 /* wait flag */ );
		if (rc == 0) {
			rc = CIFSSMBLock(xid, pTcon,
					 ((struct smbFileInfo *) file->
					  private_data)->netfid,
					 length,
					 pfLock->fl_start,
					 1 /* numUnlock */ ,
					 0 /* numLock */ , lockType,
					 0 /* wait flag */ );
			pfLock->fl_type = F_UNLCK;
			if (rc != 0)
				cERROR(1,
				       ("\nError unlocking previously locked range %d during test of lock ",
					rc));
			rc = 0;

		} else {
			/* if rc == ERR_SHARING_VIOLATION ? */
			rc = 0;	/* do not change lock type to unlock since the range is in use */
		}

		FreeXid(xid);
		return rc;
	}

	rc = CIFSSMBLock(xid, pTcon,
			 ((struct smbFileInfo *) file->private_data)->
			 netfid, length,
			 pfLock->fl_start, numUnlock, numLock, lockType,
			 0 /* wait flag */ );
	FreeXid(xid);
	return rc;
}


ssize_t cifs_write(struct file * file, const char *write_data,
		   size_t write_size, loff_t * poffset)
{
	int rc = 0;
	int bytes_written = 0;
	int total_written;
	struct cifs_sb_info *cifs_sb;
	struct smbTconInfo *pTcon;
	int xid, long_op;
	xid = GetXid();

	cifs_sb = (struct cifs_sb_info *) (&(file->f_dentry->d_sb->u));
	pTcon = cifs_sb->tcon;

	cFYI(1,
	     (" write %d bytes to offset %lld of %s \n", write_size,
	      *poffset, file->f_dentry->d_name.name));

	if (file->private_data == NULL) {
		FreeXid(xid);
		return -EBADF;
	}

	if (*poffset > file->f_dentry->d_inode->i_size)
		long_op = 2;	/* writes past end of file can take a long time */
	else
		long_op = 1;

	for (total_written = 0; write_size > total_written;
	     total_written += bytes_written) {
		rc = CIFSSMBWrite(xid, pTcon,
				  ((struct smbFileInfo *) file->
				   private_data)->netfid,
				  write_size - total_written, *poffset,
				  &bytes_written,
				  write_data + total_written, long_op);
		if (rc || (bytes_written == 0)) {
			FreeXid(xid);
			if (total_written)
				break;
			else {
				FreeXid(xid);
				return rc;
			}
		} else
			*poffset += bytes_written;
		long_op = FALSE;	/* subsequent writes should be fast - 15 seconds is plenty of time */
	}
	file->f_dentry->d_inode->i_ctime =
	    file->f_dentry->d_inode->i_mtime = CURRENT_TIME;
	mark_inode_dirty_sync(file->f_dentry->d_inode);
	return total_written;
}

int cifs_fsync(struct file *file, struct dentry *dentry, int datasync)
{
	int xid;
	int rc = 0;
	xid = GetXid();
	/* BB fill in code to flush write behind buffers - when we start supporting oplock */
	cFYI(1,
	     (" name: %s datasync: 0x%x ", dentry->d_name.name, datasync));
	FreeXid(xid);
	return rc;
}

ssize_t cifs_read(struct file * file, char *read_data, size_t read_size,
		  loff_t * poffset)
{
	int rc = -EACCES;
	int bytes_read = 0;
	int total_read;
	struct cifs_sb_info *cifs_sb;
	struct smbTconInfo *pTcon;
	int xid;

	xid = GetXid();
	cifs_sb = (struct cifs_sb_info *) (&(file->f_dentry->d_sb->u));
	pTcon = cifs_sb->tcon;

	if (file->private_data == NULL) {
		FreeXid(xid);
		return -EBADF;
	}

	for (total_read = 0; read_size > total_read;
	     total_read += bytes_read) {
		rc = CIFSSMBRead(xid, pTcon,
				 ((struct smbFileInfo *) file->
				  private_data)->netfid,
				 read_size - total_read, *poffset,
				 &bytes_read, read_data + total_read);
		if (rc || (bytes_read == 0)) {
			if (total_read) {
				break;
			} else {
				FreeXid(xid);
				return rc;
			}
		} else
			*poffset += bytes_read;
	}

	FreeXid(xid);
	return total_read;
}

void fill_in_inode(struct inode *tmp_inode,
		   FILE_DIRECTORY_INFO * pfindData, int *pobject_type)
{
	/* Linux can not store file creation time unfortunately so we ignore it */
	tmp_inode->i_atime = cifs_NTtimeToUnix(pfindData->LastAccessTime);
	tmp_inode->i_mtime = cifs_NTtimeToUnix(pfindData->LastWriteTime);
	tmp_inode->i_ctime = cifs_NTtimeToUnix(pfindData->ChangeTime);
	tmp_inode->i_mode = S_IRWXUGO;	/* 777 perms */
	cFYI(0,
	     ("\nCIFS FFIRST: Attributes came in as 0x%x",
	      pfindData->ExtFileAttributes));
	if (pfindData->ExtFileAttributes & ATTR_REPARSE) {
		*pobject_type = DT_LNK;
		tmp_inode->i_mode |= S_IFLNK;	/* BB can this and S_IFREG or S_IFDIR be set at same time as in Windows? */
	} else if (pfindData->ExtFileAttributes & ATTR_DIRECTORY) {
		*pobject_type = DT_DIR;
		tmp_inode->i_mode |= S_IFDIR;
	} else {
		*pobject_type = DT_REG;
		tmp_inode->i_mode |= S_IFREG;
	}			/* BB add code here - to validate if device or weird share type or odd device type? BB */


	tmp_inode->i_size = pfindData->EndOfFile;
	tmp_inode->i_blocks =
	    do_div(pfindData->AllocationSize, tmp_inode->i_blksize);
	cFYI(1,
	     ("\nFinddata alloc size (from smb) %lld",
	      pfindData->AllocationSize));
	if (pfindData->AllocationSize < pfindData->EndOfFile)
		cFYI(1, ("\nServer inconsistency Error: it says allocation size less than end of file "));	/* BB remove */
	cFYI(1,
	     ("\nCIFS FFIRST: Size %ld and blocks %ld ",
	      (unsigned long) tmp_inode->i_size, tmp_inode->i_blocks));
	if (S_ISREG(tmp_inode->i_mode)) {
		cFYI(1, (" File inode "));
		tmp_inode->i_op = &cifs_file_inode_ops;
		tmp_inode->i_fop = &cifs_file_ops;
	} else if (S_ISDIR(tmp_inode->i_mode)) {
		cFYI(1, (" Directory inode"));
		tmp_inode->i_op = &cifs_dir_inode_ops;
		tmp_inode->i_fop = &cifs_dir_ops;
	} else if (S_ISLNK(tmp_inode->i_mode)) {
		cFYI(1, (" Symbolic Link inode "));
		tmp_inode->i_op = &cifs_symlink_inode_ops;
	} else {
		cFYI(1, ("\nInit special inode "));
		init_special_inode(tmp_inode, tmp_inode->i_mode,
				   kdev_t_to_nr(tmp_inode->i_rdev));
	}
}

void unix_fill_in_inode(struct inode *tmp_inode,
			FILE_UNIX_INFO * pfindData, int *pobject_type)
{
	tmp_inode->i_atime = cifs_NTtimeToUnix(pfindData->LastAccessTime);
	tmp_inode->i_mtime =
	    cifs_NTtimeToUnix(pfindData->LastModificationTime);
	tmp_inode->i_ctime =
	    cifs_NTtimeToUnix(pfindData->LastStatusChange);

	/* tmp_inode->i_mtime =
	   cifs_NTtimeToUnix(pfindData->LastModificationTime);
	   tmp_inode->i_ctime =
	   cifs_NTtimeToUnix(pfindData->LastStatusChange); */
	tmp_inode->i_mode = pfindData->Permissions;
	if (pfindData->Type == UNIX_FILE) {
		*pobject_type = DT_REG;
		tmp_inode->i_mode |= S_IFREG;
	} else if (pfindData->Type == UNIX_SYMLINK) {
		*pobject_type = DT_LNK;
		tmp_inode->i_mode |= S_IFLNK;
	} else if (pfindData->Type == UNIX_DIR) {
		*pobject_type = DT_DIR;
		tmp_inode->i_mode |= S_IFDIR;
	} else if (pfindData->Type == UNIX_CHARDEV) {
		*pobject_type = DT_CHR;
		tmp_inode->i_mode |= S_IFCHR;
	} else if (pfindData->Type == UNIX_BLOCKDEV) {
		*pobject_type = DT_BLK;
		tmp_inode->i_mode |= S_IFBLK;
	} else if (pfindData->Type == UNIX_FIFO) {
		*pobject_type = DT_FIFO;
		tmp_inode->i_mode |= S_IFIFO;
	} else if (pfindData->Type == UNIX_SOCKET) {
		*pobject_type = DT_SOCK;
		tmp_inode->i_mode |= S_IFSOCK;
	}

	tmp_inode->i_uid = pfindData->Uid;
	tmp_inode->i_gid = pfindData->Gid;
	tmp_inode->i_nlink = pfindData->Nlinks;

	tmp_inode->i_size = pfindData->EndOfFile;
	tmp_inode->i_blocks =
	    do_div(pfindData->NumOfBytes, tmp_inode->i_blksize);
	cFYI(1, ("\nFinddata alloc size (from smb) %lld", pfindData->NumOfBytes));	/* BB remove */
	if (pfindData->NumOfBytes < pfindData->EndOfFile)
		cFYI(1, ("\nServer inconsistency Error: it says allocation size less than end of file "));	/* BB remove */
	cFYI(1, ("\nCIFS FFIRST: Size %ld and blocks %ld ", (unsigned long) tmp_inode->i_size, tmp_inode->i_blocks));	/* BB remove */
	if (S_ISREG(tmp_inode->i_mode)) {
		cFYI(1, (" File inode "));
		tmp_inode->i_op = &cifs_file_inode_ops;
		tmp_inode->i_fop = &cifs_file_ops;
	} else if (S_ISDIR(tmp_inode->i_mode)) {
		cFYI(1, (" Directory inode"));
		tmp_inode->i_op = &cifs_dir_inode_ops;
		tmp_inode->i_fop = &cifs_dir_ops;
	} else if (S_ISLNK(tmp_inode->i_mode)) {
		cFYI(1, (" Symbolic Link inode "));
		tmp_inode->i_op = &cifs_symlink_inode_ops;
/* tmp_inode->i_fop = *//* do not need to set to anything */
	} else {
		cFYI(1, ("\nInit special inode "));
		init_special_inode(tmp_inode, tmp_inode->i_mode,
				   kdev_t_to_nr(tmp_inode->i_rdev));
	}
}

int cifs_readdir(struct file *file, void *direntry, filldir_t filldir)
{
	int rc = 0;
	int rc2 = 0;
	int total_returned_this_readdir = 0;
	int last_index = 0;
	int xid, i;
	int existing;
	int done_search = FALSE;
	int Unicode = FALSE;
	int UnixSearch = FALSE;
	__u16 searchHandle;
	struct cifs_sb_info *cifs_sb;
	int object_type;
	struct smbTconInfo *pTcon;
	char *full_path = NULL;
	char *data;
	struct inode *tmp_inode;
	struct dentry *tmp_dentry;
	struct qstr qstring;
	T2_FFIRST_RSP_PARMS findParms;
	T2_FNEXT_RSP_PARMS findNextParms;
	FILE_DIRECTORY_INFO *pfindData;
	FILE_UNIX_INFO *pUnixFindData;

	xid = GetXid();

	data = kmalloc(4096, GFP_KERNEL);
	pfindData = (FILE_DIRECTORY_INFO *) data;
	pUnixFindData = (FILE_UNIX_INFO *) data;

	cifs_sb = (struct cifs_sb_info *) (&(file->f_dentry->d_sb->u));
	pTcon = cifs_sb->tcon;

	full_path = build_wildcard_path_from_dentry(file->f_dentry);

	cFYI(1,
	     ("\nFull path is %s with position %lld ", full_path,
	      file->f_pos));

	i = file->f_pos;
	switch (i) {
	case 0:
		if (filldir
		    (direntry, ".", 1, i, file->f_dentry->d_inode->i_ino,
		     DT_DIR) < 0) {
			cERROR(1,
			       ("\nFilldir for current directory direntry failed "));
			break;
		}
		i++;
		file->f_pos++;
		/* fallthrough */
	case 1:
		if (filldir
		    (direntry, "..", 2, i,
		     file->f_dentry->d_parent->d_inode->i_ino,
		     DT_DIR) < 0) {
			cERROR(1,
			       ("\nFilldir for parent directory direntry failed "));
			break;
		}
		i++;
		file->f_pos++;
		/* fallthrough */

		rc = CIFSFindFirst(xid, pTcon, full_path, pfindData,
				   &findParms, cifs_sb->local_nls,
				   &Unicode, &UnixSearch);
		cFYI(1, ("Count: %d  More avail entries: 0x%x ",
			 findParms.SearchCount, findParms.EndofSearch));

		if (rc == 0) {
			searchHandle = findParms.SearchHandle;
			file->private_data =
			    kmalloc(sizeof(struct smbFileInfo),
				    GFP_KERNEL);
			if (file->private_data) {
				memset(file->private_data, 0,
				       sizeof(struct smbFileInfo));
				((struct smbFileInfo *) file->
				 private_data)->netfid = searchHandle;
			}
			for (i = 2; i < findParms.SearchCount + 2; i++) {
				if (UnixSearch == FALSE) {
					cFYI(1,
					     ("\nFilename length: %d",
					      pfindData->FileNameLength));
					if (Unicode == TRUE)
						pfindData->FileNameLength =
						    cifs_strfromUCS_le
						    (pfindData->FileName,
						     (UniChar *)
						     pfindData->FileName,
						     (pfindData->
						      FileNameLength)
						     / 2,
						     cifs_sb->local_nls);
					qstring.name = pfindData->FileName;
					qstring.len =
					    pfindData->FileNameLength;
					qstring.hash =
					    full_name_hash(qstring.name,
							   qstring.len);
					tmp_dentry =
					    d_lookup(file->f_dentry,
						     &qstring);
					if (tmp_dentry) {
						cFYI(1, ("\nExisting dentry found with inode 0x%p", tmp_dentry->d_inode));	/* BB remove */
						existing = TRUE;
						tmp_inode =
						    tmp_dentry->d_inode;
						/* BB overwrite the old name? i.e. tmp_dentry->d_name and tmp_dentry->d_name.len ?? */
					} else {
						existing = FALSE;
						tmp_dentry =
						    d_alloc(file->f_dentry,
							    &qstring);
						tmp_inode =
						    new_inode(file->
							      f_dentry->
							      d_sb);
						tmp_dentry->d_op =
						    &cifs_dentry_ops;
						cFYI(1,
						     ("\nInstantiating dentry 0x%p with inode 0x%p ",
						      tmp_dentry,
						      tmp_inode));
						d_instantiate(tmp_dentry,
							      tmp_inode);
						if (!existing)
							d_rehash
							    (tmp_dentry);
					}
					tmp_dentry->d_time = jiffies;
					tmp_inode->i_blksize =
					    (pTcon->ses->maxBuf -
					     MAX_CIFS_HDR_SIZE) &
					    0xFFFFFE00;
					fill_in_inode(tmp_inode, pfindData,
						      &object_type);
					cFYI(1,
					     ("\nFile name is %s ",
					      pfindData->FileName));
					if (((pfindData->FileNameLength !=
					      1)
					     || (pfindData->FileName[0] !=
						 '.'))
					    &&
					    ((pfindData->FileNameLength !=
					      2)
					     || (pfindData->FileName[0] !=
						 '.')
					     || (pfindData->FileName[1] !=
						 '.'))) {
						filldir(direntry,
							pfindData->
							FileName,
							pfindData->
							FileNameLength, i,
							tmp_inode->i_ino,
							object_type);
						file->f_pos++;
					}
					renew_parental_timestamps(file->
								  f_dentry);
					pfindData =
					    (FILE_DIRECTORY_INFO
					     *) ((char *)
						 pfindData +
						 pfindData->
						 NextEntryOffset);
					/* BB also check to make sure that it is not beyond the end of the SMB */
				} else {	/* Unix Search */
					if (Unicode == TRUE)
						qstring.len =
						    cifs_strfromUCS_le
						    (pUnixFindData->
						     FileName, (UniChar *)
						     pUnixFindData->
						     FileName,
						     MAX_PATHCONF,
						     cifs_sb->local_nls);
					else
						qstring.len =
						    strnlen(pUnixFindData->
							    FileName,
							    MAX_PATHCONF);
					qstring.name =
					    pUnixFindData->FileName;
					qstring.hash =
					    full_name_hash(qstring.name,
							   qstring.len);
					tmp_dentry =
					    d_lookup(file->f_dentry,
						     &qstring);
					if (tmp_dentry) {
						existing = TRUE;
						cFYI(1, ("\nExisting dentry found for "));	/* BB remove */
						tmp_inode =
						    tmp_dentry->d_inode;
					} else {
						existing = FALSE;
						tmp_dentry =
						    d_alloc(file->f_dentry,
							    &qstring);
						tmp_inode =
						    new_inode(file->
							      f_dentry->
							      d_sb);
						tmp_dentry->d_op =
						    &cifs_dentry_ops;
						cFYI(1,
						     ("\nInstantiating dentry 0x%p with inode 0x%p ",
						      tmp_dentry,
						      tmp_inode));
						d_instantiate(tmp_dentry,
							      tmp_inode);
						if (!existing)
							d_rehash
							    (tmp_dentry);
					}
					tmp_dentry->d_time = jiffies;
					tmp_inode->i_blksize =
					    (pTcon->ses->maxBuf -
					     MAX_CIFS_HDR_SIZE) &
					    0xFFFFFE00;
					unix_fill_in_inode(tmp_inode,
							   pUnixFindData,
							   &object_type);
					cFYI(1,
					     ("\nFile name is %s ",
					      pUnixFindData->FileName));
					if (((qstring.len != 1)
					     || (pUnixFindData->
						 FileName[0] != '.'))
					    && ((qstring.len != 2)
						|| (pUnixFindData->
						    FileName[0] != '.')
						|| (pUnixFindData->
						    FileName[1] != '.'))) {
						filldir(direntry,
							pUnixFindData->
							FileName,
							qstring.len, i,
							tmp_inode->i_ino,
							object_type);
						file->f_pos++;
					}
					renew_parental_timestamps(file->
								  f_dentry);
					pUnixFindData =
					    (FILE_UNIX_INFO *) ((char *)
								pUnixFindData
								+
								pUnixFindData->
								NextEntryOffset);
					/* BB also check to make sure that it is not beyond the end of the SMB */
				}
			}	/* end for loop */
			last_index = i;
			total_returned_this_readdir = last_index;
			if (findParms.EndofSearch != 0)
				done_search = TRUE;
		} else {
			done_search = TRUE;
			rc = 0;	/* unless parent directory disappeared - do not return error here on 1st pass */
		}		/* end if else */
	default:
		if (file->private_data)
			searchHandle =
			    ((struct smbFileInfo *) (file->private_data))->
			    netfid;
		else {
			done_search = TRUE;
			rc = -EBADF;
			break;
		}
		while (done_search == FALSE) {
			if (total_returned_this_readdir < 127) {
				findNextParms.SearchCount =
				    127 - (total_returned_this_readdir);
			} else
				break;
			rc = CIFSFindNext(xid, pTcon, pfindData,
					  &findNextParms, searchHandle, 0,
					  &Unicode, &UnixSearch);
			if ((rc == 0) && (findNextParms.SearchCount != 0)) {
				for (i = 0; i < findNextParms.SearchCount;
				     i++, total_returned_this_readdir++) {
					if (UnixSearch == FALSE) {
						if (Unicode == TRUE)
							pfindData->
							    FileNameLength
							    =
							    cifs_strfromUCS_le
							    (pfindData->
							     FileName,
							     (UniChar *)
							     pfindData->
							     FileName,
							     (pfindData->
							      FileNameLength)
							     / 2,
							     cifs_sb->
							     local_nls);
						qstring.name =
						    pfindData->FileName;
						qstring.len =
						    pfindData->
						    FileNameLength;
						qstring.hash =
						    full_name_hash(qstring.
								   name,
								   qstring.
								   len);
						tmp_dentry =
						    d_lookup(file->
							     f_dentry,
							     &qstring);
						if (tmp_dentry) {
							existing = TRUE;
							cFYI(1, ("\nExisting dentry found for "));	/* BB remove */
							tmp_inode =
							    tmp_dentry->
							    d_inode;
						} else {
							existing = FALSE;
							tmp_dentry =
							    d_alloc(file->
								    f_dentry,
								    &qstring);
							tmp_inode =
							    new_inode
							    (file->
							     f_dentry->
							     d_sb);
							tmp_dentry->d_op =
							    &cifs_dentry_ops;
							cFYI(1,
							     ("\nAdding dentry 0x%p with inode 0x%p ",
							      tmp_dentry,
							      tmp_inode));
							d_instantiate
							    (tmp_dentry,
							     tmp_inode);
							if (!existing)
								d_rehash
								    (tmp_dentry);
						}
						tmp_dentry->d_time =
						    jiffies;

						tmp_inode->i_blksize =
						    (pTcon->ses->maxBuf -
						     MAX_CIFS_HDR_SIZE) &
						    0xFFFFFE00;

						fill_in_inode(tmp_inode,
							      pfindData,
							      &object_type);

						cFYI(1,
						     ("\nFile name is %s ",
						      pfindData->
						      FileName));
						if (((pfindData->
						      FileNameLength != 1)
						     || (pfindData->
							 FileName[0] !=
							 '.'))
						    &&
						    ((pfindData->
						      FileNameLength != 2)
						     || (pfindData->
							 FileName[0] !=
							 '.')
						     || (pfindData->
							 FileName[1] !=
							 '.'))) {
							rc2 =
							    filldir
							    (direntry,
							     pfindData->
							     FileName,
							     pfindData->
							     FileNameLength,
							     i +
							     last_index,
							     tmp_inode->
							     i_ino,
							     object_type);
							if (rc2)
								cFYI(1,
								     (" filldir rc2 = %d ",
								      rc2));
							file->f_pos++;
						}

						pfindData =
						    (FILE_DIRECTORY_INFO
						     *) ((char *) pfindData
							 +
							 pfindData->
							 NextEntryOffset);
						/* BB also check to make sure that it is not beyond the end of the SMB */
					} else {
						/* BB fill in Unix search logic info here */
						if (Unicode == TRUE)
							qstring.len =
							    cifs_strfromUCS_le
							    (pUnixFindData->
							     FileName,
							     (UniChar *)
							     pUnixFindData->
							     FileName,
							     MAX_PATHCONF,
							     cifs_sb->
							     local_nls);
						else
							qstring.len =
							    strnlen
							    (pUnixFindData->
							     FileName,
							     MAX_PATHCONF);

						qstring.name =
						    pUnixFindData->
						    FileName;
						qstring.hash =
						    full_name_hash(qstring.
								   name,
								   qstring.
								   len);
						tmp_dentry =
						    d_lookup(file->
							     f_dentry,
							     &qstring);
						if (tmp_dentry) {
							existing = TRUE;
							cFYI(1, ("\nExisting dentry found for "));	/* BB remove */
							tmp_inode =
							    tmp_dentry->
							    d_inode;
						} else {
							existing = FALSE;
							tmp_dentry =
							    d_alloc(file->
								    f_dentry,
								    &qstring);
							tmp_inode =
							    new_inode
							    (file->
							     f_dentry->
							     d_sb);
							tmp_dentry->d_op =
							    &cifs_dentry_ops;
							cFYI(1,
							     ("\nAdding dentry 0x%p with inode 0x%p ",
							      tmp_dentry,
							      tmp_inode));
							d_instantiate
							    (tmp_dentry,
							     tmp_inode);
							if (!existing)
								d_rehash
								    (tmp_dentry);
						}
						tmp_dentry->d_time =
						    jiffies;

						tmp_inode->i_blksize =
						    (pTcon->ses->maxBuf -
						     MAX_CIFS_HDR_SIZE) &
						    0xFFFFFE00;

						unix_fill_in_inode
						    (tmp_inode,
						     pUnixFindData,
						     &object_type);

						cFYI(1,
						     ("\nFile name is %s ",
						      pUnixFindData->
						      FileName));
						if (((qstring.len != 1)
						     || (pUnixFindData->
							 FileName[0] !=
							 '.'))
						    && ((qstring.len != 2)
							|| (pUnixFindData->
							    FileName[0] !=
							    '.')
							|| (pUnixFindData->
							    FileName[1] !=
							    '.'))) {
							rc2 =
							    filldir
							    (direntry,
							     pUnixFindData->
							     FileName,
							     qstring.len,
							     i +
							     last_index,
							     tmp_inode->
							     i_ino,
							     object_type);
							if (rc2)
								cFYI(1,
								     (" filldir rc2 = %d ",
								      rc2));
							file->f_pos++;
						}

						pUnixFindData =
						    (FILE_UNIX_INFO
						     *) ((char *)
							 pUnixFindData +
							 pUnixFindData->
							 NextEntryOffset);
						/* BB also check to make sure that it is not beyond the end of the SMB */
					}
				}	/* end for loop */
				if (findNextParms.EndofSearch != 0)
					done_search = TRUE;
				last_index += i;
			} else {
				rc = 0;
				break;
			}
		}		/* end while */
	}			/* end switch */
	if (data)
		kfree(data);
	if (full_path)
		kfree(full_path);
	FreeXid(xid);
	return rc;
}
