/*  $Revision: 1.11 $
**
**  Expire overview database.
*/
#include <stdio.h>
#include <sys/types.h>
#include "configdata.h"
#include "clibrary.h"
#include <ctype.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <fcntl.h>
#include <errno.h>
#include "qio.h"
#include "mydir.h"
#include "libinn.h"
#include "macros.h"
#include "paths.h"
#include "tree.h"
#include "can.h"

#define	START_LIST_SIZE	128


/*
**   Information about a line in the overview file.
*/
typedef struct _LINE {
    ARTNUM	Article;
    char	*Start;
    int		Length;
    int		Offset;
    char	*Canpos;
} LINE;


/*
**  A list of articles; re-uses space.
*/
typedef struct _LIST {
    int		Used;
    int		Size;
    ARTNUM	*Articles;
} LIST;


/*
**  A buffer; re-uses space.
*/
typedef struct _BUFFER {
    int		Used;
    int		Size;
    char	*Data;
} BUFFER;


/*
**  Information about the schema of the news overview files.
*/
typedef struct _ARTOVERFIELD {
    char	*Header;
    int		Length;
    BOOL	HasHeader;
} ARTOVERFIELD;


/*
**  Append an article to an LIST.
*/
#define LISTappend(L, a)	\
	if ((L).Size == (L).Used) {			\
	    (L).Size *= 2;				\
	    RENEW((L).Articles, ARTNUM, (L).Size);	\
	    (L).Articles[(L).Used++] = (a);		\
	}						\
	else						\
	    (L).Articles[(L).Used++] = (a)


/*
**  Global variables.
*/
STATIC char		SPOOL[] = _PATH_SPOOL;
STATIC char		*SCHEMA = _PATH_SCHEMA;
STATIC BOOL		InSpoolDir;
STATIC BOOL		Verbose;
STATIC BOOL		DoNothing;
STATIC ARTOVERFIELD	*ARTfields;
STATIC int		ARTfieldsize;
STATIC BOOL		SearchCancels;


/*
**  Sorting predicate for qsort to put articles in numeric order.
*/
STATIC int
LISTcompare(p1, p2)
    CPOINTER p1;
    CPOINTER p2;
{
    ARTNUM	*ip1;
    ARTNUM	*ip2;

    ip1 = CAST(ARTNUM*, p1);
    ip2 = CAST(ARTNUM*, p2);
    return *ip1 - *ip2;
}


/*
**  If list is big enough, and out of order, sort it.
*/
STATIC void
LISTsort(lp)
    LIST	*lp;
{
    register int	i;
    register ARTNUM	*ap;

    for (ap = lp->Articles, i = lp->Used - 1; --i >= 0; ap++)
	if (ap[0] >= ap[1]) {
	    qsort((POINTER)lp->Articles, (SIZE_T)lp->Used,
		sizeof lp->Articles[0], LISTcompare);
	    break;
	}
}


/*
**  Unlock the group.
*/
STATIC void
UnlockGroup(lfd, lockfile)
    int		lfd;
    char	*lockfile;
{
    if (lfd > 0) {
	if (unlink(lockfile) < 0 && errno != ENOENT)
	    (void)fprintf(stderr, "expireover cant unlink %s %s\n",
		    lockfile, strerror(errno));
	if (close(lfd) < 0)
	    (void)fprintf(stderr, "expireover cant close %s %s\n",
		    lockfile, strerror(errno));
	lfd = -1;
    }
}


/*
**  Sorting predicate to put lines in numeric order.
*/
STATIC int
LINEcompare(p1, p2)
    CPOINTER p1;
    CPOINTER p2;
{
    LINE	*lp1;
    LINE	*lp2;

    lp1 = CAST(LINE*, p1);
    lp2 = CAST(LINE*, p2);
    return lp1->Article - lp2->Article;
}


/*
**  Take in a sorted list of count article numbers in group, and delete
**  them from the overview file.
*/
STATIC void
RemoveLines(group, Deletes)
    char			*group;
    LIST			*Deletes;
{
    static BUFFER		B;
    static LINE			*Lines;
    static int			LineSize;
    register struct iovec	*vp;
    register LINE		*lp;
    register LINE		*end;
    register char		*p;
    register char		*next;
    register ARTNUM		*ap;
    register int		i;
    struct stat			Sb;
    struct iovec		iov[8];
    char			file[SPOOLNAMEBUFF];
    char			lockfile[SPOOLNAMEBUFF];
    int				fd;
    int				count;
    int				lfd;
    static int			field=0;
    int				iTobi;
    char			*pTobi;
    time_t			theTime;
    ARTCAN			*can;

    if (Verbose) {
	for (ap = Deletes->Articles, i = Deletes->Used; --i >= 0; ap++)
	    (void)printf("- %s/%ld\n", group, *ap);
	if (DoNothing)
	    return;
    }

    if( field==0 )
    {	for (field = 0, iTobi = 0; iTobi < ARTfieldsize; iTobi++)
	{   if (caseEQ(ARTfields[iTobi].Header, "Xcanpos")) {
		field = iTobi + 1;
		break;
	    }
	}
	
	if( field==0 )
	{   fprintf(stderr, "cant find Xcanpos in nov\n");
	    exit (-1);
	}
    }

    /* Lock the group. */
    (void)sprintf(lockfile, "%s/.LCK%s", group, _PATH_OVERVIEW);
    lfd = open(lockfile, O_WRONLY | O_TRUNC | O_CREAT, ARTFILE_MODE);
    if (lfd < 0) {
	if( errno!=ENOENT || Verbose )
	    (void)fprintf(stderr, "Can't open %s, %s\n", lockfile, strerror(errno));
	return;
    }

    /* Open file, lock it. */
    (void)sprintf(file, "%s/%s", group, _PATH_OVERVIEW);
    for ( ; ; ) {
	if ((fd = open(file, O_RDWR)) < 0) {
	    if( errno!=ENOENT || Verbose )
		(void)fprintf(stderr, "Can't open %s, %s\n", file, strerror(errno));
	    UnlockGroup(lfd, lockfile);
	    return;
	}
	if (LockFile(fd, FALSE) >= 0)
	    break;
	/* Wait for lock; close file -- might be unlinked -- and try again. */
	(void)LockFile(fd, TRUE);
	(void)close(fd);
    }

    if (fstat(fd, &Sb) < 0) {
	(void)fprintf(stderr, "Can't open %s, %s\n", file, strerror(errno));
	UnlockGroup(lfd, lockfile);
	(void)close(fd);
	return;
    }
    if (Sb.st_size == 0) {
	/* Empty file; done deleting. */
	UnlockGroup(lfd, lockfile);
	(void)close(fd);
	return;
    }

    /* Read in the whole file. */
    if (B.Size == 0) {
	B.Size = Sb.st_size + 1;
	B.Data = NEW(char, B.Size);
    }
    else if (B.Size <= Sb.st_size) {
	B.Size = Sb.st_size + 1;
	RENEW(B.Data, char, B.Size);
    }
    if (xread(fd, B.Data, Sb.st_size) < 0) {
	(void)fprintf(stderr, "Can't read %s, %s\n", file, strerror(errno));
	UnlockGroup(lfd, lockfile);
	(void)close(fd);
	return;
    }
    B.Data[Sb.st_size] = '\0';

    /* Count lines, get space. */
    for (i = 1, p = B.Data; (p = strchr(p, '\n')) != NULL && *++p; i++)
	continue;
    if (LineSize == 0) {
	LineSize = i;
	Lines = NEW(LINE, LineSize + 1);
    }
    else if (LineSize < i) {
	LineSize = i;
	RENEW(Lines, LINE, LineSize + 1);
    }

    /* Build line array. */
    for (lp = Lines, p = B.Data; ; p = next, lp++) {
	if ((next = strchr(p, '\n')) == NULL)
	    break;
	lp->Start = p;
	lp->Length = ++next - p;
	lp->Article = atol(p);

	pTobi=p;
	for( iTobi=field; --iTobi>=0 && *p; pTobi++ )
	{   if( (pTobi=strchr(pTobi, '\t'))==NULL )
		break;
	}
	if( pTobi==NULL || *pTobi=='\0' )
	{   (void)fprintf(stderr, "corrupt line in %s\n", file);
	    continue;
	}
	
	if( ARTfields[field-1].HasHeader )
	    pTobi+=ARTfields[field-1].Length+2;
	
	lp->Canpos=pTobi;
    }
    qsort((POINTER)Lines, (SIZE_T)(lp - Lines), sizeof lp[0], LINEcompare);

    /* Remove duplicates. */
    for (end = lp - 1, lp = Lines; lp < end; lp++)
	if (lp[0].Article == lp[1].Article)
	    lp->Article = 0;

    time(&theTime);

    /* Scan through lines, collecting clumps and skipping holes. */
    ap = Deletes->Articles;
    count = Deletes->Used;
    iov[0].iov_len = 0;
    for (vp = iov, lp = Lines; lp < end + 1; lp++) {
	/* An already-removed article, or one that should be? */
	if (lp->Article == 0)
	    continue;
	
	/* Skip delete items before the current one. */
	while (count > 0 && *ap < lp->Article) {
	    ap++;
	    count--;
	}

	if (count > 0 && lp->Article == *ap) {
	    while (*ap == lp->Article && count > 0) {
		ap++;
		count--;
	    }
	    continue;
	}

	/* article in an expired can? */

	if( CANdelete(lp->Canpos)<theTime )
	{   if( Verbose )
		fprintf(stderr, "dropping %.30s\n", lp->Canpos);
	
	    continue;
	}

	/* article in can still exists? */

	if( SearchCancels )
	{   if( can=CANopenByNameForRead(lp->Canpos) )
	    {   if( CANartsize(can)<=0 )
		{   if( Verbose )
			fprintf(stderr, "dropping %.30s\n", lp->Canpos);
		
		    continue;
		}
	    }
	}

	/* We're keeping this entry; see if we can add it to any
	 * in-progress iov element. */
	if (vp->iov_len) {
	    if (((char *) vp->iov_base) + vp->iov_len == lp->Start) {
		/* Contiguous. */
		vp->iov_len += lp->Length;
		continue;
	    }

	    /* Doesn't fit -- get a new element. */
	    if (++vp == ENDOF(iov)) {
		if (xwritev(lfd, iov, SIZEOF(iov)) < 0) {
		    (void)fprintf(stderr, "Can't write %s, %s\n",
			    lockfile, strerror(errno));
		    UnlockGroup(lfd, lockfile);
		    (void)close(fd);
		    return;
		}
		vp = iov;
	    }
	}

	/* Start new element. */
	vp->iov_base = lp->Start;
	vp->iov_len = lp->Length;
    }

    /* Write out remaining. */
    if (vp->iov_len)
	vp++;
    if (iov[0].iov_len && xwritev(lfd, iov, vp - iov) < 0) {
	(void)fprintf(stderr, "Can't write %s, %s\n",
		lockfile, strerror(errno));
	UnlockGroup(lfd, lockfile);
	(void)close(fd);
	return;
    }

    if (rename(lockfile, file) < 0)
	(void)fprintf(stderr, "Can't rename %s, %s\n",
		lockfile, strerror(errno));

    /* Don't call UnlockGroup; do it inline. */
    if (close(lfd) < 0)
	(void)fprintf(stderr, "expireover cant close %s %s\n",
		file, strerror(errno));
    if (close(fd) < 0)
	(void)fprintf(stderr, "expireover cant close unlinked %s %s\n",
		file, strerror(errno));
}


/*
**  Read the overview schema.
*/
static void
ARTreadschema()
{
    register FILE		*F;
    register char		*p;
    register ARTOVERFIELD	*fp;
    register int		i;
    char			buff[SMBUF];

    /* Open file, count lines. */
    if ((F = fopen(SCHEMA, "r")) == NULL) {
	(void)fprintf(stderr, "Can't open %s, %s\n", SCHEMA, strerror(errno));
	exit(1);
    }
    for (i = 0; fgets(buff, sizeof buff, F) != NULL; i++)
	continue;
    (void)fseek(F, (OFFSET_T)0, SEEK_SET);
    ARTfields = NEW(ARTOVERFIELD, i + 1);

    /* Parse each field. */
    for (fp = ARTfields; fgets(buff, sizeof buff, F) != NULL; ) {
	/* Ignore blank and comment lines. */
	if ((p = strchr(buff, '\n')) != NULL)
	    *p = '\0';
	if ((p = strchr(buff, COMMENT_CHAR)) != NULL)
	    *p = '\0';
	if (buff[0] == '\0')
	    continue;
	if ((p = strchr(buff, ':')) != NULL) {
	    *p++ = '\0';
	    fp->HasHeader = EQ(p, "full");
	}
	else
	    fp->HasHeader = FALSE;
	fp->Header = COPY(buff);
	fp->Length = strlen(buff);
	fp++;
    }
    ARTfieldsize = fp - ARTfields;
    (void)fclose(F);
}


/*
**  Expire by batch, or line at a time.
*/
STATIC void
Expire(SortedInput, qp)
    BOOL		SortedInput;
    register QIOSTATE	*qp;
{
    static LIST		List;
    register char	*line;
    register char	*p;
    char		group[SPOOLNAMEBUFF];

    if (List.Articles == NULL) {
	List.Size = START_LIST_SIZE;
	List.Articles = NEW(ARTNUM, List.Size);
    }
    List.Used = 0;

    if (SortedInput) {
	for ( ; ; ) {
	    if ((line = QIOread(qp)) == NULL) {
		if (QIOerror(qp)) {
		    (void)fprintf(stderr, "Can't read input %s\n",
			    strerror(errno));
		    break;
		}
		if (QIOtoolong(qp))
		    continue;
		break;
	    }
	    if ((p = strrchr(line, '/')) == NULL)
		continue;
	    *p++ = '\0';
	    if (List.Used == 0) {
		(void)strcpy(group, line);
		List.Used = 0;
	    }
	    else if (!EQ(line, group)) {
		LISTsort(&List);
		RemoveLines(group, &List);
		(void)strcpy(group, line);
		List.Used = 0;
	    }
	    LISTappend(List, atol(p));
	}

	/* Do the last group. */
	if (List.Used) {
	    LISTsort(&List);
	    RemoveLines(group, &List);
	}
    }
    else {
	for (List.Used = 1; ; ) {
	    if ((line = QIOread(qp)) == NULL) {
		if (QIOerror(qp)) {
		    (void)fprintf(stderr, "Can't read input %s\n",
			    strerror(errno));
		    break;
		}
		if (QIOtoolong(qp))
		    continue;
		break;
	    }
	    if ((p = strrchr(line, '/')) == NULL)
		continue;
	    *p++ = '\0';
	    List.Articles[0] = atol(p);
	    RemoveLines(line, &List);
	}
    }

    QIOclose(qp);
}


/*
**  Update using canpos.  delete entries.
*/
STATIC void
SpoolUpdate(Name)
    char		*Name;
{
    register QIOSTATE	*qp;
    register char	*line;
    register char	*p;
    LIST		Over;

    Over.Used=0;
    Over.Size=0;
    Over.Articles=NULL;

    /* Open file. */
    if (EQ(Name, "-"))
	qp = QIOfdopen(STDIN, QIO_BUFFER);
    else if ((qp = QIOopen(Name, QIO_BUFFER)) == NULL) {
	(void)fprintf(stderr, "Can't open %s, %s\n", Name, strerror(errno));
	exit(1);
    }
    
    for ( ; ; ) {
	if ((line = QIOread(qp)) == NULL) {
	    if (QIOtoolong(qp) || QIOerror(qp)) {
		(void)fprintf(stderr,
			"Line too long or error reading %s, %s\n",
			Name, strerror(errno));
		exit(1);
	    }
	    break;
	}

	/* Nip off newsgroup name, and turn it into a directory. */
	for (p = line; *p && !ISWHITE(*p) && *p != '\n'; p++)
	    if (*p == '.')
		*p = '/';
	*p = '\0';

	RemoveLines(line, &Over);
    }

    QIOclose(qp);
    exit(0);
}



/*
**  Print usage message and exit.
*/
STATIC NORETURN
Usage()
{
    (void)fprintf(stderr, "Usage:  expireover [flags] [file...]\n");
    (void)fprintf(stderr, "        no addEntries anymore, sorry\n");
    exit(1);
}


int
main(ac, av)
    int			ac;
    char		*av[];
{
    register int	i;
    QIOSTATE		*qp;
    BOOL		ReadSpool;
    BOOL		SortedInput;
    char		*Dir;
    char		*Name;

    /* Set defaults. */
    Dir = _PATH_OVERVIEWDIR;
    Name = _PATH_ACTIVE;
    ReadSpool = FALSE;
    SortedInput = FALSE;
    (void)umask(NEWSUMASK);

    /* Parse JCL. */
    while ((i = getopt(ac, av, "acD:f:nO:svz")) != EOF)
	switch (i) {
	default:
	    Usage();
	    /* NOTREACHED */
	case 'a':
	    Usage();
	    /* NOTREACHED */
	case 'c' :
	    ReadSpool = TRUE;
	    SearchCancels = TRUE;
	    break;	
	case 'D':
	    Dir = optarg;
	    break;
	case 'f':
	    Name = optarg;
	    break;
	case 'n':
	    DoNothing = TRUE;
	    break;
	case 'O':
	    SCHEMA = optarg;
	    break;
	case 's':
	    ReadSpool = TRUE;
	    break;
	case 'v':
	    Verbose = TRUE;
	    break;
	case 'z':
	    SortedInput = TRUE;
	    break;
	}
    ac -= optind;
    av += optind;
    if ((ReadSpool && ac))
	Usage();

    /* Setup. */
    if (chdir(Dir) < 0) {
	(void)fprintf(stderr, "Cant chdir to %s, %s\n", Dir, strerror(errno));
	exit(1);
    }
    InSpoolDir = EQ(Dir, SPOOL);

    ARTreadschema();

    /* Do work. */
    if (ReadSpool)
	SpoolUpdate(Name);
    if (ac == 0)
	Expire(SortedInput, QIOfdopen(STDIN, QIO_BUFFER));
    else {
	for ( ; *av; av++)
	    if (EQ(*av, "-"))
		Expire(SortedInput, QIOfdopen(STDIN, QIO_BUFFER));
	    else if ((qp = QIOopen(*av, QIO_BUFFER)) == NULL)
		(void)fprintf(stderr, "Can't open %s, %s\n",
			*av, strerror(errno));
	    else
		Expire(SortedInput, qp);
    }

    exit(0);
    /* NOTREACHED */
}

