/*
 * linux_loadagv_mrtg.c : Written 2003 by Christian Vogel,
 * hereby placed in the public domain. Please see README
 * for further information.
 */
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>

#define DEFAULT_LOADAVG "/proc/loadavg"
#define DEFAULT_UPTIME "/proc/uptime"

void usage(char *argv0){
	fprintf(stderr,"Usage: %s [options]\n",argv0);
	fprintf(stderr,"Options:     Description (default):\n");
	fprintf(stderr,"  -n n       use nth float from loadavg file (1)\n");
	fprintf(stderr,"             n=1: 1min, n=2:5min, n=3:15min average\n");
	fprintf(stderr,"  -f file    get loadavg from file (/proc/loadavg)\n");
	fprintf(stderr,"  -u file    get uptime from file (/proc/uptime)\n");
	fprintf(stderr,"  -h         this help\n");
	fprintf(stderr,"\n");
	fprintf(stderr,"Written 2003 by chris@hedonism.cx (Christian Vogel)\n");
	fprintf(stderr,"\n");
	fprintf(stderr,"This program is used as an external data-gatherer\n");
	fprintf(stderr,"for MRTG. It gathers the loadavg of the machine\n");
	fprintf(stderr,"and is currently linux-specific (uses /proc/...)\n");
	fprintf(stderr,"\n");
	fprintf(stderr,"Use is like this (mrtg.cfg):\n");
	fprintf(stderr," Target[load]: `/usr/local/mrtg/bin/linux_loadavg_mrtg`\n");
	fprintf(stderr,"\n");
	fprintf(stderr,"MRTG is (c) by Tobias Oetiker  <oetiker@ee.ethz.ch>\n");
	fprintf(stderr,"http://people.ee.ethz.ch/~oetiker/webtools/mrtg/\n");
	exit(1);
}

/* interpret uptime as number of seconds. Convert to days,hours,mins,secs */
void uptime_to_smhd(float uptime,int *s,int *m,int *h,int *d){
	*d = uptime/(60*60*24);
	uptime -= *d * (60*60*24);
	*h = uptime/(60*60);
	uptime -= *h * (60*60);
	*m = uptime/60;
	uptime -= *m * 60;
	*s = uptime;
}

/* open file *fn, read a float. return 0 on success
 * nfloat means: read the nfloat'th number. Use 3 to
 * read 15 minute loadavg from /proc/loadavg for example */
int float_from_file(char *fn,float *fl,int nfloat){
	FILE *f;
	int ret;

	if(NULL == (f=fopen(fn,"r"))){
		perror(fn);
		return -1;
	}

	do {
		ret = fscanf(f,"%f",fl);
	} while(--nfloat && (1 == ret));

	fclose(f);
	if(1 != ret){
		fprintf(stderr,"Could not parse file %s\n",fn);
		return -1;
	}
	return 0;
}

int main(int argc,char **argv){
	int i;
	int nfl = 1;
	char *fn = DEFAULT_LOADAVG;
	char *ufn = DEFAULT_UPTIME;
	float load,uptime;
	int up_sec,up_min,up_hour,up_day;
	char cbuf[128];

	while(-1 != (i=getopt(argc,argv,"f:u:n:h"))){
		switch(i){
			case 'f':
				fn = optarg;
				break;
			case 'u':
				ufn = optarg;
				break;
			case 'n':
				if(1 != sscanf(optarg,"%d",&nfl)){
					fprintf(stderr,
						"Cannot parse %s as number\n",
						optarg);
					exit(1);
				}
				break;
			case 'h':
				usage(argv[0]);
				break;
		}
	}

	/* read and print loadavg */
	if(float_from_file(fn,&load,nfl))
		exit(1);

	printf("%f\n0\n",load);

	/* read uptime */
	if(float_from_file(ufn,&uptime,1))
		exit(1);

	/* convert to day/hour/min/sec, print it */
	uptime_to_smhd(uptime,&up_sec,&up_min,&up_hour,&up_day);
	printf("Up %d:%02d:%02d:%02d\n",up_day,up_hour,up_min,up_sec);

	/* get and print hostname */
	gethostname(cbuf,sizeof(cbuf)-1);
	cbuf[sizeof(cbuf)-1]='\0';
	printf("%s\n",cbuf);

	exit(0);
}
