/*
 * Copyright (c) 2002-2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_IDSTR(id, "@(#)$Id: t-idb-0.c,v 1.48 2005/10/14 22:48:30 ca Exp $")

#include "sm/io.h"
#include "sm/mta.h"
#include "sm/cstr.h"
#include "sm/cdb.h"
#include "sm/ibdb.h"
#include "sm/qmgr.h"
#include "sm/memops.h"
#include "sm/bhtable.h"
#include "sm/test.h"

#include <stdio.h>

/*
**  Test program to write and read IBDB.
**  Uses wribdb.c to write IBDB.
*/

#define FNAME	"ibd"
#define IBDBSIZE	8192
#define INIT_SEQ	1
static int init_seq = INIT_SEQ;

/*
**  ToDo:
**	- more consistency checks for read data.
**	- cleanup files.
*/

#include "../checks/wribdb.c"

/*
**  TESTIDBR -- read INCEDB records
**
**	Parameters:
**		iter -- number of iterations (transactions)
**		maxs -- maximum size of an INCEDB file
**
**	Returns:
**		usual sm_error code.
*/

static sm_ret_T
testidbr(int iter, int maxs)
{
	int i, j, nrcpts, status;
	ibdbr_ctx_P ibdbrc;
	ibdb_ta_P ta;
	ibdb_rcpt_P rcpt;
	sessta_id_T id;
	id_count_T id_count;
	sm_str_P str;
	sm_ret_T ret;

	ret = ibdbr_open(FNAME, SM_IO_RDONLY, init_seq, maxs, IBDB_OFL_WRITE,
			&ibdbrc);
	if (sm_is_err(ret) && sm_error_value(ret) == ENOENT)
		return SM_SUCCESS;
	SM_TEST(ret == SM_SUCCESS);
	if (ret != SM_SUCCESS)
		return ret;

	str = NULL;
	rcpt = NULL;
	ta = (ibdb_ta_P) sm_zalloc(sizeof(*ta));
	SM_TEST(ta != NULL);
	if (ta == NULL)
		goto end;
	ta->ibt_ta_id = id;
	ta->ibt_mail_pa = sm_str_new(NULL, 256, 256);
	SM_TEST(ta->ibt_mail_pa != NULL);
	if (ta->ibt_mail_pa == NULL)
		goto end;

	rcpt = (ibdb_rcpt_P) sm_zalloc(sizeof(*rcpt));
	SM_TEST(rcpt != NULL);
	if (rcpt == NULL)
		goto end;
	rcpt->ibr_ta_id = id;
	rcpt->ibr_pa = sm_str_new(NULL, 256, 256);
	SM_TEST(rcpt->ibr_pa != NULL);
	if (rcpt->ibr_pa == NULL)
		goto end;

	str = sm_str_new(NULL, 256, 256);
	SM_TEST(str != NULL);
	if (str == NULL)
		goto end;

	id_count = 0;
	for (i = 0; i < iter; i++)
	{
		id_count++;

		nrcpts = (i % 5) + 1;
		for (j = 0; j < nrcpts; j++)
		{
			sm_str_clr(str);
			ret = sm_strprintf(str,
					"rcpt-%d-%d@other.Rcpt", i, j);

			ret = ibdbr_get(ibdbrc, rcpt, ta, &status);
			SM_TEST(ret == RECT_IBDB_RCPT);
			if (sm_is_err(ret))
				goto end;
			SM_TEST(sm_memeq(sm_str_getdata(str),
				sm_str_getdata(rcpt->ibr_pa),
				sm_str_getlen(rcpt->ibr_pa)));
			SM_TEST(rcpt->ibr_idx == (rcpt_idx_T) j);
		}

		ret = ibdbr_get(ibdbrc, rcpt, ta, &status);
		SM_TEST(ret == RECT_IBDB_TA);
		if (sm_is_err(ret))
			break;
		sm_cstr_free(ta->ibt_cdb_id);

		for (j = 0; j < nrcpts; j++)
		{
			sm_str_clr(str);
			ret = sm_strprintf(str,
					"rcpt-%d-%d@other.Rcpt", i, j);

			ret = ibdbr_get(ibdbrc, rcpt, ta, &status);
			SM_TEST(ret == RECT_IBDB_RCPT);
			if (sm_is_err(ret))
				goto end;
			SM_TEST(sm_memeq(sm_str_getdata(str),
				sm_str_getdata(rcpt->ibr_pa),
				sm_str_getlen(rcpt->ibr_pa)));
		}

		ret = ibdbr_get(ibdbrc, rcpt, ta, &status);
		SM_TEST(ret == RECT_IBDB_TA);
		if (sm_is_err(ret))
			break;
		sm_cstr_free(ta->ibt_cdb_id);
	}

  end:
	if (ta != NULL)
	{
		SM_STR_FREE(ta->ibt_mail_pa);
		sm_free(ta);
	}
	if (rcpt != NULL)
	{
		SM_STR_FREE(rcpt->ibr_pa);
		sm_free(rcpt);
	}
	SM_STR_FREE(str);
	ret = ibdbr_close(ibdbrc);
	SM_TEST(ret == SM_SUCCESS);
	return ret;
}

static void
usage(const char *prg)
{
	fprintf(stderr, "usage: %s [options]\n", prg);
	fprintf(stderr, "-I	generate incomplete IBDB file\n");
	fprintf(stderr, "-i n	n iterations\n");
	fprintf(stderr, "-m n	maximum size of IBDB file\n");
	fprintf(stderr, "-r	read only\n");
	fprintf(stderr, "-w	write only\n");
	exit(1);
}

int
main(int argc, char *argv[])
{
	int iter, c, maxs;
	sm_ret_T ret;
	bool readonly;
	bool writeonly;
	bool incomplete;

	iter = 100;
	maxs = IBDBSIZE;
	readonly = false;
	writeonly = false;
	incomplete = false;
	while ((c = getopt(argc, argv, "i:Im:Hrw")) != -1)
	{
		switch (c)
		{
#if SM_HEAP_CHECK
		  case 'H':
			SmHeapCheck = atoi(optarg);
			break;
#endif /* SM_HEAP_CHECK */
		  case 'I':
			incomplete = true;
			break;
		  case 'i':
			iter = strtol(optarg, NULL, 0);
			if (iter <= 0)
				return 1;
			break;
		  case 'm':
			maxs = strtol(optarg, NULL, 0);
			if (maxs <= 0)
				return 1;
			break;
		  case 'r':
			readonly = true;
			break;
		  case 'w':
			writeonly = true;
			break;
		  default:
			usage(argv[0]);
			/* NOTREACHED */
			return 1;
		}
	}

	sm_test_begin(argc, argv, "test sm_idb_0");
	if (readonly)
		ret = SM_SUCCESS;
	else
		ret = testidbwr(iter, maxs, incomplete ? WRIBDB_FL_INCOMPL : 0,
				NULL, NULL);
	if (ret == SM_SUCCESS && !incomplete && !writeonly)
		ret = testidbr(iter, maxs);
	return sm_test_end();
}
