/*
 * Copyright (c) 2002 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 *
 *	$Id: iobuf.h,v 1.5 2005/06/17 00:35:26 ca Exp $
 */

#ifndef _SM_IOBUF_H_INCLUDED_
#define _SM_IOBUF_H_INCLUDED_	1

#include "sm/generic.h"
#include "sm/magic.h"

/*
**  The sm_iobuf buffer is defined by
**  1) its structure, by
**  2) the SM_IOBUF_GET() and
**  3) SM_IOBUF_PUT() operations that automatically handle buffer empty and
**  buffer full conditions, and
**  4) by the SM_IOBUF_SPACE() operation that allows
**  the user to reserve buffer space ahead of time, to allow for situations
**  where calling SM_IOBUF_PUT() is not possible or desirable.
**
**  The SM_IOBUF buffer does not specify primitives for memory allocation or
**  deallocation. The purpose is to allow different applications to have
**  different strategies: a memory-resident buffer; a memory-mapped file; or
**  a stdio-like window to an open file. Each application provides its own
**  get(), put() and space() methods that perform the necessary magic.
**
**  This interface is pretty normal. With one exception: the number of bytes
**  left to read is negated. This is done so that we can change direction
**  between reading and writing on the fly.
*/

typedef struct sm_iobuf sm_iobuf_S;
typedef int (*SM_IOBUF_GET_FN)(sm_iobuf_S *);
typedef int (*SM_IOBUF_PUT_FN)(sm_iobuf_S *);
typedef int (*SM_IOBUF_SPACE_FN)(sm_iobuf_S *, int);

struct sm_iobuf
{
	sm_magic_T	 sm_magic;
	int		 sm_iob_flags;	/* status, see below */
	int		 sm_iob_len;	/* buffer length */
	int		 sm_iob_cnt;	/* bytes left to read/write */
	unsigned char	*sm_iob_data;	/* variable-length buffer */
	unsigned char	*sm_iob_ptr;	/* read/write position */
	SM_IOBUF_GET_FN  sm_iob_getfn;	/* read buffer empty action */
	SM_IOBUF_PUT_FN  sm_iob_putfn;	/* write buffer full action */
	SM_IOBUF_SPACE_FN sm_iob_space;	/* request for buffer space */
};

/*
**  Typically, an application will embed a sm_iobuf structure into a larger
**  structure that also contains application-specific members. This approach
**  gives us the best of both worlds. The application can still use the
**  generic sm_iobuf primitives for reading and writing sm_iobufs. The macro below
**  transforms a pointer from sm_iobuf structure to the structure that contains
**  it.
*/

#define SM_IOBUF_TO_APPL(sm_iobuf_ptr, app_type, sm_iobuf_member) \
    ((app_type *) (((char *) (sm_iobuf_ptr)) - offsetof(app_type, sm_iobuf_member)))

/*
**  Buffer status management.
*/

#define SM_IOBUF_FLAG_ERR	0x0001	/* some I/O error */
#define SM_IOBUF_FLAG_EOF	0x0002	/* end of data */
#define SM_IOBUF_FLAG_TIMEOUT	0x0004	/* timeout error */
#define SM_IOBUF_FLAG_BAD	(SM_IOBUF_FLAG_ERR|SM_IOBUF_FLAG_EOF|\
				 SM_IOBUF_FLAG_TIMEOUT)
#define SM_IOBUF_FLAG_FIXED	0x0008	/* fixed-size buffer */

#define sm_iobuf_error(v)	((v)->sm_iob_flags & SM_IOBUF_FLAG_ERR)
#define sm_iobuf_eof(v)		((v)->sm_iob_flags & SM_IOBUF_FLAG_EOF)
#define sm_iobuf_timeout(v)	((v)->sm_iob_flags & SM_IOBUF_FLAG_TIMEOUT)
#define sm_iobuf_clearerr(v)	((v)->sm_iob_flags &= ~SM_IOBUF_FLAG_BAD)

/*
**  Buffer I/O-like operations and results.
*/

#define SM_IOBUF_GET(v)	((v)->sm_iob_cnt < 0 \
			? ++(v)->sm_iob_cnt, (int) *(v)->sm_iob_ptr++ \
			: sm_iobuf_get(v))
#define SM_IOBUF_PUT(v, c)	((v)->sm_iob_cnt > 0 \
			? --(v)->sm_iob_cnt, (int) (*(v)->sm_iob_ptr++ = (c)) \
			: sm_iobuf_put((v),(c)))
#define SM_IOBUF_SPACE(v,n) ((v)->sm_iob_space((v),(n)))

#define SM_IOBUF_EOF	(-1)		/* no more space or data */

int	sm_iobuf_get(sm_iobuf_S *);
int	sm_iobuf_put(sm_iobuf_S *, int);
int	sm_iobuf_unget(sm_iobuf_S *, int);
int	sm_iobuf_read(sm_iobuf_S *, char *, int);
int	sm_iobuf_write(sm_iobuf_S *, const char *, int);

#endif /* _SM_IOBUF_H_INCLUDED_ */
