/*
 * Copyright (c) 2002 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 *
 *	$Id: iobuf.h,v 1.3 2002/02/15 02:12:17 ca Exp $
 */

/*
 *  This code is derived from postfix 1.1.1.
 *
 * LICENSE
 *	The Secure Mailer license must be distributed with this software.
 * AUTHOR(S)
 *	Wietse Venema
 *	IBM T.J. Watson Research
 *	P.O. Box 704
 *	Yorktown Heights, NY 10598, USA
 */

#ifndef _IOBUF_H_INCLUDED_
#define _IOBUF_H_INCLUDED_

#include "sm/generic.h"
#include "sm/magic.h"

/*
**  The iobuf buffer is defined by
**  1) its structure, by
**  2) the IOBUF_GET() and
**  3) IOBUF_PUT() operations that automatically handle buffer empty and
**  buffer full conditions, and
**  4) by the IOBUF_SPACE() operation that allows
**  the user to reserve buffer space ahead of time, to allow for situations
**  where calling IOBUF_PUT() is not possible or desirable.
**
**  The IOBUF buffer does not specify primitives for memory allocation or
**  deallocation. The purpose is to allow different applications to have
**  different strategies: a memory-resident buffer; a memory-mapped file; or
**  a stdio-like window to an open file. Each application provides its own
**  get(), put() and space() methods that perform the necessary magic.
**
**  This interface is pretty normal. With one exception: the number of bytes
**  left to read is negated. This is done so that we can change direction
**  between reading and writing on the fly.
*/

typedef struct iobuf iobuf_S;
typedef int (*IOBUF_GET_FN)(iobuf_S *);
typedef int (*IOBUF_PUT_FN)(iobuf_S *);
typedef int (*IOBUF_SPACE_FN)(iobuf_S *, int);

struct iobuf
{
	sm_magic_T 	 sm_magic;
	int		 iob_flags;	/* status, see below */
	int		 iob_len;	/* buffer length */
	int		 iob_cnt;	/* bytes left to read/write */
	unsigned char	*iob_data;	/* variable-length buffer */
	unsigned char	*iob_ptr;	/* read/write position */
	IOBUF_GET_FN	 iob_getfn;	/* read buffer empty action */
	IOBUF_PUT_FN	 iob_putfn;	/* write buffer full action */
	IOBUF_SPACE_FN	 iob_space;	/* request for buffer space */
};

/*
**  Typically, an application will embed a iobuf structure into a larger
**  structure that also contains application-specific members. This approach
**  gives us the best of both worlds. The application can still use the
**  generic iobuf primitives for reading and writing iobufs. The macro below
**  transforms a pointer from iobuf structure to the structure that contains
**  it.
*/

#define IOBUF_TO_APPL(iobuf_ptr, app_type, iobuf_member) \
    ((app_type *) (((char *) (iobuf_ptr)) - offsetof(app_type, iobuf_member)))

/*
**  Buffer status management.
*/

#define	IOBUF_FLAG_ERR	0x0001	/* some I/O error */
#define IOBUF_FLAG_EOF	0x0002	/* end of data */
#define IOBUF_FLAG_TIMEOUT	0x0004	/* timeout error */
#define IOBUF_FLAG_BAD	(IOBUF_FLAG_ERR|IOBUF_FLAG_EOF|\
				 IOBUF_FLAG_TIMEOUT)
#define IOBUF_FLAG_FIXED	0x0008	/* fixed-size buffer */

#define iobuf_error(v)	((v)->iob_flags & IOBUF_FLAG_ERR)
#define iobuf_eof(v)		((v)->iob_flags & IOBUF_FLAG_EOF)
#define iobuf_timeout(v)	((v)->iob_flags & IOBUF_FLAG_TIMEOUT)
#define iobuf_clearerr(v)	((v)->iob_flags &= ~IOBUF_FLAG_BAD)

/*
**  Buffer I/O-like operations and results.
*/

#define IOBUF_GET(v)	((v)->iob_cnt < 0 \
			? ++(v)->iob_cnt, (int) *(v)->iob_ptr++ \
			: iobuf_get(v))
#define IOBUF_PUT(v, c)	((v)->iob_cnt > 0 \
			? --(v)->iob_cnt, (int) (*(v)->iob_ptr++ = (c)) \
			: iobuf_put((v),(c)))
#define IOBUF_SPACE(v,n) ((v)->iob_space((v),(n)))

#define IOBUF_EOF	(-1)		/* no more space or data */

extern int	iobuf_get(iobuf_S *);
extern int	iobuf_put(iobuf_S *, int);
extern int	iobuf_unget(iobuf_S *, int);
extern int	iobuf_read(iobuf_S *, char *, int);
extern int	iobuf_write(iobuf_S *, const char *, int);

#endif /* _IOBUF_H_INCLUDED_ */
