/*
 * Copyright (c) 2000 QUALCOMM Incorporated. All rights reserved.
 * See License.txt file for terms and conditions for modification and
 * redistribution.
 *
 * Revisions:
 *
 * 08/16/00  [rg]
 *         - Added error string to standard_error.
 *
 * 07/14/00  [rg]
 *         - Errors during spool copies/appends now always abort.
 *
 * 07/05/00  [rg]
 *         - Be more patient when locking spool.
 *
 * 06/10/00  [rg]
 *         - Fixed bug causing spool duplication in server mode when
 *           KEEP_TEMP_DROP defined.
 *         - Added more trace code to update path.
 *         - Removed -Z run-time switch.
 *         - Changed BBSIZE to BUFSIZE because BBSIZE is defined on IRIX.
 *         - AUTO_DELETE now checks if msg already deleted and updates
 *           bytes_deleted; also skips loop if msgs_deleted >= msg_count.
 *         - Deleted unused variables 'uidl_buf' & 'result'.
 *
 * 05/16/00  [rg]
 *         - We now check if we need to refresh the mail lock every
 *           p->check_lock_refresh messages.
 *         - When '-Z' run-time switch used, we revert to old (pre-3.x)
 *           locking.  This is dangerous and will likely be removed prior
 *           to release.
 * 04/21/00  [rg]
 *         - Always use our maillock(), even if system supplies one.
 *         - Rename our maillock() to Qmaillock().
 *
 * 03/09/00  [rg]
 *         - Fix incorrect use of save_errno / save_error.
 *
 * 02/16/00  [rg]
 *         - Now calling touchlock periodically.
 *
 * 12/06/99  [rg]
 *         - Added #include "misc.h" to make sure odds and ends are properly
 *           defined.
 *         - Fixed typo (SYS_MAILOCK instead of SYS_MAILLOCK).
 *
 * 11/22/99  [rg]
 *         - fixed HAVE_MAILLOCK to be HAVE_MAILLOCK_H
 *         - Changed MAILLOCK to be SYS_MAILLOCK
 *         - Removed mailunlock macro; added #include for our own maillock.h,
 *         - Made use of maillock() unconditional.
 *
 */


/*
 * Copyright (c) 1989 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */


#include <sys/types.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>

#include "config.h"

#if HAVE_UNISTD_H
#  include <unistd.h>
#endif

#include <flock.h>

#if HAVE_STRINGS_H
#  include <strings.h>
#endif

#include <sys/stat.h>
#include <sys/file.h>

#include <time.h>

#include "popper.h"
#include "maillock.h"

#if !defined(L_XTND)
#  define L_XTND SEEK_END
#endif

#ifndef L_SET
#  define L_SET SEEK_SET
#endif

#if defined(HAVE_UNISTD_H)
#  include <unistd.h>
#endif

#include "misc.h"

extern int      errno;

#define         BUFSIZE          4096

static char standard_error[] =
    "Error updating primary drop.  Mail left in temporary maildrop [%s (%d)]";


void
unlink_temp_drop ( POP *p, char *filename, const char *fn, size_t ln )
{
#ifdef  KEEP_TEMP_DROP
    DEBUG_LOG3 ( p, "KEEP_TEMP_DROP set [%s:%lu]; temp drop (%s) not unlinked",
                 fn, ln, filename );
#else   /* not KEEP_TEMP_DROP */
    /*
     * Added code in pop_dropcopy.c makes unlink ok now.
     * s-dorner@uiuc.edu, 12/91 
     */
    unlink ( filename );
    DEBUG_LOG3 ( p, "Unlinked [%s:%lu] temp drop (%s)", 
                 fn, (unsigned long) ln, filename );
#endif  /* KEEP_TEMP_DROP */
}


void
truncate_file ( POP *p, int fd, const char *id, const char *fn, size_t ln )
{
    int rslt = 0;


    rslt = ftruncate ( fd, (OFF_T) 0 );
    if ( rslt == 0 ) {
        DEBUG_LOG4 ( p, "Truncated [%s:%lu] %s (%d)", 
                     fn, (unsigned long) ln, id, fd );
    }
    else {
        int e = errno;

        pop_log ( p, POP_PRIORITY, HERE,
                  "Unable to truncate [%s:%lu] %s: %s (%d) ",
                  fn, (long unsigned) ln,
                  id, strerror(e), e );
    }
}





/* 
 *  updt:   Apply changes to a user's POP maildrop
 */

int pop_updt (p)
POP     *   p;
{
    FILE                *   md = NULL;              /*  Stream pointer for 
                                                        the user's maildrop */
    int                     mfd = -1;               /*  File descriptor for
                                                        above */
    char                    buffer[BUFSIZE];        /*  Read buffer */

    MsgInfoList         *   mp = NULL;              /*  Pointer to message 
                                                        info list */
    register int            msg_num;                /*  Current message 
                                                        counter */
    register int            status_written;         /*  Status header field 
                                                        written */
    int                     nchar;                  /* Bytes read/written */

    long                    offset = 0;             /* New mail offset */

    int                     save_errno = 0;         /* Save the error value we
                                                       are trying to print. */
    int                     rslt = 0;               /* temp use when calling */
    struct stat             mybuf;                  /*  For fstat() */
    time_t                  time_locked = 0;
    int                     retrycnt    = 0;        /* for patient locking */

#ifdef DO_TIMING
    time_t                  my_timer = time(0);
#endif /* DO_TIMING */


#ifdef AUTO_DELETE
#  ifdef NOUPDATEONABORT
    /* 
       AUTO_DELETE causes messages which have been RETRd by the client to be
       unconditionally and automatically marked deleted.  This prevents clients
       from leaving mail on the server.  

       CAUTION: Be sure users are informed of this before enabling it, to 
       prevent lost mail.

       Note: NOUPDATEONABORT must also be defined; otherwise users may lose mail
       if they try and RETR it but the connection closes for any reason, the 
       client aborts, etc.
    */
    if ( p->msgs_deleted < p->msg_count )
    { /* there are msgs available for deletion */
        DEBUG_LOG0 ( p, "Auto-deleting messages which have been RETRd..." );
        for ( msg_num = 0; msg_num < p->msg_count; ++msg_num )
        { /* msg loop */
            mp = &p->mlp [ msg_num ];
            if ( mp->retr_flag && !mp->del_flag )
            {
                p->msgs_deleted  ++;
                p->bytes_deleted += mp->length;
                p->dirty          = TRUE;
                mp->del_flag      = TRUE;
                DEBUG_LOG1 ( p, ">>> msg %i automatically flagged as DELETED <<<", 
                             msg_num );
            }
    
        } /* msg loop */
    } /* there are msgs available for deletion */
#  endif /* NOUPDATEONABORT */
#endif /* AUTO_DELETE */

    DEBUG_LOG0 ( p, "Performing maildrop update..." );
    DEBUG_LOG0 ( p, "Checking to see if all messages were deleted" );

    if ( p->stats ) {
        pop_log ( p, POP_PRIORITY, HERE, "Stats: %s %d %ld %d %ld %s %s",
                  p->user, p->msgs_deleted, p->bytes_deleted,
                  p->msg_count - p->msgs_deleted,
                  p->drop_size - p->bytes_deleted,
                  p->client, p->ipaddr );
    }

    if ( p->server_mode ) {
        if ( !p->dirty ) {
            unlink_temp_drop ( p, p->temp_drop, HERE );
            DEBUG_LOG0 ( p, "Running in server mode; spool not "
                            "changed; deleting temp drop" );
#ifdef DO_TIMING
            p->clean_time = time(0) - my_timer;
#endif /* DO_TIMING */

            return ( POP_SUCCESS );
        }
    }
    if ( !p->server_mode && p->msgs_deleted == p->msg_count ) {
        /*
         * In a !server_mode the server process would be the sole user of
         * .user.pop so truncate and unlink is not a problem.
         *
         * Truncate before close, to avoid race condition.  
         */
        truncate_file ( p, fileno ( p->drop ), "temp drop", HERE );
        DEBUG_LOG1 ( p, "Non-server mode and all msgs deleted; "
                        "truncated temp drop (%d)", fileno(p->drop) );
        unlink_temp_drop ( p,  p->temp_drop, HERE );
        fclose ( p->drop );

#ifdef DO_TIMING
        p->clean_time = time(0) - my_timer;
#endif /* DO_TIMING */

        return ( POP_SUCCESS );
    }

    /* 
     * Always use Qmaillock()
     */

    rslt = Qmaillock ( p->user, 4, p->trace, HERE, p->debug );
    if ( rslt != 0 )
        return ( pop_msg ( p, POP_FAILURE, HERE, 
                           "maillock error '%s' (%d) on '%s': %s (%d)", 
                           Qlockerr(rslt), rslt, p->temp_drop, 
                           STRERROR(errno), errno ) );
    time_locked = time(0);
    
    if ( !p->server_mode ) {
        DEBUG_LOG1 ( p, "(Normal mode) opening spool \"%s\"", 
                     p->drop_name );
        /*  
         * Open the user's real maildrop 
         */
        if ( ( mfd = open   ( p->drop_name, O_RDWR | O_CREAT, 0660 ) ) == -1 ||
             ( md  = fdopen ( mfd, "r+" ) ) == NULL ) {
            Qmailunlock ( HERE );
            return pop_msg ( p, POP_FAILURE, HERE, standard_error, 
                             STRERROR(errno), errno );
        }
        
        DEBUG_LOG2 ( p, "(normal mode) mfd=%d: %s", mfd, p->drop_name );
    } 
    else {
        mfd = fileno ( p->drop );
        DEBUG_LOG2 ( p, "(Server mode) accessing spool; mfd=%d: %s", 
                     mfd, p->drop_name );
    }

    /*  
     * Lock the user's real mail drop 
     */
    while ( flock ( mfd, LOCK_EX ) == -1 && retrycnt++ < 4 )
        sleep ( retrycnt * 5 );
    if ( retrycnt == 4 ) {
        fclose ( (p->server_mode) ? p->drop : md );
        Qmailunlock ( HERE );
        return pop_msg ( p, POP_FAILURE, HERE, "flock: '%s': %s (%d)", 
                         p->drop_name,
                         STRERROR(errno), errno );
    }

    if ( !p->server_mode )  {
        /* 
         * Go to the right places 
         */
        fseek ( p->drop, 0, SEEK_END ); 
        offset = ftell ( p->drop ); 
        DEBUG_LOG1 ( p, "Drop has %lu octets", offset );
        
        Qtouchlock ( HERE );

        /*  
         * Append any messages that may have arrived during the session 
         * to the temporary maildrop 
         */
        DEBUG_LOG2 ( p, "Copying new mail from spool (%d) to temp drop (%d)",
                     mfd, fileno(p->drop) );

        while ( ( nchar = read ( mfd, buffer, BUFSIZE ) ) > 0 ) {
            if ( nchar != write ( fileno ( p->drop ), buffer, nchar ) ) {
                nchar = -1; /* write failed */
                break ;
            }
        }
        if ( nchar ) {
            save_errno = errno;
            fclose ( md );
            ftruncate ( fileno ( p->drop ), (OFF_T)offset );
            fclose ( p->drop );
            Qmailunlock ( HERE );
            return pop_msg ( p, POP_FAILURE, HERE, 
                             "Error copying new messages: '%s' : %s (%d)",
                             p->drop_name, STRERROR(save_errno), save_errno );
        }

        fflush ( md );
        rewind ( md );
        truncate_file ( p, mfd, "spool", HERE );
        lseek ( mfd, (OFF_T)0, L_SET );
        DEBUG_LOG1 ( p, "Non-server mode; copied any new msgs to temp "
                        "drop; truncated spool (%d)", mfd );

        /* 
         * Synch stdio and the kernel for the POP drop 
         */
        rewind ( p->drop );
        lseek  ( fileno ( p->drop ), (OFF_T)0, L_SET );

        /*  Transfer messages not flagged for deletion from the temporary 
         *  maildrop to the new maildrop 
         */
        DEBUG_LOG2 ( p, "Creating new maildrop \"%s\" from \"%s\"",
                     p->drop_name, p->temp_drop );

    } /* not server mode */
    else { /* SERVER_MODE */
        fstat ( fileno ( p->drop ), &mybuf );
        if ( p->msgs_deleted == p->msg_count &&
             mybuf.st_size   == p->spool_end ) {
            /* 
             * Truncate before close, to avoid race condition.  
             */
            truncate_file ( p, fileno ( p->drop ), "spool", HERE );
            DEBUG_LOG1 ( p, "Server mode; all msgs deleted and spool size "
                            "unchanged; truncated spool (%d)",
                            fileno ( p->drop) );
            unlink_temp_drop ( p,  p->temp_drop, HERE );
            flock  ( mfd, LOCK_UN );
            fclose ( p->drop );
            fclose ( p->hold );
            Qmailunlock ( HERE );

#ifdef DO_TIMING
            p->clean_time = time(0) - my_timer;
#endif /* DO_TIMING */

            return ( POP_SUCCESS );
        } /* all msgs deleted; no new ones arrived */

        md  = p->hold;   /* Really the temp drop */
        mfd = fileno ( md );
        DEBUG_LOG1 ( p, "server mode: set md to p->hold; "
                        "mfd = temp drop (%d)", mfd );
    } /* server mode */

    if ( !p->server_mode || ( p->msgs_deleted != p->msg_count ) ) {
        DEBUG_LOG5 ( p, "Server mode=%i; %i out of %i msgs deleted; "
                        "copying msgs from temp drop (%d) to spool (%d)", 
                     p->server_mode, p->msgs_deleted, p->msg_count,
                     fileno(p->drop), fileno(md) );

        for ( msg_num = 0; msg_num < p->msg_count; ++msg_num ) {

            int inheader   = 1;
            int body_lines = 0;

            /*  
             * Get a pointer to the message information list 
             */
            mp = &p->mlp [ msg_num ];
            
            /*
             * It is very important to refresh the mail lock on an
             * interval between one and three minutes.  To avoid
             * calling time(0) too often, we check every x msgs.
             */
            if ( msg_num % p->check_lock_refresh == 0 &&
                 time(0) - time_locked >= LOCK_REFRESH_INTERVAL ) {
                Qtouchlock ( HERE );
                time_locked = time(0);
            }

            if ( mp->del_flag ) {
                DEBUG_LOG1 ( p, "Message %d flagged for deletion.", mp->number );
                continue;
            }

            fseek ( p->drop, mp->offset, SEEK_SET );

            DEBUG_LOG1 ( p, "Copying message %d.", mp->number );

            /* 
             * Put the From line separator 
             */
            fgets ( buffer, MAXMSGLINELEN, p->drop );
            if ( fputs ( buffer, md ) == EOF )
                break;

            sprintf ( buffer, "X-UIDL: %s", mp->uidl_str );
            if ( fputs ( buffer, md ) == EOF )
                break;

            for ( status_written = 0, inheader = 1;
                  fgets ( buffer, MAXMSGLINELEN, p->drop );
                ) {

                if ( inheader ) { /* Header */

                    /*  
                     * A blank line signals the end of the header. 
                     */
                    if ( *buffer == '\n' ) {
#ifndef NO_STATUS
                        if ( status_written == 0 ) {
                            if ( mp->retr_flag ) {
                                sprintf ( buffer, "Status: RO\n\n" );
                            } else {
                                sprintf ( buffer, "Status: U\n\n" );
                            }
                        }
#endif /* NO_STATUS */

                        inheader       = 0;
                        body_lines     = 1;
                        status_written = 0;

                    } else if ( !strncasecmp ( buffer, "X-UIDL:", 7 ) ) {
                        continue;       /* Skip all existing UIDL lines */

                    } else if ( !strncasecmp ( buffer, "Status:", 7 ) ) {

                        /*  
                         * Update the message status 
                         */
                        if ( mp->retr_flag )
                            sprintf ( buffer, "Status: RO\n" );
                        status_written++;
                    }
                    /*  
                     * Save another header line 
                     */
                    if ( fputs ( buffer, md ) == EOF )
                        break;

                    } else { /* Body */ 
                        if ( ++body_lines > mp->body_lines )
                            break;
                        if ( fputs ( buffer, md ) == EOF )
                            break;
                    }
            }

            if ( ferror ( md ) ) {
                break;
            }

            if ( p->mmdf_separator ) {
                fputs ( p->mmdf_separator, md );
            }
        }

    /* flush and check for errors now!  The new mail will be writen
     * without stdio,  since we need not separate messages 
     */

        if ( ferror ( md ) ) {
            save_errno = errno;

            truncate_file ( p, mfd, "spool", HERE );
            fclose ( md );
            Qmailunlock ( HERE );
            fclose ( p->drop );
            if ( save_errno == EDQUOT )
                return pop_msg ( p, POP_FAILURE, HERE,
                                 "Overquota copying messages to Mailspool."
                                 " Temp drop unchanged (%d)",
                                 save_errno );
            else
                return pop_msg ( p, POP_FAILURE, HERE, 
                                 standard_error, STRERROR(save_errno),
                                 save_errno );
        }

        fflush ( md );
        if ( ferror ( md ) ) {
            int save_error = errno;

            truncate_file ( p, mfd, "spool", HERE );
            close ( mfd );
            Qmailunlock ( HERE );
            fclose ( p->drop );

            if ( save_error == EDQUOT )
                return pop_msg ( p, POP_FAILURE, HERE,
                                 "Overquota copying messages to Mailspool. "
                                 "Temp drop unchanged (%d)",
                                 save_error );
            else
                return pop_msg ( p, POP_FAILURE, HERE, 
                                 standard_error, STRERROR ( save_error ),
                                 save_error );
        }

    } /* p->msgs_deleted != p->msg_count */

    if ( p->server_mode ) {
        DEBUG_LOG2 ( p, "(server mode; p->drop (%d) is really spool; "
                        "mfd (%d) is temp drop)",
                     fileno(p->drop), mfd );

        if ( mybuf.st_size > p->spool_end ) {
            /* 
             * Go to the right places 
             */
            lseek ( fileno ( p->drop ), (OFF_T)p->spool_end, L_SET ); 
            
            DEBUG_LOG1 ( p, "%lu octets were added to spool during session",
                         mybuf.st_size - p->spool_end );

            Qtouchlock ( HERE );
            
            /*  
             * Append any messages that may have arrived during the session 
             * to the temporary maildrop 
             */
            while ( ( nchar = read ( fileno ( p->drop ), buffer, BUFSIZE ) ) > 0 )
                if ( nchar != write ( mfd, buffer, nchar ) ) {
                    nchar = -1;
                    break ;
                }
                
            DEBUG_LOG3 ( p, "%scopying new msgs from spool (%d) to "
                            "temp drop (%d)", 
                            nchar == 0 ? "" : "ERROR ",
                            fileno(p->drop), mfd );

            if ( nchar != 0 ) {
                truncate_file ( p, mfd, "temp drop", HERE );
                close ( mfd );
                Qmailunlock ( HERE );

                if ( errno == EDQUOT ) {
                    return pop_msg ( p, POP_FAILURE, HERE,
                                     "Overquota appending messages from "
                                     "mailspool to temporary drop (%d)",
                                     errno );
                } else
                    return pop_msg ( p, POP_FAILURE, HERE,
                                     "Error appending messages from mailspool "
                                     "to temporary drop: %s (%d)",
                                      STRERROR(errno), errno );
            }
        } /* mybuf.st_size > p->spool_end */

        rewind ( p->drop );
        truncate_file ( p, fileno(p->drop ), "spool", HERE );
        lseek ( fileno ( p->drop ), (OFF_T)0, L_SET );

        lseek ( mfd, (OFF_T)0, L_SET );

        Qtouchlock ( HERE );

        while ( ( nchar = read ( mfd, buffer, BUFSIZE ) ) > 0 )
            if ( nchar != write ( fileno ( p->drop ), buffer, nchar ) ) {
                nchar = -1;
                break ;
            }

        DEBUG_LOG3 ( p, "%scopying temp drop (%d) to spool (%d)",
                     nchar == 0 ? "" : "ERROR ",
                     mfd, fileno(p->drop) );

        if ( nchar != 0 ) {
            truncate_file ( p, fileno(p->drop), "spool", HERE );
            fclose ( p->drop );
            Qmailunlock ( HERE );
            fclose ( md );
            if ( errno == EDQUOT ) {
                return pop_msg ( p, POP_FAILURE, HERE,
                                 "Overquota copying messages back to mailspool (%d)", 
                                 errno );
            } else
                return pop_msg ( p, POP_FAILURE, HERE,
                                 "Error appending messages from temporary drop "
                                 "to mailspool: %s (%d)",
                                 STRERROR(errno), errno );
        } /* failed to write */

        truncate_file ( p, mfd, "temp drop", HERE );
        unlink_temp_drop ( p, p->temp_drop, HERE );
        fclose ( md );
        fclose ( p->drop );
        Qmailunlock ( HERE );
    } /* server mode */
    else {
        /* 
         * Go to start of new mail if any 
         */
        lseek ( fileno(p->drop), (OFF_T)offset, L_SET );

        Qtouchlock ( HERE );
        
        /* 
         * Copy over any new mail that arrived while processing the pop drop 
         */
        while ( (nchar = read ( fileno(p->drop), buffer, BUFSIZE ) ) > 0 )
            if ( nchar != write ( mfd, buffer, nchar ) ) {
                nchar = -1;
                break ;
            }
        if ( nchar != 0 ) {
            int save_error = errno;

            ftruncate ( mfd, (OFF_T)0 );
            fclose ( md );
            fclose ( p->drop );
            Qmailunlock ( HERE );

            if ( save_error == EDQUOT )
                return pop_msg ( p, POP_FAILURE, HERE,
                                 "Overquota copying messages to Mailspool. "
                                 "Temp drop unchanged (%d)",
                                 save_error );
            else
                return pop_msg ( p, POP_FAILURE, HERE, 
                                 standard_error, STRERROR ( save_error ),
                                 save_error );
        }

        /*  
         * Close the maildrop and empty temporary maildrop 
         */
        fclose ( md );
        ftruncate ( fileno(p->drop), (OFF_T)0 );
        DEBUG_LOG1 ( p, "truncated temp drop (%d)", fileno(p->drop) );
        unlink_temp_drop ( p,  p->temp_drop, HERE );
        fclose ( p->drop );
        Qmailunlock ( HERE );

    }

#ifdef DO_TIMING
    p->clean_time = time(0) - my_timer;
#endif /* DO_TIMING */

    return ( pop_quit ( p ) );
}



int pop_restore ( p )
POP *p;
{
  FILE    *md           = NULL;
  int     mfd           = -1;
  char    buffer [ BUFSIZE ];
  int     nchar         = 0;
  long    offset        = 0;
  int     save_errno    = 0;
  int     rslt          = 0;
  int     retrycnt      = 0;

#ifdef DO_TIMING
  time_t  my_timer = time(0);
#endif /* DO_TIMING */

    DEBUG_LOG0 ( p, "Performing maildrop restoration..." ) ;

    if ( p->server_mode ) {
        unlink_temp_drop ( p,  p->temp_drop, HERE );

#ifdef DO_TIMING
        p->clean_time = time(0) - my_timer;
#endif /* DO_TIMING */

        return ( POP_SUCCESS );
    }

    DEBUG_LOG1 ( p, "Opening mail drop \"%s\"", p->drop_name );

    rslt = Qmaillock ( p->user, 4, p->trace, HERE, p->debug );
    if ( rslt != 0 )
        return ( pop_msg ( p, POP_FAILURE, HERE, 
                           "maillock error '%s' (%d) on '%s': %s (%d)", 
                           Qlockerr(rslt), rslt, p->drop_name, 
                           STRERROR(errno), errno ) );

    if ( ( mfd = open   ( p->drop_name, O_RDWR | O_CREAT, 0660 ) ) == -1 ||
         ( md  = fdopen ( mfd, "r+" ) ) == NULL ) {
        Qmailunlock ( HERE );
        return pop_msg ( p, POP_FAILURE, HERE, standard_error, 
                         STRERROR ( errno ), errno );
    }

    while ( flock ( mfd, LOCK_EX ) == -1 && retrycnt++ < 4 )
        sleep ( retrycnt * 5 );
    if ( retrycnt == 4 ) {
        fclose ( md );
        Qmailunlock ( HERE );
        return pop_msg ( p, POP_FAILURE, HERE, "flock: '%s': %s (%d)", 
                         p->drop_name, STRERROR(errno), errno );
    }

    fseek ( p->drop, 0, SEEK_END );
    offset = ftell ( p->drop );

    /* Append any messages that may have arrived during the session 
     * to the temporary mail drop 
     */
    while ( ( nchar = read ( mfd, buffer, BUFSIZE ) ) > 0 )
        if ( nchar != write ( fileno(p->drop), buffer, nchar ) ) {
            nchar = -1;
            break;
        }

    DEBUG_LOG3 ( p, "%scopying new msgs from spool (%d) to temp drop (%d)",
                 nchar == 0 ? "" : "ERROR ",
                 mfd, fileno(p->drop) );

        if ( nchar != 0) {
            save_errno = errno;
            fclose ( md );
            ftruncate ( fileno(p->drop), (OFF_T)offset );
            DEBUG_LOG2 ( p, "ERROR; truncated temp drop (%d) to %lu",
                         fileno(p->drop), offset );
            fclose ( p->drop );
            Qmailunlock ( HERE );

            if ( save_errno == EDQUOT ) {
                return pop_msg ( p, POP_FAILURE, HERE, 
                                 "Overquota: appending messages "
                                 "to temporary drop (%d)",
                                 save_errno );
            }
            else
                return pop_msg ( p, POP_FAILURE, HERE,
                                 "Error appending messages from mailspool "
                                 "to temporary drop: %s (%d)",
                                 STRERROR(save_errno), save_errno );
       }

       fflush ( md );
       rewind ( md );
       truncate_file ( p, mfd, "spool", HERE );
       lseek ( mfd, (OFF_T)0, L_SET );

       rewind ( p->drop );
       lseek ( fileno(p->drop), (OFF_T)0, L_SET );
       
       while ( ( nchar = read ( fileno(p->drop), buffer, BUFSIZE ) ) > 0 )
            if ( nchar != write ( mfd, buffer, nchar ) ) {
                nchar = -1;
                break;
            }

        DEBUG_LOG3 ( p, "%scopying from temp drop (%d) to spool (%d)",
                 nchar == 0 ? "" : "ERROR ",
                 fileno(p->drop), mfd );

       if ( nchar != 0 ) {
           save_errno = errno;
           truncate_file ( p, mfd, "spool", HERE );
           fclose ( md );
           fclose ( p->drop );
           Qmailunlock ( HERE );

           if ( save_errno == EDQUOT )
               return pop_msg ( p, POP_FAILURE, HERE,
                                "Overquota copying messages to Mailspool. "
                                "Temp drop unchanged (%d)",
                                save_errno );
           else
               return pop_msg ( p, POP_FAILURE, HERE, standard_error,
                                STRERROR ( errno ), errno );
       }

       fclose ( md );
       truncate_file ( p, fileno(p->drop), "temp drop", HERE );
       unlink_temp_drop ( p,  p->temp_drop, HERE );
       fclose ( p->drop );
       Qmailunlock ( HERE );

#ifdef DO_TIMING
    p->clean_time = time(0) - my_timer;
#endif /* DO_TIMING */

       return pop_quit ( p );
}








