/*
 * Copyright (c) 2000 QUALCOMM Incorporated.  All rights reserved.
 * The license.txt file specifies the terms for use, 
 * modification, and redistribution.
 *
 * Revisions:
 *
 *     09/09/00  [rcg]
 *             - Now stripping domain name from user name.
 *
 *     07/14/00  [rcg]
 *             - Fixed crash on APOP command with no APOP database.
 *
 *     06/30/00  [rcg]
 *             - Added support for ~/.qpopper-options.
 *
 *     06/16/00  [rcg]
 *              - Now setting server mode on or off based on group
 *                membership.
 *
 *     06/10/00  [rcg]
 *              - Don't use datum.dptr after dbm_close.
 *
 *     06/05/00  [rcg]
 *              - Added drac call.
 *
 *     03/07/00  [rcg]
 *              - Updated authentication OK message to account for hidden
 *                messages.
 *
 *     02/09/00  [rcg]
 *              - Added extended response codes.
 *              - Added log entry if LOG_LOGIN defined.
 */

/*
 * APOP authentication, derived from MH 6.8.3
 */

#include <sys/types.h>
#include "config.h"

#ifdef DO_TIMING
#  include <time.h>
#endif /* DO_TIMING */


#ifdef APOP

#include <sys/stat.h>
#include <stdio.h>

#ifdef GDBM
#  include <gdbm.h>
#else /* not GDBM */
#  include <ndbm.h>
#endif /* GDBM */

#include <fcntl.h>
#include <pwd.h>

#if HAVE_STRINGS_H
#  include <strings.h>
#endif /* HAVE_STRINGS_H */

#if HAVE_SYS_FILE_H
# include <sys/file.h>
#endif /* HAVE_SYS_FILE_H */

#if HAVE_UNISTD_H
#  include <unistd.h>
#endif

#if HAVE_SYS_UNISTD_H
#  include <sys/unistd.h>
#endif

#include <string.h>
#include <flock.h>
#include <stdlib.h>

#include "genpath.h"
#include "popper.h"
#include "snprintf.h"


/*
 * Obscure password so a cleartext search doesn't come up with
 * something interesting.
 *
 */

char *
obscure ( string )
char *string;
{
    unsigned char *cp, *newstr;

    cp = newstr = (unsigned char *)strdup(string);

    while ( *cp )
       *cp++ ^= 0xff;

    return ( (char *)newstr );
}


extern char    *ERRMSG_PW;
extern char    *ERRMSG_ACEXP;
extern char    *ERRMSG_PWEXP;
extern char    *ERRMSG_AUTH;


int 
pop_apop ( p )
POP *p;
{
    register char *cp;
    char    buffer[BUFSIZ];
    register unsigned char *dp;
    unsigned char *ep,
                   digest[16];
    struct passwd *pw;
    int            j = 0;

#ifdef GDBM
    char apop_file[BUFSIZ];
#else /* not GDBM */
    char apop_dir[BUFSIZ];
#endif /* GDBM */

    struct stat  st;
    datum        key,
                 ddatum;
    char        *apop_str = NULL;

#ifdef GDBM
    GDBM_FILE db;
#else /* not GDBM */
    DBM    *db;
#endif /* GDBM */

    MD5_CTX mdContext;
    int f;

#ifdef DO_TIMING
    time_t my_timer = time(0);
#endif /* DO_TIMING */

#ifdef SCRAM_ONLY
    return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                             "[AUTH] You must use SCRAM-MD5 authentication "
                             "to connect to this server" ) );
#endif /* SCRAM_ONLY */

    /* 
     * Downcase user name if requested 
     */
    if ( p->downcase_user )
        downcase_uname ( p, p->pop_parm[1] );

    /* 
     * strip domain name
     */
    trim_domain ( p, p->pop_parm[1] );

    strncpy (p->user, p->pop_parm[1], sizeof(p->user) );
    p->user [ sizeof(p->user)-1 ] = 0;
    strcpy ( p->authid, p->user );  /* userid is also authentication id */

    pop_log ( p, POP_INFO, HERE, "apop \"%s\"", p->user );

#ifdef AUTHFILE
    if ( checkauthfile ( p->user ) != 0 ) {
        DEBUG_LOG2 ( p, "User %.128s not in authfile %.256s",
                     p->user, AUTHFILE );
        return ( pop_msg ( p, POP_FAILURE, HERE,
                           "[AUTH] Permission denied.", p->user ) );
    }
#endif /* AUTHFILE */

#ifdef NONAUTHFILE
    if ( checknonauthfile ( p->user ) != 0 ) {
        DEBUG_LOG2 ( p, "User %.128s in nonauthfile %.256s",
                     p->user, NONAUTHFILE );
        return ( pop_msg ( p, POP_FAILURE, HERE,
                           "[AUTH] Permission denied.", p->user ) );
    }
#endif /* NONAUTHFILE */

    pw = getpwnam ( p->user );
    if ( ( pw == NULL) || (  pw->pw_passwd == NULL )
                       || ( *pw->pw_passwd == '\0' ) ) {
        return ( pop_auth_fail ( p, POP_FAILURE, HERE, ERRMSG_PW, p->user ) );
    }

    if ( pw->pw_uid == 0 )
        return ( pop_auth_fail ( p, POP_FAILURE, HERE, ERRMSG_AUTH, p->user ) );

#ifdef GDBM
    db = gdbm_open ( APOP, 512, GDBM_READER, 0, 0 );
#else
    db = dbm_open  ( APOP, O_RDONLY, 0 );
#endif
    if ( db == NULL )
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[SYS/TEMP] POP authentication DB not "
                                 "available (user %s): %s (%d)",
                                 p->user, strerror(errno), errno ) );

#ifdef GDBM
    strncpy ( apop_file, APOP, sizeof(apop_file) - 1 );
    apop_file [ sizeof(apop_file) -1 ] = '\0';
    j = stat ( apop_file, &st  );
    if ( ( j != -1 )
         && ( ( st.st_mode & 0777 ) != 0600 ) ) {
        gdbm_close ( db );
        DEBUG_LOG4 ( p, "stat returned %d; %s mode is %o (%x)",
                     j, apop_file, st.st_mode, st.st_mode );
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[SYS/PERM] POP authentication DB "
                                 "has wrong mode (%.3o)",
                                 st.st_mode & 0777 ) );
    }
    f = open ( apop_file, O_RDONLY );
    if ( f == -1 ) {
        int e = errno;
        gdbm_close ( db );
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[SYS/TEMP] unable to lock POP "
                                 "authentication DB: %s (%d)",
                                 strerror(e), e ) );
    }
#else  
    strncpy ( apop_dir, APOP, sizeof(apop_dir) - 5 );
#  if defined(BSD44_DBM)
    strcat ( apop_dir, ".db" );
#  else
    strcat ( apop_dir, ".dir" );
#  endif /* BSD44_DBM */
    j = stat ( apop_dir, &st );
    if ( ( j != -1 )
         && ( ( st.st_mode & 0777 ) != 0600 ) ) {
        dbm_close ( db );
        DEBUG_LOG4 ( p, "stat returned %d; %s mode is %o (%x)",
                     j, apop_dir, st.st_mode, st.st_mode );
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[SYS/PERM] POP authentication DB has "
                                 "wrong mode (%.3o)",
                                 st.st_mode & 0777 ) );
    }
    f = open ( apop_dir, O_RDONLY );
    if ( f == -1 ) {
        int e = errno;
        dbm_close ( db );
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[SYS/TEMP] unable to lock POP authentication DB: %s (%d)",
                                 strerror(e), e ) );
    }
#endif 
    if ( flock ( f, LOCK_SH ) == -1 ) {
        int e = errno;
        close ( f );
#ifdef GDBM
        gdbm_close ( db );
#else
        dbm_close  ( db );
#endif
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[SYS/TEMP] unable to lock POP authentication DB: %s (%d)",
                                 strerror(e), e ) );
    }
    key.dsize = strlen  ( key.dptr = p->user ) + 1;
#ifdef GDBM
    ddatum = gdbm_fetch ( db, key );
#else
    ddatum = dbm_fetch  ( db, key );
#endif
    if ( ddatum.dptr == NULL ) {
        close ( f );
#ifdef GDBM
        gdbm_close ( db );
#else
        dbm_close  ( db );
#endif
        DEBUG_LOG1 ( p, "User %.128s not in popauth db", p->user );
        return ( pop_auth_fail ( p, POP_FAILURE, HERE, 
                                 "[AUTH] not authenticated" ) );
    }

    /*
     * (n)dbm invalidates the datum pointer after dbm_close (gdbm
     * returns a copy), so we copy and obscure it before the close.
     */
    apop_str = obscure ( ddatum.dptr );

#ifdef GDBM
    free ( ddatum.dptr );
    gdbm_close ( db );
#else
    dbm_close  ( db );
#endif
    close ( f );

    MD5Init   ( &mdContext );
    MD5Update ( &mdContext, (unsigned char *) p->md5str, strlen(p->md5str) );
    MD5Update ( &mdContext, (unsigned char *) apop_str,  strlen(apop_str)  );
    MD5Final  ( digest, &mdContext );

    cp = buffer;
    for (ep = (dp = digest) + sizeof digest / sizeof digest[0];
             dp < ep;
             cp += 2)
        (void) sprintf (cp, "%02x", *dp++ & 0xff);
    *cp = '\0';

#ifdef TRACE_APOP_KEYS
    {
    DEBUG_LOG2 ( p, "AUTH KEYS : %s to %s", apop_str, buffer );
    }
#endif /* TRACE_APOP_KEYS */

    free ( apop_str );
    apop_str = NULL;

    if ( strcmp(p->pop_parm[2], buffer) ) {
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[AUTH] authentication failure" ) );
    }

    DEBUG_LOG1 ( p, "APOP authentication ok for \"%s\"", p->user );

#ifdef DO_TIMING
    p->login_time = time(0) - my_timer;
#endif /* DO_TIMING */

    /*
     * Check if server mode should be set or reset based on group membership.
     */

#ifdef SERVER_MODE_GROUP_INCL
    if ( check_group ( p, pw, SERVER_MODE_GROUP_INCL ) ) {
        p->server_mode = TRUE;
    DEBUG_LOG2 ( p, "Set server mode for user %s; member of \"%.128s\" group",
                 p->user, SERVER_MODE_GROUP_INCL );
    }
#endif /* SERVER_MODE_GROUP_INCL */

#ifdef SERVER_MODE_GROUP_EXCL
    if ( check_group ( p, pw, SERVER_MODE_GROUP_EXCL ) ) {
        p->server_mode = FALSE;
    DEBUG_LOG2 ( p, "Set server mode OFF for user %s; member of \"%.128s\" group",
                 p->user, SERVER_MODE_GROUP_INCL );
    }
#endif /* SERVER_MODE_GROUP_EXCL */

    /*
     * Process ~/.qpopper-options file, if requested and present.
     */
    if ( p->user_opts ) {
        int         rslt;
        char        buf [ 256 ];
        struct stat stat_buf;

        rslt = Qsnprintf ( buf, sizeof(buf), "%s/.qpopper-options", pw->pw_dir );
        if ( rslt == -1 )
            pop_log ( p, POP_PRIORITY, HERE, 
                      "Unable to build user options file name for user %s",
                      p->user );
        else {
            rslt = stat ( buf, &stat_buf );
            if ( rslt == 0 ) {
                rslt = pop_config ( p, buf, TRUE );
                if ( rslt == POP_FAILURE ) {
                    pop_log ( p, POP_PRIORITY, HERE,
                              "Unable to process user options file for user %s",
                              p->user );
                }
            }
        }
    } /* p->user_opts */


    /*  
     * Make a temporary copy of the user's maildrop 
     *
     * and set the group and user id 
     */
    if ( pop_dropcopy ( p, pw ) != POP_SUCCESS ) 
        return ( POP_FAILURE );

    /*  
     * Initialize the last-message-accessed number 
     */
    p->last_msg = 0;

    p->AuthType  = apop;      /* authentication method is APOP       */
    p->AuthState = apopcmd;   /* authenticated successfully via APOP */
    /*  
     * Authorization completed successfully 
     */
#ifdef LOG_LOGIN
    pop_log ( p, POP_PRIORITY, HERE,
              "(v%s) POP login by user \"%s\" at (%s) %s",
              VERSION, p->user, p->client, p->ipaddr );
#endif /* LOG_LOGIN */

#ifdef DRAC_AUTH
    drac_it ( p );
#endif /* DRAC_AUTH */

    return ( pop_msg ( p, POP_SUCCESS, HERE,
                       "%s has %d visible message%s (%d hidden) in %d octets.",
                        p->user,
                        p->visible_msg_count,
                        p->visible_msg_count == 1 ? "" : "s",
                        (p->msg_count - p->visible_msg_count),
                        p->drop_size ) );
}

#endif /* APOP */

