/*
 * Copyright (c) 1989 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */

/*
 * Copyright (c) 2000 QUALCOMM Incorporated. All rights reserved.
 * See License.txt file for terms and conditions for modification and
 * redistribution.
 *
 * Revisions:
 *  03/10/00 [rg]
 *           - Additional change suggested by Jacques Distler for NEXT systems.
 *  02/02/00 [rg]
 *           - Made changes suggested by Jacques Distler for NEXT systems.
 *  01/14/00 [rg]
 *           - Used new genpath() call to get POP_TMPXMIT path.
 *  12/17/99 [rg]
 *           - Made sure all functions return a value, to appease some 
 *             compilers.
 *  06/24/98 [py]
 *           - Added a deprecated switch (-i) to sendmail to ignore 
 *             dots. (Need to find another method.)
 *  04/04/98 [py]
 *           -Made callstack and xmit_recv.
 *
 */

#include "config.h"

#include <sys/types.h>
#include <stdio.h>
#include <sys/file.h>

#ifdef POPSCO
#  define __SCO_WAIT3__
#  include <fcntl.h>
#endif

#include <sys/wait.h>

#if HAVE_SYS_STAT_H
#  include <sys/stat.h>
#endif

#if HAVE_FCNTL_H
#  include <fcntl.h>
#endif

#include "popper.h"
#include "genpath.h"

/*
 *  xmit:   POP XTND function to receive a message from 
 *          a client and send it in the mail
 */

FILE     *tmp;                       /* File descriptor for temporary file */
char      temp_xmit [ MAXDROPLEN ];  /* Name of the temporary filedrop */


pop_xmit (p)
POP    *  p;
{
    int   tfn;                    

    if ( p->xmitting ) 
        return pop_xmit_recv ( p, p->inp_buffer );

    /*  
     * Create a temporary file into which to copy the user's message 
     */

    if ( genpath ( p->user, 
                   temp_xmit, 
                   sizeof(temp_xmit),
                   GNPH_XMT,
                   (p->debug ? p->trace : NULL),
                   p->debug ) < 0 )
        return ( pop_msg ( p, POP_FAILURE, HERE, 
                           "Unable to get temporary message file name" ) );

    DEBUG_LOG1 ( p, "Creating temporary file for sending mail message \"%s\"",
                 temp_xmit);
    if ( ( ( tfn = mkstemp(temp_xmit) ) == -1 ) ||
         ( ( tmp = fdopen (tfn, "w+") ) == NULL)  ) {    /* failure, bail out */
        return pop_msg ( p, POP_FAILURE, HERE,
                         "Unable to create temporary message file \"%s\": "
                         "%s (%d)",
                         temp_xmit, STRERROR(errno), errno );
    }

    /*  
     * Receive the message 
     */
    DEBUG_LOG1 ( p, "Receiving mail message in \"%s\"", temp_xmit );
    p->xmitting = 1;
    Push ( &(p->InProcess), (FP)pop_xmit );
    return pop_msg ( p, POP_SUCCESS, HERE, "Start sending message." );
}

pop_xmit_recv ( p, buffer )
POP  *p;
char *buffer;
{

    /*  
     * Look for initial period 
     */
    DEBUG_LOG2 ( p, "Receiving (%d): \"%.128s\"", strlen(buffer), buffer );
    if ( *buffer == '.' ) {
        /*  
         * Exit on end of message 
         */
        if ( strcmp(buffer,".\r\n") == 0 ) {
            (void) fclose(tmp);
            p->xmitting = 0;
            (void) Pop( &(p->InProcess) );
            pop_xmit_exec(p);
        }
        /* 
         * sendmail will not remove escaped .. 
         */
        else if ( buffer[1] == '.' )
            (void)fputs ( &buffer[1], tmp );
        else    
            (void)fputs ( buffer, tmp );
    } else    
        (void)fputs ( buffer, tmp );
    return 0;
}


    
pop_xmit_exec ( p )
POP *p;
{

#ifdef NEXT
    int                    *stat;
#else
    int                     stat;
#endif /* NEXT */

    PID_T                   id, pid;
    
    DEBUG_LOG1 ( p, "Forking for \"%s\"",  MAIL_COMMAND );
    DEBUG_LOG1 ( p, "Reading from \"%s\"", temp_xmit );
    pop_log ( p, POP_DEBUG, HERE, "Pop transmit from \"%s\" on \"%s\"", 
              p->user, p->client );
    /*  
     * Send the message 
     */
    switch ( pid = fork() ) {
        case 0:
            /*  
             * Open the log file 
             */
            (void)closelog();
#ifdef SYSLOG42
            (void)openlog ( p->myname, 0 );
#else
            (void)openlog ( p->myname, LOG_PID, POP_FACILITY );
#endif
            pop_log ( p, POP_DEBUG, HERE, "Pop transmit from \"%s\" on \"%s\"", 
                      p->user, p->client );
            (void)fclose ( p->input );
            (void)fclose ( p->output );       
            (void)close(0);   /* Make sure open return stdin.
                               * to be used by MAIL_COMMAND
                               */
            if ( open ( temp_xmit, O_RDONLY, 0 ) < 0 ) {
                DEBUG_LOG3 ( p, "Unable to open \"%s\": %s (%d)",
                             temp_xmit, STRERROR(errno), errno );
                (void)_exit(1);
            }
            (void)execl ( MAIL_COMMAND, "send-mail", "-t", "-i", "-oem", NULLCP );
            (void)_exit(1);
        case -1:
            if ( !p->debug ) (void)unlink ( temp_xmit );
            return pop_msg ( p, POP_FAILURE, HERE, "Unable to execute \"%s\": %s (%d)",
                             MAIL_COMMAND, STRERROR(errno), errno );
        default:

#ifdef NEXT
            while ( ( id = wait(stat) ) >= 0 && id != pid );
#else
            id = waitpid ( pid, &stat, 0 );
#endif
            if (!p->debug) (void)unlink (temp_xmit);

#ifdef NEXT
            if ( !WIFEXITED (*stat) )
#else
            if ( (!WIFEXITED (stat) ) || ( WEXITSTATUS (stat) != 0) )
#endif
                return pop_msg ( p, POP_FAILURE, HERE, "Unable to send message" );

            return pop_msg ( p, POP_SUCCESS, HERE, "Message sent successfully" );
    }
}

pop_xmit_clean(p)
POP *p;
{
    fclose(tmp);
    if (!p->debug) (void)unlink (temp_xmit);
    return 0;
}


