/*
 * Copyright (c) 1989 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */
 
 
/*
 * Copyright (c) 2000 Qualcomm Incorporated.  All rights reserved.
 * The file License.txt specifies the terms for use, modification,
 * and redistribution.
 *
 *
 * Revisions:
 *
 * 03/01/00  [rcg]
 *          -  Now calling msg_ptr to adjust for hidden msgs and check range.
 *
 * 02/10/00  [RG]
 *          -  Don't trace message body unless TRACE_MSG_BODY defined.
 *
 * 01/18/00  [RG]
 *          -  Put octet count in OK response to RETR to appease
 *             certain poorly-written clients.
 *
 * 12/22/98  [RG]
 *  3/20/98  [PY]
 *        
 */

#include <config.h>
#include <stdio.h>
#include <sys/types.h>
#include <string.h>

#ifndef HAVE_BCOPY
#  define bcopy(src,dest,len)   (void) (memcpy(dest,src,len))
#  define bzero(dest,len)       (void) (memset(dest, (char)NULL, len))
#  define bcmp(b1,b2,n)         (void) (memcmp(b1,b2,n))
#endif /* HAVE_BCOPY */

#ifndef HAVE_INDEX
#  define index(s,c)            strchr(s,c)
#  define rindex(s,c)           strrchr(s,c)
#endif /* HAVE_INDEX */

#if HAVE_STRINGS_H
#  include <strings.h>
#endif /* HAVE_STRINGS_H */

#include "popper.h"
#include "mmangle/mime.h"
#include "mmangle/mangle.h"


static int    not_outputting; /* Used for TOP command to indicate we're done 
                                 A value = 0 Output based on lines to send
                                         < 0 Force the output nomatter
                                         = 1 No more lines to send
                              */

/* 
  Expects input one line at a time
  
  This skips some headers we shouldn't send, and inserts others
 */
static struct header_mucker {
  char   uidl_sent;
  void (*out_fn)(void *, char *, long);
  void **out_state;
  char  *uidl_str;
} header_mucker_global;

void *header_mucker_init(
  void (*out_fn)(void *, char *, long),
  void  *out_state,
  char  *uidl_str)
{
  header_mucker_global.out_fn = out_fn;
  header_mucker_global.out_state = out_state;
  header_mucker_global.uidl_sent = 0;
  header_mucker_global.uidl_str = uidl_str;
  return(&header_mucker_global);
}


/* ----------------------------------------------------------------------
   This handles some header mucking we do, namely UIDL and content-length

   Args: header_mucker_state - our state, mainly saved UIDL and output Fns
         input               - input buffer
         len                 - len of input (not used)

   Returns: nothing

   Note that input MUST be a line at a time!
   --- */
static void header_mucker_input(
    void *header_mucker_state,
    char *input, 
    long len)
{
  struct header_mucker *hm_state = (struct header_mucker *)header_mucker_state;
  char uidl_buf[MAXMSGLINELEN];

  if(!strncasecmp(input, "Content-Length:", 15) ||
     !strncasecmp(input, "X-UIDL:", 7)) { /* Skip UIDLs */
    return;
  }

  if(!hm_state->uidl_sent && (*input=='\n' ||!strncasecmp(input,"Status:",7))){
    /* -- output UIDL if we're at end of header and it hasn't been output --*/
    sprintf(uidl_buf, "X-UIDL: %s", hm_state->uidl_str);
    (*(hm_state->out_fn))(hm_state->out_state, uidl_buf, strlen(uidl_buf));
    hm_state->uidl_sent++;
  }
  (*(hm_state->out_fn))(hm_state->out_state, input, len);
}



/* ----------------------------------------------------------------------
   This part of the pipeline does the TOP command and a few other things
  
  Input can be binary and buffered any which way
  
  This implements the top command, and converts UNIX line ends to
  network line ends, and stuffs the "." character.

  The precise line count for TOP is managed here. Once it has been
  reached the flag is set to signal the input to the to stop.
 */
struct topper topper_global;

static void *topper_init(
  int   lines_to_output,
  int   do_dot_escape,
  void *out_state,
  char tp,
  unsigned long bl)
{
  topper_global.lines_to_output = lines_to_output;
  topper_global.out_state       = out_state;
  topper_global.in_header       = 1;
  topper_global.last_char       = '\0';
  topper_global.is_top          = tp;
  topper_global.body_lines      = bl;
  return(&topper_global);
}


static void topper_input(void *topper_state, char *input, long len)
{
    char          *i, *i_end = input + len, *buf;
    struct topper *t_state = (struct topper *)topper_state;
    char          *crlf = "\r\n", dot = '.';

    if(input == NULL) return;
    buf = input;
    for(i = input; i < i_end; t_state->last_char = *i, i++) {
        if(*i == '.' && t_state->last_char == '\n') {
            fwrite(buf, 1, i - buf + 1, (FILE *)t_state->out_state);
            fwrite(&dot, 1, 1, (FILE *)t_state->out_state);
            buf = i+1;
        } else if(*i == '\n') {
            fwrite(buf, 1, i - buf, (FILE *)t_state->out_state);
            fwrite(crlf, 1, 2, (FILE *)t_state->out_state);
            buf = i+1;
            if(!t_state->in_header && t_state->is_top)
                t_state->lines_to_output--;
        } 
    }
    fwrite(buf, 1, i - buf, (FILE *)t_state->out_state);
}
  


/* 
 *  send:   Send the header and a specified number of lines 
 *          from a mail message to a POP client.
 */

pop_send(p)
POP     *   p;
{
    MsgInfoList         *   mp;         /*  Pointer to message info list */
    int                     top_lines; 
    register int            msg_num;
    register int            msg_lines;
    register int            uidl_sent = 0;
    char                    buffer[MAXMSGLINELEN];
    void                   *hm_state, *t_state;
    MimeParsePtr            mimeParse = NULL;
    ManglerStateType        manglerState;
    char                    is_top = 0;  /* Set to 1 if "top" */
    int                     do_mangle; /* Contain the param number which contain
                                          mangle command */
    long                    bufLen = 0;


    /*  
     * Convert the first parameter into an integer, it's the mesage num 
     */
    msg_num = atoi ( p->pop_parm [ 1 ] );

    /*  
     * Get a pointer to the message in the message list 
     */
    mp = msg_ptr ( p, msg_num );
    if ( mp == NULL )
        return ( pop_msg ( p, POP_FAILURE, HERE, 
                           "Message %d does not exist.", msg_num ) );

    /*  
     * Is the message flagged for deletion? 
     */
    if ( mp->del_flag )
        return ( pop_msg ( p, POP_FAILURE, HERE,
                              "Message %d has been deleted.", msg_num ) );

    DEBUG_LOG2 ( p, "(message %d body contains %d lines)", 
                 msg_num, mp->body_lines );

    /* 
     * Parse arg for top command if it exists
     *
     * Also parse for "x-mangle" / "mangle" extension      
     */
    top_lines = mp->body_lines - 1; /* 'cuz body_lines include newline separating
                                       headers and message body */
    if ( strcmp ( p->pop_command, "top" ) == 0 ) {
        /*  
         * TOP command -- Convert the second parameter into an integer 
         */
        top_lines = atoi ( p->pop_parm[2] );
        if ( p->parm_count > 2 )
            pop_lower ( p->pop_parm[3] );
        do_mangle = ( p->parm_count > 2 && 
                      ( ( !strncmp ( p->pop_parm[3], "mangle",   6 ) ) ||
                        ( !strncmp ( p->pop_parm[3], "x-mangle", 8 ) ) 
                      ) 
                    )
                  ? 3 : 0;
        is_top = 1;
    } /* top */
    else { /* retr */
        /* 
         * NO_STATUS does not dirty the mailspool if a status is changed 
         */
#ifndef NO_STATUS
        /*  
         * Assume that a RETR (retrieve) command was issued 
         */
        if ( mp->retr_flag != TRUE )
            p->dirty = 1;
#endif /* NO_STATUS */

        /*  
         * Flag the message as retreived 
         */
        mp->retr_flag = TRUE;
        if ( p->parm_count > 1 )
            pop_lower ( p->pop_parm[2] );

        do_mangle = ( p->parm_count > 1 && 
                      ( !strncmp ( p->pop_parm[2], "mangle",   6 ) ||
                        !strncmp ( p->pop_parm[2], "x-mangle", 8 )  ) 
                    ) ? 2 : 0;
    } /* retr */


    /* 
     * Set up the output pipeline
     *
     * Last thing is topper, to do top, newline stuff and . escaping 
     */
    t_state = topper_init (
                            top_lines,
                            1,                   /* Stuff \r\n. as \r\n.. */
                            (void *)p->output,   /* Stream to output on */
                            is_top,
                            mp->body_lines - 1
                            );

    /* 
     * Set up the header mucker and possibly the Mime mangler 
     */
    if ( do_mangle ) {
        memset ( (void *)&manglerState, 0, sizeof(ManglerStateType) );
        manglerState.outFnConstructor 
            = manglerState.outFn
            = manglerState.outFnDestructor
            = topper_input;
        manglerState.outFnState        = t_state;
        manglerState.lastWasGoodHeader = 0;
        /* 
         * Initialize the MimeTypeInfo with the parameters supplied 
         */
        if ( FillMangleInfo ( p->pop_parm[do_mangle], 
                              top_lines == 0,
                              &manglerState.rqInfo ) == -1 ) {
            return ( pop_msg ( p, POP_FAILURE, HERE, 
                               "Syntax error in Mangle" ) );
        }
        mimeParse = MimeInit ( MangleMapper, &manglerState, p->drop );
        hm_state = header_mucker_init ( (void(*) (void *, char *, long)) MimeInput,
                                        (void *) mimeParse,
                                        mp->uidl_str );
        DEBUG_LOG0 ( p, "Mangling; initialized header_mucker: MimeInput, mimeParse" );
    } else {
        hm_state = header_mucker_init ( topper_input, t_state, mp->uidl_str );
        DEBUG_LOG0 ( p, "Not mangling; initialized header_mucker: topper_input, t_state" );
    }

    /*  
     * Position to the start of the message 
     */
    (void)fseek ( p->drop, mp->offset, 0 );

    /*  
     * Skip the first line (the sendmail "From" or MMDF line) 
     */
    (void)fgets ( buffer, MAXMSGLINELEN, p->drop );

    /* Some poorly-written clients (reported to include Netscape Messenger)
     * expect the octet count to be in the OK response to RETR, which is of
     * course a clear violation of RFC 1939.  But, in the spirit of bad
     * money driving out good, we accommodate such behavior.
     */
    if ( do_mangle || is_top )
        pop_msg ( p, POP_SUCCESS, HERE, "Message follows" );
    else
        pop_msg ( p, POP_SUCCESS, HERE, "%lu octets", mp->length );
    
    /*  
     * Stuff the message down the pipeline 
     */

    /* 
     * Send the headers 
     */
    while ( fgets ( buffer, MAXMSGLINELEN, p->drop ) ) {
        DEBUG_LOG1 ( p, "HDR *** %s", buffer );
        if ( strcmp(buffer,"\n") == 0 ) {
            header_mucker_input ( hm_state, buffer, strlen(buffer) );
            break;
        }
        header_mucker_input ( hm_state, buffer, strlen(buffer) );
    }

    /* 
     * Send the body 
     */
    ((struct topper *)t_state)->in_header = 0;
    if ( is_top ) {
        while ( fgets( buffer, MAXMSGLINELEN, p->drop )   && 
               ((struct topper*)t_state)->lines_to_output && 
               MSG_LINES-- ) {
            bufLen = strlen(buffer);
#ifdef TRACE_MSG_BODY
            DEBUG_LOG4 ( p, "*** %d(%d)[%li] %.256s", 
                         MSG_LINES, top_lines, bufLen, buffer );
#endif /* TRACE_MSG_BODY */
            header_mucker_input ( hm_state, buffer, bufLen );
        }
    }
    else { /* not TOP */
        while ( fgets ( buffer, MAXMSGLINELEN, p->drop ) && MSG_LINES-- ) {
            bufLen = strlen(buffer);
#ifdef TRACE_MSG_BODY
            DEBUG_LOG3 ( p, "*** %d[%li] %.256s", 
                         MSG_LINES, bufLen, buffer );
#endif /* TRACE_MSG_BODY */
            header_mucker_input ( hm_state, buffer, bufLen );
        }
    }
    if ( do_mangle ){
        FreeMangleInfo ( &manglerState.rqInfo );
        /* 
         * We want to force the MimeFinish to be sent out, if TOP Command 
         */
        MimeFinish ( mimeParse );
    }
    if ( hangup )
        return ( pop_msg ( p, POP_FAILURE, HERE, "SIGHUP or SIGPIPE flagged" ) );

    /* 
     * Make sure message ends with a CRLF 
     */
    if ( topper_global.last_char != '\n' ){
        DEBUG_LOG0 ( p,"*** adding trailing CRLF" );
        (void)fputs ( "\r\n", p->output );
    }

    (void)fputs  ( ".\r\n", p->output );
    (void)fflush ( p->output );
    DEBUG_LOG0   ( p, "---- sent termination sequence ----" );

    return ( POP_SUCCESS );
}





/* This is used outside this file, but not by anything in this file  */

/*
 *  sendline:   Send a line of a multi-line response to a client.
 */
pop_sendline ( p, buffer )
POP         *   p;
char        *   buffer;
{
    char        *   bp;

    /*  
     * Byte stuff lines that begin with the termination octet 
     */
    if ( *buffer == POP_TERMINATE ) 
        (void)fputc ( POP_TERMINATE, p->output );

    /*  
     * Terminate the string at a <NL> if one exists in the buffer
     */
    bp = index ( buffer, NEWLINE );
    if ( bp != NULL ) 
        *bp = 0;

    /*  
     * Send the line to the client 
     */
    (void) fputs ( buffer, p->output );

    /*  
     * Put a <CR><NL> if a newline was removed from the buffer 
     */
    if ( bp != NULL ) 
        (void) fputs ( "\r\n", p->output );
}


