/*
 * Copyright (c) 2000 QUALCOMM Incorporated.  All rights reserved.
 * The license.txt file specifies the terms for use, 
 * modification, and redistribution.
 *
 * Revisions:
 *     03/07/00  [rcg]
 *              - Updated authentication OK message to account for hidden
 *                messages.
 *
 *     02/09/00  [rcg]
 *              - Added extended response codes.
 *              - Added log entry if LOG_LOGIN defined.
 */

/*
 * APOP authentication, derived from MH 6.8.3
 */

#include <sys/types.h>
#include "config.h"


#ifdef APOP

#include <sys/stat.h>
#include <stdio.h>

#ifdef GDBM
#  include <gdbm.h>
#else /* not GDBM */
#  include <ndbm.h>
#endif /* GDBM */

#include <fcntl.h>
#include <pwd.h>

#if HAVE_STRINGS_H
#  include <strings.h>
#endif /* HAVE_STRINGS_H */

#if HAVE_SYS_FILE_H
# include <sys/file.h>
#endif /* HAVE_SYS_FILE_H */

#include <string.h>
#include <flock.h>

#include "genpath.h"
#include "popper.h"


/*
 * Obscure password so a cleartext search doesn't come up with
 * something interesting.
 *
 */

char *
obscure ( string )
char *string;
{
    unsigned char *cp, *newstr;

    cp = newstr = (unsigned char *)strdup(string);

    while ( *cp )
       *cp++ ^= 0xff;

    return ( (char *)newstr );
}


int 
pop_apop ( p )
POP *p;
{
    register char *cp;
    char    buffer[BUFSIZ];
    register unsigned char *dp;
    unsigned char *ep,
                   digest[16];
    struct passwd *pw;

#ifdef GDBM
    char apop_file[BUFSIZ];
#else /* not GDBM */
    char apop_dir[BUFSIZ];
#endif /* GDBM */

    struct stat st;
    datum   key,
            ddatum;

#ifdef GDBM
    GDBM_FILE db;
#else /* not GDBM */
    DBM    *db;
#endif /* GDBM */

    MD5_CTX mdContext;
    int f;

#ifdef SCRAM_ONLY
    return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                             "[AUTH] You must use SCRAM-MD5 authentication "
                             "to connect to this server" ) );
#endif /* SCRAM_ONLY */

    /* 
     * Downcase user name if requested 
     */
    if ( p->downcase_user )
        downcase_uname ( p, p->pop_parm[1] );

    strncpy (p->user, p->pop_parm[1], sizeof(p->user) );
    p->user [ sizeof(p->user)-1 ] = 0;
    strcpy ( p->authid, p->user );  /* userid is also authentication id */

    pop_log ( p, POP_INFO, HERE, "apop \"%s\"", p->user );

#ifdef AUTHFILE
    if ( checkauthfile ( p->user ) != 0 ) {
        DEBUG_LOG2 ( p, "User %.128s not in authfile %.256s",
                     p->user, AUTHFILE );
        return ( pop_msg ( p, POP_FAILURE, HERE,
                           "[AUTH] Permission denied.", p->user ) );
    }
#endif /* AUTHFILE */

#ifdef NONAUTHFILE
    if ( checknonauthfile ( p->user ) != 0 ) {
        DEBUG_LOG2 ( p, "User %.128s in nonauthfile %.256s",
                     p->user, NONAUTHFILE );
        return ( pop_msg ( p, POP_FAILURE, HERE,
                           "[AUTH] Permission denied.", p->user ) );
    }
#endif /* NONAUTHFILE */

    pw = getpwnam ( p->user );
    if ( ( pw == NULL) || (  pw->pw_passwd == NULL )
                       || ( *pw->pw_passwd == '\0' ) ) {
        return ( pop_auth_fail ( p, POP_FAILURE, HERE, pwerrmsg, p->user ) );
    }

    if ( pw->pw_uid == 0 )
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[AUTH] User %s login denied.", p->user ) );

#ifdef GDBM
    db = gdbm_open ( APOP, 512, GDBM_READER, 0, 0 );
#else
    db = dbm_open  ( APOP, O_RDONLY, 0 );
#endif
    if ( db == NULL )
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[SYS/TEMP] POP authentication DB not "
                                 "available (user %s): %s (%s)",
                                 p->user, strerror(errno), errno ) );

#ifdef GDBM
    strncpy ( apop_file, APOP, sizeof(apop_file) - 1 );
    apop_file [ sizeof(apop_file) -1 ] = '\0';
    if ( stat ( apop_file, &st  ) != -1
         && ( st.st_mode & 0777 ) != 0600 ) {
        gdbm_close ( db );
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[SYS/PERM] POP authentication DB "
                                 "has wrong mode (0%o)",
                                 st.st_mode & 0777 ) );
    }
    f = open ( apop_file, O_RDONLY );
    if ( f == -1 ) {
        int e = errno;
        gdbm_close ( db );
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[SYS/TEMP] unable to lock POP "
                                 "authentication DB: %s (%d)",
                                 strerror(e), e ) );
    }
#else  
    strncpy ( apop_dir, APOP, sizeof(apop_dir) - 5 );
#  if defined(BSD44_DBM)
    strcat ( apop_dir, ".db" );
#  else
    strcat ( apop_dir, ".dir" );
#  endif /* BSD44_DBM */
    if ( stat ( apop_dir, &st )   != -1
         && ( st.st_mode & 0777 ) != 0600 ) {
        dbm_close ( db );
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[SYS/PERM] POP authentication DB has wrong mode (0%o)",
                                 st.st_mode & 0777 ) );
    }
    f = open ( apop_dir, O_RDONLY );
    if ( f == -1 ) {
        int e = errno;
        dbm_close ( db );
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[SYS/TEMP] unable to lock POP authentication DB: %s (%d)",
                                 strerror(e), e ) );
    }
#endif 
    if ( flock ( f, LOCK_SH ) == -1 ) {
        int e = errno;
        close ( f );
#ifdef GDBM
        gdbm_close ( db );
#else
        dbm_close  ( db );
#endif
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[SYS/TEMP] unable to lock POP authentication DB: %s (%d)",
                                 strerror(e), e ) );
    }
    key.dsize = strlen  ( key.dptr = p->user ) + 1;
#ifdef GDBM
    ddatum = gdbm_fetch ( db, key );
#else
    ddatum = dbm_fetch  ( db, key );
#endif
    if ( ddatum.dptr == NULL ) {
        close ( f );
#ifdef GDBM
        gdbm_close ( db );
#else
        dbm_close  ( db );
#endif
        DEBUG_LOG1 ( p, "User %.128s not in popauth db", p->user );
        return ( pop_auth_fail ( p, POP_FAILURE, HERE, 
                                 "[AUTH] not authenticated" ) );
    }

#ifdef GDBM
    gdbm_close ( db );
#else
    dbm_close  ( db );
#endif
    close ( f );

    MD5Init ( &mdContext );
    MD5Update ( &mdContext, (unsigned char *)p->md5str, strlen(p->md5str) );
/*    MD5Update(&mdContext, (unsigned char *)obscure(ddatum.dptr), (ddatum.dsize - 1)); */

    MD5Update ( &mdContext, (unsigned char *)obscure(ddatum.dptr), strlen(ddatum.dptr) );
    MD5Final ( digest, &mdContext );

    cp = buffer;
    for (ep = (dp = digest) + sizeof digest / sizeof digest[0];
             dp < ep;
             cp += 2)
        (void) sprintf (cp, "%02x", *dp++ & 0xff);
    *cp = '\0';
    {
    DEBUG_LOG2(p, "AUTH KEYS : %s to %s", ddatum.dptr, buffer);
    }

    if ( strcmp(p->pop_parm[2], buffer) ) {
        return ( pop_auth_fail ( p, POP_FAILURE, HERE,
                                 "[AUTH] authentication failure" ) );
    }

    DEBUG_LOG1 ( p, "APOP authentication ok for \"%s\"", p->user );

    if ( genpath ( p->user, 
                   p->drop_name, 
                   sizeof(p->drop_name), 
                   GNPH_SPOOL,
                   (p->debug ? p->trace : NULL),
                   p->debug ) < 0 )
        return ( pop_msg ( p, POP_FAILURE, HERE, 
                           "[SYS/PERM] Unable to create temporary drop name" ) );

    /*  
     * Make a temporary copy of the user's maildrop 
     *
     * and set the group and user id 
     */
    if ( pop_dropcopy ( p, pw ) != POP_SUCCESS ) 
        return ( POP_FAILURE );

    /*  
     * Initialize the last-message-accessed number 
     */
    p->last_msg = 0;

    p->AuthType  = apop;      /* authentication method is APOP       */
    p->AuthState = apopcmd;   /* authenticated successfully via APOP */
    /*  
     * Authorization completed successfully 
     */
#ifdef LOG_LOGIN
    pop_log ( p, POP_PRIORITY, HERE,
              "(v%s) POP login by user \"%s\" at (%s) %s",
              VERSION, p->user, p->client, p->ipaddr );
#endif /* LOG_LOGIN */

    return ( pop_msg ( p, POP_SUCCESS, HERE,
                       "%s has %d visible message%s (%d hidden) in %d octets.",
                        p->user,
                        p->visible_msg_count,
                        p->visible_msg_count == 1 ? "" : "s",
                        (p->msg_count - p->visible_msg_count),
                        p->drop_size ) );
}

#endif /* APOP */

