/*
 * Copyright (c) 2000 QUALCOMM Incorporated. All rights reserved.
 * See License.txt file for terms and conditions for modification and
 * redistribution.
 *
 * Revisions: 
 *
 * 04/21/00  [rcg]
 *         - Replaced sprintf with Qsprintf.
 *
 * 03/08/00  [rcg]
 *         - Trace calls now dependent on DEBUG as well as bDebugging
 *
 * 01/31/00  [rcg]
 *         - Fix syntax error when HOMEDIRMAIL defined.
 *
 * 01/27/00  [rcg]
 *         - genpath wasn't returning correct path for .pop file
 *           when HOMEDIRMAIL set and --enable-temp-drop-dir used,
 *           or no hashed, no home dir used (latter fix sent in by
 *           Gerard Kok).
 *         - Added HERE parameter to logit() calls.
 *
 * 01/07/00  [rcg]
 *         - Added iWhich parameter to specify path to which file. 
 *
 * 12/06/99  [rcg]
 *         - Trace calls now dependent on bDebugging.
 *
 * 12/02/99  [rcg]
 *         - Trace log calls now dependent on DEBUGGING 
 *
 * 11/24/99  [rcg]
 *         - Modifed to pass in individual parameters instead of POP *.
 *         - Moved into its own file.
 *         - Applied patch contributed by Vince Vielhaber to make use of
 *           HOMEDIRMAIL easier (only need define it in popper.h, no need
 *           to edit this file as well).
 *
 */

#include "config.h"
#include "genpath.h"
#include "logit.h"
#include "popper.h"

#include <sys/types.h>
#include <errno.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <flock.h>

#if HAVE_UNISTD_H
#  include <unistd.h>
#endif

#if HAVE_SYS_UNISTD_H
#  include <sys/unistd.h>
#endif

#if HAVE_STRINGS_H
#  include <strings.h>
#endif

#include <sys/stat.h>
#include <sys/file.h>
#include <pwd.h>

#include "popper.h"



/* Hashing to a spool directory helps reduce the lookup time for sites
 * with thousands of mail spool files.  Unix uses a linear list to
 * save directory information and the following methods attempt to
 * improve the performance.
 *
 * Method 1 - add the value of the first 4 chars mod 26 and open the
 *            spool file in the directory 'a' - 'z'/user.
 *            Brian Buhrow <buhrow@cats.ucsc.edu>
 *
 * Method 2 - Use the first 2 characters to determine which mail spool
 *            to open.  Eg: /usr/spool/u/s/user.
 *            Larry Schwimmer <rosebud@cyclone.stanford.edu>
 *
 * All these methods require that local mail delivery and client programs
 * use the same algorithm.  Only one method to a customer :-)
 */

#if (HASH_SPOOL == 1)

int 
genpath ( char *pszUser, char *pszDrop, int iDropLen, GNPH_WHICH iWhich,
          void *fTrace, int bDebugging )
{
    int seed = 0;
    char dirchar[4];

    if ( !pszUser || *pszUser == '\0' ) {
        logit ( fTrace, POP_PRIORITY, HERE, "Bogus user passed to genpath" );
        return -1; /*bogus login name*/
    }

    /*Now, perform the hash*/

    switch ( strlen(pszUser) ) {
        case 1:
            seed = (pszUser[0]);
            break;
        case 2:
            seed = (pszUser[0] + pszUser[1]);
            break;
        case 3:
            seed = (pszUser[0] + pszUser[1] + pszUser[2]);
            break;
        case 4:
            seed = (pszUser[0] + pszUser[1] + pszUser[2]+ pszUser[3]);
            break;
        default:
            seed = (pszUser[0] + pszUser[1] + pszUser[2]+ 
                    pszUser[3] + pszUser[4]);
            break;
    }
    
    dirchar[0] = '/';
    dirchar[1] = (char)((seed % 26) + 'a');
    dirchar[2] = '/';
    dirchar[3] = '\0';
    
    if ( iWhich == GNPH_POP )
        strncpy ( pszDrop, POP_DROP_DIR, iDropLen );
    else
        strncpy ( pszDrop, POP_MAILDIR, iDropLen );
    strncat ( pszDrop, dirchar, iDropLen - strlen(pszDrop) );
    
    switch ( iWhich ) {
        case GNPH_SPOOL:  /* spool file */
            strncat ( pszDrop, pszUser, iDropLen - strlen(pszDrop) );
            break;
        case GNPH_POP:    /* .pop file  */
            if ( (iDropLen - strlen(pszDrop)) < 
                 (strlen(pszUser) + strlen(POP_DROP)) ) { 
                logit ( fTrace, POP_PRIORITY, HERE,
                        "Insufficient room to generate path to .pop file"
                        " for user %.100s",
                        pszUser );
                return -1;
            }
            Qsprintf ( pszDrop + strlen(pszDrop), POP_DROP, pszUser );
            break;
        case GNPH_TMP:    /* tmpxxxx    */
            strncat ( pszDrop, POP_TMPDROP, iDropLen - strlen(pszDrop) );
            break;
        case GNPH_XMT:    /* xmitxxxx   */
            strncat ( pszDrop, POP_TMPXMIT, iDropLen - strlen(pszDrop) );
            break;
        case GNPH_OLDPOP: /* old .pop file (always in POP_MAILDIR) */
            strncpy ( pszDrop, POP_MAILDIR, iDropLen );
            strncat ( pszDrop, "/",         iDropLen - strlen(pszDrop) );
            if ( (iDropLen - strlen(pszDrop)) < 
                 (strlen(pszUser) + strlen(POP_DROP)) ) { 
                logit ( fTrace, POP_PRIORITY, HERE,
                        "Insufficient room to generate path to .pop file"
                        " for user %.100s",
                        pszUser );
                return -1;
            }
            Qsprintf ( pszDrop + strlen(pszDrop), POP_DROP, pszUser );
            break;
        case GNPH_PATH:  /* Just the path, M'am */
            pszDrop [ strlen(pszDrop) -1 ] = '\0';
            break;
        default:
            logit ( fTrace, POP_PRIORITY, HERE,
                    "Bad iWhich passed to genpath: %d",
                    (int)iWhich );
            return -1;
            break;
    }
    
    if ( DEBUGGING && bDebugging ) {
        logit ( fTrace, POP_DEBUG, HERE,
                "genpath %s (%d) for user %s returning %s", 
                GNPH_DEBUG_NAME(iWhich), (int)iWhich, pszUser, pszDrop );
    }

    return 1;
}

#endif
#if (HASH_SPOOL == 2)

int 
genpath ( char *pszUser, char *pszDrop, int iDropLen, GNPH_WHICH iWhich,
          void *fTrace, int bDebugging )
{
    if ( !pszUser || *pszUser == '\0' )
        return -1;
    
    /* Make sure there is enough room for path of longest item */
    
    if ( (strlen(POP_MAILDIR) + strlen(pszUser) + 17) > iDropLen ) {
        logit ( fTrace, 
                POP_PRIORITY, 
                HERE,
                "Insufficient room to generate path for user %.100s"
                "; need %i; have %i",
                pszUser,
                (strlen(POP_MAILDIR) + strlen(pszUser) + 17),
                iDropLen );
        return -1;
    }
    
    Qsprintf ( pszDrop, "%s/%c/%c/", 
              (iWhich != GNPH_POP ? POP_MAILDIR : POP_DROP_DIR),
              *pszUser, 
              *(pszUser+1) ? *(pszUser+1) : *pszUser );

    switch ( iWhich ) {
        case GNPH_SPOOL:  /* spool file */
            strncat ( pszDrop, pszUser, iDropLen - strlen(pszDrop) );
            break;
        case GNPH_POP:    /* .pop file  */
            Qsprintf ( pszDrop + strlen(pszDrop), POP_DROP, pszUser );
            break;
        case GNPH_TMP:    /* tmpxxxx    */
            strncat ( pszDrop, POP_TMPDROP, iDropLen - strlen(pszDrop) );
            break;
        case GNPH_XMT:    /* xmitxxxx   */
            strncat ( pszDrop, POP_TMPXMIT, iDropLen - strlen(pszDrop)) ;
            break;
        case GNPH_OLDPOP: /* old .pop file (always in POP_MAILDIR) */
            strncpy ( pszDrop, POP_MAILDIR, iDropLen );
            strncat ( pszDrop, "/",         iDropLen - strlen(pszDrop) );
            if ( (iDropLen - strlen(pszDrop)) < 
                 (strlen(pszUser) + strlen(POP_DROP)) ) { 
                logit ( fTrace, POP_PRIORITY, HERE,
                        "Insufficient room to generate path to .pop file"
                        " for user %.100s",
                        pszUser );
                return -1;
            }
            Qsprintf ( pszDrop + strlen(pszDrop), POP_DROP, pszUser );
            break;
        case GNPH_PATH:  /* Just the path, M'am */
            pszDrop [ strlen(pszDrop) -1 ] = '\0';
            break;
        default:
            logit ( fTrace, POP_PRIORITY, HERE,
                    "Bad iWhich passed to genpath: %d",
                    (int)iWhich );
            return -1;
            break;
    }

    if ( DEBUGGING && bDebugging ) {
        logit ( fTrace, POP_DEBUG, HERE,
                "genpath %s (%d) for user %s returning %s", 
                GNPH_DEBUG_NAME(iWhich), (int)iWhich, pszUser, pszDrop );
    }

    return 1;
}

#endif
#if (HASH_SPOOL != 1 && HASH_SPOOL != 2)

int 
genpath ( char *pszUser, char *pszDrop, int iDropLen, GNPH_WHICH iWhich,
          void *fTrace, int bDebugging )
{
    struct passwd *pwp;


#ifdef HOMEDIRMAIL
    if ( ( pwp = getpwnam(pszUser) ) == NULL ) {
        logit ( fTrace, 
                POP_PRIORITY, 
                HERE,
                "Unable to retrieve password entry for user %s", 
                pszUser);
        return -1;
    }
    strncpy ( pszDrop, pwp->pw_dir, iDropLen );
    strncat ( pszDrop, "/",         iDropLen - strlen(pszDrop) );
    
    switch ( iWhich ) {
        case GNPH_SPOOL:  /* spool file */
            strncat ( pszDrop, HOMEDIRMAIL, iDropLen - strlen(pszDrop) );
            break;
        case GNPH_POP:    /* .pop file  */
            if ( strcmp ( POP_DROP_DIR, POP_MAILDIR ) != 0 )
                strncpy ( pszDrop, POP_DROP_DIR, iDropLen );
            if ( (iDropLen - strlen(pszDrop)) < 
                 (strlen(pszUser) + strlen(POP_DROP)) ) { 
                logit ( fTrace, POP_PRIORITY, HERE,
                        "Insufficient room to generate path to .pop file"
                        " for user %.100s",
                        pszUser );
                return -1;
            }
            Qsprintf ( pszDrop + strlen(pszDrop), POP_DROP, pszUser );
            break;
        case GNPH_TMP:    /* tmpxxxx    */
            strncat ( pszDrop, ".", iDropLen - strlen(pszDrop) );
            strncat ( pszDrop, POP_TMPDROP, iDropLen - strlen(pszDrop) );
            break;
        case GNPH_XMT:    /* xmitxxxx   */
            strncat ( pszDrop, ".", iDropLen - strlen(pszDrop) );
            strncat ( pszDrop, POP_TMPXMIT, iDropLen - strlen(pszDrop)) ;
            break;
        case GNPH_OLDPOP: /* old .pop file (always in POP_MAILDIR) */
            strncpy ( pszDrop, POP_MAILDIR, iDropLen );
            strncat ( pszDrop, "/",         iDropLen - strlen(pszDrop) );
            if ( (iDropLen - strlen(pszDrop)) < 
                 (strlen(pszUser) + strlen(POP_DROP)) ) { 
                logit ( fTrace, POP_PRIORITY, HERE,
                        "Insufficient room to generate path to .pop file"
                        " for user %.100s",
                        pszUser );
                return -1;
            }
            Qsprintf ( pszDrop + strlen(pszDrop), POP_DROP, pszUser );
            break;
        case GNPH_PATH:  /* Just the path, M'am */
            pszDrop [ strlen(pszDrop) -1 ] = '\0';
            break;
        default:
            logit ( fTrace, POP_PRIORITY, HERE,
                    "Bad iWhich passed to genpath: %d",
                    (int)iWhich );
            return -1;
            break;
    }
    
#else
    if ( iWhich == GNPH_POP )
        strncpy ( pszDrop, POP_DROP_DIR, iDropLen );
    else
        strncpy ( pszDrop, POP_MAILDIR, iDropLen );
    strncat ( pszDrop, "/", iDropLen - strlen(pszDrop) );
    
    switch ( iWhich ) {
        case GNPH_SPOOL:  /* spool file */
            strncat ( pszDrop, pszUser, iDropLen - strlen(pszDrop) );
            break;
        case GNPH_POP:    /* .pop file  */
        case GNPH_OLDPOP: /* old .pop file (always in POP_MAILDIR) */
            if ( (iDropLen - strlen(pszDrop)) < 
                 (strlen(pszUser) + strlen(POP_DROP)) ) { 
                logit ( fTrace, POP_PRIORITY, HERE,
                        "Insufficient room to generate path to .pop file"
                        " for user %.100s",
                        pszUser );
                return -1;
            }
            Qsprintf ( pszDrop + strlen(pszDrop), POP_DROP, pszUser );
            break;
        case GNPH_TMP:    /* tmpxxxx    */
            strncat ( pszDrop, ".", iDropLen - strlen(pszDrop) );
            strncat ( pszDrop, POP_TMPDROP, iDropLen - strlen(pszDrop) );
            break;
        case GNPH_XMT:    /* xmitxxxx   */
            strncat ( pszDrop, ".", iDropLen - strlen(pszDrop) );
            strncat ( pszDrop, POP_TMPXMIT, iDropLen - strlen(pszDrop)) ;
            break;
        case GNPH_PATH:  /* Just the path, M'am */
            pszDrop [ strlen(pszDrop) -1 ] = '\0';
            break;
        default:
            logit ( fTrace, POP_PRIORITY, HERE,
                    "Bad iWhich passed to genpath: %d",
                    (int)iWhich );
            return -1;
            break;
    }
    
#endif

    if ( DEBUGGING && bDebugging ) {
        logit ( fTrace, POP_DEBUG, HERE,
                "genpath %s (%d) for user %s returning %s", 
                GNPH_DEBUG_NAME(iWhich), (int)iWhich, pszUser, pszDrop );
    }

    return 1;
}

#endif /* HASH_SPOOL */




#ifdef DEBUG
const char *GNPH_DEBUG_NAME ( GNPH_WHICH iWhich )
{
   static char error_buffer[64];
   switch ( iWhich ) {
       case GNPH_SPOOL:
           strcpy ( error_buffer, "Spool" );
           break;
       case GNPH_POP:
           strcpy ( error_buffer, ".pop" );
           break;
       case GNPH_TMP:
           strcpy ( error_buffer, "TMPxxxx" );
           break;
       case GNPH_XMT:
           strcpy ( error_buffer, "XMITxxxx" );
           break;
       case GNPH_OLDPOP:
           strcpy ( error_buffer, "old .pop" );
           break;
       case GNPH_PATH:
           strcpy ( error_buffer, "path only" );
           break;
       default:
           strcpy ( error_buffer, "???" );
   }
    return ( error_buffer );
}
#endif /* DEBUG */

