/*	$NetBSD: rd.c,v 1.2 2003/08/07 16:30:56 agc Exp $ */

/*-
 * Copyright (c) 1996-2003 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Jason R. Thorpe.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the NetBSD
 *	Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Copyright (c) 1982, 1990, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * the Systems Programming Group of the University of Utah Computer
 * Science Department.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * from: Utah $Hdr: rd.c 1.44 92/12/26$
 *
 *	@(#)rd.c	8.2 (Berkeley) 5/19/94
 */

/*
 * Copyright (c) 1988 University of Utah.
 *
 * This code is derived from software contributed to Berkeley by
 * the Systems Programming Group of the University of Utah Computer
 * Science Department.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * from: Utah $Hdr: rd.c 1.44 92/12/26$
 *
 *	@(#)rd.c	8.2 (Berkeley) 5/19/94
 */

/*
 * CS80/SS80 disk driver
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: rd.c,v 1.2 2003/08/07 16:30:56 agc Exp $");

#include "rnd.h"

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <sys/callout.h>
#include <sys/conf.h>
#include <sys/device.h>
#include <sys/disk.h>
#include <sys/disklabel.h>
#include <sys/endian.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <sys/proc.h>
#include <sys/stat.h>

#if NRND > 0
#include <sys/rnd.h>
#endif

#include <dev/gpib/gpibvar.h>
#include <dev/gpib/cs80busvar.h>

#include <dev/gpib/rdreg.h>

#ifdef DEBUG
int	rddebug = 0xff;
#define RDB_FOLLOW	0x01
#define RDB_STATUS	0x02
#define RDB_IDENT	0x04
#define RDB_IO		0x08
#define RDB_ASYNC	0x10
#define RDB_ERROR	0x80
#define DPRINTF(mask, str)	if (rddebug & (mask)) printf str
#else
#define DPRINTF(mask, str)	/* nothing */
#endif

struct	rd_softc {
	struct	device sc_dev;
	gpib_chipset_tag_t sc_ic;
	gpib_handle_t sc_hdl;

	struct	disk sc_dk;

	int	sc_slave;		/* GPIB slave */
	int	sc_punit;		/* physical unit on slave */

	int	sc_flags;
#define	RDF_ALIVE	0x01
#define	RDF_SEEK	0x02
#define RDF_SWAIT	0x04
#define RDF_OPENING	0x08
#define RDF_CLOSING	0x10
#define RDF_WANTED	0x20
#define RDF_WLABEL	0x40

	u_int16_t sc_type;
	u_int8_t *sc_addr;
	int	sc_resid;
	struct	rd_iocmd sc_ioc;
	struct	bufq_state sc_tab;
	int	sc_active;
	int	sc_errcnt;

	struct	callout sc_restart_ch;

#if NRND > 0
	rndsource_element_t rnd_source;
#endif
};

#define RDUNIT(dev)			DISKUNIT(dev)
#define RDPART(dev)			DISKPART(dev)
#define RDMAKEDEV(maj, unit, part)	MAKEDISKDEV(maj, unit, part)
#define RDLABELDEV(dev)	(RDMAKEDEV(major(dev), RDUNIT(dev), RAW_PART))

#define	RDRETRY		5
#define RDWAITC		1	/* min time for timeout in seconds */

int	rderrthresh = RDRETRY-1;	/* when to start reporting errors */

/*
 * Misc. HW description, indexed by sc_type.
 * Used for mapping 256-byte sectors for 512-byte sectors
 */
const struct rdidentinfo {
	u_int16_t ri_hwid;		/* 2 byte HW id */
	u_int16_t ri_maxunum;		/* maximum allowed unit number */
	char	*ri_desc;		/* drive type description */
	int	ri_nbpt;		/* DEV_BSIZE blocks per track */
	int	ri_ntpc;		/* tracks per cylinder */
	int	ri_ncyl;		/* cylinders per unit */
	int	ri_nblocks;		/* DEV_BSIZE blocks on disk */
} rdidentinfo[] = {
	{ RD7946AID,	0,	"7945A",	NRD7945ABPT,
	  NRD7945ATRK,	968,	 108416 },

	{ RD9134DID,	1,	"9134D",	NRD9134DBPT,
	  NRD9134DTRK,	303,	  29088 },

	{ RD9134LID,	1,	"9122S",	NRD9122SBPT,
	  NRD9122STRK,	77,	   1232 },

	{ RD7912PID,	0,	"7912P",	NRD7912PBPT,
	  NRD7912PTRK,	572,	 128128 },

	{ RD7914PID,	0,	"7914P",	NRD7914PBPT,
	  NRD7914PTRK,	1152,	 258048 },

	{ RD7958AID,	0,	"7958A",	NRD7958ABPT,
	  NRD7958ATRK,	1013,	 255276 },

	{ RD7957AID,	0,	"7957A",	NRD7957ABPT,
	  NRD7957ATRK,	1036,	 159544 },

	{ RD7933HID,	0,	"7933H",	NRD7933HBPT,
	  NRD7933HTRK,	1321,	 789958 },

	{ RD9134LID,	1,	"9134L",	NRD9134LBPT,
	  NRD9134LTRK,	973,	  77840 },

	{ RD7936HID,	0,	"7936H",	NRD7936HBPT,
	  NRD7936HTRK,	698,	 600978 },

	{ RD7937HID,	0,	"7937H",	NRD7937HBPT,
	  NRD7937HTRK,	698,	1116102 },

	{ RD7914CTID,	0,	"7914CT",	NRD7914PBPT,
	  NRD7914PTRK,	1152,	 258048 },

	{ RD7946AID,	0,	"7946A",	NRD7945ABPT,
	  NRD7945ATRK,	968,	 108416 },

	{ RD9134LID,	1,	"9122D",	NRD9122SBPT,
	  NRD9122STRK,	77,	   1232 },

	{ RD7957BID,	0,	"7957B",	NRD7957BBPT,
	  NRD7957BTRK,	1269,	 159894 },

	{ RD7958BID,	0,	"7958B",	NRD7958BBPT,
	  NRD7958BTRK,	786,	 297108 },

	{ RD7959BID,	0,	"7959B",	NRD7959BBPT,
	  NRD7959BTRK,	1572,	 594216 },

	{ RD2200AID,	0,	"2200A",	NRD2200ABPT,
	  NRD2200ATRK,	1449,	 654948 },

	{ RD2203AID,	0,	"2203A",	NRD2203ABPT,
	  NRD2203ATRK,	1449,	1309896 }
};
int numrdidentinfo = sizeof(rdidentinfo) / sizeof(rdidentinfo[0]);

int	rdlookup(int, int, int);
int	rdgetinfo(struct rd_softc *);
void	rdrestart(void *);
struct buf *rdfinish(struct rd_softc *, struct buf *);

void	rdgetcompatlabel(struct rd_softc *, struct disklabel *);
void	rdgetdefaultlabel(struct rd_softc *, struct disklabel *);
void	rdrestart(void *);
void	rdustart(struct rd_softc *);
struct buf *rdfinish(struct rd_softc *, struct buf *);
void	rdcallback(void *, int);
void	rdstart(struct rd_softc *);
void	rdintr(struct rd_softc *);
int	rderror(struct rd_softc *);

int	rdmatch(struct device *, struct cfdata *, void *);
void	rdattach(struct device *, struct device *, void *);

CFATTACH_DECL(rd, sizeof(struct rd_softc),
	rdmatch, rdattach, NULL, NULL);


dev_type_open(rdopen);
dev_type_close(rdclose);
dev_type_read(rdread);
dev_type_write(rdwrite);
dev_type_ioctl(rdioctl);
dev_type_strategy(rdstrategy);
dev_type_dump(rddump);
dev_type_size(rdsize);

const struct bdevsw rd_bdevsw = {
	rdopen, rdclose, rdstrategy, rdioctl, rddump, rdsize, D_DISK
};

const struct cdevsw rd_cdevsw = {
	rdopen, rdclose, rdread, rdwrite, rdioctl,
	nostop, notty, nopoll, nommap, nokqfilter, D_DISK
};

extern struct cfdriver rd_cd;

int
rdlookup(id, slave, punit)
	int id;
	int slave;
	int punit;
{
	int i;

	for (i = 0; i < numrdidentinfo; i++) {
		if (rdidentinfo[i].ri_hwid == id)
			break;
	}
	if (i == numrdidentinfo || punit > rdidentinfo[i].ri_maxunum)
		return (-1);
	return (i);
}

int
rdmatch(parent, match, aux)
	struct device *parent;
	struct cfdata *match;
	void *aux;
{
	struct cs80bus_attach_args *ca = aux;

	if (rdlookup(ca->ca_id, ca->ca_slave, ca->ca_punit) < 0)
		return (0);
	return (1);
}

void
rdattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct rd_softc *sc = (struct rd_softc *)self;
	struct cs80bus_attach_args *ca = aux;
	struct cs80_description csd;
	char name[7];
	int type, i, n;

	sc->sc_ic = ca->ca_ic;
	sc->sc_slave = ca->ca_slave;
	sc->sc_punit = ca->ca_punit;

	if ((type = rdlookup(ca->ca_id, ca->ca_slave, ca->ca_punit)) < 0)
		return;

	if (cs80reset(parent, sc->sc_slave, sc->sc_punit)) {
		printf("\n%s: can't reset device\n", sc->sc_dev.dv_xname);
		return;
	}

	if (cs80describe(parent, sc->sc_slave, sc->sc_punit, &csd)) {
		printf("\n%s: didn't respond to describe command\n",
		    sc->sc_dev.dv_xname);
		return;
	}
	memset(name, 0, sizeof(name)); 
	for (i=0, n=0; i<3; i++) {
		name[n++] = (csd.d_name[i] >> 4) + '0';
		name[n++] = (csd.d_name[i] & 0x0f) + '0';
	}

#ifdef DEBUG
	if (rddebug & RDB_IDENT) {
		printf("\n%s: name: ('%s')\n",
		    sc->sc_dev.dv_xname, name);
		printf("  iuw %x, maxxfr %d, ctype %d\n",
		    csd.d_iuw, csd.d_cmaxxfr, csd.d_ctype);
		printf("  utype %d, bps %d, blkbuf %d, burst %d, blktime %d\n",
		    csd.d_utype, csd.d_sectsize,
		    csd.d_blkbuf, csd.d_burstsize, csd.d_blocktime);
		printf("  avxfr %d, ort %d, atp %d, maxint %d, fv %x, rv %x\n",
		    csd.d_uavexfr, csd.d_retry, csd.d_access,
		    csd.d_maxint, csd.d_fvbyte, csd.d_rvbyte);
		printf("  maxcyl/head/sect %d/%d/%d, maxvsect %d, inter %d\n",
		    csd.d_maxcylhead >> 8, csd.d_maxcylhead & 0xff,
		    csd.d_maxsect, csd.d_maxvsectl, csd.d_interleave);
		printf("%s", sc->sc_dev.dv_xname);
	}
#endif

	/*
	 * Take care of a couple of anomolies:
	 * 1. 7945A and 7946A both return same HW id
	 * 2. 9122S and 9134D both return same HW id
	 * 3. 9122D and 9134L both return same HW id
	 */
	switch (ca->ca_id) {
	case RD7946AID:
		if (memcmp(name, "079450", 6) == 0)
			type = RD7945A;
		else
			type = RD7946A;
		break;

	case RD9134LID:
		if (memcmp(name, "091340", 6) == 0)
			type = RD9134L;
		else
			type = RD9122D;
		break;

	case RD9134DID:
		if (memcmp(name, "091220", 6) == 0)
			type = RD9122S;
		else
			type = RD9134D;
		break;
	}

	sc->sc_type = type;

	/*
	 * XXX We use DEV_BSIZE instead of the sector size value pulled
	 * XXX off the driver because all of this code assumes 512 byte
	 * XXX blocks.  ICK!
	 */
	printf(": %s\n", rdidentinfo[type].ri_desc);
	printf("%s: %d cylinders, %d heads, %d blocks, %d bytes/block\n",
	    sc->sc_dev.dv_xname, rdidentinfo[type].ri_ncyl,
	    rdidentinfo[type].ri_ntpc, rdidentinfo[type].ri_nblocks,
	    DEV_BSIZE);

	bufq_alloc(&sc->sc_tab, BUFQ_FCFS);

	/*
	 * Initialize and attach the disk structure.
	 */
	memset(&sc->sc_dk, 0, sizeof(sc->sc_dk));
	sc->sc_dk.dk_name = sc->sc_dev.dv_xname;
	disk_attach(&sc->sc_dk);

	callout_init(&sc->sc_restart_ch);

	if (gpibregister(sc->sc_ic, sc->sc_slave, rdcallback, sc,
	    &sc->sc_hdl)) {
		printf("%s: can't register callback\n", sc->sc_dev.dv_xname);
		return;
	}

	sc->sc_flags = RDF_ALIVE;
#ifdef DEBUG
	/* always report errors */
	if (rddebug & RDB_ERROR)
		rderrthresh = 0;
#endif
#if NRND > 0
	/*
	 * attach the device into the random source list
	 */
	rnd_attach_source(&sc->rnd_source, sc->sc_dev.dv_xname,
			  RND_TYPE_DISK, 0);
#endif
}

/*
 * Read or constuct a disklabel
 */
int
rdgetinfo(sc)
	struct rd_softc *sc;
{
	struct disklabel *lp = sc->sc_dk.dk_label;
	struct partition *pi;
	const char *msg;

	memset(sc->sc_dk.dk_cpulabel, 0, sizeof(struct cpu_disklabel));

	rdgetdefaultlabel(sc, lp);

	/*
	 * Call the generic disklabel extraction routine
	 */
	msg = readdisklabel(RDMAKEDEV(0, sc->sc_dev.dv_unit, RAW_PART),
	    rdstrategy, lp, NULL);
	if (msg == NULL)
		return (0);

	pi = lp->d_partitions;
	printf("%s: WARNING: %s, ", sc->sc_dev.dv_xname, msg);
#ifdef COMPAT_NOLABEL
	printf("using old default partitioning\n");
	rdgetcompatlabel(sc, lp);
#else
	printf("defining '%c' partition as entire disk\n", 'a' + RAW_PART);
	pi[RAW_PART].p_size = rdidentinfo[sc->sc_type].ri_nblocks;
	lp->d_npartitions = RAW_PART+1;
	pi[0].p_size = 0;
#endif
	return (0);
}

int
rdopen(dev, flags, mode, p)
	dev_t dev;
	int flags, mode;
	struct proc *p;
{
	struct rd_softc *sc;
	int error, mask, part;

	sc = device_lookup(&rd_cd, RDUNIT(dev));
	if (sc == NULL || (sc->sc_flags & RDF_ALIVE) ==0)
		return (ENXIO);

	/*
	 * Wait for any pending opens/closes to complete
	 */
	while (sc->sc_flags & (RDF_OPENING | RDF_CLOSING))
		(void) tsleep(sc, PRIBIO, "rdopen", 0);

	/*
	 * On first open, get label and partition info.
	 * We may block reading the label, so be careful
	 * to stop any other opens.
	 */
	if (sc->sc_dk.dk_openmask == 0) {
		sc->sc_flags |= RDF_OPENING;
		error = rdgetinfo(sc);
		sc->sc_flags &= ~RDF_OPENING;
		wakeup((caddr_t)sc);
		if (error)
			return (error);
	}

	part = RDPART(dev);
	mask = 1 << part;

	/* Check that the partition exists. */
	if (part != RAW_PART && (part > sc->sc_dk.dk_label->d_npartitions ||
	    sc->sc_dk.dk_label->d_partitions[part].p_fstype == FS_UNUSED))
		return (ENXIO);

	/* Ensure only one open at a time. */
	switch (mode) {
	case S_IFCHR:
		sc->sc_dk.dk_copenmask |= mask;
		break;
	case S_IFBLK:
		sc->sc_dk.dk_bopenmask |= mask;
		break;
	}
	sc->sc_dk.dk_openmask =
	    sc->sc_dk.dk_copenmask | sc->sc_dk.dk_bopenmask;

	return (0);
}

int
rdclose(dev, flag, mode, p)
	dev_t dev;
	int flag, mode;
	struct proc *p;
{
	struct rd_softc *sc;
	struct disk *dk;
	int mask, s;

	sc = device_lookup(&rd_cd, RDUNIT(dev));
	if (sc == NULL)
		return (ENXIO);

	dk = &sc->sc_dk;

	mask = 1 << RDPART(dev);
	if (mode == S_IFCHR)
		dk->dk_copenmask &= ~mask;
	else
		dk->dk_bopenmask &= ~mask;
	dk->dk_openmask = dk->dk_copenmask | dk->dk_bopenmask;
	/*
	 * On last close, we wait for all activity to cease since
	 * the label/parition info will become invalid.  Since we
	 * might sleep, we must block any opens while we are here.
	 * Note we don't have to about other closes since we know
	 * we are the last one.
	 */
	if (dk->dk_openmask == 0) {
		sc->sc_flags |= RDF_CLOSING;
		s = splbio();
		while (sc->sc_active) {
			sc->sc_flags |= RDF_WANTED;
			(void) tsleep(&sc->sc_tab, PRIBIO, "rdclose", 0);
		}
		splx(s);
		sc->sc_flags &= ~(RDF_CLOSING | RDF_WLABEL);
		wakeup((caddr_t)sc);
	}
	return (0);
}

void
rdstrategy(bp)
	struct buf *bp;
{
	struct rd_softc *sc;
	struct partition *pinfo;
	daddr_t bn;
	int sz, s;
	int offset;

	sc = device_lookup(&rd_cd, RDUNIT(bp->b_dev));

	DPRINTF(RDB_FOLLOW,
	    ("rdstrategy(%p): dev %x, bn %" PRId64 ", bcount %ld, %c\n",
	    bp, bp->b_dev, bp->b_blkno, bp->b_bcount,
	    (bp->b_flags & B_READ) ? 'R' : 'W'));

	bn = bp->b_blkno;
	sz = howmany(bp->b_bcount, DEV_BSIZE);
	pinfo = &sc->sc_dk.dk_label->d_partitions[RDPART(bp->b_dev)];

	/* Don't perform partition translation on RAW_PART. */
	offset = (RDPART(bp->b_dev) == RAW_PART) ? 0 : pinfo->p_offset;

	if (RDPART(bp->b_dev) != RAW_PART) {
		/*
		 * XXX This block of code belongs in
		 * XXX bounds_check_with_label()
		 */

		if (bn < 0 || bn + sz > pinfo->p_size) {
			sz = pinfo->p_size - bn;
			if (sz == 0) {
				bp->b_resid = bp->b_bcount;
				goto done;
			}
			if (sz < 0) {
				bp->b_error = EINVAL;
				goto bad;
			}
			bp->b_bcount = dbtob(sz);
		}
		/*
		 * Check for write to write protected label
		 */
		if (bn + offset <= LABELSECTOR &&
#if LABELSECTOR != 0
		    bn + offset + sz > LABELSECTOR &&
#endif
		    !(bp->b_flags & B_READ) && !(sc->sc_flags & RDF_WLABEL)) {
			bp->b_error = EROFS;
			goto bad;
		}
	}
	bp->b_rawblkno = bn + offset;
	s = splbio();
	BUFQ_PUT(&sc->sc_tab, bp);
	if (sc->sc_active == 0) {
		sc->sc_active = 1;
		rdustart(sc);
	}
	splx(s);
	return;
bad:
	bp->b_flags |= B_ERROR;
done:
	biodone(bp);
}

/*
 * Called from timeout() when handling maintenance releases
 * callout from timeouts
 */
void
rdrestart(arg)
	void *arg;
{
	int s = splbio();
	rdustart((struct rd_softc *)arg);
	splx(s);
}


/* called by rdstrategy() to start a block transfer */
/* called by rdrestart() when handingly timeouts */
/* called by rdintr() */
void
rdustart(sc)
	struct rd_softc *sc;
{
	struct buf *bp;

	bp = BUFQ_PEEK(&sc->sc_tab);
	sc->sc_addr = bp->b_data;
	sc->sc_resid = bp->b_bcount;
	if (gpibrequest(sc->sc_ic, sc->sc_hdl))
		rdstart(sc);
}

struct buf *
rdfinish(sc, bp)
	struct rd_softc *sc;
	struct buf *bp;
{

	sc->sc_errcnt = 0;
	(void)BUFQ_GET(&sc->sc_tab);
	bp->b_resid = 0;
	biodone(bp);
	gpibrelease(sc->sc_ic, sc->sc_hdl);
	if ((bp = BUFQ_PEEK(&sc->sc_tab)) != NULL)
		return (bp);
	sc->sc_active = 0;
	if (sc->sc_flags & RDF_WANTED) {
		sc->sc_flags &= ~RDF_WANTED;
		wakeup((caddr_t)&sc->sc_tab);
	}
	return (NULL);
}

void
rdcallback(v, action)
	void *v;
	int action;
{
	struct rd_softc *sc = v;

	DPRINTF(RDB_FOLLOW, ("rdcallback: v=%p, action=%d\n", v, action));

	switch (action) {
	case GPIBCBF_START:
		rdstart(sc);
		break;
	case GPIBCBF_INTR:
		rdintr(sc);
		break;
#ifdef DEBUG
	default:
		DPRINTF(RDB_ERROR, ("rdcallback: unknown action %d\n",
		    action));
		break;
#endif
	}
}


/* called from rdustart() to start a transfer */
/* called from gpib interface as the initiator */
void
rdstart(sc)
	struct rd_softc *sc;
{
	struct buf *bp = BUFQ_PEEK(&sc->sc_tab);
	int part, slave, punit;

	slave = sc->sc_slave;
	punit = sc->sc_punit;

	DPRINTF(RDB_FOLLOW, ("rdstart(%s): bp %p, %c\n",
	    sc->sc_dev.dv_xname, bp, (bp->b_flags & B_READ) ? 'R' : 'W'));

again:

	part = RDPART(bp->b_dev);
	sc->sc_flags |= RDF_SEEK;
	sc->sc_ioc.c_unit = CS80CMD_SUNIT(punit);
	sc->sc_ioc.c_volume = CS80CMD_SVOL(0);
	sc->sc_ioc.c_saddr = CS80CMD_SADDR;
	sc->sc_ioc.c_hiaddr = htobe16(0);
	sc->sc_ioc.c_addr = htobe32(RDBTOS(bp->b_rawblkno));
	sc->sc_ioc.c_nop2 = CS80CMD_NOP;
	sc->sc_ioc.c_slen = CS80CMD_SLEN;
	sc->sc_ioc.c_len = htobe32(sc->sc_resid);
	sc->sc_ioc.c_cmd = bp->b_flags & B_READ ? CS80CMD_READ : CS80CMD_WRITE;

	if (gpibsend(sc->sc_ic, slave, CS80CMD_SCMD, &sc->sc_ioc.c_unit,
	    sizeof(sc->sc_ioc)-1) == sizeof(sc->sc_ioc)-1) {
		/* Instrumentation. */
		disk_busy(&sc->sc_dk);
		sc->sc_dk.dk_seek++;
		gpibawait(sc->sc_ic);
		return;
	}
	/*
	 * Experience has shown that the gpibwait in this gpibsend will
	 * occasionally timeout.  It appears to occur mostly on old 7914
	 * drives with full maintenance tracks.  We should probably
	 * integrate this with the backoff code in rderror.
	 */

	DPRINTF(RDB_ERROR,
	    ("rdstart: cmd %x adr %ul blk %" PRId64 " len %d ecnt %d\n",
	    sc->sc_ioc.c_cmd, sc->sc_ioc.c_addr, bp->b_blkno, sc->sc_resid,
	     sc->sc_errcnt));

	sc->sc_flags &= ~RDF_SEEK;
	cs80reset(sc->sc_dev.dv_parent, slave, punit);
	if (sc->sc_errcnt++ < RDRETRY)
		goto again;
	printf("%s: rdstart err: cmd 0x%x sect %uld blk %" PRId64 " len %d\n",
	       sc->sc_dev.dv_xname, sc->sc_ioc.c_cmd, sc->sc_ioc.c_addr,
	       bp->b_blkno, sc->sc_resid);
	bp->b_flags |= B_ERROR;
	bp->b_error = EIO;
	bp = rdfinish(sc, bp);
	if (bp) {
		sc->sc_addr = bp->b_data;
		sc->sc_resid = bp->b_bcount;
		if (gpibrequest(sc->sc_ic, sc->sc_hdl))
			goto again;
	}
}

void
rdintr(sc)
	struct rd_softc *sc;
{
	struct buf *bp;
	u_int8_t stat = 13;	/* in case gpibrecv fails */
	int rv, dir, restart, slave;

	slave = sc->sc_slave;
	bp = BUFQ_PEEK(&sc->sc_tab);

	DPRINTF(RDB_FOLLOW, ("rdintr(%s): bp %p, %c, flags %x\n",
	    sc->sc_dev.dv_xname, bp, (bp->b_flags & B_READ) ? 'R' : 'W',
	    sc->sc_flags));

	disk_unbusy(&sc->sc_dk, (bp->b_bcount - bp->b_resid),
		(bp->b_flags & B_READ));

	if (sc->sc_flags & RDF_SEEK) {
		sc->sc_flags &= ~RDF_SEEK;
		dir = (bp->b_flags & B_READ ? GPIB_READ : GPIB_WRITE);
		gpibxfer(sc->sc_ic, slave, CS80CMD_EXEC, sc->sc_addr,
		    sc->sc_resid, dir, dir == GPIB_READ);
		disk_busy(&sc->sc_dk);
		return;
	}
	if ((sc->sc_flags & RDF_SWAIT) == 0) {
		if (gpibpptest(sc->sc_ic, slave) == 0) {
			/* Instrumentation. */
			disk_busy(&sc->sc_dk);
			sc->sc_flags |= RDF_SWAIT;
			gpibawait(sc->sc_ic);
			return;
		}
	} else
		sc->sc_flags &= ~RDF_SWAIT;
	rv = gpibrecv(sc->sc_ic, slave, CS80CMD_QSTAT, &stat, 1);
	if (rv != 1 || stat) {
		DPRINTF(RDB_ERROR,
		    ("rdintr: receive failed (rv=%d) or bad stat %d\n", rv,
		     stat));
		restart = rderror(sc);
		if (sc->sc_errcnt++ < RDRETRY) {
			if (restart)
				rdstart(sc);
			return;
		}
		bp->b_flags |= B_ERROR;
		bp->b_error = EIO;
	}
	if (rdfinish(sc, bp) != NULL)
		rdustart(sc);
#if NRND > 0
	rnd_add_uint32(&sc->rnd_source, bp->b_blkno);
#endif
}

/*
 * Deal with errors.
 * Returns 1 if request should be restarted,
 * 0 if we should just quietly give up.
 */
int
rderror(sc)
	struct rd_softc *sc;
{
	struct cs80_stat css;
	struct buf *bp;
	daddr_t hwbn, pbn;

	DPRINTF(RDB_FOLLOW, ("rderror: sc=%p\n", sc));

	if (cs80status(sc->sc_dev.dv_parent, sc->sc_slave,
	    sc->sc_punit, &css)) {
		cs80reset(sc->sc_dev.dv_parent, sc->sc_slave, sc->sc_punit);
		return (1);
	}
#ifdef DEBUG
	if (rddebug & RDB_ERROR) {			/* status info */
		printf("\n    volume: %d, unit: %d\n",
		       (css.c_vu>>4)&0xF, css.c_vu&0xF);
		printf("    reject 0x%x\n", css.c_ref);
		printf("    fault 0x%x\n", css.c_fef);
		printf("    access 0x%x\n", css.c_aef);
		printf("    info 0x%x\n", css.c_ief);
		printf("    block,  P1-P10: ");
		printf("0x%x", *(u_int32_t *)&css.c_raw[0]);
		printf("0x%x", *(u_int32_t *)&css.c_raw[4]);
		printf("0x%x\n", *(u_int16_t *)&css.c_raw[8]);
	}
#endif
	if (css.c_fef & FEF_REXMT)
		return (1);
	if (css.c_fef & FEF_PF) {
		cs80reset(sc->sc_dev.dv_parent, sc->sc_slave, sc->sc_punit);
		return (1);
	}
	/*
	 * Unit requests release for internal maintenance.
	 * We just delay awhile and try again later.  Use expontially
	 * increasing backoff ala ethernet drivers since we don't really
	 * know how long the maintenance will take.  With RDWAITC and
	 * RDRETRY as defined, the range is 1 to 32 seconds.
	 */
	if (css.c_fef & FEF_IMR) {
		extern int hz;
		int rdtimo = RDWAITC << sc->sc_errcnt;
		DPRINTF(RDB_STATUS,
		    ("%s: internal maintenance, %d-second timeout\n",
		    sc->sc_dev.dv_xname, rdtimo));
		gpibrelease(sc->sc_ic, sc->sc_hdl);
		callout_reset(&sc->sc_restart_ch, rdtimo * hz, rdrestart, sc);
		return (0);
	}
	/*
	 * Only report error if we have reached the error reporting
	 * threshhold.  By default, this will only report after the
	 * retry limit has been exceeded.
	 */
	if (sc->sc_errcnt < rderrthresh)
		return (1);

	/*
	 * First conjure up the block number at which the error occurred.
 	 */
	bp = BUFQ_PEEK(&sc->sc_tab);
	pbn = sc->sc_dk.dk_label->d_partitions[RDPART(bp->b_dev)].p_offset;
	if ((css.c_fef & FEF_CU) || (css.c_fef & FEF_DR) ||
	    (css.c_ief & IEF_RRMASK)) {
		/*
		 * Not all errors report a block number, just use b_blkno.
		 */
		hwbn = RDBTOS(pbn + bp->b_blkno);
		pbn = bp->b_blkno;
	} else {
		hwbn = css.c_blk;
		pbn = RDSTOB(hwbn) - pbn;
	}
#ifdef DEBUG
	if (rddebug & RDB_ERROR) {			/* status info */
		printf("\n    volume: %d, unit: %d\n",
		       (css.c_vu>>4)&0xF, css.c_vu&0xF);
		printf("    reject 0x%x\n", css.c_ref);
		printf("    fault 0x%x\n", css.c_fef);
		printf("    access 0x%x\n", css.c_aef);
		printf("    info 0x%x\n", css.c_ief);
		printf("    block,  P1-P10: ");
		printf("    block: %" PRId64 ", P1-P10: ", hwbn);
		printf("0x%x", *(u_int32_t *)&css.c_raw[0]);
		printf("0x%x", *(u_int32_t *)&css.c_raw[4]);
		printf("0x%x\n", *(u_int16_t *)&css.c_raw[8]);
	}
#endif
#ifdef DEBUG
	if (rddebug & RDB_ERROR) {			/* command */
		printf("    ioc: ");
		printf("0x%x", *(u_int32_t *)&sc->sc_ioc.c_pad);
		printf("0x%x", *(u_int16_t *)&sc->sc_ioc.c_hiaddr);
		printf("0x%x", *(u_int32_t *)&sc->sc_ioc.c_addr);
		printf("0x%x", *(u_int16_t *)&sc->sc_ioc.c_nop2);
		printf("0x%x", *(u_int32_t *)&sc->sc_ioc.c_len);
		printf("0x%x\n", *(u_int16_t *)&sc->sc_ioc.c_cmd);
		return (1);
	}
#endif
	/*
	 * Now output a generic message suitable for badsect.
	 * Note that we don't use harderr because it just prints
	 * out b_blkno which is just the beginning block number
	 * of the transfer, not necessary where the error occurred.
	 */
	printf("%s%c: hard error, sector number %" PRId64 "\n",
	    sc->sc_dev.dv_xname, 'a'+RDPART(bp->b_dev), pbn);
	/*
	 * Now report the status as returned by the hardware with
	 * attempt at interpretation.
	 */
	printf("%s %s error:", sc->sc_dev.dv_xname,
	    (bp->b_flags & B_READ) ? "read" : "write");
	printf(" unit %d, volume %d R0x%x F0x%x A0x%x I0x%x\n",
	       css.c_vu&0xF, (css.c_vu>>4)&0xF,
	       css.c_ref, css.c_fef, css.c_aef, css.c_ief);
	printf("P1-P10: ");
	printf("0x%x ", *(u_int32_t *)&css.c_raw[0]);
	printf("0x%x ", *(u_int32_t *)&css.c_raw[4]);
	printf("0x%x\n", *(u_int16_t *)&css.c_raw[8]);

	return (1);
}

int
rdread(dev, uio, flags)
	dev_t dev;
	struct uio *uio;
	int flags;
{

	return (physio(rdstrategy, NULL, dev, B_READ, minphys, uio));
}

int
rdwrite(dev, uio, flags)
	dev_t dev;
	struct uio *uio;
	int flags;
{

	return (physio(rdstrategy, NULL, dev, B_WRITE, minphys, uio));
}

int
rdioctl(dev, cmd, data, flag, p)
	dev_t dev;
	u_long cmd;
	caddr_t data;
	int flag;
	struct proc *p;
{
	struct rd_softc *sc;
	struct disklabel *lp;
	int error, flags;

	sc = device_lookup(&rd_cd, RDUNIT(dev));
	if (sc == NULL)
		return (ENXIO);
	lp = sc->sc_dk.dk_label;

	DPRINTF(RDB_FOLLOW, ("rdioctl: sc=%p\n", sc));

	switch (cmd) {
	case DIOCGDINFO:
		*(struct disklabel *)data = *lp;
		return (0);

	case DIOCGPART:
		((struct partinfo *)data)->disklab = lp;
		((struct partinfo *)data)->part =
		    &lp->d_partitions[RDPART(dev)];
		return (0);

	case DIOCWLABEL:
		if ((flag & FWRITE) == 0)
			return (EBADF);
		if (*(int *)data)
			sc->sc_flags |= RDF_WLABEL;
		else
			sc->sc_flags &= ~RDF_WLABEL;
		return (0);

	case DIOCSDINFO:
		if ((flag & FWRITE) == 0)
			return (EBADF);
		return (setdisklabel(lp, (struct disklabel *)data,
		    (sc->sc_flags & RDF_WLABEL) ? 0 : sc->sc_dk.dk_openmask,
		    (struct cpu_disklabel *)0));

	case DIOCWDINFO:
		if ((flag & FWRITE) == 0)
			return (EBADF);
		error = setdisklabel(lp, (struct disklabel *)data,
		    (sc->sc_flags & RDF_WLABEL) ? 0 : sc->sc_dk.dk_openmask,
		    (struct cpu_disklabel *)0);
		if (error)
			return (error);
		flags = sc->sc_flags;
		sc->sc_flags = RDF_ALIVE | RDF_WLABEL;
		error = writedisklabel(RDLABELDEV(dev), rdstrategy, lp,
		    (struct cpu_disklabel *)0);
		sc->sc_flags = flags;
		return (error);

	case DIOCGDEFLABEL:
		rdgetdefaultlabel(sc, (struct disklabel *)data);
		return (0);
	}
	return (EINVAL);
}

void
rdgetdefaultlabel(sc, lp)
	struct rd_softc *sc;
	struct disklabel *lp;
{
	int type = sc->sc_type;

	memset((caddr_t)lp, 0, sizeof(struct disklabel));

	lp->d_type = DTYPE_GPIB;
	lp->d_secsize = DEV_BSIZE;
	lp->d_nsectors = rdidentinfo[type].ri_nbpt;
	lp->d_ntracks = rdidentinfo[type].ri_ntpc;
	lp->d_ncylinders = rdidentinfo[type].ri_ncyl;
	lp->d_secpercyl = lp->d_ntracks * lp->d_nsectors;
	lp->d_secperunit = lp->d_ncylinders * lp->d_secpercyl;

	strncpy(lp->d_typename, rdidentinfo[type].ri_desc, 16);
	strncpy(lp->d_packname, "fictitious", 16);
	lp->d_rpm = 3000;
	lp->d_interleave = 1;
	lp->d_flags = 0;

	lp->d_partitions[RAW_PART].p_offset = 0;
	lp->d_partitions[RAW_PART].p_size =
	    lp->d_secperunit * (lp->d_secsize / DEV_BSIZE);
	lp->d_partitions[RAW_PART].p_fstype = FS_UNUSED;
	lp->d_npartitions = RAW_PART + 1;

	lp->d_magic = DISKMAGIC;
	lp->d_magic2 = DISKMAGIC;
	lp->d_checksum = dkcksum(lp);
}

int
rdsize(dev)
	dev_t dev;
{
	struct rd_softc *sc;
	int psize, didopen = 0;

	sc = device_lookup(&rd_cd, RDUNIT(dev));
	if (sc == NULL || (sc->sc_flags & RDF_ALIVE) == 0)
		return (-1);

	/*
	 * We get called very early on (via swapconf)
	 * without the device being open so we may need
	 * to handle it here.
	 */
	if (sc->sc_dk.dk_openmask == 0) {
		if (rdopen(dev, FREAD | FWRITE, S_IFBLK, NULL))
			return (-1);
		didopen = 1;
	}
	psize = sc->sc_dk.dk_label->d_partitions[RDPART(dev)].p_size *
	    (sc->sc_dk.dk_label->d_secsize / DEV_BSIZE);
	if (didopen)
		(void) rdclose(dev, FREAD | FWRITE, S_IFBLK, NULL);
	return (psize);
}


static int rddoingadump;	/* simple mutex */

/*
 * Non-interrupt driven, non-dma dump routine.
 */
int
rddump(dev, blkno, va, size)
	dev_t dev;
	daddr_t blkno;
	caddr_t va;
	size_t size;
{
	struct rd_softc *sc;
	int sectorsize;		/* size of a disk sector */
	int nsects;		/* number of sectors in partition */
	int sectoff;		/* sector offset of partition */
	int totwrt;		/* total number of sectors left to write */
	int nwrt;		/* current number of sectors to write */
	int slave;
	struct disklabel *lp;
	u_int8_t stat;

	/* Check for recursive dump; if so, punt. */
	if (rddoingadump)
		return (EFAULT);
	rddoingadump = 1;

	sc = device_lookup(&rd_cd, RDUNIT(dev));
	if (sc == NULL || (sc->sc_flags & RDF_ALIVE) == 0)
		return (ENXIO);

	DPRINTF(RDB_FOLLOW, ("rddump: sc=%p\n", sc));

	slave = sc->sc_slave;

	/*
	 * Convert to disk sectors.  Request must be a multiple of size.
	 */
	lp = sc->sc_dk.dk_label;
	sectorsize = lp->d_secsize;
	if ((size % sectorsize) != 0)
		return (EFAULT);
	totwrt = size / sectorsize;
	blkno = dbtob(blkno) / sectorsize;	/* blkno in DEV_BSIZE units */

	nsects = lp->d_partitions[RDPART(dev)].p_size;
	sectoff = lp->d_partitions[RDPART(dev)].p_offset;

	/* Check transfer bounds against partition size. */
	if ((blkno < 0) || (blkno + totwrt) > nsects)
		return (EINVAL);

	/* Offset block number to start of partition. */
	blkno += sectoff;

	while (totwrt > 0) {
		nwrt = totwrt;		/* XXX */
#ifndef RD_DUMP_NOT_TRUSTED
		/*
		 * Fill out and send GPIB command.
		 */
		sc->sc_ioc.c_unit = CS80CMD_SUNIT(sc->sc_punit);
		sc->sc_ioc.c_volume = CS80CMD_SVOL(0);
		sc->sc_ioc.c_saddr = CS80CMD_SADDR;
		sc->sc_ioc.c_hiaddr = 0;
		sc->sc_ioc.c_addr = RDBTOS(blkno);
		sc->sc_ioc.c_nop2 = CS80CMD_NOP;
		sc->sc_ioc.c_slen = CS80CMD_SLEN;
		sc->sc_ioc.c_len = nwrt * sectorsize;
		sc->sc_ioc.c_cmd = CS80CMD_WRITE;
		(void) gpibsend(sc->sc_ic, slave, CS80CMD_SCMD,
		    &sc->sc_ioc.c_unit, sizeof(sc->sc_ioc)-3);
		if (gpibswait(sc->sc_ic, slave))
			return (EIO);
		/*
		 * Send the data.
		 */
		(void) gpibsend(sc->sc_ic, slave, CS80CMD_EXEC, va,
		    nwrt * sectorsize);
		(void) gpibswait(sc->sc_ic, slave);
		(void) gpibrecv(sc->sc_ic, slave, CS80CMD_QSTAT, &stat, 1);
		if (stat)
			return (EIO);
#else /* RD_DUMP_NOT_TRUSTED */
		/* Let's just talk about this first... */
		printf("%s: dump addr %p, blk %d\n", sc->sc_dev.dv_xname,
		    va, blkno);
		delay(500 * 1000);	/* half a second */
#endif /* RD_DUMP_NOT_TRUSTED */

		/* update block count */
		totwrt -= nwrt;
		blkno += nwrt;
		va += sectorsize * nwrt;
	}
	rddoingadump = 0;
	return (0);
}

#ifdef COMPAT_NOLABEL

/*
 * CS/80 partitions.  We reserve the first cylinder for a LIF
 * style boot directory (the 8k allowed in the BSD filesystem
 * is just way too small).  This boot area is outside of all but
 * the C partition.  This implies that you cannot use the C 
 * partition on a bootable disk since the filesystem would overlay
 * the boot area.  You must use the A partition.
 *
 * These maps support four basic layouts:
 *
 *	A/B/G:   This is the "traditional" setup for a bootable disk.
 *	         A is the root partition, B the swap, and G a user partition.
 *	A/D/H:   This is a setup for bootable systems requiring more swap
 *		 (e.g. those who use HPCL).  It has A as the root, D as a
 *		 larger swap, and H as a smaller user partition.
 *	A/D/E/F: Similar to A/D/H with E and F breaking H into two partitions.
 *		 E could be used for /usr and F for users.
 *	C:       This gives a single, non-bootable, large user filesystem.
 *	         Good for second drives on a machine (e.g. /usr/src).
 */
struct	size {
	daddr_t	nblocks;
	int	cyloff;
} rd7945A_sizes[8] = {
	{ RDSZ(15904),	1	},	/* A=cyl 1 thru 142 */
	{ RDSZ(20160),	143	},	/* B=cyl 143 thru 322 */
	{ RDSZ(108416),	0	},	/* C=cyl 0 thru 967 */
	{ RDSZ(40320),	143	},	/* D=cyl 143 thru 502 */
	{ RDSZ(0),	0	},	/* E=<undefined> */
	{ RDSZ(0),	0	},	/* F=<undefined> */
	{ RDSZ(72240),	323	},	/* G=cyl 323 thru 967 */
	{ RDSZ(52080),	503	},	/* H=cyl 503 thru 967 */
}, rd9134D_sizes[8] = {
	{ RDSZ(15936),	1	},	/* A=cyl 1 thru 166 */
	{ RDSZ(13056),	167	},	/* B=cyl 167 thru 302 */
	{ RDSZ(29088),	0	},	/* C=cyl 0 thru 302 */
	{ RDSZ(0),	0	},	/* D=<undefined> */
	{ RDSZ(0),	0	},	/* E=<undefined> */
	{ RDSZ(0),	0	},	/* F=<undefined> */
	{ RDSZ(0),	0	},	/* G=<undefined> */
	{ RDSZ(0),	0	},	/* H=<undefined> */
}, rd9122S_sizes[8] = {
	{ RDSZ(0),	0	},	/* A=<undefined> */
	{ RDSZ(0),	0	},	/* B=<undefined> */
	{ RDSZ(1232),	0	},	/* C=cyl 0 thru 76 */
	{ RDSZ(0),	0	},	/* D=<undefined> */
	{ RDSZ(0),	0	},	/* E=<undefined> */
	{ RDSZ(0),	0	},	/* F=<undefined> */
	{ RDSZ(0),	0	},	/* G=<undefined> */
	{ RDSZ(0),	0	},	/* H=<undefined> */
}, rd7912P_sizes[8] = {
	{ RDSZ(15904),	0	},	/* A=cyl 1 thru 71 */
	{ RDSZ(22400),	72	},	/* B=cyl 72 thru 171 */
	{ RDSZ(128128),	0	},	/* C=cyl 0 thru 571 */
	{ RDSZ(42560),	72	},	/* D=cyl 72 thru 261 */
	{ RDSZ(0),	292	},	/* E=<undefined> */
	{ RDSZ(0),	542	},	/* F=<undefined> */
	{ RDSZ(89600),	172	},	/* G=cyl 221 thru 571 */
	{ RDSZ(69440),	262	},	/* H=cyl 262 thru 571 */
}, rd7914P_sizes[8] = {
	{ RDSZ(15904),	1	},	/* A=cyl 1 thru 71 */
	{ RDSZ(40320),	72	},	/* B=cyl 72 thru 251 */
	{ RDSZ(258048),	0	},	/* C=cyl 0 thru 1151 */
	{ RDSZ(64960),	72	},	/* D=cyl 72 thru 361 */
	{ RDSZ(98560),	362	},	/* E=cyl 362 thru 801 */
	{ RDSZ(78400),	802	},	/* F=cyl 802 thru 1151 */
	{ RDSZ(201600),	252	},	/* G=cyl 221 thru 1151 */
	{ RDSZ(176960),	362	},	/* H=cyl 362 thru 1151 */
}, rd7933H_sizes[8] = {
	{ RDSZ(16146),	1	},	/* A=cyl 1 thru 27 */
	{ RDSZ(66976),	28	},	/* B=cyl 28 thru 139 */
	{ RDSZ(789958),	0	},	/* C=cyl 0 thru 1320 */
	{ RDSZ(16146),	140	},	/* D=cyl 140 thru 166 */
	{ RDSZ(165646),	167	},	/* E=cyl 167 thru 443 */
	{ RDSZ(165646),	444	},	/* F=cyl 444 thru 720 */
	{ RDSZ(706238),	140	},	/* G=cyl 140 thru 1320 */
	{ RDSZ(358800),	721	},	/* H=cyl 721 thru 1320 */
}, rd9134L_sizes[8] = {
	{ RDSZ(15920),	1	},	/* A=cyl 1 thru 199 */
	{ RDSZ(20000),	200	},	/* B=cyl 200 thru 449 */
	{ RDSZ(77840),	0	},	/* C=cyl 0 thru 972 */
	{ RDSZ(32000),	200	},	/* D=cyl 200 thru 599 */
	{ RDSZ(0),	0	},	/* E=<undefined> */
	{ RDSZ(0),	0	},	/* F=<undefined> */
	{ RDSZ(41840),	450	},	/* G=cyl 450 thru 972 */
	{ RDSZ(29840),	600	},	/* H=cyl 600 thru 972 */
}, rd7957A_sizes[8] = {
	{ RDSZ(16016),	1	},	/* A=cyl 1 thru 104 */
	{ RDSZ(24640),	105	},	/* B=cyl 105 thru 264 */
	{ RDSZ(159544),	0	},	/* C=cyl 0 thru 1035 */
	{ RDSZ(42350),	105	},	/* D=cyl 105 thru 379 */
	{ RDSZ(54824),	380	},	/* E=cyl 380 thru 735 */
	{ RDSZ(46200),	736	},	/* F=cyl 736 thru 1035 */
	{ RDSZ(118734),	265	},	/* G=cyl 265 thru 1035 */
	{ RDSZ(101024),	380	},	/* H=cyl 380 thru 1035 */
}, rd7958A_sizes[8] = {
	{ RDSZ(16128),	1	},	/* A=cyl 1 thru 64 */
	{ RDSZ(32256),	65	},	/* B=cyl 65 thru 192 */
	{ RDSZ(255276),	0	},	/* C=cyl 0 thru 1012 */
	{ RDSZ(48384),	65	},	/* D=cyl 65 thru 256 */
	{ RDSZ(100800),	257	},	/* E=cyl 257 thru 656 */
	{ RDSZ(89712),	657	},	/* F=cyl 657 thru 1012 */
	{ RDSZ(206640),	193	},	/* G=cyl 193 thru 1012 */
	{ RDSZ(190512),	257	},	/* H=cyl 257 thru 1012 */
}, rd7957B_sizes[8] = {
	{ RDSZ(16002),	1	},	/* A=cyl 1 thru 127 */
	{ RDSZ(32760),	128	},	/* B=cyl 128 thru 387 */
	{ RDSZ(159894),	0	},	/* C=cyl 0 thru 1268 */
	{ RDSZ(49140),	128	},	/* D=cyl 128 thru 517 */
	{ RDSZ(50400),	518	},	/* E=cyl 518 thru 917 */
	{ RDSZ(44226),	918	},	/* F=cyl 918 thru 1268 */
	{ RDSZ(111006),	388	},	/* G=cyl 388 thru 1268 */
	{ RDSZ(94626),	518	},	/* H=cyl 518 thru 1268 */
}, rd7958B_sizes[8] = {
	{ RDSZ(16254),	1	},	/* A=cyl 1 thru 43 */
	{ RDSZ(32886),	44	},	/* B=cyl 44 thru 130 */
	{ RDSZ(297108),	0	},	/* C=cyl 0 thru 785 */
	{ RDSZ(49140),	44	},	/* D=cyl 44 thru 173 */
	{ RDSZ(121716),	174	},	/* E=cyl 174 thru 495 */
	{ RDSZ(109620),	496	},	/* F=cyl 496 thru 785 */
	{ RDSZ(247590),	131	},	/* G=cyl 131 thru 785 */
	{ RDSZ(231336),	174	},	/* H=cyl 174 thru 785 */
}, rd7959B_sizes[8] = {
	{ RDSZ(16254),	1	},	/* A=cyl 1 thru 43 */
	{ RDSZ(49140),	44	},	/* B=cyl 44 thru 173 */
	{ RDSZ(594216),	0	},	/* C=cyl 0 thru 1571 */
	{ RDSZ(65772),	44	},	/* D=cyl 44 thru 217 */
	{ RDSZ(303912),	218	},	/* E=cyl 218 thru 1021 */
	{ RDSZ(207900),	1022	},	/* F=cyl 1022 thru 1571 */
	{ RDSZ(528444),	174	},	/* G=cyl 174 thru 1571 */
	{ RDSZ(511812),	218	},	/* H=cyl 218 thru 1571 */
}, rd2200A_sizes[8] = {
	{ RDSZ(16272),	1	},	/* A=cyl 1 thru 36 */
	{ RDSZ(49720),	37	},	/* B=cyl 37 thru 146 */
	{ RDSZ(654948),	0	},	/* C=cyl 0 thru 1448 */
	{ RDSZ(65992),	37	},	/* D=cyl 37 thru 182 */
	{ RDSZ(304648),	183	},	/* E=cyl 183 thru 856 */
	{ RDSZ(267584),	857	},	/* F=cyl 857 thru 1448 */
	{ RDSZ(588504),	147	},	/* G=cyl 147 thru 1448 */
	{ RDSZ(572232),	183	},	/* H=cyl 183 thru 1448 */
}, rd2203A_sizes[8] = {
	/* modelled after the 7937; i.e. bogus */
	{ RDSZ(16272),	1	},	/* A=cyl 1 thru 18 */
	{ RDSZ(67800),	19	},	/* B=cyl 19 thru 93 */
	{ RDSZ(1309896), 0	},	/* C=cyl 0 thru 1448 */
	{ RDSZ(16272),	94	},	/* D=cyl 19 thru 111 */
	{ RDSZ(305552),	112	},	/* E=cyl 112 thru 449 */
	{ RDSZ(305552),	450	},	/* F=cyl 450 thru 787 */
	{ RDSZ(1224920), 94	},	/* G=cyl 94 thru 1448 */
	{ RDSZ(597544),	788	},	/* H=cyl 788 thru 1448 */
}, rd7936H_sizes[8] = {
	{ RDSZ(16359),	1	},	/* A=cyl 1 thru 19 */
	{ RDSZ(67158),	20	},	/* B=cyl 20 thru 97 */
	{ RDSZ(600978),	0	},	/* C=cyl 0 thru 697 */
	{ RDSZ(16359),	98	},	/* D=cyl 98 thru 116 */
	{ RDSZ(120540),	117	},	/* E=cyl 117 thru 256 */
	{ RDSZ(120540),	256	},	/* F=cyl 256 thru 396 */
	{ RDSZ(516600),	98	},	/* G=cyl 98 thru 697 */
	{ RDSZ(259161),	397	},	/* H=cyl 397 thru 697 */
}, rd7937H_sizes[8] = {
	{ RDSZ(15990),	1	},	/* A=cyl 1 thru 10 */
	{ RDSZ(67158),	11	},	/* B=cyl 11 thru 52 */
	{ RDSZ(1116102), 0	},	/* C=cyl 0 thru 697 */
	{ RDSZ(124722),	53	},	/* D=cyl 53 thru 130 */
	{ RDSZ(163098),	131	},	/* E=cyl 131 thru 232 */
	{ RDSZ(287820),	233	},	/* F=cyl 233 thru 412 */
	{ RDSZ(1031355), 53	},	/* G=cyl 53 thru 697 */
	{ RDSZ(455715),	413	},	/* H=cyl 413 thru 697 */
};

/*
 * Indexed the same as rdidentinfo array.
 */
struct rdcompatinfo {
	struct size *sizes;	/* partition info */
} rdcompatinfo[] = {
	{ rd7945A_sizes },
	{ rd9134D_sizes },
	{ rd9122S_sizes },
	{ rd7912P_sizes },
	{ rd7914P_sizes },
	{ rd7958A_sizes },
	{ rd7957A_sizes },
	{ rd7933H_sizes },
	{ rd9134L_sizes },
	{ rd7936H_sizes },
	{ rd7937H_sizes },
	{ rd7914P_sizes },
	{ rd7945A_sizes },
	{ rd9122S_sizes },
	{ rd7957B_sizes },
	{ rd7958B_sizes },
	{ rd7959B_sizes },
	{ rd2200A_sizes },
	{ rd2203A_sizes },
};
int nrdcompatinfo = sizeof(rdcompatinfo) / sizeof(rdcompatinfo[0]);

void
rdgetcompatlabel(sc, lp)
	struct rd_softc *sc;
	struct disklabel *lp;
{
	struct rdcompatinfo *ci = &rdcompatinfo[sc->sc_type];
	const struct rdidentinfo *ri = &rdidentinfo[sc->sc_type];
	struct partition *pi;
	int dcount;
	
	rdgetdefaultlabel(sc, lp);

	lp->d_npartitions = 8;
	pi = lp->d_partitions;
	for (dcount = 0; dcount < lp->d_npartitions; dcount++) {
		pi->p_size = ci->sizes[dcount].nblocks;
		pi->p_offset = ci->sizes[dcount].cyloff * lp->d_secpercyl;
		pi->p_fsize = 1024;
		if (dcount == 1 || dcount == 3)
			pi->p_fstype = FS_SWAP;
		else if (dcount == 2)
			pi->p_fstype = FS_BOOT;
		else
			pi->p_fstype = FS_BSDFFS;
		pi->p_frag = 8;
		pi++;
	}
}

#endif /* COMPAT_NOLABEL */
