/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* YIPS @(#)$Id: handler.c,v 1.1.1.1.2.6.2.19 1998/10/29 16:45:23 sakane Exp $ */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>

#include <net/route.h>
#include <net/pfkeyv2.h>
#include <netkey/keydb.h>

#include <netinet/in.h>
#include <netinet6/ipsec.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <netdb.h>

#if !defined(HAVE_GETADDRINFO) || !defined(HAVE_GETNAMEINFO)
#include "gai.h"
#endif

#include "var.h"
#include "vmbuf.h"
#include "misc.h"
#include "schedule.h"
#include "cfparse.h"
#include "isakmp.h"
#include "isakmp_var.h"
#include "oakley.h"
#include "ipsec_doi.h"
#include "crypto.h"
#include "handler.h"
#include "pfkey.h"
#include "admin.h"
#include "debug.h"

char *local_secret;
vchar_t oakley_prime768;
vchar_t oakley_prime1024;

u_int portI = PORT_ISAKMP;
u_int portIr = PORT_ISAKMP;
int sockIs[MAXADDRS], n_soI = 0;
struct sockaddr *myaddr[MAXADDRS];

struct isakmp_ph1tab ph1tab;

static int isakmp_do_expire(struct isakmp_ph1 *);

/*
 * isakmp packet handler
 */
int
isakmp_handler(sockI)
	int sockI;
{
	struct isakmp isakmp;
	struct sockaddr_storage from;
	struct sockaddr_storage my;
	int fromlen = sizeof(from);
	int mylen = sizeof(my);
	int len;
	vchar_t *buf;
	int error = -1;

	/* read message by MSG_PEEK */
	while ((len = recvfrom(sockI, (char *)&isakmp, sizeof(isakmp),
		    MSG_PEEK, (struct sockaddr *)&from, &fromlen)) < 0) {
		if (errno == EINTR) continue;
		plog("isakmp_handler", "recvfrom (%s)\n", strerror(errno));
		goto end;
	}

	/* check isakmp header length */
	if (len < sizeof(isakmp)) {
		plog2((struct sockaddr *)&from,
			"isakmp_handler", "received invalid header length.\n");
		if ((len = recvfrom(sockI, (char *)&isakmp, sizeof(isakmp),
			    0, (struct sockaddr *)&from, &fromlen)) < 0) {
			plog("isakmp_handler", "recvfrom (%s)\n", strerror(errno));
		}
		goto end;
	}

	/* read real message */
	if ((buf = vmalloc(ntohl(isakmp.len))) == 0) {
		plog("isakmp_handler", "vmalloc (%s)\n", strerror(errno)); 
		goto end;
	}

	while ((len = recvfrom(sockI, buf->v, buf->l,
	                    0, (struct sockaddr *)&from, &fromlen)) < 0) {
		if (errno == EINTR) continue;
		plog("isakmp_handler", "recvfrom (%s)\n", strerror(errno));
		goto end;
	}

	/* XXX get my name */
	/* XXX I can't get port number... */
    {
	int dummy, dummylen = sizeof(struct sockaddr_in);
	u_short xport = 0;

	/* get real port received packet */
	if (getsockname(sockI, (struct sockaddr *)&my, &mylen) < 0) {
		plog("isakmp_handler", "getsockname (%s)\n", strerror(errno));
		goto end;
	}
	switch (my.ss_family) {
	case AF_INET:
		xport = ((struct sockaddr_in *)&my)->sin_port;
		break;
#ifdef INET6
	case AF_INET6:
		xport = ((struct sockaddr_in6 *)&my)->sin6_port;
		break;
#endif
	default:
		xport = 0;
		break;
	}

	/* get real interface received packet */
	if ((dummy = socket(from.ss_family, SOCK_DGRAM, 0)) < 0) {
		plog("isakmp_handler", "socket (%s)\n", strerror(errno));
		goto end;
	}
	
	if (connect(dummy, (struct sockaddr *)&from, fromlen) < 0) {
		plog("isakmp_handler", "connect (%s)\n", strerror(errno));
		goto end;
	}

	if (getsockname(dummy, (struct sockaddr *)&my, &mylen) < 0) {
		plog("isakmp_handler", "getsockname (%s)\n", strerror(errno));
		goto end;
	}

	/* make real sockaddr received packet */
	switch (my.ss_family) {
	case AF_INET:
		((struct sockaddr_in *)&my)->sin_port = xport;
		break;
#ifdef INET6
	case AF_INET6:
		((struct sockaddr_in6 *)&my)->sin6_port = xport;
		break;
#endif
	default:
		xport = 0;
		break;
	}

	close(dummy);
    }

	if (len != buf->l) {
		plog2((struct sockaddr *)&from,
			"isakmp_handler", "received invalid length, why ?\n");
		goto end;
	}

	YIPSDEBUG(DEBUG_NET,
		INET_NTOP((struct sockaddr *)&from, _addr1_);
		INET_NTOP((struct sockaddr *)&my, _addr2_);
		plog("isakmp_handler",
			"%d bytes message has been received "
			"from %s[%d] by %s[%d].\n", len,
			_addr1_, ntohs(_INPORTBYAF((struct sockaddr *)&from)),
			_addr2_, ntohs(_INPORTBYAF((struct sockaddr *)&my))));
	YIPSDEBUG(DEBUG_DNET, pvdump(buf));

	/* XXX: check sender whether allow or not */

	/* XXX: I don't know how to check isakmp half connection. */

	/* isakmp main routine */
	if (isakmp_main(buf, (struct sockaddr *)&from,
			(struct sockaddr *)&my) != 0) goto end;

	error = 0;

end:
	vfree(buf);

	return(error);
}

/* %%%
 *
 */
int
isakmp_init()
{
	int tmp, i;

	/* set DH MODP */
	oakley_prime768.v = strtob(OAKLEY_PRIME_MODP768, 16, &oakley_prime768.l);
	oakley_prime1024.v = strtob(OAKLEY_PRIME_MODP1024, 16, &oakley_prime1024.l);

	/* initialize a isakmp status table */
	memset((char *)&ph1tab, 0, sizeof(ph1tab));

	srandom(time(0));

	if (!n_soI) {
		/*
		 * initialize default port for ISAKMP to send, if no "listen"
		 * directive is specified in config file.
		 */
		struct addrinfo hints, *res, *res0;
		int error;
		char portbuf[10];

		YIPSDEBUG(DEBUG_INFO,
			plog("isakmp_init",
				"configuring default isakmp port.\n"));

		snprintf(portbuf, sizeof(portbuf), "%d", portI);
		memset(&hints, 0, sizeof(hints));
		hints.ai_family = PF_UNSPEC;
		hints.ai_flags = AI_PASSIVE;
		error = getaddrinfo(NULL, portbuf, &hints, &res);
		if (error) {
			plog("isakmp_init",
				"getaddrinfo (%s)\n", gai_strerror(error));
			goto err;
		}
		res0 = res;
		while (res) {
			if (sizeof(myaddr)/sizeof(myaddr[0]) <= n_soI) {
				plog("isakmp_init", "too many addresses\n");
				freeaddrinfo(res0);
				goto err;
			}
			myaddr[n_soI] = CALLOC(res->ai_addrlen,
				struct sockaddr *);
			if (myaddr[n_soI] == 0) {
				plog("isakmp_init",
					"calloc (%s)\n", strerror(errno));
				freeaddrinfo(res0);
				goto err;
			}
			memcpy(myaddr[n_soI], res->ai_addr, res->ai_addrlen);
			n_soI++;
			res = res->ai_next;
		}
		freeaddrinfo(res0);
	}

	/* initialize a port of ISAKMP */
	for (i = 0; i < n_soI; i++) {
		if ((sockIs[i] = socket(myaddr[i]->sa_family, SOCK_DGRAM, 0)) < 0) {
			plog("isakmp_init", "socket (%s)\n", strerror(errno));
			goto err;
		}

		if (setsockopt(sockIs[i], SOL_SOCKET, SO_REUSEPORT,
		               (void *)&tmp, sizeof(tmp)) < 0) {
			plog("isakmp_init",
				"setsockopt (%s)\n", strerror(errno));
			goto err;
		}

		tmp = IPSEC_LEVEL_BYPASS;
		switch (myaddr[i]->sa_family) {
		case AF_INET:
		    {
			int j;
			int tab[] = {
				IP_AUTH_TRANS_LEVEL, IP_ESP_TRANS_LEVEL,
				IP_AUTH_NETWORK_LEVEL, IP_ESP_NETWORK_LEVEL,
				-1
			};

			for (j = 0; 0 <= tab[j]; j++) {
				if (setsockopt(sockIs[i], IPPROTO_IP,
						tab[j], (void *)&tmp,
						sizeof(tmp)) < 0) {
					plog("isakmp_init", "setsockopt (%s)\n",
						strerror(errno));
					goto err;
				}
			}
			break;
		    }
#ifdef INET6
		case AF_INET6:
		    {
			int j;
			int tab[] = {
				IPV6_AUTH_TRANS_LEVEL, IPV6_ESP_TRANS_LEVEL,
				IPV6_AUTH_NETWORK_LEVEL, IPV6_ESP_NETWORK_LEVEL,
				-1
			};

			for (j = 0; 0 <= tab[j]; j++) {
				if (setsockopt(sockIs[i], IPPROTO_IPV6,
						tab[j], (void *)&tmp,
						sizeof(tmp)) < 0) {
					plog("isakmp_init", "setsockopt (%s)\n",
						strerror(errno));
					goto err;
				}
			}
			break;
		    }
#endif
		}
		tmp = 0;

		if (bind(sockIs[i], (struct sockaddr *)myaddr[i],
			myaddr[i]->sa_len) < 0) {
			plog("isakmp_init", "bind (%s)\n", strerror(errno));
			goto err;
		}

	    {
		char hostbuf[MAXHOSTNAMELEN];
		char portbuf[MAXHOSTNAMELEN];
		YIPSDEBUG(DEBUG_INFO,
			GETNAMEINFO(myaddr[i], _addr1_, _addr2_);
			plog("isakmp_init",
				"using %s[%s] as the port of isakmp.\n",
				_addr1_, _addr2_));
	    }
	}

	return(0);

err:
	isakmp_close();
	return(-1);
}

/*
 * make strings containing i_cookie + r_cookie + msgid
 */
u_char *
isakmp_pindex(index, msgid)
	isakmp_index *index;
	msgid_t *msgid;
{
	static char buf[64];
	u_char *p;
	int i, j;

	memset(buf, 0, sizeof(buf));

	/* copy index */
	p = (u_char *)index;
	for (j = 0, i = 0; i < sizeof(isakmp_index); i++) {
		sprintf((char *)&buf[j], "%02x", p[i]);
		j += 2;
		switch (i) {
		case 7: case 15:
			buf[j++] = ':';
		}
	}

	if (msgid == 0)
		return(buf);

	/* copy msgid */
	p = (u_char *)msgid;
	for (i = 0; i < sizeof(msgid_t); i++) {
		sprintf((char *)&buf[j], "%02x", p[i]);
		j += 2;
	}

	return(buf);
}

/* %%%
 */
void
isakmp_close()
{
	int num = n_soI;

	while (num--)
		(void)close(sockIs[num]);

	return;
}

int
isakmp_send(iph1, buf)
	struct isakmp_ph1 *iph1;
	vchar_t *buf;
{
	struct sockaddr *sa;
	int len;
	int i;

	sa = iph1->remote;

	/* send HDR;SA to responder */
	/* XXX */
	for (i = 0; i < n_soI; i++) {
		if (sa->sa_family == myaddr[i]->sa_family)
			break;
	}
	if (i == n_soI) {
		plog("isakmp_send", "no socket matches address family %d\n",
			sa->sa_family);
		return -1;
	}
	if ((len = sendto(sockIs[i], buf->v, buf->l, 0, sa, sa->sa_len)) < 0) {
		plog("isakmp_send", "sendto (%s)\n", strerror(errno));
		return(-1);
	}

	YIPSDEBUG(DEBUG_NET,
		GETNAMEINFO(sa, _addr1_, _addr2_);
		plog("isakmp_send",
			"%d bytes message has been sent to %s[%s].\n",
			len, _addr1_, _addr2_));
	YIPSDEBUG(DEBUG_DNET, pvdump(buf));

	return(0);
}

int
isakmp_resend_ph1(sc)
	struct sched *sc;
{
	struct isakmp_ph1 *iph1 = (struct isakmp_ph1 *)sc->ptr1;
	vchar_t *buf = (vchar_t *)sc->ptr2;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_resend_ph1", "resend packet.\n"));
	YIPSDEBUG(DEBUG_SCHED2, plog("isakmp_resend_ph1", "tick over #%s\n",
	    sched_pindex(&sc->index)));

	if (isakmp_send(iph1, buf) < 0)
		return(-1);

	sc->tick = isakmp_timer;

	return(0);
}

/* called as schedule of negotiating isakmp-sa is time over. */
int
isakmp_timeout_ph1(sc)
	struct sched *sc;
{
	struct isakmp_ph1 *iph1 = (struct isakmp_ph1 *)sc->ptr1;
	vchar_t *buf = (vchar_t *)sc->ptr2;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_timeout_ph1", "timeout to send.\n"));
	YIPSDEBUG(DEBUG_SCHED2, plog("isakmp_timeout_ph1", "try over #%s\n",
	    sched_pindex(&sc->index)));

	if (isakmp_free_ph1(iph1) < 0)
		return(-1);

	vfree(buf);

	sched_kill(sc);

	return(0);
}

int
isakmp_resend_ph2(sc)
	struct sched *sc;
{
	struct isakmp_ph2 *iph2 = (struct isakmp_ph2 *)sc->ptr1;
	vchar_t *buf = (vchar_t *)sc->ptr2;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_resend_ph2", "resend packet.\n"));
	YIPSDEBUG(DEBUG_SCHED2, plog("isakmp_resend_ph2", "tick over #%s\n",
	    sched_pindex(&sc->index)));

	if (isakmp_send(iph2->ph1, buf) < 0)
		return(-1);

	sc->tick = isakmp_timer;

	return(0);
}

/* called as schedule of negotiating ipsec-sa is time over. */
int
isakmp_timeout_ph2(sc)
	struct sched *sc;
{
	struct isakmp_ph2 *iph2 = (struct isakmp_ph2 *)sc->ptr1;
	vchar_t *buf = (vchar_t *)sc->ptr2;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_timeout_ph2", "timeout to send.\n"));
	YIPSDEBUG(DEBUG_SCHED2, plog("isakmp_timeout_ph2", "try over #%s\n",
	    sched_pindex(&sc->index)));

	if (isakmp_free_ph2(iph2) < 0)
		return(-1);

	vfree(buf);

	sched_kill(sc);

	return(0);
}

/* called as schedule of isakmp-sa is expired. */
int
isakmp_expire(sc)
	struct sched *sc;
{
	struct isakmp_ph1 *iph1 = (struct isakmp_ph1 *)sc->ptr1;

	YIPSDEBUG(DEBUG_SCHED2, plog("isakmp_expire", "try over #%s\n",
	    sched_pindex(&sc->index)));

	plog("isakmp_expire",
	    "ISAKMP-SA is expired. %s\n", isakmp_pindex(&iph1->index, 0));

	if (iph1->ph2tab.len == 0) {
		if (isakmp_do_expire(iph1) < 0) return(-1);
	} else {
		/* set flag */
		iph1->status |= ISAKMP_EXPIRED;
	}

	sched_kill(sc);

	return(0);
}

static int
isakmp_do_expire(iph1)
	struct isakmp_ph1 *iph1;
{
	/* if it's initiator, begin re-negosiation */
	if (iph1->dir == INITIATOR) {
		YIPSDEBUG(DEBUG_STAMP,
		    plog("isakmp_do_expire", "begin ISAKMP-SA re-negosiation.\n"));

		if (isakmp_begin_ident(iph1->cfp,
			iph1->local, iph1->remote) == 0) return(-1);
	}

	/* delete old status record */
	if (isakmp_free_ph1(iph1) < 0) return(-1);

	return(0);
}

/* %%%
 * functions about management of the isakmp status table
 */
/*
 * create new isakmp Phase 1 status record to handle isakmp
 */
struct isakmp_ph1 *
isakmp_new_ph1(index)
	isakmp_index *index;
{
	struct isakmp_ph1 *iph1, *c;

	if (isakmp_ph1byindex(index) != 0) {
		plog("isakmp_new_ph1",
		    "already exists. %s\n", isakmp_pindex(index, 0));
		return(0);
	}

	/* create new iph1 */
	if ((iph1 = CALLOC(sizeof(*iph1), struct isakmp_ph1 *)) == 0) {
		plog("isakmp_new_ph1", "calloc (%s)\n", strerror(errno)); 
		return(0);
	}

	memcpy((caddr_t)&iph1->index, (caddr_t)index, sizeof(*index));
	iph1->status = ISAKMP_PH1_N;

	/* add to phase 1 table */
	iph1->next = (struct isakmp_ph1 *)0;
	iph1->prev = ph1tab.tail;

	if (ph1tab.tail == 0)
		ph1tab.head = iph1;
	else
		ph1tab.tail->next = iph1;
	ph1tab.tail = iph1;
	ph1tab.len++;

	return(iph1);
}

/*
 * free from isakmp Phase 1 status table
 */
int
isakmp_free_ph1(iph1)
	struct isakmp_ph1 *iph1;
{
	struct isakmp_ph1 *c;
	int error = -1;

	/* diagnostics */
	if ((c = isakmp_ph1byindex(&iph1->index)) == 0) {
		plog("isakmp_free_ph1",
			"Why is there no status, %s\n", isakmp_pindex(&iph1->index, 0));
		goto end;
	}

	if (c != iph1) {
		plog("isakmp_free_ph1",
		    "why not equalize, %s\n", isakmp_pindex(&iph1->index, 0));
		goto end;
	}

	if (c->ph2tab.len != 0) {
		plog("isakmp_free_ph1",
		    "why ipsec status is alive, %s\n", isakmp_pindex(&iph1->index, 0));
		goto end;
	}

	/* XXX: free more !? */
	/* if (c->dhp) vfree(c->dhp); because this is static */
	if (c->dhpriv) vfree(c->dhpriv);
	if (c->dhpub) vfree(c->dhpub);
	if (c->dhpub_p) vfree(c->dhpub_p);
	if (c->dhgxy) vfree(c->dhgxy);
	if (c->nonce) vfree(c->nonce);
	if (c->nonce_p) vfree(c->nonce_p);
	if (c->skeyid) vfree(c->skeyid);
	if (c->skeyid_d) vfree(c->skeyid_d);
	if (c->skeyid_a) vfree(c->skeyid_a);
	if (c->skeyid_e) vfree(c->skeyid_e);
	if (c->key) vfree(c->key);
	if (c->hash) vfree(c->hash);
	if (c->iv) vfree(c->iv);
	if (c->ive) vfree(c->ive);
	if (c->ivd) vfree(c->ivd);
	if (c->sa) vfree(c->sa);
	/* if (c->id) vfree(c->id); */
	if (c->id_p) vfree(c->id_p);

	if (c->remote) free(c->remote);

	if (c->isa) {
		if (c->isa->spi) vfree(c->isa->spi);
		(void)free(c->isa);
	}

	/* reap from phase 1 table */
	/* middle */
	if (c->prev && c->next) {
		c->prev->next = c->next;
		c->next->prev = c->prev;
	} else
	/* tail */
	if (c->prev && c->next == 0) {
		c->prev->next = (struct isakmp_ph1 *)0;
		ph1tab.tail = c->prev;
	} else
	/* head */
	if (c->prev == 0 && c->next) {
		c->next->prev = (struct isakmp_ph1 *)0;
		ph1tab.head = c->next;
	} else {
	/* iph2->next == 0 && iph2->prev == 0 */
	/* last one */
		ph1tab.head = (struct isakmp_ph1 *)0;
		ph1tab.tail = (struct isakmp_ph1 *)0;
	}

	ph1tab.len--;

	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_free_ph1", "ISAKMP-SA negotiation is free, %s\n",
	        isakmp_pindex(&iph1->index, 0)));

	(void)free(c);

	error = 0;
end:
	return(error);
}

/*
 * search on table of isakmp Phase 1 status by index.
 */
struct isakmp_ph1 *
isakmp_ph1byindex(index)
	isakmp_index *index;
{
	struct isakmp_ph1 *c;

	for (c = ph1tab.head; c; c = c->next) {
		if (memcmp((char *)&c->index, (char *)index, sizeof(*index)) == 0)
			return(c);
	}

	return(0);
}

/*
 * search on table of isakmp Phase 1 status by i_ck in index.
 */
struct isakmp_ph1 *
isakmp_ph1byindex0(index)
	isakmp_index *index;
{
	struct isakmp_ph1 *c;

	for (c = ph1tab.head; c; c = c->next) {
		if (memcmp((char *)&c->index, (char *)index, sizeof(cookie_t)) == 0)
			return(c);
	}

	return(0);
}

/*
 * search isakmp-sa record established on table of isakmp Phase 1
 * by destination address
 */
struct isakmp_ph1 *
isakmp_ph1byaddr(addr)
	struct sockaddr *addr;
{
	struct isakmp_ph1 *c, *tmp = 0;

	for (c = ph1tab.head; c; c = c->next) {
		if (saddrcmp(c->remote, addr) == 0) {

			/* XXX Should new SA be applied if SA is not initiator ? */
			if (c->dir != INITIATOR) continue;

		 	if (! ISSET(c->status, ISAKMP_EXPIRED)) return(c);
			tmp = c;
		}
	}

	return(tmp == 0 ? 0 : tmp);
}

/*
 * create new isakmp Phase 2 status record to handle isakmp in Phase2
 */
struct isakmp_ph2 *
isakmp_new_ph2(iph1, msgid)
	struct isakmp_ph1 *iph1;
	msgid_t *msgid;
{
	struct isakmp_ph2 *iph2 = 0, *c;
	int error = -1;

	/* validation */
	if ((c = isakmp_ph2bymsgid(iph1, msgid)) != 0) {
		plog("isakmp_new_ph2",
		    "already exists, %d.\n", isakmp_pindex(&iph1->index, msgid));
		goto end;
	}

	/* create new iph2 */
	if ((iph2 = CALLOC(sizeof(*iph2), struct isakmp_ph2 *)) == 0) {
		plog("isakmp_new_ph2", "calloc (%s)\n", strerror(errno)); 
		goto end;
	}

	iph2->status = ISAKMP_PH2_N;
	memcpy((caddr_t)&iph2->msgid, (caddr_t)msgid, sizeof(msgid_t));
	iph2->ph1 = iph1;

	iph2->next = (struct isakmp_ph2 *)0;
	iph2->prev = iph1->ph2tab.tail;

	/* add to phase 2 table */
	if (iph1->ph2tab.tail == 0)
		iph1->ph2tab.head = iph2;
	else
		iph1->ph2tab.tail->next = iph2;
	iph1->ph2tab.tail = iph2;
	iph1->ph2tab.len++;

	error = 0;

end:
	if (error) {
		if (iph2) (void)isakmp_free_ph2(iph2);
		iph2 = 0;
	}

	return(iph2);
}

/*
 * free from isakmp Phase 2 status table
 */
int
isakmp_free_ph2(iph2)
	struct isakmp_ph2 *iph2;
{
	struct isakmp_ph1 *iph1;

	/* save isakmp status to check expiration. */
	iph1 = iph2->ph1;

	/* XXX: free more !? */
	/* if (iph2->dhp) vfree(iph2->dhp); because this is static */
	if (iph2->dhpriv) vfree(iph2->dhpriv);
	if (iph2->dhpub) vfree(iph2->dhpub);
	if (iph2->dhpub_p) vfree(iph2->dhpub_p);
	if (iph2->dhgxy) vfree(iph2->dhgxy);
	if (iph2->id) vfree(iph2->id);
	if (iph2->id_p) vfree(iph2->id_p);
	if (iph2->nonce) vfree(iph2->nonce);
	if (iph2->nonce_p) vfree(iph2->nonce_p);
	if (iph2->hash) vfree(iph2->hash);
	if (iph2->iv) vfree(iph2->iv);
	if (iph2->ive) vfree(iph2->ive);
	if (iph2->ivd) vfree(iph2->ivd);

	if (iph2->pst) pfkey_free_st(iph2->pst);
	if (iph2->sa) vfree(iph2->sa);
	if (iph2->isa) free(iph2->isa);

	/* reap from phase 2 table */
	/* middle */
	if (iph2->prev && iph2->next) {
		iph2->prev->next = iph2->next;
		iph2->next->prev = iph2->prev;
	} else
	/* tail */
	if (iph2->prev && iph2->next == 0) {
		iph2->prev->next = (struct isakmp_ph2 *)0;
		iph1->ph2tab.tail = iph2->prev;
	} else
	/* head */
	if (iph2->prev == 0 && iph2->next) {
		iph2->next->prev = (struct isakmp_ph2 *)0;
		iph1->ph2tab.head = iph2->next;
	} else {
	/* iph2->next == 0 && iph2->prev == 0 */
		iph1->ph2tab.head = (struct isakmp_ph2 *)0;
		iph1->ph2tab.tail = (struct isakmp_ph2 *)0;
	}

	iph1->ph2tab.len--;

	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_free_ph2", "IPsec-SA negotiation is free, %s\n",
	        isakmp_pindex(&iph1->index, &iph2->msgid)));

	(void)free(iph2);

	/* if isakmp-sa has been expired,
	 * free isakmp-sa and begin to re-negotiation.
	 */
	if (iph1->ph2tab.len == 0 && ISSET(iph1->status, ISAKMP_EXPIRED)) {
		if (isakmp_do_expire(iph1) < 0) return(-1);
	}

	return(0);
}

struct isakmp_ph2 *
isakmp_ph2bymsgid(iph1, msgid)
	struct isakmp_ph1 *iph1;
	msgid_t *msgid;
{
	struct isakmp_ph2 *c;

	for (c = iph1->ph2tab.head; c; c = c->next) {
		if (memcmp((char *)&c->msgid, (char *)msgid, sizeof(msgid_t)) == 0)
			return(c);
	}

	return(0);
}

/* %%%
 * Interface between PF_KEYv2 and ISAKMP
 */
/*
 * receive PFKEYv2 from kernel, and begin either IDENT or QUICK mode.
 * if IDENT mode finished, begin QUICK mode.
 */
int
isakmp_pfkey(pst)
	struct pfkey_st *pst;
{
	struct isakmp_conf *cfp;
	struct isakmp_ph1 *iph1 = 0;
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("isakmp_pfkey", "begin.\n"));

	/* search appropreate configuration */
	if ((cfp = isakmp_cfbypeer(pst->dst)) == &cftab) {
		/* apply anonymous, but.. */
		if (cfp->remote == 0) {
			plog("isakmp_pfkey",
				"peer address of ISAKMP couldn't be found.\n");
			goto end;
		}
	}

	/* search isakmp status table by address */
	iph1 = isakmp_ph1byaddr(pst->dst);

	if (iph1 != 0 && iph1->status == ISAKMP_PH2_N) {
#if defined(SINGLE)
		if (iph1->ph2tab.len == 0) {
#endif
			/* found ISAKMP-SA. */
			YIPSDEBUG(DEBUG_STAMP,
			    plog("isakmp_pfkey", "begin QUICK mode.\n"));

			/* begin quick mode */
			if (isakmp_begin_quick(iph1, pst) < 0) goto end;

#if defined(SINGLE)
		} else {

			/* It's on time to process other negotiation. */
			YIPSDEBUG(DEBUG_NOTIFY,
				plog("isakmp_pfkey",
			        "queued, as on time to negotiate other SA.\n"));
			isakmp_new_queue(pst, pst->dst);
		}
#endif

	} else
	if (iph1 != 0 && ISSET(iph1->status, ISAKMP_EXPIRED)) {
		/* found ISAKMP-SA, but expired. */
		YIPSDEBUG(DEBUG_STAMP,
		    plog("isakmp_pfkey", "queued, as ISAKMP-SA expired.\n"));
		isakmp_new_queue(pst, pst->dst);

	} else
	if (iph1 != 0 && (iph1->status != ISAKMP_PH2_N)) {
		/* found ISAKMP-SA, but begin negotiation. */
		YIPSDEBUG(DEBUG_STAMP,
		    plog("isakmp_pfkey", "queued, ISAKMP-SA is on negotiation.\n"));
		isakmp_new_queue(pst, pst->dst);

	} else {
		/* no ISAKMP-SA found. */
		YIPSDEBUG(DEBUG_STAMP,
		    plog("isakmp_pfkey", "queued, as no phase 1 exists.\n"));

		YIPSDEBUG(DEBUG_STAMP,
		    plog("isakmp_pfkey", "begin IDENT mode.\n"));

		isakmp_new_queue(pst, pst->dst);

		/* XXX copy port */
		((struct sockaddr_in *)pst->dst)->sin_port = 
			((struct sockaddr_in *)cfp->remote)->sin_port;

		/* begin ident mode */
		if ((iph1 = isakmp_begin_ident(cfp,
					pst->src, pst->dst)) == 0) goto end;
	}

	error = 0;

end:
	return(error);
}

int
isakmp_new_queue(pst, remote)
	struct pfkey_st *pst;
	struct sockaddr *remote;
{
	YIPSDEBUG(DEBUG_SCHED,
	    plog("isakmp_new_queue", "new PFKEY request scheduled.\n"));

	/* add to the schedule to resend, and seve back pointer. */
	pst->sc = sched_add(1, isakmp_pfkey_check,
	                    pfkey_timer, isakmp_pfkey_over,
	                    (caddr_t)pst, (caddr_t)remote);

	return(0);
}

int
isakmp_pfkey_check(sc)
	struct sched *sc;
{
	struct pfkey_st *pst = (struct pfkey_st *)sc->ptr1;
	struct sockaddr *remote = (struct sockaddr *)sc->ptr2;
	struct isakmp_ph1 *iph1;

	YIPSDEBUG(DEBUG_SCHED2,
	    plog("isakmp_pfkey_check",
	        "tick over #%s\n", sched_pindex(&sc->index)));

	iph1 = isakmp_ph1byaddr(remote);
	if (iph1 != 0
	 && iph1->status == ISAKMP_PH2_N
	 && iph1->ph2tab.len == 0) {
		/* found isakmp-sa */
		sched_kill(sc);
		if (isakmp_pfkey(pst) < 0) return(-1);
	} else
		sc->tick = 1;

	return(0);
}

/* called as schedule is time over. */
int
isakmp_pfkey_over(sc)
	struct sched *sc;
{
	struct pfkey_st *pst = (struct pfkey_st *)sc->ptr1;
	struct sockaddr *remote = (struct sockaddr *)sc->ptr2;

	YIPSDEBUG(DEBUG_STAMP,
	    plog("isakmp_pfkey_over", "timeout to negosiate SA.\n"));
	YIPSDEBUG(DEBUG_SCHED2,
	    plog("isakmp_pfkey_over", "try over #%s\n",
	        sched_pindex(&sc->index)));

	plog2(remote, "isakmp_pfkey_over",
		"no IPSEC-SA exchanged, because ISAKMP-SA could't be established.\n");

	/* XXX */
	pfkey_free_st(pst);

	sched_kill(sc);

	return(0);
}

/*
 * decision configuration by peer address.
 */
struct isakmp_conf *
isakmp_cfbypeer(remote)
	struct sockaddr *remote;
{
	struct isakmp_conf *cfp;
	u_long mask;

	YIPSDEBUG(DEBUG_MISC,
		INET_NTOP(remote, _addr1_);
		plog("isakmp_cfbypeer",
			"search with remote addr=%s.\n", _addr1_));

	for (cfp = cftab.next; cfp; cfp = cfp->next) {
		if (saddrcmp(remote, cfp->remote) == 0) {
			YIPSDEBUG(DEBUG_MISC,
				INET_NTOP(cfp->remote, _addr1_);
				plog("isakmp_cfbypeer",
					"apply configuration addr=%s[%u]\n",
					_addr1_,
					ntohs(_INPORTBYAF(cfp->remote))));
			return(cfp);
		}
	}

	YIPSDEBUG(DEBUG_MISC,
	    plog("isakmp_cfbypeer", "apply default configuration.\n"));

	return(&cftab); /* anonymous */
}

vchar_t *
isakmp_dump_sa()
{
	struct isakmp_ph1 *dst, *src;
	struct admin_com *com;
	vchar_t *buf;
	int tlen;

	tlen = sizeof(struct isakmp_ph1) * ph1tab.len
	     + sizeof(struct admin_com);

	if ((buf = vmalloc(tlen)) == 0) {
		plog("isakmp_dump_sa", "vmalloc(%s)\n", strerror(errno));
		return(0);
	}
	com = (struct admin_com *)buf->v;

	com->cmd = ADMIN_SHOW_SA;
	com->len = tlen;

	dst = (struct isakmp_ph1 *)(buf->v + sizeof(*com));
	for (src = ph1tab.head; src; src = src->next) {
		memcpy((caddr_t)dst, (caddr_t)src, sizeof(*dst));
		dst++;
	}

	return(buf);
}

