/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by WIDE Project and
 *    its contributors.
 * 4. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* YIPS @(#)$Id: pfkey_lib.c,v 1.1.1.1.2.6.2.6 1998/09/14 11:56:37 sakane Exp $ */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>

#include <net/route.h>
#include <net/pfkeyv2.h>

#include <netinet/in.h>
#include <netkey/keydb.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "var.h"
#include "vmbuf.h"
#include "isakmp.h"
#include "ipsec_doi.h"
#include "pfkey.h"
#include "pfkey_lib.h"
#include "crypto.h"
#include "debug.h"

static int so_PK;

int pfkey_aalg_type[] = {
SADB_AALG_NONE,		/* reserved */
SADB_AALG_MD5HMAC,	/* IPSECDOI_ATTR_AUTH_HMAC_MD5 */
SADB_AALG_SHA1HMAC,	/* IPSECDOI_ATTR_AUTH_HMAC_SHA1 */
SADB_AALG_MD5,		/* not supported. */
SADB_AALG_SHA,		/* not supported. */
SADB_AALG_NULL,		/* not supported. */
SADB_AALG_MAX
};

int pfkey_ealg_type[] = {
SADB_EALG_NONE,		/* reserved */
SADB_EALG_DESCBC,	/* IPSECDOI_ESP_DES_IV64 */
			/* sadb_sa_flags |= SADB_X_EXT_OLD */
SADB_EALG_DESCBC,	/* IPSECDOI_ESP_DES */
SADB_EALG_3DESCBC,	/* IPSECDOI_ESP_3DES */
SADB_EALG_RC5CBC,	/* IPSECDOI_ESP_RC5 */
0,			/* IPSECDOI_ESP_IDEA isn't supported. */
SADB_EALG_CAST128CBC,	/* IPSECDOI_ESP_CAST */
SADB_EALG_BLOWFISHCBC,	/* IPSECDOI_ESP_BLOWFISH */
0,			/* IPSECDOI_ESP_3IDEA isn't supported. */
SADB_EALG_DESCBC,	/* IPSECDOI_ESP_DES_IV32 */
			/* sadb_sa_flags |= (SADB_X_EXT_OLD|SADB_X_EXT_IV4B)*/
0,			/* IPSECDOI_ESP_RC4 isn't supported. */
SADB_EALG_NULL,		/* IPSECDOI_ESP_NULL */
SADB_EALG_MAX
};

static caddr_t key_setsadbaddr __P((caddr_t, u_int,
				u_int, caddr_t, u_int, u_int, u_int));
static caddr_t key_setsadbkey __P((caddr_t, u_int, caddr_t, u_int));
static caddr_t key_setsadbsa __P((caddr_t, caddr_t, u_int, u_int));
static caddr_t key_setsadblifetime __P((caddr_t, u_int, u_int32_t));

/* %%% */
/*
 * send SADB_REGISTER message to the kernel and receive a reply from kernel.
 * IN:
 *	satype:	type of SA in order to register.
 * OUT:
 *	0     : error occured, and set errno.
 *	others: a pointer to new allocated buffer in which supported
 *	        algorithms is.
 */
struct sadb_msg *
pfkey_register(satype)
	u_int satype;
{
	struct sadb_msg *newmsg;
	u_int len;

	YIPSDEBUG(DEBUG_PFKEY, plog("pfkey_register", "begin register.\n"));

	/* create new sadb_msg to send. */
	len = sizeof(struct sadb_msg);

	newmsg = CALLOC(len, struct sadb_msg *);
	if (newmsg == 0) {
		plog("pfkey_register", "No more memory.\n");
		errno = ENOBUFS;
		return(0);
	}

	newmsg->sadb_msg_version = PF_KEY_V2;
	newmsg->sadb_msg_type = SADB_REGISTER;
	newmsg->sadb_msg_errno = 0;
	newmsg->sadb_msg_satype = satype;
	newmsg->sadb_msg_len = _UNIT64(len);
	newmsg->sadb_msg_seq = 0;
	newmsg->sadb_msg_pid = getpid();

	/* send message */
	if ((len = pfkey_send(newmsg, len)) < 0)
		return(0);

	free(newmsg);

	/* receive message */
	if ((newmsg = pfkey_recv()) == 0)
		return(0);

	return(newmsg);
}

/*
 * allocate a new pfkey status management buffer to handle SADB_ACQUIRE.
 * IN:
 *	msg: pointer to the SADB_ACQUIRE message.
 * OUT:
 *	0     : error occured, and set errno.
 *	others: a pointer to new allocated pfkey_st buffer.
 */
struct pfkey_st *
pfkey_acquire(msg)
	struct sadb_msg *msg;
{
	struct pfkey_st *pst = 0;
	caddr_t mhp[SADB_EXT_MAX + 1];
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("pfkey_acquire", "begin\n"));

	/* allocate buffer for status management of pfkey message */
	if ((pst = pfkey_new_st()) == 0) goto err;

	pst->satype = msg->sadb_msg_satype;
	pst->seq = msg->sadb_msg_seq;
	pst->pid = msg->sadb_msg_pid;

	memset(mhp, 0, sizeof(caddr_t) * (SADB_EXT_MAX + 1));
	if (pfkey_check(msg, mhp)) goto err;

	/* copy from acquired */
    {
	struct sadb_address *addr;
	u_int len;

	addr = (struct sadb_address *)mhp[SADB_EXT_ADDRESS_SRC];
	len = _UNUNIT64(addr->sadb_address_len);
	pst->src0 = CALLOC(len, struct sadb_address *);
	if (pst->src0 == 0) goto err;
	memcpy((caddr_t)pst->src0, (caddr_t)addr, len);
	pst->src = (struct sockaddr *)((caddr_t)pst->src0 + sizeof(*addr));

	addr = (struct sadb_address *)mhp[SADB_EXT_ADDRESS_DST];
	len = _UNUNIT64(addr->sadb_address_len);
	pst->dst0 = CALLOC(len, struct sadb_address *);
	if (pst->dst0 == 0) goto err;
	memcpy((caddr_t)pst->dst0, (caddr_t)addr, len);
	pst->dst = (struct sockaddr *)((caddr_t)pst->dst0 + sizeof(*addr));
    }

	error = 0;
end:
	return(pst);
err:
	if (pst) pfkey_free_st(pst);
	pst = 0;
	goto end;
}

/*
 * send SADB_GETSPI message to the kernel and receive a reply from kernel.
 * IN:
 *	pst:
 * OUT:
 *	0     : error occured, and set errno.
 *	others: a value of SPI is network byte order.
 */
u_int32_t
pfkey_getspi(seq, satype, src, dst)
	u_int32_t seq;
	u_int satype;
	struct sockaddr *src, *dst;
{
	struct sadb_msg *newmsg = 0;
	struct sadb_address *addr = 0;
	struct sadb_spirange *spirng = 0;
	caddr_t mhp[SADB_EXT_MAX + 1];
	u_int32_t spi = 0;
	u_int len, salen;
	caddr_t p;

	YIPSDEBUG(DEBUG_PFKEY, plog("pfkey_getspi", "begin getspi.\n"));

	/* sanity check */
	if (src == 0 || dst == 0) {
		plog("pfkey_getspi", "Invalid data was passed.\n");
		errno = ENOBUFS;
		return(0);
	}
	
	salen = _SALENBYAF(src->sa_family);	/* XXX: assuming same dst. */

	/* create new sadb_msg to reply. */
	len = sizeof(struct sadb_msg)
		+ sizeof(struct sadb_address) + _ALIGN8(salen)
		+ sizeof(struct sadb_address) + _ALIGN8(salen)
		/*+ sizeof(struct sadb_spirange)*/ ;

	newmsg = CALLOC(len, struct sadb_msg *);
	if (newmsg == 0) {
		plog("pfkey_getspi", "No more memory.\n");
		errno = ENOBUFS;
		return(0);
	}

	newmsg->sadb_msg_version = PF_KEY_V2;
	newmsg->sadb_msg_type = SADB_GETSPI;
	newmsg->sadb_msg_errno = 0;
	newmsg->sadb_msg_satype = satype;
	newmsg->sadb_msg_len = _UNIT64(len);
	newmsg->sadb_msg_seq = seq;
	newmsg->sadb_msg_pid = getpid();
	p = (caddr_t)newmsg + sizeof(struct sadb_msg);

	/* set sadb_address for source */
	p = key_setsadbaddr(p, SADB_EXT_ADDRESS_SRC,
			src->sa_family,
			_INADDRBYAF(src),
			_ALENBYAF(src->sa_family) << 3,
			0,			/* XXX proto */
			0);			/* XXX port */

	/* set sadb_address for destination */
	p = key_setsadbaddr(p, SADB_EXT_ADDRESS_DST,
			dst->sa_family,
			_INADDRBYAF(dst),
			_ALENBYAF(dst->sa_family) << 3,
			0,			/* XXX proto */
			0);			/* XXX port */

	/* XXX */
	/* proccessing spi range */

	/* send message */
	if ((len = pfkey_send(newmsg, len)) < 0)
		goto end;

	free(newmsg);

	/* receive message */
	if ((newmsg = pfkey_recv()) == 0)
		goto end;

	memset(mhp, 0, sizeof(caddr_t) * (SADB_EXT_MAX + 1));
	if (pfkey_check(newmsg, mhp))
		goto end;
	
	/* sanity check */
	if (mhp[SADB_EXT_SA] == 0) {
		plog("pfkey_getspi", "no sadb_sa found.\n");
		errno = EINVAL;
		goto end;
	}

	spi = ((struct sadb_sa *)mhp[SADB_EXT_SA])->sadb_sa_spi;

end:
	if (newmsg) free(newmsg);
	return(spi);
}

/*
 * send SADB_UPDATE message to the kernel and receive a reply from kernel.
 * IN:
 *	pst:
 * OUT:
 *	0	: success
 *	negative: fail and set errno.
 */
int
pfkey_update(seq, src, dst, isa)
	u_int32_t seq;
	struct sockaddr *src, *dst;
	struct ipsec_sa *isa;
{
	struct sadb_msg *newmsg = 0;
	u_int len, salen;
	caddr_t p;
	int error = -1;

	YIPSDEBUG(DEBUG_PFKEY, plog("pfkey_update", "begin update.\n"));

	salen = _SALENBYAF(src->sa_family);	/* XXX: assuming same dst. */

	/* create new sadb_msg to reply. */
	len = sizeof(struct sadb_msg)
		+ sizeof(struct sadb_sa)
		+ sizeof(struct sadb_address) + _ALIGN8(salen)
		+ sizeof(struct sadb_address) + _ALIGN8(salen)
		+ sizeof(struct sadb_lifetime)
		+ sizeof(struct sadb_lifetime);

	if (pfkey_aalg_type[isa->hash_t] != SADB_AALG_NONE)
		len += (sizeof(struct sadb_key) + _ALIGN8(isa->keymat->l));

	if (pfkey_ealg_type[isa->enc_t] != SADB_EALG_NONE)
		len += (sizeof(struct sadb_key) + _ALIGN8(IPSEC_ENCRYPTKEYLEN));

	newmsg = CALLOC(len, struct sadb_msg *);
	if (newmsg == 0) {
		plog("pfkey_update", "No more memory.\n");
		errno = ENOBUFS;
		return(-1);
	}

	newmsg->sadb_msg_version = PF_KEY_V2;
	newmsg->sadb_msg_type = SADB_UPDATE;
	newmsg->sadb_msg_errno = 0;
	newmsg->sadb_msg_satype = isa->proto_id;
	newmsg->sadb_msg_len = _UNIT64(len);
	newmsg->sadb_msg_seq = seq;
	newmsg->sadb_msg_pid = getpid();
	p = (caddr_t)newmsg + sizeof(struct sadb_msg);

	/* set sadb_sa */
	p = key_setsadbsa(p, isa->spi->v,
		pfkey_aalg_type[isa->hash_t], pfkey_ealg_type[isa->enc_t]);

	/* set sadb_address for source */
	p = key_setsadbaddr(p, SADB_EXT_ADDRESS_SRC,
			src->sa_family,
			_INADDRBYAF(src),
			_ALENBYAF(src->sa_family) << 3,
			0,			/* XXX proto */
			0);			/* XXX port */

	/* set sadb_address for destination */
	p = key_setsadbaddr(p, SADB_EXT_ADDRESS_DST,
			dst->sa_family,
			_INADDRBYAF(dst),
			_ALENBYAF(dst->sa_family) << 3,
			0,			/* XXX proto */
			0);			/* XXX port */

	/* set authentication algorithm, if present. */
	if (pfkey_aalg_type[isa->hash_t] != SADB_AALG_NONE) {
		p = key_setsadbkey(p, SADB_EXT_KEY_AUTH,
				isa->keymat->v, isa->keymat->l);
	}

	/* set encyption algorithm, if present. */
	if (pfkey_ealg_type[isa->enc_t] != SADB_EALG_NONE) {
		p = key_setsadbkey(p, SADB_EXT_KEY_ENCRYPT,
				isa->keymat->v, IPSEC_ENCRYPTKEYLEN);
	}

	/* set sadb_lifetime for destination */
	/* XXX HARD and SOFT same lifetime. */
	p = key_setsadblifetime(p, SADB_EXT_LIFETIME_HARD, isa->ldur);
	p = key_setsadblifetime(p, SADB_EXT_LIFETIME_SOFT, isa->ldur);

	/* send message */
	if ((len = pfkey_send(newmsg, len)) < 0)
		goto end;

	free(newmsg);

	/* receive message */
	if ((newmsg = pfkey_recv()) == 0)
		goto end;

	error = newmsg->sadb_msg_errno;

end:
	if (newmsg) free(newmsg);
	return(error);
}

/*
 * send SADB_ADD message to the kernel and receive a reply from kernel.
 * IN:
 *	pst:
 * OUT:
 *	0	: success
 *	negative: fail and set errno.
 */
int
pfkey_add(satype, src, dst, isa)
	u_int satype;
	struct sockaddr *src, *dst;
	struct ipsec_sa *isa;
{
	struct sadb_msg *newmsg = 0;
	u_int len, salen;
	caddr_t p;
	int error = -1;

	YIPSDEBUG(DEBUG_PFKEY, plog("pfkey_add", "begin add.\n"));

	salen = _SALENBYAF(src->sa_family);	/* XXX: assuming same dst. */

	/* create new sadb_msg to reply. */
	len = sizeof(struct sadb_msg)
		+ sizeof(struct sadb_sa)
		+ sizeof(struct sadb_address) + _ALIGN8(salen)
		+ sizeof(struct sadb_address) + _ALIGN8(salen)
		+ sizeof(struct sadb_lifetime)
		+ sizeof(struct sadb_lifetime);

	if (pfkey_aalg_type[isa->hash_t] != SADB_AALG_NONE)
		len += (sizeof(struct sadb_key) + _ALIGN8(isa->keymat->l));

	if (pfkey_ealg_type[isa->enc_t] != SADB_EALG_NONE)
		len += (sizeof(struct sadb_key) + _ALIGN8(IPSEC_ENCRYPTKEYLEN));

	newmsg = CALLOC(len, struct sadb_msg *);
	if (newmsg == 0) {
		plog("pfkey_add", "No more memory.\n");
		errno = ENOBUFS;
		return(-1);
	}

	newmsg->sadb_msg_version = PF_KEY_V2;
	newmsg->sadb_msg_type = SADB_ADD;
	newmsg->sadb_msg_errno = 0;
	newmsg->sadb_msg_satype = satype;
	newmsg->sadb_msg_len = _UNIT64(len);
	newmsg->sadb_msg_seq = 0;
	newmsg->sadb_msg_pid = getpid();
	p = (caddr_t)newmsg + sizeof(struct sadb_msg);

	/* set sadb_sa */
	p = key_setsadbsa(p, isa->spi_p->v,
		pfkey_aalg_type[isa->hash_t], pfkey_ealg_type[isa->enc_t]);

	/* set sadb_address for source */
	p = key_setsadbaddr(p, SADB_EXT_ADDRESS_SRC,
			src->sa_family,
			_INADDRBYAF(src),
			_ALENBYAF(src->sa_family) << 3,
			0,			/* XXX proto */
			0);			/* XXX port */

	/* set sadb_address for destination */
	p = key_setsadbaddr(p, SADB_EXT_ADDRESS_DST,
			dst->sa_family,
			_INADDRBYAF(dst),
			_ALENBYAF(dst->sa_family) << 3,
			0,			/* XXX proto */
			0);			/* XXX port */

	/* set authentication algorithm, if present. */
	if (pfkey_aalg_type[isa->hash_t] != SADB_AALG_NONE) {
		p = key_setsadbkey(p, SADB_EXT_KEY_AUTH,
				isa->keymat->v, isa->keymat->l);
	}

	/* set encyption algorithm, if present. */
	if (pfkey_ealg_type[isa->enc_t] != SADB_EALG_NONE) {
		p = key_setsadbkey(p, SADB_EXT_KEY_ENCRYPT,
				isa->keymat->v, IPSEC_ENCRYPTKEYLEN);
	}

	/* set sadb_lifetime for destination. */
	/* XXX HARD and SOFT same lifetime. */
	p = key_setsadblifetime(p, SADB_EXT_LIFETIME_HARD, isa->ldur);
	p = key_setsadblifetime(p, SADB_EXT_LIFETIME_SOFT, isa->ldur);

	/* send message */
	if ((len = pfkey_send(newmsg, len)) < 0)
		goto end;

	free(newmsg);

	/* receive message */
	if ((newmsg = pfkey_recv()) == 0)
		goto end;

	error = newmsg->sadb_msg_errno;

end:
	if (newmsg) free(newmsg);
	return(error);
}

/*
 * allocate a new pfkey status management buffer to handle SADB_EXPIRE
 * IN:
 *	msg: pointer to the SADB_EXPIRE message.
 * OUT:
 *	0     : error occured, and set errno.
 *	others: a pointer to new allocated pfkey_st buffer.
 */
struct pfkey_st *
pfkey_expire(msg)
	struct sadb_msg *msg;
{
	struct pfkey_st *pst = 0;
	caddr_t mhp[SADB_EXT_MAX + 1];
	int error = -1;

	YIPSDEBUG(DEBUG_STAMP, plog("pfkey_expire", "begin\n"));

	/* allocate buffer for status management of pfkey message */
	if ((pst = pfkey_new_st()) == 0) goto err;

	pst->satype = msg->sadb_msg_satype;
	pst->seq = msg->sadb_msg_seq;
	pst->pid = msg->sadb_msg_pid;

	memset(mhp, 0, sizeof(caddr_t) * (SADB_EXT_MAX + 1));
	if (pfkey_check(msg, mhp)) goto err;

	/* XXX ignore HARD lifetime */
	if (mhp[SADB_EXT_LIFETIME_HARD] != 0) {
		plog("pfkey_expire", "ignore HARD lifetime.\n");
		goto err;
	}

	/* copy from expire */
    {
	struct sadb_address *addr;
	u_int len;

	addr = (struct sadb_address *)mhp[SADB_EXT_ADDRESS_SRC];
	len = _UNUNIT64(addr->sadb_address_len);
	pst->src0 = CALLOC(len, struct sadb_address *);
	if (pst->src0 == 0) goto err;
	memcpy((caddr_t)pst->src0, (caddr_t)addr, len);
	pst->src = (struct sockaddr *)((caddr_t)pst->src0 + sizeof(*addr));

	addr = (struct sadb_address *)mhp[SADB_EXT_ADDRESS_DST];
	len = _UNUNIT64(addr->sadb_address_len);
	pst->dst0 = CALLOC(len, struct sadb_address *);
	if (pst->dst0 == 0) goto err;
	memcpy((caddr_t)pst->dst0, (caddr_t)addr, len);
	pst->dst = (struct sockaddr *)((caddr_t)pst->dst0 + sizeof(*addr));
    }

	error = 0;
end:
	return(pst);
err:
	if (pst) pfkey_free_st(pst);
	pst = 0;
	goto end;
}

/*
 * check basic usage for sadb_msg,
 * and set the pointer to each header in this message buffer.
 * This routine is derived from netkey/key.c in KAME.
 * IN:	msg: pointer to message buffer.
 *	mhp: pointer to the buffer allocated of each header.
 * OUT:	0 if success.
 *	other if error, return errno.
 */
int
pfkey_check(msg, mhp)
	struct sadb_msg *msg;
	caddr_t *mhp;
{
	/* check version */
	if (msg->sadb_msg_version != PF_KEY_V2) {
		printf("key_check: PF_KEY version %lu is too old.\n",
		    msg->sadb_msg_version);
		return(EINVAL);
	}

	/* check errno */
	if (msg->sadb_msg_errno != 0) {
		printf("key_check: %s in kernel.\n", strerror(errno));
		return(msg->sadb_msg_errno);
	}

	/* check type */
	if (msg->sadb_msg_type > SADB_MAX) {
		printf("key_check: invalid type %u is passed.\n",
		    msg->sadb_msg_type);
		return(EINVAL);
	}

	/* check SA type */
	switch (msg->sadb_msg_satype) {
	case SADB_SATYPE_UNSPEC:
		if (msg->sadb_msg_type == SADB_FLUSH
		 || msg->sadb_msg_type == SADB_DUMP
		 || msg->sadb_msg_type == SADB_X_SPDADD
		 || msg->sadb_msg_type == SADB_X_SPDDELETE
		 || msg->sadb_msg_type == SADB_X_SPDDUMP
		 || msg->sadb_msg_type == SADB_X_SPDFLUSH)
			break;
		else
			printf("key_check: type UNSPEC is invalid.\n");
		break;
	case SADB_SATYPE_AH:
	case SADB_SATYPE_ESP:
		break;
	case SADB_SATYPE_RSVP:
	case SADB_SATYPE_OSPFV2:
	case SADB_SATYPE_RIPV2:
	case SADB_SATYPE_MIP:
		printf("key_check: type %u isn't supported.\n",
		    msg->sadb_msg_satype);
		return(EOPNOTSUPP);
	default:
		printf("key_check: invalid type %u is passed.\n",
		    msg->sadb_msg_satype);
		return(EINVAL);
	}

	mhp[0] = (caddr_t)msg;

    {
	struct sadb_ext *ext;
	int tlen, extlen;

	tlen = _UNUNIT64(msg->sadb_msg_len) - sizeof(struct sadb_msg);
	ext = (struct sadb_ext *)((caddr_t)msg + sizeof(struct sadb_msg));

	while (tlen > 0) {
		/* duplicate check */
		/* XXX: Are the KEY_AUTH and KEY_ENCRYPT ? */
		if (mhp[ext->sadb_ext_type] != 0) {
			printf("key_check: duplicate ext_type %u is passed.\n",
				ext->sadb_ext_type);
			return(EINVAL);
		}

		/* set pointer */
		switch (ext->sadb_ext_type) {
		case SADB_EXT_SA:
		case SADB_EXT_LIFETIME_CURRENT:
		case SADB_EXT_LIFETIME_HARD:
		case SADB_EXT_LIFETIME_SOFT:
		case SADB_EXT_ADDRESS_SRC:
		case SADB_EXT_ADDRESS_DST:
		case SADB_EXT_ADDRESS_PROXY:
		case SADB_EXT_KEY_AUTH:
			/* must to be chek weak keys. */
		case SADB_EXT_KEY_ENCRYPT:
			/* must to be chek weak keys. */
		case SADB_EXT_IDENTITY_SRC:
		case SADB_EXT_IDENTITY_DST:
		case SADB_EXT_SENSITIVITY:
		case SADB_EXT_PROPOSAL:
		case SADB_EXT_SUPPORTED_AUTH:
		case SADB_EXT_SUPPORTED_ENCRYPT:
		case SADB_EXT_SPIRANGE:
		case SADB_X_EXT_POLICY:
			mhp[ext->sadb_ext_type] = (caddr_t)ext;
			break;
		default:
			printf("key_check: invalid ext_type %u is passed.\n", ext->sadb_ext_type);
			return(EINVAL);
		}

		extlen = _UNUNIT64(ext->sadb_ext_len);
		tlen -= extlen;
		ext = (struct sadb_ext *)((caddr_t)ext + extlen);
	}
    }

	/* check field of upper layer protocol and address family */
	if (mhp[SADB_EXT_ADDRESS_SRC] && mhp[SADB_EXT_ADDRESS_DST]) {
		struct sadb_address *src0, *dst0;
		struct sockaddr *src, *dst;

		src0 = (struct sadb_address *)(mhp[SADB_EXT_ADDRESS_SRC]);
		dst0 = (struct sadb_address *)(mhp[SADB_EXT_ADDRESS_DST]);
		src = (struct sockaddr *)((caddr_t)src0 + sizeof(*src0));
		dst = (struct sockaddr *)((caddr_t)dst0 + sizeof(*dst0));

		if (src0->sadb_address_proto != dst0->sadb_address_proto) {
			printf("key_check: upper layer protocol mismatched.\n");
			return(EINVAL);
		}

		if (src->sa_family != dst->sa_family) {
			printf("key_check: address family mismatched.\n");
			return(EINVAL);
		}
		if (src->sa_family != AF_INET
		 && src->sa_family != AF_INET6) {
			printf("key_check: invalid address family.\n");
			return(EINVAL);
		}
	}

	return(0);
}

/* %%%: basic routines */
/*
 * open a socket.
 * OUT:
 *	-1: fail.
 *	others : success and return value of socket.
 */
int
pfkey_open()
{
	if ((so_PK = socket(PF_KEY, SOCK_RAW, PF_KEY_V2)) < 0)
		return(-1);

	return(so_PK);
}

/*
 * close a socket.
 * OUT:
 *	 0: success.
 *	-1: fail.
 */
void
pfkey_close()
{
	(void)close(so_PK);

	return;
}

/*
 * receive sadb_msg data, and return pointer to new buffer allocated.
 * Must free this buffer later.
 * OUT:
 *    0     : error occured.
 *    others: a pointer to sadb_msg structure.
 */
struct sadb_msg *
pfkey_recv()
{
	struct sadb_msg buf, *newmsg;
	int len, reallen;

	while ((len = recv(so_PK, (caddr_t)&buf, sizeof(buf), MSG_PEEK)) < 0) {
		if (errno == EINTR) continue;
		return(0);
	}

	if (len < sizeof(buf)) {
		recv(so_PK, (caddr_t)&buf, sizeof(buf), 0);
		errno = EINVAL;
		return(0);
	}

	/* read real message */
	reallen = _UNUNIT64(buf.sadb_msg_len);
	if ((newmsg = CALLOC(reallen, struct sadb_msg *)) == 0)
		return(0);

	while ((len = recv(so_PK, (caddr_t)newmsg, reallen, 0)) < 0) {
		if (errno == EINTR) continue;
		return(0);
	}

	if (len != reallen) {
		errno = EINVAL;
		return(0);
	}

	YIPSDEBUG(DEBUG_NET,
	    plog("pfkey_recv", "message has been received, cmd=%d len=%d\n",
	    newmsg->sadb_msg_type, newmsg->sadb_msg_len));
	YIPSDEBUG(DEBUG_DNET, kdebug_sadb(newmsg));

	return(newmsg);
}

/*
 * send message to a socket.
 * OUT:
 *	 others: success and return length to send.
 *	-1     : fail.
 */
int
pfkey_send(msg, len)
	struct sadb_msg *msg;
	int len;
{
	if ((len = send(so_PK, (caddr_t)msg, len, 0)) < 0)
		return(-1);

	YIPSDEBUG(DEBUG_NET,
		plog("pfkey_send", "message has been sent, len=%d\n", len));
	YIPSDEBUG(DEBUG_DNET, kdebug_sadb(msg));

	return(len);
}

/* %%% pfkey status management routines */
/*
 * create new status record of pfkey message
 */
struct pfkey_st *
pfkey_new_st()
{
	struct pfkey_st *pst;

	/* create new status */
	if ((pst = CALLOC(sizeof(*pst), struct pfkey_st *)) == 0) {
		plog("pfkey_new_st", "malloc (%s)\n", strerror(errno)); 
		return(0);
	}

	return(pst);
}

/*
 * free record from pfkey stauts table
 */
void
pfkey_free_st(st)
	struct pfkey_st *st;
{
	/* XXX: free more !? */
	if (st->sc) sched_kill(st->sc);
	if (st->src0) free(st->src0);
	if (st->dst0) free(st->dst0);

	(void)free(st);

	return;
}

/*
 * set sadb_address
 */
static caddr_t
key_setsadbaddr(buf, type, af, addr, pref, proto, port)
	caddr_t buf, addr;
	u_int type, af, pref, proto, port;
{
	struct sadb_address addr0;
	caddr_t p = buf; /* save */
	u_int len = sizeof(struct sadb_address) + _ALIGN8(_SALENBYAF(af));

	addr0.sadb_address_len = _UNIT64(len);
	addr0.sadb_address_exttype = type;
	addr0.sadb_address_proto = proto;
	addr0.sadb_address_prefixlen = pref;
	memcpy(p, (caddr_t)&addr0, sizeof(struct sadb_address));

	p += sizeof(struct sadb_address);

	switch (af) {
	case AF_INET:
	{
		struct sockaddr_in in;

		bzero((caddr_t)&in, sizeof(in));
		in.sin_len = sizeof(in);
		in.sin_family = af;
		in.sin_port = port;
		memcpy((caddr_t)&in.sin_addr, addr, _ALENBYAF(af));
		memcpy(p, (caddr_t)&in, in.sin_len);
	}
		break;
#ifdef INET6
	case AF_INET6:
	{
		struct sockaddr_in6 in6;

		bzero((caddr_t)&in6, sizeof(in6));
		in6.sin6_len = sizeof(in6);
		in6.sin6_family = af;
		in6.sin6_port = port;
		memcpy((caddr_t)&in6.sin6_addr, addr, _ALENBYAF(af));
		memcpy(p, (caddr_t)&in6, in6.sin6_len);
	}
#endif /* INET6 */
		break;
	}

	return(buf + len);
}

/*
 * set sadb_key
 */
static caddr_t
key_setsadbkey(buf, type, key, keylen)
	caddr_t buf, key;
	u_int type, keylen;
{
	struct sadb_key m_key;
	caddr_t p = buf; /* save */
	u_int vkeylen = _ALIGN8(keylen);
	u_int len = sizeof(struct sadb_key) + vkeylen;

	m_key.sadb_key_len = _UNIT64(len);
	m_key.sadb_key_exttype = type;
	m_key.sadb_key_bits = keylen * 8;
	m_key.sadb_key_reserved = 0;

	memcpy(p, (caddr_t)&m_key, sizeof(struct sadb_key));
	p += sizeof(struct sadb_key);
	memcpy(p, key, keylen);
	p += vkeylen;

	return(buf + len);
}

/*
 * set sadb_sa
 */
static caddr_t
key_setsadbsa(buf, spi, aalg, ealg)
	caddr_t buf, spi;
	u_int aalg, ealg;
{
	struct sadb_sa *sa;
	u_int len = sizeof(struct sadb_sa);

	sa = (struct sadb_sa *)buf;
	sa->sadb_sa_len = _UNIT64(len);
	sa->sadb_sa_exttype = SADB_EXT_SA;
	memcpy((caddr_t)&sa->sadb_sa_spi, spi, sizeof(sa->sadb_sa_spi));
	sa->sadb_sa_replay = 0;
	sa->sadb_sa_state = 0;
	sa->sadb_sa_auth = aalg;
	sa->sadb_sa_encrypt = ealg;

	return(buf + len);
}

static caddr_t
key_setsadblifetime(buf, type, time)
	caddr_t buf;
	u_int type;
	u_int32_t time;
{
	struct sadb_lifetime *lft;
	u_int len = sizeof(struct sadb_lifetime);

	lft = (struct sadb_lifetime *)buf;
	lft->sadb_lifetime_len = _UNIT64(len);
	lft->sadb_lifetime_exttype = type;
	lft->sadb_lifetime_allocations = 0;
	lft->sadb_lifetime_bytes = 0;
	lft->sadb_lifetime_addtime = time;
	lft->sadb_lifetime_usetime = 0;

	return(buf + len);
}

