/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by WIDE Project and
 *    its contributors.
 * 4. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* YIPS @(#)$Id: session.c,v 1.1.1.1.2.1 1998/07/01 01:41:42 sakane Exp $ */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>

#include "var.h"
#include "extern.h"
#include "debug.h"

int session(void);
void signal_handler(int);
int close_sockets(void);

extern int isakmp_handler(int);
extern void isakmp_close(void);
extern int pfkey_handler(void);
extern int pfkey_close(void);
extern int admin_handler(void);
extern int admin_close(void);

int
session()
{
	static fd_set mask0;
	int nfds, i;
	fd_set rfds;
	struct timeval timeout, *tm;
	int num = 0;

	FD_ZERO(&mask0);

	FD_SET(sockA, &mask0);
	FD_SET(sockP, &mask0);

	nfds = (sockA > sockP ? sockA : sockP);
	for (i = 0; i < n_soI; i++) {
		FD_SET(sockIs[i], &mask0);
		nfds = (nfds > sockIs[i] ? nfds : sockIs[i]);
	}
	nfds++;

	/* initialize */
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
	tm = &timeout;

	while (1) {
		rfds = mask0;

		while (select(nfds, &rfds, (fd_set *)0, (fd_set *)0, tm) < 0) {
			YIPSDEBUG(DEBUG_NET,
			    plog("session", "return select() with result-code=%d\n", errno));
			switch (errno) {
			case EINTR: continue;
			default:
				plog("session", "select (%s)\n", strerror(errno));
				return(-1);
			}
		}

		if(FD_ISSET(sockA, &rfds)) {
			admin_handler();
		}

		for (i = 0; i < n_soI; i++) {
			if(FD_ISSET(sockIs[i], &rfds))
				isakmp_handler(sockIs[i]);
		}

		if(FD_ISSET(sockP, &rfds)) {
			pfkey_handler();
		}

		num = schedular(num);
		if (num > 0) {
			timeout.tv_sec = 1; /* XXX */
			timeout.tv_usec = 0;
			tm = &timeout;
		} else
		if (num == 0)
			tm = 0;
		else {
			plog("session", "error in scheduling\n");
			return(-1);
		}
	}
}

int signals[] = {
SIGHUP, SIGINT, SIGQUIT, SIGILL,
SIGTRAP, SIGABRT, SIGEMT, SIGFPE,
SIGKILL, SIGBUS, /*SIGSEGV,*/ SIGSYS,
SIGPIPE, SIGALRM, SIGTERM, SIGTSTP,
SIGTTIN, SIGTTOU, SIGUSR1, SIGUSR2,
0
};

void
signal_handler(sig)
	int sig;
{
	int i;

	switch (sig) {
	case 0:
		for (i = 0; signals[i] != 0; i++)
			if (set_signal(signals[i], signal_handler) < 0)
				exit(-1);
		break;

	case SIGHUP:
		if (re_cfparse() != 0)
			exit(1);
		break;

	default:
		plog("signal_handler", "caught signal, %d\n", sig);
		close_sockets();
		exit(0);
		break;
	}
}

int
set_signal(sig, func)
	int sig;
	void (*func)();
{
	struct sigaction sa;

	memset((caddr_t)&sa, 0, sizeof(sa));
	sa.sa_handler = func;
	sa.sa_flags = SA_RESTART;

	if (sigemptyset(&sa.sa_mask) < 0) {
		perror("sigemptyset");
		return(-1);
	}

	sigaction(sig, &sa, (struct sigaction *)0);

	return(0);
}

int
close_sockets()
{
	isakmp_close();
	(void)pfkey_close();
	(void)admin_close();
}

