/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by WIDE Project and
 *    its contributors.
 * 4. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* YIPS @(#)$Id: ipsec_doi.c,v 1.1.1.1.2.2 1998/07/02 06:59:35 sakane Exp $ */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>

#include <netinet/in.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>

#include "var.h"
#include "vmbuf.h"
#include "misc.h"
#include "cfparse.h"
#include "isakmp.h"
#include "ipsec_doi.h"
#include "oakley.h"
#include "extern.h"
#include "debug.h"

char *oakley_modp768;
char *oakley_modp1024;
int oakley_modp768_len;
int oakley_modp1024_len;

int isakmp_port_type = 0;
int isakmp_id_type = IPSECDOI_ID_IPV4_ADDR;

u_int32 ipsecdoi_set_ldur(int type, vchar_t *);
vchar_t *ipsecdoi_make_mysa(struct isakmp_cf_sa *, u_int32);

/*
 * validate SA payload.
 */
int
ipsecdoi_sap2isa(isa, sap)
	caddr_t isa;
	struct isakmp_pl_sa *sap;
{
	int tlen, proplen;
	caddr_t bp;
	u_int32 doi, sit;
	int error = -1;

	bp = (caddr_t)sap;
	tlen = sap->h.len;

	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_sap2isa", "total SA len=%d\n", tlen));
	YIPSDEBUG(DEBUG_DSA, pdump(bp, tlen, YDUMP_HEX));

	doi = ntohl(sap->doi);
	if (doi != IPSEC_DOI) {
		plog("ipsecdoi_sap2isa", "invalid value of DOI 0x%08x.\n", doi);
		goto end;
	}

	sit = ntohl(sap->sit);
	switch (sit) {
	case IPSECDOI_SIT_IDENTITY_ONLY:
		break;

	case IPSECDOI_SIT_SECRECY:
	case IPSECDOI_SIT_INTEGRITY:
		plog("ipsecdoi_sap2isa",
		    "situation 0x%08x unsupported yet.\n", sit);
		goto end;

	default:
		plog("ipsecdoi_sap2isa", "invalid situation 0x%08x.\n", sit);
		goto end;
	}

	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_sap2isa", "doi=%d, sit=0x%08x\n", doi, sit));

	bp += sizeof(*sap);
	tlen -= sizeof(*sap);

	while (tlen > 0) {
		/* get proposal */
		if ((proplen = ipsecdoi_get_prop(bp, isa)) < 0) goto end;

		tlen -= proplen;
		bp += proplen;

		/* XXX */
	}

	error = 0;

end:
	return(error);
}

/*
 *   explore proposal payload.
 *   return length of proposal payload.
 */
int
ipsecdoi_get_prop(bp, isa)
	caddr_t bp, isa;
{
	struct isakmp_pl_p *prop;
	int tlen, proplen, trnslen;
	int error = -1;

	prop = (struct isakmp_pl_p *)bp;
	proplen = tlen = ntohs(prop->h.len);

	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_get_prop", " total proposal len=%d\n", tlen));

	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_get_prop",
	        "prop#=%d, prot-id=%d, spi-size=%d, #trns=%d\n",
	        prop->p_no, prop->prot_id, prop->spi_size, prop->num_t));

	bp += sizeof(*prop);
	tlen -= sizeof(*prop);

	/* allocate buffer for SA */
	switch (prop->prot_id) {
	case IPSECDOI_PROTO_ISAKMP:
		/* save spi */
		if ((((struct oakley_sa *)isa)->spi
		        = vmalloc(prop->spi_size)) == 0) {
			plog("ipsecdoi_get_prop", "vmalloc (%s)\n", strerror(errno));
			goto end;
		}
		memcpy(((struct oakley_sa *)isa)->spi->v, bp, prop->spi_size);

		YIPSDEBUG(DEBUG_SA,
		    plog("ipsecdoi_get_prop", "spi=");
			pvdump(((struct oakley_sa *)isa)->spi));

		/* save proto_id */
		((struct oakley_sa *)isa)->proto_id = prop->prot_id;

		break;

	case IPSECDOI_PROTO_IPSEC_AH:
	case IPSECDOI_PROTO_IPSEC_ESP:
		/* save spi */
		if ((((struct ipsec_sa *)isa)->spi
		        = vmalloc(prop->spi_size)) == 0) {
			plog("ipsecdoi_get_prop", "vmalloc (%s)\n", strerror(errno));
			goto end;
		}
		memcpy(((struct ipsec_sa *)isa)->spi->v, bp, prop->spi_size);

		YIPSDEBUG(DEBUG_SA,
		    plog("ipsecdoi_get_prop", "spi=");
			pvdump(((struct ipsec_sa *)isa)->spi));

		/* save proto_id */
		((struct ipsec_sa *)isa)->proto_id = prop->prot_id;

		break;

	case IPSECDOI_PROTO_IPCOMP:
		plog("ipsecdoi_get_prop", "IPCOMP isn't supported yet.\n");
		break;

	default:
		plog("ipsecdoi_get_prop",
		    "invalid protocol id %d.\n", prop->prot_id);
		break;
	}

	bp += prop->spi_size;
	tlen -= prop->spi_size;

	while (tlen > 0) {

		/* get transform */
		switch (prop->prot_id) {
		case IPSECDOI_PROTO_ISAKMP:
			YIPSDEBUG(DEBUG_SA,
			    plog("ipsecdoi_get_prop", "protocol OAKLEY\n"));
			if ((trnslen = ipsecdoi_get_isakmp(bp, isa)) < 0)
				 goto end;
			break;

		case IPSECDOI_PROTO_IPSEC_AH:
			YIPSDEBUG(DEBUG_SA,
			    plog("ipsecdoi_get_prop", "protocol AH\n"));
			if ((trnslen = ipsecdoi_get_ah(bp, isa)) < 0)
				 goto end;
			break;

		case IPSECDOI_PROTO_IPSEC_ESP:
			YIPSDEBUG(DEBUG_SA,
			    plog("ipsecdoi_get_prop", "protocol ESP\n"));
			if ((trnslen = ipsecdoi_get_esp(bp, isa)) < 0)
				 goto end;
			break;

		case IPSECDOI_PROTO_IPCOMP:
			plog("ipsecdoi_get_prop",
			    "IPCOMP %d isn't supported yet.\n", prop->prot_id);
			break;

		default:
			plog("ipsecdoi_get_prop",
			    "invalid protocol id %d.\n", prop->prot_id);
			break;
		}

		tlen -= trnslen;
		bp += trnslen;
	}

	error = proplen;

end:
	return(error);
}

/*
 *   explore AH transform payload.
 *   return length of transform payload.
 */
int
ipsecdoi_get_ah(bp, isa)
	caddr_t bp, isa;
{
	struct isakmp_pl_t *trns;
	struct ipsec_sa *sa = (struct ipsec_sa *)isa;
	int tlen, trnslen;
	int error = -1;

	trns = (struct isakmp_pl_t *)bp;
	trnslen = tlen = ntohs(trns->h.len);

	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_get_ah", "total transform len=%d\n", tlen));
	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_get_ah",
	        "trns#=%d, trns-id=%d\n", trns->t_no, trns->t_id));

	/* save transform id */
	sa->t_id = trns->t_id;

	switch (trns->t_id) {
	case IPSECDOI_AH_MD5:
		YIPSDEBUG(DEBUG_SA, plog("ipsecdoi_get_ah", "transform MD5\n"));
		break;

	case IPSECDOI_AH_SHA:
		YIPSDEBUG(DEBUG_SA, plog("ipsecdoi_get_ah", "transform SHA\n"));
		break;

	case IPSECDOI_AH_DES:
		plog("ipsecdoi_get_ah", "AH_DES %d isn't supported.\n", trns->t_id);
		break;

	default:
		plog("ipsecdoi_get_ah", "invalid transform-id %d.\n", trns->t_id);
		goto end;
	}

	bp += sizeof(*trns);
	tlen -= sizeof(*trns);

	if (!isakmp_co_pluto) {
		if (ipsecdoi_get_attr(bp, isa, tlen))
			goto end;
	} else {
		if (ipsecdoi_get_pluto_attr(bp, isa, tlen))
			goto end;
	}

	error = trnslen;

end:
	return(error);
}

/*
 *   explore ESP transform payload.
 *   return length of transform payload.
 */
int
ipsecdoi_get_esp(bp, isa)
	caddr_t bp, isa;
{
	struct isakmp_pl_t *trns;
	struct ipsec_sa *sa = (struct ipsec_sa *)isa;
	int tlen, trnslen;
	int error = -1;

	trns = (struct isakmp_pl_t *)bp;
	trnslen = tlen = ntohs(trns->h.len);

	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_get_esp", "total transform len=%d\n", tlen));
	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_get_esp",
	        "trns#=%d, trns-id=%d\n", trns->t_no, trns->t_id));

	/* save transform id */
	sa->t_id = trns->t_id;

	switch (trns->t_id) {
	case IPSECDOI_ESP_NULL:
		YIPSDEBUG(DEBUG_SA,
		    plog("ipsecdoi_get_esp", "transform NULL\n"));
		break;
	case IPSECDOI_ESP_DES:
		YIPSDEBUG(DEBUG_SA,
		    plog("ipsecdoi_get_esp", "transform DES\n"));
		break;
	case IPSECDOI_ESP_3DES:
		YIPSDEBUG(DEBUG_SA,
		    plog("ipsecdoi_get_esp", "transform 3DES\n"));
		break;
	case IPSECDOI_ESP_DES_IV32:
		YIPSDEBUG(DEBUG_SA,
		    plog("ipsecdoi_get_esp", "transform DES IV32\n"));
		break;
	case IPSECDOI_ESP_DES_IV64:
		YIPSDEBUG(DEBUG_SA,
		    plog("ipsecdoi_get_esp", "transform DES IV64\n"));
		break;
	case IPSECDOI_ESP_RC5:
	case IPSECDOI_ESP_IDEA:
	case IPSECDOI_ESP_CAST:
	case IPSECDOI_ESP_BLOWFISH:
	case IPSECDOI_ESP_3IDEA:
	case IPSECDOI_ESP_ARCFOUR:
		plog("ipsecdoi_get_esp",
		    "ESP transform %d isn't supported.\n", trns->t_id);
		break;
	default:
		plog("ipsecdoi_get_esp", "invalid transform-id %d.\n", trns->t_id);
		goto end;
	}

	bp += sizeof(*trns);
	tlen -= sizeof(*trns);

	if (!isakmp_co_pluto) {
		if (ipsecdoi_get_attr(bp, isa, tlen))
			goto end;
	} else {
		if (ipsecdoi_get_pluto_attr(bp, isa, tlen))
			goto end;
	}

	error = trnslen;

end:
	return(error);
}

static char *ipsec_attr_name[] = {
"",
"SA Life Type",
"SA Life Duration",
"Group Description",
"Encription Mode",
"Authentication Algorithm",
"Key Length",
"Key Rounds",
"Compression Dictionary Size",
"Compression Private Algorithm"
};

int
ipsecdoi_get_attr(bp, isa, tlen)
	caddr_t bp, isa;
	int tlen;
{
	struct isakmp_data *d;
	struct ipsec_sa *sa = (struct ipsec_sa *)isa;
	int flag, type, lorv;
	vchar_t *lbuf = 0;
	int error = -1;

	d = (struct isakmp_data *)bp;

	/* default */
	sa->life_t = IPSECDOI_ATTR_SA_LTYPE_DEFAULT;

	while (tlen > 0) {

		type = ntohs(d->type) & ~ISAKMP_GEN_MASK;
		flag = ntohs(d->type) & ISAKMP_GEN_MASK;
		lorv = ntohs(d->lorv);

		YIPSDEBUG(DEBUG_DSA,
		    plog("ipsecdoi_get_attr",
		        "type=%d, flag=0x%04x, lorv=0x%04x\n", type, flag, lorv));

		switch (type) {
		case IPSECDOI_ATTR_ENC_MODE:
			switch (lorv) {
			case IPSECDOI_ATTR_ENC_MODE_TUNNEL:
				YIPSDEBUG(DEBUG_SA,
				    plog("ipsecdoi_get_attr", "Tunnel mode\n"));
				break;
			case IPSECDOI_ATTR_ENC_MODE_TRNS:
				YIPSDEBUG(DEBUG_SA,
				    plog("ipsecdoi_get_attr", "Transport mode\n"));
				break;
			default:
				plog("ipsecdoi_get_attr",
				    "invalid encryption mode %d.\n", lorv);
				goto end;
			}
			sa->enc_m = (u_int8)lorv;
			break;

		case IPSECDOI_ATTR_AUTH:
			switch (lorv) {
			case IPSECDOI_ATTR_AUTH_HMAC_MD5:
				YIPSDEBUG(DEBUG_SA,
				    plog("ipsecdoi_get_attr", "HMAC-MD5\n"));
				break;
			case IPSECDOI_ATTR_AUTH_HMAC_SHA1:
				YIPSDEBUG(DEBUG_SA,
				    plog("ipsecdoi_get_attr", "HMAC-SHA1\n"));
				break;
			case IPSECDOI_ATTR_AUTH_DES_MAC:
			case IPSECDOI_ATTR_AUTH_KPDK:
				plog("ipsecdoi_get_attr",
				    "ignore authentication algorithm %d isn't supported.\n", lorv);
				break;
			default:
				plog("ipsecdoi_get_attr",
				    "invalid authentication algorithm %d.\n", lorv);
				goto end;
			}
			sa->hash_t = (u_int8)lorv;
			break;

		case IPSECDOI_ATTR_SA_LTYPE:
			if (! flag) goto end;
			switch (lorv) {
			case IPSECDOI_ATTR_SA_LTYPE_SEC:
				YIPSDEBUG(DEBUG_SA,
				    plog("ipsecdoi_get_attr", "SEC life type\n"));
				break;
			case IPSECDOI_ATTR_SA_LTYPE_KB:
			    plog("ipsecdoi_get_attr",
				    "ignore KB life type isn't supported.\n");
				break;
			default:
				plog("ipsecdoi_get_attr", "invalid life type %d.\n", lorv);
				goto end;
			}
			sa->life_t = (u_int8)lorv;
			break;

		case IPSECDOI_ATTR_SA_LDUR:
			if (flag) {
				if ((lbuf = vmalloc(sizeof(d->lorv))) == 0) {
					plog("ipsecdoi_get_attr",
					    "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				lorv = htons(lorv);
				memcpy(lbuf->v, (char *)&lorv, sizeof(d->lorv));
			} else { /* i.e. ISAKMP_GEN_TLV */
				if ((lbuf = vmalloc(lorv)) == 0) {
					plog("ipsecdoi_get_attr", "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				memcpy(lbuf->v, d + sizeof(*d), lorv);
			}
			break;

		case IPSECDOI_ATTR_GRP_DESC:
		case IPSECDOI_ATTR_KEY_LENGTH:
		case IPSECDOI_ATTR_KEY_ROUNDS:
		case IPSECDOI_ATTR_COMP_DICT_SIZE:
		case IPSECDOI_ATTR_COMP_PRIVALG:
			if (type < IPSECDOI_ATTR_COMP_PRIVALG)
				plog("ipsecdoi_get_attr",
				    "ignore attribute type %d [%s] isn't supported.\n",
				    type, ipsec_attr_name[type]);
			else
				plog("ipsecdoi_get_attr",
				    "ignore attribute type %d isn't supported.\n", type);
			break;

		default:
			plog("ipsecdoi_get_isakmp", "invalid attribute type %d.\n", type);
			goto end;
		}

		if (flag) {
			tlen -= sizeof(*d);
			d = (struct isakmp_data *)((char *)d + sizeof(*d));
		} else {
			tlen -= (sizeof(*d) + lorv);
			d = (struct isakmp_data *)((char *)d + sizeof(*d) + lorv);
		}
	}

	/* check life type and life duration, and make life time of SA */
	if (lbuf)
		sa->ldur = ipsecdoi_set_ldur(sa->life_t, lbuf);
	else
		sa->ldur = IPSECDOI_ATTR_SA_LDUR_DEFAULT;

	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_get_attr", "life duration 0x%x\n", sa->ldur));

	error = 0;

end:
	if (lbuf) vfree(lbuf);

	return(error);
}

static char *isakmp_attr_name[] = {
"",
"Encryption Algorithm",
"Hash Algorithm",
"Authentication Method",
"Group Description",
"Group Type",
"Group Prime/Irreducible Polynomial",
"Group Generator One",
"Group Generator Two",
"Group Curve A",
"Group Curve B",
"Life Type",
"Life Duration",
"PRF",
"Key Length",
"Field Size",
"Group Order",
};

/*
 *   explore OAKLEY transform payload.
 *   return length of transform payload.
 */
int
ipsecdoi_get_isakmp(bp, isa)
	caddr_t bp, isa;
{
	struct oakley_sa *sa = (struct oakley_sa *)isa;
	struct isakmp_pl_t *trns;
	struct isakmp_data *d;
	int tlen, trnslen;
	u_int flag, type, lorv;
	vchar_t *lbuf = 0;
	int error = -1;

	/* default */
	sa->life_t = OAKLEY_ATTR_SA_LTYPE_DEFAULT;

	trns = (struct isakmp_pl_t *)bp;
	trnslen = tlen = ntohs(trns->h.len);

	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_get_isakmp", "total transform len=%d\n", tlen));
	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_get_isakmp", "trns#=%d, trns-id=%d\n",
	        trns->t_no, trns->t_id));

	switch (trns->t_id) {
	case IPSECDOI_KEY_OAKLEY:
		break;
	default:
		plog("ipsecdoi_get_isakmp", "invalid transform-id %d.\n", trns->t_id);
		goto end;
	}

	bp += sizeof(*trns);
	tlen -= sizeof(*trns);

	d = (struct isakmp_data *)bp;

	while (tlen > 0) {

		type = ntohs(d->type) & ~ISAKMP_GEN_MASK;
		flag = ntohs(d->type) & ISAKMP_GEN_MASK;
		lorv = ntohs(d->lorv);

		YIPSDEBUG(DEBUG_DSA,
		    plog("ipsecdoi_get_isakmp", "type=%d, flag=0x%04x, lorv=0x%04x\n",
		        type, flag, lorv));

		switch (type) {
		case OAKLEY_ATTR_ENC_ALG:
			if (! flag) goto end;
			switch (lorv) {
			case OAKLEY_ATTR_ENC_ALG_DES:
				YIPSDEBUG(DEBUG_SA, plog("ipsecdoi_get_isakmp", "DES\n"));
				break;
			case OAKLEY_ATTR_ENC_ALG_3DES:
				YIPSDEBUG(DEBUG_SA, plog("ipsecdoi_get_isakmp", "3DES\n"));
				break;
			case OAKLEY_ATTR_ENC_ALG_IDEA:
			case OAKLEY_ATTR_ENC_ALG_BL:
			case OAKLEY_ATTR_ENC_ALG_RC5:
			case OAKLEY_ATTR_ENC_ALG_CAST:
				plog("ipsecdoi_get_isakmp",
				    "encryption algorithm %d isn't supported.\n", lorv);
				goto end;
			default:
				plog("ipsecdoi_get_isakmp",
				    "invalied encryption algorithm %d.\n", lorv);
				goto end;
			}
			sa->enc_t = (u_int8)lorv;
			break;

		case OAKLEY_ATTR_HASH_ALG:
			if (! flag) goto end;
			switch (lorv) {
			case OAKLEY_ATTR_HASH_ALG_MD5:
				YIPSDEBUG(DEBUG_SA, plog("ipsecdoi_get_isakmp", "MD5\n"));
				break;
			case OAKLEY_ATTR_HASH_ALG_SHA:
				YIPSDEBUG(DEBUG_SA, plog("ipsecdoi_get_isakmp", "SHA1\n"));
				break;
			case OAKLEY_ATTR_HASH_ALG_TIGER:
				plog("ipsecdoi_get_isakmp",
				    "hash algorithm %d isn't supported.\n", lorv);
				goto end;
			default:
				plog("ipsecdoi_get_isakmp",
				    "invalid hash algorithm %d.\n", lorv);
				goto end;
			}
			sa->hash_t = (u_int8)lorv;
			break;

		case OAKLEY_ATTR_AUTH_METHOD:
			if (! flag) goto end;
			switch (lorv) {
			case OAKLEY_ATTR_AUTH_METHOD_PSKEY:
				YIPSDEBUG(DEBUG_SA, plog("ipsecdoi_get_isakmp", "pre-shared key\n"));
				break;
			case OAKLEY_ATTR_AUTH_METHOD_DSS:
			case OAKLEY_ATTR_AUTH_METHOD_RSA:
			case OAKLEY_ATTR_AUTH_METHOD_RSAENC:
			case OAKLEY_ATTR_AUTH_METHOD_RSAREV:
				plog("ipsecdoi_get_isakmp",
				    "authentication method %d isn't supported.\n", lorv);
				goto end;
			default:
				plog("ipsecdoi_get_isakmp",
				    "invalid authentication method %d.\n", lorv);
				goto end;
			}
			sa->auth_t = (u_int8)lorv;
			break;

		case OAKLEY_ATTR_GRP_DESC:
			if (! flag) goto end;
			switch (lorv) {
			case OAKLEY_ATTR_GRP_DESC_MODP768:
				YIPSDEBUG(DEBUG_SA, plog("ipsecdoi_get_isakmp", "MODP 768\n"));
				break;
			case OAKLEY_ATTR_GRP_DESC_MODP1024:
				YIPSDEBUG(DEBUG_SA, plog("ipsecdoi_get_isakmp", "MODP 1024\n"));
				break;
			case OAKLEY_ATTR_GRP_DESC_EC2N155:
			case OAKLEY_ATTR_GRP_DESC_EC2N185:
				plog("ipsecdoi_get_isakmp",
				    "DH group %d isn't supported.\n", lorv);
				goto end;
			default:
				plog("ipsecdoi_get_isakmp", "invalid DH group %d.\n", lorv);
				goto end;
			}
			sa->dhgrp = (u_int8)lorv;
			break;

		case OAKLEY_ATTR_SA_LTYPE:
			if (! flag) goto end;
			switch (lorv) {
			case OAKLEY_ATTR_SA_LTYPE_SEC:
				YIPSDEBUG(DEBUG_SA,
				    plog("ipsecdoi_get_isakmp", "SEC life type\n"));
				break;
			case OAKLEY_ATTR_SA_LTYPE_KB:
			    plog("ipsecdoi_get_isakmp", "KB life type isn't supported.\n");
				goto end;
			default:
				plog("ipsecdoi_get_isakmp", "invalid life type %d.\n", lorv);
				goto end;
			}
			sa->life_t = (u_int8)lorv;
			break;

		case OAKLEY_ATTR_SA_LDUR:
			if (flag) {
				if ((lbuf = vmalloc(sizeof(d->lorv))) == 0) {
					plog("ipsecdoi_get_isakmp", "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				lorv = htons(lorv);
				memcpy(lbuf->v, (char *)&lorv, sizeof(d->lorv));
			} else { /* i.e. ISAKMP_GEN_TLV */
				if ((lbuf = vmalloc(lorv)) == 0) {
					plog("ipsecdoi_get_isakmp", "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				memcpy(lbuf->v, d + sizeof(*d), lorv);
			}
			break;

		case OAKLEY_ATTR_PRF:
			if (! flag) goto end;
			YIPSDEBUG(DEBUG_SA, plog("ipsecdoi_get_isakmp", "PRF\n"));
			sa->prf_t = (u_int8)lorv;
			break;

		case OAKLEY_ATTR_KEY_LEN:
            plog("ipsecdoi_get_isakmp",
                "Key Length was received. ignored.\n");
            break;

		case OAKLEY_ATTR_GRP_TYPE:
		case OAKLEY_ATTR_GRP_PI:
		case OAKLEY_ATTR_GRP_GEN_ONE:
		case OAKLEY_ATTR_GRP_GEN_TWO:
		case OAKLEY_ATTR_GRP_CURVE_A:
		case OAKLEY_ATTR_GRP_CURVE_B:

		case OAKLEY_ATTR_FIELD_SIZE:
		case OAKLEY_ATTR_GRP_ORDER:
			if (type < OAKLEY_ATTR_GRP_ORDER)
				plog("ipsecdoi_get_isakmp",
				    "attribute type %d [%s] isn't supported.\n",
				    type, isakmp_attr_name[type]);
			else
				plog("ipsecdoi_get_isakmp",
				    "attribute type %d isn't supported.\n", type);
			goto end;

		default:
			plog("ipsecdoi_get_isakmp", "invalid attribute type %d.\n", type);
			goto end;
		}

		if (flag) {
			tlen -= sizeof(*d);
			d = (struct isakmp_data *)((char *)d + sizeof(*d));
		} else {
			tlen -= (sizeof(*d) + lorv);
			d = (struct isakmp_data *)((char *)d + sizeof(*d) + lorv);
		}
	}

	/* check life type and life duration, and make life time of SA */
	if (lbuf)
		sa->ldur = ipsecdoi_set_ldur(sa->life_t, lbuf);
	else
		sa->ldur = OAKLEY_ATTR_SA_LDUR_DEFAULT;

	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_get_isakmp", "life duration 0x%08x\n", sa->ldur));

	error = trnslen;

end:
	if (lbuf) vfree(lbuf);

	return(error);
}

u_int32
ipsecdoi_set_ldur(type, buf)
	int type;
	vchar_t *buf;
{
	u_int32 ldur;

	if (type == 0 || buf == 0) return(0);

	switch (buf->l) {
	case 2:
		ldur = ntohs(*(u_int16 *)buf->v);
		break;
	case 4:
		ldur = ntohl(*(u_int32 *)buf->v);
		break;
	default:
		plog("ipsecdoi_set_ldur", "length %d of life duration isn't supported.\n", buf->l);
		return(0);
	}

	return(ldur);
}

#if 0
/*
 * make ID payload minus general header into isakmp sa.
 */
vchar_t *
ipsecdoi_make_id(type)
	int type;
{
	int len, idlen;
	vchar_t *id = 0;
	int error = -1;

	/* XXX */
	char *str = "127.0.0.1";
	struct in_addr in;

	switch (type) {
	case IPSECDOI_ID_IPV4_ADDR:
		idlen = 4;
		break;
	case IPSECDOI_ID_IPV4_ADDR_SUBNET:
		idlen = 8;
		break;
	case IPSECDOI_ID_IPV6_ADDR:
	case IPSECDOI_ID_IPV4_ADDR_RANGE:
		idlen = 16;
		break;
	case IPSECDOI_ID_IPV6_ADDR_SUBNET:
	case IPSECDOI_ID_IPV6_ADDR_RANGE:
		idlen = 32;
		break;
	case IPSECDOI_ID_FQDN:
	case IPSECDOI_ID_USER_FQDN:
	case IPSECDOI_ID_DER_ASN1_DN:
	case IPSECDOI_ID_DER_ASN1_GN:
	case IPSECDOI_ID_KEY_ID:
		idlen = 0;
		plog("ipsecdoi_make_id", "ID type %d isn't supported.\n", type);
		goto end;
	default:
		plog("ipsecdoi_make_id", "Invalied ID type %d.\n", type);
		goto end;
	}

	len = sizeof(struct isakmp_pl_id) + idlen - sizeof(struct isakmp_gen);
	if ((id = vmalloc(len)) == 0) {
		plog("ipsecdoi_make_id", "vmalloc (%s)\n", strerror(errno));
		goto end;
	}

	id->v[0] = type;

	if (isakmp_port_type)
		id->v[1] = 17; /* udp */
	else
		id->v[1] = 0;

	if (!isakmp_co_pluto)
		*(u_int16 *)&id->v[2] = htons(portI);
	else
		*(u_int16 *)&id->v[2] = 0;

	/* XXX */
	inet_aton(str, &in);
	memcpy(id->v + 4, (char *)&in.s_addr, idlen);

	error = 0;

end:
	if (error) {
		if (id) {
			vfree(id);
			id = 0;
		}
	}

	return(id);
}
#endif

#if 0
/* ipsecdoi_ipsec_mysa
 *   The order of values is network byte order.
 */
int
ipsecdoi_ipsec_mysa()
{
	struct isakmp_pl_sa *sa;
	struct isakmp_pl_p *prop;
	struct isakmp_pl_t *trns;
	struct isakmp_data *data;
	char *p;
	int tlen, num_p;
	int error = -1;

	num_p = 1;

	tlen = 0x34;

	if ((ipsec_mysa = vmalloc(tlen)) == 0) {
		plog("ipsecdoi_ipsec_mysa", "vmalloc (%s)\n", strerror(errno)); 
		return(-1);
	}
	p = ipsec_mysa->v;

	/* create SA payload */
	sa = (struct isakmp_pl_sa *)p;
	sa->h.np = ISAKMP_NPTYPE_NONE;
	sa->h.len = htons(tlen);
	sa->doi = htonl(IPSEC_DOI);
	sa->sit = htonl(IPSECDOI_SIT_IDENTITY_ONLY);
	p += sizeof(*sa);

	/* create 1st proposal */
	prop = (struct isakmp_pl_p *)p;
	prop->h.np     = ISAKMP_NPTYPE_NONE;
	prop->h.len    = htons(0x28);
	prop->p_no     = 1;
	prop->prot_id  = IPSECDOI_PROTO_IPSEC_ESP;
	prop->spi_size = 8;
	prop->num_t    = 1;
	p += sizeof(*prop);

	*(u_int32 *)p = htonl(0x00000101);
	p += 4;
	*(u_int32 *)p = htonl(0);
	p += 4;

	trns = (struct isakmp_pl_t *)p;
	trns->h.np  = ISAKMP_NPTYPE_NONE;
	trns->h.len = htons(0x18);
	trns->t_no  = 1;
	trns->t_id  = IPSECDOI_ESP_3DES;
	p += sizeof(*trns);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | IPSECDOI_ATTR_SA_LTYPE);
	data->lorv = htons(IPSECDOI_ATTR_SA_LTYPE_SEC);
	p += sizeof(*data);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | IPSECDOI_ATTR_SA_LDUR);
	data->lorv = htons(0x001e);
	p += sizeof(*data);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | IPSECDOI_ATTR_ENC_MODE);
	data->lorv = htons(IPSECDOI_ATTR_ENC_MODE_TRNS);
	p += sizeof(*data);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | IPSECDOI_ATTR_AUTH);
	data->lorv = htons(IPSECDOI_ATTR_AUTH_HMAC_SHA1);
	p += sizeof(*data);

end:
	return(0);
}
#endif

vchar_t *
ipsecdoi_make_mysa(cf_sa, spi)
	struct isakmp_cf_sa *cf_sa;
	u_int32 spi;
{
	vchar_t *mysa;
	struct isakmp_cf_p *cf_p;
	struct isakmp_cf_t *cf_t;
	struct isakmp_pl_sa *sa;
	struct isakmp_pl_p *prop;
	struct isakmp_pl_t *trns;
	struct isakmp_pl_data *data;
	u_int8 *np_p = 0, *np_t = 0;
	caddr_t p;

	if ((mysa = vmalloc(cf_sa->len)) == 0) {
		plog("ipsecdoi_make_mysa", "vmalloc (%s)\n", strerror(errno)); 
		return(0);
	}
	p = mysa->v;

	/* create SA payload */
	sa = (struct isakmp_pl_sa *)p;
	sa->h.np = ISAKMP_NPTYPE_NONE;
	sa->h.len = htons(cf_sa->len);
	sa->doi = cf_sa->doi;
	sa->sit = cf_sa->sit;
	p += sizeof(*sa);

	for (cf_p = cf_sa->p; cf_p; cf_p = cf_p->next) {

		if (np_p) *np_p = ISAKMP_NPTYPE_P;

		/* create proposal */
		prop = (struct isakmp_pl_p *)p;
		prop->h.np     = ISAKMP_NPTYPE_NONE;
		prop->h.len    = htons(cf_p->len);
		prop->p_no     = cf_p->p_no;
		prop->prot_id  = cf_p->prot_id;
		prop->spi_size = cf_p->spi_size;
		prop->num_t    = cf_p->num_t;
		p += sizeof(*prop);

		if (cf_p->spi_size != 0) {
			if (cf_p->spi == 0) {
				plog("ipsecdoi_make_mysa", "no spi exists, why ?\n");
				free(mysa);
				return(0);
			} else
			if (cf_p->spi_size != cf_p->spi->l) {
				plog("ipsecdoi_make_mysa", "spi length mismatch, why ?\n");
				free(mysa);
				return(0);
			}

			if (spi) {
				spi = htonl(spi);
				memcpy(p, (char *)&spi, sizeof(spi));
			} else
				memcpy(p, cf_p->spi->v, cf_p->spi_size);

			p += cf_p->spi_size;
		}

		for (cf_t = cf_p->t; cf_t; cf_t = cf_t->next) {

			if (np_t) *np_t = ISAKMP_NPTYPE_T;

			/* create transform */
			trns = (struct isakmp_pl_t *)p;
			trns->h.np  = ISAKMP_NPTYPE_NONE;
			trns->h.len = htons(cf_t->len);
			trns->t_no  = cf_t->t_no;
			trns->t_id  = cf_t->t_id;
			p += sizeof(*trns);

			memcpy(p, cf_t->data->v, cf_t->data->l);
			p += cf_t->data->l;

			/* save buffer to pre-next payload */
			np_t = &trns->h.np;
		}

		/* save buffer to pre-next payload */
		np_p = &prop->h.np;
		np_t = 0;
	}

	return(mysa);
}

#if 0
/* oakley_isakmp_mysa
 *   see 3.4 Security Association Payload
 *   The order of values is network byte order.
 */
int
ipsecdoi_isakmp_mysa()
{
	struct isakmp_pl_sa *sa;
	struct isakmp_pl_p *prop;
	struct isakmp_pl_t *trns;
	struct isakmp_data *data;
	char *p;
	int tlen, num_p;
	int error = -1;

	num_p = 1;

#if 0
	/* create the storage of the pointer of my proposal */
	if ((myp = (vchar_t **)calloc(num_p, sizeof(vchar_t **))) == 0) goto end;

	/* create my proposal */
	for (i = 0; i < num_p; i++) {
		if ((myp[i] = (vchar_t *)vmalloc(sizeof(vchar_t *))) == 0) {
			while (i--) 
				vfree(i);
			goto end;
		}
	}
#endif

	/* XXX: must modify below. */
/*
0000003c
 00000001 00000001
00000030
 01010801
 00000100 00000000
00000020
 01010000
 80010001 80020002 80030001 80040001 800b0001 800c0e10
*/
	tlen = 0x3c;

	if ((mysa = vmalloc(tlen)) == 0) {
		plog("ipsecdoi_isakmp_mysa", "vmalloc (%s)\n", strerror(errno)); 
		return(-1);
	}
	p = mysa->v;

	/* create SA payload */
	sa = (struct isakmp_pl_sa *)p;
	sa->h.np = ISAKMP_NPTYPE_NONE;
	sa->h.len = htons(tlen);
	sa->doi = htonl(IPSEC_DOI);
	sa->sit = htonl(IPSECDOI_SIT_IDENTITY_ONLY);
	p += sizeof(*sa);

	/* create 1st proposal */
	prop = (struct isakmp_pl_p *)p;
	prop->h.np     = ISAKMP_NPTYPE_NONE;
	prop->h.len    = htons(0x30);
	prop->p_no     = 1;
	prop->prot_id  = IPSECDOI_PROTO_ISAKMP;
	prop->num_t    = 1;

	/* XXX: why is there SPI ? Pluto is incurrect ? */
	prop->spi_size = 8;
	p += sizeof(*prop);

	*(u_int32 *)p = htonl(0x00000100);
	p += 4;
	*(u_int32 *)p = htonl(0);
	p += 4;

	trns = (struct isakmp_pl_t *)p;
	trns->h.np  = ISAKMP_NPTYPE_NONE;
	trns->h.len = htons(0x20);
	trns->t_no  = 1;
	trns->t_id  = IPSECDOI_KEY_OAKLEY;
	p += sizeof(*trns);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | OAKLEY_ATTR_ENC_ALG);
	data->lorv = htons(OAKLEY_ATTR_ENC_ALG_DES);
	p += sizeof(*data);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | OAKLEY_ATTR_HASH_ALG);
	data->lorv = htons(OAKLEY_ATTR_HASH_ALG_SHA);
	p += sizeof(*data);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | OAKLEY_ATTR_AUTH_METHOD);
	data->lorv = htons(OAKLEY_ATTR_AUTH_METHOD_PSKEY);
	p += sizeof(*data);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | OAKLEY_ATTR_GRP_DESC);
	data->lorv = htons(OAKLEY_ATTR_GRP_DESC_MODP768);
	p += sizeof(*data);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | OAKLEY_ATTR_SA_LTYPE);
	data->lorv = htons(OAKLEY_ATTR_SA_LTYPE_SEC);
	p += sizeof(*data);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | OAKLEY_ATTR_SA_LDUR);
	data->lorv = htons(0x001e);
	p += sizeof(*data);
	
end:
	return(0);
}
#endif

/*
 * for pluto
 */
#include "pluto.h"
int
ipsecdoi_get_pluto_attr(bp, isa, tlen)
	caddr_t bp, isa;
	int tlen;
{
	struct isakmp_data *d;
	struct ipsec_sa *sa = (struct ipsec_sa *)isa;
	int flag, type, lorv;
	vchar_t *lbuf = 0;
	int error = -1;

	d = (struct isakmp_data *)bp;

	/* default */
	sa->life_t = IPSECDOI_ATTR_SA_LTYPE_DEFAULT;

	while (tlen > 0) {

		type = ntohs(d->type) & ~ISAKMP_GEN_MASK;
		flag = ntohs(d->type) & ISAKMP_GEN_MASK;
		lorv = ntohs(d->lorv);

		YIPSDEBUG(DEBUG_DSA,
		    plog("ipsecdoi_get_pluto_attr", "type=%d, flag=0x%04x, lorv=0x%04x\n",
		        type, flag, lorv));

		switch (type) {
		case ENCAPSULATION_MODE:
			switch (lorv) {
			case ENCAPSULATION_MODE_TUNNEL:
				YIPSDEBUG(DEBUG_SA,
				    plog("ipsecdoi_get_pluto_attr", "Tunnel mode\n"));
				break;
			case ENCAPSULATION_MODE_TRANSPORT:
				YIPSDEBUG(DEBUG_SA,
				    plog("ipsecdoi_get_pluto_attr", "Transport mode\n"));
				break;
			default:
				plog("ipsecdoi_get_pluto_attr", "invalid encryption mode %d.\n", lorv);
				goto end;
			}
			sa->enc_t = (u_int8)lorv;
			break;

		case HMAC_ALGORITHM:
			switch (lorv) {
			case HMAC_ALGORITHM_MD5:
				YIPSDEBUG(DEBUG_SA,
				    plog("ipsecdoi_get_pluto_attr", "HMAC-MD5\n"));
				break;
			case HMAC_ALGORITHM_SHA1:
				YIPSDEBUG(DEBUG_SA,
				    plog("ipsecdoi_get_pluto_attr", "HMAC-SHA1\n"));
				break;
			default:
				plog("ipsecdoi_get_pluto_attr", "invalid authentication algorithm %d.\n", lorv);
				goto end;
			}
			sa->hash_t = (u_int8)lorv;
			break;

		case SA_LIFE_TYPE:
			if (! flag) goto end;
			switch (lorv) {
			case SA_LIFE_TYPE_SECONDS:
				YIPSDEBUG(DEBUG_SA,
				    plog("ipsecdoi_get_pluto_attr", "SEC life type\n"));
				break;
			case SA_LIFE_TYPE_KBYTES:
				YIPSDEBUG(DEBUG_SA,
				    plog("ipsecdoi_get_pluto_attr", "KB life type isn't supported.\n"));
			default:
				plog("ipsecdoi_get_pluto_attr", "invalid life type %d.\n", lorv);
				goto end;
			}
			sa->life_t = (u_int8)lorv;
			break;

		case SA_LIFE_DURATION:
			if (flag) {
				if ((lbuf = vmalloc(sizeof(d->lorv))) == 0) {
					plog("ipsecdoi_get_pluto_attr", "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				lorv = htons(lorv);
				memcpy(lbuf->v, (char *)&lorv, sizeof(d->lorv));
			} else { /* i.e. ISAKMP_GEN_TLV */
				if ((lbuf = vmalloc(lorv)) == 0) {
					plog("ipsecdoi_get_pluto_attr", "vmalloc (%s)\n", strerror(errno));
					goto end;
				}
				memcpy(lbuf->v, d + sizeof(*d), lorv);
			}
			break;

		case REPLAY_PROTECTION:
			YIPSDEBUG(DEBUG_SA, plog("ipsecdoi_get_pluto_attr", "ignore sign of replay protection\n"));
			break;

		case AUTH_KEY_LIFE_TYPE:
		case AUTH_KEY_LIFE_DURATION:
		case ENC_KEY_LIFE_TYPE:
		case ENC_KEY_LIFE_DURATION:
		case GROUP_DESCRIPTION:
		case CA_DISTINGUISHED_NAME:
			plog("ipsecdoi_get_pluto_attr", "attribute type %d isn't supported.\n", type);
			goto end;
			break;

		default:
			plog("ipsecdoi_get_pluto_attr", "invalid attribute type %d.\n", type);
			goto end;
		}

		if (flag) {
			tlen -= sizeof(*d);
			d = (struct isakmp_data *)((char *)d + sizeof(*d));
		} else {
			tlen -= (sizeof(*d) + lorv);
			d = (struct isakmp_data *)((char *)d + sizeof(*d) + lorv);
		}
	}

	/* check life type and life duration, and make life time of SA */
	if (lbuf)
		sa->ldur = ipsecdoi_set_ldur(sa->life_t, lbuf);
	else
		sa->ldur = IPSECDOI_ATTR_SA_LDUR_DEFAULT;

	YIPSDEBUG(DEBUG_SA,
	    plog("ipsecdoi_get_pluto_attr", "life duration 0x%x\n", sa->ldur));

	error = 0;

end:
	if (lbuf) vfree(lbuf);

	return(error);
}

#if 0
/* ipsecdoi_ipsec_pluto_mysa
 *   The order of values is network byte order.
 */
int
ipsecdoi_ipsec_pluto_mysa()
{
	struct isakmp_pl_sa *sa;
	struct isakmp_pl_p *prop;
	struct isakmp_pl_t *trns;
	struct isakmp_data *data;
	char *p;
	int tlen, num_p;
	int error = -1;

	num_p = 1;

	/* XXX: must modify below. */
/* 
0a000038
 00000001 00000001
0000002c
 01030801
 00000101 00000000
0000001c
 01030000
 80050001 80067080 80070001 800a0002 800b0001
*/
	tlen = 0x38;

	if ((ipsec_mysa = vmalloc(tlen)) == 0) {
		plog("ipsecdoi_ipsec_pluto_mysa", "vmalloc (%s)\n", strerror(errno)); 
		return(-1);
	}
	p = ipsec_mysa->v;

	/* create SA payload */
	sa = (struct isakmp_pl_sa *)p;
	sa->h.np = ISAKMP_NPTYPE_NONE;
	sa->h.len = htons(tlen);
	sa->doi = htonl(IPSEC_DOI);
	sa->sit = htonl(IPSECDOI_SIT_IDENTITY_ONLY);
	p += sizeof(*sa);

	/* create 1st proposal */
	prop = (struct isakmp_pl_p *)p;
	prop->h.np     = ISAKMP_NPTYPE_NONE;
	prop->h.len    = htons(0x2c);
	prop->p_no     = 1;
	prop->prot_id  = IPSECDOI_PROTO_IPSEC_ESP;
	prop->spi_size = 8;
	prop->num_t    = 1;
	p += sizeof(*prop);

	*(u_int32 *)p = htonl(0x00000101);
	p += 4;
	*(u_int32 *)p = htonl(0);
	p += 4;

	trns = (struct isakmp_pl_t *)p;
	trns->h.np  = ISAKMP_NPTYPE_NONE;
	trns->h.len = htons(0x1c);
	trns->t_no  = 1;
	trns->t_id  = IPSECDOI_ESP_3DES;
	p += sizeof(*trns);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | SA_LIFE_TYPE);
	data->lorv = htons(SA_LIFE_TYPE_SECONDS);
	p += sizeof(*data);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | SA_LIFE_DURATION);
	data->lorv = htons(0x7080);
	p += sizeof(*data);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | REPLAY_PROTECTION);
	data->lorv = htons(REPLAY_PROTECTION_REQUIRED);
	p += sizeof(*data);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | ENCAPSULATION_MODE);
	data->lorv = htons(ENCAPSULATION_MODE_TRANSPORT);
	p += sizeof(*data);

	data = (struct isakmp_data *)p;
	data->type = htons(ISAKMP_GEN_TV | HMAC_ALGORITHM);
	data->lorv = htons(HMAC_ALGORITHM_MD5);
	p += sizeof(*data);
	
end:
	return(0);
}
#endif

