/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* YIPS @(#)$Id: handler.c,v 1.25 2000/06/28 05:59:32 sakane Exp $ */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "var.h"
#include "misc.h"
#include "vmbuf.h"
#include "plog.h"
#include "sockmisc.h"
#include "debug.h"

#include "schedule.h"
#include "algorithm.h"
#include "policy.h"
#include "proposal.h"
#include "isakmp_var.h"
#include "isakmp.h"
#include "isakmp_inf.h"
#include "oakley.h"
#include "remoteconf.h"
#include "handler.h"

static LIST_HEAD(_ph1tree_, ph1handle) ph1tree;
static LIST_HEAD(_ph2tree_, ph2handle) ph2tree;

/*
 * functions about management of the isakmp status table
 */
/* %%% management phase 1 handler */
/*
 * search for isakmpsa handler with isakmp index.
 */
struct ph1handle *
getph1byindex(index)
	isakmp_index *index;
{
	struct ph1handle *p;

	LIST_FOREACH(p, &ph1tree, chain) {
		if (memcmp(&p->index, index, sizeof(*index)) == 0)
			return p;
	}

	return NULL;
}

/*
 * search for isakmp handler by i_ck in index.
 */
struct ph1handle *
getph1byindex0(index)
	isakmp_index *index;
{
	struct ph1handle *p;

	LIST_FOREACH(p, &ph1tree, chain) {
		if (memcmp(&p->index, index, sizeof(cookie_t)) == 0)
			return p;
	}

	return NULL;
}

/*
 * search for isakmpsa handler by remote address.
 * don't use port number to search because this function search
 * with phase 2's destinaion.
 */
struct ph1handle *
getph1byaddr(remote)
	struct sockaddr *remote;
{
	struct ph1handle *p;

	LIST_FOREACH(p, &ph1tree, chain) {
		if (cmpsaddrwop(remote, p->remote) == 0)
			return p;
	}

	return NULL;
}

/*
 * dump isakmp-sa
 */
vchar_t *
dumpph1()
{
	struct ph1handle *iph1;
	struct ph1dump *pd;
	int cnt = 0;
	vchar_t *buf;

	/* get length of buffer */
	LIST_FOREACH(iph1, &ph1tree, chain)
		cnt++;

	buf = vmalloc(cnt * sizeof(struct ph1dump));
	if (buf == NULL) {
		plog(logp, LOCATION, NULL,
			"failed to get buffer\n");
		return NULL;
	}
	pd = (struct ph1dump *)buf->v;

	LIST_FOREACH(iph1, &ph1tree, chain) {
		memcpy(&pd->index, &iph1->index, sizeof(iph1->index));
		pd->status = iph1->status;
		pd->side = iph1->side;
		memcpy(&pd->remote, iph1->remote, iph1->remote->sa_len);
		memcpy(&pd->local, iph1->local, iph1->local->sa_len);
		pd->version = iph1->version;
		pd->etype = iph1->etype;
		pd->created = iph1->created;
		pd->ph2cnt = iph1->ph2cnt;
		pd++;
	}

	return buf;
}

/*
 * create new isakmp Phase 1 status record to handle isakmp in Phase1
 */
struct ph1handle *
newph1()
{
	struct ph1handle *iph1;

	/* create new iph1 */
	iph1 = CALLOC(sizeof(*iph1), struct ph1handle *);
	if (iph1 == NULL)
		return NULL;

	iph1->status = PHASE1ST_SPAWN;

	return iph1;
}

/*
 * delete new isakmp Phase 1 status record to handle isakmp in Phase1
 */
void
delph1(iph1)
	struct ph1handle *iph1;
{
	if (iph1->remote) {
		free(iph1->remote);
		iph1->remote = NULL;
	}
	if (iph1->local) {
		free(iph1->local);
		iph1->local = NULL;
	}

	if (iph1->authstr) {
		vfree(iph1->authstr);
		iph1->authstr = NULL;
	}

	if (iph1->sce)
		SCHED_KILL(iph1->sce);
	if (iph1->scr)
		SCHED_KILL(iph1->scr);

	if (iph1->dhpriv) {
		vfree(iph1->dhpriv);
		iph1->dhpriv = NULL;
	}
	if (iph1->dhpub) {
		vfree(iph1->dhpub);
		iph1->dhpub = NULL;
	}
	if (iph1->dhpub_p) {
		vfree(iph1->dhpub_p);
		iph1->dhpub_p = NULL;
	}
	if (iph1->dhgxy) {
		vfree(iph1->dhgxy);
		iph1->dhgxy = NULL;
	}
	if (iph1->nonce) {
		vfree(iph1->nonce);
		iph1->nonce = NULL;
	}
	if (iph1->nonce_p) {
		vfree(iph1->nonce_p);
		iph1->nonce_p = NULL;
	}
	if (iph1->skeyid) {
		vfree(iph1->skeyid);
		iph1->skeyid = NULL;
	}
	if (iph1->skeyid_d) {
		vfree(iph1->skeyid_d);
		iph1->skeyid_d = NULL;
	}
	if (iph1->skeyid_a) {
		vfree(iph1->skeyid_a);
		iph1->skeyid_a = NULL;
	}
	if (iph1->skeyid_e) {
		vfree(iph1->skeyid_e);
		iph1->skeyid_e = NULL;
	}
	if (iph1->key) {
		vfree(iph1->key);
		iph1->key = NULL;
	}
	if (iph1->hash) {
		vfree(iph1->hash);
		iph1->hash = NULL;
	}
	if (iph1->sig) {
		vfree(iph1->sig);
		iph1->sig = NULL;
	}
	if (iph1->sig_p) {
		vfree(iph1->sig_p);
		iph1->sig_p = NULL;
	}
	if (iph1->cert) {
		vfree(iph1->cert);
		iph1->cert = NULL;
	}
	if (iph1->cert_p) {
		vfree(iph1->cert_p);
		iph1->cert_p = NULL;
	}
	if (iph1->crl_p) {
		vfree(iph1->crl_p);
		iph1->crl_p = NULL;
	}
	if (iph1->id) {
		vfree(iph1->id);
		iph1->id = NULL;
	}
	if (iph1->id_p) {
		vfree(iph1->id_p);
		iph1->id_p = NULL;
	}

	if (iph1->ivm) {
		oakley_delivm(iph1->ivm);
		iph1->ivm = NULL;
	}

	if (iph1->sa) {
		vfree(iph1->sa);
		iph1->sa = NULL;
	}
	if (iph1->sa_ret) {
		vfree(iph1->sa_ret);
		iph1->sa_ret = NULL;
	}

	free(iph1);
}

/*
 * create new isakmp Phase 1 status record to handle isakmp in Phase1
 */
int
insph1(iph1)
	struct ph1handle *iph1;
{
	/* validity check */
	if (iph1->remote == NULL) {
		plog(logp, LOCATION, NULL,
			"invalid isakmp SA handler. no remote address.\n");
		return -1;
	}
	LIST_INSERT_HEAD(&ph1tree, iph1, chain);

	return 0;
}

void
remph1(iph1)
	struct ph1handle *iph1;
{
	LIST_REMOVE(iph1, chain);
}

/*
 * flush isakmp-sa
 */
void
flushph1()
{
	struct ph1handle *p, *next;

	for (p = LIST_FIRST(&ph1tree); p; p = next) {
		next = LIST_NEXT(p, chain);
		remph1(p);
		delph1(p);
	}
}

void
initph1tree()
{
	LIST_INIT(&ph1tree);
}

/* %%% management phase 2 handler */
/*
 * search ph2handle with policyindex.
 */
#if 0
struct ph2handle *
getph2byspidx(spidx)
	struct policyindex *spidx;
{
	struct ph2handle *p;

	LIST_FOREACH(p, &ph2tree, chain) {
		/*
		 * there are ph2handle independent on policy
		 * such like informational exchange.
		 */
		if (p->spidx == NULL)
			continue;
		if (cmpspidx(spidx, p->spidx) == 0)
			return p;
	}

	return NULL;
}
#endif

/*
 * search ph2handle with policy id.
 */
struct ph2handle *
getph2byspid(spid)
      u_int32_t spid;
{
	struct ph2handle *p;

	LIST_FOREACH(p, &ph2tree, chain) {
		/*
		 * there are ph2handle independent on policy
		 * such like informational exchange.
		 */
		if (p->spid == spid)
			return p;
	}

	return NULL;
}

/*
 * search ph2handle with sequence number.
 */
struct ph2handle *
getph2byseq(seq)
	u_int32_t seq;
{
	struct ph2handle *p;

	LIST_FOREACH(p, &ph2tree, chain) {
		if (p->seq == seq)
			return p;
	}

	return NULL;
}

/*
 * search ph2handle with message id.
 */
struct ph2handle *
getph2bymsgid(iph1, msgid)
	struct ph1handle *iph1;
	u_int32_t msgid;
{
	struct ph2handle *p;

	LIST_FOREACH(p, &ph2tree, chain) {
		if (p->msgid == msgid)
			return p;
	}

	return NULL;
}

/*
 * call by pk_recvexpire().
 */
struct ph2handle *
getph2bysaidx(src, dst, proto_id, spi)
	struct sockaddr *src, *dst;
	u_int proto_id;
	u_int32_t spi;
{
	struct ph2handle *iph2;
	struct saproto *pr;

	LIST_FOREACH(iph2, &ph2tree, chain) {
		if (iph2->approval == NULL)
			continue;
		for (pr = iph2->approval->head; pr != NULL; pr = pr->next) {
			if (proto_id != pr->proto_id)
				break;
			if (spi == pr->spi || spi == pr->spi_p)
				return iph2;
		}
	}

	return NULL;
}

/*
 * create new isakmp Phase 2 status record to handle isakmp in Phase2
 */
struct ph2handle *
newph2()
{
	struct ph2handle *iph2 = NULL;

	/* create new iph2 */
	iph2 = CALLOC(sizeof(*iph2), struct ph2handle *);
	if (iph2 == NULL)
		return NULL;

	iph2->status = PHASE1ST_SPAWN;

	return iph2;
}

/* initialize ph2handle */
void
initph2(iph2)
	struct ph2handle *iph2;
{
	/* NOTE: don't initialize src/dst */

	if (iph2->sce)
		SCHED_KILL(iph2->sce);
	if (iph2->scr)
		SCHED_KILL(iph2->scr);

	/* clear spi, keep variables in the proposal */
	if (iph2->proposal) {
		struct saproto *pr;
		for (pr = iph2->proposal->head; pr != NULL; pr = pr->next)
			pr->spi = 0;
	}

	/* clear approval */
	if (iph2->approval) {
		flushsaprop(iph2->approval);
		iph2->approval = NULL;
	}

	if (iph2->pfsgrp) {
		oakley_dhgrp_free(iph2->pfsgrp);
		iph2->pfsgrp = NULL;
	}

	if (iph2->dhpriv) {
		vfree(iph2->dhpub_p);
		iph2->dhpub_p = NULL;
	}
	if (iph2->dhpub) {
		vfree(iph2->dhpub_p);
		iph2->dhpub_p = NULL;
	}
	if (iph2->dhgxy) {
		vfree(iph2->dhgxy);
		iph2->dhgxy = NULL;
	}
	if (iph2->id) {
		vfree(iph2->id);
		iph2->id = NULL;
	}
	if (iph2->id_p) {
		vfree(iph2->id_p);
		iph2->id_p = NULL;
	}
	if (iph2->nonce_p) {
		vfree(iph2->nonce_p);
		iph2->nonce_p = NULL;
	}

	if (iph2->sa) {
		vfree(iph2->sa);
		iph2->sa = NULL;
	}
	if (iph2->sa_ret) {
		vfree(iph2->sa_ret);
		iph2->sa_ret = NULL;
	}

	if (iph2->ivm) {
		oakley_delivm(iph2->ivm);
		iph2->ivm = NULL;
	}
}

/*
 * delete new isakmp Phase 2 status record to handle isakmp in Phase2
 */
void
delph2(iph2)
	struct ph2handle *iph2;
{
	initph2(iph2);

	if (iph2->proposal) {
		flushsaprop(iph2->proposal);
		iph2->proposal = NULL;
	}

	if (iph2->src) {
		free(iph2->src);
		iph2->src = NULL;
	}
	if (iph2->dst) {
		free(iph2->dst);
		iph2->dst = NULL;
	}

	free(iph2);
}

/*
 * create new isakmp Phase 2 status record to handle isakmp in Phase2
 */
int
insph2(iph2)
	struct ph2handle *iph2;
{
	LIST_INSERT_HEAD(&ph2tree, iph2, chain);

	return 0;
}

void
remph2(iph2)
	struct ph2handle *iph2;
{
	LIST_REMOVE(iph2, chain);
}

void
initph2tree()
{
	LIST_INIT(&ph2tree);
}

void
flushph2()
{
	struct ph2handle *p, *next;

	for (p = LIST_FIRST(&ph2tree); p; p = next) {
		next = LIST_NEXT(p, chain);

		/* send delete information */
		isakmp_info_send_d2(p);

		unbindph12(p);
		remph2(p);
		delph2(p);
	}
}

/* %%% */
void
bindph12(iph1, iph2)
	struct ph1handle *iph1;
	struct ph2handle *iph2;
{
	iph2->ph1 = iph1;
	LIST_INSERT_HEAD(&iph1->ph2tree, iph2, ph1bind);
}

void
unbindph12(iph2)
	struct ph2handle *iph2;
{
	if (iph2->ph1 != NULL) {
		iph2->ph1 = NULL;
		LIST_REMOVE(iph2, ph1bind);
	}
}
