/*-
 * Copyright (c) 1993-1994 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and the Network Research Group at
 *      Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * EM6 (Elastic multicast over ipv6/Kame patch for FreeBSD2.2.8)
 * Copyright (C) 1999 FUJITSU LABORATORIES LTD.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: net.cc,v 1.2 96/06/07 13:37:07 mccanne Exp $ (LBL)";

#include <stdlib.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#ifdef WIN32
#include <winsock.h>
#else
#include <sys/param.h>
#include <netdb.h>
#include <sys/socket.h>
#include <sys/uio.h>
#endif
#include "net.h"
#include "crypt.h"

#include "net-addr.h"

#if defined(EM6)
#include <em6lib.h>
extern em6_nozzle_t nozzle;
extern char* em6_destlist;
extern struct sockaddr_in6 sin_em6;
extern int dest_port[];
void pack_nozzle(void);

#if defined(EM6GROUP) || defined(EM6EXPLORE)
extern void adios();
#endif /* EM6GROUP || EM6EXPLORE */

#if defined(EM6GROUP)
#include "Tcl.h"
extern char* em6grp_vic_grp_URL;
int em6grp_nozzle_update(void);
#endif /* EM6GROUP */

#if defined(EM6EXPLORE)
#include <curses.h>
extern bool nozzle_exploring;
extern int exploring_count;
em6_nozzle_t other_nozzle = -1;
int other_exploring_count = 0;
bool other_nozzle_exploring = FALSE;
#endif /* EM6EXPLORE */

#define MAXPACKETSIZE (1500-28)

static int em6_sendmsg(int s, struct msghdr* mh, int flags){
	u_char wrkbuf[MAXPACKETSIZE];
	int len = mh->msg_iovlen;
	struct iovec* iov = mh->msg_iov;
	u_char* cp;
	u_char* ep;
	int result = 0;

	for (cp = wrkbuf, ep = wrkbuf + MAXPACKETSIZE; --len >= 0; ++iov) {
		int plen = iov->iov_len;
		if (cp + plen >= ep) {
			errno = E2BIG;
			return (-1);
		}
		memcpy(cp, iov->iov_base, plen);
		cp += plen;
	}

#if defined(EM6GROUP)
	if(em6grp_vic_grp_URL != NULL)
	  em6grp_nozzle_update();
#endif /* EM6GROUP */

	sin_em6.sin6_port   = dest_port[s];

	if ( nozzle != -1 ) {
	  result = em6_nozzle_sendto(s, (char*)wrkbuf, cp - wrkbuf, flags, 
				     &sin_em6, sizeof(struct sockaddr_in6),
				     nozzle);
	}

	return (result);
}
#endif /* EM6 */

/*
 * Linux does not have sendmsg
 */
#if defined(__linux__) || defined(WIN32)
#define MAXPACKETSIZE (1500-28)

static int
sendmsg(int s, struct msghdr* mh, int flags)
{
	u_char wrkbuf[MAXPACKETSIZE];
	int len = mh->msg_iovlen;
	struct iovec* iov = mh->msg_iov;
	u_char* cp;
	u_char* ep;

	for (cp = wrkbuf, ep = wrkbuf + MAXPACKETSIZE; --len >= 0; ++iov) {
		int plen = iov->iov_len;
		if (cp + plen >= ep) {
			errno = E2BIG;
			return (-1);
		}
		memcpy(cp, iov->iov_base, plen);
		cp += plen;
	}
	return (send(s, (char*)wrkbuf, cp - wrkbuf, flags));
}
#endif


/*
 * Address::operator==() -- compare two addresses
 */
int Address::operator==(const Address & addr) const {
	return (memcmp(*this, addr, length()) == 0);
}

/*
 * Address::alloc() -- allocate address with type based upon name
 */
Address * Address::alloc(const char * name) {
	return AddressType::alloc(name); 
}

/*
 * Address::defautl_alloc() -- allocate address with default type
 */
Address * Address::default_alloc() {
	char name[MAXHOSTNAMELEN];
	gethostname(name, sizeof(name));
	Address * result = alloc(name);
	return (result ? result : new Address());
}


Network::Network() :
	addr_(*(Address::default_alloc())),
	local_(*(Address::default_alloc())),
	lport_(0),
	port_(0),
	ttl_(0),
	rsock_(-1),
	ssock_(-1),
	noloopback_broken_(0),
	crypt_(0)
{
}

Network::Network(Address & addr, Address & local) :
	addr_(addr),
	local_(local),
	lport_(0),
	port_(0),
	ttl_(0),
	rsock_(-1),
	ssock_(-1),
	noloopback_broken_(0),
	crypt_(0)
{
}

Network::~Network()
{
	if (&addr_) delete &addr_; 
	if (&local_) delete &local_;
}

int Network::command(int argc, const char*const* argv)
{
	if (argc == 2) {
		Tcl& tcl = Tcl::instance();
		char* cp = tcl.buffer();
		if (strcmp(argv[1], "addr") == 0 || 
		    strcmp(argv[1], "interface") == 0 ||
		    strcmp(argv[1], "port") == 0 ||
		    strcmp(argv[1], "ttl") == 0 ||
		    strcmp(argv[1], "ismulticast") == 0)
			strcpy(cp, "0");
		else
			return (TclObject::command(argc, argv));
		tcl.result(cp);
		return (TCL_OK);
	} else if (argc == 3) {
		if (strcmp(argv[1], "crypt") == 0) {
			/*
			 * 'crypt ""' will turn of encryption because
			 * lookup() will return null.
			 */
			crypt_ = (Crypt*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
	}
	return (TclObject::command(argc, argv));
}

void Network::nonblock(int fd)
{       
#ifdef WIN32
	u_long flag = 1;
	if (ioctlsocket(fd, FIONBIO, &flag) == -1) {
		fprintf(stderr, "ioctlsocket: FIONBIO: %lu\n", GetLastError());
		exit(1);
	}
#else
        int flags = fcntl(fd, F_GETFL, 0);
#if defined(hpux) || defined(__hpux)
        flags |= O_NONBLOCK;
#else
        flags |= O_NONBLOCK|O_NDELAY;
#endif
        if (fcntl(fd, F_SETFL, flags) == -1) {
                perror("fcntl: F_SETFL");
                exit(1);
        }
#endif
}

u_char* Network::wrkbuf_;
int Network::wrkbuflen_;

void Network::expand_wrkbuf(int len)
{
	if (wrkbuflen_ == 0)
		wrkbuf_ = (u_char*)malloc(len);
	else
		wrkbuf_ = (u_char*)realloc((u_char*)wrkbuf_, len);
	wrkbuflen_ = len;
}

void Network::dosend(u_char* buf, int len, int fd)
{
#if defined(EM6)
        int cc;
        if (em6_destlist == NULL
#if defined(EM6GROUP)
	    && em6grp_vic_grp_URL == NULL
#endif /* EM6GROUP */
	    )
	  cc = ::send(fd, (char*)buf, len, 0);
	else {
#if defined(EM6GROUP)
	  if(em6grp_vic_grp_URL != NULL)
	    em6grp_nozzle_update();
#endif /* EM6GROUP */
	  sin_em6.sin6_port   = dest_port[fd];

	  if ( nozzle != -1 )
	    cc = em6_nozzle_sendto(fd,(char*)buf,len,0,&sin_em6,
				   sizeof(struct sockaddr_in6), nozzle);
	  else
	    cc = 0;
	}
#else /* EM6 */
	int cc = ::send(fd, (char*)buf, len, 0);
#endif /* EM6 */
	if (cc < 0) {
		switch (errno) {
		case ECONNREFUSED:
			/* no one listening at some site - ignore */
#if defined(__osf__) || defined(_AIX)
			/*
			 * Due to a bug in kern/uipc_socket.c, on several
			 * systems, datagram sockets incorrectly persist
			 * in an error state on receipt of an ICMP
			 * port-unreachable.  This causes unicast connection
			 * rendezvous problems, and worse, multicast
			 * transmission problems because several systems
			 * incorrectly send port unreachables for 
			 * multicast destinations.  Our work around
			 * is to simply close and reopen the socket
			 * (by calling reset() below).
			 *
			 * This bug originated at CSRG in Berkeley
			 * and was present in the BSD Reno networking
			 * code release.  It has since been fixed
			 * in 4.4BSD and OSF-3.x.  It is know to remain
			 * in AIX-4.1.3.
			 *
			 * A fix is to change the following lines from
			 * kern/uipc_socket.c:
			 *
			 *	if (so_serror)
			 *		snderr(so->so_error);
			 *
			 * to:
			 *
			 *	if (so->so_error) {
			 * 		error = so->so_error;
			 *		so->so_error = 0;
			 *		splx(s);
			 *		goto release;
			 *	}
			 *
			 */
			reset();
#endif
			break;

		case ENETUNREACH:
		case EHOSTUNREACH:
			/*
			 * These "errors" are totally meaningless.
			 * There is some broken host sending
			 * icmp unreachables for multicast destinations.
			 * UDP probably aborted the send because of them --
			 * try exactly once more.  E.g., the send we
			 * just did cleared the errno for the previous
			 * icmp unreachable, so we should be able to
			 * send now.
			 */
#if defined(EM6)
		  if (em6_destlist == NULL
#if defined(EM6GROUP)
		      && em6grp_vic_grp_URL == NULL
#endif /* EM6GROUP */
		      ){
#endif /* EM6 */
			(void)::send(ssock_, (char*)buf, len, 0);
#if defined(EM6)
		  } else {
#if defined(EM6GROUP)
		    if(em6grp_vic_grp_URL != NULL)
		      em6grp_nozzle_update();
#endif /* EM6GROUP */
		    sin_em6.sin6_port   = dest_port[ssock_];

		    if ( nozzle != -1 )
		      (void)em6_nozzle_sendto(ssock_,(char*)buf,len,0,&sin_em6,
					      sizeof(struct sockaddr_in6),
					      nozzle);
		  }
#endif /* EM6 */
			break;

		default:
			perror("send");
			return;
		}
	}
}

void Network::send(u_char* buf, int len)
{
	if (crypt_)
		buf = crypt_->Encrypt(buf, len);
	dosend(buf, len, ssock_);
}

/*
 * Copy a scatter/gather packet to the work buffer and
 * return it's length.
 */
int Network::cpmsg(const msghdr& mh)
{
	int len, i;
	for (len = 0, i = 0; i < int(mh.msg_iovlen); ++i)
		len += mh.msg_iov[i].iov_len;

	if (len > wrkbuflen_)
		expand_wrkbuf(len);
	u_char* cp = wrkbuf_;
	for (i = 0; i < int(mh.msg_iovlen); ++i) {
		int cc = mh.msg_iov[i].iov_len;
		memcpy(cp, mh.msg_iov[i].iov_base, cc);
		cp += cc;
	}
	return (len);
}

void Network::send(const msghdr& mh)
{
#if defined(EM6)
	int cc;
#endif /* EM6 */

	if (crypt_) {
		int cc = cpmsg(mh);
		send(wrkbuf_, cc);
		return;
	}

#if defined(EM6)
	if ( em6_destlist == NULL
#if defined(EM6GROUP)
	     && em6grp_vic_grp_URL == NULL
#endif /* EM6GROUP */
	     ) {
	  cc = ::sendmsg(ssock_, (msghdr*)&mh, 0);
	} else
	  cc = em6_sendmsg(ssock_, (msghdr*)&mh, 0);
#else /* EM6 */
	int cc = ::sendmsg(ssock_, (msghdr*)&mh, 0);
#endif /* EM6 */

	if (cc < 0) {
		switch (errno) {
		case ECONNREFUSED:
			/* no one listening at some site - ignore */
#if defined(__osf__) || defined(_AIX)
			reset();
#endif
			break;

		case ENETUNREACH:
		case EHOSTUNREACH:
			/*
			 * These "errors" are totally meaningless.
			 * There is some broken host sending
			 * icmp unreachables for multicast destinations.
			 * UDP probably aborted the send because of them --
			 * try exactly once more.  E.g., the send we
			 * just did cleared the errno for the previous
			 * icmp unreachable, so we should be able to
			 * send now.
			 */
#if defined(EM6)
			if (em6_destlist == NULL
#if defined(EM6GROUP)
		            && em6grp_vic_grp_URL == NULL
#endif /* EM6GROUP */
			    )
#endif /* EM6 */
			(void)::sendmsg(ssock_, (msghdr*)&mh, 0);
#if defined(EM6)
			else
				(void) em6_sendmsg(ssock_, (msghdr*)&mh, 0);
#endif /* EM6 */
			break;

		default:
#if defined(EM6)
		        if (em6_destlist == NULL
#if defined(EM6GROUP)
			    && em6grp_vic_grp_URL == NULL
#endif /* EM6GROUP */
			    )
#endif /* EM6 */
			perror("sendmsg");
#if defined(EM6)
			else
			  perror("em6_sendmsg");
#endif /* EM6 */
			return;
		}
	}
}



int Network::dorecv(u_char* buf, int len, u_int32_t& from, int fd)
{
	sockaddr_in sfrom;
	int fromlen = sizeof(sfrom);
	int cc = ::recvfrom(fd, (char*)buf, len, 0,
			    (sockaddr*)&sfrom, &fromlen);
	if (cc < 0) {
		if (errno != EWOULDBLOCK)
			perror("recvfrom");
		return (-1);
	}
	from = sfrom.sin_addr.s_addr;
	if (noloopback_broken_ &&  memcmp(local_, &from, fromlen) == 0
	    && sfrom.sin_port == lport_)
		return (0);

	return (cc);
}



int Network::recv(u_char* buf, int len, u_int32_t& from)
{
	if (crypt_) {
		if (len > wrkbuflen_)
			expand_wrkbuf(len);
		int cc = dorecv(wrkbuf_, len, from, rsock_);
		return (crypt_->Decrypt(wrkbuf_, cc, buf));
	}
	return (dorecv(buf, len, from, rsock_));
}


int Network::recv(u_char* buf, int len, Address & from)
{
	if (crypt_) {
		if (len > wrkbuflen_)
			expand_wrkbuf(len);
		int cc = dorecv(wrkbuf_, len, from, rsock_);
		return (crypt_->Decrypt(wrkbuf_, cc, buf));
	}
	return (dorecv(buf, len, from, rsock_));
}

void Network::reset()
{
}

#if defined(EM6)
int em6grp_nozzle_update(void){
#if defined(EM6GROUP) || defined(EM6EXPLORE)
  int error;
#endif /* EM6GROUP || EM6EXPLORE */
#if defined(EM6GROUP)
  struct in6_addr node;
  struct addrinfo hint, *ip;
  char *tcl_join_nodes, *tcl_leave_nodes, *c, *d;
  char *join_nodes, *leave_nodes;

  Tcl& tcl = Tcl::instance();
  tcl.evalc("Em6grp_vic_chk_notify");

  if ( nozzle == -1 ) {
    pack_nozzle();
    return(0);
  }

  tcl_join_nodes = tcl.getvar("em6grp_vic_join_nodes");
  if (tcl_join_nodes != NULL && strlen(tcl_join_nodes) != 0 ) {
    printf("em6grp_nozzle_update: join_nodes = [%s]\n", tcl_join_nodes);

    d = join_nodes = (char*)malloc(strlen(tcl_join_nodes) + 1);
    strcpy(join_nodes, tcl_join_nodes);

    do{
      c = strchr(join_nodes, ',');
      if (c != NULL)
	*c = '\0';
      bzero((char *)&hint,sizeof(hint));
      hint.ai_flags = 0;
      hint.ai_family = AF_INET6;
      if ((error = getaddrinfo(join_nodes, NULL, &hint, &ip)) != 0) {
	fprintf(stderr, 
		"nozzle_update(join): invalid addrress[%s](%d)\n", 
		join_nodes, error);
	exit(1);
      }

      node = ((struct sockaddr_in6 *)(ip->ai_addr))->sin6_addr;
      join_nodes = c + 1;

      if( (error = em6_nozzle_addnode(nozzle, &node)) != 0){
	fprintf(stderr,
		"em6grp_nozzle_update: em6_nozzle_addnode error(%d).\n",
		error);
	adios();
      }

#if defined(EM6EXPLORE)      
      if((error = em6_nozzle_expstart(nozzle, NULL)) != 0){
	fprintf(stderr, "em6grp_nozzle_update: ");
	fprintf(stderr, "em6_nozzle_expstart failed(%d).\n", error);
	adios();
      }
      exploring_count++;
      fprintf(stderr, "em6grp_nozzle_update: ");
      fprintf(stderr, "em6_nozzle_expstart start(%d).\n", exploring_count);
      nozzle_exploring = TRUE;
#endif /* EM6EXPLORE */

    } while(c != NULL);

    free(d);
  }


  tcl_leave_nodes = tcl.getvar("em6grp_vic_leave_nodes");
  if (tcl_leave_nodes != NULL && strlen(tcl_leave_nodes) != 0 ) {
    printf("em6grp_nozzle_update: leave_nodes = [%s]\n", tcl_leave_nodes);

    d = leave_nodes = (char*)malloc(strlen(tcl_leave_nodes) + 1);
    strcpy(leave_nodes, tcl_leave_nodes);

    do{
      c = strchr(leave_nodes, ',');
      if (c != NULL)
	*c = '\0';
      bzero((char *)&hint,sizeof(hint));
      hint.ai_flags = 0;
      hint.ai_family = AF_INET6;
      if ((error = getaddrinfo(leave_nodes, NULL, &hint, &ip)) != 0) {
	fprintf(stderr, 
		"em6grp_nozzle_update(leave): invalid addrress[%s](%d)\n", 
		leave_nodes, error);
	exit(1);
      }

      node = ((struct sockaddr_in6 *)(ip->ai_addr))->sin6_addr;
      leave_nodes = c + 1;

      if( (error = em6_nozzle_delnode(nozzle, &node)) != 0){
	fprintf(stderr,
		"em6grp_nozzle_update: em6_nozzle_delnode error(%d).\n",
		error);
	adios();
      }
    } while(c != NULL);

    free(d);
  }
#endif /* EM6GROUP */

#if defined(EM6EXPLORE)
  if(nozzle_exploring){
    error = em6_nozzle_exppoll(nozzle, 0);

    if(error < 0){
      fprintf(stderr, "em6grp_nozzle_update: ");
      fprintf(stderr, "em6_nozzle_exppoll failed(%d).\n", error);
      perror("em6_nozzle_exppoll");
      exit(1);
    }

    if(error == 0){
      fprintf(stderr, "em6grp_nozzle_update: ");
      fprintf(stderr, "em6_nozzle_exppoll continue(%d).\n", exploring_count);
    } else {
      fprintf(stderr, "em6grp_nozzle_update: ");
      fprintf(stderr, "em6_nozzle_exppoll succeeded(%d).\n", exploring_count);
      nozzle_exploring = FALSE;
    }
  }

  if(other_nozzle_exploring){
    error = em6_nozzle_exppoll(other_nozzle, 0);

    if(error < 0){
      fprintf(stderr, "em6grp_nozzle_update: ");
      fprintf(stderr, "em6_nozzle_exppoll(other) failed(%d).\n", error);
      perror("em6_nozzle_exppoll");
      exit(1);
    }

    if(error == 0){
      fprintf(stderr, "em6grp_nozzle_update: ");
      fprintf(stderr, "em6_nozzle_exppoll(other) continue(%d).\n", 
	      other_exploring_count);
    } else {
      fprintf(stderr, "em6grp_nozzle_update: ");
      fprintf(stderr, "em6_nozzle_exppoll(other) succeeded(%d).\n", 
	      other_exploring_count);
      other_nozzle_exploring = FALSE;
    }
  }else{
    if(em6_nozzle_needexplore(&other_nozzle) == 0){
      if((error = em6_nozzle_expstart(other_nozzle, NULL)) != 0){
	fprintf(stderr, "em6grp_nozzle_update: ");
	fprintf(stderr, "em6_nozzle_expstart(other) failed(%d).\n", error);
    	exit(1);
      }
      other_exploring_count++;
      fprintf(stderr, "em6grp_nozzle_update: ");
      fprintf(stderr, "em6_nozzle_expstart(other) start(%d).\n", 
	      other_exploring_count);
      other_nozzle_exploring = TRUE;
    }
  }    
#endif /* EM6EXPLORE */

  return(0);
}
#endif /* EM6 */
