/* wptW32API.cpp - Common W32 API functions
 *	Copyright (C) 2001, 2002, 2003 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <shellapi.h>
#include <shlobj.h>
#include <commctrl.h>

#include "wptNLS.h"
#include "wptW32API.h"
#include "wptErrors.h"
#include "wptVersion.h"
#include "wptTypes.h"


/*
 * The the text of a menu item.
 */
void
set_menu_text (HMENU menu, int m_uid, const char *text)
{
    MENUITEMINFO mii;
    char menu_text[80];
    
    memset (&mii, 0, sizeof (mii));
    mii.cbSize = sizeof (mii);
    mii.fMask = MIIM_TYPE;
    mii.dwTypeData = menu_text;
    mii.cch = sizeof (menu_text);
    GetMenuItemInfo (menu, m_uid, FALSE, &mii);
    
    memset (&mii, 0, sizeof mii);
    mii.cbSize = sizeof mii;
    mii.fMask = MIIM_TYPE;
    mii.fType = MFT_STRING;
    mii.dwTypeData = (char *) text;
    SetMenuItemInfo (menu, m_uid, FALSE, &mii);
} /* set_menu_text */


void
set_menu_state (HMENU menu, int m_uid, int state)
{	
    MENUITEMINFO mii;

    memset( &mii, 0, sizeof (mii) );
    mii.cbSize = sizeof (mii);
    mii.fMask = MIIM_STATE;
    mii.fState = state;
    SetMenuItemInfo (menu, m_uid, FALSE, &mii);
} /* set_menu_state */



const char *
get_filename_dlg (HWND hwnd, int id, const char * title, 
		  const char * filter, const char * name)
{
    static char file[512] = "";
    OPENFILENAME open;
    
    if (name && strlen (name) < (sizeof (file)-1))
	strcpy (file, name);
    else
	memset (file, 0, sizeof (file));
    if (!filter)
	filter = _("All Files (*.*)\0*.*");
    memset (&open, 0, sizeof (open));
    open.lStructSize = sizeof (OPENFILENAME);
    open.hInstance = glob_hinst;
    open.lpstrTitle = title;
    open.lpstrFilter = filter;
    open.hwndOwner = hwnd;
    open.lpstrFile = file;
    open.nMaxFile = sizeof (file) - 1;
    open.Flags = 0;

    if (id == 0 && GetOpenFileName (&open))
	return open.lpstrFile;
    else if (id == 1 && GetSaveFileName (&open))
	return open.lpstrFile;
    
    return NULL;
} /* get_filename_dlg */


const char * 
get_folder_dlg (HWND hwnd, const char * title, const char * name)
{
    static char folder[MAX_PATH] = "";
    BROWSEINFO bi;
    ITEMIDLIST * il;

    memset (&bi, 0, sizeof (bi));
    bi.hwndOwner = hwnd;
    if (title)
	bi.lpszTitle = title;
    if (name && strlen (name) < MAX_PATH-1)
	strcpy (folder, name);
    else
	memset (folder, 0, sizeof (folder));
    il = SHBrowseForFolder (&bi);
    if (il) {
	SHGetPathFromIDList (il, folder);
	return folder;
    }
    return NULL;
}


char*
get_clip_text (HWND hwnd)
{
    HANDLE clipmem;
    char *cliptxt, *p;  
    int len;
    
    if (OpenClipboard (hwnd) == FALSE)
	return NULL;
    clipmem = GetClipboardData (CF_TEXT);
    if (clipmem == NULL) {
	p = NULL;
	goto leave;
    }
    cliptxt = (char *) GlobalLock (clipmem);
    if (cliptxt == NULL) {
	p = NULL;
	goto leave;
    }
    
    len =  strlen (cliptxt);
    p = new char[len + 1];
    if (!p)
	BUG (NULL);
    memcpy (p, cliptxt, len);
    p[len] = '\0';
    GlobalUnlock (clipmem);
    
leave:
    CloseClipboard ();
    return p;
} /* get_clip_text */


int
set_clip_text (HWND hwnd, const char *text, int nbytes)
{    
    HANDLE clipmem;
    int rc = 0;
    char *p;
    
    if (OpenClipboard (hwnd) == FALSE)	
	return WPTERR_CLIP_OPEN;
    EmptyClipboard ();

    clipmem = GlobalAlloc (GHND, nbytes + 1);
    if (clipmem == NULL)
	BUG (NULL);
    p = (char *) GlobalLock (clipmem);
    if (p == NULL) {
	rc = WPTERR_GENERAL;;
	goto leave;	
    }
    memcpy (p, text, nbytes);
    p[nbytes] = '\0';
    
    GlobalUnlock (clipmem);
    SetClipboardData (CF_TEXT, clipmem);
    
leave:
    CloseClipboard ();
    return rc;
} /* set_clip_text */


int
set_clip_text2 (HWND hwnd, const char *text, int nbytes, int head_foot)
{
    char *p, *new_text;

    p = get_clip_text (hwnd);
    if (!p)
	return WPTERR_CLIP_GET;
    new_text = new char [strlen (p)+strlen (text)+8];
    if (!new_text)
	BUG (0);
    if (head_foot == 0)
	sprintf (new_text, "%s\r\n%s\r\n\r\n", text, p);
    else
	sprintf (new_text, "%s\n%s\n\n", p, text);
    set_clip_text (hwnd, new_text, strlen (new_text)+1);
    free_if_alloc (p);
    free_if_alloc (new_text);
    return 0;
}


char*
make_filename( const char *path, const char *file, const char *ext )
{
    char *p;
    size_t size = 0;

    if( path && *path )
	size += strlen( path );
    if( file && *file )
	size += strlen( file );
    if( ext && *ext )
	size += strlen( ext );
    p = new char[size + 4];
    memset( p, 0, size );
    if( path ) {
	strcat( p, path );
	if( path[strlen( path ) -1] != '\\' )
	    strcat( p, "\\" );
    }
    if( file )
	strcat( p, file );
    if( ext ) {
	strcat( p, "." );
	strcat( p, ext );
    }
    return p;
} /* make_filename */


/* return 0 if it exists, otherwise >0. */
int
file_exist_check (const char * fname)
{	
    HANDLE fh;	

    fh = CreateFile( fname, GENERIC_READ, FILE_SHARE_READ, 
		     NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );
    if( fh == INVALID_HANDLE_VALUE )
	return WPTERR_FILE_EXIST;
    CloseHandle( fh );
    return 0;
} /* file_exist_check */


int
dir_exist_check( const char *dir )
{
    struct stat statbuf;    
    
    if( stat( dir, &statbuf ) == -1 )
	return WPTERR_GENERAL;
    if( statbuf.st_mode & _S_IFDIR )
	return 0;
    return WPTERR_GENERAL;
} /* dir_exist_check */


size_t
get_file_size( const char *fname )
{
    size_t fsize;
    HANDLE fh;

    fh = CreateFile( fname, GENERIC_READ, FILE_SHARE_READ,
		    NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );	
    if( fh == INVALID_HANDLE_VALUE )
	return 0;
    fsize = GetFileSize( fh, NULL );
    if( fsize == 0xFFFFFFFF )
	fsize = 0;
    CloseHandle( fh );
    return fsize;
} /* get_file_size */


int
init_file_lock( LOCK *ctx, const char *file )
{
    
    ctx->size = get_file_size( file );
    ctx->file = m_strdup( file );
    ctx->fh = CreateFile( file, GENERIC_READ, FILE_SHARE_READ, NULL, 
			 OPEN_ALWAYS, 0, NULL );
    if( ctx->fh == INVALID_HANDLE_VALUE )
	return WPTERR_GENERAL;
    if( LockFile( ctx->fh, 0, 0, ctx->size, 0 ) == FALSE ) {
	CloseHandle( ctx->fh );
	ctx->fh = INVALID_HANDLE_VALUE;
	ctx->size = 0;
	free( ctx->file );
	return WPTERR_GENERAL;
    }
    return 0;
} /* init_file_lock */


void
release_file_lock( LOCK *ctx )
{
    free_if_alloc( ctx->file );
    ctx->file = NULL;
    ctx->size = 0;
    CloseHandle( ctx->fh );
} /* release_file_lock */


int
dialog_box_param( HINSTANCE hinst, LPCTSTR name, HWND parent, DLGPROC fnc, 
				  LPARAM param, LPCTSTR title, int title_id )
{
#ifndef LANG_DE
    if( FindWindowEx( GetDesktopWindow(), NULL, NULL, title ) )
	return -1;	
#else
    char strdesc[256];
    LoadString( glob_hinst, title_id, strdesc, sizeof (strdesc) -1 );
    if( FindWindowEx( GetDesktopWindow(), NULL, NULL, strdesc ) )
	return -1;
#endif
    
    return DialogBoxParam( hinst, name, parent, fnc, param );
} /* dialog_box_param */


int
msg_box( HWND hwnd, const char *text, const char *title, int mode )
{
    mode |= MB_SETFOREGROUND;
    mode |= MB_TASKMODAL;
    mode |= MB_TOPMOST;
    return MessageBox(hwnd, text, title, mode);
} /* msg_box */


void
set_active_window( HWND dlg )
{	
    activ_hwnd = dlg;
} /* set_active_window */

void
reset_active_window( void )
{	
    activ_hwnd = NULL;
} /* reset_active_window */


static DWORD CALLBACK 
reminder_thread( void *ctx )
{
    reminder_ctx_s *c = (reminder_ctx_s *)ctx;

    Sleep( c->msecs );
    SetForegroundWindow( activ_hwnd );

    return 0;
} /* reminder_thread */

HANDLE
window_reminder( struct reminder_ctx_s *ctx )
{
    DWORD tid = 0;
    
    return CreateThread( NULL, 0, reminder_thread, ctx, 0, &tid );
} /* window_reminder */


char *
m_strdup( const char *str )
{
    char * p = new char[strlen( str ) + 1];
    if( p )
	strcpy( p, str );
    return p;
} /* m_strdup */


void
center_window2 (HWND hwndChild, HWND style) 
{     
    HWND hwndParent;
    RECT rChild, rParent;    
    HDC hdc;
    int wChild, hChild, wParent, hParent;     
    int wScreen, hScreen, xNew, yNew;
    int flags = SWP_NOSIZE | SWP_NOZORDER;

    hwndParent = GetDesktopWindow ();
    GetWindowRect (hwndChild, &rChild);     
    wChild = rChild.right - rChild.left;     
    hChild = rChild.bottom - rChild.top;

    GetWindowRect (hwndParent, &rParent);     
    wParent = rParent.right - rParent.left;     
    hParent = rParent.bottom - rParent.top;      
    
    hdc = GetDC (hwndChild);     
    wScreen = GetDeviceCaps (hdc, HORZRES);     
    hScreen = GetDeviceCaps (hdc, VERTRES);     
    ReleaseDC (hwndChild, hdc);      
    xNew = rParent.left + ((wParent - wChild) /2);     
    if (xNew < 0)
	xNew = 0;
    else if ((xNew+wChild) > wScreen) 
	xNew = wScreen - wChild;
    yNew = rParent.top  + ((hParent - hChild) /2);
    if (yNew < 0)
	yNew = 0;
    else if ((yNew+hChild) > hScreen)
	yNew = hScreen - hChild;
    if (style == HWND_TOPMOST || style == HWND_NOTOPMOST)
	flags = SWP_NOMOVE | SWP_NOSIZE;
    SetWindowPos (hwndChild, style? style : NULL, xNew, yNew, 0, 0, flags);
		  
}


void
center_window (HWND hwndChild) 
{
    center_window2 (hwndChild, NULL);
}
