/* wptKeyManagerDlg.cpp - WinPT Key Manager
 *	Copyright (C) 2000-2004 Timo Schulz
 *	Copyright (C) 2004 Andreas Jobs
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <commctrl.h>

#include "../resource.h"
#include "wptTypes.h"
#include "wptGPG.h"
#include "wptCommonCtl.h"
#include "wptContext.h" /* for passphrase_s */
#include "wptDlgs.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptKeylist.h"
#include "wptNLS.h"
#include "wptErrors.h"
#include "wptHotkey.h"
#include "wptKeyManager.h"
#include "wptKeyserver.h"
#include "wptKeyEdit.h"
#include "wptRegistry.h"


#define WINDOWCLASS_SEPARATOR_CHILD "WINPT_SEP_CHILD"
#define SEPARATOR_HEIGHT            5

int treeview_add_item (HWND tree, HTREEITEM parent, const char * text);

static subclass_s keylist_proc;
static int km_index = -1;
static int km_def_pos = 0;

HIMAGELIST glob_imagelist;

struct km_info {
    /* Window positions */
    int pos_x, pos_y;
    int ypos_sep;
    int ypercent_sep;

    HWND hwnd_sep;
    HWND toolbar;
    HWND statbar;

    listview_ctrl_t lv;
    int keylist_sortby;
};

struct mybuttons {
    long icon;
    long command;
    char *text;
};

struct mybuttons myb[] = {
    {IMI_KEY_DELETE, ID_KEYMISC_DELETE, "Delete"},
    {IMI_KEY_PROPS,  ID_KEYMISC_PROPS,	"Properties"},
    {IMI_KEY_SIGN,   ID_KEYMISC_SIGN,	"Sign"},
    {IMI_KEY_IMPORT, ID_KEYMISC_IMPORT, "Import"},
    {IMI_KEY_EXPORT, ID_KEYMISC_EXPORT, "Export"},
};


static BOOL CALLBACK
keylist_subclass_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    switch (msg) 
    {
    case WM_LBUTTONDBLCLK:
        send_cmd_id( keylist_proc.dlg, ID_KEYMISC_PROPS );
        break;
        
    case WM_KEYUP:
        int virt_key = (int)wparam;
        if (virt_key == VK_SPACE) 
	{
            send_cmd_id( keylist_proc.dlg, ID_KEYMISC_PROPS );
            break;
        }
        if (virt_key == VK_DELETE) 
	{
            send_cmd_id( keylist_proc.dlg, ID_KEYMISC_DELETE );
            break;
        }
	if (virt_key == VK_INSERT)
	{
	    send_cmd_id (keylist_proc.dlg, ID_KEYMISC_PASTE);
	    break;
	}
        break;
    }
    
    return CallWindowProc( keylist_proc.old, dlg, msg, wparam, lparam );
} /* keylist_subclass_proc */


static HWND
load_toolbar (HWND dlg, struct km_info * kmi)
{
    HWND tbwnd;
    TBSAVEPARAMS tbsp;
    TBBUTTON tb_buttons[] = {
        /*{imagelist_getindex(IMI_EXIT),       ID_KEYMISC_QUIT,   TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0L, 0},*/
	{imagelist_getindex(IMI_KEY_DELETE), ID_KEYMISC_DELETE, TBSTATE_ENABLED, TBSTYLE_BUTTON, {0}, 0L, 0},
        {imagelist_getindex(IMI_KEY_PROPS),  ID_KEYMISC_PROPS,  TBSTATE_ENABLED, TBSTYLE_BUTTON, {0}, 0L, 0},
	{imagelist_getindex(IMI_KEY_SIGN),   ID_KEYMISC_SIGN,   TBSTATE_ENABLED, TBSTYLE_BUTTON, {0}, 0L, 0},
        {0,                                  0,                 0,               TBSTYLE_SEP,    {0}, 0L, 0},
	{imagelist_getindex(IMI_KEY_IMPORT), ID_KEYMISC_IMPORT, TBSTATE_ENABLED, TBSTYLE_BUTTON, {0}, 0L, 0},
        {imagelist_getindex(IMI_KEY_EXPORT), ID_KEYMISC_EXPORT, TBSTATE_ENABLED, TBSTYLE_BUTTON, {0}, 0L, 0},
     };
    
    tbwnd = CreateWindowEx (0, TOOLBARCLASSNAME, NULL,
			    WS_CHILD|TBSTYLE_TOOLTIPS|TBSTYLE_FLAT|CCS_ADJUSTABLE,
			    0, 0, 0, 0, dlg, (HMENU)IDR_WINPT_KMTB, glob_hinst, NULL);
    if (tbwnd) {
	SendMessage (tbwnd, TB_BUTTONSTRUCTSIZE, (WPARAM) sizeof(TBBUTTON), 0); 
        SendMessage (tbwnd, TB_SETIMAGELIST, 0, (LPARAM)glob_imagelist);
	        SendMessage (tbwnd, TB_AUTOSIZE, 0, 0);
        ShowWindow (tbwnd, SW_SHOW);

        /* Restore toolbar */
        tbsp.hkr = HKEY_CURRENT_USER;
        tbsp.pszSubKey = "Software\\WinPT";
        tbsp.pszValueName = "KM_toolbar";
        if (SendMessage(tbwnd, TB_SAVERESTORE, FALSE, (LPARAM)&tbsp ) == 0)
            SendMessage (tbwnd, TB_ADDBUTTONS, sizeof(tb_buttons) / sizeof(tb_buttons[0]), (LONG)&tb_buttons[0]);
     }
     return tbwnd;
} /* load_toolbar */


static void
do_center_window (HWND dlg, struct km_info * kmi)
{
    RECT rect;
    char * p;
    int pos_x = 0, pos_y = 0;
        
    /* Find bottom of keylist */
    GetWindowRect (GetDlgItem(dlg, IDC_KEYMISC_KEYLIST), &rect);
    MapWindowPoints (NULL, dlg, (POINT*)&rect, 2);

    kmi->ypos_sep = rect.bottom;

    p = get_reg_entry( HKEY_CURRENT_USER, "Software\\WinPT", "KM_Pos_X" );
    if( p && !strcmp( p, " " ) ) {
	free_if_alloc( p );	
	center_window( dlg );
	return;
    }
    else if( p )
	pos_x = atol( p );

    p = get_reg_entry( HKEY_CURRENT_USER, "Software\\WinPT", "KM_Pos_Y" );
    if( p && !strcmp( p, " " ) ) {
	free_if_alloc( p );
	center_window( dlg );
	return;
    }
    else if( p )
	pos_y = atol( p );

    if( !pos_y && !pos_x ) {
	center_window( dlg );
	return;
    }
    
    if( pos_x > GetSystemMetrics( SM_CXSCREEN ) 
	|| pos_y > GetSystemMetrics( SM_CYSCREEN ) ) {
	pos_x = pos_y = 0;
    }
    GetClientRect( dlg, &rect );
    MoveWindow( dlg, pos_x, pos_y, rect.right, rect.bottom, TRUE );
}


static void
do_resize_window( HWND dlg, struct km_info *kmi)
{
    HWND h;
    RECT rclient, rect;
    BOOL bRepaint = FALSE;

    /* Get rect of client area and make life easier */
    GetClientRect( dlg, &rclient );

    /* Move toolbar to the top of the window */
    if (kmi->toolbar) {
        GetWindowRect(kmi->toolbar, &rect);
        ScreenToClient(dlg, (POINT*)&rect);
        ScreenToClient(dlg, (POINT*)&(rect.right));

        rclient.top += rect.bottom - rect.top;
        MoveWindow (kmi->toolbar, 0, 0, rclient.right - rclient.left, 
		    rect.bottom - rect.top, bRepaint);
    }

    /* Move statusbar to the bottom of the window */
    if (kmi->statbar) {
        GetWindowRect( kmi->statbar, &rect );
        ScreenToClient(dlg, (POINT*)&rect);
        ScreenToClient(dlg, (POINT*)&(rect.right));

        rclient.bottom -= rect.bottom - rect.top;
        MoveWindow (kmi->statbar, 0, rclient.bottom, rclient.right - rclient.left, 
		    rect.bottom - rect.top, bRepaint);
    }

    // Obtain separator information and move it to the desired posistion
    if (kmi->ypercent_sep)
        kmi->ypos_sep = (rclient.bottom - rclient.top) * kmi->ypercent_sep / 100;
    else
        kmi->ypercent_sep = kmi->ypos_sep * 100 / (rclient.bottom - rclient.top);
        
    // Don't move away
    if (kmi->ypos_sep+5 > rclient.bottom)
        kmi->ypos_sep = rclient.bottom - 5;
    if (kmi->ypos_sep < rclient.top)
        kmi->ypos_sep = rclient.top;
    MoveWindow (kmi->hwnd_sep, 0, kmi->ypos_sep, (rclient.right - rclient.left), 5, bRepaint);
        
    // Place the keylist above the separator
    h = GetDlgItem( dlg, IDC_KEYMISC_KEYLIST );
    MoveWindow (h, rclient.left, rclient.top, rclient.right - rclient.left, 
		kmi->ypos_sep - rclient.top, bRepaint);
    rclient.top = kmi->ypos_sep + 5 + 8;

    /* Place the group text and the group box below the separator */
    h = GetDlgItem( dlg, IDC_KEYMISC_GTEXT );
    MoveWindow( h, rclient.left, rclient.top, 100, 14, bRepaint);
    rclient.top += 18;

    h = GetDlgItem( dlg, IDC_KEYMISC_GROUP );
    MoveWindow (h, rclient.left, rclient.top, rclient.right - rclient.left, 
		(rclient.bottom < rclient.top) ? 0 : rclient.bottom - rclient.top, bRepaint);
    
    /* Repaint the whole thing */
    InvalidateRect (dlg, NULL, TRUE);
} /* do_resize_window */


static void
do_create_minpopup (HWND dlg)
{
    HMENU hm;
    MENUITEMINFO mi;
    char * s;
    POINT p;
    
    if (gpg_read_only)
	return;
    hm = CreatePopupMenu ();
    if (!hm)
	BUG( NULL );
    memset (&mi, 0, sizeof mi);
    mi.cbSize = sizeof mi;
    s = (char *)_("Paste Key from Clipboard");
    mi.fType = MF_STRING;
    mi.dwTypeData = s;
    mi.cch = strlen (s);
    mi.fMask = MIIM_DATA | MIIM_ID | MIIM_TYPE;
    mi.wID = ID_KEYCTX_PASTE;
    InsertMenuItem (hm, 0, FALSE, &mi);
    GetCursorPos (&p);
    TrackPopupMenu (hm, 0, p.x, p.y, 0, dlg, NULL);
    DestroyMenu (hm);
} /* do_create_minpopup */


static void
do_check_cache( listview_ctrl_t lv, HWND dlg, HWND sb  )
{
    gpgme_keycache_t cache;

    if( keycache_get_reload( ) ) {
	keycache_reload( dlg );
	keycache_set_reload( 0 );
	cache = keycache_get_ctx( 1 );
	if( !cache )
	    BUG( dlg );
	keylist_reload( lv, cache, KEYLIST_LIST, GPGME_ATTR_USERID );
	km_complete_status_bar (sb, lv, km_def_pos);
    }
} /* do_check_cache */


long CALLBACK
separator_wnd_proc( HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam )
{
    static POINT last_mouse_pos;

    if (msg == WM_CREATE)
        SetWindowLong (hwnd, GWL_USERDATA, (long)(((CREATESTRUCT*)lparam)->lpCreateParams));

    switch (msg) {
    case WM_PAINT:
        PAINTSTRUCT ps;
        RECT rect;
        HPEN hpen;

        GetClientRect (hwnd, &rect);
        BeginPaint (hwnd, &ps);

        // Background
        FillRect (ps.hdc, &rect, (HBRUSH)(COLOR_3DFACE+1));

        // The lines from the light into the dark
        MoveToEx(ps.hdc, 0,0, NULL);
        if ((hpen = CreatePen(PS_SOLID, 0, GetSysColor(COLOR_3DHILIGHT))) != NULL) {
            SelectObject(ps.hdc, (LPVOID)hpen);
            LineTo(ps.hdc, rect.right, 0);
            DeleteObject(hpen);
        }
	MoveToEx(ps.hdc, 0, 1, NULL);
	if ((hpen = CreatePen(PS_SOLID, 0, GetSysColor(COLOR_3DLIGHT))) != NULL) {
             SelectObject(ps.hdc, (LPVOID)hpen);
             LineTo(ps.hdc, rect.right, rect.bottom);
             DeleteObject(hpen);
         }

        MoveToEx(ps.hdc, 0, rect.bottom-1, NULL);
        if ((hpen = CreatePen(PS_SOLID, 0, GetSysColor(COLOR_3DSHADOW))) != NULL) {
            SelectObject(ps.hdc, (LPVOID)hpen);
            LineTo(ps.hdc, rect.right, rect.bottom-1);
            DeleteObject(hpen);
        }
        MoveToEx(ps.hdc, 0, rect.bottom, NULL);
        if ((hpen = CreatePen(PS_SOLID, 0, GetSysColor(COLOR_3DDKSHADOW))) != NULL) {
            SelectObject(ps.hdc, (LPVOID)hpen);
            LineTo(ps.hdc, rect.right, rect.bottom);
            DeleteObject(hpen);
        }

        EndPaint (hwnd, &ps);
        return 0;

    case WM_LBUTTONDOWN:
        last_mouse_pos.x = LOWORD(lparam);
        last_mouse_pos.y = HIWORD(lparam);
        ClientToScreen (hwnd, &last_mouse_pos);
        SetCapture (hwnd);
        return 0;

    case WM_LBUTTONUP:
        ReleaseCapture ();
        return 0;

    case WM_MOUSEMOVE:
        if (wparam == MK_LBUTTON) {
            struct km_info *kmi;
            POINT p;
            RECT rect;

            if ((kmi = (struct km_info *)GetWindowLong (hwnd, GWL_USERDATA)) == NULL)
                break;

            // Calculate mouse movement
            p.x = LOWORD(lparam);
            p.y = HIWORD(lparam);
            ClientToScreen (hwnd, &p);

            GetWindowRect (hwnd, &rect);
            rect.top += (short)(p.y - last_mouse_pos.y);
            rect.bottom += (short)(p.y - last_mouse_pos.y);

            last_mouse_pos.y = p.y;

            // Apply mouse movement to window. Beware the MoveWindow is relaive 
	    // to parent NOT screen
            MapWindowPoints (NULL, GetParent(hwnd), (POINT*)&rect, 2);
            kmi->ypos_sep = rect.top;
            kmi->ypercent_sep = 0; // This forces do_resize_window to use abs. position
	    do_resize_window (GetParent(hwnd), kmi);
	    return 0;
        }
    }

    return DefWindowProc (hwnd, msg, wparam, lparam);
}


static HWND
regist_sep_wnd (HWND dlg, struct km_info * kmi)
{
    WNDCLASS wndclass;
    HWND h;

    wndclass.style         = CS_DBLCLKS | CS_HREDRAW | CS_VREDRAW;
    wndclass.lpfnWndProc   = separator_wnd_proc;
    wndclass.cbClsExtra    = 0;
    wndclass.cbWndExtra    = sizeof (long);
    wndclass.hInstance     = glob_hinst;
    wndclass.hIcon         = NULL;
    wndclass.hCursor       = LoadCursor (NULL, IDC_SIZENS);
    wndclass.hbrBackground = (HBRUSH) GetStockObject (LTGRAY_BRUSH);
    wndclass.lpszMenuName  = 0;
    wndclass.lpszClassName = WINDOWCLASS_SEPARATOR_CHILD;
    RegisterClass (&wndclass);

    h = CreateWindowEx (WS_EX_WINDOWEDGE, WINDOWCLASS_SEPARATOR_CHILD, NULL,
			WS_CHILD | WS_CLIPCHILDREN | WS_VISIBLE, 
			0, 400, 2000, 5, dlg, (HMENU) 0, glob_hinst, kmi);    
    ShowWindow (h, SW_SHOW);
    return h;
} /* regist_sep_wnd */



#define enable_button(hwnd, cid) \
    SendMessage ((hwnd), TB_ENABLEBUTTON, (cid), MAKELONG (key_selected, 0))

void
update_ui_items (HWND hwnd, listview_ctrl_t lv)
{
    int key_selected = 0, key_has_sec = 0;
    int i, key_inv = 0;
    HWND hwnd_child;
    HMENU hmenu;
    
    /* Get some states */
    key_selected = SendMessage (GetDlgItem (hwnd, IDC_KEYMISC_KEYLIST), 
					   LVM_GETSELECTEDCOUNT, 0, 0) 
					   ? TRUE : FALSE;
    key_has_sec = FALSE;
    if (key_selected) {
        i = listview_get_curr_pos (lv);
        key_has_sec = km_check_for_seckey( lv, i, NULL ) ? TRUE : FALSE;	
	key_inv = km_get_key_status (lv, i) & KM_FLAG_REVOKED;
    }

    /* Enable / disable toolbar buttons */
    hwnd_child = GetDlgItem(hwnd, IDR_WINPT_KMTB);
    enable_button (hwnd_child, ID_KEYMISC_DELETE);
    enable_button (hwnd_child, ID_KEYMISC_PROPS);
    enable_button (hwnd_child, ID_KEYMISC_SIGN);
    enable_button (hwnd_child, ID_KEYMISC_EXPORT);

    /* Enable / disable menu items */
    hmenu = GetMenu (hwnd);
    set_menu_state (hmenu, ID_KEYMISC_EXPORT, key_selected ? MF_ENABLED : MF_GRAYED);
    set_menu_state (hmenu, ID_KEYMISC_EXPORT_PRIVKEY, key_has_sec ? MF_ENABLED : MF_GRAYED);
    set_menu_state (hmenu, ID_KEYMISC_REVCERT, key_has_sec ? MF_ENABLED : MF_GRAYED);
    set_menu_state (hmenu, ID_KEYMISC_DELETE, key_selected ? MF_ENABLED : MF_GRAYED);
    set_menu_state (hmenu, ID_KEYMISC_PROPS, key_selected ? MF_ENABLED : MF_GRAYED);
    set_menu_state (hmenu, ID_KEYMISC_SIGN, key_selected && !key_inv ? MF_ENABLED : MF_GRAYED);    
} /* update_ui_items */


static void
popup_gpg_readonly (HWND dlg, HMENU hm)
{
    int resids[] = {
	ID_KEYCTX_PASTE,
	ID_KEYCTX_EDIT,
	ID_KEYCTX_SIGN,
	ID_KEYCTX_REV,
	ID_KEYCTX_DEL,
	ID_KEYCTX_ENABLE,
	ID_KEYCTX_DISABLE,
	ID_KEYCTX_RECVFROM,
	ID_KEYCTX_ADDKEY,
	ID_KEYCTX_ADDPHOTO,
	ID_KEYCTX_ADDUID,
	ID_KEYCTX_ADDREV,
	0
    };
    int i;

    if (gpg_read_only == 0)
	return;
    for (i=0; resids[i] != 0; i++)
	set_menu_state (hm, resids[i], MF_GRAYED);
}


static void
menu_gpg_readonly (HWND dlg, HMENU hm, int id)
{
    int key_resids[] = 
    {
	ID_KEYMISC_SIGN,
	ID_KEYMISC_DELETE,
	ID_KEYMISC_REVCERT,
	ID_KEYMISC_IMPORT_HTTP,
	ID_KEYMISC_IMPORT,
	ID_KEYMISC_REFRESH_KEYS,
	ID_KEYMISC_REBUILD,
	ID_KEYMISC_KEYWIZARD,
	ID_KEYMISC_CARDNEW,
	ID_KEYMISC_NEWKEY,
	0,
    };
    int edit_resids[] = 
    {
	ID_KEYMISC_PASTE,
	ID_KEYMISC_OT,
	ID_KEYMISC_EDITKEY,
	0
    };
    int * resids;
    int i;

    
    if (gpg_read_only == 0)
	return;
    switch (id)
    {
    case 0: return;
    case 3: resids = key_resids; break;
    case 1: resids = edit_resids;break;
    }

    for (i=0; resids[i] != 0; i++)
	set_menu_state (hm, resids[i], MF_GRAYED);
}


BOOL CALLBACK
keymanager_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    struct km_info *kmi;
    static km_group_s *gc = NULL;
    static HMENU menu = NULL;
    static int refresh_keys = 0;    
    INITCOMMONCONTROLSEX icex;
    HWND kl;
    HMENU hm;
    gpgme_keycache_t c;
    gpgme_key_t pk;
    km_group_cb_s gcb;    
    struct winpt_key_s k = {0};
    struct http_file_s * http;
    refresh_cache_s rcs = {0};
    char keyid[48], uid[128], type[32], * q;
    const char *t, * host;
    u16 port = 0;
    int idx = 0, i=0, rc;
    size_t size = 0;

    if ((msg != WM_INITDIALOG) 
	&& ((kmi = (struct km_info*)GetWindowLong (dlg, GWL_USERDATA)) == NULL))
        return FALSE;

    switch (msg) {
    case WM_INITDIALOG:
        kmi = new struct km_info;
        memset (kmi, 0, sizeof (struct km_info));
        icex.dwSize = sizeof (INITCOMMONCONTROLSEX);
        icex.dwICC  = ICC_BAR_CLASSES;
        InitCommonControlsEx (&icex);
	kmi->hwnd_sep = regist_sep_wnd (dlg, kmi);
        imagelist_load (dlg);

#ifndef LANG_DE
        SetWindowText( dlg, _("Key Manager") );
#endif        
        menu = LoadMenu( glob_hinst, (LPCSTR)IDR_WINPT_KEYMISC );
#ifndef LANG_DE
        set_menu_text( menu, ID_KEYMISC_QUIT, _("&Quit") );
        set_menu_text( menu, ID_KEYMISC_UID, _("User ID") );
        set_menu_text( menu, ID_KEYMISC_NEWKEY, _("&Expert") );
	set_menu_text( menu, ID_KEYMISC_KEYWIZARD, _("&Normal") );
        set_menu_text( menu, ID_KEYMISC_EDIT, _("Edit") );
        set_menu_text( menu, ID_KEYMISC_SIGN, _("&Sign") );
        set_menu_text( menu, ID_KEYMISC_DELETE, _("&Delete") );
        set_menu_text( menu, ID_KEYMISC_REVCERT, _("&Revoke") );
        set_menu_text( menu, ID_KEYMISC_CHECKSIGS, _("&List Signatures") );
        set_menu_text( menu, ID_KEYMISC_TRUSTPATH, _("List Trust Path") );
        set_menu_text( menu, ID_KEYMISC_PROPS, _("&Properties") );
        set_menu_text( menu, ID_KEYMISC_GPGOPT, _("Options") );
        set_menu_text( menu, ID_KEYMISC_GPGPREFS, _("Preferences") );
        set_menu_text( menu, ID_KEYMISC_SENDRECV, _("Keyserver") );
        set_menu_text( menu, ID_KEYMISC_RECACHE, _("Re&load Key Cache") );
        set_menu_text( menu, ID_KEYMISC_REBUILD, _("R&everify Signatures") );
	set_menu_text( menu, ID_KEYMISC_INFO, _("Info") );
	set_menu_text (menu, ID_KEYMISC_HELP, _("Help"));
#endif
        SetMenu( dlg, menu );
        if( keyring_check_last_access() )
            keycache_set_reload( 1 );
        if( keycache_get_reload( ) )
            keycache_reload( dlg );
        c = keycache_get_ctx( KEYCACHE_PUB );
        if( !c )
            BUG( NULL );
        kmi->keylist_sortby = GPGME_ATTR_USERID;
        Header_SetImageList(ListView_GetHeader(GetDlgItem( dlg, IDC_KEYMISC_KEYLIST )), 
			    glob_imagelist);
        kmi->lv = keylist_load( GetDlgItem( dlg, IDC_KEYMISC_KEYLIST ), c, 
				NULL, KEYLIST_LIST, kmi->keylist_sortby );
        /* init subclassing for the listview */
        kl = GetDlgItem( dlg, IDC_KEYMISC_KEYLIST );
        keylist_proc.dlg = dlg;
        keylist_proc.current = (WNDPROC)keylist_subclass_proc;
        keylist_proc.old = (WNDPROC)GetWindowLong( kl, GWL_WNDPROC );
        if( keylist_proc.old ) {
            if( !SetWindowLong( kl, GWL_WNDPROC, (LONG)keylist_proc.current) ) {
                msg_box( dlg, _("Could not set keylist window procedure."), _("Key Manager"), MB_ERR );
		BUG( NULL );
	    }
        }
        km_groups_new( &gc, GetDlgItem( dlg, IDC_KEYMISC_GROUP ) );
        km_groups_load( gc );
        SetForegroundWindow( dlg );
	
	kmi->statbar = CreateStatusWindow( WS_CHILD | WS_VISIBLE | CCS_BOTTOM, "status bar", dlg, 100000 );
	ShowWindow( kmi->statbar, SW_SHOW );
	km_update_default_key_str (kmi->statbar, &km_def_pos);
	km_complete_status_bar( kmi->statbar, kmi->lv, km_def_pos );

        SetWindowLong (dlg, GWL_USERDATA, (LONG)kmi);
        kmi->toolbar = load_toolbar (dlg, kmi);

	do_center_window( dlg, kmi );
        do_resize_window( dlg, kmi );
        update_ui_items (dlg, kmi->lv);
        return TRUE;
        
    case WM_DESTROY:
	if (kmi->lv) {
            keylist_delete (kmi->lv);
            kmi->lv = NULL;       
	    
	 if (gc) {
            km_groups_release (gc);
            gc = NULL;
        }
       imagelist_destroy ();

       char buf[32];
       ltoa (kmi->pos_x, buf, 10);
       set_reg_entry( HKEY_CURRENT_USER, "Software\\WinPT", "KM_Pos_X", buf );
       ltoa (kmi->pos_y, buf, 10);
       set_reg_entry (HKEY_CURRENT_USER, "Software\\WinPT", "KM_Pos_Y", buf);
       /* Remove runtime information. This should be the last action taken here. */
       delete kmi; kmi = NULL;
       SetWindowLong (dlg, GWL_USERDATA, NULL);	   
       keycache_set_reload (refresh_keys);
       return FALSE;

    case WM_MOVE:	
        /* kmi->pos_x = (int)(short)LOWORD(lparam);
           kmi->pos_y = (int)(short)HIWORD(lparam); */
        RECT r;
        GetWindowRect (dlg, &r);
	kmi->pos_x = r.left;
	kmi->pos_y = r.top;	
	break;
        

    case WM_RBUTTONUP:
	do_create_minpopup (dlg);
	break;

    case WM_NOTIFY:		
        NMHDR * notify;
        POINT p;
        HMENU popup;
        
        notify = (NMHDR *)lparam;
	if (notify == NULL)
	    break;
	switch (notify->code) 
	{
        case TBN_QUERYDELETE:
	    SetWindowLong(dlg, DWL_MSGRESULT, TRUE);
            return TRUE;
        
        case TBN_QUERYINSERT:
	    SetWindowLong(dlg, DWL_MSGRESULT, TRUE);
            return TRUE;

        case TBN_GETBUTTONINFO:
            LPTBNOTIFY lpTbNotify;
            lpTbNotify = (LPTBNOTIFY)lparam;
            if (lpTbNotify->iItem < (sizeof(myb) / sizeof(mybuttons))) {
                lpTbNotify->tbButton.iBitmap = imagelist_getindex(myb[lpTbNotify->iItem].icon);
                lpTbNotify->tbButton.idCommand = myb[lpTbNotify->iItem].command;
                lpTbNotify->tbButton.fsState = TBSTATE_ENABLED;
                lpTbNotify->tbButton.fsStyle = TBSTYLE_BUTTON;
                lpTbNotify->tbButton.iString = 0;
                strncpy (lpTbNotify->pszText, myb[lpTbNotify->iItem].text, lpTbNotify->cchText);
                SetWindowLong(dlg, DWL_MSGRESULT, TRUE);
            }
            return TRUE;

        case TBN_RESET: /* Restore last saved toolbar status */
            TBSAVEPARAMS tbsp;

            tbsp.hkr = HKEY_CURRENT_USER;
            tbsp.pszSubKey = "Software\\WinPT";
            tbsp.pszValueName = "KM_toolbar";
            SendMessage (notify->hwndFrom, TB_SAVERESTORE, FALSE, (LPARAM)&tbsp); 
            break;

        case TBN_BEGINADJUST: /* User is about to change the toolbar. Save it */
            tbsp.hkr = HKEY_CURRENT_USER;
            tbsp.pszSubKey = "Software\\WinPT";
            tbsp.pszValueName = "KM_toolbar";
            SendMessage (notify->hwndFrom, TB_SAVERESTORE, TRUE, (LPARAM)&tbsp); 
            break;

        case TBN_ENDADJUST: /* User has finished customizing the toolbar. Save it. */
            tbsp.hkr = HKEY_CURRENT_USER;
            tbsp.pszSubKey = "Software\\WinPT";
            tbsp.pszValueName = "KM_toolbar";
            SendMessage (notify->hwndFrom, TB_SAVERESTORE, TRUE, (LPARAM)&tbsp); 
            break;

        case TTN_GETDISPINFO:
            LPTOOLTIPTEXT lpttt;
            lpttt = (LPTOOLTIPTEXT)lparam;

            lpttt->hinst = NULL;
            switch (lpttt->hdr.idFrom) {
            case ID_KEYMISC_DELETE:
                lpttt->lpszText = (char *)_("Delete key from keyring"); 
		break;

            case ID_KEYMISC_PROPS:
                lpttt->lpszText = (char *)_("Show key properties"); 
		break;

            case ID_KEYMISC_SIGN:
                lpttt->lpszText = (char *)_("Sign key"); 
		break;

            case ID_KEYMISC_IMPORT:
                lpttt->lpszText = (char *)_("Import key to keyring");
		break;

            case ID_KEYMISC_EXPORT:
                lpttt->lpszText = (char *)_("Export key to a file"); 
		break;
            }
            return TRUE;
            
        case LVN_ITEMCHANGED:
            if (((LPNMLISTVIEW)lparam)->uNewState) /* item selected? */
	    { 
                update_ui_items (dlg, kmi->lv);
                return TRUE;
            }
            break;

	case NM_RCLICK:
            if (notify->idFrom == IDC_KEYMISC_KEYLIST) 
	    {
		if (listview_get_curr_pos (kmi->lv) == -1)
		    return TRUE; /* Popup only when a item was selected */
		do_check_cache (kmi->lv, dlg, kmi->statbar);
		GetCursorPos (&p);
		hm = LoadMenu (glob_hinst, MAKEINTRESOURCE (IDR_WINPT_KEYMISC_CTX));
		popup = GetSubMenu (hm, 0);
	    #ifndef LANG_DE
		set_menu_text( popup, ID_KEYCTX_UID_COPY, _("Copy User ID to Clipboard") );
		set_menu_text( popup, ID_KEYCTX_COPY, _("Copy Key to Clipboard") );
		set_menu_text( popup, ID_KEYCTX_PASTE, _("Paste Key from Clipboard") );            
		set_menu_text( popup, ID_KEYCTX_RECVFROM, _("Refresh from Keyserver") );
	    #endif
		idx = listview_get_curr_pos (kmi->lv);
		if (km_check_for_seckey (kmi->lv, idx, &i) && i == 0) 
		{
		    set_menu_state (popup, ID_KEYCTX_SETDEFKEY, MF_ENABLED);
		    set_menu_state (popup, ID_KEYCTX_MAXTRUST, MF_ENABLED);
		}
		if( !km_check_for_seckey( kmi->lv, idx, NULL ) ) {
		    set_menu_state( popup, ID_KEYCTX_REV, MF_DISABLED|MF_GRAYED );
		    set_menu_state( popup, ID_KEYCTX_ADDKEY, MF_DISABLED|MF_GRAYED );
		    set_menu_state( popup, ID_KEYCTX_ADDUID, MF_DISABLED|MF_GRAYED );
		    set_menu_state( popup, ID_KEYCTX_ADDREV, MF_DISABLED|MF_GRAYED );
		    set_menu_state( popup, ID_KEYCTX_ADDPHOTO, MF_DISABLED|MF_GRAYED );
		    set_menu_state (popup, ID_KEYCTX_SETPREFKS, MF_DISABLED|MF_GRAYED);
		}
		else if( km_check_for_seckey( kmi->lv, idx, NULL ) 
		      && km_key_is_v3( kmi->lv, idx ) ) 
		{
		    /* PGP 2 keys, version 3 have no no support for photo-id's,
		       designated revokers and secondary keys. */
		    set_menu_state( popup, ID_KEYCTX_ADDKEY, MF_DISABLED|MF_GRAYED );
		    set_menu_state( popup, ID_KEYCTX_ADDREV, MF_DISABLED|MF_GRAYED );
		    set_menu_state( popup, ID_KEYCTX_ADDPHOTO, MF_DISABLED|MF_GRAYED );
		}
		if( km_get_key_status( kmi->lv, idx ) & KM_FLAG_DISABLED )
		    set_menu_state( popup, ID_KEYCTX_DISABLE, MF_DISABLED|MF_GRAYED );
		else
		    set_menu_state( popup, ID_KEYCTX_ENABLE, MF_DISABLED|MF_GRAYED );
		if (km_get_key_status (kmi->lv, idx) & KM_FLAG_REVOKED)
		    set_menu_state (popup, ID_KEYCTX_SIGN, MF_DISABLED|MF_GRAYED);
		if( mapi_init() )
		    set_menu_state( popup, ID_KEYCTX_SENDMAIL, MF_DISABLED|MF_GRAYED );
		popup_gpg_readonly (dlg, popup);
		TrackPopupMenu( popup, TPM_RIGHTALIGN, p.x, p.y, 0, dlg, NULL );
		DestroyMenu( popup );
		DestroyMenu( hm );
		return TRUE;
	    }
	    if( notify->idFrom == IDC_KEYMISC_GROUP ) {
		HWND tree = GetDlgItem( dlg, IDC_KEYMISC_GROUP );
		if( TreeView_GetSelection( tree ) ) {
		    GetCursorPos( &p );
		    hm = LoadMenu( glob_hinst, MAKEINTRESOURCE(IDR_WINPT_GROUP_CTX) );
		    popup = GetSubMenu( hm, 0 );
		    if( km_index == -1 )
			set_menu_state( popup, ID_GROUP_PASTE, MF_DISABLED|MF_GRAYED );
		    set_menu_text( popup, ID_GROUP_PASTE, _("Paste into this group") );
		    set_menu_text( popup, ID_GROUP_DELETE, _("Delete") );
		    TrackPopupMenu( popup, TPM_RIGHTALIGN, p.x, p.y, 0, dlg, NULL );
		    DestroyMenu( popup );
		    DestroyMenu( hm );
		    return TRUE;
		}
	    }
	    break;

        case LVN_COLUMNCLICK:
	    if (notify->idFrom == IDC_KEYMISC_KEYLIST) {
                NMLISTVIEW * p = (LPNMLISTVIEW) lparam;
                int sortby = 0;
                switch (p->iSubItem) {
                case 0:  sortby = GPGME_ATTR_USERID; break;
                case 1:  sortby = GPGME_ATTR_KEYID; break;
                case 2:  sortby = GPGME_ATTR_IS_SECRET; break;
                case 3:  sortby = GPGME_ATTR_LEN; break;
                case 5:  sortby = GPGME_ATTR_VALIDITY; break;
		case 6:  sortby = GPGME_ATTR_OTRUST; break;
                case 7:  sortby = GPGME_ATTR_CREATED; break;
		case 8:  sortby = GPGME_ATTR_ALGO; break;
                default: return TRUE; //sortby = GPGME_ATTR_USERID; break;
                }

                if ((kmi->keylist_sortby & ~KEYLIST_SORT_DESC) == sortby)
                    kmi->keylist_sortby ^= KEYLIST_SORT_DESC;
                else
                    kmi->keylist_sortby = sortby;
                keylist_sort (kmi->lv, kmi->keylist_sortby);
                return TRUE;
            }
            break;
        }
        break;
	}

    case WM_WINDOWPOSCHANGING:
        if (((WINDOWPOS*)lparam)->cx < 400)
            ((WINDOWPOS*)lparam)->cx = 400;
        if (((WINDOWPOS*)lparam)->cy < 200)
            ((WINDOWPOS*)lparam)->cy = 200;
         return TRUE;
        
    case WM_SIZE:
        do_resize_window (dlg, kmi);
        return TRUE;
        
    case WM_SYSCOMMAND:
        if( LOWORD (wparam) == SC_CLOSE )
            EndDialog( dlg, TRUE );
        return FALSE;
        
    case WM_MENUSELECT:
	menu_gpg_readonly (dlg, (HMENU)lparam, LOWORD (wparam));
	break;

    case WM_COMMAND:
        if( gnupg_access_keyring( 1 ) ) {
            msg_box( dlg, _("Could not access public keyring"), _("Key Manager"), MB_ERR );
            return FALSE;
        }
	do_check_cache( kmi->lv, dlg, kmi->statbar );
        switch( LOWORD( wparam ) ) {
        case ID_KEYMISC_QUIT:
            EndDialog( dlg, TRUE );
            return TRUE;
	    
	case ID_KEYMISC_MAIL:
	    DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_MAIL, GetDesktopWindow (),
			    winpt_mail_proc, NULL);
	    break;
            
        case ID_KEYMISC_DELETE:
            if (km_delete_keys (kmi->lv, dlg))
                return FALSE;
	    refresh_keys = 1;
            return TRUE;
            
        case ID_KEYMISC_SIGN:
            gpgme_key_t key;

            if ( (idx = listview_get_curr_pos( kmi->lv )) == -1 ) {
                msg_box( dlg, _("Please select a key."),  _("Key Manager"),
                         MB_ERR );
                return FALSE;
            }
            if( !km_check_key_status( kmi->lv, idx ) )
                return FALSE;
            listview_get_item_text( kmi->lv, idx, 1, keyid, sizeof keyid );
            listview_get_item_text( kmi->lv, idx, 2, type, sizeof type -1 );
	    rc = get_pubkey( keyid, &key );
	    if( rc )
		BUG( dlg );
            if( key )
                dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_KEYSIGN, dlg,
                                  keysign_dlg_proc, (LPARAM)key, 
                                  _("Key Signing"), IDS_WINPT_KEYSIGN);
            return TRUE;
            
        case ID_KEYMISC_REVCERT:
	    idx = listview_get_curr_pos( kmi->lv );
            if( idx == -1 ) {
                msg_box( dlg, _("Please select a key."), _("Key Manager"), MB_ERR );
                return FALSE;
            }
            listview_get_item_text( kmi->lv, idx, 0, uid, sizeof uid -1 );
            listview_get_item_text( kmi->lv, idx, 1, keyid, sizeof keyid-1 );
            if ( !km_check_for_seckey( kmi->lv, idx, NULL ) ) {
                msg_box( dlg, _("There is no secret key available!"), _("Key Manager"), MB_ERR );
                return FALSE;
            }
            
            {
                char t[128];
                listview_get_item_text( kmi->lv, idx, 5, t, sizeof t -1 );
                if( strchr( t, 'R' ) ) {
                    msg_box( dlg, _("Key already revoked!"), _("Key Manager"), MB_INFO );
                    return FALSE;
                }
            }
            
	    q = km_quote_uid (uid);
            k.key_pair = 1;
            k.keyid = keyid;
            k.uid = q;
	    k.is_protected = km_check_if_protected( kmi->lv, idx );
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_KEYREVOKE, dlg,
                             key_revoke_dlg_proc, (LPARAM)&k,
                             _("Key Revocation"), IDS_WINPT_KEYREVOKE );
	    free_if_alloc( q );
            return TRUE;
            
        case ID_KEYMISC_TRUSTPATH:
	    idx = listview_get_curr_pos( kmi->lv );
	    if( idx == -1 ) {
                msg_box( dlg, _("Please select a key."), _("Key Manager"), MB_ERR );
                return FALSE;
            }
            listview_get_item_text( kmi->lv, idx, 0, uid, sizeof uid -1 );
            listview_get_item_text( kmi->lv, idx, 1, keyid, sizeof keyid -1 );
            if( km_check_for_seckey( kmi->lv, idx, NULL ) ) {
                msg_box( dlg, _("It does not make any sense with a key pair!"), _("Key Manager"), MB_OK );
                return FALSE;
            }
            k.keyid = keyid;
            k.uid = uid;
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_KEYTRUST, dlg, 
                              keytrust_dlg_proc, (LPARAM)&k, 
                              _("List Trust Path"), IDS_WINPT_KEYTRUST );
            return TRUE;
            
        case ID_KEYMISC_CHECKSIGS:
	    idx = listview_get_curr_pos( kmi->lv );
            if( idx == -1 ) {
                msg_box( dlg, _("Please select a key."), _("Key Manager"), MB_ERR );
                return FALSE;
            }
            listview_get_item_text( kmi->lv, idx, 0, uid, sizeof uid-1 );
            listview_get_item_text( kmi->lv, idx, 1, keyid, sizeof keyid-1 );
            k.keyid = keyid;
            k.uid = uid;
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_KEYSIG, dlg, 
                              keysig_dlg_proc, (LPARAM)&k,
                              _("Key Signature List" ), IDS_WINPT_KEYSIG );
            return TRUE;
            
        case ID_KEYMISC_PROPS:
	    idx = listview_get_curr_pos( kmi->lv );
            if( idx == -1 ) {
                msg_box( dlg, _("Please select a key."), _("Key Manager"), MB_ERR );
                return FALSE;
            }
            listview_get_item_text( kmi->lv, idx, 1, keyid, sizeof keyid -1 );
            listview_get_item_text( kmi->lv, idx, 2, type, sizeof type -1 );
            k.key_pair = 0;
            k.keyid = keyid;
            if( !strcmp( type, "pub/sec" ) || !strcmp( type, "pub/crd" ) )
                k.key_pair = 1;
	    k.callback.ctl = kmi->lv;
	    k.callback.idx = idx;
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_KEYPROPS, dlg,
                              keyprops_dlg_proc, (LPARAM)&k,
                              _("Key Properties"), IDS_WINPT_KEYPROPS );
	    if (k.callback.new_val != 0)
	    {
		t = get_key_trust_str (k.callback.new_val);
		listview_add_sub_item (kmi->lv, idx, 6, t);
	    }
            return TRUE;
            
        case ID_KEYMISC_RECACHE:
            /* If there is already a reload request, don't bother the user with a message. */
            if( keycache_get_reload() == 1 )
                idx = IDYES;
            else {
                char t[256];
                _snprintf( t, sizeof t -1,
                           _("This is only useful when the keyring has been "
                             "modified (sign a key...).\n"
                             "Do you really want to reload the keycache?") );
                idx = msg_box( dlg, t, _("Key Manager"), MB_YESNO );
            }
            if( idx == IDYES ) {
                rcs.kr_reload = rcs.kr_update = 1;
                rcs.tr_update = 0;
                DialogBoxParam( glob_hinst, (LPCSTR)IDD_WINPT_KEYCACHE, dlg,
                                keycache_dlg_proc, (LPARAM)&rcs );
                c = keycache_get_ctx( 1 );
		if( !c )
		    BUG( dlg );
                keylist_reload( kmi->lv, c, KEYLIST_LIST, GPGME_ATTR_USERID );
		refresh_keys = 0;
            }
            return TRUE;
            
        case ID_KEYMISC_REBUILD:
            rcs.kr_reload = rcs.kr_update = 1;
            rcs.tr_update = 1;
            DialogBoxParam( glob_hinst, (LPCSTR)IDD_WINPT_KEYCACHE, dlg,
                            keycache_dlg_proc, (LPARAM)&rcs );
            c = keycache_get_ctx( 1 );
	    if( !c )
		BUG( dlg );
            keylist_reload( kmi->lv, c, KEYLIST_LIST, GPGME_ATTR_USERID );
            return TRUE;
            
        case ID_KEYMISC_NEWKEY:
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_KEYGEN, dlg,
                              keygen_dlg_proc, NULL, _("Key Generation"), 
			      IDS_WINPT_KEYGEN );
            if( keycache_get_reload( ) )
                send_cmd_id( dlg, ID_KEYMISC_RECACHE );
            return TRUE;

	case ID_KEYMISC_CARDNEW:
	    if( !scard_support ) {
		msg_box( dlg, _("Smart Card support is not available."), _("Key Manager"), MB_INFO );
		return FALSE;
	    }
	    dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_CARD_KEYGEN, dlg,
			      card_keygen_dlg_proc, NULL, _("Card Key Generation"),
			      IDS_WINPT_CARD_KEYGEN );
	    if( keycache_get_reload() )
		send_cmd_id( dlg, ID_KEYMISC_RECACHE );
	    return TRUE;

	case ID_KEYMISC_KEYWIZARD:
	    dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_KEYWIZARD, dlg,
			      keygen_wizard_dlg_proc, NULL, _("Key Generation Wizard"),
			      IDS_WINPT_KEYWIZARD );
	    if( keycache_get_reload() )
		send_cmd_id( dlg, ID_KEYMISC_RECACHE );
	    return TRUE;
            
        case ID_KEYMISC_SENDRECV:
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_KEYSERVER, dlg,
                              keyserver_dlg_proc, NULL, _("Keyserver Access"), 
			      IDS_WINPT_KEYSERVER );
            return TRUE;
            
        case ID_KEYMISC_GPGPREFS:
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_GPGPREFS, dlg,
                              gpgprefs_dlg_proc, NULL, _("GnuPG Preferences"), 
			      IDS_WINPT_GPGPREFS );
            return TRUE;
            
        case ID_KEYMISC_GPGOPT:
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_GPGOPT, dlg,
                              gpgopt_dlg_proc, NULL, _("GnuPG Options" ), 
			      IDS_WINPT_GPGOPT );
            return TRUE;
            
        case ID_KEYMISC_IMPORT:
            t = get_filename_dlg( dlg, FILE_OPEN, _("Choose Name of the Key File"), NULL, NULL );
            if( t )
                km_file_import( dlg, t );
            return TRUE;

	case ID_KEYMISC_IMPORT_HTTP:
	    http = (struct http_file_s *)get_http_file_dlg (dlg);
	    if (http->cancel == 0)
		km_http_import (dlg, http->url);
	    delete http;
	    break;
            
        case ID_KEYMISC_EXPORT:
	    idx = listview_get_curr_pos( kmi->lv );
            if( idx == -1 ) {
                msg_box( dlg, _("Please select a key."), _("Key Manager"), MB_ERR );
                return FALSE;
            }
	    if( listview_count_items( kmi->lv, 1 ) > 1 )
		strcpy( keyid, _("Exported_GPG_Keys.asc") );
	    else {
		listview_get_item_text( kmi->lv, idx, 1, keyid, sizeof keyid-1 );
		strcat( keyid, ".asc" );
	    }
            t = get_filename_dlg( dlg, FILE_SAVE, _("Choose Name for Key File"), NULL, keyid );
            if( !t || overwrite_file( t ) == 0 )
                return FALSE;
            km_file_export( dlg, kmi->lv, t );
            return TRUE;
            
        case ID_KEYMISC_EXPORT_PRIVKEY:
	    idx = listview_get_curr_pos( kmi->lv );
            if( idx == -1 ) {
                msg_box( dlg, _("Please select a key."), _("Key Manager"), MB_ERR );
                return FALSE;
            }
            if( !km_check_for_seckey( kmi->lv, idx, NULL ) ) {
                msg_box( dlg, _("There is no corresponding secret key for this key."), 
                        _("Key Manager"), MB_ERR );
                return FALSE;
            }
            if( listview_count_items( kmi->lv, 1 ) > 1 ) {
                msg_box( dlg, _("You can only export one secret key."), _("Key Manager"), MB_ERR );
                return FALSE;
            }
            idx = msg_box( dlg, 
                          _("This operation will export your *SECRET* key!\n\n"
                            "Never send this key to ANYONE, it should be available\n"
                            "ONLY on your machine and you may use this function\n"
                            "to copy the key to a safe place.\n\n"
                            "Do you really want to export the key?"),
                          _("WARNING"), MB_INFO|MB_YESNO );
            if( idx == IDYES ) {
		idx = listview_get_curr_pos( kmi->lv );
		listview_get_item_text( kmi->lv, idx, 1, keyid, sizeof (keyid)-8 );
		strcat( keyid, ".sec.asc" );
                t = get_filename_dlg( dlg, 1, _("Choose Name for Key File"), NULL, keyid );
                if( t && !file_exist_check( t ) ) {
		    idx = log_box( _("Key Manager"), MB_INFO|MB_YESNO,
				   _("File %s already exists.\nOverwrite?"), t );
		    if( idx == IDNO )
			return FALSE;
                }
		if( t )
		    km_privkey_export( dlg, kmi->lv, t );
            }
            break;

	case ID_KEYMISC_INFO:
	    dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_ABOUT, glob_hwnd,
			      about_winpt_dlg_proc, NULL, _("About WinPT"),
			      IDS_WINPT_ABOUT );
	    break;

	case ID_KEYMISC_HELP:
	    ShellExecute (dlg, "open", "winpt.chm", NULL, NULL, SW_SHOW);
	    break;

	case ID_KEYMISC_OT:
	    dialog_box_param( glob_hinst, (LPCTSTR)IDD_WINPT_OWNERTRUST, glob_hwnd,
			      ownertrust_dlg_proc, NULL, 
			      _("Ownertrust"), IDS_WINPT_OWNERTRUST );
	    break;

	case ID_KEYMISC_EDITKEY:
	    idx = listview_get_curr_pos (kmi->lv);
	    if( idx == -1 )
		break;
	    listview_get_item_text( kmi->lv, idx, 1, keyid, sizeof keyid-1 );
	    k.is_protected = km_check_if_protected( kmi->lv, idx );
	    k.key_pair = km_check_for_seckey( kmi->lv, idx, NULL );
	    k.keyid = keyid;
	    k.flags = km_get_key_status( kmi->lv, idx );
	    dialog_box_param( glob_hinst, (LPCTSTR)IDD_WINPT_KEYEDIT, dlg,
			      keyedit_main_dlg_proc, (LPARAM)&k,
			      _("Key Edit"), IDS_KEYCTX_EDIT );
	    break;
            
        case ID_GROUP_NEW:
            memset( &gcb, 0, sizeof gcb );
            gcb.gc = gc;
            dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_GROUP, glob_hwnd, 
                             group_manager_dlg_proc, (LPARAM)&gcb, _("New Group"), 
			     IDS_WINPT_GROUP );
            if( gcb.use_name )
                treeview_add_item( GetDlgItem(dlg, IDC_KEYMISC_GROUP), NULL, gcb.name );
            return TRUE;
            
        case ID_KEYMISC_COPY:
            km_index = listview_get_curr_pos( kmi->lv );
            km_clip_export( dlg, kmi->lv );
            break;
            
	case ID_KEYMISC_SELALL:
	    listview_select_all (kmi->lv);
	    break;

        case ID_KEYMISC_PASTE:
            km_index = -1;
            km_clip_import (dlg);
            break;
            
	case ID_KEYCTX_SETPREFKS:
	    /* XXX: todo */
	    break;

        case ID_KEYMISC_REFRESH_KEYS:
	    if (listview_count_items (kmi->lv, 1) == 0)
	    {
		msg_box (dlg, _("No key was selected, select all by default."), _("Key Manager"), MB_INFO);
		listview_select_all (kmi->lv);
	    }
            km_refresh_from_keyserver (kmi->lv, dlg);
            break;
                        
	/** Context menu **/
	case ID_KEYCTX_EDIT:
	    send_cmd_id( dlg, ID_KEYMISC_EDITKEY );
	    break;

        case ID_KEYCTX_PROPS:
	    send_cmd_id( dlg, ID_KEYMISC_PROPS );
	    break;

        case ID_KEYCTX_UIDS: 
	    send_cmd_id( dlg, ID_KEYMISC_UID ); 
	    break;

        case ID_KEYCTX_SIGN: 
	    send_cmd_id( dlg, ID_KEYMISC_SIGN ); 
	    break;

        case ID_KEYCTX_DEL: 
	    send_cmd_id (dlg, ID_KEYMISC_DELETE); 
	    break;

        case ID_KEYCTX_REV: 
	    send_cmd_id( dlg, ID_KEYMISC_REVCERT ); 
	    break;

	case ID_KEYCTX_SENDMAIL:
	    km_send_to_mail_recipient( kmi->lv, dlg );
	    break;

        case ID_KEYCTX_KS_DEFAULT:
	    host = kserver_get_hostname (0, -1, &port);
	    km_send_to_keyserver (kmi->lv, dlg, host, port);
	    break;

	case ID_KEYCTX_ADDKEY:
	    idx = listview_get_curr_pos( kmi->lv );
            listview_get_item_text( kmi->lv, idx, 1, keyid, sizeof keyid - 1 );
            k.key_pair = km_check_for_seckey( kmi->lv, idx, NULL );
	    k.is_protected = km_check_if_protected( kmi->lv, idx );
	    k.keyid = keyid;
	    keyedit_add_subkey( &k, dlg, NULL );
	    break;

	case ID_KEYCTX_ADDUID:
	    idx = listview_get_curr_pos( kmi->lv );
	    listview_get_item_text( kmi->lv, idx, 1, keyid, sizeof keyid-1 );
	    k.key_pair = km_check_for_seckey( kmi->lv, idx, NULL );
	    k.is_protected = km_check_if_protected( kmi->lv, idx );
	    k.keyid = keyid;
	    keyedit_add_userid( &k, dlg, NULL );
	    break;

	case ID_KEYCTX_ADDREV:
	    idx = listview_get_curr_pos( kmi->lv );
	    listview_get_item_text( kmi->lv, idx, 1, keyid, sizeof keyid-1 );	    
	    k.keyid = keyid;
	    k.is_protected = km_check_if_protected( kmi->lv, idx );
	    k.key_pair = km_check_for_seckey( kmi->lv, idx, NULL );
	    keyedit_add_revoker( &k, dlg );
	    break;

	case ID_KEYCTX_ADDPHOTO:
	    idx = listview_get_curr_pos( kmi->lv );
	    listview_get_item_text( kmi->lv, idx, 1, keyid, sizeof keyid-1 );
	    k.keyid = keyid;
	    k.is_protected = km_check_if_protected( kmi->lv, idx );
	    k.key_pair = km_check_for_seckey( kmi->lv, idx, NULL );
	    keyedit_add_photo( &k, dlg );
	    break;

	case ID_KEYCTX_KS_NL:
	case ID_KEYCTX_KS_PL:
	case ID_KEYCTX_KS_AT:
	case ID_KEYCTX_KS_DE:
	case ID_KEYCTX_KS_DK:
	case ID_KEYCTX_KS_CZ:
	case ID_KEYCTX_KS_ES:
	case ID_KEYCTX_KS_UK:
	    host = kserver_get_hostname (LOWORD (wparam) - 40107, 0, &port);
	    km_send_to_keyserver (kmi->lv, dlg, host, port);
	    break;

        case ID_KEYCTX_RECVFROM: 
	    km_refresh_from_keyserver (kmi->lv, dlg);
	    break;

	case ID_KEYCTX_UID_COPY:
	    /* XXX: add generic function to support multiple selection
	            with a callback */
	    idx = listview_get_curr_pos( kmi->lv );
	    listview_get_item_text( kmi->lv, idx, 0, uid, sizeof uid-1 );
	    set_clip_text( NULL, uid, strlen( uid ) );
	    break;

	case ID_KEYCTX_KEYID_COPY:
	    idx = listview_get_curr_pos( kmi->lv );
	    listview_get_item_text( kmi->lv, idx, 1, uid, sizeof uid-1 );
	    set_clip_text( NULL, uid, strlen( uid ) );
	    break;

	case ID_KEYCTX_FPR_COPY:
	    idx = listview_get_curr_pos( kmi->lv );
	    listview_get_item_text( kmi->lv, idx, 1, uid, sizeof uid-1 );
	    get_pubkey( uid, &pk );
	    if( pk ) {
		const char * s = get_key_fpr( pk );
		set_clip_text( NULL, s? s : "", s? strlen( s ): 0 );
	    }
	    break;

	case ID_KEYCTX_KINFO_COPY:
	    idx = listview_get_curr_pos( kmi->lv );
	    listview_get_item_text( kmi->lv, idx, 1, uid, sizeof uid-1 );
	    km_set_clip_info( uid );	    
	    break;

        case ID_KEYCTX_COPY:
            km_index = listview_get_curr_pos (kmi->lv);
            km_clip_export (dlg, kmi->lv);
            break;

        case ID_KEYCTX_PASTE:  
            km_index = -1;
            km_clip_import (dlg);
            break;

	case ID_KEYCTX_DISABLE:
	    idx = listview_get_curr_pos (kmi->lv);
	    km_enable_disable_key (kmi->lv, dlg, idx, 0);
	    break;

	case ID_KEYCTX_ENABLE:
	    idx = listview_get_curr_pos (kmi->lv);
	    km_enable_disable_key (kmi->lv, dlg, idx, 1);
	    break;

        case ID_KEYCTX_LISTSIGS: 
	    send_cmd_id (dlg, ID_KEYMISC_CHECKSIGS);
	    break;

	case ID_KEYCTX_MAXTRUST:
	    idx = listview_get_curr_pos (kmi->lv);
	    km_set_implicit_trust (dlg, kmi->lv, idx);
	    break;

        case ID_KEYCTX_SETDEFKEY:
            idx = listview_get_curr_pos (kmi->lv);
	    if (km_check_key_status (kmi->lv, idx)) 
	    {
		listview_get_item_text (kmi->lv, idx, 1, keyid, sizeof keyid -1);
		rc = set_gnupg_default_key (keyid);
		if (rc)
		    msg_box( dlg, winpt_strerror (rc), _("Key Manager"), MB_ERR);
		km_update_default_key_str (kmi->statbar, &km_def_pos);
	    }
            break;
            
        case ID_GROUP_PASTE:
            km_groups_add (gc, kmi->lv, km_index);
            break;
            
        case ID_GROUP_DELETE:
            km_groups_del (gc);
            break;
        }	
        break;
    }
    
    return FALSE;
} /* keymanager_dlg_proc */


