/* wptCardDlg.cpp - Smart Card support
 *	Copyright (C) 2003, 2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <commctrl.h>
#include <ctype.h>

#include "../resource.h"
#include "wptTypes.h"
#include "wptW32API.h"
#include "wptErrors.h"
#include "wptRegistry.h"
#include "wptVersion.h"
#include "wptCommonCtl.h"
#include "wptDlgs.h"
#include "wptGPG.h"
#include "wptUTF8.h"
#include "wptCard.h"

int keygen_check_date (SYSTEMTIME * st);

static const char * sex[] = {"Male", "Female", "Undefined", NULL};
static const char * lang[] = {"Undefined", "cs", "de", "en", "es", "fr", "hu",
			      "it", "nl", "pt", "ro", "ru", "zh", "at", 
			      NULL};

static pin_cb_ctx_s pincb;

struct {
    int ctlid;
    gpgme_attr_t attr;
    const char * err;
} attr_tab[] = {
    {IDC_CEDIT_AID,    GPGME_ATTR_CARD_AID,    ""},
    {IDC_CEDIT_VENDOR, GPGME_ATTR_CARD_VENDOR, "No Vendor"},
    {IDC_CEDIT_VERSION,GPGME_ATTR_CARD_VER,    "No Version"},
    {IDC_CEDIT_SERIAL, GPGME_ATTR_CARD_SERIAL, "No Serial-No"},
    {IDC_CEDIT_NAME,   GPGME_ATTR_CARD_NAME,   "No Name"},
    {IDC_CEDIT_NAME2,  GPGME_ATTR_CARD_NAME2,  "No Surname"},
    {IDC_CEDIT_KEYURL, GPGME_ATTR_CARD_URL,    "No Key-URL"},
    {IDC_CEDIT_LOGIN,  GPGME_ATTR_CARD_LOGIN,  "No Login name"},
    {0},
};


#define card_get_string(card, what) \
    gpgme_card_get_string_attr ((card), (what), NULL, 0)
#define card_get_ulong(card, what)  \
    gpgme_card_get_ulong_attr ((card), (what), NULL, 0)
#define card_get_fpr(card, idx)     \
    gpgme_card_get_string_attr ((card), GPGME_ATTR_CARD_FPR, NULL, (idx)-1)


static int
idx_from_lang( const char * _lang )
{
    const char * s;
    int i;

    if (!_lang)
	return 0;
    for (i=0; (s = lang[i]); i++) {
	if( !strcmp( _lang, s ) )
	    return i;
    }
    return 0;
} /* idx_from_lang */


int
show_card_status (void)
{
    int rc = 0;
    int cardstat;

    cardstat = pcsc_get_card_status ();
    if ((cardstat & CARD_STATE_UNAWARE) || (cardstat & CARD_STATE_UNAVAIL))
	rc = WPTERR_NOREADER;
    else if (cardstat & CARD_STATE_EMPTY)
	rc = WPTERR_NOCARD;	
    if (rc) {
	msg_box (NULL, winpt_strerror (rc), _("Card Manager"), MB_ERR);
	return -1;
    }
    return 0;
} /* show_card_status */


gpgme_card_t
gpg_load_scard (void)
{
    gpgme_error_t rc;
    gpgme_card_t card = NULL;
    gpgme_editcard_t ec;
    gpgme_ctx_t ctx;
    const char * s;
    struct card_cb_s cb = {0};

    rc = gpgme_new (&ctx);
    if (!rc)
	rc = gpgme_editcard_new (&ec);
    if (rc)
	BUG (0);
    gpgme_enable_logging (ctx);
    gpgme_editcard_set_callback (ec, card_callback, &cb);
    gpgme_set_edit_ctx (ctx, ec, 0);
    rc = gpgme_op_statuscard (ctx, &card);
    if (rc) {
	gpgme_show_error (NULL, rc, ctx, _("Card Manager"), MB_ERR);
	goto leave;
    }
    s = gpgme_card_get_string_attr (card, GPGME_ATTR_CARD_AID, NULL, 0);
    if (!s || strncmp (s, "D276000124", 10)) {
	msg_box (NULL, winpt_strerror (WPTERR_NOPGPCARD), "WinPT", MB_ERR);
	gpgme_card_release (card);
	card = NULL;
    }

leave:
    gpgme_editcard_release (ec);
    gpgme_release (ctx);
    return card;
} /* gpg_load_scard */


static void
print_fpr (HWND dlg, int id, const char * fpr)
{
    char buf[128], dig[2];    
    size_t i, c;

    if (!fpr)
	strcpy( buf, _("No Fingerprint") );
    else {
	memset( buf, 0, sizeof (buf) );
	for( i=0, c=0; i < strlen( fpr ); i++ ) {
	    dig[0] = fpr[i]; dig[1] = 0;
	    strcat( buf, dig );
	    if( ++c == 4 ) {
		strcat( buf, " " );
		c=0;
	    }
	}
    }
    SetDlgItemText( dlg, id, buf );
} /* print_fpr */


static int
card_status( HWND dlg, gpgme_card_t card )
{
    static int fprbuf[] = {0, IDC_CEDIT_FPR1, IDC_CEDIT_FPR2, IDC_CEDIT_FPR3, 0};
    const char * s;
    int idx=0;
    u32 t;

    s = card_get_string( card, GPGME_ATTR_CARD_AID );
    if( !s ) {
	msg_box( dlg, _("No OpenPGP smart card detected."), "WinPT", MB_ERR );
	return -1;
    }
    SetDlgItemText( dlg, IDC_CEDIT_AID, s );

    t = card_get_ulong( card, GPGME_ATTR_CARD_SIGCOUNT );
    SetDlgItemInt( dlg, IDC_CEDIT_SIGCOUNT, t, TRUE );

    for( idx=1; fprbuf[idx]; idx++ ) {
	s = card_get_fpr( card, idx );
	print_fpr( dlg, fprbuf[idx], s );
    }

    for( idx=1; attr_tab[idx].attr; idx++ ) {
	s = card_get_string( card, attr_tab[idx].attr );
	SetDlgItemText( dlg, attr_tab[idx].ctlid, s && *s? s : attr_tab[idx].err );
    }

    s = card_get_string( card, GPGME_ATTR_CARD_LANG );
    idx = idx_from_lang( s );    
    SendDlgItemMessage( dlg, IDC_CEDIT_LANG, CB_SETCURSEL, (WPARAM)idx, 0 );

    t = card_get_ulong( card, GPGME_ATTR_CARD_SEX );
    switch( t ) {
    case 'm': idx=0; break;
    case 'f': idx=1; break;
    default :
    case 'u': idx=2; break;
    }
    SendDlgItemMessage( dlg, IDC_CEDIT_SEX, CB_SETCURSEL, (WPARAM)idx, 0 );

    return 0;
} /* card_status */


static void
prepare_dialog( HWND dlg )
{
    const char * s;
    int i;

    for( i=0; (s = sex[i]); i++ )
	SendDlgItemMessage( dlg, IDC_CEDIT_SEX, CB_ADDSTRING, 0, (LPARAM) s );
    SendDlgItemMessage( dlg, IDC_CEDIT_SEX, CB_SETCURSEL, 0, 0 );
    for( i=0; (s = lang[i]); i++ )
	SendDlgItemMessage( dlg, IDC_CEDIT_LANG, CB_ADDSTRING, 0, (LPARAM)s );
    SendDlgItemMessage( dlg, IDC_CEDIT_LANG, CB_SETCURSEL, 0, 0 );
} /* prepare_dialog */


static int
check_string( const char * str, int flags )
{
    size_t i;
    for( i=0; i < strlen( str ); i++ ) {
	if( flags & 0x02 && !isalpha( str[i] ) )
	    return -1;
    }
    return 0;
}


static int
do_proc_card_cmds( HWND dlg, struct pin_cb_ctx_s * pincb, gpgme_card_t card )
{
    static struct {
	int id;
	int cmd;
	int us_ascii;
	int changed;
    } idctl[] = {
	{IDC_CEDIT_NAME,  GPGME_EDITCARD_NAME,  1,  0},
	{IDC_CEDIT_LANG2, GPGME_EDITCARD_LANG,  1,  0},
	{IDC_CEDIT_SEX2,  GPGME_EDITCARD_SEX,   1|1,0},
	{IDC_CEDIT_KEYURL,GPGME_EDITCARD_KEYURL,1,  0},
	{IDC_CEDIT_LOGIN, GPGME_EDITCARD_LOGIN, 1,  0},
	{0}
    };
    gpgme_editcard_t ec;
    gpgme_ctx_t ctx;
    gpgme_error_t rc;
    char buf[256], tmp[128];
    int errc=0;
    int i, id, n=0;

    /* XXX rewrite the entire function */
    for( i=0; idctl[i].id; i++ ) /* reset */
	idctl[i].changed = 0;
    
    if( SendMessage( GetDlgItem( dlg, IDC_CEDIT_LANG2 ), WM_GETTEXTLENGTH, 0, 0 ) ) {
	idctl[1].changed = 1; 
	n++;
    }
    if( SendMessage( GetDlgItem( dlg, IDC_CEDIT_SEX2 ), WM_GETTEXTLENGTH, 0, 0 ) ) {
	idctl[2].changed = 1; 
	n++;
    }
    
    if( SendDlgItemMessage( dlg, IDC_CEDIT_NAME2, EM_GETMODIFY, 0, 0 ) ) {
	idctl[0].changed = 1;
	n++;
    }
    for( i=0; (id = idctl[i].id); i++ ) {
	if( SendDlgItemMessage( dlg, id, EM_GETMODIFY, 0, 0 ) ) {
	    idctl[i].changed = 1;
	    n++;
	}
    }
    if( !pincb || !card ) /* just return the changed elements */
	return n;
    if( !n )
	return 0;
    
    rc = gpgme_editcard_new( &ec );
    if( !rc )
	rc = gpgme_new( &ctx );
    if( rc )
	BUG( NULL );
    gpgme_editcard_control( ec, GPGME_EDITCARD_APIN, pincb->apin );
    gpgme_editcard_control( ec, GPGME_EDITCARD_UPIN, pincb->upin );
    for( i=0; idctl[i].id; i++ ) {
	if( idctl[i].changed ) {
	    GetDlgItemText( dlg, idctl[i].id, buf, sizeof (buf)-1 );
	    if (idctl[i].us_ascii && is_8bit_string (buf)) {
		msg_box (dlg, _("Only plain ASCII is currently allowed."),
		         _("Card Edit"), MB_ERR);
		errc--; continue;
	    }
	    if( (idctl[i].us_ascii & 2) && check_string( buf, 2 ) ) {
		msg_box( dlg, _("Only alphabetic characters are allowed."),
			 _("Card Edit"), MB_ERR );
		errc--; continue;
	    }
	    if( idctl[i].cmd == GPGME_EDITCARD_NAME ) {
		/* The "name" command actually needs two fields */
		GetDlgItemText( dlg, IDC_CEDIT_NAME2, tmp, sizeof tmp-1 );
		gpgme_editcard_control( ec, GPGME_EDITCARD_NAME2, tmp );
	    }
	    gpgme_editcard_control( ec, idctl[i].cmd, buf );
	    gpgme_set_edit_ctx( ctx, ec, idctl[i].cmd );
	    rc = gpgme_op_editcard( ctx );
	    if( rc ) {
		msg_box( dlg, _("Could not modify card attribute."), 
		         _("Card Edit"), MB_ERR );
		errc--;
	    }
	}
    }
    if( !errc ) {
	/* if the operation(s) succeeded, reset the modify flag for each control */
	for( i = 0; idctl[i].id; i++ )
	    SendDlgItemMessage( dlg, idctl[i].id, EM_SETMODIFY, (WPARAM)(UINT)FALSE, 0 );
	msg_box( dlg, _("Card attribute changed."), _("Card Edit"), MB_OK );
	SetDlgItemText( dlg, IDC_CEDIT_LANG2, "" );
	SetDlgItemText( dlg, IDC_CEDIT_SEX2, "" );
    }
    gpgme_editcard_release( ec );
    gpgme_release( ctx );
    return errc;
} /* do_proc_card_cmds */


void
free_pincb (struct pin_cb_ctx_s * ctx)
{
    if (!ctx)
	return;
    free_if_alloc (ctx->info_text);
    free_if_alloc (ctx->upin);
    free_if_alloc (ctx->apin);
} /* free_pincb */


static int
do_askpin( HWND dlg, gpgme_edit_card_t which, gpgme_card_t card,
	   struct pin_cb_ctx_s * pincb )
{
    const char * s, * fmt;
    const char * n1, * n2, * serial;
    char * p;
    size_t n;

    if( (which == GPGME_EDITCARD_CHAPIN && pincb->apin) ||
        (which == GPGME_EDITCARD_CHUPIN && pincb->upin) )
	return 0;
    
    if( which == GPGME_EDITCARD_CHAPIN )
	s = _("Please enter the 'Admin PIN'");
    else if( which == GPGME_EDITCARD_CHUPIN )
	s = _("Please enter the 'User PIN'");
    else
	s = _("Please enter the PIN");
    pincb->which = which;
    free_if_alloc( pincb->info_text );
    if( card ) {
	fmt = _("%s\nName: %s %s\nSerial-No: %s\n");
	n1 = card_get_string( card, GPGME_ATTR_CARD_NAME );
	n2 = card_get_string( card, GPGME_ATTR_CARD_NAME2 );
	if( !n1 || !n2 ) {
	    n1 = "No"; n2 = "Name";
	}    
	serial = card_get_string( card, GPGME_ATTR_CARD_SERIAL );
	if( !serial )
	    serial = "No Serial";
	n = strlen( n1 ) + strlen( n2 ) + strlen( fmt ) + strlen( serial ) + 3;
	p = pincb->info_text = new char[strlen( s )+n+1 ];
	if( !p )
	    BUG (0);
	sprintf( p, fmt, s, n1, n2, serial );
    }
    else {
	p = pincb->info_text = m_strdup (s);
	if (!p)
	    BUG (0);
    }
    DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_PIN, dlg,
		    pin_cb_dlg_proc, (LPARAM)pincb);
    if( !pincb->apin && !pincb->upin) {
	safe_free (pincb->info_text);
	return -1;
    }
    return 0;
} /* do_askpin */


BOOL CALLBACK
card_edit_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static gpgme_card_t card;    
    char tmp[128];
    size_t n=0;

    switch (msg) {
    case WM_INITDIALOG:
	card = (gpgme_card_t)lparam;
	if (!card)
	    BUG (0);
	prepare_dialog (dlg);
	if (card_status (dlg, card ))
	    EndDialog (dlg, TRUE);
	center_window (dlg);
	SetForegroundWindow (dlg);
	return TRUE;

    case WM_DESTROY:
	free_if_alloc (pincb.info_text);
	free_if_alloc (pincb.apin);
	free_if_alloc (pincb.upin);
	memset (&pincb, 0, sizeof pincb);
	break;

    case WM_COMMAND:
	switch( HIWORD( wparam ) ) {
	case CBN_KILLFOCUS:
	case CBN_EDITCHANGE:
	case CBN_EDITUPDATE:
	    int ctlid = GetDlgCtrlID( (HWND)lparam );
	    int dstid = 0;

	    switch (ctlid) {
	    case IDC_CEDIT_LANG: dstid = IDC_CEDIT_LANG2; break;
	    case IDC_CEDIT_SEX:  dstid = IDC_CEDIT_SEX2; break;
	    }	    
	    GetDlgItemText (dlg, ctlid, tmp, 127);
	    SetDlgItemText (dlg, dstid, tmp);
	    break;
	}
	switch( LOWORD( wparam ) ) {
	case IDC_CEDIT_CHPIN:
	    DialogBoxParam( glob_hinst, (LPCTSTR)IDD_WINPT_CARD_CHPIN, dlg,
			    card_changepin_dlg_proc, NULL );
	    break;

	case IDC_CEDIT_NEWKEYS:
	    if (item_get_text_length (dlg, IDC_CEDIT_FPR1) > 0) {
		int id = msg_box (dlg, 
		    _("This operation will override the keys on the card.\n"
		      "Still proceed?"), _("Card Edit"), MB_WARN|MB_YESNO);
		if (id == IDNO)
		    return FALSE;
	    }
	    DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_CARD_KEYGEN,
			    glob_hwnd, card_keygen_dlg_proc, NULL);
	    break;

	case IDOK:
	    n = do_proc_card_cmds( dlg, NULL, NULL );
	    if( n ) {
		if( do_askpin( dlg, GPGME_EDITCARD_CHAPIN, card, &pincb ) )
		    EndDialog( dlg, FALSE );
		if( do_askpin( dlg, GPGME_EDITCARD_CHUPIN, card, &pincb ) )
		    EndDialog( dlg, FALSE );
	    }
	    do_proc_card_cmds( dlg, &pincb, card );
	    free_pincb( &pincb );
	    if( !n )
		EndDialog( dlg, TRUE );
	    break;

	case IDCANCEL:
	    EndDialog( dlg, FALSE );
	    break;
	}
	break;
    }

    return FALSE;
} /* card_edit_dlg_proc */



static int /* fixme: works only roughly */
calc_days (int y2, int m2, int d2, 
	   int y1, int m1, int d1)

{
    int n=0;
    
    if ((y2-y1) > 0)
	n += (y2-y1)*365;
    if ((m2-m1) > 0)
	n += (m2-m1)*30;
    if ((d2-d1) > 0)
	n += (d2-d1);
    else if ((d2-d1) < 0)
	n -= (d1-d2);
    return n;
}


static void
keygen_fill_algbox (HWND dlg)
{
    SendDlgItemMessage (dlg, IDC_CKEYGEN_ALG, CB_ADDSTRING, 0, (LPARAM)(const char*)"RSA");
    SendDlgItemMessage (dlg, IDC_CKEYGEN_ALG, CB_SETCURSEL, 0, 0);
}


BOOL CALLBACK
card_keygen_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static int state = 0;
    static int pwd_state = 0;
    gpgme_ctx_t ctx;
    gpgme_editcard_t crd;
    gpgme_error_t err;
    char name[128], email[128], comment[128], expdate[64];
    char pass[128];
    int card_flags = GPGME_CARDFLAG_NONE;
    int expires=0;
    size_t n;

    switch (msg) {
    case WM_INITDIALOG:
	state = 0;
	pwd_state = 1;
	center_window (dlg);
	CheckDlgButton (dlg, IDC_CKEYGEN_REPLACE, BST_CHECKED);
	CheckDlgButton (dlg, IDC_CKEYGEN_NEVER, BST_CHECKED);
	CheckDlgButton (dlg, IDC_CKEYGEN_BACKUP, BST_CHECKED);
	EnableWindow (GetDlgItem (dlg, IDC_CKEYGEN_VALID), FALSE);
	keygen_fill_algbox (dlg);
	SetFocus (GetDlgItem (dlg, IDC_CKEYGEN_NAME));
	SetForegroundWindow (dlg);
	return FALSE;

    case WM_COMMAND:
	if (HIWORD (wparam) == BN_CLICKED) {
	    switch (LOWORD (wparam)) {
	    case IDC_CKEYGEN_NEVER:	    
		state ^= 1;
		EnableWindow (GetDlgItem (dlg, IDC_CKEYGEN_VALID), state);
		break;

	    case IDC_CKEYGEN_BACKUP:
		pwd_state ^= 1;
		EnableWindow (GetDlgItem (dlg, IDC_CKEYGEN_PASS), pwd_state);
		break;
	    }
	}

	switch (LOWORD (wparam)) {
	case IDOK:
	    n = item_get_text_length (dlg, IDC_CKEYGEN_NAME);
	    if (!n) {
		msg_box (dlg, _("Please enter your name."), _("Card Edit"), MB_ERR);
		return FALSE;
	    }
	    if (n < 5) {
		msg_box (dlg, _("Name must be at least 5 characters long."),
			 _("Card Edit"), MB_INFO);
		return FALSE;
	    }
	    n = item_get_text_length (dlg, IDC_CKEYGEN_EMAIL);
	    if (!n) {
		msg_box (dlg, _("Please enter your e-mail address."),
			 _("Card Edit"), MB_ERR);
		return FALSE;
	    }
	    GetDlgItemText (dlg, IDC_CKEYGEN_NAME, name, sizeof (name)-1);
	    GetDlgItemText (dlg, IDC_CKEYGEN_EMAIL, email, sizeof (email)-1);
	    if (!strchr (email, '@') || n < 3) {
		msg_box (dlg, _("Please enter a valid e-mail address."),
		         _("Card Edit"), MB_ERR);
		return FALSE;
	    }
	    n = GetDlgItemText (dlg, IDC_CKEYGEN_PASS, pass, sizeof (pass)-1);
	    if (!n) {
		msg_box (dlg, _("Please enter an off-card passphrase."), _("Card Edit"), MB_ERR);
		return FALSE;
	    }
	    n = item_get_text_length (dlg, IDC_CKEYGEN_COMMENT);
	    if (n > 0)
		GetDlgItemText (dlg, IDC_CKEYGEN_COMMENT, comment, sizeof (comment)-1);
	    if (is_8bit_string (name) ||is_8bit_string (comment)) {
		msg_box (dlg, _("Please use plain ASCII charset for the fields."), 
			 _("Card Edit"), MB_INFO);
		return FALSE;
	    }
	    memset (&pincb, 0, sizeof (pincb));
	    if (do_askpin (dlg, GPGME_EDITCARD_CHAPIN, NULL, &pincb)) {
		free_pincb (&pincb);
		return FALSE;
	    }
	    if (do_askpin (dlg, GPGME_EDITCARD_CHUPIN, NULL, &pincb)) {
		free_pincb (&pincb);
		return FALSE;
	    }
	    err = gpgme_new (&ctx);
	    if (!err)
		err = gpgme_editcard_new (&crd);
	    if (err)
		BUG (0);
	    expires = !IsDlgButtonChecked (dlg, IDC_CKEYGEN_NEVER);
	    if (expires) {
		SYSTEMTIME st, ct;
		DateTime_GetSystemtime (GetDlgItem (dlg, IDC_CKEYGEN_VALID), &st);
		if (!keygen_check_date (&st)) {
		    msg_box (dlg, _("The date you have chosen lies in the past."),
			     _("Card Edit"), MB_ERR);
		    gpgme_release (ctx);
		    gpgme_editcard_release (crd);
		    return FALSE;
		}
		GetSystemTime (&ct);
		/* XXX this is not very precise */
		sprintf (expdate, "%d", calc_days (st.wYear, st.wMonth, st.wDay,
						   ct.wYear, ct.wMonth, ct.wDay));
	    }
	    if (IsDlgButtonChecked (dlg, IDC_CKEYGEN_REPLACE))
		card_flags |= GPGME_CARDFLAG_REPLACE;
	    if (IsDlgButtonChecked (dlg, IDC_CKEYGEN_BACKUP))
		card_flags |= GPGME_CARDFLAG_BAKENC;
	    gpgme_editcard_set_keygen_params (crd, card_flags, name, email,
					      n? comment : NULL, 
					      expires? expdate : NULL);
	    gpgme_editcard_set_passwd (crd, pass);
	    gpgme_editcard_control (crd, GPGME_EDITCARD_APIN, pincb.apin);
	    gpgme_editcard_control (crd, GPGME_EDITCARD_UPIN, pincb.upin);
	    gpgme_set_edit_ctx (ctx, crd, GPGME_EDITCARD_GENKEY);
	    SetCursor( LoadCursor (NULL, IDC_WAIT));
	    err = gpgme_op_editcard (ctx);
	    SetCursor (LoadCursor (NULL, IDC_ARROW));
	    if (err == GPGME_Canceled)
		msg_box (dlg, _("Operation was canceled. It seems that there are "
		                "existing\nkeys on the cards. You need to mark the "
				"'Overwrite' flag."), _("Card Edit"), MB_INFO);
	    else if (err)
		msg_box (dlg, "The operation does not succeed.\n"
			      "Please make sure you entered the right PIN's."
			      , _("Card Edit"), MB_ERR);
	    else
		msg_box (dlg, _("Keys successfully created."), 
		         _("Card Edit"), MB_OK);
	    memset (pass, 0, sizeof (pass));
	    free_pincb (&pincb);
	    gpgme_release (ctx);
	    gpgme_editcard_release (crd);
	    break;

	case IDCANCEL:
	    EndDialog (dlg, FALSE);
	    return FALSE;
	}
	break;
    }
    return FALSE;
} /* card_keygen_dlg_proc */


static int
check_pin_len (int which, int flag, int pinlen)
{
    if (!pinlen) {
	if (flag)
	    msg_box (NULL, _("Please enter the old card PIN."), _("Card Edit"), MB_ERR);
	else
	    msg_box (NULL, _("Please enter the new card PIN."), _("Card Edit"), MB_ERR);
	return -1;
    }
    if (which == GPGME_EDITCARD_CHAPIN
	&& pinlen < 8) {
	msg_box (NULL, _("PIN must be minimal 8 characters."), _("Card Edit"), MB_ERR);
	return -1;
    }
    if (which == GPGME_EDITCARD_CHUPIN
	&& pinlen < 6) {
	msg_box (NULL, _("PIN must be minimal 6 characters."), _("Card Edit"), MB_ERR);
	return -1;
    }
    return 0;
}


BOOL CALLBACK
card_changepin_dlg_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{    
    gpgme_ctx_t ctx;
    gpgme_editcard_t chpin;
    gpgme_error_t rc;
    gpgme_edit_card_t which;
    char pold[128], pnew[128];
    size_t n;

    switch( msg ) {
    case WM_INITDIALOG:
	center_window( dlg );
	CheckDlgButton( dlg, IDC_CHPIN_ISWORK, BST_CHECKED );
	SetForegroundWindow( dlg );
	break;

    case WM_COMMAND:
	switch( LOWORD( wparam ) ) {
	case IDOK:
	    if( IsDlgButtonChecked( dlg, IDC_CHPIN_ISADMIN ) )
		which = GPGME_EDITCARD_CHAPIN;
	    else if( IsDlgButtonChecked( dlg, IDC_CHPIN_ISWORK ) )
		which = GPGME_EDITCARD_CHUPIN;
	    else
		BUG (0);

	    n = item_get_text_length( dlg, IDC_CHPIN_OLDPIN );
	    if (check_pin_len (which, 1, n))
		return FALSE;
	    n = item_get_text_length( dlg, IDC_CHPIN_NEWPIN );
	    if (check_pin_len (which, 0, n))
		return FALSE;
	    GetDlgItemText( dlg, IDC_CHPIN_OLDPIN, pold, sizeof (pold)-1 );
	    GetDlgItemText( dlg, IDC_CHPIN_NEWPIN, pnew, sizeof (pnew)-1 );
	    rc = gpgme_new( &ctx );
	    if( !rc )
		rc = gpgme_editcard_new( &chpin );
	    if (rc)
		BUG (0);
	    gpgme_changepin_set( chpin, which, pold, pnew );
	    gpgme_set_edit_ctx( ctx, chpin, which );
	    rc = gpgme_op_changepin( ctx );
	    if( rc )
		msg_box( dlg, gpgme_strerror( rc ), _("Card Edit"), MB_ERR );
	    else {
		msg_box( dlg, _("PIN successfully changed."), _("Card Edit"), MB_OK );
		SetDlgItemText( dlg, IDC_CHPIN_NEWPIN, "" );
		SetDlgItemText( dlg, IDC_CHPIN_OLDPIN, "" );
	    }
	    gpgme_release( ctx );
	    gpgme_editcard_release( chpin );
	    break;

	case IDCANCEL:
	    SetDlgItemText( dlg, IDC_CHPIN_NEWPIN, "" );
	    SetDlgItemText( dlg, IDC_CHPIN_OLDPIN, "" );
	    EndDialog( dlg, FALSE );
	    break;
	}
	break;
    }

    return FALSE;
} /* card_changepin_dlg_proc */