/* PTD.cpp - Privacy Tray Dynamic
 *	Copyright (C) 2002-2005 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <stdio.h>

HINSTANCE glob_hinst;

/* We need a special section in the DLL for storing our shared data */
#pragma data_seg(".SHARDAT")
static HHOOK cbt_hook = NULL;
static DWORD tray_proc_id = 0;
static HWND shell_traywnd = 0;
static HWND curr_focus = NULL;
static HWND rebar_w32 = NULL;
static HWND mstask_swc = NULL;
static HWND systab_c32 = NULL;
#pragma data_seg()

static unsigned * journ_keys = 0;
static unsigned key_idx = 0;
static unsigned nkeys = 0;
static HHOOK journ_hook = NULL;


static LRESULT CALLBACK
PTD_CBT_proc (int code, WPARAM wparam, LPARAM lparam)
{
    HWND curr_hwnd = NULL;	
    DWORD proc_id = 0;
    
    if (code >= 0 && code == HCBT_SETFOCUS) {
        /* A window is about to receive the keyboard focus */		
        
        curr_hwnd = (HWND)wparam;
        GetWindowThreadProcessId( curr_hwnd, &proc_id );
        /* Only if the current window is different from this window list,
           we set the new focus. For more information read the hint in 
	   PTD_initialize. */
        if ((proc_id != tray_proc_id)
             && (curr_hwnd != NULL)
             && (curr_hwnd != curr_focus)
             && (curr_hwnd != shell_traywnd)
             && (curr_hwnd != systab_c32)
             && (curr_hwnd != mstask_swc)
             && (curr_hwnd != rebar_w32)) {
            curr_focus = curr_hwnd;
        }
    }
    
    return  CallNextHookEx (cbt_hook , code, wparam, lparam);
} /* PTD_CBT_proc */


HWND
PTD_get_curr_hwnd (void)
{
    return curr_focus;
} /* PTD_get_curr_hwnd */


BOOL
PTD_initialize (void)
{
    tray_proc_id = GetCurrentProcessId( );
    cbt_hook = SetWindowsHookEx( WH_CBT, PTD_CBT_proc, glob_hinst, 0 );
    if( cbt_hook == NULL )		
        return FALSE;
    
    /* Okay, what are we doing here:
     * In the past I used the Spy++ application to find out why the old current window
     * doesn't work on so much systems. Now we try to use the computer based training
     * hook, more precise the HCBT_SETFOCUS to find out the last window focus. But to
     * succeed we need to ignore some windows because otherwise our own window or a
     * window from the taskbar would be recognized.
     *
     * Here is what the Spy++ says about the windows order:
     * Shell_TrayWnd
     *  \ ReBarWindow32
     *                \ MSTaskSwWClass
     *                               \ SysTabControl32
     * 
     * As a result we need to ignore those windows to prevent failured in the code.
     */
    shell_traywnd = FindWindowEx (NULL, NULL, "Shell_TrayWnd", NULL);
    rebar_w32 = FindWindowEx (shell_traywnd, NULL, "ReBarWindow32", NULL);
    mstask_swc = FindWindowEx (rebar_w32, NULL, "MSTaskSwWClass", NULL);
    systab_c32 = FindWindowEx (mstask_swc, NULL, "SysTabControl32", NULL);
    
    /*_log_box ("tray_proc_id=%u cbt_hook=%p shell_traywnd=%p rebar_w32=%p "
		"mstask_swc=%p systab_c32=%p",
		tray_proc_id, cbt_hook, shell_traywnd, rebar_w32, 
		mstask_swc, systab_c32);*/
    
    return TRUE;
} /* PTD_initialize */


void
PTD_delete (void)
{
    if (!cbt_hook)
	return;
    UnhookWindowsHookEx (cbt_hook);
    cbt_hook = NULL;
    shell_traywnd = NULL;
    tray_proc_id = 0;
} /* PTD_delete */


int
PTD_is_used (void)
{
    return shell_traywnd && cbt_hook;
} /* PTD_is_used */


LRESULT CALLBACK 
PTD_playback_proc (int code, WPARAM wparam, LPARAM lparam)
{
    LPEVENTMSG em;
    
    if (code == HC_SKIP) {
        key_idx++;
        if (key_idx == nkeys) {
            UnhookWindowsHookEx (journ_hook);
            journ_hook = NULL;
        }
        return 0;
    }
    else if (code == HC_GETNEXT) {
        /* Here is an overview what the event message struct can contain:
         * message - WM_KEYUP, WM_KEYDOWN, WM_SYSKEYUP, WM_SYSKEYDOWN:
         * paramL - specifies the virtual key code of the key that was pressed.
         * paramH - specifies the scan code.				
         */
        em = (LPEVENTMSG )lparam;
        if (journ_keys[key_idx] & 0x8000)
            em->message = journ_keys[key_idx] & 0x4000 ? WM_SYSKEYDOWN : WM_KEYDOWN;
        else 
            em->message = journ_keys[key_idx] & 0x4000 ? WM_SYSKEYUP : WM_KEYUP;
        em->paramL = LOBYTE( journ_keys[key_idx] )
            | (MapVirtualKey (LOBYTE (journ_keys[key_idx]), 0) << 8);
        em->paramH = 1;
        em->time = GetTickCount ();
        return 0;
    }
    
    return CallNextHookEx (journ_hook, code, wparam, lparam);
} /* PTD_Playback_proc */


BOOL
PTD_keyb_send (UINT * keys, UINT n_keys)
{
    MSG msg;

    journ_keys = keys;
    key_idx = 0;
    nkeys =  n_keys;    
    
    if (journ_hook)
        return FALSE;
    
    while (GetAsyncKeyState (VK_MENU) & 0x8000
            || GetAsyncKeyState (VK_SHIFT) & 0x8000
            || GetAsyncKeyState (VK_CONTROL) & 0x8000) {
        if (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE)) {
            if (msg.message == WM_QUIT) {
                PostMessage (msg.hwnd, msg.message, msg.wParam, msg.lParam);
                return FALSE;
            }
            TranslateMessage (&msg);
            DispatchMessage (&msg);
        }
    }
    journ_hook = SetWindowsHookEx (WH_JOURNALPLAYBACK,
                                   (HOOKPROC)PTD_playback_proc,
                                   glob_hinst, 0);
    if (journ_hook == NULL)
        return FALSE;
    
    while (journ_hook) {
        if (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE)) {
            if (msg.message == WM_QUIT) {
                if (journ_hook)
                    UnhookWindowsHookEx (journ_hook);
                PostMessage (msg.hwnd, msg.message, msg.wParam, msg.wParam);
            }
            TranslateMessage (&msg);
            DispatchMessage (&msg);
        }
    }
    
    return TRUE;
} /* PTD_keyb_send */


const char *
PTD_get_version (void)
{
    return "0.8.0";
} /* PTD_get_version */

		
BOOL WINAPI 
DllMain (HINSTANCE hinst, DWORD reason, LPVOID reserv)
{
    switch (reason)  {
    case DLL_PROCESS_ATTACH: 
	glob_hinst = hinst; 
	break;
    case DLL_THREAD_ATTACH:  
	break;
    case DLL_THREAD_DETACH:  
	break;
    case DLL_PROCESS_DETACH: 
	break;
    }
    return TRUE;
} /* DllMain */
