/* sign.c -  signing functions
 *	Copyright (C) 2000 Werner Koch (dd9jn)
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "util.h"
#include "context.h"
#include "ops.h"
#include "status-table.h"

struct  sign_result_s {
    void * last_pw_handle;
    char * userid_hint;
    char * passphrase_info;
    char cardno[32+1];
    int bad_passphrase;
    int idea_cipher;
    int no_passphrase;
    int okay;	
};


void
_gpgme_release_sign_result( _sign_result_t res )
{
    if( res ) {
	safe_free( res->userid_hint );
	safe_free( res->passphrase_info );
	safe_free( res );
    }
}


static void
sign_status_handler( gpgme_ctx_t ctx, gpg_status_code_t code, char *args )
{
    char * p;
    int i=0;

    if( ctx->out_of_core )
        return;
    if( ctx->result_type == RESULT_TYPE_NONE ) {
        assert( !ctx->result.sign );
        ctx->result.sign = calloc(  1, sizeof *ctx->result.sign );
        if( !ctx->result.sign ) {
            ctx->out_of_core = 1;
            return;
        }
        ctx->result_type = RESULT_TYPE_SIGN;
    }
    assert( ctx->result_type == RESULT_TYPE_SIGN );
    
    switch( code ) {
    case STATUS_EOF:
        break;
        
    case STATUS_USERID_HINT:
        safe_free( ctx->result.sign->userid_hint );
	ctx->result.sign->userid_hint = strdup( args );
        if( !ctx->result.sign ) {
            ctx->out_of_core = 1;
	    return;
	}
        break;
        
    case STATUS_BAD_PASSPHRASE:
        DEBUG0( "Bad passphrase - once again please\n" );
        ctx->result.sign->bad_passphrase++;		 
        break;
        
    case STATUS_GOOD_PASSPHRASE:
        ctx->result.sign->bad_passphrase = 0;
        break;

    case STATUS_RSA_OR_IDEA:
	ctx->result.sign->idea_cipher = 1;
	break;
        
    case STATUS_NEED_PASSPHRASE:
    case STATUS_NEED_PASSPHRASE_SYM:
        safe_free( ctx->result.sign->passphrase_info );
	ctx->result.sign->passphrase_info = strdup( args );
        if( !ctx->result.sign->passphrase_info ) {
            ctx->out_of_core = 1;
	    return;
	}
        break;
        
    case STATUS_MISSING_PASSPHRASE:
        DEBUG0( "missing passphrase - stop\n" );
        ctx->result.sign->no_passphrase = 1;
        break;
        
    case STATUS_SIG_CREATED:
        ctx->result.sign->okay = 1;
        break;

    case STATUS_PROGRESS:
	if( ctx->cb.progress )
	    _gpgme_progress_handler( ctx, args );
	break;

    case STATUS_CARDCTRL:
	if( args[i++] != '3' )
	    break;
	i++;
	p = ctx->result.sign->cardno;
	for( ; i-1 < DIM(ctx->result.sign->cardno) && args[i]; i++ )
	    *p++ = args[i];
	*p = 0;
	break;
        
    default:
        break;
    }
}


static const char *
command_handler( void *opaque, gpg_status_code_t code, const char *key )
{
    gpgme_ctx_t c = opaque;
    
    if( c->result_type == RESULT_TYPE_NONE ) {
        assert( !c->result.sign );
        c->result.sign = calloc( 1, sizeof *c->result.sign );
        if( !c->result.sign ) {
            c->out_of_core = 1;
            return NULL;
        }		
        c->result_type = RESULT_TYPE_SIGN;
    }
    
    if( !code ) {
        /* We have been called for cleanup */
        if( c->cb.passphrase ) {
            c->cb.passphrase( c->cb.passphrase_value, 0,
                              &c->result.sign->last_pw_handle );
        }
        return NULL;
    }
    
    if( !key || !c->cb.passphrase )
        return NULL;
    
    if( code == STATUS_GET_HIDDEN 
	&& (!strcmp( key, "passphrase.enter" )
	||  !strcmp( key, "passphrase.pin.ask" )) ) {
        const char * userid_hint = c->result.sign->userid_hint;
        const char * passphrase_info = c->result.sign->passphrase_info;
	const char * cardno = c->result.sign->cardno;
        int bad_passphrase = c->result.sign->bad_passphrase;
	int is_card=0;
        char * buf;
        const char * s;
        
        c->result.sign->bad_passphrase = 0;
	is_card = !strcmp( key, "passphrase.pin.ask" );
        if( !userid_hint )
            userid_hint = "[User ID hint missing]";
        if( !passphrase_info )
            passphrase_info = "[passphrase info missing]";
        buf = malloc( 20 + strlen( userid_hint )
		         + strlen( passphrase_info ) + 3 );
        if( !buf ) {
            c->out_of_core = 1;
            return NULL;
        }
        sprintf( buf, "%s\n%s\n%s",
                 bad_passphrase? "TRY_AGAIN":"ENTER_PASSPHRASE",
                 userid_hint, passphrase_info );
        
        s = c->cb.passphrase( c->cb.passphrase_value,
			      is_card? cardno : buf,
			      &c->result.sign->last_pw_handle );
        safe_free( buf );
        return s;
    }    
    
    return NULL;
}


static gpgme_error_t
file_sign_start (gpgme_ctx_t ctx, int signenc, int sigmode,
		 gpgme_recipients_t recp,
		 const char * input, const char * output)
{
    gpgme_error_t rc = 0;
    gpgme_key_t key;
    
    fail_on_pending_request (ctx);
    ctx->pending = 1;
    
    _gpgme_gpg_release (&ctx->gpg);
    rc = _gpgme_gpg_new (&ctx->gpg);
    if( rc )
        return rc;
        
    if( ctx->use_logging )
	_gpgme_gpg_set_logging_handler( ctx->gpg, ctx );
    _gpgme_gpg_set_status_handler( ctx->gpg, sign_status_handler, ctx );
    if( !ctx->use_pass_fd )
     _gpgme_gpg_set_command_handler( ctx->gpg, command_handler,  ctx );
    else {
        rc = _gpgme_add_passphrase( ctx );
        if( rc ) {
            _gpgme_gpg_release( &ctx->gpg );
            return rc;
        }
    }
    
    _gpgme_gpg_add_arg (ctx->gpg, "--yes");
    key = gpgme_signers_enum (ctx, 0);
    if (key) 
    {
	const char * s;
	s = gpgme_key_get_string_attr (key, GPGME_ATTR_KEYID, NULL, 0);
	if (!s)	
	    return mk_error (Invalid_Value);
        _gpgme_gpg_add_arg (ctx->gpg, "-u");
	_gpgme_gpg_add_arg_concat (ctx->gpg, s, "!");
	if (signenc) 
	{
	    _gpgme_gpg_add_arg (ctx->gpg, "--encrypt-to");
	    _gpgme_gpg_add_arg (ctx->gpg, s);
	}
    }
    if( ctx->use_armor )
        _gpgme_gpg_add_arg( ctx->gpg, "--armor" );
    if( ctx->cb.progress )
	_gpgme_gpg_add_arg( ctx->gpg, "--enable-progress-filter" );
    if( output && *output ) {
        _gpgme_gpg_add_arg( ctx->gpg, "--output" );
        _gpgme_gpg_add_arg( ctx->gpg, output );
    }
    else
	_gpgme_gpg_add_arg( ctx->gpg, "--no-mangle-dos-filenames" );
    if( signenc ) {
	if (sigmode != GPGME_SIG_MODE_NORMAL)
	    return mk_error (Invalid_Value);
        if (!recp)
            return mk_error (No_Recipients);
	if (ctx->force_trust || _gpgme_recipients_all_valid (recp))
	    _gpgme_gpg_add_arg (ctx->gpg, "--always-trust");
        _gpgme_gpg_add_arg (ctx->gpg, "--encrypt");
        _gpgme_append_gpg_args_from_recipients (recp, ctx->gpg);
    }
    if (ctx->cb.progress)
	_gpgme_gpg_add_arg (ctx->gpg, "--enable-progress-filter");
    
    switch( sigmode ) {
    case GPGME_SIG_MODE_NORMAL: 
	_gpgme_gpg_add_arg( ctx->gpg, "--sign" );
	break;
    case GPGME_SIG_MODE_DETACH: 
	_gpgme_gpg_add_arg( ctx->gpg, "--detach-sign" );
	break;        
    case GPGME_SIG_MODE_CLEAR:
        _gpgme_gpg_add_arg( ctx->gpg, "--clearsign" ); 
        if( ctx->use_textmode )
            _gpgme_gpg_add_arg( ctx->gpg, "--textmode" );
        break;
    default:
        _gpgme_gpg_add_arg( ctx->gpg, "--sign" ); 
        break;
    }	
    
    _gpgme_gpg_add_arg( ctx->gpg, input );
    rc = _gpgme_gpg_spawn( ctx->gpg,  ctx );
    if( rc ) {
        ctx->pending = 0;
        _gpgme_gpg_release( &ctx->gpg );
    }
    
    return rc;	
} /* file_sign_start */


static gpgme_error_t
sign_start( gpgme_ctx_t ctx, gpgme_data_t in, gpgme_data_t out, gpgme_sigmode_t mode )
{
    int rc = 0;
    int i;
    gpgme_key_t key;
    
    fail_on_pending_request( ctx );
    ctx->pending = 1;
    
    _gpgme_release_result( ctx );
    ctx->out_of_core = 0;
    
    if( mode != GPGME_SIG_MODE_NORMAL
        && mode != GPGME_SIG_MODE_DETACH
        && mode != GPGME_SIG_MODE_CLEAR )
        return mk_error( Invalid_Value );
    
    /* create a process object */
    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
        goto leave;
    
    _gpgme_gpg_set_status_handler( ctx->gpg, sign_status_handler, ctx );
    if( ctx->cb.passphrase ) {
        rc = _gpgme_gpg_set_command_handler( ctx->gpg, command_handler, ctx );
        if( rc )
            goto leave;
    } 
    else if( ctx->passphrase_value ) {
        rc = _gpgme_add_passphrase( ctx );
        if( rc )
            goto leave;
    }
    
    /* build the commandline */
    if( mode == GPGME_SIG_MODE_CLEAR )
        _gpgme_gpg_add_arg ( ctx->gpg, "--clearsign" );
    else {
        _gpgme_gpg_add_arg ( ctx->gpg, "--sign" );
        if( mode == GPGME_SIG_MODE_DETACH )
            _gpgme_gpg_add_arg ( ctx->gpg, "--detach" );
        if( ctx->use_armor )
            _gpgme_gpg_add_arg ( ctx->gpg, "--armor" );
        if( ctx->use_textmode)
            _gpgme_gpg_add_arg (ctx->gpg, "--textmode");
    }
    for (i = 0; (key = gpgme_signers_enum (ctx, i)); i++) 
    {
        const char *s;
	s = gpgme_key_get_string_attr( key, GPGME_ATTR_KEYID, NULL, 0);
        if (s) 
	{
            _gpgme_gpg_add_arg (ctx->gpg, "-u");
            _gpgme_gpg_add_arg_concat (ctx->gpg, s, "!");
        }
        gpgme_key_unref (key);
    }
    
    /* Check the supplied data */
    if( gpgme_data_get_type( in ) == GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( No_Data );
        goto leave;
    }
    _gpgme_data_set_mode(in, GPGME_DATA_MODE_OUT );
    if( !out || gpgme_data_get_type( out ) != GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( Invalid_Value );
        goto leave;
    }
    _gpgme_data_set_mode( out, GPGME_DATA_MODE_IN );
    
    if( ctx->use_tmpfiles ) {
        _gpgme_gpg_add_arg( ctx->gpg, "--output" );
        _gpgme_gpg_add_arg( ctx->gpg, _gpgme_get_tmpfile( 0 ) );
        _gpgme_data_write_to_tmpfile( in );
        _gpgme_gpg_add_arg( ctx->gpg, _gpgme_get_tmpfile( 1 ) );
    }
    else {
        /* Tell the gpg object about the data */
        _gpgme_gpg_add_data( ctx->gpg, in, 0 );
        _gpgme_gpg_add_data( ctx->gpg, out, 1 );
    }
    
    /* and kick off the process */
    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );
    
leave:
    if( rc ) {
        ctx->pending = 0; 
        _gpgme_gpg_release( &ctx->gpg ); 
    }
    return rc;
}


static gpgme_error_t
get_sign_result( gpgme_ctx_t ctx )
{
    struct sign_result_s * res;
    gpgme_error_t err = 0;    

    if( ctx->result_type != RESULT_TYPE_SIGN )
	err = mk_error( General_Error );
    else if( ctx->out_of_core )
        err = mk_error( Out_Of_Core );
    else {
        assert ( ctx->result.sign );
	res = ctx->result.sign;
	if( res->no_passphrase )
	    err = mk_error( No_Passphrase );
	else if( res->bad_passphrase )
	    err = mk_error( Bad_Passphrase );
	else if( res->idea_cipher )
	    err = mk_error( Cipher_IDEA );
	else if( gpgme_get_process_rc( ctx ) )
	    err = mk_error( Interal_GPG_Problem );
	else if( !res->okay )
	    err = mk_error( Signing_Failed );        
    }
    return err;
} /* get_sign_result */


/**
 * gpgme_op_sign:
 * @c: The context
 * @in: Data to be signed
 * @out: Detached signature
 * @mode: Signature creation mode
 * 
 * Create a detached signature for @in and write it to @out.
 * The data will be signed using either the default key or the ones
 * defined through @c.
 * The defined modes for signature create are:
 * <literal>
 * GPGME_SIG_MODE_NORMAL (or 0) 
 * GPGME_SIG_MODE_DETACH
 * GPGME_SIG_MODE_CLEAR
 * </literal>
 * Note that the settings done by gpgme_set_armor() and gpgme_set_textmode()
 * are ignore for @mode GPGME_SIG_MODE_CLEAR.
 * 
 * Return value: 0 on success or an error code.
 **/
gpgme_error_t
gpgme_op_sign( gpgme_ctx_t ctx, gpgme_data_t in, gpgme_data_t out, gpgme_sigmode_t mode )
{
    gpgme_error_t err;
    
    err = sign_start( ctx, in, out, mode );
    if ( !err ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
        if( ctx->use_tmpfiles ) {
            _gpgme_data_read_from_tmpfile( out );
            _gpgme_del_tmpfiles( ctx->wipe_fnc );
        }
        err = get_sign_result( ctx );
    }
    return err;
} /* gpgme_op_sign */


gpgme_error_t
gpgme_op_file_sign( gpgme_ctx_t ctx, gpgme_sigmode_t mode, 
		    const char * input, const char * output )
{
    gpgme_error_t err;
    
    err = file_sign_start( ctx, 0, mode, NULL, input, output );
    if( !err ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
	err = get_sign_result( ctx );
    }
    return err;
} /* gpgme_file_sign */


gpgme_error_t
gpgme_op_files_sign( gpgme_ctx_t ctx, gpgme_sigmode_t mode,
		     const char ** files, size_t nfiles )
{
    gpgme_error_t err;
    size_t i;

    for( i=0; i < nfiles; i++ ) {
	err = gpgme_op_file_sign( ctx, mode, files[i], NULL );
	if( err )
	    break;
    }
    return err;
} /* gpgme_op_files_sign */


gpgme_error_t
gpgme_op_file_sign_encrypt( gpgme_ctx_t ctx, gpgme_recipients_t recp,
			    const char * input, const char * output )
{
    gpgme_error_t err;    
    
    err = file_sign_start( ctx, 1, GPGME_SIG_MODE_NORMAL, recp, input, output );
    if( !err ) {
	gpgme_wait( ctx, 1 );
	ctx->pending = 0;
	err = get_sign_result( ctx );
    }
    return err;
} /* gpgme_op_file_sign_encrypt */


gpgme_error_t
gpgme_op_files_sign_encrypt( gpgme_ctx_t ctx, gpgme_recipients_t recp,
			     const char ** files, size_t nfiles )
{
    gpgme_error_t err;
    size_t i;

    for( i=0; i < nfiles; i++ ) {
	err = gpgme_op_file_sign_encrypt( ctx, recp, files[i], NULL );
	if( err )
	    break;
    }
    return err;
} /* gpgme_op_files_sign_encrypt */


gpgme_error_t
gpgme_op_clip_sign( gpgme_ctx_t ctx, gpgme_sigmode_t mode, 
		    const char * keyid, int wraplen )
{
    gpgme_error_t err;
    gpgme_data_t plain = NULL;
    gpgme_data_t sig = NULL;
    gpgme_key_t key = NULL;
    
    if (!keyid)
        return mk_error (General_Error);
    
    gpgme_control (ctx, GPGME_CTRL_ARMOR, 1);
    err = gpgme_data_new_from_clipboard (&plain);
    if (err)
        return err;
    if (wraplen)
        gpgme_data_wrap_lines (&plain, wraplen);
    gpgme_data_write (plain, "\r\n", 1);
	
    key = _gpgme_key_new_fromkeyid (keyid);
    if (!key) {
	gpgme_data_release (sig);
	return mk_error (General_Error);
    }
    
    err = gpgme_data_new (&sig);
    if (!err)
	err = gpgme_signers_add (ctx, key);
    if (!err)
	err = gpgme_op_sign (ctx, plain, sig, mode);
    if (!err) {
	gpgme_data_change_version (&sig);
	gpgme_data_release_and_set_clipboard (sig);
    }

    gpgme_data_release (plain);
    gpgme_key_release (key);
    return err;
} /* gpgme_op_clip_sign */
