/* w95_slow.cpp  95/98 specific slow seeder implementation
 *	Copyright (C) 1999 Martin Grap
 *
 * This file is part of winseed.
 *
 * winseed is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * winseed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include<windows.h>
#include<tlhelp32.h>

#include<string.h>
#include<stdio.h>

#include<util.h>
#include<md5_internal.h>
#include<w95_slow.h>

static const UINT32 WIN95_SLOW_SEEDER_EXPECTED_SIZE = (128 * 1024);


win95_slow_seeder::win95_slow_seeder(tool_help_loader *th)
	: seeder(WIN95_SLOW_SEEDER_EXPECTED_SIZE)
{
	tool_help = th;
}

win95_slow_seeder::win95_slow_seeder(UINT32 internal_buffer_size, tool_help_loader *th)
	: seeder(internal_buffer_size)
{
	tool_help = th;
}

win95_slow_seeder::~win95_slow_seeder()
{
	if (tool_help != NULL)
	{
		delete tool_help;
	}
}

UINT32 win95_slow_seeder::get_expected_seed_size()
{
	return WIN95_SLOW_SEEDER_EXPECTED_SIZE;
}

UINT32 win95_slow_seeder::walk_heaps_of_heaplist(mem_writer& w, DWORD pid, DWORD hid)
{
	UINT32 result = PCP_SUCCESS;
	HEAPENTRY32 heapentry;
	CMD5Internal md5;
	bool stop = false;
	DWORD error;

	do
	{
		heapentry.dwSize = sizeof(HEAPENTRY32);

		if (tool_help->ThHeap32First(&heapentry, pid, hid) == FALSE)
		{
			/* 
				Assumption: If the process with id pid has terminated between the
				time the process snapshot has been taken and the time this member is
				called then this function should fail. On the other hand this is not 
				really an error we therefore ignore it by returning success.
			*/
			break;
		}

		w.add((UINT8 *)&heapentry, sizeof(HEAPENTRY32));

		/*
			walk through all remaining heaps of the list
		*/
		do 
		{
			heapentry.dwSize = sizeof(HEAPENTRY32);

			if (tool_help->ThHeap32Next(&heapentry) == FALSE)
			{
				stop = true;
				error = GetLastError();

				if ((error != ERROR_NO_MORE_FILES) && (error != ERROR_INVALID_DATA ))
				{
					/* 
						The documentation says that ERROR_INVALID_DATA can be caused 
						either by a corrupted heap or by modifications to the heap which
						prevent the function to proceed. The documentation does, on the other
						hand, not mention whether these modifications are an indication
						of an error or whether they are "normal". I assumed that they would
						be normal and it may therefore happen that we return PCP_SUCCESS
						even though the heap is courrupted. I simply wanted to prevent the
						situation that this member function fails half of the time because
						Word does some wierd stuff to its heaps ... .
					*/
					result = PCP_SEEDER_FAILED;
				}
			}
			else
			{
				md5.md5_mem((UINT8 *)&heapentry, sizeof(HEAPENTRY32));

				/* 
					Add only a digest of the structure. This "compresses"
					the output a little bit. I doubt that the struct contains
					more than 80 bits of entropy we should therefore not lose
					anything.
				*/
				w.add(md5.ctx_digest, 10);
			}

		} while(!stop);

	} while (0);

	return result;
}

UINT32 win95_slow_seeder::walk_heaplist_of_process(mem_writer& w,DWORD pid)
{
	UINT32 result = PCP_SUCCESS;

	HANDLE snapshot = (HANDLE)-1;
	HEAPLIST32 heaplist;
	bool stop = false;

	do
	{
		if ((snapshot = tool_help->ThCreateToolhelp32Snapshot(TH32CS_SNAPHEAPLIST, pid)) == (HANDLE)-1)
		{
			/* 
				Assumption: If the process with id pid has terminated between the
				time the process snapshot has been taken and the time this member is
				called then CreateToolhelp32Snapshot fails. On the other hand this
				is not really an error we therefore ignore it.
			*/
			break;
		}

		heaplist.dwSize = sizeof(HEAPLIST32);

		if (tool_help->ThHeap32ListFirst(snapshot, &heaplist) == FALSE)
		{
			/*
				I can not imagine that a process does not have a single heaplist. Windows
				tells us exactly that if GetLastError is equal to ERROR_NO_MORE_FILES.
				In this case we still return PCP_SUCCESS.
			*/
			if (GetLastError() != ERROR_NO_MORE_FILES)
			{
				result = PCP_SEEDER_FAILED;
			}
			break;
		}

		w.add((UINT8 *)&heaplist, sizeof(HEAPLIST32));

		/*
			walk heaps of first heaplist
		*/
		if ((result = walk_heaps_of_heaplist(w, pid, heaplist.th32HeapID)) != PCP_SUCCESS)
		{
			break;
		}

		do
		{
			heaplist.dwSize = sizeof(HEAPLIST32);

			if (tool_help->ThHeap32ListNext(snapshot, &heaplist) == FALSE)
			{
				stop = true;

				/*
					Do not return an error if we simply reached the end of the list of
					heaplists.  
				*/
				if (GetLastError() != ERROR_NO_MORE_FILES)
				{
					result = PCP_SEEDER_FAILED;
				}
			}
			else
			{
				w.add((UINT8 *)&heaplist,sizeof(HEAPLIST32));

				/*
					walk heaps of current list
				*/
				if ((result = walk_heaps_of_heaplist(w, pid, heaplist.th32HeapID)) != PCP_SUCCESS)
				{
					stop = true;
				}
			}

		} while(!stop);

	} while (0);

	if (snapshot != (HANDLE)-1)
	{
		CloseHandle(snapshot);
	}

	return result;
}


UINT32 win95_slow_seeder::walk_modules_of_process(mem_writer& w,DWORD pid)
{
	UINT32 result = PCP_SUCCESS;
	HANDLE snapshot = (HANDLE)-1;
	MODULEENTRY32 module_entry;
	CMD5Internal md5;
	bool stop = false;

	do
	{
		if ((snapshot = tool_help->ThCreateToolhelp32Snapshot(TH32CS_SNAPMODULE, pid)) == (HANDLE)-1)
		{
			/* 
				Assumption: If the process with id pid has terminated between the
				time the process snapshot has been taken and the time this member is
				called then CreateToolhelp32Snapshot fails. On the other hand this
				is not really an error we therefore ignore it.
			*/
			break;
		}

		module_entry.dwSize = sizeof(MODULEENTRY32);

		if (tool_help->ThModule32First(snapshot, &module_entry) == FALSE)
		{
			/*
				If there are no modules for this process (in my opinion there should)
				at least be the .exe itself) we stop but report no error.
			*/
			if (GetLastError() != ERROR_NO_MORE_FILES)
			{
				result = PCP_SEEDER_FAILED;
			}
			break;
		}

		w.add((UINT8 *)&module_entry, sizeof(MODULEENTRY32));

		/*
			walk through all modules of the process
		*/
		do
		{
			module_entry.dwSize = sizeof(MODULEENTRY32);

			if (tool_help->ThModule32Next(snapshot, &module_entry) == FALSE)
			{
				stop = true;

				/*
					If we get ERROR_NO_MORE_FILES we just reached the end of 
					the module list. This is not an error! In any other case
					we return a failure indication.
				*/
				if (GetLastError() != ERROR_NO_MORE_FILES)
				{
					result = PCP_SEEDER_FAILED;
				}

			}
			else
			{
				md5.md5_mem((UINT8 *)&module_entry, sizeof(MODULEENTRY32));

				/* 
					Add only a digest of the structure. This "compresses"
					the output a little bit, as the strucutre contains 
					the path and the filename of the module (each 260
					= _MAX_PATH bytes) 
				*/
				w.add(md5.ctx_digest, 16);
			}

		} while (!stop);

	} while (0);

	if (snapshot != (HANDLE)-1)
	{
		CloseHandle(snapshot);
	}

	return result;
}

UINT32 win95_slow_seeder::walk_threads(mem_writer& w)
{
	UINT32 result = PCP_SUCCESS;
	HANDLE snapshot = (HANDLE)-1;
	THREADENTRY32 thread_entry;
	bool stop = false;

	do
	{
		if ((snapshot = tool_help->ThCreateToolhelp32Snapshot(TH32CS_SNAPTHREAD, 0)) == (HANDLE)-1)
		{
			result = PCP_SEEDER_FAILED;
			break;
		}

		thread_entry.dwSize = sizeof(THREADENTRY32);

		if (tool_help->ThThread32First(snapshot, &thread_entry) == FALSE)
		{
			/* 
				Maybe there are no threads, i.e. GetLastError == ERROR_NO_MORE_FILES. 
				Hard to imagine, but who knows! If there are no threads we still can
				return PCP_SUCCESS
			*/
			if (GetLastError() != ERROR_NO_MORE_FILES)
			{
				result = PCP_SEEDER_FAILED;
			}
			break;
		}

		w.add((UINT8 *)&thread_entry, sizeof(THREADENTRY32));

		/*
			loop through all threads in the system
		*/
		do 
		{
			thread_entry.dwSize = sizeof(THREADENTRY32);

			if (tool_help->ThThread32Next(snapshot, &thread_entry) == FALSE)
			{
				stop = true;
				
				/*
					If we get ERROR_NO_MORE_FILES we reached the end of the list.
					If we get something different something went really wrong
				*/
				if (GetLastError() != ERROR_NO_MORE_FILES)
				{
					result = PCP_SEEDER_FAILED;
				}
			}
			else
			{
				w.add((UINT8 *)&thread_entry, sizeof(THREADENTRY32));
			}

		} while (!stop);

	} while (0);

	if (snapshot != (HANDLE)-1)
	{
		CloseHandle(snapshot);
	}

	return result;
}

UINT32 win95_slow_seeder::walk_processes(mem_writer& w)
{
	UINT32 result = PCP_SUCCESS;
	HANDLE snapshot = (HANDLE)-1;
	PROCESSENTRY32 process_entry;
	bool stop = false;

	do
	{
		if ((snapshot = tool_help->ThCreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0)) == (HANDLE)-1)
		{
			result = PCP_SEEDER_FAILED;
			break;
		}

		process_entry.dwSize = sizeof(PROCESSENTRY32);

		if (tool_help->ThProcess32First(snapshot, &process_entry) == FALSE)
		{
			/*
				If there are no processes in the snapshot (seems pretty impossible to me) 
				we do not return an error. But if Windows 98 insists that there are no
				processes, that's OK for me :-).
			*/
			if (GetLastError() != ERROR_NO_MORE_FILES)
			{
				result = PCP_SEEDER_FAILED;
			}
			break;
		}

		w.add((UINT8 *)&process_entry, sizeof(PROCESSENTRY32));

		/*
			walk module list of first process in list
		*/
		if ((result = walk_modules_of_process(w, process_entry.th32ProcessID)) != PCP_SUCCESS)
		{
			break;
		}

		/*
			walk heap list of first process in list
		*/
		if ((result = walk_heaplist_of_process(w, process_entry.th32ProcessID)) != PCP_SUCCESS)
		{
			break;
		}

		/*
			walk through remaining processes
		*/
		do 
		{
			process_entry.dwSize = sizeof(PROCESSENTRY32);

			if(tool_help->ThProcess32Next(snapshot, &process_entry) == FALSE)
			{
				stop = true;

				/*
					End of list is reached when we get ERROR_NO_MORE_FILES. Anything
					else is a "real" error.
				*/
				if (GetLastError() != ERROR_NO_MORE_FILES)
				{
					result = PCP_SEEDER_FAILED;
				}
			}
			else
			{
				w.add((UINT8 *)&process_entry, sizeof(PROCESSENTRY32));

				/* 
					walk modules of current process in list
				*/
				if ((result = walk_modules_of_process(w, process_entry.th32ProcessID)) != PCP_SUCCESS)
				{
					stop = true;
				}
				else
				{
					/*
						walk heaplist of current process in list
					*/
					if ((result = walk_heaplist_of_process(w, process_entry.th32ProcessID)) != PCP_SUCCESS)
					{
						stop = true;
					}
				}
			}

		} while(!stop);

	} while (0);

	if (snapshot != (HANDLE)-1)
	{
		CloseHandle(snapshot);
	}

	return result;
}

UINT32 win95_slow_seeder::get_seed(UINT8 *seed, UINT32& desired_length)
{
	UINT8 *Buffer = NULL;
	UINT32 result = PCP_SUCCESS, bytes_to_copy;
	mem_writer w;
	CMD5Internal md5;

	do 
	{
		if (seed == NULL)
		{
			result = PCP_NULL_POINTER;
			break;
		}

		if ((Buffer = new UINT8[get_internal_seed_size()]) == NULL)
		{
			result = PCP_SEEDER_NO_MEM;
			break;
		}

		w.reset(Buffer, get_internal_seed_size());

		/* 
			walk threads in system 
		*/
		if ((result = walk_threads(w)) != PCP_SUCCESS)
		{
			break;
		}

		if (w.is_full())
		{
			result = PCP_SEEDER_TOO_SMALL;
			break;
		}

		/* 
			walk all processes in system 
		*/
		if ((result = walk_processes(w)) != PCP_SUCCESS)
		{
			break;
		}

		if (w.is_full())
		{
			result = PCP_SEEDER_TOO_SMALL;
			break;
		}
	}
	while(0);

	if (result == PCP_SUCCESS)
	{
		bytes_to_copy = min(w.get_pos(), desired_length);
		memcpy(seed, Buffer, bytes_to_copy);

		/* 
			In case some people do not want to retrieve the full size seed 
			we xor the hash (which depends on the full seed) to the first  
			16 bytes of the returned buffer. This (hopefully) makes sure   
			that no entropy is lost even if some seed bytes are not        
			returned to the caller.                                        
		*/
		md5.md5_mem(Buffer, get_internal_seed_size());
		pcp_util::mem_xor(seed, bytes_to_copy, md5.ctx_digest, 16);

		desired_length = bytes_to_copy;
	}
	else
	{
		desired_length = 0;
	}

	if (Buffer != NULL)
	{
		pcp_util::zero_memory(Buffer, get_internal_seed_size());
		delete[] Buffer;
	}

	return result;
}
