/* Seahorse
 *
 * Copyright (C) 1999, 2000 Anthony Mulcahy
 *   
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <errno.h>
#include <fcntl.h>
#include <pty.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>


#include "gpg-interface.h"
#include "seahorsesig.h"

static void
show_error (char *error_msg);


static int
gpg_exec			       (char	*argv[],
					char	**output);

static int
gpg_exec_with_passwd		       (char	*argv[],
					int	clipboard,
					int	passwd_fds[],
					char	*passphrase,
					char	*clipboard_string,
					char	**output);


static void
show_error (char *error_msg)
{
	errorbox = gnome_error_dialog ("can't fork");
	gtk_signal_connect (GTK_OBJECT (errorbox), "clicked",
			GTK_SIGNAL_FUNC (on_errorbox_clicked), NULL);
	gtk_signal_connect (GTK_OBJECT (errorbox), "close",
			GTK_SIGNAL_FUNC (on_errorbox_clicked), NULL);
	gtk_widget_show (errorbox);
	for (;;)
		gtk_main_iteration();
}


int
gpg_exec			       (char	*argv[],
					char	**output)
{
	fd_set fdset;
	int op_fds[2];
	int status;
	int select_result, read_len;
	pid_t child;
	char *buf;
	size_t size, alloc_size;
	struct timeval timeout;

	pipe (op_fds);
	
	if (!(child = fork ())) {
		/* in child */

		dup2 (op_fds[1], STDOUT_FILENO);
		dup2 (op_fds[1], STDERR_FILENO);
		close (op_fds[1]);

		execvp ("gpg", argv);

	} else if (child < 0)
		show_error ("can't fork");

	FD_ZERO (&fdset);
	FD_SET (op_fds[0], &fdset);
	close (op_fds[1]);

	timeout.tv_sec = 60;
	timeout.tv_usec = 0;

	size = 0;
	alloc_size = 1024;

	buf = malloc (alloc_size);
	if (buf == NULL)
		show_error ("can't allocate buf");
	
	for (;;) {
		if ((select_result = select (FD_SETSIZE, &fdset, NULL, NULL, &timeout)) < 0) {
			if (errno == EINTR)
				continue;
			break /* otherwise stop */;
		}
		if (select_result == 0)
			break /* timeout 60 seconds, no output for 1 minute */;
		if (size + 1024 > alloc_size) {
			alloc_size += 1024;
			buf = realloc (buf , alloc_size);
			if (buf == NULL)
				show_error ("can't allocate buf\n");
		}
		if ((read_len =  read (op_fds[0], &buf[size], alloc_size - size - 1)) < 0) {
			if (errno == EINTR)
				continue;
			break;
		}
		if (read_len == 0)
			break /* EOF */;
		size += read_len;
	}

	buf[size] = 0;
	close (op_fds[0]);

	*output = buf;
	if (waitpid (child, &status, WNOHANG) < 0)
		show_error ("error in waitpid");

	if (WIFEXITED (status)) {
		return WEXITSTATUS (status);
	}
	else return 1;
}


int
gpg_exec_with_passwd		       (char	*argv[],
					int	clipboard,
					int	passwd_fds[],
					char	*passphrase,
					char	*clipboard_string,
					char	**output)
{
	fd_set fdset;
	int ip_fds[2];
	int op_fds[2];
	int status;
	int select_result, read_len;
	pid_t child;
	char *buf;
	size_t size, alloc_size;
	struct timeval timeout;

	pipe (ip_fds);
	pipe (op_fds);
	
	if (!(child = fork ())) {
		/* in child */

		close (ip_fds[1]);
		close (op_fds[0]);
		close (passwd_fds[1]);

		dup2 (ip_fds[0], STDIN_FILENO);
		dup2 (op_fds[1], STDOUT_FILENO);
		dup2 (op_fds[1], STDERR_FILENO);
		close (ip_fds[0]);
		close (op_fds[1]);

		execvp ("gpg", argv);

	} else if (child < 0)
		show_error ("can't fork");

	FD_ZERO (&fdset);
	FD_SET (op_fds[0], &fdset);
	close (ip_fds[0]);
	close (op_fds[1]);
	close (passwd_fds[0]);

	timeout.tv_sec = 60;
	timeout.tv_usec = 0;

	size = 0;
	alloc_size = 1024;

	buf = malloc (alloc_size);
	if (buf == NULL)
		show_error ("can't allocate buf");
	
	write (passwd_fds[1], passphrase, strlen (passphrase));
	write (passwd_fds[1], "\n", 2);
	close (passwd_fds[1]);

	if (clipboard) {
		write (ip_fds[1], clipboard_string,
				strlen (clipboard_string));
	}
	close (ip_fds[1]);

	for (;;) {
		if ((select_result = select (FD_SETSIZE, &fdset, NULL, NULL, &timeout)) < 0) {
			if (errno == EINTR)
				continue;
			break /* otherwise stop */;
		}
		if (select_result == 0)
			break /* timeout 60 seconds, no output for 1 minute */;
		if (size + 1024 > alloc_size) {
			alloc_size += 1024;
			buf = realloc (buf , alloc_size);
			if (buf == NULL)
				show_error ("can't allocate buf\n");
		}
		if ((read_len =  read (op_fds[0], &buf[size], alloc_size - size - 1)) < 0) {
			if (errno == EINTR)
				continue;
			break;
		}
		if (read_len == 0)
			break /* EOF */;
		size += read_len;
	}

	buf[size] = 0;
	close (op_fds[0]);

	*output = buf;
	if (waitpid (child, &status, WNOHANG) < 0)
		show_error ("error in waitpid");

	if (WIFEXITED (status)) {
		return WEXITSTATUS (status);
	}
	else return 1;
}


int
gpg_sign_encrypt		       (int	sign,
					int	detachedsign,
					int	encrypt,
					int	ascii_arm,
					int	clipboard,
					char	*filename,
					char	*clipboard_string,
					char	*passphrase,
					char	*username,
					char	**output)
{
	char *argv[16];
	char passwd_fd[8];
	int i = 0;
	int passwd_fds[2];

	pipe (passwd_fds);
	sprintf (passwd_fd, "%d", passwd_fds[0]);

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--yes";
	argv[i++] = "--batch";
	argv[i++] = "--comment";
	argv[i++] = "Seahorse v" VERSION " http://www1.kcn.ne.jp/~anthony/seahorse";

	if (sign) {
		argv[i++] = "--sign";
		argv[i++] = "--passphrase-fd";
		argv[i++] = passwd_fd;
	} else if (detachedsign) {
		argv[i++] = "--detach-sign";
		argv[i++] = "--passphrase-fd";
		argv[i++] = passwd_fd;
	}

	if (encrypt) {
		argv[i++] = "--encrypt";
		argv[i++] = "--recipient";
		argv[i++] = username;
	}

	if (ascii_arm)
		argv[i++] = "--armor";

	if (!clipboard) {
		argv[i++] = filename;
	}
	
	argv[i++] = NULL;

	return gpg_exec_with_passwd (argv, clipboard, passwd_fds, passphrase, clipboard_string, output);
}


int
gpg_decrypt			       (int	decrypt,
					int	clipboard,
					int	outputToFile,
					char	*filename,
					char	*clipboard_string,
					char	*passphrase,
					char	**output)
{

	char *argv[12];
	char passwd_fd[8];
	int i = 0;
	int passwd_fds[2];
	char *outfile;

	pipe (passwd_fds);
	sprintf (passwd_fd, "%d", passwd_fds[0]);

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--yes";
	argv[i++] = "--batch";

	if (!clipboard) {
		if ((outfile = malloc (strlen (filename) + 5)) == NULL)
			show_error ("Can't allocate outfile in gpg_decrypt");
		strcpy (outfile, filename);
		if (!strcmp (outfile + strlen (outfile) - 4, ".gpg")) {
				outfile[strlen (outfile) - 4] = '\0';
				}
		else if (!strcmp (outfile + strlen (outfile) - 4, ".asc")) {
				outfile[strlen (outfile) - 4] = '\0';
		}
		if (outputToFile) {
			argv[i++] = "-o";
			argv[i++] = outfile;
		}
	}
			
	if (decrypt) {
		argv[i++] = "--decrypt";
		argv[i++] = "--passphrase-fd";
		argv[i++] = passwd_fd;
	}
	else argv[i++] = "--verify";

	if (!clipboard)
		argv[i++] = filename;
	argv[i++] = NULL;

	return gpg_exec_with_passwd (argv, clipboard, passwd_fds, passphrase, clipboard_string, output);
}


int
gpg_list_keys			       (char	**output)
{
	char *argv[7];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--yes";
	argv[i++] = "--batch";
	argv[i++] = "--list-keys";
	argv[i++] = NULL;

	return gpg_exec (argv, output);
}


int
gpg_recv_key			       (char *keyserver,
					char *keyid,
					char **output)
{
	char *argv[10];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--yes";
	argv[i++] = "--batch";
	argv[i++] = "--keyserver";
	argv[i++] = keyserver;
	argv[i++] = "--recv-keys";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_exec (argv, output);
}


int
gpg_export_key			       (char	*keyfile,
					char	*keyid,
					int	ascii_arm,
					char	**output)
{
	char *argv[11];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--yes";
	argv[i++] = "--batch";
	if (ascii_arm)
		argv[i++] = "--armor";
	argv[i++] = "-o";
	argv[i++] = keyfile;
	argv[i++] = "--export";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_exec (argv, output);
}


int
gpg_import_key			       (char *keyfile,
					char **output)
{
	char *argv[8];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--yes";
	argv[i++] = "--batch";
	argv[i++] = "--import";
	argv[i++] = keyfile;
	argv[i++] = NULL;

	return gpg_exec (argv, output);
}


int
gpg_send_key			       (char *keyserver,
					char *keyid,
					char **output)
{
	char *argv[9];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--yes";
	argv[i++] = "--batch";
	argv[i++] = "--keyserver";
	argv[i++] = keyserver;
	argv[i++] = "--send-keys";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_exec (argv, output);
}


int
gpg_gen_key			       (char	*keytype,
					int	keysize,
					int	expiration,
					char	*realname,
					char	*email,
					char	*comment,
					char	*passphrase,
					GtkWidget *log_textbox)
{
	char buf[2];
	char prvchr = '\0';
	fd_set ready;
	int cntr = 1;
	int done = 0;
	int master;
	int status;
	pid_t child;


	if ((child = forkpty (&master, NULL, NULL, NULL)) < 0) {
		show_error ("ptypair");
	} else if (child == 0) {
		close (master);

		execlp ("gpg", "gpg", "--gen-key", 0);
		exit (1);
	}


	do {
		FD_ZERO (&ready);
		FD_SET (master, &ready);
		select (FD_SETSIZE, &ready, NULL, NULL, NULL);

		if (FD_ISSET (master, &ready)) {
			
			if (read (master, buf, 1) > 0) {
				/*write (STDOUT_FILENO, buf, 1);*/

				if (buf[0] == ' ') {
					if ((prvchr == '?') && (cntr == 1)) {
						write (master, keytype, 1);
						write (master, "\n", 1);
						cntr++;
					} else if ((prvchr == ')') && (cntr == 2)) {
						write (master, "1024\n", 5);
						cntr++;
					} else if ((prvchr == ')') && (cntr == 3)) {
						write (master, "0\n", 2);
						cntr++;
					} else if ((prvchr == '?') && (cntr == 4)) {
						write (master, "y\n", 2);
						cntr++;
					} else if ((prvchr == ':') && (cntr == 5)) {
						write (master, realname, strlen (realname));
						write (master, "\n", 1);
						cntr++;
					} else if ((prvchr == ':') && (cntr == 6)) {
						write (master, email, strlen (email));
						write (master, "\n", 1);
						cntr++;
					} else if ((prvchr == ':') && (cntr == 7)) {
						write (master, comment, strlen (comment));
						write (master, "\n", 1);
						cntr++;
					} else if ((prvchr == '?') && (cntr == 8)) {
						write (master, "o\n", 2);
						cntr++;
					/* Sometimes the passphrase isn't read*/
					/* correctly so 5 attempts are made to*/
					/* write it */
					} else if ((prvchr == ':') && (8 < cntr)
						    && (cntr < 13)) {
						write (master, passphrase,
						       strlen (passphrase));
						write (master, "\n", 1);
						cntr++;
					}
				}
				prvchr = buf[0];
				gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, buf, 1);
				while (gtk_events_pending())
				    gtk_main_iteration();

			} else done = 1;
		}

	} while (!done);

	while (!waitpid (child, &status, 0)) {}

	if (WIFEXITED (status)) {
		return WEXITSTATUS (status);
	}
	else return 1;
}


